
/*******************************************************************************
 * The information contained in this file is confidential and proprietary to
 * QLogic Corporation.  No part of this file may be reproduced or
 * distributed, in any form or by any means for any purpose, without the
 * express written permission of QLogic Corporation.
 *
 * (c) COPYRIGHT 2015 QLogic Corporation, ALL RIGHTS RESERVED.
 *******************************************************************************/
/* **********************************************************
 * Copyright 2015 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * qfle3_mod.c --
 *
 *      Kernel module implementation for native qfle3 driver.
 */

#include "qfle3.h"
#include "qfle3_ecore_init.h"
#include "qfle3_ecore_init_ops.h"
#include "qfle3_57712_int_offsets.h"
#include "qfle3_sm.h"
#include "qfle3_mgmt_api.h"
#ifdef QFLE3_CNIC
#include "qfle3_cnic.h"
#endif

extern vmk_uint32 DRSS[QFLE3_MAX_NUM_OF_PFS];
extern vmk_uint32 RSS[QFLE3_MAX_NUM_OF_PFS];
extern vmk_uint32 dropless_fc;
extern vmk_uint32 sw_fcoe_pri;
extern vmk_uint32 auto_recovery;
extern vmk_uint32 enable_live_grcdump;

#if (ESX_DDK_VERSION >= 2017)
extern vmk_uint32 rss_engine_nr[QFLE3_MAX_NUM_OF_PFS];
#endif


#ifdef QFLE3_SRIOV

extern int relative_pf_num;
extern vmk_uint32 enable_fcoe_queue;
extern vmk_uint32 enable_dcbx;
extern vmk_uint32 iscsi_ol_mtu;
extern vmk_uint32 iscsi_ol_rx_bd_rings;
extern vmk_uint32 iscsi_ol_tx_bd_rings;
char *channel_tlvs_string[] = {
   "CHANNEL_TLV_NONE", /* ends tlv sequence */
   "CHANNEL_TLV_ACQUIRE",
   "CHANNEL_TLV_INIT",
   "CHANNEL_TLV_SETUP_Q",
   "CHANNEL_TLV_SET_Q_FILTERS",
   "CHANNEL_TLV_ACTIVATE_Q",
   "CHANNEL_TLV_DEACTIVATE_Q",
   "CHANNEL_TLV_TEARDOWN_Q",
   "CHANNEL_TLV_CLOSE",
   "CHANNEL_TLV_RELEASE",
   "CHANNEL_TLV_UPDATE_RSS_DEPRECATED",
   "CHANNEL_TLV_PF_RELEASE_VF",
   "CHANNEL_TLV_LIST_END",
   "CHANNEL_TLV_FLR",
   "CHANNEL_TLV_PF_SET_MAC",
   "CHANNEL_TLV_PF_SET_VLAN",
   "CHANNEL_TLV_UPDATE_RSS",
   "CHANNEL_TLV_PHYS_PORT_ID",
   "CHANNEL_TLV_UPDATE_TPA",
   "CHANNEL_TLV_FP_HSI_SUPPORT",
   "CHANNEL_TLV_MAX"
};
#endif //QFLE3_SRIOV


int qfle3_get_link_cfg_idx(struct qfle3_adapter *adapter);
vmk_Bool qfle3_trylock_leader_lock(struct qfle3_adapter *adapter);
void qfle3_recovery_func(vmk_AddrCookie data);
VMK_ReturnStatus qfle3_release_leader_lock(struct qfle3_adapter *adapter);
static void qfle3_drv_info_iscsi_stat(qfle3_adapter *adapter);
static void qfle3_drv_info_fcoe_stat(qfle3_adapter *adapter);
static void qfle3_init_uplink_data(qfle3_adapter * adapter);
void qfle3_read_mf_cfg(struct qfle3_adapter *adapter);
void qfle3_init_rss_tables(qfle3_adapter *adapter);
void qfle3_calculate_filters_count(qfle3_adapter * adapter);
void qfle3_parity_recover(void *adapter);
void qfle3_disable_close_the_gate(struct qfle3_adapter *adapter)
{
   vmk_uint32 val;

   //QFLE3_DBG(NETIF_MSG_IFDOWN, "Disabling \"close the gates\"\n");

   val = REG_RD(adapter, MISC_REG_AEU_GENERAL_MASK);
   val &= ~(MISC_AEU_GENERAL_MASK_REG_AEU_PXP_CLOSE_MASK |
	    MISC_AEU_GENERAL_MASK_REG_AEU_NIG_CLOSE_MASK);
   REG_WR(adapter, MISC_REG_AEU_GENERAL_MASK, val);
}

vmk_Bool
qfle3_reset_is_done(struct qfle3_adapter *adapter, int engine)
{
   vmk_uint32 val = REG_RD(adapter, QFLE3_RECOVERY_GLOB_REG);
   vmk_uint32 bit = engine ?
      QFLE3_PATH1_RST_IN_PROG_BIT : QFLE3_PATH0_RST_IN_PROG_BIT;

   /* return false if bit is set */
   return (val & bit) ? VMK_FALSE: VMK_TRUE;
}


void qfle3_set_os_driver_state(struct qfle3_adapter *adapter, vmk_uint32 state)
{
   vmk_uint32 cur;

   if (!IS_MF_BD(adapter) || !SHMEM2_HAS(adapter, os_driver_state))
      return;

   cur = SHMEM2_RD(adapter, os_driver_state[QFLE3_FW_MB_IDX(adapter)]);
   QFLE3_DBG(QFLE3_DBG_LOAD, "Driver state %08x-->%08x\n",
	     cur, state);

   SHMEM2_WR(adapter, os_driver_state[QFLE3_FW_MB_IDX(adapter)], state);
}


/* acquire split MCP access lock register */
static int
qfle3_acquire_alr(struct qfle3_adapter *adapter)
{
   vmk_uint32 j, val;
   int rc = 0;

   for (j = 0; j < 1000; j++) {
      REG_WR(adapter, MCP_REG_MCPR_ACCESS_LOCK, MCPR_ACCESS_LOCK_LOCK);
      val = REG_RD(adapter, MCP_REG_MCPR_ACCESS_LOCK);
      if (val & MCPR_ACCESS_LOCK_LOCK)
	 break;

      vmk_WorldSleep(10000);
   }
   if (!(val & MCPR_ACCESS_LOCK_LOCK)) {
      QFLE3_ERR("Cannot acquire MCP access lock register\n");
      rc = VMK_BUSY;
   }

   return rc;
}

/* release split MCP access lock register */
static void
qfle3_release_alr(struct qfle3_adapter *adapter)
{
   REG_WR(adapter, MCP_REG_MCPR_ACCESS_LOCK, 0);
}

static inline vmk_uint16
qfle3_igu_ack_int(struct qfle3_adapter *adapter)
{
   vmk_uint32 igu_addr = (BAR_IGU_INTMEM + IGU_REG_SISR_MDPC_WMASK_LSB_UPPER * 8);
   vmk_uint32 result = REG_RD(adapter, igu_addr);

   QFLE3_DBG(QFLE3_DBG_INTR, "read 0x%08x from IGU addr 0x%x\n", result, igu_addr);

   vmk_CPUMemFenceReadWrite();
   return result;
}

inline vmk_uint16
qfle3_ack_int(struct qfle3_adapter * adapter)
{
   vmk_CPUMemFenceReadWrite();
   return qfle3_igu_ack_int(adapter);
}

static void
qfle3_igu_int_disable(struct qfle3_adapter *adapter)
{
   vmk_uint32 val = REG_RD(adapter, IGU_REG_PF_CONFIGURATION);

   val &= ~(IGU_PF_CONF_MSI_MSIX_EN | IGU_PF_CONF_INT_LINE_EN | IGU_PF_CONF_ATTN_BIT_EN);

   // QFLE3_DBG(QFLE3_DBG_IFDOWN, "write %x to IGU\n", val);

   /*
    * flush all outstanding writes
    */
   vmk_CPUMemFenceReadWrite();

   REG_WR(adapter, IGU_REG_PF_CONFIGURATION, val);
   if (REG_RD(adapter, IGU_REG_PF_CONFIGURATION) != val)
      QFLE3_ERR("Proper val not read from IGU!\n");
}

void
qfle3_int_disable(struct qfle3_adapter *adapter)
{
   adapter->hw_int_disabled = VMK_TRUE;
   qfle3_igu_int_disable(adapter);
   
}

static void
qfle3_igu_int_enable(struct qfle3_adapter *adapter)
{
   vmk_uint32 val;
   vmk_Bool msix = (adapter->flags & USING_MSIX_FLAG) ? VMK_TRUE : VMK_FALSE;
   vmk_Bool single_msix =
      (adapter->flags & USING_SINGLE_MSIX_FLAG) ? VMK_TRUE : VMK_FALSE;
   vmk_Bool msi = (adapter->flags & USING_MSI_FLAG) ? VMK_TRUE : VMK_FALSE;

   val = REG_RD(adapter, IGU_REG_PF_CONFIGURATION);

   if (msix) {
      val &= ~(IGU_PF_CONF_INT_LINE_EN | IGU_PF_CONF_SINGLE_ISR_EN);
      val |= (IGU_PF_CONF_MSI_MSIX_EN | IGU_PF_CONF_ATTN_BIT_EN);

      if (single_msix)
	 val |= IGU_PF_CONF_SINGLE_ISR_EN;
   } else if (msi) {
      val &= ~IGU_PF_CONF_INT_LINE_EN;
      val |= (IGU_PF_CONF_MSI_MSIX_EN |
	      IGU_PF_CONF_ATTN_BIT_EN | IGU_PF_CONF_SINGLE_ISR_EN);
   } else {
      val &= ~IGU_PF_CONF_MSI_MSIX_EN;
      val |= (IGU_PF_CONF_INT_LINE_EN |
	      IGU_PF_CONF_ATTN_BIT_EN | IGU_PF_CONF_SINGLE_ISR_EN);
   }

   /*
    * Clean previous status - need to configure igu prior to ack
    */
   if ((!msix) || single_msix) {
      REG_WR(adapter, IGU_REG_PF_CONFIGURATION, val);
      qfle3_ack_int(adapter);
   }

   val |= IGU_PF_CONF_FUNC_EN;

   QFLE3_DBG(QFLE3_DBG_INTR, "write 0x%x to IGU  mode %s\n",
	     val, (msix ? "MSI-X" : (msi ? "MSI" : "INTx")));

   REG_WR(adapter, IGU_REG_PF_CONFIGURATION, val);

   if (val & IGU_PF_CONF_INT_LINE_EN) {
      vmk_uint32 pci_command, new;
      VMK_ReturnStatus status;

      status = vmk_PCIReadConfig(vmk_ModuleCurrentID,
				 adapter->pdev,
				 VMK_PCI_CONFIG_ACCESS_16, PCI_COMMAND, &pci_command);

      if (status != VMK_OK) {
	 QFLE3_ERR("Error reading PCI command register\n");
      }

      new = pci_command & ~PCI_COMMAND_INTX_DISABLE;

      if (new != pci_command) {
	 status = vmk_PCIWriteConfig(vmk_ModuleCurrentID,
				     adapter->pdev,
				     VMK_PCI_CONFIG_ACCESS_16, PCI_COMMAND, new);
	 if (status != VMK_OK) {
	    QFLE3_ERR("Error writing PCI command register\n");
	 }
      }
   }

   vmk_CPUMemFenceReadWrite();

   /*
    * init leading/trailing edge
    */
   if (IS_MF(adapter)) {
      val = (0xee0f | (1 << (QFLE3_VN(adapter) + 4)));
      if (adapter->port.pmf)
	 /*
	  * enable nig and gpio3 attention
	  */
	 val |= 0x1100;
   } else
      val = 0xffff;

   REG_WR(adapter, IGU_REG_TRAILING_EDGE_LATCH, val);
   REG_WR(adapter, IGU_REG_LEADING_EDGE_LATCH, val);

   /*
    * Make sure that interrupts are indeed enabled from here on
    */
   vmk_CPUMemFenceReadWrite();
}

void
qfle3_int_enable(struct qfle3_adapter *adapter)
{
   
   qfle3_igu_int_enable(adapter);
   adapter->hw_int_disabled = VMK_FALSE;
}

static inline void
qfle3_hc_ack_sb(struct qfle3_adapter *adapter, vmk_uint8 sb_id,
		vmk_uint8 storm, vmk_uint16 index, vmk_uint8 op, vmk_uint8 update)
{
   vmk_uint32 hc_addr = (HC_REG_COMMAND_REG + QFLE3_PORT(adapter) * 32 +
			 COMMAND_REG_INT_ACK);
   struct igu_ack_register igu_ack;

   igu_ack.status_block_index = index;
   igu_ack.sb_id_and_flags =
      ((sb_id << IGU_ACK_REGISTER_STATUS_BLOCK_ID_SHIFT) |
       (storm << IGU_ACK_REGISTER_STORM_ID_SHIFT) |
       (update << IGU_ACK_REGISTER_UPDATE_INDEX_SHIFT) |
       (op << IGU_ACK_REGISTER_INTERRUPT_MODE_SHIFT));

   REG_WR(adapter, hc_addr, (*(vmk_uint32 *) & igu_ack));

   /*
    * Make sure that ACK is written
    */
   vmk_CPUMemFenceReadWrite();
}

static inline void
qfle3_igu_ack_sb_gen(struct qfle3_adapter *adapter, vmk_uint8 igu_sb_id,
		     vmk_uint8 segment, vmk_uint16 index, vmk_uint8 op,
		     vmk_uint8 update, vmk_uint32 igu_addr)
{
   struct igu_regular cmd_data = { 0 };

   cmd_data.sb_id_and_flags =
      ((index << IGU_REGULAR_SB_INDEX_SHIFT) |
       (segment << IGU_REGULAR_SEGMENT_ACCESS_SHIFT) |
       (update << IGU_REGULAR_BUPDATE_SHIFT) | (op << IGU_REGULAR_ENABLE_INT_SHIFT));

   //QFLE3_DBG(QFLE3_DBG_INTR, "write 0x%08x to IGU addr 0x%x\n",
   //  cmd_data.sb_id_and_flags, igu_addr);
   REG_WR(adapter, igu_addr, cmd_data.sb_id_and_flags);

   /*
    * Make sure that ACK is written
    */
   vmk_CPUMemFenceReadWrite();
}



void
qfle3_igu_ack_sb(struct qfle3_adapter *adapter, vmk_uint8 igu_sb_id, vmk_uint8 segment,
		 vmk_uint16 index, vmk_uint8 op, vmk_uint8 update)
{
   vmk_uint32 igu_addr = adapter->igu_base_addr;
   igu_addr += (IGU_CMD_INT_ACK_BASE + igu_sb_id) * 8;
   qfle3_igu_ack_sb_gen(adapter, igu_sb_id, segment, index, op, update, igu_addr);
}

inline void
qfle3_ack_sb(struct qfle3_adapter *adapter, vmk_uint8 igu_sb_id, vmk_uint8 storm,
	     vmk_uint16 index, vmk_uint8 op, vmk_uint8 update)
{
   if (adapter->hw_info.int_block == INT_BLOCK_HC)
      qfle3_hc_ack_sb(adapter, igu_sb_id, storm, index, op, update);
   else {
      vmk_uint8 segment;

      if (CHIP_INT_MODE_IS_BC(adapter))
	 segment = storm;
      else if (igu_sb_id != adapter->igu_dsb_id)
	 segment = IGU_SEG_ACCESS_DEF;
      else if (storm == ATTENTION_ID)
	 segment = IGU_SEG_ACCESS_ATTN;
      else
	 segment = IGU_SEG_ACCESS_DEF;
      qfle3_igu_ack_sb(adapter, igu_sb_id, segment, index, op, update);
   }
}

static void
qfle3_fill(struct qfle3_adapter *adapter, vmk_uint32 addr, int fill, vmk_uint32 len)
{
   vmk_uint32 i;
   if (!(len % 4) && !(addr % 4))
      for (i = 0; i < len; i += 4)
	 REG_WR(adapter, addr + i, fill);
   else
      for (i = 0; i < len; i++)
	 REG_WR8(adapter, addr + i, fill);
}

static void
qfle3_wr_sp_sb_data(struct qfle3_adapter *adapter,
		    struct hc_sp_status_block_data *sp_sb_data)
{
   int func = QFLE3_FUNC(adapter);
   int i;
   for (i = 0; i < sizeof(struct hc_sp_status_block_data) / sizeof(vmk_uint32); i++)
      REG_WR(adapter, BAR_CSTRORM_INTMEM +
	     CSTORM_SP_STATUS_BLOCK_DATA_OFFSET(func) +
	     i * sizeof(vmk_uint32), *((vmk_uint32 *) sp_sb_data + i));
}

static void
qfle3_zero_sp_sb(struct qfle3_adapter *adapter)
{
   int func = QFLE3_FUNC(adapter);
   struct hc_sp_status_block_data sp_sb_data;
   vmk_Memset(&sp_sb_data, 0, sizeof(struct hc_sp_status_block_data));

   sp_sb_data.state = SB_DISABLED;
   sp_sb_data.p_func.vf_valid = VMK_FALSE;

   qfle3_wr_sp_sb_data(adapter, &sp_sb_data);

   qfle3_fill(adapter, BAR_CSTRORM_INTMEM +
	      CSTORM_SP_STATUS_BLOCK_OFFSET(func), 0, CSTORM_SP_STATUS_BLOCK_SIZE);
   qfle3_fill(adapter, BAR_CSTRORM_INTMEM +
	      CSTORM_SP_SYNC_BLOCK_OFFSET(func), 0, CSTORM_SP_SYNC_BLOCK_SIZE);
}

static void
qfle3_init_def_sb(struct qfle3_adapter *adapter)
{
   struct host_sp_status_block *def_sb = adapter->def_status_blk;
   vmk_IOA mapping = adapter->def_status_blk_mapping;
   int igu_sp_sb_index;
   int igu_seg_id;
   int port = QFLE3_PORT(adapter);
   int func = QFLE3_FUNC(adapter);
   int reg_offset, reg_offset_en5;
   vmk_uint64 section;
   int index;
   struct hc_sp_status_block_data sp_sb_data;
   vmk_Memset(&sp_sb_data, 0, sizeof(struct hc_sp_status_block_data));

   if (CHIP_INT_MODE_IS_BC(adapter)) {
      igu_sp_sb_index = DEF_SB_IGU_ID;
      igu_seg_id = HC_SEG_ACCESS_DEF;
   } else {
      igu_sp_sb_index = adapter->igu_dsb_id;
      igu_seg_id = IGU_SEG_ACCESS_DEF;
   }

   /*
    * ATTN
    */
   section = ((vmk_uint64) mapping) + vmk_offsetof(struct host_sp_status_block,
						   atten_status_block);
   def_sb->atten_status_block.status_block_id = igu_sp_sb_index;

   adapter->attn_state = 0;

   reg_offset = (port ? MISC_REG_AEU_ENABLE1_FUNC_1_OUT_0 :
		 MISC_REG_AEU_ENABLE1_FUNC_0_OUT_0);
   reg_offset_en5 = (port ? MISC_REG_AEU_ENABLE5_FUNC_1_OUT_0 :
		     MISC_REG_AEU_ENABLE5_FUNC_0_OUT_0);
   for (index = 0; index < MAX_DYNAMIC_ATTN_GRPS; index++) {
      int sindex;
      /*
       * take care of sig[0]..sig[4]
       */
      for (sindex = 0; sindex < 4; sindex++)
	 adapter->attn_group[index].sig[sindex] =
	    REG_RD(adapter, reg_offset + sindex * 0x4 + 0x10 * index);

      /*
       * enable5 is separate from the rest of the registers,
       * and therefore the address skip is 4
       * and not 16 between the different groups
       */
      adapter->attn_group[index].sig[4] = REG_RD(adapter, reg_offset_en5 + 0x4 * index);
   }

   if (adapter->hw_info.int_block == INT_BLOCK_HC) {
      reg_offset = (port ? HC_REG_ATTN_MSG1_ADDR_L : HC_REG_ATTN_MSG0_ADDR_L);

      REG_WR(adapter, reg_offset, U64_LO(section));
      REG_WR(adapter, reg_offset + 4, U64_HI(section));
   } else if (!CHIP_IS_E1x(adapter)) {
      REG_WR(adapter, IGU_REG_ATTN_MSG_ADDR_L, U64_LO(section));
      REG_WR(adapter, IGU_REG_ATTN_MSG_ADDR_H, U64_HI(section));
   }

   section = ((vmk_uint64) mapping) + vmk_offsetof(struct host_sp_status_block, sp_sb);

   qfle3_zero_sp_sb(adapter);

   /*
    * PCI guarantees endianity of regpairs
    */
   sp_sb_data.state = SB_ENABLED;
   sp_sb_data.host_sb_addr.lo = U64_LO(section);
   sp_sb_data.host_sb_addr.hi = U64_HI(section);
   sp_sb_data.igu_sb_id = igu_sp_sb_index;
   sp_sb_data.igu_seg_id = igu_seg_id;
   sp_sb_data.p_func.pf_id = func;
   sp_sb_data.p_func.vnic_id = QFLE3_VN(adapter);
   sp_sb_data.p_func.vf_id = 0xff;

   qfle3_wr_sp_sb_data(adapter, &sp_sb_data);

   qfle3_ack_sb(adapter, adapter->igu_dsb_id, USTORM_ID, 0, IGU_INT_ENABLE, 0);
}

#define QFLE3_DEF_SB_ATT_IDX    0x0001
#define QFLE3_DEF_SB_IDX    0x0002

static vmk_uint16
qfle3_update_dsb_idx(struct qfle3_adapter *adapter)
{
   struct host_sp_status_block *def_sb = adapter->def_status_blk;
   vmk_uint16 rc = 0;

   vmk_CPUMemFenceReadWrite();
   if (adapter->def_att_idx != def_sb->atten_status_block.attn_bits_index) {
      adapter->def_att_idx = def_sb->atten_status_block.attn_bits_index;
      rc |= QFLE3_DEF_SB_ATT_IDX;
   }

   if (adapter->def_idx != def_sb->sp_sb.running_index) {
      adapter->def_idx = def_sb->sp_sb.running_index;
      rc |= QFLE3_DEF_SB_IDX;
   }

   /*
    * Do not reorder: indices reading should complete before handling
    */
   vmk_CPUMemFenceReadWrite();
   return rc;
}


static VMK_ReturnStatus 
qfle3_init_sp_ring(struct qfle3_adapter *adapter)
{
   VMK_ReturnStatus status = VMK_OK;
   status = qfle3_create_spinlock("spq_lock",
			 QFLE3_NAME_TO_STRING(adapter->pdev_name),
			 QFLE3_LOCK_RANK_HIGH, &adapter->spq_lock);
   if (status != VMK_OK)
      return status;

   vmk_AtomicWrite64(&adapter->cq_spq_left, MAX_SPQ_PENDING);

   adapter->spq_prod_idx = 0;
   adapter->dsb_sp_prod = QFLE3_SP_DSB_INDEX;
   adapter->spq_prod_bd = adapter->spq;
   adapter->spq_last_bd = adapter->spq_prod_bd + MAX_SP_DESC_CNT;
   return VMK_OK;
}

static void
qfle3_init_eq_ring(struct qfle3_adapter *adapter)
{
   int i;
   for (i = 1; i <= NUM_EQ_PAGES; i++) {
      union event_ring_elem *elem = &adapter->eq_ring[EQ_DESC_CNT_PAGE * i - 1];

      elem->next_page.addr.hi =
	 htole32(U64_HI(adapter->eq_mapping + QFLE3_PAGE_SIZE * (i % NUM_EQ_PAGES)));
      elem->next_page.addr.lo =
	 htole32(U64_LO(adapter->eq_mapping + QFLE3_PAGE_SIZE * (i % NUM_EQ_PAGES)));
   }
   adapter->eq_cons = 0;
   adapter->eq_prod = NUM_EQ_DESC;
   adapter->eq_cons_sb = QFLE3_EQ_INDEX;
   /*
    * we want a warning message before it gets wrought...
    */
   vmk_AtomicWrite64(&adapter->eq_spq_left,
		     MIN_OF(MAX_SP_DESC_CNT - MAX_SPQ_PENDING, NUM_EQ_DESC) - 1);
}

static inline vmk_uint8
qfle3_get_path_func_num(struct qfle3_adapter *adapter)
{
   vmk_uint8 func_num = 0, i;

   for (i = 0; i < E1H_FUNC_MAX / 2; i++) {
      vmk_uint32 func_config =
	 MF_CFG_RD(adapter,
		   func_mf_config[QFLE3_PORT(adapter) + 2 * i].
		   config);
      func_num +=
	 ((func_config & FUNC_MF_CFG_FUNC_HIDE) ? 0 : 1);
   }

   return func_num;
}

static inline void
qfle3_init_sp_objs(struct qfle3_adapter *adapter)
{
   /* RX_MODE controlling object */
   ecore_init_rx_mode_obj(adapter, &adapter->rx_mode_obj);

   /* multicast configuration controlling object */
   ecore_init_mcast_obj(adapter, &adapter->mcast_obj, adapter->fp->cl_id, adapter->fp->cid,
			QFLE3_FUNC(adapter), QFLE3_FUNC(adapter),
			QFLE3_SP(adapter, mcast_rdata),
			QFLE3_SP_MAPPING(adapter, mcast_rdata),
			ECORE_FILTER_MCAST_PENDING, &adapter->sp_state,
			ECORE_OBJ_TYPE_RX);

   /* Setup CAM credit pools */
   ecore_init_mac_credit_pool(adapter, &adapter->macs_pool, QFLE3_FUNC(adapter),
			      qfle3_get_path_func_num(adapter));

   ecore_init_vlan_credit_pool(adapter, &adapter->vlans_pool, QFLE3_FUNC(adapter),
			       qfle3_get_path_func_num(adapter));
#if (ESX_DDK_VERSION < 2017)
   /* RSS configuration object */
   ecore_init_rss_config_obj(adapter, &adapter->netq_rss_conf_obj, adapter->fp->cl_id,
			     adapter->fp->cid, QFLE3_FUNC(adapter), QFLE3_FUNC(adapter),
			     QFLE3_SP(adapter, rss_rdata),
			     QFLE3_SP_MAPPING(adapter, rss_rdata),
			     ECORE_FILTER_RSS_CONF_PENDING, &adapter->sp_state,
			     ECORE_OBJ_TYPE_RX);

   ecore_init_rss_config_obj(adapter, &adapter->defq_rss_conf_obj, adapter->fp->cl_id,
                             adapter->fp->cid, QFLE3_FUNC(adapter),
			     QFLE3_FUNC(adapter) + (2 * E2_FUNC_MAX),
                             QFLE3_SP(adapter, defq_rss_rdata),
                             QFLE3_SP_MAPPING(adapter, defq_rss_rdata),
                             ECORE_FILTER_RSS_CONF_PENDING, &adapter->sp_state,
                             ECORE_OBJ_TYPE_RX);
#endif
#if 0
   adapter->vlan_credit = PF_VLAN_CREDIT_E2(adapter, qfle3_get_path_func_num(adapter));
#endif
}

static inline vmk_uint32
reg_poll(struct qfle3_adapter *adapter, vmk_uint32 reg, vmk_uint32 expected, int ms,
	 int wait)
{
   vmk_uint32 val;

   do {
      val = REG_RD(adapter, reg);
      if (val == expected)
	 break;
      ms -= wait;
      vmk_WorldSleep(wait);

   } while (ms > 0);

   return val;
}

static void
qfle3_clean_pglue_errors(struct qfle3_adapter *adapter)
{
   if (!CHIP_IS_E1x(adapter))
      REG_WR(adapter, PGLUE_B_REG_WAS_ERROR_PF_7_0_CLR, 1 << QFLE3_ABS_FUNC(adapter));
}

VMK_ReturnStatus
qfle3_acquire_hw_lock(struct qfle3_adapter *adapter, vmk_uint32 resource)
{
   vmk_uint32 lock_status;
   vmk_uint32 resource_bit = (1 << resource);
   vmk_int32 func = QFLE3_FUNC(adapter);
   vmk_uint32 hw_lock_control_reg;
   vmk_uint32 cnt;

   /*
    * Validating that the resource is within range
    */
   if (resource > HW_LOCK_MAX_RESOURCE_VALUE) {
      QFLE3_ERR("resource(0x%x) > HW_LOCK_MAX_RESOURCE_VALUE(0x%x)\n",
		resource, HW_LOCK_MAX_RESOURCE_VALUE);
      return VMK_FAILURE;
   }

   if (func <= 5) {
      hw_lock_control_reg = (MISC_REG_DRIVER_CONTROL_1 + func * 8);
   } else {
      hw_lock_control_reg = (MISC_REG_DRIVER_CONTROL_7 + (func - 6) * 8);
   }

   /*
    * Validating that the resource is not already taken
    */
   lock_status = REG_RD(adapter, hw_lock_control_reg);
   if (lock_status & resource_bit) {
      QFLE3_ERR("lock_status 0x%x  resource_bit 0x%x\n", lock_status, resource_bit);
      return VMK_FAILURE;
   }
   /*
    * Try for 5 second every 5ms
    */
   for (cnt = 0; cnt < 1000; cnt++) {
      /*
       * Try to acquire the lock
       */
      REG_WR(adapter, hw_lock_control_reg + 4, resource_bit);
      lock_status = REG_RD(adapter, hw_lock_control_reg);
      if (lock_status & resource_bit)
	 return VMK_OK;

      vmk_WorldSleep(5000);
   }
   QFLE3_ERR("Timeout\n");
   return VMK_FAILURE;
}

VMK_ReturnStatus
qfle3_release_hw_lock(struct qfle3_adapter * adapter, vmk_uint32 resource)
{
   vmk_uint32 lock_status;
   vmk_uint32 resource_bit = (1 << resource);
   vmk_uint32 func = QFLE3_FUNC(adapter);
   vmk_uint32 hw_lock_control_reg;

   /*
    * Validating that the resource is within range
    */
   if (resource > HW_LOCK_MAX_RESOURCE_VALUE) {
      QFLE3_ERR("resource(0x%x) > HW_LOCK_MAX_RESOURCE_VALUE(0x%x)\n",
		resource, HW_LOCK_MAX_RESOURCE_VALUE);
      return VMK_FAILURE;
   }

   if (func <= 5) {
      hw_lock_control_reg = (MISC_REG_DRIVER_CONTROL_1 + func * 8);
   } else {
      hw_lock_control_reg = (MISC_REG_DRIVER_CONTROL_7 + (func - 6) * 8);
   }

   /*
    * Validating that the resource is currently taken
    */
   lock_status = REG_RD(adapter, hw_lock_control_reg);
   if (!(lock_status & resource_bit)) {
      QFLE3_INFO
	 ("lock_status 0x%x resource_bit 0x%x. Unlock was called but lock wasn't taken!\n",
	  lock_status, resource_bit);
      return VMK_FAILURE;
   }

   REG_WR(adapter, hw_lock_control_reg, resource_bit);
   return VMK_OK;
}

/* HW Lock for shared dual port PHYs */
void
qfle3_acquire_phy_lock(struct qfle3_adapter *adapter)
{
   vmk_SemaLock(&adapter->port.phy_mutex);

   qfle3_acquire_hw_lock(adapter, HW_LOCK_RESOURCE_MDIO);
}

void
qfle3_release_phy_lock(struct qfle3_adapter *adapter)
{
   qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_MDIO);

   vmk_SemaUnlock(&adapter->port.phy_mutex);
}

static int
qfle3_init_firmware(struct qfle3_adapter *adapter)
{
   ecore_init_e2_firmware(adapter);
   adapter->iro_arr = e2_iro_arr;
   return 0;
}

static void
qfle3_release_firmware(struct qfle3_adapter *adapter)
{
   /*
    * Do nothing
    */
   return;
}

/* gzip service functions */
static int
qfle3_gunzip_init(struct qfle3_adapter *adapter)
{
   adapter->gz_buf =
      qfle3_alloc_dma_mapping(adapter, adapter->dmaEngineCoherent,
			      FW_BUF_SIZE, &adapter->gz_dma_addr);

   if (!adapter->gz_buf) {
      QFLE3_ERR("Failed to allocate memory " "for firmware decompression buffer.\n");
      return -1;
   }
   return 0;
}

/* gzip service functions */
static void
qfle3_gunzip_end(struct qfle3_adapter *adapter)
{
   if (adapter->gz_buf) {
      qfle3_free_dma_mapping(adapter,
			     adapter->dmaEngineCoherent,
			     adapter->gz_buf, adapter->gz_dma_addr, FW_BUF_SIZE);
      adapter->gz_buf = NULL;
   }
}

void
qfle3_ilt_wr(struct qfle3_adapter *adapter, vmk_uint32 index, vmk_IOA addr)
{
   int reg;
   vmk_uint32 wb_write[2];

   reg = PXP2_REG_RQ_ONCHIP_AT_B0 + index * 8;

   wb_write[0] = ONCHIP_ADDR1(addr);
   wb_write[1] = ONCHIP_ADDR2(addr);
   REG_WR_DMAE(adapter, reg, wb_write, 2);
}

void
qfle3_igu_clear_sb_gen(struct qfle3_adapter *adapter, vmk_uint8 func,
		       vmk_uint8 idu_sb_id, vmk_Bool is_pf)
{
   vmk_uint32 data, ctl, cnt = 100;
   vmk_uint32 igu_addr_data = IGU_REG_COMMAND_REG_32LSB_DATA;
   vmk_uint32 igu_addr_ctl = IGU_REG_COMMAND_REG_CTRL;
   vmk_uint32 igu_addr_ack = IGU_REG_CSTORM_TYPE_0_SB_CLEANUP + (idu_sb_id / 32) * 4;
   vmk_uint32 sb_bit = 1 << (idu_sb_id % 32);
   vmk_uint32 func_encode = func | (is_pf ? 1 : 0) << IGU_FID_ENCODE_IS_PF_SHIFT;
   vmk_uint32 addr_encode = IGU_CMD_E2_PROD_UPD_BASE + idu_sb_id;

   /*
    * Not supported in BC mode
    */
   if (CHIP_INT_MODE_IS_BC(adapter))
      return;

   data = (IGU_USE_REGISTER_cstorm_type_0_sb_cleanup
	   << IGU_REGULAR_CLEANUP_TYPE_SHIFT) |
      IGU_REGULAR_CLEANUP_SET | IGU_REGULAR_BCLEANUP;

   ctl = addr_encode << IGU_CTRL_REG_ADDRESS_SHIFT |
      func_encode << IGU_CTRL_REG_FID_SHIFT |
      IGU_CTRL_CMD_TYPE_WR << IGU_CTRL_REG_TYPE_SHIFT;

   REG_WR(adapter, igu_addr_data, data);
   vmk_CPUMemFenceReadWrite();

   REG_WR(adapter, igu_addr_ctl, ctl);
   vmk_CPUMemFenceReadWrite();
   /*
    * wait for clean up to finish
    */
   while (!(REG_RD(adapter, igu_addr_ack) & sb_bit) && --cnt)
      vmk_WorldSleep(20 * 1000);

   if (!(REG_RD(adapter, igu_addr_ack) & sb_bit)) {
      QFLE3_ERR
	 ("Unable to finish IGU cleanup: idu_sb_id %d offset %d bit %d (cnt %d)\n",
	  idu_sb_id, idu_sb_id / 32, idu_sb_id % 32, cnt);
   }
}

static void
qfle3_igu_clear_sb(struct qfle3_adapter *adapter, vmk_uint8 idu_sb_id)
{
   qfle3_igu_clear_sb_gen(adapter, QFLE3_FUNC(adapter), idu_sb_id, VMK_TRUE /*PF*/);
   QFLE3_DBG(QFLE3_DBG_HW, "Clear IGU SB called\n");
}

static void
qfle3_clear_func_ilt(struct qfle3_adapter *adapter, vmk_uint32 func)
{
   vmk_uint32 i, base = FUNC_ILT_BASE(func);
   for (i = base; i < base + ILT_PER_FUNC; i++)
      qfle3_ilt_wr(adapter, i, 0);
}

void
qfle3_pf_disable(struct qfle3_adapter *adapter)
{
   vmk_uint32 val = REG_RD(adapter, IGU_REG_PF_CONFIGURATION);
   val &= ~IGU_PF_CONF_FUNC_EN;

   REG_WR(adapter, IGU_REG_PF_CONFIGURATION, val);
   REG_WR(adapter, PGLUE_B_REG_INTERNAL_PFID_ENABLE_MASTER, 0);
   REG_WR(adapter, CFC_REG_WEAK_ENABLE_PF, 0);
}

static void
qfle3_reset_func(struct qfle3_adapter *adapter)
{
   int port = QFLE3_PORT(adapter);
   int func = QFLE3_FUNC(adapter);
   int i;

   QFLE3_DBG(QFLE3_DBG_HW, "Reset func called\n");

   /*
    * Disable the function in the FW
    */
   REG_WR8(adapter, BAR_XSTRORM_INTMEM + XSTORM_FUNC_EN_OFFSET(func), 0);
   REG_WR8(adapter, BAR_CSTRORM_INTMEM + CSTORM_FUNC_EN_OFFSET(func), 0);
   REG_WR8(adapter, BAR_TSTRORM_INTMEM + TSTORM_FUNC_EN_OFFSET(func), 0);
   REG_WR8(adapter, BAR_USTRORM_INTMEM + USTORM_FUNC_EN_OFFSET(func), 0);

   /*
    * FP SBs
    */
#if 0
   for_each_eth_queue(adapter, i) {
      struct qfle3_fastpath *fp = &adapter->fp[i];
      REG_WR8(adapter, BAR_CSTRORM_INTMEM +
	      CSTORM_STATUS_BLOCK_DATA_STATE_OFFSET(fp->fw_sb_id), SB_DISABLED);
   }
#endif

   /*
    * SP SB
    */
   REG_WR8(adapter, BAR_CSTRORM_INTMEM +
	   CSTORM_SP_STATUS_BLOCK_DATA_STATE_OFFSET(func), SB_DISABLED);

   for (i = 0; i < XSTORM_SPQ_DATA_SIZE / 4; i++)
      REG_WR(adapter, BAR_XSTRORM_INTMEM + XSTORM_SPQ_DATA_OFFSET(func), 0);

   /*
    * Configure IGU
    */
   if (adapter->hw_info.int_block == INT_BLOCK_HC) {
      REG_WR(adapter, HC_REG_LEADING_EDGE_0 + port * 8, 0);
      REG_WR(adapter, HC_REG_TRAILING_EDGE_0 + port * 8, 0);
   } else {
      REG_WR(adapter, IGU_REG_LEADING_EDGE_LATCH, 0);
      REG_WR(adapter, IGU_REG_TRAILING_EDGE_LATCH, 0);
   }

   /*
    * Clear ILT
    */
   qfle3_clear_func_ilt(adapter, func);

   /*
    * Timers workaround bug for E2: if this is vnic-3,
    * * we need to set the entire ilt range for this timers.
    */
   if (!CHIP_IS_E1x(adapter) && QFLE3_VN(adapter) == 3) {
      struct ilt_client_info ilt_cli;
      /*
       * use dummy TM client
       */
      vmk_Memset(&ilt_cli, 0, sizeof(struct ilt_client_info));
      ilt_cli.start = 0;
      ilt_cli.end = ILT_NUM_PAGE_ENTRIES - 1;
      ilt_cli.client_num = ILT_CLIENT_TM;

      ecore_ilt_boundry_init_op(adapter, &ilt_cli, 0, INITOP_CLEAR);
   }


   /*
    * this assumes that reset_port() called before reset_func()
    */
   if (!CHIP_IS_E1x(adapter))
      qfle3_pf_disable(adapter);

   adapter->dmae_ready = 0;
}

void qfle3_calc_fc_adv(struct qfle3_adapter *adapter)
{
   vmk_uint8 cfg_idx = qfle3_get_link_cfg_idx(adapter);

   adapter->port.advertising[cfg_idx] &= ~(ELINK_ADVERTISED_Asym_Pause |
					   ELINK_ADVERTISED_Pause);
   switch (adapter->link_vars.ieee_fc &
	   MDIO_COMBO_IEEE0_AUTO_NEG_ADV_PAUSE_MASK) {
   case MDIO_COMBO_IEEE0_AUTO_NEG_ADV_PAUSE_BOTH:
      adapter->port.advertising[cfg_idx] |= (ELINK_ADVERTISED_Asym_Pause |
					     ELINK_ADVERTISED_Pause);
      break;

   case MDIO_COMBO_IEEE0_AUTO_NEG_ADV_PAUSE_ASYMMETRIC:
      adapter->port.advertising[cfg_idx] |= ELINK_ADVERTISED_Asym_Pause;
      break;

   default:
      break;
   }
}
static void qfle3_init_dropless_fc(struct qfle3_adapter *adapter)
{
   u32 pause_enabled = 0;
   u32 pause_old=0;

   if (!CHIP_IS_E1(adapter) && adapter->dropless_fc && adapter->link_vars.link_up) {
      if (adapter->link_vars.flow_ctrl & ELINK_FLOW_CTRL_TX)
         pause_enabled = 1;

      pause_old = REG_RD(adapter, BAR_USTRORM_INTMEM +
         USTORM_ETH_PAUSE_ENABLED_OFFSET(QFLE3_PORT(adapter)));
      QFLE3_DBG(QFLE3_DBG_LINK, "dropless_fc was  %s\n",
         pause_old ? "enabled" : "disabled");
      REG_WR(adapter, BAR_USTRORM_INTMEM +
            USTORM_ETH_PAUSE_ENABLED_OFFSET(QFLE3_PORT(adapter)),
             pause_enabled);
   }

   QFLE3_DBG(QFLE3_DBG_LINK, "dropless_fc is %s\n",
      pause_enabled ? "enabled" : "disabled");
}

void
qfle3_link_status_update(struct qfle3_adapter *adapter)
{
   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_QUIESCED))
      return;
   /* read updated dcb configuration */
   qfle3_dcbx_pmf_update(adapter);

   elink_link_status_update(&adapter->link_params, &adapter->link_vars);
//change to enable stats
   qfle3_init_dropless_fc(adapter);

   if (adapter->link_vars.link_up){
	   QFLE3_DBG(QFLE3_DBG_STATS, "Executing qfle3_stats_handle for STATS_EVENT_LINK_UP\n");
	   qfle3_stats_handle(adapter, STATS_EVENT_LINK_UP);
   }
   else {
	   QFLE3_DBG(QFLE3_DBG_STATS, "Executing qfle3_stats_handle for STATS_EVENT_STOP\n");
	   qfle3_stats_handle(adapter, STATS_EVENT_STOP);
   }
   /* indicate link status */
   qfle3_acquire_phy_lock(adapter);
   qfle3_report_link(adapter);
   qfle3_release_phy_lock(adapter);
}

void
qfle3_link_set(struct qfle3_adapter *adapter)
{
   qfle3_acquire_phy_lock(adapter);
   elink_phy_init(&adapter->link_params, &adapter->link_vars);
   qfle3_release_phy_lock(adapter);

   qfle3_init_dropless_fc(adapter);

   qfle3_calc_fc_adv(adapter);
}

static void
qfle3_link_reset(struct qfle3_adapter *adapter)
{
   qfle3_acquire_phy_lock(adapter);
   elink_lfa_reset(&adapter->link_params, &adapter->link_vars);
   qfle3_release_phy_lock(adapter);
}

void
qfle3_force_link_reset(struct qfle3_adapter *adapter)
{
   qfle3_acquire_phy_lock(adapter);
   elink_link_reset(&adapter->link_params, &adapter->link_vars, 1);
   qfle3_release_phy_lock(adapter);
}

static void
qfle3_reset_port(struct qfle3_adapter *adapter)
{
   int port = QFLE3_PORT(adapter);
   vmk_uint32 val;

   /*
    * Reset physical Link
    */
   qfle3_link_reset(adapter);

   REG_WR(adapter, NIG_REG_MASK_INTERRUPT_PORT0 + port * 4, 0);

   /*
    * Do not rcv packets to BRB
    */
   REG_WR(adapter, NIG_REG_LLH0_BRB1_DRV_MASK + port * 4, 0x0);
   /*
    * Do not direct rcv packets that are not for MCP to the BRB
    */
   REG_WR(adapter, (port ? NIG_REG_LLH1_BRB1_NOT_MCP : NIG_REG_LLH0_BRB1_NOT_MCP), 0x0);

   /*
    * Configure AEU
    */
   REG_WR(adapter, MISC_REG_AEU_MASK_ATTN_FUNC_0 + port * 4, 0);

   vmk_WorldSleep(100 * 1000);
   /*
    * Check for BRB port occupancy
    */
   val = REG_RD(adapter, BRB1_REG_PORT_NUM_OCC_BLOCKS_0 + port * 4);
   if (val)
      QFLE3_WARN("BRB1 is not empty  %d blocks are occupied\n", val);

}

static void
qfle3_reset_common(struct qfle3_adapter *adapter)
{
   vmk_uint32 val = 0x1400;

   /*
    * reset_common
    */
   REG_WR(adapter, GRCBASE_MISC + MISC_REGISTERS_RESET_REG_1_CLEAR, 0xd3ffff7f);

   if (CHIP_IS_E3(adapter)) {
      val |= MISC_REGISTERS_RESET_REG_2_MSTAT0;
      val |= MISC_REGISTERS_RESET_REG_2_MSTAT1;
   }

   REG_WR(adapter, GRCBASE_MISC + MISC_REGISTERS_RESET_REG_2_CLEAR, val);
}

static void
qfle3_init_searcher(struct qfle3_adapter *adapter)
{
   int port = QFLE3_PORT(adapter);
   ecore_src_init_t2(adapter, adapter->t2, adapter->t2_mapping, SRC_CONN_NUM);
   /*
    * T1 hash bits value determines the T1 number of entries
    */
   REG_WR(adapter, SRC_REG_NUMBER_HASH_BITS0 + port * 4, SRC_HASH_BITS);
}

/*
 * FLR Support for E2
 *
 * qfle3_pf_flr_clnup() is called during nic_load in the per function HW
 * initialization.
 */
#define FLR_WAIT_USEC       10000       /* 10 milliseconds */
#define FLR_WAIT_INTERVAL   50  /* usec */
#define FLR_POLL_CNT        (FLR_WAIT_USEC/FLR_WAIT_INTERVAL)   /* 200 */

struct pbf_pN_buf_regs {
   int pN;
   vmk_uint32 init_crd;
   vmk_uint32 crd;
   vmk_uint32 crd_freed;
};

struct pbf_pN_cmd_regs {
   int pN;
   vmk_uint32 lines_occup;
   vmk_uint32 lines_freed;
};

static void
qfle3_pbf_pN_buf_flushed(struct qfle3_adapter *adapter,
			 struct pbf_pN_buf_regs *regs, vmk_uint32 poll_count)
{
   vmk_uint32 init_crd, crd, crd_start, crd_freed, crd_freed_start;
   vmk_uint32 cur_cnt = poll_count;

   crd_freed = crd_freed_start = REG_RD(adapter, regs->crd_freed);
   crd = crd_start = REG_RD(adapter, regs->crd);
   init_crd = REG_RD(adapter, regs->init_crd);

   QFLE3_DBG(QFLE3_DBG_SP, "INIT CREDIT[%d] : %x\n", regs->pN, init_crd);
   QFLE3_DBG(QFLE3_DBG_SP, "CREDIT[%d]      : s:%x\n", regs->pN, crd);
   QFLE3_DBG(QFLE3_DBG_SP, "CREDIT_FREED[%d]: s:%x\n", regs->pN, crd_freed);

   while ((crd != init_crd) && ((vmk_uint32) SUB_S32(crd_freed, crd_freed_start) <
				(init_crd - crd_start))) {
      if (cur_cnt--) {
	 vmk_DelayUsecs(FLR_WAIT_INTERVAL);
	 crd = REG_RD(adapter, regs->crd);
	 crd_freed = REG_RD(adapter, regs->crd_freed);
      } else {
	 QFLE3_DBG(QFLE3_DBG_SP, "PBF tx buffer[%d] timed out\n", regs->pN);
	 QFLE3_DBG(QFLE3_DBG_SP, "CREDIT[%d]      : c:%x\n", regs->pN, crd);
	 QFLE3_DBG(QFLE3_DBG_SP, "CREDIT_FREED[%d]: c:%x\n", regs->pN, crd_freed);
	 break;
      }
   }
   QFLE3_DBG(QFLE3_DBG_SP, "Waited %d*%d usec for PBF tx buffer[%d]\n",
	     poll_count - cur_cnt, FLR_WAIT_INTERVAL, regs->pN);
}

static void
qfle3_pbf_pN_cmd_flushed(struct qfle3_adapter *adapter,
			 struct pbf_pN_cmd_regs *regs, vmk_uint32 poll_count)
{
   vmk_uint32 occup, to_free, freed, freed_start;
   vmk_uint32 cur_cnt = poll_count;

   occup = to_free = REG_RD(adapter, regs->lines_occup);
   freed = freed_start = REG_RD(adapter, regs->lines_freed);

   QFLE3_DBG(QFLE3_DBG_SP, "OCCUPANCY[%d]   : s:%x\n", regs->pN, occup);
   QFLE3_DBG(QFLE3_DBG_SP, "LINES_FREED[%d] : s:%x\n", regs->pN, freed);

   while (occup && ((vmk_uint32) SUB_S32(freed, freed_start) < to_free)) {
      if (cur_cnt--) {
	 vmk_DelayUsecs(FLR_WAIT_INTERVAL);
	 occup = REG_RD(adapter, regs->lines_occup);
	 freed = REG_RD(adapter, regs->lines_freed);
      } else {
	 QFLE3_DBG(QFLE3_DBG_SP, "PBF cmd queue[%d] timed out\n", regs->pN);
	 QFLE3_DBG(QFLE3_DBG_SP, "OCCUPANCY[%d]   : s:%x\n", regs->pN, occup);
	 QFLE3_DBG(QFLE3_DBG_SP, "LINES_FREED[%d] : s:%x\n", regs->pN, freed);
	 break;
      }
   }
   QFLE3_DBG(QFLE3_DBG_SP, "Waited %d*%d usec for PBF cmd queue[%d]\n",
	     poll_count - cur_cnt, FLR_WAIT_INTERVAL, regs->pN);
}

static vmk_uint32
qfle3_flr_clnup_reg_poll(struct qfle3_adapter *adapter, vmk_uint32 reg,
			 vmk_uint32 expected, vmk_uint32 poll_count)
{
   vmk_uint32 cur_cnt = poll_count;
   vmk_uint32 val;

   while ((val = REG_RD(adapter, reg)) != expected && cur_cnt--)
      vmk_DelayUsecs(FLR_WAIT_INTERVAL);

   return val;
}

int
qfle3_flr_clnup_poll_hw_counter(struct qfle3_adapter *adapter, vmk_uint32 reg,
				char *msg, vmk_uint32 poll_cnt)
{
   vmk_uint32 val = qfle3_flr_clnup_reg_poll(adapter, reg, 0, poll_cnt);
   if (val != 0) {
      QFLE3_DBG(QFLE3_DBG_HW, "%s usage count=%d\n", msg, val);
      return 1;
   }
   return 0;
}

vmk_uint32
qfle3_flr_clnup_poll_count(struct qfle3_adapter * adapter)
{
   /*
    * adjust polling timeout
    */
   if (CHIP_REV_IS_EMUL(adapter))
      return FLR_POLL_CNT * 2000;

   if (CHIP_REV_IS_FPGA(adapter))
      return FLR_POLL_CNT * 120;

   return FLR_POLL_CNT;
}

void
qfle3_tx_hw_flushed(struct qfle3_adapter *adapter, vmk_uint32 poll_count)
{
   struct pbf_pN_cmd_regs cmd_regs[] = {
      {0, (CHIP_IS_E3B0(adapter)) ? PBF_REG_TQ_OCCUPANCY_Q0 : PBF_REG_P0_TQ_OCCUPANCY,
       (CHIP_IS_E3B0(adapter)) ?
       PBF_REG_TQ_LINES_FREED_CNT_Q0 : PBF_REG_P0_TQ_LINES_FREED_CNT},
      {1, (CHIP_IS_E3B0(adapter)) ? PBF_REG_TQ_OCCUPANCY_Q1 : PBF_REG_P1_TQ_OCCUPANCY,
       (CHIP_IS_E3B0(adapter)) ?
       PBF_REG_TQ_LINES_FREED_CNT_Q1 : PBF_REG_P1_TQ_LINES_FREED_CNT},
      {4, (CHIP_IS_E3B0(adapter)) ? PBF_REG_TQ_OCCUPANCY_LB_Q : PBF_REG_P4_TQ_OCCUPANCY,
       (CHIP_IS_E3B0(adapter)) ?
       PBF_REG_TQ_LINES_FREED_CNT_LB_Q : PBF_REG_P4_TQ_LINES_FREED_CNT}
   };


   struct pbf_pN_buf_regs buf_regs[] = {
      {0, (CHIP_IS_E3B0(adapter)) ? PBF_REG_INIT_CRD_Q0 : PBF_REG_P0_INIT_CRD,
       (CHIP_IS_E3B0(adapter)) ? PBF_REG_CREDIT_Q0 : PBF_REG_P0_CREDIT,
       (CHIP_IS_E3B0(adapter)) ?
       PBF_REG_INTERNAL_CRD_FREED_CNT_Q0 : PBF_REG_P0_INTERNAL_CRD_FREED_CNT},
      {1, (CHIP_IS_E3B0(adapter)) ? PBF_REG_INIT_CRD_Q1 : PBF_REG_P1_INIT_CRD,
       (CHIP_IS_E3B0(adapter)) ? PBF_REG_CREDIT_Q1 : PBF_REG_P1_CREDIT,
       (CHIP_IS_E3B0(adapter)) ?
       PBF_REG_INTERNAL_CRD_FREED_CNT_Q1 : PBF_REG_P1_INTERNAL_CRD_FREED_CNT},
      {4, (CHIP_IS_E3B0(adapter)) ? PBF_REG_INIT_CRD_LB_Q : PBF_REG_P4_INIT_CRD,
       (CHIP_IS_E3B0(adapter)) ? PBF_REG_CREDIT_LB_Q : PBF_REG_P4_CREDIT,
       (CHIP_IS_E3B0(adapter)) ?
       PBF_REG_INTERNAL_CRD_FREED_CNT_LB_Q : PBF_REG_P4_INTERNAL_CRD_FREED_CNT},
   };

   int i;

   /*
    * Verify the command queues are flushed P0, P1, P4
    */
   for (i = 0; i < ARRAY_SIZE(cmd_regs); i++)
      qfle3_pbf_pN_cmd_flushed(adapter, &cmd_regs[i], poll_count);

   /*
    * Verify the transmission buffers are flushed P0, P1, P4
    */
   for (i = 0; i < ARRAY_SIZE(buf_regs); i++)
      qfle3_pbf_pN_buf_flushed(adapter, &buf_regs[i], poll_count);
}

#define OP_GEN_PARAM(param)                                             \
   (((param) << SDM_OP_GEN_COMP_PARAM_SHIFT) & SDM_OP_GEN_COMP_PARAM)

#define OP_GEN_TYPE(type)                                               \
   (((type) << SDM_OP_GEN_COMP_TYPE_SHIFT) & SDM_OP_GEN_COMP_TYPE)

#define OP_GEN_AGG_VECT(index)                                          \
   (((index) << SDM_OP_GEN_AGG_VECT_IDX_SHIFT) & SDM_OP_GEN_AGG_VECT_IDX)

int
qfle3_send_final_clnup(struct qfle3_adapter *adapter, vmk_uint8 clnup_func,
		       vmk_uint32 poll_cnt)
{
   vmk_uint32 op_gen_command = 0;
   vmk_uint32 comp_addr = BAR_CSTRORM_INTMEM +
      CSTORM_FINAL_CLEANUP_COMPLETE_OFFSET(clnup_func);
   int ret = 0;

   if (REG_RD(adapter, comp_addr)) {
      QFLE3_ERR("Cleanup complete was not 0 before sending\n");
      return 1;
   }

   op_gen_command |= OP_GEN_PARAM(XSTORM_AGG_INT_FINAL_CLEANUP_INDEX);
   op_gen_command |= OP_GEN_TYPE(XSTORM_AGG_INT_FINAL_CLEANUP_COMP_TYPE);
   op_gen_command |= OP_GEN_AGG_VECT(clnup_func);
   op_gen_command |= 1 << SDM_OP_GEN_AGG_VECT_IDX_VALID_SHIFT;

   QFLE3_DBG(QFLE3_DBG_SP, "sending FW Final cleanup\n");
   REG_WR(adapter, XSDM_REG_OPERATION_GEN, op_gen_command);

   if (qfle3_flr_clnup_reg_poll(adapter, comp_addr, 1, poll_cnt) != 1) {
      QFLE3_ERR("FW final cleanup did not succeed\n");
      QFLE3_ERR("At timeout completion address contained %x\n",
		(REG_RD(adapter, comp_addr)));
      //qfle3_panic();
      return 1;
   }
   /*
    * Zero completion for next FLR
    */
   REG_WR(adapter, comp_addr, 0);

   return ret;
}

vmk_uint8
qfle3_is_pcie_pending(vmk_PCIDevice dev)
{
   vmk_uint32 status;

   qfle3_pcie_cap_read_word(dev, PCI_EXP_DEVSTA, &status);
   return status & PCI_EXP_DEVSTA_TRPND;
}

static int
qfle3_poll_hw_usage_counters(struct qfle3_adapter *adapter, vmk_uint32 poll_cnt)
{
   /*
    * wait for CFC PF usage-counter to zero (includes all the VFs)
    */
   if (qfle3_flr_clnup_poll_hw_counter(adapter,
				       CFC_REG_NUM_LCIDS_INSIDE_PF,
				       "CFC PF usage counter timed out", poll_cnt))
      return 1;

   /*
    * Wait for DQ PF usage-counter to zero (until DQ cleanup)
    */
   if (qfle3_flr_clnup_poll_hw_counter(adapter,
				       DORQ_REG_PF_USAGE_CNT,
				       "DQ PF usage counter timed out", poll_cnt))
      return 1;

   /*
    * Wait for QM PF usage-counter to zero (until QM cleanup)
    */
   if (qfle3_flr_clnup_poll_hw_counter(adapter,
				       QM_REG_PF_USG_CNT_0 + 4 * QFLE3_FUNC(adapter),
				       "QM PF usage counter timed out", poll_cnt))
      return 1;

   /*
    * Wait for Timer PF usage-counters to zero (until Timer cleanup)
    */
   if (qfle3_flr_clnup_poll_hw_counter(adapter,
				       TM_REG_LIN0_VNIC_UC + 4 * QFLE3_PORT(adapter),
				       "Timers VNIC usage counter timed out", poll_cnt))
      return 1;
   if (qfle3_flr_clnup_poll_hw_counter(adapter,
				       TM_REG_LIN0_NUM_SCANS + 4 * QFLE3_PORT(adapter),
				       "Timers NUM_SCANS usage counter timed out",
				       poll_cnt))
      return 1;

   /*
    * Wait DMAE PF usage counter to zero
    */
   if (qfle3_flr_clnup_poll_hw_counter(adapter,
				       dmae_reg_go_c[INIT_DMAE_C(adapter)],
				       "DMAE command register timed out", poll_cnt))
      return 1;

   return 0;
}

static void
qfle3_hw_enable_status(struct qfle3_adapter *adapter)
{
   vmk_uint32 val;

   val = REG_RD(adapter, CFC_REG_WEAK_ENABLE_PF);
   QFLE3_DBG(QFLE3_DBG_HW, "CFC_REG_WEAK_ENABLE_PF is 0x%x\n", val);

   val = REG_RD(adapter, PBF_REG_DISABLE_PF);
   QFLE3_DBG(QFLE3_DBG_HW, "PBF_REG_DISABLE_PF is 0x%x\n", val);

   val = REG_RD(adapter, IGU_REG_PCI_PF_MSI_EN);
   QFLE3_DBG(QFLE3_DBG_HW, "IGU_REG_PCI_PF_MSI_EN is 0x%x\n", val);

   val = REG_RD(adapter, IGU_REG_PCI_PF_MSIX_EN);
   QFLE3_DBG(QFLE3_DBG_HW, "IGU_REG_PCI_PF_MSIX_EN is 0x%x\n", val);

   val = REG_RD(adapter, IGU_REG_PCI_PF_MSIX_FUNC_MASK);
   QFLE3_DBG(QFLE3_DBG_HW, "IGU_REG_PCI_PF_MSIX_FUNC_MASK is 0x%x\n", val);

   val = REG_RD(adapter, PGLUE_B_REG_SHADOW_BME_PF_7_0_CLR);
   QFLE3_DBG(QFLE3_DBG_HW, "PGLUE_B_REG_SHADOW_BME_PF_7_0_CLR is 0x%x\n", val);

   val = REG_RD(adapter, PGLUE_B_REG_FLR_REQUEST_PF_7_0_CLR);
   QFLE3_DBG(QFLE3_DBG_HW, "PGLUE_B_REG_FLR_REQUEST_PF_7_0_CLR is 0x%x\n", val);

   val = REG_RD(adapter, PGLUE_B_REG_INTERNAL_PFID_ENABLE_MASTER);
   QFLE3_DBG(QFLE3_DBG_HW, "PGLUE_B_REG_INTERNAL_PFID_ENABLE_MASTER is 0x%x\n", val);
}

static int
qfle3_pf_flr_clnup(struct qfle3_adapter *adapter)
{
   vmk_uint32 poll_cnt = qfle3_flr_clnup_poll_count(adapter);

   QFLE3_DBG(QFLE3_DBG_HW, "Cleanup after FLR PF[%d]\n", QFLE3_ABS_FUNC(adapter));

   /*
    * Re-enable PF target read access
    */
   REG_WR(adapter, PGLUE_B_REG_INTERNAL_PFID_ENABLE_TARGET_READ, 1);

   /*
    * Poll HW usage counters
    */
   QFLE3_DBG(QFLE3_DBG_HW, "Polling usage counters\n");
   if (qfle3_poll_hw_usage_counters(adapter, poll_cnt))
      return VMK_BUSY;

   /*
    * Zero the igu 'trailing edge' and 'leading edge'
    */

   /*
    * Send the FW cleanup command
    */
   if (qfle3_send_final_clnup(adapter, (vmk_uint8) QFLE3_FUNC(adapter), poll_cnt))
      return VMK_BUSY;

   /*
    * ATC cleanup
    */

   /*
    * Verify TX hw is flushed
    */
   qfle3_tx_hw_flushed(adapter, poll_cnt);

   /*
    * Wait 100ms (not adjusted according to platform)
    */
   vmk_WorldSleep(100 * 1000);

   /*
    * Verify no pending pci transactions
    */
   if (qfle3_is_pcie_pending(adapter->pdev))
      QFLE3_ERR("PCIE Transactions still pending\n");

   /*
    * Debug
    */
   qfle3_hw_enable_status(adapter);

   /*
    * Master enable - Due to WB DMAE writes performed before this
    * * register is re-initialized as part of the regular function init
    */
   REG_WR(adapter, PGLUE_B_REG_INTERNAL_PFID_ENABLE_MASTER, 1);
   return 0;
}
#define MCPR_TRACE_BUFFER_SIZE (0x800)
#define SCRATCH_BUFFER_SIZE(adapter) \
	(CHIP_IS_E1(adapter) ? 0x10000 : (CHIP_IS_E1H(adapter) ? 0x20000 : 0x28000))
#define MCPR_SCRATCH_BASE(A) \
      (CHIP_IS_E1x(A) ? MCP_REG_MCPR_SCRATCH : MCP_A_REG_MCPR_SCRATCH)
static void qfle3_fw_dump(struct qfle3_adapter *adapter)
{
   
	u32 addr, val;
	u32 mark, offset;
	__be32 data[9];
	int word;
	u32 trace_shmem_base;
      char* logline;
      int pos = 0;
      
#define QFLE3_DUMP(fmt, args...) \
          pos += vmk_Sprintf(logline+pos, fmt "", ##args)
#define QFLE3_DUMP_LN(fmt, args...) \
             vmk_Sprintf(logline+pos, fmt "\n", ##args); pos=0;\
             vmk_LogNoLevel(VMK_LOG_URGENCY_NORMAL, "%s", logline)

	QFLE3_INFO("bc %d.%d.%d\n",
		(adapter->hw_info.bc_ver & 0xff0000) >> 16,
		(adapter->hw_info.bc_ver & 0xff00) >> 8,
		(adapter->hw_info.bc_ver & 0xff));
   logline = qfle3_heap_alloc(0x1000);

   if (!logline)
      return;

//	if (pci_channel_offline(adapter->pdev)) {
//		BNX2X_ERR("Cannot dump MCP info while in PCI error\n");
//		return;
//	}

	val = REG_RD(adapter, MCP_REG_MCPR_CPU_PROGRAM_COUNTER);
	if (val == REG_RD(adapter, MCP_REG_MCPR_CPU_PROGRAM_COUNTER))
		QFLE3_ERR("MCP PC at 0x%x\n", val);

	if (QFLE3_PATH(adapter) == 0)
		trace_shmem_base = adapter->hw_info.shmem_base;
	else
		trace_shmem_base = SHMEM2_RD(adapter, other_shmem_base_addr);

	/* sanity */
	if (trace_shmem_base < MCPR_SCRATCH_BASE(adapter) + MCPR_TRACE_BUFFER_SIZE ||
	    trace_shmem_base >= MCPR_SCRATCH_BASE(adapter) +
				SCRATCH_BUFFER_SIZE(adapter)) {
		QFLE3_ERR("Unable to dump trace buffer (mark %x)\n",
			  trace_shmem_base);
      qfle3_heap_free(logline);
		return;
	}

	addr = trace_shmem_base - MCPR_TRACE_BUFFER_SIZE;

	/* validate TRCB signature */
	mark = REG_RD(adapter, addr);
	if (mark != MFW_TRACE_SIGNATURE) {
		QFLE3_ERR("Trace buffer signature is missing.");
      
      qfle3_heap_free(logline);
		return ;
	}

	/* read cyclic buffer pointer */
	addr += 4;
	mark = REG_RD(adapter, addr);
	mark = MCPR_SCRATCH_BASE(adapter) + ((mark + 0x3) & ~0x3) - 0x08000000;
	if (mark >= trace_shmem_base || mark < addr + 4) {
		QFLE3_ERR("Mark doesn't fall inside Trace Buffer\n");
      
      qfle3_heap_free(logline);
		return;
	}
	QFLE3_DUMP_LN("begin fw dump (mark 0x%x)\n", mark);
	/* dump buffer after the mark */
	for (offset = mark; offset < trace_shmem_base; offset += 0x8*4) {
		for (word = 0; word < 8; word++)
			data[word] = vmk_CPUToBE32(REG_RD(adapter, offset + 4*word));
		data[8] = 0x0;
		QFLE3_DUMP("%s", (char *)data);
	}

	/* dump buffer before the mark */
	for (offset = addr + 4; offset <= mark; offset += 0x8*4) {
		for (word = 0; word < 8; word++)
			data[word] = vmk_CPUToBE32(REG_RD(adapter, offset + 4*word));
		data[8] = 0x0;
		QFLE3_DUMP("%s", (char *)data);
	}
	QFLE3_DUMP_LN("\nend of fw dump\n");
   qfle3_heap_free(logline);
}
static int
qfle3_init_hw_func(struct qfle3_adapter *adapter)
{
   int port = QFLE3_PORT(adapter);
   int func = QFLE3_FUNC(adapter);
   int init_phase = PHASE_PF0 + func;
   struct ecore_ilt *ilt = QFLE3_ILT(adapter);
   vmk_uint16 cdu_ilt_start;
   vmk_uint32 addr;
   //vmk_uint32 val;
   //vmk_uint32 main_mem_base, main_mem_size, main_mem_prty_clr;
   int i, rc;
   //int main_mem_width;

   QFLE3_DBG(QFLE3_DBG_HW, "starting func init  func %d\n", func);

   /*
    * FLR cleanup - hmmm
    */
   rc = qfle3_pf_flr_clnup(adapter);
   if (rc) {
#if 0
      qfle3_fw_dump(adapter);
      qfle3_idle_chk(adapter);
#endif
      return rc;
   }

   /*
    * set MSI reconfigure capability
    */
#if 0
   if (adapter->hw_info.int_block == INT_BLOCK_HC) {
      addr = (port ? HC_REG_CONFIG_1 : HC_REG_CONFIG_0);
      val = REG_RD(adapter, addr);
      val |= HC_CONFIG_0_REG_MSI_ATTN_EN_0;
      REG_WR(adapter, addr, val);
   }
#endif

   ecore_init_block(adapter, BLOCK_PXP, init_phase);
   ecore_init_block(adapter, BLOCK_PXP2, init_phase);
   QFLE3_DBG(QFLE3_DBG_HW, "Init phase pxp/pxp2 done.\n");

   ilt = QFLE3_ILT(adapter);
   cdu_ilt_start = ilt->clients[ILT_CLIENT_CDU].start;

#ifdef QFLE3_SRIOV
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Checking if Page CIDS are needed to be configured for SRIOV\n");
   if (!adapter->vfdb){
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Adapter vfdb not allocated yet\n");
   }
   else {
      cdu_ilt_start += QFLE3_FIRST_VF_CID / ILT_PAGE_CIDS;
      cdu_ilt_start = qfle3_iov_init_ilt(adapter, cdu_ilt_start);
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Page CIDS configured successfully for SRIOV\n"); 
   }
#endif //QFLE3_SRIOV

   /*
    * since QFLE3_FIRST_VF_CID > 0 the PF L2 cids precedes
    * * those of the VFs, so start line should be reset
    */
   cdu_ilt_start = ilt->clients[ILT_CLIENT_CDU].start;
   for (i = 0; i < L2_ILT_LINES(adapter); i++) {
      ilt->lines[cdu_ilt_start + i].page = adapter->context[i].vcxt;
      ilt->lines[cdu_ilt_start + i].page_mapping = adapter->context[i].cxt_mapping;
      ilt->lines[cdu_ilt_start + i].size = adapter->context[i].size;
   }

   ecore_ilt_init_op(adapter, INITOP_SET);

   if (!CONFIGURE_NIC_MODE(adapter)) {
      qfle3_init_searcher(adapter);
      REG_WR(adapter, PRS_REG_NIC_MODE, 0);
      QFLE3_INFO("NIC MODE disabled\n");
   } else {
      /*
       * Set NIC mode
       */
      REG_WR(adapter, PRS_REG_NIC_MODE, 1);
      QFLE3_INFO("NIC MODE configured\n");
   }


   if (!CHIP_IS_E1x(adapter)) {
      vmk_uint32 pf_conf = IGU_PF_CONF_FUNC_EN;

      /*
       * Turn on a single ISR mode in IGU if driver is going to use
       * * INT#x or MSI
       */
      if (!(adapter->flags & USING_MSIX_FLAG))
	 pf_conf |= IGU_PF_CONF_SINGLE_ISR_EN;
      /*
       * Timers workaround bug: function init part.
       * Need to wait 20msec after initializing ILT,
       * needed to make sure there are no requests in
       * one of the PXP internal queues with "old" ILT addresses
       */
      vmk_WorldSleep(20 * 1000);
      /*
       * Master enable - Due to WB DMAE writes performed before this
       * register is re-initialized as part of the regular function
       * init
       */
      REG_WR(adapter, PGLUE_B_REG_INTERNAL_PFID_ENABLE_MASTER, 1);
      /*
       * Enable the function in IGU
       */
      REG_WR(adapter, IGU_REG_PF_CONFIGURATION, pf_conf);
      QFLE3_DBG(QFLE3_DBG_HW, "Function enabled in IGU.\n");
   }

   adapter->dmae_ready = 1;

   ecore_init_block(adapter, BLOCK_PGLUE_B, init_phase);

   qfle3_clean_pglue_errors(adapter);

   QFLE3_DBG(QFLE3_DBG_HW, "Cleaned PGLUE errors.\n");

   ecore_init_block(adapter, BLOCK_ATC, init_phase);
   ecore_init_block(adapter, BLOCK_DMAE, init_phase);
   ecore_init_block(adapter, BLOCK_NIG, init_phase);
   ecore_init_block(adapter, BLOCK_SRC, init_phase);
   ecore_init_block(adapter, BLOCK_MISC, init_phase);
   ecore_init_block(adapter, BLOCK_TCM, init_phase);
   ecore_init_block(adapter, BLOCK_UCM, init_phase);
   ecore_init_block(adapter, BLOCK_CCM, init_phase);
   ecore_init_block(adapter, BLOCK_XCM, init_phase);
   ecore_init_block(adapter, BLOCK_TSEM, init_phase);
   ecore_init_block(adapter, BLOCK_USEM, init_phase);
   ecore_init_block(adapter, BLOCK_CSEM, init_phase);
   ecore_init_block(adapter, BLOCK_XSEM, init_phase);

   if (!CHIP_IS_E1x(adapter))
      REG_WR(adapter, QM_REG_PF_EN, 1);

   if (!CHIP_IS_E1x(adapter)) {
      REG_WR(adapter, TSEM_REG_VFPF_ERR_NUM, QFLE3_MAX_NUM_OF_VFS + func);
      REG_WR(adapter, USEM_REG_VFPF_ERR_NUM, QFLE3_MAX_NUM_OF_VFS + func);
      REG_WR(adapter, CSEM_REG_VFPF_ERR_NUM, QFLE3_MAX_NUM_OF_VFS + func);
      REG_WR(adapter, XSEM_REG_VFPF_ERR_NUM, QFLE3_MAX_NUM_OF_VFS + func);
   }
   ecore_init_block(adapter, BLOCK_QM, init_phase);

   ecore_init_block(adapter, BLOCK_TM, init_phase);
   ecore_init_block(adapter, BLOCK_DORQ, init_phase);
   REG_WR(adapter, DORQ_REG_MODE_ACT, 1);       /* no dpm */
   QFLE3_DBG(QFLE3_DBG_HW,"DorQ enabled.\n");

#ifdef QFLE3_SRIOV
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: About to execute qfle3_iov_init_dq\n");
   qfle3_iov_init_dq(adapter);
#endif //QFLE3_SRIOV

   ecore_init_block(adapter, BLOCK_BRB1, init_phase);
   ecore_init_block(adapter, BLOCK_PRS, init_phase);
   ecore_init_block(adapter, BLOCK_TSDM, init_phase);
   ecore_init_block(adapter, BLOCK_CSDM, init_phase);
   ecore_init_block(adapter, BLOCK_USDM, init_phase);
   ecore_init_block(adapter, BLOCK_XSDM, init_phase);
   ecore_init_block(adapter, BLOCK_UPB, init_phase);
   ecore_init_block(adapter, BLOCK_XPB, init_phase);
   ecore_init_block(adapter, BLOCK_PBF, init_phase);
   if (!CHIP_IS_E1x(adapter))
      REG_WR(adapter, PBF_REG_DISABLE_PF, 0);

   ecore_init_block(adapter, BLOCK_CDU, init_phase);

   ecore_init_block(adapter, BLOCK_CFC, init_phase);

   if (!CHIP_IS_E1x(adapter))
      REG_WR(adapter, CFC_REG_WEAK_ENABLE_PF, 1);

   if (IS_MF(adapter)) {
      if (!(IS_MF_UFP(adapter) && QFLE3_IS_MF_SD_PROTOCOL_FCOE(adapter))) {
	 REG_WR(adapter, NIG_REG_LLH0_FUNC_EN + port * 8, 1);
	 REG_WR(adapter, NIG_REG_LLH0_FUNC_VLAN_ID + port * 8, adapter->mf_ov);
      }
   }

   ecore_init_block(adapter, BLOCK_MISC_AEU, init_phase);
   QFLE3_DBG(QFLE3_DBG_HW,"AEU enabled.\n");

   /*
    * HC init per function
    */
   if (adapter->hw_info.int_block == INT_BLOCK_HC) {
      if (CHIP_IS_E1H(adapter)) {
	 REG_WR(adapter, MISC_REG_AEU_GENERAL_ATTN_12 + func * 4, 0);

	 REG_WR(adapter, HC_REG_LEADING_EDGE_0 + port * 8, 0);
	 REG_WR(adapter, HC_REG_TRAILING_EDGE_0 + port * 8, 0);
      }
      ecore_init_block(adapter, BLOCK_HC, init_phase);

   } else {
      int num_segs, sb_idx, prod_offset;

      REG_WR(adapter, MISC_REG_AEU_GENERAL_ATTN_12 + func * 4, 0);

      if (!CHIP_IS_E1x(adapter)) {
	 REG_WR(adapter, IGU_REG_LEADING_EDGE_LATCH, 0);
	 REG_WR(adapter, IGU_REG_TRAILING_EDGE_LATCH, 0);
      }

      ecore_init_block(adapter, BLOCK_IGU, init_phase);

      if (!CHIP_IS_E1x(adapter)) {
	 int dsb_idx = 0;
	 /**
	  * Producer memory:
	  * E2 mode: address 0-135 match to the mapping memory;
	  * 136 - PF0 default prod; 137 - PF1 default prod;
	  * 138 - PF2 default prod; 139 - PF3 default prod;
	  * 140 - PF0 attn prod;    141 - PF1 attn prod;
	  * 142 - PF2 attn prod;    143 - PF3 attn prod;
	  * 144-147 reserved.
	  *
	  * E1.5 mode - In backward compatible mode;
	  * for non default SB; each even line in the memory
	  * holds the U producer and each odd line hold
	  * the C producer. The first 128 producers are for
	  * NDSB (PF0 - 0-31; PF1 - 32-63 and so on). The last 20
	  * producers are for the DSB for each PF.
	  * Each PF has five segments: (the order inside each
	  * segment is PF0; PF1; PF2; PF3) - 128-131 U prods;
	  * 132-135 C prods; 136-139 X prods; 140-143 T prods;
	  * 144-147 attn prods;
	  */
	 /*
	  * non-default-status-blocks
	  */

	 num_segs = CHIP_INT_MODE_IS_BC(adapter) ?
	    IGU_BC_NDSB_NUM_SEGS : IGU_NORM_NDSB_NUM_SEGS;
	 for (sb_idx = 0; sb_idx < adapter->igu_sb_cnt; sb_idx++) {
	    prod_offset = (adapter->igu_base_sb + sb_idx) * num_segs;

	    for (i = 0; i < num_segs; i++) {
	       addr = IGU_REG_PROD_CONS_MEMORY + (prod_offset + i) * 4;
	       REG_WR(adapter, addr, 0);
	    }
	    /*
	     * send consumer update with value 0
	     */
	    qfle3_ack_sb(adapter, adapter->igu_base_sb + sb_idx,
			 USTORM_ID, 0, IGU_INT_NOP, 1);
	    qfle3_igu_clear_sb(adapter, adapter->igu_base_sb + sb_idx);
	 }

	 /*
	  * default-status-blocks
	  */
	 num_segs = CHIP_INT_MODE_IS_BC(adapter) ?
	    IGU_BC_DSB_NUM_SEGS : IGU_NORM_DSB_NUM_SEGS;

	 if (CHIP_MODE_IS_4_PORT(adapter))
	    dsb_idx = QFLE3_FUNC(adapter);
	 else
	    dsb_idx = QFLE3_VN(adapter);

	 prod_offset = (CHIP_INT_MODE_IS_BC(adapter) ?
			IGU_BC_BASE_DSB_PROD + dsb_idx :
			IGU_NORM_BASE_DSB_PROD + dsb_idx);

	 /*
	  * igu prods come in chunks of E1HVN_MAX (4) -
	  * does not matters what is the current chip mode
	  */
	 for (i = 0; i < (num_segs * E1HVN_MAX); i += E1HVN_MAX) {
	    addr = IGU_REG_PROD_CONS_MEMORY + (prod_offset + i) * 4;
	    REG_WR(adapter, addr, 0);

	 }
	 /*
	  * send consumer update with 0
	  */
	 if (CHIP_INT_MODE_IS_BC(adapter)) {
	    qfle3_ack_sb(adapter, adapter->igu_dsb_id, USTORM_ID, 0, IGU_INT_NOP, 1);
	    qfle3_ack_sb(adapter, adapter->igu_dsb_id, CSTORM_ID, 0, IGU_INT_NOP, 1);
	    qfle3_ack_sb(adapter, adapter->igu_dsb_id, XSTORM_ID, 0, IGU_INT_NOP, 1);
	    qfle3_ack_sb(adapter, adapter->igu_dsb_id, TSTORM_ID, 0, IGU_INT_NOP, 1);
	    qfle3_ack_sb(adapter, adapter->igu_dsb_id, ATTENTION_ID, 0, IGU_INT_NOP, 1);
	 } else {
	    qfle3_ack_sb(adapter, adapter->igu_dsb_id, USTORM_ID, 0, IGU_INT_NOP, 1);
	    qfle3_ack_sb(adapter, adapter->igu_dsb_id, ATTENTION_ID, 0, IGU_INT_NOP, 1);
	 }
	 qfle3_igu_clear_sb(adapter, adapter->igu_dsb_id);

	 /*
	  * !!! These should become driver const once
	  * *  rf-tool supports split-68 const
	  */
	 REG_WR(adapter, IGU_REG_SB_INT_BEFORE_MASK_LSB, 0);
	 REG_WR(adapter, IGU_REG_SB_INT_BEFORE_MASK_MSB, 0);
	 REG_WR(adapter, IGU_REG_SB_MASK_LSB, 0);
	 REG_WR(adapter, IGU_REG_SB_MASK_MSB, 0);
	 REG_WR(adapter, IGU_REG_PBA_STATUS_LSB, 0);
	 REG_WR(adapter, IGU_REG_PBA_STATUS_MSB, 0);
      }
   }

   /*
    * Reset PCIE errors for debug
    */
   REG_WR(adapter, 0x2114, 0xffffffff);
   REG_WR(adapter, 0x2120, 0xffffffff);

#if 0
   /*
    * Disable PCIE-core timeout as ESX intercepts the PCIE transactions
    * * and this may cause false "PCIE timeout" errors.
    */
   qfle3_disable_pcie_tout(adapter);
#endif

#if 0
#ifdef QFLE3_STOP_ON_ERROR
   /*
    * Enable STORMs SP logging
    */
   REG_WR8(adapter, BAR_USTRORM_INTMEM +
	   USTORM_RECORD_SLOW_PATH_OFFSET(QFLE3_FUNC(adapter)), 1);
   REG_WR8(adapter, BAR_TSTRORM_INTMEM +
	   TSTORM_RECORD_SLOW_PATH_OFFSET(QFLE3_FUNC(adapter)), 1);
   REG_WR8(adapter, BAR_CSTRORM_INTMEM +
	   CSTORM_RECORD_SLOW_PATH_OFFSET(QFLE3_FUNC(adapter)), 1);
   REG_WR8(adapter, BAR_XSTRORM_INTMEM +
	   XSTORM_RECORD_SLOW_PATH_OFFSET(QFLE3_FUNC(adapter)), 1);
#endif
#endif

   elink_phy_probe(&adapter->link_params);

   return 0;
}

static int
qfle3_init_hw_port(struct qfle3_adapter *adapter)
{
   int port = QFLE3_PORT(adapter);
   int init_phase = port ? PHASE_PORT1 : PHASE_PORT0;
   //vmk_uint32 low, high;
   vmk_uint32 val, reg;

   QFLE3_DBG(QFLE3_DBG_HW, "starting port init  port %d\n", port);

   REG_WR(adapter, NIG_REG_MASK_INTERRUPT_PORT0 + port * 4, 0);

   ecore_init_block(adapter, BLOCK_MISC, init_phase);
   ecore_init_block(adapter, BLOCK_PXP, init_phase);
   ecore_init_block(adapter, BLOCK_PXP2, init_phase);

   /*
    * Timers bug workaround: disables the pf_master bit in pglue at
    * * common phase, we need to enable it here before any dmae access are
    * * attempted. Therefore we manually added the enable-master to the
    * * port phase (it also happens in the function phase)
    */
   if (!CHIP_IS_E1x(adapter))
      REG_WR(adapter, PGLUE_B_REG_INTERNAL_PFID_ENABLE_MASTER, 1);
   ecore_init_block(adapter, BLOCK_ATC, init_phase);
   ecore_init_block(adapter, BLOCK_DMAE, init_phase);
   ecore_init_block(adapter, BLOCK_PGLUE_B, init_phase);
   ecore_init_block(adapter, BLOCK_QM, init_phase);

   ecore_init_block(adapter, BLOCK_TCM, init_phase);
   ecore_init_block(adapter, BLOCK_UCM, init_phase);
   ecore_init_block(adapter, BLOCK_CCM, init_phase);
   ecore_init_block(adapter, BLOCK_XCM, init_phase);

   /*
    * QM cid (connection) count
    */
   ecore_qm_init_cid_count(adapter, adapter->qm_cid_count, INITOP_SET);

   if (CNIC_SUPPORT(adapter)) {
      ecore_init_block(adapter, BLOCK_TM, init_phase);
      REG_WR(adapter, TM_REG_LIN0_SCAN_TIME + port * 4, 20);
      REG_WR(adapter, TM_REG_LIN0_MAX_ACTIVE_CID + port * 4, 31);
   }

   ecore_init_block(adapter, BLOCK_DORQ, init_phase);

   ecore_init_block(adapter, BLOCK_BRB1, init_phase);

   if (CHIP_MODE_IS_4_PORT(adapter))
      REG_WR(adapter, (QFLE3_PORT(adapter) ?
		       BRB1_REG_MAC_GUARANTIED_1 : BRB1_REG_MAC_GUARANTIED_0), 40);

   ecore_init_block(adapter, BLOCK_PRS, init_phase);
   if (CHIP_IS_E3B0(adapter)) {
      if (IS_MF_AFEX(adapter)) {
	 /*
	  * configure headers for AFEX mode
	  */
	 REG_WR(adapter, QFLE3_PORT(adapter) ?
		PRS_REG_HDRS_AFTER_BASIC_PORT_1 : PRS_REG_HDRS_AFTER_BASIC_PORT_0, 0xE);
	 REG_WR(adapter, QFLE3_PORT(adapter) ?
		PRS_REG_HDRS_AFTER_TAG_0_PORT_1 : PRS_REG_HDRS_AFTER_TAG_0_PORT_0, 0x6);
	 REG_WR(adapter, QFLE3_PORT(adapter) ?
		PRS_REG_MUST_HAVE_HDRS_PORT_1 : PRS_REG_MUST_HAVE_HDRS_PORT_0, 0xA);
      } else {
	 /*
	  * Ovlan exists only if we are in multi-function +
	  * * switch-dependent mode, in switch-independent there
	  * * is no ovlan headers
	  */
	 REG_WR(adapter, QFLE3_PORT(adapter) ?
		PRS_REG_HDRS_AFTER_BASIC_PORT_1 :
		PRS_REG_HDRS_AFTER_BASIC_PORT_0, (adapter->path_has_ovlan ? 7 : 6));
      }
   }
   ecore_init_block(adapter, BLOCK_TSDM, init_phase);
   ecore_init_block(adapter, BLOCK_CSDM, init_phase);
   ecore_init_block(adapter, BLOCK_USDM, init_phase);
   ecore_init_block(adapter, BLOCK_XSDM, init_phase);

   ecore_init_block(adapter, BLOCK_TSEM, init_phase);
   ecore_init_block(adapter, BLOCK_USEM, init_phase);
   ecore_init_block(adapter, BLOCK_CSEM, init_phase);
   ecore_init_block(adapter, BLOCK_XSEM, init_phase);

   ecore_init_block(adapter, BLOCK_UPB, init_phase);
   ecore_init_block(adapter, BLOCK_XPB, init_phase);

   ecore_init_block(adapter, BLOCK_PBF, init_phase);

   ecore_init_block(adapter, BLOCK_CDU, init_phase);
   ecore_init_block(adapter, BLOCK_CFC, init_phase);

   ecore_init_block(adapter, BLOCK_HC, init_phase);

   ecore_init_block(adapter, BLOCK_IGU, init_phase);

   ecore_init_block(adapter, BLOCK_MISC_AEU, init_phase);
   /*
    * init aeu_mask_attn_func_0/1:
    * *  - SF mode: bits 3-7 are masked. Only bits 0-2 are in use
    * *  - MF mode: bit 3 is masked. Bits 0-2 are in use as in SF
    * *             bits 4-7 are used for "per vn group attention"
    */
   val = IS_MF(adapter) ? 0xF7 : 0x7;
   /*
    * Enable DCBX attention for all but E1
    */
   val |= CHIP_IS_E1(adapter) ? 0 : 0x10;
   REG_WR(adapter, MISC_REG_AEU_MASK_ATTN_FUNC_0 + port * 4, val);

   /*
    * SCPAD_PARITY should NOT trigger close the gates
    */
   reg = port ? MISC_REG_AEU_ENABLE4_NIG_1 : MISC_REG_AEU_ENABLE4_NIG_0;
   REG_WR(adapter, reg,
	  REG_RD(adapter, reg) & ~AEU_INPUTS_ATTN_BITS_MCP_LATCHED_SCPAD_PARITY);

   reg = port ? MISC_REG_AEU_ENABLE4_PXP_1 : MISC_REG_AEU_ENABLE4_PXP_0;
   REG_WR(adapter, reg,
	  REG_RD(adapter, reg) & ~AEU_INPUTS_ATTN_BITS_MCP_LATCHED_SCPAD_PARITY);

   ecore_init_block(adapter, BLOCK_NIG, init_phase);

   if (!CHIP_IS_E1x(adapter)) {
      /*
       * Bit-map indicating which L2 hdrs may appear after the
       * * basic Ethernet header
       */
      if (IS_MF_AFEX(adapter))
	 REG_WR(adapter, QFLE3_PORT(adapter) ?
		NIG_REG_P1_HDRS_AFTER_BASIC : NIG_REG_P0_HDRS_AFTER_BASIC, 0xE);
      else
	 REG_WR(adapter, QFLE3_PORT(adapter) ?
		NIG_REG_P1_HDRS_AFTER_BASIC :
		NIG_REG_P0_HDRS_AFTER_BASIC, IS_MF_SD(adapter) ? 7 : 6);

      if (CHIP_IS_E3(adapter))
	 REG_WR(adapter, QFLE3_PORT(adapter) ?
		NIG_REG_LLH1_MF_MODE : NIG_REG_LLH_MF_MODE, IS_MF(adapter));
   }

   if (!CHIP_IS_E3(adapter))
      REG_WR(adapter, NIG_REG_XGXS_SERDES0_MODE_SEL + port * 4, 1);

   if (!CHIP_IS_E1(adapter)) {
      /*
       * 0x2 disable mf_ov, 0x1 enable
       */
      REG_WR(adapter, NIG_REG_LLH0_BRB1_DRV_MASK_MF + port * 4,
	     (IS_MF_SD(adapter) ? 0x1 : 0x2));

      if (!CHIP_IS_E1x(adapter)) {
	 val = 0;
	 switch (adapter->mf_mode) {
	 case MULTI_FUNCTION_SD:
	    val = 1;
	    break;
	 case MULTI_FUNCTION_SI:
	 case MULTI_FUNCTION_AFEX:
	    val = 2;
	    break;
	 }

	 REG_WR(adapter, (QFLE3_PORT(adapter) ? NIG_REG_LLH1_CLS_TYPE :
			  NIG_REG_LLH0_CLS_TYPE), val);
      }

      REG_WR(adapter, NIG_REG_LLFC_ENABLE_0 + port * 4, 0);
      REG_WR(adapter, NIG_REG_LLFC_OUT_EN_0 + port * 4, 0);
      REG_WR(adapter, NIG_REG_PAUSE_ENABLE_0 + port * 4, 1);
   }

   /*
    * If SPIO5 is set to generate interrupts, enable it for this port
    */
   val = REG_RD(adapter, MISC_REG_SPIO_EVENT_EN);
   if (val & MISC_SPIO_SPIO5) {
      vmk_uint32 reg_addr = (port ? MISC_REG_AEU_ENABLE1_FUNC_1_OUT_0 :
			     MISC_REG_AEU_ENABLE1_FUNC_0_OUT_0);
      val = REG_RD(adapter, reg_addr);
      val |= AEU_INPUTS_ATTN_BITS_SPIO5;
      REG_WR(adapter, reg_addr, val);
   }

   return 0;
}

static void
qfle3_config_endianity(struct qfle3_adapter *adapter, vmk_uint32 val)
{
   REG_WR(adapter, PXP2_REG_RQ_QM_ENDIAN_M, val);
   REG_WR(adapter, PXP2_REG_RQ_TM_ENDIAN_M, val);
   REG_WR(adapter, PXP2_REG_RQ_SRC_ENDIAN_M, val);
   REG_WR(adapter, PXP2_REG_RQ_CDU_ENDIAN_M, val);
   REG_WR(adapter, PXP2_REG_RQ_DBG_ENDIAN_M, val);

   /*
    * make sure this value is 0
    */
   REG_WR(adapter, PXP2_REG_RQ_HC_ENDIAN_M, 0);

   REG_WR(adapter, PXP2_REG_RD_QM_SWAP_MODE, val);
   REG_WR(adapter, PXP2_REG_RD_TM_SWAP_MODE, val);
   REG_WR(adapter, PXP2_REG_RD_SRC_SWAP_MODE, val);
   REG_WR(adapter, PXP2_REG_RD_CDURD_SWAP_MODE, val);
}

static void
qfle3_set_endianity(struct qfle3_adapter *adapter)
{
#ifdef __BIG_ENDIAN
   qfle3_config_endianity(adapter, 1);
#else
   qfle3_config_endianity(adapter, 0);
#endif
}

#if 0
static void
qfle3_reset_endianity(struct qfle3_adapter *adapter)
{
   qfle3_config_endianity(adapter, 0);
}
#endif

static void
qfle3_init_pxp(struct qfle3_adapter *adapter)
{
   vmk_uint32 devctl;
   int r_order, w_order;

   qfle3_pcie_cap_read_word(adapter->pdev, PCI_EXP_DEVCTL, &devctl);
   QFLE3_DBG(QFLE3_DBG_HW, "read 0x%x from devctl\n", devctl);
   w_order = ((devctl & PCI_EXP_DEVCTL_PAYLOAD) >> 5);
   if (adapter->mrrs == -1)
      r_order = ((devctl & PCI_EXP_DEVCTL_READRQ) >> 12);
   else {
      QFLE3_DBG(QFLE3_DBG_HW, "force read order to %d\n", adapter->mrrs);
      r_order = adapter->mrrs;
   }

   ecore_init_pxp_arb(adapter, r_order, w_order);
}

static vmk_uint32
qfle3_get_pretend_reg(struct qfle3_adapter *adapter)
{
   vmk_uint32 base = PXP2_REG_PGL_PRETEND_FUNC_F0;
   vmk_uint32 stride = PXP2_REG_PGL_PRETEND_FUNC_F1 - base;
   return base + (QFLE3_ABS_FUNC(adapter)) * stride;
}

int
qfle3_pretend_func(struct qfle3_adapter *adapter, vmk_uint16 pretend_func_val)
{
   vmk_uint32 pretend_reg;

   if (CHIP_IS_E1H(adapter) && pretend_func_val >= E1H_FUNC_MAX)
      return -1;

   /*
    * get my own pretend register
    */
   pretend_reg = qfle3_get_pretend_reg(adapter);
   REG_WR(adapter, pretend_reg, pretend_func_val);
   pretend_reg = REG_RD(adapter, pretend_reg);
   return 0;
}

static void
qfle3_enable_blocks_attention(struct qfle3_adapter *adapter)
{
   vmk_uint32 val;

   REG_WR(adapter, PXP_REG_PXP_INT_MASK_0, 0);
   if (!CHIP_IS_E1x(adapter))
      REG_WR(adapter, PXP_REG_PXP_INT_MASK_1, 0x40);
   else
      REG_WR(adapter, PXP_REG_PXP_INT_MASK_1, 0);
   REG_WR(adapter, DORQ_REG_DORQ_INT_MASK, 0);
   REG_WR(adapter, CFC_REG_CFC_INT_MASK, 0);
   /*
    * mask read length error interrupts in brb for parser
    * (parsing unit and 'checksum and crc' unit)
    * these errors are legal (PU reads fixed length and CAC can cause
    * read length error on truncated packets)
    */
   REG_WR(adapter, BRB1_REG_BRB1_INT_MASK, 0xFC00);
   REG_WR(adapter, QM_REG_QM_INT_MASK, 0);
   REG_WR(adapter, TM_REG_TM_INT_MASK, 0);
   REG_WR(adapter, XSDM_REG_XSDM_INT_MASK_0, 0);
   REG_WR(adapter, XSDM_REG_XSDM_INT_MASK_1, 0);
   REG_WR(adapter, XCM_REG_XCM_INT_MASK, 0);
/*  REG_WR(adapter, XSEM_REG_XSEM_INT_MASK_0, 0); */
/*  REG_WR(adapter, XSEM_REG_XSEM_INT_MASK_1, 0); */
   REG_WR(adapter, USDM_REG_USDM_INT_MASK_0, 0);
   REG_WR(adapter, USDM_REG_USDM_INT_MASK_1, 0);
   REG_WR(adapter, UCM_REG_UCM_INT_MASK, 0);
/*  REG_WR(adapter, USEM_REG_USEM_INT_MASK_0, 0); */
/*  REG_WR(adapter, USEM_REG_USEM_INT_MASK_1, 0); */
   REG_WR(adapter, GRCBASE_UPB + PB_REG_PB_INT_MASK, 0);
   REG_WR(adapter, CSDM_REG_CSDM_INT_MASK_0, 0);
   REG_WR(adapter, CSDM_REG_CSDM_INT_MASK_1, 0);
   REG_WR(adapter, CCM_REG_CCM_INT_MASK, 0);
/*  REG_WR(adapter, CSEM_REG_CSEM_INT_MASK_0, 0); */
/*  REG_WR(adapter, CSEM_REG_CSEM_INT_MASK_1, 0); */

   val = PXP2_PXP2_INT_MASK_0_REG_PGL_CPL_AFT |
      PXP2_PXP2_INT_MASK_0_REG_PGL_CPL_OF | PXP2_PXP2_INT_MASK_0_REG_PGL_PCIE_ATTN;
   if (!CHIP_IS_E1x(adapter))
      val |= PXP2_PXP2_INT_MASK_0_REG_PGL_READ_BLOCKED |
	 PXP2_PXP2_INT_MASK_0_REG_PGL_WRITE_BLOCKED;
   REG_WR(adapter, PXP2_REG_PXP2_INT_MASK_0, val);

   REG_WR(adapter, TSDM_REG_TSDM_INT_MASK_0, 0);
   REG_WR(adapter, TSDM_REG_TSDM_INT_MASK_1, 0);
   REG_WR(adapter, TCM_REG_TCM_INT_MASK, 0);
/*  REG_WR(adapter, TSEM_REG_TSEM_INT_MASK_0, 0); */

   if (!CHIP_IS_E1x(adapter))
      /*
       * enable VFC attentions: bits 11 and 12, bits 31:13 reserved
       */
      REG_WR(adapter, TSEM_REG_TSEM_INT_MASK_1, 0x07ff);

   REG_WR(adapter, CDU_REG_CDU_INT_MASK, 0);
   REG_WR(adapter, DMAE_REG_DMAE_INT_MASK, 0);
/*  REG_WR(adapter, MISC_REG_MISC_INT_MASK, 0); */
   REG_WR(adapter, PBF_REG_PBF_INT_MASK, 0x18); /* bit 3,4 masked */
}

static int
qfle3_set_spio(struct qfle3_adapter *adapter, int spio, vmk_uint32 mode)
{
   vmk_uint32 spio_reg;

   /*
    * Only 2 SPIOs are configurable
    */
   if ((spio != MISC_SPIO_SPIO4) && (spio != MISC_SPIO_SPIO5)) {
      QFLE3_ERR("Invalid SPIO 0x%x\n", spio);
      return VMK_BAD_PARAM;
   }

   qfle3_acquire_hw_lock(adapter, HW_LOCK_RESOURCE_SPIO);
   /*
    * read SPIO and mask except the float bits
    */
   spio_reg = (REG_RD(adapter, MISC_REG_SPIO) & MISC_SPIO_FLOAT);

   switch (mode) {
   case MISC_SPIO_OUTPUT_LOW:
      QFLE3_DBG(QFLE3_DBG_HW, "Set SPIO 0x%x -> output low\n", spio);
      /*
       * clear FLOAT and set CLR
       */
      spio_reg &= ~(spio << MISC_SPIO_FLOAT_POS);
      spio_reg |= (spio << MISC_SPIO_CLR_POS);
      break;

   case MISC_SPIO_OUTPUT_HIGH:
      QFLE3_DBG(QFLE3_DBG_HW, "Set SPIO 0x%x -> output high\n", spio);
      /*
       * clear FLOAT and set SET
       */
      spio_reg &= ~(spio << MISC_SPIO_FLOAT_POS);
      spio_reg |= (spio << MISC_SPIO_SET_POS);
      break;

   case MISC_SPIO_INPUT_HI_Z:
      QFLE3_DBG(QFLE3_DBG_HW, "Set SPIO 0x%x -> input\n", spio);
      /*
       * set FLOAT
       */
      spio_reg |= (spio << MISC_SPIO_FLOAT_POS);
      break;

   default:
      break;
   }
   REG_WR(adapter, MISC_REG_SPIO, spio_reg);
   qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_SPIO);

   return 0;
}

static void
qfle3_setup_fan_failure_detection(struct qfle3_adapter *adapter)
{
   int is_required;
   vmk_uint32 val;
   int port;

   is_required = 0;
   val = SHMEM_RD(adapter, dev_info.shared_hw_config.config2) &
      SHARED_HW_CFG_FAN_FAILURE_MASK;

   if (val == SHARED_HW_CFG_FAN_FAILURE_ENABLED)
      is_required = 1;

   /*
    * The fan failure mechanism is usually related to the PHY type since
    * the power consumption of the board is affected by the PHY. Currently,
    * fan is required for most designs with SFX7101, BCM8727 and BCM8481.
    */
   else if (val == SHARED_HW_CFG_FAN_FAILURE_PHY_TYPE)
      for (port = PORT_0; port < PORT_MAX; port++) {
	 is_required |=
	    elink_fan_failure_det_req(adapter,
				      adapter->hw_info.shmem_base,
				      adapter->hw_info.shmem2_base, port);
      }
   QFLE3_DBG(QFLE3_DBG_HW, "fan detection setting: %d\n", is_required);

   if (is_required == 0)
      return;

   /*
    * Fan failure is indicated by SPIO 5
    */
   qfle3_set_spio(adapter, MISC_SPIO_SPIO5, MISC_SPIO_INPUT_HI_Z);

   /*
    * set to active low mode
    */
   val = REG_RD(adapter, MISC_REG_SPIO_INT);
   val |= (MISC_SPIO_SPIO5 << MISC_SPIO_INT_OLD_SET_POS);
   REG_WR(adapter, MISC_REG_SPIO_INT, val);

   /*
    * enable interrupt to signal the IGU
    */
   val = REG_RD(adapter, MISC_REG_SPIO_EVENT_EN);
   val |= MISC_SPIO_SPIO5;
   REG_WR(adapter, MISC_REG_SPIO_EVENT_EN, val);
}

/**
 * qfle3_init_hw_common - initialize the HW at the COMMON phase.
 *
 * @adapter:     driver handle
 */
static int
qfle3_init_hw_common(struct qfle3_adapter *adapter)
{
   vmk_uint32 val;

   QFLE3_DBG(QFLE3_DBG_HW, "starting common init  func %d\n", QFLE3_ABS_FUNC(adapter));

   /*
    * take the RESET lock to protect undi_unload flow from accessing
    * registers while we're resetting the chip
    */
   qfle3_acquire_hw_lock(adapter, HW_LOCK_RESOURCE_RESET);

   qfle3_reset_common(adapter);
   REG_WR(adapter, GRCBASE_MISC + MISC_REGISTERS_RESET_REG_1_SET, 0xffffffff);

   val = 0xfffc;
   if (CHIP_IS_E3(adapter)) {
      val |= MISC_REGISTERS_RESET_REG_2_MSTAT0;
      val |= MISC_REGISTERS_RESET_REG_2_MSTAT1;
   }
   REG_WR(adapter, GRCBASE_MISC + MISC_REGISTERS_RESET_REG_2_SET, val);

   qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_RESET);

   ecore_init_block(adapter, BLOCK_MISC, PHASE_COMMON);

   QFLE3_DBG(QFLE3_DBG_HW, "Initialized MISC block \n");

   if (!CHIP_IS_E1x(adapter)) {
      vmk_uint8 abs_func_id;

      /**
       * 4-port mode or 2-port mode we need to turn of master-enable
       * for everyone, after that, turn it back on for self.
       * so, we disregard multi-function or not, and always disable
       * for all functions on the given path, this means 0,2,4,6 for
       * path 0 and 1,3,5,7 for path 1
       */
      for (abs_func_id = QFLE3_PATH(adapter);
	   abs_func_id < E2_FUNC_MAX * 2; abs_func_id += 2) {
	 if (abs_func_id == QFLE3_ABS_FUNC(adapter)) {
	    REG_WR(adapter, PGLUE_B_REG_INTERNAL_PFID_ENABLE_MASTER, 1);
	    continue;
	 }

	 qfle3_pretend_func(adapter, abs_func_id);
	 /*
	  * clear pf enable
	  */
	 qfle3_pf_disable(adapter);
	 qfle3_pretend_func(adapter, QFLE3_ABS_FUNC(adapter));
      }
   }

   ecore_init_block(adapter, BLOCK_PXP, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW,"Initialized PXP block \n");

   ecore_init_block(adapter, BLOCK_PXP2, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "Initialized PXP2 block \n");

   qfle3_init_pxp(adapter);
   qfle3_set_endianity(adapter);
   ecore_ilt_init_page_size(adapter, INITOP_SET);

   /*
    * let the HW do it's magic ...
    */
   vmk_WorldSleep(100 * 1000);
   /*
    * finish PXP init
    */
   val = REG_RD(adapter, PXP2_REG_RQ_CFG_DONE);
   if (val != 1) {
      QFLE3_ERR("PXP2 CFG failed\n");
      return VMK_BUSY;
   }
   val = REG_RD(adapter, PXP2_REG_RD_INIT_DONE);
   if (val != 1) {
      QFLE3_ERR("PXP2 RD_INIT failed\n");
      return VMK_BUSY;
   }

   /*
    * Timers bug workaround E2 only. We need to set the entire ILT to
    * * have entries with value "0" and valid bit on.
    * * This needs to be done by the first PF that is loaded in a path
    * * (i.e. common phase)
    */

   if (!CHIP_IS_E1x(adapter)) {
/* In E2 there is a bug in the timers block that can cause function 6 / 7
 * (i.e. vnic3) to start even if it is marked as "scan-off".
 * This occurs when a different function (func2,3) is being marked
 * as "scan-off". Real-life scenario for example: if a driver is being
 * load-unloaded while func6,7 are down. This will cause the timer to access
 * the ilt, translate to a logical address and send a request to read/write.
 * Since the ilt for the function that is down is not valid, this will cause
 * a translation error which is unrecoverable.
 * The Workaround is intended to make sure that when this happens nothing fatal
 * will occur. The workaround:
 *  1.  First PF driver which loads on a path will:
 *      a.  After taking the chip out of reset, by using pretend,
 *          it will write "0" to the following registers of
 *          the other vnics.
 *          REG_WR(pdev, PGLUE_B_REG_INTERNAL_PFID_ENABLE_MASTER, 0);
 *          REG_WR(pdev, CFC_REG_WEAK_ENABLE_PF,0);
 *          REG_WR(pdev, CFC_REG_STRONG_ENABLE_PF,0);
 *          And for itself it will write '1' to
 *          PGLUE_B_REG_INTERNAL_PFID_ENABLE_MASTER to enable
 *          dmae-operations (writing to pram for example.)
 *          note: can be done for only function 6,7 but cleaner this
 *            way.
 *      b.  Write zero+valid to the entire ILT.
 *      c.  Init the first_timers_ilt_entry, last_timers_ilt_entry of
 *          VNIC3 (of that port). The range allocated will be the
 *          entire ILT. This is needed to prevent  ILT range error.
 *  2.  Any PF driver load flow:
 *      a.  ILT update with the physical addresses of the allocated
 *          logical pages.
 *      b.  Wait 20msec. - note that this timeout is needed to make
 *          sure there are no requests in one of the PXP internal
 *          queues with "old" ILT addresses.
 *      c.  PF enable in the PGLC.
 *      d.  Clear the was_error of the PF in the PGLC. (could have
 *          occurred while driver was down)
 *      e.  PF enable in the CFC (WEAK + STRONG)
 *      f.  Timers scan enable
 *  3.  PF driver unload flow:
 *      a.  Clear the Timers scan_en.
 *      b.  Polling for scan_on=0 for that PF.
 *      c.  Clear the PF enable bit in the PXP.
 *      d.  Clear the PF enable in the CFC (WEAK + STRONG)
 *      e.  Write zero+valid to all ILT entries (The valid bit must
 *          stay set)
 *      f.  If this is VNIC 3 of a port then also init
 *          first_timers_ilt_entry to zero and last_timers_ilt_entry
 *          to the last entry in the ILT.
 *
 *  Notes:
 *  Currently the PF error in the PGLC is non recoverable.
 *  In the future the there will be a recovery routine for this error.
 *  Currently attention is masked.
 *  Having an MCP lock on the load/unload process does not guarantee that
 *  there is no Timer disable during Func6/7 enable. This is because the
 *  Timers scan is currently being cleared by the MCP on FLR.
 *  Step 2.d can be done only for PF6/7 and the driver can also check if
 *  there is error before clearing it. But the flow above is simpler and
 *  more general.
 *  All ILT entries are written by zero+valid and not just PF6/7
 *  ILT entries since in the future the ILT entries allocation for
 *  PF-s might be dynamic.
 */
      struct ilt_client_info ilt_cli;
      struct ecore_ilt ilt;
      vmk_Memset(&ilt_cli, 0, sizeof(struct ilt_client_info));
      vmk_Memset(&ilt, 0, sizeof(struct ecore_ilt));

      /*
       * initialize dummy TM client
       */
      ilt_cli.start = 0;
      ilt_cli.end = ILT_NUM_PAGE_ENTRIES - 1;
      ilt_cli.client_num = ILT_CLIENT_TM;

      /*
       * Step 1: set zeroes to all ilt page entries with valid bit on
       * * Step 2: set the timers first/last ilt entry to point
       * * to the entire range to prevent ILT range error for 3rd/4th
       * * vnic (this code assumes existence of the vnic)
       * *
       * * both steps performed by call to qfle3_ilt_client_init_op()
       * * with dummy TM client
       * *
       * * we must use pretend since PXP2_REG_RQ_##blk##_FIRST_ILT
       * * and his brother are split registers
       */
      qfle3_pretend_func(adapter, (QFLE3_PATH(adapter) + 6));
      ecore_ilt_client_init_op_ilt(adapter, &ilt, &ilt_cli, INITOP_CLEAR);
      qfle3_pretend_func(adapter, QFLE3_ABS_FUNC(adapter));

      REG_WR(adapter, PXP2_REG_RQ_DRAM_ALIGN, QFLE3_PXP_DRAM_ALIGN);
      REG_WR(adapter, PXP2_REG_RQ_DRAM_ALIGN_RD, QFLE3_PXP_DRAM_ALIGN);
      REG_WR(adapter, PXP2_REG_RQ_DRAM_ALIGN_SEL, 1);
   }
   REG_WR(adapter, PXP2_REG_RQ_DISABLE_INPUTS, 0);
   REG_WR(adapter, PXP2_REG_RD_DISABLE_INPUTS, 0);

   if (!CHIP_IS_E1x(adapter)) {
      int factor = CHIP_REV_IS_EMUL(adapter) ? 1000 :
	 (CHIP_REV_IS_FPGA(adapter) ? 400 : 0);
      ecore_init_block(adapter, BLOCK_PGLUE_B, PHASE_COMMON);

      ecore_init_block(adapter, BLOCK_ATC, PHASE_COMMON);

      /*
       * let the HW do it's magic ...
       */
      do {
	 vmk_WorldSleep(200 * 1000);
	 val = REG_RD(adapter, ATC_REG_ATC_INIT_DONE);
      } while (factor-- && (val != 1));

      if (val != 1) {
	 QFLE3_ERR("ATC_INIT failed\n");
	 return VMK_BUSY;
      }
   }

   ecore_init_block(adapter, BLOCK_DMAE, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "DMAE init done\n");

#ifdef QFLE3_SRIOV
   if (adapter->sriov_pos){
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: adapter->sriov_pos is %d\n", adapter->sriov_pos);
      REG_WR(adapter, DMAE_REG_BACKWARD_COMP_EN, 0);
   }
#endif //QFLE3_SRIOV

   /*
    * clean the DMAE memory
    */
   adapter->dmae_ready = 1;
   ecore_init_fill(adapter, TSEM_REG_PRAM, 0, 8, 1);
   QFLE3_DBG(QFLE3_DBG_HW, "init fill TSEM_REG_PRAM done\n");

   ecore_init_block(adapter, BLOCK_TCM, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "TCM init done\n");

   ecore_init_block(adapter, BLOCK_UCM, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "UCM init done\n");

   ecore_init_block(adapter, BLOCK_CCM, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "CCM init done\n");

   ecore_init_block(adapter, BLOCK_XCM, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "XCM init done\n");

   qfle3_read_dmae(adapter, XSEM_REG_PASSIVE_BUFFER, 3);
   qfle3_read_dmae(adapter, CSEM_REG_PASSIVE_BUFFER, 3);
   qfle3_read_dmae(adapter, TSEM_REG_PASSIVE_BUFFER, 3);
   qfle3_read_dmae(adapter, USEM_REG_PASSIVE_BUFFER, 3);

   ecore_init_block(adapter, BLOCK_QM, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "QM init done\n");

   /*
    * QM queues pointers table
    */
   ecore_qm_init_ptr_table(adapter, adapter->qm_cid_count, INITOP_SET);

   /*
    * soft reset pulse
    */
   REG_WR(adapter, QM_REG_SOFT_RESET, 1);
   REG_WR(adapter, QM_REG_SOFT_RESET, 0);

   if (CNIC_SUPPORT(adapter))
      ecore_init_block(adapter, BLOCK_TM, PHASE_COMMON);

   ecore_init_block(adapter, BLOCK_DORQ, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "DORQ init done\n");

   if (!CHIP_REV_IS_SLOW(adapter))
      /*
       * enable hw interrupt from doorbell Q
       */
      REG_WR(adapter, DORQ_REG_DORQ_INT_MASK, 0);

   ecore_init_block(adapter, BLOCK_BRB1, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "BRB1 init done\n");

   ecore_init_block(adapter, BLOCK_PRS, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "PRS init done\n");
   REG_WR(adapter, PRS_REG_A_PRSU_20, 0xf);

   if (!CHIP_IS_E1(adapter))
      REG_WR(adapter, PRS_REG_E1HOV_MODE, adapter->path_has_ovlan);

   if (!CHIP_IS_E1x(adapter) && !CHIP_IS_E3B0(adapter)) {
      if (IS_MF_AFEX(adapter)) {
	 /*
	  * configure that VNTag and VLAN headers must be
	  * * received in afex mode
	  */
	 REG_WR(adapter, PRS_REG_HDRS_AFTER_BASIC, 0xE);
	 REG_WR(adapter, PRS_REG_MUST_HAVE_HDRS, 0xA);
	 REG_WR(adapter, PRS_REG_HDRS_AFTER_TAG_0, 0x6);
	 REG_WR(adapter, PRS_REG_TAG_ETHERTYPE_0, 0x8926);
	 REG_WR(adapter, PRS_REG_TAG_LEN_0, 0x4);
      } else {
	 /*
	  * Bit-map indicating which L2 hdrs may appear
	  * * after the basic Ethernet header
	  */
	 REG_WR(adapter, PRS_REG_HDRS_AFTER_BASIC, adapter->path_has_ovlan ? 7 : 6);
      }
   }

   ecore_init_block(adapter, BLOCK_TSDM, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "TSDM init done\n");
   ecore_init_block(adapter, BLOCK_CSDM, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "CSDM init done\n");
   ecore_init_block(adapter, BLOCK_USDM, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "USDM init done\n");
   ecore_init_block(adapter, BLOCK_XSDM, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "XSDM init done\n");

   if (!CHIP_IS_E1x(adapter)) {
      /*
       * reset VFC memories
       */
      REG_WR(adapter, TSEM_REG_FAST_MEMORY + VFC_REG_MEMORIES_RST,
	     VFC_MEMORIES_RST_REG_CAM_RST | VFC_MEMORIES_RST_REG_RAM_RST);
      REG_WR(adapter, XSEM_REG_FAST_MEMORY + VFC_REG_MEMORIES_RST,
	     VFC_MEMORIES_RST_REG_CAM_RST | VFC_MEMORIES_RST_REG_RAM_RST);

      vmk_WorldSleep(20 * 1000);
   }
   QFLE3_DBG(QFLE3_DBG_HW, "Reset VFC done\n");
   ecore_init_block(adapter, BLOCK_TSEM, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "TSEM init done\n");
   ecore_init_block(adapter, BLOCK_USEM, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "RSEM init done\n");
   ecore_init_block(adapter, BLOCK_CSEM, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "CSEM init done\n");
   ecore_init_block(adapter, BLOCK_XSEM, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "XSEM init done\n");

   /*
    * sync semi rtc
    */
   REG_WR(adapter, GRCBASE_MISC + MISC_REGISTERS_RESET_REG_1_CLEAR, 0x80000000);
   REG_WR(adapter, GRCBASE_MISC + MISC_REGISTERS_RESET_REG_1_SET, 0x80000000);

   ecore_init_block(adapter, BLOCK_UPB, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "UPB init done\n");
   ecore_init_block(adapter, BLOCK_XPB, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "XPB init done\n");
   ecore_init_block(adapter, BLOCK_PBF, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "PBF init done\n");

   if (!CHIP_IS_E1x(adapter)) {
      if (IS_MF_AFEX(adapter)) {
	 /*
	  * configure that VNTag and VLAN headers must be
	  * * sent in afex mode
	  */
	 REG_WR(adapter, PBF_REG_HDRS_AFTER_BASIC, 0xE);
	 REG_WR(adapter, PBF_REG_MUST_HAVE_HDRS, 0xA);
	 REG_WR(adapter, PBF_REG_HDRS_AFTER_TAG_0, 0x6);
	 REG_WR(adapter, PBF_REG_TAG_ETHERTYPE_0, 0x8926);
	 REG_WR(adapter, PBF_REG_TAG_LEN_0, 0x4);
      } else {
	 REG_WR(adapter, PBF_REG_HDRS_AFTER_BASIC, adapter->path_has_ovlan ? 7 : 6);
      }
   }

   REG_WR(adapter, SRC_REG_SOFT_RST, 1);

   ecore_init_block(adapter, BLOCK_SRC, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "SRC init done\n");
   if (CNIC_SUPPORT(adapter)) {
      REG_WR(adapter, SRC_REG_KEYSEARCH_0, 0x63285672);
      REG_WR(adapter, SRC_REG_KEYSEARCH_1, 0x24b8f2cc);
      REG_WR(adapter, SRC_REG_KEYSEARCH_2, 0x223aef9b);
      REG_WR(adapter, SRC_REG_KEYSEARCH_3, 0x26001e3a);
      REG_WR(adapter, SRC_REG_KEYSEARCH_4, 0x7ae91116);
      REG_WR(adapter, SRC_REG_KEYSEARCH_5, 0x5ce5230b);
      REG_WR(adapter, SRC_REG_KEYSEARCH_6, 0x298d8adf);
      REG_WR(adapter, SRC_REG_KEYSEARCH_7, 0x6eb0ff09);
      REG_WR(adapter, SRC_REG_KEYSEARCH_8, 0x1830f82f);
      REG_WR(adapter, SRC_REG_KEYSEARCH_9, 0x01e46be7);
   }
   REG_WR(adapter, SRC_REG_SOFT_RST, 0);

   if (sizeof(union cdu_context) != 1024)
      QFLE3_ERR("please adjust the size of cdu_context(%ld)\n",
		(long) sizeof(union cdu_context));

   ecore_init_block(adapter, BLOCK_CDU, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "CDU init done\n");
   val = (4 << 24) + (0 << 12) + 1024;
   REG_WR(adapter, CDU_REG_CDU_GLOBAL_PARAMS, val);

   ecore_init_block(adapter, BLOCK_CFC, PHASE_COMMON);
   REG_WR(adapter, CFC_REG_INIT_REG, 0x7FF);
   QFLE3_DBG(QFLE3_DBG_HW, "CFC init done\n");
   /*
    * enable context validation interrupt from CFC
    */
   REG_WR(adapter, CFC_REG_CFC_INT_MASK, 0);

   /*
    * set the thresholds to prevent CFC/CDU race
    */
   REG_WR(adapter, CFC_REG_DEBUG0, 0x20020000);

   ecore_init_block(adapter, BLOCK_HC, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "HC init done\n");

   if (!CHIP_IS_E1x(adapter) && QFLE3_NOMCP(adapter))
      REG_WR(adapter, IGU_REG_RESET_MEMORIES, 0x36);

   ecore_init_block(adapter, BLOCK_IGU, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "IGU init done\n");
   ecore_init_block(adapter, BLOCK_MISC_AEU, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "AEU init done\n");

   /*
    * Reset PCIE errors for debug
    */
   REG_WR(adapter, 0x2814, 0xffffffff);
   REG_WR(adapter, 0x3820, 0xffffffff);

   if (!CHIP_IS_E1x(adapter)) {
      REG_WR(adapter, PCICFG_OFFSET + PXPCS_TL_CONTROL_5,
	     (PXPCS_TL_CONTROL_5_ERR_UNSPPORT1 | PXPCS_TL_CONTROL_5_ERR_UNSPPORT));
      REG_WR(adapter, PCICFG_OFFSET + PXPCS_TL_FUNC345_STAT,
	     (PXPCS_TL_FUNC345_STAT_ERR_UNSPPORT4 |
	      PXPCS_TL_FUNC345_STAT_ERR_UNSPPORT3 |
	      PXPCS_TL_FUNC345_STAT_ERR_UNSPPORT2));
      REG_WR(adapter, PCICFG_OFFSET + PXPCS_TL_FUNC678_STAT,
	     (PXPCS_TL_FUNC678_STAT_ERR_UNSPPORT7 |
	      PXPCS_TL_FUNC678_STAT_ERR_UNSPPORT6 |
	      PXPCS_TL_FUNC678_STAT_ERR_UNSPPORT5));
   }

   ecore_init_block(adapter, BLOCK_NIG, PHASE_COMMON);
   QFLE3_DBG(QFLE3_DBG_HW, "NIG init done\n");
   if (!CHIP_IS_E1(adapter)) {
      /*
       * in E3 this done in per-port section
       */
      if (!CHIP_IS_E3(adapter))
	 REG_WR(adapter, NIG_REG_LLH_MF_MODE, IS_MF(adapter));
   }

   /*
    * finish CFC init
    */
   val = reg_poll(adapter, CFC_REG_LL_INIT_DONE, 1, 100, 10);
   if (val != 1) {
      QFLE3_ERR("CFC LL_INIT failed\n");
      return VMK_BUSY;
   }
   val = reg_poll(adapter, CFC_REG_AC_INIT_DONE, 1, 100, 10);
   if (val != 1) {
      QFLE3_ERR("CFC AC_INIT failed\n");
      return VMK_BUSY;
   }
   val = reg_poll(adapter, CFC_REG_CAM_INIT_DONE, 1, 100, 10);
   if (val != 1) {
      QFLE3_ERROR("CFC CAM_INIT failed\n");
      return VMK_BUSY;
   }
   REG_WR(adapter, CFC_REG_DEBUG0, 0);

   qfle3_setup_fan_failure_detection(adapter);

   /*
    * clear PXP2 attentions
    */
   val = REG_RD(adapter, PXP2_REG_PXP2_INT_STS_CLR_0);

   qfle3_enable_blocks_attention(adapter);
   if (!CHIP_REV_IS_SLOW(adapter))
      ecore_enable_blocks_parity(adapter);

   if (SHMEM2_HAS(adapter, netproc_fw_ver))
      SHMEM2_WR(adapter, netproc_fw_ver, REG_RD(adapter, XSEM_REG_PRAM));

   return 0;
}

static int
qfle3_init_hw_common_chip(struct qfle3_adapter *adapter)
{
   vmk_uint32 shmem_base[2], shmem2_base[2];
   int rc = qfle3_init_hw_common(adapter);

   if (rc) {
      QFLE3_ERR("failed to init hw\n");
      return rc;
   }

   /*
    * Avoid common init in case MFW supports LFA
    */
   if (SHMEM2_RD(adapter, size) >
       (vmk_uint32) vmk_offsetof(struct shmem2_region,
				 lfa_host_addr[QFLE3_PORT(adapter)])) {
      QFLE3_DBG(QFLE3_DBG_LINK, "Avoid common phy init as MFW supports LFA\n");
      return 0;
   }

   shmem_base[0] = adapter->hw_info.shmem_base;
   shmem2_base[0] = adapter->hw_info.shmem2_base;
   if (!CHIP_IS_E1x(adapter)) {
      shmem_base[1] = SHMEM2_RD(adapter, other_shmem_base_addr);
      shmem2_base[1] = SHMEM2_RD(adapter, other_shmem2_base_addr);
   }
   qfle3_acquire_phy_lock(adapter);
   elink_common_init_phy(adapter, shmem_base, shmem2_base, adapter->hw_info.chip_id, 0);
   qfle3_release_phy_lock(adapter);

   return 0;
}

static struct ecore_func_sp_drv_ops qfle3_func_sp_drv = {
   .init_hw_cmn_chip = qfle3_init_hw_common_chip,
   .init_hw_cmn = qfle3_init_hw_common,
   .init_hw_port = qfle3_init_hw_port,
   .init_hw_func = qfle3_init_hw_func,

   .reset_hw_cmn = qfle3_reset_common,
   .reset_hw_port = qfle3_reset_port,
   .reset_hw_func = qfle3_reset_func,

   .gunzip_init = qfle3_gunzip_init,
   .gunzip_end = qfle3_gunzip_end,

   .init_fw = qfle3_init_firmware,
   .release_fw = qfle3_release_firmware,
};

VMK_ReturnStatus
qfle3_compare_fw_ver(struct qfle3_adapter *adapter, vmk_uint32 load_code)
{
   /*
    * is another pf loaded on this engine?
    */
   if (load_code != FW_MSG_CODE_DRV_LOAD_COMMON_CHIP &&
       load_code != FW_MSG_CODE_DRV_LOAD_COMMON) {
      /*
       * build my FW version dword
       */
      vmk_uint32 my_fw = (BCM_5710_FW_MAJOR_VERSION) +
	 (BCM_5710_FW_MINOR_VERSION << 8) +
	 (BCM_5710_FW_REVISION_VERSION << 16) + (BCM_5710_FW_ENGINEERING_VERSION << 24);

      /*
       * read loaded FW from chip
       */
      vmk_uint32 loaded_fw = REG_RD(adapter, XSEM_REG_PRAM);

      QFLE3_DBG(QFLE3_DBG_LOAD, "loaded fw %x, my fw %x\n", loaded_fw, my_fw);

      /*
       * abort nic load if version mismatch
       */
      if (my_fw != loaded_fw) {
	 QFLE3_ERR
	    ("qfle3 with FW %x was already loaded which mismatches my %x FW. Aborting\n",
	     loaded_fw, my_fw);
	 return VMK_BUSY;
      }
   }
   return VMK_OK;
}

/* MFW/MCP related functions */

vmk_uint32
qfle3_mfw_command(struct qfle3_adapter *adapter, vmk_uint32 command, vmk_uint32 param)
{
   int mb_idx = QFLE3_FW_MB_IDX(adapter);
   vmk_uint32 seq;
   vmk_uint32 load_code = 0;
   vmk_uint32 cnt = 1;
   vmk_uint8 delay = 10;

   vmk_SemaLock(&adapter->mfw_cmd_mutex);
   seq = ++adapter->fw_seq;
   seq &= 0xffff;  // mask off upper bits because seq number is only 16 bits.
   adapter->fw_seq = seq;
   SHMEM_WR(adapter, func_mb[mb_idx].drv_mb_param, param);
   SHMEM_WR(adapter, func_mb[mb_idx].drv_mb_header, (command | seq));

   QFLE3_DBG(QFLE3_DBG_MCP,
	     "wrote command (%x) to FW MB param 0x%08x\n", (command | seq), param);

   do {
      vmk_WorldSleep(delay * 1000);

      load_code = SHMEM_RD(adapter, func_mb[mb_idx].fw_mb_header);

      /*
       * Give the FW up to 5 second (500*10ms)
       */
   } while ((seq != (load_code & FW_MSG_SEQ_NUMBER_MASK)) && (cnt++ < 500));

   QFLE3_DBG(QFLE3_DBG_MCP,
	     "after %d ms] read (%x) seq is (%x) from FW MB\n",
	     cnt * delay, load_code, seq);

   /*
    * is this a reply to our command?
    */
   if (seq == (load_code & FW_MSG_SEQ_NUMBER_MASK)) {
      load_code &= FW_MSG_CODE_MASK;
   } else {
      /*
       * FW BUG!
       */
      QFLE3_ERR("FW failed to respond! Command (%x) param (%x)\n", command, param);
      load_code = 0;
   }
   vmk_SemaUnlock(&adapter->mfw_cmd_mutex);

   return load_code;
}


vmk_uint32
qfle3_send_unload_req(struct qfle3_adapter *adapter, vmk_uint32 unload_mode)
{
   VMK_ReturnStatus status;
   vmk_uint32 pmc;
   vmk_uint8 entry;
   vmk_uint32 reset_code;

   int port = QFLE3_PORT(adapter);

   if (unload_mode == UNLOAD_NORMAL)
      reset_code = DRV_MSG_CODE_UNLOAD_REQ_WOL_DIS;
   
   else if (adapter->flags & NO_WOL_FLAG){
      reset_code = DRV_MSG_CODE_UNLOAD_REQ_WOL_MCP;
      QFLE3_DBG(QFLE3_DBG_LOAD, "NO WOL enabled\n");
   } else if (adapter->wol) {

      QFLE3_DBG(QFLE3_DBG_LOAD, "Enabling WOL in hardware\n");
      vmk_uint32 emac_base = port ? GRCBASE_EMAC1 : GRCBASE_EMAC0;
      vmk_EthAddress *mac_addr = &adapter->hwMacAddr;
      vmk_uint32 val;

      /* The mac address is written to entries 1-4 to
       * preserve entry 0 which is used by the PMF
       */
      entry = ((adapter->pf_id >> 1) + 1)*8;

      val = ((*mac_addr)[0] << 8) | (*mac_addr)[1];
      EMAC_WR(adapter, EMAC_REG_EMAC_MAC_MATCH + entry, val);

      val = ((*mac_addr)[2] << 24) | ((*mac_addr)[3] << 16) |
	 ((*mac_addr)[4] << 8) | (*mac_addr)[5];
      EMAC_WR(adapter, EMAC_REG_EMAC_MAC_MATCH + entry + 4, val);

      /* Enable the PME and clear the status */

      status = vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev, VMK_PCI_CONFIG_ACCESS_16, adapter->pm_cap + PCI_PM_CTRL , &pmc);

      pmc |= PCI_PM_CTRL_PME_ENABLE | PCI_PM_CTRL_PME_STATUS;
      status = vmk_PCIWriteConfig(vmk_ModuleCurrentID, adapter->pdev,VMK_PCI_CONFIG_ACCESS_16,adapter->pm_cap + PCI_PM_CTRL,pmc);

      reset_code = DRV_MSG_CODE_UNLOAD_REQ_WOL_EN;

   }else {
      reset_code = DRV_MSG_CODE_UNLOAD_REQ_WOL_DIS;
   }

   /* Send the request to the MCP */
   reset_code = qfle3_mfw_command(adapter, reset_code, 0);

   return reset_code;
}

static VMK_ReturnStatus
qfle3_nic_load_request(struct qfle3_adapter *adapter, vmk_uint32 * load_code)
{
   vmk_uint32 param;

   /*
    * init fw_seq
    */
   adapter->fw_seq =
      (SHMEM_RD(adapter, func_mb[QFLE3_FW_MB_IDX(adapter)].drv_mb_header) &
       DRV_MSG_SEQ_NUMBER_MASK);
   QFLE3_DBG(QFLE3_DBG_LOAD, "fw_seq 0x%08x\n", adapter->fw_seq);

#if 0
   /*
    * Get current FW pulse sequence
    */
   adapter->fw_drv_pulse_wr_seq =
      (SHMEM_RD(adapter, func_mb[QFLE3_FW_MB_IDX(adapter)].drv_pulse_mb) &
       DRV_PULSE_SEQ_MASK);
   QFLE3_DEV_INFO("drv_pulse 0x%x\n", adapter->fw_drv_pulse_wr_seq);
#endif

   param = DRV_MSG_CODE_LOAD_REQ_WITH_LFA;

   if (IS_MF_SD(adapter) && qfle3_port_after_undi(adapter))
      param |= DRV_MSG_CODE_LOAD_REQ_FORCE_LFA;

   /*
    * load request
    */
   (*load_code) = qfle3_mfw_command(adapter, DRV_MSG_CODE_LOAD_REQ, param);

   /*
    * if mcp fails to respond we must abort
    */
   if (!(*load_code)) {
      QFLE3_ERR("MCP response failure, aborting\n");
      return VMK_BUSY;
   }

   /*
    * If mcp refused (e.g. other port is in diagnostic mode) we
    * * must abort
    */
   if ((*load_code) == FW_MSG_CODE_DRV_LOAD_REFUSED) {
      QFLE3_ERR("MCP refused load request, aborting\n");
      return VMK_BUSY;
   }
   return VMK_OK;
}

vmk_uint32
elink_cb_reg_read(struct qfle3_adapter * adapter, vmk_uint32 reg_addr)
{
   return (REG_RD(adapter, reg_addr));
}

void
elink_cb_reg_write(struct qfle3_adapter *adapter, vmk_uint32 reg_addr, vmk_uint32 val)
{

   REG_WR(adapter, reg_addr, val);
}

/* wb_write - pointer to 2 32 bits vars to be passed to the DMAE*/
void
elink_cb_reg_wb_write(struct qfle3_adapter *adapter, vmk_uint32 offset,
		      vmk_uint32 * wb_write, vmk_uint16 len)
{
   QFLE3_DBG(QFLE3_DBG_LINK, "Function not implemented yet\n");
}

void
elink_cb_reg_wb_read(struct qfle3_adapter *adapter, vmk_uint32 offset,
		     vmk_uint32 * wb_write, vmk_uint16 len)
{
   QFLE3_DBG(QFLE3_DBG_LINK, "Function not implemented yet\n");
}

vmk_uint8
elink_cb_gpio_write(struct qfle3_adapter *adapter,
		    vmk_uint16 gpio_num, vmk_uint8 mode, vmk_uint8 port)
{
   /* The GPIO should be swapped if swap register is set and active */
   int gpio_port = (REG_RD(adapter, NIG_REG_PORT_SWAP) &&
		    REG_RD(adapter, NIG_REG_STRAP_OVERRIDE)) ^ port;
   int gpio_shift = gpio_num +
      (gpio_port ? MISC_REGISTERS_GPIO_PORT_SHIFT : 0);
   vmk_uint32 gpio_mask = (1 << gpio_shift);
   vmk_uint32 gpio_reg;

   if (gpio_num > MISC_REGISTERS_GPIO_3) {
      QFLE3_ERR("Invalid GPIO %d\n", gpio_num);
      return -1;
   }

   qfle3_acquire_hw_lock(adapter, HW_LOCK_RESOURCE_GPIO);
   /* read GPIO and mask except the float bits */
   gpio_reg = (REG_RD(adapter, MISC_REG_GPIO) & MISC_REGISTERS_GPIO_FLOAT);

   switch (mode) {
   case MISC_REGISTERS_GPIO_OUTPUT_LOW:
      QFLE3_DBG(QFLE3_DBG_LINK,
		"Set GPIO %d (shift %d) -> output low\n",
		gpio_num, gpio_shift);
      /* clear FLOAT and set CLR */
      gpio_reg &= ~(gpio_mask << MISC_REGISTERS_GPIO_FLOAT_POS);
      gpio_reg |=  (gpio_mask << MISC_REGISTERS_GPIO_CLR_POS);
      break;

   case MISC_REGISTERS_GPIO_OUTPUT_HIGH:
      QFLE3_DBG(QFLE3_DBG_LINK,
		"Set GPIO %d (shift %d) -> output high\n",
		gpio_num, gpio_shift);
      /* clear FLOAT and set SET */
      gpio_reg &= ~(gpio_mask << MISC_REGISTERS_GPIO_FLOAT_POS);
      gpio_reg |=  (gpio_mask << MISC_REGISTERS_GPIO_SET_POS);
      break;

   case MISC_REGISTERS_GPIO_INPUT_HI_Z:
      QFLE3_DBG(QFLE3_DBG_LINK,
		"Set GPIO %d (shift %d) -> input\n",
		gpio_num, gpio_shift);
      /* set FLOAT */
      gpio_reg |= (gpio_mask << MISC_REGISTERS_GPIO_FLOAT_POS);
      break;

   default:
      break;
   }

   REG_WR(adapter, MISC_REG_GPIO, gpio_reg);
   qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_GPIO);

   return 0;
}

vmk_uint8
elink_cb_gpio_mult_write(struct qfle3_adapter * adapter, vmk_uint8 pins, vmk_uint8 mode)
{
   vmk_uint32 gpio_reg = 0;
   int rc = 0;

   /* Any port swapping should be handled by caller. */

   qfle3_acquire_hw_lock(adapter, HW_LOCK_RESOURCE_GPIO);
   /* read GPIO and mask except the float bits */
   gpio_reg = REG_RD(adapter, MISC_REG_GPIO);
   gpio_reg &= ~(pins << MISC_REGISTERS_GPIO_FLOAT_POS);
   gpio_reg &= ~(pins << MISC_REGISTERS_GPIO_CLR_POS);
   gpio_reg &= ~(pins << MISC_REGISTERS_GPIO_SET_POS);

   switch (mode) {
   case MISC_REGISTERS_GPIO_OUTPUT_LOW:
      QFLE3_DBG(QFLE3_DBG_LINK, "Set GPIO 0x%x -> output low\n", pins);
      /* set CLR */
      gpio_reg |= (pins << MISC_REGISTERS_GPIO_CLR_POS);
      break;

   case MISC_REGISTERS_GPIO_OUTPUT_HIGH:
      QFLE3_DBG(QFLE3_DBG_LINK, "Set GPIO 0x%x -> output high\n", pins);
      /* set SET */
      gpio_reg |= (pins << MISC_REGISTERS_GPIO_SET_POS);
      break;

   case MISC_REGISTERS_GPIO_INPUT_HI_Z:
      QFLE3_DBG(QFLE3_DBG_LINK, "Set GPIO 0x%x -> input\n", pins);
      /* set FLOAT */
      gpio_reg |= (pins << MISC_REGISTERS_GPIO_FLOAT_POS);
      break;

   default:
      QFLE3_ERR("Invalid GPIO mode assignment %d\n", mode);
      rc = VMK_BAD_PARAM;
      break;
   }

   if (rc == 0)
      REG_WR(adapter, MISC_REG_GPIO, gpio_reg);

   qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_GPIO);
   return rc;
}


vmk_uint32
elink_cb_gpio_read(struct qfle3_adapter * adapter, vmk_uint16 gpio_num, vmk_uint8 port)
{

   /* The GPIO should be swapped if swap register is set and active */
   int gpio_port = (REG_RD(adapter, NIG_REG_PORT_SWAP) &&
		    REG_RD(adapter, NIG_REG_STRAP_OVERRIDE)) ^ port;
   int gpio_shift = gpio_num +
      (gpio_port ? MISC_REGISTERS_GPIO_PORT_SHIFT : 0);
   vmk_uint32 gpio_mask = (1 << gpio_shift);
   vmk_uint32 gpio_reg;
   int value;

   if (gpio_num > MISC_REGISTERS_GPIO_3) {
      QFLE3_ERR("Invalid GPIO %d\n", gpio_num);
      return -1;
   }

   /* read GPIO value */
   gpio_reg = REG_RD(adapter, MISC_REG_GPIO);

   /* get the requested pin value */
   if ((gpio_reg & gpio_mask) == gpio_mask)
      value = 1;
   else
      value = 0;

   return value;
}

vmk_uint8
elink_cb_gpio_int_write(struct qfle3_adapter * adapter,
			vmk_uint16 gpio_num, vmk_uint8 mode, vmk_uint8 port)
{
   /* The GPIO should be swapped if swap register is set and active */
   int gpio_port = (REG_RD(adapter, NIG_REG_PORT_SWAP) &&
		    REG_RD(adapter, NIG_REG_STRAP_OVERRIDE)) ^ port;
   int gpio_shift = gpio_num +
      (gpio_port ? MISC_REGISTERS_GPIO_PORT_SHIFT : 0);
   vmk_uint32 gpio_mask = (1 << gpio_shift);
   vmk_uint32 gpio_reg;

   if (gpio_num > MISC_REGISTERS_GPIO_3) {
      QFLE3_ERR("Invalid GPIO %d\n", gpio_num);
      return -1;
   }

   qfle3_acquire_hw_lock(adapter, HW_LOCK_RESOURCE_GPIO);
   /* read GPIO int */
   gpio_reg = REG_RD(adapter, MISC_REG_GPIO_INT);

   switch (mode) {
   case MISC_REGISTERS_GPIO_INT_OUTPUT_CLR:
      QFLE3_DBG(QFLE3_DBG_LINK,
		"Clear GPIO INT %d (shift %d) -> output low\n",
		gpio_num, gpio_shift);
      /* clear SET and set CLR */
      gpio_reg &= ~(gpio_mask << MISC_REGISTERS_GPIO_INT_SET_POS);
      gpio_reg |=  (gpio_mask << MISC_REGISTERS_GPIO_INT_CLR_POS);
      break;

   case MISC_REGISTERS_GPIO_INT_OUTPUT_SET:
      QFLE3_DBG(QFLE3_DBG_LINK,
		"Set GPIO INT %d (shift %d) -> output high\n",
		gpio_num, gpio_shift);
      /* clear CLR and set SET */
      gpio_reg &= ~(gpio_mask << MISC_REGISTERS_GPIO_INT_CLR_POS);
      gpio_reg |=  (gpio_mask << MISC_REGISTERS_GPIO_INT_SET_POS);
      break;

   default:
      break;
   }

   REG_WR(adapter, MISC_REG_GPIO_INT, gpio_reg);
   qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_GPIO);

   return 0;
}


vmk_uint32
elink_cb_fw_command(struct qfle3_adapter * adapter, vmk_uint32 command, vmk_uint32 param)
{

   return qfle3_mfw_command(adapter, command, param);
}


/* Delay */
void
elink_cb_udelay(struct qfle3_adapter *adapter, vmk_uint32 microsecond)
{
   vmk_DelayUsecs(microsecond);
}

/* This function is called every 1024 bytes downloading of phy firmware.
   Driver can use it to print to screen indication for download progress */
void
elink_cb_download_progress(struct qfle3_adapter *adapter, vmk_uint32 cur,
			   vmk_uint32 total)
{
   QFLE3_DBG(QFLE3_DBG_LINK, "Function not implemented yet\n");
}


void
elink_cb_event_log(struct qfle3_adapter *adapter, const elink_log_id_t log_id, ...)
{
   QFLE3_DBG(QFLE3_DBG_LINK, "Function not implemented yet\n");
}

vmk_uint8
elink_cb_path_id(struct qfle3_adapter *adapter)
{
   QFLE3_DBG(QFLE3_DBG_LINK, "Function not implemented yet\n");
   return 0;
}

void
elink_cb_notify_link_changed(struct qfle3_adapter *adapter)
{
   REG_WR(adapter,
	  MISC_REG_AEU_GENERAL_ATTN_12 + QFLE3_FUNC(adapter)*sizeof(vmk_uint32), 1);
}

int qfle3_get_cur_phy_idx(struct qfle3_adapter *adapter)
{
   vmk_uint32 sel_phy_idx = 0;
   if (adapter->link_params.num_phys <= 1)
      return ELINK_INT_PHY;

   if (adapter->link_vars.link_up) {
      sel_phy_idx = ELINK_EXT_PHY1;
      /* In case link is SERDES, check if the ELINK_EXT_PHY2 is the one */
      if ((adapter->link_vars.link_status & LINK_STATUS_SERDES_LINK) &&
	  (adapter->link_params.phy[ELINK_EXT_PHY2].supported & ELINK_SUPPORTED_FIBRE))
	 sel_phy_idx = ELINK_EXT_PHY2;
   } else {

      switch (elink_phy_selection(&adapter->link_params)) {
      case PORT_HW_CFG_PHY_SELECTION_HARDWARE_DEFAULT:
      case PORT_HW_CFG_PHY_SELECTION_FIRST_PHY:
      case PORT_HW_CFG_PHY_SELECTION_FIRST_PHY_PRIORITY:
	 sel_phy_idx = ELINK_EXT_PHY1;
	 break;
      case PORT_HW_CFG_PHY_SELECTION_SECOND_PHY:
      case PORT_HW_CFG_PHY_SELECTION_SECOND_PHY_PRIORITY:
	 sel_phy_idx = ELINK_EXT_PHY2;
	 break;
      }
   }

   return sel_phy_idx;
}

int qfle3_get_link_cfg_idx(struct qfle3_adapter *adapter)
{
   vmk_uint32 sel_phy_idx = qfle3_get_cur_phy_idx(adapter);
   /*
    * The selected activated PHY is always after swapping (in case PHY
    * swapping is enabled). So when swapping is enabled, we need to reverse
    * the configuration
    */

   if (adapter->link_params.multi_phy_config &
       PORT_HW_CFG_PHY_SWAPPED_ENABLED) {
      if (sel_phy_idx == ELINK_EXT_PHY1)
	 sel_phy_idx = ELINK_EXT_PHY2;
      else if (sel_phy_idx == ELINK_EXT_PHY2)
	 sel_phy_idx = ELINK_EXT_PHY1;
   }
   return ELINK_LINK_CONFIG_IDX(sel_phy_idx);
}

int qfle3_initial_phy_init(struct qfle3_adapter *adapter, int load_mode)
{
   int rc, cfx_idx = qfle3_get_link_cfg_idx(adapter);
   vmk_uint16 req_line_speed = adapter->link_params.req_line_speed[cfx_idx];

   adapter->link_params.req_fc_auto_adv = ELINK_FLOW_CTRL_BOTH;

   qfle3_acquire_phy_lock(adapter);

   if (load_mode == QFLE3_LOAD_DIAG) {
      struct elink_params *lp = &adapter->link_params;
      lp->loopback_mode = ELINK_LOOPBACK_XGXS;
      /* Prefer doing PHY loopback at highest speed, if possible */
      if (lp->req_line_speed[cfx_idx] < ELINK_SPEED_20000) {
	 if (lp->speed_cap_mask[cfx_idx] &
	     PORT_HW_CFG_SPEED_CAPABILITY_D0_20G)
	    lp->req_line_speed[cfx_idx] =
	       ELINK_SPEED_20000;
	 else if (lp->speed_cap_mask[cfx_idx] &
		  PORT_HW_CFG_SPEED_CAPABILITY_D0_10G)
	    lp->req_line_speed[cfx_idx] =
	       ELINK_SPEED_10000;
	 else
	    lp->req_line_speed[cfx_idx] =
	       ELINK_SPEED_1000;
      }
   }

   if (load_mode == QFLE3_LOAD_LOOPBACK_EXT) {
      struct elink_params *lp = &adapter->link_params;
      lp->loopback_mode = ELINK_LOOPBACK_EXT;
   }

   rc = elink_phy_init(&adapter->link_params, &adapter->link_vars);

   qfle3_release_phy_lock(adapter);

#if 0
   qfle3_init_dropless_fc(adapter);
#endif

   qfle3_calc_fc_adv(adapter);

   if (adapter->link_vars.link_up) {
      QFLE3_DBG(QFLE3_DBG_STATS, "Executing qfle3_stats_handle for STATS_EVENT_LINK_UP\n");
      qfle3_stats_handle(adapter, STATS_EVENT_LINK_UP);
      qfle3_acquire_phy_lock(adapter);
      qfle3_report_link(adapter);
      qfle3_release_phy_lock(adapter);
   }

   adapter->link_params.req_line_speed[cfx_idx] = req_line_speed;
   return rc;
}

vmk_uint16 qfle3_get_mf_speed(struct qfle3_adapter *adapter)
{
   vmk_uint16 line_speed = adapter->link_vars.line_speed;
#ifdef QFLE3_NPAR
   if (IS_MF(adapter)) {
      vmk_uint16 maxCfg = qfle3_extract_max_cfg(adapter,
						adapter->mf_config[QFLE3_VN(adapter)]);

      /* Calculate the current MAX line speed limit for the MF
       * devices
       */
      if (IS_MF_PERCENT_BW(adapter))
	 line_speed = (line_speed * maxCfg) / 100;
      else {
	 vmk_uint16 vn_max_rate = maxCfg * 100;

	 if (vn_max_rate < line_speed)
	    line_speed = vn_max_rate;
      }
   }
#endif

   return line_speed;
}

static void qfle3_fill_report_data(struct qfle3_adapter *adapter,
				   struct qfle3_link_report_data *data)
{
   vmk_Memset(data, 0, sizeof(*data));

   /* Fill the report data: effective line speed */
   data->line_speed = qfle3_get_mf_speed(adapter);

   /* Link is down */
   if (!adapter->link_vars.link_up || (adapter->flags & MF_FUNC_DIS))
      ECORE_SET_BIT(QFLE3_LINK_REPORT_LINK_DOWN,
		    &data->link_report_flags);

#if 0
   if (!QFLE3_NUM_ETH_QUEUES(adapter))
      ECORE_SET_BIT(QFLE3_LINK_REPORT_LINK_DOWN,
		    &data->link_report_flags);
#endif

   /* Full DUPLEX */
   if (adapter->link_vars.duplex == DUPLEX_FULL)
      ECORE_SET_BIT(QFLE3_LINK_REPORT_FD,
		    &data->link_report_flags);

   /* Rx Flow Control is ON */
   if (adapter->link_vars.flow_ctrl & ELINK_FLOW_CTRL_RX)
      ECORE_SET_BIT(QFLE3_LINK_REPORT_RX_FC_ON,
		    &data->link_report_flags);

   /* Tx Flow Control is ON */
   if (adapter->link_vars.flow_ctrl & ELINK_FLOW_CTRL_TX)
      ECORE_SET_BIT(QFLE3_LINK_REPORT_TX_FC_ON,
		    &data->link_report_flags);
}

/* Calculates the sum of vn_min_rates.
   It's needed for further normalizing of the min_rates.
   Returns:
     sum of vn_min_rates.
       or
     0 - if all the min_rates are 0.
     In the later case fairness algorithm should be deactivated.
     If not all min_rates are zero then those that are zeroes will be set to 1.
 */
static void qfle3_calc_vn_min(struct qfle3_adapter *adapter,
                      struct cmng_init_input *input)
{
    int all_zero = 1;
    int vn;

    for (vn = VN_0; vn < QFLE3_MAX_VN_NUM(adapter); vn++) {
        u32 vn_cfg = adapter->mf_config[vn];
        u32 vn_min_rate = ((vn_cfg & FUNC_MF_CFG_MIN_BW_MASK) >>
                   FUNC_MF_CFG_MIN_BW_SHIFT) * 100;

        /* Skip hidden vns */
        if (vn_cfg & FUNC_MF_CFG_FUNC_HIDE)
            vn_min_rate = 0;
        /* If min rate is zero - set it to 1 */
        else if (!vn_min_rate)
            vn_min_rate = DEF_MIN_RATE;
        else
            all_zero = 0;

        input->vnic_min_rate[vn] = vn_min_rate;
    }

    /* if ETS or all min rates are zeros - disable fairness */
#if 0
    if (QFLE3_IS_ETS_ENABLED(adapter)) {
        input->flags.cmng_enables &=
                    ~CMNG_FLAGS_PER_PORT_FAIRNESS_VN;
        QFLE3_INFO("Fairness will be disabled due to ETS\n");}
#endif
    if (all_zero) {
        input->flags.cmng_enables &=
                    ~CMNG_FLAGS_PER_PORT_FAIRNESS_VN;
        QFLE3_DBG(QFLE3_DBG_DCB|QFLE3_DBG_LINK, "All MIN values are zeroes fairness will be disabled\n");
    } else
        input->flags.cmng_enables |=
                    CMNG_FLAGS_PER_PORT_FAIRNESS_VN;
}



static void qfle3_calc_vn_max(struct qfle3_adapter *adapter, int vn,
                    struct cmng_init_input *input)
{
    u16 vn_max_rate;
    u32 vn_cfg = adapter->mf_config[vn];

    if (vn_cfg & FUNC_MF_CFG_FUNC_HIDE)
        vn_max_rate = 0;
    else {
        u32 maxCfg = qfle3_extract_max_cfg(adapter, vn_cfg);

        if (IS_MF_PERCENT_BW(adapter)) {
            /* maxCfg in percents of linkspeed */
            vn_max_rate = (adapter->link_vars.line_speed * maxCfg) / 100;
        } else /* SD modes */
            /* maxCfg is absolute in 100Mb units */
            vn_max_rate = maxCfg * 100;
    }

    QFLE3_DBG(QFLE3_DBG_DCB,"vn %d: vn_max_rate %d\n", vn, vn_max_rate);

    input->vnic_max_rate[vn] = vn_max_rate;
}

static int qfle3_get_cmng_fns_mode(struct qfle3_adapter *adapter)
{
    if (CHIP_REV_IS_SLOW(adapter))
        return CMNG_FNS_NONE;
    if (IS_MF(adapter))
        return CMNG_FNS_MINMAX;

    return CMNG_FNS_NONE;
}

static void qfle3_cmng_fns_init(struct qfle3_adapter *adapter, vmk_uint8 read_cfg, vmk_uint8 cmng_type)
{
    struct cmng_init_input input;
    extern u32 fairness_threshold;
    
    vmk_Memset(&input, 0, sizeof(struct cmng_init_input));
    input.port_rate = adapter->link_vars.line_speed;
    
    if (fairness_threshold) {
        QFLE3_INFO("Configuring Fairness Thresholds\n");
        input.size_thr = 3200;
        input.fairness_thr = 80000;
    } else {
        input.size_thr = 0;
        input.fairness_thr = 0;
    }
    
    if (cmng_type == CMNG_FNS_MINMAX && input.port_rate) {
        int vn;

        /* read mf conf from shmem */
        if (read_cfg)
            qfle3_read_mf_cfg(adapter);

        /* vn_weight_sum and enable fairness if not 0 */
        qfle3_calc_vn_min(adapter, &input);

        /* calculate and set min-max rate for each vn */
        if (adapter->port.pmf)
            for (vn = VN_0; vn < QFLE3_MAX_VN_NUM(adapter); vn++)
                qfle3_calc_vn_max(adapter, vn, &input);

        /* always enable rate shaping and fairness */
        input.flags.cmng_enables |=
                    CMNG_FLAGS_PER_PORT_RATE_SHAPING_VN;


        ecore_init_cmng(&input, &adapter->cmng);
        return;
    }

    /* rate shaping and fairness are disabled */
    QFLE3_DBG(QFLE3_DBG_DCB|QFLE3_DBG_LINK, "rate shaping and fairness are disabled\n");
}


static inline void
__storm_memset_struct(struct qfle3_adapter *adapter,
                      vmk_uint32 addr, vmk_ByteCount size, vmk_uint32 * data)
{
   int i;
   for (i = 0; i < size / 4; i++)
      REG_WR(adapter, addr + (i * 4), data[i]);
}


static void storm_memset_cmng(struct qfle3_adapter *adapter,
                   struct cmng_init *cmng,
                   u8 port)
{
    int vn;
    vmk_ByteCount size = sizeof(struct cmng_struct_per_port);

    u32 addr = BAR_XSTRORM_INTMEM +
            XSTORM_CMNG_PER_PORT_VARS_OFFSET(port);

    __storm_memset_struct(adapter, addr, size, (u32 *)&cmng->port);

    for (vn = VN_0; vn < QFLE3_MAX_VN_NUM(adapter); vn++) {
        int func = func_by_vn(adapter, vn);

        addr = BAR_XSTRORM_INTMEM +
               XSTORM_RATE_SHAPING_PER_VN_VARS_OFFSET(func);
        size = sizeof(struct rate_shaping_vars_per_vn);
        __storm_memset_struct(adapter, addr, size,
                      (u32 *)&cmng->vnic.vnic_max_rate[vn]);

        addr = BAR_XSTRORM_INTMEM +
               XSTORM_FAIRNESS_PER_VN_VARS_OFFSET(func);
        size = sizeof(struct fairness_vars_per_vn);
        __storm_memset_struct(adapter, addr, size,
                      (u32 *)&cmng->vnic.vnic_min_rate[vn]);
    }
}

/* init cmng mode in HW according to local configuration */
void qfle3_set_local_cmng(struct qfle3_adapter *adapter)
{
    int cmng_fns = qfle3_get_cmng_fns_mode(adapter);

    if (cmng_fns != CMNG_FNS_NONE) {
        qfle3_cmng_fns_init(adapter, VMK_FALSE, cmng_fns);
        storm_memset_cmng(adapter, &adapter->cmng, QFLE3_PORT(adapter));
    } else {
        /* rate shaping and fairness are disabled */
        QFLE3_DBG(QFLE3_DBG_DCB|QFLE3_DBG_LINK, "single function mode without fairness\n");
    }
}

#ifdef QFLE3_SRIOV

static void
qfle3_stats_update_func(vmk_AddrCookie data)
{
   qfle3_adapter *adapter = data.ptr;

   vmk_CPUMemFenceReadWrite();
   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_LOADED) && !vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)) {

      if (adapter->recovery_state == QFLE3_RECOVERY_DONE) {
         QFLE3_DBG(QFLE3_DBG_STATS, "Collecting statistics on function 0x%x\n", *(vmk_uint32*)&adapter->pdev_addr);
         qfle3_stats_handle(adapter, STATS_EVENT_UPDATE);
      }

      qfle3_schedule_helper(adapter,
         adapter->stats_collection_helper, qfle3_stats_update_func, 2000);
   }
}

void qfle3_read_mf_cfg(struct qfle3_adapter *adapter)
{
   int vn, n = (CHIP_MODE_IS_4_PORT(adapter) ? 2 : 1);

   if (QFLE3_NOMCP(adapter))
      return; /* what should be the default value in this case */

   /* For 2 port configuration the absolute function number formula
    * is:
    *      abs_func = 2 * vn + BP_PORT + BP_PATH
    *
    *      and there are 4 functions per port
    *
    * For 4 port configuration it is
    *      abs_func = 4 * vn + 2 * BP_PORT + BP_PATH
    *
    *      and there are 2 functions per port
    */
   for (vn = 0; vn < QFLE3_MAX_VN_NUM(adapter); vn++) {
      int /*abs*/func = n * (2 * vn + QFLE3_PORT(adapter)) + QFLE3_PATH(adapter);

      if (func >= E1H_FUNC_MAX)
         break;

      adapter->mf_config[vn] =
         MF_CFG_RD(adapter, func_mf_config[func].config);
   }
   if (adapter->mf_config[QFLE3_VN(adapter)] & FUNC_MF_CFG_FUNC_DISABLED) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: mf_cfg function disabled\n");
      adapter->flags |= MF_FUNC_DIS;
   } else {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: mf_cfg function enabled\n");
      adapter->flags &= ~MF_FUNC_DIS;
   }
}


static void qfle3_iov_link_update(struct qfle3_adapter *adapter)
{
   int vfid;

   if(!adapter->vfdb){
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Adapter vfdb not allocated yet\n");
      return;
   }

   for(vfid=0; vfid < adapter->vfdb->sriov.nr_virtfn; vfid++)
      qfle3_iov_link_update_vf(adapter, vfid);
}

#endif  //QFLE3_SRIOV

void
qfle3_report_link(qfle3_adapter *adapter)
{
   struct qfle3_link_report_data cur_data;
   vmk_UplinkSharedData *shared_data = &adapter->uplinkSharedData;
   vmk_LinkStatus *linkInfo = &shared_data->link;
   VMK_ReturnStatus status = VMK_OK;
   vmk_int32 i, qid;
   struct qfle3_fastpath *fp;
   vmk_UplinkSharedQueueData *sqd = NULL;

   
   //qfle3_acquire_phy_lock(adapter);

#ifdef QFLE3_SRIOV
   /* reread mf_cfg */
   qfle3_read_mf_cfg(adapter);
#endif //QFLE3_SRIOV

   /* Read the current link report info */
   qfle3_fill_report_data(adapter, &cur_data);

   /* Don't report link down or exactly the same link status twice */
   if (!vmk_Memcmp(&cur_data, &adapter->last_reported_link, sizeof(cur_data)) ||
       (ECORE_TEST_BIT(QFLE3_LINK_REPORT_LINK_DOWN,
		       &adapter->last_reported_link.link_report_flags) &&
	ECORE_TEST_BIT(QFLE3_LINK_REPORT_LINK_DOWN,
		       &cur_data.link_report_flags))) {
      //qfle3_release_phy_lock(adapter);
      QFLE3_DBG(QFLE3_DBG_LINK, "Not reporting same link status (%s) again",
		ECORE_TEST_BIT(QFLE3_LINK_REPORT_LINK_DOWN,
			       &cur_data.link_report_flags) ? "Down" : "Up");
      return;
   }

   adapter->link_cnt++;


   if (ECORE_TEST_BIT(QFLE3_LINK_REPORT_LINK_DOWN,
		      &cur_data.link_report_flags)) {
      if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_QUIESCED)) {
      	 goto out;
      }

      adapter->linkUp = VMK_FALSE;
      QFLE3_DBG(QFLE3_DBG_LINK, "Pausing TX Queues On Link down\n");
      FOREACH_TX_VMK_QUEUE(i,qid,fp)
         sqd = QFLE3_TX_QUEUE_SHARED_DATA(adapter);
         sqd += i;
         if ((fp->fp_state == QFLE3_SM_Q_STARTED) && 
            fp->soft_state.q_started_by_vmk &&
            (sqd->state == VMK_UPLINK_QUEUE_STATE_STARTED))
            qfle3_tq_pause (adapter, i);
      }

      QFLE3_SHARED_DATA_WRITE_BEGIN(adapter);
      linkInfo->speed = cur_data.line_speed;
      linkInfo->state =  VMK_LINK_STATE_DOWN;
      linkInfo->duplex = VMK_LINK_DUPLEX_HALF;
      QFLE3_SHARED_DATA_WRITE_END(adapter);

      if(adapter->uplink != NULL) {//QFLE3_NPAR
         status = vmk_UplinkUpdateLinkState(adapter->uplink, linkInfo);
         if (status != VMK_OK) {
   	 QFLE3_ERR( "vmk_UplinkUpdateLinkState failed: %#x, uplink: %p",
   		    status, adapter->uplink);
         }
      }
#ifdef QFLE3_CNIC
      if (CNIC_LOADED(adapter)) {
         adapter->cnicOps->cnicLinkUpdate(adapter->cnic_data, 2, 0);
         
         QFLE3_DBG(QFLE3_DBG_CNIC| QFLE3_DBG_LINK, "CNIC Link state reported\n");
      }
#endif
      QFLE3_INFO("NIC Link is Down\n");
   } else {
      const char *duplex;
      const char *flow;

      if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_QUIESCED)) {
      	 goto out;
      }

      //start the queues
      QFLE3_DBG(QFLE3_DBG_QUEUE|QFLE3_DBG_LINK, "Resuming TX Queues On Link up\n");
      FOREACH_TX_VMK_QUEUE(i,qid,fp)
         sqd = QFLE3_TX_QUEUE_SHARED_DATA(adapter);
         sqd += i;
         if ((fp->fp_state == QFLE3_SM_Q_STARTED) &&
            fp->soft_state.q_started_by_vmk &&
            (sqd->state == VMK_UPLINK_QUEUE_STATE_STOPPED))
            qfle3_tq_resume (adapter, i);
      }

      if (ECORE_TEST_BIT(QFLE3_LINK_REPORT_FD,
			 &cur_data.link_report_flags))
      	 duplex = "full";
      else
      	 duplex = "half";

      /* Handle the FC at the end so that only these flags would be
       * possibly set. This way we may easily check if there is no FC
       * enabled.
       */
      if (cur_data.link_report_flags) {
      	 if (ECORE_TEST_BIT(QFLE3_LINK_REPORT_RX_FC_ON,
      			    &cur_data.link_report_flags)) {
      	    if (ECORE_TEST_BIT(QFLE3_LINK_REPORT_TX_FC_ON,
      			       &cur_data.link_report_flags))
      	       flow = "ON - receive & transmit";
      	    else
      	       flow = "ON - receive";
      	 } else {
      	    flow = "ON - transmit";
      	 }
      } else {
      	 flow = "none";
      }

      vmk_WorldSleep(VMK_USEC_PER_SEC / 10);

      QFLE3_SHARED_DATA_WRITE_BEGIN(adapter);
      linkInfo->speed = cur_data.line_speed;
      linkInfo->state =  VMK_LINK_STATE_UP;
      linkInfo->duplex = (!vmk_Strcmp(duplex, "full") ? 
                           VMK_LINK_DUPLEX_FULL: VMK_LINK_DUPLEX_HALF);
      QFLE3_SHARED_DATA_WRITE_END(adapter);
      if(adapter->uplink != NULL) {//QFLE3_NPAR
         status = vmk_UplinkUpdateLinkState(adapter->uplink, linkInfo);
         if (status != VMK_OK) {
            QFLE3_ERR( "vmk_UplinkUpdateLinkState failed: %#x, uplink: %p",
               status, adapter->uplink);
	    goto out;
         }
      }
      adapter->linkUp = VMK_TRUE;

#ifdef QFLE3_CNIC
      if (CNIC_LOADED(adapter)) {
         adapter->cnicOps->cnicLinkUpdate(adapter->cnic_data, 1, cur_data.line_speed);
         QFLE3_DBG(QFLE3_DBG_CNIC| QFLE3_DBG_LINK, "CNIC Link state reported\n");
      }
#endif
      QFLE3_INFO("NIC Link is Up, %d Mb/s %s duplex, Flow control: %s\n",
		 cur_data.line_speed, duplex, flow);
   }
 
   if(adapter->uplink != NULL)
	   vmk_Memcpy(&adapter->last_reported_link, &cur_data, sizeof(cur_data));
#ifdef QFLE3_SRIOV
   qfle3_iov_link_update(adapter);
#endif //QFLE3_SRIOV

  out:
   //qfle3_release_phy_lock(adapter);
   return;
}

static vmk_Bool
qfle3_is_contextless_ramrod(int cmd, int cmd_type)
{
   if ((cmd_type == NONE_CONNECTION_TYPE) ||
       (cmd == RAMROD_CMD_ID_ETH_FORWARD_SETUP) ||
       (cmd == RAMROD_CMD_ID_ETH_CLASSIFICATION_RULES) ||
       (cmd == RAMROD_CMD_ID_ETH_FILTER_RULES) ||
       (cmd == RAMROD_CMD_ID_ETH_MULTICAST_RULES) ||
       (cmd == RAMROD_CMD_ID_ETH_SET_MAC) || (cmd == RAMROD_CMD_ID_ETH_RSS_UPDATE))
      return VMK_TRUE;
   else
      return VMK_FALSE;
}

struct eth_spe *
qfle3_sp_get_next(struct qfle3_adapter *adapter)
{
   struct eth_spe *next_spe = adapter->spq_prod_bd;

   if (adapter->spq_prod_bd == adapter->spq_last_bd) {
      adapter->spq_prod_bd = adapter->spq;
      adapter->spq_prod_idx = 0;
      QFLE3_DBG(QFLE3_DBG_SP,"end of spq\n");
   } else {
      adapter->spq_prod_bd++;
      adapter->spq_prod_idx++;
   }
   return next_spe;
}

void
qfle3_sp_prod_update(struct qfle3_adapter *adapter)
{
   int func = QFLE3_FUNC(adapter);

   /*
    * Make sure that BD data is updated before writing the producer:
    * BD data is written to the memory, the producer is read from the
    * memory, thus we need a full memory barrier to ensure the ordering.
    */
   vmk_CPUMemFenceReadWrite();

   REG_WR16(adapter, BAR_XSTRORM_INTMEM + XSTORM_SPQ_PROD_OFFSET(func),
	    adapter->spq_prod_idx);
   vmk_CPUMemFenceReadWrite();
}

vmk_int32
ecore_sp_post(struct qfle3_adapter *adapter,
	      vmk_int32 command, vmk_int32 cid, vmk_IOA mapping, vmk_int32 cmd_type)
{
   vmk_uint16 type;
   struct eth_spe *spe;
   vmk_uint32 data_hi = U64_HI(mapping);
   vmk_uint32 data_lo = U64_LO(mapping);
   vmk_Bool common = qfle3_is_contextless_ramrod(command, cmd_type);

   vmk_SpinlockLock(adapter->spq_lock);

   if (common) {
      if (!vmk_AtomicRead64(&adapter->eq_spq_left)) {
      QFLE3_ERR("BUG! EQ ring full!\n");
      vmk_SpinlockUnlock(adapter->spq_lock);
      return VMK_BUSY;
      }
   } else if (!vmk_AtomicRead64(&adapter->cq_spq_left)) {
      QFLE3_ERR("BUG! SPQ ring full!\n");
      vmk_SpinlockUnlock(adapter->spq_lock);
      return VMK_BUSY;
   }

   spe = qfle3_sp_get_next(adapter);

   /*
    * CID needs port number to be encoded int it
    */
   spe->hdr.conn_and_cmd_data =
      htole32((command << SPE_HDR_CMD_ID_SHIFT) | HW_CID(adapter, cid));

   /*
    * In some cases, type may already contain the func-id
    * * mainly in SRIOV related use cases, so we add it here only
    * * if it's not already set.
    */
   if (!(cmd_type & SPE_HDR_FUNCTION_ID)) {
      type = (cmd_type << SPE_HDR_CONN_TYPE_SHIFT) & SPE_HDR_CONN_TYPE;
      type |= ((QFLE3_FUNC(adapter) << SPE_HDR_FUNCTION_ID_SHIFT) & SPE_HDR_FUNCTION_ID);
   } else {
      type = cmd_type;
   }

   spe->hdr.type = htole16(type);

   spe->data.update_data_addr.hi = htole32(data_hi);
   spe->data.update_data_addr.lo = htole32(data_lo);

   /*
    * It's ok if the actual decrement is issued towards the memory
    * somewhere between the spin_lock and spin_unlock. Thus no
    * more explicit memory barrier is needed.
    */
   if (common)
      vmk_AtomicDec64(&adapter->eq_spq_left);
   else
      vmk_AtomicDec64(&adapter->cq_spq_left);

   QFLE3_DBG(QFLE3_DBG_SP,
	     "SPQE[%x] (%x:%x)  (cmd, common?) (%d,%d)  hw_cid %x  data (%x:%x) type(0x%x) left (CQ, EQ) (%lx,%lx)\n",
	     adapter->spq_prod_idx, (vmk_uint32) U64_HI(adapter->spq_mapping),
	     (vmk_uint32) (U64_LO(adapter->spq_mapping) + (void *) adapter->spq_prod_bd -
			   (void *) adapter->spq), command, common, HW_CID(adapter, cid),
	     data_hi, data_lo, type, vmk_AtomicRead64(&adapter->cq_spq_left),
	     vmk_AtomicRead64(&adapter->eq_spq_left));

   qfle3_sp_prod_update(adapter);
   vmk_SpinlockUnlock(adapter->spq_lock);

   return 0;
}

void
ecore_storm_memset_struct(struct qfle3_adapter *adapter,
			  vmk_uint32 addr, vmk_ByteCount size, vmk_uint32 * data)
{
   vmk_int32 i;
   for (i = 0; i < size / 4; i++)
      REG_WR(adapter, addr + (i * 4), data[i]);
}

vmk_uint32
calc_crc32(vmk_uint8 * crc32_packet,
	   vmk_uint32 crc32_length, vmk_uint32 crc32_seed, vmk_uint8 complement)
{
   vmk_uint32 byte = 0;
   vmk_uint32 bit = 0;
   vmk_uint8 msb = 0;
   vmk_uint32 temp = 0;
   vmk_uint32 shft = 0;
   vmk_uint8 current_byte = 0;
   vmk_uint32 crc32_result = crc32_seed;
   const vmk_uint32 CRC32_POLY = 0x1edc6f41;

   if ((crc32_packet == NULL) || (crc32_length == 0) || ((crc32_length % 8) != 0)) {
      return (crc32_result);
   }

   for (byte = 0; byte < crc32_length; byte = byte + 1) {
      current_byte = crc32_packet[byte];
      for (bit = 0; bit < 8; bit = bit + 1) {
	 /*
	  * msb = crc32_result[31];
	  */
	 msb = (vmk_uint8) (crc32_result >> 31);

	 crc32_result = crc32_result << 1;

	 /*
	  * it (msb != current_byte[bit])
	  */
	 if (msb != (0x1 & (current_byte >> bit))) {
	    crc32_result = crc32_result ^ CRC32_POLY;
	    /*
	     * crc32_result[0] = 1
	     */
	    crc32_result |= 1;
	 }
      }
   }

   temp = crc32_result;
   shft = sizeof(crc32_result) * 8 - 1;

   for (crc32_result >>= 1; crc32_result; crc32_result >>= 1) {
      temp <<= 1;
      temp |= crc32_result & 1;
      shft--;
   }

   /*
    * temp[31-bit] = crc32_result[bit]
    */
   temp <<= shft;

   /*
    * Swap
    */
   /*
    * crc32_result = {temp[7:0], temp[15:8], temp[23:16], temp[31:24]}
    */
   {
      vmk_uint32 t0, t1, t2, t3;
      t0 = (0x000000ff & (temp >> 24));
      t1 = (0x0000ff00 & (temp >> 8));
      t2 = (0x00ff0000 & (temp << 8));
      t3 = (0xff000000 & (temp << 24));
      crc32_result = t0 | t1 | t2 | t3;
   }

   /*
    * Complement
    */
   if (complement) {
      crc32_result = ~crc32_result;
   }

   return (crc32_result);
}

static void
storm_memset_hc_timeout(struct qfle3_adapter *adapter, vmk_uint8 port,
			vmk_uint8 fw_sb_id, vmk_uint8 sb_index, vmk_uint8 ticks)
{
   vmk_uint32 addr = BAR_CSTRORM_INTMEM +
      CSTORM_STATUS_BLOCK_DATA_TIMEOUT_OFFSET(fw_sb_id, sb_index);
   REG_WR8(adapter, addr, ticks);
}

static void
storm_memset_hc_disable(struct qfle3_adapter *adapter, vmk_uint8 port,
			vmk_uint16 fw_sb_id, vmk_uint8 sb_index, vmk_uint8 disable)
{
   vmk_uint32 enable_flag = disable ? 0 : (1 << HC_INDEX_DATA_HC_ENABLED_SHIFT);
   vmk_uint32 addr = BAR_CSTRORM_INTMEM +
      CSTORM_STATUS_BLOCK_DATA_FLAGS_OFFSET(fw_sb_id, sb_index);
   vmk_uint8 flags = REG_RD8(adapter, addr);
   /*
    * clear and set
    */
   flags &= ~HC_INDEX_DATA_HC_ENABLED;
   flags |= enable_flag;
   REG_WR8(adapter, addr, flags);
}

void
qfle3_update_coalesce_sb_index(struct qfle3_adapter *adapter, vmk_uint8 fw_sb_id,
			       vmk_uint8 sb_index, vmk_uint8 disable, vmk_uint16 usec)
{
   int port = QFLE3_PORT(adapter);
   vmk_uint8 ticks = usec / QFLE3_BTR;

   storm_memset_hc_timeout(adapter, port, fw_sb_id, sb_index, ticks);

   disable = disable ? 1 : (usec ? 0 : 1);
   storm_memset_hc_disable(adapter, port, fw_sb_id, sb_index, disable);
}

static void qfle3_update_coalesce_sb(struct qfle3_adapter *adapter, vmk_uint8 fw_sb_id,
				     vmk_uint16 tx_usec, vmk_uint16 rx_usec)
{
   qfle3_update_coalesce_sb_index(adapter, fw_sb_id, HC_INDEX_ETH_RX_CQ_CONS,
				  VMK_FALSE, rx_usec);

   qfle3_update_coalesce_sb_index(adapter, fw_sb_id,
				  HC_INDEX_ETH_TX_CQ_CONS_COS0, VMK_FALSE,
				  tx_usec);
   qfle3_update_coalesce_sb_index(adapter, fw_sb_id,
				  HC_INDEX_ETH_TX_CQ_CONS_COS1, VMK_FALSE,
				  tx_usec);
   qfle3_update_coalesce_sb_index(adapter, fw_sb_id,
				  HC_INDEX_ETH_TX_CQ_CONS_COS2, VMK_FALSE,
				  tx_usec);
}

void
qfle3_set_ctx_validation(struct qfle3_adapter *adapter,
			 struct eth_context *cxt, vmk_uint32 cid)
{
   if (!cxt) {
      QFLE3_ERR("bad context pointer %p\n", cxt);
      return;
   }

   /*
    * ustorm cxt validation
    */
   cxt->ustorm_ag_context.cdu_usage =
      CDU_RSRVD_VALUE_TYPE_A(HW_CID(adapter, cid),
			     CDU_REGION_NUMBER_UCM_AG, ETH_CONNECTION_TYPE);
   /*
    * xcontext validation
    */
   cxt->xstorm_ag_context.cdu_reserved =
      CDU_RSRVD_VALUE_TYPE_A(HW_CID(adapter, cid),
			     CDU_REGION_NUMBER_XCM_AG, ETH_CONNECTION_TYPE);
}

void
qfle3_reg_wr_ind(struct qfle3_adapter *adapter, vmk_uint32 addr, vmk_uint32 val)
{
   vmk_PCIWriteConfig(vmk_ModuleCurrentID, adapter->pdev,
		      VMK_PCI_CONFIG_ACCESS_32, PCICFG_GRC_ADDRESS, addr);
   vmk_PCIWriteConfig(vmk_ModuleCurrentID, adapter->pdev,
		      VMK_PCI_CONFIG_ACCESS_32, PCICFG_GRC_DATA, val);
   vmk_PCIWriteConfig(vmk_ModuleCurrentID, adapter->pdev,
		      VMK_PCI_CONFIG_ACCESS_32, PCICFG_GRC_ADDRESS, 0);
}

static vmk_uint32
qfle3_reg_rd_ind(struct qfle3_adapter *adapter, vmk_uint32 addr)
{
   vmk_uint32 val;

   vmk_PCIWriteConfig(vmk_ModuleCurrentID, adapter->pdev,
		      VMK_PCI_CONFIG_ACCESS_32, PCICFG_GRC_ADDRESS, addr);
   vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev,
		     VMK_PCI_CONFIG_ACCESS_32, PCICFG_GRC_DATA, &val);
   vmk_PCIWriteConfig(vmk_ModuleCurrentID, adapter->pdev,
		      VMK_PCI_CONFIG_ACCESS_32, PCICFG_GRC_ADDRESS, 0);

   return val;
}

void
qfle3_post_dmae(struct qfle3_adapter *adapter, struct dmae_command *dmae, vmk_int32 idx)
{
   vmk_uint32 cmd_offset;
   vmk_int32 i;

   cmd_offset = (DMAE_REG_CMD_MEM + sizeof(struct dmae_command) * idx);
   for (i = 0; i < (sizeof(struct dmae_command) / 4); i++) {
      REG_WR(adapter, cmd_offset + i * 4, *(((vmk_uint32 *) dmae) + i));
   }
   REG_WR(adapter, dmae_reg_go_c[idx], 1);
}

vmk_int32
qfle3_issue_dmae_with_comp(struct qfle3_adapter *adapter,
			   struct dmae_command *dmae, vmk_uint32 * comp)
{
   vmk_int32 cnt = 4000;
   vmk_int32 rc = 0;

   vmk_SpinlockLock(adapter->dmae_lock);

   /*
    * reset completion
    */
   *comp = 0;

   /*
    * post the command on the channel used for initializations
    */
   qfle3_post_dmae(adapter, dmae, INIT_DMAE_C(adapter));

   /*
    * wait for completion
    */
   vmk_DelayUsecs(5);
   while ((*comp & ~DMAE_PCI_ERR_FLAG) != DMAE_COMP_VAL) {

      if (!cnt) {
	 QFLE3_ERR("DMAE timeout: comp %x!\n", *comp);
	 rc = DMAE_TIMEOUT;
	 goto unlock;
      }
      cnt--;
      vmk_DelayUsecs(50);
   }
   if (*comp & DMAE_PCI_ERR_FLAG) {
      QFLE3_ERR("DMAE PCI Error!\n");
      rc = DMAE_PCI_ERROR;
   }
  unlock:

   vmk_SpinlockUnlock(adapter->dmae_lock);

   return rc;
}

vmk_uint32
qfle3_dmae_opcode_add_comp(vmk_uint32 opcode, vmk_uint8 comp_type)
{
   return (opcode | ((comp_type << DMAE_COMMAND_C_DST_SHIFT) |
		     DMAE_COMMAND_C_TYPE_ENABLE));
}

vmk_uint32
qfle3_dmae_opcode(struct qfle3_adapter * adapter,
		  vmk_uint8 src_type, vmk_uint8 dst_type,
		  vmk_Bool with_comp, vmk_uint8 comp_type)
{
   vmk_uint32 opcode = 0;

   opcode |= ((src_type << DMAE_COMMAND_SRC_SHIFT) |
	      (dst_type << DMAE_COMMAND_DST_SHIFT));

   opcode |= (DMAE_CMD_SRC_RESET | DMAE_CMD_DST_RESET);

   opcode |= (QFLE3_PORT(adapter) ? DMAE_CMD_PORT_1 : DMAE_CMD_PORT_0);
   opcode |= ((QFLE3_VN(adapter) << DMAE_CMD_E1HVN_SHIFT) |
	      (QFLE3_VN(adapter) << DMAE_COMMAND_DST_VN_SHIFT));
   opcode |= (DMAE_COM_SET_ERR << DMAE_COMMAND_ERR_POLICY_SHIFT);

   opcode |= DMAE_CMD_ENDIANITY_DW_SWAP;

   if (with_comp)
      opcode = qfle3_dmae_opcode_add_comp(opcode, comp_type);
   return opcode;
}

void
qfle3_prep_dmae_with_comp(struct qfle3_adapter *adapter,
			  struct dmae_command *dmae,
			  vmk_uint8 src_type, vmk_uint8 dst_type)
{
   vmk_Memset(dmae, 0, sizeof(struct dmae_command));

   /*
    * set the opcode
    */
   dmae->opcode = qfle3_dmae_opcode(adapter,
				    src_type, dst_type, VMK_TRUE, DMAE_COMP_PCI);

   /*
    * fill in the completion parameters
    */
   dmae->comp_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, wb_comp));
   dmae->comp_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, wb_comp));
   dmae->comp_val = DMAE_COMP_VAL;
}

void
qfle3_read_dmae(struct qfle3_adapter *adapter, vmk_uint32 src_addr, vmk_uint32 len32)
{
   int rc;
   struct dmae_command dmae;

   if (!adapter->dmae_ready) {
      vmk_uint32 *data = QFLE3_SP(adapter, wb_data[0]);
      int i;

      if (CHIP_IS_E1(adapter))
	 for (i = 0; i < len32; i++)
	    data[i] = qfle3_reg_rd_ind(adapter, src_addr + i * 4);
      else
	 for (i = 0; i < len32; i++)
	    data[i] = REG_RD(adapter, src_addr + i * 4);

      return;
   }

   /*
    * set opcode and fixed command fields
    */
   qfle3_prep_dmae_with_comp(adapter, &dmae, DMAE_SRC_GRC, DMAE_DST_PCI);

   /*
    * fill in addresses and len
    */
   dmae.src_addr_lo = src_addr >> 2;
   dmae.src_addr_hi = 0;
   dmae.dst_addr_lo = U64_LO(QFLE3_SP_MAPPING(adapter, wb_data));
   dmae.dst_addr_hi = U64_HI(QFLE3_SP_MAPPING(adapter, wb_data));
   dmae.len = len32;

   /*
    * issue the command and wait for completion
    */
   rc = qfle3_issue_dmae_with_comp(adapter, &dmae, QFLE3_SP(adapter, wb_comp));
   if (rc) {
      QFLE3_ERR("DMAE returned failure %d\n", rc);
   }
}

void
qfle3_write_dmae(struct qfle3_adapter *adapter,
		 vmk_IOA dma_addr, vmk_uint32 dst_addr, vmk_uint32 len32)
{
   vmk_int32 rc;
   struct dmae_command dmae;

   if (!adapter->dmae_ready) {
      vmk_uint32 *data = QFLE3_SP(adapter, wb_data[0]);

      if (CHIP_IS_E1(adapter))
	 ecore_init_ind_wr(adapter, dst_addr, data, len32);
      else
	 ecore_init_str_wr(adapter, dst_addr, data, len32);

      return;
   }

   /*
    * set opcode and fixed command fields
    */
   qfle3_prep_dmae_with_comp(adapter, &dmae, DMAE_SRC_PCI, DMAE_DST_GRC);

   /*
    * fill in addresses and len
    */
   dmae.src_addr_lo = U64_LO(dma_addr);
   dmae.src_addr_hi = U64_HI(dma_addr);
   dmae.dst_addr_lo = dst_addr >> 2;
   dmae.dst_addr_hi = 0;
   dmae.len = len32;

   /*
    * issue the command and wait for completion
    */

   rc = qfle3_issue_dmae_with_comp(adapter, &dmae, QFLE3_SP(adapter, wb_comp));
   if (rc) {
      QFLE3_ERR("DMAE returned failure %d\n", rc);
   }
}

void
qfle3_write_dmae_phys_len(struct qfle3_adapter *adapter,
			  vmk_IOA phys_addr, vmk_uint32 addr, vmk_uint32 len)
{
   vmk_int32 dmae_wr_max = DMAE_LEN32_WR_MAX(adapter);
   vmk_int32 offset = 0;

   while (len > dmae_wr_max) {
      qfle3_write_dmae(adapter, phys_addr + offset, addr + offset, dmae_wr_max);
      offset += dmae_wr_max * 4;
      len -= dmae_wr_max;
   }

   qfle3_write_dmae(adapter, phys_addr + offset, addr + offset, len);
}


static void
__storm_memset_dma_mapping(struct qfle3_adapter *adapter,
			   vmk_uint32 addr, vmk_IOA mapping)
{
   REG_WR(adapter, addr, U64_LO(mapping));
   REG_WR(adapter, addr + 4, U64_HI(mapping));
}

static void
storm_memset_spq_addr(struct qfle3_adapter *adapter, vmk_IOA mapping, vmk_uint16 abs_fid)
{
   vmk_uint32 addr = XSEM_REG_FAST_MEMORY + XSTORM_SPQ_PAGE_BASE_OFFSET(abs_fid);

   __storm_memset_dma_mapping(adapter, addr, mapping);
}



static void
storm_memset_vf_to_pf(struct qfle3_adapter *adapter, vmk_uint16 abs_fid,
		      vmk_uint16 pf_id)
{
   REG_WR8(adapter, BAR_XSTRORM_INTMEM + XSTORM_VF_TO_PF_OFFSET(abs_fid), pf_id);
   REG_WR8(adapter, BAR_CSTRORM_INTMEM + CSTORM_VF_TO_PF_OFFSET(abs_fid), pf_id);
   REG_WR8(adapter, BAR_TSTRORM_INTMEM + TSTORM_VF_TO_PF_OFFSET(abs_fid), pf_id);
   REG_WR8(adapter, BAR_USTRORM_INTMEM + USTORM_VF_TO_PF_OFFSET(abs_fid), pf_id);
}

static void
storm_memset_func_en(struct qfle3_adapter *adapter, vmk_uint16 abs_fid, vmk_uint8 enable)
{
   REG_WR8(adapter, BAR_XSTRORM_INTMEM + XSTORM_FUNC_EN_OFFSET(abs_fid), enable);
   REG_WR8(adapter, BAR_CSTRORM_INTMEM + CSTORM_FUNC_EN_OFFSET(abs_fid), enable);
   REG_WR8(adapter, BAR_TSTRORM_INTMEM + TSTORM_FUNC_EN_OFFSET(abs_fid), enable);
   REG_WR8(adapter, BAR_USTRORM_INTMEM + USTORM_FUNC_EN_OFFSET(abs_fid), enable);
}

static void
storm_memset_eq_data(struct qfle3_adapter *adapter,
		     struct event_ring_data *eq_data, vmk_uint16 pfid)
{
   vmk_ByteCount size = sizeof(struct event_ring_data);

   vmk_uint32 addr = BAR_CSTRORM_INTMEM + CSTORM_EVENT_RING_DATA_OFFSET(pfid);

   __storm_memset_struct(adapter, addr, size, (vmk_uint32 *) eq_data);
}

static void
storm_memset_eq_prod(struct qfle3_adapter *adapter, vmk_uint16 eq_prod, vmk_uint16 pfid)
{
   vmk_uint32 addr = BAR_CSTRORM_INTMEM + CSTORM_EVENT_RING_PROD_OFFSET(pfid);
   REG_WR16(adapter, addr, eq_prod);
}

#ifdef QFLE3_SRIOV
static  vmk_PCIVFDeviceOps qfle3_vf_ops = {
        .removeVF = qfle3_vf_unregister,
};
/* General service functions */
static void storm_memset_vf_mbx_ack(struct qfle3_adapter *adapter, vmk_uint16 abs_fid)
{
   vmk_uint32 addr = BAR_CSTRORM_INTMEM + CSTORM_VF_PF_CHANNEL_STATE_OFFSET(abs_fid);
   REG_WR8(adapter, addr, VF_PF_CHANNEL_STATE_READY);
}

static void storm_memset_vf_mbx_valid(struct qfle3_adapter *adapter, vmk_uint16 abs_fid)
{
   vmk_uint32 addr = BAR_CSTRORM_INTMEM + CSTORM_VF_PF_CHANNEL_VALID_OFFSET(abs_fid);
   REG_WR8(adapter, addr, 1);
}

static inline vmk_Bool ether_addr_equal(const vmk_uint8 *addr1, const vmk_uint8 *addr2)
{
   return vmk_Memcmp(addr1, addr2, ETH_ALEN) == 0;
}

/* called by qfle3_init_hw_func, returns the next ilt line */
int qfle3_iov_init_ilt(struct qfle3_adapter *adapter, vmk_uint16 line)
{
   int i;
   struct ecore_ilt *ilt = QFLE3_ILT(adapter);

   if(!adapter->vfdb){
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Adapter vfdb not allocated yet\n");
      return line;
   }

   /*Set Vfs Line limit*/
   for (i = 0; i < QFLE3_VF_CIDS/ILT_PAGE_CIDS; i++) {
      struct hw_dma *hw_cxt = &(adapter->vfdb->context[i]);

      ilt->lines[line+i].page = hw_cxt->addr;
      ilt->lines[line+i].page_mapping = hw_cxt->mapping;
      ilt->lines[line+i].size = hw_cxt->size; /* doesn't matter */
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: ilt->lines[line+i].page is %p ilt->lines[line+i].page_mapping 0x%lx ilt->lines[line+i].size %d\n", 
            ilt->lines[line+i].page, ilt->lines[line+i].page_mapping, ilt->lines[line+i].size);
   }
   return line+i;
}


#endif  //QFLE3_SRIOV

static void
qfle3_update_eq_prod(struct qfle3_adapter *adapter, vmk_uint16 prod)
{
   /*
    * No memory barriers
    */
   storm_memset_eq_prod(adapter, prod, QFLE3_FUNC(adapter));
   vmk_CPUMemFenceReadWrite();
}

void
qfle3_func_init(struct qfle3_adapter *adapter, struct qfle3_func_init_params *p)
{
#if 0
   if (CHIP_IS_E1x(adapter)) {
      struct tstorm_eth_function_common_config tcfg = { 0 };

      storm_memset_func_cfg(adapter, &tcfg, p->func_id);
   }
#endif

   /*
    * Enable the function in the FW
    */
   storm_memset_vf_to_pf(adapter, p->func_id, p->pf_id);
   storm_memset_func_en(adapter, p->func_id, 1);

   /*
    * spq
    */
   if (p->spq_active) {
      storm_memset_spq_addr(adapter, p->spq_map, p->func_id);
      REG_WR(adapter, XSEM_REG_FAST_MEMORY +
	     XSTORM_SPQ_PROD_OFFSET(p->func_id), p->spq_prod);
   }
}

static void
qfle3_pf_init(struct qfle3_adapter *adapter)
{
   struct qfle3_func_init_params func_init = { 0 };
   struct event_ring_data eq_data = { {0} };

   if (!CHIP_IS_E1x(adapter)) {
      /*
       * reset IGU PF statistics: MSIX + ATTN
       */
      /*
       * PF
       */
      REG_WR(adapter, IGU_REG_STATISTIC_NUM_MESSAGE_SENT +
	     QFLE3_IGU_STAS_MSG_VF_CNT * 4 +
	     (CHIP_MODE_IS_4_PORT(adapter) ?
	      QFLE3_FUNC(adapter) : QFLE3_VN(adapter)) * 4, 0);
      /*
       * ATTN
       */
      REG_WR(adapter, IGU_REG_STATISTIC_NUM_MESSAGE_SENT +
	     QFLE3_IGU_STAS_MSG_VF_CNT * 4 +
	     QFLE3_IGU_STAS_MSG_PF_CNT * 4 +
	     (CHIP_MODE_IS_4_PORT(adapter) ?
	      QFLE3_FUNC(adapter) : QFLE3_VN(adapter)) * 4, 0);
   }

   func_init.spq_active = VMK_TRUE;
   func_init.pf_id = QFLE3_FUNC(adapter);
   func_init.func_id = QFLE3_FUNC(adapter);
   func_init.spq_map = adapter->spq_mapping;
   func_init.spq_prod = adapter->spq_prod_idx;

   qfle3_func_init(adapter, &func_init);

   vmk_Memset(&(adapter->cmng), 0, sizeof(struct cmng_struct_per_port));

   /*
    * Congestion management values depend on the link rate
    * There is no active link so initial link rate is set to 10 Gadapters.
    * When the link comes up The congestion management values are
    * re-calculated according to the actual link rate.
    */
   adapter->link_vars.line_speed = ELINK_SPEED_10000;
   qfle3_cmng_fns_init(adapter, VMK_TRUE, qfle3_get_cmng_fns_mode(adapter));

   /*
    * Only the PMF sets the HW
    */
   if (adapter->port.pmf)
      storm_memset_cmng(adapter, &adapter->cmng, QFLE3_PORT(adapter));

   /*
    * init Event Queue - PCI bus guarantees correct endianity
    */
   eq_data.base_addr.hi = U64_HI(adapter->eq_mapping);
   eq_data.base_addr.lo = U64_LO(adapter->eq_mapping);
   eq_data.producer = adapter->eq_prod;
   eq_data.index_id = HC_SP_INDEX_EQ_CONS;
   eq_data.sb_id = DEF_SB_ID;
   storm_memset_eq_data(adapter, &eq_data, QFLE3_FUNC(adapter));
}

static void
qfle3_init_internal_common(struct qfle3_adapter *adapter)
{
   int i;

   /*
    * Zero this manually as its initialization is
    * currently missing in the initTool
    */
   for (i = 0; i < (USTORM_AGG_DATA_SIZE >> 2); i++)
      REG_WR(adapter, BAR_USTRORM_INTMEM + USTORM_AGG_DATA_OFFSET + i * 4, 0);
   if (!CHIP_IS_E1x(adapter)) {
      REG_WR8(adapter, BAR_CSTRORM_INTMEM + CSTORM_IGU_MODE_OFFSET,
	      CHIP_INT_MODE_IS_BC(adapter) ? HC_IGU_BC_MODE : HC_IGU_NBC_MODE);
   }
}

static void
qfle3_init_internal(struct qfle3_adapter *adapter, vmk_uint32 load_code)
{
   switch (load_code) {
   case FW_MSG_CODE_DRV_LOAD_COMMON:
   case FW_MSG_CODE_DRV_LOAD_COMMON_CHIP:
      qfle3_init_internal_common(adapter);
      /*
       * no break
       */

   case FW_MSG_CODE_DRV_LOAD_PORT:
      /*
       * nothing to do
       */
      /*
       * no break
       */

   case FW_MSG_CODE_DRV_LOAD_FUNCTION:
      /*
       * internal memory per function is
       * initialized inside qfle3_pf_init
       */
      break;

   default:
      QFLE3_ERR("Unknown load_code (0x%x) from MCP\n", load_code);
      break;
   }
}

vmk_Bool
qfle3_clear_pf_load(struct qfle3_adapter *adapter)
{
   vmk_uint32 val1, val;
   vmk_uint32 mask = QFLE3_PATH(adapter) ? QFLE3_PATH1_LOAD_CNT_MASK :
      QFLE3_PATH0_LOAD_CNT_MASK;
   vmk_uint32 shift = QFLE3_PATH(adapter) ? QFLE3_PATH1_LOAD_CNT_SHIFT :
      QFLE3_PATH0_LOAD_CNT_SHIFT;

   qfle3_acquire_hw_lock(adapter, HW_LOCK_RESOURCE_RECOVERY_REG);
   val = REG_RD(adapter, QFLE3_RECOVERY_GLOB_REG);
   //QFLE3_DBG(QFLE3_DBG_IFDOWN, "Old GEN_REG_VAL=0x%08x\n", val);

   /*
    * get the current counter value
    */
   val1 = (val & mask) >> shift;

   /*
    * clear bit of that PF
    */
   val1 &= ~(1 << adapter->pf_num);

   /*
    * clear the old value
    */
   val &= ~mask;

   /*
    * set the new one
    */
   val |= ((val1 << shift) & mask);

   REG_WR(adapter, QFLE3_RECOVERY_GLOB_REG, val);
   qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_RECOVERY_REG);
   return val1 != 0;
}

void
qfle3_set_pf_load(struct qfle3_adapter *adapter)
{
   vmk_uint32 val1, val;
   vmk_uint32 mask = QFLE3_PATH(adapter) ? QFLE3_PATH1_LOAD_CNT_MASK :
      QFLE3_PATH0_LOAD_CNT_MASK;
   vmk_uint32 shift = QFLE3_PATH(adapter) ? QFLE3_PATH1_LOAD_CNT_SHIFT :
      QFLE3_PATH0_LOAD_CNT_SHIFT;

   qfle3_acquire_hw_lock(adapter, HW_LOCK_RESOURCE_RECOVERY_REG);

   val = REG_RD(adapter, QFLE3_RECOVERY_GLOB_REG);
   val1 = (val & mask) >> shift;
   val1 |= (1 << adapter->pf_num);
   val &= ~mask;
   val |= ((val1 << shift) & mask);

   REG_WR(adapter, QFLE3_RECOVERY_GLOB_REG, val);
   qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_RECOVERY_REG);
}

vmk_uint16
qfle3_cid_ilt_lines(struct qfle3_adapter *adapter)
{
#ifdef QFLE3_SRIOV
   if(adapter->vfdb){
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: hence returning lies accordingly\n");
      return ((QFLE3_FIRST_VF_CID + QFLE3_VF_CIDS)/ILT_PAGE_CIDS);
   }
   else
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Adapter vfdb not allocated yet\n");
#endif //QFLE3_SRIOV
   return L2_ILT_LINES(adapter);
}

static VMK_ReturnStatus
qfle3_alloc_cdu_cxt_mem(struct qfle3_adapter *adapter)
{
   vmk_uint32 i;
   vmk_uint32 cxt_size;
   vmk_uint32 allocated;
   VMK_ReturnStatus status = VMK_OK;

   cxt_size = sizeof(union cdu_context) * QFLE3_L2_CID_COUNT(adapter);
   QFLE3_DBG(QFLE3_DBG_LOAD, "allocating %u number of cdu cxt \n", QFLE3_L2_CID_COUNT(adapter));

   for (i = 0, allocated = 0; allocated < cxt_size; i++) {
      adapter->context[i].size = MIN_OF(CDU_ILT_PAGE_SZ, (cxt_size - allocated));

      adapter->context[i].vcxt =
      qfle3_alloc_dma_mapping(adapter,
            adapter->dmaEngineCoherent,
            adapter->context[i].size,
            &adapter->context[i].cxt_mapping);
      if (!adapter->context[i].vcxt) {
         status = VMK_NO_MEMORY;
         break;
      }
      
      QFLE3_DBG(QFLE3_DBG_LOAD, "allocating %u context page of size %lu \n", i, adapter->context[i].size);
      allocated += adapter->context[i].size;
   }

   if (status != VMK_OK) {
      //Handle error
   }

   return status;
}

void
qfle3_setup_ilt(struct qfle3_adapter *adapter)
{
   struct ilt_client_info *ilt_client;
   struct ecore_ilt *ilt = QFLE3_ILT(adapter);
   vmk_uint16 line = 0;

   ilt->start_line = FUNC_ILT_BASE(QFLE3_FUNC(adapter));
   
	QFLE3_DBG(QFLE3_DBG_START, "ilt starts at line %d\n", ilt->start_line);
   /*
    * CDU
    */
   ilt_client = &ilt->clients[ILT_CLIENT_CDU];
   ilt_client->client_num = ILT_CLIENT_CDU;
   ilt_client->page_size = CDU_ILT_PAGE_SZ;
   ilt_client->flags = ILT_CLIENT_SKIP_MEM;
   ilt_client->start = line;
   line += qfle3_cid_ilt_lines(adapter);
#ifdef QFLE3_CNIC
   if (CNIC_SUPPORT(adapter))
      line += CNIC_ILT_LINES;
#endif
   ilt_client->end = line - 1;

	QFLE3_DBG(QFLE3_DBG_START, "ilt client[CDU]: start %d, end %d, psz 0x%x, flags 0x%x, hw psz %d\n",
	   ilt_client->start,
	   ilt_client->end,
	   ilt_client->page_size,
	   ilt_client->flags,
	   ILOG2(ilt_client->page_size >> 12));
   /*
    * QM
    */
   if (QM_INIT(adapter->qm_cid_count)) {
      ilt_client = &ilt->clients[ILT_CLIENT_QM];
      ilt_client->client_num = ILT_CLIENT_QM;
      ilt_client->page_size = QM_ILT_PAGE_SZ;
      ilt_client->flags = 0;
      ilt_client->start = line;

      line += (vmk_uint16) (DIV_ROUND_UP(adapter->qm_cid_count * QM_QUEUES_PER_FUNC * 4,
					 QM_ILT_PAGE_SZ));

      ilt_client->end = line - 1;
      
		QFLE3_DBG(QFLE3_DBG_START,
		   "ilt client[QM]: start %d, end %d, psz 0x%x, flags 0x%x, hw psz %d\n",
		   ilt_client->start,
		   ilt_client->end,
		   ilt_client->page_size,
		   ilt_client->flags,
		   ILOG2(ilt_client->page_size >> 12));
   }
#ifdef QFLE3_CNIC
	if (CNIC_SUPPORT(adapter)) {
		/* SRC */
		ilt_client = &ilt->clients[ILT_CLIENT_SRC];
		ilt_client->client_num = ILT_CLIENT_SRC;
		ilt_client->page_size = SRC_ILT_PAGE_SZ;
		ilt_client->flags = 0;
		ilt_client->start = line;
		line += SRC_ILT_LINES;
		ilt_client->end = line - 1;

 
       QFLE3_DBG(QFLE3_DBG_START,
          "ilt client[SRC]: start %d, end %d, psz 0x%x, flags 0x%x, hw psz %d\n",
          ilt_client->start,
          ilt_client->end,
          ilt_client->page_size,
          ilt_client->flags,
          ILOG2(ilt_client->page_size >> 12));

		/* TM */
		ilt_client = &ilt->clients[ILT_CLIENT_TM];
		ilt_client->client_num = ILT_CLIENT_TM;
		ilt_client->page_size = TM_ILT_PAGE_SZ;
		ilt_client->flags = 0;
		ilt_client->start = line;
		line += TM_ILT_LINES;
		ilt_client->end = line - 1;

   QFLE3_DBG(QFLE3_DBG_START,
      "ilt client[TM]: start %d, end %d, psz 0x%x, flags 0x%x, hw psz %d\n",
      ilt_client->start,
      ilt_client->end,
      ilt_client->page_size,
      ilt_client->flags,
      ILOG2(ilt_client->page_size >> 12));

 	}
#endif
   VMK_ASSERT(line < ILT_MAX_LINES);

}

static void
qfle3_zero_drv_ver_ncsi_oem(struct qfle3_adapter *adapter)
{
   vmk_uint32 ncsi_oem_data_addr;
   if (adapter->load_code == FW_MSG_CODE_DRV_LOAD_COMMON_CHIP) {
      if (SHMEM2_HAS(adapter, ncsi_oem_data_addr)) {
	 ncsi_oem_data_addr = SHMEM2_RD(adapter, ncsi_oem_data_addr);
	 if (ncsi_oem_data_addr) {
	    REG_WR(adapter,
		   ncsi_oem_data_addr +
		   vmk_offsetof(struct glob_ncsi_oem_data, driver_version), 0);
	 }
      }
   }
}

static int
qfle3_reset_hw(struct qfle3_adapter *adapter, vmk_uint32 load_code)
{
   struct ecore_func_state_params func_params = {NULL};

   /* Prepare parameters for function state transitions */
   ECORE_SET_BIT(RAMROD_COMP_WAIT, &func_params.ramrod_flags);

   func_params.f_obj = &adapter->func_obj;
   func_params.cmd = ECORE_F_CMD_HW_RESET;

   func_params.params.hw_init.load_phase = load_code;

   return ecore_func_state_change(adapter, &func_params);
}

static VMK_ReturnStatus
qfle3_init_hw(struct qfle3_adapter *adapter, vmk_uint32 load_code)
{
   int rc = 0;
   VMK_ReturnStatus status = VMK_OK;
   struct ecore_func_state_params func_params = { NULL };

   /*
    * Prepare parameters for function state transitions
    */
   SET_BIT(RAMROD_COMP_WAIT, &func_params.ramrod_flags);

   func_params.f_obj = &adapter->func_obj;
   func_params.cmd = ECORE_F_CMD_HW_INIT;

   func_params.params.hw_init.load_phase = load_code;

   rc = ecore_func_state_change(adapter, &func_params);
   if (rc) {
      status = VMK_FAILURE;
   }
   return status;
}

static inline VMK_ReturnStatus
qfle3_func_stop(struct qfle3_adapter *adapter)
{
   struct ecore_func_state_params func_params = {NULL};
   int rc;

   /* Prepare parameters for function state transitions */
   ECORE_SET_BIT(RAMROD_COMP_WAIT, &func_params.ramrod_flags);
   func_params.f_obj = &adapter->func_obj;
   func_params.cmd = ECORE_F_CMD_STOP;

   /*
    * Try to stop the function the 'good way'. If fails (in case
    * of a parity error during qfle3_chip_cleanup()) and we are
    * not in a debug mode, perform a state transaction in order to
    * enable further HW_RESET transaction.
    */
   rc = ecore_func_state_change(adapter, &func_params);
   if (rc) {
#ifdef QFLE3_STOP_ON_ERROR
      return VMK_FAILURE;
#else
      QFLE3_ERR("FUNC_STOP ramrod failed. Running a dry transaction\n");
      ECORE_SET_BIT(RAMROD_DRV_CLR_ONLY, &func_params.ramrod_flags);
      rc = ecore_func_state_change(adapter, &func_params);
      if (rc) {
	 return VMK_FAILURE;
      }
#endif
   }
   return VMK_OK;
}

void
qfle3_get_c2s_mapping(qfle3_adapter *adapter,
                               vmk_uint8 *c2s_map, vmk_uint8 *c2s_default)
{
   int mfw_vn = QFLE3_FW_MB_IDX(adapter);
   vmk_uint32 tmp;

   /* If the shmem shouldn't affect configuration, reflect */
   if(!IS_MF_BD(adapter)) {
      int i;

      for (i = 0; i < QFLE3_MAX_PRIORITY; i++)
              c2s_map[i] = i;

      *c2s_default = 0;

      return;
   }

   tmp = SHMEM2_RD(adapter, c2s_pcp_map_lower[mfw_vn]);
   tmp = vmk_BE32ToCPU(tmp);
   c2s_map[0] = tmp & 0xff;
   c2s_map[1] = (tmp >> 8) & 0xff;
   c2s_map[2] = (tmp >> 16) & 0xff;
   c2s_map[3] = (tmp >> 24) & 0xff;

   tmp = SHMEM2_RD(adapter, c2s_pcp_map_upper[mfw_vn]);
   tmp = vmk_BE32ToCPU(tmp);
   c2s_map[4] = tmp & 0xff;
   c2s_map[5] = (tmp >> 8) & 0xff;
   c2s_map[6] = (tmp >> 16) & 0xff;
   c2s_map[7] = (tmp >> 24) & 0xff;

   tmp = SHMEM2_RD(adapter, c2s_pcp_map_default[mfw_vn]);
   tmp = vmk_BE32ToCPU(tmp);
   *c2s_default = (tmp >> (8 * mfw_vn)) & 0xff;
}


static inline VMK_ReturnStatus
qfle3_func_start(struct qfle3_adapter *adapter)
{
   int rc = 0;
   VMK_ReturnStatus status = VMK_OK;
   struct ecore_func_state_params func_params = { NULL };
   struct ecore_func_start_params *start_params = &func_params.params.start;

   /*
    * Prepare parameters for function state transitions
    */
   SET_BIT(RAMROD_COMP_WAIT, &func_params.ramrod_flags);

   func_params.f_obj = &adapter->func_obj;
   func_params.cmd = ECORE_F_CMD_START;

   /*
    * Function parameters
    */
   start_params->mf_mode = adapter->mf_mode;
   start_params->sd_vlan_tag = adapter->mf_ov;

   if (IS_MF_BD(adapter)) {
      start_params->sd_vlan_eth_type = 0x88A8; //8021AD
      REG_WR(adapter, PRS_REG_VLAN_TYPE_0, 0x88A8);
      REG_WR(adapter, PBF_REG_VLAN_TYPE_0, 0x88A8);
      REG_WR(adapter, NIG_REG_LLH_E1HOV_TYPE_1, 0x88A8);

      qfle3_get_c2s_mapping(adapter, start_params->c2s_pri,
                                      &start_params->c2s_pri_default);
      start_params->c2s_pri_valid = 1;
   }
   if (adapter->sw_fcoe){
      start_params->network_cos_mode = STATIC_COS;// for Multi_COS, otherwize, OVERRIDE_COS;
   } else {
   start_params->network_cos_mode = OVERRIDE_COS;
   }

   start_params->vxlan_dst_port = vmk_BE16ToCPU(vmk_UplinkVXLANPortNBOGet());
   start_params->inner_rss = 1;

   if (adapter->vxlan_filters_en) {
      start_params->inner_clss_vxlan = INNER_CLSS_USE_VNI;
   }
   if (QFLE3_IS_GENEVE_OFFLOAD_ENABLED(adapter)) {
      start_params->geneve_dst_port = adapter->geneve_udp_dst_port;
      start_params->inner_rss = 1;
   }

   if (IS_MF_UFP(adapter) && QFLE3_IS_MF_SD_PROTOCOL_FCOE(adapter)) {
      start_params->class_fail_ethtype = 0x8914; //FIP
      start_params->class_fail = 1;
      start_params->no_added_tags = 1;
   }

   rc = ecore_func_state_change(adapter, &func_params);
   if (rc) {
      status = VMK_FAILURE;
   }
   return status;
}

static int
qfle3_set_qm_cid_count(struct qfle3_adapter *adapter)
{
   int cid_count = QFLE3_L2_MAX_CID(adapter);
#ifdef QFLE3_SRIOV
   if(adapter->vfdb)
      cid_count += QFLE3_VF_CIDS;
   else
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Adapter vfdb not allocated yet\n");
#endif //QFLE3_SRIOV

#ifdef QFLE3_CNIC 
   if (CNIC_SUPPORT(adapter))
      cid_count += CNIC_CID_MAX;
   QFLE3_DBG(QFLE3_DBG_LOAD|QFLE3_DBG_CNIC, "l2 max cid is %u, total cid_count %u \n", 
      QFLE3_L2_MAX_CID(adapter), ROUNDUP(cid_count, QM_CID_ROUND));
   QFLE3_DBG(QFLE3_DBG_LOAD|QFLE3_DBG_CNIC, "max queue count is %u\n", QFLE3_MAX_RSS_COUNT(adapter));
   QFLE3_DBG(QFLE3_DBG_LOAD|QFLE3_DBG_CNIC, "igu_sb_cnt is %u\n", (adapter)->igu_sb_cnt);
   QFLE3_DBG(QFLE3_DBG_LOAD|QFLE3_DBG_CNIC, "uio_cid_pad is %u\n", UIO_CID_PAD(adapter));
   QFLE3_DBG(QFLE3_DBG_LOAD|QFLE3_DBG_CNIC, "start cnic eth cid is %u\n", QFLE3_CNIC_START_ETH_CID(adapter));
   
#endif   
   return ROUNDUP(cid_count, QM_CID_ROUND);
}


inline void
qfle3_get_pci_device_info(qfle3_adapter * adapter)
{
   vmk_uint16 capOffset = 0;
   VMK_ReturnStatus status;
   vmk_uint32 pmc;

   vmk_DeviceGetRegistrationData(adapter->device, (vmk_AddrCookie *) & adapter->pdev);
   vmk_PCIQueryDeviceID(adapter->pdev, &adapter->pdev_id);
   vmk_PCIQueryDeviceAddr(adapter->pdev, &adapter->pdev_addr);
   vmk_NameFormat(&adapter->pdev_name, QFLE3_SBDF_FMT,
		  adapter->pdev_addr.seg, adapter->pdev_addr.bus,
		  adapter->pdev_addr.dev, adapter->pdev_addr.fn);

   QFLE3_DBG(QFLE3_DBG_LOAD,
	     "PCI device " QFLE3_PCIID_FMT " detected",
	     adapter->pdev_id.deviceID, adapter->pdev_id.vendorID,
	     adapter->pdev_id.subDeviceID, adapter->pdev_id.subVendorID);
   status = vmk_PCIFindCapability(adapter->pdev, PCI_CAP_ID_PM, &capOffset);
   if (status == VMK_OK) {
      adapter->pm_cap = capOffset;
      vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev, VMK_PCI_CONFIG_ACCESS_16, adapter->pm_cap + PCI_PM_PMC, &pmc);
      adapter->flags |= (pmc & PCI_PM_CAP_PME_D3cold) ? 0 : NO_WOL_FLAG;
   }else {
      QFLE3_ERR("Cannot find power management capability.\n");
      adapter->pm_cap = 0;
   }


}

VMK_ReturnStatus
qfle3_is_device_supported(qfle3_adapter * adapter)
{
   VMK_ReturnStatus status = VMK_OK;

   switch (adapter->pdev_id.deviceID) {
   case CHIP_NUM_57712:
   case CHIP_NUM_57800:
   case CHIP_NUM_57810:
   case CHIP_NUM_57811:
   case CHIP_NUM_57840_4_10:
   case CHIP_NUM_57840_2_20:
#ifdef QFLE3_NPAR
   case CHIP_NUM_57840_MF:
   case CHIP_NUM_57712_MF:
   case CHIP_NUM_57800_MF:
   case CHIP_NUM_57810_MF:
   case CHIP_NUM_57811_MF:
#endif
      break;
   default:
      QFLE3_DBG(QFLE3_DBG_LOAD,
		"PCI device ID 0x%x not supported by the driver",
		adapter->pdev_id.deviceID);
      status = VMK_FAILURE;
   }
   return status;
}

VMK_ReturnStatus
qfle3_disable_bus_master(qfle3_adapter * adapter)
{
   vmk_uint32 cmd;
   VMK_ReturnStatus status = VMK_OK;

   status = vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev,
			      VMK_PCI_CONFIG_ACCESS_16, QFLE3_PCIR_COMMAND, &cmd);
   if (status != VMK_OK) {
      QFLE3_ERR("Failed to read PCI config. offset: %u (%x)",
		QFLE3_PCIR_COMMAND, status);
      goto out;
   }

   if (!(cmd & QFLE3_PCI_REG_COMMAND_BUS_MASTER)) {
      QFLE3_DBG(QFLE3_DBG_LOAD, "Disable bus mastering");
      cmd &= ~QFLE3_PCI_REG_COMMAND_BUS_MASTER;
      status = vmk_PCIWriteConfig(vmk_ModuleCurrentID, adapter->pdev,
				  VMK_PCI_CONFIG_ACCESS_16, QFLE3_PCIR_COMMAND, cmd);
      if (status != VMK_OK) {
	 QFLE3_ERR("Failed to write PCI config. "
		   "offset: %u, data: %u (%x)", QFLE3_PCIR_COMMAND, cmd, status);
	 goto out;
      }
   }

   return VMK_OK;

  out:
   QFLE3_ERR("Failed to disable bus mastering!");
   return status;
}

static inline VMK_ReturnStatus
qfle3_enable_bus_master(qfle3_adapter * adapter)
{
   vmk_uint32 cmd;
   VMK_ReturnStatus status = VMK_OK;

   status = vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev,
			      VMK_PCI_CONFIG_ACCESS_16, QFLE3_PCIR_COMMAND, &cmd);
   if (status != VMK_OK) {
      QFLE3_ERR("Failed to read PCI config. offset: %u (%x)",
		QFLE3_PCIR_COMMAND, status);
      goto out;
   }

   if (!(cmd & QFLE3_PCI_REG_COMMAND_BUS_MASTER)) {
      QFLE3_DBG(QFLE3_DBG_LOAD, "Enable bus mastering");
      cmd |= QFLE3_PCI_REG_COMMAND_BUS_MASTER;
      status = vmk_PCIWriteConfig(vmk_ModuleCurrentID, adapter->pdev,
				  VMK_PCI_CONFIG_ACCESS_16, QFLE3_PCIR_COMMAND, cmd);
      if (status != VMK_OK) {
	 QFLE3_ERR("Failed to write PCI config. "
		   "offset: %u, data: %u (%x)", QFLE3_PCIR_COMMAND, cmd, status);
	 goto out;
      }
   }

   return VMK_OK;

  out:
   QFLE3_ERR("Failed to enable bus mastering!");
   return status;
}

static VMK_ReturnStatus
qfle3_map_io_resource(qfle3_adapter * adapter)
{
   VMK_ReturnStatus status;
   status = vmk_PCIMapIOResource(vmk_ModuleCurrentID,
				 adapter->pdev, 0, NULL, (vmk_VA *) & adapter->reg_base);
   if (status != VMK_OK) {
      QFLE3_ERR("Failed to map BAR0 (%x)", status);
      goto fail_bar0;
   }

   status = vmk_PCIMapIOResource(vmk_ModuleCurrentID,
				 adapter->pdev, 2, NULL, (vmk_VA *) & adapter->db_base);
   if (status != VMK_OK) {
      QFLE3_ERR("Failed to map BAR2 (%x)", status);
      goto fail_bar2;
   }

   return VMK_OK;

  fail_bar2:
   vmk_PCIUnmapIOResource(vmk_ModuleCurrentID, adapter->pdev, 0);
  fail_bar0:
   return status;
}

static void
qfle3_unmap_io_resource(qfle3_adapter * adapter)
{
   VMK_ReturnStatus status;
   
   status = vmk_PCIUnmapIOResource(vmk_ModuleCurrentID, adapter->pdev, 2);
   QFLE3_DBG(QFLE3_DBG_LOAD, "vmk_PCIUnmapIOResource 2, status %d", status);
   status = vmk_PCIUnmapIOResource(vmk_ModuleCurrentID, adapter->pdev, 0);
   QFLE3_DBG(QFLE3_DBG_LOAD, "vmk_PCIUnmapIOResource 0, status %d", status);
}


static VMK_ReturnStatus
qfle3_alloc_bit_vectors(qfle3_adapter * adapter)
{
   /*
    * Device state bit vector
    */
   adapter->state = vmk_BitVectorAlloc(qfle3_mod_info.heapID, QFLE3_STATE_MAX);
   if (!adapter->state) {
      QFLE3_ERR("Failed to allocate bit vectors");
      return VMK_NO_MEMORY;
   }

   vmk_BitVectorZap(adapter->state);
   return VMK_OK;
}

static void
qfle3_free_bit_vectors(qfle3_adapter * adapter)
{
   vmk_BitVectorFree(qfle3_mod_info.heapID, adapter->state);
}

static inline void
qfle3_destroy_dma_engines(qfle3_adapter * adapter)
{
   if (adapter->dmaEngine) {
      vmk_DMAEngineDestroy(adapter->dmaEngine);
   }

   if (adapter->dmaEngineCoherent) {
      vmk_DMAEngineDestroy(adapter->dmaEngineCoherent);
   }

   if (adapter->dmaEngine_rxbuf) {
      vmk_DMAEngineDestroy(adapter->dmaEngine_rxbuf);
   }
}

static VMK_ReturnStatus
qfle3_create_dma_engines(qfle3_adapter * adapter)
{
   VMK_ReturnStatus status;
   vmk_DMAConstraints constraints;
   vmk_DMAEngineProps props;

   constraints.addressMask = VMK_ADDRESS_MASK_64BIT;
   constraints.maxTransfer = 0;
   constraints.sgMaxEntries = 0;
   constraints.sgElemMaxSize = 0;
   constraints.sgElemSizeMult = 0;
   constraints.sgElemAlignment = 0;
   constraints.sgElemStraddle = 0;

   /*
    * DMA engine for dma mapping
    */
   vmk_NameFormat(&props.name, "qfle3-%s", QFLE3_NAME_TO_STRING(adapter->pdev_name));
   props.module = vmk_ModuleCurrentID;
   props.device = adapter->device;
   props.constraints = &constraints;
   props.bounce = NULL;
   props.flags = VMK_DMA_ENGINE_FLAGS_NONE;
   status = vmk_DMAEngineCreate(&props, &adapter->dmaEngine);
   if (status != VMK_OK) {
      QFLE3_ERR("Failed to create DMA engine (%x)", status);
      goto fail_engine;
   }

   /*
    * DMA engine for coherent dma mapping
    */
   vmk_NameFormat(&props.name, "qfle3-co-%s", QFLE3_NAME_TO_STRING(adapter->pdev_name));
   props.flags = VMK_DMA_ENGINE_FLAGS_COHERENT;
   status = vmk_DMAEngineCreate(&props, &adapter->dmaEngineCoherent);
   if (status != VMK_OK) {
      QFLE3_ERR("Failed to create coherent DMA engine (%x)", status);
      goto fail_coherent_engine;
   }

   /*
    * DMA engine for rx buffer dma mapping
    */
   vmk_NameFormat(&props.name, "qfle3-rxbuf-%s",
		  QFLE3_NAME_TO_STRING(adapter->pdev_name));

   constraints.sgElemAlignment = 16;
   props.flags = VMK_DMA_ENGINE_FLAGS_NONE;
   status = vmk_DMAEngineCreate(&props, &adapter->dmaEngine_rxbuf);
   if (status != VMK_OK) {
      QFLE3_ERR("Failed to create rx buffer DMA engine (%x)", status);
      goto fail_rxbuf_engine;
   }

   return VMK_OK;

  fail_rxbuf_engine:
   vmk_DMAEngineDestroy(adapter->dmaEngineCoherent);
  fail_coherent_engine:
   vmk_DMAEngineDestroy(adapter->dmaEngine);
  fail_engine:
   return status;
}

static VMK_ReturnStatus
qfle3_prev_mcp_done(struct qfle3_adapter *adapter)
{
	vmk_uint32	rc = qfle3_mfw_command(adapter, DRV_MSG_CODE_UNLOAD_DONE,
			DRV_MSG_CODE_UNLOAD_SKIP_LINK_RESET);
	if (!rc) {
		QFLE3_ERR("MCP response failure, aborting\n");
		return VMK_BUSY;
	}

	return VMK_OK;
}

static struct qfle3_prev_path_list *
qfle3_prev_path_get_entry(struct qfle3_adapter *adapter)
{
	struct qfle3_prev_path_list *tmp_list;
	struct vmk_ListLinks *list_itr;

	VMK_LIST_FORALL(&qfle3_mod_info.prev_list, list_itr) {
      tmp_list = VMK_LIST_ENTRY(list_itr, struct qfle3_prev_path_list, link);
		if (adapter->pdev_addr.dev == tmp_list->slot &&
				adapter->pdev_addr.bus == tmp_list->bus &&
				QFLE3_PATH(adapter) == tmp_list->path)
			return tmp_list;
	}

	return NULL;
}

vmk_Bool
qfle3_port_after_undi(struct qfle3_adapter *adapter)
{
	struct qfle3_prev_path_list *entry;
	vmk_Bool val;

	vmk_SemaLock(&qfle3_mod_info.prev_mutex);

	entry = qfle3_prev_path_get_entry(adapter);
	val = !!(entry && (entry->undi & (1 << QFLE3_PORT(adapter))));

	vmk_SemaUnlock(&qfle3_mod_info.prev_mutex);

	return val;
}

static vmk_Bool
qfle3_prev_is_path_marked(struct qfle3_adapter *adapter)
{
	struct qfle3_prev_path_list *tmp_list;
	vmk_Bool rc = VMK_FALSE;
	VMK_ReturnStatus status;

	status = vmk_SemaTryLock(&qfle3_mod_info.prev_mutex);
	if (status != VMK_OK)
		return VMK_FALSE;

	tmp_list = qfle3_prev_path_get_entry(adapter);
	if (tmp_list) {
		if (tmp_list->aer) {
			QFLE3_DBG(QFLE3_DBG_HW, "Path %d was marked by AER\n",
					QFLE3_PATH(adapter));
		} else {
			rc = VMK_TRUE;
			QFLE3_DBG(QFLE3_DBG_HW, "Path %d was already cleaned from previous drivers\n",
					QFLE3_PATH(adapter));
		}
	}

	vmk_SemaUnlock(&qfle3_mod_info.prev_mutex);
	return rc;
}

static VMK_ReturnStatus
qfle3_prev_mark_path(struct qfle3_adapter *adapter, vmk_Bool after_undi)
{
	VMK_ReturnStatus status;
	struct qfle3_prev_path_list *tmp_list;

	status = vmk_SemaTryLock(&qfle3_mod_info.prev_mutex);
	if (status != VMK_OK) {
		QFLE3_ERR("Failed to take lock\n");
		return status;
	}

	/* Check whether the entry for this path already exists */
	tmp_list = qfle3_prev_path_get_entry(adapter);
	if (tmp_list) {
		if (!tmp_list->aer) {
			QFLE3_ERR("Re-Marking the path.\n");
		} else {
			QFLE3_DBG(QFLE3_DBG_HW, "Removing AER indication from path %d\n",
					QFLE3_PATH(adapter));
			tmp_list->aer = 0;
		}
		vmk_SemaUnlock(&qfle3_mod_info.prev_mutex);
		return VMK_OK;

	}
	vmk_SemaUnlock(&qfle3_mod_info.prev_mutex);

	/* Create an entry for this path and add it */
	tmp_list = qfle3_heap_alloc(sizeof(struct qfle3_prev_path_list));
	if (!tmp_list) {
		QFLE3_ERR("Failed to allocate 'qfle3_prev_path_list'\n");
		return VMK_NO_MEMORY;
	}

	tmp_list->bus = adapter->pdev_addr.bus;
	tmp_list->slot = adapter->pdev_addr.dev;
	tmp_list->path = QFLE3_PATH(adapter);
	tmp_list->aer = 0;
	tmp_list->undi = after_undi ? (1 << QFLE3_PORT(adapter)) : 0;

	status = vmk_SemaTryLock(&qfle3_mod_info.prev_mutex);
	if (status != VMK_OK) {
		QFLE3_ERR("Failed to take lock\n");
		qfle3_heap_free(tmp_list);
	} else {
		QFLE3_DBG(QFLE3_DBG_HW, "Marked path [%d] - finished previous unload\n",
				QFLE3_PATH(adapter));

		vmk_ListInsert(&tmp_list->link, vmk_ListAtFront(&qfle3_mod_info.prev_list));
		vmk_SemaUnlock(&qfle3_mod_info.prev_mutex);
	}

	return status;
}

static vmk_Bool
qfle3_prev_unload_close_umac(struct qfle3_adapter *adapter, vmk_uint8 port,
		vmk_uint32 reset_reg,
		struct qfle3_mac_vals *vals)
{
	vmk_uint32 mask = MISC_REGISTERS_RESET_REG_2_UMAC0 << port;
	vmk_uint32 base_addr;

	if (!(mask & reset_reg))
		return VMK_FALSE;

	QFLE3_DBG(QFLE3_DBG_LOAD, "Disable umac Rx %02x\n", port);
	base_addr = port ? GRCBASE_UMAC1 : GRCBASE_UMAC0;
	vals->umac_addr[port] = base_addr + UMAC_REG_COMMAND_CONFIG;
	vals->umac_val[port] = REG_RD(adapter, vals->umac_addr[port]);
	REG_WR(adapter, vals->umac_addr[port], 0);

	return VMK_TRUE;
}

static void
qfle3_prev_unload_close_mac(struct qfle3_adapter *adapter,
		struct qfle3_mac_vals *vals)
{
	vmk_uint32 val, base_addr, offset, mask, reset_reg;
	vmk_Bool mac_stopped = VMK_FALSE;
	vmk_uint8 port = QFLE3_PORT(adapter);

	/* reset addresses as they also mark which values were changed */
	vmk_Memset(vals, 0, sizeof(*vals));

	reset_reg = REG_RD(adapter, MISC_REG_RESET_REG_2);

	if (!CHIP_IS_E3(adapter)) {
		val = REG_RD(adapter, NIG_REG_BMAC0_REGS_OUT_EN + port * 4);
		mask = MISC_REGISTERS_RESET_REG_2_RST_BMAC0 << port;
		if ((mask & reset_reg) && val) {
			vmk_uint32 wb_data[2];
			QFLE3_DBG(QFLE3_DBG_LOAD, "Disable bmac Rx\n");
			base_addr = QFLE3_PORT(adapter) ? NIG_REG_INGRESS_BMAC1_MEM
				: NIG_REG_INGRESS_BMAC0_MEM;
			offset = CHIP_IS_E2(adapter) ? BIGMAC2_REGISTER_BMAC_CONTROL
				: BIGMAC_REGISTER_BMAC_CONTROL;

			/*
			 * use rd/wr since we cannot use dmae. This is safe
			 * since MCP won't access the bus due to the request
			 * to unload, and no function on the path can be
			 * loaded at this time.
			 */
			wb_data[0] = REG_RD(adapter, base_addr + offset);
			wb_data[1] = REG_RD(adapter, base_addr + offset + 0x4);
			vals->bmac_addr = base_addr + offset;
			vals->bmac_val[0] = wb_data[0];
			vals->bmac_val[1] = wb_data[1];
			wb_data[0] &= ~ELINK_BMAC_CONTROL_RX_ENABLE;
			REG_WR(adapter, vals->bmac_addr, wb_data[0]);
			REG_WR(adapter, vals->bmac_addr + 0x4, wb_data[1]);
		}
		QFLE3_DBG(QFLE3_DBG_LOAD, "Disable emac Rx\n");
		vals->emac_addr = NIG_REG_NIG_EMAC0_EN + QFLE3_PORT(adapter)*4;
		vals->emac_val = REG_RD(adapter, vals->emac_addr);
		REG_WR(adapter, vals->emac_addr, 0);
		mac_stopped = VMK_TRUE;
	} else {
		if (reset_reg & MISC_REGISTERS_RESET_REG_2_XMAC) {
			QFLE3_DBG(QFLE3_DBG_LOAD, "Disable xmac Rx [%2x]\n", QFLE3_PORT(adapter));
			base_addr = QFLE3_PORT(adapter) ? GRCBASE_XMAC1 : GRCBASE_XMAC0;
			val = REG_RD(adapter, base_addr + XMAC_REG_PFC_CTRL_HI);
			REG_WR(adapter, base_addr + XMAC_REG_PFC_CTRL_HI,
					val & ~(1 << 1));
			REG_WR(adapter, base_addr + XMAC_REG_PFC_CTRL_HI,
					val | (1 << 1));
			vals->xmac_addr = base_addr + XMAC_REG_CTRL;
			vals->xmac_val = REG_RD(adapter, vals->xmac_addr);
			REG_WR(adapter, vals->xmac_addr, 0);
			mac_stopped = VMK_TRUE;
		}

		mac_stopped |= qfle3_prev_unload_close_umac(adapter, 0,
				reset_reg, vals);
		mac_stopped |= qfle3_prev_unload_close_umac(adapter, 1,
				reset_reg, vals);
	}

	if (mac_stopped)
		vmk_WorldSleep(20 * 1000);
}

#define QFLE3_PREV_UNDI_PROD_ADDR(p) (BAR_TSTRORM_INTMEM + 0x1508 + ((p) << 4))
#define QFLE3_PREV_UNDI_PROD_ADDR_H(f) (BAR_TSTRORM_INTMEM + \
                                        0x1848 + ((f) << 4))
#define QFLE3_PREV_UNDI_RCQ(val)        ((val) & 0xffff)
#define QFLE3_PREV_UNDI_BD(val)         ((val) >> 16 & 0xffff)
#define QFLE3_PREV_UNDI_PROD(rcq, bd)   ((bd) << 16 | (rcq))

#define BCM_5710_UNDI_FW_MF_MAJOR       (0x07)
#define BCM_5710_UNDI_FW_MF_MINOR       (0x08)
#define BCM_5710_UNDI_FW_MF_VERS        (0x05)

static vmk_Bool
qfle3_prev_is_after_undi(struct qfle3_adapter *adapter)
{
	/* UNDI marks its presence in DORQ -
	 * it initializes CID offset for normal bell to 0x7
	 */
	if (!(REG_RD(adapter, MISC_REG_RESET_REG_1) &
				MISC_REGISTERS_RESET_REG_1_RST_DORQ))
		return VMK_FALSE;

	if (REG_RD(adapter, DORQ_REG_NORM_CID_OFST) == 0x7) {
		QFLE3_INFO("UNDI previously loaded\n");
		return VMK_TRUE;
	}

	return VMK_FALSE;
}

static void
qfle3_prev_unload_undi_inc(struct qfle3_adapter *adapter, vmk_uint8 inc)
{
	vmk_uint32 rcq, bd;
	vmk_uint32 addr, tmp_reg;

	if (QFLE3_FUNC(adapter) < 2)
		addr = QFLE3_PREV_UNDI_PROD_ADDR(QFLE3_PORT(adapter));
	else
		addr = QFLE3_PREV_UNDI_PROD_ADDR_H(QFLE3_FUNC(adapter) - 2);

	tmp_reg = REG_RD(adapter, addr);

	rcq = QFLE3_PREV_UNDI_RCQ(tmp_reg) + inc;
	bd = QFLE3_PREV_UNDI_BD(tmp_reg) + inc;

	tmp_reg = QFLE3_PREV_UNDI_PROD(rcq, bd);
	REG_WR(adapter, addr, tmp_reg);

	QFLE3_DBG(QFLE3_DBG_LOAD, "UNDI producer [%d/%d][%08x] rings bd -> 0x%04x, rcq -> 0x%04x\n",
			QFLE3_PORT(adapter), QFLE3_FUNC(adapter), addr, bd, rcq);
}

static VMK_ReturnStatus
qfle3_prev_unload_common(struct qfle3_adapter *adapter)
{
	vmk_uint32 reset_reg, tmp_reg = 0;
	vmk_Bool prev_undi = VMK_FALSE;
	struct qfle3_mac_vals mac_vals;
	VMK_ReturnStatus status = VMK_OK;

	/* It is possible a previous function received 'common' answer,
	 * but hasn't loaded yet, therefore creating a scenario of
	 * multiple functions receiving 'common' on the same path.
	 */
	QFLE3_DBG(QFLE3_DBG_LOAD, "Common unload Flow\n");

	vmk_Memset(&mac_vals, 0, sizeof(mac_vals));

	if (qfle3_prev_is_path_marked(adapter)){
      
           QFLE3_DBG(QFLE3_DBG_LOAD, "qfle3_prev_is_path_marked\n");
           return qfle3_prev_mcp_done(adapter);
        }

	reset_reg = REG_RD(adapter, MISC_REG_RESET_REG_1);

	/* Reset should be performed after BRB is emptied */
	if (reset_reg & MISC_REGISTERS_RESET_REG_1_RST_BRB1) {
		vmk_uint32 timer_count = 1000;

		/* Close the MAC Rx to prevent BRB from filling up */
		qfle3_prev_unload_close_mac(adapter, &mac_vals);

		/* close LLH filters for both ports towards the BRB */
		elink_set_rx_filter(&adapter->link_params, 0);
		adapter->link_params.port ^= 1;
		elink_set_rx_filter(&adapter->link_params, 0);
		adapter->link_params.port ^= 1;

		/* Check if the UNDI driver was previously loaded */
		if (qfle3_prev_is_after_undi(adapter)) {
			vmk_uint32 temp_value = 0;
			prev_undi = VMK_TRUE;
			/* clear the UNDI indication */
			REG_WR(adapter, DORQ_REG_NORM_CID_OFST, 0);
			/* clear possible idle check errors */
			temp_value = REG_RD(adapter, NIG_REG_NIG_INT_STS_CLR_0);
		}
		if (!CHIP_IS_E1x(adapter))
			/* block FW from writing to host */
			REG_WR(adapter, PGLUE_B_REG_INTERNAL_PFID_ENABLE_MASTER, 0);

		/* wait until BRB is empty */
		tmp_reg = REG_RD(adapter, BRB1_REG_NUM_OF_FULL_BLOCKS);
		while (timer_count) {
			vmk_uint32 prev_brb = tmp_reg;

			tmp_reg = REG_RD(adapter, BRB1_REG_NUM_OF_FULL_BLOCKS);
			if (!tmp_reg)
				break;

			QFLE3_DBG(QFLE3_DBG_LOAD, "BRB still has 0x%08x\n", tmp_reg);

			/* reset timer as long as BRB actually gets emptied */
			if (prev_brb > tmp_reg)
				timer_count = 1000;
			else
				timer_count--;

			/* If UNDI resides in memory, manually increment it */
			if (prev_undi)
				qfle3_prev_unload_undi_inc(adapter, 1);

			vmk_DelayUsecs(10);
		}

		if (!timer_count)
			QFLE3_ERR("Failed to empty BRB, hope for the best\n");
	}
        QFLE3_DBG(QFLE3_DBG_LOAD, "reset common\n");

	/* No packets are in the pipeline, path is ready for reset */
	qfle3_reset_common(adapter);

	if (mac_vals.xmac_addr)
		REG_WR(adapter, mac_vals.xmac_addr, mac_vals.xmac_val);
	if (mac_vals.umac_addr[0])
		REG_WR(adapter, mac_vals.umac_addr[0], mac_vals.umac_val[0]);
	if (mac_vals.umac_addr[1])
		REG_WR(adapter, mac_vals.umac_addr[1], mac_vals.umac_val[1]);
	if (mac_vals.emac_addr)
		REG_WR(adapter, mac_vals.emac_addr, mac_vals.emac_val);
	if (mac_vals.bmac_addr) {
		REG_WR(adapter, mac_vals.bmac_addr, mac_vals.bmac_val[0]);
		REG_WR(adapter, mac_vals.bmac_addr + 4, mac_vals.bmac_val[1]);
	}

	status = qfle3_prev_mark_path(adapter, prev_undi);
	if (status != VMK_OK) {
		qfle3_prev_mcp_done(adapter);
		return status;
	}

	return qfle3_prev_mcp_done(adapter);
}


static VMK_ReturnStatus
qfle3_wait_for_pending_pci_transaction(qfle3_adapter *adapter)
{
	int i;

	/* Wait for Transaction Pending bit clean */
	for (i = 0; i < 4; i++) {
		if (i)
			vmk_WorldSleep((1 << (i - 1)) * 100 * 1000);

		if (!qfle3_is_pcie_pending(adapter->pdev))
			return VMK_OK;
	}

	return VMK_FAILURE;
}

static VMK_ReturnStatus
qfle3_do_flr(struct qfle3_adapter *adapter)
{
	VMK_ReturnStatus status;

	/* only bootcode REQ_BC_VER_4_INITIATE_FLR and onwards support flr */
	if (adapter->hw_info.bc_ver < REQ_BC_VER_4_INITIATE_FLR) {
		QFLE3_ERR("FLR not supported by BC_VER: 0x%x\n",
				adapter->hw_info.bc_ver);
		return VMK_BAD_PARAM;
	}

	status = qfle3_wait_for_pending_pci_transaction(adapter);
	if (status != VMK_OK)
		QFLE3_ERR("Transaction is not cleared; proceeding with reset anyway\n");

	QFLE3_INFO("Initiating FLR\n");
	qfle3_mfw_command(adapter, DRV_MSG_CODE_INITIATE_FLR, 0);

	return VMK_OK;
}

static VMK_ReturnStatus
qfle3_prev_unload_uncommon(struct qfle3_adapter *adapter)
{
	VMK_ReturnStatus status;

	QFLE3_INFO("Uncommon unload Flow\n");

	/* Test if previous unload process was already finished for this path */
	if (qfle3_prev_is_path_marked(adapter))
		return qfle3_prev_mcp_done(adapter);

	QFLE3_DBG(QFLE3_DBG_LOAD, "Path is unmarked\n");

	/* Cannot proceed with FLR if UNDI is loaded, since FW does not match */
	if (qfle3_prev_is_after_undi(adapter))
		goto out;

	/* If function has FLR capabilities, and existing FW version matches
	 * the one required, then FLR will be sufficient to clean any residue
	 * left by previous driver
	 */
	status = qfle3_compare_fw_ver(adapter,
			FW_MSG_CODE_DRV_LOAD_FUNCTION);

	if (status == VMK_OK) {
		/* fw version is good */
		QFLE3_DBG(QFLE3_DBG_LOAD, "FW version matches our own. Attempting FLR\n");
		status = qfle3_do_flr(adapter);
	}

	if (status == VMK_OK) {
		/* FLR was performed */
		QFLE3_DBG(QFLE3_DBG_LOAD, "FLR successful\n");
		return VMK_OK;
	}

	QFLE3_INFO("Could not FLR\n");
out:
	/* Close the MCP request, return failure*/
	status = qfle3_prev_mcp_done(adapter);
	if (status != VMK_OK)
		status = VMK_STATUS_PENDING;

	return status;
}
vmk_Bool qfle3_isFirstFuncOnDev(qfle3_adapter * adapter, vmk_PCIDeviceAddr pciaddr)
{
	struct vmk_ListLinks *item;
   qfle3_adapter *list_entry;
   VMK_LIST_FORALL (&qfle3_mod_info.adapterList, item) {
      
      list_entry = VMK_LIST_ENTRY(item, struct qfle3_adapter, adapterLink);
      QFLE3_DBG(QFLE3_DBG_LOAD, "compare adapter addr 0x%04x:%02x%02x.%d to 0x%04x:%02x%02x.%d\n",pciaddr.seg, pciaddr.bus, pciaddr.dev,pciaddr.fn, 
      list_entry->pdev_addr.seg,list_entry->pdev_addr.bus,list_entry->pdev_addr.dev,list_entry->pdev_addr.fn);
      if ((pciaddr.seg == list_entry->pdev_addr.seg) &&
         (pciaddr.bus == list_entry->pdev_addr.bus) &&
         (pciaddr.dev == list_entry->pdev_addr.dev)){
         QFLE3_DBG(QFLE3_DBG_LOAD, "Not first function on Dev  0x%04x:%02x%02x.%d \n",pciaddr.seg, pciaddr.bus, pciaddr.dev,pciaddr.fn);
         return VMK_FALSE;
      }
      
   }
   
   QFLE3_DBG(QFLE3_DBG_LOAD, "First function on Dev\n");
   return VMK_TRUE;
}

vmk_Bool qfle3_isFirstFuncOnDevPath(qfle3_adapter * adapter, vmk_PCIDeviceAddr pciaddr)
{
	struct vmk_ListLinks *item;
   qfle3_adapter *list_entry;
   VMK_LIST_FORALL (&qfle3_mod_info.adapterList, item) {
      
      list_entry = VMK_LIST_ENTRY(item, struct qfle3_adapter, adapterLink);
      QFLE3_DBG(QFLE3_DBG_LOAD, "compare adapter addr 0x%04x:%02x%02x.%d to 0x%04x:%02x%02x.%d\n",pciaddr.seg, pciaddr.bus, pciaddr.dev,pciaddr.fn, 
      list_entry->pdev_addr.seg,list_entry->pdev_addr.bus,list_entry->pdev_addr.dev,list_entry->pdev_addr.fn);
      if ((pciaddr.seg == list_entry->pdev_addr.seg) &&
         (pciaddr.bus == list_entry->pdev_addr.bus) &&
         (pciaddr.dev == list_entry->pdev_addr.dev) &&
         (QFLE3_PATH(adapter) == QFLE3_PATH(list_entry))){
         QFLE3_DBG(QFLE3_DBG_LOAD, "Not first function on Dev Path 0x%04x:%02x%02x.%d  Path %d\n",pciaddr.seg, pciaddr.bus, pciaddr.dev,pciaddr.fn, QFLE3_PATH(adapter));
         return VMK_FALSE;
      }
      
   }
   
   QFLE3_DBG(QFLE3_DBG_LOAD, "First function on Dev Path\n");
   return VMK_TRUE;
}
static VMK_ReturnStatus
qfle3_cleanup_prev_unload_state(qfle3_adapter * adapter)
{
	VMK_ReturnStatus status = VMK_OK;
	vmk_int32 time_counter = 10;
	vmk_uint32 fw, hw_lock_reg, hw_lock_val;

	QFLE3_INFO("Entering Previous Unload Flow\n");

	/* clear hw from errors which may have resulted from an interrupted
	 * dmae transaction.
	 */
	qfle3_clean_pglue_errors(adapter);

	/* Release previously held locks */
	hw_lock_reg = (QFLE3_FUNC(adapter) <= 5) ?
		(MISC_REG_DRIVER_CONTROL_1 + QFLE3_FUNC(adapter) * 8) :
		(MISC_REG_DRIVER_CONTROL_7 + (QFLE3_FUNC(adapter) - 6) * 8);

	hw_lock_val = REG_RD(adapter, hw_lock_reg);
	if (hw_lock_val) {
		if (hw_lock_val & HW_LOCK_RESOURCE_NVRAM) {
			QFLE3_DBG(QFLE3_DBG_LOAD, "Release Previously held NVRAM lock\n");
			REG_WR(adapter, MCP_REG_MCPR_NVM_SW_ARB,
					(MCPR_NVM_SW_ARB_ARB_REQ_CLR1 << QFLE3_PORT(adapter)));
		}

		QFLE3_DBG(QFLE3_DBG_LOAD, "Release Previously held hw lock\n");
		REG_WR(adapter, hw_lock_reg, 0xffffffff);
	} else
		QFLE3_DBG(QFLE3_DBG_LOAD, "No need to release hw/nvram locks\n");

	if (MCPR_ACCESS_LOCK_LOCK & REG_RD(adapter, MCP_REG_MCPR_ACCESS_LOCK)) {
		QFLE3_DBG(QFLE3_DBG_LOAD, "Release previously held alr\n");
		qfle3_release_alr(adapter);
	}


	do {
		vmk_int32 aer = 0;
		/* Lock MCP using an unload request */
		fw = qfle3_mfw_command(adapter, DRV_MSG_CODE_UNLOAD_REQ_WOL_DIS, 0);
		if (!fw) {
			QFLE3_ERR("MCP response failure, aborting\n");
			status = VMK_BUSY;
			break;
		}

		status = vmk_SemaTryLock(&qfle3_mod_info.prev_mutex);
		if (status != VMK_OK) {
			QFLE3_ERR("Cannot check for AER; Failed to take lock\n");
		} else {
			/* If Path is marked by EEH, ignore unload status */
			aer = !!(qfle3_prev_path_get_entry(adapter) &&
					qfle3_prev_path_get_entry(adapter)->aer);
			vmk_SemaUnlock(&qfle3_mod_info.prev_mutex);
		}
      
      QFLE3_DBG(QFLE3_DBG_LOAD, "Unloading on PCI device 0x%04x:%02x%02x.%d ", adapter->pdev_addr.seg,adapter->pdev_addr.bus,
         adapter->pdev_addr.dev, adapter->pdev_addr.fn);      
      QFLE3_DBG(QFLE3_DBG_LOAD, "Unload code 0x%x \n", fw);

      // BFS work around
      if (adapter->recovery_state == QFLE3_RECOVERY_DONE) {
         if (qfle3_isFirstFuncOnDevPath(adapter,adapter->pdev_addr))
            if (fw != FW_MSG_CODE_DRV_UNLOAD_COMMON) {
               QFLE3_DBG(QFLE3_DBG_LOAD, "Unload code on First funcion on Path %d is not UNLOAD_COMMON, forcing to UNLOAD_COMMON\n",
                  QFLE3_PATH(adapter));
               status = qfle3_prev_unload_common(adapter);
               break;
            }
      }

      // normal flow of logic   
      if (fw == FW_MSG_CODE_DRV_UNLOAD_COMMON || aer){
         status = qfle3_prev_unload_common(adapter);
         break;
      }

      /* non-common reply from MCP might require looping */
      status = qfle3_prev_unload_uncommon(adapter);
      if (status != VMK_STATUS_PENDING)
         break;

		vmk_WorldSleep(20 * 1000);
	} while (--time_counter);

	if (!time_counter || status != VMK_OK) {
		QFLE3_INFO("Unloading previous driver did not occur, Possibly due to MF UNDI\n");
		status = VMK_FAILURE;
	}

	/* Mark function if its port was used to boot from SAN */
	if (qfle3_port_after_undi(adapter))
		adapter->link_params.feature_config_flags |=
			ELINK_FEATURE_CONFIG_BOOT_FROM_SAN;

	QFLE3_DBG(QFLE3_DBG_LOAD, "Finished Previous Unload Flow [%d]\n", status);

	return status;
}

static void
qfle3_set_init_mode_flags(struct qfle3_adapter *adapter)
{
   vmk_uint32 flags = 0;

   if (CHIP_REV_IS_FPGA(adapter))
      SET_FLAGS(flags, MODE_FPGA);
   else if (CHIP_REV_IS_EMUL(adapter))
      SET_FLAGS(flags, MODE_EMUL);
   else
      SET_FLAGS(flags, MODE_ASIC);

   if (CHIP_MODE_IS_4_PORT(adapter))
      SET_FLAGS(flags, MODE_PORT4);
   else
      SET_FLAGS(flags, MODE_PORT2);

   if (CHIP_IS_E2(adapter))
      SET_FLAGS(flags, MODE_E2);
   else if (CHIP_IS_E3(adapter)) {
      SET_FLAGS(flags, MODE_E3);
      if (CHIP_REV(adapter) == CHIP_REV_Ax)
	 SET_FLAGS(flags, MODE_E3_A0);
      else                      /*if (CHIP_REV(adapter) == CHIP_REV_Bx) */
	 SET_FLAGS(flags, MODE_E3_B0 | MODE_COS3);
   }

   if (IS_MF(adapter)) {
      SET_FLAGS(flags, MODE_MF);
      switch (adapter->mf_mode) {
      case MULTI_FUNCTION_SD:
	 SET_FLAGS(flags, MODE_MF_SD);
	 break;
      case MULTI_FUNCTION_SI:
	 SET_FLAGS(flags, MODE_MF_SI);
	 break;
      case MULTI_FUNCTION_AFEX:
	 SET_FLAGS(flags, MODE_MF_AFEX);
	 break;
      }
   } else
      SET_FLAGS(flags, MODE_SF);

#if defined(__LITTLE_ENDIAN)
   SET_FLAGS(flags, MODE_LITTLE_ENDIAN);
#else                           /*(__BIG_ENDIAN) */
   SET_FLAGS(flags, MODE_BIG_ENDIAN);
#endif
   INIT_MODE_FLAGS(adapter) = flags;
}

void qfle3_update_max_mf_config(qfle3_adapter *adapter, vmk_uint32 value)
{
    /* load old values */
    vmk_uint32 mf_cfg = adapter->mf_config[QFLE3_VN(adapter)];

    if (value != qfle3_extract_max_cfg(adapter, mf_cfg)) {
        /* leave all but MAX value */
        mf_cfg &= ~FUNC_MF_CFG_MAX_BW_MASK;

        /* set new MAX value */
        mf_cfg |= (value << FUNC_MF_CFG_MAX_BW_SHIFT)
                & FUNC_MF_CFG_MAX_BW_MASK;

        qfle3_mfw_command(adapter, DRV_MSG_CODE_SET_MF_BW, mf_cfg);
    }
}

static void validate_set_si_mode(qfle3_adapter * adapter)
{
    vmk_uint8 func = QFLE3_ABS_FUNC(adapter);
    vmk_uint32 val;

    val = MF_CFG_RD(adapter, func_mf_config[func].mac_upper);

    /* check for legal mac (upper bytes)*/
    if (val != 0xffff) {
        adapter->mf_mode = MULTI_FUNCTION_SI;
        adapter->mf_config[QFLE3_VN(adapter)] =
            MF_CFG_RD(adapter, func_mf_config[func].config);
    } else
        QFLE3_INFO("illegal MAC address for SI\n");
}

static void
qfle3_link_settings_supported(struct qfle3_adapter *adapter, vmk_uint32 switch_cfg)
{
   vmk_uint32 cfg_size = 0, idx, port = QFLE3_PORT(adapter);

   /*
    * Aggregation of supported attributes of all external phys
    */
   adapter->port.supported[0] = 0;
   adapter->port.supported[1] = 0;
   switch (adapter->link_params.num_phys) {
   case 1:
      adapter->port.supported[0] = adapter->link_params.phy[ELINK_INT_PHY].supported;
      cfg_size = 1;
      break;
   case 2:
      adapter->port.supported[0] = adapter->link_params.phy[ELINK_EXT_PHY1].supported;
      cfg_size = 1;
      break;
   case 3:
      if (adapter->link_params.multi_phy_config & PORT_HW_CFG_PHY_SWAPPED_ENABLED) {
	 adapter->port.supported[1] = adapter->link_params.phy[ELINK_EXT_PHY1].supported;
	 adapter->port.supported[0] = adapter->link_params.phy[ELINK_EXT_PHY2].supported;
      } else {
	 adapter->port.supported[0] = adapter->link_params.phy[ELINK_EXT_PHY1].supported;
	 adapter->port.supported[1] = adapter->link_params.phy[ELINK_EXT_PHY2].supported;
      }
      cfg_size = 2;
      break;
   }
   if (!(adapter->port.supported[0] || adapter->port.supported[1])) {
      QFLE3_ERR
	 ("NVRAM config error. BAD phy config. PHY1 config 0x%x, PHY2 config 0x%x\n",
	  SHMEM_RD(adapter, dev_info.port_hw_config[port].external_phy_config),
	  SHMEM_RD(adapter, dev_info.port_hw_config[port].external_phy_config2));
      return;
   }

   if (CHIP_IS_E3(adapter))
      adapter->port.phy_addr = REG_RD(adapter, MISC_REG_WC0_CTRL_PHY_ADDR);
   else {
      switch (switch_cfg) {
      case ELINK_SWITCH_CFG_1G:
	 adapter->port.phy_addr =
	    REG_RD(adapter, NIG_REG_SERDES0_CTRL_PHY_ADDR + port * 0x10);
	 break;
      case ELINK_SWITCH_CFG_10G:
	 adapter->port.phy_addr =
	    REG_RD(adapter, NIG_REG_XGXS0_CTRL_PHY_ADDR + port * 0x18);
	 break;
      default:
	 QFLE3_ERR("BAD switch_cfg link_config 0x%x\n", adapter->port.link_config[0]);
	 return;
      }
   }
   QFLE3_DBG(QFLE3_DBG_LOAD, "phy_addr 0x%x\n", adapter->port.phy_addr);
   /*
    * mask what we support according to speed_cap_mask per configuration
    */
   for (idx = 0; idx < cfg_size; idx++) {
      if (!(adapter->link_params.speed_cap_mask[idx] &
	    PORT_HW_CFG_SPEED_CAPABILITY_D0_10M_HALF))
	 adapter->port.supported[idx] &= ~ELINK_SUPPORTED_10baseT_Half;

      if (!(adapter->link_params.speed_cap_mask[idx] &
	    PORT_HW_CFG_SPEED_CAPABILITY_D0_10M_FULL))
	 adapter->port.supported[idx] &= ~ELINK_SUPPORTED_10baseT_Full;

      if (!(adapter->link_params.speed_cap_mask[idx] &
	    PORT_HW_CFG_SPEED_CAPABILITY_D0_100M_HALF))
	 adapter->port.supported[idx] &= ~ELINK_SUPPORTED_100baseT_Half;

      if (!(adapter->link_params.speed_cap_mask[idx] &
	    PORT_HW_CFG_SPEED_CAPABILITY_D0_100M_FULL))
	 adapter->port.supported[idx] &= ~ELINK_SUPPORTED_100baseT_Full;

      if (!(adapter->link_params.speed_cap_mask[idx] &
	    PORT_HW_CFG_SPEED_CAPABILITY_D0_1G))
#if 1
	 adapter->port.supported[idx] &= ~(ELINK_SUPPORTED_1000baseT_Full);
#else
      adapter->port.supported[idx] &= ~(ELINK_SUPPORTED_1000baseT_Half |
					ELINK_SUPPORTED_1000baseT_Full);
#endif

      if (!(adapter->link_params.speed_cap_mask[idx] &
	    PORT_HW_CFG_SPEED_CAPABILITY_D0_2_5G))
	 adapter->port.supported[idx] &= ~ELINK_SUPPORTED_2500baseX_Full;

      if (!(adapter->link_params.speed_cap_mask[idx] &
	    PORT_HW_CFG_SPEED_CAPABILITY_D0_10G))
	 adapter->port.supported[idx] &= ~ELINK_SUPPORTED_10000baseT_Full;

      if (!(adapter->link_params.speed_cap_mask[idx] &
	    PORT_HW_CFG_SPEED_CAPABILITY_D0_20G))
	 adapter->port.supported[idx] &= ~ELINK_SUPPORTED_20000baseKR2_Full;
   }

   QFLE3_DBG(QFLE3_DBG_LOAD, "supported 0x%x 0x%x\n", adapter->port.supported[0],
	      adapter->port.supported[1]);
}

static void
qfle3_link_settings_requested(struct qfle3_adapter *adapter)
{
   vmk_uint32 link_config, idx, cfg_size = 0;

   adapter->port.advertising[0] = 0;
   adapter->port.advertising[1] = 0;
   switch (adapter->link_params.num_phys) {
   case 1:
   case 2:
      cfg_size = 1;
      break;
   case 3:
      cfg_size = 2;
      break;
   }
   for (idx = 0; idx < cfg_size; idx++) {
      adapter->link_params.req_duplex[idx] = DUPLEX_FULL;
      link_config = adapter->port.link_config[idx];
      switch (link_config & PORT_FEATURE_LINK_SPEED_MASK) {
      case PORT_FEATURE_LINK_SPEED_AUTO:
	 if (adapter->port.supported[idx] & ELINK_SUPPORTED_Autoneg) {
	    adapter->link_params.req_line_speed[idx] = ELINK_SPEED_AUTO_NEG;
	    adapter->port.advertising[idx] |= adapter->port.supported[idx];
	    if (adapter->link_params.phy[ELINK_EXT_PHY1].type ==
		PORT_HW_CFG_XGXS_EXT_PHY_TYPE_BCM84833)
	       adapter->port.advertising[idx] |=
		  (ELINK_SUPPORTED_100baseT_Half | ELINK_SUPPORTED_100baseT_Full);
	 } else {
	    /*
	     * force 10G, no AN
	     */
	    adapter->link_params.req_line_speed[idx] = ELINK_SPEED_10000;
	    adapter->port.advertising[idx] |=
	       (ELINK_ADVERTISED_10000baseT_Full | ELINK_ADVERTISED_FIBRE);
	    continue;
	 }
	 break;
      case PORT_FEATURE_LINK_SPEED_10M_FULL:
	 if (adapter->port.supported[idx] & ELINK_SUPPORTED_10baseT_Full) {
	    adapter->link_params.req_line_speed[idx] = ELINK_SPEED_10;
	    adapter->port.advertising[idx] |=
	       (ELINK_ADVERTISED_10baseT_Full | ELINK_ADVERTISED_TP);
	 } else {
	    QFLE3_ERR
	       ("NVRAM config error. Invalid link_config 0x%x  speed_cap_mask 0x%x\n",
		link_config, adapter->link_params.speed_cap_mask[idx]);
	    return;
	 }
	 break;

      case PORT_FEATURE_LINK_SPEED_10M_HALF:
	 if (adapter->port.supported[idx] & ELINK_SUPPORTED_10baseT_Half) {
	    adapter->link_params.req_line_speed[idx] = ELINK_SPEED_10;
	    adapter->link_params.req_duplex[idx] = DUPLEX_HALF;
	    adapter->port.advertising[idx] |=
	       (ELINK_ADVERTISED_10baseT_Half | ELINK_ADVERTISED_TP);
	 } else {
	    QFLE3_ERR
	       ("NVRAM config error. Invalid link_config 0x%x  speed_cap_mask 0x%x\n",
		link_config, adapter->link_params.speed_cap_mask[idx]);
	    return;
	 }
	 break;

      case PORT_FEATURE_LINK_SPEED_100M_FULL:
	 if (adapter->port.supported[idx] & ELINK_SUPPORTED_100baseT_Full) {
	    adapter->link_params.req_line_speed[idx] = ELINK_SPEED_100;
	    adapter->port.advertising[idx] |=
	       (ELINK_ADVERTISED_100baseT_Full | ELINK_ADVERTISED_TP);
	 } else {
	    QFLE3_ERR
	       ("NVRAM config error. Invalid link_config 0x%x  speed_cap_mask 0x%x\n",
		link_config, adapter->link_params.speed_cap_mask[idx]);
	    return;
	 }
	 break;

      case PORT_FEATURE_LINK_SPEED_100M_HALF:
	 if (adapter->port.supported[idx] & ELINK_SUPPORTED_100baseT_Half) {
	    adapter->link_params.req_line_speed[idx] = ELINK_SPEED_100;
	    adapter->link_params.req_duplex[idx] = DUPLEX_HALF;
	    adapter->port.advertising[idx] |=
	       (ELINK_ADVERTISED_100baseT_Half | ELINK_ADVERTISED_TP);
	 } else {
	    QFLE3_ERR
	       ("NVRAM config error. Invalid link_config 0x%x  speed_cap_mask 0x%x\n",
		link_config, adapter->link_params.speed_cap_mask[idx]);
	    return;
	 }
	 break;

      case PORT_FEATURE_LINK_SPEED_1G:
	 if (adapter->port.supported[idx] & ELINK_SUPPORTED_1000baseT_Full) {
	    adapter->link_params.req_line_speed[idx] = ELINK_SPEED_1000;
	    adapter->port.advertising[idx] |=
	       (ELINK_ADVERTISED_1000baseT_Full | ELINK_ADVERTISED_TP);
	 } else if (adapter->port.supported[idx] & ELINK_SUPPORTED_1000baseKX_Full) {
	    adapter->link_params.req_line_speed[idx] = ELINK_SPEED_1000;
	    adapter->port.advertising[idx] |= ELINK_ADVERTISED_1000baseKX_Full;
	 } else {
	    QFLE3_ERR
	       ("NVRAM config error. Invalid link_config 0x%x  speed_cap_mask 0x%x\n",
		link_config, adapter->link_params.speed_cap_mask[idx]);
	    return;
	 }
	 break;

      case PORT_FEATURE_LINK_SPEED_2_5G:
	 if (adapter->port.supported[idx] & ELINK_SUPPORTED_2500baseX_Full) {
	    adapter->link_params.req_line_speed[idx] = ELINK_SPEED_2500;
	    adapter->port.advertising[idx] |=
	       (ELINK_ADVERTISED_2500baseX_Full | ELINK_ADVERTISED_TP);
	 } else {
	    QFLE3_ERR
	       ("NVRAM config error. Invalid link_config 0x%x  speed_cap_mask 0x%x\n",
		link_config, adapter->link_params.speed_cap_mask[idx]);
	    return;
	 }
	 break;
      case PORT_FEATURE_LINK_SPEED_10G_CX4:
	 if (adapter->port.supported[idx] & ELINK_SUPPORTED_10000baseT_Full) {
	    adapter->link_params.req_line_speed[idx] = ELINK_SPEED_10000;
	    adapter->port.advertising[idx] |=
	       (ELINK_ADVERTISED_10000baseT_Full | ELINK_ADVERTISED_FIBRE);
	 } else if (adapter->port.supported[idx] & ELINK_SUPPORTED_10000baseKR_Full) {
	    adapter->link_params.req_line_speed[idx] = ELINK_SPEED_10000;
	    adapter->port.advertising[idx] |=
	       (ELINK_ADVERTISED_10000baseKR_Full | ELINK_ADVERTISED_FIBRE);
	 } else {
	    QFLE3_ERR
	       ("NVRAM config error. Invalid link_config 0x%x  speed_cap_mask 0x%x\n",
		link_config, adapter->link_params.speed_cap_mask[idx]);
	    return;
	 }
	 break;
      case PORT_FEATURE_LINK_SPEED_20G:
	 adapter->link_params.req_line_speed[idx] = ELINK_SPEED_20000;

	 break;
      default:
	 QFLE3_ERR("NVRAM config error. BAD link speed link_config 0x%x\n",
		   link_config);
	 adapter->link_params.req_line_speed[idx] = ELINK_SPEED_AUTO_NEG;
	 adapter->port.advertising[idx] = adapter->port.supported[idx];
	 break;
      }
      adapter->link_params.req_flow_ctrl[idx] = (link_config &
						 PORT_FEATURE_FLOW_CONTROL_MASK);
      if (adapter->link_params.req_flow_ctrl[idx] == ELINK_FLOW_CTRL_AUTO) {
	 if (!(adapter->port.supported[idx] & ELINK_SUPPORTED_Autoneg))
	    adapter->link_params.req_flow_ctrl[idx] = ELINK_FLOW_CTRL_NONE;
	 else
	    adapter->link_params.req_fc_auto_adv = ELINK_FLOW_CTRL_BOTH;

      }

      QFLE3_DBG(QFLE3_DBG_LOAD, "req_line_speed %d  req_duplex %d req_flow_ctrl 0x%x advertising 0x%x\n",
	  adapter->link_params.req_line_speed[idx],
	  adapter->link_params.req_duplex[idx], adapter->link_params.req_flow_ctrl[idx],
	  adapter->port.advertising[idx]);
   }
}

static void
qfle3_get_link_port_config(qfle3_adapter * adapter)
{
   vmk_uint32 port = QFLE3_PORT(adapter);
   vmk_uint32 config;
   vmk_uint32 ext_phy_type;
   vmk_uint32 ext_phy_config;
   vmk_uint32 eee_mode;

   adapter->link_params.adapter = adapter;
   adapter->link_params.port = port;

   adapter->link_params.lane_config =
      SHMEM_RD(adapter, dev_info.port_hw_config[port].lane_config);

   adapter->link_params.speed_cap_mask[0] =
      SHMEM_RD(adapter,
	       dev_info.port_hw_config[port].speed_capability_mask) &
      PORT_HW_CFG_SPEED_CAPABILITY_D0_MASK;

   adapter->link_params.speed_cap_mask[1] =
      SHMEM_RD(adapter,
	       dev_info.port_hw_config[port].speed_capability_mask2) &
      PORT_HW_CFG_SPEED_CAPABILITY_D0_MASK;

   adapter->port.link_config[0] =
      SHMEM_RD(adapter, dev_info.port_feature_config[port].link_config);

   adapter->port.link_config[1] =
      SHMEM_RD(adapter, dev_info.port_feature_config[port].link_config2);

   adapter->link_params.multi_phy_config =
      SHMEM_RD(adapter, dev_info.port_hw_config[port].multi_phy_config);

   config = SHMEM_RD(adapter, dev_info.port_feature_config[port].config);

   adapter->wol = (!(adapter->flags & NO_WOL_FLAG) &&
         (config & PORT_FEATURE_WOL_ENABLED));
//TODO: uncomment this out
#if 0
   if ((config & PORT_FEAT_CFG_STORAGE_PERSONALITY_MASK) ==
         PORT_FEAT_CFG_STORAGE_PERSONALITY_FCOE && !IS_MF(adapter))
      adapter->flags |= NO_ISCSI_FLAG | NO_ISCSI_OOO_FLAG;
   if ((config & PORT_FEAT_CFG_STORAGE_PERSONALITY_MASK) ==
         PORT_FEAT_CFG_STORAGE_PERSONALITY_ISCSI && !IS_MF(adapter))
      adapter->flags |= NO_FCOE_FLAG;
#endif
   adapter->link_params.switch_cfg = (adapter->port.link_config[0] &
         PORT_FEATURE_CONNECTED_SWITCH_MASK);




   elink_phy_probe(&adapter->link_params);

   qfle3_link_settings_supported(adapter, adapter->link_params.switch_cfg);

   qfle3_link_settings_requested(adapter);

   ext_phy_config = SHMEM_RD(adapter, dev_info.port_hw_config[port].external_phy_config);
   ext_phy_type = ELINK_XGXS_EXT_PHY_TYPE(ext_phy_config);
#if 0
   if (ext_phy_type == PORT_HW_CFG_XGXS_EXT_PHY_TYPE_DIRECT)
      adapter->mdio.prtad = adapter->port.phy_addr;

   else if ((ext_phy_type != PORT_HW_CFG_XGXS_EXT_PHY_TYPE_FAILURE) &&
	    (ext_phy_type != PORT_HW_CFG_XGXS_EXT_PHY_TYPE_NOT_CONN))
      adapter->mdio.prtad = ELINK_XGXS_EXT_PHY_ADDR(ext_phy_config);
#endif

   if (config & PORT_FEAT_CFG_AUTOGREEEN_ENABLED)
      adapter->link_params.feature_config_flags |=
	 ELINK_FEATURE_CONFIG_AUTOGREEEN_ENABLED;

   eee_mode = (((SHMEM_RD(adapter, dev_info.port_feature_config[port].eee_power_mode)) &
		PORT_FEAT_CFG_EEE_POWER_MODE_MASK) >>
	       PORT_FEAT_CFG_EEE_POWER_MODE_SHIFT);
   if (eee_mode != PORT_FEAT_CFG_EEE_POWER_MODE_DISABLED) {
      adapter->link_params.eee_mode = ELINK_EEE_MODE_ADV_LPI |
	 ELINK_EEE_MODE_ENABLE_LPI | ELINK_EEE_MODE_OUTPUT_TIME;
   } else {
      adapter->link_params.eee_mode = 0;
   }
}

static VMK_ReturnStatus
qfle3_get_mf_config(qfle3_adapter * adapter)
{
   vmk_uint32 val,val2;
   vmk_uint32 vn;
   VMK_ReturnStatus status = VMK_OK;
   int /*abs*/func = QFLE3_ABS_FUNC(adapter);

   vn = QFLE3_VN(adapter);

   if (SHMEM2_HAS(adapter, mf_cfg_addr))
      adapter->hw_info.mf_cfg_base = SHMEM2_RD(adapter, mf_cfg_addr);
   else
      adapter->hw_info.mf_cfg_base = adapter->hw_info.shmem_base +
              vmk_offsetof(struct shmem_region, func_mb) +
              E1H_FUNC_MAX * sizeof(struct drv_func_mb);

   /*
   * Initialize MF configuration
   */
   adapter->mf_ov = 0;
   adapter->mf_mode = 0;
   adapter->mf_sub_mode = 0;

   if (adapter->hw_info.mf_cfg_base != SHMEM_MF_CFG_ADDR_NONE) {
      /*
       * get mf configuration
       */
      val = SHMEM_RD(adapter, dev_info.shared_feature_config.config);
      val &= SHARED_FEAT_CFG_FORCE_SF_MODE_MASK;

      switch (val) {
      case SHARED_FEAT_CFG_FORCE_SF_MODE_SWITCH_INDEPT:
		validate_set_si_mode(adapter);
	    break;
      case SHARED_FEAT_CFG_FORCE_SF_MODE_AFEX_MODE:
             if ((MF_CFG_RD(adapter, func_mf_config[func].mac_upper) != 0xffff) &&
                           (SHMEM2_HAS(adapter, afex_driver_support))) {
                     adapter->mf_mode = MULTI_FUNCTION_AFEX;
                     adapter->mf_config[vn] = MF_CFG_RD(adapter, func_mf_config[func].config);
         } else {
                     QFLE3_WARN("can not configure afex mode\n");
         }
         break;

      case SHARED_FEAT_CFG_FORCE_SF_MODE_MF_ALLOWED:
             val = MF_CFG_RD(adapter, func_mf_config[FUNC_0].e1hov_tag);
             val &= FUNC_MF_CFG_E1HOV_TAG_MASK;

             if (val != FUNC_MF_CFG_E1HOV_TAG_DEFAULT) {
                  adapter->mf_mode = MULTI_FUNCTION_SD;
                  adapter->mf_config[vn] = MF_CFG_RD(adapter, func_mf_config[func].config);
         } else
                  QFLE3_WARN("Illegal OV for SD\n");
         break;

      case SHARED_FEAT_CFG_FORCE_SF_MODE_BD_MODE:
             adapter->mf_mode = MULTI_FUNCTION_SD;
         adapter->mf_sub_mode = SUB_MF_MODE_BD;
         adapter->mf_config[vn] = MF_CFG_RD(adapter, func_mf_config[func].config);

         if (SHMEM2_HAS(adapter, mtu_size)) {
                  int mtu_idx = QFLE3_FW_MB_IDX(adapter);
            vmk_uint16 mtu_size;
            vmk_uint16 mtu;

            mtu = SHMEM2_RD(adapter, mtu_size[mtu_idx]);
            mtu_size = (vmk_uint16)mtu;
            QFLE3_DBG(QFLE3_DBG_LOAD, "Read MTU size %04x [%08x]\n", mtu_size, mtu);

         /* if valid: update device mtu */
            if (((mtu_size + ETH_HLEN) >= ETH_MIN_PACKET_SIZE) &&
                  (mtu_size <= ETH_MAX_JUMBO_PACKET_SIZE))
               adapter->mtu = mtu_size;
         }
         break;

      case SHARED_FEAT_CFG_FORCE_SF_MODE_UFP_MODE:
         adapter->mf_mode = MULTI_FUNCTION_SD;
         adapter->mf_sub_mode = SUB_MF_MODE_UFP;
         adapter->mf_config[vn] = MF_CFG_RD(adapter, func_mf_config[func].config);
         break;

      case SHARED_FEAT_CFG_FORCE_SF_MODE_FORCED_SF:
	 adapter->mf_info.mf_config[vn] = 0;
	 break;
      case SHARED_FEAT_CFG_FORCE_SF_MODE_EXTENDED_MODE:
	     val2 = SHMEM_RD(adapter,
                    dev_info.shared_hw_config.config_3);
                val2 &= SHARED_HW_CFG_EXTENDED_MF_MODE_MASK;
                switch (val2) {
                case SHARED_HW_CFG_EXTENDED_MF_MODE_NPAR1_DOT_5:
                    validate_set_si_mode(adapter);
                    adapter->mf_sub_mode =
                            SUB_MF_MODE_NPAR1_DOT_5;
                    break;
                default:
                    /* Unknown configuration */
                    adapter->mf_config[vn] = 0;
                    QFLE3_ERR("unknown extended MF mode 0x%x\n",
                               val);
                }
	 break;
      default:
	 /*
	  * Unknown configuration: reset mf_config
	  */
	 adapter->mf_info.mf_config[vn] = 0;
	 QFLE3_INFO("unknown MF mode 0x%x\n", val);
      }
   }
   switch (adapter->mf_mode) {
   case MULTI_FUNCTION_SD:
      val = MF_CFG_RD(adapter, func_mf_config[func].e1hov_tag) &
              FUNC_MF_CFG_E1HOV_TAG_MASK;
      if (val != FUNC_MF_CFG_E1HOV_TAG_DEFAULT) {
         adapter->mf_ov = val;
         adapter->path_has_ovlan = VMK_TRUE;

         QFLE3_DBG(QFLE3_DBG_LOAD, "MF OV for func %d is %d (0x%04x)\n",
                    func, adapter->mf_ov, adapter->mf_ov);
      } else if ((adapter->mf_sub_mode == SUB_MF_MODE_UFP) ||
         (adapter->mf_sub_mode == SUB_MF_MODE_BD)) {
              QFLE3_ERR("Unexpected - no valid MF OV for func %d in UFP/BD mode\n",
                              func);
              adapter->path_has_ovlan = VMK_TRUE;
      } else {
         QFLE3_ERR("No valid MF OV for func %d, aborting\n", func);
                           return VMK_FAILURE;
      }
      break;
   case MULTI_FUNCTION_AFEX:
      QFLE3_INFO("func %d is in MF afex mode\n", func);
      break;
   case MULTI_FUNCTION_SI:
      QFLE3_INFO("func %d is in MF switch-independent mode\n", func);
      break;
   default:
      if (vn) {
         QFLE3_ERR("VN %d is in a single function mode, aborting\n", vn);
              return VMK_FAILURE;
      }
      break;
   }

   /* check if other port on the path needs ovlan:
    * Since MF configuration is shared between ports
    * Possible mixed modes are only
    * {SF, SI} {SF, SD} {SD, SF} {SI, SF}
    */
   if (CHIP_MODE_IS_4_PORT(adapter) &&
       !adapter->path_has_ovlan && !IS_MF(adapter) &&
       adapter->hw_info.mf_cfg_base != SHMEM_MF_CFG_ADDR_NONE) {
      vmk_uint8 other_port = !QFLE3_PORT(adapter);
      vmk_uint8 other_func = QFLE3_PATH(adapter) + 2*other_port;
      val = MF_CFG_RD(adapter, func_mf_config[other_func].e1hov_tag);
      if (val != FUNC_MF_CFG_E1HOV_TAG_DEFAULT)
         adapter->path_has_ovlan = VMK_TRUE;
   }

   QFLE3_INFO("%s function mode\n",
                   IS_MF(adapter) ? "multi" : "single");

#define NO_MCP_WA_CFG_SET_ADDR          (0xA0000)
#define NO_MCP_WA_CFG_SET_MAGIC         (0x88AA55FF)
#define NO_MCP_WA_MULTI_VNIC_MODE       (0xA0004)
#define NO_MCP_WA_VNICS_PER_PORT(port)  (0xA0008 + 4*(port))
#define NO_MCP_WA_OVLAN(func)           (0xA0010 + 4*(func))
#define NO_MCP_WA_FORCE_5710            (0xA0030)
#define NO_MCP_WA_VALID_LIC_ADDR        (0xA0040)
#define NO_MCP_WA_VALID_LIC_MAGIC       (0xCCAAFFEE)
#define NO_MCP_WA_TOE_LIC               (0xA0048)
#define NO_MCP_WA_ISCSI_LIC             (0xA0050)
#define NO_MCP_WA_RDMA_LICv         (0xA0058)
#define NO_MCP_WA_CLC_SHMEM             (0xAF900)

    if (QFLE3_NOMCP(adapter) && !CHIP_IS_E1x(adapter)) {
        val = REG_RD(adapter,  /*adapter->common.shmem_base*/ + NO_MCP_WA_CFG_SET_ADDR);
        /* should have a magic number written if configuration was set otherwise, use default above */
        if (val == NO_MCP_WA_CFG_SET_MAGIC) {
            val = REG_RD(adapter, /*adapter->common.shmem_base*/ + NO_MCP_WA_FORCE_5710);
            if (!val) {
                vmk_uint32 vnic_mode = REG_RD(adapter, NO_MCP_WA_MULTI_VNIC_MODE);
                vnic_mode &= SHARED_FEAT_CFG_FORCE_SF_MODE_MASK;
                switch (vnic_mode) {
                case SHARED_FEAT_CFG_FORCE_SF_MODE_SWITCH_INDEPT:
                    if (MF_CFG_RD(adapter, func_mf_config[func].mac_upper) != 0xffff)
                        adapter->mf_mode = MULTI_FUNCTION_SI;
                    else
                        QFLE3_ERR("No valid MAC for func %d\n",
                            func);

                           break;
                case SHARED_FEAT_CFG_FORCE_SF_MODE_MF_ALLOWED:
                   break;
               default:
                    /*SF*/
                    break;
                }

                if (CHIP_MODE_IS_4_PORT(adapter) &&
                    !adapter->path_has_ovlan &&
                    !IS_MF(adapter)) {
                    vmk_uint8 other_port = !QFLE3_PORT(adapter);
                    vmk_uint8 other_func = QFLE3_PATH(adapter) + 2*other_port;
                    val = REG_RD(adapter, NO_MCP_WA_OVLAN(other_func)) & FUNC_MF_CFG_E1HOV_TAG_MASK;
                    if (val != FUNC_MF_CFG_E1HOV_TAG_DEFAULT)
                        adapter->path_has_ovlan = 1;
                }
            } else
                   QFLE3_ERR("NO_MCP_WA_FORCE_5710\n");
        } else
               QFLE3_ERR("NO_MCP_WA_CFG_SET_MAGIC not found\n");

        QFLE3_INFO("%s function mode\n",
             IS_MF(adapter) ? "multi" : "single");

           if (!IS_MF(adapter))
            if (vn) {
                   QFLE3_ERR("VN %d is in a single function mode, aborting\n",
                    vn);
                       return VMK_FAILURE;
            }
    }
   return status;
}

#define MCP_TIMEOUT      5000   /* 5 seconds (in ms) */
#define MCP_ONE_TIMEOUT  100    /* 100 ms */

static VMK_ReturnStatus
qfle3_init_shmem(qfle3_adapter * adapter)
{
   int cnt = 0;
   vmk_uint32 val = 0;

   do {
      adapter->hw_info.shmem_base = REG_RD(adapter, MISC_REG_SHARED_MEM_ADDR);
      if (adapter->hw_info.shmem_base) {
	 val = SHMEM_RD(adapter, validity_map[QFLE3_PORT(adapter)]);
	 if (val & SHR_MEM_VALIDITY_MB)
	    return VMK_OK;
      }

      vmk_WorldSleep(MCP_ONE_TIMEOUT * 1000);

   } while (cnt++ < (MCP_TIMEOUT / MCP_ONE_TIMEOUT));

   QFLE3_ERR("BAD MCP validity signature\n");

   return VMK_FAILURE;
}

static VMK_ReturnStatus
qfle3_get_igu_cam_info(struct qfle3_adapter *adapter)
{
   vmk_uint32 val;
   vmk_uint32 igu_sb_id;
   vmk_uint16 cap_offset;
   vmk_uint32 control = 0;
   vmk_uint8 fid, igu_sb_cnt = 0;
   vmk_uint32 pfid = QFLE3_FUNC(adapter);
   VMK_ReturnStatus status = VMK_OK;

   adapter->igu_sb_cnt = 1;
   adapter->igu_base_addr = BAR_IGU_INTMEM;

   status = vmk_PCIFindCapability(adapter->pdev, PCI_CAP_ID_MSIX, &cap_offset);

   if (status != VMK_OK || !cap_offset) {
      QFLE3_INFO("No msix capability found\n");
   } else {
      QFLE3_DBG(QFLE3_DBG_LOAD, "MSIX capability found\n");

      status = vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev,
				 VMK_PCI_CONFIG_ACCESS_16,
				 cap_offset + PCI_MSIX_FLAGS, &control);
      if (status == VMK_OK) {

         adapter->igu_sb_cnt = control & PCI_MSIX_FLAGS_QSIZE;
      }
   }
   
   QFLE3_INFO("PCI function %d:%d:%d:%d supports %d Interrupt vectors\n", 
      adapter->pdev_addr.seg, adapter->pdev_addr.bus, adapter->pdev_addr.dev,
      adapter->pdev_addr.fn, adapter->igu_sb_cnt);

   adapter->igu_base_sb = 0xff;
   if (CHIP_INT_MODE_IS_BC(adapter)) {
      int vn = QFLE3_VN(adapter);
      igu_sb_cnt = adapter->igu_sb_cnt;
      adapter->igu_base_sb = (CHIP_MODE_IS_4_PORT(adapter) ? pfid : vn) * FP_SB_MAX_E1x;

      adapter->igu_dsb_id = E1HVN_MAX * FP_SB_MAX_E1x +
	 (CHIP_MODE_IS_4_PORT(adapter) ? pfid : vn);

      return status;
   }

   for (igu_sb_id = 0; igu_sb_id < IGU_REG_MAPPING_MEMORY_SIZE; igu_sb_id++) {
      val = REG_RD(adapter, IGU_REG_MAPPING_MEMORY + igu_sb_id * 4);
      if (!(val & IGU_REG_MAPPING_MEMORY_VALID))
         continue;

      fid = ((val) & IGU_REG_MAPPING_MEMORY_FID_MASK) >>
         IGU_REG_MAPPING_MEMORY_FID_SHIFT;

      if ((fid & IGU_FID_ENCODE_IS_PF)) {
         if ((fid & IGU_FID_PF_NUM_MASK) != pfid)
            continue;
         if ((((val) & IGU_REG_MAPPING_MEMORY_VECTOR_MASK) >>
            IGU_REG_MAPPING_MEMORY_VECTOR_SHIFT) == 0)
            /*
            * default status block
            */
            adapter->igu_dsb_id = igu_sb_id;
         else {
            if (adapter->igu_base_sb == 0xff)
               adapter->igu_base_sb = igu_sb_id;
            igu_sb_cnt++;
         }
      }
   }

   adapter->igu_sb_cnt = MIN_OF(adapter->igu_sb_cnt, igu_sb_cnt);

   if (igu_sb_cnt == 0) {
      QFLE3_ERR("CAM configuration error\n");
      status = VMK_FAILURE;
   }
   QFLE3_INFO("Interrupt vectors supported %d\n", adapter->igu_sb_cnt);
   return status;
}

void
qfle3_set_mac_buf(vmk_uint8 *mac_buf, vmk_uint32 mac_lo, vmk_uint16 mac_hi)
{
   __be16 mac_hi_be = vmk_CPUToBE16(mac_hi);
   __be32 mac_lo_be = vmk_CPUToBE32(mac_lo);

   vmk_Memcpy(mac_buf, &mac_hi_be, sizeof(mac_hi_be));
   vmk_Memcpy(mac_buf + sizeof(mac_hi_be), &mac_lo_be, sizeof(mac_lo_be));
}

static void
qfle3_get_mac_hwinfo(struct qfle3_adapter *adapter)
{
   vmk_uint32 val, val2;
   int func = QFLE3_ABS_FUNC(adapter);
   int port = QFLE3_PORT(adapter);

   if (IS_MF(adapter)) {
      val2 = MF_CFG_RD(adapter, func_mf_config[func].mac_upper);
      val = MF_CFG_RD(adapter, func_mf_config[func].mac_lower);
      if ((val2 != FUNC_MF_CFG_UPPERMAC_DEFAULT) &&
	  (val != FUNC_MF_CFG_LOWERMAC_DEFAULT))
	 qfle3_set_mac_buf(adapter->hwMacAddr, val, val2);

   } else {
      /* in SF read MACs from port configuration */
      val2 = SHMEM_RD(adapter, dev_info.port_hw_config[port].mac_upper);
      val = SHMEM_RD(adapter, dev_info.port_hw_config[port].mac_lower);
      qfle3_set_mac_buf(adapter->hwMacAddr, val, val2);

   }
#ifdef QFLE3_CNIC
   if (CNIC_SUPPORT(adapter))
      qfle3_get_cnic_mac_hwinfo(adapter);
#endif

#ifdef QFLE3_SRIOV
   /* Read physical port identifier from shmem */
   if(!QFLE3_NOMCP(adapter)) {
      val2 = SHMEM_RD(adapter, dev_info.port_hw_config[port].mac_upper);
      val = SHMEM_RD(adapter, dev_info.port_hw_config[port].mac_lower);
      qfle3_set_mac_buf(adapter->phys_port_id, val, val2);
      adapter->flags |= HAS_PHYS_PORT_ID;
   }
#endif  //QFLE3_SRIOV

   vmk_Memcpy(adapter->link_params.mac_addr, adapter->hwMacAddr, VMK_ETH_ADDR_LENGTH);
}

static VMK_ReturnStatus
qfle3_get_hw_info(struct qfle3_adapter *adapter)
{
   vmk_uint32 id;
   vmk_uint32 val;
   vmk_uint32 boot_mode;
   vmk_uint32 pci_cfg_dword;
   VMK_ReturnStatus status = VMK_OK;

   /*
    * Validate that chip access is feasible
    */
   if (REG_RD(adapter, MISC_REG_CHIP_NUM) == 0xffffffff) {
      QFLE3_ERR("Chip read returns all Fs. Preventing probe from continuing\n");
      goto err;
   }

   val = REG_RD(adapter, MISC_REG_CHIP_NUM);
   id = ((val & 0xffff) << 16);
   val = REG_RD(adapter, MISC_REG_CHIP_REV);
   id |= ((val & 0xf) << 12);

   val = REG_RD(adapter, PCICFG_OFFSET + PCI_ID_VAL3);
   id |= (((val >> 24) & 0xf) << 4);
   val = REG_RD(adapter, MISC_REG_BOND_ID);
   id |= (val & 0xf);
   adapter->hw_info.chip_id = id;

   /*
    * force 57811 according to MISC register
    */
   if (REG_RD(adapter, MISC_REG_CHIP_TYPE) & MISC_REG_CHIP_TYPE_57811_MASK) {
      if (CHIP_IS_57810(adapter))
	 adapter->hw_info.chip_id = (CHIP_NUM_57811 << 16) |
	    (adapter->hw_info.chip_id & 0x0000FFFF);
      else if (CHIP_IS_57810_MF(adapter))
	 adapter->hw_info.chip_id = (CHIP_NUM_57811_MF << 16) |
	    (adapter->hw_info.chip_id & 0x0000FFFF);
      adapter->hw_info.chip_id |= 0x1;
   }

   adapter->db_stride = (1 << QFLE3_DB_SHIFT);

   adapter->link_params.chip_id = adapter->hw_info.chip_id;

   val = REG_RD(adapter, MISC_REG_PORT4MODE_EN_OVWR);
   if ((val & 1) == 0)
      val = REG_RD(adapter, MISC_REG_PORT4MODE_EN);
   else
      val = (val >> 1) & 1;

   adapter->hw_info.port_mode = val ? CHIP_4_PORT_MODE : CHIP_2_PORT_MODE;

   status = vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev,
			      VMK_PCI_CONFIG_ACCESS_32,
			      PCICFG_ME_REGISTER, &pci_cfg_dword);
   if (status != VMK_OK) {
      goto err;
   }

   adapter->pf_num = (vmk_uint8)((pci_cfg_dword & ME_REG_ABS_PF_NUM) >>
				 ME_REG_ABS_PF_NUM_SHIFT);

   QFLE3_DBG(QFLE3_DBG_LOAD, "ABS PF number: %d\n", adapter->pf_num);

   if (CHIP_MODE_IS_4_PORT(adapter))
      adapter->pf_id = (adapter->pf_num >> 1);  /* 0..3 */
   else
      adapter->pf_id = (adapter->pf_num & 0x6); /* 0, 2, 4, 6 */

   adapter->db_size = (1 << QFLE3_DB_SHIFT);


   val = REG_RD(adapter, MCP_REG_MCPR_NVM_CFG4);
   adapter->hw_info.flash_size = (QFLE3_NVRAM_1MB_SIZE <<
				  (val & MCPR_NVM_CFG4_FLASH_SIZE));

   QFLE3_DBG(QFLE3_DBG_LOAD, "flash_size 0x%x (%d)\n",
	      adapter->hw_info.flash_size, adapter->hw_info.flash_size);

   qfle3_init_shmem(adapter);

   adapter->hw_info.shmem2_base = REG_RD(adapter, (QFLE3_PATH(adapter) ?
						   MISC_REG_GENERIC_CR_1 :
						   MISC_REG_GENERIC_CR_0));

   adapter->link_params.shmem_base = adapter->hw_info.shmem_base;
   adapter->link_params.shmem2_base = adapter->hw_info.shmem2_base;
   if (SHMEM2_RD(adapter, size) >
       (vmk_uint32) vmk_offsetof(struct shmem2_region,
				 lfa_host_addr[QFLE3_PORT(adapter)]))
      adapter->link_params.lfa_base =
	 REG_RD(adapter,
		adapter->hw_info.shmem2_base +
		(vmk_uint32) vmk_offsetof(struct shmem2_region,
					  lfa_host_addr[QFLE3_PORT(adapter)]));
   else
      adapter->link_params.lfa_base = 0;

   if (!adapter->hw_info.shmem_base) {
      QFLE3_ERR("MCP not active\n");
      adapter->flags |= NO_MCP_FLAG;
      goto err;
   }

   adapter->hw_info.hw_config = SHMEM_RD(adapter, dev_info.shared_hw_config.config);

   adapter->link_params.hw_led_mode = ((adapter->hw_info.hw_config &
					SHARED_HW_CFG_LED_MODE_MASK) >>
				       SHARED_HW_CFG_LED_MODE_SHIFT);

   adapter->link_params.feature_config_flags = 0;
   val = SHMEM_RD(adapter, dev_info.shared_feature_config.config);
   if (val & SHARED_FEAT_CFG_OVERRIDE_PREEMPHASIS_CFG_ENABLED)
      adapter->link_params.feature_config_flags |=
	 ELINK_FEATURE_CONFIG_OVERRIDE_PREEMPHASIS_ENABLED;
   else
      adapter->link_params.feature_config_flags &=
	 ~ELINK_FEATURE_CONFIG_OVERRIDE_PREEMPHASIS_ENABLED;

   val = SHMEM_RD(adapter, dev_info.bc_rev) >> 8;
   adapter->hw_info.bc_ver = val;
   if (val < QFLE3_BC_VER) {
      /*
       * for now only warn
       * * later we might need to enforce this
       */
      QFLE3_ERR("This driver needs bc_ver %X but found %X, please upgrade BC\n",
		QFLE3_BC_VER, val);
   }

   adapter->link_params.feature_config_flags |=
      (val >= REQ_BC_VER_4_VRFY_FIRST_PHY_OPT_MDL) ?
      ELINK_FEATURE_CONFIG_BC_SUPPORTS_OPT_MDL_VRFY : 0;

   adapter->link_params.feature_config_flags |=
      (val >= REQ_BC_VER_4_VRFY_SPECIFIC_PHY_OPT_MDL) ?
      ELINK_FEATURE_CONFIG_BC_SUPPORTS_DUAL_PHY_OPT_MDL_VRFY : 0;
   adapter->link_params.feature_config_flags |=
      (val >= REQ_BC_VER_4_VRFY_AFEX_SUPPORTED) ?
      ELINK_FEATURE_CONFIG_BC_SUPPORTS_AFEX : 0;
   adapter->link_params.feature_config_flags |=
      (val >= REQ_BC_VER_4_SFP_TX_DISABLE_SUPPORTED) ?
      ELINK_FEATURE_CONFIG_BC_SUPPORTS_SFP_TX_DISABLED : 0;

   adapter->link_params.feature_config_flags |=
      (val >= REQ_BC_VER_4_MT_SUPPORTED) ? ELINK_FEATURE_CONFIG_MT_SUPPORT : 0;

#if 0
   adapter->flags |= (val >= REQ_BC_VER_4_PFC_STATS_SUPPORTED) ?
      BC_SUPPORTS_PFC_STATS : 0;
#endif
   adapter->flags |= (val >= REQ_BC_VER_4_FCOE_FEATURES) ? BC_SUPPORTS_FCOE_FEATURES : 0;

   adapter->flags |= (val >= REQ_BC_VER_4_DCBX_ADMIN_MSG_NON_PMF) ?
      BC_SUPPORTS_DCBX_MSG_NON_PMF : 0;
#if 0
   adapter->flags |= (val >= REQ_BC_VER_4_RMMOD_CMD) ? BC_SUPPORTS_RMMOD_CMD : 0;
#endif

   adapter->flags |= (val >= REQ_BC_VER_4_UPDATE_DRIVER_STATE) ? MCP_SUPPORTS_S_CHANNEL_DOWN : 0;

   boot_mode = SHMEM_RD(adapter,
			dev_info.port_feature_config[QFLE3_PORT(adapter)].mba_config) &
      PORT_FEATURE_MBA_BOOT_AGENT_TYPE_MASK;
   switch (boot_mode) {
   case PORT_FEATURE_MBA_BOOT_AGENT_TYPE_PXE:
      adapter->hw_info.boot_mode = FEATURE_ETH_BOOTMODE_PXE;
      break;
   case PORT_FEATURE_MBA_BOOT_AGENT_TYPE_ISCSIB:
      adapter->hw_info.boot_mode = FEATURE_ETH_BOOTMODE_ISCSI;
      break;
   case PORT_FEATURE_MBA_BOOT_AGENT_TYPE_FCOE_BOOT:
      adapter->hw_info.boot_mode = FEATURE_ETH_BOOTMODE_FCOE;
      break;
   case PORT_FEATURE_MBA_BOOT_AGENT_TYPE_NONE:
      adapter->hw_info.boot_mode = FEATURE_ETH_BOOTMODE_NONE;
      break;
   }

   qfle3_acquire_hw_lock(adapter, HW_LOCK_RESOURCE_RESET);

   val = REG_RD(adapter, IGU_REG_BLOCK_CONFIGURATION);

   adapter->hw_info.int_block = INT_BLOCK_IGU;

   if (val & IGU_BLOCK_CONFIGURATION_REG_BACKWARD_COMP_EN) {
      int tout = 5000;

      val &= ~(IGU_BLOCK_CONFIGURATION_REG_BACKWARD_COMP_EN);
      REG_WR(adapter, IGU_REG_BLOCK_CONFIGURATION, val);
      REG_WR(adapter, IGU_REG_RESET_MEMORIES, 0x7f);

      while (tout && REG_RD(adapter, IGU_REG_RESET_MEMORIES)) {
	 tout--;
	 vmk_WorldSleep(1000);
      }

      if (REG_RD(adapter, IGU_REG_RESET_MEMORIES)) {
	 QFLE3_DBG(QFLE3_DBG_INTR, "Forcing Normal Mode failed!!!\n");
	 qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_RESET);
	 goto err;
      }
   }

   if (val & IGU_BLOCK_CONFIGURATION_REG_BACKWARD_COMP_EN) {
      QFLE3_DBG(QFLE3_DBG_LOAD, "IGU Backward Compatible Mode\n");
      adapter->hw_info.int_block |= INT_BLOCK_MODE_BW_COMP;
   } else
      QFLE3_DBG(QFLE3_DBG_LOAD, "IGU Normal Mode\n");

   status = qfle3_get_igu_cam_info(adapter);
   qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_RESET);
   if (status != VMK_OK)
      goto err;

   adapter->base_fw_ndsb = adapter->igu_base_sb;

   status = qfle3_get_mf_config(adapter);
   if (status != VMK_OK)
      goto err;

   qfle3_get_link_port_config(adapter);

   qfle3_get_mac_hwinfo(adapter);
#ifdef QFLE3_CNIC
   qfle3_get_cnic_info(adapter);
#endif
  err:
   return status;
}
static vmk_Bool qfle3_get_dropless_info(struct qfle3_adapter *adapter)
{
   int tmp;
   u32 cfg;

//   if (IS_VF(adapter))
//   	return 0;

   if (IS_MF(adapter) && !CHIP_IS_E1x(adapter)) {
      /* Take function: tmp = func */
      tmp = QFLE3_ABS_FUNC(adapter);
      cfg = MF_CFG_RD(adapter, func_ext_config[tmp].func_cfg);
      cfg = !!(cfg & MACP_FUNC_CFG_PAUSE_ON_HOST_RING);
   } else {
      /* Take port: tmp = port */
      tmp = QFLE3_PORT(adapter);
      cfg = SHMEM_RD(adapter,
                dev_info.port_hw_config[tmp].generic_features);
      cfg = !!(cfg & PORT_HW_CFG_PAUSE_ON_HOST_RING_ENABLED);

      QFLE3_INFO("DROPLESS FC cfg 0x%x", cfg);
   }
   return cfg;
}
void
qfle3_init_queue_count(qfle3_adapter * adapter)
{
   vmk_uint32 txqCnt = QFLE3_DEVICE_MAX_TX_QUEUES;
   vmk_uint32 rxqCnt = QFLE3_DEVICE_MAX_RX_QUEUES;
   vmk_uint32 rssCnt = QFLE3_DEVICE_MAX_RSS_QUEUES;
   vmk_uint32 defRssCnt = 0;
#if (ESX_DDK_VERSION >= 2017)
   vmk_uint32 max_txqCnt, max_rxqCnt;
   vmk_uint32 rss_engine_cnt = QFLE3_DEVICE_MAX_RSS_ENGINE;
#endif
   if (QFLE3_FUNC(adapter) >= 2) {
       txqCnt = QFLE3_DEVICE_MAX_TX_QUEUES/4;
       rxqCnt = QFLE3_DEVICE_MAX_RX_QUEUES/4;
#if (ESX_DDK_VERSION < 2017)
       rssCnt = QFLE3_DEVICE_MAX_RSS_QUEUES/2;
#endif
   }
   /*
    * Check queue number forced by module parameters
    */
   if (qfle3_txqueue_count < 0 || qfle3_txqueue_count > QFLE3_DEVICE_MAX_TX_QUEUES) {
      QFLE3_INFO("Module param txqueue_nr (%u) out of range. Using default value(0).\n", qfle3_txqueue_count);
      qfle3_txqueue_count = 0;
   }
   QFLE3_INFO("Module param txqueue_nr (%u) \n", qfle3_txqueue_count);
   if (qfle3_rxqueue_count < 0 || qfle3_rxqueue_count > QFLE3_DEVICE_MAX_RX_QUEUES) {
      QFLE3_INFO("Module param rxqueue_nr (%u) out of range. Using default value(0) for selecting optimal number of queues.\n", qfle3_rxqueue_count);
      qfle3_rxqueue_count = 0;
   }
   QFLE3_INFO("Module param rxqueue_nr (%u)\n", qfle3_rxqueue_count);
   if (RSS[relative_pf_num] < 0 || RSS[relative_pf_num] > QFLE3_DEVICE_MAX_RSS_QUEUES) {
      QFLE3_INFO("Module param RSS (%u) out of range. Using default value(0) for selecting optimal number of queues.\n", RSS[relative_pf_num]);
      RSS[relative_pf_num] = 0;
   }
   QFLE3_INFO("Module param RSS (%u)\n", RSS[relative_pf_num]);
   QFLE3_INFO("Module param txqueue_nr (%d)  rxqueue_nr (%d).\n", qfle3_txqueue_count, qfle3_txqueue_count);
   if (qfle3_txqueue_count > 0) {
      txqCnt = MIN(txqCnt, qfle3_txqueue_count);
   }
   if (qfle3_rxqueue_count > 0) {
      rxqCnt = MIN(rxqCnt, qfle3_rxqueue_count);
   }
    if (RSS[relative_pf_num] > 0) {
      rssCnt = MIN(rssCnt, RSS[relative_pf_num]);
   }

   if (DRSS[relative_pf_num] < 5 && DRSS[relative_pf_num] > 1) {
      defRssCnt = DRSS[relative_pf_num];
   } else if (DRSS[relative_pf_num] != 0) {
      QFLE3_ERR("Given module param DRSS (%u) out of range, driver will try enabling it to max. %d.\n",
             DRSS[relative_pf_num], QFLE3_DEVICE_MAX_DEF_RSS_QUEUES);
      defRssCnt = DRSS[relative_pf_num] = QFLE3_DEVICE_MAX_DEF_RSS_QUEUES;
   }
   
   QFLE3_INFO("txqCnt (%d)  rxqCnt (%d). rssCnt (%d) defRssCnt (%d) relative PF(%d)\n", txqCnt, rxqCnt, rssCnt, defRssCnt, relative_pf_num);

#if (ESX_DDK_VERSION >= 2017)
   if ((rss_engine_nr[relative_pf_num] <= QFLE3_DEVICE_MAX_RSS_ENGINE) && (rss_engine_nr[relative_pf_num] >= 0)) {
      rss_engine_cnt = rss_engine_nr[relative_pf_num];
   } else {
      QFLE3_ERR("Given module param rss_engine_nr (%u) out of range, driver will try enabling it to max. %d.\n",
                    rss_engine_nr[relative_pf_num], QFLE3_DEVICE_MAX_RSS_ENGINE);
      rss_engine_cnt = QFLE3_DEVICE_MAX_RSS_ENGINE;
   }
#endif

   if (QFLE3_FUNC(adapter) >= 2) {
     defRssCnt = DRSS[relative_pf_num] = 0;
     QFLE3_INFO("DRSS is not supported on this NPAR function \n");
#if (ESX_DDK_VERSION >= 2017)
    rss_engine_cnt = (rss_engine_cnt >= 1)? 1:0;
    QFLE3_INFO("Multiple NetQ RSS is not supported on this NPAR function \n");
#endif
    }
#if (ESX_DDK_VERSION >= 2017)
   rss_engine_cnt = (rss_engine_cnt > rxqCnt)? rxqCnt : rss_engine_cnt;
   rss_engine_cnt = ((rss_engine_cnt >= rxqCnt) && (defRssCnt<2))? rss_engine_cnt-1 : rss_engine_cnt;
   rss_engine_cnt = (rxqCnt<2)? 0: rss_engine_cnt;
#endif

#if (ESX_DDK_VERSION < 2017)
   /*
    * Check with physical CPU cores
    */
   vmk_UplinkQueueGetNumQueuesSupported(txqCnt, rxqCnt, &txqCnt, &rxqCnt);
   QFLE3_DBG(QFLE3_DBG_LOAD,
             "vmk_UplinkQueueGetNumQueuesSupported txqNum: %d, rxqNum: %d.", txqCnt, rxqCnt);

   adapter->num_rssqs_nd = rssCnt;
   adapter->num_rssqs_def = defRssCnt;
   adapter->num_txqs_drv = adapter->num_txqs_vmk = txqCnt;
   adapter->num_rxqs_vmk = rxqCnt;
   adapter->num_rxqs_drv = adapter->num_rxqs_vmk +
                  (adapter->num_rssqs_nd - 1) + ((adapter->num_rssqs_def > 0) ? (adapter->num_rssqs_def - 1) : 0);
   QFLE3_INFO("Estimated txqNum:%d, rxqNum:%d, Default RSS Queues:%d, Netqueue RSS Queues:%d, Total NIC RX Queues on adapter:%d.\n",
                   adapter->num_txqs_vmk, adapter->num_rxqs_vmk, adapter->num_rssqs_def, adapter->num_rssqs_nd, adapter->num_rxqs_drv);

#else
   adapter->num_txqs_drv = adapter->num_txqs_vmk = txqCnt;
   adapter->num_rxqs_vmk = rxqCnt;
adjust_queues:
   if(rss_engine_cnt <=1){
        defRssCnt = 0;
        QFLE3_INFO("Total RSS Engine Count less than or equal to 1 hence disabling DRSS \n");
   }
   if(rss_engine_cnt <=0){
      rssCnt = 1;
      QFLE3_INFO("Total RSS Engine Count less than or equal to 0 hence disabling Netqueue RSS \n");
   }


   adapter->num_rssqs_nd = rssCnt;
   adapter->num_rssqs_def = defRssCnt;
   adapter->num_rss_engines = rss_engine_cnt;
   adapter->num_rxqs_drv = adapter->num_rxqs_vmk + (((adapter->num_rss_engines > 1) && (adapter->num_rssqs_def > 1)) ? (adapter->num_rssqs_def - 1) : 0);
                                                                                     // Include Secondary RSS Queues associated with Default Queue
   adapter->num_rxqs_drv = adapter->num_rxqs_drv + (((adapter->num_rss_engines > 1) && (adapter->num_rssqs_def > 1)) ?
                        ((adapter->num_rss_engines - 1) * (adapter->num_rssqs_nd - 1)): (adapter->num_rss_engines * (adapter->num_rssqs_nd - 1))); 
										     // Include Secondary NetQueue RSS Queues

   QFLE3_INFO("Estimated Number of RSS Engines: %d and Total number of RX Queues are %d\n", adapter->num_rss_engines, adapter->num_rxqs_drv);
   /*
    * Check with physical CPU cores
    */
   vmk_UplinkQueueGetNumQueuesSupported(adapter->num_txqs_drv, adapter->num_rxqs_drv, &max_txqCnt, &max_rxqCnt);
   if((max_txqCnt < adapter->num_txqs_vmk) || (max_rxqCnt < adapter->num_rxqs_drv)){
      QFLE3_INFO("Physical CPU cores can only support %d TX Queues and %d RX Queues hence re-adjusting Queue Count\n", max_txqCnt, max_rxqCnt); 
      adapter->num_txqs_drv = adapter->num_txqs_vmk = max_txqCnt;
      if(rss_engine_cnt)
         rss_engine_cnt = rss_engine_cnt - 1;
      else
         adapter->num_rxqs_vmk = adapter->num_rxqs_vmk - (adapter->num_rxqs_drv - max_rxqCnt);
      if(adapter->num_rxqs_vmk<=2){
         rss_engine_cnt = 0;
         defRssCnt = rssCnt = 0;
         adapter->num_rxqs_vmk = max_rxqCnt;
      }
      goto adjust_queues;
   }
   
   QFLE3_INFO("Estimated txqNum:%d, rxqNum:%d, Default RSS Queues:%d, Netqueue RSS Queues:%d, RSS Engines:%d, Total NIC RX Queues on adapter:%d.\n",
                   adapter->num_txqs_vmk, adapter->num_rxqs_vmk, adapter->num_rssqs_def, adapter->num_rssqs_nd, adapter->num_rss_engines, adapter->num_rxqs_drv);
#endif
#ifdef QFLE3_CNIC
   if (CNIC_SUPPORT(adapter))
   adapter->num_cnicqs = 3;
   else
      adapter->num_cnicqs = 0;
#else
   adapter->num_cnicqs = 0;
#endif

}

static VMK_ReturnStatus
qfle3_alloc_msix_interrupts(qfle3_adapter * adapter)
{
   VMK_ReturnStatus status;
   vmk_uint8 numIntrs = 0;
   vmk_uint32 numIntrsAlloced;
   qfle3_intr *intr = &adapter->intr;
   //vmk_uint32 minimum = 0;
#if (ESX_DDK_VERSION >= 2017)
   vmk_uint8 def_rss_enable=0;
#endif

   if(adapter->igu_sb_cnt < 3 + CNIC_SUPPORT(adapter)){
                QFLE3_INFO("Too few MSI-X vectors available %d. Need atleast one for TX and RX Queue Each."
                                " One for default SB and %d for CNIC\n", adapter->igu_sb_cnt,  CNIC_SUPPORT(adapter));
                return(VMK_FAILURE);
   }

   /* one vector for default SB, one vector for CNIC */
   if (adapter->igu_sb_cnt < (QFLE3_NUM_RX_ETH_QUEUES(adapter) + 
                              QFLE3_NUM_TX_ETH_QUEUES(adapter)  + 1 + CNIC_SUPPORT(adapter))) {
         QFLE3_INFO("Not enough MSI-X vectors available to support %d out of %d ethernet queues",
                    adapter->igu_sb_cnt, QFLE3_NUM_RX_ETH_QUEUES(adapter) + 
                              QFLE3_NUM_TX_ETH_QUEUES(adapter) + 1 + CNIC_SUPPORT(adapter));
         if (adapter->igu_sb_cnt < 10) {
            adapter->num_rssqs_nd = 1;
            adapter->num_rssqs_def = 1;
#if (ESX_DDK_VERSION >= 2017)
            adapter->num_rss_engines = 0;
#endif
         } else {
#if (ESX_DDK_VERSION >= 2017)
            if(adapter->num_rss_engines>2)
               adapter->num_rss_engines = adapter->num_rss_engines-1;

            if ((adapter->num_rssqs_def > 1) && (adapter->num_rss_engines > 1))
               def_rss_enable = 1;
         }
recalculate:
           adapter->num_txqs_drv = adapter->num_txqs_vmk = (adapter->igu_sb_cnt - 1 - CNIC_SUPPORT(adapter)
                   - (def_rss_enable?((adapter->num_rss_engines -1)*(adapter->num_rssqs_nd -1)):((adapter->num_rss_engines)*(adapter->num_rssqs_nd -1)))
                   - (def_rss_enable?(adapter->num_rssqs_def-1):0)) / 2;
           adapter->num_rxqs_vmk = (adapter->igu_sb_cnt - 1 - CNIC_SUPPORT(adapter)
                   - (def_rss_enable?((adapter->num_rss_engines -1)*(adapter->num_rssqs_nd -1)):((adapter->num_rss_engines)*(adapter->num_rssqs_nd -1)))
                   - (def_rss_enable?(adapter->num_rssqs_def-1):0)) / 2;

           adapter->num_rxqs_drv = adapter->num_rxqs_vmk
                   + (def_rss_enable? (adapter->num_rssqs_def - 1) : 0)
                   + (def_rss_enable? ( adapter->num_rss_engines- 1) * (adapter->num_rssqs_nd - 1): (adapter->num_rss_engines * (adapter->num_rssqs_nd - 1)));

        QFLE3_INFO("adapter->igu_sb_cnt is %d QFLE3_NUM_RX_ETH_QUEUES(adapter) %d QFLE3_NUM_TX_ETH_QUEUES(adapter) %d \n",
                                        adapter->igu_sb_cnt, QFLE3_NUM_RX_ETH_QUEUES(adapter), QFLE3_NUM_TX_ETH_QUEUES(adapter));

        if(!adapter->num_rxqs_vmk || !adapter->num_txqs_vmk){
                adapter->num_rss_engines = 0;
                adapter->num_rssqs_nd = 1;
                adapter->num_rssqs_def = 1;
                adapter->num_txqs_drv = adapter->num_txqs_vmk = 1;
                adapter->num_rxqs_drv = adapter->num_rxqs_vmk = 1;
        }


          if(adapter->num_rxqs_vmk <= adapter->num_rss_engines){
                adapter->num_rss_engines = ((adapter->num_rss_engines >1) ? (adapter->num_rss_engines-1) : 0);
                if(adapter->num_rss_engines<=1){
                        adapter->num_rssqs_def = 0;
                        def_rss_enable = 0;
                }
                if(adapter->num_rss_engines==0){
                        adapter->num_rssqs_nd = 1;
                }
                goto recalculate;
          }

          if (adapter->igu_sb_cnt < (QFLE3_NUM_RX_ETH_QUEUES(adapter) +
                              QFLE3_NUM_TX_ETH_QUEUES(adapter)  + 1 + CNIC_SUPPORT(adapter))) {
                adapter->num_rssqs_def = 0;
                adapter->num_rss_engines = 1;
                QFLE3_INFO("Not enough MSI-X vectors available, Disabling DRSS\n");
                goto recalculate;
         }

#else
         if (adapter->num_rssqs_nd > 1)
            adapter->num_rssqs_nd = 2;
	    }
            if (adapter->num_rssqs_def > 1)
               adapter->num_rssqs_def = 2;
recalculate:
            adapter->num_txqs_drv = adapter->num_txqs_vmk = (adapter->igu_sb_cnt - 1 - CNIC_SUPPORT(adapter) - (adapter->num_rssqs_nd -1) - (adapter->num_rssqs_def-1)) / 2;
            adapter->num_rxqs_vmk = (adapter->igu_sb_cnt - 1 - CNIC_SUPPORT(adapter) -(adapter->num_rssqs_nd -1) - (adapter->num_rssqs_def-1) ) / 2;
	    adapter->num_rxqs_drv = adapter->num_rxqs_vmk + adapter->num_rssqs_nd - 1 +
                ((adapter->num_rssqs_def > 1) ? (adapter->num_rssqs_def - 1) : 0);

            if (adapter->igu_sb_cnt < (QFLE3_NUM_RX_ETH_QUEUES(adapter) +
                              QFLE3_NUM_TX_ETH_QUEUES(adapter)  + 1 + CNIC_SUPPORT(adapter))) {
		adapter->num_rssqs_def = 1;
                QFLE3_INFO("Not enough MSI-X vectors available, Disabling DRSS\n");
		goto recalculate;
            }
#endif
      
   }
#if (ESX_DDK_VERSION >= 2017)
   QFLE3_INFO("Actual txqNum:%d, rxqNum:%d, rssNum:%d, defRssNum:%d cnicNum:%d"
                "RSS Engine:%d Total NIC RX Queues on adapter:%d\n",
                 adapter->num_txqs_vmk,
                 adapter->num_rxqs_vmk,
                 adapter->num_rssqs_nd,
                 adapter->num_rssqs_def,
                 adapter->num_cnicqs,
                 adapter->num_rss_engines,
                 adapter->num_rxqs_drv);
#else
    QFLE3_INFO("Actual txqNum:%d, rxqNum:%d, rssNum:%d, defRssNum:%d cnicNum:%d"
                " Total NIC RX Queues on adapter:%d\n",
                 adapter->num_txqs_vmk,
                 adapter->num_rxqs_vmk,
                 adapter->num_rssqs_nd,
                 adapter->num_rssqs_def,
                 adapter->num_cnicqs,
                 adapter->num_rxqs_drv);
#endif
   /*
    */
   numIntrs = QFLE3_NUM_RX_ETH_QUEUES(adapter) + QFLE3_NUM_TX_ETH_QUEUES(adapter);
   /*
    * Default Status Block
    */
   numIntrs += 1;

   /*
    * CNIC Status Block
    */
   numIntrs += CNIC_SUPPORT(adapter);
   

   status = vmk_PCIAllocIntrCookie(vmk_ModuleCurrentID,
               adapter->pdev,
               VMK_PCI_INTERRUPT_TYPE_MSIX,
               numIntrs, numIntrs, NULL,
               intr->cookies, &numIntrsAlloced);
   if (status == VMK_OK) {
      VMK_ASSERT(numIntrs == numIntrsAlloced);
      intr->numIntrs = numIntrsAlloced;
      QFLE3_DBG(QFLE3_DBG_LOAD, "Allocated %d MSI-X vectors.", intr->numIntrs);
      goto Done;
   }

   QFLE3_ERR("Failed to allocate %d MSI-X vectors (%x)", numIntrs, status);
Done:
   return status;
}

static void
qfle3_free_interrupts(qfle3_adapter * adapter)
{
   if (adapter->intr.numIntrs) {
      vmk_PCIFreeIntrCookie(vmk_ModuleCurrentID, adapter->pdev);
      adapter->intr.numIntrs = 0;
   }
}

static VMK_ReturnStatus
qfle3_register_sp_interrupt(qfle3_adapter * adapter)
{

   VMK_ReturnStatus (*ack_handler) (void *, vmk_IntrCookie);
   void (*intr_handler) (void *, vmk_IntrCookie);

   switch (adapter->intr.intrType) {
   case QFLE3_IT_MSIX:
      ack_handler = qfle3_interrupt_ack;
      intr_handler = qfle3_sp_interrupt;
      break;
   case QFLE3_IT_MSI:
      ack_handler = qfle3_single_interrupt_ack;
      intr_handler = qfle3_single_interrupt;
      break;
   case QFLE3_IT_INTX:
      ack_handler = qfle3_single_interrupt_ack;
      intr_handler = qfle3_single_interrupt;
      break;
   default:
      QFLE3_ERR("Unknown interrupt type");
      return VMK_FAILURE;
   }

   return qfle3_register_interrupt(adapter, 0,
				   adapter->pdev_name,
				   adapter,
				   ack_handler, intr_handler);
}

static VMK_ReturnStatus
qfle3_alloc_interrupts(qfle3_adapter * adapter)
{
   VMK_ReturnStatus status = VMK_OK;
   const char *intrName[] = { "AUTO", "INT-X", "MSI", "MSI-X" };
   qfle3_intr *intr = &adapter->intr;

   qfle3_init_queue_count(adapter);

   if (qfle3_intr_mode < 0 || qfle3_intr_mode > QFLE3_IT_MSIX) {
      QFLE3_INFO("Module param intr_mode(%u) out of range. Using default value(0).\n", qfle3_intr_mode);
      qfle3_intr_mode = QFLE3_IT_AUTO;
   }

   intr->intrType = qfle3_intr_mode;

   if (intr->intrType == QFLE3_IT_AUTO) {
      QFLE3_INFO("Interrupt type AUTO detected. Try MSI-X.");
      intr->intrType = QFLE3_IT_MSIX;
   }

   if (intr->intrType == QFLE3_IT_MSIX) {
      status = qfle3_alloc_msix_interrupts(adapter);
      if (status == VMK_OK) {
         adapter->flags |=  USING_MSIX_FLAG;
         goto done;
      } else {
//         intr->intrType = QFLE3_IT_MSI;
         QFLE3_ERR("Could not allocate %d MSI-X vectors", intr->numIntrs);
         return status;
         /*
         * Fall through
         */
      }
   }

   if (intr->intrType == QFLE3_IT_MSI) {
      status = vmk_PCIAllocIntrCookie(vmk_ModuleCurrentID,
         adapter->pdev,
         VMK_PCI_INTERRUPT_TYPE_MSI,
         1, 1, NULL, &intr->cookies[0], NULL);
      if (status == VMK_OK) {
	 intr->numIntrs = 1;
	 adapter->num_txqs_drv = adapter->num_txqs_vmk = 1;
	 adapter->num_rxqs_vmk = 1;
	 adapter->num_rssqs_nd = 1;
         if(adapter->num_rssqs_def > 1)
	    adapter->num_rssqs_def = 1;
	 adapter->flags |=  USING_MSI_FLAG;
	 QFLE3_INFO("Actual txqNum: %d, rxqNum: %d.",
               adapter->num_txqs_vmk,
               adapter->num_rxqs_vmk);
    adapter->flags |= NO_ISCSI_OOO_FLAG;
    adapter->flags |= NO_ISCSI_FLAG;
    adapter->flags |= NO_FCOE_FLAG;
	 goto done;
      } else {
         QFLE3_ERR("Failed to allocate MSI PCI interrupt (%x)", status);
         intr->intrType = QFLE3_IT_INTX;
         /*
         * Fall through
         */
      }
   }

   if (intr->intrType == QFLE3_IT_INTX) {

      status = vmk_PCIAllocIntrCookie(vmk_ModuleCurrentID,
            adapter->pdev,
            VMK_PCI_INTERRUPT_TYPE_LEGACY,
            1, 1, NULL, &intr->cookies[0], NULL);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to allocate legacy PCI interrupt (%x)", status);
         goto fail_intx;
      }
      intr->numIntrs = 1;
      adapter->num_txqs_drv = adapter->num_txqs_vmk = 1;
      adapter->num_rxqs_vmk = 1;
      adapter->num_rssqs_nd = 1;
      if(adapter->num_rssqs_def > 1)
         adapter->num_rssqs_def = 1;
      QFLE3_INFO("Actual txqNum: %d, rxqNum: %d.",
               adapter->num_txqs_vmk,
               adapter->num_rxqs_vmk);
      adapter->flags |= NO_ISCSI_OOO_FLAG;
      adapter->flags |= NO_ISCSI_FLAG;
      adapter->flags |= NO_FCOE_FLAG;
   }
  done:
   QFLE3_INFO("Using %d %s PCI interrupt", intr->numIntrs, intrName[intr->intrType]);
  fail_intx:
   return status;
}
/* fw stats memory used for DMA, include the ramrod to Device and returned data from Device*/
static VMK_ReturnStatus
qfle3_free_fw_stats_mem(qfle3_adapter *adapter)
{
   if (adapter->fw_stats) {
      qfle3_free_dma_mapping(adapter,
			     adapter->dmaEngineCoherent,
			     adapter->fw_stats,
			     adapter->fw_stats_mapping,
			     adapter->fw_stats_req_size + adapter->fw_stats_data_size);
      adapter->fw_stats = NULL;
   }

   adapter->fw_stats_num = 0;

   adapter->fw_stats_req_size = 0;
   adapter->fw_stats_req = NULL;
   adapter->fw_stats_req_mapping = 0;

   adapter->fw_stats_data_size = 0;
   adapter->fw_stats_data = NULL;
   adapter->fw_stats_data_mapping = 0;

   return VMK_OK;
}
static VMK_ReturnStatus
qfle3_alloc_fw_stats_mem(qfle3_adapter *adapter)
{
   vmk_uint8  num_queue_stats;
   vmk_uint32 num_groups;
   
// int is_fcoe_stats = NO_FCOE(adapter) ? 0 : 1; //QFLE3_SRIOV_TODO
   vmk_uint16  vf_headroom = 0;

   /* number of queues for statistics is number of eth queues */
   num_queue_stats = QFLE3_NUM_ETH_QUEUES(adapter)+1;

/* Total number of FW statistics requests =
 * 1 for port stats + 1 for PF stats + potential 2 for FCoE (fcoe proper
 * and fcoe l2 queue) stats + num of queues (which includes another 1
 * for fcoe l2 queue if applicable)
 */
   adapter->fw_stats_num = (2 +/* is_fcoe_stats +*/ num_queue_stats);

/*
 * Request is built from stats_query_header and an array of
 * stats_query_cmd_group each of which contains STATS_QUERY_CMD_COUNT
 * rules. The real number or requests is configured in the
 * stats_query_header.
 */

/* When SRIOV is enabled adapter->vfdb is allocated during attach
 * before the allocation of fw_stats_mem. vf_headroom should have 
 * appropriate value to adjust fw_stats_req_size to account for
 * VF stats request issued from both VFs as well as PF.
*/
#ifdef QFLE3_SRIOV
   if(adapter->vfdb)
      vf_headroom = (adapter->vfdb->sriov.nr_virtfn * QFLE3_CIDS_PER_VF)*2;
   else
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Adapter vfdb not allocated yet\n");
#endif //QFLE3_SRIOV

   num_groups =
      (((adapter->fw_stats_num + vf_headroom) / STATS_QUERY_CMD_COUNT) +
       (((adapter->fw_stats_num  + vf_headroom) % STATS_QUERY_CMD_COUNT) ? 1 : 0));

   QFLE3_DBG(QFLE3_DBG_LOAD, "stats fw_stats_num %d, vf headroom %d, num_groups %d\n",
      adapter->fw_stats_num, vf_headroom, num_groups);

   adapter->fw_stats_req_size = (sizeof(struct stats_query_header) +
				 (num_groups * sizeof(struct stats_query_cmd_group)));

/*
 * Data for statistics requests + stats_counter.
 * stats_counter holds per-STORM counters that are incremented when
 * STORM has finished with the current request. Memory for FCoE
 * offloaded statistics are counted anyway, even if they will not be sent.
 * VF stats are not accounted for here as the data of VF stats is stored
 * in memory allocated by the VF, not here.
 */
   adapter->fw_stats_data_size = (sizeof(struct stats_counter) +
				  sizeof(struct per_port_stats) +
				  sizeof(struct per_pf_stats) +
				  //sizeof(struct fcoe_statistics_params) +    //QFLE3_SRIOV_TODO
				  (sizeof(struct per_queue_stats) * num_queue_stats));
   adapter->fw_stats = qfle3_alloc_dma_mapping(adapter,
					       adapter->dmaEngineCoherent,
					       adapter->fw_stats_req_size + adapter->fw_stats_data_size,
					       &adapter->fw_stats_mapping);
   if (!adapter->fw_stats) {
      QFLE3_ERR("allocation of fw stats memory fails!\n");
      return VMK_FAILURE;
   }

   /* set up the shortcuts */

   adapter->fw_stats_req = (struct qfle3_fw_stats_req *)adapter->fw_stats;
   adapter->fw_stats_req_mapping = adapter->fw_stats_mapping;

   adapter->fw_stats_data = (struct qfle3_fw_stats_data *)((vmk_uint8 *)adapter->fw_stats + adapter->fw_stats_req_size);
   adapter->fw_stats_data_mapping = (adapter->fw_stats_mapping + adapter->fw_stats_req_size);

   QFLE3_DBG(QFLE3_DBG_LOAD, "statistics request base address set to %lx\n", adapter->fw_stats_req_mapping);

   QFLE3_DBG(QFLE3_DBG_LOAD, "statistics data base address set to %lx\n", adapter->fw_stats_data_mapping);

   return VMK_OK;

}

static int
qfle3_free_hsi_mem(qfle3_adapter * adapter)
{

   if (adapter->def_status_blk) {
      qfle3_free_dma_mapping(adapter,
			     adapter->dmaEngineCoherent,
			     adapter->def_status_blk,
			     adapter->def_status_blk_mapping,
			     sizeof(struct host_sp_status_block));
      adapter->def_status_blk = NULL;
   }

   if (adapter->spq) {
      qfle3_free_dma_mapping(adapter,
			     adapter->dmaEngineCoherent,
			     adapter->spq, adapter->spq_mapping, QFLE3_PAGE_SIZE);
      adapter->spq = NULL;
   }

   if (adapter->eq_ring) {
      qfle3_free_dma_mapping(adapter,
			     adapter->dmaEngineCoherent,
			     adapter->eq_ring,
			     adapter->eq_mapping, QFLE3_PAGE_SIZE * NUM_EQ_PAGES);
      adapter->eq_ring = NULL;
   }
   /* free the fw stats memory defined */
   qfle3_free_fw_stats_mem(adapter);

   return (0);
}


static VMK_ReturnStatus
qfle3_alloc_hsi_mem(qfle3_adapter * adapter)
{
   adapter->def_status_blk =
      qfle3_alloc_dma_mapping(adapter,
			      adapter->dmaEngineCoherent,
			      sizeof(struct host_sp_status_block),
			      &adapter->def_status_blk_mapping);

   if (!adapter->def_status_blk) {
      QFLE3_ERR("Failed to alloc default status block\n");
      goto err;
   }

   adapter->spq =
      qfle3_alloc_dma_mapping(adapter,
			      adapter->dmaEngineCoherent,
			      QFLE3_PAGE_SIZE, &adapter->spq_mapping);
   if (!adapter->spq) {
      QFLE3_ERR("Failed to alloc spq mem\n");
      goto err;
   }

   /*
    * EQ
    */
   adapter->eq_ring = qfle3_alloc_dma_mapping(adapter,
					      adapter->dmaEngineCoherent,
					      QFLE3_PAGE_SIZE * NUM_EQ_PAGES,
					      &adapter->eq_mapping);
   if (!adapter->eq_ring) {
      QFLE3_ERR("Failed to alloc eq mem\n");
      goto err;
   }

   if (VMK_OK != qfle3_alloc_fw_stats_mem(adapter)) {
      QFLE3_ERR("Failed to allc fw stats mem\n");
      goto err;
   }

   return VMK_OK;
  err:
   qfle3_free_hsi_mem(adapter);
   return VMK_FAILURE;
}

static void
qfle3_free_probe_mem(struct qfle3_adapter *adapter)
{
   vmk_int32 i;

   if (adapter->sp)
      qfle3_free_dma_mapping(adapter,
			     adapter->dmaEngineCoherent,
			     adapter->sp,
			     adapter->sp_mapping, sizeof(struct qfle3_slowpath));

   for (i = 0; i < ILT_MAX_L2_LINES; i++) {
      if (adapter->context[i].vcxt)
      	 qfle3_free_dma_mapping(adapter,
				adapter->dmaEngineCoherent,
				adapter->context[i].vcxt,
				adapter->context[i].cxt_mapping,
				adapter->context[i].size);
   }

   ecore_ilt_mem_op(adapter, ILT_MEMOP_FREE);

   if (adapter->ilt->lines)
      qfle3_heap_free(adapter->ilt->lines);
#ifdef QFLE3_SRIOV
   qfle3_iov_free_mem(adapter);
#endif //QFLE3_SRIOV
   qfle3_free_hsi_mem(adapter);
}

static VMK_ReturnStatus
qfle3_alloc_probe_mem(struct qfle3_adapter *adapter)
{
   VMK_ReturnStatus status = VMK_OK;

   adapter->sp =
      qfle3_alloc_dma_mapping(adapter,
			      adapter->dmaEngineCoherent,
			      sizeof(struct qfle3_slowpath), &adapter->sp_mapping);
   if (!adapter->sp) {
      QFLE3_ERR("Error allocating sp mem\n");
      status = VMK_NO_MEMORY;
      goto err;
   }

   status = qfle3_alloc_cdu_cxt_mem(adapter);
   if (status != VMK_OK) {
      goto err;
   }

   adapter->ilt->lines = qfle3_heap_alloc(sizeof(struct ilt_line) * ILT_MAX_LINES);
   if (!adapter->ilt->lines) {
      status = VMK_FAILURE;
      QFLE3_ERR("Error allocating ilt lines\n");
      goto err;
   }

   if (ecore_ilt_mem_op(adapter, ILT_MEMOP_ALLOC)) {
      status = VMK_NO_MEMORY;
      goto err;
   }
#ifdef QFLE3_SRIOV
   if (qfle3_iov_alloc_mem(adapter)) {
      status = VMK_NO_MEMORY;
      goto err;
   }
#endif  //QFLE3_SRIOV

   qfle3_schedule_helper(adapter,
      adapter->stats_collection_helper, qfle3_stats_update_func, 2000);


   /*
    * slowpath, status block
    */
   status = qfle3_alloc_hsi_mem(adapter);
   if (status != VMK_OK) {
      qfle3_cancel_helper_requests(adapter, adapter->stats_collection_helper);
      goto err;
   }
   return VMK_OK;
  err:
   qfle3_free_probe_mem(adapter);
   return status;
}


#ifdef QFLE3_SRIOV

VMK_ReturnStatus qfle3_passthru_config(struct qfle3_adapter *adapter, vmk_uint32 vfIdx, int change, void *data)
{
   vmk_NetVFCfgInfo cfginfo;
   vmk_uint32 *new_mtu = 0;

   vmk_Memset(&cfginfo, 0, sizeof(cfginfo));

   switch (change) {

   case VMK_CFG_MAC_CHANGED:
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: MAC Address changed\n");
      if (ETH_ALEN != sizeof(cfginfo.macAddr)) {
         QFLE3_ERR("Invalid MAC address: " QFLE3_MAC_FMT"\n", QFLE3_MAC_PRN_LIST(cfginfo.macAddr));
         return VMK_FAILURE;
      }
      cfginfo.cfgChanged = VMK_CFG_MAC_CHANGED;
      vmk_Memcpy(cfginfo.macAddr, ((vmk_uint8 *)data), ETH_ALEN);
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: Guest OS requesting MAC addr: " QFLE3_MAC_FMT " for VF %d\n", QFLE3_MAC_PRN_LIST(cfginfo.macAddr), vfIdx);
      break;

   case VMK_CFG_MTU_CHANGED:
      new_mtu = (vmk_uint32 *)data;
      cfginfo.cfgChanged = VMK_CFG_MTU_CHANGED;
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: Guest OS requesting MTU change to %d for VF %d\n",*new_mtu, vfIdx);
      vmk_Memcpy(&cfginfo.mtu, new_mtu, sizeof(cfginfo.mtu));
      break;

   case VMK_CFG_GUEST_VLAN_ADD:

   case VMK_CFG_GUEST_VLAN_REMOVE:
   {
      vmk_uint8 guest_vid[512];
      vmk_uint16 array_index;
      vmk_uint8 bit_index;
      vmk_uint8 bit;
      vmk_uint16 gvid = *((vmk_uint16 *)data);

      cfginfo.cfgChanged = change;

      vmk_Memset(guest_vid, 0, 512 * sizeof(vmk_uint8));

      array_index = (gvid >> 3) & 0x1FF;
      bit_index = gvid & 0x7;
      bit = (1 << bit_index);
      guest_vid[array_index] |= bit;
      vmk_Memcpy(cfginfo.vlan.guestVlans, guest_vid, sizeof(cfginfo.vlan.guestVlans));
      break;
   }

   default:
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: Invalid VF configuration change request.\n");
      return VMK_FAILURE;
   }

   return vmk_NetPTConfigureVF(&cfginfo, adapter->vfdb->vfs[vfIdx].vf_sbdf);
}

int
qfle3_esx_set_mac_passthru_config(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
              struct qfle3_vf_mac_vlan_filters *fl)
{
   struct qfle3_vf_mac_vlan_filter *entry;
   int rc = 0;
   int i;
   struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vf->index];

   for (i = 0; i < fl->count; i++) {
      entry = &fl->filters[i];

      if (entry->type != QFLE3_VF_FILTER_MAC)
         continue;

      vmk_Memcpy(adapter_esx_vf->mac_from_config, entry->mac, ETH_ALEN);

   }      
   rc = 0;
   adapter_esx_vf->passthru_state = 0;

   return rc;
}

VMK_ReturnStatus
qfle3_esx_vf_reset_all_vmk_gvlans(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   vmk_NetVFCfgInfo cfginfo;
   vmk_uint8 guest_vid[512];

   vmk_Memset(&cfginfo, 0, sizeof(cfginfo));

   cfginfo.cfgChanged = VMK_CFG_GUEST_VLAN_REMOVE;

   vmk_Memset(guest_vid, 0xFF, 512 * sizeof(vmk_uint8));

   vmk_Memcpy(cfginfo.vlan.guestVlans, guest_vid, sizeof(cfginfo.vlan.guestVlans));

   return vmk_NetPTConfigureVF(&cfginfo, adapter->vfdb->vfs[vf->index].vf_sbdf);

}

int
qfle3_esx_vf_request_all_vmk_gvlans(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   vmk_NetVFCfgInfo cfginfo;
   vmk_uint8 guest_vid[512];

   vmk_Memset(&cfginfo, 0, sizeof(cfginfo));

   cfginfo.cfgChanged = VMK_CFG_GUEST_VLAN_ADD;

   vmk_Memset(guest_vid, 0xFF, 512 * sizeof(vmk_uint8));

   vmk_Memcpy(cfginfo.vlan.guestVlans, guest_vid, sizeof(cfginfo.vlan.guestVlans));

   return vmk_NetPTConfigureVF(&cfginfo, adapter->vfdb->vfs[vf->index].vf_sbdf);
}

void
qfle3_esx_vf_remove_all_gvlans(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vf->index];

   vmk_Memset(&adapter_esx_vf->requested_gvlans, 0, 64 * sizeof(vmk_uint64));
}

static vmk_Bool qfle3_validate_vf_sp_objs(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                                            vmk_Bool print_err)
{
   if(!(vf->vfqs[LEADING_IDX].sp_initialized)) {
      QFLE3_ERR("SRIOV: BC: Slowpath objects not yet initialized!\n");
      return VMK_FALSE;
   }
   return VMK_TRUE;
}

static int qfle3_vf_mac_vlan_config(struct qfle3_adapter *adapter,
                struct qfle3_virtf *vf, int qid,
                struct qfle3_vf_mac_vlan_filter *filter,
                vmk_Bool drv_only)
{
   struct ecore_vlan_mac_ramrod_params ramrod;
   int rc;

  QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: vf[%d] - %s a %s filter\n",
      vf->abs_vfid, filter->add ? "Adding" : "Deleting",
      (filter->type == QFLE3_VF_FILTER_VLAN_MAC) ? "VLAN-MAC" :
      (filter->type == QFLE3_VF_FILTER_MAC) ? "MAC" : "VLAN");

   if (filter->type == QFLE3_VF_FILTER_MAC)
       QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: mac address is" QFLE3_MAC_FMT "\n", QFLE3_MAC_PRN_LIST(filter->mac));

   /* Prepare ramrod params */
   vmk_Memset(&ramrod, 0, sizeof(struct ecore_vlan_mac_ramrod_params));
   if (filter->type == QFLE3_VF_FILTER_VLAN_MAC) {
      ramrod.vlan_mac_obj = &qfle3_vfq(vf, qid, vlan_mac_obj);
      ramrod.user_req.u.vlan.vlan = filter->vid;
      vmk_Memcpy(&ramrod.user_req.u.mac.mac, filter->mac, ETH_ALEN);
      ECORE_SET_BIT(ECORE_ETH_MAC, &ramrod.user_req.vlan_mac_flags);
   } else if (filter->type == QFLE3_VF_FILTER_VLAN) {
      ramrod.vlan_mac_obj = &qfle3_vfq(vf, qid, vlan_obj);
      ramrod.user_req.u.vlan.vlan = filter->vid;
   } else {
      ECORE_SET_BIT(ECORE_ETH_MAC, &ramrod.user_req.vlan_mac_flags);
      ramrod.vlan_mac_obj = &qfle3_vfq(vf, qid, mac_obj);
      vmk_Memcpy(&ramrod.user_req.u.mac.mac, filter->mac, ETH_ALEN);
   }
   ramrod.user_req.cmd = filter->add ? ECORE_VLAN_MAC_ADD :
                   ECORE_VLAN_MAC_DEL;

   ECORE_SET_BIT(RAMROD_EXEC, &ramrod.ramrod_flags);
   if (drv_only)
      ECORE_SET_BIT(RAMROD_DRV_CLR_ONLY, &ramrod.ramrod_flags);
   else
      ECORE_SET_BIT(RAMROD_COMP_WAIT, &ramrod.ramrod_flags);

   /* Add/Remove the filter */
   rc = ecore_config_vlan_mac(adapter, &ramrod);
   if (rc && rc != ECORE_EXISTS) {
      QFLE3_ERR("SRIOV: BC: Failed to %s %s\n",
           filter->add ? "add" : "delete",
           (filter->type == QFLE3_VF_FILTER_VLAN_MAC) ?
            "VLAN-MAC" :
           (filter->type == QFLE3_VF_FILTER_MAC) ?
            "MAC" : "VLAN");
      return rc;
   }

   return 0;
}

int qfle3_vf_mac_vlan_config_list(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
              struct qfle3_vf_mac_vlan_filters *filters,
              int qid, vmk_Bool drv_only)
{
   int rc = 0, i;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: vf[%d]\n", vf->abs_vfid);

   if (!qfle3_validate_vf_sp_objs(adapter, vf, VMK_TRUE)){
      qfle3_heap_free(filters);
      return -1;
   }

   if (ecore_get_q_logical_state(adapter, &(vf->vfqs[qid].sp_obj)) == ECORE_Q_LOGICAL_STATE_STOPPED) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: queue was already stopped. Aborting filter modification at this time\n");
      qfle3_heap_free(filters);
      return rc;
   }

   /* Prepare ramrod params */
   for (i = 0; i < filters->count; i++) {
      rc = qfle3_vf_mac_vlan_config(adapter, vf, qid, &filters->filters[i], drv_only);
      if (rc)
         break;
   }

   /* Rollback if needed */
   if (i != filters->count) {
      QFLE3_ERR("SRIOV: BC: Managed only %d/%d filters - rolling back\n",
           i, filters->count + 1);
      while (--i >= 0) {
         filters->filters[i].add = !filters->filters[i].add;
         qfle3_vf_mac_vlan_config(adapter, vf, qid, &filters->filters[i], drv_only);
      }
   }

   /* It's our responsibility to free the filters */
   qfle3_heap_free(filters);

   return rc;
}


static void
qfle3_esx_check_and_add_gvlan(struct qfle3_adapter *adapter, struct qfle3_virtf *vf, vmk_uint16 vid, vmk_uint8 qid)
{
   int rc = 0;
   vmk_uint64 bit = 0;
   vmk_uint8 array_index = 0;
   vmk_uint64 bit_index = 0;
   struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vf->index];

   array_index = (vid >> 6);
   bit_index = vid & 0x3F;
   bit = ((vmk_uint64)1 << bit_index);

   if ((adapter_esx_vf->allowed_gvlans[array_index] & bit) && !(adapter_esx_vf->requested_gvlans[array_index] & bit)) {
      vmk_ByteCount fsz;
      struct qfle3_vf_mac_vlan_filters *fl;

      fsz = sizeof(struct qfle3_vf_mac_vlan_filter) + sizeof(struct qfle3_vf_mac_vlan_filters);

      fl = qfle3_heap_alloc(fsz);
      if (!fl) {
         QFLE3_ERR("SRIOV: Failed to alloc memory for GVLAN. VF: %d, GVLAN: %u.\n", vf->index, vid);
         return;
      }

      fl->filters[0].vid = vid;
      fl->filters[0].type = QFLE3_VF_FILTER_VLAN;
      fl->filters[0].add = VMK_TRUE;
      fl->count = 1;
      rc  = qfle3_vf_mac_vlan_config_list(adapter, vf, fl, qid, VMK_FALSE);
      if (rc) {
         QFLE3_ERR("SRIOV: Failed to program GVLAN. VF: %d, GVLAN: %u.\n", vf->index, vid);
         return;
      }
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Programmed GVLAN. VF: %d, GVLAN: %u.\n", vf->index, vid);
   }

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Added GVLAN to requested bitmap. VF: %d, GVLAN: %u.\n", vf->index, vid);
   adapter_esx_vf->requested_gvlans[array_index] |= bit;
}

static void
qfle3_esx_check_and_remove_gvlan(struct qfle3_adapter *adapter, struct qfle3_virtf *vf, vmk_uint16 vid, vmk_uint8 qid)
{
   int rc = 0;
   vmk_uint64 bit = 0;
   vmk_uint8 array_index = 0;
   vmk_uint64 bit_index = 0;
   struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vf->index];

   array_index = (vid >> 6);
   bit_index = vid & 0x3F;
   bit = ((vmk_uint64)1 << bit_index);

   if ((adapter_esx_vf->allowed_gvlans[array_index] & bit) && (adapter_esx_vf->requested_gvlans[array_index] & bit)) {
      vmk_ByteCount fsz;
      struct qfle3_vf_mac_vlan_filters *fl;

      fsz = sizeof(struct qfle3_vf_mac_vlan_filter) + sizeof(struct qfle3_vf_mac_vlan_filters);

      fl = qfle3_heap_alloc(fsz);
      if (!fl) {
         QFLE3_ERR("SRIOV: Failed to alloc memory for GVLAN. VF: %d, GVLAN: %u.\n", vf->index, vid);
         return;
      }

      fl->filters[0].vid = vid;
      fl->filters[0].type = QFLE3_VF_FILTER_VLAN;
      fl->filters[0].add = VMK_FALSE;
      fl->count = 1;
      rc  = qfle3_vf_mac_vlan_config_list(adapter, vf, fl, qid, VMK_FALSE);
      if (rc) {
         QFLE3_ERR("SRIOV: Failed to remove GVLAN. VF: %d, GVLAN: %u.\n", vf->index, vid);
         adapter_esx_vf->requested_gvlans[array_index] |= bit;
         return;
      }
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Removed GVLAN. VF: %d, GVLAN: %u.\n", vf->index, vid);
      adapter_esx_vf->requested_gvlans[array_index] &= ~bit;
   }

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Removed GVLAN from requested bitmap. VF: %d, GVLAN: %u.\n", vf->index, vid);
}


int qfle3_esx_vf_gvlans_requested(struct qfle3_adapter *adapter, vmk_uint8 vf_idx)
{
   struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vf_idx];
   vmk_uint64 tmp_gvlans[64] = {0};

   if (vmk_Memcmp(&adapter_esx_vf->requested_gvlans, &tmp_gvlans, 64 * sizeof(vmk_uint64)))
      return 1;

   return 0;
}


int qfle3_esx_vf_gvlans_allowed(struct qfle3_adapter *adapter, vmk_uint8 vf_idx)
{
   struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vf_idx];
   vmk_uint64 tmp_gvlans[64] = {0};

   if (vmk_Memcmp(&adapter_esx_vf->allowed_gvlans, &tmp_gvlans, 64 * sizeof(vmk_uint64)))
      return 1;

   return 0;
}

int
qfle3_esx_vf_any_vlan_allowed(struct qfle3_adapter *adapter, vmk_uint8 vf_idx)
{
   struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vf_idx];
   struct qfle3_virtf *vf = QFLE3_VF(adapter, vf_idx);

   if (!qfle3_esx_vf_gvlans_allowed(adapter, vf_idx)) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: ACCEPT_ANY_VLAN not allowed for VF:%d. GVLANs not allowed.\n", vf_idx);
      return 0;
   }

   if (((vf->cfg_flags & VF_CFG_VLAN_FILTER) && adapter_esx_vf->any_vlan_req) || !qfle3_esx_vf_gvlans_requested(adapter, vf_idx)) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: ACCEPT_ANY_VLAN allowed for VF:%d. CFG_VLAN_FILTER:%d, any_vlan_req:%d, gvlans_req:%d.\n", 
         vf_idx, !!(vf->cfg_flags & VF_CFG_VLAN_FILTER), adapter_esx_vf->any_vlan_req, qfle3_esx_vf_gvlans_requested(adapter, vf_idx));
      return 1;
   }

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: ACCEPT_ANY_VLAN not allowed for VF:%d. CFG_VLAN_FILTER:%d, any_vlan_req:%d, gvlans_req:%d.\n",
      vf_idx, !!(vf->cfg_flags & VF_CFG_VLAN_FILTER), adapter_esx_vf->any_vlan_req, qfle3_esx_vf_gvlans_requested(adapter, vf_idx));

   return 0;
}


int
qfle3_esx_set_vlan_passthru_config(struct qfle3_adapter *adapter, struct qfle3_virtf *vf, struct vfpf_set_q_filters_tlv *tlv)
{
   int i;
   int rc = 0;
   struct ecore_queue_update_params *update_params;
   struct ecore_queue_state_params q_params = {NULL};

   for (i = 0; i < tlv->n_mac_vlan_filters; i++) {
      struct vfpf_q_mac_vlan_filter *msg_filter = &tlv->filters[i];

      if (!(msg_filter->flags & VFPF_Q_FILTER_VLAN_TAG_VALID))
         continue;

      if (msg_filter->flags & VFPF_Q_FILTER_SET) {
         qfle3_esx_check_and_add_gvlan(adapter, vf, msg_filter->vlan_tag, tlv->vf_qid);
      } else {
         qfle3_esx_check_and_remove_gvlan(adapter, vf, msg_filter->vlan_tag, tlv->vf_qid);
      }
   }

   if (qfle3_esx_vf_gvlans_allowed(adapter, vf->index)) {
      if (vf->state != VF_ENABLED ||
          ecore_get_q_logical_state(adapter, &qfle3_leading_vfq(vf, sp_obj)) != ECORE_Q_LOGICAL_STATE_ACTIVE) {
         return 0;
      }
      ECORE_SET_BIT(RAMROD_COMP_WAIT, &q_params.ramrod_flags);
      q_params.cmd = ECORE_Q_CMD_UPDATE;
      q_params.q_obj = &qfle3_leading_vfq(vf, sp_obj);
      update_params = &q_params.params.update;
      ECORE_SET_BIT(ECORE_Q_UPDATE_DEF_VLAN_EN_CHNG, &update_params->update_flags);
      ECORE_SET_BIT(ECORE_Q_UPDATE_SILENT_VLAN_REM_CHNG, &update_params->update_flags);

      ECORE_CLEAR_BIT(ECORE_Q_UPDATE_DEF_VLAN_EN, &update_params->update_flags);
      ECORE_CLEAR_BIT(ECORE_Q_UPDATE_SILENT_VLAN_REM, &update_params->update_flags);

      /* Update the Queue state */
      rc = ecore_queue_state_change(adapter, &q_params);
      if (rc)
         QFLE3_ERR("SRIOV: Queue state change failed for guest VLAN configuration.\n");
      else
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Queue state changed to clear default VLAN.\n");
   }

   return 0;
}

static
struct qfle3_virtf *qfle3_vf_by_abs_fid(struct qfle3_adapter *adapter, vmk_uint16 abs_vfid)
{
        vmk_uint16 idx = (vmk_uint16)qfle3_vf_idx_by_abs_fid(adapter, abs_vfid);
        return (idx < (adapter->vfdb->sriov.nr_virtfn)) ? QFLE3_VF(adapter, idx) : NULL;
}

static vmk_uint8 qfle3_iov_is_vf_cid(struct qfle3_adapter *adapter, vmk_uint16 cid)
{
   return ((cid >= QFLE3_FIRST_VF_CID) &&
      ((cid - QFLE3_FIRST_VF_CID) < QFLE3_VF_CIDS));
}

static void qfle3_vf_handle_classification_eqe(struct qfle3_adapter *adapter,
               struct qfle3_vf_queue *vfq,  union event_ring_elem *elem)
{
   unsigned long ramrod_flags = 0;
   int rc = 0;

   /* Always push next commands out, don't wait here */
   SET_BIT(RAMROD_CONT, &ramrod_flags);

   switch (elem->message.data.eth_event.echo >> ECORE_SWCID_SHIFT) {
   case ECORE_FILTER_MAC_PENDING:
      rc = vfq->mac_obj.complete(adapter, &vfq->mac_obj, elem,
                  &ramrod_flags);
      break;
   case ECORE_FILTER_VLAN_PENDING:
      rc = vfq->vlan_obj.complete(adapter, &vfq->vlan_obj, elem,
                   &ramrod_flags);
      break;
   default:
      QFLE3_ERR("SRIOV: Unsupported classification command: %d\n",
           elem->message.data.eth_event.echo);
      return;
   }
   if (rc < 0)
      QFLE3_ERR("SRIOV: Failed to schedule new commands: %d\n", rc);
   else if (rc > 0)
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Scheduled next pending commands...\n");
}

static
void qfle3_vf_handle_mcast_eqe(struct qfle3_adapter *adapter,
                struct qfle3_virtf *vf)
{
   struct ecore_mcast_ramrod_params rparam = {NULL};
   int rc;

   rparam.mcast_obj = &vf->mcast_obj;
   vf->mcast_obj.raw.clear_pending(&vf->mcast_obj.raw);

   /* If there are pending mcast commands - send them */
   if (vf->mcast_obj.check_pending(&vf->mcast_obj)) {
      rc = ecore_config_mcast(adapter, &rparam, ECORE_MCAST_CMD_CONT);
      if (rc < 0)
         QFLE3_ERR("SRIOV: Failed to send pending mcast commands: %d\n",
              rc);
   }
}

static void qfle3_vf_handle_filters_eqe(struct qfle3_adapter *adapter,
             struct qfle3_virtf *vf)
{
   vmk_CPUMemFenceReadWrite();
   ECORE_CLEAR_BIT(ECORE_FILTER_RX_MODE_PENDING, &vf->filter_state);
   vmk_CPUMemFenceReadWrite();
}

static void qfle3_vf_handle_rss_update_eqe(struct qfle3_adapter *adapter,
                  struct qfle3_virtf *vf)
{
   vf->rss_conf_obj.raw.clear_pending(&vf->rss_conf_obj.raw);
}


int qfle3_iov_eq_sp_event(struct qfle3_adapter *adapter, union event_ring_elem *elem)
{
   struct qfle3_virtf *vf;
   int qidx = 0, abs_vfid;
   vmk_uint8 opcode;
   vmk_uint16 cid = 0xffff;

   if (!adapter->vfdb){
      return 1;
   }

   /* first get the cid - the only events we handle here are cfc-delete
    * and set-mac completion
    */
   opcode = elem->message.opcode;

   switch (opcode) {
   case EVENT_RING_OPCODE_CFC_DEL:
      cid = SW_CID(elem->message.data.cfc_del_event.cid);
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: checking cfc-del comp cid=%d\n", cid);
      break;
   case EVENT_RING_OPCODE_CLASSIFICATION_RULES:
   case EVENT_RING_OPCODE_MULTICAST_RULES:
   case EVENT_RING_OPCODE_FILTERS_RULES:
   case EVENT_RING_OPCODE_RSS_UPDATE_RULES:
      cid = (elem->message.data.eth_event.echo &
             ECORE_SWCID_MASK);
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: checking filtering comp cid=%d\n", cid);
      break;
   case EVENT_RING_OPCODE_VF_FLR:
      abs_vfid = elem->message.data.vf_flr_event.vf_id;
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Got VF FLR notification abs_vfid=%d\n",abs_vfid);
      goto get_vf;
   case EVENT_RING_OPCODE_MALICIOUS_VF:
      abs_vfid = elem->message.data.malicious_vf_event.vf_id;
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Got VF MALICIOUS notification abs_vfid=%d err_id=0x%x\n",
           abs_vfid, elem->message.data.malicious_vf_event.err_id);
      goto get_vf;
   default:
      return 1;
   }

   /* check if the cid is the VF range */
   if (!qfle3_iov_is_vf_cid(adapter, cid)) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: cid is outside vf range: %d\n", cid);
      return 1;
   }

   /* extract vf and rxq index from vf_cid - relies on the following:
    * 1. vfid on cid reflects the true abs_vfid
    * 2. The max number of VFs (per path) is 64
    */
   qidx = cid & ((1 << QFLE3_VF_CID_WND)-1);
   abs_vfid = (cid >> QFLE3_VF_CID_WND) & (QFLE3_MAX_NUM_OF_VFS-1);
get_vf:
   vf = qfle3_vf_by_abs_fid(adapter, abs_vfid);

   if (!vf) {
      QFLE3_ERR("SRIOV: EQ completion for unknown VF, cid %d, abs_vfid %d\n",
           cid, abs_vfid);
      return 0;
   }

   switch (opcode) {
   case EVENT_RING_OPCODE_CFC_DEL:
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: got VF [%d:%d] cfc delete ramrod\n",
         vf->abs_vfid, qidx);
      vfq_get(vf, qidx)->sp_obj.complete_cmd(adapter,
                         &vfq_get(vf,qidx)->sp_obj,ECORE_Q_CMD_CFC_DEL);
      break;
   case EVENT_RING_OPCODE_CLASSIFICATION_RULES:
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: got VF [%d:%d] set mac/vlan ramrod\n",
         vf->abs_vfid, qidx);
      qfle3_vf_handle_classification_eqe(adapter, vfq_get(vf, qidx), elem);
      break;
   case EVENT_RING_OPCODE_MULTICAST_RULES:
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: got VF [%d:%d] set mcast ramrod\n",
         vf->abs_vfid, qidx);
      qfle3_vf_handle_mcast_eqe(adapter, vf);
      break;
   case EVENT_RING_OPCODE_FILTERS_RULES:
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: got VF [%d:%d] set rx-mode ramrod\n",
         vf->abs_vfid, qidx);
      qfle3_vf_handle_filters_eqe(adapter, vf);
      break;
   case EVENT_RING_OPCODE_RSS_UPDATE_RULES:
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: got VF [%d:%d] RSS update ramrod\n",
         vf->abs_vfid, qidx);
      qfle3_vf_handle_rss_update_eqe(adapter, vf);
   case EVENT_RING_OPCODE_VF_FLR:
   case EVENT_RING_OPCODE_MALICIOUS_VF:
      /* Do nothing for now */
      return 0;
   }

   return 0;
}



/* must be called after the number of PF queues and the number of VFs are
 * both known
 */
static void qfle3_iov_static_resc(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   struct vf_pf_resc_request *resc = &vf->alloc_resc;

   /* will be set only during VF-ACQUIRE */
   resc->num_rxqs = 0;
   resc->num_txqs = 0;
   resc->num_mac_filters = VF_MAC_CREDIT_CNT;
   resc->num_vlan_filters = VF_VLAN_CREDIT_CNT;
   /* no real limitation */
   resc->num_mc_filters = 0;
   /* num_sbs already set */
   resc->num_sbs = vf->sb_count;
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Setting VF Resources to num_mac_filters:%d and num_vlan_filters:%d\n",
                         resc->num_mac_filters, resc->num_vlan_filters);

}

static vmk_Bool qfle3_tlv_supported(vmk_uint16 tlvtype)
{
        return CHANNEL_TLV_NONE < tlvtype && tlvtype < CHANNEL_TLV_MAX;
}


void qfle3_lock_vf_pf_channel(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                                 enum channel_tlvs tlv)
{
   /* we don't lock the channel for unsupported tlvs */
   if (!qfle3_tlv_supported(tlv)) {
      QFLE3_ERR("SRIOV: BC: Attempting to lock with unsupported tlv. Aborting\n");
      return;
   }
   /* lock the channel */
   vmk_SemaLock(&vf->op_mutex);

   /* record the locking op */
   vf->op_current = tlv;

   /* log the lock */
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF[%d]: vf pf channel locked by %s\n", vf->abs_vfid, channel_tlvs_string[tlv]);    
}

void qfle3_unlock_vf_pf_channel(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                                   enum channel_tlvs expected_tlv)
{
   enum channel_tlvs current_tlv;
   if (!vf) {
      QFLE3_ERR("SRIOV: BC: VF was %p\n", vf);
      return;
   }

   current_tlv = vf->op_current;
   /* we don't unlock the channel for unsupported tlvs */
   if (!qfle3_tlv_supported(expected_tlv))
      return;
   
   if ( expected_tlv != vf->op_current )
      QFLE3_ERR("SRIOV: BC: WARNING: lock mismatch: expected %s found %s\n" , channel_tlvs_string[expected_tlv], channel_tlvs_string[vf->op_current]);  

   /* record the locking op */
   vf->op_current = CHANNEL_TLV_NONE;

   /* unlock the channel */
   vmk_SemaUnlock(&vf->op_mutex);

   /* log the unlock */
  QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF[%d]: vf pf channel unlocked by %s\n", vf->abs_vfid, channel_tlvs_string[current_tlv]);	 
}

/* this works only on !E1h */
static int qfle3_copy32_vf_dmae(struct qfle3_adapter *adapter, vmk_uint8 from_vf, ecore_dma_addr_t pf_addr,
                                  vmk_uint8 vfid, vmk_uint32 vf_addr_hi, vmk_uint32 vf_addr_lo, vmk_uint32 len32)
{
   struct dmae_command dmae;

   if (CHIP_IS_E1x(adapter)) {
      QFLE3_ERR("SRIOV: BC: Chip revision does not support VFs\n");
      return DMAE_NOT_RDY;
   }
   
   if (!adapter->dmae_ready) {
      QFLE3_ERR("SRIOV: BC: DMAE is not ready, can not copy\n");
      return DMAE_NOT_RDY;
   }
   /* set opcode and fixed command fields */
   qfle3_prep_dmae_with_comp(adapter, &dmae, DMAE_SRC_PCI, DMAE_DST_PCI);

   if (from_vf) {
      dmae.opcode_iov = (vfid << DMAE_COMMAND_SRC_VFID_SHIFT) |
         (DMAE_SRC_VF << DMAE_COMMAND_SRC_VFPF_SHIFT) |
         (DMAE_DST_PF << DMAE_COMMAND_DST_VFPF_SHIFT);

      dmae.opcode |= (DMAE_C_DST << DMAE_COMMAND_C_FUNC_SHIFT);
      dmae.src_addr_lo = vf_addr_lo;
      dmae.src_addr_hi = vf_addr_hi;
      dmae.dst_addr_lo = U64_LO(pf_addr);
      dmae.dst_addr_hi = U64_HI(pf_addr);
   } else {
      dmae.opcode_iov = (vfid << DMAE_COMMAND_DST_VFID_SHIFT) |
         (DMAE_DST_VF << DMAE_COMMAND_DST_VFPF_SHIFT) |
         (DMAE_SRC_PF << DMAE_COMMAND_SRC_VFPF_SHIFT);

      dmae.opcode |= (DMAE_C_SRC << DMAE_COMMAND_C_FUNC_SHIFT);
      dmae.src_addr_lo = U64_LO(pf_addr);
      dmae.src_addr_hi = U64_HI(pf_addr);
      dmae.dst_addr_lo = vf_addr_lo;
      dmae.dst_addr_hi = vf_addr_hi;
   }
   dmae.len = len32;
   /* issue the command and wait for completion */
   return qfle3_issue_dmae_with_comp(adapter, &dmae, QFLE3_SP(adapter, wb_comp));
} 

static void qfle3_vf_prep_rx_mode(struct qfle3_adapter *adapter, vmk_uint8 qid,
                                    struct ecore_rx_mode_ramrod_params *ramrod,
                                    struct qfle3_virtf *vf, unsigned long accept_flags)
{
   struct qfle3_vf_queue *vfq = vfq_get(vf, qid);
   vmk_Memset(ramrod, 0, sizeof(*ramrod));
   ramrod->cid = vfq->cid;
   ramrod->cl_id = vfq_cl_id(vf, vfq);
   ramrod->rx_mode_obj = &adapter->rx_mode_obj;
   ramrod->func_id = FW_VF_HANDLE(vf->abs_vfid);
   ramrod->rx_accept_flags = accept_flags;
   ramrod->tx_accept_flags = accept_flags;
   ramrod->pstate = &vf->filter_state;
   ramrod->state = ECORE_FILTER_RX_MODE_PENDING;

   ECORE_SET_BIT(ECORE_FILTER_RX_MODE_PENDING, &vf->filter_state);
   ECORE_SET_BIT(RAMROD_RX, &ramrod->ramrod_flags);
   ECORE_SET_BIT(RAMROD_TX, &ramrod->ramrod_flags);

   ramrod->rdata = qfle3_vf_sp(adapter, vf, rx_mode_rdata.e2);
   ramrod->rdata_mapping = qfle3_vf_sp_map(adapter,  vf, rx_mode_rdata.e2);

}

int qfle3_vf_rxmode(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                       int qid, unsigned long accept_flags)
{
   struct ecore_rx_mode_ramrod_params ramrod;
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: vf[%d]\n", vf->abs_vfid);
   qfle3_vf_prep_rx_mode(adapter, qid, &ramrod, vf, accept_flags);
   ECORE_SET_BIT(RAMROD_COMP_WAIT, &ramrod.ramrod_flags);
   vfq_get(vf, qid)->accept_flags = ramrod.rx_accept_flags;
   return ecore_config_rx_mode(adapter, &ramrod);
}

static int qfle3_vf_vlan_mac_clear(struct qfle3_adapter *adapter, struct  qfle3_virtf *vf,
                                     int qid, vmk_Bool drv_only, int type)
{
   struct ecore_vlan_mac_ramrod_params ramrod;
   int rc;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC:vf[%d] - deleting all %s\n", vf->abs_vfid,
               (type == QFLE3_VF_FILTER_VLAN_MAC) ? "VLAN-MACs" :
               (type == QFLE3_VF_FILTER_MAC) ? "MACs" : "VLANs");

   /* Prepare ramrod params */
   vmk_Memset(&ramrod, 0, sizeof(struct ecore_vlan_mac_ramrod_params));
   if (type == QFLE3_VF_FILTER_VLAN_MAC) {
      ECORE_SET_BIT(ECORE_ETH_MAC, &ramrod.user_req.vlan_mac_flags);
      ramrod.vlan_mac_obj = &(vf->vfqs[qid].vlan_mac_obj);
   } else if (type == QFLE3_VF_FILTER_MAC) {
      ECORE_SET_BIT(ECORE_ETH_MAC, &ramrod.user_req.vlan_mac_flags);
      ramrod.vlan_mac_obj = &(vf->vfqs[qid].mac_obj);
   } else {
      ramrod.vlan_mac_obj = &(vf->vfqs[qid].vlan_obj);
   }
   ramrod.user_req.cmd = ECORE_VLAN_MAC_DEL;

   ECORE_SET_BIT(RAMROD_EXEC, &ramrod.ramrod_flags);
   if(drv_only)
      ECORE_SET_BIT(RAMROD_DRV_CLR_ONLY, &ramrod.ramrod_flags);
   else
      ECORE_SET_BIT(RAMROD_COMP_WAIT, &ramrod.ramrod_flags);
   /* Start deleting */
   rc = ramrod.vlan_mac_obj->delete_all(adapter, ramrod.vlan_mac_obj, &ramrod.user_req.vlan_mac_flags,
                                           &ramrod.ramrod_flags);

   if (rc) {
      QFLE3_ERR("Failed to delete all %s\n",
                   (type == QFLE3_VF_FILTER_VLAN_MAC) ? "VLAN-MACs" :
                   (type == QFLE3_VF_FILTER_MAC) ? "MACs" : "VLANs");
      return rc;
   }
   return 0;
}

int qfle3_vf_mcast(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                      qfle3_mac_addr_t *mcasts, int mc_num, vmk_Bool drv_only)
{
   struct ecore_mcast_list_elem *mc = NULL;
   struct ecore_mcast_ramrod_params mcast;
   int rc=0;
   int  i;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: vf[%d]\n", vf->abs_vfid);

   /* Prepare Multicast command */
   vmk_Memset(&mcast, 0, sizeof(struct ecore_mcast_ramrod_params));
   mcast.mcast_obj = &vf->mcast_obj;
   if (drv_only)
      ECORE_SET_BIT(RAMROD_DRV_CLR_ONLY, &mcast.ramrod_flags);
   else
      ECORE_SET_BIT(RAMROD_COMP_WAIT, &mcast.ramrod_flags);
   if(mc_num) {
      mc = qfle3_heap_alloc(mc_num * sizeof(struct ecore_mcast_list_elem));
      if (!mc) {
         QFLE3_ERR("SRIOV: BC: Cannot Configure multicasts due to lack of memory\n");
         return -1;
      }
   }
   /* clear existing mcasts */
   mcast.mcast_list_len = vf->mcast_list_len;
   vf->mcast_list_len = mc_num;
   rc = ecore_config_mcast(adapter, &mcast, ECORE_MCAST_CMD_DEL);
   if (rc) {
      QFLE3_ERR("SRIOV: BC: Failed to remove multicasts\n");
      qfle3_heap_free(mc);
      return rc;
   }
   /* update mcast list on the ramrod params */
   if (mc_num) {
      ECORE_LIST_INIT(&mcast.mcast_list);
      for (i = 0; i < mc_num; i++) {
         mc[i].mac = mcasts[i];
         ECORE_LIST_PUSH_TAIL(&mc[i].link, &mcast.mcast_list);
      }

      /* add new mcasts */
      mcast.mcast_list_len = mc_num;
      rc = ecore_config_mcast(adapter, &mcast, ECORE_MCAST_CMD_ADD);
      if (rc)
         QFLE3_ERR("SRIOV: BC: Faled to add multicasts\n");
         qfle3_heap_free(mc);
   }
   return rc;
}

static int qfle3_vf_queue_destroy(struct qfle3_adapter *adapter, struct qfle3_virtf *vf, int qid)
{
   enum ecore_queue_cmd cmds[] = {ECORE_Q_CMD_HALT,
                                  ECORE_Q_CMD_TERMINATE,
                                  ECORE_Q_CMD_CFC_DEL};
   struct ecore_queue_state_params q_params;
   int rc,i;
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: vf[%d]\n", vf->abs_vfid);

   /* Prepare ramrod information */
   vmk_Memset(&q_params, 0, sizeof(struct ecore_queue_state_params));
   q_params.q_obj = &(vf->vfqs[qid].sp_obj);
   ECORE_SET_BIT(RAMROD_COMP_WAIT, &q_params.ramrod_flags);

   if (ecore_get_q_logical_state(adapter, q_params.q_obj) == ECORE_Q_LOGICAL_STATE_STOPPED) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: queue was already stopped. Aborting gracefully\n");
      goto out;
   }
   /* Run Queue 'destruction' ramrods */
   for (i = 0; i < ARRAY_SIZE(cmds); i++) {
      q_params.cmd = cmds[i];
      rc = ecore_queue_state_change(adapter, &q_params);
      if (rc) {
         QFLE3_ERR("SRIOV: Failed to run Queue command %d\n", cmds[i]);
         return rc;
      }
   }
out:
   /* Clean Context */
   if (vf->vfqs[qid].cxt) {
      vf->vfqs[qid].cxt->ustorm_ag_context.cdu_usage = 0;
      vf->vfqs[qid].cxt->xstorm_ag_context.cdu_reserved = 0;
   }
   return 0;
} 

int qfle3_vf_queue_teardown(struct qfle3_adapter *adapter, struct qfle3_virtf *vf, int qid)
{
   int rc=0;
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: vf[%d:%d]\n", vf->abs_vfid, qid);

   if (ecore_get_q_logical_state(adapter, &(vf->vfqs[qid].sp_obj)) == ECORE_Q_LOGICAL_STATE_STOPPED) {   
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: queue was already stopped. Aborting filter removal at this time\n");
      return rc;
   }


   /* Remove all classification configuration for leading queue */
   if (qid == LEADING_IDX) {
      rc = qfle3_vf_rxmode(adapter, vf, qid, 0);
      if (rc)
         goto op_err;
      /* Remove filtering if feasible */
      if (qfle3_validate_vf_sp_objs(adapter, vf, VMK_TRUE)) {
         rc = qfle3_vf_vlan_mac_clear(adapter, vf, qid, VMK_FALSE, QFLE3_VF_FILTER_VLAN_MAC);
         if(rc)
            goto op_err;
         rc = qfle3_vf_vlan_mac_clear(adapter, vf, qid, VMK_FALSE, QFLE3_VF_FILTER_VLAN);
         if(rc)
            goto op_err;
         rc = qfle3_vf_vlan_mac_clear(adapter, vf, qid, VMK_FALSE, QFLE3_VF_FILTER_MAC);
         if(rc)
            goto op_err;
         rc = qfle3_vf_mcast(adapter, vf, NULL, 0, VMK_FALSE);
         if(rc)
            goto op_err;
      }
   }
   /* Destroy queue */
   rc = qfle3_vf_queue_destroy(adapter, vf, qid);
   if (rc)
      goto op_err;

   if (qid == LEADING_IDX) {
      struct ecore_vlan_mac_obj *obj;
      obj = &(vf->vfqs[qid].mac_obj);
      QFLE3_DBG(QFLE3_DBG_IOV, "Destroying spinlock exe_queue.lock for mac_obj\n");
      qfle3_destroy_spinlock(obj->exe_queue.lock);
      obj = &(vf->vfqs[qid].vlan_mac_obj);
      QFLE3_DBG(QFLE3_DBG_IOV, "Destroying spinlock exe_queue.lock for vlan_mac_obj\n");
      qfle3_destroy_spinlock(obj->exe_queue.lock);
      obj = &(vf->vfqs[qid].vlan_obj);
      QFLE3_DBG(QFLE3_DBG_IOV, "Destroying spinlock exe_queue.lock for vlan_obj\n");
      qfle3_destroy_spinlock(obj->exe_queue.lock);
   }

  /* Terminate queue */
   if(((vf)->vfqs[qid].sp_obj.state) != ECORE_Q_STATE_RESET) {
      struct ecore_queue_state_params qstate;

      vmk_Memset(&qstate, 0, sizeof(struct ecore_queue_state_params));
      qstate.q_obj = &((vf)->vfqs[qid].sp_obj);
      qstate.q_obj->state = ECORE_Q_STATE_STOPPED;
      qstate.cmd = ECORE_Q_CMD_TERMINATE;
      ECORE_SET_BIT(RAMROD_COMP_WAIT, &qstate.ramrod_flags);
      rc = ecore_queue_state_change(adapter, &qstate);
      if (rc)
         goto op_err;
   }

   return rc;

op_err:
   QFLE3_ERR("SRIOV: BC: vf[%d:%d] error: rc %d\n",vf->abs_vfid, qid, rc);
   return rc;
}

static void qfle3_vf_igu_disable(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   vmk_uint32 val;

   /* clear the VF configuration - pretend */
   qfle3_pretend_func(adapter, HW_VF_HANDLE(adapter, vf->abs_vfid));
   val = REG_RD(adapter, IGU_REG_VF_CONFIGURATION);
   val &= ~(IGU_VF_CONF_MSI_MSIX_EN | IGU_VF_CONF_SINGLE_ISR_EN |
              IGU_VF_CONF_FUNC_EN | IGU_VF_CONF_PARENT_MASK);
   REG_WR(adapter, IGU_REG_VF_CONFIGURATION, val);
   qfle3_pretend_func(adapter, QFLE3_ABS_FUNC(adapter));
}

static void qfle3_vf_qtbl_set_q(struct qfle3_adapter *adapter, vmk_uint8 abs_vfid, vmk_uint8 qid,
                                   vmk_uint8 enable)
{
   vmk_uint32 reg = PXP_REG_HST_ZONE_PERMISSION_TABLE + qid * 4;
   vmk_uint32 val = enable ? (abs_vfid | (1 << 6)) : 0;
   REG_WR(adapter, reg, val);
}

static void qfle3_vf_clr_qtbl(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   int i;
   for( i=0; i < vf->alloc_resc.num_rxqs; i++)
      qfle3_vf_qtbl_set_q(adapter, vf->abs_vfid,
                            vfq_qzone_id(vf, vfq_get(vf, i)), VMK_FALSE);
}

struct set_vf_state_cookie {
   struct qfle3_virtf *vf;
   vmk_uint8 state;
};

static void qfle3_set_vf_state(void *cookie)
{
    struct set_vf_state_cookie *p = (struct set_vf_state_cookie *)cookie;

    p->vf->state = p->state;
}

int qfle3_vf_close(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   int rc = 0, i;
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF[%d]\n", vf->abs_vfid);
   /* Close all queues */
   for (i = 0; i < vf_rxq_count(vf); i++) {
      rc = qfle3_vf_queue_teardown(adapter, vf, i);
      if (rc)
         goto op_err;
   }
   /* disable the interrupts */
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Disabling igu\n");
   qfle3_vf_igu_disable(adapter, vf);
   /* disable the VF */
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Clearing qtbl\n");
   qfle3_vf_clr_qtbl(adapter, vf);

   /* need to make sure there are no outstanding stats ramrods which may
   * cause the device to access the VF's stats buffer which it will free
    * as soon as we return from the close flow.*/
   {                                                      
      struct set_vf_state_cookie cookie;
      cookie.vf = vf;
      cookie.state = VF_ACQUIRED;
      rc = qfle3_stats_safe_exec(adapter, qfle3_set_vf_state, &cookie);
      if (rc)
         goto op_err;
   }
   vf->state = VF_ACQUIRED;  
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Set state to acquired\n");
   return 0;

op_err:
   QFLE3_ERR("SRIOV: BC: vf[%d] CLOSE error: rc %d\n", vf->abs_vfid, rc);
   return rc;
}

/*FLR Routines: */
static void qfle3_vf_free_resc(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   /* reset the state variables */
   qfle3_iov_static_resc(adapter, vf);
   vf->state = VF_FREE;
}

/* VF release can be called either: 1. The VF was acquired but
 * not enabled 2. the vf was enabled or in the process of being
 * enabled
 */
int qfle3_vf_free(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   int rc;
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF[%d] STATE: %s\n", vf->abs_vfid,
               vf->state == VF_FREE ? "Free" :
               vf->state == VF_ACQUIRED ? "Acquired" :
               vf->state == VF_ENABLED ? "Enabled" :
               vf->state == VF_RESET ? "Reset" :
               "Unknown");
   switch (vf->state) {
   case VF_ENABLED:
      rc = qfle3_vf_close(adapter, vf);
      if(rc)
         goto op_err;
         /* Fallthrough to release resources */
   case VF_ACQUIRED:
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: About to free resources\n");
      qfle3_vf_free_resc(adapter, vf);
      break;
   case VF_FREE:
   case VF_RESET:
   default:
      break;
   }
   return 0;
op_err:
   QFLE3_ERR("SRIOV: BC: VF[%d] RELEASE error: rc %d\n", vf->abs_vfid, rc);
   return rc;
}



/* VF release ~ VF close + VF release-resources
 * Release is the ultimate SW shutdown and is called whenever an
 * irrecoverable error is encountered.
 */
int qfle3_vf_release(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   int rc;
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: PF releasing vf %d\n", vf->abs_vfid);
   qfle3_lock_vf_pf_channel(adapter, vf, CHANNEL_TLV_PF_RELEASE_VF);
   rc = qfle3_vf_free(adapter, vf);
   if (rc)
      QFLE3_ERR("SRIOV: BC: VF[%d] Failed to allocate resources for release op- rc=%d\n",
                      vf->abs_vfid, rc);
   qfle3_unlock_vf_pf_channel(adapter, vf, CHANNEL_TLV_PF_RELEASE_VF);
   return rc;
}

static vmk_Bool qfle3_vf_mbx_is_windows_vm(struct qfle3_adapter *adapter, struct vfpf_acquire_tlv *acquire)
{
   /* Windows driver does one of three things:
   * 1. Old driver doesn't have bulletin board address set.
   * 2. 'Middle' driver sends mc_num == 32.
   * 3. New driver sets the OS field  */
   if (!acquire->bulletin_addr || acquire->resc_request.num_mc_filters == 32 ||
      ((acquire->vfdev_info.vf_os & VF_OS_MASK) == VF_OS_WINDOWS))
       return VMK_TRUE;

   return VMK_FALSE;
}

/* Fins a TLV by type in a TLV buffer; If found, returns pointer to the TLV */
static void *qfle3_search_tlv_list(struct qfle3_adapter *adapter, void *tlvs_list,
                                      enum channel_tlvs req_tlv)
{
   struct channel_tlv *tlv = (struct channel_tlv *)tlvs_list;
   do {
      if (tlv->type == req_tlv)
      return tlv;
      if (!tlv->length) {
         QFLE3_ERR("SRIOV: BC: Found TLV with length 0\n");
         return NULL;
      }
      tlvs_list += tlv->length;
      tlv = (struct channel_tlv *)tlvs_list;
   } while (tlv->type != CHANNEL_TLV_LIST_END);

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: TLV list does not contain %s TLV\n", channel_tlvs_string[req_tlv]);  

   return NULL;
}

static int qfle3_vf_mbx_acquire_chk_dorq(struct qfle3_adapter *adapter,
                                         struct qfle3_virtf *vf,
                                         struct qfle3_vf_mbx *mbx)
{
   /* Linux drivers which correctly set the doorbell size also
   * send a physical port request
   */
   if (qfle3_search_tlv_list(adapter, &mbx->msg->req, CHANNEL_TLV_PHYS_PORT_ID))
      return 0;

   /* Issue does not exist in windows VMs */
   if (qfle3_vf_mbx_is_windows_vm(adapter, &mbx->msg->req.acquire))
      return 0;

   return -1;
}

vmk_uint8 qfle3_vf_max_queue_cnt(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   return (vmk_uint8)MIN( ((vmk_uint8)MIN( vf_sb_count(vf), QFLE3_CIDS_PER_VF)), QFLE3_VF_MAX_QUEUES);
}


static int qfle3_vf_chk_avail_resc(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                            struct vf_pf_resc_request *req_resc)
{
        vmk_uint8 rxq_cnt = vf_rxq_count(vf) ? : qfle3_vf_max_queue_cnt(adapter, vf);
        vmk_uint8 txq_cnt = vf_txq_count(vf) ? : qfle3_vf_max_queue_cnt(adapter, vf);

        return ((req_resc->num_rxqs <= rxq_cnt) &&
                (req_resc->num_txqs <= txq_cnt) &&
                (req_resc->num_sbs <= vf_sb_count(vf))   &&
                (req_resc->num_mac_filters <= vf_mac_rules_cnt(vf)) &&
                (req_resc->num_vlan_filters <= vf_vlan_rules_cnt(vf)));
}

static void qfle3_vfq_init(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                           struct qfle3_vf_queue *q)
{
   vmk_uint8 cl_id = vfq_cl_id(vf, q);
   vmk_uint8 func_id = FW_VF_HANDLE(vf->abs_vfid);
   unsigned long q_type = 0;

   ECORE_SET_BIT(ECORE_Q_TYPE_HAS_TX, &q_type);
   ECORE_SET_BIT(ECORE_Q_TYPE_HAS_RX, &q_type);

   /* Queue State object */
   ecore_init_queue_obj(adapter, &q->sp_obj,
                        cl_id, &q->cid, 1, func_id,
                        qfle3_vf_sp(adapter, vf, q_data),
                        qfle3_vf_sp_map(adapter, vf, q_data),
                        q_type);

   /* sp indication is set only when vlan/mac/etc. are initialized */
   q->sp_initialized = VMK_FALSE;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Initialized vf %d's queue object. func id set to %d. cid set to 0x%x\n",
              vf->abs_vfid, q->sp_obj.func_id, q->cid);
}

/* CORE VF API */
int qfle3_vf_acquire(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                     struct vf_pf_resc_request *resc)
{
   int base_vf_cid = (adapter->vfdb->sriov.first_vf_in_pf + vf->index) * QFLE3_CIDS_PER_VF;

   union cdu_context *base_cxt = (union cdu_context *)
      QFLE3_VF_CXT_PAGE(adapter, base_vf_cid/ILT_PAGE_CIDS)->addr + (base_vf_cid & (ILT_PAGE_CIDS-1));
   int i;

   /* if state is 'acquired' the VF was not released or FLR'd, in
   * this case the returned resources match the acquired already
   * acquired resources. Verify that the requested numbers do
   * not exceed the already acquired numbers.  */
   if (vf->state == VF_ACQUIRED) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF[%d] Trying to re-acquire resources (VF was not released or FLR'd)\n",
                   vf->abs_vfid);

      if (!qfle3_vf_chk_avail_resc(adapter, vf, resc)) {
         QFLE3_ERR("SRIOV: BC: VF[%d] When re-acquiring resources, requested numbers must be <= then previously acquired numbers\n",
                            vf->abs_vfid);
         return -1;
      }
      return 0;
   }
   /* Otherwise vf state must be 'free' or 'reset' */
   if (vf->state != VF_FREE && vf->state != VF_RESET) {
     QFLE3_ERR("SRIOV: BC: VF[%d] Can not acquire a VF with state %d\n",
                vf->abs_vfid, vf->state);
     return -1;
   }
   /* static allocation:
   *the global maximum number are fixed per VF. Fail the request if
   * requested number exceed these globals
   */
   if (!qfle3_vf_chk_avail_resc(adapter, vf, resc)) {
      QFLE3_ERR("SRIOV: BC: cannot fulfill vf resource request. Placing maximal available values in response\n");
                /* set the max resource in the vf */
         return -1;
   }
   /* Set resources counters - 0 request means max available */
   vf->alloc_resc.num_sbs = resc->num_sbs;
   vf->alloc_resc.num_rxqs = resc->num_rxqs ? : qfle3_vf_max_queue_cnt(adapter, vf);
   vf->alloc_resc.num_txqs = resc->num_txqs ? : qfle3_vf_max_queue_cnt(adapter, vf);

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Fulfilling vf request: sb count %d, tx_count %d, rx_count %d, mac_rules_count %d, vlan_rules_count %d\n",
      (vf)->alloc_resc.num_sbs, (vf)->alloc_resc.num_rxqs, (vf)->alloc_resc.num_txqs, (vf)->alloc_resc.num_mc_filters, (vf)->alloc_resc.num_vlan_filters);

   /* Initialize the queues */
   if (!vf->vfqs) {
      QFLE3_ERR("SRIOV: BC: vf->vfqs was not allocated\n");
      return -1;
   }

   for(i=0; i < vf->alloc_resc.num_rxqs; i++) {
      struct qfle3_vf_queue *q = vfq_get(vf, i);

      if (!q) {
         QFLE3_ERR("SRIOV: BC: q number %d was not allocated\n", i);
         return -1;
         }

         q->index = i;
         q->cxt = &((base_cxt + i)->eth);
         q->cid = QFLE3_FIRST_VF_CID + base_vf_cid + i;

         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VFQ[%d:%d]: index %d, cid 0x%x, cxt %p\n",
                   vf->abs_vfid, i, q->index, q->cid, q->cxt);

         /* init SP objects */
         qfle3_vfq_init(adapter, vf, q);
   }
   vf->state = VF_ACQUIRED;
   return 0;
}

static inline int qfle3_pfvf_status_codes(int rc)
{
        switch (rc) {
        case 0:
                return PFVF_STATUS_SUCCESS;
        case -1:
                return PFVF_STATUS_NO_RESOURCE;
        default:
                return PFVF_STATUS_FAILURE;
        }
}

/**
 * qfle3_fill_fw_str - Fill buffer with FW version string.
 *
 * @adapter:        driver handle
 * @buf:       character buffer to fill with the fw name
 * @buf_len:   length of the above buffer
 *
 */
void qfle3_fill_fw_str(struct qfle3_adapter *adapter, char *buf, vmk_ByteCount buf_len)    
{
   vmk_uint8 phy_fw_ver[PHY_FW_VER_LEN];
   phy_fw_ver[0] = '\0';
   elink_get_ext_phy_fw_version(&adapter->link_params, phy_fw_ver, PHY_FW_VER_LEN);
   vmk_Strncpy(buf, adapter->fw_ver, buf_len);
   vmk_Snprintf(buf + vmk_Strnlen(adapter->fw_ver,8), 32 - vmk_Strnlen(adapter->fw_ver,8),
                         "bc %d.%d.%d%s%s",
                         (adapter->hw_info.bc_ver & 0xff0000) >> 16,
                         (adapter->hw_info.bc_ver & 0xff00) >> 8,
                         (adapter->hw_info.bc_ver & 0xff),
                         ((phy_fw_ver[0] != '\0') ? " phy " : ""), phy_fw_ver);
}

/* place a given tlv on the tlv buffer at a given offset */
static void qfle3_add_tlv(struct qfle3_adapter *adapter, void *tlvs_list,
                         vmk_uint16 offset, vmk_uint16 type, vmk_uint16 length)
{
   struct channel_tlv *tl = (struct channel_tlv *)(tlvs_list + offset);
   tl->type = type;
   tl->length = length;
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Adding tl->type %d : %s of Length %d\n", tl->type, channel_tlvs_string[tl->type], tl->length);
}

static void qfle3_vf_mbx_resp_phys_port(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                                        void *buffer, vmk_uint16 *offset)
{
   struct vfpf_port_phys_id_resp_tlv *port_id;

   if (!(adapter->flags & HAS_PHYS_PORT_ID)){
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: adapter->flags NOT SET FOR HAS_PHYS_PORT_ID\n");
      return;
   }

   qfle3_add_tlv(adapter, buffer, *offset, CHANNEL_TLV_PHYS_PORT_ID,
                      sizeof(struct vfpf_port_phys_id_resp_tlv));

   port_id = (struct vfpf_port_phys_id_resp_tlv*)(((vmk_uint8 *)buffer) + *offset);
   vmk_Memcpy(port_id->id, adapter->phys_port_id, ETH_ALEN);

   /* Offset should continue representing the offset to the tail
   * of TLV data (outside this function scope)
   */
   *offset += sizeof(struct vfpf_port_phys_id_resp_tlv);

}

static void qfle3_vf_mbx_resp_fp_hsi_ver(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                                         void *buffer, vmk_uint16 *offset)
{
        struct vfpf_fp_hsi_resp_tlv *fp_hsi;

        qfle3_add_tlv(adapter, buffer, *offset, CHANNEL_TLV_FP_HSI_SUPPORT,
                      sizeof(struct vfpf_fp_hsi_resp_tlv));

        fp_hsi = (struct vfpf_fp_hsi_resp_tlv*)(((vmk_uint8 *)buffer) + *offset);
        fp_hsi->is_supported = (vf->fp_hsi > ETH_FP_HSI_VERSION) ? 0 : 1;

        /* Offset should continue representing the offset to the tail
         * of TLV data (outside this function scope)
         */
        *offset += sizeof(struct vfpf_fp_hsi_resp_tlv);
}

/* list the types and lengths of the tlvs on the buffer */
static void qfle3_dp_tlv_list(struct qfle3_adapter *adapter, void *tlvs_list)
{
   int i = 1;
   struct channel_tlv *tlv = (struct channel_tlv *)tlvs_list;

   while (tlv->type != CHANNEL_TLV_LIST_END) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: TLV number: %d TLV Type: %s  length: %d\n", tlv->type,  channel_tlvs_string[tlv->type], tlv->length);  

      /* advance to next tlv */
      tlvs_list += tlv->length;

      /* cast general tlv list pointer to channel tlv header*/
      tlv = (struct channel_tlv *)tlvs_list;

      i++;

      /* break condition for this loop */
      if (i > MAX_TLVS_IN_LIST) {
         QFLE3_ERR("SRIOV: BC: corrupt tlvs");
         return;
      }
   }

   /* output last tlv */
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: TLV number: %d TLV Type: %s  length: %d\n", tlv->type,  channel_tlvs_string[tlv->type], tlv->length);
}

static void qfle3_vf_mbx_resp_send_msg(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                                       int vf_rc)
{
   struct qfle3_vf_mbx *mbx = QFLE3_VF_MBX(adapter, vf->index);;
   struct pfvf_general_resp_tlv *resp = &mbx->msg->resp.general_resp;
   ecore_dma_addr_t pf_addr;
   vmk_uint64 vf_addr;
   int rc;

   qfle3_dp_tlv_list(adapter, resp);
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Mailbox vf address hi 0x%x, lo 0x%x, offset 0x%x\n",
           mbx->vf_addr_hi, mbx->vf_addr_lo, mbx->first_tlv.resp_msg_offset);

   resp->hdr.status = qfle3_pfvf_status_codes(vf_rc);
   /* send response */
   vf_addr = HILO_U64(mbx->vf_addr_hi, mbx->vf_addr_lo) +
             mbx->first_tlv.resp_msg_offset;
   pf_addr = mbx->msg_mapping +
            vmk_offsetof(struct qfle3_vf_mbx_msg, resp);

   /* Copy the response buffer. The first u64 is written afterwards, as
   * the vf is sensitive to the header being written
   */
   vf_addr += sizeof(vmk_uint64);
   pf_addr += sizeof(vmk_uint64);
   rc = qfle3_copy32_vf_dmae(adapter, VMK_FALSE, pf_addr, vf->abs_vfid,
                                  U64_HI(vf_addr),
                                  U64_LO(vf_addr),
                                  (sizeof(union pfvf_tlvs) - sizeof(vmk_uint64))/4);
   if (rc) {
      QFLE3_ERR("SRIOV: BC: Failed to copy response body to VF %d\n",vf->abs_vfid);
      goto mbx_error;
   }
   vf_addr -= sizeof(vmk_uint64);
   pf_addr -= sizeof(vmk_uint64);
   /* ack the FW */
   storm_memset_vf_mbx_ack(adapter, vf->abs_vfid);
   vmk_CPUMemFenceReadWrite();
   /* copy the response header including status-done field,
   * must be last dmae, must be after FW is acked
   */
    rc = qfle3_copy32_vf_dmae(adapter, VMK_FALSE, pf_addr, vf->abs_vfid,
                                  U64_HI(vf_addr),
                                  U64_LO(vf_addr),
                                  sizeof(vmk_uint64)/4);

   /* unlock channel mutex */
   qfle3_unlock_vf_pf_channel(adapter, vf, mbx->first_tlv.tl.type);
   if (rc) {
      QFLE3_ERR("SRIOV: BC: Failed to copy response status to VF %d\n",
                          vf->abs_vfid);
      goto mbx_error;
   }
   return;

mbx_error:
   qfle3_vf_release(adapter, vf);
}


static void qfle3_vf_mbx_acquire_resp(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                                      struct qfle3_vf_mbx *mbx, int vfop_status)
{
   int i;
   struct pfvf_acquire_resp_tlv *resp = &mbx->msg->resp.acquire_resp;
   struct pf_vf_resc *resc = &resp->resc;
   vmk_uint8 status = qfle3_pfvf_status_codes(vfop_status);
   vmk_uint16 length;

   vmk_Memset(resp, 0, sizeof(*resp));

   /* fill in pfdev info */
   resp->pfdev_info.chip_num = adapter->hw_info.chip_id;
   resp->pfdev_info.db_size = adapter->db_size;
   resp->pfdev_info.indices_per_sb = HC_SB_MAX_INDICES_E2;
   resp->pfdev_info.pf_cap = (PFVF_CAP_RSS |
                              PFVF_CAP_TPA |
                              PFVF_CAP_TPA_UPDATE |
                              PFVF_CAP_VLAN_FILTER);
   resp->pfdev_info.pf_cap |= PFVF_CAP_ALLOW_MAC;
   qfle3_esx_vf_reset_all_vmk_gvlans(adapter, vf);              
   qfle3_esx_vf_request_all_vmk_gvlans(adapter, vf);            
   qfle3_fill_fw_str(adapter, resp->pfdev_info.fw_ver,  
                          sizeof(resp->pfdev_info.fw_ver));

   if (status == PFVF_STATUS_NO_RESOURCE || status == PFVF_STATUS_SUCCESS) {
      /* set resources numbers, if status equals NO_RESOURCE these
      * are max possible numbers
      */
      resc->num_rxqs = vf_rxq_count(vf) ? : qfle3_vf_max_queue_cnt(adapter, vf);
      resc->num_txqs = vf_txq_count(vf) ? : qfle3_vf_max_queue_cnt(adapter, vf);
      resc->num_sbs = vf_sb_count(vf);
      resc->num_mac_filters = vf_mac_rules_cnt(vf);
      resc->num_vlan_filters = vf_vlan_rules_cnt(vf);
      resc->num_mc_filters = 0;
      resc->pf_link_speed = adapter->link_vars.line_speed;
      resc->pf_link_supported = adapter->port.supported[0];
      if (status == PFVF_STATUS_SUCCESS) {
         /* fill in the allocated resources */
         struct pf_vf_bulletin_content *bulletin = QFLE3_VF_BULLETIN(adapter, vf->index);
         for( i=0; i < vf->alloc_resc.num_rxqs; i++)
            resc->hw_qid[i] = vfq_qzone_id(vf, vfq_get(vf, i));

         for( i=0; i < vf->alloc_resc.num_sbs; i++) {
            resc->hw_sbs[i].hw_sb_id = vf_igu_sb(vf, i);
            resc->hw_sbs[i].sb_qid = vf_hc_qzone(vf, i);
         }

         /* if a mac has been set for this vf, supply it */
         if (bulletin->valid_bitmap & 1 << MAC_ADDR_VALID) {
            vmk_Memcpy(resc->current_mac_addr, bulletin->mac, ETH_ALEN);
         }
      }
   }

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF[%d] ACQUIRE_RESPONSE: pfdev_info- chip_num=0x%x, db_size=%d, idx_per_sb=%d, pf_cap=0x%x\n",
          vf->abs_vfid, resp->pfdev_info.chip_num, resp->pfdev_info.db_size, resp->pfdev_info.indices_per_sb, resp->pfdev_info.pf_cap);
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF[%d] ACQUIRE_RESPONSE: resources- n_rxq-%d, n_txq-%d, n_sbs-%d, n_macs-%d, n_vlans-%d, n_mcs-%d, fw_ver: '%s'\n",
          vf->abs_vfid, resc->num_rxqs, resc->num_txqs, resc->num_sbs, resc->num_mac_filters, resc->num_vlan_filters, resc->num_mc_filters, resp->pfdev_info.fw_ver);


   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: hw_qids- [ ");       
        for (i = 0; i < vf_rxq_count(vf); i++)
                QFLE3_DBG(QFLE3_DBG_IOV, "%d ", resc->hw_qid[i]);
        QFLE3_DBG(QFLE3_DBG_IOV, "], sb_info- [ ");
        for (i = 0; i < vf_sb_count(vf); i++)
           QFLE3_DBG(QFLE3_DBG_IOV,  "%d:%d ", resc->hw_sbs[i].hw_sb_id, resc->hw_sbs[i].sb_qid);
        QFLE3_DBG(QFLE3_DBG_IOV,  "]\n");
        /* prepare response */
   length = sizeof(struct pfvf_acquire_resp_tlv);
   qfle3_add_tlv(adapter, &mbx->msg->resp, 0, CHANNEL_TLV_ACQUIRE, length);

   /* Handle possible VF requests for physical port identifiers.
   * 'length' should continue to indicate the offset of the first empty
   * place in the buffer (i.e., where next TLV should be inserted)
   */
   if (qfle3_search_tlv_list(adapter, &mbx->msg->req, CHANNEL_TLV_PHYS_PORT_ID)){
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Add CHANNEL_TLV_PHYS_PORT_ID in response\n");
      qfle3_vf_mbx_resp_phys_port(adapter, vf, &mbx->msg->resp, &length);
   }

   /* `New' vfs will want to know if fastpath HSI is supported, since
   * if that's not the case they could print into system log the fact
   * the driver version must be updated.
   */
   qfle3_vf_mbx_resp_fp_hsi_ver(adapter, vf, &mbx->msg->resp, &length);
   qfle3_add_tlv(adapter, &mbx->msg->resp, length, CHANNEL_TLV_LIST_END,sizeof(struct channel_list_end_tlv));

   /* send the response */
   qfle3_vf_mbx_resp_send_msg(adapter, vf, vfop_status);
}


static void qfle3_vf_mbx_acquire(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                                 struct qfle3_vf_mbx *mbx)
{
   int rc;
   struct vfpf_acquire_tlv *acquire = &mbx->msg->req.acquire;

   /* log vfdef info */
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF[%d] ACQUIRE: vfdev_info- vf_id %d, vf_os %d resources- n_rxq-%d, n_txq-%d, n_sbs-%d, n_macs-%d, n_vlans-%d, n_mcs-%d\n",
      vf->abs_vfid, acquire->vfdev_info.vf_id, acquire->vfdev_info.vf_os,
      acquire->resc_request.num_rxqs, acquire->resc_request.num_txqs,
      acquire->resc_request.num_sbs, acquire->resc_request.num_mac_filters,
      acquire->resc_request.num_vlan_filters,
      acquire->resc_request.num_mc_filters);

   /* Prevent VFs with old drivers from loading, since they calculate
   * CIDs incorrectly requiring a VF-flr [VM reboot] in order to recover
   * while being upgraded.*/
   rc = qfle3_vf_mbx_acquire_chk_dorq(adapter, vf, mbx);
   if (rc) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF [%d] - Can't support acquire request due to doorbell mismatch. Please update VM driver\n",
                   vf->abs_vfid);
      goto out;
   }
   /* Verify the VF fastpath HSI can be supported by the loaded FW.
   * Linux vfs should be oblivious to changes between v0 and v2.*/
   if (qfle3_vf_mbx_is_windows_vm(adapter, &mbx->msg->req.acquire))
      vf->fp_hsi = acquire->vfdev_info.fp_hsi_ver;
   else
      vf->fp_hsi = (vmk_uint8)MAX(acquire->vfdev_info.fp_hsi_ver,ETH_FP_HSI_VER_2);
   
   if (vf->fp_hsi > ETH_FP_HSI_VERSION) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF [%d] - Can't support acquire request since VF requests a FW version which is too new [%02x > %02x]\n",
                   vf->abs_vfid, acquire->vfdev_info.fp_hsi_ver, ETH_FP_HSI_VERSION);
      rc = -1;
      goto out;
   }
   /* acquire the resources */
   rc = qfle3_vf_acquire(adapter, vf, &acquire->resc_request);

   /* store address of vf's bulletin board */
   vf->bulletin_map = acquire->bulletin_addr;
   if (acquire->vfdev_info.caps & VF_CAP_SUPPORT_EXT_BULLETIN) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC:VF[%d] supports long bulletin boards\n",vf->abs_vfid);
      vf->cfg_flags |= VF_CFG_EXT_BULLETIN;
   } else {
      vf->cfg_flags &= ~VF_CFG_EXT_BULLETIN;
   }

   if (acquire->vfdev_info.caps & VF_CAP_SUPPORT_VLAN_FILTER) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC:VF[%d] supports vlan filtering\n",vf->abs_vfid);
      vf->cfg_flags |= VF_CFG_VLAN_FILTER;
   } else {
      vf->cfg_flags &= ~VF_CFG_VLAN_FILTER;
   }

out:
   /* response */
   qfle3_vf_mbx_acquire_resp(adapter, vf, mbx, rc);
}

static void qfle3_vf_igu_ack_sb(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                                vmk_uint8 igu_sb_id, vmk_uint8 segment, vmk_uint16 index, vmk_uint8 op,
                                vmk_uint8 update)
{
   /* acking a VF sb through the PF - use the GRC */
   vmk_uint32 ctl;
   vmk_uint32 igu_addr_data = IGU_REG_COMMAND_REG_32LSB_DATA;
   vmk_uint32 igu_addr_ctl = IGU_REG_COMMAND_REG_CTRL;
   vmk_uint32 func_encode = vf->abs_vfid;
   vmk_uint32 addr_encode = IGU_CMD_E2_PROD_UPD_BASE + igu_sb_id;
   struct igu_regular cmd_data = {0};

   cmd_data.sb_id_and_flags =
              ((index << IGU_REGULAR_SB_INDEX_SHIFT) |
              (segment << IGU_REGULAR_SEGMENT_ACCESS_SHIFT) |
              (update << IGU_REGULAR_BUPDATE_SHIFT) |
              (op << IGU_REGULAR_ENABLE_INT_SHIFT));

   ctl = addr_encode << IGU_CTRL_REG_ADDRESS_SHIFT         |
              func_encode << IGU_CTRL_REG_FID_SHIFT             |
              IGU_CTRL_CMD_TYPE_WR << IGU_CTRL_REG_TYPE_SHIFT;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: write 0x%08x to IGU(via GRC) addr 0x%x\n",
              cmd_data.sb_id_and_flags, igu_addr_data);
   REG_WR(adapter, igu_addr_data, cmd_data.sb_id_and_flags);
   vmk_CPUMemFenceReadWrite();
   
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC:write 0x%08x to IGU(via GRC) addr 0x%x\n",
           ctl, igu_addr_ctl);
   REG_WR(adapter, igu_addr_ctl, ctl);
   vmk_CPUMemFenceReadWrite();
}


static void qfle3_vf_igu_reset(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   int i;
   vmk_uint32 val;

   /* Set VF masks and configuration - pretend */
   qfle3_pretend_func(adapter, HW_VF_HANDLE(adapter, vf->abs_vfid));

   REG_WR(adapter, IGU_REG_SB_INT_BEFORE_MASK_LSB, 0);
   REG_WR(adapter, IGU_REG_SB_INT_BEFORE_MASK_MSB, 0);
   REG_WR(adapter, IGU_REG_SB_MASK_LSB, 0);
   REG_WR(adapter, IGU_REG_SB_MASK_MSB, 0);
   REG_WR(adapter, IGU_REG_PBA_STATUS_LSB, 0);
   REG_WR(adapter, IGU_REG_PBA_STATUS_MSB, 0);

   val = REG_RD(adapter, IGU_REG_VF_CONFIGURATION);
   val |= (IGU_VF_CONF_FUNC_EN | IGU_VF_CONF_MSI_MSIX_EN);
   val &= ~IGU_VF_CONF_PARENT_MASK;
   val |= (QFLE3_ABS_FUNC(adapter) >> 1) << IGU_VF_CONF_PARENT_SHIFT;
   REG_WR(adapter, IGU_REG_VF_CONFIGURATION, val);

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: value in IGU_REG_VF_CONFIGURATION of vf %d after write is 0x%08x\n",
              vf->abs_vfid, val);
   qfle3_pretend_func(adapter, QFLE3_ABS_FUNC(adapter));

   /* iterate over all queues, clear sb consumer */
   for (i = 0; i < (vf)->alloc_resc.num_sbs; i++) {
      vmk_uint8 igu_sb_id = vf_igu_sb(vf, i);

      /* zero prod memory */
      REG_WR(adapter, IGU_REG_PROD_CONS_MEMORY + igu_sb_id * 4, 0);

      /* clear sb state machine */
      qfle3_igu_clear_sb_gen(adapter, vf->abs_vfid, igu_sb_id,
                                       VMK_FALSE /* VF */);

      /* disable + update */
      qfle3_vf_igu_ack_sb(adapter, vf, igu_sb_id, USTORM_ID, 0,
                                    IGU_INT_DISABLE, 1);
   }
}

static void qfle3_vf_enable_traffic(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   /* Reset vf in IGU  interrupts are still disabled */
   qfle3_vf_igu_reset(adapter, vf);

   /* pretend to enable the vf with the PBF */
   qfle3_pretend_func(adapter, HW_VF_HANDLE(adapter, vf->abs_vfid));
   REG_WR(adapter, PBF_REG_DISABLE_VF, 0);
   qfle3_pretend_func(adapter, QFLE3_ABS_FUNC(adapter));
}

/* This table was generated by the "crctable" program */
static const unsigned int crc32table[256] = {
                   /* 0x00 */ 0x00000000, 0x77073096, 0xEE0E612C, 0x990951BA,
                   /* 0x04 */ 0x076DC419, 0x706AF48F, 0xE963A535, 0x9E6495A3,
                   /* 0x08 */ 0x0EDB8832, 0x79DCB8A4, 0xE0D5E91E, 0x97D2D988,
                   /* 0x0C */ 0x09B64C2B, 0x7EB17CBD, 0xE7B82D07, 0x90BF1D91,
                   /* 0x10 */ 0x1DB71064, 0x6AB020F2, 0xF3B97148, 0x84BE41DE,
                   /* 0x14 */ 0x1ADAD47D, 0x6DDDE4EB, 0xF4D4B551, 0x83D385C7,
                   /* 0x18 */ 0x136C9856, 0x646BA8C0, 0xFD62F97A, 0x8A65C9EC,
                   /* 0x1C */ 0x14015C4F, 0x63066CD9, 0xFA0F3D63, 0x8D080DF5,
                   /* 0x20 */ 0x3B6E20C8, 0x4C69105E, 0xD56041E4, 0xA2677172,
                   /* 0x24 */ 0x3C03E4D1, 0x4B04D447, 0xD20D85FD, 0xA50AB56B,
                   /* 0x28 */ 0x35B5A8FA, 0x42B2986C, 0xDBBBC9D6, 0xACBCF940,
                   /* 0x2C */ 0x32D86CE3, 0x45DF5C75, 0xDCD60DCF, 0xABD13D59,
                   /* 0x30 */ 0x26D930AC, 0x51DE003A, 0xC8D75180, 0xBFD06116,
                   /* 0x34 */ 0x21B4F4B5, 0x56B3C423, 0xCFBA9599, 0xB8BDA50F,
                   /* 0x38 */ 0x2802B89E, 0x5F058808, 0xC60CD9B2, 0xB10BE924,
                   /* 0x3C */ 0x2F6F7C87, 0x58684C11, 0xC1611DAB, 0xB6662D3D,
                   /* 0x40 */ 0x76DC4190, 0x01DB7106, 0x98D220BC, 0xEFD5102A,
                   /* 0x44 */ 0x71B18589, 0x06B6B51F, 0x9FBFE4A5, 0xE8B8D433,
                   /* 0x48 */ 0x7807C9A2, 0x0F00F934, 0x9609A88E, 0xE10E9818,
                   /* 0x4C */ 0x7F6A0DBB, 0x086D3D2D, 0x91646C97, 0xE6635C01,
                   /* 0x50 */ 0x6B6B51F4, 0x1C6C6162, 0x856530D8, 0xF262004E,
                   /* 0x54 */ 0x6C0695ED, 0x1B01A57B, 0x8208F4C1, 0xF50FC457,
                   /* 0x58 */ 0x65B0D9C6, 0x12B7E950, 0x8BBEB8EA, 0xFCB9887C,
                   /* 0x5C */ 0x62DD1DDF, 0x15DA2D49, 0x8CD37CF3, 0xFBD44C65,
                   /* 0x60 */ 0x4DB26158, 0x3AB551CE, 0xA3BC0074, 0xD4BB30E2,
                   /* 0x64 */ 0x4ADFA541, 0x3DD895D7, 0xA4D1C46D, 0xD3D6F4FB,
                   /* 0x68 */ 0x4369E96A, 0x346ED9FC, 0xAD678846, 0xDA60B8D0,
                   /* 0x6C */ 0x44042D73, 0x33031DE5, 0xAA0A4C5F, 0xDD0D7CC9,
                   /* 0x70 */ 0x5005713C, 0x270241AA, 0xBE0B1010, 0xC90C2086,
                   /* 0x74 */ 0x5768B525, 0x206F85B3, 0xB966D409, 0xCE61E49F,
                   /* 0x78 */ 0x5EDEF90E, 0x29D9C998, 0xB0D09822, 0xC7D7A8B4,
                   /* 0x7C */ 0x59B33D17, 0x2EB40D81, 0xB7BD5C3B, 0xC0BA6CAD,
                   /* 0x80 */ 0xEDB88320, 0x9ABFB3B6, 0x03B6E20C, 0x74B1D29A,
                   /* 0x84 */ 0xEAD54739, 0x9DD277AF, 0x04DB2615, 0x73DC1683,
                   /* 0x88 */ 0xE3630B12, 0x94643B84, 0x0D6D6A3E, 0x7A6A5AA8,
                   /* 0x8C */ 0xE40ECF0B, 0x9309FF9D, 0x0A00AE27, 0x7D079EB1,
                   /* 0x90 */ 0xF00F9344, 0x8708A3D2, 0x1E01F268, 0x6906C2FE,
                   /* 0x94 */ 0xF762575D, 0x806567CB, 0x196C3671, 0x6E6B06E7,
                   /* 0x98 */ 0xFED41B76, 0x89D32BE0, 0x10DA7A5A, 0x67DD4ACC,
                   /* 0x9C */ 0xF9B9DF6F, 0x8EBEEFF9, 0x17B7BE43, 0x60B08ED5,
                   /* 0xA0 */ 0xD6D6A3E8, 0xA1D1937E, 0x38D8C2C4, 0x4FDFF252,
                   /* 0xA4 */ 0xD1BB67F1, 0xA6BC5767, 0x3FB506DD, 0x48B2364B,
                   /* 0xA8 */ 0xD80D2BDA, 0xAF0A1B4C, 0x36034AF6, 0x41047A60,
                   /* 0xAC */ 0xDF60EFC3, 0xA867DF55, 0x316E8EEF, 0x4669BE79,
                   /* 0xB0 */ 0xCB61B38C, 0xBC66831A, 0x256FD2A0, 0x5268E236,
                   /* 0xB4 */ 0xCC0C7795, 0xBB0B4703, 0x220216B9, 0x5505262F,
                   /* 0xB8 */ 0xC5BA3BBE, 0xB2BD0B28, 0x2BB45A92, 0x5CB36A04,
                   /* 0xBC */ 0xC2D7FFA7, 0xB5D0CF31, 0x2CD99E8B, 0x5BDEAE1D,
                   /* 0xC0 */ 0x9B64C2B0, 0xEC63F226, 0x756AA39C, 0x026D930A,
                   /* 0xC4 */ 0x9C0906A9, 0xEB0E363F, 0x72076785, 0x05005713,
                   /* 0xC8 */ 0x95BF4A82, 0xE2B87A14, 0x7BB12BAE, 0x0CB61B38,
                   /* 0xCC */ 0x92D28E9B, 0xE5D5BE0D, 0x7CDCEFB7, 0x0BDBDF21,
                   /* 0xD0 */ 0x86D3D2D4, 0xF1D4E242, 0x68DDB3F8, 0x1FDA836E,
                   /* 0xD4 */ 0x81BE16CD, 0xF6B9265B, 0x6FB077E1, 0x18B74777,
                   /* 0xD8 */ 0x88085AE6, 0xFF0F6A70, 0x66063BCA, 0x11010B5C,
                   /* 0xDC */ 0x8F659EFF, 0xF862AE69, 0x616BFFD3, 0x166CCF45,
                   /* 0xE0 */ 0xA00AE278, 0xD70DD2EE, 0x4E048354, 0x3903B3C2,
                   /* 0xE4 */ 0xA7672661, 0xD06016F7, 0x4969474D, 0x3E6E77DB,
                   /* 0xE8 */ 0xAED16A4A, 0xD9D65ADC, 0x40DF0B66, 0x37D83BF0,
                   /* 0xEC */ 0xA9BCAE53, 0xDEBB9EC5, 0x47B2CF7F, 0x30B5FFE9,
                   /* 0xF0 */ 0xBDBDF21C, 0xCABAC28A, 0x53B39330, 0x24B4A3A6,
                   /* 0xF4 */ 0xBAD03605, 0xCDD70693, 0x54DE5729, 0x23D967BF,
                   /* 0xF8 */ 0xB3667A2E, 0xC4614AB8, 0x5D681B02, 0x2A6F2B94,
                   /* 0xFC */ 0xB40BBE37, 0xC30C8EA1, 0x5A05DF1B, 0x2D02EF8D,
                                            };
unsigned int mm_crc32(unsigned char *address, unsigned int size, unsigned int crc)
{
  for (; (size > 0); size--)
    /* byte loop */
    crc = (((crc >> 8) & 0x00FFFFFF) ^
           crc32table[(crc ^ *address++) & 0x000000FF]);

  return(crc);
}

/* crc is the first field in the bulletin board. compute the crc over the
 * entire bulletin board excluding the crc field itself. Use the length field
 * as the Bulletin Board was posted by a PF with possibly a different version
 * from the vf which will sample it. Therefore, the length is computed by the
 * PF and then used blindly by the VF.
 */
vmk_uint32 qfle3_crc_vf_bulletin(struct pf_vf_bulletin_content *bulletin)
{
   return mm_crc32((vmk_uint8 *)bulletin + sizeof(bulletin->crc), 
      (vmk_uint32)(bulletin->length - sizeof(bulletin->crc)), (vmk_uint32)BULLETIN_CRC_SEED);
}

void qfle3_vf_bulletin_finalize(struct pf_vf_bulletin_content *bulletin,
                                vmk_Bool support_long)
{
   /* Older VFs contain a bug where they can't check CRC for bulletin
   * boards of length greater than legacy size.
    */
   bulletin->length = support_long ? BULLETIN_CONTENT_SIZE : BULLETIN_CONTENT_LEGACY_SIZE;
   bulletin->crc = qfle3_crc_vf_bulletin(bulletin);
}

/* propagate local bulletin board to vf */
int qfle3_post_vf_bulletin(struct qfle3_adapter *adapter, int vf)
{
   struct pf_vf_bulletin_content *bulletin = QFLE3_VF_BULLETIN(adapter, vf);
   ecore_dma_addr_t pf_addr = QFLE3_VF_BULLETIN_DMA(adapter)->mapping +
                vf * BULLETIN_CONTENT_SIZE;
   ecore_dma_addr_t vf_addr = adapter->vfdb->vfs[vf].bulletin_map;
   int rc;

   /* can only update vf after init took place */
   if (((adapter->vfdb->vfs[vf].state) != VF_ENABLED) && ((adapter->vfdb->vfs[vf].state) != VF_ACQUIRED))
      return 0;

   bulletin->version++;
   qfle3_vf_bulletin_finalize(bulletin, ((adapter->vfdb->vfs[vf].cfg_flags) & VF_CFG_EXT_BULLETIN) ? 
                                                                     VMK_TRUE : VMK_FALSE);

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Sending bulletin board: version %d valid bit map 0x%lx mac %02x:%02x:%02x:%02x:%02x:%02x "
               "vlan %d bulletin->crc %x\n", bulletin->version, bulletin->valid_bitmap,
               bulletin->mac[0],bulletin->mac[1],bulletin->mac[2],bulletin->mac[3],bulletin->mac[4],bulletin->mac[5],
               bulletin->vlan, bulletin->crc);

   /* propagate bulletin board via dmae to vm memory */ 
   rc = qfle3_copy32_vf_dmae(adapter, VMK_FALSE, pf_addr,
                                  adapter->vfdb->vfs[vf].abs_vfid, U64_HI(vf_addr),
                                  U64_LO(vf_addr), bulletin->length / 4); 
   return rc;
}
/* clears vf error in all semi blocks */
static void qfle3_vf_semi_clear_err(struct qfle3_adapter *adapter, vmk_uint8 abs_vfid)
{
   REG_WR(adapter, TSEM_REG_VFPF_ERR_NUM, abs_vfid);
   REG_WR(adapter, USEM_REG_VFPF_ERR_NUM, abs_vfid);
   REG_WR(adapter, CSEM_REG_VFPF_ERR_NUM, abs_vfid);
   REG_WR(adapter, XSEM_REG_VFPF_ERR_NUM, abs_vfid);
}

static void qfle3_vf_pglue_clear_err(struct qfle3_adapter *adapter, vmk_uint8 abs_vfid)
{
   vmk_uint32 was_err_group = (2 * QFLE3_PATH(adapter) + abs_vfid) >> 5;
   vmk_uint32 was_err_reg = 0;
   switch (was_err_group) {
      case 0:
         was_err_reg = PGLUE_B_REG_WAS_ERROR_VF_31_0_CLR;
         break;
      case 1:
         was_err_reg = PGLUE_B_REG_WAS_ERROR_VF_63_32_CLR;
         break;
      case 2:
         was_err_reg = PGLUE_B_REG_WAS_ERROR_VF_95_64_CLR;
         break;
      case 3:
         was_err_reg = PGLUE_B_REG_WAS_ERROR_VF_127_96_CLR;
         break;
   }
   REG_WR(adapter, was_err_reg, 1 << (abs_vfid & 0x1f));
}

static void qfle3_vf_enable_internal(struct qfle3_adapter *adapter, vmk_uint8 enable)
{
   REG_WR(adapter, PGLUE_B_REG_INTERNAL_VFID_ENABLE, enable ? 1 : 0);
}

void qfle3_vf_enable_access(struct qfle3_adapter *adapter, vmk_uint8 abs_vfid)
{
   /* set the VF-PF association in the FW */
   storm_memset_vf_to_pf(adapter, FW_VF_HANDLE(abs_vfid), QFLE3_FUNC(adapter));
   storm_memset_func_en(adapter, FW_VF_HANDLE(abs_vfid), 1);
   /* clear vf errors*/
   qfle3_vf_semi_clear_err(adapter, abs_vfid);
   qfle3_vf_pglue_clear_err(adapter, abs_vfid);
   /* internal vf-enable - pretend */
   qfle3_pretend_func(adapter, HW_VF_HANDLE(adapter, abs_vfid));
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Enabling internal access for vf %x\n", abs_vfid);
   qfle3_vf_enable_internal(adapter, VMK_TRUE);
   qfle3_pretend_func(adapter, QFLE3_ABS_FUNC(adapter));
}


int qfle3_vf_init(struct qfle3_adapter *adapter, struct qfle3_virtf *vf, ecore_dma_addr_t *sb_map)
{
   struct qfle3_func_init_params func_init = {0};
   int i;

   /* the sb resources are initialized at this point, do the
   * FW/HW initializations
   */
   for(i=0; i < vf->alloc_resc.num_sbs; i++) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Initializing sb %d for vf %d\n", i, vf->index);
      qfle3_init_sb(adapter, (vmk_IOA)sb_map[i], vf->abs_vfid, VMK_TRUE,
                    vf_igu_sb(vf, i), vf_igu_sb(vf, i));
   }
   /* Sanity checks */
   if (vf->state != VF_ACQUIRED) {
      QFLE3_ERR("SRIOV: BC: VF[%d] is not in VF_ACQUIRED, but %d\n",
                 vf->abs_vfid, vf->state);
      return -1;
   }

   /* let FLR complete ... */
   vmk_WorldSleep(100 * 1000);

   /* reset IGU VF statistics: MSIX */
   REG_WR(adapter, IGU_REG_STATISTIC_NUM_MESSAGE_SENT + vf->abs_vfid * 4 , 0);
   /* function setup */
   func_init.pf_id = QFLE3_FUNC(adapter);
   func_init.func_id = FW_VF_HANDLE(vf->abs_vfid);
   qfle3_func_init(adapter, &func_init);
   /* Enable the vf */
   qfle3_vf_enable_access(adapter, vf->abs_vfid);
   qfle3_vf_enable_traffic(adapter, vf);

   /* queue protection table */
   for(i=0; i < vf->alloc_resc.num_rxqs; i++)
      qfle3_vf_qtbl_set_q(adapter, vf->abs_vfid,
                                    vfq_qzone_id(vf, vfq_get(vf, i)), VMK_TRUE);

   vf->state = VF_ENABLED;
   /* update vf bulletin board */
   qfle3_post_vf_bulletin(adapter, vf->index);
   return 0;
}

static int qfle3_vf_op_prep(struct qfle3_adapter *adapter, int vfidx,
                            struct qfle3_virtf **vf,
                            struct pf_vf_bulletin_content **bulletin,
                            vmk_Bool test_queue)
{
  /* if (adapter->state != QFLE3_STATE_OPEN) {          QFLE3_SRIOV_TODO
      QFLE3_ERR("SRIOV: BC: PF is down - can't utilize iov-related functionality\n");
      return -1;
   }*/

   if (!adapter->vfdb) {
      QFLE3_ERR("SRIOV: BC: sriov is disabled - can't utilize iov-related functionality\n");
      return -1;
   }

   if (vfidx >= adapter->vfdb->sriov.nr_virtfn) {
      QFLE3_ERR("SRIOV: BC: VF is uninitialized - can't utilize iov-related functionality. vfidx was %d QFLE3_NR_VIRTFN was %d\n",
                          vfidx, adapter->vfdb->sriov.nr_virtfn);
      return -1;
   }
   /* init members */
   *vf = ( (adapter->vfdb)  && (adapter->vfdb->vfs) )? &(adapter)->vfdb->vfs[vfidx] : NULL;
   *bulletin = QFLE3_VF_BULLETIN(adapter, vfidx);

   if (!*vf) {
      QFLE3_ERR("SRIOV: BC: Unable to get VF structure for vfidx %d\n", vfidx);
      return -1;
        }

   if (test_queue && !(*vf)->vfqs) {
      QFLE3_ERR("SRIOV: BC: vfqs struct is null. Was this invoked before dynamically enabling SR-IOV? vfidx was %d\n",
                          vfidx);
      return -1;
   }

   if (!*bulletin) {
      QFLE3_ERR("SRIOV: BC: Bulletin Board struct is null for vfidx %d\n",
                          vfidx);
      return -1;
   }
   return 0;
}

static int qfle3_max_speed_cap(struct qfle3_adapter *adapter)
{
   vmk_uint32 supported = adapter->port.supported[qfle3_get_link_cfg_idx(adapter)];

   if (supported & (ELINK_SUPPORTED_20000baseMLD2_Full | ELINK_SUPPORTED_20000baseKR2_Full))
      return 20000;

   return 10000; /* assume lowest supported speed is 10G */
}

int qfle3_iov_link_update_vf(struct qfle3_adapter *adapter, int idx)
{
   struct qfle3_link_report_data *state = &adapter->last_reported_link;
   struct pf_vf_bulletin_content *bulletin;
   struct qfle3_virtf *vf;
   vmk_Bool update = VMK_TRUE;
   int rc = 0;

   /* sanity and init */
   rc = qfle3_vf_op_prep(adapter, idx, &vf, &bulletin, VMK_FALSE);
   if (rc)
      return rc;

   vmk_SemaLock(&adapter->vfdb->bulletin_mutex);

   if (vf->link_cfg == IFLA_VF_LINK_STATE_AUTO) {
      bulletin->valid_bitmap |= 1 << LINK_VALID;
      bulletin->link_speed = state->line_speed;
      bulletin->link_flags = 0;
      if (ECORE_TEST_BIT(QFLE3_LINK_REPORT_LINK_DOWN, &state->link_report_flags))
         bulletin->link_flags |= VFPF_LINK_REPORT_LINK_DOWN;
      if (ECORE_TEST_BIT(QFLE3_LINK_REPORT_FD,&state->link_report_flags))
         bulletin->link_flags |= VFPF_LINK_REPORT_FULL_DUPLEX;
      if (ECORE_TEST_BIT(QFLE3_LINK_REPORT_RX_FC_ON,&state->link_report_flags))
         bulletin->link_flags |= VFPF_LINK_REPORT_RX_FC_ON;
      if (ECORE_TEST_BIT(QFLE3_LINK_REPORT_TX_FC_ON,&state->link_report_flags))
         bulletin->link_flags |= VFPF_LINK_REPORT_TX_FC_ON;
   } else if (vf->link_cfg == IFLA_VF_LINK_STATE_DISABLE &&
               !(bulletin->link_flags & VFPF_LINK_REPORT_LINK_DOWN)) {
      bulletin->valid_bitmap |= 1 << LINK_VALID;
      bulletin->link_flags |= VFPF_LINK_REPORT_LINK_DOWN;
   } else if (vf->link_cfg == IFLA_VF_LINK_STATE_ENABLE &&
               (bulletin->link_flags & VFPF_LINK_REPORT_LINK_DOWN)) {
      bulletin->valid_bitmap |= 1 << LINK_VALID;
      bulletin->link_speed = qfle3_max_speed_cap(adapter);
      bulletin->link_flags &= ~VFPF_LINK_REPORT_LINK_DOWN;
   } else {
      update = VMK_FALSE;
   }
   if (update) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: vf %d mode %u speed %d flags %x\n", idx,
                   vf->link_cfg, bulletin->link_speed, bulletin->link_flags);

      /* Post update on VF's bulletin board */
      rc = qfle3_post_vf_bulletin(adapter, idx);
      if (rc) {
         QFLE3_ERR("SRIOV: BC: Failed to update VF[%d] bulletin\n", idx);
         goto out;
      }
   }

out:
   vmk_SemaUnlock(&adapter->vfdb->bulletin_mutex);
   return rc;
}

static void qfle3_vf_mbx_resp_single_tlv(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   struct qfle3_vf_mbx *mbx = &adapter->vfdb->mbxs[vf->index];
   vmk_uint16 length, type;

   /* prepare response */
   type = mbx->first_tlv.tl.type;
   length = type == CHANNEL_TLV_ACQUIRE ?
                sizeof(struct pfvf_acquire_resp_tlv) :
                sizeof(struct pfvf_general_resp_tlv);
   qfle3_add_tlv(adapter, &mbx->msg->resp, 0, type, length);
   qfle3_add_tlv(adapter, &mbx->msg->resp, length, CHANNEL_TLV_LIST_END, sizeof(struct channel_list_end_tlv));
}

static void qfle3_vf_mbx_resp(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                              int rc)
{
   qfle3_vf_mbx_resp_single_tlv(adapter, vf);
   qfle3_vf_mbx_resp_send_msg(adapter, vf, rc);
}

static void qfle3_vf_mbx_init_vf(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                              struct qfle3_vf_mbx *mbx)
{
   struct vfpf_init_tlv *init = &mbx->msg->req.init;
   int rc;

   /* record ghost addresses from vf message */
   vf->fw_stat_map = init->stats_addr;
   vf->stats_stride = init->stats_stride;
   rc = qfle3_vf_init(adapter, vf, (ecore_dma_addr_t *)init->sb_addr);

   /* set VF multiqueue statistics collection mode */
   if (init->flags & VFPF_INIT_FLG_STATS_COALESCE)
      vf->cfg_flags |= VF_CFG_STATS_COALESCE;

   /* Update VF's view of link state */
   if (vf->cfg_flags & VF_CFG_EXT_BULLETIN)
      qfle3_iov_link_update_vf(adapter, vf->index);

   /* response */
   qfle3_vf_mbx_resp(adapter, vf, rc);
}

static void qfle3_leading_vfq_init(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                                   struct qfle3_vf_queue *q)
{
   vmk_uint8 cl_id = vfq_cl_id(vf, q);
   vmk_uint8 func_id = FW_VF_HANDLE(vf->abs_vfid);

   /* mac */
   ecore_init_mac_obj(adapter, &q->mac_obj,
                           cl_id, q->cid, func_id,
                           qfle3_vf_sp(adapter, vf, mac_rdata),
                           qfle3_vf_sp_map(adapter, vf, mac_rdata),
                           ECORE_FILTER_MAC_PENDING,
                           &vf->filter_state,
                           ECORE_OBJ_TYPE_RX_TX,
                           &vf->vf_macs_pool);
   /* vlan */
   ecore_init_vlan_obj(adapter, &q->vlan_obj,
                            cl_id, q->cid, func_id,
                            qfle3_vf_sp(adapter, vf, vlan_rdata),
                            qfle3_vf_sp_map(adapter, vf, vlan_rdata),
                            ECORE_FILTER_VLAN_PENDING,
                            &vf->filter_state,
                            ECORE_OBJ_TYPE_RX_TX,
                            &vf->vf_vlans_pool);
   /* vlan-mac */
   ecore_init_vlan_mac_obj(adapter, &q->vlan_mac_obj,
                                cl_id, q->cid, func_id,
                                qfle3_vf_sp(adapter, vf, vlan_mac_rdata),
                                qfle3_vf_sp_map(adapter, vf, vlan_mac_rdata),
                                ECORE_FILTER_VLAN_MAC_PENDING,
                                &vf->filter_state,
                                ECORE_OBJ_TYPE_RX_TX,
                                &vf->vf_macs_pool,
                                &vf->vf_vlans_pool);
   /* mcast */
   ecore_init_mcast_obj(adapter, &vf->mcast_obj, cl_id,
                             q->cid, func_id, func_id,
                             qfle3_vf_sp(adapter, vf, mcast_rdata),
                             qfle3_vf_sp_map(adapter, vf, mcast_rdata),
                             ECORE_FILTER_MCAST_PENDING,
                             &vf->filter_state,
                             ECORE_OBJ_TYPE_RX_TX);
   /* rss */
   ecore_init_rss_config_obj(adapter, &vf->rss_conf_obj, cl_id, q->cid,
                                  func_id, func_id,
                                  qfle3_vf_sp(adapter, vf, rss_rdata),
                                  qfle3_vf_sp_map(adapter, vf, rss_rdata),
                                  ECORE_FILTER_RSS_CONF_PENDING,
                                  &vf->filter_state,
                                  ECORE_OBJ_TYPE_RX_TX);
  
   vf->leading_rss = cl_id;
   q->is_leading = VMK_TRUE;
   q->sp_initialized = VMK_TRUE;
}

/* convert MBX queue-flags to standard SP queue-flags */
static void qfle3_vf_mbx_set_q_flags(struct qfle3_adapter *adapter, vmk_uint32 mbx_q_flags,
                                     unsigned long *sp_q_flags)
{
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: mbx_q_flags 0x%x\n", mbx_q_flags);
   if (mbx_q_flags & VFPF_QUEUE_FLG_TPA)
      SET_BIT(ECORE_Q_FLG_TPA, sp_q_flags);
   if (mbx_q_flags & VFPF_QUEUE_FLG_TPA_IPV6)
       SET_BIT(ECORE_Q_FLG_TPA_IPV6, sp_q_flags);
   if (mbx_q_flags & VFPF_QUEUE_FLG_TPA_GRO)
       SET_BIT(ECORE_Q_FLG_TPA_GRO, sp_q_flags);
   if (mbx_q_flags & VFPF_QUEUE_FLG_STATS)
       SET_BIT(ECORE_Q_FLG_STATS, sp_q_flags);
   if (mbx_q_flags & VFPF_QUEUE_FLG_VLAN)
       SET_BIT(ECORE_Q_FLG_VLAN, sp_q_flags);
   if (mbx_q_flags & VFPF_QUEUE_FLG_COS)
       SET_BIT(ECORE_Q_FLG_COS, sp_q_flags);
   if (mbx_q_flags & VFPF_QUEUE_FLG_HC)
       SET_BIT(ECORE_Q_FLG_HC, sp_q_flags);
   if (mbx_q_flags & VFPF_QUEUE_FLG_DHC)
       SET_BIT(ECORE_Q_FLG_DHC, sp_q_flags);
   if (mbx_q_flags & VFPF_QUEUE_FLG_LEADING_RSS)
       SET_BIT(ECORE_Q_FLG_LEADING_RSS, sp_q_flags);

   /* outer vlan removal is set according to PF's multi function mode */
    if (IS_MF_SD(adapter))
       SET_BIT(ECORE_Q_FLG_OV, sp_q_flags);

}

/* VFOP operations states */
void qfle3_vfop_qctor_dump_tx(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                              struct ecore_queue_init_params *init_params,
                              struct ecore_queue_setup_params *setup_params,
                              vmk_uint16 q_idx, vmk_uint16 sb_idx)
{
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF[%d] Q_SETUP: txq[%d]-- vfsb=%d, sb-index=%d, hc-rate=%d, flags=0x%lx, traffic-type=%d",
           vf->abs_vfid, q_idx, sb_idx, init_params->tx.sb_cq_index,
           init_params->tx.hc_rate, setup_params->flags, setup_params->txq_params.traffic_type);
}

void qfle3_vfop_qctor_dump_rx(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                            struct ecore_queue_init_params *init_params,
                            struct ecore_queue_setup_params *setup_params,
                            vmk_uint16 q_idx, vmk_uint16 sb_idx)
{
   struct ecore_rxq_setup_params *rxq_params = &setup_params->rxq_params;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF[%d] Q_SETUP: rxq[%d]-- vfsb=%d, sb-index=%d, hc-rate=%d, mtu=%d, buf-size=%d\n",
           vf->abs_vfid, q_idx, sb_idx, init_params->rx.sb_cq_index, init_params->rx.hc_rate, setup_params->gen_params.mtu, rxq_params->buf_sz);
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF[%d] sge-size=%d, max_sge_pkt=%d, tpa-agg-size=%d, flags=0x%lx, drop-flags=0x%x, cache-log=%d\n",
           vf->abs_vfid, rxq_params->sge_buf_sz, rxq_params->max_sges_pkt, rxq_params->tpa_agg_sz, setup_params->flags, rxq_params->drop_flags, rxq_params->cache_line_log);
}

void qfle3_vfop_qctor_prep(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                           struct qfle3_vf_queue *q, struct qfle3_vf_queue_construct_params *p,
                           unsigned long q_type)
{
   struct ecore_queue_init_params *init_p = &p->qstate.params.init;
   struct ecore_queue_setup_params *setup_p = &p->prep_qsetup;
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: vf[%d]\n", vf->abs_vfid);
   /* INIT */
   /* Enable host coalescing in the transition to INIT state */
   if (ECORE_TEST_BIT(ECORE_Q_FLG_HC, &init_p->rx.flags))
      SET_BIT(ECORE_Q_FLG_HC_EN, &init_p->rx.flags);

   if (ECORE_TEST_BIT(ECORE_Q_FLG_HC, &init_p->tx.flags))
      SET_BIT(ECORE_Q_FLG_HC_EN, &init_p->tx.flags);

   /* FW SB ID */
   init_p->rx.fw_sb_id = vf_igu_sb(vf, q->sb_idx);
   init_p->tx.fw_sb_id = vf_igu_sb(vf, q->sb_idx);

   /* context */
   init_p->cxts[0] = q->cxt;

   /* SETUP */

   /* Setup-op general parameters */
   vf->sp_cl_id = qfle3_fp(adapter, 0, cl_id);
   setup_p->gen_params.spcl_id = vf->sp_cl_id;
   setup_p->gen_params.stat_id = vfq_stat_id(vf, q);
   setup_p->gen_params.fp_hsi = vf->fp_hsi;

   /* Setup-op flags:
   * collect statistics, zero statistics, local-switching, security,
   * OV for Flex10, RSS and MCAST for leading
   */
   if (ECORE_TEST_BIT(ECORE_Q_FLG_STATS, &setup_p->flags))
      SET_BIT(ECORE_Q_FLG_ZERO_STATS, &setup_p->flags);

   /* for VFs, enable tx switching, bd coherency, and mac address
   * anti-spoofing
   */
   SET_BIT(ECORE_Q_FLG_TX_SWITCH, &setup_p->flags);
   SET_BIT(ECORE_Q_FLG_TX_SEC, &setup_p->flags);
   SET_BIT(ECORE_Q_FLG_ANTI_SPOOF, &setup_p->flags);
   /* Setup-op rx parameters */
   if (ECORE_TEST_BIT(ECORE_Q_TYPE_HAS_RX, &q_type)) {
      struct ecore_rxq_setup_params *rxq_p = &setup_p->rxq_params;

      rxq_p->cl_qzone_id = vfq_qzone_id(vf, q);
      rxq_p->fw_sb_id = vf_igu_sb(vf, q->sb_idx);
      rxq_p->rss_engine_id = FW_VF_HANDLE(vf->abs_vfid);

      if (ECORE_TEST_BIT(ECORE_Q_FLG_TPA, &setup_p->flags))
         rxq_p->max_tpa_queues = QFLE3_VF_MAX_TPA_AGG_QUEUES;
   }

   /* Setup-op tx parameters */
   if (ECORE_TEST_BIT(ECORE_Q_TYPE_HAS_TX, &q_type)) {
      setup_p->txq_params.tss_leading_cl_id = vf->leading_rss;
      setup_p->txq_params.fw_sb_id = vf_igu_sb(vf, q->sb_idx);
   }
}

static int qfle3_vf_queue_create(struct qfle3_adapter *adapter,
                                 struct qfle3_virtf *vf, int qid,
                                 struct qfle3_vf_queue_construct_params *qctor)
{
   struct ecore_queue_state_params *q_params;
   int rc = 0;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: vf[%d:%d]\n", vf->abs_vfid, qid);

   /* Prepare ramrod information */
   q_params = &qctor->qstate;
   q_params->q_obj = &qfle3_vfq(vf, qid, sp_obj);
   ECORE_SET_BIT(RAMROD_COMP_WAIT, &q_params->ramrod_flags);   

   if (ecore_get_q_logical_state(adapter, q_params->q_obj) ==
       ECORE_Q_LOGICAL_STATE_ACTIVE) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: queue was already up. Aborting gracefully\n");
      goto out;
   }

   /* Run Queue 'construction' ramrods */
   q_params->cmd = ECORE_Q_CMD_INIT;
   rc = ecore_queue_state_change(adapter, q_params);
   if (rc)
      goto out;

   vmk_Memcpy(&q_params->params.setup, &qctor->prep_qsetup,
              sizeof(struct ecore_queue_setup_params));
   q_params->cmd = ECORE_Q_CMD_SETUP;
   rc = ecore_queue_state_change(adapter, q_params);
   if (rc)
      goto out;

   /* enable interrupts */
   qfle3_vf_igu_ack_sb(adapter, vf, vf_igu_sb(vf, vf->vfqs[qid].sb_idx),
                            USTORM_ID, 0, IGU_INT_ENABLE, 0);
out:
   return rc;
}



static int qfle3_set_vf_vlan_filter(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                vmk_uint16 vlan, vmk_Bool add)
{
   struct ecore_vlan_mac_ramrod_params ramrod_param;
   unsigned long ramrod_flags = 0;
   int rc = 0;

   /* configure the new vlan to device */
   vmk_Memset(&ramrod_param, 0, sizeof(ramrod_param));
   ECORE_SET_BIT(RAMROD_COMP_WAIT, &ramrod_flags);
   ramrod_param.vlan_mac_obj = &qfle3_leading_vfq(vf, vlan_obj);
   ramrod_param.ramrod_flags = ramrod_flags;
   ramrod_param.user_req.u.vlan.vlan = vlan;
   ramrod_param.user_req.cmd = add ? ECORE_VLAN_MAC_ADD : ECORE_VLAN_MAC_DEL;
   

   if (!qfle3_esx_vf_gvlans_allowed(adapter, vf->index))   
      rc = ecore_config_vlan_mac(adapter, &ramrod_param);

   if (rc) {
      QFLE3_ERR("SRIOV: PT: Failed to configure vlan\n"); 
      return -1;
   }
   return 0;
}


int qfle3_pt_set_vf_vlan(struct qfle3_adapter *adapter, int vfidx, vmk_uint16 vlan, vmk_uint8 qos)
{
   struct pf_vf_bulletin_content *bulletin = NULL;
   struct ecore_vlan_mac_obj *vlan_obj;
   unsigned long vlan_mac_flags = 0;
   unsigned long ramrod_flags = 0;
   struct qfle3_virtf *vf = NULL;
   int i, rc;

   if (vlan > 4095) {
      QFLE3_ERR("SRIOV: PT: illegal vlan value %d\n", vlan);
      return -1;
   }

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: Configuring VF %d with VLAN %d qos %d\n",
      vfidx, vlan, 0);

   /* sanity and init */
   rc = qfle3_vf_op_prep(adapter, vfidx, &vf, &bulletin, VMK_TRUE);
   if (rc)
      return rc;

   /* update PF's copy of the VF's bulletin. No point in posting the vlan
    * to the VF since it doesn't have anything to do with it. But it useful
    * to store it here in case the VF is not up yet and we can only
    * configure the vlan later when it does. Treat vlan id 0 as remove the
    * Host tag.
    */
   vmk_SemaLock(&adapter->vfdb->bulletin_mutex);

   if (vlan > 0)
      bulletin->valid_bitmap |= 1 << VLAN_VALID;
   else
      bulletin->valid_bitmap &= ~(1 << VLAN_VALID);
   bulletin->vlan = vlan;

   /* Post update on VF's bulletin board */
   rc = qfle3_post_vf_bulletin(adapter, vfidx);
   if (rc)
      QFLE3_ERR("SRIOV: PT: Failed to update VF[%d] bulletin\n", vfidx);
   vmk_SemaUnlock(&adapter->vfdb->bulletin_mutex);

   /* is vf initialized and queue set up? */

   if (vf->state != VF_ENABLED ||
       ecore_get_q_logical_state(adapter, &qfle3_leading_vfq(vf, sp_obj)) !=
       ECORE_Q_LOGICAL_STATE_ACTIVE)
      return rc;

   /* User should be able to see error in system logs */
   if (!qfle3_validate_vf_sp_objs(adapter, vf, VMK_TRUE))
      return -1;

   /* must lock vfpf channel to protect against vf flows */
   qfle3_lock_vf_pf_channel(adapter, vf, CHANNEL_TLV_PF_SET_VLAN);

   /* remove existing vlans */
   ECORE_SET_BIT(RAMROD_COMP_WAIT, &ramrod_flags);
   vlan_obj = &qfle3_leading_vfq(vf, vlan_obj);
   
   if (!qfle3_esx_vf_gvlans_allowed(adapter, vf->index))                
      rc = vlan_obj->delete_all(adapter, vlan_obj, &vlan_mac_flags,
            &ramrod_flags);
   if (rc) {
      QFLE3_ERR("SRIOV: PT: Failed to delete vlans\n");
      rc = -1;
      goto out;
   }

   rc = qfle3_set_vf_vlan_filter(adapter, vf, vlan, VMK_TRUE);
   if (rc)
      goto out;

   /* send queue update ramrods to configure default vlan and
    * silent vlan removal
    */
   for(i=0; i < vf->alloc_resc.num_rxqs; i++) {
      struct ecore_queue_state_params q_params = {NULL};
      struct ecore_queue_update_params *update_params;

      q_params.q_obj = &qfle3_vfq(vf, i, sp_obj);

      /* validate the Q is UP */
      if (ecore_get_q_logical_state(adapter, q_params.q_obj) !=
          ECORE_Q_LOGICAL_STATE_ACTIVE)
         continue;

      ECORE_SET_BIT(RAMROD_COMP_WAIT, &q_params.ramrod_flags);
      q_params.cmd = ECORE_Q_CMD_UPDATE;
      update_params = &q_params.params.update;
      ECORE_SET_BIT(ECORE_Q_UPDATE_DEF_VLAN_EN_CHNG, &update_params->update_flags);
      ECORE_SET_BIT(ECORE_Q_UPDATE_SILENT_VLAN_REM_CHNG, &update_params->update_flags);
      if (vlan == 0) {
         /* if vlan is 0 then we want to leave the VF traffic
          * untagged, and leave the incoming traffic untouched
          * (i.e. do not remove any vlan tags).
          */
         ECORE_CLEAR_BIT(ECORE_Q_UPDATE_DEF_VLAN_EN, &update_params->update_flags);
         ECORE_CLEAR_BIT(ECORE_Q_UPDATE_SILENT_VLAN_REM, &update_params->update_flags);
      } else {
         /* configure default vlan to vf queue and set silent
          * vlan removal (the vf remains unaware of this vlan).
          */
         ECORE_SET_BIT(ECORE_Q_UPDATE_DEF_VLAN_EN, &update_params->update_flags);
         ECORE_SET_BIT(ECORE_Q_UPDATE_SILENT_VLAN_REM, &update_params->update_flags);
         update_params->def_vlan = vlan;
         update_params->silent_removal_value = vlan & VLAN_VID_MASK;
         update_params->silent_removal_mask = VLAN_VID_MASK;
      }

      /* Update the Queue state */
      rc = ecore_queue_state_change(adapter, &q_params);
      if (rc) {
         QFLE3_ERR("SRIOV: PT: Failed to configure default VLAN queue %d\n", i);
         goto out;
      }
   }
out:
   qfle3_unlock_vf_pf_channel(adapter, vf, CHANNEL_TLV_PF_SET_VLAN);

   if (rc)
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: updated VF[%d] vlan configuration (vlan = %d)\n", vfidx, vlan);

   return rc;
}

int qfle3_set_vf_vlan(struct qfle3_adapter *adapter, int vfidx, vmk_uint16 vlan, vmk_uint8 qos)
{
   struct pf_vf_bulletin_content *bulletin = NULL;
   struct ecore_vlan_mac_obj *vlan_obj;
   unsigned long vlan_mac_flags = 0;
   unsigned long ramrod_flags = 0;
   struct qfle3_virtf *vf = NULL;
   int i, rc;

   if (vlan > 4095) {
      QFLE3_ERR("SRIOV: PF: illegal vlan value %d\n", vlan);
      return -1;
   }

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PF: Configuring VF %d with VLAN %d qos %d\n",
      vfidx, vlan, 0);

   /* sanity and init */
   rc = qfle3_vf_op_prep(adapter, vfidx, &vf, &bulletin, VMK_TRUE);
   if (rc)
      return rc;

   /* update PF's copy of the VF's bulletin. No point in posting the vlan
    * to the VF since it doesn't have anything to do with it. But it useful
    * to store it here in case the VF is not up yet and we can only
    * configure the vlan later when it does. Treat vlan id 0 as remove the
    * Host tag.
    */
   vmk_SemaLock(&adapter->vfdb->bulletin_mutex);

   if (vlan > 0)
      bulletin->valid_bitmap |= 1 << VLAN_VALID;
   else
      bulletin->valid_bitmap &= ~(1 << VLAN_VALID);
   bulletin->vlan = vlan;

   /* Post update on VF's bulletin board */
   rc = qfle3_post_vf_bulletin(adapter, vfidx);
   if (rc)
      QFLE3_ERR("SRIOV: PF: Failed to update VF[%d] bulletin\n", vfidx);
   vmk_SemaUnlock(&adapter->vfdb->bulletin_mutex);

   /* is vf initialized and queue set up? */

   if (vf->state != VF_ENABLED ||
       ecore_get_q_logical_state(adapter, &qfle3_leading_vfq(vf, sp_obj)) !=
       ECORE_Q_LOGICAL_STATE_ACTIVE)
      return rc;

   /* User should be able to see error in system logs */
   if (!qfle3_validate_vf_sp_objs(adapter, vf, VMK_TRUE))
      return -1;

   /* remove existing vlans */
   ECORE_SET_BIT(RAMROD_COMP_WAIT, &ramrod_flags);
   vlan_obj = &qfle3_leading_vfq(vf, vlan_obj);

   if (!qfle3_esx_vf_gvlans_allowed(adapter, vf->index))
      rc = vlan_obj->delete_all(adapter, vlan_obj, &vlan_mac_flags,
            &ramrod_flags);
   if (rc) {
      QFLE3_ERR("SRIOV: PF: Failed to delete vlans\n");
      rc = -1;
      goto out;
   }

   rc = qfle3_set_vf_vlan_filter(adapter, vf, vlan, VMK_TRUE);
   if (rc)
      goto out;
   /* send queue update ramrods to configure default vlan and
    * silent vlan removal
    */
   for(i=0; i < vf->alloc_resc.num_rxqs; i++) {
      struct ecore_queue_state_params q_params = {NULL};
      struct ecore_queue_update_params *update_params;

      q_params.q_obj = &qfle3_vfq(vf, i, sp_obj);

      /* validate the Q is UP */
      if (ecore_get_q_logical_state(adapter, q_params.q_obj) !=
          ECORE_Q_LOGICAL_STATE_ACTIVE)
         continue;

      ECORE_SET_BIT(RAMROD_COMP_WAIT, &q_params.ramrod_flags);
      q_params.cmd = ECORE_Q_CMD_UPDATE;
      update_params = &q_params.params.update;
      ECORE_SET_BIT(ECORE_Q_UPDATE_DEF_VLAN_EN_CHNG, &update_params->update_flags);
      ECORE_SET_BIT(ECORE_Q_UPDATE_SILENT_VLAN_REM_CHNG, &update_params->update_flags);
      if (vlan == 0) {
         /* if vlan is 0 then we want to leave the VF traffic
          * untagged, and leave the incoming traffic untouched
          * (i.e. do not remove any vlan tags).
          */
         ECORE_CLEAR_BIT(ECORE_Q_UPDATE_DEF_VLAN_EN, &update_params->update_flags);
         ECORE_CLEAR_BIT(ECORE_Q_UPDATE_SILENT_VLAN_REM, &update_params->update_flags);
      } else {
         /* configure default vlan to vf queue and set silent
          * vlan removal (the vf remains unaware of this vlan).
          */
         ECORE_SET_BIT(ECORE_Q_UPDATE_DEF_VLAN_EN, &update_params->update_flags);
         ECORE_SET_BIT(ECORE_Q_UPDATE_SILENT_VLAN_REM, &update_params->update_flags);
         update_params->def_vlan = vlan;
         update_params->silent_removal_value = vlan & VLAN_VID_MASK;
         update_params->silent_removal_mask = VLAN_VID_MASK;
      }

      /* Update the Queue state */
      rc = ecore_queue_state_change(adapter, &q_params);
      if (rc) {
         QFLE3_ERR("SRIOV: PF: Failed to configure default VLAN queue %d\n", i);
         goto out;
      }
   }
out:

   if (rc)
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: updated VF[%d] vlan configuration (vlan = %d)\n", vfidx, vlan);

   return rc;
}

void qfle3_pf_set_vfs_vlan(struct qfle3_adapter *adapter)
{
   int vfidx;
   struct pf_vf_bulletin_content *bulletin;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Configuring vlan for VFs from sp-task\n");
   for(vfidx=0; vfidx < adapter->vfdb->sriov.nr_virtfn; vfidx++) {
      bulletin = QFLE3_VF_BULLETIN(adapter, vfidx);
      if (bulletin->valid_bitmap & (1 << VLAN_VALID))
         qfle3_set_vf_vlan(adapter, vfidx, bulletin->vlan, 0);
   }
}

int qfle3_vf_queue_setup(struct qfle3_adapter *adapter, struct qfle3_virtf *vf, int qid,
                         struct qfle3_vf_queue_construct_params *qctor)
{
   int rc;
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: vf[%d:%d]\n", vf->abs_vfid, qid);

   rc = qfle3_vf_queue_create(adapter, vf, qid, qctor);
   if (rc)
      goto op_err;

   qfle3_pf_set_vfs_vlan(adapter);      
   
   return 0;
op_err:
   QFLE3_ERR("SRIOV: BC: QSETUP[%d:%d] error: rc %d\n", vf->abs_vfid, qid, rc);
   return rc;
}

void
qfle3_esx_passthru_config_setup_filters_finalize(struct qfle3_adapter *adapter, vmk_uint16 vf_idx, int rc, int state)
{
   adapter->esx_vf[(vf_idx)].passthru_rc = rc;
   adapter->esx_vf[(vf_idx)].passthru_state = (state);
}

static void qfle3_vf_mbx_setup_q(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                                 struct qfle3_vf_mbx *mbx)
{
   struct vfpf_setup_q_tlv *setup_q = &mbx->msg->req.setup_q;
   struct qfle3_vf_queue_construct_params qctor;
   int rc = 0;
   vmk_uint32 mtu;
   struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vf->index];

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: vf[%d]\n", vf->abs_vfid);

   QFLE3_DBG(QFLE3_DBG_IOV, "setup_q->txq.txq_addr: 0x%lx setup_q->txq.vf_sb: %d setup_q->txq.sb_index: %d setup_q->txq.hc_rate: %d setup_q->txq.flags: 0x%x setup_q->txq.stat_id: %d\n ",
                   setup_q->txq.txq_addr, setup_q->txq.vf_sb, setup_q->txq.sb_index, setup_q->txq.hc_rate, setup_q->txq.flags, setup_q->txq.stat_id);
   QFLE3_DBG(QFLE3_DBG_IOV, "setup_q->txq.traffic_type: %d setup_q->txq.padding: %d\n",setup_q->txq.traffic_type, setup_q->txq.padding);
   QFLE3_DBG(QFLE3_DBG_IOV, "setup_q->rxq.rcq_addr: 0x%lx setup_q->rxq.rcq_np_addr: 0x%lx setup_q->rxq.rxq_addr: 0x%lx setup_q->rxq.sge_addr: 0x%lx \n",
                    setup_q->rxq.rcq_addr, setup_q->rxq.rcq_np_addr, setup_q->rxq.rxq_addr, setup_q->rxq.sge_addr);
   QFLE3_DBG(QFLE3_DBG_IOV, "setup_q->rxq.vf_sb: %d setup_q->rxq.sb_index: %d setup_q->rxq.hc_rate: %d setup_q->rxq.mtu: %d setup_q->rxq.buf_sz: %d setup_q->rxq.flags: 0x%x"                             "setup_q->rxq.stat_id: %d\n", setup_q->rxq.vf_sb, setup_q->rxq.sb_index, setup_q->rxq.hc_rate, setup_q->rxq.mtu, setup_q->rxq.buf_sz, setup_q->rxq.flags, setup_q->rxq.stat_id);
   QFLE3_DBG(QFLE3_DBG_IOV, "setup_q->rxq.sge_buf_sz: %d setup_q->rxq.tpa_agg_sz: %d setup_q->rxq.max_sge_pkt: %d setup_q->rxq.drop_flags: %d setup_q->rxq.cache_line_log:"
                  " %d setup_q->rxq.padding: %d",
                  setup_q->rxq.sge_buf_sz,  setup_q->rxq.tpa_agg_sz,  setup_q->rxq.max_sge_pkt,  setup_q->rxq.drop_flags,  setup_q->rxq.cache_line_log,  setup_q->rxq.padding);

   /* verify vf_qid */
   if (setup_q->vf_qid >= vf_rxq_count(vf)) {
      QFLE3_ERR("SRIOV: BC: vf_qid %d invalid, max queue count is %d\n",
                setup_q->vf_qid, vf_rxq_count(vf));
      rc = -1;
      goto response;
   }

   /* tx queues must be setup alongside rx queues thus if the rx queue
   * is not marked as valid there's nothing to do.
   */
   if (setup_q->param_valid & (VFPF_RXQ_VALID|VFPF_TXQ_VALID)) {
      struct qfle3_vf_queue *q = vfq_get(vf, setup_q->vf_qid);
      unsigned long q_type = 0;
      struct ecore_queue_init_params *init_p;
      struct ecore_queue_setup_params *setup_p;

      if (qfle3_vfq_is_leading(q)){
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: vf[%d] Leading vfq\n", vf->abs_vfid);
         qfle3_leading_vfq_init(adapter, vf, q);
      }

     /*Get mtu check from  vmkernel*/
      if (adapter_esx_vf->old_mtu == QFLE3_ESX_PASSTHRU_MTU_UNINITIALIZED) {
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF: %d: Initial MTU value set to %d\n", vf->index, setup_q->rxq.mtu);
      } else if (adapter_esx_vf->old_mtu != setup_q->rxq.mtu){
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF: %d: Requesting mtu Change to %d\n", vf->index, setup_q->rxq.mtu);
         mtu = setup_q->rxq.mtu;
         adapter_esx_vf->flags &= ~QFLE3_ESX_PASSTHRU_ALLOW_MTU;
         qfle3_passthru_config(adapter, vf->index, VMK_CFG_MTU_CHANGED, &mtu);
      }

      /* re-init the VF operation context */
      vmk_Memset(&qctor, 0 , sizeof(struct qfle3_vf_queue_construct_params));
      setup_p = &qctor.prep_qsetup;
      init_p =  &qctor.qstate.params.init;

      /* activate immediately */
      ECORE_SET_BIT(ECORE_Q_FLG_ACTIVE, &setup_p->flags);

      if (setup_q->param_valid & VFPF_TXQ_VALID) {
         struct ecore_txq_setup_params *txq_params = &setup_p->txq_params;
         
         ECORE_SET_BIT(ECORE_Q_TYPE_HAS_TX, &q_type);

         /* save sb resource index */
         q->sb_idx = setup_q->txq.vf_sb;

         /* tx init */
         init_p->tx.hc_rate = setup_q->txq.hc_rate;
         init_p->tx.sb_cq_index = setup_q->txq.sb_index;

         qfle3_vf_mbx_set_q_flags(adapter, setup_q->txq.flags, &init_p->tx.flags);

         /* tx setup - flags */
         qfle3_vf_mbx_set_q_flags(adapter, setup_q->txq.flags, &setup_p->flags);
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: general setup_p->flags 0x%lx init_p->tx.flags is 0x%lx init_p->rx.flags is 0x%lx\n", setup_p->flags, init_p->tx.flags, init_p->rx.flags);

         /* tx setup - general, nothing */

         /* tx setup - tx */
         txq_params->dscr_map = setup_q->txq.txq_addr;
         txq_params->sb_cq_index = setup_q->txq.sb_index;
         txq_params->traffic_type = setup_q->txq.traffic_type;
         qfle3_vfop_qctor_dump_tx(adapter, vf, init_p, setup_p,
                                                 q->index, q->sb_idx);
         
 
      }

      if (setup_q->param_valid & VFPF_RXQ_VALID) {
         struct ecore_rxq_setup_params *rxq_params = &setup_p->rxq_params;
       
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Setting RX Queue for vf[%d]\n", vf->abs_vfid);
         ECORE_SET_BIT(ECORE_Q_TYPE_HAS_RX, &q_type);

        /* Note: there is no support for different SBs
        * for TX and RX
        */
        q->sb_idx = setup_q->rxq.vf_sb;
        /* rx init */
        init_p->rx.hc_rate = setup_q->rxq.hc_rate;
        init_p->rx.sb_cq_index = setup_q->rxq.sb_index;
        qfle3_vf_mbx_set_q_flags(adapter, setup_q->rxq.flags, &init_p->rx.flags);

        /* rx setup - flags */
        qfle3_vf_mbx_set_q_flags(adapter, setup_q->rxq.flags, &setup_p->flags);
        
        /* rx setup - general */
        setup_p->gen_params.mtu = setup_q->rxq.mtu;

        /* rx setup - rx */
        rxq_params->drop_flags = setup_q->rxq.drop_flags;
        rxq_params->dscr_map = setup_q->rxq.rxq_addr;
        rxq_params->sge_map = setup_q->rxq.sge_addr;
        rxq_params->rcq_map = setup_q->rxq.rcq_addr;
        rxq_params->rcq_np_map = setup_q->rxq.rcq_np_addr;
        rxq_params->buf_sz = setup_q->rxq.buf_sz;
        rxq_params->tpa_agg_sz = setup_q->rxq.tpa_agg_sz;
        rxq_params->max_sges_pkt = setup_q->rxq.max_sge_pkt;
        rxq_params->sge_buf_sz = setup_q->rxq.sge_buf_sz;
        rxq_params->cache_line_log = setup_q->rxq.cache_line_log;
        rxq_params->sb_cq_index = setup_q->rxq.sb_index;

        /* rx setup - multicast engine */
        if (qfle3_vfq_is_leading(q)) {
           vmk_uint8 mcast_id = FW_VF_HANDLE(vf->abs_vfid);
           
           rxq_params->mcast_engine_id = mcast_id;
           ECORE_SET_BIT(ECORE_Q_FLG_MCAST, &setup_p->flags);
           QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: general setup_p->flags 0x%lx init_p->tx.flags is 0x%lx init_p->rx.flags is 0x%lx\n", setup_p->flags, init_p->tx.flags, init_p->rx.flags);
        }
       
 
        qfle3_vfop_qctor_dump_rx(adapter, vf, init_p, setup_p, q->index, q->sb_idx);
       
     }
     /* complete the preparations */

      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: init_p->tx.flags: 0x%lx init_p->tx.hc_rate: 0x%x init_p->tx.fw_sb_id: %d init_p->tx.sb_cq_index: %d\n", 
            init_p->tx.flags, init_p->tx.hc_rate, init_p->tx.fw_sb_id, init_p->tx.sb_cq_index);
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: init_p->rx.flags: 0x%lx init_p->rx.hc_rate: 0x%x init_p->rx.fw_sb_id: %d init_p->rx.sb_cq_index: %d init_p->max_cos: %d\n",
            init_p->rx.flags, init_p->rx.hc_rate, init_p->rx.fw_sb_id, init_p->rx.sb_cq_index, init_p->max_cos);
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: setup_p->gen_params.stat_id: %d setup_p->gen_params.spcl_id: %d setup_p->gen_params.mtu: %d"
            "setup_p->gen_params.cos: %d setup_p->gen_params.fp_hsi: %d\n", 
            setup_p->gen_params.stat_id, setup_p->gen_params.spcl_id, setup_p->gen_params.mtu, setup_p->gen_params.cos, setup_p->gen_params.fp_hsi);
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: setup_p->pause_params.bd_th_lo: 0x%x setup_p->pause_params.bd_th_hi: 0x%x setup_p->pause_params.rcq_th_lo: 0x%x" 
            "setup_p->pause_params.rcq_th_hi: 0x%x\n", setup_p->pause_params.bd_th_lo, setup_p->pause_params.bd_th_hi, setup_p->pause_params.rcq_th_lo, setup_p->pause_params.rcq_th_hi);
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: setup_p->pause_params.sge_th_lo: 0x%x setup_p->pause_params.sge_th_hi: 0x%x setup_p->pause_params.pri_map: 0x%x\n", 
                  setup_p->pause_params.sge_th_lo, setup_p->pause_params.sge_th_hi, setup_p->pause_params.pri_map);


     qfle3_vfop_qctor_prep(adapter, vf, q, &qctor, q_type);

      if (adapter_esx_vf->old_mtu != QFLE3_ESX_PASSTHRU_MTU_UNINITIALIZED && adapter_esx_vf->old_mtu != setup_p->gen_params.mtu){
         int i=5;
         while(i){
            if (adapter_esx_vf->flags & QFLE3_ESX_PASSTHRU_ALLOW_MTU){
               QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF: %d Allowed to set MTU to %d VF requested %d\n", vf->index, adapter_esx_vf->mtu_from_config, setup_p->gen_params.mtu);
               setup_p->gen_params.mtu = adapter_esx_vf->mtu_from_config;
               adapter_esx_vf->flags &= ~QFLE3_ESX_PASSTHRU_ALLOW_MTU;
               break;
            } else {
               vmk_WorldSleep(10000);
               QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF: %d waiting for VMkernel approval for MTU change %d\n", vf->index, i);
               --i ;
               continue;
            }
         }
         if (i==0 && !(adapter_esx_vf->flags & QFLE3_ESX_PASSTHRU_ALLOW_MTU)){
               QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF: %d Not alowed to set MTU to %d reverting to 1500\n", vf->index, setup_p->gen_params.mtu);
               setup_p->gen_params.mtu = 1500;

         }
      } else {
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF: %d Initializing MTU to %d\n", vf->index, setup_p->gen_params.mtu);
      }

      rc = qfle3_vf_queue_setup(adapter, vf, q->index, &qctor);
      if (rc)
         goto response;

      adapter_esx_vf->old_mtu=setup_p->gen_params.mtu;
  }
response:
        QFLE3_ESX_SET_PASSTHRU_RC_STATE(adapter, vf->index, rc, 0);
        qfle3_vf_mbx_resp(adapter, vf, rc);
}

static int qfle3_vf_filters_contain(struct vfpf_set_q_filters_tlv *filters,
                                    vmk_uint32 flags)
{
   int i, cnt = 0;
   for (i = 0; i < filters->n_mac_vlan_filters; i++)
      if  ((filters->filters[i].flags & flags) == flags)
         cnt++;

   return cnt;
}

static int qfle3_filters_validate_mac(struct qfle3_adapter *adapter, struct qfle3_virtf *vf, struct vfpf_set_q_filters_tlv *filters)
{
   struct pf_vf_bulletin_content *bulletin = QFLE3_VF_BULLETIN(adapter, vf->index);
   int rc = 0;

   /* if a mac was already set for this VF via the set vf mac ndo, we only
    * accept mac configurations of that mac. Why accept them at all?
    * because PF may have been unable to configure the mac at the time
    * since queue was not set up.
    */
   if (bulletin->valid_bitmap & 1 << MAC_ADDR_VALID) {
      struct vfpf_q_mac_vlan_filter *filter = NULL;
      int i;

      for (i = 0; i < filters->n_mac_vlan_filters; i++) {
         if (!(filters->filters[i].flags &
               VFPF_Q_FILTER_DEST_MAC_VALID))
            continue;

         /* once a mac was set by ndo can only accept
          * a single mac...
          */
         if (filter) {
            QFLE3_ERR("SRIOV: VF[%d] requested the addition of multiple macs after set_vf_mac ndo was called [%d filters]\n",vf->abs_vfid,
                 filters->n_mac_vlan_filters);
            rc = -1;
            goto response;
         }

         filter = &filters->filters[i];
      }

      /* ...and only the mac set by the ndo */
      if (filter && !ether_addr_equal(filter->mac, bulletin->mac)) {
         QFLE3_ERR("SRIOV: VF[%d] requested the addition of a mac address not matching the one configured by set_vf_mac ndo\n", vf->abs_vfid);

         QFLE3_ERR("ndo mac " QFLE3_MAC_FMT "\n", QFLE3_MAC_PRN_LIST(bulletin->mac));

         QFLE3_ERR("req mac " QFLE3_MAC_FMT "\n", QFLE3_MAC_PRN_LIST(filter->mac));

         rc = -1;
         goto response;
      }
   }

response:
   return rc;
}

static int qfle3_filters_validate_vlan(struct qfle3_adapter *adapter,
                                       struct qfle3_virtf *vf,
                                       struct vfpf_set_q_filters_tlv *filters)
{
   struct pf_vf_bulletin_content *bulletin = QFLE3_VF_BULLETIN(adapter, vf->index);
   int rc = 0;

   /* if vlan was set by hypervisor we don't allow guest to config vlan */
   if (bulletin->valid_bitmap & 1 << VLAN_VALID) {
      /* search for vlan filters */

      if (qfle3_vf_filters_contain(filters, VFPF_Q_FILTER_VLAN_TAG_VALID)) {
         QFLE3_ERR("SRIOV: BC: VF[%d] attempted to configure vlan but one was already set by Hypervisor. Aborting request\n",
                                  vf->abs_vfid);
         rc = -1;
         goto response;
      }
   }

   /* verify vf_qid */
   if (filters->vf_qid > vf_rxq_count(vf)) {
      rc = -1;
      goto response;
   }

response:
   return rc;
}

static int qfle3_vf_mbx_macvlan_list(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                                     struct vfpf_set_q_filters_tlv *tlv,
                                     struct qfle3_vf_mac_vlan_filters **pfl,
                                     vmk_uint32 type_flag)
{
        int i, j;
        struct qfle3_vf_mac_vlan_filters *fl = NULL;
        vmk_ByteCount fsz;

        fsz = tlv->n_mac_vlan_filters * sizeof(struct qfle3_vf_mac_vlan_filter) +
              sizeof(struct qfle3_vf_mac_vlan_filters);

        fl = ECORE_ZALLOC(fsz, GFP_ATOMIC, adapter);
        if (!fl)
           return -1;

        for (i = 0, j = 0; i < tlv->n_mac_vlan_filters; i++) {
                struct vfpf_q_mac_vlan_filter *msg_filter = &tlv->filters[i];

                if ((msg_filter->flags & type_flag) != type_flag)
                        continue;
                vmk_Memset(&fl->filters[j], 0, sizeof(fl->filters[j]));
                if (type_flag & VFPF_Q_FILTER_DEST_MAC_VALID) {
                        fl->filters[j].mac = msg_filter->mac;
                        fl->filters[j].type |= QFLE3_VF_FILTER_MAC;
                        QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Recieved Filter mac " QFLE3_MAC_FMT " filter type: QFLE3_VF_FILTER_MAC\n", QFLE3_MAC_PRN_LIST(msg_filter->mac)); 
                }
                if (type_flag & VFPF_Q_FILTER_VLAN_TAG_VALID) {
                        fl->filters[j].vid = msg_filter->vlan_tag;
                        fl->filters[j].type |= QFLE3_VF_FILTER_VLAN;   
                        QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Recieved Filter VLAN %d filter type: QFLE3_VF_FILTER_VLAN\n",  msg_filter->vlan_tag);
                }
                fl->filters[j].add = !!(msg_filter->flags & VFPF_Q_FILTER_SET);
                QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: filter is of type %d\n", fl->filters[j].add);
                fl->count++;
                j++;
        }
        if (!fl->count)
                ECORE_FREE(adapter, fl, sizeof(*f1));
        else
                *pfl = fl;

        return 0;
}


static int qfle3_vf_mbx_qfilters(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   int rc = 0;

   struct vfpf_set_q_filters_tlv *msg =
                &QFLE3_VF_MBX(adapter, vf->index)->msg->req.set_q_filters;


   /* check for any mac/vlan changes */
   if (msg->flags & VFPF_SET_Q_FILTERS_MAC_VLAN_CHANGED) {
      /* build vlan-mac list */
      struct qfle3_vf_mac_vlan_filters *fl = NULL;

      /* build vlan-mac list */
      rc = qfle3_vf_mbx_macvlan_list(adapter, vf, msg, &fl,
                                               VFPF_VLAN_MAC_FILTER);
      if (rc)
         goto op_err;

      if (fl) {
         rc = qfle3_esx_set_mac_passthru_config(adapter, vf, fl);        
         if (rc)
           goto op_err;

         rc = qfle3_esx_set_vlan_passthru_config(adapter, vf, msg);
         if (rc)
            goto op_err;

         /* set vlan-mac list */
         rc = qfle3_vf_mac_vlan_config_list(adapter, vf, fl, msg->vf_qid, VMK_FALSE);
         if (rc)
            goto op_err;
         }
      /* build mac list */
      fl = NULL;
      rc = qfle3_vf_mbx_macvlan_list(adapter, vf, msg, &fl, VFPF_MAC_FILTER);
      if (rc)
         goto op_err;

      if (fl) {
          rc = qfle3_esx_set_mac_passthru_config(adapter, vf, fl);    
          if (rc)
                goto op_err;
           
             /* set mac list */
         rc = qfle3_vf_mac_vlan_config_list(adapter, vf, fl, msg->vf_qid, VMK_FALSE);
         if (rc)
            goto op_err;
      }


      rc = qfle3_esx_set_vlan_passthru_config(adapter, vf, msg);
      if (rc)
         goto op_err;
   }

   if (msg->flags & VFPF_SET_Q_FILTERS_RX_MASK_CHANGED) {
      unsigned long accept = 0;
      struct pf_vf_bulletin_content *bulletin = QFLE3_VF_BULLETIN(adapter, vf->index);  

      /* Ignore VF requested mode; instead set a regular mode */
      if (msg->rx_mask !=  VFPF_RX_MASK_ACCEPT_NONE) {
         ECORE_SET_BIT(ECORE_ACCEPT_UNICAST, &accept);
         ECORE_SET_BIT(ECORE_ACCEPT_MULTICAST, &accept);
         ECORE_SET_BIT(ECORE_ACCEPT_BROADCAST, &accept);
      }

      /* any_vlan is not configured if HV is forcing VLAN
      * any_vlan is configured if
      *   1. VF does not support vlan filtering
      *   OR
      *   2. VF supports vlan filtering and explicitly requested it
      */
      if (msg->rx_mask & VFPF_RX_MASK_ACCEPT_ANY_VLAN)
         adapter->esx_vf[vf->index].any_vlan_req = 1;
      else
         adapter->esx_vf[vf->index].any_vlan_req = 0;

      if (!(bulletin->valid_bitmap & (1 << VLAN_VALID)) && qfle3_esx_vf_any_vlan_allowed(adapter, vf->index))    
          ECORE_SET_BIT(ECORE_ACCEPT_ANY_VLAN, &accept);
	
      /* set rx-mode */
      rc = qfle3_vf_rxmode(adapter, vf, msg->vf_qid, accept);
      if (rc)
         goto op_err;
   }

   if (msg->flags & VFPF_SET_Q_FILTERS_MULTICAST_CHANGED) {
      /* set mcasts */
      rc = qfle3_vf_mcast(adapter, vf, msg->multicast, msg->n_multicast, VMK_FALSE);
      if (rc)
         goto op_err;
   }
   
   qfle3_esx_passthru_config_setup_filters_finalize(adapter, vf->index, rc, 0);  

op_err:
   if (rc)
      QFLE3_ERR("SRIOV: BC: QFILTERS[%d:%d] error: rc %d\n", vf->abs_vfid, msg->vf_qid, rc);
   return rc;
}

static void qfle3_vf_mbx_dp_q_filter(struct qfle3_adapter *adapter,  int idx, struct vfpf_q_mac_vlan_filter *filter)
{
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: MAC-VLAN[%d] -- flags=0x%x\n", idx, filter->flags);
   if (filter->flags & VFPF_Q_FILTER_VLAN_TAG_VALID)
      QFLE3_DBG(QFLE3_DBG_IOV, "vlan=%d", filter->vlan_tag);
   if (filter->flags & VFPF_Q_FILTER_DEST_MAC_VALID)
      QFLE3_DBG(QFLE3_DBG_IOV, "MAC=" QFLE3_MAC_FMT "\n", QFLE3_MAC_PRN_LIST(filter->mac));
}

static void qfle3_vf_mbx_dp_q_filters(struct qfle3_adapter *adapter, struct vfpf_set_q_filters_tlv *filters)
{
    int i;

    if (filters->flags & VFPF_SET_Q_FILTERS_MAC_VLAN_CHANGED)
        for (i = 0; i < filters->n_mac_vlan_filters; i++)
            qfle3_vf_mbx_dp_q_filter(adapter,  i, &filters->filters[i]);

    if (filters->flags & VFPF_SET_Q_FILTERS_RX_MASK_CHANGED)
        QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: RX-MASK=0x%x\n", filters->rx_mask);

    if (filters->flags & VFPF_SET_Q_FILTERS_MULTICAST_CHANGED)
        for (i = 0; i < filters->n_multicast; i++)
            QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: MULTICAST=" QFLE3_MAC_FMT "\n", QFLE3_MAC_PRN_LIST(filters->multicast[i]));
}


static void qfle3_vf_mbx_set_q_filters(struct qfle3_adapter *adapter,
                                       struct qfle3_virtf *vf,
                                       struct qfle3_vf_mbx *mbx)
{
   struct vfpf_set_q_filters_tlv *filters = &mbx->msg->req.set_q_filters;
   int rc;

   rc = qfle3_filters_validate_mac(adapter, vf, filters);
   if (rc)
      goto response;

   rc = qfle3_filters_validate_vlan(adapter, vf, filters);
   if (rc)
      goto response;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF[%d] Q_FILTERS: queue[%d]\n", vf->abs_vfid, filters->vf_qid);

   /* print q_filter message */
   qfle3_vf_mbx_dp_q_filters(adapter, filters);    

   rc = qfle3_vf_mbx_qfilters(adapter, vf);
response:
   qfle3_vf_mbx_resp(adapter, vf, rc);
}



static void qfle3_vf_mbx_teardown_q(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                struct qfle3_vf_mbx *mbx)
{
   int qid = mbx->msg->req.q_op.vf_qid;
   int rc;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF[%d] Q_TEARDOWN: vf_qid=%d\n",
      vf->abs_vfid, qid);

   rc = qfle3_vf_queue_teardown(adapter, vf, qid);
   qfle3_vf_mbx_resp(adapter, vf, rc);
}


int
qfle3_esx_save_statsitics(struct qfle3_adapter *adapter, u8 vfID)
{
    int i;

    if (!adapter->vfdb)
        return 0;

    for (i = 0; i < QFLE3_VF_MAX_QUEUES; i++) {
        struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vfID];

        struct per_queue_stats *fw_stats = (struct per_queue_stats *)
               ((u8 *)adapter_esx_vf->vf_fw_stats +
                (i * QFLE3_ESX_VF_FW_STATS_SIZE));
        struct per_queue_stats *old_fw_stats =
               (struct per_queue_stats *)
               ((u8 *)adapter_esx_vf->old_vf_fw_stats +
                (i * QFLE3_ESX_VF_FW_STATS_SIZE));

        struct xstorm_per_queue_stats *xstorm =
            &fw_stats->xstorm_queue_statistics;
        struct xstorm_per_queue_stats *old_xstorm =
            &old_fw_stats->xstorm_queue_statistics;

        struct tstorm_per_queue_stats *tstorm =
            &fw_stats->tstorm_queue_statistics;
        struct tstorm_per_queue_stats *old_tstorm =
            &old_fw_stats->tstorm_queue_statistics;

        struct ustorm_per_queue_stats *ustorm =
            &fw_stats->ustorm_queue_statistics;
        struct ustorm_per_queue_stats *old_ustorm =
            &old_fw_stats->ustorm_queue_statistics;

        ADD_64(old_tstorm->rcv_ucast_bytes.hi,
               le32toh(tstorm->rcv_ucast_bytes.hi),
               old_tstorm->rcv_ucast_bytes.lo,
               le32toh(tstorm->rcv_ucast_bytes.lo));
        old_tstorm->rcv_ucast_pkts +=
            le32toh(tstorm->rcv_ucast_pkts);
        old_tstorm->checksum_discard +=
            le32toh(tstorm->checksum_discard);
        ADD_64(old_tstorm->rcv_bcast_bytes.hi,
               le32toh(tstorm->rcv_bcast_bytes.hi),
               old_tstorm->rcv_bcast_bytes.lo,
               le32toh(tstorm->rcv_bcast_bytes.lo));
        old_tstorm->rcv_bcast_pkts +=
            le32toh(tstorm->rcv_bcast_pkts);
        old_tstorm->pkts_too_big_discard +=
            le32toh(tstorm->pkts_too_big_discard);
        ADD_64(old_tstorm->rcv_mcast_bytes.hi,
               le32toh(tstorm->rcv_mcast_bytes.hi),
               old_tstorm->rcv_mcast_bytes.lo,
               le32toh(tstorm->rcv_mcast_bytes.lo));
        old_tstorm->rcv_mcast_pkts +=
            le32toh(tstorm->rcv_mcast_pkts);
        old_tstorm->ttl0_discard +=
            le32toh(tstorm->ttl0_discard);
        old_tstorm->no_buff_discard +=
            le32toh(tstorm->no_buff_discard);

        ADD_64(old_ustorm->ucast_no_buff_bytes.hi,
               le32toh(ustorm->ucast_no_buff_bytes.hi),
               old_ustorm->ucast_no_buff_bytes.lo,
               le32toh(ustorm->ucast_no_buff_bytes.lo));
        ADD_64(old_ustorm->mcast_no_buff_bytes.hi,
               le32toh(ustorm->mcast_no_buff_bytes.hi),
               old_ustorm->mcast_no_buff_bytes.lo,
               le32toh(ustorm->mcast_no_buff_bytes.lo));
        ADD_64(old_ustorm->bcast_no_buff_bytes.hi,
               le32toh(ustorm->bcast_no_buff_bytes.hi),
               old_ustorm->bcast_no_buff_bytes.lo,
               le32toh(ustorm->bcast_no_buff_bytes.lo));
        old_ustorm->ucast_no_buff_pkts +=
            le32toh(ustorm->ucast_no_buff_pkts);
        old_ustorm->mcast_no_buff_pkts +=
            le32toh(ustorm->mcast_no_buff_pkts);
        old_ustorm->bcast_no_buff_pkts +=
            le32toh(ustorm->bcast_no_buff_pkts);
        old_ustorm->coalesced_pkts +=
            le32toh(ustorm->coalesced_pkts);
        ADD_64(old_ustorm->coalesced_bytes.hi,
               le32toh(ustorm->coalesced_bytes.hi),
               old_ustorm->coalesced_bytes.lo,
               le32toh(ustorm->coalesced_bytes.lo));
        old_ustorm->coalesced_events +=
            le32toh(ustorm->coalesced_events);
        old_ustorm->coalesced_aborts +=
            le32toh(ustorm->coalesced_aborts);


        ADD_64(old_xstorm->ucast_bytes_sent.hi,
               le32toh(xstorm->ucast_bytes_sent.hi),
               old_xstorm->ucast_bytes_sent.lo,
               le32toh(xstorm->ucast_bytes_sent.lo));
        ADD_64(old_xstorm->mcast_bytes_sent.hi,
               le32toh(xstorm->mcast_bytes_sent.hi),
               old_xstorm->mcast_bytes_sent.lo,
               le32toh(xstorm->mcast_bytes_sent.lo));
        ADD_64(old_xstorm->bcast_bytes_sent.hi,
               le32toh(xstorm->bcast_bytes_sent.hi),
               old_xstorm->bcast_bytes_sent.lo,
               le32toh(xstorm->bcast_bytes_sent.lo));
        old_xstorm->ucast_pkts_sent +=
            le32toh(xstorm->ucast_pkts_sent);
        old_xstorm->mcast_pkts_sent +=
            le32toh(xstorm->mcast_pkts_sent);
        old_xstorm->bcast_pkts_sent +=
            le32toh(xstorm->bcast_pkts_sent);
        old_xstorm->error_drop_pkts +=
            le32toh(xstorm->error_drop_pkts);
    }

    return 0;
}

void
qfle3_esx_vf_close(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   qfle3_esx_vf_remove_all_gvlans(adapter, vf);      
   qfle3_esx_save_statsitics(adapter, vf->index);
   return;
}


static void qfle3_vf_mbx_close_vf(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
              struct qfle3_vf_mbx *mbx)
{
   int rc;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF[%d] VF_CLOSE\n", vf->abs_vfid);

   qfle3_esx_vf_close(adapter, vf);
   rc = qfle3_vf_close(adapter, vf);
   qfle3_vf_mbx_resp(adapter, vf, rc);
}


void qfle3_esx_vf_release(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   int i;

   qfle3_esx_vf_reset_all_vmk_gvlans(adapter, vf);  

   for (i = 0; i < adapter->vfdb->sriov.nr_virtfn; i++) {
      struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[i];

      adapter_esx_vf->old_mtu = QFLE3_ESX_PASSTHRU_MTU_UNINITIALIZED;
      vmk_Memset(adapter_esx_vf->old_vf_fw_stats, 0, QFLE3_ESX_VF_FW_TOTAL_STATS_SIZE);
      vmk_Memset(adapter_esx_vf->vf_fw_stats, 0, QFLE3_ESX_VF_FW_TOTAL_STATS_SIZE); 
   }
}

static void qfle3_vf_mbx_release_vf(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                struct qfle3_vf_mbx *mbx)
{
   int rc;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: VF[%d] VF_RELEASE\n", vf->abs_vfid);

   qfle3_esx_vf_release(adapter, vf);
   rc = qfle3_vf_free(adapter, vf);
   qfle3_vf_mbx_resp(adapter, vf, rc);
}

int qfle3_vf_rss_update(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                        struct ecore_config_rss_params *rss)
{
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: vf[%d]\n", vf->abs_vfid);
   SET_BIT(RAMROD_COMP_WAIT, &rss->ramrod_flags);
   return ecore_config_rss(adapter, rss);
}


static void qfle3_vf_mbx_update_rss(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                struct qfle3_vf_mbx *mbx)
{
   struct ecore_config_rss_params rss;
   struct vfpf_rss_tlv *rss_tlv = &mbx->msg->req.update_rss;
   int rc = 0;

   if (rss_tlv->ind_table_size != T_ETH_INDIRECTION_TABLE_SIZE || rss_tlv->rss_key_size != T_ETH_RSS_KEY) {
      QFLE3_ERR("SRIOV: BC: Failing rss configuration of vf %d due to size mismatch\n",
           vf->index);
      rc = -1;
      goto mbx_resp;
   }

   vmk_Memset(&rss, 0, sizeof(struct ecore_config_rss_params));

   /* set vfop params according to rss tlv */
   vmk_Memcpy(rss.ind_table, rss_tlv->ind_table,
          T_ETH_INDIRECTION_TABLE_SIZE);
   vmk_Memcpy(rss.rss_key, rss_tlv->rss_key, sizeof(rss_tlv->rss_key));
   rss.rss_obj = &vf->rss_conf_obj;
   rss.rss_result_mask = rss_tlv->rss_result_mask;

   /* flags handled individually for backward/forward compatability */
   rss.rss_flags = 0;
   rss.ramrod_flags = 0;

   if (rss_tlv->rss_flags & VFPF_RSS_MODE_DISABLED)
      ECORE_SET_BIT(ECORE_RSS_MODE_DISABLED, &rss.rss_flags);
   if (rss_tlv->rss_flags & VFPF_RSS_MODE_REGULAR)
      ECORE_SET_BIT(ECORE_RSS_MODE_REGULAR, &rss.rss_flags);
   if (rss_tlv->rss_flags & VFPF_RSS_SET_SRCH)
      ECORE_SET_BIT(ECORE_RSS_SET_SRCH, &rss.rss_flags);
   if (rss_tlv->rss_flags & VFPF_RSS_IPV4)
      ECORE_SET_BIT(ECORE_RSS_IPV4, &rss.rss_flags);
   if (rss_tlv->rss_flags & VFPF_RSS_IPV4_TCP)
      ECORE_SET_BIT(ECORE_RSS_IPV4_TCP, &rss.rss_flags);
   if (rss_tlv->rss_flags & VFPF_RSS_IPV4_UDP)
      ECORE_SET_BIT(ECORE_RSS_IPV4_UDP, &rss.rss_flags);
   if (rss_tlv->rss_flags & VFPF_RSS_IPV6)
      ECORE_SET_BIT(ECORE_RSS_IPV6, &rss.rss_flags);
   if (rss_tlv->rss_flags & VFPF_RSS_IPV6_TCP)
      ECORE_SET_BIT(ECORE_RSS_IPV6_TCP, &rss.rss_flags);
   if (rss_tlv->rss_flags & VFPF_RSS_IPV6_UDP)
      ECORE_SET_BIT(ECORE_RSS_IPV6_UDP, &rss.rss_flags);

   if ((!(rss_tlv->rss_flags & VFPF_RSS_IPV4_TCP) && rss_tlv->rss_flags & VFPF_RSS_IPV4_UDP) ||
       (!(rss_tlv->rss_flags & VFPF_RSS_IPV6_TCP) && rss_tlv->rss_flags & VFPF_RSS_IPV6_UDP)) {
      QFLE3_ERR("SRIOV: BC: about to hit a FW assert. aborting...\n");
      rc = -1;
      goto mbx_resp;
   }

   rc = qfle3_vf_rss_update(adapter, vf, &rss);
mbx_resp:
   qfle3_vf_mbx_resp(adapter, vf, rc);
}

static int qfle3_validate_tpa_params(struct qfle3_adapter *adapter,
                   struct vfpf_tpa_tlv *tpa_tlv)
{
   int rc = 0;

   if (tpa_tlv->tpa_client_info.max_sges_for_packet >
       U_ETH_MAX_SGES_FOR_PACKET) {
      rc = -1;
      QFLE3_ERR("SRIOV: BC: TPA update: max_sges received %d, max is %d\n",
           tpa_tlv->tpa_client_info.max_sges_for_packet,
           U_ETH_MAX_SGES_FOR_PACKET);
   }

   if (tpa_tlv->tpa_client_info.max_tpa_queues > MAX_AGG_QS(adapter)) {
      rc = -1;
      QFLE3_ERR("SRIOV: BC: TPA update: max_tpa_queues received %d, max is %d\n",
           tpa_tlv->tpa_client_info.max_tpa_queues,
           MAX_AGG_QS(adapter));
   }

   return rc;
}


int qfle3_vf_tpa_update(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
         struct vfpf_tpa_tlv *tlv, struct ecore_queue_update_tpa_params *params)
{
   vmk_uint64 *sge_addr = tlv->tpa_client_info.sge_addr;
   struct ecore_queue_state_params qstate;
   int qid, rc = 0;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: vf[%d]\n", vf->abs_vfid);

   /* Set ramrod params */
   vmk_Memset(&qstate, 0, sizeof(struct ecore_queue_state_params));
   vmk_Memcpy(&qstate.params.update_tpa, params, sizeof(struct ecore_queue_update_tpa_params));
   qstate.cmd = ECORE_Q_CMD_UPDATE_TPA;
   SET_BIT(RAMROD_COMP_WAIT, &qstate.ramrod_flags);

   for (qid = 0; qid < vf_rxq_count(vf); qid++) {
      qstate.q_obj = &(vf->vfqs[qid].sp_obj);
      qstate.params.update_tpa.sge_map = sge_addr[qid];
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: sge_addr[%d:%d] %08x:%08x\n", vf->abs_vfid, qid, U64_HI(sge_addr[qid]),
                      U64_LO(sge_addr[qid]));
      rc = ecore_queue_state_change(adapter, &qstate);
      if (rc) {
         QFLE3_ERR("SRIOV: BC: Failed to configure sge_addr %08x:%08x for [%d:%d]\n",
              U64_HI(sge_addr[qid]), U64_LO(sge_addr[qid]), vf->abs_vfid, qid);
         return rc;
      }
   }
   return rc;
}


static void qfle3_vf_mbx_update_tpa(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                struct qfle3_vf_mbx *mbx)
{
   struct ecore_queue_update_tpa_params vf_op_params;
   struct vfpf_tpa_tlv *tpa_tlv = &mbx->msg->req.update_tpa;
   int rc = 0;

   vmk_Memset(&vf_op_params, 0, sizeof(vf_op_params));

   if (qfle3_validate_tpa_params(adapter, tpa_tlv))
      goto mbx_resp;

   vf_op_params.complete_on_both_clients =
      tpa_tlv->tpa_client_info.complete_on_both_clients;
   vf_op_params.dont_verify_thr =
      tpa_tlv->tpa_client_info.dont_verify_thr;
   vf_op_params.max_agg_sz =
      tpa_tlv->tpa_client_info.max_agg_size;
   vf_op_params.max_sges_pkt =
      tpa_tlv->tpa_client_info.max_sges_for_packet;
   vf_op_params.max_tpa_queues =
      tpa_tlv->tpa_client_info.max_tpa_queues;
   vf_op_params.sge_buff_sz =
      tpa_tlv->tpa_client_info.sge_buff_size;
   vf_op_params.sge_pause_thr_high =
      tpa_tlv->tpa_client_info.sge_pause_thr_high;
   vf_op_params.sge_pause_thr_low =
      tpa_tlv->tpa_client_info.sge_pause_thr_low;
   vf_op_params.tpa_mode =
      tpa_tlv->tpa_client_info.tpa_mode;
   vf_op_params.update_ipv4 =
      tpa_tlv->tpa_client_info.update_ipv4;
   vf_op_params.update_ipv6 =
      tpa_tlv->tpa_client_info.update_ipv6;

   rc = qfle3_vf_tpa_update(adapter, vf, tpa_tlv, &vf_op_params);

mbx_resp:
   qfle3_vf_mbx_resp(adapter, vf, rc);
}


/* dispatch request */
static void qfle3_vf_mbx_request(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                                      struct qfle3_vf_mbx *mbx)
{ 
  
   int i;
   /* check if tlv type is known */
   if (qfle3_tlv_supported(mbx->first_tlv.tl.type)) {
      /* Lock the per vf op mutex and note the locker's identity.
      * The unlock will take place in mbx response.*/
      qfle3_lock_vf_pf_channel(adapter, vf, mbx->first_tlv.tl.type);

      /* switch on the opcode */
      switch (mbx->first_tlv.tl.type) {
         case CHANNEL_TLV_ACQUIRE:
            qfle3_vf_mbx_acquire(adapter, vf, mbx);
            return;
         case CHANNEL_TLV_INIT:
            qfle3_vf_mbx_init_vf(adapter, vf, mbx);
            return;
         case CHANNEL_TLV_SETUP_Q:
            qfle3_vf_mbx_setup_q(adapter, vf, mbx);
            return;
         case CHANNEL_TLV_SET_Q_FILTERS:
            qfle3_vf_mbx_set_q_filters(adapter, vf, mbx);
            return;
         case CHANNEL_TLV_TEARDOWN_Q:
            qfle3_vf_mbx_teardown_q(adapter, vf, mbx);
            return;
         case CHANNEL_TLV_CLOSE:
            qfle3_vf_mbx_close_vf(adapter, vf, mbx);
            return;
         case CHANNEL_TLV_RELEASE:
            qfle3_vf_mbx_release_vf(adapter, vf, mbx);
            return;
         case CHANNEL_TLV_UPDATE_RSS:
            qfle3_vf_mbx_update_rss(adapter, vf, mbx);
            return;
         case CHANNEL_TLV_UPDATE_TPA:
            qfle3_vf_mbx_update_tpa(adapter, vf, mbx);
            return;
         }
   } else {
      /* unknown TLV - this may belong to a VF driver from the future
      * - a version written after this PF driver was written, which
      * supports features unknown as of yet. Too bad since we don't
      * support them. Or this may be because someone wrote a crappy
      * VF driver and is sending garbage over the channel.
      */
      QFLE3_ERR("SRIOV: BC; Unknown TLV. type %d length %d vf->state was %d. first 20 bytes of mailbox buffer:\n",
                   mbx->first_tlv.tl.type, mbx->first_tlv.tl.length, vf->state);
       for(i = 0; i < 20; i++)
       QFLE3_ERR("SRIOV: BC: %x ", mbx->first_tlv.tl.type);
   }        
   /* can we respond to VF (do we have an address for it?) */
   if (vf->state == VF_ACQUIRED || vf->state == VF_ENABLED) {
      /* notify the VF that we do not support this request */
      qfle3_vf_mbx_resp(adapter, vf, PFVF_STATUS_NOT_SUPPORTED);
   } else {
      /* can't send a response since this VF is unknown to us
      * just ack the FW to release the mailbox and unlock
      * the channel.*/
      storm_memset_vf_mbx_ack(adapter, vf->abs_vfid);
      vmk_CPUMemFenceReadWrite();
      qfle3_unlock_vf_pf_channel(adapter, vf, mbx->first_tlv.tl.type);
   }
}

/* handle new vf-pf messages */
void qfle3_vf_mbx(struct qfle3_adapter *adapter)
{
   struct qfle3_vfdb *vfdb = adapter->vfdb;
   vmk_uint64 events;
   vmk_uint8 vf_idx;
   int rc;

   if(!vfdb)
      return;
   
   vmk_SemaLock(&vfdb->event_mutex);
   events = vfdb->event_occur;
   vfdb->event_occur = 0;
   vmk_SemaUnlock(&vfdb->event_mutex);

   for( vf_idx=0; vf_idx < adapter->vfdb->sriov.nr_virtfn; vf_idx++) {
      struct qfle3_vf_mbx *mbx = &(adapter->vfdb->mbxs[vf_idx]);
      struct qfle3_virtf *vf = QFLE3_VF(adapter, vf_idx);;
      /* Handle VFs which have pending events */
      if (!(events & (1ULL << vf_idx)))
         continue;
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Handling vf pf event vfid %d, address: [%x:%x], resp_offset 0x%x\n",
                     vf_idx, mbx->vf_addr_hi, mbx->vf_addr_lo, mbx->first_tlv.resp_msg_offset);

      /* dmae to get the VF request */
      rc = qfle3_copy32_vf_dmae(adapter, VMK_TRUE, mbx->msg_mapping, vf->abs_vfid, mbx->vf_addr_hi,
                                   mbx->vf_addr_lo, sizeof(union vfpf_tlvs)/4);

      if(rc) {
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Failed to copy request VF %d\n", vf->abs_vfid);
         qfle3_vf_release(adapter, vf);
         return;
      }

      /* process the VF message header */
      mbx->first_tlv = mbx->msg->req.first_tlv;

      /* Clean response buffer to refrain from falsely
      * seeing chains. */
      vmk_Memset(&mbx->msg->resp, 0, sizeof(union pfvf_tlvs));

      /* dispatch the request (will prepare the response) */
      qfle3_vf_mbx_request(adapter, vf, mbx);
   } 
}

static void qfle3_vf_flr_clnup_hw(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   vmk_uint32 poll_cnt = qfle3_flr_clnup_poll_count(adapter);

   /* DQ usage counter */
   qfle3_pretend_func(adapter, HW_VF_HANDLE(adapter, vf->abs_vfid));
   qfle3_flr_clnup_poll_hw_counter(adapter, DORQ_REG_VF_USAGE_CNT,
                                      "DQ VF usage counter timed out", poll_cnt);
   qfle3_pretend_func(adapter, QFLE3_ABS_FUNC(adapter));

   /* FW cleanup command - poll for the results */
   if(qfle3_send_final_clnup(adapter, (vmk_uint8)FW_VF_HANDLE(vf->abs_vfid),poll_cnt))
      QFLE3_ERR("SRIOV: BC: VF[%d] Final cleanup timed-out\n", vf->abs_vfid);
   
   /* verify TX hw is flushed */
   qfle3_tx_hw_flushed(adapter, poll_cnt);
}

void qfle3_vf_remove_all_gvlans(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vf->index];
   vmk_Memset(&adapter_esx_vf->requested_gvlans, 0, 64 * sizeof(vmk_uint64));
}

static int qfle3_vf_queue_flr(struct qfle3_adapter *adapter, struct qfle3_virtf *vf,
                                 int qid)
{
   int rc=0;
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: FLR Request from VF[%d:%d]\n", vf->abs_vfid, qid);

   /* If needed, clean the filtering data base */
   if ((qid == LEADING_IDX) && (qfle3_validate_vf_sp_objs(adapter,  vf, VMK_FALSE))) {
      if (ecore_get_q_logical_state(adapter, &(vf->vfqs[qid].sp_obj)) == ECORE_Q_LOGICAL_STATE_STOPPED) {
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: queue was already stopped. Aborting lock deletion at this time \n");
      } else {
         rc = qfle3_vf_vlan_mac_clear(adapter, vf, qid, VMK_TRUE, QFLE3_VF_FILTER_VLAN_MAC);

         if(rc)
            goto op_err;

         rc = qfle3_vf_vlan_mac_clear(adapter, vf, qid, VMK_TRUE, QFLE3_VF_FILTER_VLAN);

         if(rc)
            goto op_err;
      
         rc = qfle3_vf_vlan_mac_clear(adapter, vf, qid, VMK_TRUE, QFLE3_VF_FILTER_MAC);

         if(rc)
            goto op_err;

        
         struct ecore_vlan_mac_obj *obj;
         obj = &(vf->vfqs[qid].mac_obj);
         QFLE3_DBG(QFLE3_DBG_IOV, "Destroying spinlock exe_queue.lock for mac_obj\n");
         qfle3_destroy_spinlock(obj->exe_queue.lock);
         obj = &(vf->vfqs[qid].vlan_mac_obj);
         QFLE3_DBG(QFLE3_DBG_IOV, "Destroying spinlock exe_queue.lock for vlan_mac_obj\n");
         qfle3_destroy_spinlock(obj->exe_queue.lock);
         obj = &(vf->vfqs[qid].vlan_obj);
         QFLE3_DBG(QFLE3_DBG_IOV, "Destroying spinlock exe_queue.lock for vlan_obj\n");
         qfle3_destroy_spinlock(obj->exe_queue.lock);
      }
   }

   /* Terminate queue */
   if(((vf)->vfqs[qid].sp_obj.state) != ECORE_Q_STATE_RESET) {
      struct ecore_queue_state_params qstate;
         
      vmk_Memset(&qstate, 0, sizeof(struct ecore_queue_state_params));
      qstate.q_obj = &((vf)->vfqs[qid].sp_obj);
      qstate.q_obj->state = ECORE_Q_STATE_STOPPED;
      qstate.cmd = ECORE_Q_CMD_TERMINATE;
      ECORE_SET_BIT(RAMROD_COMP_WAIT, &qstate.ramrod_flags);
      rc = ecore_queue_state_change(adapter, &qstate);
      if (rc)
         goto op_err;
   }

      qfle3_esx_vf_remove_all_gvlans(adapter, vf);       
      qfle3_esx_vf_reset_all_vmk_gvlans(adapter, vf);   

   return 0;
op_err:
   QFLE3_ERR("SRIOV: BC: vf[%d:%d] error: rc %d\n", vf->abs_vfid, qid, rc);
   return rc;

}

/* enable vf_pf mailbox (aka vf-pf-channel) */
void qfle3_vf_enable_mbx(struct qfle3_adapter *adapter, vmk_uint8 abs_vfid)
{
   /* enable the mailbox in the FW */
   storm_memset_vf_mbx_ack(adapter, abs_vfid);
   storm_memset_vf_mbx_valid(adapter, abs_vfid);
   /* enable the VF access to the mailbox */
   qfle3_vf_enable_access(adapter, abs_vfid);
}
   
static void qfle3_vf_flr(struct qfle3_adapter *adapter, struct qfle3_virtf *vf)
{
   int rc=0;
   int i;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: FLR Request from VF[%d]\n", vf->abs_vfid);

   /* the cleanup operations are valid if and only if the VF
   * was first acquired.
   */

   for (i = 0; i < (vf)->alloc_resc.num_rxqs; i++) {
      rc = qfle3_vf_queue_flr(adapter, vf, i);
      if (rc)
         goto out;
   }

   /* remove multicasts */
   qfle3_vf_mcast(adapter, vf, NULL, 0, VMK_TRUE);
   
   /* dispatch final cleanup and wait for HW queues to flush */
   qfle3_vf_flr_clnup_hw(adapter, vf);

   /* release VF resources */
   qfle3_vf_free_resc(adapter, vf);

   /* re-open the mailbox */
   qfle3_vf_enable_mbx(adapter, vf->abs_vfid);

   return;

out:
   QFLE3_ERR("SRIOV: BC: vf[%d:%d] failed flr: rc %d\n",
                vf->abs_vfid, i, rc);
}


static void qfle3_vf_flr_clnup(struct qfle3_adapter *adapter)
{
   struct qfle3_virtf *vf;
   int i;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Executing qfle3_vf_flr_clnup\n");
   for(i = 0; i < adapter->vfdb->sriov.nr_virtfn; i++) {
      /* VF should be RESET & in FLR cleanup states */
      if((adapter->vfdb->vfs[i].state != VF_RESET) || !(adapter->vfdb->vfs[i].flr_clnup_stage))
         continue;

      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Next vf to cleanup: %d. Num of vfs: %d\n",
                    i, adapter->vfdb->sriov.nr_virtfn);

      vf =  ( ((adapter->vfdb) && (adapter->vfdb->vfs))? &(adapter->vfdb->vfs[i]) : NULL);

      /* lock the vf pf channel */
      qfle3_lock_vf_pf_channel(adapter, vf, CHANNEL_TLV_FLR);

      /* invoke the VF FLR SM */
      qfle3_vf_flr(adapter, vf);

      /* mark the VF to be ACKED and continue */
      vf->flr_clnup_stage = VMK_FALSE;
      qfle3_unlock_vf_pf_channel(adapter, vf, CHANNEL_TLV_FLR);
   }

   /* Acknowledge the handled VFs.
   * we are acknowledge all the vfs which an flr was requested for, even
   * if amongst them there are such that we never opened, since the mcp
   * will interrupt us immediately again if we only ack some of the bits,
   * resulting in an endless loop. This can happen for example in KVM
   * where an 'all ones' flr request is sometimes given by hyper visor
   */

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: DRV_STATUS_VF_DISABLED ACK for vfs 0x%x 0x%x\n",
                 adapter->vfdb->flrd_vfs[0], adapter->vfdb->flrd_vfs[1]);

   for (i = 0; i < FLRD_VFS_DWORDS; i++)
      SHMEM2_WR(adapter, drv_ack_vf_disabled[QFLE3_FW_MB_IDX(adapter)][i], adapter->vfdb->flrd_vfs[i]);

   elink_cb_fw_command(adapter, DRV_MSG_CODE_VF_DISABLED_DONE, 0);

   /* clear the acked bits - better yet if the MCP implemented
    * write to clear semantics
   */
   for (i = 0; i < FLRD_VFS_DWORDS; i++)   
      SHMEM2_WR(adapter, drv_ack_vf_disabled[QFLE3_FW_MB_IDX(adapter)][i],0);
}

void qfle3_vf_handle_flr_event(struct qfle3_adapter *adapter)
{
   int i;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Executing qfle3_vf_handle_flr_event\n");
   /* Read FLR'd VFs */
   for (i = 0; i < FLRD_VFS_DWORDS; i++)
      adapter->vfdb->flrd_vfs[i] = SHMEM2_RD(adapter, mcp_vf_disabled[i]);

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: DRV_STATUS_VF_DISABLED received for vfs 0x%x 0x%x\n",
              adapter->vfdb->flrd_vfs[0], adapter->vfdb->flrd_vfs[1]);

   for(i=0;i<(adapter->vfdb->sriov.nr_virtfn);i++) {
      struct qfle3_virtf *vf = QFLE3_VF(adapter, i);
      vmk_uint32 reset = 0;

      if(vf->abs_vfid < 32)
         reset = adapter->vfdb->flrd_vfs[0] & (1 << vf->abs_vfid);
      else
         reset = adapter->vfdb->flrd_vfs[1] & (1 << (vf->abs_vfid - 32));

      if (reset) {
         /* set as reset and ready for cleanup */
         vf->state = VF_RESET;
         vf->flr_clnup_stage = VMK_TRUE;

         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Initiating Final cleanup for VF %d\n", vf->abs_vfid);
      }
   }
   /* do the FLR cleanup for all marked VFs*/
   qfle3_vf_flr_clnup(adapter);
}

static void
qfle3_vf_set_igu_info(struct qfle3_adapter *adapter, vmk_uint8 igu_sb_id, vmk_uint8 abs_vfid)
{
   struct qfle3_virtf *vf = qfle3_vf_by_abs_fid(adapter, abs_vfid);
   if (vf) {
      /* the first igu entry belonging to VFs of this PF */
      if(!adapter->vfdb->first_vf_igu_entry)
         adapter->vfdb->first_vf_igu_entry = igu_sb_id;

      /* the first igu entry belonging to this VF */
      if (!(vf->alloc_resc.num_sbs))
         vf->igu_base_id = igu_sb_id;

      ++(vf->alloc_resc.num_sbs);
      ++(vf->sb_count);
   }
   (adapter->vfdb->vf_sbs_pool)++;
}

static int
qfle3_get_vf_igu_cam_info(struct qfle3_adapter *adapter)
{
   int sb_id;
   vmk_uint32 val;
   vmk_uint8 fid, current_pf = 0;

   /* IGU in normal mode - read CAM */
   for (sb_id = 0; sb_id < IGU_REG_MAPPING_MEMORY_SIZE; sb_id++) {
      val = REG_RD(adapter, IGU_REG_MAPPING_MEMORY + sb_id * 4);
      if (!(val & IGU_REG_MAPPING_MEMORY_VALID))
         continue;
      fid = GET_FIELD((val), IGU_REG_MAPPING_MEMORY_FID);
      if (fid & IGU_FID_ENCODE_IS_PF)
         current_pf = fid & IGU_FID_PF_NUM_MASK;
      else if (current_pf == QFLE3_FUNC(adapter))
         qfle3_vf_set_igu_info(adapter, sb_id,(fid & IGU_FID_VF_NUM_MASK));

      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: %s[%d], igu_sb_id=%d, msix=%d\n", ((fid & IGU_FID_ENCODE_IS_PF) ? "PF" : "VF"),    
         ((fid & IGU_FID_ENCODE_IS_PF) ? (fid & IGU_FID_PF_NUM_MASK) :(fid & IGU_FID_VF_NUM_MASK)), sb_id,
         GET_FIELD((val), IGU_REG_MAPPING_MEMORY_VECTOR));
   }

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: vf_sbs_pool is %d\n",adapter->vfdb->vf_sbs_pool);

   return adapter->vfdb->vf_sbs_pool;
}

static void qfle3_disable_sriov(struct qfle3_adapter *adapter)
{
   if(adapter->num_vf_actual){
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Disabling vfs.\n");
   vmk_PCIDisableVFs(adapter->pdev);
   adapter->num_vf_actual=0;
   }
  return;

}

#define IGU_ENTRY_SIZE 4

void qfle3_iov_remove_one(struct qfle3_adapter *adapter)
{
   int vf_idx;
   int rc,i;
   struct qfle3_sriov *iov;

   /* if SRIOV is not enabled there's nothing to do */
   if(!adapter->vfdb){
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Adapter vfdb not allocated yet\n");
      return;
   }

   qfle3_disable_sriov(adapter);
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Disabled SRIOV\n");
   if (adapter->bc_service_worldid) {
      adapter->flush_req=1;
      rc =  vmk_WorldForceWakeup(adapter->bc_service_worldid);      
      if (rc != VMK_OK)
         QFLE3_ERR("SRIOV: BC: Wake up failed, for SRIOV WQ\n");
      vmk_WorldDestroy(adapter->bc_service_worldid);
      vmk_WorldWaitForDeath(adapter->bc_service_worldid);
      adapter->bc_service_worldid=0;
   }
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Back Channel kernel Word Deleted\n");
   qfle3_destroy_spinlock(adapter->sriov_bc_lock);
   if(adapter->iov_task_state) {
      vmk_BitVectorFree(qfle3_mod_info.heapID, adapter->iov_task_state);
   } 
 
   iov = &(adapter->vfdb->sriov);

   for (i=0; i< iov->nr_virtfn; i++) {
      struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[i];

      qfle3_heap_free(adapter_esx_vf->old_vf_fw_stats);      
      vmk_SemaDestroy(&adapter->vfdb->vfs[i].op_mutex);
   }      
   vmk_SemaDestroy(&adapter->vfdb->event_mutex);
   vmk_SemaDestroy(&adapter->vfdb->bulletin_mutex);

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Back Channel Locks Destroyed\n");
   adapter->vfdb->sriov.nr_virtfn=0;
   /* disable access to all VFs */
   for(vf_idx = 0; vf_idx < adapter->vfdb->sriov.total; vf_idx++) {
      qfle3_pretend_func(adapter, HW_VF_HANDLE(adapter, adapter->vfdb->sriov.first_vf_in_pf + vf_idx));   
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Disabling internal access for vf %d\n", adapter->vfdb->sriov.first_vf_in_pf + vf_idx);
      qfle3_vf_enable_internal(adapter,0);
      qfle3_pretend_func(adapter, QFLE3_ABS_FUNC(adapter));
   }
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: About to free SRIOV Heaps\n");
   if(adapter->esx_vf)
      qfle3_heap_free(adapter->esx_vf);

   if(adapter->vfdb) {
      qfle3_heap_free(adapter->vfdb->vfqs);   
      qfle3_heap_free(adapter->vfdb->vfs);
      qfle3_heap_free(adapter->vfdb);
   }
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Heap Freed \n");
   adapter->vfdb = NULL;
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: adapter->vfdb is now NULL\n");
   adapter->flags &= ~TX_SWITCHING;
}


static int qfle3_enable_sriov(struct qfle3_adapter *adapter)
{
   int rc = 0; 
   int vf_idx, sb_idx, vfq_idx, qcount, first_vf;
   vmk_uint32 igu_entry, address;
   vmk_uint16 num_vf_queues;
   VMK_ReturnStatus status;
   int i=0;

   if(!adapter->vfdb->sriov.nr_virtfn){
      QFLE3_ERR("SRIOV: No of VFs is Zero\n");
      return -1;
   }

   if(!adapter->vfdb){
      QFLE3_ERR("SRIOV: Failed to configure SR-IOV since vfdb was not allocated.\n");
      return -1;
   }

   /*if(adapter->state != QFLE3_STATE_OPEN){                     //QFLE3_SRIOV_TODO
      QFLE3_ERR("SRIOV: VF num configuration via sysfs not supported while PF is down\n");
      return -1;
   }*/


   first_vf = adapter->vfdb->sriov.first_vf_in_pf;

   /* statically distribute vf sb pool between VFs */
   num_vf_queues = (vmk_uint16)min(QFLE3_VF_MAX_QUEUES,
                          adapter->vfdb->vf_sbs_pool / adapter->vfdb->sriov.nr_virtfn);

   /* zero previous values learned from igu cam */
   for (vf_idx = 0; vf_idx < adapter->vfdb->sriov.nr_virtfn; vf_idx++) {
      struct qfle3_virtf *vf = QFLE3_VF(adapter, vf_idx);

      vf->sb_count = 0;
      vf_sb_count(QFLE3_VF(adapter, vf_idx)) = 0;
   }
   adapter->vfdb->vf_sbs_pool = 0;

   /* prepare IGU cam */
   sb_idx = adapter->vfdb->first_vf_igu_entry;
   address = IGU_REG_MAPPING_MEMORY + sb_idx * IGU_ENTRY_SIZE;
   for (vf_idx = first_vf; vf_idx < (first_vf + adapter->vfdb->sriov.nr_virtfn); vf_idx++) {
      for (vfq_idx = 0; vfq_idx < num_vf_queues; vfq_idx++) {
         igu_entry = vf_idx << IGU_REG_MAPPING_MEMORY_FID_SHIFT |
            vfq_idx << IGU_REG_MAPPING_MEMORY_VECTOR_SHIFT |
            IGU_REG_MAPPING_MEMORY_VALID;
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: assigning sb %d to vf %d\n",
            sb_idx, vf_idx);
         REG_WR(adapter, address, igu_entry);
         sb_idx++;
         address += IGU_ENTRY_SIZE;
      }
   }

   /* reinitilize vf database according to igu cam */
   qfle3_get_vf_igu_cam_info(adapter);

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: vf_sbs_pool %d, num_vf_queues %d\n",
      QFLE3_VFDB(adapter)->vf_sbs_pool, num_vf_queues);

   qcount = 0;
   for(vf_idx=0; vf_idx < adapter->vfdb->sriov.nr_virtfn; vf_idx++) {
      struct qfle3_virtf *vf = QFLE3_VF(adapter, vf_idx);

      /* set local queue arrays */
      vf->vfqs = &adapter->vfdb->vfqs[qcount];
      qcount += vf_sb_count(vf);
      qfle3_iov_static_resc(adapter, vf);
   }

   /* prepare msix vectors in VF configuration space */
   for (vf_idx = first_vf; vf_idx < first_vf + adapter->vfdb->sriov.nr_virtfn; vf_idx++) {
      qfle3_pretend_func(adapter, HW_VF_HANDLE(adapter, vf_idx));
      REG_WR(adapter, PCICFG_OFFSET + GRC_CONFIG_REG_VF_MSIX_CONTROL,
             num_vf_queues - 1);
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: set msix vec num in VF %d cfg space to %d\n",
         vf_idx, num_vf_queues - 1);
   }
   qfle3_pretend_func(adapter, QFLE3_ABS_FUNC(adapter));

   /* enable sriov. This will probe all the VFs, and consequentially cause
    * the "acquire" messages to appear on the VF PF channel.
    */
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: about to call enable sriov\n");
   qfle3_disable_sriov(adapter);               
   status = vmk_PCIEnableVFs(adapter->pdev, &adapter->vfdb->sriov.nr_virtfn);
   if(status!=VMK_OK){
      QFLE3_ERR("SRIOV: Failed to enable SRIOV %d returned.\n", rc);
      adapter->num_vf_actual=adapter->vfdb->sriov.nr_virtfn;
      qfle3_iov_remove_one(adapter);
      return -1;
   }

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: VFs enabled successfully %d\n", adapter->vfdb->sriov.nr_virtfn);

   adapter->num_vf_actual = adapter->vfdb->sriov.nr_virtfn;
   for(i = 0; i < adapter->vfdb->sriov.nr_virtfn; i++) {
      vmk_PCIDevice tmp_vf;
      status = vmk_PCIGetVFPCIDevice(adapter->pdev, i, &tmp_vf);
      if (status != VMK_OK) {
         QFLE3_ERR("SRIOV: Unable to get device handle for VF.\n");
         qfle3_iov_remove_one(adapter);
         return -1;
       }
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Successfully got VF Device Info for vf %d\n", i);
      status = vmk_PCIQueryDeviceAddr(tmp_vf, &adapter->vfdb->vfs[i].vf_sbdf);

      if (status != VMK_OK) {
         QFLE3_ERR("SRIOV: Query VF pci device addr failed\n");
         qfle3_iov_remove_one(adapter);
         return -1;
      }
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Successfully got VF Device Address for vf %d\n", i);
   }
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Successfully got all Vf Info\n");

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: sriov enabled (%d vfs)\n", adapter->num_vf_actual);
   adapter->flags |= TX_SWITCHING;
   return 0;
}


int qfle3_vf_idx_by_abs_fid(struct qfle3_adapter *adapter, vmk_uint16 abs_vfid)
{
        int idx;

        for(idx=0; idx<((adapter)->vfdb->sriov.nr_virtfn); idx++)
                if (((adapter)->vfdb->vfs[idx].abs_vfid) == abs_vfid)
                        break;
        return idx;
}

/* must be called after PF bars are mapped */
static VMK_ReturnStatus
qfle3_iov_init_one(struct qfle3_adapter *adapter)
{

   int i;
   struct qfle3_sriov *iov;
   vmk_uint32 val;

   adapter->vfdb = NULL;

   if (CHIP_IS_E1x(adapter)) {
      QFLE3_ERR("SRIOV: Adapter model doesn't support SRIOV\n");
      return VMK_FAILURE;
   }

   if (adapter->intr.intrType != QFLE3_IT_MSIX){
      QFLE3_ERR("SRIOV: MSI/INTx mode is incompatible with SRIOV\n");
      return VMK_FAILURE;
   }

   if (QFLE3_L2_MAX_CID(adapter) >= QFLE3_FIRST_VF_CID) {
      QFLE3_ERR("SRIOV: cids %d are overspilling into vf space (starts at %d). Abort SRIOV\n",
         QFLE3_L2_MAX_CID(adapter), QFLE3_FIRST_VF_CID);
      return VMK_FAILURE;
   }

   /* verify igu is in normal mode */
   if (CHIP_INT_MODE_IS_BC(adapter)) {
      QFLE3_ERR("SRIOV: IGU not normal mode,  SRIOV can not be enabled\n");
      return VMK_FAILURE;
   }

   /* allocate the vfs database */
   adapter->vfdb = qfle3_heap_alloc(sizeof(struct qfle3_vfdb));
   if (!adapter->vfdb) {
      QFLE3_ERR("SRIOV: Failed to allocate vf database\n");
      goto failed;
   }
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: adapter->vfdb heap allocated \n");

   /* get the sriov info */
   iov = &(adapter->vfdb->sriov);
   iov->pos = adapter->sriov_pos;
   
   vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev, VMK_PCI_CONFIG_ACCESS_16, iov->pos + PCI_SRIOV_CTRL, &iov->ctrl);
   vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev, VMK_PCI_CONFIG_ACCESS_16, iov->pos + PCI_SRIOV_TOTAL_VF, &iov->total);
   vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev, VMK_PCI_CONFIG_ACCESS_16, iov->pos + PCI_SRIOV_INITIAL_VF, &iov->initial);
   vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev, VMK_PCI_CONFIG_ACCESS_16, iov->pos + PCI_SRIOV_VF_OFFSET, &iov->offset);
   vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev, VMK_PCI_CONFIG_ACCESS_16, iov->pos + PCI_SRIOV_VF_STRIDE, &iov->stride);
   vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev, VMK_PCI_CONFIG_ACCESS_16, iov->pos + PCI_SRIOV_SUP_PGSIZE, &iov->pgsz);
   vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev, VMK_PCI_CONFIG_ACCESS_16, iov->pos + PCI_SRIOV_CAP, &iov->cap);
   vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev, VMK_PCI_CONFIG_ACCESS_16, iov->pos + PCI_SRIOV_FUNC_LINK, &iov->link);

   /* get the number of SRIOV bars */
   iov->nres = 0;

   /* read the first_vfid */
   val=REG_RD(adapter, PCICFG_OFFSET + GRC_CONFIG_REG_PF_INIT_VF);
   iov->first_vf_in_pf = ((val & GRC_CR_PF_INIT_VF_PF_FIRST_VF_NUM_MASK)
                           * 8) - (QFLE3_MAX_NUM_OF_VFS * QFLE3_PATH(adapter));
   
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: IOV info[%d]: first vf %d, nres %d, cap 0x%x, ctrl 0x%x, total %d, initial %d, num vfs %d, offset %d, stride %d, page size 0x%x\n",
              QFLE3_FUNC(adapter), iov->first_vf_in_pf, iov->nres, iov->cap, iov->ctrl, iov->total, iov->initial, iov->nr_virtfn, iov->offset, iov->stride, iov->pgsz);

   /* SR-IOV capability was enabled but there are no VFs*/
   if (iov->total == 0){
      QFLE3_ERR("SRIOV: Total VFs are Zero\n");
      goto failed;
   }

   /* module param can limit the resources allocated to VFs*/
   iov->nr_virtfn = (vmk_uint16)min(iov->total, adapter->num_vf_req);
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: iov->nr_virtfn is %d and Num Of Vfs requested is %d\n",
      iov->nr_virtfn, adapter->num_vf_req);

   /*Allocate the VF Array*/
   adapter->vfdb->vfs = qfle3_heap_alloc(sizeof(struct qfle3_virtf) * iov->nr_virtfn);
   if(!adapter->vfdb->vfs) {
      QFLE3_ERR("SRIOV: Failed to allocate VF Array\n");
      goto failed;
   }
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: adapter->vfdb->vfs heap allocated \n");

   /* Initial VF init - index and abs_vfid - nr_virtfn must be set  */
   for (i=0; i< iov->nr_virtfn; i++) {
      adapter->vfdb->vfs[i].index = i;
      adapter->vfdb->vfs[i].abs_vfid = iov->first_vf_in_pf + i;
      adapter->vfdb->vfs[i].state = VF_FREE;
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
      vmk_SemaCreate(&adapter->vfdb->vfs[i].op_mutex, qfle3_mod_info.heapID, "op_mutex", 1);
#else
      vmk_SemaCreate(&adapter->vfdb->vfs[i].op_mutex, vmk_ModuleCurrentID, "op_mutex", 1);
#endif
      adapter->vfdb->vfs[i].op_current = CHANNEL_TLV_NONE;

   }

   /* re-read the IGU CAM for VFs - index and abs_vfid must be set */
   if(!qfle3_get_vf_igu_cam_info(adapter)) {
      QFLE3_ERR("SRIOV: No entries in IGU CAM for vfs\n");
      goto failed;
   }

   /* allocate the queue arrays for all VFs */
   adapter->vfdb->vfqs = qfle3_heap_alloc(QFLE3_MAX_NUM_VF_QUEUES * sizeof(struct qfle3_vf_queue));
   if(!adapter->vfdb->vfqs) {
      QFLE3_ERR("SRIOV: Failed to allocate vf queue array\n");
      goto failed;
   }
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: adapter->vfdb->vfqs heap allocated \n");
   /* Prepare the VFs event synchronization mechanism */
   
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
   vmk_SemaCreate(&adapter->vfdb->event_mutex, qfle3_mod_info.heapID, "event_mutex", 1);
   vmk_SemaCreate(&adapter->vfdb->bulletin_mutex, qfle3_mod_info.heapID, "bulletin_mutex", 1);
#else
   vmk_SemaCreate(&adapter->vfdb->event_mutex, vmk_ModuleCurrentID, "event_mutex", 1);
   vmk_SemaCreate(&adapter->vfdb->bulletin_mutex, vmk_ModuleCurrentID, "bulletin_mutex", 1);

#endif
   adapter->esx_vf = qfle3_heap_alloc(sizeof(struct qfle3_esx_vf) * iov->nr_virtfn);
   if (!adapter->esx_vf) {
      QFLE3_ERR("SRIOV: Failed to allocate SRIOV passthru vf database\n");
      goto failed;
   }
   
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: adapter->esx_vf heap allocated \n");

   for (i=0; i<iov->nr_virtfn; i++) {
     struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[i];
     int j=0;
     adapter_esx_vf->old_mtu = QFLE3_ESX_PASSTHRU_MTU_UNINITIALIZED;

     adapter_esx_vf->old_vf_fw_stats = qfle3_heap_alloc(QFLE3_ESX_VF_FW_TOTAL_STATS_SIZE);
     if(!adapter_esx_vf->old_vf_fw_stats) {
        QFLE3_ERR("SRIOV: Failed to allocate old_vf_fw_stats for VF %d\n", i);
        for(j=i-1; j>=0; j--){
            struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[j];
            qfle3_heap_free(adapter_esx_vf->old_vf_fw_stats);
         }
     goto failed;
     }

   }

   if (SHMEM2_HAS(adapter, sriov_switch_mode))
      SHMEM2_WR(adapter, sriov_switch_mode, SRIOV_SWITCH_MODE_VEB);

   return VMK_OK;

failed:
   if(adapter->esx_vf)
      qfle3_heap_free(adapter->esx_vf);
   if (adapter->vfdb) {
      qfle3_heap_free(adapter->vfdb->vfqs);
      qfle3_heap_free(adapter->vfdb->vfs);
      qfle3_heap_free(adapter->vfdb);
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: adapter->vfdb->vfqs, adapter->vfdb->vfs and adapter->vfdb heaps are freed\n");
   }
   adapter->vfdb = NULL;

   return VMK_FAILURE;

}


void qfle3_schedule_iov_task(struct qfle3_adapter *adapter, enum qfle3_iov_flag flag)
{
   int rc;
   if(adapter->bc_service_worldid == VMK_INVALID_WORLD_ID) {
      QFLE3_ERR("SRIOV: World not allocated yet.\n");
      return;
   }
   vmk_SpinlockLock(adapter->sriov_bc_lock);
   vmk_BitVectorSet(adapter->iov_task_state, flag);
   vmk_CPUMemFenceReadWrite();
   vmk_SpinlockUnlock(adapter->sriov_bc_lock);

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Waking the WorkQueue \n");
   rc = vmk_WorldForceWakeup(adapter->bc_service_worldid);

   if(rc!=VMK_OK) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Unable to Wakeup kernel world for backchannel %d for IOV Flag %d\n", rc, flag);
   }
   return;
}

static VMK_ReturnStatus qfle3_sriov_task(void *data)
{
   struct qfle3_adapter *adapter = (struct qfle3_adapter *)data;
   vmk_uint8 flag=1;
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: About to execute Back Channel request\n");

   while(flag) {

      vmk_SpinlockLock(adapter->sriov_bc_lock);

      if (vmk_BitVectorAtomicTestAndClear(adapter->iov_task_state,QFLE3_IOV_HANDLE_FLR)){
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: About to execute VF FLR \n");
         vmk_SpinlockUnlock(adapter->sriov_bc_lock);
         qfle3_vf_handle_flr_event(adapter);
      } 

     else if (vmk_BitVectorAtomicTestAndClear(adapter->iov_task_state,QFLE3_IOV_HANDLE_VF_MSG)){
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: About to execute VF MBX\n");
         vmk_SpinlockUnlock(adapter->sriov_bc_lock);
         qfle3_vf_mbx(adapter);
      }
     
      else {
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: No Back Channel Processing Flag Set\n");
         vmk_SpinlockUnlock(adapter->sriov_bc_lock);
      }

      vmk_SpinlockLock(adapter->sriov_bc_lock);
      if (vmk_BitVectorIsZero(adapter->iov_task_state)) {
         vmk_SpinlockUnlock(adapter->sriov_bc_lock);
      
         if ( vmk_WorldWait(VMK_EVENT_NONE, VMK_LOCK_INVALID,   
                     VMK_TIMEOUT_UNLIMITED_MS,"qfle3 backchannel thread sleeping") == VMK_BAD_PARAM) {
            QFLE3_ERR("SRIOV: BC: Failed to sleep backchannel world \n");
            return VMK_BAD_PARAM;
            } 
         } else {
         vmk_SpinlockUnlock(adapter->sriov_bc_lock);
      }
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Backchannel word sleeping\n");
      if (adapter->flush_req==1){
         flag=0;
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Descheduled Backchannel World\n");
      }
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Backchannel word processing Done\n");
   }
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Backchannel word flushed\n");
   return VMK_OK;
}

/* IOV global initialization routines  */
void qfle3_iov_init_dq(struct qfle3_adapter *adapter)
{
   if(!adapter->vfdb){
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Adapter vfdb not allocated yet\n");
      return;
   }

   /* Set the DQ such that the CID reflect the abs_vfid */
   REG_WR(adapter, DORQ_REG_VF_NORM_VF_BASE, 0);
   REG_WR(adapter, DORQ_REG_MAX_RVFID_SIZE, ILOG2(QFLE3_MAX_NUM_OF_VFS));

   /* Set VFs starting CID. If its > 0 the preceding CIDs are belong to
    * the PF L2 queues
    */
   REG_WR(adapter, DORQ_REG_VF_NORM_CID_BASE, QFLE3_FIRST_VF_CID);

   /* The VF window size is the log2 of the max number of CIDs per VF */
   REG_WR(adapter, DORQ_REG_VF_NORM_CID_WND_SIZE, QFLE3_VF_CID_WND);

   /* The VF doorbell size  0 - *B, 4 - 128B. We set it here to match
   * the Pf doorbell size although the 2 are independent.
   */
   REG_WR(adapter, DORQ_REG_VF_NORM_CID_OFST, 3);

   /* No security checks for now -
   * configure single rule (out of 16) mask = 0x1, value = 0x0,
   * CID range 0 - 0x1ffff
   */
   REG_WR(adapter, DORQ_REG_VF_TYPE_MASK_0, 1);
   REG_WR(adapter, DORQ_REG_VF_TYPE_VALUE_0, 0);
   REG_WR(adapter, DORQ_REG_VF_TYPE_MIN_MCID_0, 0);
   REG_WR(adapter, DORQ_REG_VF_TYPE_MAX_MCID_0, 0x1ffff);

   /* set the VF doorbell threshold. This threshold represents the amount
   * of doorbells allowed in the main DORQ fifo for a specific VF.
   */
   REG_WR(adapter, DORQ_REG_VF_USAGE_CT_LIMIT, 64);

}

void qfle3_vf_mbx_schedule(struct qfle3_adapter *adapter,
			   struct vf_pf_event_data *vfpf_event)
{
   vmk_uint8 vf_idx;
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: vf pf event received: vfid %d, address_hi %x, address lo %x",
      vfpf_event->vf_id, vfpf_event->msg_addr_hi, vfpf_event->msg_addr_lo);

   /* Sanity checks consider removing later */
   /* check if the vf_id is valid */
   if ((vfpf_event->vf_id - adapter->vfdb->sriov.first_vf_in_pf) > adapter->vfdb->sriov.nr_virtfn) {
      QFLE3_ERR("SRIOV: BC: Illegal vf_id %d max allowed: %d\n", 
                 vfpf_event->vf_id, adapter->vfdb->sriov.nr_virtfn);
      return;
   }

   vf_idx = qfle3_vf_idx_by_abs_fid(adapter, vfpf_event->vf_id);

   /* Update VFDB with current message and schedule its handling */
   vmk_SemaLock(&adapter->vfdb->event_mutex);
   adapter->vfdb->mbxs[vf_idx].vf_addr_hi = vfpf_event->msg_addr_hi;
   adapter->vfdb->mbxs[vf_idx].vf_addr_lo = vfpf_event->msg_addr_lo;
   adapter->vfdb->event_occur |= (1ULL << vf_idx);
   vmk_SemaUnlock(&adapter->vfdb->event_mutex);
   
      qfle3_schedule_iov_task(adapter, QFLE3_IOV_HANDLE_VF_MSG);
}

int qfle3_iov_alloc_mem(struct qfle3_adapter *adapter)
{
   vmk_ByteCount tot_size;
   int i, rc=0;

   if(!adapter->vfdb){
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Adapter vfdb not allocated\n");
      return rc;
   }

   /* allocate vfs hw contexts */
   tot_size = ((adapter->vfdb->sriov.first_vf_in_pf) + (adapter->vfdb->sriov.nr_virtfn))*
      QFLE3_CIDS_PER_VF * sizeof(union cdu_context);

   for (i=0; i<QFLE3_VF_CIDS/ILT_PAGE_CIDS; i++) {
      struct hw_dma *cxt = &(adapter->vfdb->context[i]);
      cxt->size = (vmk_ByteCount)min(tot_size, CDU_ILT_PAGE_SZ);
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: i is %d\n", i);
      if(cxt->size) {
         cxt->addr = qfle3_alloc_dma_mapping(adapter,
                                             adapter->dmaEngineCoherent,
                                             cxt->size,
					     &cxt->mapping);
         if(!cxt->addr)
            goto alloc_mem_err;

      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: cxt->mapping allocated successfuly cxt->addr is %p mapping is 0x%lx size id %d\n", cxt->addr, cxt->mapping, (vmk_uint32)cxt->size);
      } else {
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: setting cxt->addr to NULL\n");
         cxt->addr = NULL;
         cxt->mapping = 0;
      }
      tot_size -= cxt->size;
   }
   
   /* allocate vfs ramrods dma memory - client_init and set_mac */
   tot_size = (adapter->vfdb->sriov.nr_virtfn) * sizeof(struct qfle3_vf_sp);;
   adapter->vfdb->sp_dma.addr = qfle3_alloc_dma_mapping(adapter,
                                                        adapter->dmaEngineCoherent,
                                                        tot_size,
                                                        &(adapter->vfdb->sp_dma.mapping));
   if(!(adapter->vfdb->sp_dma.addr))
      goto alloc_mem_err;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: adapter->vfdb->sp_dma.mapping allocated successfuly address is %p mapping is 0x%lx size id %d\n",
					adapter->vfdb->sp_dma.addr, adapter->vfdb->sp_dma.mapping, (vmk_uint32)tot_size);
   adapter->vfdb->sp_dma.size = tot_size;

   /* allocate mailboxes */
   tot_size = (adapter->vfdb->sriov.nr_virtfn) * MBX_MSG_ALIGNED_SIZE;
   adapter->vfdb->mbx_dma.addr = qfle3_alloc_dma_mapping(adapter,
                                                         adapter->dmaEngineCoherent,
                                                         tot_size,
                                                         &(adapter->vfdb->mbx_dma.mapping));
   if(!(adapter->vfdb->mbx_dma.addr))
      goto alloc_mem_err;
   
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: adapter->vfdb->mbx_dma.mapping allocated successfuly address is %p mapping is 0x%lx size id %d\n",
					adapter->vfdb->mbx_dma.addr, adapter->vfdb->mbx_dma.mapping, (vmk_uint32)tot_size); 
   adapter->vfdb->mbx_dma.size = tot_size;

   /* allocate local bulletin boards */
   tot_size = (adapter->vfdb->sriov.nr_virtfn) * BULLETIN_CONTENT_SIZE;
   adapter->vfdb->bulletin_dma.addr = qfle3_alloc_dma_mapping(adapter,
                                                              adapter->dmaEngineCoherent,
                                                              tot_size,
                                                              &(adapter->vfdb->bulletin_dma.mapping));
   if(!adapter->vfdb->bulletin_dma.addr)
      goto alloc_mem_err;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: adapter->vfdb->bulletin_dma.mapping allocated successfuly address is %p mapping is 0x%lx size id %d\n",
					adapter->vfdb->bulletin_dma.addr, adapter->vfdb->bulletin_dma.mapping, (vmk_uint32)tot_size);
   adapter->vfdb->bulletin_dma.size = tot_size;

   for (i = 0; i< adapter->vfdb->sriov.nr_virtfn; i++) {
      struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[i];

      adapter_esx_vf->vf_fw_stats = qfle3_alloc_dma_mapping(adapter,
                                                            adapter->dmaEngineCoherent,
                                                            QFLE3_ESX_VF_FW_TOTAL_STATS_SIZE,
                                                            &(adapter_esx_vf->vf_fw_stats_mapping));
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: adapter_esx_vf->vf_fw_stats allocated successfully address is %p mapping is 0x%lx size id %ld for vfid %d\n",
			adapter_esx_vf->vf_fw_stats, adapter_esx_vf->vf_fw_stats_mapping, QFLE3_ESX_VF_FW_TOTAL_STATS_SIZE, i);
   }
   
   return 0;

alloc_mem_err:
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Error while memory mapping \n");
   return -1;
}

void qfle3_iov_free_mem(struct qfle3_adapter *adapter)
{
   int i;
   struct qfle3_sriov *iov;

   if(!adapter->vfdb){
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Adapter vfdb not allocated\n");
      return;
   }
   iov = &(adapter->vfdb->sriov);
   for (i = 0; i < iov->nr_virtfn; i++) {
      struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[i];

      if (adapter_esx_vf->vf_fw_stats){
         qfle3_free_dma_mapping(adapter, adapter->dmaEngineCoherent,
                               adapter_esx_vf->vf_fw_stats,
                               adapter_esx_vf->vf_fw_stats_mapping,
                               QFLE3_ESX_VF_FW_TOTAL_STATS_SIZE);
         adapter_esx_vf->vf_fw_stats = NULL;
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Adapter memory for VF statistics freed successfully for vfid %d\n", i);
      }
   }

   /* free vfs hw contexts */
   for (i = 0; i < QFLE3_VF_CIDS/ILT_PAGE_CIDS; i++) {
      struct hw_dma *cxt = &adapter->vfdb->context[i];

      if (cxt->addr){
         qfle3_free_dma_mapping(adapter, adapter->dmaEngineCoherent,
                               cxt->addr, cxt->mapping, cxt->size);
         cxt->addr = NULL;
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: cxt->mapping freed successfuly\n");
      }
   }

   if (adapter->vfdb->sp_dma.addr) {
      qfle3_free_dma_mapping(adapter, adapter->dmaEngineCoherent,
                              adapter->vfdb->sp_dma.addr, adapter->vfdb->sp_dma.mapping, adapter->vfdb->sp_dma.size);
      adapter->vfdb->sp_dma.addr = NULL;
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: adapter->vfdb->sp_dma.mapping freed successfuly\n");
   }

   if (adapter->vfdb->mbx_dma.addr) {
      qfle3_free_dma_mapping(adapter, adapter->dmaEngineCoherent,
                              adapter->vfdb->mbx_dma.addr, adapter->vfdb->mbx_dma.mapping, adapter->vfdb->mbx_dma.size);
      adapter->vfdb->mbx_dma.addr = NULL;
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: adapter->vfdb->mbx_dma.mapping freed successfuly\n");
   }
   if (adapter->vfdb->bulletin_dma.addr) {
      qfle3_free_dma_mapping(adapter, adapter->dmaEngineCoherent,
                               adapter->vfdb->bulletin_dma.addr, adapter->vfdb->bulletin_dma.mapping, adapter->vfdb->bulletin_dma.size);
      adapter->vfdb->bulletin_dma.addr = NULL;
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: adapter->vfdb->bulletin_dma.mapping freed successfuly\n");
   }                            
}


/* Must be called only after VF-Enable*/
/*inline int
qfle3_vf_bus(struct qfle3_adapter *adapter, int vfid)
{
   vmk_uint32 sbdf;
   struct pci_dev *vf;

   vf = vmklnx_get_vf(adapter->pdev, vfid, &sbdf);
   if (vf == NULL)
      return -1;

   return vf->bus->number;
}
*/
/* Must be called only after VF-Enable*/
/*inline int
qfle3_vf_devfn(struct qfle3_adapter *adapter, int vfid)
{
   vmk_uint32 sbdf;
   struct pci_dev *vf;

   vf = vmklnx_get_vf(adapter->pdev, vfid, &sbdf);
   if (vf == NULL)
      return -1;

   return vf->devfn;
}

*/
/* called by qfle3_nic_load */
int qfle3_iov_nic_init(struct qfle3_adapter *adapter)
{
   int vfid;

   if (!adapter->vfdb) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Adapter vfdb not allocated\n");
      return 0;
   }

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: num of vfs: %d\n", adapter->vfdb->sriov.nr_virtfn);

   /* let FLR complete ... */
   vmk_WorldSleep(100);

   /* initialize vf database */
   for(vfid=0; vfid < adapter->vfdb->sriov.nr_virtfn; vfid++) {
      struct qfle3_virtf *vf = QFLE3_VF(adapter, vfid);

      int base_vf_cid = (adapter->vfdb->sriov.first_vf_in_pf + vfid) * QFLE3_CIDS_PER_VF;
      
      union cdu_context *base_cxt = (union cdu_context *)
         QFLE3_VF_CXT_PAGE(adapter, base_vf_cid/ILT_PAGE_CIDS)->addr + (base_vf_cid & (ILT_PAGE_CIDS-1));

      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: VF[%d] Max IGU SBs: %d, base vf cid 0x%x, base cid 0x%x, base cxt %p\n",
         vf->abs_vfid, vf_sb_count(vf), base_vf_cid, QFLE3_FIRST_VF_CID + base_vf_cid, base_cxt);

      /* init statically provisioned resources */
      qfle3_iov_static_resc(adapter, vf);

      /* queues are initialized during VF-ACQUIRE */
      vf->filter_state = 0;

      ecore_init_credit_pool(&vf->vf_vlans_pool, 0, vf_vlan_rules_cnt(vf));
      ecore_init_credit_pool(&vf->vf_macs_pool, 0, vf_mac_rules_cnt(vf));

      /*  init mcast object - This object will be re-initialized
       *  during VF-ACQUIRE with the proper cl_id and cid.
       *  It needs to be initialized here so that it can be safely
       *  handled by a subsequent FLR flow.
       */
      vf->mcast_list_len = 0;
      ecore_init_mcast_obj(adapter, &vf->mcast_obj, 0xFF,
                 0xFF, 0xFF, 0xFF, qfle3_vf_sp(adapter, vf, mcast_rdata),
                 qfle3_vf_sp_map(adapter, vf, mcast_rdata), ECORE_FILTER_MCAST_PENDING,
                 &vf->filter_state,  ECORE_OBJ_TYPE_RX_TX);

      /* set the mailbox message addresses */
      QFLE3_VF_MBX(adapter, vfid)->msg = (struct qfle3_vf_mbx_msg *)
         (((vmk_uint8 *)QFLE3_VF_MBX_DMA(adapter)->addr) + vfid * MBX_MSG_ALIGNED_SIZE);

      QFLE3_VF_MBX(adapter, vfid)->msg_mapping = QFLE3_VF_MBX_DMA(adapter)->mapping +
         vfid * MBX_MSG_ALIGNED_SIZE;

      /* Enable vf mailbox */
      qfle3_vf_enable_mbx(adapter, vf->abs_vfid);
   }

   return 0;
}

VMK_ReturnStatus qfle3_vf_unregister(vmk_PCIDevice vf)
{
   vmk_PCIUnregisterVF(vf);
   return VMK_OK;
}

#endif  //QFLE3_SRIOV

static VMK_ReturnStatus
qfle3_attach_dev(vmk_Device device)
{
   VMK_ReturnStatus status;
   struct qfle3_adapter *adapter = NULL;
   //char dump_file_name[64] = {0};
   QFLE3_SMCMD_STATUS cmd_status;
   //extern vmk_uint32 mp_co_rxusecs;
   //extern vmk_uint32 mp_co_txusecs;
   extern vmk_uint32 rx_bd_rings;
   extern vmk_uint32 tx_bd_rings;
   extern vmk_uint32 disable_cnic;
   extern vmk_uint32 tx_to_delay;
   
#ifdef QFLE3_SRIOV
   vmk_WorldProps world_props;
   int rc;
   int pos = VMK_PCI_CFG_SPACE_SIZE;
   vmk_uint32 header;
   int ttl;
#if (ESX_DDK_VERSION >= 2017)
   vmk_uint32 tmp;
#else
   extern vmk_uint32 max_vfs[QFLE3_MAX_NUM_OF_PFS];
#endif //ESX_DDK_VERSION
#endif //QFLE3_SRIOV
   vmk_AddrCookie data;

   adapter = qfle3_heap_alloc(sizeof(*adapter));
   if (adapter == NULL) {
      vmk_LogMessage("Failed to allocate qfle3 adapter object");
      status = VMK_NO_MEMORY;
      goto fail_adapter_alloc;
   }
#ifdef QFLE3_CNIC
   adapter->cnicEthDev = NULL;
   adapter->cnicEthDev = qfle3_heap_alloc(sizeof(struct cnic_eth_dev));
   
   if (adapter->cnicEthDev == NULL) {
      QFLE3_ERR("Failed to allocate qfle3 adapter object");
      status = VMK_NO_MEMORY;
      goto fail_cnic_alloc;
   }
#endif
   adapter->device = device;
   adapter->debug_mask = qfle3_debugMask;

   QFLE3_DBG(QFLE3_DBG_KERNEL, "DriverOps.attach: debugMask %u, now %u\n",qfle3_debugMask, adapter->debug_mask );

   

   adapter->lb_pkt = NULL;
   adapter->reboot_in_progress = 0;
   if (enable_tpa > 1 || enable_tpa < 0) {
	   QFLE3_INFO("Module param enable_lro(%u) out of range, Using default value 1.\n", enable_tpa);
	   enable_tpa = 1;
   }

   if(enable_tpa)
      adapter->lro_enable = VMK_TRUE;

   adapter->dcbx_sb.enabled = enable_dcbx;
   
   qfle3_get_pci_device_info(adapter);
   QFLE3_DBG(QFLE3_DBG_LOAD, "DriverOps.attach, %s\n", adapter->pdev_name.string);
#ifdef QFLE3_CNIC
   if (disable_cnic) {
      adapter->cnic_support = 0;
      
      QFLE3_INFO("Diabling CNIC Support\n");
   } else
   adapter->cnic_support = 1;
#else
   adapter->cnic_support = 0;
   QFLE3_INFO("Diabling CNIC Support\n");

#endif
   status = qfle3_is_device_supported(adapter);
   if (status != VMK_OK) {
      goto fail_unsupported_dev;
   }
   adapter->max_cos = QFLE3_MULTI_TX_COS;

   status = qfle3_alloc_bit_vectors(adapter);
   if (status != VMK_OK) {
      goto fail_bv;
   }
   vmk_BitVectorSet(adapter->state, QFLE3_STATE_BIT_QUIESCED);

   status = qfle3_map_io_resource(adapter);
   if (status != VMK_OK) {
      goto fail_io_map;
   }

   status = qfle3_enable_bus_master(adapter);
   if (status != VMK_OK) {
      goto fail_bus_master;
   }

      /*
    * Disable indirect addressing mode
    */
   status = vmk_PCIWriteConfig(vmk_ModuleCurrentID, adapter->pdev,
                               VMK_PCI_CONFIG_ACCESS_32, PCICFG_GRC_ADDRESS, 0);
   if (status != VMK_OK) {
      QFLE3_ERR("Failed to write PCI config. "
                "offset: %u, data: %u (%x)", PCICFG_GRC_ADDRESS, 0, status);
      goto fail_addr_mode;
   }

   REG_WR(adapter, PXP2_REG_PGL_ADDR_88_F0, 0);
   REG_WR(adapter, PXP2_REG_PGL_ADDR_8C_F0, 0);
   REG_WR(adapter, PXP2_REG_PGL_ADDR_90_F0, 0);
   REG_WR(adapter, PXP2_REG_PGL_ADDR_94_F0, 0);

   REG_WR(adapter, PGLUE_B_REG_INTERNAL_PFID_ENABLE_TARGET_READ, 1);

   status = qfle3_get_hw_info(adapter);
   if (status != VMK_OK)
      goto fail_hw_info;

   if (!CNIC_SUPPORT(adapter)){
      adapter->flags |= NO_ISCSI_FLAG;
      adapter->flags |= NO_ISCSI_OOO_FLAG;
      adapter->flags |= NO_FCOE_FLAG;
   }
   if (NO_FCOE(adapter) && NO_ISCSI(adapter)){ // no offloaded storage supported by the card
      // enable_fcoe_queue will take effect
      if (enable_fcoe_queue) {
         adapter->sw_fcoe = 1;
         QFLE3_INFO ("Software FCoE capability is enabled\n");
      }else{
         adapter->sw_fcoe = 0;
      }
   }else {
      QFLE3_INFO ("Card supports offload storage. Module parameter enable_fcoe_queue is disabled for this function \n");
      adapter->sw_fcoe = 0;
   }

//   adapter->dropless_fc = dropless_fc | qfle3_get_dropless_info(adapter);
   adapter->dropless_fc = dropless_fc | adapter->sw_fcoe | qfle3_get_dropless_info(adapter);
   if (adapter->dropless_fc)
      QFLE3_INFO("Dropless Flow Control Enabled");
   
   status = qfle3_alloc_interrupts(adapter);
   if (status != VMK_OK) {
      goto fail_intr_alloc;
   }

   /*
    * setup memory for comunicating with hardware
    */
   status = qfle3_create_dma_engines(adapter);
   if (status != VMK_OK) {
      goto fail_dma_engine;
   }

#ifdef QFLE3_SRIOV
   status = vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev, VMK_PCI_CONFIG_ACCESS_32,
                                pos, &header);
   if ((status == VMK_OK) && header){
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Value of pos is %d \n", pos);
      ttl = ((VMK_PCI_CFG_SPACE_EXP_SIZE - VMK_PCI_CFG_SPACE_SIZE) / 8);
      while (ttl-- > 0) {
      if ((header & 0x0000ffff) == PCI_EXT_CAP_ID_SRIOV) {
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Value of pos is %d \n", pos);
         break;
      }
      pos = ((header >> 20) & 0xffc);
      if (pos < VMK_PCI_CFG_SPACE_SIZE){
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: pos is %d which is less than VMK_PCI_CFG_SPACE_SIZE %d\n", pos, VMK_PCI_CFG_SPACE_SIZE);
         pos=0;
         break;
      }
      if((vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev, VMK_PCI_CONFIG_ACCESS_32,
         pos, &header)) == VMK_FAILURE) {
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Error while reading PCIConfig to get SRIOV pos\n");
         pos=0;
         break;
      }
    }
      if (!pos) {
         QFLE3_ERR("SRIOV: No PCIe support for SRIOV\n");
      }else {
        adapter->sriov_pos = pos;
      }
   }

#if (ESX_DDK_VERSION >= 2017)
   status = vmk_DeviceGetParamMaxVfs(device, &tmp);
   if (status == VMK_OK && tmp>0) {
      adapter->num_vf_req = (vmk_uint16)tmp;
   }
#else      
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: pf_num is %d, relative_pf_num is %d and max_vfs[relative_pf_num] is %d \n",
                                    adapter->pf_num, relative_pf_num, max_vfs[relative_pf_num]);
   if ((relative_pf_num < QFLE3_MAX_NUM_OF_PFS) && (max_vfs[relative_pf_num]>0) && (max_vfs[relative_pf_num]<= QFLE3_MAX_NUM_OF_VFS)) {
      adapter->num_vf_req = max_vfs[relative_pf_num];
   }
#endif //ESX_DDK_VERSION

   if ((adapter->num_vf_req > 0) && adapter->sriov_pos) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Number of VFs requested are %d\n", adapter->num_vf_req);
      status = qfle3_iov_init_one(adapter);
      if (status != VMK_OK){
         QFLE3_ERR("SRIOV: Failed to get SRIOV Info \n");
      } else {
         rc=0;
            rc = qfle3_enable_sriov(adapter);  
            if(rc)
               QFLE3_ERR("SRIOV: Failed to enable SRIOV\n");
            else {

               adapter->iov_task_state = vmk_BitVectorAlloc(qfle3_mod_info.heapID, 32);
               qfle3_create_spinlock("sriov_bc_lock",
                                        QFLE3_NAME_TO_STRING(adapter->pdev_name),
                                        QFLE3_LOCK_RANK_LOW, &adapter->sriov_bc_lock);
               world_props.name = "bc_service_world";
               world_props.moduleID = vmk_ModuleCurrentID;
               world_props.startFunction = qfle3_sriov_task;
               world_props.data = (void *)adapter;
               world_props.schedClass = VMK_WORLD_SCHED_CLASS_DEFAULT;
               world_props.heapID = qfle3_mod_info.heapID;
               QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: About to create Kernel World for SRIOV\n");
               status = vmk_WorldCreate(&world_props,&adapter->bc_service_worldid);
               if (status != VMK_OK) {
                  QFLE3_ERR("SRIOV: Unable to create kernel world for backchannel. SRIOV Cannot be enabled\n");

                  qfle3_destroy_spinlock(adapter->sriov_bc_lock);
		  vmk_BitVectorFree(qfle3_mod_info.heapID, adapter->iov_task_state);
                  qfle3_iov_remove_one(adapter);
               } else {
                  QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Kernel World created successfully \n");
               }
            }
      }

   } else {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: No. of Vfs not specified or out of range.\n");
   }

   relative_pf_num++;

#endif //QFLE3_SRIOV
   status = qfle3_create_helper(adapter, &adapter->stats_collection_helper, "stats_collection_helper");
   if (status != VMK_OK)
      QFLE3_DBG(QFLE3_DBG_STATS, "Stats collection Helper creation Failure\n");

//   // grab the leadership before qfle3 drivers do
//   // need to reconsider if MF mode is enabled
//   if (!qfle3_trylock_leader_lock(adapter))
//      QFLE3_ERR( "SRIOV: Failed to acquire recovery leadership!!!  Recovery could fail!");
//   else
//      adapter->is_leader = 1;

   adapter->mrrs = -1;
   if (hw_vlan_en > 1 || hw_vlan_en < 0) {
	   QFLE3_INFO("Module param hw_vlan(%u) out of range, Using default value 1.\n", hw_vlan_en);
	   hw_vlan_en = 1;
   }
   adapter->hw_vlan_en = hw_vlan_en;

   if (qfle3_offload_en > 15 || qfle3_offload_en < 0) {
	   QFLE3_INFO("Module param offload_flags(%u) out of range, Using default value 15.\n", qfle3_offload_en);
	   qfle3_offload_en = 15;
   }
   adapter->offloadflags = qfle3_offload_en;

   adapter->accept_any_vlan = 1;
   adapter->tx_bd_num_pages = TX_BD_DEFAULT_NUM_PAGES;
   adapter->rx_bd_num_pages = RX_BD_DEFAULT_NUM_PAGES;
   if (!(tx_bd_rings & (tx_bd_rings-1)) && !(rx_bd_rings & (rx_bd_rings-1))) {
      if((tx_bd_rings  <= TX_BD_USABLE_PER_PAGE * TX_BD_MAX_PAGES) && (tx_bd_rings >= TX_BD_USABLE_PER_PAGE * TX_BD_MIN_PAGES)){
         adapter->tx_bd_num_pages = tx_bd_rings/TX_BD_USABLE_PER_PAGE;
      } else {
         QFLE3_INFO("Module param txring_bd_nr(%u) out of range. Using default value %lu.\n", tx_bd_rings,
                     TX_BD_DEFAULT_NUM_PAGES * TX_BD_USABLE_PER_PAGE);
         adapter->tx_bd_num_pages = TX_BD_DEFAULT_NUM_PAGES;
      }


      if ((rx_bd_rings >  RX_BD_USABLE_PER_PAGE * RX_BD_MAX_PAGES) || (rx_bd_rings < RX_BD_USABLE_PER_PAGE * RX_BD_MIN_PAGES)) {
         QFLE3_INFO("Module param rxring_bd_nr(%u) out of range. Using default value %lu.\n", rx_bd_rings,
            RX_BD_DEFAULT_NUM_PAGES * RX_BD_USABLE_PER_PAGE);
         adapter->rx_bd_num_pages = RX_BD_DEFAULT_NUM_PAGES;
      }else {
         adapter->rx_bd_num_pages = rx_bd_rings/RX_BD_USABLE_PER_PAGE;
      }
   } else {
      QFLE3_ERR("Module parameter for tx/rx BD ring is not power of 2 tx:%d, rx:%d, ignoring.\n", tx_bd_rings, rx_bd_rings);
   }
   
   adapter->tx_ring_size = adapter->tx_bd_num_pages * TX_BD_USABLE_PER_PAGE;
   QFLE3_INFO("Initializing TX BD ring size to %d\n", adapter->tx_ring_size);

   adapter->rx_ring_size = adapter->rx_bd_num_pages * RX_BD_USABLE_PER_PAGE;
   QFLE3_INFO("Initializing RX BD ring size to %d\n", adapter->rx_ring_size);

#if 0
   if (mp_co_rxusecs > QFLE3_MAX_COALESCE_TOUT)
      adapter->rx_ticks  = QFLE3_MAX_COALESCE_TOUT;
   else
      adapter->rx_ticks  = mp_co_rxusecs;

   if (mp_co_txusecs > QFLE3_MAX_COALESCE_TOUT)
      adapter->tx_ticks  = QFLE3_MAX_COALESCE_TOUT;
   else
      adapter->tx_ticks  = mp_co_txusecs;
#else
      adapter->rx_ticks  = (25 / QFLE3_BTR) * QFLE3_BTR;
      adapter->tx_ticks  = (50 / QFLE3_BTR) * QFLE3_BTR;
#endif
      if (enable_vxlan_filters > 1 || enable_vxlan_filters < 0) {
         QFLE3_INFO("Module param hw_vlan(%u) out of range, Using default value 1.\n", hw_vlan_en);
         enable_vxlan_filters = 0;
      }

      adapter->vxlan_filters_en = enable_vxlan_filters;

   if (tx_to_delay)
      adapter->tx_to_delay = tx_to_delay;

   adapter->fwdmp_flags = GRC_DMP_DISABLED;

   if (enable_live_grcdump & GRC_DMP_PARITY_ENABLED) {
      /* Enable parity error dump only on one PF 
       * under each engine.
       */
       adapter->fwdmp_flags |= GRC_DMP_FRESH_ENABLED;
      if (qfle3_is_grc_disabled(adapter)) {
         adapter->fwdmp_flags |= GRC_DMP_PARITY_ENABLED;
         QFLE3_INFO("GRC parity dump Enabled");
      }
   }

   if (enable_live_grcdump & GRC_DMP_ERROR_ENABLED)
      adapter->fwdmp_flags |= GRC_DMP_ERROR_ENABLED;

   if (enable_live_grcdump & GRC_DMP_TXTO_ENABLED) {
      adapter->fwdmp_flags |= GRC_DMP_TXTO_ENABLED;
      QFLE3_INFO("GRC Tx timeout dump Enabled");
   }

   if (enable_live_grcdump & GRC_DMP_STSTO_ENABLED) {
      adapter->fwdmp_flags |= GRC_DMP_STSTO_ENABLED;
      QFLE3_INFO("GRC Stats timeout dump Enabled");
   }
   QFLE3_INFO("GRC dump settings: 0x%x.\n", adapter->fwdmp_flags);
   

   if (QFLE3_IS_GENEVE_OFFLOAD_ENABLED(adapter)) {
      adapter->geneve_udp_dst_port = vmk_BE16ToCPU(vmk_GenevePortGet());
   }
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
   status = vmk_SemaCreate(&adapter->port.phy_mutex,
                           qfle3_mod_info.heapID, "phy_mutex", 1);
#else
   status = vmk_SemaCreate(&adapter->port.phy_mutex,
                           vmk_ModuleCurrentID, "phy_mutex", 1);
#endif
   if (status != VMK_OK) {
      vmk_Warning(qfle3_mod_info.logID, "Failed to PHY Mutex");
      goto fail_phy_mutex_create;
   }
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
   status = vmk_SemaCreate(&adapter->mfw_cmd_mutex,
                           qfle3_mod_info.heapID, "mfw_cmd_mutex", 1);
#else
   status = vmk_SemaCreate(&adapter->mfw_cmd_mutex,
                           vmk_ModuleCurrentID, "mfw_cmd_mutex", 1);
#endif
   if (status != VMK_OK) {
      vmk_Warning(qfle3_mod_info.logID, "Failed to create MFW CMD Mutex");
      goto fail_mfw_mutex_create;
   }
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
   status = vmk_SemaCreate(&adapter->drv_info_mutex,
                           qfle3_mod_info.heapID, "drv_info_mutex", 1);
#else
   status = vmk_SemaCreate(&adapter->drv_info_mutex,
                           vmk_ModuleCurrentID, "drv_info_mutex", 1);
#endif
   if (status != VMK_OK) {
      vmk_Warning(qfle3_mod_info.logID, "Failed to create drv info Mutex");
      goto fail_drv_info_mutex_create;
   }

#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
   status = vmk_SemaCreate(&adapter->cnic_mutex,
                           qfle3_mod_info.heapID, "cnic_mutex", 1);
#else
   status = vmk_SemaCreate(&adapter->cnic_mutex,
                           vmk_ModuleCurrentID, "cnic_mutex", 1);
#endif
   if (status != VMK_OK) {
      vmk_Warning(qfle3_mod_info.logID, "Failed to CNIC Mutex");
      goto fail_cnic_mutex_create;
   }

   qfle3_set_init_mode_flags(adapter);

   adapter->ilt = qfle3_heap_alloc(sizeof(struct ecore_ilt));
   if (!adapter->ilt) {
      status = VMK_NO_MEMORY;
      goto fail_ilt_alloc;
   }

/* create spinlock for statemachine */
   status = qfle3_create_spinlock("sm_lock",
                                  QFLE3_NAME_TO_STRING(adapter->pdev_name),
                                  QFLE3_LOCK_RANK_HIGH, &adapter->sm_lock);
   if (status != VMK_OK) {
      vmk_Warning(qfle3_mod_info.logID, "Failed to create SM lock");
      goto fail_sm_lock;
   }

   status = qfle3_create_spinlock("uplinkSharedDataLock",
                                  QFLE3_NAME_TO_STRING(adapter->pdev_name),
                                  QFLE3_LOCK_RANK_HIGH, &adapter->uplinkSharedDataLock);
   if (status != VMK_OK) {
      vmk_Warning(qfle3_mod_info.logID, "Failed to create uplinkSharedData lock");
      goto fail_uplinkSharedDataLock;
   }

   status = qfle3_create_spinlock("q_lock",
                                  QFLE3_NAME_TO_STRING(adapter->pdev_name),
                                  QFLE3_LOCK_RANK_LOW, &adapter->q_lock);
   if (status != VMK_OK) {
      vmk_Warning(qfle3_mod_info.logID, "Failed to create Q lock");
      goto fail_q_lock;
   }
   /* Initialize the stats feature of firmware */
   status = qfle3_create_spinlock("stats_lock",
			 QFLE3_NAME_TO_STRING(adapter->pdev_name),
			 QFLE3_LOCK_RANK_LOW, &adapter->stats_lock);
   if (status != VMK_OK) {
      goto fail_stats_lock;
   }

   status = qfle3_create_helper(adapter, &adapter->recovery_helper, "recovery_helper");
   if (status != VMK_OK)
      goto fail_recovery_helper_create;
   adapter->sp_rtnl_state = vmk_BitVectorAlloc(qfle3_mod_info.heapID, 16);
   if (!adapter->sp_rtnl_state) {
      QFLE3_ERR("Failed to allocate bit vectors");
      goto fail_rtnl_state;
   }

   vmk_BitVectorZap(adapter->sp_rtnl_state);
   
   adapter->recovery_status = vmk_BitVectorAlloc(qfle3_mod_info.heapID, 16);
   if (!adapter->recovery_status) {
      QFLE3_ERR("Failed to allocate bit vectors");
      goto fail_recovery_state;
   }

   vmk_BitVectorZap(adapter->recovery_status);

   status = qfle3_create_helper(adapter, 
                                 &adapter->dcbx_sb.dcbx_task, "dcbx_task");
   if (status != VMK_OK)
      goto fail_dcbx_helper_create;

//   qfle3_init_uplink_data(adapter);

   status = qfle3_init_shared_queue_info(adapter);
   if (status != VMK_OK) {
      QFLE3_ERR("DriverOps.start qfle3_init_shared_queue_info failed!");
      return VMK_FAILURE;
   }
//   qfle3_init_uplink_data(adapter);
//   qfle3_init_rss_tables(adapter);
   if (vmk_ListIsEmpty(&qfle3_mod_info.adapterList)) {
      vmk_uint32 val;
      // clear the Recovery load register
      
      qfle3_acquire_hw_lock(adapter, HW_LOCK_RESOURCE_RECOVERY_REG);
      val = REG_RD(adapter, QFLE3_RECOVERY_GLOB_REG);
      val = val & ~QFLE3_PATH0_LOAD_CNT_MASK;
      val = val & ~QFLE3_PATH1_LOAD_CNT_MASK;
      REG_WR(adapter, QFLE3_RECOVERY_GLOB_REG, val);
      qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_RECOVERY_REG);
   }
   qfle3_sm_change_state(adapter, QFLE3_SM_STANDBY);
/* call statemachine with command */
   cmd_status = qfle3_sm_cmd(adapter, QFLE3_SMCMD_BRINGUP, 0);
   if (cmd_status != QFLE3_SMCMD_STATUS_COMPLETED) {
      status = VMK_FAILURE;
      goto fail_load;
   }

   data.ptr = adapter;
   vmk_TimerSchedule(qfle3_mod_info.timer_queue,
                     &qfle3_timer,
                     data,
                     VMK_USEC_PER_SEC, 
                     VMK_TIMER_DEFAULT_TOLERANCE, 
                     VMK_TIMER_ATTR_PERIODIC, 
                     VMK_LOCKDOMAIN_INVALID, 
                     VMK_SPINLOCK_UNRANKED,
                     &adapter->periodic_timer);

   /* Register reboot handler needed for WoL configuration */
   if (qfle3_mod_info.reboot_handler == NULL) {
      status = vmk_RegisterRebootHandler(qfle3_reboot_handler, &qfle3_mod_info);
      if (status == VMK_OK) {
         QFLE3_INFO("Registering reboot handler\n");
         qfle3_mod_info.reboot_handler = qfle3_reboot_handler;
      } else {
         QFLE3_ERR("Failed to register reboot handler. "
                   "WoL configuration won't be performed.");
         /* Ignore the failure as it is non-fatal. */
      }
   }

   /* Init the firmware dump enabled for all adapters, disable later for panicPoll worker */
   adapter->fwdmp_enable = 1;
//   if (auto_recovery) {
//      adapter->fwdmp_flags |= GRC_DMP_PARITY_ENABLED;
//   }
/*
 * register to device layer
 */
   vmk_ListInsert(&adapter->adapterLink, vmk_ListAtRear(&qfle3_mod_info.adapterList));
   vmk_DeviceSetAttachedDriverData(device, adapter);

   QFLE3_INFO("Chip Revision: %s, FW Version: %d_%d_%d\n",
             CHIP_IS_E2(adapter) ? "everest2" : "everest3",
             BCM_5710_FW_MAJOR_VERSION,
             BCM_5710_FW_MINOR_VERSION,
             BCM_5710_FW_REVISION_VERSION);
   QFLE3_INFO("Driver version is: %s\n", QFLE3_DRIVER_VERSION);
   vmk_BitVectorSet(adapter->state, QFLE3_STATE_BIT_ATTACHED);
   QFLE3_DBG(QFLE3_DBG_KERNEL, "qfle3_attach_dev END, mailbox id %d",QFLE3_FW_MB_IDX(adapter));

   return VMK_OK;

  fail_load:
   qfle3_destroy_helper(adapter, adapter->dcbx_sb.dcbx_task);
   adapter->dcbx_sb.dcbx_task = 0;
  fail_dcbx_helper_create:
   vmk_BitVectorFree(qfle3_mod_info.heapID, adapter->recovery_status);
   adapter->recovery_status = NULL;
  fail_recovery_state:
   vmk_BitVectorFree(qfle3_mod_info.heapID, adapter->sp_rtnl_state);
   adapter->sp_rtnl_state = NULL;
  fail_rtnl_state:
   qfle3_destroy_helper (adapter, adapter->recovery_helper);

   adapter->recovery_helper = 0;
  fail_recovery_helper_create:
   qfle3_destroy_spinlock(adapter->stats_lock);
  fail_stats_lock:
     qfle3_destroy_spinlock(adapter->q_lock);
  fail_q_lock:
   qfle3_destroy_spinlock(adapter->uplinkSharedDataLock);
  fail_uplinkSharedDataLock:
   qfle3_destroy_spinlock(adapter->sm_lock);
  fail_sm_lock:
   qfle3_heap_free(adapter->ilt);
  fail_ilt_alloc:
     vmk_SemaDestroy(&adapter->cnic_mutex);
  fail_cnic_mutex_create:
   vmk_SemaDestroy(&adapter->drv_info_mutex);
  fail_drv_info_mutex_create:
   vmk_SemaDestroy(&adapter->mfw_cmd_mutex);
  fail_mfw_mutex_create:
   vmk_SemaDestroy(&adapter->port.phy_mutex);
  fail_phy_mutex_create:
   qfle3_destroy_dma_engines(adapter);
   qfle3_destroy_helper(adapter, adapter->stats_collection_helper);
  fail_dma_engine:
   qfle3_free_interrupts(adapter);
  fail_intr_alloc:
  fail_bus_master:
   qfle3_unmap_io_resource(adapter);
  fail_hw_info:
  fail_addr_mode:
  fail_bv:
  fail_io_map:
   qfle3_free_bit_vectors(adapter);
  fail_unsupported_dev:
#ifdef QFLE3_CNIC
   qfle3_heap_free(adapter->cnicEthDev);
  fail_cnic_alloc:
#endif
   qfle3_heap_free(adapter);
  fail_adapter_alloc:
   
   QFLE3_ERR( "qfle3_attach_dev Failed");
   return status;
}

static void
qfle3_nic_load_dcc(qfle3_adapter *adapter, int load_code)
{
   if (((load_code == FW_MSG_CODE_DRV_LOAD_COMMON) ||
         (load_code == FW_MSG_CODE_DRV_LOAD_COMMON_CHIP)) &&
         (adapter->hw_info.shmem2_base)) {
      if (SHMEM2_HAS(adapter, dcc_support))
              SHMEM2_WR(adapter, dcc_support,
                              (SHMEM_DCC_SUPPORT_DISABLE_ENABLE_PF_TLV |
                               SHMEM_DCC_SUPPORT_BANDWIDTH_ALLOCATION_TLV));
      #if 0
      if (SHMEM2_HAS(adapter, afex_driver_support))
              SHMEM2_WR(adapter, afex_driver_support,
                              SHMEM_AFEX_SUPPORTED_VERSION_ONE);
      #endif
   }

   /* Set AFEX default VLAN tag to an invalid value */
   //adapter->afex_def_vlan_tag = -1;
}

static vmk_uint32
qfle3_version_str_to_int(vmk_uint8 *version, vmk_Bool format)
{
   vmk_uint8 vals[4];
   int i = 0;

   if (format) {
      i = vmk_Sscanf(version, "1.%c%hhd.%hhd.%hhd",
                 &vals[0], &vals[1], &vals[2], &vals[3]);
      if (i > 0)
         vals[0] -= '0';
   } else {
      i = vmk_Sscanf(version, "%hhd.%hhd.%hhd.%hhd",
                 &vals[0], &vals[1], &vals[2], &vals[3]);
   }

   while (i < 4)
      vals[i++] = 0;

   return (vals[0] << 24) | (vals[1] << 16) | (vals[2] << 8) | vals[3];
}

void qfle3_update_mng_version(struct qfle3_adapter *adapter)
{
   u32 iscsiver = DRV_VER_NOT_LOADED;
   u32 fcoever = DRV_VER_NOT_LOADED;
   u32 ethver = DRV_VER_NOT_LOADED;
   int idx = QFLE3_FW_MB_IDX(adapter);
   u8 *version;

   if (!SHMEM2_HAS(adapter, func_os_drv_ver))
      return;

   vmk_SemaLock(&adapter->drv_info_mutex); 
    /* Must not proceed when `qfle3_handle_drv_info_req' is feasible */
   if (adapter->drv_info_mng_owner)
      goto out;

  /* if (adapter->state != QFLE3_STATE_OPEN)
        goto out;
  */
    /* Parse ethernet driver version */
   ethver = qfle3_version_str_to_int(QFLE3_DRIVER_VERSION, VMK_FALSE);
   if (!CNIC_LOADED(adapter))
      goto out;

   /* Try getting storage driver version via cnic */
   vmk_Memset(&adapter->sp->drv_info_to_mcp, 0,
           sizeof(union drv_info_to_mcp));
   qfle3_drv_info_iscsi_stat(adapter);
   version = adapter->sp->drv_info_to_mcp.iscsi_stat.version;
   iscsiver = qfle3_version_str_to_int(version, VMK_FALSE);

   vmk_Memset(&adapter->sp->drv_info_to_mcp, 0,
           sizeof(union drv_info_to_mcp));
   qfle3_drv_info_fcoe_stat(adapter);
   version = adapter->sp->drv_info_to_mcp.fcoe_stat.version;
   fcoever = qfle3_version_str_to_int(version, VMK_FALSE);

out:
   SHMEM2_WR(adapter, func_os_drv_ver[idx].versions[DRV_PERS_ETHERNET], ethver);
   SHMEM2_WR(adapter, func_os_drv_ver[idx].versions[DRV_PERS_ISCSI], iscsiver);
   SHMEM2_WR(adapter, func_os_drv_ver[idx].versions[DRV_PERS_FCOE], fcoever);

   vmk_SemaUnlock(&adapter->drv_info_mutex);

   QFLE3_INFO("Setting driver version: ETH [%08x] iSCSI [%08x] FCoE [%08x]\n",
       ethver, iscsiver, fcoever);
}
static void qfle3_detect_tx_hang(struct qfle3_adapter *adapter)
{
	u8 cos;
	struct qfle3_fp_txdata *txdata;
   
   vmk_int32 i, qid;
   struct qfle3_fastpath *fp;
   vmk_Bool adapter_stuck = VMK_FALSE;
   int donot_recover_flag = 0;
   if (vmk_BitVectorTest(adapter->sp_rtnl_state, QFLE3_SP_RTNL_DONOT_RECOVER) || 
      vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)){      
      QFLE3_ERR("Hold off on recovery\n");
      donot_recover_flag = 1;
   }
   FOREACH_TX_ETH_QUEUE(i,qid,fp)

		for_each_cos_in_tx_queue(fp, cos) {
			txdata = fp->txdata_ptr[cos];
         if (donot_recover_flag) {
            txdata->queue_stuck = 0;
         }
         if (adapter->error_status || adapter_stuck) {
            // reset the queue_stuck counter when we are in error condition
            
				QFLE3_ERR("Recovery in progress, reset queue_stuck for q %d.%d\n", qid,cos);
            txdata->queue_stuck = 0;
            continue;
         }
			if (txdata->tx_pkt_cons == txdata->prev_tx_pkt_cons &&
			    txdata->tx_pkt_prod != txdata->tx_pkt_cons) {

				if (txdata->queue_stuck < adapter->tx_to_delay) {
					txdata->queue_stuck++;
					continue;
				}

				QFLE3_ERR("Tx Queue %d.%d stuck. tx_pkt_cons: %d, tx_pkt_prod: %d\n",
					  i, cos, txdata->tx_pkt_cons,
					  txdata->tx_pkt_prod);
            
            if (adapter_stuck  == VMK_FALSE){
               
               donot_recover_flag = vmk_BitVectorAtomicTestAndSet(adapter->sp_rtnl_state, QFLE3_SP_RTNL_DONOT_RECOVER);
               if (donot_recover_flag) {
                  QFLE3_ERR("Driver is not able to recover from Tx Timeout at this point");
                  
                  QFLE3_TRIGGER_TX_TO(adapter, QFLE3_ERR_TXQ_STUCK);
                  adapter_stuck = VMK_TRUE;
               }
            }
            txdata->queue_stuck = 0;
				continue;
			}
			txdata->prev_tx_pkt_cons = txdata->tx_pkt_cons;
			txdata->queue_stuck = 0;
		}
	}
}
void qfle3_tx_disable(struct qfle3_adapter * adapter)
{
//   vmk_int32 i, qid;
//   struct qfle3_fastpath *fp;
//   FOREACH_TX_ETH_QUEUE(i,qid,fp)
//      qfle3_tq_pause(adapter, qid);
//   }
   qfle3_link_down(adapter);
   qfle3_int_disable(adapter);
}

void qfle3_timer(vmk_TimerCookie data)
{
   struct qfle3_adapter *adapter = data.ptr;

//   if (vmk_BitVectorTest(adapter->sp_rtnl_state, QFLE3_SP_RTNL_DETECT_TX_TIMEOUT)) {
      if ((!vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)) && 
           (adapter->recovery_state == QFLE3_RECOVERY_DONE)) {
         
         QFLE3_DBG(QFLE3_DBG_TX, "Timer fired\n");
      // do not check tx hang if the adapter is in recovery mode or reset mode
      	if (!adapter->error_status && adapter->tx_to_delay)
      		qfle3_detect_tx_hang(adapter);
      }
//   }
   
   if (vmk_BitVectorTest(adapter->sp_rtnl_state, QFLE3_SP_RTNL_DONOT_RECOVER)) {
      vmk_TimerCycles timeNow = vmk_GetTimerCycles();
      vmk_TimerRelCycles cyclesRelative = timeNow - adapter->last_recovery_start;
      if (vmk_TimerTCToMS(cyclesRelative) >= 60000) {
         adapter->last_recovery_start = 0;
         
         QFLE3_INFO("Re-Enabling Recovery");
         vmk_BitVectorAtomicTestAndClear(adapter->sp_rtnl_state, QFLE3_SP_RTNL_DONOT_RECOVER);
      }
   }
   
}


void
qfle3_update_mfw_dump(struct qfle3_adapter *adapter)
{
   vmk_uint32 drv_ver;
   vmk_uint32 valid_dump;
   vmk_TimeVal tv;


   if (!SHMEM2_HAS(adapter, drv_info))
		return;

   /* Update Driver load time*/
   vmk_GetTimeOfDay(&tv);
   SHMEM2_WR(adapter, drv_info.epoc, tv.sec);

   drv_ver = qfle3_version_str_to_int(QFLE3_DRIVER_VERSION, VMK_FALSE);
   SHMEM2_WR(adapter, drv_info.drv_ver, drv_ver);

   SHMEM2_WR(adapter, drv_info.fw_ver, REG_RD(adapter, XSEM_REG_PRAM));

   /* Check & notify On-Chip dump. */
   valid_dump = SHMEM2_RD(adapter, drv_info.valid_dump);

   if (valid_dump & FIRST_DUMP_VALID)
      QFLE3_DBG(QFLE3_DBG_LOAD, "A valid On-Chip MFW dump found on 1st partition\n");

   if (valid_dump & SECOND_DUMP_VALID)
      QFLE3_DBG(QFLE3_DBG_LOAD, "A valid On-Chip MFW dump found on 2nd partition\n");
}

VMK_ReturnStatus qfle3_load(struct qfle3_adapter *adapter)
{
   VMK_ReturnStatus status;
   vmk_uint32 load_code;
   vmk_uint32 val;

   QFLE3_INFO("qfle3_load\n");


   /*
    * init fw_seq
    */
   adapter->fw_seq =
      SHMEM_RD(adapter,
	       func_mb[QFLE3_FW_MB_IDX(adapter)].drv_mb_header) &
      DRV_MSG_SEQ_NUMBER_MASK;
   QFLE3_DBG(QFLE3_DBG_LOAD, "fw_seq 0x%08x\n", adapter->fw_seq);

   status = qfle3_cleanup_prev_unload_state(adapter);
   if (status != VMK_OK)
      goto fail_prev_state;

   adapter->qm_cid_count = qfle3_set_qm_cid_count(adapter);
   QFLE3_DBG(QFLE3_DBG_LOAD, "qm_cid_count %d\n", adapter->qm_cid_count);

   qfle3_setup_ilt(adapter);

   status = qfle3_alloc_probe_mem(adapter);
   if (status != VMK_OK)
      goto fail_probe_mem;

   status = qfle3_create_helper(adapter, &adapter->sp_helper, "sp_helper");
   if (status != VMK_OK)
      goto fail_sp_helper_create;

   status = qfle3_create_helper(adapter,
				&adapter->periodic_task, "period_fn");
   if (status != VMK_OK)
      goto fail_periodic_helper_create;

   qfle3_set_pf_load(adapter);

   status = qfle3_nic_load_request(adapter, &load_code);
   if (status != VMK_OK)
      goto fail_load_req;
   
   QFLE3_DBG(QFLE3_DBG_LOAD, "Loading on PCI device 0x%04x:%02x%02x.%d  code 0x%x", adapter->pdev_addr.seg,adapter->pdev_addr.bus,
      adapter->pdev_addr.dev, adapter->pdev_addr.fn, load_code);   
   if (adapter->recovery_state == QFLE3_RECOVERY_DONE) {
      if (qfle3_isFirstFuncOnDevPath(adapter,adapter->pdev_addr) &&
            ((load_code != FW_MSG_CODE_DRV_LOAD_COMMON) && 
             (load_code != FW_MSG_CODE_DRV_LOAD_COMMON_CHIP))) { // BFS work around code
             
         if (qfle3_isFirstFuncOnDev(adapter,adapter->pdev_addr))
            load_code = FW_MSG_CODE_DRV_LOAD_COMMON_CHIP;
         else
            load_code = FW_MSG_CODE_DRV_LOAD_COMMON;
         
         QFLE3_DBG(QFLE3_DBG_LOAD, "DEBUG CODE: forcing load code to code 0x%x", load_code);      
      }
   }

   adapter->load_code = load_code;

   status = qfle3_compare_fw_ver(adapter, load_code);
   if (status != VMK_OK) {
      qfle3_mfw_command(adapter, DRV_MSG_CODE_LOAD_DONE, 0);
      goto fail_fw_ver;
   }

   if ((load_code == FW_MSG_CODE_DRV_LOAD_COMMON) ||
       (load_code == FW_MSG_CODE_DRV_LOAD_COMMON_CHIP) ||
       (load_code == FW_MSG_CODE_DRV_LOAD_PORT)) {
      adapter->port.pmf = 1;
      qfle3_schedule_helper(adapter,
                            adapter->periodic_task, qfle3_periodic_task_func, 0);
   } else {
      adapter->port.pmf = 0;
   }

   qfle3_zero_drv_ver_ncsi_oem(adapter);

   adapter->dmae_ready = 0;

   status = qfle3_create_spinlock("dmae_lock",
				  QFLE3_NAME_TO_STRING(adapter->pdev_name),
				  QFLE3_LOCK_RANK_HIGH, &adapter->dmae_lock);
   if (status != VMK_OK)
      goto fail_dmae_lock;

   ecore_init_func_obj(adapter, &adapter->func_obj,
		       QFLE3_SP(adapter, func_rdata),
		       QFLE3_SP_MAPPING(adapter, func_rdata),
		       QFLE3_SP(adapter, func_afex_rdata),
		       QFLE3_SP_MAPPING(adapter, func_afex_rdata), &qfle3_func_sp_drv);

   if (adapter->func_obj.one_pending_mutex == NULL) {
      status = VMK_FAILURE;
      goto fail_init_func_obj;
   }

   status = qfle3_init_hw(adapter, load_code);
   if (status != VMK_OK) {
      QFLE3_ERR("HW init failed, aborting\n");
      qfle3_mfw_command(adapter, DRV_MSG_CODE_LOAD_DONE, 0);
      goto fail_init_hw;
   }

   elink_init_mod_abs_int(adapter, &adapter->link_vars, adapter->hw_info.chip_id,
			  adapter->hw_info.shmem_base,
			  adapter->hw_info.shmem2_base, QFLE3_PORT(adapter));

   /*
    * initialize the default status block and sp ring
    */
   qfle3_init_def_sb(adapter);

   qfle3_update_dsb_idx(adapter);

   status = qfle3_init_sp_ring(adapter);
   if (status != VMK_OK) {
      QFLE3_ERR("qfle3_init_sp_ring failed \n");
      qfle3_mfw_command(adapter, DRV_MSG_CODE_LOAD_DONE, 0);
      goto fail_init_sp_ring;
   }

   qfle3_init_eq_ring(adapter);

   qfle3_init_sp_objs(adapter);

   qfle3_init_internal(adapter, load_code);

   qfle3_pf_init(adapter);
//   /* Initialize the stats feature of firmware */
//   status = qfle3_create_spinlock("stats_lock",
//			 QFLE3_NAME_TO_STRING(adapter->pdev_name),
//			 QFLE3_LOCK_RANK_LOW, &adapter->stats_lock);
//   if (status != VMK_OK) {
//      goto fail_stats_lock;
//   }

   // set the mask bit for CSEM CAM parity
   val = REG_RD(adapter,CSEM_REG_CSEM_PRTY_MASK_1);
   val |= CSEM_CSEM_PRTY_MASK_1_REG_CAM_PARITY;
   REG_WR(adapter,CSEM_REG_CSEM_PRTY_MASK_1,val);
   
   adapter->stats_init = VMK_TRUE;
   qfle3_stats_init(adapter);

   status = qfle3_register_sp_interrupt(adapter);

   if (status != VMK_OK) {
      QFLE3_ERR("Failed to register slowpath interrupt!\n");
      goto fail_intr_reg;
   }

   status = vmk_IntrEnable(adapter->intr.cookies[0]);
   if (status != VMK_OK) {
      QFLE3_ERR("Failed to unmask slowpath interrupt!\n");
      goto fail_intr_unmask;
   }

   qfle3_int_enable(adapter);

   qfle3_nic_load_dcc(adapter, load_code);

   status = qfle3_func_start(adapter);

   if (status != VMK_OK) {
      QFLE3_ERR("Function start failed!\n");
      qfle3_mfw_command(adapter, DRV_MSG_CODE_LOAD_DONE, 0);
      goto fail_func_start;
   }
  
   qfle3_update_mng_version(adapter);

   load_code = qfle3_mfw_command(adapter, DRV_MSG_CODE_LOAD_DONE, 0);
   if (!load_code) {
      QFLE3_ERR("MCP response failure, aborting\n");
      status = VMK_BUSY;
      goto fail_mfw_load;
   }
   // driver loaded
   vmk_BitVectorSet(adapter->state, QFLE3_STATE_BIT_LOADED);
   
   qfle3_calculate_filters_count(adapter);
   
   if (adapter->port.pmf)
      qfle3_update_drv_flags(adapter, 1 << DRV_FLAGS_PORT_MASK, 0);
   else {
      qfle3_link_status_update(adapter);
   }
   /* L3 and L4 header is 60, the mtu should not be less than 60. */
   if (load_mtu < 60 || load_mtu > 9000) {
	   QFLE3_INFO("Module param load_mtu(%u) out of range. Using default value (1500).\n", load_mtu);
	   load_mtu = 1500;
   }
   adapter->mtu = load_mtu;

   if (SHMEM2_HAS(adapter, drv_capabilities_flag)) {
      /* mark driver is loaded in shmem2 */
      vmk_uint32 val;
      val = SHMEM2_RD(adapter, drv_capabilities_flag[QFLE3_FW_MB_IDX(adapter)]);
      val &= ~DRV_FLAGS_MTU_MASK;
      val |= (adapter->mtu << DRV_FLAGS_MTU_SHIFT);
      SHMEM2_WR(adapter, drv_capabilities_flag[QFLE3_FW_MB_IDX(adapter)],
		val | DRV_FLAGS_CAPABILITIES_LOADED_SUPPORTED |
		DRV_FLAGS_CAPABILITIES_LOADED_L2);
   }

   qfle3_update_mfw_dump(adapter);

   /* Initiate DCBX FSM */
   if (adapter->port.pmf) {
      qfle3_dcbx_start(adapter);
      QFLE3_DBG(QFLE3_DBG_LOAD, "Starting DCBX.\n");
   }  
   
   qfle3_set_os_driver_state(adapter, OS_DRIVER_STATE_ACTIVE);

   return VMK_OK;

  fail_mfw_load:
  fail_func_start:
   qfle3_int_disable(adapter);
   //Do we need to do this for INTx? May disable interrupts for other devices.
   vmk_IntrDisable(adapter->intr.cookies[0]);
  fail_intr_unmask:
   QFLE3_DBG(QFLE3_DBG_INTR, "Unregister interrupt # %d\n", 2);
   qfle3_unregister_interrupt(adapter, 0);
   fail_intr_reg:
   qfle3_destroy_spinlock(adapter->spq_lock);
   qfle3_destroy_spinlock(adapter->stats_lock);
  fail_init_sp_ring:
   qfle3_gunzip_end(adapter);
  fail_init_hw:
   vmk_SemaDestroy(&adapter->func_obj.one_pending_mutex);
  fail_init_func_obj:
   qfle3_destroy_spinlock(adapter->dmae_lock);
  fail_dmae_lock:
  fail_fw_ver:
   qfle3_mfw_command(adapter, DRV_MSG_CODE_UNLOAD_REQ_WOL_MCP, 0);
   qfle3_mfw_command(adapter, DRV_MSG_CODE_UNLOAD_DONE, 0);
   adapter->port.pmf = 0;
  fail_load_req:
   qfle3_clear_pf_load(adapter);
   qfle3_destroy_helper(adapter, adapter->periodic_task);
  fail_periodic_helper_create:
   qfle3_destroy_helper(adapter, adapter->sp_helper);
  fail_sp_helper_create:
   qfle3_free_probe_mem(adapter);
  fail_probe_mem:
  fail_prev_state:

   return status;

}

void
qfle3_reboot_handler(void *data)
{
   qfle3_mod_info_t *p_drv = (qfle3_mod_info_t *)data;
   vmk_ListLinks *itempPtr = NULL;
   vmk_ListLinks *nic_list = &p_drv->adapterList;

   VMK_LIST_FORALL(nic_list, itempPtr) {

      vmk_uint32 reset_code = DRV_MSG_CODE_UNLOAD_REQ_WOL_DIS;
      vmk_Bool loaded;
#if (VMKAPI_REVISION > VMK_API_2_4_0_0)
      vmk_uint32 ret;
      VMK_ReturnStatus status;
#endif
      qfle3_adapter *adapter = VMK_LIST_ENTRY(itempPtr, qfle3_adapter, adapterLink);

      adapter->phy_initialized = 0;
      adapter->port.pmf = 0;
      /*Stop statistics collection from hardware */
      qfle3_stats_handle(adapter, STATS_EVENT_STOP);
      adapter->reboot_in_progress = 1;

      qfle3_set_os_driver_state(adapter, OS_DRIVER_STATE_DISABLED);

      if (SHMEM2_HAS(adapter, drv_capabilities_flag)) {
         vmk_uint32 val;
         val = SHMEM2_RD(adapter, drv_capabilities_flag[QFLE3_FW_MB_IDX(adapter)]);
         SHMEM2_WR(adapter, drv_capabilities_flag[QFLE3_FW_MB_IDX(adapter)],
		val & ~DRV_FLAGS_CAPABILITIES_LOADED_L2);
      }

      reset_code = qfle3_send_unload_req(adapter, UNLOAD_NORMAL);
      QFLE3_INFO("Value of reset_code is 0x%x \n", reset_code);
#if (VMKAPI_REVISION > VMK_API_2_4_0_0)
      status = qfle3_func_stop(adapter);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to stop function.\n");
      }
      
      ret = qfle3_reset_hw(adapter, reset_code);
      if (ret)
         QFLE3_ERR("HW_RESET failed\n");
      
#endif

      /* Report UNLOAD_DONE to MCP */
      qfle3_mfw_command(adapter, DRV_MSG_CODE_UNLOAD_DONE, 0);

      loaded = qfle3_clear_pf_load(adapter);
      if (loaded == VMK_FALSE) {
         if (qfle3_reset_is_done(adapter, QFLE3_PATH(adapter))){
            QFLE3_ERR("PF is not loaded, calling qfle3_disable_close_the_gate()\n");
            qfle3_disable_close_the_gate(adapter);
         }
      }

      qfle3_int_disable(adapter);
   }
}

VMK_ReturnStatus qfle3_unload(qfle3_adapter *adapter/*, int unload_mode, vmk_Bool keep_link*/)
{

   vmk_uint32 ret;
   vmk_Bool loaded;
   vmk_uint32 reset_code;
   VMK_ReturnStatus status;

   QFLE3_INFO("qfle3_unload, adapter %p\n", adapter);
   
   adapter->phy_initialized = 0;
   adapter->port.pmf = 0;
   qfle3_set_os_driver_state(adapter, OS_DRIVER_STATE_DISABLED);

   if (SHMEM2_HAS(adapter, drv_capabilities_flag)) {
      vmk_uint32 val;
      val = SHMEM2_RD(adapter, drv_capabilities_flag[QFLE3_FW_MB_IDX(adapter)]);
      SHMEM2_WR(adapter, drv_capabilities_flag[QFLE3_FW_MB_IDX(adapter)],
		val & ~DRV_FLAGS_CAPABILITIES_LOADED_L2);
   }
   
#define IS_PF(x) (x)!=NULL
	if (IS_PF(adapter) && adapter->recovery_state != QFLE3_RECOVERY_DONE &&
	    (!vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_LOADED) ||
	    vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_ERROR))) {
		/* We can get here if the driver has been unloaded
		 * during parity error recovery and is either waiting for a
		 * leader to complete or for other functions to unload and
		 * then ifdown has been issued. In this case we want to
		 * unload and let other functions to complete a recovery
		 * process.
		 */
      
		adapter->recovery_state = QFLE3_RECOVERY_DONE;
		adapter->is_leader = 0;
      
		qfle3_release_leader_lock(adapter);
		vmk_CPUMemFenceReadWrite();

		QFLE3_INFO("Releasing a leadership...\n");
		QFLE3_ERR("Can't unload in Quiesced or error state\n");
		return VMK_FAILURE;
	}
	/* Nothing to do during unload if previous bnx2x_nic_load()
	 * have not completed successfully - all resources are released.
	 *
	 * we can get here only after unsuccessful ndo_* callback, during which
	 * dev->IFF_UP flag is still on.
	 */
	if (!vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_LOADED)  || 
      vmk_BitVectorTest(adapter->state,QFLE3_STATE_BIT_ERROR)){
      
      QFLE3_ERR("Can't unload in Quiesced or error state %x\n", adapter->state->vector[0]);
		return VMK_OK;
	}
   
   if (adapter->recovery_state == QFLE3_RECOVERY_DONE){
      
      vmk_BitVectorAtomicTestAndClear(adapter->state, QFLE3_STATE_BIT_LOADED);
      reset_code = qfle3_send_unload_req(adapter, UNLOAD_NORMAL);
      
      QFLE3_ERR("qfle3_unload, reset code %x\n", reset_code);
      status = qfle3_func_stop(adapter);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to stop function.\n");
      }
      ret = qfle3_reset_hw(adapter, reset_code);
      if (ret)
         QFLE3_ERR("HW_RESET failed\n");

      /* Report UNLOAD_DONE to MCP */
      qfle3_mfw_command(adapter, DRV_MSG_CODE_UNLOAD_DONE, 0);

      loaded = qfle3_clear_pf_load(adapter);
      if (loaded == VMK_FALSE) {
         if (qfle3_reset_is_done(adapter, QFLE3_PATH(adapter))){
   	 qfle3_disable_close_the_gate(adapter);
         }
      }
      qfle3_int_disable(adapter);
   }else {
   
      QFLE3_DBG(QFLE3_DBG_LOAD, "qfle3_unload in RECOVERY");
   // unload in recovery mode
      reset_code = qfle3_send_unload_req(adapter, UNLOAD_RECOVERY);
      qfle3_pf_disable(adapter);
      /* Report UNLOAD_DONE to MCP */
      qfle3_mfw_command(adapter, DRV_MSG_CODE_UNLOAD_DONE, 0);
      
      loaded = qfle3_clear_pf_load(adapter);
      
      QFLE3_ERR ("Driver state is %s in MFW", loaded?"loaded":"unloaded");
      if (loaded == VMK_FALSE) {
         if (qfle3_reset_is_done(adapter, QFLE3_PATH(adapter))){
            
            QFLE3_DBG(QFLE3_DBG_LOAD, "Driver CLosing gate");
            qfle3_disable_close_the_gate(adapter);
         }
      }
      
   }

   vmk_SemaDestroy(&adapter->func_obj.one_pending_mutex);

   vmk_IntrDisable(adapter->intr.cookies[0]);
   QFLE3_DBG(QFLE3_DBG_INTR|QFLE3_DBG_LOAD, "Unregister interrupt # %d\n", 0);
   qfle3_unregister_interrupt(adapter, 0);

   qfle3_destroy_spinlock(adapter->spq_lock);

   qfle3_destroy_spinlock(adapter->dmae_lock);
   status = qfle3_destroy_helper(adapter, adapter->sp_helper);
   if (status != VMK_OK) {
      QFLE3_WARN("Failed to destroy slow path helper thread.\n");
   }

   adapter->sp_helper = 0;

   status = qfle3_destroy_helper(adapter, adapter->periodic_task);
   if (status != VMK_OK) {
      QFLE3_WARN("Failed to destroy periodic helper thread.\n");
   }
   
   qfle3_free_probe_mem(adapter);

   adapter->periodic_task = 0;

   return VMK_OK;
}



static VMK_ReturnStatus
qfle3_detach_dev(vmk_Device device)
{
   qfle3_adapter *adapter;
   int i;
   vmk_DeviceGetAttachedDriverData(device, (vmk_AddrCookie *) & adapter);

   QFLE3_DBG(QFLE3_DBG_KERNEL, "DriverOps.detach\n");
   vmk_BitVectorClear(adapter->state, QFLE3_STATE_BIT_ATTACHED);
   qfle3_sm_cmd(adapter, QFLE3_SMCMD_TOSTANDBY, 0);
   qfle3_sm_change_state(adapter, QFLE3_SM_NOTREADY);
   // Before we try to unload, check to make sure that there are no storage related
   // connections and also make sure SRIOV is removed

   if (adapter->cnicEthDev->nof_ofld_conns){
      QFLE3_ERR("Unable to detach, must remove storage connections first %d\n", 
         adapter->cnicEthDev->nof_ofld_conns);
      return VMK_FAILURE;
   }
   if (adapter->vfdb) {
      for (i=0; i<adapter->vfdb->sriov.nr_virtfn; i++) {
         if (adapter->vfdb->vfs[i].state != VF_FREE){
            QFLE3_ERR("Unable to detach, VF[%d] is not free \n",i);
            return VMK_FAILURE;
         }
      }
   }
   
   qfle3_destroy_shared_queue_info(adapter);
   
   // There are no more legitimate reason for us to fail detach now.
   if (qfle3_mod_info.reboot_handler) {
      vmk_UnregisterRebootHandler(qfle3_mod_info.reboot_handler);
      qfle3_mod_info.reboot_handler = NULL;
//      vmk_LogMessage("Reboot Handler called for %d adapters.  Unregistering now.", qfle3_mod_info.reboot_adapter_cnt);
   }else{
      vmk_LogMessage("no Reboot Handler registered");
   }

   //vmk_MgmtUnregisterInstanceCallbacks(vmkmgmt_api_handler, (vmk_uint64)adapter);

   vmk_ListRemove(&adapter->adapterLink);
//   vmk_CPUMemFenceReadWrite();
//   qfle3_stats_handle(adapter, STATS_EVENT_STOP);
//   qfle3_save_statistics(adapter);

   if (VMK_OK != qfle3_destroy_helper (adapter, adapter->stats_collection_helper))
	   QFLE3_ERR("Unable to destroy stats_collection_helper\n");

   adapter->stats_collection_helper = 0;

#ifdef QFLE3_SRIOV
   if(adapter->vfdb){
	   qfle3_iov_free_mem(adapter);
      qfle3_iov_remove_one(adapter);
   } 
   relative_pf_num--;
#endif //QFLE3_SRIOV

   vmk_TimerCancel(adapter->periodic_timer, VMK_TRUE);

   vmk_BitVectorFree(qfle3_mod_info.heapID, adapter->sp_rtnl_state);
   adapter->sp_rtnl_state = NULL;
   
   vmk_BitVectorFree(qfle3_mod_info.heapID, adapter->recovery_status);
   adapter->recovery_status = NULL;
   if (VMK_OK != qfle3_destroy_helper (adapter, adapter->recovery_helper))
      QFLE3_ERR("Unable to destroy recovery helper\n");

   adapter->recovery_helper = 0;

   if (VMK_OK != qfle3_destroy_helper(adapter, adapter->dcbx_sb.dcbx_task))
      QFLE3_ERR("Unable to destroy dcbx_task helper\n");

   adapter->dcbx_sb.dcbx_task = 0;
   qfle3_set_os_driver_state(adapter, OS_DRIVER_STATE_NOT_LOADED);
   if (adapter->is_leader){
      qfle3_release_leader_lock(adapter);
      
      adapter->is_leader = 0;
   }

   qfle3_destroy_spinlock(adapter->sm_lock);

   qfle3_destroy_spinlock(adapter->uplinkSharedDataLock);

   qfle3_destroy_spinlock(adapter->q_lock);
   
   // stats lock destroy, is this the right time?
   qfle3_destroy_spinlock(adapter->stats_lock);


/*
   if (adapter->fwdmp_vmk_handle) {
      vmk_DumpDeleteFileCallback(adapter->fwdmp_vmk_handle);
   }
*/
   if (adapter->fwdmp_buf)
      qfle3_heap_free (adapter->fwdmp_buf);

   qfle3_heap_free(adapter->ilt);

   vmk_SemaDestroy(&adapter->port.phy_mutex);

   vmk_SemaDestroy(&adapter->mfw_cmd_mutex);

   vmk_SemaDestroy(&adapter->drv_info_mutex);
   
   vmk_SemaDestroy(&adapter->cnic_mutex);

   qfle3_destroy_dma_engines(adapter);

   qfle3_free_interrupts(adapter);

   qfle3_disable_bus_master(adapter);

   qfle3_unmap_io_resource(adapter);

   qfle3_free_bit_vectors(adapter);

#ifdef QFLE3_CNIC
   qfle3_heap_free(adapter->cnicEthDev);
#endif
   qfle3_heap_free(adapter);
   return (VMK_OK);
}


#ifdef QFLE3_SRIOV



int qfle3_del_all_macs(struct qfle3_adapter *adapter,
             struct ecore_vlan_mac_obj *mac_obj,
             int mac_type, vmk_Bool wait_for_comp)
{
   int rc;
   unsigned long ramrod_flags = 0, vlan_mac_flags = 0;

   /* Wait for completion of requested */
   if (wait_for_comp)
      ECORE_SET_BIT(RAMROD_COMP_WAIT, &ramrod_flags);
   
   if (adapter->recovery_state != QFLE3_RECOVERY_DONE){
      // Perform a dry cleanup
      ECORE_SET_BIT(RAMROD_DRV_CLR_ONLY, &ramrod_flags);
   }
   /* Set the mac type of addresses we want to clear */
   ECORE_SET_BIT(mac_type, &vlan_mac_flags);

   rc = mac_obj->delete_all(adapter, mac_obj, &vlan_mac_flags, &ramrod_flags);
   if (rc < 0)
      QFLE3_ERR("SRIOV: Failed to delete MACs: %d\n", rc);

   return rc;
}


/* New mac for VF. Consider these cases:
 * 1. VF hasn't been acquired yet - save the mac in local bulletin board and
 *    supply at acquire.
 * 2. VF has already been acquired but has not yet initialized - store in local
 *    bulletin board. mac will be posted on VF bulletin board after VF init. VF
 *    will configure this mac when it is ready.
 * 3. VF has already initialized but has not yet setup a queue - post the new
 *    mac on VF's bulletin board right now. VF will configure this mac when it
 *    is ready.
 * 4. VF has already set a queue - delete any macs already configured for this
 *    queue and manually config the new mac.
 * In any event, once this function has been called refuse any attempts by the
 * VF to configure any mac for itself except for this mac. In case of a race
 * where the VF fails to see the new post on its bulletin board before sending a
 * mac configuration request, the PF will simply fail the request and VF can try
 * again after consulting its bulletin board.
 */
int qfle3_set_vf_mac(vmk_AddrCookie data, int vfidx, vmk_uint8 *mac)
{
   struct qfle3_adapter *adapter = (qfle3_adapter *)data.ptr;
   int rc, q_logical_state;
   struct qfle3_virtf *vf = NULL;
   struct pf_vf_bulletin_content *bulletin = NULL;

   if (!ECORE_IS_VALID_ETHER_ADDR(mac)) {
      QFLE3_ERR("SRIOV: PT: MAC address invalid\n");
      return -1;
   }

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: MAC address IS vALID\n");
   /* sanity and init */
   rc = qfle3_vf_op_prep(adapter, vfidx, &vf, &bulletin, VMK_TRUE);
   if (rc)
      return rc;

   vmk_SemaLock(&adapter->vfdb->bulletin_mutex);

   /* update PF's copy of the VF's bulletin. Will no longer accept mac
    * configuration requests from vf unless match this mac
    */
   bulletin->valid_bitmap |= 1 << MAC_ADDR_VALID;
   vmk_Memcpy(bulletin->mac, mac, ETH_ALEN);

   /* Post update on VF's bulletin board */
   rc = qfle3_post_vf_bulletin(adapter, vfidx);

   /* release lock before checking return code */
   vmk_SemaUnlock(&adapter->vfdb->bulletin_mutex);

   if (rc) {
      QFLE3_ERR("SRIOV: PT: Failed to update VF[%d] bulletin\n", vfidx);
      return rc;
   }
   
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: Updated VF[%d] bulletin\n", vfidx);
   q_logical_state =
      ecore_get_q_logical_state(adapter, &qfle3_leading_vfq(vf, sp_obj));
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: vf->state is %d and q_logical_state is %d \n", vf->state, q_logical_state);
   if (vf->state == VF_ENABLED &&
       q_logical_state == ECORE_Q_LOGICAL_STATE_ACTIVE) {
      /* configure the mac in device on this vf's queue */
      unsigned long ramrod_flags = 0;
      struct ecore_vlan_mac_obj *mac_obj;
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: WE are here \n");
      /* User should be able to see failure reason in system logs */
      if (!qfle3_validate_vf_sp_objs(adapter, vf, VMK_TRUE))
         return -1;

      /* must lock vfpf channel to protect against vf flows */
      qfle3_lock_vf_pf_channel(adapter, vf, CHANNEL_TLV_PF_SET_MAC);

      /* remove existing eth macs */
      mac_obj  = &qfle3_leading_vfq(vf, mac_obj);
      rc = qfle3_del_all_macs(adapter, mac_obj, ECORE_ETH_MAC, VMK_TRUE);
      if (rc) {
         QFLE3_ERR("SRIOV: PT: Failed to delete eth macs\n");
         rc = -1;
         goto out;
      }

      /* remove existing uc list macs */
      rc = qfle3_del_all_macs(adapter, mac_obj, ECORE_UC_LIST_MAC, VMK_TRUE);
      if (rc) {
         QFLE3_ERR("SRIOV: PT: Failed to delete uc_list macs\n");
         rc = -1;
         goto out;
      }

      /* configure the new mac to device */
      ECORE_SET_BIT(RAMROD_COMP_WAIT, &ramrod_flags);
      qfle3_set_mac_one(adapter, (vmk_uint8 *)&bulletin->mac, mac_obj, VMK_TRUE,
              ECORE_ETH_MAC, &ramrod_flags);

out:
      qfle3_unlock_vf_pf_channel(adapter, vf, CHANNEL_TLV_PF_SET_MAC);
   }

   return rc;
}

static int
qfle3_pt_vf_set_mac(vmk_AddrCookie data, vmk_VFID vmkVf, vmk_uint8 *new_mac)
{
   int rc=0;
   struct qfle3_adapter *adapter = (qfle3_adapter *)data.ptr;
   struct qfle3_virtf *vf = QFLE3_VF(adapter, vmkVf);
   struct pf_vf_bulletin_content *bulletin = NULL;

   if (is_zero_ether_addr(new_mac)) {
      struct ecore_vlan_mac_obj *mac_obj = &qfle3_vfq(vf, 0, mac_obj);

      if (vf == NULL || vf->state != VF_ENABLED)
         return VMK_OK;

      /* must lock vfpf channel to protect against vf flows */
      qfle3_lock_vf_pf_channel(adapter, vf, CHANNEL_TLV_PF_SET_MAC);

      /* remove existing eth macs */
      rc = qfle3_del_all_macs(adapter, mac_obj, ECORE_ETH_MAC, VMK_TRUE);
      if (rc) {
         QFLE3_ERR("SRIOV: PT: Failed to delete eth macs\n");
         rc = VMK_FAILURE;
         goto error;
      }

      /* remove existing uc list macs */
      rc = qfle3_del_all_macs(adapter, mac_obj, ECORE_UC_LIST_MAC, VMK_TRUE);
      if (rc) {
         QFLE3_ERR("SRIOV: PT: Failed to delete uc_list macs\n");
         rc = VMK_FAILURE;
         goto error;
      }

      /* update PF's copy of the VF's bulletin.
       */

      bulletin = QFLE3_VF_BULLETIN(adapter, vmkVf);
      if (!bulletin) {
         QFLE3_ERR("SRIOV: PT: Bulletin Board struct is null for vfidx %d\n",
                vmkVf);
         rc = VMK_FAILURE;
         goto error;
      }

      bulletin->valid_bitmap |= 1 << MAC_ADDR_VALID;
      vmk_Memcpy(bulletin->mac, new_mac, ETH_ALEN);
      rc = VMK_OK;
error:
      qfle3_unlock_vf_pf_channel(adapter, vf, CHANNEL_TLV_PF_SET_MAC);
      return rc;
   }
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Mac is not NULL hence executing qfle3_set_vf_mac\n");

   rc = qfle3_set_vf_mac(data, vmkVf, new_mac);

   if (!rc) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: Using mac:%x:%x:%x:%x:%x:%x for VF: %d\n",
         new_mac[0],new_mac[1],new_mac[2],new_mac[3],new_mac[4],new_mac[5], vf->index);
      return VMK_OK;
   } else {
      QFLE3_ERR("SRIOV: PT: Failed to set mac:%x:%x:%x:%x:%x:%x for VF: %d\n",
         new_mac[0],new_mac[1],new_mac[2],new_mac[3],new_mac[4],new_mac[5], vf->index);

      /*  Silently fail the setting of the MAC */
      return VMK_OK;
   }
}


VMK_ReturnStatus
qfle3_pt_vf_set_default_vlan(vmk_AddrCookie data,
             vmk_NetPTOPVFSetDefaultVlanArgs *args)
{
   struct qfle3_adapter *adapter = (qfle3_adapter *)data.ptr;
   int rc;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: Using default VLAN for VF: %d: "
      "enabled: %d vid:%d priority:%d\n",
      args->vf, args->enable, args->vid, args->prio);

   rc = qfle3_pt_set_vf_vlan(adapter, args->vf, args->enable ? args->vid : 0,
                args->prio);
   if (rc == 0)
      return VMK_OK;
   else
      return VMK_FAILURE;
}


static int qfle3_modify_vf_gvlan_range(vmk_AddrCookie data, vmk_NetPTOPVFVlanRangeArgs *args, vmk_NetPTOP op)
{
   int i;
   vmk_uint16 vid;
   vmk_Bool add;
   vmk_ByteCount fsz;
   int rc = 0;
   vmk_uint64 bit = 0;
   vmk_uint64 bit_index = 0;
   vmk_uint8 array_index = 0;
   int num_filters = 0;
   unsigned long accept_flags;
   vmk_uint16 last_gvid = args->last;
   vmk_uint16 first_gvid = args->first;
   struct qfle3_adapter *adapter = (qfle3_adapter *)data.ptr;
   struct qfle3_virtf *vf = QFLE3_VF(adapter, args->vf);
   struct qfle3_vf_mac_vlan_filters *fl = NULL;
   struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vf->index];
   struct ecore_queue_state_params q_params = {NULL};
   struct ecore_queue_update_params *update_params;

   add = (op == VMK_NETPTOP_VF_ADD_VLAN_RANGE) ?  VMK_TRUE : VMK_FALSE;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: %s vlan range %d to %d\n", add ? "Add" : "Remove", args->first, args->last);

   num_filters = last_gvid - first_gvid + 1;
   fsz = num_filters * sizeof(struct qfle3_vf_mac_vlan_filter) + sizeof(struct qfle3_vf_mac_vlan_filters);

   fl = qfle3_heap_alloc(fsz);
   if (!fl) {
      QFLE3_ERR("SRIOV: PT: Failed to alloc memory for GVLAN range. VF: %d, first: %u, last: %u.\n", vf->index, first_gvid, last_gvid);
      return VMK_FAILURE;
   }

   /* must lock vfpf channel to protect against vf flows */
   qfle3_lock_vf_pf_channel(adapter, vf, CHANNEL_TLV_PF_SET_VLAN);

   /* If GVLANs are going to be programmed, then we need to
    * explicitly program VLAN 0, to allow untagged traffic.
    * The VLAN 0 would be requested only by Linux guest OS.
    * The Windows guest OS does[should] NOT request for VLAN 0.
    */
   if (add)
      adapter_esx_vf->allowed_gvlans[0] |= 0x1;
   else
      adapter_esx_vf->allowed_gvlans[0] &= ~0x1;

   for (i = 0, vid = first_gvid; vid <= last_gvid; vid++) {
      array_index = (vid >> 6);
      bit_index = vid & 0x3F;
      bit = ((vmk_uint64)1 << bit_index);

      if (adapter_esx_vf->requested_gvlans[array_index] & bit) {
         fl->filters[i].vid = vid;
         fl->filters[i].type = QFLE3_VF_FILTER_VLAN;
         fl->filters[i].add = add;
         fl->count++;
         i++;
      }

      if (add)
         adapter_esx_vf->allowed_gvlans[array_index] |= bit;
      else
         adapter_esx_vf->allowed_gvlans[array_index] &= ~bit;
   }

   if (vf->state != VF_ENABLED ||
       ecore_get_q_logical_state(adapter, &qfle3_leading_vfq(vf, sp_obj)) != ECORE_Q_LOGICAL_STATE_ACTIVE) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: VF %d not enabled Yet or VF Queues Not created Yet \n.", vf->index);
      rc =  VMK_FAILURE;
      goto out;
   }

   if (!qfle3_leading_vfq(vf, sp_initialized)) {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: VF %d Leading VF Queue Not created Yet \n", vf->index);
      rc =  VMK_FAILURE;
      goto out;
   }

   rc  = qfle3_vf_mac_vlan_config_list(adapter, vf, fl, qfle3_leading_vfq(vf, index), VMK_FALSE);

   fl = NULL;

   if (rc) {
      QFLE3_ERR("SRIOV: Failed to configure GVLAN range. VF: %d, first: %u, last: %u.\n", vf->index, first_gvid, last_gvid); 
      rc =  VMK_FAILURE;
      goto out;
   }

   /* need to remove/add the VF's accept_any_vlan bit */
   accept_flags = qfle3_leading_vfq(vf, accept_flags);
   if (qfle3_esx_vf_any_vlan_allowed(adapter, vf->index))
      ECORE_SET_BIT(ECORE_ACCEPT_ANY_VLAN, &accept_flags);
   else
      ECORE_CLEAR_BIT(ECORE_ACCEPT_ANY_VLAN, &accept_flags);

   qfle3_vf_rxmode(adapter, vf, LEADING_IDX, accept_flags);

   /* send queue update ramrod to configure default vlan and silent
    * vlan removal
    */
   if (qfle3_esx_vf_gvlans_allowed(adapter, vf->index)) {
      ECORE_SET_BIT(RAMROD_COMP_WAIT, &q_params.ramrod_flags);
      q_params.cmd = ECORE_Q_CMD_UPDATE;
      q_params.q_obj = &qfle3_leading_vfq(vf, sp_obj);
      update_params = &q_params.params.update;
      ECORE_SET_BIT(ECORE_Q_UPDATE_DEF_VLAN_EN_CHNG, &update_params->update_flags);
      ECORE_SET_BIT(ECORE_Q_UPDATE_SILENT_VLAN_REM_CHNG, &update_params->update_flags);
      ECORE_CLEAR_BIT(ECORE_Q_UPDATE_DEF_VLAN_EN, &update_params->update_flags);
      ECORE_CLEAR_BIT(ECORE_Q_UPDATE_SILENT_VLAN_REM, &update_params->update_flags);

      /* Update the Queue state */
      rc = ecore_queue_state_change(adapter, &q_params);
      if (rc) {
         QFLE3_ERR("SRIOV: Queue state change failed for guest VLAN configuration.\n");
         goto out;
      }
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Queue state changed to clear default VLAN.\n");
   }
out:
   if (fl)
      qfle3_heap_free(fl);

   qfle3_unlock_vf_pf_channel(adapter, vf, CHANNEL_TLV_PF_SET_VLAN);
   return rc;
}



static void qfle3_esx_get_tx_vf_stats(struct qfle3_adapter *adapter, 
                      vmk_VFID vfID,
                      vmk_uint8 numTxQueues,
                      vmk_NetVFTXQueueStats *vf_tqStats)
{
    int i, j;
    struct qfle3_virtf *vf = QFLE3_VF(adapter, vfID);
    struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vfID];

    for (i = 0; i < vf_txq_count(vf); i++) {
        struct per_queue_stats *fw_stats = (struct per_queue_stats *)
               ((u8 *)adapter_esx_vf->vf_fw_stats + (i * QFLE3_ESX_VF_FW_STATS_SIZE));
        
        struct xstorm_per_queue_stats *xstorm = &fw_stats->xstorm_queue_statistics;

        struct per_queue_stats *old_fw_stats =
               (struct per_queue_stats *)
               ((u8 *)adapter_esx_vf->old_vf_fw_stats + (i * QFLE3_ESX_VF_FW_STATS_SIZE));
        
        struct xstorm_per_queue_stats *old_xstorm = &old_fw_stats->xstorm_queue_statistics;

        j = (numTxQueues != vf_txq_count(vf)) ? 0 : i;

        vf_tqStats[j].unicastPkts += QFLE3_ESX_GET_STORM_STAT_32(
                            xstorm,
                            ucast_pkts_sent);
        vf_tqStats[j].unicastBytes += QFLE3_ESX_GET_STORM_STAT_64(
                            xstorm,
                            ucast_bytes_sent);
        vf_tqStats[j].multicastPkts += QFLE3_ESX_GET_STORM_STAT_32(
                            xstorm,
                            mcast_pkts_sent);
        vf_tqStats[j].multicastBytes += QFLE3_ESX_GET_STORM_STAT_64(
                            xstorm,
                            mcast_bytes_sent);
        vf_tqStats[j].broadcastPkts += QFLE3_ESX_GET_STORM_STAT_32(
                            xstorm,
                            bcast_pkts_sent);
        vf_tqStats[j].broadcastBytes += QFLE3_ESX_GET_STORM_STAT_64(
                            xstorm,
                            bcast_bytes_sent);
        vf_tqStats[j].discards = 0;
        vf_tqStats[j].TSOPkts = 0;
        vf_tqStats[j].TSOBytes = 0;
    }

}

static void qfle3_esx_get_rx_vf_stats(struct qfle3_adapter *adapter,
                      vmk_VFID vfID,
                      vmk_uint8 numRxQueues,
                      vmk_NetVFRXQueueStats *vf_rqStats)
{
    int i, j;
    struct qfle3_virtf *vf = QFLE3_VF(adapter, vfID);
    struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vfID];

    for (i = 0; i < vf_rxq_count(vf); i++) {
        struct per_queue_stats *fw_stats = (struct per_queue_stats *)
               ((u8 *)adapter_esx_vf->vf_fw_stats +
                (i *  QFLE3_ESX_VF_FW_STATS_SIZE));
        struct tstorm_per_queue_stats *tstorm =
            &fw_stats->tstorm_queue_statistics;
        struct ustorm_per_queue_stats *ustorm =
            &fw_stats->ustorm_queue_statistics;

        struct per_queue_stats *old_fw_stats =
               (struct per_queue_stats *)
               ((u8 *)adapter_esx_vf->old_vf_fw_stats +
                (i *  QFLE3_ESX_VF_FW_STATS_SIZE));
        struct tstorm_per_queue_stats *old_tstorm =
            &old_fw_stats->tstorm_queue_statistics;
        struct ustorm_per_queue_stats *old_ustorm =
            &old_fw_stats->ustorm_queue_statistics;

        j = (numRxQueues != vf_rxq_count(vf)) ? 0 : i;
        vf_rqStats[j].unicastPkts += QFLE3_ESX_GET_STORM_STAT_32(
                            tstorm,
                            rcv_ucast_pkts);
        vf_rqStats[j].unicastBytes += QFLE3_ESX_GET_STORM_STAT_64(
                            tstorm,
                            rcv_ucast_bytes);
        vf_rqStats[j].multicastPkts += QFLE3_ESX_GET_STORM_STAT_32(
                            tstorm,
                            rcv_mcast_pkts);
        vf_rqStats[j].multicastBytes += QFLE3_ESX_GET_STORM_STAT_64(
                            tstorm,
                            rcv_mcast_bytes);
        vf_rqStats[j].broadcastPkts += QFLE3_ESX_GET_STORM_STAT_32(
                            tstorm,
                            rcv_bcast_pkts);
        vf_rqStats[j].broadcastBytes += QFLE3_ESX_GET_STORM_STAT_64(
                            tstorm,
                            rcv_bcast_bytes);
        vf_rqStats[j].outOfBufferDrops += QFLE3_ESX_GET_STORM_STAT_32(
                            tstorm,
                            no_buff_discard);
        vf_rqStats[j].errorDrops += QFLE3_ESX_GET_STORM_STAT_32(
                            tstorm,
                            checksum_discard) +
                     QFLE3_ESX_GET_STORM_STAT_32(
                            tstorm,
                            pkts_too_big_discard) +
                     QFLE3_ESX_GET_STORM_STAT_32(
                            tstorm,
                            ttl0_discard);
        vf_rqStats[j].LROPkts += QFLE3_ESX_GET_STORM_STAT_32(
                            ustorm,
                            coalesced_pkts);
        vf_rqStats[j].LROBytes += QFLE3_ESX_GET_STORM_STAT_64(
                            ustorm,
                            coalesced_bytes);
    }
}


VMK_ReturnStatus
qfle3_esx_vf_get_stats(vmk_AddrCookie data, vmk_VFID vfID, vmk_uint8 numTxQueues,
				vmk_uint8 numRxQueues, vmk_NetVFTXQueueStats *tqStats, vmk_NetVFRXQueueStats *rqStats)
{
   qfle3_adapter *adapter = (qfle3_adapter *)data.ptr;
   struct qfle3_virtf *vf = QFLE3_VF(adapter, vfID);
   vmk_NetVFRXQueueStats *vf_rqStats;
   vmk_NetVFTXQueueStats *vf_tqStats;

   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: Returning VF stats VF: %d, TX queues: %d/%d, RX queues %d/%d\n",
       vfID, numTxQueues, vf_txq_count(vf), numRxQueues, vf_rxq_count(vf));

   if(tqStats == NULL){
      QFLE3_DBG(QFLE3_DBG_IOV, "tqStats == NULL\n");
      return VMK_FAILURE;
   }

   if(rqStats == NULL){
      QFLE3_DBG(QFLE3_DBG_IOV, "rqStats == NULL\n");
      return VMK_FAILURE;
   }

   if (vfID > adapter->vfdb->sriov.nr_virtfn){
      QFLE3_DBG(QFLE3_DBG_IOV,"Requesting VF: %d but only configured %d VF's\n",
         vfID, adapter->vfdb->sriov.nr_virtfn);
      return VMK_FAILURE;
   }

   if(vf->state != VF_ENABLED){
      QFLE3_DBG(QFLE3_DBG_IOV,"VF %d not enabled so no stats for it\n",  vfID);
      return VMK_OK;
   }

   vf_tqStats = qfle3_heap_alloc(numTxQueues * sizeof(vmk_NetVFTXQueueStats));
   if(!vf_tqStats){
      QFLE3_DBG(QFLE3_DBG_IOV, "Failed to allocate vf_tqStats \n");
      return VMK_NO_MEMORY;
   }
   
   vf_rqStats = qfle3_heap_alloc(numRxQueues * sizeof(vmk_NetVFRXQueueStats));   
   if(!vf_rqStats){
      QFLE3_DBG(QFLE3_DBG_IOV, "Failed to allocate vf_rqStats \n");
      return VMK_NO_MEMORY;
   } 


   qfle3_esx_get_tx_vf_stats(adapter, vfID, numTxQueues, vf_tqStats);
   qfle3_esx_get_rx_vf_stats(adapter, vfID, numRxQueues, vf_rqStats);

   vmk_Memcpy(tqStats, vf_tqStats, sizeof(vmk_NetVFTXQueueStats));
   vmk_Memcpy(rqStats, vf_rqStats, sizeof(vmk_NetVFRXQueueStats));

   qfle3_heap_free(vf_tqStats);
   qfle3_heap_free(vf_rqStats);

   return VMK_OK;

}

VMK_ReturnStatus qfle3_passthru_ops(vmk_AddrCookie data, vmk_NetPTOP op, void *pargs)
{
   qfle3_adapter *adapter = (qfle3_adapter *)data.ptr;
   VMK_ReturnStatus ret = VMK_OK;
  
   QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: VF Passthrough operation called for OP 0x%x\n", op);
   if(!adapter->esx_vf) {
      QFLE3_ERR("SRIOV: PT: Failing Passthru OP %d. PF may not be up.\n", op);
      return VMK_FAILURE;
   }

   switch (op) {
   case VMK_NETPTOP_VF_SET_MAC:
   {
      vmk_NetPTOPVFSetMacArgs *args = pargs;
      struct qfle3_virtf *vf = QFLE3_VF(adapter, args->vf);
      struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vf->index];
      int rc=0;
      
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: VF Passthrough operation called for VMK_NETPTOP_VF_SET_MAC \n");
      if (vmk_Memcmp(adapter_esx_vf->last_mac, args->mac, ETH_ALEN) != 0) {
            QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Calling Passthru OP to change MAC Address via Hypervisor: VF: %d, MAC: %x:%x:%x:%x:%x:%x\n",
               args->vf, args->mac[0], args->mac[1], args->mac[2], args->mac[3], args->mac[4], args->mac[5]);
            rc = qfle3_pt_vf_set_mac(data, args->vf, args->mac);
            if (rc == VMK_OK)
               vmk_Memcpy(adapter_esx_vf->last_mac, args->mac, ETH_ALEN);
      } else {
            QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Calling Passthru OP to change MAC Address via Hypervisor: VF: %d, MAC: %x:%x:%x:%x:%x:%x but unchaged Hypervisor assigning same MAC\n",
               args->vf, args->mac[0], args->mac[1], args->mac[2], args->mac[3], args->mac[4], args->mac[5]);
            rc = 0;
      }

      adapter_esx_vf->passthru_state = 0;

      return rc;
   }
   case VMK_NETPTOP_VF_SET_DEFAULT_VLAN:
   {
      vmk_NetPTOPVFSetDefaultVlanArgs *args = pargs;
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: VF Passthrough operation called for VMK_NETPTOP_VF_SET_DEFAULT_VLAN \n");
      return qfle3_pt_vf_set_default_vlan(data, args);
   }
   case VMK_NETPTOP_VF_ADD_VLAN_RANGE:
   case VMK_NETPTOP_VF_DEL_VLAN_RANGE:
   {
      int rc = 0;
      vmk_NetPTOPVFVlanRangeArgs *args = pargs;
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: VF Passthrough operation called for VMK_NETPTOP_VF_ADD_VLAN_RANGE or VMK_NETPTOP_VF_DEL_VLAN_RANGE \n");
      rc = qfle3_modify_vf_gvlan_range(data, args, op);
      return rc;
   }
   case VMK_NETPTOP_VF_SET_MTU:
   {
      vmk_NetPTOPVFSetMtuArgs *args = pargs;
      struct qfle3_virtf *vf = QFLE3_VF(adapter, args->vf);
      struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vf->index];

      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: VF %d Passthrough operation called for VMK_NETPTOP_VF_SET_MTU for mtu %d \n", args->vf, args->mtu);
      adapter_esx_vf->flags |= QFLE3_ESX_PASSTHRU_ALLOW_MTU;
      adapter_esx_vf->mtu_from_config = args->mtu;

      adapter_esx_vf->passthru_state = 0;
      return VMK_OK;
   }
   case VMK_NETPTOP_VF_SET_ANTISPOOF:
   {
      vmk_NetPTOPVFSetAntispoofArgs *args = pargs;
      struct qfle3_virtf *vf = QFLE3_VF(adapter, args->vf);
      struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vf->index];

      adapter_esx_vf->flags |= QFLE3_ESX_PASSTHRU_VF_IN_USE;

      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: Unhandled OP VMK_NETPTOP_VF_SET_ANTISPOOF 0x%x\n", op);
      return VMK_FAILURE;
   }
   case VMK_NETPTOP_VF_QUIESCE:
   {
      vmk_NetPTOPVFSimpleArgs *args = pargs;
      struct qfle3_virtf *vf = QFLE3_VF(adapter, args->vf);
      struct qfle3_esx_vf *adapter_esx_vf = &adapter->esx_vf[vf->index];
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: VF Passthrough operation called for VMK_NETPTOP_VF_QUIESCE\n");

      adapter_esx_vf->flags &= ~QFLE3_ESX_PASSTHRU_VF_IN_USE;
      return VMK_OK;
   }
   case VMK_NETPTOP_VF_GET_QUEUE_STATS:
   {
      vmk_NetPTOPVFGetQueueStatsArgs *args = pargs;
      return qfle3_esx_vf_get_stats(data, args->vf, args->numTxQueues,
                                    args->numRxQueues, args->tqStats, args->rqStats);

   }
   default:
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: PT: Unhandled OP 0x%x\n", op);
      return VMK_FAILURE;
   }

   return ret;
}


#endif //QFLE3_SRIOV

static VMK_ReturnStatus
qfle3_scan_dev(vmk_Device device)
{
   VMK_ReturnStatus status = VMK_OK;
   qfle3_adapter *adapter;
   vmk_Name busName;
   vmk_DeviceID deviceID;
   vmk_DeviceProps deviceProps;
#ifdef QFLE3_CNIC
#ifdef QFLE3_CREATE_CNIC_QS

   vmk_DeviceID cnicdeviceID;
   vmk_DeviceProps cnicdeviceProps;

   extern vmk_DeviceOps qfle3_cnic_device_ops;
#endif
#endif
   vmk_DeviceGetAttachedDriverData(device, (vmk_AddrCookie *) & adapter);
   QFLE3_DBG(QFLE3_DBG_KERNEL, "DriverOps.scan, %s\n", adapter->pdev_name.string);

   vmk_NameInitialize(&busName, VMK_LOGICAL_BUS_NAME);
   vmk_BusTypeFind(&busName, &deviceID.busType);
   status = vmk_LogicalCreateBusAddress(qfle3_mod_info.driverID, device, 0,
					&deviceID.busAddress, &deviceID.busAddressLen);
   if (status != VMK_OK) {
      QFLE3_ERR("Failed to create logical bus address for Uplink device(%x)", status);
      goto release_bus_type;
   }
   QFLE3_DBG(QFLE3_DBG_LOAD, "uplink device bus address %s", deviceID.busAddress);
   
   deviceID.busIdentifier = VMK_UPLINK_DEVICE_IDENTIFIER;
   deviceID.busIdentifierLen = sizeof(VMK_UPLINK_DEVICE_IDENTIFIER) - 1;
   deviceProps.registeringDriver = qfle3_mod_info.driverID;
   deviceProps.deviceID = &deviceID;
   deviceProps.deviceOps = &qfle3_uplink_device_ops;
   deviceProps.registeringDriverData.ptr = adapter;
   deviceProps.registrationData.ptr = &adapter->uplinkRegData;
   status = vmk_DeviceRegister(&deviceProps, device, &adapter->uplinkDevice);
   if ((status != VMK_OK) && (status != VMK_EXISTS)) {
      QFLE3_ERR("Failed to register uplink device (%x)", status);
      goto release_bus_address;
   } else {
      status = VMK_OK;
   }

#ifdef QFLE3_SRIOV
   if(adapter->vfdb){
      int i,vfid;

      adapter->vf_cb.ptr = &qfle3_passthru_ops;
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Inside qfle3_scan_dev, to register Passthru Operations\n"); 
      for(i=0; i<adapter->num_vf_actual; i++) {
         vmk_PCIDevice tmp_vf;
         vmk_PCIDevice pdev = adapter->pdev;
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Eexucting vmk_PCIGetVFPCIDevice for VF %d\n", i);
         status = vmk_PCIGetVFPCIDevice(pdev, i, &tmp_vf);
         VMK_ASSERT (status == VMK_OK);

         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Eexucting vmk_PCIRegisterVF for VF %d\n", i);
         status = vmk_PCIRegisterVF(tmp_vf, pdev, qfle3_mod_info.driverID, &qfle3_vf_ops);
         VMK_ASSERT (status == VMK_OK);

         vmk_PCISetVFPrivateData(tmp_vf,adapter->vf_cb);
      }

      qfle3_iov_nic_init(adapter);
      for(vfid=0; vfid < adapter->vfdb->sriov.nr_virtfn; vfid++) {
         struct qfle3_virtf *vf = QFLE3_VF(adapter, vfid);

         /* fill in the BDF and bars */
         vf->bus = adapter->vfdb->vfs[vfid].vf_sbdf.bus;
         vf->devfn = adapter->vfdb->vfs[vfid].vf_sbdf.fn;
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: VF info[%d]: bus 0x%x, devfn 0x%x\n", vf->abs_vfid, vf->bus, vf->devfn);
     }
   }
   
#endif //QFLE3_SRIOV

   QFLE3_DBG(QFLE3_DBG_LOAD, "Uplink device %p registered", adapter->uplinkDevice);
   release_bus_address:
   vmk_LogicalFreeBusAddress(qfle3_mod_info.driverID, deviceID.busAddress);
   
   release_bus_type:
    vmk_BusTypeRelease(deviceID.busType);

   /* CNIC logical device registration */
#ifdef QFLE3_CNIC
#ifdef QFLE3_CREATE_CNIC_QS

   if (NO_FCOE(adapter)&& NO_ISCSI(adapter)){
      QFLE3_ERR("Skip creating of CNIC device because NO FCOE or ISCSI support found!");
      return status;
   }
   
   vmk_BusTypeFind(&busName, &cnicdeviceID.busType);
   if (status != VMK_OK) {
      QFLE3_ERR("Failed to find busType logical (%x)", status);
      return status;
   }
   status = vmk_LogicalCreateBusAddress(qfle3_mod_info.driverID, device, 1, 
   					&cnicdeviceID.busAddress, &cnicdeviceID.busAddressLen);
   if (status != VMK_OK) {
      QFLE3_ERR("Failed to create logical bus address for CNIC device(%x)", status);
      goto release_cnic_bus_type;
   }
   
   QFLE3_DBG(QFLE3_DBG_LOAD, "cnic device bus address %s", cnicdeviceID.busAddress);
   adapter->nicOps.cnicProbeNic = &qfle3_cnic_probe;
   adapter->nicOps.cnicLL2Tx = &cnicLL2Tx;
   cnicdeviceID.busIdentifier = QFLE3_CNIC_DEVICE_IDENTIFIER;
   cnicdeviceID.busIdentifierLen = sizeof(QFLE3_CNIC_DEVICE_IDENTIFIER) - 1;
   cnicdeviceProps.registeringDriver = qfle3_mod_info.driverID;
   cnicdeviceProps.deviceID = &cnicdeviceID;
   cnicdeviceProps.deviceOps = &qfle3_cnic_device_ops;
   cnicdeviceProps.registeringDriverData.ptr = adapter;
   cnicdeviceProps.registrationData.ptr = &adapter->nicOps;

   status = vmk_DeviceRegister(&cnicdeviceProps, device, &adapter->cnicDevice);

   if ((status != VMK_OK) && (status != VMK_EXISTS)) {
      QFLE3_ERR("Failed to register cnic device (%x)", status);
      goto release_cnic_bus_address;
   } else {
      status = VMK_OK;
   }

   QFLE3_INFO("Cnic device %p registered", adapter->cnicDevice);

   
   /*
    * Fall through
    */

  release_cnic_bus_address:
  vmk_LogicalFreeBusAddress(qfle3_mod_info.driverID, cnicdeviceID.busAddress);
  
  release_cnic_bus_type:
   vmk_BusTypeRelease(cnicdeviceID.busType);
#endif
#endif  
   QFLE3_DBG(QFLE3_DBG_KERNEL, "DriverOps.scan, Done\n");

   return status;
}

static void
qfle3_init_macaddress(qfle3_adapter * adapter)
{
   vmk_UplinkSharedData *sd = &adapter->uplinkSharedData;

   QFLE3_SHARED_DATA_WRITE_BEGIN(adapter);
   vmk_Memcpy(sd->macAddr, adapter->hwMacAddr, VMK_ETH_ADDR_LENGTH);
   vmk_Memcpy(sd->hwMacAddr, adapter->hwMacAddr, VMK_ETH_ADDR_LENGTH);
   QFLE3_SHARED_DATA_WRITE_END(adapter);
}

static void
qfle3_init_uplink_data(qfle3_adapter * adapter)
{
   int i;
   vmk_UplinkSharedData *sharedData;
   vmk_UplinkRegData *regData;
   char phy_fw_ver[PHY_FW_VER_LEN] = {'\0'};
   int cfg_idx = qfle3_get_link_cfg_idx(adapter);
   int phy_idx = qfle3_get_cur_phy_idx(adapter);
   struct elink_phy *phy;
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
   vmk_LinkMediaType media;
#endif
   phy = &adapter->link_params.phy[phy_idx];

   adapter->supportedModesArraySz = 0;
   i = 0;
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
   switch (phy->media_type) {
	   case ELINK_ETH_PHY_SFPP_10G_FIBER:
		   media = VMK_LINK_MEDIA_BASE_SFI;
		   break;
	   case ELINK_ETH_PHY_KR:
		   if (phy->supported & (ELINK_SUPPORTED_20000baseKR2_Full) )
			   media = VMK_LINK_MEDIA_BASE_KR2;
		   else if (phy->supported & (ELINK_SUPPORTED_20000baseMLD2_Full) )
			   media = VMK_LINK_MEDIA_BASE_MLD2;
		   else 
			   media = VMK_LINK_MEDIA_BASE_KR;
		   break;
	   case ELINK_ETH_PHY_DA_TWINAX:
		   media = VMK_LINK_MEDIA_BASE_TWINAX;
		   break;
	   case ELINK_ETH_PHY_BASE_T:
		   media = VMK_LINK_MEDIA_BASE_T;
		   break;
	   case ELINK_ETH_PHY_NOT_PRESENT:
		   media = VMK_LINK_MEDIA_NONE;
		   break;
	   case ELINK_ETH_PHY_CX4:
	   case ELINK_ETH_PHY_SFP_1G_FIBER:
	   case ELINK_ETH_PHY_XFP_FIBER:
	   case ELINK_ETH_PHY_UNSPECIFIED:
	   default:
		   media = VMK_LINK_MEDIA_UNKNOWN;
		   break;
   }
#endif
   if (adapter->link_params.req_line_speed[cfg_idx] == ELINK_SPEED_AUTO_NEG) {
      adapter->supportedModes[i].speed = VMK_LINK_SPEED_AUTO;
      adapter->supportedModes[i].duplex = VMK_LINK_DUPLEX_AUTO;
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
      adapter->supportedModes[i].media = media;
#endif
      i++;
   }

   if (adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_10baseT_Half) {
      adapter->supportedModes[i].speed = VMK_LINK_SPEED_10_MBPS;
      adapter->supportedModes[i].duplex = VMK_LINK_DUPLEX_HALF;
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
      adapter->supportedModes[i].media = media;
#endif
      i++;
   }

   if (adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_10baseT_Full) {
      adapter->supportedModes[i].speed = VMK_LINK_SPEED_10_MBPS;
      adapter->supportedModes[i].duplex = VMK_LINK_DUPLEX_FULL;
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
      adapter->supportedModes[i].media = media;
#endif
      i++;
   }

   if (adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_100baseT_Half) {
      adapter->supportedModes[i].speed = VMK_LINK_SPEED_100_MBPS;
      adapter->supportedModes[i].duplex = VMK_LINK_DUPLEX_HALF;
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
      adapter->supportedModes[i].media = media;
#endif
      i++;
   }

   if (adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_100baseT_Full) {
      adapter->supportedModes[i].speed = VMK_LINK_SPEED_100_MBPS;
      adapter->supportedModes[i].duplex = VMK_LINK_DUPLEX_FULL;
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
      adapter->supportedModes[i].media = media;
#endif
      i++;
   }

#if 0
//TODO find out if we support Half duplex on 1000MBPS.
   if (adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_1000baseT_Half) {
      adapter->supportedModes[i].speed = VMK_LINK_SPEED_1000_MBPS;
      adapter->supportedModes[i].duplex = VMK_LINK_DUPLEX_HALF;
      i++;
   }
#endif

   if (adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_1000baseT_Full) {
		if ((phy->media_type == ELINK_ETH_PHY_BASE_T) ||
				(phy->media_type == ELINK_ETH_PHY_KR) ||
				(phy->media_type == ELINK_ETH_PHY_SFP_1G_FIBER)) {
			adapter->supportedModes[i].speed = VMK_LINK_SPEED_1000_MBPS;
			adapter->supportedModes[i].duplex = VMK_LINK_DUPLEX_FULL;
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
			adapter->supportedModes[i].media = media;
#endif
			i++;
		}
   }

#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
   if (adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_2500baseX_Full) {
      adapter->supportedModes[i].speed = VMK_LINK_SPEED_2500_MBPS;
      adapter->supportedModes[i].duplex = VMK_LINK_DUPLEX_FULL;
      adapter->supportedModes[i].media = media;
      i++;
   }
#endif
   if (adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_10000baseT_Full) {
      adapter->supportedModes[i].speed = VMK_LINK_SPEED_10000_MBPS;
      adapter->supportedModes[i].duplex = VMK_LINK_DUPLEX_FULL;
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
      adapter->supportedModes[i].media = media;
#endif
      i++;
   }

   if (adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_20000baseKR2_Full) {
      adapter->supportedModes[i].speed = VMK_LINK_SPEED_20000_MBPS;
      adapter->supportedModes[i].duplex = VMK_LINK_DUPLEX_FULL;
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
      adapter->supportedModes[i].media = media;
#endif
      i++;
   }

   adapter->supportedModesArraySz = i;

   sharedData = &adapter->uplinkSharedData;
   vmk_VersionedAtomicInit(&sharedData->lock);
   sharedData->flags = 0;
   sharedData->state = VMK_UPLINK_STATE_ENABLED;
   sharedData->mtu = adapter->mtu;
   sharedData->supportedModes = adapter->supportedModes;
   sharedData->supportedModesArraySz = adapter->supportedModesArraySz;
   vmk_NameInitialize(&sharedData->driverInfo.driver, QFLE3_DRIVER_NAME);
   vmk_NameInitialize(&sharedData->driverInfo.version, QFLE3_DRIVER_VERSION);
   vmk_NameInitialize(&sharedData->driverInfo.moduleInterface, "native");

   elink_get_ext_phy_fw_version(&adapter->link_params, phy_fw_ver, PHY_FW_VER_LEN);

   vmk_NameFormat(&sharedData->driverInfo.firmwareVersion,
		  "Storm: %u.%u.%u.%u MFW: %u.%u.%u",
		  BCM_5710_FW_MAJOR_VERSION,
		  BCM_5710_FW_MINOR_VERSION,
		  BCM_5710_FW_REVISION_VERSION,
		  BCM_5710_FW_ENGINEERING_VERSION,
		  (adapter->hw_info.bc_ver & 0xff0000) >> 16,
		  (adapter->hw_info.bc_ver & 0xff00) >> 8,
		  (adapter->hw_info.bc_ver & 0xff));

   QFLE3_INFO("Storm %u.%u.%u.%u MFW %u.%u.%u%s%s",
	      BCM_5710_FW_MAJOR_VERSION,
	      BCM_5710_FW_MINOR_VERSION,
	      BCM_5710_FW_REVISION_VERSION,
	      BCM_5710_FW_ENGINEERING_VERSION,
	      (adapter->hw_info.bc_ver & 0xff0000) >> 16,
	      (adapter->hw_info.bc_ver & 0xff00) >> 8,
	      (adapter->hw_info.bc_ver & 0xff),
	      ((phy_fw_ver[0] != '\0') ? " PHY " : ""), phy_fw_ver);

   /*
    * MAC address is already poll out from ecore in dev_attach()
    */
   qfle3_init_macaddress(adapter);

   regData = &adapter->uplinkRegData;
   regData->apiRevision = VMKAPI_REVISION;
   regData->moduleID = vmk_ModuleCurrentID;
   regData->ops = qfle3_uplink_ops;
   regData->sharedData = sharedData;
   regData->driverData.ptr = adapter;
}


void
qfle3_set_fp_rx_buf_size(struct qfle3_adapter *adapter, struct qfle3_fastpath *fp)
{
   vmk_uint32  mtu;
   
   QFLE3_DBG(QFLE3_DBG_LOAD, "mtu = %d\n", adapter->mtu);

#ifdef QFLE3_CNIC
   if (IS_FCOE_FP(fp))
      mtu = QFLE3_FCOE_MINI_JUMBO_MTU;
   else if (IS_OOO_FP(fp) || IS_FWD_FP(fp))
      mtu = iscsi_ol_mtu;
   else
#endif
      mtu = adapter->mtu;
      
   /*
    * get the Rx buffer size for RX frames
    */
   fp->rx_buf_size =
      (IP_HEADER_ALIGNMENT_PADDING + ETH_OVERHEAD + mtu + VMK_L1_CACHELINE_SIZE);

   QFLE3_DBG(QFLE3_DBG_LOAD, "rx_buf_size for fp[%02d] = %d\n",
	     fp->qid, fp->rx_buf_size);

   /*
    * get the mbuf allocation size for RX frames
    */
   if (fp->rx_buf_size <= 2048) {
      fp->rx_buf_alloc_size = 2048;
   } else if (fp->rx_buf_size <= (3 * 1024)) {
      fp->rx_buf_alloc_size = (3 * 1024);
   } else if (fp->rx_buf_size <= BCM_PAGE_SIZE) {
      fp->rx_buf_alloc_size = BCM_PAGE_SIZE;
   } else {
      fp->rx_buf_alloc_size = (9 * 1024); //jumbo 9k
   }

   QFLE3_DBG(QFLE3_DBG_LOAD, "mbuf_alloc_size for fp[%02d] = %d\n",
	     fp->qid, fp->rx_buf_alloc_size);
}

void qfle3_update_coalesce(struct qfle3_adapter *adapter)
{
   int i;

   for(i = 0; i < QFLE3_NUM_ETH_QUEUES(adapter); i++) {
      qfle3_update_coalesce_sb(adapter, adapter->fp[i].fw_sb_id,
			       adapter->tx_ticks, adapter->rx_ticks);
   }
}

void
qfle3_init_rss_tables(qfle3_adapter *adapter)
{
   vmk_int32 i;
   const vmk_uint64 key[] = { 0xbeac01fa6a42b73bULL, 0x8030f20c77cb2da3ULL,
			      0xae7b30b4d0ca2bcbULL, 0x43a38fb04167253dULL, 0x255b0ec26d5a56daULL };

#if (ESX_DDK_VERSION >= 2017)
   vmk_uint32 j;
   for (i = 0; i < T_ETH_INDIRECTION_TABLE_SIZE; i++) {
      /* raw ind tbl entry should have a value between
       * 0 to adapter->esx.rss_p_size - 1
       */
      for(j=1; j<=QFLE3_DEVICE_MAX_RSS_ENGINE; j++){
         if (adapter->num_rssqs_nd > 0)
            adapter->netq_rss_raw_ind_tbl[j][i] = i % adapter->num_rssqs_nd;
      } if (adapter->num_rssqs_def > 1)
         adapter->defq_rss_raw_ind_tbl[i] = i % adapter->num_rssqs_def;
   }
   for(j=1; j<=QFLE3_DEVICE_MAX_RSS_ENGINE; j++){
      vmk_Memcpy(adapter->netq_rss_key_tbl[j], &key, sizeof(adapter->netq_rss_key_tbl[j]));
   }
   if (adapter->num_rssqs_def > 1)
      vmk_Memcpy(adapter->defq_rss_key_tbl, &key, sizeof(adapter->defq_rss_key_tbl));

#else

   for (i = 0; i < T_ETH_INDIRECTION_TABLE_SIZE; i++) {
      /* raw ind tbl entry should have a value between
       * 0 to adapter->esx.rss_p_size - 1
       */
      if (adapter->num_rssqs_nd > 0)
         adapter->netq_rss_raw_ind_tbl[i] = i % adapter->num_rssqs_nd;
      if (adapter->num_rssqs_def > 0)
         adapter->defq_rss_raw_ind_tbl[i] = i % adapter->num_rssqs_def;
   }
   /* Initialize RSS keys with random numbers */
   vmk_Memcpy(adapter->netq_rss_key_tbl, &key, sizeof(adapter->netq_rss_key_tbl));
   if (adapter->num_rssqs_def > 0)
      vmk_Memcpy(adapter->defq_rss_key_tbl, &key, sizeof(adapter->defq_rss_key_tbl));
#endif
}

VMK_ReturnStatus
qfle3_start_dev(vmk_Device device)
{
   qfle3_adapter *adapter;
   QFLE3_SMCMD_STATUS sm_status = QFLE3_SMCMD_STATUS_WRONGSTATE;

   vmk_DeviceGetAttachedDriverData(device, (vmk_AddrCookie *) & adapter);
   QFLE3_DBG(QFLE3_DBG_KERNEL, "DriverOps.start, %s\n", adapter->pdev_name.string);
   
   sm_status = qfle3_sm_cmd(adapter, QFLE3_SMCMD_START, 0);
   if (sm_status != QFLE3_SMCMD_STATUS_COMPLETED){
      QFLE3_ERR("DriverOps.start qfle3_smcmd_start failed!");
      goto err_failed;
   }
   QFLE3_DBG(QFLE3_DBG_KERNEL, "DriverOps.start, %s Done\n", adapter->pdev_name.string);
   return VMK_OK;
err_failed:
   return VMK_FAILURE;
   
}

static VMK_ReturnStatus
qfle3_quiesce_dev(vmk_Device device)
{
   qfle3_adapter *adapter;
   QFLE3_SMCMD_STATUS sm_status = QFLE3_SMCMD_STATUS_WRONGSTATE;

   vmk_DeviceGetAttachedDriverData(device, (vmk_AddrCookie *) & adapter);
   QFLE3_DBG(QFLE3_DBG_KERNEL, "DriverOps.quiesce, %s\n", adapter->pdev_name.string);

   sm_status = qfle3_sm_cmd(adapter, QFLE3_SMCMD_STOP, 0);
   if (sm_status != QFLE3_SMCMD_STATUS_COMPLETED) {
      QFLE3_ERR("DriverOps.quiece qfle3_smcmd_stop failed!");
   }
//   vmk_BitVectorSet(adapter->state, QFLE3_STATE_BIT_QUIESCED);
   
   QFLE3_DBG(QFLE3_DBG_KERNEL, "DriverOps.quiesce, %s Done\n", adapter->pdev_name.string);
   return VMK_OK;
}

static void
qfle3_forget_dev(vmk_Device device)
{
   qfle3_adapter *adapter;
   vmk_DeviceGetAttachedDriverData(device, (vmk_AddrCookie *) & adapter);
   QFLE3_DBG(QFLE3_DBG_KERNEL, "DriverOps.forget, %s\n", adapter->pdev_name.string);
   return;
}

static int
ecore_gunzip(struct qfle3_adapter *adapter, const vmk_uint8 * zbuf, int len)
{
   return 0;
}

/*
 * Global Data
 */
vmk_DriverOps qfle3_drv_ops = {
   .attachDevice = qfle3_attach_dev,
   .detachDevice = qfle3_detach_dev,
   .scanDevice = qfle3_scan_dev,
   .startDevice = qfle3_start_dev,
   .quiesceDevice = qfle3_quiesce_dev,
   .forgetDevice = qfle3_forget_dev,
};

int qfle3_set_q_rx_mode(struct qfle3_adapter *adapter, vmk_uint8 cl_id,
			unsigned long rx_mode_flags,
			unsigned long rx_accept_flags,
			unsigned long tx_accept_flags,
			unsigned long ramrod_flags)
{
   struct ecore_rx_mode_ramrod_params ramrod_param;
   int rc;

   vmk_Memset(&ramrod_param, 0, sizeof(ramrod_param));

   /* Prepare ramrod parameters */
   ramrod_param.cid = 0;
   ramrod_param.cl_id = cl_id;
   ramrod_param.rx_mode_obj = &adapter->rx_mode_obj;
   ramrod_param.func_id = QFLE3_FUNC(adapter);

   ramrod_param.pstate = &adapter->sp_state;
   ramrod_param.state = ECORE_FILTER_RX_MODE_PENDING;

   ramrod_param.rdata = QFLE3_SP(adapter, rx_mode_rdata);
   ramrod_param.rdata_mapping = QFLE3_SP_MAPPING(adapter, rx_mode_rdata);

   ECORE_SET_BIT(ECORE_FILTER_RX_MODE_PENDING, &adapter->sp_state);

   ramrod_param.ramrod_flags = ramrod_flags;
   ramrod_param.rx_mode_flags = rx_mode_flags;

   ramrod_param.rx_accept_flags = rx_accept_flags;
   ramrod_param.tx_accept_flags = tx_accept_flags;

   rc = ecore_config_rx_mode(adapter, &ramrod_param);
   if (rc < 0) {
      QFLE3_ERR("Set rx_mode %d failed\n", adapter->rx_mode);
      return rc;
   }

   return 0;
}

int qfle3_fill_accept_flags(struct qfle3_adapter *adapter, vmk_uint32 rx_mode,
			    unsigned long *rx_accept_flags,
			    unsigned long *tx_accept_flags)
{
   /* Clear the flags first */
   *rx_accept_flags = 0;
   *tx_accept_flags = 0;

   switch (rx_mode) {
   case QFLE3_RX_MODE_NONE:
      /*
       * 'drop all' supersedes any accept flags that may have been
       * passed to the function.
       */
      break;
   case QFLE3_RX_MODE_NORMAL:
      ECORE_SET_BIT(ECORE_ACCEPT_UNICAST, rx_accept_flags);
      ECORE_SET_BIT(ECORE_ACCEPT_MULTICAST, rx_accept_flags);
      ECORE_SET_BIT(ECORE_ACCEPT_BROADCAST, rx_accept_flags);

      /* internal switching mode */
      ECORE_SET_BIT(ECORE_ACCEPT_UNICAST, tx_accept_flags);
      ECORE_SET_BIT(ECORE_ACCEPT_MULTICAST, tx_accept_flags);
      ECORE_SET_BIT(ECORE_ACCEPT_BROADCAST, tx_accept_flags);

      if (adapter->accept_any_vlan) {
	 ECORE_SET_BIT(ECORE_ACCEPT_ANY_VLAN, rx_accept_flags);
	 ECORE_SET_BIT(ECORE_ACCEPT_ANY_VLAN, tx_accept_flags);
      }

      break;
   case QFLE3_RX_MODE_ALLMULTI:
      ECORE_SET_BIT(ECORE_ACCEPT_UNMATCHED, rx_accept_flags);
      ECORE_SET_BIT(ECORE_ACCEPT_UNICAST, rx_accept_flags);
      ECORE_SET_BIT(ECORE_ACCEPT_ALL_MULTICAST, rx_accept_flags);
      ECORE_SET_BIT(ECORE_ACCEPT_BROADCAST, rx_accept_flags);

      /* internal switching mode */
      ECORE_SET_BIT(ECORE_ACCEPT_UNICAST, tx_accept_flags);
      ECORE_SET_BIT(ECORE_ACCEPT_ALL_MULTICAST, tx_accept_flags);
      ECORE_SET_BIT(ECORE_ACCEPT_BROADCAST, tx_accept_flags);

      if (adapter->accept_any_vlan) {
	 ECORE_SET_BIT(ECORE_ACCEPT_ANY_VLAN, rx_accept_flags);
	 ECORE_SET_BIT(ECORE_ACCEPT_ANY_VLAN, tx_accept_flags);
      }

      break;
   case QFLE3_RX_MODE_PROMISC:
      /* According to definition of SI mode, iface in promisc mode
       * should receive matched and unmatched (in resolution of port)
       * unicast packets.
       */
      ECORE_SET_BIT(ECORE_ACCEPT_UNMATCHED, rx_accept_flags);
      ECORE_SET_BIT(ECORE_ACCEPT_UNICAST, rx_accept_flags);
      ECORE_SET_BIT(ECORE_ACCEPT_ALL_MULTICAST, rx_accept_flags);
      ECORE_SET_BIT(ECORE_ACCEPT_BROADCAST, rx_accept_flags);

      /* internal switching mode */
      ECORE_SET_BIT(ECORE_ACCEPT_ALL_MULTICAST, tx_accept_flags);
      ECORE_SET_BIT(ECORE_ACCEPT_BROADCAST, tx_accept_flags);

#ifdef QFLE3_NPAR
      if (IS_MF_SI(adapter))
	 ECORE_SET_BIT(ECORE_ACCEPT_ALL_UNICAST, tx_accept_flags);
      else
#endif
	 ECORE_SET_BIT(ECORE_ACCEPT_UNICAST, tx_accept_flags);

      ECORE_SET_BIT(ECORE_ACCEPT_ANY_VLAN, rx_accept_flags);
      ECORE_SET_BIT(ECORE_ACCEPT_ANY_VLAN, tx_accept_flags);

      break;
   default:
      QFLE3_ERR("Unknown rx_mode: %d\n", rx_mode);
      return VMK_BAD_PARAM;
   }

   return 0;
}

int qfle3_set_storm_rx_mode(struct qfle3_adapter *adapter)
{
   unsigned long rx_mode_flags = 0, ramrod_flags = 0;
   unsigned long rx_accept_flags = 0, tx_accept_flags = 0;
   int rc;
   
	if (!NO_FCOE(adapter))
		/* Configure rx_mode of FCoE Queue */
		ECORE_SET_BIT(ECORE_RX_MODE_FCOE_ETH, &rx_mode_flags);

   rc = qfle3_fill_accept_flags(adapter, adapter->rx_mode, &rx_accept_flags,
				&tx_accept_flags);
   if (rc)
      return rc;

   ECORE_SET_BIT(RAMROD_RX, &ramrod_flags);
   ECORE_SET_BIT(RAMROD_TX, &ramrod_flags);

   return qfle3_set_q_rx_mode(adapter, adapter->fp->cl_id, rx_mode_flags,
			      rx_accept_flags, tx_accept_flags,
			      ramrod_flags);
}

void
qfle3_set_rx_mode(qfle3_adapter *adapter)
{
   qfle3_set_storm_rx_mode(adapter);
}

int qfle3_set_mac_one(struct qfle3_adapter *adapter, vmk_uint8 *mac,
		      struct ecore_vlan_mac_obj *obj, vmk_Bool set,
		      int mac_type, unsigned long *ramrod_flags)
{
   int rc;
   struct ecore_vlan_mac_ramrod_params ramrod_param;

   QFLE3_DBG(QFLE3_DBG_SP, "set_mac_one %s "QFLE3_MAC_FMT, set?"set":"remove", QFLE3_MAC_PRN_LIST(mac));

   vmk_Memset(&ramrod_param, 0, sizeof(ramrod_param));

   /* Fill general parameters */
   ramrod_param.vlan_mac_obj = obj;
   ramrod_param.ramrod_flags = *ramrod_flags;

   /* Fill a user request section if needed */
   if (!ECORE_TEST_BIT(RAMROD_CONT, ramrod_flags)) {
      vmk_Memcpy(ramrod_param.user_req.u.mac.mac, mac, ETH_ALEN);
      ECORE_SET_BIT(mac_type, &ramrod_param.user_req.vlan_mac_flags);

      /* Set the command: ADD or DEL */
      if (set)
	 ramrod_param.user_req.cmd = ECORE_VLAN_MAC_ADD;
      else
	 ramrod_param.user_req.cmd = ECORE_VLAN_MAC_DEL;
   }

   rc = ecore_config_vlan_mac(adapter, &ramrod_param);

   if (rc == ECORE_EXISTS) {
      QFLE3_DBG(QFLE3_DBG_SP, "MAC already exists: %d\n", rc);
      /* do not treat adding same MAC as error */
      rc = 0;
   } else if (rc != 0)
      QFLE3_ERR("%s MAC failed\n", (set ? "Set" : "Del"));

   return rc;
}

int qfle3_set_vxlan_fltr_one(struct qfle3_adapter *adapter, vmk_uint8 *outer_mac,
				struct ecore_vlan_mac_obj *obj, vmk_Bool set,
				int mac_type, unsigned long *ramrod_flags,
				vmk_uint8 *inner_mac, vmk_uint32 vxlan_id)
{
   int rc;
   struct ecore_vlan_mac_ramrod_params ramrod_param;
#if 0
   if (!is_valid_ether_addr(inner_mac)) {
      DP(QFLE3_MSG_SP, "Returning since NULL inner mac being programmed.\n");
      /* Do not treat as error. */
      return 0;
   }
#endif

   vmk_Memset(&ramrod_param, 0, sizeof(ramrod_param));

        /* Fill general parameters */
   ramrod_param.vlan_mac_obj = obj;
   ramrod_param.ramrod_flags = *ramrod_flags;

   /* Fill a user request section if needed */
   if (!ECORE_TEST_BIT(RAMROD_CONT, ramrod_flags)) {
      struct ecore_vxlan_fltr_ramrod_data *ramrod_data;

      ramrod_data = &ramrod_param.user_req.u.vxlan_fltr;
      vmk_Memcpy(ramrod_data->innermac, inner_mac, ETH_ALEN);
      ramrod_data->vni = vxlan_id;

      /* Set the command: ADD or DEL */
      if (set)
         ramrod_param.user_req.cmd = ECORE_VLAN_MAC_ADD;
      else
         ramrod_param.user_req.cmd = ECORE_VLAN_MAC_DEL;
   }

   rc = ecore_config_vlan_mac(adapter, &ramrod_param);

   if (rc == ECORE_EXISTS) {
      QFLE3_DBG(QFLE3_DBG_SP, "VXLAN filter already exists: %d\n", rc);
           /* do not treat adding same MAC as error */
           rc = 0;
   } else if (rc != 0)
           QFLE3_ERR("%s VXLAN Filter failed\n", (set ? "Set" : "Del"));

   return rc;
}


int
qfle3_stop_queue(struct qfle3_adapter *adapter, int index)
{
   struct qfle3_fastpath *fp = &adapter->fp[index];
	struct qfle3_fp_txdata *txdata = fp->txdata_ptr[0];
   struct ecore_queue_state_params q_params = {NULL};
   int rc, tx_index;

   QFLE3_DBG(QFLE3_DBG_SP, "Stopping queue %d cid %d\n", index, fp->cid);

   q_params.q_obj = &QFLE3_SP_OBJ(adapter, fp).q_obj;
   /* We want to wait for completion in this context */
   ECORE_SET_BIT(RAMROD_COMP_WAIT, &q_params.ramrod_flags);
   
   if (adapter->recovery_state != QFLE3_RECOVERY_DONE){
      ECORE_SET_BIT(RAMROD_DRV_CLR_ONLY, &q_params.ramrod_flags);
   }
   
   if (!IS_FWD_FP(fp)) {

      /* close tx-only connections */
      for (tx_index = FIRST_TX_ONLY_COS_INDEX;
           tx_index < fp->max_cos;
           tx_index++){

         /* ascertain this is a normal queue*/
         txdata = fp->txdata_ptr[tx_index];

         QFLE3_DBG(QFLE3_DBG_SP, "stopping tx-only queue fp[%d.%d],%d\n",
                        fp->qid, txdata->cos, txdata->txq_index);

         /* send halt terminate on tx-only connection */
         q_params.cmd = ECORE_Q_CMD_TERMINATE;
         vmk_Memset(&q_params.params.terminate, 0,
                sizeof(q_params.params.terminate));
         q_params.params.terminate.cid_index = tx_index;

         rc = ecore_queue_state_change(adapter, &q_params);
         if (rc) {
            QFLE3_ERR ("Terminate CMD FAILED fp[%d.%d],%d\n",fp->qid, txdata->cos, txdata->txq_index);
            return rc;
         }
         QFLE3_DBG(QFLE3_DBG_SP, "Terminate CMD was sucessful queue fp[%d.%d],%d\n",
                        fp->qid, txdata->cos, txdata->txq_index);
         /* send halt terminate on tx-only connection */
         q_params.cmd = ECORE_Q_CMD_CFC_DEL;
         vmk_Memset(&q_params.params.cfc_del, 0,
                sizeof(q_params.params.cfc_del));
         q_params.params.cfc_del.cid_index = tx_index;
         rc = ecore_queue_state_change(adapter, &q_params);
         if (rc) {
            QFLE3_ERR ("CFC DEL CMD FAILED fp[%d.%d],%d\n",fp->qid, txdata->cos, txdata->txq_index);
            return rc;
         }

         QFLE3_DBG(QFLE3_DBG_SP, "CFC DEL CMD was sucessful queue fp[%d.%d],%d\n",
                        fp->qid, txdata->cos, txdata->txq_index);
      }
   /* Stop the primary connection: */
   /* ...halt the connection */

      txdata = fp->txdata_ptr[0];
      QFLE3_DBG(QFLE3_DBG_SP, "stopping primary connection queue fp[%d.0],%d\n",
                     fp->qid, txdata->txq_index);
   q_params.cmd = ECORE_Q_CMD_HALT;
   rc = ecore_queue_state_change(adapter, &q_params);
      if (rc) {
         QFLE3_ERR ("HALT CMD FAILED fp[%d.%d],%d\n",fp->qid, txdata->cos, txdata->txq_index);
      return rc;
      }
      QFLE3_DBG(QFLE3_DBG_SP, "HALT CMD was sucessful queue fp[%d.0],%d\n",
                     fp->qid, txdata->txq_index);
   /* ...terminate the connection */
   q_params.cmd = ECORE_Q_CMD_TERMINATE;
   vmk_Memset(&q_params.params.terminate, 0,
	      sizeof(q_params.params.terminate));
   q_params.params.terminate.cid_index = FIRST_TX_COS_INDEX;
   rc = ecore_queue_state_change(adapter, &q_params);
      if (rc) {
         QFLE3_ERR ("Terminate CMD FAILED fp[%d.%d],%d\n",fp->qid, txdata->cos, txdata->txq_index);
      return rc;
      }

      QFLE3_DBG(QFLE3_DBG_SP, "Terminate CMD was sucessful queue fp[%d.%d],%d\n",
                     fp->qid, txdata->cos, txdata->txq_index);
   /* ...delete cfc entry */
   }
   q_params.cmd = ECORE_Q_CMD_CFC_DEL;
   vmk_Memset(&q_params.params.cfc_del, 0,
	      sizeof(q_params.params.cfc_del));
   q_params.params.cfc_del.cid_index = FIRST_TX_COS_INDEX;
   rc = ecore_queue_state_change(adapter, &q_params);
   if (rc) {
      QFLE3_ERR ("CFC DEL CMD FAILED fp[%d.%d],%d\n",fp->qid, txdata->cos, txdata->txq_index);
      return rc;
   }

   QFLE3_DBG(QFLE3_DBG_SP, "CFC DEL CMD was sucessful queue fp[%d.%d],%d\n",
                  fp->qid, txdata->cos, txdata->txq_index);
   return rc;
}

inline void
qfle3_update_fpsb_idx(struct qfle3_fastpath *fp)
{
   vmk_CPUMemFenceReadWrite();
   fp->fp_hc_idx = fp->sb_running_index[SM_RX_ID];
}

static void qfle3_wr_fp_sb_data(struct qfle3_adapter *adapter,
				int fw_sb_id,
				vmk_uint32 *sb_data_p,
				vmk_uint32 data_size)
{
   int index;
   for (index = 0; index < data_size; index++)
      REG_WR(adapter, BAR_CSTRORM_INTMEM +
	     CSTORM_STATUS_BLOCK_DATA_OFFSET(fw_sb_id) +
	     sizeof(vmk_uint32)*index,
	     *(sb_data_p + index));
}

static void qfle3_zero_fp_sb(struct qfle3_adapter *adapter, int fw_sb_id)
{
   vmk_uint32 *sb_data_p;
   vmk_uint32 data_size = 0;
   struct hc_status_block_data_e2 sb_data_e2;

   /* disable the function first */
   vmk_Memset(&sb_data_e2, 0, sizeof(struct hc_status_block_data_e2));
   sb_data_e2.common.state = SB_DISABLED;
   sb_data_e2.common.p_func.vf_valid = VMK_FALSE;
   sb_data_p = (vmk_uint32 *)&sb_data_e2;
   data_size = sizeof(struct hc_status_block_data_e2)/sizeof(vmk_uint32);
   qfle3_wr_fp_sb_data(adapter, fw_sb_id, sb_data_p, data_size);

   qfle3_fill(adapter, BAR_CSTRORM_INTMEM +
	      CSTORM_STATUS_BLOCK_OFFSET(fw_sb_id), 0,
	      CSTORM_STATUS_BLOCK_SIZE);
   qfle3_fill(adapter, BAR_CSTRORM_INTMEM +
	      CSTORM_SYNC_BLOCK_OFFSET(fw_sb_id), 0,
	      CSTORM_SYNC_BLOCK_SIZE);
}

static void qfle3_setup_ndsb_state_machine(struct hc_status_block_sm *hc_sm,
					   int igu_sb_id, int igu_seg_id)
{
   hc_sm->igu_sb_id = igu_sb_id;
   hc_sm->igu_seg_id = igu_seg_id;
   hc_sm->timer_value = 0xFF;
   hc_sm->time_to_expire = 0xFFFFFFFF;
}

/* allocates state machine ids. */
static void qfle3_map_sb_state_machines(struct hc_index_data *index_data)
{
   /* zero out state machine indices */
   /* rx indices */
   index_data[HC_INDEX_ETH_RX_CQ_CONS].flags &= ~HC_INDEX_DATA_SM_ID;

   /* tx indices */
   index_data[HC_INDEX_OOO_TX_CQ_CONS].flags &= ~HC_INDEX_DATA_SM_ID;
   index_data[HC_INDEX_ETH_TX_CQ_CONS_COS0].flags &= ~HC_INDEX_DATA_SM_ID;
   index_data[HC_INDEX_ETH_TX_CQ_CONS_COS1].flags &= ~HC_INDEX_DATA_SM_ID;
   index_data[HC_INDEX_ETH_TX_CQ_CONS_COS2].flags &= ~HC_INDEX_DATA_SM_ID;

   /* map indices */
   /* rx indices */
   index_data[HC_INDEX_ETH_RX_CQ_CONS].flags |=
      SM_RX_ID << HC_INDEX_DATA_SM_ID_SHIFT;

   /* tx indices */
   index_data[HC_INDEX_OOO_TX_CQ_CONS].flags |=
      SM_TX_ID << HC_INDEX_DATA_SM_ID_SHIFT;
   index_data[HC_INDEX_ETH_TX_CQ_CONS_COS0].flags |=
      SM_TX_ID << HC_INDEX_DATA_SM_ID_SHIFT;
   index_data[HC_INDEX_ETH_TX_CQ_CONS_COS1].flags |=
      SM_TX_ID << HC_INDEX_DATA_SM_ID_SHIFT;
   index_data[HC_INDEX_ETH_TX_CQ_CONS_COS2].flags |=
      SM_TX_ID << HC_INDEX_DATA_SM_ID_SHIFT;
}

void qfle3_init_sb(struct qfle3_adapter *adapter, vmk_IOA mapping, int vfid,
		   vmk_uint8 vf_valid, int fw_sb_id, int igu_sb_id)
{
   int igu_seg_id;

   struct hc_status_block_data_e2 sb_data_e2;
   struct hc_status_block_sm  *hc_sm_p;
   int data_size;
   vmk_uint32 *sb_data_p;

   if (CHIP_INT_MODE_IS_BC(adapter))
      igu_seg_id = HC_SEG_ACCESS_NORM;
   else
      igu_seg_id = IGU_SEG_ACCESS_NORM;

   qfle3_zero_fp_sb(adapter, fw_sb_id);

   vmk_Memset(&sb_data_e2, 0, sizeof(struct hc_status_block_data_e2));
   sb_data_e2.common.state = SB_ENABLED;
   sb_data_e2.common.p_func.pf_id = QFLE3_FUNC(adapter);
   sb_data_e2.common.p_func.vf_id = vfid;
   sb_data_e2.common.p_func.vf_valid = vf_valid;
   sb_data_e2.common.p_func.vnic_id = QFLE3_VN(adapter);
   sb_data_e2.common.same_igu_sb_1b = VMK_TRUE;
   sb_data_e2.common.host_sb_addr.hi = U64_HI(mapping);
   sb_data_e2.common.host_sb_addr.lo = U64_LO(mapping);
   hc_sm_p = sb_data_e2.common.state_machine;
   sb_data_p = (vmk_uint32 *)&sb_data_e2;
   data_size = sizeof(struct hc_status_block_data_e2)/sizeof(vmk_uint32);
   qfle3_map_sb_state_machines(sb_data_e2.index_data);

   qfle3_setup_ndsb_state_machine(&hc_sm_p[SM_RX_ID],
				  igu_sb_id, igu_seg_id);
   qfle3_setup_ndsb_state_machine(&hc_sm_p[SM_TX_ID],
				  igu_sb_id, igu_seg_id);

   //QFLE3_DBG(QFLE3_DBG_START, "Init FW SB %d\n", fw_sb_id);

   /* write indices to HW - PCI guarantees endianity of regpairs */
   qfle3_wr_fp_sb_data(adapter, fw_sb_id, sb_data_p, data_size);
}

unsigned long qfle3_get_common_flags(struct qfle3_adapter *adapter,
					    struct qfle3_fastpath *fp,
					    vmk_Bool zero_stats)
{
   unsigned long flags = 0;

   /* PF driver will always initialize the Queue to an ACTIVE state */
   SET_BIT(ECORE_Q_FLG_ACTIVE, &flags);

   /* tx only connections collect statistics (on the same index as the
    * parent connection). The statistics are zeroed when the parent
    * connection is initialized.
    */
/* set here to enable statistics */
   //TODO: reenable statistics for FCOE
   if ((!IS_FWD_FP(fp)) && (!IS_OOO_FP(fp)) && (!IS_FCOE_FP(fp)))
   {
      SET_BIT(ECORE_Q_FLG_STATS, &flags);
      if (zero_stats)
         SET_BIT(ECORE_Q_FLG_ZERO_STATS, &flags);
      
      if (adapter->flags & TX_SWITCHING)
         SET_BIT(ECORE_Q_FLG_TX_SWITCH, &flags);
   }
   SET_BIT(ECORE_Q_FLG_PCSUM_ON_PKT, &flags);
   SET_BIT(ECORE_Q_FLG_TUN_INC_INNER_IP_ID, &flags);

#ifdef QFLE3_STOP_ON_ERROR
   SET_BIT(ECORE_Q_FLG_TX_SEC, &flags);
#endif


   return flags;
}

static unsigned long qfle3_get_q_flags(struct qfle3_adapter *adapter,
				       struct qfle3_fastpath *fp,
				       vmk_Bool leading)
{
   unsigned long flags = 0;

   /* clculate other queue flags */
   if (IS_MF_SD(adapter))
      SET_BIT(ECORE_Q_FLG_OV, &flags);

   if (IS_OOO_FP(fp))
      SET_BIT(ECORE_Q_FLG_OOO, &flags);

   if (IS_FCOE_FP(fp)) {
      SET_BIT(ECORE_Q_FLG_FCOE, &flags);
         /* For FCoE - force usage of default priority (for afex) */
      SET_BIT(ECORE_Q_FLG_FORCE_DEFAULT_PRI, &flags);
   }
   
 	if (IS_FCOE_FP(fp) && IS_MF_SD(adapter) && !IS_MF_UFP(adapter) && !IS_MF_BD(adapter))
		SET_BIT(ECORE_Q_FLG_REFUSE_OUTBAND_VLAN, &flags);

   if (fp->tpa_enable) {
      SET_BIT(ECORE_Q_FLG_TPA, &flags);
      SET_BIT(ECORE_Q_FLG_TPA_IPV6, &flags);
#if 0
      if (fp->tpa_enable)
	 SET_BIT(ECORE_Q_FLG_TPA_GRO, &flags);
#endif
   }

   if (leading) {
      SET_BIT(ECORE_Q_FLG_LEADING_RSS, &flags);
      SET_BIT(ECORE_Q_FLG_MCAST, &flags);
   }

   if ((!IS_FCOE_FP(fp)) && adapter->hw_vlan_en) {
      SET_BIT(ECORE_Q_FLG_VLAN, &flags);
   }

   /* configure silent vlan removal */
   if (IS_MF_AFEX(adapter))
      SET_BIT(ECORE_Q_FLG_SILENT_VLAN_REM, &flags);

#if 0
   /* configure silent vlan removal */
   if (adapter->dcb_state == QFLE3_DCB_STATE_ON)
      SET_BIT(ECORE_Q_FLG_SILENT_VLAN_REM, &flags);
#endif

   /* merge with common flags */
   return flags | qfle3_get_common_flags(adapter, fp, VMK_TRUE);
}


static void qfle3_pf_q_prep_init(struct qfle3_adapter *adapter,
				 struct qfle3_fastpath *fp, struct ecore_queue_init_params *init_params)
{
   int cxt_index, cxt_offset;
   vmk_uint8 cos;

   if (!IS_FCOE_FP(fp)) {
      SET_BIT(ECORE_Q_FLG_HC, &init_params->rx.flags);
      SET_BIT(ECORE_Q_FLG_HC, &init_params->tx.flags);

      /* If HC is supported, enable host coalescing in the transition
       * to INIT state.
       */
      SET_BIT(ECORE_Q_FLG_HC_EN, &init_params->rx.flags);
      SET_BIT(ECORE_Q_FLG_HC_EN, &init_params->tx.flags);

      /* HC rate */
      init_params->rx.hc_rate = adapter->rx_ticks ?
         (1000000 / adapter->rx_ticks) : 0;
      init_params->tx.hc_rate = adapter->tx_ticks ?
         (1000000 / adapter->tx_ticks) : 0;

      /* FW SB ID */
      init_params->rx.fw_sb_id = init_params->tx.fw_sb_id =
         fp->fw_sb_id;

      /*
       * CQ index among the SB indices: FCoE clients uses the default
       * SB, therefore it's different.
       */
      init_params->rx.sb_cq_index = HC_INDEX_ETH_RX_CQ_CONS;
      init_params->tx.sb_cq_index = HC_INDEX_ETH_FIRST_TX_CQ_CONS;
   }
   /* set maximum number of COSs supported by this queue */
   init_params->max_cos = fp->max_cos;


   QFLE3_DBG(QFLE3_DBG_START, "fp: %d setting queue params max cos to: %d\n",
	     fp->qid, init_params->max_cos);

   for (cos = FIRST_TX_COS_INDEX; cos < init_params->max_cos; cos++) {
      cxt_index = fp->txdata_ptr[cos]->cid / ILT_PAGE_CIDS;
      cxt_offset = fp->txdata_ptr[cos]->cid - (cxt_index *
			   ILT_PAGE_CIDS);
      init_params->cxts[cos] =
      &adapter->context[cxt_index].vcxt[cxt_offset].eth;

      QFLE3_DBG(QFLE3_DBG_START, "fp: %d cos %d context index %d, context offset %d\n",
           fp->qid, cos, cxt_index, cxt_offset);
   }
}

void qfle3_pf_q_prep_general(struct qfle3_adapter *adapter,
				    struct qfle3_fastpath *fp, struct ecore_general_setup_params *gen_init,
				    vmk_uint8 cos)
{
   /* set firmware stat_id for stats */
   gen_init->stat_id = qfle3_stats_id(fp);
   gen_init->spcl_id = fp->cl_id;

	if (IS_FCOE_FP(fp))
		gen_init->mtu = QFLE3_FCOE_MINI_JUMBO_MTU;
	else
		gen_init->mtu = adapter->mtu;


   gen_init->cos = cos;

   gen_init->fp_hsi = ETH_FP_HSI_VERSION;
   /* TODO: Add default_vlan configuration here
    * when VIF supoort is added.
    */
}

static void qfle3_pf_rx_q_prep(struct qfle3_adapter *adapter,
			       struct qfle3_fastpath *fp, struct rxq_pause_params *pause,
			       struct ecore_rxq_setup_params *rxq_init)
{
   vmk_uint8 max_sge = 0;
   vmk_uint16 sge_sz = 0;
   vmk_uint16 tpa_agg_size = 0;

   if (fp->tpa_enable) {
      pause->sge_th_lo = SGE_TH_LO(adapter);
      pause->sge_th_hi = SGE_TH_HI(adapter);

      /* validate SGE ring has enough to cross high threshold */
      ECORE_DBG_BREAK_IF(adapter->dropless_fc &&
         pause->sge_th_hi + FW_PREFETCH_CNT >
         RX_SGE_USABLE_PER_PAGE * RX_SGE_NUM_PAGES);

      tpa_agg_size = TPA_AGG_SIZE;
#if 0
      /*
       * The minimum size for tpa aggregation size is
       * RX buffer size + MTU
       */
      if (max_agg_size_param >= (fp->rx_buf_size + adapter->dev->mtu))
	 tpa_agg_size = min(max_agg_size_param, tpa_agg_size);
#endif
      max_sge = SGE_PAGE_ALIGN(adapter->mtu) >>
         SGE_PAGE_SHIFT;
      max_sge = ((max_sge + PAGES_PER_SGE - 1) &
         (~(PAGES_PER_SGE-1))) >> PAGES_PER_SGE_SHIFT;
      sge_sz = (vmk_uint16)min(SGE_PAGES, 0xffff);
   }

   pause->bd_th_lo = BD_TH_LO(adapter);
   pause->bd_th_hi = BD_TH_HI(adapter);

   pause->rcq_th_lo = RCQ_TH_LO(adapter);
   pause->rcq_th_hi = RCQ_TH_HI(adapter);
   /*
    * validate that rings have enough entries to cross
    * high thresholds
    */
   ECORE_DBG_BREAK_IF(adapter->dropless_fc &&
      pause->bd_th_hi + FW_PREFETCH_CNT >
      adapter->rx_ring_size);
   ECORE_DBG_BREAK_IF(adapter->dropless_fc &&
      pause->rcq_th_hi + FW_PREFETCH_CNT >
      RCQ_NUM_PAGES * RCQ_USABLE_PER_PAGE);

   if (adapter->sw_fcoe) {// not only for default rx queue
      pause->pri_map = 1 | 1 << sw_fcoe_pri;  /* 1 << 3 for FCoE priority*/
   } else {
   pause->pri_map = 1;
   }

   /* rxq setup */
   rxq_init->dscr_map = fp->rx_chain_ioa;
   rxq_init->sge_map = fp->rx_sge_chain_ioa;
   rxq_init->rcq_map = fp->rcq_chain_ioa;
   rxq_init->rcq_np_map = fp->rcq_chain_ioa + BCM_PAGE_SIZE;

   /* This should be a maximum number of data bytes that may be
    * placed on the BD (not including paddings).
    */
   rxq_init->buf_sz = fp->rx_buf_size - QFLE3_FW_RX_ALIGN_START -
      QFLE3_FW_RX_ALIGN_END - IP_HEADER_ALIGNMENT_PADDING;

   rxq_init->cl_qzone_id = fp->cl_qzone_id;
   rxq_init->tpa_agg_sz = tpa_agg_size;
   rxq_init->sge_buf_sz = sge_sz;
   rxq_init->max_sges_pkt = max_sge;
   rxq_init->rss_engine_id = fp->rss_engine_id;
   rxq_init->mcast_engine_id = QFLE3_FUNC(adapter);

   /* Maximum number or simultaneous TPA aggregation for this Queue.
    *
    * For PF Clients it should be the maximum available number.
    * VF driver(s) may want to define it to a smaller value.
    */
   rxq_init->max_tpa_queues = MAX_AGG_QS(adapter);

   rxq_init->cache_line_log = QFLE3_RX_ALIGN_SHIFT;
   rxq_init->fw_sb_id = fp->fw_sb_id;

	if (IS_FCOE_FP(fp))
		rxq_init->sb_cq_index = HC_SP_INDEX_ETH_FCOE_RX_CQ_CONS;
	else
		rxq_init->sb_cq_index = HC_INDEX_ETH_RX_CQ_CONS;
   /* configure silent vlan removal
    * if multi function mode is afex, then mask default vlan
    */
   if (IS_MF_AFEX(adapter)) {
      rxq_init->silent_removal_value = adapter->afex_def_vlan_tag;
      rxq_init->silent_removal_mask = VLAN_VID_MASK;
   } else {
      rxq_init->silent_removal_value = 0;
      rxq_init->silent_removal_mask = VLAN_PRIO_MASK | VLAN_VID_MASK;
   }
}

void qfle3_pf_tx_q_prep(struct qfle3_adapter *adapter,
			       struct qfle3_fastpath *fp, struct ecore_txq_setup_params *txq_init,
			       vmk_uint8 cos)
{
   txq_init->dscr_map = fp->txdata_ptr[cos]->tx_chain_ioa;
   txq_init->sb_cq_index = HC_INDEX_ETH_FIRST_TX_CQ_CONS + cos;
   txq_init->traffic_type = LLFC_TRAFFIC_TYPE_NW;
   txq_init->fw_sb_id = fp->fw_sb_id;

   /*
    * set the tss leading client id for TX classification ==
    * leading RSS client id
    */
   txq_init->tss_leading_cl_id = qfle3_fp(adapter, 0, cl_id);//fp->cl_id;

   
	if (IS_FCOE_FP(fp)) {
		txq_init->sb_cq_index = HC_SP_INDEX_ETH_FCOE_TX_CQ_CONS;
		txq_init->traffic_type = LLFC_TRAFFIC_TYPE_FCOE;
	}
}

int qfle3_setup_tx_only(struct qfle3_adapter *adapter, struct qfle3_fastpath *fp,
			struct ecore_queue_state_params *q_params,
			struct ecore_queue_setup_tx_only_params *tx_only_params,
			int tx_index, vmk_Bool leading)
{
	vmk_Memset(tx_only_params, 0, sizeof(*tx_only_params));

	/* Set the command */
	q_params->cmd = ECORE_Q_CMD_SETUP_TX_ONLY;

	/* Set tx-only QUEUE flags: don't zero statistics */
	tx_only_params->flags = qfle3_get_common_flags(adapter, fp, VMK_FALSE);

	/* choose the index of the cid to send the slow path on */
	tx_only_params->cid_index = tx_index;

	/* Set general TX_ONLY_SETUP parameters */
	qfle3_pf_q_prep_general(adapter, fp, &tx_only_params->gen_params, tx_index);

	/* Set Tx TX_ONLY_SETUP parameters */
	qfle3_pf_tx_q_prep(adapter, fp, &tx_only_params->txq_params, tx_index);


	QFLE3_DBG(QFLE3_DBG_START,
	   "preparing to send tx-only ramrod for connection: cos %d, primary cid %d, cid %d, client id %d, sp-client id %d, flags %lx\n",
	   tx_index, q_params->q_obj->cids[FIRST_TX_COS_INDEX],
	   q_params->q_obj->cids[tx_index], q_params->q_obj->cl_id,
	   tx_only_params->gen_params.spcl_id, tx_only_params->flags);

	/* send the ramrod */
	return ecore_queue_state_change(adapter, q_params);
}

int qfle3_setup_queue(struct qfle3_adapter *adapter, struct qfle3_fastpath *fp,
		      vmk_Bool leading)
{
   struct ecore_queue_state_params q_params = {NULL};
   struct ecore_queue_setup_params *setup_params =
      &q_params.params.setup;
   struct ecore_queue_setup_tx_only_params *tx_only_params =
                                        &q_params.params.tx_only;
   int rc;
   vmk_uint8 tx_index;

   QFLE3_DBG(QFLE3_DBG_START, "setting up queue %d\n", fp->qid);

   /* reset IGU state skip FCoE L2 queue */
   if (!IS_FCOE_FP(fp))
      qfle3_ack_sb(adapter, fp->igu_sb_id, USTORM_ID, 0,
   		IGU_INT_ENABLE, 0);

   q_params.q_obj = &QFLE3_SP_OBJ(adapter, fp).q_obj;
   /* We want to wait for completion in this context */
   SET_BIT(RAMROD_COMP_WAIT, &q_params.ramrod_flags);

   /* Prepare the INIT parameters */
   qfle3_pf_q_prep_init(adapter, fp, &q_params.params.init);

   /* Set the command */
   q_params.cmd = ECORE_Q_CMD_INIT;

   /* Change the state to INIT */
   rc = ecore_queue_state_change(adapter, &q_params);
   if (rc) {
      QFLE3_ERR("Queue(%d) INIT failed\n", fp->qid);
      return rc;
   }

   QFLE3_DBG(QFLE3_DBG_SP, "init complete %d \n", fp->qid);

#ifdef QFLE3_CNIC
   if ((IS_FWD_FP(fp)))
      return qfle3_setup_tx_only(adapter, fp, &q_params, tx_only_params,
				 FIRST_TX_COS_INDEX, leading);
#endif

   /* Now move the Queue to the SETUP state... */
   vmk_Memset(setup_params, 0, sizeof(*setup_params));

   /* Set QUEUE flags */
   setup_params->flags = qfle3_get_q_flags(adapter, fp, leading);

   /* Set general SETUP parameters */
   qfle3_pf_q_prep_general(adapter, fp, &setup_params->gen_params,
			   FIRST_TX_COS_INDEX);

   qfle3_pf_rx_q_prep(adapter, fp, &setup_params->pause_params,
		      &setup_params->rxq_params);
   
	if (!IS_OOO_FP(fp))
      qfle3_pf_tx_q_prep(adapter, fp, &setup_params->txq_params,
		      FIRST_TX_COS_INDEX);

   /* Set the command */
   q_params.cmd = ECORE_Q_CMD_SETUP;

#ifdef QFLE3_CNIC   
   if (IS_FCOE_FP(fp))
      adapter->fcoe_init = VMK_TRUE;

   if (IS_OOO_FP(fp))
      adapter->ooo_init = VMK_TRUE;
#endif

   /* Change the state to SETUP */
   rc = ecore_queue_state_change(adapter, &q_params);
   if (rc) {
      QFLE3_ERR("Queue(%d) SETUP failed\n", fp->qid);
      return rc;
   }

   QFLE3_DBG(QFLE3_DBG_START,"Queue(%d.0) SETUP success\n", fp->qid);

	/* loop through the relevant tx-only indices */
	for (tx_index = FIRST_TX_ONLY_COS_INDEX;
	      tx_index < fp->max_cos;
	      tx_index++) {

		/* prepare and send tx-only ramrod*/
		rc = qfle3_setup_tx_only(adapter, fp, &q_params,
					  tx_only_params, tx_index, leading);
		if (rc) {
			QFLE3_ERR("Queue(%d.%d) TX_ONLY_SETUP failed\n",
				  fp->qid, tx_index);
			return rc;
		}

      QFLE3_DBG(QFLE3_DBG_START,"Queue(%d.%d) TX_ONLY_SETUP success\n", fp->qid, tx_index);
   }
   return rc;
}

static struct ecore_queue_sp_obj *
qfle3_cid_to_q_obj(struct qfle3_adapter *adapter, vmk_uint32 cid)
{
   struct ecore_queue_sp_obj * q_obj = NULL;
   QFLE3_DBG(QFLE3_DBG_SP, "retrieving fp from cid %d\n", cid);
#ifdef QFLE3_CNIC   
   if (/*CNIC_LOADED(adapter) && */(cid == QFLE3_FCOE_ETH_CID(adapter)))
      q_obj = &qfle3_fcoe_sp_obj(adapter, q_obj);
   else if (/*CNIC_LOADED(adapter) && */(cid == QFLE3_OOO_ETH_CID(adapter)))
      q_obj = &qfle3_ooo_sp_obj(adapter, q_obj);
   else if (/*CNIC_LOADED(adapter) && */(cid == QFLE3_FWD_ETH_CID(adapter)))
      q_obj = &qfle3_fwd_sp_obj(adapter, q_obj);
   else
#endif
      q_obj = &adapter->sp_objs[CID_TO_FP(cid, adapter)].q_obj;

   return q_obj;

}

#ifdef QFLE3_CNIC
/* Called with netif_addr_lock_bh() taken.
 * Sets an rx_mode config for an iSCSI ETH client.
 * Doesn't block.
 * Completion should be checked outside.
 */
void qfle3_set_iscsi_eth_rx_mode(struct qfle3_adapter *adapter, vmk_Bool start)
{
	unsigned long accept_flags = 0, ramrod_flags = 0;
	vmk_uint8 cl_id = qfle3_cnic_eth_cl_id(adapter, QFLE3_ISCSI_ETH_CL_ID_IDX);
	int sched_state = ECORE_FILTER_ISCSI_ETH_STOP_SCHED;

	if (start) {
		/* Start accepting on iSCSI L2 ring. Accept all multicasts
		 * because it's the only way for UIO Queue to accept
		 * multicasts (in non-promiscuous mode only one Queue per
		 * function will receive multicast packets (leading in our
		 * case).
		 */
		ECORE_SET_BIT(ECORE_ACCEPT_UNICAST, &accept_flags);
		ECORE_SET_BIT(ECORE_ACCEPT_ALL_MULTICAST, &accept_flags);
		ECORE_SET_BIT(ECORE_ACCEPT_BROADCAST, &accept_flags);
		ECORE_SET_BIT(ECORE_ACCEPT_ANY_VLAN, &accept_flags);

		/* Clear STOP_PENDING bit if START is requested */
		ECORE_CLEAR_BIT(ECORE_FILTER_ISCSI_ETH_STOP_SCHED, &adapter->sp_state);

		sched_state = ECORE_FILTER_ISCSI_ETH_START_SCHED;
	} else
		/* Clear START_PENDING bit if STOP is requested */
		ECORE_CLEAR_BIT(ECORE_FILTER_ISCSI_ETH_START_SCHED, &adapter->sp_state);

	if (ECORE_TEST_BIT(ECORE_FILTER_RX_MODE_PENDING, &adapter->sp_state)) {
		ECORE_SET_BIT(sched_state, &adapter->sp_state);
                QFLE3_DBG(QFLE3_DBG_CNIC, "Exit iscsi_eth_rx_mode: already rx-mode pending\n");
	} else {
		ECORE_SET_BIT(RAMROD_RX, &ramrod_flags);
		qfle3_set_q_rx_mode(adapter, cl_id, 0, accept_flags, 0,
				    ramrod_flags);
                QFLE3_DBG(QFLE3_DBG_CNIC, "Exit iscsi_eth_rx_mode: Success\n");
	}
}
int qfle3_set_fcoe_eth_rx_mode(struct qfle3_adapter *adapter, vmk_Bool start)
{
	unsigned long accept_flags = 0, ramrod_flags = 0;
   
	vmk_uint8 cl_id = qfle3_fcoe_fp(adapter)->cl_id;
	int sched_state = ECORE_FILTER_FCOE_ETH_STOP_SCHED;

        QFLE3_DBG(QFLE3_DBG_CNIC, "Enter fcoe_eth_rx_mode: start=0x%x\n", start);	
     if (adapter->recovery_state != QFLE3_RECOVERY_DONE){
        ECORE_SET_BIT(RAMROD_DRV_CLR_ONLY, &ramrod_flags);
     }

   if (start) {
		/* Start accepting on FCoE L2 ring */
		ECORE_SET_BIT(ECORE_ACCEPT_UNICAST, &accept_flags);
		ECORE_SET_BIT(ECORE_ACCEPT_MULTICAST, &accept_flags);
		ECORE_SET_BIT(ECORE_ACCEPT_ANY_VLAN, &accept_flags);

		/* Clear STOP_PENDING bit if START is requested */
		ECORE_CLEAR_BIT(ECORE_FILTER_FCOE_ETH_STOP_SCHED, &adapter->sp_state);

		sched_state = ECORE_FILTER_FCOE_ETH_START_SCHED;
	} else
		/* Clear START_PENDING bit if STOP is requested  */
		ECORE_CLEAR_BIT(ECORE_FILTER_FCOE_ETH_START_SCHED, &adapter->sp_state);

	if (ECORE_TEST_BIT(ECORE_FILTER_RX_MODE_PENDING, &adapter->sp_state)) {
		ECORE_SET_BIT(sched_state, &adapter->sp_state);
                QFLE3_DBG(QFLE3_DBG_CNIC, "Exit fcoe_eth_rx_mode: already rx-mode pending\n");
		return VMK_FALSE;
	} else {
		ECORE_SET_BIT(RAMROD_RX, &ramrod_flags);
      
                QFLE3_DBG(QFLE3_DBG_CNIC, "fcoe_eth_rx_mode: flags 0x%lx\n", accept_flags);  
		qfle3_set_q_rx_mode(adapter, cl_id, 0, accept_flags, 0,
				    ramrod_flags);
                QFLE3_DBG(QFLE3_DBG_CNIC, "Exit fcoe_eth_rx_mode: Success\n");  
		return VMK_TRUE;
	}
}
#endif
static void
qfle3_handle_rx_mode_eqe(struct qfle3_adapter *adapter)
{
   ECORE_CLEAR_BIT(ECORE_FILTER_RX_MODE_PENDING, &adapter->sp_state);
   QFLE3_DBG(QFLE3_DBG_SP, "Clear rx mode pending bit, sp_state 0x%lu\n", adapter->sp_state);

   /* Send rx_mode command again if was requested */
   if (ECORE_TEST_AND_CLEAR_BIT(ECORE_FILTER_RX_MODE_SCHED, &adapter->sp_state))
      qfle3_set_storm_rx_mode(adapter);
#ifdef QFLE3_CNIC
   else if (ECORE_TEST_AND_CLEAR_BIT(ECORE_FILTER_ISCSI_ETH_START_SCHED,
			       &adapter->sp_state))
      qfle3_set_iscsi_eth_rx_mode(adapter, VMK_TRUE);
   else if (ECORE_TEST_AND_CLEAR_BIT(ECORE_FILTER_ISCSI_ETH_STOP_SCHED,
			       &adapter->sp_state))
      qfle3_set_iscsi_eth_rx_mode(adapter, VMK_FALSE);
   else if (ECORE_TEST_AND_CLEAR_BIT(ECORE_FILTER_FCOE_ETH_START_SCHED,
			       &adapter->sp_state))
      qfle3_set_fcoe_eth_rx_mode(adapter, VMK_TRUE);
   else if (ECORE_TEST_AND_CLEAR_BIT(ECORE_FILTER_FCOE_ETH_STOP_SCHED,
			       &adapter->sp_state))
      qfle3_set_fcoe_eth_rx_mode(adapter, VMK_FALSE);
#endif
}

static void
qfle3_handle_mcast_eqe(struct qfle3_adapter *adapter)
{
}

static void
qfle3_handle_classification_eqe(struct qfle3_adapter *adapter,
				union event_ring_elem *elem)
{
   unsigned long ramrod_flags = 0;
   int rc = 0;
   vmk_uint32 cid = elem->message.data.eth_event.echo & ECORE_SWCID_MASK;
   struct ecore_vlan_mac_obj *vlan_mac_obj;

   /* Always push next commands out, don't wait here */
   ECORE_SET_BIT(RAMROD_CONT, &ramrod_flags);

   switch (le32toh(elem->message.data.eth_event.echo)
	   >> ECORE_SWCID_SHIFT) {
   case ECORE_FILTER_MAC_PENDING:
      QFLE3_DBG(QFLE3_DBG_SP, "Got SETUP_MAC completions cid = %d\n", cid);
#ifdef QFLE3_CNIC
//      if (CNIC_LOADED(adapter) && 
//      if (cid == QFLE3_ISCSI_ETH_CID(adapter))
//	 vlan_mac_obj = &adapter->iscsi_l2_mac_obj;
    if (cid == QFLE3_FCOE_ETH_CID(adapter))
       vlan_mac_obj = &adapter->sp_objs[FCOE_IDX(adapter)].mac_obj;
      else
#endif
      vlan_mac_obj = &adapter->sp_objs[cid].mac_obj;

      break;
#if 0
   case ECORE_FILTER_VLAN_PENDING:
      QFLE3_DBG(QFLE3_DBG_SP, "Got SETUP_VLAN completions\n");
      vlan_mac_obj = &adapter->sp_objs[cid].vlan_obj;
#endif
      break;
   case ECORE_FILTER_MCAST_PENDING:
      QFLE3_DBG(QFLE3_DBG_SP, "Got SETUP_MCAST completions\n");
      /* This is only relevant for 57710 where multicast MACs are
       * configured as unicast MACs using the same ramrod.
       */
      qfle3_handle_mcast_eqe(adapter);
      return;
   case ECORE_FILTER_VXLAN_PENDING:
      QFLE3_DBG(QFLE3_DBG_SP, "Got SETUP_VXLAN filter completions for cid %d\n", cid);
      vlan_mac_obj = &adapter->sp_objs[cid].vxlan_filter_obj;
      break;
   default:
      QFLE3_ERR("Unsupported classification command: %d\n",
		elem->message.data.eth_event.echo);
      return;
   }

   rc = vlan_mac_obj->complete(adapter, vlan_mac_obj, elem, &ramrod_flags);

   if (rc < 0)
      QFLE3_ERR("Failed to schedule new commands: %d\n", rc);
   else if (rc > 0)
      QFLE3_DBG(QFLE3_DBG_SP, "Scheduled next pending commands...\n");
}

static void
qfle3_eq_int(struct qfle3_adapter *adapter)
{
   vmk_uint16 hw_cons, sw_cons, sw_prod;
   union event_ring_elem *elem;
   vmk_uint8 echo;
   vmk_uint32 cid;
   vmk_uint8 opcode;
#ifdef QFLE3_SRIOV
   int rc;
#endif //QFLE3_SRIOV
   int spqe_cnt = 0;
   struct ecore_queue_sp_obj *q_obj;
   struct ecore_func_sp_obj *f_obj = &adapter->func_obj;
   struct ecore_raw_obj *rss_raw; 
#if (ESX_DDK_VERSION >= 2017)
   vmk_uint32 j,rss_engine_id;
#endif

   hw_cons = le16toh(*adapter->eq_cons_sb);

   /*
    * The hw_cos range is 1-255, 257 - the sw_cons range is 0-254, 256.
    * * when we get the next-page we need to adjust so the loop
    * * condition below will be met. The next element is the size of a
    * * regular element and hence incrementing by 1
    */
   if ((hw_cons & EQ_DESC_MAX_PAGE) == EQ_DESC_MAX_PAGE)
      hw_cons++;

   /*
    * This function may never run in parallel with itself for a
    * * specific adapter, thus there is no need in "paired" read memory
    * * barrier here.
    */
   sw_cons = adapter->eq_cons;
   sw_prod = adapter->eq_prod;

   QFLE3_DBG(QFLE3_DBG_SP, "EQ:  hw_cons %u  sw_cons %u adapter->eq_spq_left %lx\n",
	     hw_cons, sw_cons, vmk_AtomicRead64(&adapter->eq_spq_left));

   for (; sw_cons != hw_cons;
	sw_prod = NEXT_EQ_IDX(sw_prod), sw_cons = NEXT_EQ_IDX(sw_cons)) {

      elem = &adapter->eq_ring[EQ_DESC(sw_cons)];
#ifdef QFLE3_SRIOV
      rc = qfle3_iov_eq_sp_event(adapter, elem);
      if (!rc) {
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: qfle3_iov_eq_sp_event returned %d\n",rc);
         goto next_spqe;
      }
#endif //QFLE3_SRIOV
      /*
       * elem CID originates from FW; actually LE
       */
      cid = SW_CID((__le32)
		   elem->message.data.cfc_del_event.cid);
      opcode = elem->message.opcode;

      /*
       * handle eq element
       */
      switch (opcode) {
#ifdef QFLE3_SRIOV
      case EVENT_RING_OPCODE_VF_PF_CHANNEL:
         QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: BC: Received a VF->PF Backchannel request.\n");
         qfle3_vf_mbx_schedule(adapter,
                              &elem->message.data.vf_pf_event);
         continue;
#else

      case EVENT_RING_OPCODE_VF_PF_CHANNEL:
	 //qfle3_vf_mbx_schedule(adapter,
	 //                     &elem->message.data.vf_pf_event);
	 continue;
#endif
      case EVENT_RING_OPCODE_STAT_QUERY:
	 //QFLE3_DBG_AND((QFLE3_DBG_SP | QFLE3_MSG_STATS),
	 //      "got statistics comp event %d\n",
	 //     adapter->stats_comp++);
	 /*
	  * nothing to do with stats comp
	  */
	 goto next_spqe;

      case EVENT_RING_OPCODE_CFC_DEL:
	 /*
	  * handle according to cid range
	  */
	 /*
	  * we may want to verify here that the adapter state is
	  * HALTING
	  */
	 QFLE3_DBG(QFLE3_DBG_SP, "got delete ramrod for MULTI[%d]\n", cid);
#ifdef QFLE3_CNIC
	 if (CNIC_LOADED(adapter) && !qfle3_cnic_handle_cfc_del(adapter, cid, elem))
	    goto next_spqe;
#endif
	 q_obj = qfle3_cid_to_q_obj(adapter, cid);

	 if (q_obj->complete_cmd(adapter, q_obj, ECORE_Q_CMD_CFC_DEL))
	    break;

	 goto next_spqe;

      case EVENT_RING_OPCODE_STOP_TRAFFIC:
	 QFLE3_DBG(QFLE3_DBG_SP | QFLE3_DBG_DCB, "got STOP TRAFFIC\n");
	 qfle3_dcbx_set_params(adapter, QFLE3_DCBX_STATE_TX_PAUSED);
	 if (f_obj->complete_cmd(adapter, f_obj, ECORE_F_CMD_TX_STOP))
	    break;
	 goto next_spqe;

      case EVENT_RING_OPCODE_START_TRAFFIC:
	 QFLE3_DBG(QFLE3_DBG_SP | QFLE3_DBG_DCB, "got START TRAFFIC\n");
	 qfle3_dcbx_set_params(adapter, QFLE3_DCBX_STATE_TX_RELEASED);
	 if (f_obj->complete_cmd(adapter, f_obj, ECORE_F_CMD_TX_START))
	    break;
	 goto next_spqe;

      case EVENT_RING_OPCODE_FUNCTION_UPDATE:
	 echo = elem->message.data.function_update_event.echo;
	 if (echo == SWITCH_UPDATE) {
	    QFLE3_DBG(QFLE3_DBG_SP, "got FUNC_SWITCH_UPDATE ramrod\n");
	    if (f_obj->complete_cmd(adapter, f_obj, ECORE_F_CMD_SWITCH_UPDATE))
	       break;

#if 0
	 } else {
	    int cmd = QFLE3_SP_RTNL_AFEX_F_UPDATE;

	    QFLE3_DBG(QFLE3_DBG_SP | QFLE3_DBG_MCP,
		      "AFEX: ramrod completed FUNCTION_UPDATE\n");
	    f_obj->complete_cmd(adapter, f_obj, ECORE_F_CMD_AFEX_UPDATE);

	    /*
	     * We will perform the Queues update from
	     * * sp_rtnl task as all Queue SP operations
	     * * should run under rtnl_lock.
	     */
	    qfle3_schedule_sp_rtnl(adapter, cmd, 0);
#endif
	 }

	 goto next_spqe;

      case EVENT_RING_OPCODE_AFEX_VIF_LISTS:
	 f_obj->complete_cmd(adapter, f_obj, ECORE_F_CMD_AFEX_VIFLISTS);
	 //qfle3_after_afex_vif_lists(adapter, elem);
	 goto next_spqe;

      case EVENT_RING_OPCODE_FORWARD_SETUP:
			q_obj = &qfle3_fwd_sp_obj(adapter, q_obj);
			if (q_obj->complete_cmd(adapter, q_obj,
						ECORE_Q_CMD_SETUP_TX_ONLY))
				break;

			goto next_spqe;
      case EVENT_RING_OPCODE_FUNCTION_START:
	 QFLE3_DBG(QFLE3_DBG_SP, "got FUNC_START ramrod\n");
	 if (f_obj->complete_cmd(adapter, f_obj, ECORE_F_CMD_START))
	    break;

	 goto next_spqe;

      case EVENT_RING_OPCODE_FUNCTION_STOP:
	 QFLE3_DBG(QFLE3_DBG_SP, "got FUNC_STOP ramrod\n");
	 if (f_obj->complete_cmd(adapter, f_obj, ECORE_F_CMD_STOP))
	    break;

	 goto next_spqe;
      }

      //Fix this later.
      //switch (opcode | adapter->state) {
      switch (opcode & 0xFFFF) {
	 //case (EVENT_RING_OPCODE_RSS_UPDATE_RULES | QFLE3_STATE_OPEN):
	 //case (EVENT_RING_OPCODE_RSS_UPDATE_RULES | QFLE3_STATE_OPENING_WAIT4_PORT):
	 //case (EVENT_RING_OPCODE_RSS_UPDATE_RULES | QFLE3_STATE_CLOSING_WAIT4_HALT):
      case (EVENT_RING_OPCODE_RSS_UPDATE_RULES):
	 cid = elem->message.data.eth_event.echo & ECORE_SWCID_MASK;
	 QFLE3_DBG(QFLE3_DBG_SP, "got RSS_UPDATE ramrod. CID %d\n", cid);
#if (ESX_DDK_VERSION >= 2017)
         if(cid == 0){
            rss_raw = &adapter->defq_rss_conf_obj.raw;
            rss_raw->clear_pending(rss_raw);
         } else {
            for(rss_engine_id=0; rss_engine_id<(QFLE3_MAX_RSS_ENGINE_IDS-1); rss_engine_id++){
               if(adapter->rss_engine[rss_engine_id]==cid)
                  break;
            }
            if(rss_engine_id >=(QFLE3_MAX_RSS_ENGINE_IDS-1)){
                QFLE3_ERR("RSS Engine ID not configured for Netqueue with cid: %d\n", cid);
                break;
            }
            j = rss_engine_id - (QFLE3_FUNC(adapter) * 2 * QFLE3_DEVICE_MAX_RSS_ENGINE);
            QFLE3_DBG(QFLE3_DBG_SP, "rss_engine_id is %d, j is %d\n", rss_engine_id, j);
            rss_raw = &adapter->netq_rss_conf_obj[j].raw;
            rss_raw->clear_pending(rss_raw);
         }
#else
	 rss_raw = &adapter->netq_rss_conf_obj.raw;
	 rss_raw->clear_pending(rss_raw);
#endif
	 break;

	 //case (EVENT_RING_OPCODE_SET_MAC | QFLE3_STATE_OPEN):
	 //case (EVENT_RING_OPCODE_SET_MAC | QFLE3_STATE_DIAG):
	 //case (EVENT_RING_OPCODE_SET_MAC | QFLE3_STATE_CLOSING_WAIT4_HALT):
	 //case (EVENT_RING_OPCODE_CLASSIFICATION_RULES | QFLE3_STATE_OPEN):
	 //case (EVENT_RING_OPCODE_CLASSIFICATION_RULES | QFLE3_STATE_DIAG):
	 //case (EVENT_RING_OPCODE_CLASSIFICATION_RULES | QFLE3_STATE_CLOSING_WAIT4_HALT):
      case (EVENT_RING_OPCODE_SET_MAC):
      case (EVENT_RING_OPCODE_CLASSIFICATION_RULES):
	 QFLE3_DBG(QFLE3_DBG_SP, "got (un)set vlan/mac ramrod\n");
	 qfle3_handle_classification_eqe(adapter, elem);
	 break;

	 //case (EVENT_RING_OPCODE_MULTICAST_RULES | QFLE3_STATE_OPEN):
	 //case (EVENT_RING_OPCODE_MULTICAST_RULES | QFLE3_STATE_DIAG):
	 //case (EVENT_RING_OPCODE_MULTICAST_RULES | QFLE3_STATE_CLOSING_WAIT4_HALT):
      case (EVENT_RING_OPCODE_MULTICAST_RULES):
	 QFLE3_DBG(QFLE3_DBG_SP, "got mcast ramrod\n");
	 qfle3_handle_mcast_eqe(adapter);
	 break;

	 //case (EVENT_RING_OPCODE_FILTERS_RULES | QFLE3_STATE_OPEN):
	 //case (EVENT_RING_OPCODE_FILTERS_RULES | QFLE3_STATE_DIAG):
	 //case (EVENT_RING_OPCODE_FILTERS_RULES | QFLE3_STATE_CLOSING_WAIT4_HALT):
      case (EVENT_RING_OPCODE_FILTERS_RULES):
	 QFLE3_DBG(QFLE3_DBG_SP, "got rx_mode ramrod\n");
	 qfle3_handle_rx_mode_eqe(adapter);
	 break;
      default:
	 /*
	  * unknown event log error and continue
	  */
	 QFLE3_ERR("Unknown EQ event %d\n", elem->message.opcode);
      }
     next_spqe:
      spqe_cnt++;
   }                            /* for */

   vmk_CPUMemFenceReadWrite();
   vmk_AtomicAdd64(&adapter->eq_spq_left, spqe_cnt);

   adapter->eq_cons = sw_cons;
   adapter->eq_prod = sw_prod;
   /*
    * Make sure that above mem writes were issued towards the memory
    */
   vmk_CPUMemFenceReadWrite();

   /*
    * update producer
    */
   qfle3_update_eq_prod(adapter, adapter->eq_prod);
}

enum storms {
   XSTORM,
   TSTORM,
   CSTORM,
   USTORM,
   MAX_STORMS
};

#define STORMS_NUM 4
#define REGS_IN_ENTRY 4

static inline int
qfle3_get_assert_list_entry(struct qfle3_adapter *adapter,
			    enum storms storm, int entry)
{
   switch (storm) {
   case XSTORM:
      return XSTORM_ASSERT_LIST_OFFSET(entry);
   case TSTORM:
      return TSTORM_ASSERT_LIST_OFFSET(entry);
   case CSTORM:
      return CSTORM_ASSERT_LIST_OFFSET(entry);
   case USTORM:
      return USTORM_ASSERT_LIST_OFFSET(entry);
   case MAX_STORMS:
   default:
      QFLE3_ERR("unknown storm\n");
   }
   return VMK_FAILURE;
}

static int qfle3_mc_assert(struct qfle3_adapter *adapter)
{
   char last_idx;
   int i, j, rc = 0;
   enum storms storm;
   vmk_uint32 regs[REGS_IN_ENTRY];
   vmk_uint32 bar_storm_intmem[STORMS_NUM] = {
      BAR_XSTRORM_INTMEM,
      BAR_TSTRORM_INTMEM,
      BAR_CSTRORM_INTMEM,
      BAR_USTRORM_INTMEM
   };
   vmk_uint32 storm_assert_list_index[STORMS_NUM] = {
      XSTORM_ASSERT_LIST_INDEX_OFFSET,
      TSTORM_ASSERT_LIST_INDEX_OFFSET,
      CSTORM_ASSERT_LIST_INDEX_OFFSET,
      USTORM_ASSERT_LIST_INDEX_OFFSET
   };
   char *storms_string[STORMS_NUM] = {
      "XSTORM",
      "TSTORM",
      "CSTORM",
      "USTORM"
   };

   for (storm = XSTORM; storm < MAX_STORMS; storm++) {
      last_idx = REG_RD8(adapter, bar_storm_intmem[storm] +
			 storm_assert_list_index[storm]);
      if (last_idx)
	 QFLE3_ERR("%s_ASSERT_LIST_INDEX 0x%x\n",
		   storms_string[storm], last_idx);
      /* print the asserts */
      for (i = 0; i < STROM_ASSERT_ARRAY_SIZE; i++) {

	 /* read a single assert entry */
	 for (j = 0; j < REGS_IN_ENTRY; j++)
	    regs[j] = REG_RD(adapter, bar_storm_intmem[storm] +
			     qfle3_get_assert_list_entry(adapter,
							 storm,
							 i) +
			     sizeof(vmk_uint32) * j);

	 /* log entry if it contains a valid assert */
	 if (regs[0] != COMMON_ASM_INVALID_ASSERT_OPCODE) {
	    QFLE3_ERR("%s_ASSERT_INDEX 0x%x = 0x%08x 0x%08x 0x%08x 0x%08x\n",
		      storms_string[storm], i, regs[3],
		      regs[2], regs[1], regs[0]);
	    rc++;
	 } else {
	    break;
	 }
      }
   }

   QFLE3_INFO("Chip Revision: %s, FW Version: %d_%d_%d\n",
	     CHIP_IS_E2(adapter) ? "everest2" : "everest3",
	     BCM_5710_FW_MAJOR_VERSION,
	     BCM_5710_FW_MINOR_VERSION,
	     BCM_5710_FW_REVISION_VERSION);

//   VMK_ASSERT(0);
   return rc;
}

static void
_print_parity(struct qfle3_adapter *adapter, vmk_uint32 reg)
{
   QFLE3_ERROR(" [0x%08x] ", REG_RD(adapter, reg));
}

static void
_print_next_block(int idx, const char *blk)
{
   QFLE3_ERROR("%s%s", idx ? ", " : "", blk);
}

static vmk_Bool
qfle3_check_blocks_with_parity0(struct qfle3_adapter *adapter, vmk_uint32 sig,
				int *par_num, vmk_Bool print)
{
   vmk_uint32 cur_bit;
   vmk_Bool res;
   int i;

   res = VMK_FALSE;

   for (i = 0; sig; i++) {
      cur_bit = (0x1UL << i);
      if (sig & cur_bit) {
	 res |= VMK_TRUE;       /* Each bit is real error! */

	 if (print) {
	    switch (cur_bit) {
	    case AEU_INPUTS_ATTN_BITS_BRB_PARITY_ERROR:
	       _print_next_block((*par_num)++, "BRB");
	       _print_parity(adapter, BRB1_REG_BRB1_PRTY_STS);
	       break;
	    case AEU_INPUTS_ATTN_BITS_PARSER_PARITY_ERROR:
	       _print_next_block((*par_num)++, "PARSER");
	       _print_parity(adapter, PRS_REG_PRS_PRTY_STS);
	       break;
	    case AEU_INPUTS_ATTN_BITS_TSDM_PARITY_ERROR:
	       _print_next_block((*par_num)++, "TSDM");
	       _print_parity(adapter, TSDM_REG_TSDM_PRTY_STS);
	       break;
	    case AEU_INPUTS_ATTN_BITS_SEARCHER_PARITY_ERROR:
	       _print_next_block((*par_num)++, "SEARCHER");
	       _print_parity(adapter, SRC_REG_SRC_PRTY_STS);
	       break;
	    case AEU_INPUTS_ATTN_BITS_TCM_PARITY_ERROR:
	       _print_next_block((*par_num)++, "TCM");
	       _print_parity(adapter, TCM_REG_TCM_PRTY_STS);
	       break;
	    case AEU_INPUTS_ATTN_BITS_TSEMI_PARITY_ERROR:
	       _print_next_block((*par_num)++, "TSEMI");
	       _print_parity(adapter, TSEM_REG_TSEM_PRTY_STS_0);
	       _print_parity(adapter, TSEM_REG_TSEM_PRTY_STS_1);
	       break;
	    case AEU_INPUTS_ATTN_BITS_PBCLIENT_PARITY_ERROR:
	       _print_next_block((*par_num)++, "XPB");
	       _print_parity(adapter, GRCBASE_XPB + PB_REG_PB_PRTY_STS);
	       break;
	    }
	 }

	 /*
	  * Clear the bit
	  */
	 sig &= ~cur_bit;
      }
   }

   return res;
}

static vmk_Bool
qfle3_check_blocks_with_parity1(struct qfle3_adapter *adapter, vmk_uint32 sig,
				int *par_num, vmk_Bool * global, vmk_Bool print)
{
   vmk_uint32 cur_bit;
   vmk_Bool res;
   int i;

   res = VMK_FALSE;

   for (i = 0; sig; i++) {
      cur_bit = (0x1UL << i);
      if (sig & cur_bit) {
	 res |= VMK_TRUE;       /* Each bit is real error! */
	 switch (cur_bit) {
	 case AEU_INPUTS_ATTN_BITS_PBF_PARITY_ERROR:
	    if (print) {
	       _print_next_block((*par_num)++, "PBF");
	       _print_parity(adapter, PBF_REG_PBF_PRTY_STS);
	    }
	    break;
	 case AEU_INPUTS_ATTN_BITS_QM_PARITY_ERROR:
	    if (print) {
	       _print_next_block((*par_num)++, "QM");
	       _print_parity(adapter, QM_REG_QM_PRTY_STS);
	    }
	    break;
	 case AEU_INPUTS_ATTN_BITS_TIMERS_PARITY_ERROR:
	    if (print) {
	       _print_next_block((*par_num)++, "TM");
	       _print_parity(adapter, TM_REG_TM_PRTY_STS);
	    }
	    break;
	 case AEU_INPUTS_ATTN_BITS_XSDM_PARITY_ERROR:
	    if (print) {
	       _print_next_block((*par_num)++, "XSDM");
	       _print_parity(adapter, XSDM_REG_XSDM_PRTY_STS);
	    }
	    break;
	 case AEU_INPUTS_ATTN_BITS_XCM_PARITY_ERROR:
	    if (print) {
	       _print_next_block((*par_num)++, "XCM");
	       _print_parity(adapter, XCM_REG_XCM_PRTY_STS);
	    }
	    break;
	 case AEU_INPUTS_ATTN_BITS_XSEMI_PARITY_ERROR:
	    if (print) {
	       _print_next_block((*par_num)++, "XSEMI");
	       _print_parity(adapter, XSEM_REG_XSEM_PRTY_STS_0);
	       _print_parity(adapter, XSEM_REG_XSEM_PRTY_STS_1);
	    }
	    break;
	 case AEU_INPUTS_ATTN_BITS_DOORBELLQ_PARITY_ERROR:
	    if (print) {
	       _print_next_block((*par_num)++, "DOORBELLQ");
	       _print_parity(adapter, DORQ_REG_DORQ_PRTY_STS);
	    }
	    break;
	 case AEU_INPUTS_ATTN_BITS_NIG_PARITY_ERROR:
	    if (print) {
	       _print_next_block((*par_num)++, "NIG");
	       if (CHIP_IS_E1x(adapter)) {
		  _print_parity(adapter, NIG_REG_NIG_PRTY_STS);
	       } else {
		  _print_parity(adapter, NIG_REG_NIG_PRTY_STS_0);
		  _print_parity(adapter, NIG_REG_NIG_PRTY_STS_1);
	       }
	    }
	    break;
	 case AEU_INPUTS_ATTN_BITS_VAUX_PCI_CORE_PARITY_ERROR:
	    if (print)
	       _print_next_block((*par_num)++, "VAUX PCI CORE");
	    *global = VMK_TRUE;
	    break;
	 case AEU_INPUTS_ATTN_BITS_DEBUG_PARITY_ERROR:
	    if (print) {
	       _print_next_block((*par_num)++, "DEBUG");
	       _print_parity(adapter, DBG_REG_DBG_PRTY_STS);
	    }
	    break;
	 case AEU_INPUTS_ATTN_BITS_USDM_PARITY_ERROR:
	    if (print) {
	       _print_next_block((*par_num)++, "USDM");
	       _print_parity(adapter, USDM_REG_USDM_PRTY_STS);
	    }
	    break;
	 case AEU_INPUTS_ATTN_BITS_UCM_PARITY_ERROR:
	    if (print) {
	       _print_next_block((*par_num)++, "UCM");
	       _print_parity(adapter, UCM_REG_UCM_PRTY_STS);
	    }
	    break;
	 case AEU_INPUTS_ATTN_BITS_USEMI_PARITY_ERROR:
	    if (print) {
	       _print_next_block((*par_num)++, "USEMI");
	       _print_parity(adapter, USEM_REG_USEM_PRTY_STS_0);
	       _print_parity(adapter, USEM_REG_USEM_PRTY_STS_1);
	    }
	    break;
	 case AEU_INPUTS_ATTN_BITS_UPB_PARITY_ERROR:
	    if (print) {
	       _print_next_block((*par_num)++, "UPB");
	       _print_parity(adapter, GRCBASE_UPB + PB_REG_PB_PRTY_STS);
	    }
	    break;
	 case AEU_INPUTS_ATTN_BITS_CSDM_PARITY_ERROR:
	    if (print) {
	       _print_next_block((*par_num)++, "CSDM");
	       _print_parity(adapter, CSDM_REG_CSDM_PRTY_STS);
	    }
	    break;
	 case AEU_INPUTS_ATTN_BITS_CCM_PARITY_ERROR:
	    if (print) {
	       _print_next_block((*par_num)++, "CCM");
	       _print_parity(adapter, CCM_REG_CCM_PRTY_STS);
	    }
	    break;
	 }

	 /*
	  * Clear the bit
	  */
	 sig &= ~cur_bit;
      }
   }

   return res;
}

static vmk_Bool
qfle3_check_blocks_with_parity2(struct qfle3_adapter *adapter,
				vmk_uint32 sig, int *par_num, vmk_Bool print)
{
   vmk_uint32 cur_bit;
   vmk_Bool res;
   int i;

   res = VMK_FALSE;

   for (i = 0; sig; i++) {
      cur_bit = (0x1UL << i);
      if (sig & cur_bit) {
	 res = VMK_TRUE;        /* Each bit is real error! */
	 if (print) {
	    switch (cur_bit) {

	    case AEU_INPUTS_ATTN_BITS_CSEMI_PARITY_ERROR:
	       _print_next_block((*par_num)++, "CSEMI");
	       _print_parity(adapter, CSEM_REG_CSEM_PRTY_STS_0);
	       _print_parity(adapter, CSEM_REG_CSEM_PRTY_STS_1);
	       break;
	    case AEU_INPUTS_ATTN_BITS_PXP_PARITY_ERROR:
	       _print_next_block((*par_num)++, "PXP");
	       _print_parity(adapter, PXP_REG_PXP_PRTY_STS);
	       _print_parity(adapter, PXP2_REG_PXP2_PRTY_STS_0);
	       _print_parity(adapter, PXP2_REG_PXP2_PRTY_STS_1);
	       break;
	    case AEU_INPUTS_ATTN_BITS_PXPPCICLOCKCLIENT_PARITY_ERROR:
	       _print_next_block((*par_num)++, "PXPPCICLOCKCLIENT");
	       break;

	    case AEU_INPUTS_ATTN_BITS_CFC_PARITY_ERROR:
	       _print_next_block((*par_num)++, "CFC");
	       _print_parity(adapter, CFC_REG_CFC_PRTY_STS);
	       break;
	    case AEU_INPUTS_ATTN_BITS_CDU_PARITY_ERROR:
	       _print_next_block((*par_num)++, "CDU");
	       _print_parity(adapter, CDU_REG_CDU_PRTY_STS);
	       break;
	    case AEU_INPUTS_ATTN_BITS_DMAE_PARITY_ERROR:
	       _print_next_block((*par_num)++, "DMAE");
	       _print_parity(adapter, DMAE_REG_DMAE_PRTY_STS);
	       break;
	    case AEU_INPUTS_ATTN_BITS_IGU_PARITY_ERROR:
	       _print_next_block((*par_num)++, "IGU");
	       if (CHIP_IS_E1x(adapter))
		  _print_parity(adapter, HC_REG_HC_PRTY_STS);
	       else
		  _print_parity(adapter, IGU_REG_IGU_PRTY_STS);
	       break;
	    case AEU_INPUTS_ATTN_BITS_MISC_PARITY_ERROR:
	       _print_next_block((*par_num)++, "MISC");
	       _print_parity(adapter, MISC_REG_MISC_PRTY_STS);
	       break;
	    }
	 }

	 /*
	  * Clear the bit
	  */
	 sig &= ~cur_bit;
      }
   }

   return res;
}

static vmk_Bool
qfle3_check_blocks_with_parity3(struct qfle3_adapter *adapter,
				vmk_uint32 sig, int *par_num, vmk_Bool * global,
				vmk_Bool print)
{
   vmk_Bool res = VMK_FALSE;
   vmk_uint32 cur_bit;
   int i;

   for (i = 0; sig; i++) {
      cur_bit = (0x1UL << i);
      if (sig & cur_bit) {
	 switch (cur_bit) {
	 case AEU_INPUTS_ATTN_BITS_MCP_LATCHED_ROM_PARITY:
	    if (print)
	       _print_next_block((*par_num)++, "MCP ROM");
	    *global = VMK_TRUE;
	    res = VMK_TRUE;
	    break;
	 case AEU_INPUTS_ATTN_BITS_MCP_LATCHED_UMP_RX_PARITY:
	    if (print)
	       _print_next_block((*par_num)++, "MCP UMP RX");
	    *global = VMK_TRUE;
	    res = VMK_TRUE;
	    break;
	 case AEU_INPUTS_ATTN_BITS_MCP_LATCHED_UMP_TX_PARITY:
	    if (print)
	       _print_next_block((*par_num)++, "MCP UMP TX");
	    *global = VMK_TRUE;
	    res = VMK_TRUE;
	    break;
	 case AEU_INPUTS_ATTN_BITS_MCP_LATCHED_SCPAD_PARITY:
	    (*par_num)++;
	    /*
	     * clear latched SCPAD PATIRY from MCP
	     */
	    REG_WR(adapter, MISC_REG_AEU_CLR_LATCH_SIGNAL, 1UL << 10);
	    break;
	 }

	 /*
	  * Clear the bit
	  */
	 sig &= ~cur_bit;
      }
   }

   return res;
}

static vmk_Bool
qfle3_check_blocks_with_parity4(struct qfle3_adapter *adapter, vmk_uint32 sig,
				int *par_num, vmk_Bool print)
{
   vmk_uint32 cur_bit;
   vmk_Bool res;
   int i;

   res = VMK_FALSE;

   for (i = 0; sig; i++) {
      cur_bit = (0x1UL << i);
      if (sig & cur_bit) {
	 res = VMK_TRUE;        /* Each bit is real error! */
	 if (print) {
	    switch (cur_bit) {
	    case AEU_INPUTS_ATTN_BITS_PGLUE_PARITY_ERROR:
	       _print_next_block((*par_num)++, "PGLUE_B");
	       _print_parity(adapter, PGLUE_B_REG_PGLUE_B_PRTY_STS);
	       break;
	    case AEU_INPUTS_ATTN_BITS_ATC_PARITY_ERROR:
	       _print_next_block((*par_num)++, "ATC");
	       _print_parity(adapter, ATC_REG_ATC_PRTY_STS);
	       break;
	    }
	 }
	 /*
	  * Clear the bit
	  */
	 sig &= ~cur_bit;
      }
   }

   return res;
}

static vmk_Bool
qfle3_parity_attn(struct qfle3_adapter *adapter, vmk_Bool * global, vmk_Bool print,
		  vmk_uint32 * sig)
{
   vmk_Bool res = VMK_FALSE;

   if ((sig[0] & HW_PRTY_ASSERT_SET_0) ||
       (sig[1] & HW_PRTY_ASSERT_SET_1) ||
       (sig[2] & HW_PRTY_ASSERT_SET_2) ||
       (sig[3] & HW_PRTY_ASSERT_SET_3) || (sig[4] & HW_PRTY_ASSERT_SET_4)) {
      int par_num = 0;

      QFLE3_DBG(QFLE3_DBG_HW, "Was parity error: HW block parity attention:\n"
		"[0]:0x%08x [1]:0x%08x [2]:0x%08x [3]:0x%08lx [4]:0x%08x\n",
		sig[0] & HW_PRTY_ASSERT_SET_0,
		sig[1] & HW_PRTY_ASSERT_SET_1,
		sig[2] & HW_PRTY_ASSERT_SET_2,
		sig[3] & HW_PRTY_ASSERT_SET_3, sig[4] & HW_PRTY_ASSERT_SET_4);
      if (print) {
	 if (((sig[0] & HW_PRTY_ASSERT_SET_0) ||
	      (sig[1] & HW_PRTY_ASSERT_SET_1) ||
	      (sig[2] & HW_PRTY_ASSERT_SET_2) ||
	      (sig[4] & HW_PRTY_ASSERT_SET_4)) ||
	     (sig[3] & HW_PRTY_ASSERT_SET_3_WITHOUT_SCPAD)) {
	    QFLE3_ERR("Parity errors detected in blocks: ");
	 } else {
	    print = VMK_FALSE;
	 }
      }

      res |= qfle3_check_blocks_with_parity0(adapter,
					     sig[0] & HW_PRTY_ASSERT_SET_0, &par_num,
					     print);
      res |=
	 qfle3_check_blocks_with_parity1(adapter, sig[1] & HW_PRTY_ASSERT_SET_1,
					 &par_num, global, print);
      res |=
	 qfle3_check_blocks_with_parity2(adapter, sig[2] & HW_PRTY_ASSERT_SET_2,
					 &par_num, print);
      res |=
	 qfle3_check_blocks_with_parity3(adapter, sig[3] & HW_PRTY_ASSERT_SET_3,
					 &par_num, global, print);
      res |=
	 qfle3_check_blocks_with_parity4(adapter, sig[4] & HW_PRTY_ASSERT_SET_4,
					 &par_num, print);

      if (print)
	 QFLE3_INFO("\n");
   }

   return res;
}

vmk_Bool
qfle3_chk_parity_attn(struct qfle3_adapter * adapter, vmk_Bool * global, vmk_Bool print)
{
   struct attn_route attn = { {0} };
   int port = QFLE3_PORT(adapter);

   attn.sig[0] = REG_RD(adapter, MISC_REG_AEU_AFTER_INVERT_1_FUNC_0 + port * 4);
   attn.sig[1] = REG_RD(adapter, MISC_REG_AEU_AFTER_INVERT_2_FUNC_0 + port * 4);
   attn.sig[2] = REG_RD(adapter, MISC_REG_AEU_AFTER_INVERT_3_FUNC_0 + port * 4);
   attn.sig[3] = REG_RD(adapter, MISC_REG_AEU_AFTER_INVERT_4_FUNC_0 + port * 4);
   /*
    * Since MCP attentions can't be disabled inside the block, we need to
    * * read AEU registers to see whether they're currently disabled
    */
   attn.sig[3] &= ((REG_RD(adapter,
			   !port ? MISC_REG_AEU_ENABLE4_FUNC_0_OUT_0
			   : MISC_REG_AEU_ENABLE4_FUNC_1_OUT_0) &
		    MISC_AEU_ENABLE_MCP_PRTY_BITS) | ~MISC_AEU_ENABLE_MCP_PRTY_BITS);

   if (!CHIP_IS_E1x(adapter))
      attn.sig[4] = REG_RD(adapter, MISC_REG_AEU_AFTER_INVERT_5_FUNC_0 + port * 4);

   return qfle3_parity_attn(adapter, global, print, attn.sig);
}

static void
qfle3_pmf_update(struct qfle3_adapter *adapter)
{
   int port = QFLE3_PORT(adapter);
   vmk_uint32 val;

   adapter->port.pmf = 1;
   QFLE3_DBG(QFLE3_DBG_MCP, "pmf %d\n", adapter->port.pmf);

   /*
    * We need the barrier to ensure the ordering between the writing to
    * adapter->port.pmf here and reading it from the qfle3_periodic_task_func().
    */
   vmk_CPUMemFenceReadWrite();

   /* queue a periodic task */
   qfle3_schedule_helper(adapter,
			 adapter->periodic_task, qfle3_periodic_task_func, 0);

   qfle3_dcbx_pmf_update(adapter);

   /* enable nig attention */
   val = (0xff0f | (1 << (QFLE3_VN(adapter) + 4)));
   if (adapter->hw_info.int_block == INT_BLOCK_HC) {
      REG_WR(adapter, HC_REG_TRAILING_EDGE_0 + port*8, val);
      REG_WR(adapter, HC_REG_LEADING_EDGE_0 + port*8, val);
   } else {
      REG_WR(adapter, IGU_REG_TRAILING_EDGE_LATCH, val);
      REG_WR(adapter, IGU_REG_LEADING_EDGE_LATCH, val);
   }
   
   QFLE3_DBG(QFLE3_DBG_STATS, "Executing qfle3_stats_handle for STATS_EVENT_PMF\n");
   qfle3_stats_handle(adapter, STATS_EVENT_PMF);
}

/**
 * qfle3_get_leader_lock_resource - get the recovery leader resource id
 *
 * @adapter:	driver handle
 *
 * Returns the recovery leader resource id according to the engine this function
 * belongs to. Currently only only 2 engines is supported.
 */
static int qfle3_get_leader_lock_resource(struct qfle3_adapter *adapter)
{
   if (QFLE3_PATH(adapter))
      return HW_LOCK_RESOURCE_RECOVERY_LEADER_1;
   else
      return HW_LOCK_RESOURCE_RECOVERY_LEADER_0;
}

VMK_ReturnStatus qfle3_release_leader_lock(struct qfle3_adapter *adapter)
{
   return qfle3_release_hw_lock(adapter, qfle3_get_leader_lock_resource(adapter));
}


/* Return true if succeeded to acquire the lock */
static vmk_Bool qfle3_trylock_hw_lock(struct qfle3_adapter *adapter, vmk_uint32 resource)
{
   vmk_uint32 lock_status;
   vmk_uint32 resource_bit = (1 << resource);
   vmk_uint32 func = QFLE3_FUNC(adapter);
   vmk_uint32 hw_lock_control_reg;

   QFLE3_DBG(QFLE3_DBG_HW,
	     "Trying to take a lock on resource %d\n", resource);

   /* Validating that the resource is within range */
   if (resource > HW_LOCK_MAX_RESOURCE_VALUE) {
      QFLE3_DBG(QFLE3_DBG_HW,
		"resource(0x%x) > HW_LOCK_MAX_RESOURCE_VALUE(0x%x)\n",
		resource, HW_LOCK_MAX_RESOURCE_VALUE);
      return VMK_FALSE;
   }

   if (func <= 5)
      hw_lock_control_reg = (MISC_REG_DRIVER_CONTROL_1 + func*8);
   else
      hw_lock_control_reg =
	 (MISC_REG_DRIVER_CONTROL_7 + (func - 6)*8);

   /* Try to acquire the lock */
   REG_WR(adapter, hw_lock_control_reg + 4, resource_bit);
   lock_status = REG_RD(adapter, hw_lock_control_reg);
   if (lock_status & resource_bit)
      return VMK_TRUE;

   QFLE3_DBG(QFLE3_DBG_HW,
	     "Failed to get a lock on resource %d\n", resource);
   return VMK_FALSE;
}

/**
 * qfle3_trylock_leader_lock- try to acquire a leader lock.
 *
 * @adapter: driver handle
 *
 * Tries to acquire a leader lock for current engine.
 */
vmk_Bool qfle3_trylock_leader_lock(struct qfle3_adapter *adapter)
{
   return qfle3_trylock_hw_lock(adapter, qfle3_get_leader_lock_resource(adapter));
}

/*
 * Set RESET_IN_PROGRESS for the current engine.
 *
 * should be run under rtnl lock
 */
void qfle3_set_reset_in_progress(struct qfle3_adapter *adapter)
{
   vmk_uint32 val;
   vmk_uint32 bit = QFLE3_PATH(adapter) ?
      QFLE3_PATH1_RST_IN_PROG_BIT : QFLE3_PATH0_RST_IN_PROG_BIT;
   qfle3_acquire_hw_lock(adapter, HW_LOCK_RESOURCE_RECOVERY_REG);
   val = REG_RD(adapter, QFLE3_RECOVERY_GLOB_REG);

   /* Set the bit */
   val |= bit;
   REG_WR(adapter, QFLE3_RECOVERY_GLOB_REG, val);
   qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_RECOVERY_REG);
}
/*
 * Set the GLOBAL_RESET bit.
 *
 * Should be run under rtnl lock
 */
void qfle3_set_reset_global(struct qfle3_adapter *adapter)
{
   vmk_uint32 val;
   qfle3_acquire_hw_lock(adapter, HW_LOCK_RESOURCE_RECOVERY_REG);
   val = REG_RD(adapter, QFLE3_RECOVERY_GLOB_REG);
   REG_WR(adapter, QFLE3_RECOVERY_GLOB_REG, val | QFLE3_GLOBAL_RESET_BIT);
   qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_RECOVERY_REG);
}
/*
 * Checks the GLOBAL_RESET bit.
 *
 * should be run under rtnl lock
 */
static vmk_Bool  qfle3_reset_is_global(struct qfle3_adapter *adapter)
{
   vmk_uint32 val = REG_RD(adapter, QFLE3_RECOVERY_GLOB_REG);
   QFLE3_DBG(QFLE3_DBG_HW, "GEN_REG_VAL=0x%08x\n", val);
   return (val & QFLE3_GLOBAL_RESET_BIT) ? VMK_TRUE : VMK_FALSE;
}

/*
 * Clear the GLOBAL_RESET bit.
 *
 * Should be run under rtnl lock
 */
static void qfle3_clear_reset_global(struct qfle3_adapter *adapter)
{
   vmk_uint32 val;
   qfle3_acquire_hw_lock(adapter, HW_LOCK_RESOURCE_RECOVERY_REG);
   val = REG_RD(adapter, QFLE3_RECOVERY_GLOB_REG);
   REG_WR(adapter, QFLE3_RECOVERY_GLOB_REG, val & (~QFLE3_GLOBAL_RESET_BIT));
   qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_RECOVERY_REG);
}

/*
 * Clear RESET_IN_PROGRESS bit for the current engine.
 *
 * Should be run under rtnl lock
 */
static void qfle3_set_reset_done(struct qfle3_adapter *adapter)
{
   vmk_uint32 val;
   vmk_uint32 bit = QFLE3_PATH(adapter) ?
      QFLE3_PATH1_RST_IN_PROG_BIT : QFLE3_PATH0_RST_IN_PROG_BIT;
   qfle3_acquire_hw_lock(adapter, HW_LOCK_RESOURCE_RECOVERY_REG);
   val = REG_RD(adapter, QFLE3_RECOVERY_GLOB_REG);

   /* Clear the bit */
   val &= ~bit;
   REG_WR(adapter, QFLE3_RECOVERY_GLOB_REG, val);

   qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_RECOVERY_REG);
}

/* Close gates #2, #3 and #4: */
static void qfle3_set_234_gates(struct qfle3_adapter *adapter, vmk_Bool close)
{
   vmk_uint32 val;

   /* Gates #2 and #4a are closed/opened for "not E1" only */
   if (!CHIP_IS_E1(adapter)) {
      /* #4 */
      REG_WR(adapter, PXP_REG_HST_DISCARD_DOORBELLS, !!close);
      /* #2 */
      REG_WR(adapter, PXP_REG_HST_DISCARD_INTERNAL_WRITES, !!close);
   }

   /* #3 */
   if (CHIP_IS_E1x(adapter)) {
      /* Prevent interrupts from HC on both ports */
      val = REG_RD(adapter, HC_REG_CONFIG_1);
      REG_WR(adapter, HC_REG_CONFIG_1,
	     (!close) ? (val | HC_CONFIG_1_REG_BLOCK_DISABLE_1) :
	     (val & ~(vmk_uint32)HC_CONFIG_1_REG_BLOCK_DISABLE_1));

      val = REG_RD(adapter, HC_REG_CONFIG_0);
      REG_WR(adapter, HC_REG_CONFIG_0,
	     (!close) ? (val | HC_CONFIG_0_REG_BLOCK_DISABLE_0) :
	     (val & ~(vmk_uint32)HC_CONFIG_0_REG_BLOCK_DISABLE_0));
   } else {
      /* Prevent incoming interrupts in IGU */
      val = REG_RD(adapter, IGU_REG_BLOCK_CONFIGURATION);

      REG_WR(adapter, IGU_REG_BLOCK_CONFIGURATION,
	     (!close) ?
	     (val | IGU_BLOCK_CONFIGURATION_REG_BLOCK_ENABLE) :
	     (val & ~(vmk_uint32)IGU_BLOCK_CONFIGURATION_REG_BLOCK_ENABLE));
   }

   QFLE3_DBG(QFLE3_DBG_HW, "%s gates #2, #3 and #4\n",
	     close ? "closing" : "opening");
   vmk_CPUMemFenceReadWrite ();
}
/*
 * Reset the whole chip except for:
 *      - PCIE core
 *      - PCI Glue, PSWHST, PXP/PXP2 RF (all controlled by
 *              one reset bit)
 *      - IGU
 *      - MISC (including AEU)
 *      - GRC
 *      - RBCN, RBCP
 */
static void qfle3_process_kill_chip_reset(struct qfle3_adapter *adapter, vmk_Bool global)
{
   vmk_uint32 not_reset_mask1, reset_mask1, not_reset_mask2, reset_mask2;
   vmk_uint32 global_bits2, stay_reset2;

   /*
    * Bits that have to be set in reset_mask2 if we want to reset 'global'
    * (per chip) blocks.
    */
   global_bits2 =
      MISC_REGISTERS_RESET_REG_2_RST_MCP_N_RESET_CMN_CPU |
      MISC_REGISTERS_RESET_REG_2_RST_MCP_N_RESET_CMN_CORE;

   /* Don't reset the following blocks.
    * Important: per port blocks (such as EMAC, BMAC, UMAC) can't be
    *            reset, as in 4 port device they might still be owned
    *            by the MCP (there is only one leader per path).
    */
   not_reset_mask1 =
      MISC_REGISTERS_RESET_REG_1_RST_HC |
      MISC_REGISTERS_RESET_REG_1_RST_PXPV |
      MISC_REGISTERS_RESET_REG_1_RST_PXP;

   not_reset_mask2 =
      MISC_REGISTERS_RESET_REG_2_RST_PCI_MDIO |
      MISC_REGISTERS_RESET_REG_2_RST_EMAC0_HARD_CORE |
      MISC_REGISTERS_RESET_REG_2_RST_EMAC1_HARD_CORE |
      MISC_REGISTERS_RESET_REG_2_RST_MISC_CORE |
      MISC_REGISTERS_RESET_REG_2_RST_RBCN |
      MISC_REGISTERS_RESET_REG_2_RST_GRC  |
      MISC_REGISTERS_RESET_REG_2_RST_MCP_N_RESET_REG_HARD_CORE |
      MISC_REGISTERS_RESET_REG_2_RST_MCP_N_HARD_CORE_RST_B |
      MISC_REGISTERS_RESET_REG_2_RST_ATC |
      MISC_REGISTERS_RESET_REG_2_PGLC |
      MISC_REGISTERS_RESET_REG_2_RST_BMAC0 |
      MISC_REGISTERS_RESET_REG_2_RST_BMAC1 |
      MISC_REGISTERS_RESET_REG_2_RST_EMAC0 |
      MISC_REGISTERS_RESET_REG_2_RST_EMAC1 |
      MISC_REGISTERS_RESET_REG_2_UMAC0 |
      MISC_REGISTERS_RESET_REG_2_UMAC1;

   /*
    * Keep the following blocks in reset:
    *  - all xxMACs are handled by the qfle3_link code.
    */
   stay_reset2 =
      MISC_REGISTERS_RESET_REG_2_XMAC |
      MISC_REGISTERS_RESET_REG_2_XMAC_SOFT;

   /* Full reset masks according to the chip */
   reset_mask1 = 0xffffffff;

   if (CHIP_IS_E1(adapter))
      reset_mask2 = 0xffff;
   else if (CHIP_IS_E1H(adapter))
      reset_mask2 = 0x1ffff;
   else if (CHIP_IS_E2(adapter))
      reset_mask2 = 0xfffff;
   else /* CHIP_IS_E3 */
      reset_mask2 = 0x3ffffff;

   /* Don't reset global blocks unless we need to */
   if (!global)
      reset_mask2 &= ~global_bits2;

   /*
    * In case of attention in the QM, we need to reset PXP
    * (MISC_REGISTERS_RESET_REG_2_RST_PXP_RQ_RD_WR) before QM
    * because otherwise QM reset would release 'close the gates' shortly
    * before resetting the PXP, then the PSWRQ would send a write
    * request to PGLUE. Then when PXP is reset, PGLUE would try to
    * read the payload data from PSWWR, but PSWWR would not
    * respond. The write queue in PGLUE would stuck, dmae commands
    * would not return. Therefore it's important to reset the second
    * reset register (containing the
    * MISC_REGISTERS_RESET_REG_2_RST_PXP_RQ_RD_WR bit) before the
    * first one (containing the MISC_REGISTERS_RESET_REG_1_RST_QM
    * bit).
    */
   REG_WR(adapter, GRCBASE_MISC + MISC_REGISTERS_RESET_REG_2_CLEAR,
	  reset_mask2 & (~not_reset_mask2));

   REG_WR(adapter, GRCBASE_MISC + MISC_REGISTERS_RESET_REG_1_CLEAR,
	  reset_mask1 & (~not_reset_mask1));

   vmk_CPUMemFenceReadWrite ();

   REG_WR(adapter, GRCBASE_MISC + MISC_REGISTERS_RESET_REG_2_SET,
	  reset_mask2 & (~stay_reset2));

   vmk_CPUMemFenceReadWrite ();

   REG_WR(adapter, GRCBASE_MISC + MISC_REGISTERS_RESET_REG_1_SET, reset_mask1);
   vmk_CPUMemFenceReadWrite ();
}

/**
 * qfle3_er_poll_igu_vq - poll for pending writes bit.
 * It should get cleared in no more than 1s.
 *
 * @adapter:	driver handle
 *
 * It should get cleared in no more than 1s. Returns 0 if
 * pending writes bit gets cleared.
 */
static int qfle3_er_poll_igu_vq(struct qfle3_adapter *adapter)
{
   vmk_uint32 cnt = 1000;
   vmk_uint32 pend_bits = 0;

   do {
      pend_bits  = REG_RD(adapter, IGU_REG_PENDING_BITS_STATUS);

      if (pend_bits == 0)
	 break;

      vmk_DelayUsecs (2000);
   } while (cnt-- > 0);

   if (cnt <= 0) {
      QFLE3_ERR("Still pending IGU requests pend_bits=%x!\n",
		pend_bits);
      return -1;
   }

   return 0;
}
#define SHARED_MF_CLP_MAGIC  0x80000000 /* `magic' bit */

static void qfle3_clp_reset_prep(struct qfle3_adapter *adapter, vmk_uint32 *magic_val)
{
   /* Do some magic... */
   vmk_uint32 val = MF_CFG_RD(adapter, shared_mf_config.clp_mb);
   *magic_val = val & SHARED_MF_CLP_MAGIC;
   MF_CFG_WR(adapter, shared_mf_config.clp_mb, val | SHARED_MF_CLP_MAGIC);
}
/**
 * qfle3_clp_reset_done - restore the value of the `magic' bit.
 *
 * @adapter:		driver handle
 * @magic_val:	old value of the `magic' bit.
 */
static void qfle3_clp_reset_done(struct qfle3_adapter *adapter, vmk_uint32 magic_val)
{
   /* Restore the `magic' bit value... */
   vmk_uint32 val = MF_CFG_RD(adapter, shared_mf_config.clp_mb);
   MF_CFG_WR(adapter, shared_mf_config.clp_mb,
	     (val & (~SHARED_MF_CLP_MAGIC)) | magic_val);
}


static VMK_ReturnStatus qfle3_reset_mcp_comp(struct qfle3_adapter *adapter, vmk_uint32 magic_val)
{
   VMK_ReturnStatus rc = qfle3_init_shmem(adapter);

   /* Restore the `magic' bit value */
   if (!CHIP_IS_E1(adapter))
      qfle3_clp_reset_done(adapter, magic_val);

   return rc;
}



/**
 * qfle3_reset_mcp_prep - prepare for MCP reset.
 *
 * @adapter:		driver handle
 * @magic_val:	old value of 'magic' bit.
 *
 * Takes care of CLP configurations.
 */
static void qfle3_reset_mcp_prep(struct qfle3_adapter *adapter, vmk_uint32 *magic_val)
{
   vmk_uint32 shmem;
   vmk_uint32 validity_offset;

   QFLE3_DBG(QFLE3_DBG_HW, "Starting\n");

   /* Set `magic' bit in order to save MF config */
   if (!CHIP_IS_E1(adapter))
      qfle3_clp_reset_prep(adapter, magic_val);

   /* Get shmem offset */
   shmem = REG_RD(adapter, MISC_REG_SHARED_MEM_ADDR);
   validity_offset =
      vmk_offsetof(struct shmem_region, validity_map[QFLE3_PORT(adapter)]);

   /* Clear validity map flags */
   if (shmem > 0)
      REG_WR(adapter, shmem + validity_offset, 0);
}

static void qfle3_pxp_prep(struct qfle3_adapter *adapter)
{
   if (!CHIP_IS_E1(adapter)) {
      REG_WR(adapter, PXP2_REG_RD_START_INIT, 0);
      REG_WR(adapter, PXP2_REG_RQ_RBC_DONE, 0);
      vmk_CPUMemFenceReadWrite ();
   }
}


static int qfle3_process_kill(struct qfle3_adapter *adapter, vmk_Bool global)
{
   int cnt = 1000;
   vmk_uint32 val = 0;
   vmk_uint32 sr_cnt, blk_cnt, port_is_idle_0, port_is_idle_1, pgl_exp_rom2;
   vmk_uint32 tags_63_32 = 0;

   /* Empty the Tetris buffer, wait for 1s */
   do {
      sr_cnt  = REG_RD(adapter, PXP2_REG_RD_SR_CNT);
      blk_cnt = REG_RD(adapter, PXP2_REG_RD_BLK_CNT);
      port_is_idle_0 = REG_RD(adapter, PXP2_REG_RD_PORT_IS_IDLE_0);
      port_is_idle_1 = REG_RD(adapter, PXP2_REG_RD_PORT_IS_IDLE_1);
      pgl_exp_rom2 = REG_RD(adapter, PXP2_REG_PGL_EXP_ROM2);
      if (CHIP_IS_E3(adapter))
	 tags_63_32 = REG_RD(adapter, PGLUE_B_REG_TAGS_63_32);

      if ((sr_cnt == 0x7e) && (blk_cnt == 0xa0) &&
	  ((port_is_idle_0 & 0x1) == 0x1) &&
	  ((port_is_idle_1 & 0x1) == 0x1) &&
	  (pgl_exp_rom2 == 0xffffffff) &&
	  (!CHIP_IS_E3(adapter) || (tags_63_32 == 0xffffffff)))
	 break;
      vmk_DelayUsecs (2000);
   } while (cnt-- > 0);

   if (cnt <= 0) {
      QFLE3_ERR("Tetris buffer didn't get empty or there are still outstanding read requests after 1s!\n");
      QFLE3_ERR("sr_cnt=0x%08x, blk_cnt=0x%08x, port_is_idle_0=0x%08x, port_is_idle_1=0x%08x, pgl_exp_rom2=0x%08x\n",
		sr_cnt, blk_cnt, port_is_idle_0, port_is_idle_1,
		pgl_exp_rom2);
      return -1;
   }

   vmk_CPUMemFenceReadWrite ();

   /* Close gates #2, #3 and #4 */
   qfle3_set_234_gates(adapter, VMK_TRUE);

   /* Poll for IGU VQs for 57712 and newer chips */
   if (qfle3_er_poll_igu_vq(adapter))
      return -1;

   /* TBD: Indicate that "process kill" is in progress to MCP */

   /* Clear "unprepared" bit */
   REG_WR(adapter, MISC_REG_UNPREPARED, 0);
   vmk_CPUMemFenceReadWrite ();

   /* Make sure all is written to the chip before the reset */
//		mmiowb();

   /* Wait for 1ms to empty GLUE and PCI-E core queues,
    * PSWHST, GRC and PSWRD Tetris buffer.
    */
   vmk_DelayUsecs (2000);

   /* Prepare to chip reset: */
   /* MCP */
   if (global)
      qfle3_reset_mcp_prep(adapter, &val);

   /* PXP */
   qfle3_pxp_prep(adapter);
   vmk_CPUMemFenceReadWrite ();

   /* reset the chip */
   qfle3_process_kill_chip_reset(adapter, global);
   vmk_CPUMemFenceReadWrite ();

   /* clear errors in PGB */
   if (!CHIP_IS_E1x(adapter))
      REG_WR(adapter, PGLUE_B_REG_LATCHED_ERRORS_CLR, 0x7f);

   /* Recover after reset: */
   /* MCP */
   if (global && (qfle3_reset_mcp_comp(adapter, val) != VMK_OK))
      return -1;

   /* TBD: Add resetting the NO_MCP mode DB here */

   /* Open the gates #2, #3 and #4 */
   qfle3_set_234_gates(adapter, VMK_FALSE);

   /* TBD: IGU/AEU preparation bring back the AEU/IGU to a
    * reset state, re-enable attentions. */

   return 0;
}


/*
 * Read the load status for the current engine.
 *
 * should be run under rtnl lock
 */
static vmk_Bool qfle3_get_load_status(struct qfle3_adapter *adapter, int engine)
{
   vmk_uint32 mask = (engine ? QFLE3_PATH1_LOAD_CNT_MASK :
		      QFLE3_PATH0_LOAD_CNT_MASK);
   vmk_uint32 shift = (engine ? QFLE3_PATH1_LOAD_CNT_SHIFT :
		       QFLE3_PATH0_LOAD_CNT_SHIFT);
   vmk_uint32 val = REG_RD(adapter, QFLE3_RECOVERY_GLOB_REG);

   QFLE3_DBG(QFLE3_DBG_HW, "GLOB_REG=0x%08x\n", val);

   val = (val & mask) >> shift;

   QFLE3_DBG(QFLE3_DBG_HW, "load mask for engine %d = 0x%x\n",
	     engine, val);

   return val != 0;
}
static int qfle3_leader_reset(struct qfle3_adapter *adapter)
{
   int rc = 0;
   vmk_Bool global = qfle3_reset_is_global(adapter);
   vmk_uint32 load_code;

   /* if not going to reset MCP - load "fake" driver to reset HW while
    * driver is owner of the HW
    */
   if (!global && !QFLE3_NOMCP(adapter)) {
      load_code = qfle3_mfw_command(adapter, DRV_MSG_CODE_LOAD_REQ,
				    DRV_MSG_CODE_LOAD_REQ_WITH_LFA);
      if (!load_code) {
	 QFLE3_ERR("MCP response failure, aborting\n");
	 rc = -1;
	 goto exit_leader_reset;
      }
      if ((load_code != FW_MSG_CODE_DRV_LOAD_COMMON_CHIP) &&
	  (load_code != FW_MSG_CODE_DRV_LOAD_COMMON)) {
	 QFLE3_ERR("MCP unexpected resp, aborting\n");
	 rc = -1;
	 goto exit_leader_reset2;
      }
      load_code = qfle3_mfw_command(adapter, DRV_MSG_CODE_LOAD_DONE, 0);
      if (!load_code) {
	 QFLE3_ERR("MCP response failure, aborting\n");
	 rc = -1;
	 goto exit_leader_reset2;
      }
   }

   /* Try to recover after the failure */
   if (qfle3_process_kill(adapter, global)) {
      QFLE3_ERR("Something bad had happen on engine %d! Aii!\n",
		QFLE3_PATH(adapter));
      rc = -1;
      goto exit_leader_reset2;
   }

   /*
    * Clear RESET_IN_PROGRES and RESET_GLOBAL bits and update the driver
    * state.
    */
   qfle3_set_reset_done(adapter);
   if (global)
      qfle3_clear_reset_global(adapter);

  exit_leader_reset2:
   /* unload "fake driver" if it was loaded */
   if (!global && !QFLE3_NOMCP(adapter)) {
      qfle3_mfw_command(adapter, DRV_MSG_CODE_UNLOAD_REQ_WOL_MCP, 0);
      qfle3_mfw_command(adapter, DRV_MSG_CODE_UNLOAD_DONE, 0);
   }
  exit_leader_reset:
   adapter->is_leader = 0;
   qfle3_release_leader_lock(adapter);
   vmk_CPUMemFenceReadWrite ();
   return rc;
}


static void qfle3_recovery_failed(struct qfle3_adapter *adapter)
{
//		netdev_err(adapter->dev, "Recovery has failed. Power cycle is needed.\n");

   /* Mark link down */
   qfle3_tx_disable(adapter);
   QFLE3_ERR("ESX_REC_FAIL DS:0x%x RS:%d ES:0x%x\n",
      adapter->state->vector[0], adapter->recovery_state, adapter->error_status);
   /*
    * Block ifup for all function on this engine until "process kill"
    * or power cycle.
    */
   qfle3_set_reset_in_progress(adapter);

//		/* Shut down the power */
   adapter->recovery_state = QFLE3_RECOVERY_FAILED;

   vmk_CPUMemFenceReadWrite ();
}


void qfle3_parity_recover(void *adap)
{
   vmk_Bool global = VMK_FALSE;
//	vmk_uint32 error_recovered, error_unrecovered;
   vmk_Bool is_parity;
   struct qfle3_adapter *adapter = (struct qfle3_adapter *)adap;

   QFLE3_INFO("Run Recovery Helper Task\n");
   while (1) {
      
      switch (adapter->recovery_state) {
      case QFLE3_RECOVERY_INIT:
         QFLE3_INFO( "State is QFLE3_RECOVERY_INIT error 0x%x\n", adapter->error_status);
                  
         is_parity = qfle3_chk_parity_attn(adapter, &global, VMK_FALSE);
//         WARN_ON(!is_parity);
          
      	 /* Try to get a LEADER_LOCK HW lock */
         if (qfle3_trylock_leader_lock (adapter)) { //	 if (adapter->is_leader) {
         
            QFLE3_INFO( " Taking the leader role in recovery");
            qfle3_set_reset_in_progress(adapter);
            /*
             * Check if there is a global attention and if
             * there was a global attention, set the global
             * reset bit.
             */
            if (global)
               qfle3_set_reset_global(adapter);
            
            adapter->is_leader = 1;
         }else{
         
            QFLE3_INFO( " Taking the non-leader role in recovery");
         }
         /* Stop the driver */
         if (qfle3_sm_cmd (adapter, QFLE3_SMCMD_TOSTANDBY, 0) != QFLE3_SMCMD_STATUS_COMPLETED)
         {
            QFLE3_ERR("Unable to execute QFLE3_SMCMD_TOSTANDBY\n");
            return;
         }
         
         QFLE3_INFO( "Adapter in Standby Mode, Wait for others");

         adapter->recovery_state = QFLE3_RECOVERY_WAIT;

         /* Ensure "is_leader", MCP command sequence and
          * "recovery_state" update values are seen on other
          * CPUs.
          */
         vmk_CPUMemFenceReadWrite ();
         QFLE3_INFO("Moving to QFLE3_RECOVERY_WAIT\n");
         
//	      if (adapter->recovery_count > 10){
//            QFLE3_INFO("Recovery Abort. count %d, ERROR 0x%x \n", adapter->recovery_count, adapter->error_status);
//            return;
//	      }
//         if (adapter->recovery_count && (adapter->error_status != QFLE3_ERR_PARITY)){
//            QFLE3_INFO("Recovery Abort. count %d, ERROR 0x%x \n", adapter->recovery_count, adapter->error_status);
//            return;
//         }
         break;

      case QFLE3_RECOVERY_WAIT:
         QFLE3_INFO("State is QFLE3_RECOVERY_WAIT\n");
         if (adapter->is_leader) {
            int other_engine = QFLE3_PATH(adapter) ? 0 : 1;
            vmk_Bool other_load_status =
               qfle3_get_load_status(adapter, other_engine);
            vmk_Bool load_status =
               qfle3_get_load_status(adapter, QFLE3_PATH(adapter));
            global = qfle3_reset_is_global(adapter);
            QFLE3_INFO("We are the leader\n");
            /* In case of a parity in a global block, let
             * the first leader that performs a
             * leader_reset() reset the global blocks in
             * order to clear global attentions. Otherwise
             * the gates will remain closed for that
             * engine.
             */
             
            QFLE3_INFO("Leader load_status 0x%x, global %d, other status 0x%x\n",
               load_status, global, other_load_status);
            if (load_status ||
               (global && other_load_status)) {
               /* Wait until all other functions get
                * down.
                */
                
               QFLE3_INFO("Leader Scheduling qfle3_parity_recover, load_status 0x%x, global %d, other status 0x%x\n",
                  load_status, global, other_load_status);
               qfle3_schedule_helper (adapter, adapter->recovery_helper, qfle3_recovery_func, 10000);
               return;
            } else {
               /* If all other functions got down -
                * try to bring the chip back to
                * normal. In any case it's an exit
                * point for a leader.
                */
                
               QFLE3_INFO("Leader Performing %s reset\n", global?"global":"engine");
               if (qfle3_leader_reset(adapter)) {
                  
                  QFLE3_ERR("Recovery Failed\n");
                  qfle3_recovery_failed(adapter);
                  return;
               }

               /* If we are here, means that the
                * leader has succeeded and doesn't
                * want to be a leader any more. Try
                * to continue as a none-leader.
                */
                
               QFLE3_INFO("Continue as a non-leader\n");
               break;
            }
         } else { /* non-leader */
         
            QFLE3_INFO("Non-leader\n");
            if (!qfle3_reset_is_done(adapter, QFLE3_PATH(adapter))) {
               /* Try to get a LEADER_LOCK HW lock as
                * long as a former leader may have
                * been unloaded by the user or
                * released a leadership by another
                * reason.
                */
                
               QFLE3_INFO("Leader has not finished reset.  Try to take leader\n");
               if (qfle3_trylock_leader_lock(adapter)) {
                  /* I'm a leader now! Restart a
                   * switch case.
                   */
                  adapter->is_leader = 1;
                  
                  QFLE3_INFO("Took Leader\n");
                  break;
               }
               QFLE3_INFO("Scheduling qfle3_parity_recover\n");

               qfle3_schedule_helper (adapter, adapter->recovery_helper, qfle3_recovery_func, 10000);
               return;
            } else {
            
               struct cnic_ctl_info ctl;
               QFLE3_INFO("Leader has finished Reset\n");
               /*
                * If there was a global attention, wait
                * for it to be cleared.
                */
               if (qfle3_reset_is_global(adapter)) {
                  
                  QFLE3_INFO("reset is global Scheduling qfle3_parity_recover\n");
                  qfle3_schedule_helper (adapter, adapter->recovery_helper, qfle3_recovery_func, 10000);
                  return;
               }

//             error_recovered =
//                adapter->eth_stats.recoverable_error;
//             error_unrecovered =
//                adapter->eth_stats.unrecoverable_error;
               adapter->recovery_state = QFLE3_RECOVERY_NIC_LOADING;
               QFLE3_INFO("State is QFLE3_RECOVERY_NIC_LOADING\n");
               if (qfle3_sm_cmd(adapter, QFLE3_SMCMD_BRINGUP, 0)!= QFLE3_SMCMD_STATUS_COMPLETED) {
                  QFLE3_ERR("Recovery failed. Power cycle may be needed");

						QFLE3_ERR("ESX_REC_FAIL DS:0x%x RS:%d ES:0x%x",
							  adapter->state->vector[0], adapter->recovery_state,
							  adapter->error_status);
//						qfle3_set_power_state(
//							bp, PCI_D3hot);
                  vmk_CPUMemFenceReadWrite ();
                  return;
               }
               if (qfle3_sm_cmd(adapter, QFLE3_SMCMD_START, 0) != QFLE3_SMCMD_STATUS_COMPLETED) {
                  QFLE3_ERR("Unable to start adapter after reset \n");
                  vmk_CPUMemFenceReadWrite ();
                  qfle3_sm_cmd(adapter, QFLE3_SMCMD_TOSTANDBY, 0);
                  return;
               } 
               if (qfle3_sm_cmd(adapter, QFLE3_SMCMD_RESUMEIO, 0) != QFLE3_SMCMD_STATUS_COMPLETED) {
                  QFLE3_ERR("Unable to start Uplink after reset \n");
                  vmk_CPUMemFenceReadWrite ();
                  qfle3_sm_cmd(adapter, QFLE3_SMCMD_TOSTANDBY, 0);
                  return;
               } 
               
                  QFLE3_INFO("Adapter is up, State is QFLE3_RECOVERY_DONE\n");
                  adapter->recovery_state = QFLE3_RECOVERY_DONE;
//                   error_recovered++;
                  adapter->error_status = 0;
                  vmk_CPUMemFenceReadWrite ();
//             adapter->eth_stats.recoverable_error =
//                error_recovered;
//             dapter->eth_stats.unrecoverable_error =
//                error_unrecovered;
               QFLE3_INFO("Adapter is up, State is QFLE3_RECOVERY_DONE\n");
               vmk_BitVectorAtomicTestAndClear(adapter->state, QFLE3_STATE_BIT_RESETTING);
               // send Recovery done command to cnic
               ctl.cmd = CNIC_CTL_NIC_RECOVERY_DONE_CMD;
               
               if (adapter->cnicOps)
                  adapter->cnicOps->cnicNotify(adapter->cnic_data, &ctl);
               return;
            }
         }
         default:
            
            QFLE3_ERR( "Shouldn't be here\n");
            return;
      }
   }
}


static void
qfle3_attn_int_deasserted0(struct qfle3_adapter *adapter, vmk_uint32 attn)
{
   int port = QFLE3_PORT(adapter);
   int reg_offset;
   vmk_uint32 val;

   reg_offset = (port ? MISC_REG_AEU_ENABLE1_FUNC_1_OUT_0 :
		 MISC_REG_AEU_ENABLE1_FUNC_0_OUT_0);

   if (attn & AEU_INPUTS_ATTN_BITS_SPIO5) {

      val = REG_RD(adapter, reg_offset);
      val &= ~AEU_INPUTS_ATTN_BITS_SPIO5;
      REG_WR(adapter, reg_offset, val);

      QFLE3_ERR("SPIO5 hw attention\n");

      /*
       * Fan failure attention
       */
#if 0
      qfle3_hw_reset_phy(&adapter->link_params);
      qfle3_fan_failure(adapter);
#endif
   }

   if ((attn & adapter->link_vars.aeu_int_mask) && adapter->port.pmf) {
      //QFLE3_INFO("module detect int\n");
      qfle3_acquire_phy_lock(adapter);
      elink_handle_module_detect_int(&adapter->link_params);
      qfle3_release_phy_lock(adapter);
   }

   if (attn & HW_INTERRUT_ASSERT_SET_0) {

      val = REG_RD(adapter, reg_offset);
      val &= ~(attn & HW_INTERRUT_ASSERT_SET_0);
      REG_WR(adapter, reg_offset, val);

      QFLE3_ERR("FATAL HW block attention set0 0x%x\n",
		(vmk_uint32) (attn & HW_INTERRUT_ASSERT_SET_0));
      qfle3_panic(adapter);
   }
}

static void
qfle3_attn_int_deasserted1(struct qfle3_adapter *adapter, vmk_uint32 attn)
{
   vmk_uint32 val;

   if (attn & AEU_INPUTS_ATTN_BITS_DOORBELLQ_HW_INTERRUPT) {

      val = REG_RD(adapter, DORQ_REG_DORQ_INT_STS_CLR);
      QFLE3_ERR("DB hw attention 0x%x\n", val);
      /*
       * DORQ discard attention
       */
      if (val & 0x2)
	 QFLE3_ERR("FATAL error from DORQ\n");
   }

   if (attn & HW_INTERRUT_ASSERT_SET_1) {

      int port = QFLE3_PORT(adapter);
      int reg_offset;

      reg_offset = (port ? MISC_REG_AEU_ENABLE1_FUNC_1_OUT_1 :
		    MISC_REG_AEU_ENABLE1_FUNC_0_OUT_1);

      val = REG_RD(adapter, reg_offset);
      val &= ~(attn & HW_INTERRUT_ASSERT_SET_1);
      REG_WR(adapter, reg_offset, val);

      QFLE3_ERR("FATAL HW block attention set1 0x%x\n",
		(vmk_uint32) (attn & HW_INTERRUT_ASSERT_SET_1));
      qfle3_panic(adapter);
   }
}

static void
qfle3_attn_int_deasserted2(struct qfle3_adapter *adapter, vmk_uint32 attn)
{
   vmk_uint32 val;

   if (attn & AEU_INPUTS_ATTN_BITS_CFC_HW_INTERRUPT) {

      val = REG_RD(adapter, CFC_REG_CFC_INT_STS_CLR);
      QFLE3_ERR("CFC hw attention 0x%x\n", val);
      /*
       * CFC error attention
       */
      if (val & 0x2)
	 QFLE3_ERR("FATAL error from CFC\n");
   }

   if (attn & AEU_INPUTS_ATTN_BITS_PXP_HW_INTERRUPT) {
      val = REG_RD(adapter, PXP_REG_PXP_INT_STS_CLR_0);
      QFLE3_ERR("PXP hw attention-0 0x%x\n", val);
      /*
       * RQ_USDMQFLE3_DBG_FIFO_OVERFLOW
       */
      if (val & 0x18000)
	 QFLE3_ERR("FATAL error from PXP\n");

      if (!CHIP_IS_E1x(adapter)) {
	 val = REG_RD(adapter, PXP_REG_PXP_INT_STS_CLR_1);
	 QFLE3_ERR("PXP hw attention-1 0x%x\n", val);
      }
   }
#ifndef QFLE3_UPSTREAM          /* ! QFLE3_UPSTREAM */
#define PXP2_EOP_ERROR_BIT   PXP2_PXP2_INT_STS_CLR_0_REG_WR_PGLUE_EOP_ERROR
#define AEU_PXP2_HW_INT_BIT  AEU_INPUTS_ATTN_BITS_PXPPCICLOCKCLIENT_HW_INTERRUPT

   if (attn & AEU_PXP2_HW_INT_BIT) {
      /*
       * CQ47854 workaround do not panic on
       * *  PXP2_PXP2_INT_STS_0_REG_WR_PGLUE_EOP_ERROR
       */
      if (!CHIP_IS_E1x(adapter)) {
	 vmk_uint32 val0, mask0, val1, mask1;
	 mask0 = REG_RD(adapter, PXP2_REG_PXP2_INT_MASK_0);
	 val1 = REG_RD(adapter, PXP2_REG_PXP2_INT_STS_1);
	 mask1 = REG_RD(adapter, PXP2_REG_PXP2_INT_MASK_1);
	 val0 = REG_RD(adapter, PXP2_REG_PXP2_INT_STS_0);
	 /*
	  * If the olny PXP2_EOP_ERROR_BIT is set in
	  * STS0 and STS1 - clear it
	  *
	  * probably we lose additional attentions between
	  * STS0 and STS_CLR0, in this case user will not
	  * be notified about them
	  */
	 if (val0 & mask0 & PXP2_EOP_ERROR_BIT && !(val1 & mask1))
	    val0 = REG_RD(adapter, PXP2_REG_PXP2_INT_STS_CLR_0);

	 /*
	  * print the register, since no one can restore it
	  */
	 QFLE3_ERR("PXP2_REG_PXP2_INT_STS_CLR_0 0x%08x\n", val0);

	 /*
	  * if PXP2_PXP2_INT_STS_0_REG_WR_PGLUE_EOP_ERROR
	  * then notify
	  */

	 if (val0 & PXP2_EOP_ERROR_BIT) {
	    QFLE3_ERR("PXP2_WR_PGLUE_EOP_ERROR\n");
	    /*
	     * if only
	     * * PXP2_PXP2_INT_STS_0_REG_WR_PGLUE_EOP_ERROR
	     * * is set then clear attention from PXP2
	     * * block without panic
	     */
	    if (((val0 & mask0) == PXP2_EOP_ERROR_BIT) && ((val1 & mask1) == 0))
	       attn &= ~AEU_PXP2_HW_INT_BIT;
	 }
      }
   }
#endif
   if (attn & HW_INTERRUT_ASSERT_SET_2) {

      int port = QFLE3_PORT(adapter);
      int reg_offset;

      reg_offset = (port ? MISC_REG_AEU_ENABLE1_FUNC_1_OUT_2 :
		    MISC_REG_AEU_ENABLE1_FUNC_0_OUT_2);

      val = REG_RD(adapter, reg_offset);
      val &= ~(attn & HW_INTERRUT_ASSERT_SET_2);
      REG_WR(adapter, reg_offset, val);

      QFLE3_ERR("FATAL HW block attention set2 0x%x\n",
		(vmk_uint32) (attn & HW_INTERRUT_ASSERT_SET_2));
      qfle3_panic(adapter);
   }
}

/* called due to MCP event (on pmf):
 *      reread new bandwidth configuration
 *      configure FW
 *      notify others function about the change
 */
static void
qfle3_config_mf_bw(qfle3_adapter *adapter)
{
   if(!IS_MF(adapter)) {
      QFLE3_INFO("Ignoring MF BW config in single function mode\n");
      return;
   }
   if (adapter->link_vars.link_up) {
	  qfle3_cmng_fns_init(adapter, VMK_TRUE, CMNG_FNS_MINMAX);
      qfle3_link_sync_notify(adapter);
   }
   storm_memset_cmng(adapter, &adapter->cmng, QFLE3_PORT(adapter));
}

static void
qfle3_set_mf_bw(qfle3_adapter *adapter)
{
   qfle3_config_mf_bw(adapter);
   qfle3_mfw_command(adapter, DRV_MSG_CODE_SET_MF_BW_ACK, 0);
}

#define DRV_INFO_ETH_STAT_NUM_MACS_REQUIRED 3
#ifdef QFLE3_CNIC

int qfle3_cnic_notify(struct qfle3_adapter * adapter, int cmd);
static void qfle3_drv_info_iscsi_stat(qfle3_adapter *adapter)
{
    struct qfle3_dcbx_app_params *app = &adapter->dcbx_sb.port_params.app;
    struct iscsi_stats_info *iscsi_stat =
        &adapter->sp->drv_info_to_mcp.iscsi_stat;

    if (!CNIC_LOADED(adapter))
        return;

    vmk_Memcpy(iscsi_stat->mac_local + MAC_PAD, adapter->iscsi_mac,
           ETH_ALEN);

    iscsi_stat->qos_priority =
        app->traffic_type_priority[LLFC_TRAFFIC_TYPE_ISCSI];

    /* ask L5 driver to add data to the struct */
    qfle3_cnic_notify(adapter, CNIC_CTL_ISCSI_STATS_GET_CMD);
}

static void qfle3_drv_info_fcoe_stat(qfle3_adapter *adapter)
{
    struct qfle3_dcbx_app_params *app = &adapter->dcbx_sb.port_params.app;
    struct fcoe_stats_info *fcoe_stat =
        &adapter->sp->drv_info_to_mcp.fcoe_stat;

    if (!CNIC_LOADED(adapter))
        return;

    vmk_Memcpy(fcoe_stat->mac_local + MAC_PAD, adapter->fip_mac, ETH_ALEN);

    fcoe_stat->qos_priority =
        app->traffic_type_priority[LLFC_TRAFFIC_TYPE_FCOE];

    /* insert FCoE stats from ramrod response */
    if (!NO_FCOE(adapter)) {
        struct tstorm_per_queue_stats *fcoe_q_tstorm_stats =
            &adapter->fw_stats_data->queue_stats[FCOE_IDX(adapter)].
            tstorm_queue_statistics;

        struct xstorm_per_queue_stats *fcoe_q_xstorm_stats =
            &adapter->fw_stats_data->queue_stats[FCOE_IDX(adapter)].
            xstorm_queue_statistics;

        struct fcoe_statistics_params *fw_fcoe_stat =
            &adapter->fw_stats_data->fcoe;

        ADD_64_LE(fcoe_stat->rx_bytes_hi, LE32_0,
              fcoe_stat->rx_bytes_lo,
              fw_fcoe_stat->rx_stat0.fcoe_rx_byte_cnt);

        ADD_64_LE(fcoe_stat->rx_bytes_hi,
              fcoe_q_tstorm_stats->rcv_ucast_bytes.hi,
              fcoe_stat->rx_bytes_lo,
              fcoe_q_tstorm_stats->rcv_ucast_bytes.lo);

        ADD_64_LE(fcoe_stat->rx_bytes_hi,
              fcoe_q_tstorm_stats->rcv_bcast_bytes.hi,
              fcoe_stat->rx_bytes_lo,
              fcoe_q_tstorm_stats->rcv_bcast_bytes.lo);

        ADD_64_LE(fcoe_stat->rx_bytes_hi,
              fcoe_q_tstorm_stats->rcv_mcast_bytes.hi,
              fcoe_stat->rx_bytes_lo,
              fcoe_q_tstorm_stats->rcv_mcast_bytes.lo);

        ADD_64_LE(fcoe_stat->rx_frames_hi, LE32_0,
              fcoe_stat->rx_frames_lo,
              fw_fcoe_stat->rx_stat0.fcoe_rx_pkt_cnt);

        ADD_64_LE(fcoe_stat->rx_frames_hi, LE32_0,
              fcoe_stat->rx_frames_lo,
              fcoe_q_tstorm_stats->rcv_ucast_pkts);

        ADD_64_LE(fcoe_stat->rx_frames_hi, LE32_0,
              fcoe_stat->rx_frames_lo,
              fcoe_q_tstorm_stats->rcv_bcast_pkts);

        ADD_64_LE(fcoe_stat->rx_frames_hi, LE32_0,
              fcoe_stat->rx_frames_lo,
              fcoe_q_tstorm_stats->rcv_mcast_pkts);

        ADD_64_LE(fcoe_stat->tx_bytes_hi, LE32_0,
              fcoe_stat->tx_bytes_lo,
              fw_fcoe_stat->tx_stat.fcoe_tx_byte_cnt);

        ADD_64_LE(fcoe_stat->tx_bytes_hi,
              fcoe_q_xstorm_stats->ucast_bytes_sent.hi,
              fcoe_stat->tx_bytes_lo,
              fcoe_q_xstorm_stats->ucast_bytes_sent.lo);

        ADD_64_LE(fcoe_stat->tx_bytes_hi,
              fcoe_q_xstorm_stats->bcast_bytes_sent.hi,
              fcoe_stat->tx_bytes_lo,
              fcoe_q_xstorm_stats->bcast_bytes_sent.lo);

        ADD_64_LE(fcoe_stat->tx_bytes_hi,
              fcoe_q_xstorm_stats->mcast_bytes_sent.hi,
              fcoe_stat->tx_bytes_lo,
              fcoe_q_xstorm_stats->mcast_bytes_sent.lo);

        ADD_64_LE(fcoe_stat->tx_frames_hi, LE32_0,
              fcoe_stat->tx_frames_lo,
              fw_fcoe_stat->tx_stat.fcoe_tx_pkt_cnt);


        ADD_64_LE(fcoe_stat->tx_frames_hi, LE32_0,
              fcoe_stat->tx_frames_lo,
              fcoe_q_xstorm_stats->ucast_pkts_sent);

        ADD_64_LE(fcoe_stat->tx_frames_hi, LE32_0,
              fcoe_stat->tx_frames_lo,
              fcoe_q_xstorm_stats->bcast_pkts_sent);

        ADD_64_LE(fcoe_stat->tx_frames_hi, LE32_0,
              fcoe_stat->tx_frames_lo,
              fcoe_q_xstorm_stats->mcast_pkts_sent);
    }
    /* ask L5 driver to add data to the struct */
    qfle3_cnic_notify(adapter, CNIC_CTL_FCOE_STATS_GET_CMD);
}
#endif
static void
qfle3_drv_info_ether_stat(qfle3_adapter *adapter)
{
   struct eth_stats_info *ether_stat =
        &adapter->sp->drv_info_to_mcp.ether_stat;
   struct ecore_vlan_mac_obj *mac_obj =
         &adapter->sp_objs->mac_obj;
   vmk_int32 i;

   vmk_StringLCopy(ether_stat->version, QFLE3_DRIVER_VERSION,
           ETH_STAT_INFO_VERSION_LEN, NULL);

   /* get DRV_INFO_ETH_STAT_NUM_MACS_REQUIRED macs, placing them in the
    * mac_local field in ether_stat struct. The base address is offset by 2
    * bytes to account for the field being 8 bytes but a mac address is
    * only 6 bytes. Likewise, the stride for the get_n_elements function is
    * 2 bytes to compensate from the 6 bytes of a mac to the 8 bytes
    * allocated by the ether_stat struct, so the macs will land in their
    * proper positions.
    */
   for (i = 0; i < DRV_INFO_ETH_STAT_NUM_MACS_REQUIRED; i++)
      vmk_Memset(ether_stat->mac_local + i, 0,
         sizeof(ether_stat->mac_local[0]));

   if (mac_obj->exe_queue.lock) {
      mac_obj->get_n_elements(adapter, &adapter->sp_objs[0].mac_obj,
                           DRV_INFO_ETH_STAT_NUM_MACS_REQUIRED,
                           ether_stat->mac_local + MAC_PAD, MAC_PAD,
                           ETH_ALEN);
   }
   ether_stat->mtu_size = adapter->mtu;
   if (adapter->uplink &&
       vmk_UplinkCapIsEnabled(adapter->uplink, VMK_UPLINK_CAP_IPV4_CSO) &&
       vmk_UplinkCapIsEnabled(adapter->uplink, VMK_UPLINK_CAP_IPV4_CSO))
      ether_stat->feature_flags |= FEATURE_ETH_CHKSUM_OFFLOAD_MASK;

   if (adapter->uplink &&
       vmk_UplinkCapIsEnabled(adapter->uplink, VMK_UPLINK_CAP_IPV4_TSO) &&
       vmk_UplinkCapIsEnabled(adapter->uplink, VMK_UPLINK_CAP_IPV4_TSO))
      ether_stat->feature_flags |= FEATURE_ETH_LSO_MASK;

   ether_stat->feature_flags |= adapter->hw_info.boot_mode;

//QFLE3_TODO Check whether promiscuous mode is really enabled or not
   ether_stat->promiscuous_mode = 1;

   ether_stat->txq_size = adapter->tx_ring_size;
   ether_stat->rxq_size = adapter->rx_ring_size;

   ether_stat->lso_max_size = QFLE3_MAX_LSO_SIZE;
   ether_stat->lso_min_seg_cnt = QFLE3_MIN_LSO_SEG_CNT;

#ifdef QFLE3_SRIOV
      ether_stat->vf_cnt = (adapter->vfdb) ? adapter->vfdb->sriov.nr_virtfn : 0;
#endif //QFLE3_SRIOV
}

#define QFLE3_UPDATE_DRV_INFO_IND_LENGTH        (20)
#define QFLE3_UPDATE_DRV_INFO_IND_COUNT         (25)

static void
qfle3_handle_drv_info_req(qfle3_adapter *adapter)
{
   enum drv_info_opcode op_code;
   vmk_uint32 drv_info_ctl = SHMEM2_RD(adapter, drv_info_control);
   vmk_Bool release = VMK_FALSE;
   int wait;

   /* if drv_info version supported by MFW doesn't match - send NACK */
   if ((drv_info_ctl & DRV_INFO_CONTROL_VER_MASK) != DRV_INFO_CUR_VER) {
      qfle3_mfw_command(adapter, DRV_MSG_CODE_DRV_INFO_NACK, 0);
      return;
   }


   op_code = (drv_info_ctl & DRV_INFO_CONTROL_OP_CODE_MASK) >>
              DRV_INFO_CONTROL_OP_CODE_SHIFT;

   /* Must prevent other flows from accessing drv_info_to_mcp */
   vmk_SemaLock(&adapter->drv_info_mutex);

   vmk_Memset(&adapter->sp->drv_info_to_mcp, 0,
              sizeof(union drv_info_to_mcp));

   switch (op_code) {
   case ETH_STATS_OPCODE:
      qfle3_drv_info_ether_stat(adapter);
      break;
#ifdef QFLE3_CNIC
   case FCOE_STATS_OPCODE:
      qfle3_drv_info_fcoe_stat(adapter);
      break;
   case ISCSI_STATS_OPCODE:
      qfle3_drv_info_iscsi_stat(adapter);
      break;
#endif
   default:
      /* if op code isn't supported - send NACK */
      qfle3_mfw_command(adapter, DRV_MSG_CODE_DRV_INFO_NACK, 0);
      goto out;
   }

   /* if we got drv_info attn from MFW then these fields are defined in
    * shmem2 for sure
    */
   SHMEM2_WR(adapter, drv_info_host_addr_lo,
                U64_LO(QFLE3_SP_MAPPING(adapter, drv_info_to_mcp)));
   SHMEM2_WR(adapter, drv_info_host_addr_hi,
                U64_HI(QFLE3_SP_MAPPING(adapter, drv_info_to_mcp)));

   qfle3_mfw_command(adapter, DRV_MSG_CODE_DRV_INFO_ACK, 0);


   /* Since possible management wants both this and get_driver_version
    * need to wait until management notifies us it finished utilizing
    * the buffer.
    */
   if (!SHMEM2_HAS(adapter, mfw_drv_indication)) {
           QFLE3_DBG(QFLE3_DBG_MCP,
                           "Management does not support indication\n");
   } else if (!adapter->drv_info_mng_owner) {
      vmk_uint32 bit = MFW_DRV_IND_READ_DONE_OFFSET((QFLE3_ABS_FUNC(adapter) >> 1));

      for (wait = 0; wait < QFLE3_UPDATE_DRV_INFO_IND_COUNT; wait++) {
         vmk_uint32 indication = SHMEM2_RD(adapter, mfw_drv_indication);

         /* Management is done; need to clear indication */
         if (indication & bit) {
            SHMEM2_WR(adapter, mfw_drv_indication,
                      indication & ~bit);
            release = VMK_TRUE;
            break;
         }

         vmk_WorldSleep(QFLE3_UPDATE_DRV_INFO_IND_LENGTH * 1000);
      }
   }
   if (!release) {
      QFLE3_DBG(QFLE3_DBG_MCP,
                "Management did not release indication\n");
      adapter->drv_info_mng_owner = VMK_TRUE;
   }
out:
   vmk_SemaUnlock(&adapter->drv_info_mutex);
}

static void
qfle3_e1h_disable(qfle3_adapter *adapter)
{
   vmk_int32 port = QFLE3_PORT(adapter);

//TODO disable TX Queues;
   //qfle3_tx_disable(adapter);

   REG_WR(adapter, NIG_REG_LLH0_FUNC_EN + port*8, 0);
}

static void
qfle3_e1h_enable(qfle3_adapter *adapter)
{
   int port = QFLE3_PORT(adapter);

   if (!(IS_MF_UFP(adapter) && QFLE3_IS_MF_SD_PROTOCOL_FCOE(adapter)))
      REG_WR(adapter, NIG_REG_LLH0_FUNC_EN + port*8, 1);

   /* Tx queue should be only re-enabled */
#if 0
   netif_tx_wake_all_queues(adapter->dev);
#endif

#if 0
#if defined(QFLE3_ESX_CNA) /* ! QFLE3_UPSTREAM */
   if (adapter->flags & CNA_ENABLED &&
       QFLE3_IS_NETQ_TX_QUEUE_ALLOCATED(qfle3_fcoe_fp(adapter)))
      netif_tx_wake_all_queues(adapter->cnadev);
#endif
#endif
}

static void
qfle3_oem_event(qfle3_adapter *adapter, vmk_uint32 event)
{
   vmk_uint32 cmd_ok, cmd_fail;

   /* sanity */
   if (event & DRV_STATUS_DCC_EVENT_MASK &&
                   event & DRV_STATUS_OEM_EVENT_MASK) {
      QFLE3_ERR("Received simultaneous events %08x\n", event);
      return;
   }

   if (event & DRV_STATUS_DCC_EVENT_MASK) {
      cmd_fail = DRV_MSG_CODE_DCC_FAILURE;
      cmd_ok = DRV_MSG_CODE_DCC_OK;
   } else /* if (event & DRV_STATUS_OEM_EVENT_MASK) */ {
      cmd_fail = DRV_MSG_CODE_OEM_FAILURE;
      cmd_ok = DRV_MSG_CODE_OEM_OK;
   }

   QFLE3_DBG(QFLE3_DBG_MCP, "oem_event 0x%x\n", event);

   if (event & (DRV_STATUS_DCC_DISABLE_ENABLE_PF |
                DRV_STATUS_OEM_DISABLE_ENABLE_PF)) {
      /*
       * This is the only place besides the function initialization
       * where the adapter->flags can change so it is done without any
       * locks
       */
      if (adapter->mf_config[QFLE3_VN(adapter)] & FUNC_MF_CFG_FUNC_DISABLED) {
         QFLE3_DBG(QFLE3_DBG_MCP, "mf_cfg function disabled\n");
         adapter->flags |= MF_FUNC_DIS;
         qfle3_e1h_disable(adapter);
      } else {
         QFLE3_DBG(QFLE3_DBG_MCP, "mf_cfg function enabled\n");
         adapter->flags &= ~MF_FUNC_DIS;
         qfle3_e1h_enable(adapter);
      }
      event &= ~(DRV_STATUS_DCC_DISABLE_ENABLE_PF |
                 DRV_STATUS_OEM_DISABLE_ENABLE_PF);
   }

   if (event & (DRV_STATUS_DCC_BANDWIDTH_ALLOCATION |
                DRV_STATUS_OEM_BANDWIDTH_ALLOCATION)) {
      qfle3_config_mf_bw(adapter);
      event &= ~(DRV_STATUS_DCC_BANDWIDTH_ALLOCATION |
                      DRV_STATUS_OEM_BANDWIDTH_ALLOCATION);
   }

   /* Report results to MCP */
   if (event)
      qfle3_mfw_command(adapter, cmd_fail, 0);
   else
      qfle3_mfw_command(adapter, cmd_ok, 0);
}

static void
qfle3_handle_update_svid_cmd(qfle3_adapter *adapter)
{
       struct ecore_func_switch_update_params *switch_update_params;
   struct ecore_func_state_params func_params;

   vmk_Memset(&func_params, 0, sizeof(struct ecore_func_state_params));
   switch_update_params = &func_params.params.switch_update;
   func_params.f_obj = &adapter->func_obj;
   func_params.cmd = ECORE_F_CMD_SWITCH_UPDATE;

   if (IS_MF_UFP(adapter) || IS_MF_BD(adapter)) {
      int func = QFLE3_ABS_FUNC(adapter);
      vmk_uint32 val;

      /* Re-learn the S-tag from shmem */
      val = MF_CFG_RD(adapter, func_mf_config[func].e1hov_tag) &
              FUNC_MF_CFG_E1HOV_TAG_MASK;
      if (val != FUNC_MF_CFG_E1HOV_TAG_DEFAULT) {
         adapter->mf_ov = val;
      } else {
         QFLE3_ERR("Got an SVID event, but no tag is configured in shmem\n");
         goto fail;
      }

      /* Configure new S-tag in LLH */
      REG_WR(adapter, NIG_REG_LLH0_FUNC_VLAN_ID + QFLE3_PORT(adapter) * 8,
                      adapter->mf_ov);

      /* Send Ramrod to update FW of change */
      ECORE_SET_BIT(ECORE_F_UPDATE_SD_VLAN_TAG_CHNG,
                      &switch_update_params->changes);
      switch_update_params->vlan = adapter->mf_ov;

      if (ecore_func_state_change(adapter, &func_params) < 0) {
              QFLE3_ERR("Failed to configure FW of S-tag Change to %02x\n",
                              adapter->mf_ov);
         goto fail;
      } else {
         QFLE3_DBG(QFLE3_DBG_MCP, "Configured S-tag %02x\n", adapter->mf_ov);
      }
   } else {
           /* not supported by SW yet */
           goto fail;
   }

   qfle3_mfw_command(adapter, DRV_MSG_CODE_OEM_UPDATE_SVID_OK, 0);
   return;
fail:
   qfle3_mfw_command(adapter, DRV_MSG_CODE_OEM_UPDATE_SVID_FAILURE, 0);
}

static void
qfle3_attn_int_deasserted3(struct qfle3_adapter *adapter, vmk_uint32 attn)
{
   vmk_uint32 val = 0;
   QFLE3_DBG(QFLE3_DBG_HW, "deasserted 3 0x%x, EVEREST_GEN_ATTN_IN_USE_MASK 0x%lx MC assert bits 0x%lx\n", 
      attn, (vmk_uint64)EVEREST_GEN_ATTN_IN_USE_MASK,(vmk_uint64)QFLE3_MC_ASSERT_BITS);

   if (adapter->reboot_in_progress)
        return;

   if (attn & EVEREST_GEN_ATTN_IN_USE_MASK) {

      if (attn & QFLE3_PMF_LINK_ASSERT) {
	 int func = QFLE3_FUNC(adapter);

	 REG_WR(adapter, MISC_REG_AEU_GENERAL_ATTN_12 + func * 4, 0);
	 QFLE3_DBG((QFLE3_DBG_HW | QFLE3_DBG_LINK), "LINK ATTENTION! 0x%x\n", attn);
#ifdef QFLE3_SRIOV
	 
         qfle3_read_mf_cfg(adapter);
	 adapter->mf_config[QFLE3_VN(adapter)] = MF_CFG_RD(adapter,
							   func_mf_config[QFLE3_ABS_FUNC
									  (adapter)].
							   config);
	 val = SHMEM_RD(adapter, func_mb[QFLE3_FW_MB_IDX(adapter)].drv_status);

#endif //QFLE3_SRIOV
	 if (val & (DRV_STATUS_DCC_EVENT_MASK | DRV_STATUS_OEM_EVENT_MASK))
	    qfle3_oem_event(adapter,
			    (val & (DRV_STATUS_DCC_EVENT_MASK |
				    DRV_STATUS_OEM_EVENT_MASK)));

	 if (val & DRV_STATUS_SET_MF_BW)
	    qfle3_set_mf_bw(adapter);

	 if (val & DRV_STATUS_DRV_INFO_REQ)
	    qfle3_handle_drv_info_req(adapter);

#ifdef QFLE3_SRIOV

         if (val & DRV_STATUS_VF_DISABLED){
            QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Scheduling VF FLR\n");
            qfle3_schedule_iov_task(adapter, QFLE3_IOV_HANDLE_FLR);
         }

#endif //QFLE3_SRIOV

	 if ((adapter->port.pmf == 0) && (val & DRV_STATUS_PMF))
	    qfle3_pmf_update(adapter);

	 if (adapter->port.pmf &&
	     (val & DRV_STATUS_DCBX_NEGOTIATION_RESULTS))
	    qfle3_dcbx_set_params(adapter, QFLE3_DCBX_STATE_NEG_RECEIVED);
#if 0
	 if (val & DRV_STATUS_AFEX_EVENT_MASK)
	    qfle3_handle_afex_cmd(adapter, val & DRV_STATUS_AFEX_EVENT_MASK);

	 if (val & DRV_STATUS_EEE_NEGOTIATION_RESULTS)
	    qfle3_handle_eee_event(adapter);

#endif
	 if (val & DRV_STATUS_OEM_UPDATE_SVID)
	    qfle3_handle_update_svid_cmd(adapter);

	 if (adapter->link_vars.periodic_flags & ELINK_PERIODIC_FLAGS_LINK_EVENT) {
	    /*
	     * sync with link
	     */
	    qfle3_acquire_phy_lock(adapter);
	    adapter->link_vars.periodic_flags &= ~ELINK_PERIODIC_FLAGS_LINK_EVENT;
	    qfle3_release_phy_lock(adapter);
#ifdef QFLE3_NPAR
	    if (IS_MF(adapter))
	       qfle3_link_sync_notify(adapter);
#endif
	    qfle3_acquire_phy_lock(adapter);
	    qfle3_report_link(adapter);
	    qfle3_release_phy_lock(adapter);
	 }
	 /*
	  * Always call it here: qfle3_report_link() will
	  * * prevent the link indication duplication.
	  */
         qfle3_link_status_update(adapter);
      } else if (attn & QFLE3_MC_ASSERT_BITS) {
         int donot_recover_flag;
         
         QFLE3_ERR("MC assert!\n");
         donot_recover_flag = vmk_BitVectorAtomicTestAndSet(adapter->sp_rtnl_state, QFLE3_SP_RTNL_DONOT_RECOVER);
         if (donot_recover_flag) {
            QFLE3_ERR("Driver is not able to recover from MC assert error at this point");
         }else {
            adapter->last_recovery_start = vmk_GetTimerCycles();
         }

         QFLE3_TRIGGER_TX_TO(adapter,QFLE3_ERR_MC_ASSERT);
         qfle3_mc_assert(adapter);
         REG_WR(adapter, MISC_REG_AEU_GENERAL_ATTN_10, 0);
         REG_WR(adapter, MISC_REG_AEU_GENERAL_ATTN_9, 0);
         REG_WR(adapter, MISC_REG_AEU_GENERAL_ATTN_8, 0);
         REG_WR(adapter, MISC_REG_AEU_GENERAL_ATTN_7, 0);
//         qfle3_panic(adapter);

      } else if (attn & QFLE3_MCP_ASSERT) {
         int donot_recover_flag;
         
         QFLE3_ERR("MCP assert!\n");
         donot_recover_flag = vmk_BitVectorAtomicTestAndSet(adapter->sp_rtnl_state, QFLE3_SP_RTNL_DONOT_RECOVER);
         if (donot_recover_flag) {
            QFLE3_ERR("Driver is not able to recover from MCP assert error at this point");
         }else {
            adapter->last_recovery_start = vmk_GetTimerCycles();
         }
         QFLE3_TRIGGER_TX_TO(adapter,QFLE3_ERR_MCP_ASSERT);
         REG_WR(adapter, MISC_REG_AEU_GENERAL_ATTN_11, 0);
         qfle3_fw_dump(adapter);

      } else
         QFLE3_ERR("Unknown HW assert! (attn 0x%x)\n", attn);
   }

   if (attn & EVEREST_LATCHED_ATTN_IN_USE_MASK) {
      QFLE3_ERR("LATCHED attention 0x%08x (masked)\n", attn);
      if (attn & QFLE3_GRC_TIMEOUT) {
         val = CHIP_IS_E1(adapter) ? 0 : REG_RD(adapter, MISC_REG_GRC_TIMEOUT_ATTN);
         QFLE3_ERR("GRC time-out 0x%08x\n", val);
      }
      if (attn & QFLE3_GRC_RSV) {
         val = CHIP_IS_E1(adapter) ? 0 : REG_RD(adapter, MISC_REG_GRC_RSV_ATTN);
         QFLE3_ERR("GRC reserved 0x%08x\n", val);
      }
      REG_WR(adapter, MISC_REG_AEU_CLR_LATCH_SIGNAL, 0x7ff);
   }
}

static void
qfle3_attn_int_deasserted4(struct qfle3_adapter *adapter, vmk_uint32 attn)
{
   vmk_uint32 val;
   if (attn & AEU_INPUTS_ATTN_BITS_PGLUE_HW_INTERRUPT) {

      val = REG_RD(adapter, PGLUE_B_REG_PGLUE_B_INT_STS_CLR);
      QFLE3_ERR("PGLUE hw attention 0x%x\n", val);
      if (val & PGLUE_B_PGLUE_B_INT_STS_REG_ADDRESS_ERROR)
         QFLE3_ERR("PGLUE_B_PGLUE_B_INT_STS_REG_ADDRESS_ERROR\n");
      if (val & PGLUE_B_PGLUE_B_INT_STS_REG_INCORRECT_RCV_BEHAVIOR)
         QFLE3_ERR("PGLUE_B_PGLUE_B_INT_STS_REG_INCORRECT_RCV_BEHAVIOR\n");
      if (val & PGLUE_B_PGLUE_B_INT_STS_REG_WAS_ERROR_ATTN)
         QFLE3_ERR("PGLUE_B_PGLUE_B_INT_STS_REG_WAS_ERROR_ATTN\n");
      if (val & PGLUE_B_PGLUE_B_INT_STS_REG_VF_LENGTH_VIOLATION_ATTN)
         QFLE3_ERR("PGLUE_B_PGLUE_B_INT_STS_REG_VF_LENGTH_VIOLATION_ATTN\n");
      if (val & PGLUE_B_PGLUE_B_INT_STS_REG_VF_GRC_SPACE_VIOLATION_ATTN)
         QFLE3_ERR("PGLUE_B_PGLUE_B_INT_STS_REG_VF_GRC_SPACE_VIOLATION_ATTN\n");
      if (val & PGLUE_B_PGLUE_B_INT_STS_REG_VF_MSIX_BAR_VIOLATION_ATTN)
         QFLE3_ERR("PGLUE_B_PGLUE_B_INT_STS_REG_VF_MSIX_BAR_VIOLATION_ATTN\n");
      if (val & PGLUE_B_PGLUE_B_INT_STS_REG_TCPL_ERROR_ATTN)
         QFLE3_ERR("PGLUE_B_PGLUE_B_INT_STS_REG_TCPL_ERROR_ATTN\n");
      if (val & PGLUE_B_PGLUE_B_INT_STS_REG_TCPL_IN_TWO_RCBS_ATTN)
         QFLE3_ERR("PGLUE_B_PGLUE_B_INT_STS_REG_TCPL_IN_TWO_RCBS_ATTN\n");
      if (val & PGLUE_B_PGLUE_B_INT_STS_REG_CSSNOOP_FIFO_OVERFLOW)
         QFLE3_ERR("PGLUE_B_PGLUE_B_INT_STS_REG_CSSNOOP_FIFO_OVERFLOW\n");
   }

   if (attn & AEU_INPUTS_ATTN_BITS_ATC_HW_INTERRUPT) {
      val = REG_RD(adapter, ATC_REG_ATC_INT_STS_CLR);
      QFLE3_ERR("ATC hw attention 0x%x\n", val);
      if (val & ATC_ATC_INT_STS_REG_ADDRESS_ERROR)
	 QFLE3_ERR("ATC_ATC_INT_STS_REG_ADDRESS_ERROR\n");
      if (val & ATC_ATC_INT_STS_REG_ATC_TCPL_TO_NOT_PEND)
	 QFLE3_ERR("ATC_ATC_INT_STS_REG_ATC_TCPL_TO_NOT_PEND\n");
      if (val & ATC_ATC_INT_STS_REG_ATC_GPA_MULTIPLE_HITS)
	 QFLE3_ERR("ATC_ATC_INT_STS_REG_ATC_GPA_MULTIPLE_HITS\n");
      if (val & ATC_ATC_INT_STS_REG_ATC_RCPL_TO_EMPTY_CNT)
	 QFLE3_ERR("ATC_ATC_INT_STS_REG_ATC_RCPL_TO_EMPTY_CNT\n");
      if (val & ATC_ATC_INT_STS_REG_ATC_TCPL_ERROR)
	 QFLE3_ERR("ATC_ATC_INT_STS_REG_ATC_TCPL_ERROR\n");
      if (val & ATC_ATC_INT_STS_REG_ATC_IREQ_LESS_THAN_STU)
	 QFLE3_ERR("ATC_ATC_INT_STS_REG_ATC_IREQ_LESS_THAN_STU\n");
   }

   if (attn & (AEU_INPUTS_ATTN_BITS_PGLUE_PARITY_ERROR |
	       AEU_INPUTS_ATTN_BITS_ATC_PARITY_ERROR)) {
      QFLE3_ERR("FATAL parity attention set4 0x%x\n",
		(vmk_uint32) (attn & (AEU_INPUTS_ATTN_BITS_PGLUE_PARITY_ERROR |
				      AEU_INPUTS_ATTN_BITS_ATC_PARITY_ERROR)));
   }
}

static void
qfle3_attn_int_deasserted(struct qfle3_adapter *adapter, vmk_uint32 deasserted)
{
   struct attn_route attn, *group_mask;
   int port = QFLE3_PORT(adapter);
   int index;
   vmk_uint32 reg_addr;
   vmk_uint32 val;
   vmk_uint32 aeu_mask;
   vmk_Bool global = VMK_FALSE;
   struct cnic_ctl_info ctl;
   /*
    * need to take HW lock because MCP or other port might also
    * try to handle this event
    */
   qfle3_acquire_alr(adapter);

   if (qfle3_chk_parity_attn(adapter, &global, VMK_TRUE)) {

      if (qfle3_parity_error(adapter)) {
         qfle3_release_alr(adapter);
         QFLE3_INFO("Parity Error cancelled. We are already in recovery \n");
         return;
      }
      
      
      adapter->recovery_state = QFLE3_RECOVERY_INIT;
      vmk_BitVectorAtomicTestAndSet(adapter->state, QFLE3_STATE_BIT_RESETTING);
      
      // send Recovery command to cnic
      ctl.cmd = CNIC_CTL_NIC_RECOVERY_CMD;
      
      if (adapter->cnicOps)
         adapter->cnicOps->cnicNotify(adapter->cnic_data, &ctl);
      
      QFLE3_INFO("Parity Error Occurred \n");

      qfle3_schedule_helper (adapter, adapter->recovery_helper, qfle3_recovery_func, 0);
      
      /*
       * Disable HW interrupts
       */
      qfle3_int_disable(adapter);
      /*
       * In case of parity errors don't handle attentions so that
       * * other function would "see" parity errors.
       */
      qfle3_release_alr(adapter);
      return;
   }
   attn.sig[0] = REG_RD(adapter, MISC_REG_AEU_AFTER_INVERT_1_FUNC_0 + port * 4);
   attn.sig[1] = REG_RD(adapter, MISC_REG_AEU_AFTER_INVERT_2_FUNC_0 + port * 4);
   attn.sig[2] = REG_RD(adapter, MISC_REG_AEU_AFTER_INVERT_3_FUNC_0 + port * 4);
   attn.sig[3] = REG_RD(adapter, MISC_REG_AEU_AFTER_INVERT_4_FUNC_0 + port * 4);
   attn.sig[4] = REG_RD(adapter, MISC_REG_AEU_AFTER_INVERT_5_FUNC_0 + port * 4);

   QFLE3_DBG(QFLE3_DBG_HW, "attn: %08x %08x %08x %08x %08x\n",
	     attn.sig[0], attn.sig[1], attn.sig[2], attn.sig[3], attn.sig[4]);

   for (index = 0; index < MAX_DYNAMIC_ATTN_GRPS; index++) {
      if (deasserted & (1 << index)) {
         group_mask = &adapter->attn_group[index];

         QFLE3_DBG(QFLE3_DBG_HW, "group[%d]: %08x %08x %08x %08x %08x\n",
            index,
            group_mask->sig[0], group_mask->sig[1],
            group_mask->sig[2], group_mask->sig[3], group_mask->sig[4]);

         qfle3_attn_int_deasserted4(adapter, attn.sig[4] & group_mask->sig[4]);
         qfle3_attn_int_deasserted3(adapter, attn.sig[3] & group_mask->sig[3]);
         qfle3_attn_int_deasserted1(adapter, attn.sig[1] & group_mask->sig[1]);
         qfle3_attn_int_deasserted2(adapter, attn.sig[2] & group_mask->sig[2]);
         qfle3_attn_int_deasserted0(adapter, attn.sig[0] & group_mask->sig[0]);
      }
   }
   qfle3_release_alr(adapter);

   if (adapter->hw_info.int_block == INT_BLOCK_HC)
      reg_addr = (HC_REG_COMMAND_REG + port * 32 + COMMAND_REG_ATTN_BITS_CLR);
   else
      reg_addr = (BAR_IGU_INTMEM + IGU_CMD_ATTN_BIT_CLR_UPPER * 8);

   val = ~deasserted;
   QFLE3_DBG(QFLE3_DBG_HW, "about to mask 0x%08x at %s addr 0x%x\n", val,
	     (adapter->hw_info.int_block == INT_BLOCK_HC) ? "HC" : "IGU", reg_addr);
   REG_WR(adapter, reg_addr, val);

   if (~adapter->attn_state & deasserted)
      QFLE3_ERR("IGU ERROR\n");

   reg_addr = port ? MISC_REG_AEU_MASK_ATTN_FUNC_1 : MISC_REG_AEU_MASK_ATTN_FUNC_0;

   qfle3_acquire_hw_lock(adapter, HW_LOCK_RESOURCE_PORT0_ATT_MASK + port);
   aeu_mask = REG_RD(adapter, reg_addr);

   QFLE3_DBG(QFLE3_DBG_HW, "aeu_mask %x  newly deasserted %x\n", aeu_mask, deasserted);
   aeu_mask |= (deasserted & 0x3ff);
   QFLE3_DBG(QFLE3_DBG_HW, "new mask %x\n", aeu_mask);

   REG_WR(adapter, reg_addr, aeu_mask);
   qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_PORT0_ATT_MASK + port);

   QFLE3_DBG(QFLE3_DBG_HW, "attn_state %x\n", adapter->attn_state);
   adapter->attn_state &= ~deasserted;
   QFLE3_DBG(QFLE3_DBG_HW, "new state %x\n", adapter->attn_state);
}

static void
qfle3_link_attn(struct qfle3_adapter *adapter)
{
   QFLE3_DBG(QFLE3_DBG_LINK, "Link attention\n");

   QFLE3_DBG(QFLE3_DBG_STATS, "Executing qfle3_stats_handle for STATS_EVENT_STOP\n");
   qfle3_stats_handle(adapter, STATS_EVENT_STOP);

   elink_link_update(&adapter->link_params, &adapter->link_vars);

   qfle3_init_dropless_fc(adapter);

   if (adapter->link_vars.link_up) {
      if (adapter->link_vars.mac_type != ELINK_MAC_TYPE_EMAC) {
            struct host_port_stats *pstats;

            pstats = QFLE3_SP(adapter, port_stats);
            /* reset old mac stats */
            vmk_Memset(&(pstats->mac_stx[0]), 0, sizeof(struct mac_stx));
      }
      if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_LOADED)) {
            QFLE3_DBG(QFLE3_DBG_STATS, "Executing qfle3_stats_handle for STATS_EVENT_LINK_UP\n");
            qfle3_stats_handle(adapter, STATS_EVENT_LINK_UP);
      }
   }

   if (adapter->link_vars.link_up && adapter->link_vars.line_speed) {
      qfle3_set_local_cmng(adapter);
   }

   qfle3_report_link(adapter);

#ifdef QFLE3_NPAR
   if (IS_MF(adapter))
      qfle3_link_sync_notify(adapter);
#endif
}

static void
qfle3_attn_int_asserted(struct qfle3_adapter *adapter, vmk_uint32 asserted)
{
   int port = QFLE3_PORT(adapter);
   vmk_uint32 aeu_addr = port ? MISC_REG_AEU_MASK_ATTN_FUNC_1 :
      MISC_REG_AEU_MASK_ATTN_FUNC_0;
   vmk_uint32 nig_int_mask_addr = port ? NIG_REG_MASK_INTERRUPT_PORT1 :
      NIG_REG_MASK_INTERRUPT_PORT0;
   vmk_uint32 aeu_mask;
   vmk_uint32 nig_mask = 0;
   vmk_uint32 reg_addr;

   if (adapter->attn_state & asserted)
      QFLE3_ERR("IGU ERROR\n");

   qfle3_acquire_hw_lock(adapter, HW_LOCK_RESOURCE_PORT0_ATT_MASK + port);
   aeu_mask = REG_RD(adapter, aeu_addr);

   QFLE3_DBG(QFLE3_DBG_HW, "aeu_mask %x  newly asserted %x\n", aeu_mask, asserted);
   aeu_mask &= ~(asserted & 0x3ff);
   QFLE3_DBG(QFLE3_DBG_HW, "new mask %x\n", aeu_mask);

   REG_WR(adapter, aeu_addr, aeu_mask);
   qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_PORT0_ATT_MASK + port);

   QFLE3_DBG(QFLE3_DBG_HW, "attn_state %x\n", adapter->attn_state);
   adapter->attn_state |= asserted;
   QFLE3_DBG(QFLE3_DBG_HW, "new state %x\n", adapter->attn_state);

   if (asserted & ATTN_HARD_WIRED_MASK) {
      if (asserted & ATTN_NIG_FOR_FUNC) {

	 qfle3_acquire_phy_lock(adapter);

	 /*
	  * save nig interrupt mask
	  */
	 nig_mask = REG_RD(adapter, nig_int_mask_addr);

	 /*
	  * If nig_mask is not set, no need to call the update
	  * * function.
	  */
	 if (nig_mask) {
	    REG_WR(adapter, nig_int_mask_addr, 0);

	    qfle3_link_attn(adapter);
	 }

	 /*
	  * handle unicore attn?
	  */
      }
      if (asserted & ATTN_SW_TIMER_4_FUNC)
	 QFLE3_DBG(QFLE3_DBG_HW, "ATTN_SW_TIMER_4_FUNC!\n");

      if (asserted & GPIO_2_FUNC)
	 QFLE3_DBG(QFLE3_DBG_HW, "GPIO_2_FUNC!\n");

      if (asserted & GPIO_3_FUNC)
	 QFLE3_DBG(QFLE3_DBG_HW, "GPIO_3_FUNC!\n");

      if (asserted & GPIO_4_FUNC)
	 QFLE3_DBG(QFLE3_DBG_HW, "GPIO_4_FUNC!\n");

      if (port == 0) {
	 if (asserted & ATTN_GENERAL_ATTN_1) {
	    QFLE3_DBG(QFLE3_DBG_HW, "ATTN_GENERAL_ATTN_1!\n");
	    REG_WR(adapter, MISC_REG_AEU_GENERAL_ATTN_1, 0x0);
	 }
	 if (asserted & ATTN_GENERAL_ATTN_2) {
	    QFLE3_DBG(QFLE3_DBG_HW, "ATTN_GENERAL_ATTN_2!\n");
	    REG_WR(adapter, MISC_REG_AEU_GENERAL_ATTN_2, 0x0);
	 }
	 if (asserted & ATTN_GENERAL_ATTN_3) {
	    QFLE3_DBG(QFLE3_DBG_HW, "ATTN_GENERAL_ATTN_3!\n");
	    REG_WR(adapter, MISC_REG_AEU_GENERAL_ATTN_3, 0x0);
	 }
      } else {
	 if (asserted & ATTN_GENERAL_ATTN_4) {
	    QFLE3_DBG(QFLE3_DBG_HW, "ATTN_GENERAL_ATTN_4!\n");
	    REG_WR(adapter, MISC_REG_AEU_GENERAL_ATTN_4, 0x0);
	 }
	 if (asserted & ATTN_GENERAL_ATTN_5) {
	    QFLE3_DBG(QFLE3_DBG_HW, "ATTN_GENERAL_ATTN_5!\n");
	    REG_WR(adapter, MISC_REG_AEU_GENERAL_ATTN_5, 0x0);
	 }
	 if (asserted & ATTN_GENERAL_ATTN_6) {
	    QFLE3_DBG(QFLE3_DBG_HW, "ATTN_GENERAL_ATTN_6!\n");
	    REG_WR(adapter, MISC_REG_AEU_GENERAL_ATTN_6, 0x0);
	 }
      }

   }
   /*
    * if hardwired
    */
   if (adapter->hw_info.int_block == INT_BLOCK_HC)
      reg_addr = (HC_REG_COMMAND_REG + port * 32 + COMMAND_REG_ATTN_BITS_SET);
   else
      reg_addr = (BAR_IGU_INTMEM + IGU_CMD_ATTN_BIT_SET_UPPER * 8);

   QFLE3_DBG(QFLE3_DBG_HW, "about to mask 0x%08x at %s addr 0x%x\n", asserted,
	     (adapter->hw_info.int_block == INT_BLOCK_HC) ? "HC" : "IGU", reg_addr);
   REG_WR(adapter, reg_addr, asserted);

   /*
    * now set back the mask
    */
   if (asserted & ATTN_NIG_FOR_FUNC) {
      /*
       * Verify that IGU ack through BAR was written before restoring
       * * NIG mask. This loop should exit after 2-3 iterations max.
       */
      if (adapter->hw_info.int_block != INT_BLOCK_HC) {
	 vmk_uint32 cnt = 0, igu_acked;
	 do {
	    igu_acked = REG_RD(adapter, IGU_REG_ATTENTION_ACK_BITS);
	 } while (((igu_acked & ATTN_NIG_FOR_FUNC) == 0) &&
		  (++cnt < MAX_IGU_ATTN_ACK_TO));
	 if (!igu_acked)
	    QFLE3_DBG(QFLE3_DBG_HW, "Failed to verify IGU ack on time\n");
	 vmk_CPUMemFenceReadWrite();
      }
      REG_WR(adapter, nig_int_mask_addr, nig_mask);
      qfle3_release_phy_lock(adapter);
   }
}

static void
qfle3_attn_int(struct qfle3_adapter *adapter)
{
   /*
    * read local copy of bits
    */
   vmk_uint32 attn_bits = le32toh(adapter->def_status_blk->atten_status_block.attn_bits);
   vmk_uint32 attn_ack =
      le32toh(adapter->def_status_blk->atten_status_block.attn_bits_ack);
   vmk_uint32 attn_state = adapter->attn_state;

   /*
    * look for changed bits
    */
   vmk_uint32 asserted = attn_bits & ~attn_ack & ~attn_state;
   vmk_uint32 deasserted = ~attn_bits & attn_ack & attn_state;

   QFLE3_DBG(QFLE3_DBG_HW,
	     "attn_bits %x  attn_ack %x  asserted %x  deasserted %x\n",
	     attn_bits, attn_ack, asserted, deasserted);

   if (~(attn_bits ^ attn_ack) & (attn_bits ^ attn_state))
      QFLE3_ERROR("BAD attention state\n");

   
//   if (adapter->trigger_error == QFLE3_TRIGGER_FATAL_HW_ERROR){
//      qfle3_panic(adapter);
//      adapter->trigger_error = 0;
//   }

   /*
    * handle bits that were raised
    */
   if (asserted)
      qfle3_attn_int_asserted(adapter, asserted);

   if (deasserted)
      qfle3_attn_int_deasserted(adapter, deasserted);
}
void
qfle3_periodic_task_func(vmk_AddrCookie data)
{
   qfle3_adapter *adapter = data.ptr;

   qfle3_acquire_phy_lock(adapter);

   vmk_CPUMemFenceReadWrite();
   if (adapter->port.pmf) {
      if (adapter->phy_initialized)
         elink_period_func(&adapter->link_params, &adapter->link_vars);

      qfle3_schedule_helper(adapter,
			    adapter->periodic_task, qfle3_periodic_task_func, 1000);
   }

   qfle3_release_phy_lock(adapter);
   //QFLE3_DBG(QFLE3_DBG_SP, "periodic task end.\n");
}
static inline int
qfle3_has_rx_work(struct qfle3_fastpath *fp)
{
   vmk_uint16 rx_cq_cons_sb;
   struct qfle3_adapter *adapter = fp->adapter;
   
   //vmk_LogMessage("rx_cq_cons_sb is %d, rx_cq_cons %d ", le16toh(*fp->rx_cq_cons_sb), fp->rx_cq_cons);
   /*
    * status block fields can change
    */
   rx_cq_cons_sb = le16toh(*fp->rx_cq_cons_sb);
   if ((rx_cq_cons_sb & RCQ_MAX) == RCQ_MAX)
      rx_cq_cons_sb++;
   return (fp->rx_cq_cons != rx_cq_cons_sb);
}

//static inline vmk_int32
//qfle3_tx_queue_has_work(struct qfle3_fastpath *fp)
//{
//   vmk_uint16 hw_cons;
//   //vmk_LogMessage("tx_cons_sb is %d, tx_pkt_cons %d", le16toh(*fp->tx_cons_sb), fp->tx_pkt_cons);
//   /*
//    * status block fields can change
//    */
//   hw_cons = le16toh(*fp->tx_cons_sb);
//   return (hw_cons != fp->tx_pkt_cons);
//}

//static inline vmk_uint8
//qfle3_has_tx_work(struct qfle3_fastpath *fp)
//{
//   /*
//    * expand this for multi-cos if ever supported
//    */
//   return (qfle3_tx_queue_has_work(fp)) ? VMK_TRUE : VMK_FALSE;
//}

int
qfle3_has_fcoe_rx_work(struct qfle3_fastpath *fp)
{
//   vmk_uint16 rx_cq_cons_sb;
//   struct qfle3_adapter *adapter = fp->adapter;
//   
//   vmk_LogMessage("rx_cq_cons_sb is %d, rx_cq_cons %d", le16toh(*fp->rx_cq_cons_sb), fp->rx_cq_cons);
//   /*
//    * status block fields can change
//    */
//   rx_cq_cons_sb = le16toh(*fp->rx_cq_cons_sb);
//   if ((rx_cq_cons_sb & RCQ_MAX) == RCQ_MAX)
//      rx_cq_cons_sb++;
//   return (fp->rx_cq_cons != rx_cq_cons_sb);

   u16 cons;
   union eth_rx_cqe *cqe;
   struct eth_fast_path_rx_cqe *cqe_fp;

   cons = fp->rx_cq_cons;
   cqe = &fp->rcq_chain[cons];
   cqe_fp = &cqe->fast_path_cqe;
   
   //vmk_LogMessage("rx_cq_cons_sb is %d, rx_cq_cons %d cqe marker %x", le16toh(*fp->rx_cq_cons_sb), fp->rx_cq_cons, cqe_fp->marker);
   return (QFLE3_IS_CQE_COMPLETED(cqe_fp));

}

void
qfle3_sp_helper_func(vmk_AddrCookie data)
{
   qfle3_adapter *adapter = data.ptr;
   QFLE3_DBG(QFLE3_DBG_SP, "sp task invoked\n");

   /*
    * make sure the atomic interrupt_occurred has been written
    */
   vmk_CPUMemFenceReadWrite();

   /*
    * what work needs to be performed?
    */
   vmk_uint16 status = qfle3_update_dsb_idx(adapter);

   QFLE3_DBG(QFLE3_DBG_SP, "status %x\n", status);
   /*
    * HW attentions
    */

   if (status & QFLE3_DEF_SB_ATT_IDX) {
      qfle3_attn_int(adapter);
      status &= ~QFLE3_DEF_SB_ATT_IDX;
   }

   /*
    * SP events: STAT_QUERY and others
    */
   if (status & QFLE3_DEF_SB_IDX) {
#ifdef QFLE3_CNIC      
      struct qfle3_fastpath *fp = qfle3_fcoe_fp(adapter);
      VMK_ReturnStatus poll_status;

      if (FCOE_INIT(adapter)){
         
         QFLE3_DBG(QFLE3_DBG_SP, "CHECK FCOE work %x\n", status);
            
         if (qfle3_has_rx_work(fp)||qfle3_has_tx_work(fp)) {
            QFLE3_DBG(QFLE3_DBG_SP, "FCOE TX/RX work %x\n", status);
            
            /* run netpoll on this queue */
            poll_status = vmk_NetPollActivate(fp->netpoll);
            if(poll_status != VMK_OK) {
               QFLE3_ERROR("Failed to activate FCOE poll");
            }
         }
      }
#endif      
      /*
       * Handle EQ completions
       */
      qfle3_eq_int(adapter);
      
      QFLE3_DBG(QFLE3_DBG_SP, "after EQ int\n");
      qfle3_ack_sb(adapter, adapter->igu_dsb_id, USTORM_ID,
		   le16toh(adapter->def_idx), IGU_INT_NOP, 1);

      status &= ~QFLE3_DEF_SB_IDX;
   }

   /*
    * if status is non zero then perhaps something went wrong
    */
   if (VMK_UNLIKELY(status))
      QFLE3_ERROR("got an unknown interrupt! (status 0x%x)\n", status);

   /*
    * ack status block only if something was actually handled
    */
   qfle3_ack_sb(adapter, adapter->igu_dsb_id, ATTENTION_ID,
		le16toh(adapter->def_att_idx), IGU_INT_ENABLE, 1);

#if 0
   /*
    * afex - poll to check if VIFSET_ACK should be sent to MFW
    */
   if (test_and_clear_bit(QFLE3_AFEX_PENDING_VIFSET_MCP_ACK, &adapter->sp_state)) {
      qfle3_report_link(adapter);
      qfle3_mfw_command(adapter, DRV_MSG_CODE_AFEX_VIFSET_ACK, 0);
   }
#endif
#ifdef QFLE3_CNIC
//   vmk_SemaLock(&adapter->cnic_mutex);
//   if (adapter->cnicIRQHandler != NULL)
//      adapter->cnicIRQHandler(adapter->cnic_data, adapter->def_status_blk);
//   vmk_SemaUnlock(&adapter->cnic_mutex);
#endif
}


#ifdef QFLE3_SRIOV

static struct qfle3_virtf *qfle3_vf_by_cid(struct qfle3_adapter *adapter,int vf_cid)
{
   /* extract the vf from vf_cid - relies on the following:
   * 1. vfid on cid reflects the true abs_vfid
   * 2. The max number of VFs (per path) is 64
   */
   int abs_vfid = (vf_cid >> QFLE3_VF_CID_WND) & (QFLE3_MAX_NUM_OF_VFS-1); 
   return qfle3_vf_by_abs_fid(adapter, abs_vfid);
}

void qfle3_iov_set_queue_sp_obj(struct qfle3_adapter *adapter, int vf_cid,
                                    struct ecore_queue_sp_obj **q_obj)
{
   struct qfle3_virtf *vf;
   if(!(adapter->vfdb)){
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Adapter vfdb not allocated yet\n");
      return;
   }

   vf = qfle3_vf_by_cid(adapter, vf_cid);
   if (vf) {
      /* extract queue index from vf_cid - relies on the following:
      * 1. vfid on cid reflects the true abs_vfid
      * 2. The max number of VFs (per path) is 64
      */
      int q_index = vf_cid & ((1 << QFLE3_VF_CID_WND)-1);
      *q_obj = &qfle3_vfq(vf, q_index, sp_obj);
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Found Vf with matching cid %d\n", vf_cid);
   } else {
      QFLE3_ERR("SRIOV: No vf matching cid %d\n", vf_cid);
   }


}
#endif //QFLE3_SRIOV

void qfle3_sp_event(struct qfle3_fastpath *fp, union eth_rx_cqe *rr_cqe)
{
   struct qfle3_adapter *adapter = fp->adapter;
   int cid = SW_CID(rr_cqe->ramrod_cqe.conn_and_cmd_data);
   int command = CQE_CMD(rr_cqe->ramrod_cqe.conn_and_cmd_data);
   enum ecore_queue_cmd drv_cmd = ECORE_Q_CMD_MAX;
   struct ecore_queue_sp_obj *q_obj = &QFLE3_SP_OBJ(adapter, fp).q_obj;

   QFLE3_DBG(QFLE3_DBG_SP,
	     "fp %d  cid %d  got ramrod #%d  type is %d\n",
	     fp->qid, cid, command,
	     rr_cqe->ramrod_cqe.ramrod_type);

#ifdef QFLE3_SRIOV
   /* If cid is within VF range, replace the slowpath object with the
    * one corresponding to this VF
    */
   if (cid >= QFLE3_FIRST_VF_CID  &&
       cid < QFLE3_FIRST_VF_CID + QFLE3_VF_CIDS)
      qfle3_iov_set_queue_sp_obj(adapter, cid, &q_obj);
#endif //QFLE3_SRIOV

   switch (command) {
   case (RAMROD_CMD_ID_ETH_CLIENT_UPDATE):
      QFLE3_DBG(QFLE3_DBG_SP, "got UPDATE ramrod. CID %d\n", cid);
      drv_cmd = ECORE_Q_CMD_UPDATE;
      break;

   case (RAMROD_CMD_ID_ETH_CLIENT_SETUP):
      QFLE3_DBG(QFLE3_DBG_SP, "got MULTI[%d] setup ramrod\n", cid);
      drv_cmd = ECORE_Q_CMD_SETUP;
      break;

   case (RAMROD_CMD_ID_ETH_TX_QUEUE_SETUP):
      QFLE3_DBG(QFLE3_DBG_SP, "got MULTI[%d] tx-only setup ramrod\n", cid);
      drv_cmd = ECORE_Q_CMD_SETUP_TX_ONLY;
      break;

   case (RAMROD_CMD_ID_ETH_HALT):
      QFLE3_DBG(QFLE3_DBG_SP, "got MULTI[%d] halt ramrod\n", cid);
      drv_cmd = ECORE_Q_CMD_HALT;
      break;
   case (RAMROD_CMD_ID_ETH_TERMINATE):
      QFLE3_DBG(QFLE3_DBG_SP, "got MULTI[%d] terminate ramrod\n", cid);
      drv_cmd = ECORE_Q_CMD_TERMINATE;
      break;

   case (RAMROD_CMD_ID_ETH_EMPTY):
      QFLE3_DBG(QFLE3_DBG_SP, "got MULTI[%d] empty ramrod\n", cid);
      drv_cmd = ECORE_Q_CMD_EMPTY;
      break;

   case (RAMROD_CMD_ID_ETH_TPA_UPDATE):
      QFLE3_DBG(QFLE3_DBG_SP, "got tpa update ramrod CID=%d\n", cid);
      drv_cmd = ECORE_Q_CMD_UPDATE_TPA;
      break;

   default:
      QFLE3_ERR("unexpected MC reply (%d) on fp[%d]\n",
		command, fp->qid);
      return;
   }

   if ((drv_cmd != ECORE_Q_CMD_MAX) &&
       q_obj->complete_cmd(adapter, q_obj, drv_cmd)) {
      /* q_obj->complete_cmd() failure means that this was
       * an unexpected completion.
       *
       * In this case we don't want to increase the adapter->spq_left
       * because apparently we haven't sent this command the first
       * place.
       */
#ifdef QFLE3_STOP_ON_ERROR
      qfle3_panic();
#else

      QFLE3_ERR("unexpected completion (%d) on fp[%d]\n",
		command, fp->qid);
      return;
   }
#endif

   vmk_CPUMemFenceReadWrite();
   vmk_AtomicInc64(&adapter->cq_spq_left);
   /* push the change in adapter->spq_left and towards the memory */
   vmk_CPUMemFenceReadWrite();

   QFLE3_DBG(QFLE3_DBG_SP, "adapter->cq_spq_left %lx\n", vmk_AtomicRead64(&adapter->cq_spq_left));

   if ((drv_cmd == ECORE_Q_CMD_UPDATE) && (IS_FCOE_FP(fp)) &&
       (!!ECORE_TEST_BIT(ECORE_AFEX_FCOE_Q_UPDATE_PENDING, &adapter->sp_state))) {
      /* if Q update ramrod is completed for last Q in AFEX vif set
       * flow, then ACK MCP at the end
       *
       * mark pending ACK to MCP bit.
       * prevent case that both bits are cleared.
       * At the end of load/unload driver checks that
       * sp_state is cleared, and this order prevents
       * races
       */
      vmk_CPUMemFenceReadWrite();
      ECORE_SET_BIT(ECORE_AFEX_PENDING_VIFSET_MCP_ACK, &adapter->sp_state);
      vmk_CPUMemFenceReadWrite();
      ECORE_CLEAR_BIT(ECORE_AFEX_FCOE_Q_UPDATE_PENDING, &adapter->sp_state);
      vmk_CPUMemFenceReadWrite();

      /* schedule the sp task as mcp ack is required */
      qfle3_schedule_helper(adapter, adapter->sp_helper, qfle3_sp_helper_func, 0);
   }

   return;
}

static struct dmp_config dmpcfg;

static void esx_reg_wr_ind(struct qfle3_adapter *adapter, vmk_uint32 addr, vmk_uint32 val)
{
   VMK_ReturnStatus status;

   status = vmk_PCIWriteConfig(vmk_ModuleCurrentID,
			       adapter->pdev,
			       VMK_PCI_CONFIG_ACCESS_32, PCICFG_GRC_ADDRESS, addr);
   if (status != VMK_OK) {
      QFLE3_ERROR("Error writing PCI command register\n");
   }

   status = vmk_PCIWriteConfig(vmk_ModuleCurrentID,
			       adapter->pdev,
			       VMK_PCI_CONFIG_ACCESS_32, PCICFG_GRC_DATA, val);
   if (status != VMK_OK) {
      QFLE3_ERROR("Error writing PCI command register\n");
   }

   status = vmk_PCIWriteConfig(vmk_ModuleCurrentID,
			       adapter->pdev,
			       VMK_PCI_CONFIG_ACCESS_32, PCICFG_GRC_ADDRESS, 0);
   if (status != VMK_OK) {
      QFLE3_ERROR("Error writing PCI command register\n");
   }

}

static vmk_uint32 esx_reg_rd_ind(struct qfle3_adapter *adapter, vmk_uint32 addr)
{
   VMK_ReturnStatus status;
   vmk_uint32 val;

   status = vmk_PCIWriteConfig(vmk_ModuleCurrentID,
			       adapter->pdev,
			       VMK_PCI_CONFIG_ACCESS_32, PCICFG_GRC_ADDRESS, addr);
   if (status != VMK_OK) {
      QFLE3_ERROR("Error writing PCI command register\n");
   }

   status = vmk_PCIReadConfig(vmk_ModuleCurrentID,
			      adapter->pdev,
			      VMK_PCI_CONFIG_ACCESS_32, PCICFG_GRC_DATA, &val);

   if (status != VMK_OK) {
      QFLE3_ERROR("Error reading PCI command register\n");
   }

   status = vmk_PCIWriteConfig(vmk_ModuleCurrentID,
			       adapter->pdev,
			       VMK_PCI_CONFIG_ACCESS_32, PCICFG_GRC_ADDRESS, 0);
   if (status != VMK_OK) {
      QFLE3_ERROR("Error writing PCI command register\n");
   }

   return val;
}

static vmk_uint32 *
dmp_stop_timer(struct qfle3_adapter *adapter, vmk_uint32 *dst, struct chip_core_dmp *dmp)
{
   vmk_uint32 i, j;
   vmk_uint32 *timer_scan_reg;

   if ((dmp->fw_hdr.dmp_size + (2 * dmpcfg.regs_timer_count)) >=
       DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0) {
      SET_FLAGS(dmp->fw_hdr.flags,
		FWDMP_FLAGS_SPACE_NEEDED);
      return dst;
   }
   if (dmp->fw_hdr.flags & FWDMP_FLAGS_LIVE_DUMP) {
      /* driver shouldn't read the timer in online
	 since it could cause attention. However,
	 it should fills the buffer and move on */
      for (i = 0;
	   i < 2 * dmpcfg.regs_timer_count;
	   i++) {
	 *dst++ = 0;
	 dmp->fw_hdr.dmp_size += 4;
      }
      return dst;
   }
   for (i = 0; i < dmpcfg.regs_timer_count; i++) {
      *dst = RD_IND(adapter, dmpcfg.regs_timer_status_addrs[i]);
      timer_scan_reg = dst + dmpcfg.regs_timer_count;
      if (*dst == 1) {
	 WR_IND(adapter, dmpcfg.regs_timer_status_addrs[i], 0);
	 for (j = 0; j < DRV_DUMP_MAX_TIMER_PENDING; j++) {
	    *timer_scan_reg = RD_IND(adapter,
				     dmpcfg.regs_timer_scan_addrs[i]);
	    if (*timer_scan_reg == 0)
	       break;
	 }
      } else {
	 *timer_scan_reg = DRV_DUMP_TIMER_SCAN_DONT_CARE;
      }
      dst++;
      dmp->fw_hdr.dmp_size += 4;
   }
   dst += dmpcfg.regs_timer_count;
   dmp->fw_hdr.dmp_size += 4 * dmpcfg.regs_timer_count;
   return dst;
}

static vmk_uint32 *
read_idle_chk(struct qfle3_adapter *adapter, vmk_uint32 *dst, struct chip_core_dmp *dmp)
{
   vmk_uint32 i, j;

   /* Read the idle chk registers */
   for (i = 0; i < IDLE_REGS_COUNT; i++)
      if (dmpcfg.mode == (idle_addrs[i].info & dmpcfg.mode))
	 for (j = 0; j < idle_addrs[i].size; j++) {
	    if ((dmp->fw_hdr.dmp_size + 4) >
		DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0) {
	       SET_FLAGS(dmp->fw_hdr.flags,
			 FWDMP_FLAGS_SPACE_NEEDED);
	       break;
	    }
	    *dst++ = RD_IND(adapter,
			    idle_addrs[i].addr + j*4);
	    dmp->fw_hdr.dmp_size += 4;
	 }
   return dst;
}

static void
dmp_rollback_timer(struct qfle3_adapter *adapter,
		   struct chip_core_dmp *dmp,
		   vmk_uint32 *tmr_status)
{
   vmk_uint32 i;

   if (dmp->fw_hdr.flags & FWDMP_FLAGS_LIVE_DUMP)
      return;
   for (i = 0; i < dmpcfg.regs_timer_count; i++) {
      if (*tmr_status == 1)
	 WR_IND(adapter, dmpcfg.regs_timer_status_addrs[i], 1);
      tmr_status++;
   }
}


static vmk_uint32 *
read_mcp_traces(struct qfle3_adapter *adapter, vmk_uint32 *dst, struct chip_core_dmp *dmp)
{
   vmk_uint32 trace_shmem_base;
   vmk_uint32 addr;
   vmk_uint32 i;

   if (dmp->fw_hdr.dmp_size + DBG_DMP_TRACE_BUFFER_SIZE >
       DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0) {
      SET_FLAGS(dmp->fw_hdr.flags, FWDMP_FLAGS_SPACE_NEEDED);
      return dst;
   }

   if (QFLE3_PATH(adapter) == 0)
      trace_shmem_base = adapter->hw_info.shmem_base;
   else
      trace_shmem_base = SHMEM2_RD(adapter, other_shmem_base_addr);

   addr = trace_shmem_base - DBG_DMP_TRACE_BUFFER_SIZE;
   for (i = 0; i < DBG_DMP_TRACE_BUFFER_SIZE;) {
      if ((dmp->fw_hdr.dmp_size + 4)
	  > DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0) {
	 SET_FLAGS(dmp->fw_hdr.flags, FWDMP_FLAGS_SPACE_NEEDED);
	 break;
      }
      *dst++ = RD_IND(adapter, addr + i);
      i += 4;
      dmp->fw_hdr.dmp_size += 4;
   }
   return dst;
}

static vmk_uint32 *
read_regular_regs(struct qfle3_adapter *adapter, vmk_uint32 *dst, struct chip_core_dmp *dmp)
{
   vmk_uint32 i, j;

   /* Read the regular address registers */
   for (i = 0; i < REGS_COUNT; i++)
      if (dmpcfg.mode == (reg_addrs[i].info & dmpcfg.mode))
	 for (j = 0; j < reg_addrs[i].size; j++) {
	    if ((dmp->fw_hdr.dmp_size + 4) >
		DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0) {
	       SET_FLAGS(dmp->fw_hdr.flags,
			 FWDMP_FLAGS_SPACE_NEEDED);
	       break;
	    }
	    *dst++ = RD_IND(adapter, reg_addrs[i].addr + j*4);
	    dmp->fw_hdr.dmp_size += 4;
	 }
   return dst;
}

static vmk_uint32 *
read_wregs(struct qfle3_adapter *adapter, vmk_uint32 *dst, struct chip_core_dmp *dmp)
{
   vmk_uint32 i, j, k;
   vmk_uint32 reg_add_read = 0;
   const struct wreg_addr *pwreg_addrs = dmpcfg.pwreg_addrs;

   for (i = 0; i < dmpcfg.wregs_count; i++) {
      if (dmpcfg.mode == (pwreg_addrs[i].info & dmpcfg.mode)) {
	 reg_add_read = pwreg_addrs[i].addr;
	 for (j = 0; j < pwreg_addrs[i].size; j++) {
	    if ((dmp->fw_hdr.dmp_size + 4) >
		DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0) {
	       SET_FLAGS(dmp->fw_hdr.flags,
			 FWDMP_FLAGS_SPACE_NEEDED);
	       break;
	    }
	    *dst++ = RD_IND(adapter, reg_add_read);
	    reg_add_read += 4;
	    dmp->fw_hdr.dmp_size += 4;
	    for (k = 0;
		 k < pwreg_addrs[i].read_regs_count; k++) {
	       if ((dmp->fw_hdr.dmp_size + 4) >
		   DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0) {
		  SET_FLAGS(dmp->fw_hdr.flags,
			    FWDMP_FLAGS_SPACE_NEEDED);
		  break;
	       }
	       *dst++ = RD_IND(adapter,
			       pwreg_addrs[i].read_regs[k]);
	       dmp->fw_hdr.dmp_size += 4;
	    }
	 }
      }
   }
   return dst;
}

static vmk_uint32 *
read_page_mode(struct qfle3_adapter *adapter, vmk_uint32 *dst, struct chip_core_dmp *dmp)
{
   vmk_uint32 index_page_vals       = 0;
   vmk_uint32 index_page_write_regs = 0;
   const struct reg_addr *page_read_regs = dmpcfg.page_read_regs;
   vmk_uint32 i, j;

   /* If one of the array size is zero, this means that
      page mode is disabled. */
   if ((0 == dmpcfg.page_mode_values_count) ||
       (0 == dmpcfg.page_write_regs_count) ||
       (0 == dmpcfg.page_read_regs_count))
      return dst;

   for (index_page_vals = 0;
	index_page_vals < dmpcfg.page_mode_values_count;
	index_page_vals++) {
      for (index_page_write_regs = 0;
	   index_page_write_regs < dmpcfg.page_write_regs_count;
	   index_page_write_regs++) {
	 WR_IND(adapter,
		dmpcfg.page_write_regs[index_page_write_regs],
		dmpcfg.page_vals[index_page_vals]);
      }
      for (i = 0; i < dmpcfg.page_read_regs_count; i++) {
	 if (dmpcfg.mode ==
	     (page_read_regs[i].info & dmpcfg.mode)) {
	    for (j = 0; j < page_read_regs[i].size; j++) {
	       if ((dmp->fw_hdr.dmp_size + 4) >
		   DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0) {
		  SET_FLAGS(dmp->fw_hdr.flags,
			    FWDMP_FLAGS_SPACE_NEEDED);
		  break;
	       }
	       *dst++ = RD_IND(adapter,
			       page_read_regs[i].addr + j*4);
	       dmp->fw_hdr.dmp_size += 4;
	    }
	 }
      }
   }
   return dst;
}

static void
get_vfc_info(struct qfle3_adapter *adapter,
	     const struct vfc_general **xvfc_info,
	     const struct vfc_general **tvfc_info)
{
   if (CHIP_IS_E1(adapter)) {
      *xvfc_info = &vfc_general_x_e1;
      *tvfc_info = &vfc_general_t_e1;
   } else if (CHIP_IS_E1H(adapter)) {
      *xvfc_info = &vfc_general_x_e1h;
      *tvfc_info = &vfc_general_t_e1h;
   } else if (CHIP_IS_E2(adapter)) {
      *xvfc_info = &vfc_general_x_e2;
      *tvfc_info = &vfc_general_t_e2;
   } else if (CHIP_IS_E3(adapter)) {
      *xvfc_info = &vfc_general_x_e3;
      *tvfc_info = &vfc_general_t_e3;
   } else {
      *xvfc_info = NULL;
      *tvfc_info = NULL;
   }
}

static void get_vfc_ops(struct qfle3_adapter *adapter,
			const struct vfc_read_task **xvfc_ops,
			const struct vfc_read_task **tvfc_ops)
{
   if (CHIP_IS_E2(adapter)) {
      *xvfc_ops = &vfc_task_x_e2;
      *tvfc_ops = &vfc_task_t_e2;
   } else if (CHIP_IS_E3(adapter)) {
      *xvfc_ops = &vfc_task_x_e3;
      *tvfc_ops = &vfc_task_t_e3;
   } else {
      *xvfc_ops = NULL;
      *tvfc_ops = NULL;
   }
}

static void
init_extension_header(vmk_uint32 data_type,
		      vmk_uint32 data_source,
		      struct extension_hdr *header)
{
   header->hdr_signature = HDR_SIGNATURE;
   header->hdr_size = (sizeof(*header) -
		       sizeof(header->hdr_size)) / sizeof(vmk_uint32);
   header->data_type = data_type;
   header->data_source = data_source;
}

static vmk_uint8
wait_for_reg_value_equals(struct qfle3_adapter *adapter,
			  vmk_uint32 offset,
			  vmk_uint32 mask,
			  vmk_uint32 expected_value,
			  vmk_uint32 timeout_us)
{
   vmk_uint32 reg_value = 0;
   vmk_uint32 wait_cnt = 0;
   vmk_uint32 wait_cnt_limit = timeout_us/DEFAULT_WAIT_INTERVAL_MICSEC;

   reg_value = RD_IND(adapter, offset);
   while (((reg_value & mask) != expected_value) &&
	  (wait_cnt++ != wait_cnt_limit)) {
      vmk_DelayUsecs(DEFAULT_WAIT_INTERVAL_MICSEC);
      reg_value = RD_IND(adapter, offset);
   }
   return ((reg_value & mask) == expected_value);
}

static vmk_uint32 *
read_vfc_block(struct qfle3_adapter *adapter,
	       const struct vfc_general *vfc_info,
	       const struct vfc_read_task *vfc_ops,
	       struct extension_hdr *header,
	       vmk_uint32 *dst,
	       struct chip_core_dmp *dmp,
	       vmk_uint8 *rc)
{
   vmk_uint32 cur_op_idx, i;
   const struct vfc_read_write_vector *current_entry = NULL;
   vmk_uint32 *dst_start = dst;

   if (!vfc_info->valid) {
      *rc = VMK_TRUE;
      return dst;
   }
   if (!wait_for_reg_value_equals(adapter, vfc_info->vfc_status,
				  0xFFFFFFFF, 0, 1000)) {
      header->error = VMK_TRUE;
      *rc = VMK_TRUE;
      goto exit;
   }
   if (dmp->fw_hdr.dmp_size + sizeof(*header) >
       DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0) {
      SET_FLAGS(dmp->fw_hdr.flags, FWDMP_FLAGS_SPACE_NEEDED);
      *rc = VMK_TRUE;
      return dst;
   }
   dst += sizeof(*header)/sizeof(vmk_uint32);
   dmp->fw_hdr.dmp_size += sizeof(*header);

   for (cur_op_idx = 0; cur_op_idx < vfc_ops->array_size; ++cur_op_idx) {
      current_entry = &vfc_ops->read_write_vectors[cur_op_idx];
      for (i = 0; i < current_entry->write_value_num_valid; ++i) {
	 WR_IND(adapter,
		vfc_info->vfc_data_write,
		current_entry->write_value[i]);
      }
      WR_IND(adapter,
	     vfc_info->vfc_address, current_entry->address_value);
      for (i = 0; i < current_entry->read_size; ++i) {
	 if (!wait_for_reg_value_equals(adapter, vfc_info->vfc_status,
					RI_VFC_IS_READY, RI_VFC_IS_READY, 1000)) {
	    vmk_uint32 reg_value;

	    reg_value =
	       RD_IND(adapter, vfc_info->vfc_status);
	    header->error = VMK_TRUE;
      	    *rc = VMK_TRUE;
	    goto exit;
	 }
	 if ((dmp->fw_hdr.dmp_size + 4) >
	     DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0) {
	    SET_FLAGS(dmp->fw_hdr.flags,
		      FWDMP_FLAGS_SPACE_NEEDED);
      	    *rc = VMK_TRUE;
	    break;
	 }
	 *dst++ = RD_IND(adapter, vfc_info->vfc_data_read);
	 header->data_size++;
	 dmp->fw_hdr.dmp_size += 4;
      }
   }
  exit:
   vmk_Memcpy((vmk_uint8 *)dst_start, header, sizeof(*header));
   return dst;
}

static vmk_uint32 *
read_vfc(struct qfle3_adapter *adapter, vmk_uint32 *dst, struct chip_core_dmp *dmp, vmk_uint8 *status)
{
   vmk_uint8 rc = VMK_FALSE;
   struct extension_hdr xvfc_header = {0};
   struct extension_hdr tvfc_header = {0};

   const struct vfc_general *xvfc_info = NULL;
   const struct vfc_general *tvfc_info = NULL;
   const struct vfc_read_task *xvfc_ops = NULL;
   const struct vfc_read_task *tvfc_ops = NULL;

   get_vfc_info(adapter, &xvfc_info, &tvfc_info);
   get_vfc_ops(adapter, &xvfc_ops, &tvfc_ops);
   if ((xvfc_info == NULL) || (tvfc_info == NULL)) {
      *status = VMK_TRUE;
      return dst;
   }
   init_extension_header(RI_TYPE_VFC, RI_SRC_XSTORM, &xvfc_header);
   init_extension_header(RI_TYPE_VFC, RI_SRC_TSTORM, &tvfc_header);
   dst = read_vfc_block(adapter,
			xvfc_info, xvfc_ops, &xvfc_header, dst, dmp, &rc);
   if (rc) {
      *status = rc;
      return dst;
   }
   dst = read_vfc_block(adapter,
			tvfc_info, tvfc_ops, &tvfc_header, dst, dmp, &rc);
   *status = rc;
   return dst;
}

static const struct igu_data *get_igu_info(struct qfle3_adapter *adapter)
{
   if (CHIP_IS_E1(adapter))
      return &igu_address_e1;
   else if (CHIP_IS_E1H(adapter))
      return &igu_address_e1h;
   else if (CHIP_IS_E2(adapter))
      return &igu_address_e2;
   else if (CHIP_IS_E3(adapter))
      return &igu_address_e3;
   else
      return NULL;
}

static vmk_uint32 *
read_igu(struct qfle3_adapter *adapter, vmk_uint32 *dst, struct chip_core_dmp *dmp, vmk_uint8 *status)
{
   struct extension_hdr igu_header = {0};
   const struct igu_data *igu_info = get_igu_info(adapter);
   vmk_uint32 iter_cnt = 0;
   vmk_uint32 more_data = 0;
   vmk_uint32 *dst_start = dst;

   init_extension_header(RI_TYPE_IGU, RI_OTHER_BLOCK, &igu_header);
   if (!igu_info) {
      *status = VMK_FALSE;
      return dst;
   }
   if (!igu_info->valid) {
      *status = VMK_TRUE;
      return dst;
   }
   more_data = RD_IND(adapter, igu_info->is_data_valid);
   if (!more_data) {
      *status = VMK_TRUE;
      return dst;
   }
   if (dmp->fw_hdr.dmp_size + sizeof(igu_header) >
       DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0) {
      SET_FLAGS(dmp->fw_hdr.flags, FWDMP_FLAGS_SPACE_NEEDED);
      *status = VMK_FALSE;
      return dst;
   }
   dst += sizeof(igu_header)/sizeof(vmk_uint32);
   dmp->fw_hdr.dmp_size += sizeof(igu_header);
   igu_header.additional_data = RD_IND(adapter, igu_info->is_last_commands);

   do {
      if ((dmp->fw_hdr.dmp_size + 4) >
	  DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0) {
	 SET_FLAGS(dmp->fw_hdr.flags,
		   FWDMP_FLAGS_SPACE_NEEDED);
	 igu_header.error = VMK_TRUE;
	 *status = VMK_FALSE;
	 break;
      }
      *dst++ = RD_IND(adapter, igu_info->data[0]);
      ++igu_header.data_size;
      dmp->fw_hdr.dmp_size += 4;
      if ((dmp->fw_hdr.dmp_size + 4) >
	  DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0) {
	 SET_FLAGS(dmp->fw_hdr.flags,
		   FWDMP_FLAGS_SPACE_NEEDED);
	 igu_header.error = VMK_TRUE;
	 *status = VMK_FALSE;
	 break;
      }
      *dst++ = RD_IND(adapter, igu_info->data[1]);
      ++igu_header.data_size;
      dmp->fw_hdr.dmp_size += 4;
      more_data = RD_IND(adapter, igu_info->is_data_valid);
      ++iter_cnt;
   } while (more_data && (iter_cnt < igu_info->max_size));

   vmk_Memcpy((vmk_uint8 *)dst_start, &igu_header, sizeof(igu_header));
   return dst;
}

/* Stall / Resume STORMs */
static void
stall_storms(struct qfle3_adapter *adapter, vmk_Bool stall)
{
	int val = stall ? 1 : 0;

	WR_IND(adapter, DRV_DUMP_XSTORM_WAITP_ADDRESS, val);
	WR_IND(adapter, DRV_DUMP_TSTORM_WAITP_ADDRESS, val);
	WR_IND(adapter, DRV_DUMP_USTORM_WAITP_ADDRESS, val);
	WR_IND(adapter, DRV_DUMP_CSTORM_WAITP_ADDRESS, val);
}

static vmk_uint32 *
read_additional_blocks(struct qfle3_adapter *adapter, vmk_uint32 *dst, struct chip_core_dmp *dmp)
{
   vmk_uint8 dmp_status = VMK_FALSE;

   /* Stall STORMS before reading VFC. */
   stall_storms(adapter, VMK_TRUE);
   
   dst = read_vfc(adapter, dst, dmp, &dmp_status);
   
   /* Resume STORMS. */
   stall_storms(adapter, VMK_FALSE);
   
   if (!dmp_status)
      dst = read_igu(adapter, dst, dmp, &dmp_status);
   return dst;
}

static int
init_dump_header(struct qfle3_adapter *adapter,
		 struct dump_hdr *dmp_hdr,
		 struct dmp_config *dmpcfg,
		 struct chip_core_dmp *dmp)
{
   dmp_hdr->hdr_size = (sizeof(struct dump_hdr)/4) - 1;
   dmp_hdr->hd_param_all = hd_param_all;
   dmp_hdr->idle_chk = 1;
   dmp_hdr->x_storm_wait_p_status =
      RD_IND(adapter, DRV_DUMP_XSTORM_WAITP_ADDRESS);
   dmp_hdr->t_storm_wait_p_status =
      RD_IND(adapter, DRV_DUMP_TSTORM_WAITP_ADDRESS);
   dmp_hdr->u_storm_wait_p_status =
      RD_IND(adapter, DRV_DUMP_USTORM_WAITP_ADDRESS);
   dmp_hdr->c_storm_wait_p_status =
      RD_IND(adapter, DRV_DUMP_CSTORM_WAITP_ADDRESS);

   if (CHIP_IS_E2(adapter)) {
	   if (dmp->fw_hdr.flags & FWDMP_FLAGS_LIVE_DUMP)
		   dmp_hdr->info = RI_E2_ONLINE;
	   else
		   dmp_hdr->info = RI_E2_OFFLINE;
	   dmpcfg->wregs_count = WREGS_COUNT_E2;
	   dmpcfg->pwreg_addrs = wreg_addrs_e2;
	   dmpcfg->regs_timer_count = TIMER_REGS_COUNT_E2;
	   dmpcfg->regs_timer_status_addrs = timer_status_regs_e2;
	   dmpcfg->regs_timer_scan_addrs = timer_scan_regs_e2;
	   dmpcfg->page_mode_values_count =
		   PAGE_MODE_VALUES_E2;
	   dmpcfg->page_vals = page_vals_e2;
	   dmpcfg->page_write_regs_count = PAGE_WRITE_REGS_E2;
	   dmpcfg->page_write_regs = page_write_regs_e2;
	   dmpcfg->page_read_regs_count = PAGE_READ_REGS_E2;
	   dmpcfg->page_read_regs = page_read_regs_e2;

   } else if (CHIP_IS_E3A0(adapter)) {

      if (dmp->fw_hdr.flags & FWDMP_FLAGS_LIVE_DUMP)
		  dmp_hdr->info = RI_E3_ONLINE;
      else
	 dmp_hdr->info = RI_E3_OFFLINE;
      dmpcfg->wregs_count = WREGS_COUNT_E3;
      dmpcfg->pwreg_addrs = wreg_addrs_e3;
      dmpcfg->regs_timer_count = TIMER_REGS_COUNT_E3;
      dmpcfg->regs_timer_status_addrs = timer_status_regs_e3;
      dmpcfg->regs_timer_scan_addrs = timer_scan_regs_e3;
      dmpcfg->page_mode_values_count =
	 PAGE_MODE_VALUES_E3;
      dmpcfg->page_vals = page_vals_e3;
      dmpcfg->page_write_regs_count = PAGE_WRITE_REGS_E3;
      dmpcfg->page_write_regs = page_write_regs_e3;
      dmpcfg->page_read_regs_count = PAGE_READ_REGS_E3;
      dmpcfg->page_read_regs = page_read_regs_e3;
   } else if (CHIP_IS_E3B0(adapter)) {
      if (dmp->fw_hdr.flags & FWDMP_FLAGS_LIVE_DUMP)
	 dmp_hdr->info = RI_E3B0_ONLINE;
      else
	 dmp_hdr->info = RI_E3B0_OFFLINE;
      dmpcfg->wregs_count = WREGS_COUNT_E3B0;
      dmpcfg->pwreg_addrs = wreg_addrs_e3b0;
      dmpcfg->regs_timer_count = TIMER_REGS_COUNT_E3B0;
      dmpcfg->regs_timer_status_addrs = timer_status_regs_e3b0;
      dmpcfg->regs_timer_scan_addrs = timer_scan_regs_e3b0;
      dmpcfg->page_mode_values_count =
	 PAGE_MODE_VALUES_E3;
      dmpcfg->page_vals = page_vals_e3;
      dmpcfg->page_write_regs_count = PAGE_WRITE_REGS_E3;
      dmpcfg->page_write_regs = page_write_regs_e3;
      dmpcfg->page_read_regs_count = PAGE_READ_REGS_E3;
      dmpcfg->page_read_regs = page_read_regs_e3;
   } else {
      QFLE3_ERR("firmware dump chip not supported\n");
      return 1;
   }

   switch (QFLE3_PATH(adapter)) {
   case 0:
      dmp_hdr->info |= RI_PATH0_DUMP;
      break;
   case 1:
      dmp_hdr->info |= RI_PATH1_DUMP;
      break;
   default:
      QFLE3_ERR("unknown path ID (%x)\n", QFLE3_PATH(adapter));
   }
   dmpcfg->mode = dmp_hdr->info & ~(RI_PATH1_DUMP |  RI_PATH0_DUMP);
   dmp_hdr->reserved = 0;
   return 0;
}

void disable_pause(struct qfle3_adapter *adapter)
{
   if (CHIP_IS_E1x(adapter) || CHIP_IS_E2(adapter)) {
      WR_IND(adapter, NIG_REG_BMAC0_PAUSE_OUT_EN, 0);
      WR_IND(adapter, NIG_REG_BMAC1_PAUSE_OUT_EN, 0);
      WR_IND(adapter, NIG_REG_EMAC0_PAUSE_OUT_EN, 0);
      WR_IND(adapter, NIG_REG_EMAC1_PAUSE_OUT_EN, 0);
   } else {
      WR_IND(adapter, NIG_REG_P0_MAC_PAUSE_OUT_EN, 0);
      WR_IND(adapter, NIG_REG_P1_MAC_PAUSE_OUT_EN, 0);
   }
}

void
qfle3_get_grcdump(struct qfle3_adapter *adapter, vmk_uint32 *dst, vmk_Bool live_dump)
{
   vmk_uint32 *tmr_status;
   struct dump_hdr dmp_hdr = {0};
   struct chip_core_dmp *dmp;

   /* build the fw dump header */
   dmp = (struct chip_core_dmp *)dst;
   vmk_Strncpy(dmp->fw_hdr.name,
	       QFLE3_NAME_TO_STRING(adapter->pdev_name), sizeof(dmp->fw_hdr.name));
   dmp->fw_hdr.adapter = (void *)adapter;
   dmp->fw_hdr.chip_id = adapter->hw_info.chip_id;
   dmp->fw_hdr.len = sizeof(struct fw_dmp_hdr);
   dmp->fw_hdr.ver = QFLE3_ESX_FW_DMP_VER;
   dmp->fw_hdr.dmp_size = dmp->fw_hdr.len;
   dmp->fw_hdr.flags = 0;

   if (live_dump)
      SET_FLAGS(dmp->fw_hdr.flags, FWDMP_FLAGS_LIVE_DUMP);

   vmk_Memset(&dmpcfg, 0, sizeof(struct dmp_config));
   vmk_Memset(&dmp_hdr, 0, sizeof(struct dump_hdr));
   dst = dmp->fw_dmp_buf;
   ecore_disable_blocks_parity(adapter);

   /* build the GRC dump header */
   if (init_dump_header(adapter, &dmp_hdr, &dmpcfg, dmp))
      goto end_dump;

   vmk_Memcpy(dst, &dmp_hdr, sizeof(struct dump_hdr));
   dst += dmp_hdr.hdr_size + 1;
   dmp->fw_hdr.dmp_size += (dmp_hdr.hdr_size + 1) * 4;

   /* stop the timers before idle check. */
   tmr_status = dst;
   dst = dmp_stop_timer(adapter, dst, dmp);
   if (dmp->fw_hdr.flags & FWDMP_FLAGS_SPACE_NEEDED)
      goto end_dump;

   /* dump 1st idle check */
   dst = read_idle_chk(adapter, dst, dmp);
   if (dmp->fw_hdr.flags & FWDMP_FLAGS_SPACE_NEEDED)
      goto end_dump;

   /* dump 2nd idle check */
   dst = read_idle_chk(adapter, dst, dmp);

   /* Enable the timers after idle check. */
   dmp_rollback_timer(adapter, dmp, tmr_status);
   if (dmp->fw_hdr.flags & FWDMP_FLAGS_SPACE_NEEDED)
      goto end_dump;

   /* dump mcp traces */
   dst = read_mcp_traces(adapter, dst, dmp);
   if (dmp->fw_hdr.flags & FWDMP_FLAGS_SPACE_NEEDED)
      goto end_dump;

   /* dump regular address registers */
   dst = read_regular_regs(adapter, dst, dmp);
   if (dmp->fw_hdr.flags & FWDMP_FLAGS_SPACE_NEEDED)
      goto end_dump;

   /* dump wide bus registers */
   dst = read_wregs(adapter, dst, dmp);
   if (dmp->fw_hdr.flags & FWDMP_FLAGS_SPACE_NEEDED)
      goto end_dump;

   /* dump read page mode registers */
   dst = read_page_mode(adapter, dst, dmp);
   if (dmp->fw_hdr.flags & FWDMP_FLAGS_SPACE_NEEDED)
      goto end_dump;

   /* dump additional blocks */
   if (!live_dump)
      dst = read_additional_blocks(adapter, dst, dmp);

   if (((dmp->fw_hdr.dmp_size + 4) <=
	DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0)) {
      *dst++ = QFLE3_FWDMP_MARKER_END;
      dmp->fw_hdr.dmp_size += 4;
   }
	QFLE3_INFO("GRC-dump size 0x%x live:%d\n",
		    dmp->fw_hdr.dmp_size, live_dump);
  end_dump:
   
   return;
}

VMK_ReturnStatus
qfle3_fw_dump_handler(void *cookie, vmk_Bool live_dump)
{
   qfle3_adapter *adapter = NULL;
   vmk_ListLinks *link = NULL;
   struct chip_core_dmp *dmp;
   VMK_ReturnStatus status = VMK_OK;
   vmk_int32 times = 0;
   dmp = qfle3_heap_alloc(DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0);
   
   if (dmp == NULL){
      QFLE3_ERR("Failed to allocate heap for qfle3_fw_dump_handler.");
      return VMK_FAILURE;
   }
   VMK_LIST_FORALL(&qfle3_mod_info.adapterList, link){
      if (times == 16)
         return status;
      adapter = VMK_LIST_ENTRY(link, struct qfle3_adapter, adapterLink);
      if (QFLE3_PANIC_FWDMP_ENABLED(adapter)) {
//         if (adapter->fwdmp_flags & GRC_DMP_STOP_STATS) {
//            qfle3_stats_handle(adapter, STATS_EVENT_STOP);
//            adapter->fwdmp_flags &= ~GRC_DMP_STOP_STATS;
//         }
//         dmp = (struct chip_core_dmp *)adapter->fwdmp_buf;
         vmk_Memset(dmp, 0, DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0);
         qfle3_get_grcdump(adapter, (vmk_uint32 *)dmp, live_dump);

         QFLE3_INFO("Writing %u bytes to dump device. Instance %i\n", dmp->fw_hdr.dmp_size, ++times);
         /* Write the FW-dump to the system dump device. */
         status = vmk_DumpRange(qfle3_mod_info.dumpFile,
                                (vmk_VA)dmp,
                                dmp->fw_hdr.dmp_size);

         if (status != VMK_OK) {
            QFLE3_ERR("Failed to write %u bytes to dump device. [%s]\n",
                      dmp->fw_hdr.dmp_size,
                      vmk_StatusToString(status));
         }
      }
   }
   
   qfle3_heap_free(dmp);
   dmp = NULL;
   return status;
}
VMK_ReturnStatus
qfle3_recovfw_dump_handler(void *cookie, vmk_Bool live_dump)
{
   qfle3_adapter *adapter = NULL;
   vmk_ListLinks *link = NULL;
   struct chip_core_dmp *dmp;
   VMK_ReturnStatus status = VMK_OK;
   vmk_int32 times = 0;

   VMK_LIST_FORALL(&qfle3_mod_info.adapterList, link){
      if (times == 16)
         return status;
      adapter = VMK_LIST_ENTRY(link, struct qfle3_adapter, adapterLink);
      if (adapter->fwdmp_buf == NULL)
         continue;
      dmp = (struct chip_core_dmp *)adapter->fwdmp_buf;

      QFLE3_INFO("Writing %u bytes to dump device. Instance %i\n", dmp->fw_hdr.dmp_size, ++times);
      /* Write the FW-dump to the system dump device. */
      status = vmk_DumpRange(qfle3_mod_info.recoveryDumpFile,
                             (vmk_VA)adapter->fwdmp_buf,
                             dmp->fw_hdr.dmp_size);

      if (status != VMK_OK) {
         QFLE3_ERR("Failed to write %u bytes to dump device. [%s]\n",
                   dmp->fw_hdr.dmp_size,
                   vmk_StatusToString(status));
      }
      qfle3_heap_free(adapter->fwdmp_buf);
      adapter->fwdmp_buf = NULL;
   }
   return status;
}


void qfle3_trigger_grcdump(qfle3_adapter *adapter, vmk_int32 context)
{
	int trigger = 0, sched_task = 0;
	QFLE3_ERR("Enterd  fwdmp_flags 0x%x ES 0x%x\n", adapter->fwdmp_flags, adapter->error_status);

   /* Tx timeout trigger handler */
   if (context == QFLE3_CXT_TXTO_TRIGGER) {
      adapter->error_status |= QFLE3_ERR_NETDEV_TO;
      
         /* Trigger recovery on other PFs of same engine.
          * - Speedup recovery trigger on other PFs.
          * - Explicit trigger if the error is local.
          */
         if (!(adapter->error_status & QFLE3_ERR_PEER_RESET))
            qfle3_reset_engine_pfs(adapter);
   }
   if ((adapter->fwdmp_flags & GRC_DMP_COLLECT_VALID) ||
      	    (adapter->error_status & QFLE3_ERR_PEER_RESET)) {
		/* Discard dump trigger. */
		QFLE3_ERR("Discard CT:0x%x GDF:0x%x ES:0x%x\n",
			  context, adapter->fwdmp_flags, adapter->error_status);
		return;
	}
	switch (context) {
	case QFLE3_CXT_PARITY_TRIGGER:
		if (adapter->fwdmp_flags & GRC_DMP_PARITY_ENABLED) {
			trigger = 1;
         adapter->fwdmp_flags |= GRC_DMP_STOP_STATS;
		}
		break;
   case QFLE3_CXT_TXTO_TRIGGER:
      if (adapter->fwdmp_flags & GRC_DMP_TXTO_ENABLED) {
         trigger = 1;
         adapter->fwdmp_flags |= (GRC_DMP_STOP_STATS | GRC_DMP_CLOSE_DEVICE);
         adapter->force_link_down = VMK_TRUE;
      } else if ( adapter->error_status) {
			/* Trigger dump in ALL cases */
			trigger = 1;
			if (!(adapter->error_status & QFLE3_ERR_STATS_TO))
				adapter->fwdmp_flags |= GRC_DMP_STOP_STATS;
      }
      break;
   case QFLE3_CXT_STSTO_TRIGGER:
      if (adapter->fwdmp_flags & GRC_DMP_STSTO_ENABLED) {
         trigger = 1;
         sched_task = 1;
         adapter->fwdmp_flags |= GRC_DMP_CLOSE_DEVICE;
         adapter->force_link_down = VMK_TRUE;
      } else {
         /* Regular dump, if not GRC dump. */
         qfle3_panic(adapter);
         // we should just trigger assert at this point
      }
      break;
	default:
		break;
	}

	if (trigger) {
		/* Mark for actual GRC dump collection. */
		adapter->fwdmp_flags |= GRC_DMP_COLLECT_TRIGGER;
		QFLE3_ERR("Triggered GRC dump cxt:%x st:%x\n",
			  context, sched_task);

		if (sched_task) {
			/* Explicit task schedule. */
         qfle3_schedule_helper (adapter, adapter->recovery_helper, qfle3_recovery_func, 0);
		}
	}
   QFLE3_ERR("GRC flags:0x%x trigger:%d\n", adapter->fwdmp_flags, trigger);

	return;
}
void qfle3_stop_engine_stats(qfle3_adapter *adapter)
{
	struct vmk_ListLinks *item;
   qfle3_adapter *t_bp;

	/* Stop the stats on given function, and also on
	 * all the engines under same engine. This will
	 * avoid assertion on other functions due to stats
	 * not getting updated during parity recovery.
	 */
	qfle3_stats_handle(adapter, STATS_EVENT_STOP);
   
   VMK_LIST_FORALL (&qfle3_mod_info.adapterList, item) {
      t_bp = VMK_LIST_ENTRY(item, struct qfle3_adapter, adapterLink);
		if (t_bp == NULL || t_bp == adapter)
			continue;

		/* If this entry is currently not in ready state,
		 * then do not attempt issueing stats stop request.
		 * Otherwise it would access freed resources.
		 */
      if (!vmk_BitVectorTest(t_bp->state, QFLE3_STATE_BIT_ATTACHED) ||
          (t_bp->recovery_state != QFLE3_RECOVERY_DONE)) {
         continue;
      }

		if ((t_bp->pdev_addr.seg == adapter->pdev_addr.seg) &&
		    (t_bp->pdev_addr.bus == adapter->pdev_addr.bus) &&
		    (t_bp->pdev_addr.dev == adapter->pdev_addr.dev) &&
		    (QFLE3_PATH(adapter) == QFLE3_PATH(t_bp))) {
         QFLE3_ERR("Stopping stats collection on engine function %d:%d:%02d.%02d\n",t_bp->pdev_addr.seg, t_bp->pdev_addr.bus,
            t_bp->pdev_addr.dev, t_bp->pdev_addr.fn);
			/* Found a function under same engine.*/
			qfle3_stats_handle(t_bp, STATS_EVENT_STOP);
		}
	}
}

void
qfle3_recovery_collect_grcdump(qfle3_adapter *adapter)
{
   struct chip_core_dmp *dmp;
   VMK_ReturnStatus status;

   if (!QFLE3_PANIC_FWDMP_ENABLED(adapter)) {
      return;
   }
   if (adapter->fwdmp_buf) {
      
      adapter->fwdmp_flags &= ~GRC_DMP_COLLECT_TRIGGER;
      QFLE3_ERR("GRC dump already exists.");
      return;
   }
   
   if (!(adapter->fwdmp_flags & GRC_DMP_COLLECT_TRIGGER)) {
      return;
   }
   if (qfle3_mod_info.recoveryDumpFile == NULL) {
   
      status = vmk_DumpAddFileCallback(vmk_ModuleCurrentID,
                                    qfle3_mod_info.heapID,
                                    "qfle3_recovery_fwdmp",
                                    qfle3_recovfw_dump_handler,
                                    NULL,
                                    "qfle3_recovery_fwdmp",
                                    &qfle3_mod_info.recoveryDumpFile);
      
      if (status != VMK_OK) {
         QFLE3_ERROR("Failed to add recoveryDump file callback.");
      }
   }
   if (adapter->fwdmp_flags & GRC_DMP_STOP_STATS) {
      qfle3_stop_engine_stats(adapter);
      adapter->fwdmp_flags &= ~GRC_DMP_STOP_STATS;
   }
   adapter->fwdmp_buf = qfle3_heap_alloc(DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0);
   if (adapter->fwdmp_buf == NULL){
      QFLE3_ERR("Failed to allocate heap for qfle3_fw_dump_handler.");
      return;
   }
   vmk_Memset(adapter->fwdmp_buf,0,DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0);
   dmp = (struct chip_core_dmp *)adapter->fwdmp_buf;
   
   QFLE3_INFO("Starting GRC dump.\n");
   qfle3_get_grcdump(adapter, adapter->fwdmp_buf, VMK_FALSE);
   dmp->fw_hdr.flags |= FWDMP_FLAGS_RECOVERY_DUMP;

   ecore_clear_blocks_parity(adapter);
   ecore_enable_blocks_parity(adapter);
   disable_pause(adapter);

   adapter->fwdmp_flags |= GRC_DMP_COLLECT_VALID;
   QFLE3_INFO("GRC Dump saved at %p.", adapter->fwdmp_buf);
   
   if (adapter->fwdmp_flags & GRC_DMP_CLOSE_DEVICE) {
      // Set Link down
      qfle3_tx_disable(adapter);
      /* Reset the rtnl_state and mark
       * Link down.
       */
      #if 0
      smp_mb__before_atomic();
      bp->sp_rtnl_state = 0;
      smp_mb__after_atomic();
      
      netif_tx_disable(bp->dev);
      netif_carrier_off(bp->dev);
      bp->dev->full_duplex = 0;
      bp->link_vars.link_up = 0;
      
      if (bp->flags & CNA_ENABLED) {
         netif_tx_disable(bp->cnadev);
         netif_carrier_off(bp->cnadev);
      }
      
      netdev_err(bp->dev, "GRC-dump Link Down !\n");
    #endif
   }

   adapter->fwdmp_flags &= ~GRC_DMP_COLLECT_TRIGGER;
   return;
}


void
qfle3_recovery_func(vmk_AddrCookie data)
{
   qfle3_adapter *adapter = data.ptr;
   
   qfle3_recovery_collect_grcdump (adapter);

   if (VMK_UNLIKELY(adapter->recovery_state != QFLE3_RECOVERY_DONE)) {
      
      QFLE3_INFO("NOT TX Timeout\n");
      // this means parity error
      if (!auto_recovery) {
         QFLE3_ERR("recovery flow called but STOP_ON_ERROR defined so reset not done to allow debug dump,\n"
              "you will need to reboot when done\n");
         goto do_not_reset;
      }
      
		/*
		 * Clear all pending SP commands as we are going to reset the
		 * function anyway.
		 */

      vmk_BitVectorClear(adapter->sp_rtnl_state, QFLE3_SP_RTNL_TX_TIMEOUT);
      vmk_CPUMemFenceReadWrite();
      
      qfle3_tx_error (adapter);
      qfle3_parity_recover(adapter);
      return;
   }

   if (vmk_BitVectorAtomicTestAndClear(adapter->sp_rtnl_state, QFLE3_SP_RTNL_TX_TIMEOUT)) {
      QFLE3_INFO("TX Timeout\n");
      
      if (!auto_recovery) {
         QFLE3_ERR("recovery flow called but STOP_ON_ERROR defined so reset not done to allow debug dump,\n"
              "you will need to reboot when done\n");
         goto do_not_reset;
      }
//      if(adapter->error_status & QFLE3_ERR_STATS_TO){
//         QFLE3_ERR("Do not attempt recovery for Stats Timeout \n");
//         goto do_not_reset;
//      }
      vmk_BitVectorClear(adapter->sp_rtnl_state, QFLE3_SP_RTNL_TX_TIMEOUT);

      vmk_CPUMemFenceReadWrite();
      qfle3_tx_error(adapter);
      return;
   }
   
do_not_reset:
   qfle3_link_down(adapter);
   adapter->recovery_state = QFLE3_RECOVERY_FAILED;
   if (qfle3_sm_cmd (adapter, QFLE3_SMCMD_TOSTANDBY, 0) != QFLE3_SMCMD_STATUS_COMPLETED)
   {
      QFLE3_ERR("Unable to execute QFLE3_SMCMD_TOSTANDBY\n");
   }
   adapter->error_status = QFLE3_ERR_NONE;
//   STOP_RECOVERY(adapter);
   QFLE3_ERR("ESX_REC_INFO: Leaving device in closed state.\n");
   return;
}











static const char *qfle3_sm_string[QFLE3_SM_MAXSTATE]=
{
   "SM_NotReady",
   "SM_Standby",
   "SM_HWREADY",
   "SM_DEVREADY",
   "SM_PORTREADY",
   "SM_FUNCREADY",
   "SM_ACTIVATED",
   "SM_IOSTARTED",
};

static const char* qfle3_sm_emptystring = "EMPTY_STRING";

const char* qfle3_sm_get_string(vmk_uint16 sm_state)
{
   if (sm_state < QFLE3_SM_MAXSTATE)
      return qfle3_sm_string[sm_state];
   return qfle3_sm_emptystring;
}

static const char *qfle3_sm_cmd_string[QFLE3_SMCMD_MAXCMD]=
{
   "CMD_Unknown",
   "CMD_Bringup",
   "CMD_Funcreset",
   "CMD_Start",
   "CMD_Pauseio",
   "CMD_Resumeio",
   "CMD_Stop",
   "CMD_Tostandby",
   "CMD_Uplinkreset",
   "CMD_Create_Q",
   "CMD_Start_Q",
   "CMD_Pause_Q",
   "CMD_Remove_Q",
   "CMD_Stop_Q",
   "CMD_Resume_Q",
};

static const char* qfle3_sm_get_cmd_string(vmk_uint16 sm_cmd)
{
   if (sm_cmd < QFLE3_SMCMD_MAXCMD)
      return qfle3_sm_cmd_string[sm_cmd];
   return qfle3_sm_emptystring;
}

static const char *qfle3_sm_cmdstatus_string[QFLE3_SMCMD_STATUS_WRONGSTATE+1]=
{
   "Status_Completed",
   "Status_Inprogress",
   "Status_Error",
   "Status_Notready",
   "Status_WrongState",
};

static const char* qfle3_sm_get_cmdstatus_string(vmk_uint16 sm_cmdstatus)
{
   if (sm_cmdstatus <= QFLE3_SMCMD_STATUS_WRONGSTATE)
      return qfle3_sm_cmdstatus_string[sm_cmdstatus];
   return qfle3_sm_emptystring;
}

void qfle3_sm_cmd_complete(struct qfle3_adapter *adapter, vmk_uint16 sm_cmd)
{
   vmk_uint16 last_cmd;

   vmk_SpinlockLock(adapter->sm_lock);
   last_cmd = adapter->last_cmd;
   if (last_cmd == sm_cmd)
      adapter->cmd_status = QFLE3_SMCMD_STATUS_COMPLETED;
   vmk_SpinlockUnlock(adapter->sm_lock);

   if (last_cmd != sm_cmd){
      QFLE3_ERROR("Completing the WRONG cmd %s, last cmd %s\n",qfle3_sm_get_cmd_string(sm_cmd), qfle3_sm_get_cmd_string(adapter->last_cmd));
      return;
   }else {
      QFLE3_DBG(QFLE3_DBG_SM, "Completing cmd %s\n",qfle3_sm_get_cmd_string(sm_cmd));
      return;
   }
}

void qfle3_sm_cmd_error(struct qfle3_adapter *adapter, vmk_uint16 sm_cmd)
{
   vmk_uint16 last_cmd;

   vmk_SpinlockLock(adapter->sm_lock);
   last_cmd = adapter->last_cmd;
   if (last_cmd == sm_cmd)
      adapter->cmd_status = QFLE3_SMCMD_STATUS_ERROR;
   vmk_SpinlockUnlock(adapter->sm_lock);

   if (last_cmd != sm_cmd){
      QFLE3_INFO("Erroring the WRONG cmd %s, last cmd %s\n",qfle3_sm_get_cmd_string(sm_cmd), qfle3_sm_get_cmd_string(adapter->last_cmd));
      return;
   }else {
      QFLE3_INFO("Erroring cmd %s\n",qfle3_sm_get_cmd_string(sm_cmd));
      return;
   }
}


/* Call this function to get the state of the SM.  Returns:
 * QFLE3_SM_STANDBY
 * QFLE3_SM_HWREADY
 * QFLE3_SM_DEVREADY
 * QFLE3_SM_PORTREADY
 * QFLE3_SM_FUNCREADY
 *
 * If the adapter is in the progress of completing a command, then return
 * QFLE3_SM_NOTREADY
 */
vmk_uint16 qfle3_sm_getstate(struct qfle3_adapter *adapter)
{
   vmk_uint16	cmd_status;
   vmk_uint16	sm_state;

   vmk_SpinlockLock(adapter->sm_lock);
   cmd_status = adapter->cmd_status;
   sm_state = adapter->sm_state;
   vmk_SpinlockUnlock(adapter->sm_lock);

   if (cmd_status != QFLE3_SMCMD_STATUS_COMPLETED)
      return QFLE3_SM_NOTREADY;
   return sm_state;
}



void qfle3_sm_change_state(struct qfle3_adapter *adapter, vmk_uint16 new_state)
{

   VMK_ASSERT(adapter != NULL);
   VMK_ASSERT(new_state != QFLE3_SM_NOTREADY);
   VMK_ASSERT(new_state < QFLE3_SM_MAXSTATE);


   QFLE3_INFO("Changing adapter state from %s to %s\n",qfle3_sm_get_string(adapter->sm_state), qfle3_sm_get_string(new_state));
   vmk_SpinlockLock(adapter->sm_lock);
   adapter->sm_state = new_state;
   vmk_SpinlockUnlock(adapter->sm_lock);

}

#if 0
void qfle3_deinit_statemachine(struct qfle3_adapter *adapter)
{
   vmk_Bool loaded;
   vmk_uint32 reset_code;


   reset_code = qfle3_send_unload_req(adapter, DRV_MSG_CODE_UNLOAD_REQ_WOL_DIS);

   /* Report UNLOAD_DONE to MCP */
   qfle3_mfw_command(adapter, DRV_MSG_CODE_UNLOAD_DONE, 0);

   loaded = qfle3_clear_pf_load(adapter);
   if (loaded == VMK_FALSE) {
      if (qfle3_reset_is_done(adapter, QFLE3_PATH(adapter)))
	 qfle3_disable_close_the_gate(adapter);
   }

}

#endif


static VMK_ReturnStatus qfle3_sm_start(struct qfle3_adapter *adapter)
{

   VMK_ReturnStatus status = VMK_FAILURE;
   vmk_UplinkSharedQueueInfo *sqi;
   sqi = &adapter->uplinkQueueInfo;
   vmk_uint32 i;
   vmk_NetPoll netpoll;

   QFLE3_DBG(QFLE3_DBG_SM, "SM_START Entering");

   qfle3_init_uplink_data(adapter);
   qfle3_init_rss_tables(adapter);

   for (i=0; i< ILT_MAX_L2_LINES; i++) {
      if (adapter->context[i].vcxt) {
         vmk_Memset(adapter->context[i].vcxt,0,sizeof(union cdu_context));
      }
   }
   vmk_CPUMemFenceReadWrite();
   status = qfle3_alloc_rxqs_mem(adapter);
   if (status != VMK_OK) {
      QFLE3_ERR("DriverOps.start rx queue allocate failed!");
      goto failed_rxq_mem;

   }
   status = qfle3_alloc_txqs_mem(adapter);
   if (status != VMK_OK) {
      QFLE3_ERR("DriverOps.start tx queue allocate failed!");
      goto failed_txq_mem;
   }

   qfle3_update_coalesce(adapter);

   /*
    * Allocate default RX/TX queues
    */
   status = qfle3_rq_alloc(adapter, &sqi->defaultRxQueueID, &netpoll, 0);
   if (status != VMK_OK) {
      QFLE3_ERR("Failed to create default RX queue (%x)\n", status);
      goto   failed_alloc_rq;
   }
   status = qfle3_tq_alloc(adapter, &sqi->defaultTxQueueID, &netpoll);
   if (status != VMK_OK) {
      QFLE3_ERR("Failed to create default TX queue (%x)\n", status);
      goto failed_alloc_tq;
   }

   status = qfle3_register_fp_interrupts(adapter);
   if (status != VMK_OK) {
      goto fail_intr_init;
   }
   // create the CNIC queues
   if (!vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING) || 
      (adapter->recovery_state != QFLE3_RECOVERY_DONE)) {
      
      // NOT in reset state means this is a device start
      // recovery state not DONE means we are in recovery
#ifdef QFLE3_CNIC
#ifdef QFLE3_CREATE_CNIC_QS
      
      if (CNIC_SUPPORT(adapter)) {
      status = qfle3_alloc_cnicqs_mem(adapter);
      if (status != VMK_OK) {
         QFLE3_ERR("DriverOps.start CNIC SB allocate failed!");
         goto failed_cnicqs_mem;
      }
      status = qfle3_create_cnicqs(adapter);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to create CNIC queues (%x)", status);
         goto failed_cnicqs;
      }
      status = qfle3_start_cnicqs(adapter);
      if (status != VMK_OK) {
         QFLE3_ERR("Failed to start CNIC queues (%x)", status);
         goto failed_cnicq_start;
      }
      }
#endif
#endif
   } else {
   // Hopefully the only condition left is when called by uplinkreset
   // we don't disturb the cnic queues in this case.
   }
   QFLE3_DBG(QFLE3_DBG_SM, "SM_START exiting, success!");
   vmk_BitVectorAtomicTestAndClear(adapter->state, QFLE3_STATE_BIT_QUIESCED);

   return VMK_OK;

#ifdef QFLE3_CNIC
#ifdef QFLE3_CREATE_CNIC_QS
   failed_cnicq_start:
      qfle3_remove_cnicqs(adapter);
   failed_cnicqs:
      qfle3_free_cnicqs_mem(adapter);
   failed_cnicqs_mem:
      qfle3_sm_cmd(adapter, QFLE3_SMCMD_STOP, 0);
#endif
#endif
fail_intr_init:
   qfle3_tq_free(adapter, sqi->defaultTxQueueID);
failed_alloc_tq:
   qfle3_rq_free(adapter, sqi->defaultRxQueueID);
failed_alloc_rq:
   qfle3_free_txqs_mem(adapter);
failed_txq_mem:
   qfle3_free_rxqs_mem(adapter);
failed_rxq_mem:

   QFLE3_DBG(QFLE3_DBG_SM, "SM_START exiting, error!");
   return status;
}

static VMK_ReturnStatus qfle3_sm_stop(struct qfle3_adapter *adapter)
{
        
   int i,fp_num;
   struct qfle3_fastpath *fpq;
   vmk_UplinkSharedQueueData *sqd = NULL;
   vmk_UplinkSharedQueueInfo *sqi;
   VMK_ReturnStatus status;

   sqi = &adapter->uplinkQueueInfo;
   QFLE3_DBG(QFLE3_DBG_SM, "SM_STOP Entering");
   vmk_BitVectorAtomicTestAndSet(adapter->state, QFLE3_STATE_BIT_QUIESCED);
   QFLE3_DBG(QFLE3_DBG_STATS, "Executing qfle3_stats_handle for STATS_EVENT_STOP\n");
   qfle3_stats_handle(adapter, STATS_EVENT_STOP);
   
   if(!vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)) {
      if (adapter->vfdb) {
         for (i=0; i<adapter->vfdb->sriov.nr_virtfn; i++) {
            if (adapter->vfdb->vfs[i].state != VF_FREE){
               QFLE3_ERR("Unable to quiece device, VF[%d] is not free \n",i);
            }
         }
      }
   }
   // just incase queues were created/started before IOstart
   FOREACH_TX_VMK_QUEUE(i,fp_num,fpq)
      sqd = QFLE3_TX_QUEUE_SHARED_DATA(adapter);
      sqd += i;
      if ((sqd->flags & VMK_UPLINK_QUEUE_FLAG_IN_USE) &&
         !(sqd->flags & VMK_UPLINK_QUEUE_FLAG_DEFAULT)) {
	 QFLE3_DBG(QFLE3_DBG_SM, "Invalidating  TX q %d\n", i);
         status = vmk_UplinkQueueInvalidate(adapter->uplink, sqd->qid);
	 if (status == VMK_FAILURE) {
	 	QFLE3_ERR("Invalidating queue not supported\n");
	 }
         if (fpq->fp_state == QFLE3_SM_Q_STARTED)
            qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_STOP_Q, i, QFLE3_SM_PARAM_TXQ);
         if ((fpq->fp_state == QFLE3_SM_Q_CREATED) &&
             !(sqd->flags & VMK_UPLINK_QUEUE_FLAG_DEFAULT))
            qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_REMOVE_Q, i, QFLE3_SM_PARAM_TXQ);
	 if(i);
	    qfle3_tq_free(adapter,sqd->qid);
      }
   }

   FOREACH_RX_VMK_QUEUE(i,fp_num,fpq)
      sqd = QFLE3_RX_QUEUE_SHARED_DATA(adapter);
      sqd += i;
      if ((sqd->flags & VMK_UPLINK_QUEUE_FLAG_IN_USE) &&
         !(sqd->flags & VMK_UPLINK_QUEUE_FLAG_DEFAULT)) {
            QFLE3_DBG(QFLE3_DBG_SM, "Invalidating  RX q %d\n", i);
            status = vmk_UplinkQueueInvalidate(adapter->uplink, sqd->qid);
         if (status == VMK_FAILURE) {
            QFLE3_ERR("Invalidating queue not supported\n");
         }
         if (fpq->fp_state == QFLE3_SM_Q_STARTED)
            qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_STOP_Q, i, QFLE3_SM_PARAM_RXQ);
         if ((fpq->fp_state == QFLE3_SM_Q_CREATED) &&
             !(sqd->flags & VMK_UPLINK_QUEUE_FLAG_DEFAULT))
            qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_REMOVE_Q, i, QFLE3_SM_PARAM_RXQ);
	 if (i)
            qfle3_rq_free(adapter,sqd->qid);
      }
   }
   
   if (!vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING) || 
      (adapter->recovery_state != QFLE3_RECOVERY_DONE)) {
      // NOT in reset state means this is a device quiesce
      // recovery state not DONE means we are in recovery
#ifdef QFLE3_CNIC
#ifdef QFLE3_CREATE_CNIC_QS
      if (CNIC_SUPPORT(adapter)){
         qfle3_stop_cnicqs(adapter);
         qfle3_remove_cnicqs(adapter);
         qfle3_free_cnicqs_mem(adapter);
      }
#endif
#endif
   } else {
   // Hopefully the only condition left is when called by uplinkreset
   // we don't disturb the cnic queues in this case.
   }
   if (adapter->cnicEthDev->nof_ofld_conns){
      QFLE3_ERR("Unable to quiece device, must remove storage connections first %d\n", 
         adapter->cnicEthDev->nof_ofld_conns);
   }

   qfle3_rq_free(adapter, sqi->defaultRxQueueID);
   qfle3_tq_free(adapter, sqi->defaultTxQueueID);
   qfle3_tq_cleanup_all(adapter);
   qfle3_unregister_fp_interrupts(adapter);
   qfle3_free_txqs_mem(adapter);
   qfle3_free_rxqs_mem(adapter);


   QFLE3_DBG(QFLE3_DBG_SM, "SM_STOP Exiting");
   return VMK_OK;
}
static QFLE3_SMCMD_STATUS qfle3_cmd_bringup(struct qfle3_adapter *adapter, vmk_uint32 param)
{

   VMK_ReturnStatus status = VMK_FAILURE;

   QFLE3_DBG(QFLE3_DBG_SM, "qfle3_cmd_bringup %s \n",qfle3_sm_get_string(adapter->sm_state));
   status = qfle3_load(adapter);
   if (status == VMK_OK) {
      qfle3_sm_change_state(adapter, QFLE3_SM_HWREADY);
      return QFLE3_SMCMD_STATUS_COMPLETED;
   } else {
      return QFLE3_SMCMD_STATUS_ERROR;
   }
}

static QFLE3_SMCMD_STATUS qfle3_cmd_start(struct qfle3_adapter *adapter, vmk_uint32 param)
{
   VMK_ReturnStatus status = VMK_FAILURE;

   QFLE3_DBG(QFLE3_DBG_SM, "qfle3_cmd_start %s \n",qfle3_sm_get_string(adapter->sm_state));
//   status = qfle3_load(adapter);
//   if (status != VMK_OK) 
//      goto load_fail;
   
   status = qfle3_sm_start(adapter);
   if (status != VMK_OK) 
      goto start_fail;

   QFLE3_DBG(QFLE3_DBG_SM, "qfle3_sm_start success \n");
   qfle3_sm_change_state(adapter, QFLE3_SM_ACTIVATED);
   return QFLE3_SMCMD_STATUS_COMPLETED;
start_fail:
//   qfle3_unload(adapter);
//load_fail:
   return QFLE3_SMCMD_STATUS_ERROR;
}
static QFLE3_SMCMD_STATUS qfle3_cmd_stop(struct qfle3_adapter *adapter, vmk_uint32 param)
{
   VMK_ReturnStatus status = VMK_FAILURE;

   QFLE3_DBG(QFLE3_DBG_SM, "qfle3_cmd_stop %s \n",qfle3_sm_get_string(adapter->sm_state));
   status = qfle3_sm_stop(adapter);
//   status = qfle3_unload(adapter);
   if (status == VMK_OK) {
      qfle3_sm_change_state(adapter, QFLE3_SM_HWREADY);
      return QFLE3_SMCMD_STATUS_COMPLETED;
   } else {
      return QFLE3_SMCMD_STATUS_ERROR;
   }
}
static QFLE3_SMCMD_STATUS qfle3_cmd_resume(struct qfle3_adapter *adapter, vmk_uint32 param)
{
   VMK_ReturnStatus status = VMK_FAILURE;

   QFLE3_DBG(QFLE3_DBG_SM, "qfle3_cmd_resume %s \n",qfle3_sm_get_string(adapter->sm_state));
   
   if (adapter->sm_state == QFLE3_SM_ACTIVATED){
      status = qfle3_activate_dev(adapter);
   }else if (adapter->sm_state == QFLE3_SM_IOSTARTED) {
      status = VMK_OK;
   }

   if (status == VMK_OK) {
      qfle3_sm_change_state(adapter, QFLE3_SM_IOSTARTED);
      return QFLE3_SMCMD_STATUS_COMPLETED;
   } else {
      return QFLE3_SMCMD_STATUS_ERROR;
   }
}
static QFLE3_SMCMD_STATUS qfle3_cmd_pause(struct qfle3_adapter *adapter, vmk_uint32 param)
{
   
   QFLE3_DBG(QFLE3_DBG_SM, "qfle3_cmd_pause %s \n",qfle3_sm_get_string(adapter->sm_state));
   if (adapter->sm_state == QFLE3_SM_IOSTARTED){
       qfle3_deactive_dev(adapter);

       qfle3_sm_change_state(adapter, QFLE3_SM_ACTIVATED);
   }

   return QFLE3_SMCMD_STATUS_COMPLETED;
}

// call this function at the bottom of reset, before bring it the uplink up again.
void qfle3_apply_uplink_changes(struct qfle3_adapter *adapter)
{

   vmk_UplinkSharedData *sd = &adapter->uplinkSharedData;
   if (adapter->uplinkCachedNewData & QFLE3_UPLINK_MTU_CHANGED) {

      QFLE3_DBG(QFLE3_DBG_UPLINK, "Change MTU from %d to %d\n", adapter->mtu, adapter->uplinkCachedData.mtu);
      adapter->mtu = adapter->uplinkCachedData.mtu;

      QFLE3_SHARED_DATA_WRITE_BEGIN(adapter);
      sd->mtu = adapter->mtu;
      QFLE3_SHARED_DATA_WRITE_END(adapter);
   }

   if (adapter->uplinkCachedNewData & QFLE3_UPLINK_RINGSIZE_CHANGED) {

      adapter->rx_bd_num_pages = adapter->uplinkCachedData.rx_ring_page_num;
      adapter->tx_bd_num_pages = adapter->uplinkCachedData.tx_ring_page_num;
      adapter->rx_ring_size = adapter->rx_bd_num_pages * RX_BD_TOTAL_PER_PAGE;
      adapter->tx_ring_size = adapter->tx_bd_num_pages * TX_BD_TOTAL_PER_PAGE;
      QFLE3_DBG(QFLE3_DBG_UPLINK, "Setting rx_ring_size to %d, and tx_ring_size to %d\n", adapter->rx_ring_size, adapter->tx_ring_size);
   }

   if (adapter->uplinkCachedNewData & QFLE3_UPLINK_VLAN_CONFIG_CHANGED) {
      adapter->hw_vlan_en = adapter->uplinkCachedData.en_vlan;
   }

   adapter->uplinkCachedNewData = 0;

}
static QFLE3_SMCMD_STATUS qfle3_cmd_uplinkreset(struct qfle3_adapter *adapter, vmk_uint32 param)
{
   VMK_ReturnStatus status = VMK_FAILURE;

   QFLE3_DBG(QFLE3_DBG_SM, "uplink reset in state %s \n",qfle3_sm_get_string(adapter->sm_state));
   if (adapter->sm_state == QFLE3_SM_ACTIVATED)
   {
      status = qfle3_sm_stop(adapter);
      qfle3_apply_uplink_changes(adapter);
      status |= qfle3_sm_start(adapter);

      QFLE3_DBG(QFLE3_DBG_SM, "uplink reset in state %s status %d\n",qfle3_sm_get_string(adapter->sm_state), (status == VMK_OK) ? 1 : 0);
   } else if (adapter->sm_state == QFLE3_SM_IOSTARTED){
      qfle3_deactive_dev(adapter);
      status = qfle3_sm_stop(adapter);
      qfle3_apply_uplink_changes(adapter);
      status |= qfle3_sm_start(adapter);
      if (status != VMK_OK)
         goto sm_start_fail;
      status |= qfle3_activate_dev(adapter);      
   } else if (adapter->sm_state == QFLE3_SM_STANDBY) {

      qfle3_apply_uplink_changes(adapter);
   }
sm_start_fail:
   vmk_BitVectorClear(adapter->state, QFLE3_STATE_BIT_RESETTING);
   if (status == VMK_OK) {
      return QFLE3_SMCMD_STATUS_COMPLETED;
   } else {
      return QFLE3_SMCMD_STATUS_ERROR;
   }
}


static QFLE3_SMCMD_STATUS qfle3_cmd_tostandby(struct qfle3_adapter *adapter, vmk_uint32 param)
{
   VMK_ReturnStatus status = VMK_FAILURE;

   QFLE3_DBG(QFLE3_DBG_SM, "qfle3_cmd_tostandby  in state %s \n",qfle3_sm_get_string(adapter->sm_state));
   if (adapter->sm_state == QFLE3_SM_ACTIVATED)
   {
      status = qfle3_sm_stop(adapter);
      status |= qfle3_unload(adapter);
   } else if (adapter->sm_state == QFLE3_SM_IOSTARTED){
      qfle3_deactive_dev(adapter);
      status = qfle3_sm_stop(adapter);
      status |= qfle3_unload(adapter);
   } else if (adapter->sm_state == QFLE3_SM_HWREADY) {
      status = qfle3_unload(adapter);
   }

   if (status == VMK_OK) {
      qfle3_sm_change_state(adapter, QFLE3_SM_STANDBY);
      return QFLE3_SMCMD_STATUS_COMPLETED;
   } else {
      return QFLE3_SMCMD_STATUS_ERROR;
   }
}
static QFLE3_SMCMD_STATUS qfle3_cmd_create_q(struct qfle3_adapter *adapter, int idx, vmk_uint32 param)
{
   VMK_ReturnStatus status;

   QFLE3_DBG(QFLE3_DBG_SM, "qfle3_cmd_create_q  idx is %d \n",idx);
   
   
   if (param & QFLE3_SM_PARAM_RXQ) {
      status = qfle3_rq_create(adapter, idx);
      if (status != VMK_OK)
         return QFLE3_SMCMD_STATUS_ERROR;

      return QFLE3_SMCMD_STATUS_COMPLETED;

   } else {

      QFLE3_DBG(QFLE3_DBG_SM, "qfle3_cmd_create_q  actual index is %d \n",QFLE3_TX_Q_NUM_TO_FP_NUM(adapter, idx));
      status = qfle3_tq_create(adapter, idx);
      if (status != VMK_OK)
         return QFLE3_SMCMD_STATUS_ERROR;

      return QFLE3_SMCMD_STATUS_COMPLETED;
   }

}
static QFLE3_SMCMD_STATUS qfle3_cmd_start_q(struct qfle3_adapter *adapter, int idx, vmk_uint32 param)
{
   VMK_ReturnStatus status = VMK_OK;


   QFLE3_DBG(QFLE3_DBG_SM, "qfle3_cmd_start_q  idx is %d \n",idx);
   

   if (param & QFLE3_SM_PARAM_RXQ) {

      status = qfle3_rq_start (adapter, idx);
      if (status != VMK_OK)
         return QFLE3_SMCMD_STATUS_ERROR;

      return QFLE3_SMCMD_STATUS_COMPLETED;
   } else {
      status = qfle3_tq_start (adapter, idx);
      if (status == VMK_OK) {
         return QFLE3_SMCMD_STATUS_COMPLETED;
      } else {
         return QFLE3_SMCMD_STATUS_ERROR;
      }      

   }
}

static QFLE3_SMCMD_STATUS qfle3_cmd_stop_q(struct qfle3_adapter *adapter, int id, vmk_uint32 param)
{

   if (param & QFLE3_SM_PARAM_RXQ) {

      qfle3_rq_stop(adapter, id);

      return QFLE3_SMCMD_STATUS_COMPLETED;
   } else {
      qfle3_tq_stop(adapter, id);
      return QFLE3_SMCMD_STATUS_COMPLETED;
   }
}


static QFLE3_SMCMD_STATUS qfle3_cmd_remove_q(struct qfle3_adapter *adapter, int idx, vmk_uint32 param)
{
   

   struct qfle3_fastpath *q;
   
   if (param & QFLE3_SM_PARAM_RXQ) {
      
      QFLE3_DBG(QFLE3_DBG_QUEUE, "Releasing Q idx %d\n", idx);
      qfle3_rq_release(adapter, idx);
   } else {
      q =  &adapter->fp[QFLE3_TX_Q_NUM_TO_FP_NUM(adapter,idx)];
      qfle3_assert_tx_ring(q);
      QFLE3_DBG(QFLE3_DBG_QUEUE, "Releasing Q idx %ld\n", q-&adapter->fp[0]);
      if (q->fp_state != QFLE3_SM_Q_CREATED) {
         QFLE3_ERR("Releasing TQ in the wrong state %d, Q idx %ld\n", q->fp_state,
            q - &adapter->fp[QFLE3_DEFAULT_TX_QID(adapter)]);
      }
      q->fp_state = 0;
   }
   return QFLE3_SMCMD_STATUS_COMPLETED;
}

struct cmd_table_entry {
   vmk_uint16	cmd;
   QFLE3_SMCMD_STATUS (*cmdfn) (struct qfle3_adapter *adapter, vmk_uint32 param);
   vmk_uint16  supported_state;
};

static struct cmd_table_entry cmd_table_entry[QFLE3_SMCMD_CREATE_Q] = {
   {QFLE3_SMCMD_UNKNOWN,      NULL, 0},
   {QFLE3_SMCMD_BRINGUP,      &qfle3_cmd_bringup, QFLE3_SM_SUPPORT(QFLE3_SM_STANDBY) | QFLE3_SM_SUPPORT(QFLE3_SM_IOSTARTED)},
   {QFLE3_SMCMD_START,        &qfle3_cmd_start, QFLE3_SM_SUPPORT(QFLE3_SM_HWREADY) | QFLE3_SM_SUPPORT(QFLE3_SM_IOSTARTED)},
   {QFLE3_SMCMD_RESUMEIO,     &qfle3_cmd_resume, QFLE3_SM_SUPPORT(QFLE3_SM_ACTIVATED) | QFLE3_SM_SUPPORT(QFLE3_SM_IOSTARTED)},
   {QFLE3_SMCMD_PAUSEIO,      &qfle3_cmd_pause, QFLE3_SM_SUPPORT(QFLE3_SM_ACTIVATED) | QFLE3_SM_SUPPORT(QFLE3_SM_IOSTARTED)},
   {QFLE3_SMCMD_STOP,         &qfle3_cmd_stop, QFLE3_SM_SUPPORT(QFLE3_SM_ACTIVATED) | QFLE3_SM_SUPPORT(QFLE3_SM_IOSTARTED)},
   {QFLE3_SMCMD_TOSTANDBY,    &qfle3_cmd_tostandby, QFLE3_SM_SUPPORT(QFLE3_SM_ACTIVATED) | QFLE3_SM_SUPPORT(QFLE3_SM_IOSTARTED) | QFLE3_SM_SUPPORT(QFLE3_SM_HWREADY)},
   {QFLE3_SMCMD_UPLINKRESET,  &qfle3_cmd_uplinkreset, QFLE3_SM_SUPPORT(QFLE3_SM_ACTIVATED) | QFLE3_SM_SUPPORT(QFLE3_SM_IOSTARTED) | QFLE3_SM_SUPPORT(QFLE3_SM_STANDBY)},

};

struct q_cmd_table_entry {
   vmk_uint16  cmd;
   QFLE3_SMCMD_STATUS (*qcmdfn) (struct qfle3_adapter *adapter, int idx, vmk_uint32 param);
   vmk_uint16 supported_state;
};

static struct q_cmd_table_entry q_cmd_table_entry [QFLE3_SMCMD_MAXCMD-QFLE3_SMCMD_CREATE_Q] = {

   {QFLE3_SMCMD_CREATE_Q,     &qfle3_cmd_create_q, QFLE3_SM_SUPPORT(QFLE3_SM_ACTIVATED) | QFLE3_SM_SUPPORT(QFLE3_SM_IOSTARTED) | QFLE3_SM_SUPPORT(QFLE3_SM_STANDBY) | QFLE3_SM_SUPPORT(QFLE3_SM_HWREADY)},
   {QFLE3_SMCMD_START_Q,      &qfle3_cmd_start_q, QFLE3_SM_SUPPORT(QFLE3_SM_ACTIVATED) | QFLE3_SM_SUPPORT(QFLE3_SM_IOSTARTED) | QFLE3_SM_SUPPORT(QFLE3_SM_STANDBY) | QFLE3_SM_SUPPORT(QFLE3_SM_HWREADY) },
   {QFLE3_SMCMD_REMOVE_Q,     &qfle3_cmd_remove_q, QFLE3_SM_SUPPORT(QFLE3_SM_ACTIVATED) | QFLE3_SM_SUPPORT(QFLE3_SM_IOSTARTED) | QFLE3_SM_SUPPORT(QFLE3_SM_STANDBY)},
   {QFLE3_SMCMD_STOP_Q,       &qfle3_cmd_stop_q, QFLE3_SM_SUPPORT(QFLE3_SM_ACTIVATED) | QFLE3_SM_SUPPORT(QFLE3_SM_IOSTARTED) | QFLE3_SM_SUPPORT(QFLE3_SM_STANDBY)},
};

QFLE3_SMCMD_STATUS qfle3_sm_cmd(struct qfle3_adapter *adapter, vmk_uint16 cmd, vmk_uint32 param)
{
   //vmk_uint16 state;
   vmk_uint16 i;
   QFLE3_SMCMD_STATUS status;
   QFLE3_SMCMD_STATUS (*fnptr) (struct qfle3_adapter *adapter, vmk_uint32 param);

   QFLE3_DBG(QFLE3_DBG_SM, "Current SM state %s, cmd received is  %s \n",qfle3_sm_get_string(adapter->sm_state), qfle3_sm_get_cmd_string(cmd));
   QFLE3_DBG(QFLE3_DBG_SM, "sm_state %s, previous state %s last cmd is  %s completion status %s\n",qfle3_sm_get_string(adapter->sm_state),
	     qfle3_sm_get_string(adapter->prev_state), qfle3_sm_get_cmd_string(adapter->last_cmd), qfle3_sm_get_cmdstatus_string(adapter->cmd_status));

   fnptr = NULL;
   status = QFLE3_SMCMD_STATUS_COMPLETED;


   vmk_SpinlockLock(adapter->sm_lock);
   if ( (adapter->cmd_status != QFLE3_SMCMD_STATUS_COMPLETED) &&
	(adapter->cmd_status != QFLE3_SMCMD_STATUS_ERROR))
      status = QFLE3_SMCMD_STATUS_NOTREADY;
   else {
      for(i = 0; i< QFLE3_SMCMD_CREATE_Q; i++) {
	 if ((cmd_table_entry[i].cmd == cmd) && (cmd_table_entry[i].supported_state & QFLE3_SM_SUPPORT(adapter->sm_state))){
	    adapter->prev_state = adapter->sm_state;
	    adapter->last_cmd = cmd;
	    adapter->cmd_status = QFLE3_SMCMD_STATUS_INPROGRESS;
	    fnptr = cmd_table_entry[i].cmdfn;
	    break;
	 }
      }

   }
   vmk_SpinlockUnlock(adapter->sm_lock);

   if (status == QFLE3_SMCMD_STATUS_NOTREADY) {
      QFLE3_ERR("Statemachine is Not Ready.  Rejecting cmd %s \n", qfle3_sm_get_cmd_string(cmd));

      qfle3_sm_cmd_error(adapter, cmd);
      return status;
   }
   if (fnptr == NULL) {
      QFLE3_ERR("Current SM state %s, does not support cmd %s !!!\n", qfle3_sm_get_string(adapter->sm_state), qfle3_sm_get_cmd_string(cmd));

      qfle3_sm_cmd_error(adapter, cmd);
      return QFLE3_SMCMD_STATUS_WRONGSTATE;
   }
   status = ((*fnptr)(adapter, param));
   QFLE3_DBG(QFLE3_DBG_SM, "cmd %s status is %s\n", qfle3_sm_get_cmd_string(cmd), qfle3_sm_get_cmdstatus_string(adapter->cmd_status));
   if (status == QFLE3_SMCMD_STATUS_COMPLETED)
      qfle3_sm_cmd_complete(adapter, cmd);
   else
      qfle3_sm_cmd_error(adapter, cmd);
   return adapter->cmd_status;
}

QFLE3_SMCMD_STATUS qfle3_sm_q_cmd(struct qfle3_adapter *adapter, vmk_uint16 cmd, int idx, vmk_uint32 param)
{
   //vmk_uint16 state;
   vmk_uint16 i;
   QFLE3_SMCMD_STATUS status;

   QFLE3_SMCMD_STATUS (*qfnptr) (struct qfle3_adapter *adapter, int idx, vmk_uint32 param);

   QFLE3_DBG(QFLE3_DBG_SM, "Current SM state %s, Q cmd received is  %s \n",qfle3_sm_get_string(adapter->sm_state), qfle3_sm_get_cmd_string(cmd));

   qfnptr = NULL;
   status = QFLE3_SMCMD_STATUS_COMPLETED;

//	   vmk_SpinlockLock(adapter->q_lock);
   for(i = 0; i< QFLE3_SMCMD_MAXCMD-QFLE3_SMCMD_CREATE_Q; i++) {
      if ((q_cmd_table_entry[i].cmd == cmd) && (q_cmd_table_entry[i].supported_state & QFLE3_SM_SUPPORT(adapter->sm_state))){
      qfnptr = q_cmd_table_entry[i].qcmdfn;
      break;
      }
   }

   if (qfnptr == NULL) {
      QFLE3_ERR("Current SM state %s, does not support cmd %s !!!\n", qfle3_sm_get_string(adapter->sm_state), qfle3_sm_get_cmd_string(cmd));
//	      vmk_SpinlockUnlock (adapter->q_lock);
      return QFLE3_SMCMD_STATUS_WRONGSTATE;
   }
   status = ((*qfnptr)(adapter, idx, param));
   QFLE3_DBG(QFLE3_DBG_SM, "cmd %s status is %s\n", qfle3_sm_get_cmd_string(cmd), qfle3_sm_get_cmdstatus_string(status));
//	   vmk_SpinlockUnlock (adapter->q_lock);
   return status;
}

int qfle3_phy_read(struct qfle3_adapter *adapter, int prtad,
               int devad, vmk_uint16 addr)
{
    u16 value;
    int rc;
    QFLE3_DBG(QFLE3_DBG_LINK, "mdio_read: prtad 0x%x, devad 0x%x, addr 0x%x\n",
       prtad, devad, addr);

    qfle3_acquire_phy_lock(adapter);
    rc = elink_phy_read(&adapter->link_params, prtad, devad, addr, &value);
    qfle3_release_phy_lock(adapter);
    QFLE3_DBG(QFLE3_DBG_LINK, "mdio_read_val 0x%x rc = 0x%x\n", value, rc);

    if (!rc)
        rc = value;
    else 
        return -1;
    return rc;
}

int qfle3_phy_write(struct qfle3_adapter *adapter, int prtad, int devad,
                vmk_uint16 addr, vmk_uint16 value)
{
    int rc;
    QFLE3_DBG(QFLE3_DBG_LINK,
       "mdio_write: prtad 0x%x, devad 0x%x, addr 0x%x, value 0x%x\n",
       prtad, devad, addr, value);

    qfle3_acquire_phy_lock(adapter);
    rc = elink_phy_write(&adapter->link_params, prtad, devad, addr, value);
    qfle3_release_phy_lock(adapter);
    return rc;
}
#ifdef QFLE3_CNIC
static void qfle3_ilt_init_op_cnic(qfle3_adapter *adapter, vmk_uint8 initop)
{
	if (CONFIGURE_NIC_MODE(adapter))
		ecore_ilt_client_id_init_op(adapter, ILT_CLIENT_SRC, initop);
	ecore_ilt_client_id_init_op(adapter, ILT_CLIENT_TM, initop);
}

int qfle3_init_hw_func_cnic(qfle3_adapter *adapter)
{
	int rc;

	qfle3_ilt_init_op_cnic(adapter, INITOP_SET);

	if (CONFIGURE_NIC_MODE(adapter)) {
		/* Configure searcher as part of function hw init */
		qfle3_init_searcher(adapter);

		/* Reset NIC mode */
		rc = qfle3_reset_nic_mode(adapter);
		if (rc)
			QFLE3_ERR("Can't change NIC mode!\n");
		return rc;
	}

	return 0;
}
VMK_ReturnStatus qfle3_alloc_cnicqs_mem(struct qfle3_adapter * adapter)
{
//   VMK_ReturnStatus status;
   struct qfle3_fastpath *fp;
   adapter->cnic_sb.e2_sb = qfle3_alloc_dma_mapping(adapter, 
                              adapter->dmaEngineCoherent, 
                              sizeof(struct host_hc_status_block_e2),
                              &adapter->cnic_sb_mapping);
   if (!adapter->cnic_sb.e2_sb)
      goto alloc_mem_err;

   if (CONFIGURE_NIC_MODE(adapter) && !adapter->t2) {
      
      QFLE3_INFO("Allocating Searcher T2 table for NIC mode!\n");
      /* allocate searcher T2 table, as it wasn't allocated before */
      adapter->t2 = qfle3_alloc_dma_mapping(adapter, adapter->dmaEngineCoherent,
                                          SRC_T2_SZ, &adapter->t2_mapping);
      if (!adapter->t2)
         goto alloc_mem_err;
   }

   /* write address to which L5 should insert its values */
   adapter->cnicEthDev->addr_drv_info_to_mcp =
      &adapter->sp->drv_info_to_mcp;

   if (ecore_ilt_mem_op_cnic(adapter, ILT_MEMOP_ALLOC))
      goto alloc_mem_err;



   if (!NO_FCOE(adapter)){
      /* FCoE */
      fp = qfle3_fcoe_fp(adapter);
      fp->qid = FCOE_IDX(adapter);
      fp->adapter = adapter;
      qfle3_set_fp_rx_buf_size(adapter, fp);
      fp->max_cos = 1;
      fp->txdata_ptr[FIRST_TX_COS_INDEX] = &adapter->qfle3_txq[FCOE_TXQ_IDX(adapter)];
      fp->txdata_ptr[FIRST_TX_COS_INDEX]->cos = FIRST_TX_COS_INDEX;
      fp->txdata_ptr[FIRST_TX_COS_INDEX]->txq_index = FCOE_TXQ_IDX(adapter);
      fp->txdata_ptr[FIRST_TX_COS_INDEX]->parent_fp = fp;
      if (qfle3_alloc_cnic_queue_mem(adapter, fp) != VMK_OK){
         /* we will fail load process instead of mark
          * NO_FCOE_FLAG
          */
         goto alloc_mem_err;
      }

   }
   if (!NO_ISCSI_OOO(adapter)) {
      
      /* ISCSI */
      fp = qfle3_ooo_fp(adapter);
      fp->qid = OOO_IDX(adapter);
      fp->adapter = adapter;
      qfle3_set_fp_rx_buf_size(adapter, fp);
      fp->max_cos = 1;
      fp->txdata_ptr[FIRST_TX_COS_INDEX] = &adapter->qfle3_txq[OOO_TXQ_IDX(adapter)];
      fp->txdata_ptr[FIRST_TX_COS_INDEX]->cos = FIRST_TX_COS_INDEX;
      fp->txdata_ptr[FIRST_TX_COS_INDEX]->txq_index = OOO_TXQ_IDX(adapter);
      fp->txdata_ptr[FIRST_TX_COS_INDEX]->parent_fp = fp;
      if (qfle3_alloc_cnic_queue_mem(adapter, fp) != VMK_OK){
         /* we will fail load process instead of mark
          * NO_FCOE_FLAG
          */
         goto alloc_mem_err;
      }
      
      fp = qfle3_fwd_fp(adapter);
      fp->qid = FWD_IDX(adapter);
      fp->adapter = adapter;
      fp->max_cos = 1;
      fp->txdata_ptr[FIRST_TX_COS_INDEX] = &adapter->qfle3_txq[FWD_TXQ_IDX(adapter)];
      fp->txdata_ptr[FIRST_TX_COS_INDEX]->cos = FIRST_TX_COS_INDEX;
      fp->txdata_ptr[FIRST_TX_COS_INDEX]->txq_index = FWD_TXQ_IDX(adapter);
      fp->txdata_ptr[FIRST_TX_COS_INDEX]->parent_fp = fp;
      if (qfle3_alloc_cnic_queue_mem(adapter, fp) != VMK_OK){
         /* we will fail load process instead of mark
          * NO_FCOE_FLAG
          */
         goto alloc_mem_err;
      }
   }
   if (qfle3_init_hw_func_cnic(adapter))
      goto alloc_mem_err;

   return VMK_OK;

alloc_mem_err:
   qfle3_free_cnicqs_mem(adapter);
   QFLE3_ERR("Can't allocate memory\n");
   return VMK_NO_MEMORY;
}


void qfle3_free_cnicqs_mem(struct qfle3_adapter * adapter)
{
   
   struct qfle3_fastpath *fp;
   ecore_ilt_mem_op_cnic(adapter, ILT_MEMOP_FREE);


//   QFLE3_PCI_FREE(adapter->t2, adapter->t2_mapping, SRC_T2_SZ);
   if (!NO_FCOE(adapter)){
      /* FCoE */
      fp = qfle3_fcoe_fp(adapter);
      
      qfle3_free_cnic_queue_mem(fp);
   }
   if (!NO_ISCSI_OOO(adapter)) {
      /* ISCSI */
      fp = qfle3_ooo_fp(adapter);
      qfle3_free_cnic_queue_mem(fp);
      
      fp = qfle3_fwd_fp(adapter);
      qfle3_free_cnic_queue_mem(fp);
      
   }
   
   if (adapter->cnic_sb.e2_sb){
      qfle3_free_dma_mapping(adapter, adapter->dmaEngineCoherent, 
         adapter->cnic_sb.e2_sb, adapter->cnic_sb_mapping,
                sizeof(union qfle3_host_hc_status_block));
      adapter->cnic_sb.e2_sb = NULL;
   }
   if (adapter->t2) {
      qfle3_free_dma_mapping(adapter,adapter->dmaEngineCoherent,
         adapter->t2,adapter->t2_mapping,SRC_T2_SZ);
      adapter->t2 = NULL;
   }
}


#endif
int qfle3_is_grc_disabled(struct qfle3_adapter * adapter)
{
	struct vmk_ListLinks *item;
   qfle3_adapter *t_bp;
   struct chip_core_dmp * dmp;
	/* Enable GRC dump only on one PCI function under an engine.
	 * Check if any of the PCI functions which are driven by
	 * same engine as given function, already has the GRC dump
	 * enabled.
	 */
	
   VMK_LIST_FORALL (&qfle3_mod_info.adapterList, item) {
      t_bp = VMK_LIST_ENTRY(item, struct qfle3_adapter, adapterLink);
		if (t_bp == NULL || t_bp == adapter)
			continue;
      dmp = (struct chip_core_dmp *)t_bp->fwdmp_buf;

		if ((t_bp->pdev_addr.seg == adapter->pdev_addr.seg) &&
		    (t_bp->pdev_addr.bus == adapter->pdev_addr.bus) &&
		    (t_bp->pdev_addr.dev == adapter->pdev_addr.dev) &&
		    (QFLE3_PATH(adapter) == QFLE3_PATH(t_bp)) &&
			(t_bp->fwdmp_flags & GRC_DMP_PARITY_ENABLED)) {
			/* Found an already enabled function. */
			return 0;
		}
	}

	/* None enabled. */
	return 1;
}

/* Trigger reset on other PFs that belong to same engine.
 */
void qfle3_reset_engine_pfs(struct qfle3_adapter * adapter)
{
   
	struct vmk_ListLinks *item;
   qfle3_adapter *t_bp;
   int peer_count = 0;
   
//	if (!auto_recovery)
//		return;

   VMK_LIST_FORALL (&qfle3_mod_info.adapterList, item) {
      t_bp = VMK_LIST_ENTRY(item, struct qfle3_adapter, adapterLink);
		if (t_bp == NULL || t_bp == adapter)
			continue;
//      QFLE3_ERR("Informing PEER function %d:%d:%02d.%02d\n",t_bp->pdev_addr.seg, t_bp->pdev_addr.bus,
//         t_bp->pdev_addr.dev, t_bp->pdev_addr.fn);
		if (!vmk_BitVectorTest(t_bp->state, QFLE3_STATE_BIT_ATTACHED) ||
		    (t_bp->recovery_state != QFLE3_RECOVERY_DONE)) {
			continue;
		}

		if ((t_bp->pdev_addr.seg == adapter->pdev_addr.seg) &&
		    (t_bp->pdev_addr.bus == adapter->pdev_addr.bus) &&
		    (t_bp->pdev_addr.dev == adapter->pdev_addr.dev) &&
		    (QFLE3_PATH(adapter) == QFLE3_PATH(t_bp))) {
         QFLE3_ERR("Triggering PEER reset on engine function %d:%d:%02d.%02d\n",t_bp->pdev_addr.seg, t_bp->pdev_addr.bus,
            t_bp->pdev_addr.dev, t_bp->pdev_addr.fn);
			/* Trigger tx timeout on this PF.*/
			QFLE3_TRIGGER_TX_TO(t_bp, QFLE3_ERR_PEER_RESET);
         peer_count++;
		}
      
	}
   QFLE3_ERR("TX TO triggered on %d peers\n",peer_count);

}
void qfle3_rec_prepare(struct qfle3_adapter * adapter)
{
//	/* Avoid redundant Tx timeout callback.*/
//	vmk_GetTimeOfDay(&adapter->trans_start);

	/* Mark Link Down.*/
   qfle3_link_down(adapter);
   
	adapter->link_vars.link_up = 0;
	/* Release rtnl lock and wait for some time to allow
	 * link status to be fetched by ethtool callback.
	 * This will allow failover at upper layer.
	 */
//	rtnl_unlock();
	vmk_DelayUsecs(800);
//	rtnl_lock();

	/* Ensure PCI Bus Master is enabled.*/
   qfle3_enable_bus_master(adapter);
}

void qfle3_schedule_sp_rtnl(struct qfle3_adapter * adapter, enum sp_rtnl_flag flag, vmk_uint32 verbose)
{
   vmk_CPUMemFenceReadWrite();
   vmk_BitVectorSet(adapter->sp_rtnl_state,flag);
   vmk_CPUMemFenceReadWrite();
	QFLE3_DBG((QFLE3_DBG_SP| verbose), "Scheduling sp_rtnl task [Flag: %d]\n",
	   flag);   
   qfle3_schedule_helper (adapter, adapter->recovery_helper, qfle3_recovery_func, 0);
}

/* Tx error handler.
 */
void qfle3_tx_error(struct qfle3_adapter * adapter)
{
   
//   int i, qid;
   
//   struct qfle3_fastpath *tq;
   struct cnic_ctl_info ctl;

	/* If this is called as part of recovery resume,
	 * then, error recovery is already in progress.
	 */
	if (adapter->recovery_state == QFLE3_RECOVERY_WAIT) {
		QFLE3_ERR("ESX_REC_WAIT DS:0x%x ES:0x%x.\n",
		   adapter->state->vector[0], adapter->error_status);
		return;
	}

   QFLE3_ERR("ESX_REC_INIT DS:0x%x RS:%d ES:0x%x to:%d rec:%d.\n",
        adapter->state->vector[0], adapter->recovery_state, adapter->error_status,
        adapter->tx_to_delay, auto_recovery);

   qfle3_rec_prepare(adapter);
   
   /* Log driver dump. */
   if ((adapter->error_status & QFLE3_ERR_NETDEV_TO) &&
       !(adapter->error_status & QFLE3_ERR_PEER_RESET) &&
       (!adapter->panic)) {
//      qfle3_panic_dump(adapter, VMK_FALSE);
//      QFLE3_INFO("qfle3_panic_dump\n");

   }

   /* Release tx buffers to unblock upper layers which
    * may be waiting for completions.
    */
//   FOREACH_TX_ETH_QUEUE(i,qid,tq) 
//         qfle3_free_tx_queue_pkt(adapter, tq);
//   }
	/* Inititate recovery, in case of TX timeout. */
	if (adapter->error_status & QFLE3_ERR_NETDEV_TO) {
      if (auto_recovery) {
         
         QFLE3_INFO("Attempt to recovery\n");
         adapter->panic = 0;
         adapter->recovery_state = QFLE3_RECOVERY_INIT;
         
         vmk_BitVectorAtomicTestAndSet(adapter->state, QFLE3_STATE_BIT_RESETTING);
         
         // send Recovery command to cnic
         ctl.cmd = CNIC_CTL_NIC_RECOVERY_CMD;
         
         if (adapter->cnicOps)
            adapter->cnicOps->cnicNotify(adapter->cnic_data, &ctl);
         qfle3_int_disable(adapter);
         qfle3_parity_recover(adapter);
      } else {
         qfle3_force_link_reset(adapter);
         if (qfle3_sm_cmd (adapter, QFLE3_SMCMD_TOSTANDBY, 0) != QFLE3_SMCMD_STATUS_COMPLETED)
         {
            QFLE3_ERR("Unable to execute QFLE3_SMCMD_TOSTANDBY\n");
         }
         adapter->error_status = QFLE3_ERR_NONE;
         QFLE3_ERR("ESX_REC_INFO: Leaving device in closed state.\n");
      }
   	QFLE3_ERR("ESX_REC_INFO DS:0x%x RS:%d ES:0x%x.\n",
   		  adapter->state->vector[0], adapter->recovery_state, adapter->error_status);
	}
   return;
}
void qfle3_tx_timeout(struct qfle3_adapter * adapter)
{
//   int donot_recover_flag;
//   
//	donot_recover_flag = vmk_BitVectorAtomicTestAndSet(adapter->sp_rtnl_state, QFLE3_SP_RTNL_DONOT_RECOVER);
//   if (donot_recover_flag) {
//      QFLE3_ERR("Driver is not able to recover from parity error at this point");
//   }else {
//      adapter->last_recovery_start = vmk_GetTimerCycles();
//   }

   QFLE3_ERR("TX Timeout Triggered on engine function %d:%d:%02d.%02d  DS:0x%x RS:%d ES:0x%x\n",
      adapter->pdev_addr.seg, adapter->pdev_addr.bus, adapter->pdev_addr.dev, adapter->pdev_addr.fn,
        adapter->state->vector[0], adapter->recovery_state, adapter->error_status);
   

	if (adapter->recovery_state != QFLE3_RECOVERY_DONE) {
		/* Probably triggered by netdev, while recovery
		 * is already in progress. Hence discard.
		 */
		QFLE3_ERR("ESX_REC_BSY DS:0x%x RS:%d ES:0x%x\n",
			  adapter->state->vector[0], adapter->recovery_state, adapter->error_status);

//      /* Avoid redundant Tx timeout callback.*/
//      vmk_GetTimeOfDay(&adapter->trans_start);
//    current timeout is set to 60 seconds.

		return;
	}

// for now treat everything same as TX TO
	qfle3_trigger_grcdump(adapter, QFLE3_CXT_TXTO_TRIGGER);

//	/* This allows the netif to be shutdown gracefully before resetting */
//	bnx2x_schedule_sp_rtnl(adapter, BNX2X_SP_RTNL_TX_TIMEOUT, 0);
   vmk_CPUMemFenceReadWrite();
   vmk_BitVectorSet(adapter->sp_rtnl_state,QFLE3_SP_RTNL_TX_TIMEOUT);
   vmk_CPUMemFenceReadWrite();
//   if (!donot_recover_flag) {
      qfle3_schedule_helper (adapter, adapter->recovery_helper, qfle3_recovery_func, 1000);
//   }
}
#if 0
/* Returns 0, if recovery is needed. Else returns 1.*/
int qfle3_parity_check(struct qfle3_adapter * adapter)
{
	int cancel_recovery = 0;
	QFLE3_ERR("Enterd\n");
	if (!START_RECOVERY(adapter)) {
      // There is no ongoing recovery attempt,
      // we will attempt to recover
		adapter->error_status = QFLE3_ERR_PARITY;
		qfle3_trigger_grcdump(adapter, QFLE3_CXT_PARITY_TRIGGER);
      adapter->recovery_count++;
	} else {
		adapter->error_status |= QFLE3_ERR_PARITY;
		cancel_recovery = 1;
	}

	QFLE3_ERR("ESX_REC_PRT DS:0x%x ES:0x%x RS:%d cr:%d\n",
		  adapter->state->vector[0], adapter->error_status,
		  adapter->recovery_state, cancel_recovery);
	QFLE3_ERR("exit\n");
	return cancel_recovery;
}
static u16 qfle3_free_tx_pkt(struct qfle3_adapter * adapter, 
   struct qfle3_fp_txdata *txdata,
			     u16 idx, unsigned int *pkts_compl,
			     unsigned int *bytes_compl)
{
	struct sw_tx_bd *tx_buf = &txdata->tx_buf_ring[idx];
	struct eth_tx_start_bd *tx_start_bd;
	struct eth_tx_bd *tx_data_bd;
	struct sk_buff *skb = tx_buf->skb;
	u16 bd_idx = TX_BD(tx_buf->first_bd), new_cons;
	int nbd;
	u16 split_bd_len = 0;


//	DP(NETIF_MSG_TX_DONE, "fp[%d]: pkt_idx %d  buff @(%p)->skb %p\n",
//	   txdata->txq_index, idx, tx_buf, skb);

	tx_start_bd = &txdata->tx_desc_ring[bd_idx].start_bd;

	nbd = le16toh(tx_start_bd->nbd) - 1;
	new_cons = nbd + tx_buf->first_bd;

	/* Get the next bd */
	bd_idx = TX_BD(NEXT_TX_IDX(bd_idx));

	/* Skip a parse bd... */
	--nbd;
	bd_idx = TX_BD(NEXT_TX_IDX(bd_idx));

	if (tx_buf->flags & BNX2X_HAS_SECOND_PBD) {
		/* Skip second parse bd... */
		--nbd;
		bd_idx = TX_BD(NEXT_TX_IDX(bd_idx));
	}

	/* TSO header+data bds share a common mapping. see bnx2x_tx_split() */
	if (tx_buf->flags & QFLE3_TSO_SPLIT_BD) {
		tx_data_bd = &txdata->tx_desc_ring[bd_idx].reg_bd;
		split_bd_len = BD_UNMAP_LEN(tx_data_bd);
		--nbd;
		bd_idx = TX_BD(NEXT_TX_IDX(bd_idx));
	}

	/* unmap first bd */
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 34)) /* BNX2X_UPSTREAM */
	dma_unmap_single(&bp->pdev->dev, BD_UNMAP_ADDR(tx_start_bd),
			 BD_UNMAP_LEN(tx_start_bd) + split_bd_len,
			 DMA_TO_DEVICE);
#else
	pci_unmap_single(bp->pdev, BD_UNMAP_ADDR(tx_start_bd),
			 BD_UNMAP_LEN(tx_start_bd) + split_bd_len,
			 PCI_DMA_TODEVICE);
#endif

	/* now free frags */
	while (nbd > 0) {

		tx_data_bd = &txdata->tx_desc_ring[bd_idx].reg_bd;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 34)) /* BNX2X_UPSTREAM */
		dma_unmap_page(&bp->pdev->dev, BD_UNMAP_ADDR(tx_data_bd),
			       BD_UNMAP_LEN(tx_data_bd), DMA_TO_DEVICE);
#else
		pci_unmap_page(bp->pdev, BD_UNMAP_ADDR(tx_data_bd),
			       BD_UNMAP_LEN(tx_data_bd), PCI_DMA_TODEVICE);
#endif
		if (--nbd)
			bd_idx = TX_BD(NEXT_TX_IDX(bd_idx));
	}

	/* release skb */
	WARN_ON(!skb);
	if (likely(skb)) {
		(*pkts_compl)++;
		(*bytes_compl) += skb->len;
		dev_kfree_skb_any(skb);
	}

	tx_buf->first_bd = 0;
	tx_buf->skb = NULL;

	return new_cons;
}
static void qfle3_free_tx_queue_pkt(qfle3_adapter * adapter,
					  struct qfle3_fastpath *fp)
{
   u8 cos;

   for_each_cos_in_tx_queue(fp, cos) {
      struct qfle3_fp_txdata *txdata = fp->txdata_ptr[cos];
      unsigned pkts_compl = 0, bytes_compl = 0;

      u16 sw_prod = txdata->tx_pkt_prod;
      u16 sw_cons = txdata->tx_pkt_cons;

      while (sw_cons != sw_prod) {
         qfle3_free_tx_pkt(adapter, txdata, TX_BD(sw_cons),
                 &pkts_compl, &bytes_compl);
         sw_cons++;
      }
      txdata->tx_pkt_cons = sw_cons;
}
#endif
/* Handler for parity error trigger.
 * Returns 0, if recovery is needed. Else returns 1.
 */
int qfle3_parity_error(struct qfle3_adapter * adapter)
{
   
   int cancel_recovery = 0;
   int donot_recover_flag;
	/* If donot_recover_flag is set, or recovery is in progress, 
	 * do not init recovery again.
	 */
	donot_recover_flag = vmk_BitVectorAtomicTestAndSet(adapter->sp_rtnl_state, QFLE3_SP_RTNL_DONOT_RECOVER);
   if (donot_recover_flag) {
      QFLE3_ERR("Driver is not able to recover from parity error at this point");
      return 1;
   }
   adapter->last_recovery_start = vmk_GetTimerCycles();
   
   if (adapter->error_status)
      cancel_recovery = 1;
   
	/* Parity error notified */
	adapter->error_status |= QFLE3_ERR_PARITY;

   QFLE3_ERR("ESX_REC_PRT DS:0x%x RS:%d ES:0x%x to:%d rec:%d cr:%d.\n",
        adapter->state->vector[0], adapter->recovery_state, adapter->error_status,
        adapter->tx_to_delay, auto_recovery, cancel_recovery);

   if (!cancel_recovery)
      qfle3_trigger_grcdump(adapter, QFLE3_CXT_PARITY_TRIGGER);
   
   return cancel_recovery;
}

void qfle3_panic(struct qfle3_adapter * adapter)
{
   
	QFLE3_ERR("driver assert RS:%d ES:0x%x P:%d\n",
		  adapter->recovery_state, adapter->error_status, adapter->panic);
   
	/* Ignore panic, while engine reset is in progress */
	if ((adapter->recovery_state == QFLE3_RECOVERY_DONE) &&
	    !(adapter->error_status & QFLE3_ERR_PEER_RESET)) {
		adapter->panic = 1;
//		qfle3_panic_dump(adapter, VMK_FALSE);
   QFLE3_INFO("qfle3_panic_dump\n");

	}

	QFLE3_TRIGGER_TX_TO(adapter, QFLE3_ERR_PANIC);
}

void qfle3_panic_dump(struct qfle3_adapter * adapter, vmk_Bool disable_int)
{
   int i, qid, j;
   struct qfle3_fastpath *fp;
	struct hc_sp_status_block_data sp_sb_data;
	int func = QFLE3_FUNC(adapter);
   extern vmk_uint32 psod_on_panic;
   char logline[1024];
   int pos = 0;
   
#define QFLE3_DUMP(fmt, args...) \
       pos += vmk_Sprintf(logline+pos, fmt "", ##args)
#define QFLE3_DUMP_LN(fmt, args...) \
          vmk_Sprintf(logline+pos, fmt "\n", ##args); pos=0;\
          vmk_LogNoLevel(VMK_LOG_URGENCY_NORMAL, "%s", logline)
   
	if (IS_PF(adapter) && disable_int)
		qfle3_int_disable(adapter);

	adapter->stats_state = STATS_STATE_DISABLED;
	adapter->eth_stats.unrecoverable_error++;
	QFLE3_DBG(QFLE3_DBG_STATS, "stats_state - DISABLED\n");


	QFLE3_ERR("begin crash dump -----------------\n");

	/* Indices */
	/* Common */
	if (IS_PF(adapter)) {
		struct host_sp_status_block *def_sb = adapter->def_status_blk;
		int data_size, cstorm_offset;

		QFLE3_DUMP_LN("def_idx(0x%x)  def_att_idx(0x%x)  attn_state(0x%x)  spq_prod_idx(0x%x) next_stats_cnt(0x%x)\n",
			  adapter->def_idx, adapter->def_att_idx, adapter->attn_state,
			  adapter->spq_prod_idx, adapter->stats_counter);
		QFLE3_DUMP_LN("DSB: attn bits(0x%x)  ack(0x%x)  id(0x%x)  idx(0x%x)\n",
			  def_sb->atten_status_block.attn_bits,
			  def_sb->atten_status_block.attn_bits_ack,
			  def_sb->atten_status_block.status_block_id,
			  def_sb->atten_status_block.attn_bits_index);
		QFLE3_DUMP("     def (");
		for (i = 0; i < HC_SP_SB_MAX_INDICES; i++)
			QFLE3_DUMP("0x%x%s",
				def_sb->sp_sb.index_values[i],
				(i == HC_SP_SB_MAX_INDICES - 1) ? ")  " : " ");
      QFLE3_DUMP_LN("\n");
		data_size = sizeof(struct hc_sp_status_block_data) /
			    sizeof(u32);
		cstorm_offset = CSTORM_SP_STATUS_BLOCK_DATA_OFFSET(func);
		for (i = 0; i < data_size; i++)
			*((u32 *)&sp_sb_data + i) =
				REG_RD(adapter, BAR_CSTRORM_INTMEM + cstorm_offset +
					   i * sizeof(u32));

		QFLE3_DUMP_LN("igu_sb_id(0x%x)  igu_seg_id(0x%x) pf_id(0x%x)  vnic_id(0x%x)  vf_id(0x%x)  vf_valid (0x%x) state(0x%x)\n",
			sp_sb_data.igu_sb_id,
			sp_sb_data.igu_seg_id,
			sp_sb_data.p_func.pf_id,
			sp_sb_data.p_func.vnic_id,
			sp_sb_data.p_func.vf_id,
			sp_sb_data.p_func.vf_valid,
			sp_sb_data.state);
	}
   FOREACH_ETH_QUEUE(i,qid,fp) 
		int loop;
		struct hc_status_block_data_e2 sb_data_e2;
		struct hc_status_block_data_e1x sb_data_e1x;
		struct hc_status_block_sm  *hc_sm_p =
			CHIP_IS_E1x(adapter) ?
			sb_data_e1x.common.state_machine :
			sb_data_e2.common.state_machine;
		struct hc_index_data *hc_index_p =
			CHIP_IS_E1x(adapter) ?
			sb_data_e1x.index_data :
			sb_data_e2.index_data;
		u8 data_size, cos;
		u32 *sb_data_p;
		struct qfle3_fp_txdata txdata;

		QFLE3_DUMP_LN("fp[%d] fp %p qid %d\n",i, fp, qid);
		if (!fp->rx_cq_cons_sb)
			continue;
		/* Rx */
      
		QFLE3_DUMP_LN("rx\n");
		QFLE3_DUMP_LN("fp%d: rx_bd_prod(0x%x)  rx_bd_cons(0x%x)  rx_comp_prod(0x%x)  rx_comp_cons(0x%x)  *rx_cons_sb(0x%x)\n",
			  i, fp->rx_bd_prod, fp->rx_bd_cons,
			  fp->rx_cq_prod,
			  fp->rx_cq_cons, le16toh(*fp->rx_cq_cons_sb));
		QFLE3_DUMP_LN("     rx_sge_prod(0x%x)  last_max_sge(0x%x)  fp_hc_idx(0x%x)\n",
			  fp->rx_sge_prod, fp->last_max_sge,
			  le16toh(fp->fp_hc_idx));
		QFLE3_DUMP_LN("tx\n");
		/* Tx */
		for_each_cos_in_tx_queue(fp, cos)
		{
         
         QFLE3_DUMP_LN("tx[%d]\n", cos);
			if (!fp->txdata_ptr[cos])
				break;

			txdata = *fp->txdata_ptr[cos];

			if (!txdata.tx_cons_sb)
				continue;

			QFLE3_DUMP_LN("fp%d: tx_pkt_prod(0x%x)  tx_pkt_cons(0x%x)  tx_bd_prod(0x%x)  tx_bd_cons(0x%x)  *tx_cons_sb(0x%x)\n",
				  i, txdata.tx_pkt_prod,
				  txdata.tx_pkt_cons, txdata.tx_bd_prod,
				  txdata.tx_bd_cons,
				  le16toh(*txdata.tx_cons_sb));
		}
		QFLE3_DUMP_LN("cos\n");
		loop = CHIP_IS_E1x(adapter) ?
			HC_SB_MAX_INDICES_E1X : HC_SB_MAX_INDICES_E2;

		/* host sb data */

		if (IS_FCOE_FP(fp))
			continue;

		QFLE3_DUMP("     run indexes (");
		for (j = 0; j < HC_SB_MAX_SM; j++)
			QFLE3_DUMP("0x%x%s",
			       fp->sb_running_index[j],
			       (j == HC_SB_MAX_SM - 1) ? ")" : " ");
      QFLE3_DUMP_LN("\n");
		QFLE3_DUMP("     indexes (");
		for (j = 0; j < loop; j++)
			QFLE3_DUMP("0x%x%s",
			       fp->sb_index_values[j],
			       (j == loop - 1) ? ")" : " ");
      QFLE3_DUMP_LN("\n");
		/* VF cannot access FW refelection for status block */
//		if (IS_VF(adapter))
//			continue;

		/* fw sb data */
		data_size = CHIP_IS_E1x(adapter) ?
			sizeof(struct hc_status_block_data_e1x) :
			sizeof(struct hc_status_block_data_e2);
		data_size /= sizeof(u32);
		sb_data_p = CHIP_IS_E1x(adapter) ?
			(u32 *)&sb_data_e1x :
			(u32 *)&sb_data_e2;
		/* copy sb data in here */
		for (j = 0; j < data_size; j++)
			*(sb_data_p + j) = REG_RD(adapter, BAR_CSTRORM_INTMEM +
				CSTORM_STATUS_BLOCK_DATA_OFFSET(fp->fw_sb_id) +
				j * sizeof(u32));

		if (!CHIP_IS_E1x(adapter)) {
			QFLE3_DUMP_LN("pf_id(0x%x)  vf_id(0x%x)  vf_valid(0x%x) vnic_id(0x%x)  same_igu_sb_1b(0x%x) state(0x%x)\n",
				sb_data_e2.common.p_func.pf_id,
				sb_data_e2.common.p_func.vf_id,
				sb_data_e2.common.p_func.vf_valid,
				sb_data_e2.common.p_func.vnic_id,
				sb_data_e2.common.same_igu_sb_1b,
				sb_data_e2.common.state);
		} else {
			QFLE3_DUMP_LN("pf_id(0x%x)  vf_id(0x%x)  vf_valid(0x%x) vnic_id(0x%x)  same_igu_sb_1b(0x%x) state(0x%x)\n",
				sb_data_e1x.common.p_func.pf_id,
				sb_data_e1x.common.p_func.vf_id,
				sb_data_e1x.common.p_func.vf_valid,
				sb_data_e1x.common.p_func.vnic_id,
				sb_data_e1x.common.same_igu_sb_1b,
				sb_data_e1x.common.state);
		}

		/* SB_SMs data */
		for (j = 0; j < HC_SB_MAX_SM; j++) {
			QFLE3_DUMP_LN("SM[%d] __flags (0x%x) igu_sb_id (0x%x)  igu_seg_id(0x%x) time_to_expire (0x%x) timer_value(0x%x)\n",
				j, hc_sm_p[j].__flags,
				hc_sm_p[j].igu_sb_id,
				hc_sm_p[j].igu_seg_id,
				hc_sm_p[j].time_to_expire,
				hc_sm_p[j].timer_value);
		}

		/* Indices data */
		for (j = 0; j < loop; j++) {
			QFLE3_DUMP_LN("INDEX[%d] flags (0x%x) timeout (0x%x)\n", j,
			       hc_index_p[j].flags,
			       hc_index_p[j].timeout);
		}
	}
	if (IS_PF(adapter)) {
      
//		int tmp_msg_en = adapter->msg_enable;
		qfle3_fw_dump(adapter);
//		bp->msg_enable |= NETIF_MSG_HW;
//		QFLE3_ERR("Idle check (1st round) ----------\n");
//		qfle3_idle_chk(adapter);
//		QFLE3_ERR("Idle check (2nd round) ----------\n");
//		bnx2x_idle_chk(adapter);
//		bp->msg_enable = tmp_msg_en;
//		qfle3_mc_assert(adapter);
	}
	QFLE3_ERR("end crash dump -----------------\n");

	if (psod_on_panic) {
		vmk_DelayUsecs(100000);
      vmk_PanicWithModuleID(vmk_ModuleCurrentID, "qfle3_panic\n");
	}
}

