/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

#include "qfle3i.h"

extern vmk_LogComponent qfle3i_vmkLog;
extern vmk_MgmtApiSignature kvSig;
extern char qfle3i_version_str[40];

vmk_DMAConstraints qfle3i_dmaConstraint = {
   .sgMaxEntries     = ISCSI_MAX_SG_PER_CMD,
   /* Required scatter/gather element alignment */
   .sgElemAlignment  = 4,
   /* If non-zero, scatter/gather element length multiple */
   .sgElemSizeMult   = 0,
   .sgElemMaxSize    = 0xffffffff,
   /* Max # of blocks per i/o */
   .maxTransfer      = 0xffffffff,
   /* Address mask to get boundary limit for scatter/gather element */
   .sgElemStraddle   = VMK_ADDRESS_MASK_32BIT+1,
   .addressMask      = VMK_ADDRESS_MASK_64BIT,
};


void qfle3i_scsi_scan(unsigned long data)
{
	struct qfle3i_hba *hba = NULL;
	vmk_ListLinks *current, *next_ptr;
	struct qfle3i_conn *conn = NULL;
	struct qfle3i_conn *tmp_conn = NULL;
	struct qfle3i_sess *sess = NULL;
	VMK_ReturnStatus status = VMK_OK;
	vmk_uint32 target_id;
	vmk_uint32 channel_id;

	hba = (struct qfle3i_hba *)data;

	PRINT_NOTICE(hba, "Enter...\n");

	if (qfle3i_adapter_ready(hba) != VMK_OK) {
		PRINT_ERR(hba, "adapter not ready.\n");
		return;
	}

check:
	if(vmk_SpinlockLock(hba->scsi_scan_list_lck) == VMK_DEATH_PENDING)
		goto exit_world;

	// Debug patch for CQ90118, to check busy loop.
	if (ql_vmk_is_list_looped(&hba->scsi_scan_list, __func__)) {
		vmk_SpinlockUnlock(hba->scsi_scan_list_lck);
		PRINT_ERR(hba, "hba->scsi_scan_list corrupted. ASSERT here\n");
		sess = NULL;
		hba = NULL;
		conn = NULL;
		PRINT_NOTICE(hba, "work on adapter:%s sess:%p ep_iscsi_cid:0x%x "
					   "conn:%p target_id:%d \n",
			(char *)hba->scsiAdapter->name.string, sess,
			conn->ep->ep_iscsi_cid, conn, sess->target_id);
	}

	/* This is the condition to exit from the tasklet */
	if (vmk_ListIsEmpty(&hba->scsi_scan_list) == VMK_TRUE){
		vmk_SpinlockUnlock(hba->scsi_scan_list_lck);
		goto exit_world;
	}

	PRINT_NOTICE(hba, "atleast one target pending for scan...\n");
	conn = VMK_LIST_ENTRY(vmk_ListFirst(&hba->scsi_scan_list),
						struct qfle3i_conn, scsi_scan_list);
	vmk_ListRemove(&conn->scsi_scan_list);
	vmk_SpinlockUnlock(hba->scsi_scan_list_lck);

	sess = conn->sess;
	/* In stress testcase, we may stuck in vmk_ScsiScanAndClaimPaths() while 
	   sess and its parameters gets destroyed, better to save these values,
	   which will be needed for vmk_ScsiNotifyPathStateChangeAsync() function.
	*/
	channel_id = sess->channel_id;
	target_id = sess->target_id;

	PRINT_NOTICE(hba, "work on adapter:%s sess:%p ep_iscsi_cid:0x%x "
			  "conn:%p channel_id:%d target_id:%d \n",
			(char *)hba->scsiAdapter->name.string, sess,
			conn->ep->ep_iscsi_cid, conn, channel_id, target_id);

	status = vmk_ScsiScanAndClaimPaths(
			&hba->scsiAdapter->name, channel_id,
			target_id, VMK_SCSI_PATH_ANY_LUN);

	PRINT_NOTICE(hba, "Invoked scan for channel:%d target_id:%d\n",
						channel_id, target_id);

	if (status == VMK_OK) {
		status = vmk_ScsiNotifyPathStateChangeAsync(
				hba->scsiAdapter,
				channel_id,
				target_id, -1);
		PRINT_NOTICE(hba, "Notified Path State Change: "
				"Target port ACTIVE: Channel=0x%x TGT ID=%d\n",
				channel_id, target_id);
	}

	if (status != VMK_OK)
		PRINT_NOTICE(hba, "scsi_scan path failed. status:%s\n",
				vmk_StatusToString(status));
/*
	if (status != VMK_OK) {
		PRINT_NOTICE(hba, "scsi_scan path failed. status:%s\n",
							vmk_StatusToString(status));
		sess->scsi_scan_counter++;
		if (sess->scsi_scan_counter <= MAX_SCSISCAN_COUNTER)
			qfle3i_queue_scsi_scan(conn);
	} else
		sess->scsi_scan_counter = 0;
*/
	goto check;
exit_world:
	PRINT_NOTICE(hba, "Exit...\n");
	return;
}


VMK_ReturnStatus qfle3i_queue_scsi_scan(struct qfle3i_conn *conn)
{
	struct qfle3i_sess *sess = conn->sess;
	struct qfle3i_endpoint *ep = conn->ep;
	struct qfle3i_hba *hba = sess->hba;

	PRINT_NOTICE(hba, "Enter...\n");

	if (conn->state == CONN_STATE_FFP_STATE) {
		PRINT_NOTICE(hba, "scanning for sess %p conn %p hba %p ep %p ep_iscsi_cid:0x%x\n",
				sess, conn, hba, ep, ep->ep_iscsi_cid);

		if(vmk_SpinlockLock(hba->scsi_scan_list_lck) == VMK_DEATH_PENDING) {
			PRINT_NOTICE(hba, "scsi_scan_list_lck is in VMK_DEATH_PENDING\n");
			return VMK_DEATH_PENDING;
		}

		vmk_ListInsert(&conn->scsi_scan_list,
				vmk_ListAtRear(&hba->scsi_scan_list));

		vmk_SpinlockUnlock(hba->scsi_scan_list_lck);
	}
	PRINT_NOTICE(hba, "Exit...\n");
	return VMK_OK;
}


static VMK_ReturnStatus qfle3i_queuecommand(
			void            *client_data,
			vmk_ScsiCommand *cmd,
			void            *device_data)
{
	struct qfle3i_scsi_task *scsi_task;
	struct qfle3i_conn *conn;

	if ((!device_data) || (!client_data) ||(!cmd))
		return VMK_BAD_PARAM;

	qfle3i_iscsilun *iscsilun = (qfle3i_iscsilun *)device_data;
	qfle3i_sess *sess = iscsilun->sess;
	qfle3i_hba *hba = (qfle3i_hba *)client_data;

	if ((!sess) || sess->hba != hba)
		goto dev_not_found;

	if (!hba->is_nic_up)
		goto dev_not_found;

	if (sess->state == QFLE3I_SESS_DESTROYED)
		goto dev_offline;

	if (vmk_AtomicRead64(&sess->device_offline)) {
		QFLE3I_DBG(DBG_IO, hba, "device is offline, cmd:%p, cmdID:0x%lx\n",
					cmd, cmd->cmdId.serialNumber);
		goto dev_offline;
	}

	if (sess->state == QFLE3I_SESS_IN_SHUTDOWN ||
	    sess->state == QFLE3I_SESS_IN_LOGOUT ||
		!sess->lead_conn || !sess->if_unblocked)
		/* delay offline indication till session is destroyed */
		goto cmd_not_accepted;

	/* TMF command is active on this ilun, so return from here */
	if (iscsilun->tmf_in_progress){
		QFLE3I_DBG(DBG_IO, hba, "TMF in progress, cmd:%p cmdID:0x%lx\n",
				cmd, cmd->cmdId.serialNumber);
		cmd->status.host = VMK_SCSI_HOST_BUS_BUSY;
		cmd->status.device = VMK_SCSI_DEVICE_GOOD;
		cmd->bytesXferred = 0;
		cmd->done(cmd);
		return VMK_OK;
	}

	vmk_SpinlockLock(sess->lock);
	if (sess->recovery_state) {
		 if (sess->recovery_state & ISCSI_SESS_RECOVERY_START) {
			vmk_SpinlockUnlock(sess->lock);
			goto cmd_not_accepted;
		} else {
			vmk_SpinlockUnlock(sess->lock);
			goto dev_not_found;
		}
	}

	conn = sess->lead_conn;
	if (!conn ||
	    !conn->ep ||
	    conn->ep->state != EP_STATE_ULP_UPDATE_COMPL) {
		vmk_SpinlockUnlock(sess->lock);
		goto dev_not_found;
	}

	scsi_task = qfle3i_alloc_scsi_task(sess);
	if (!scsi_task) {
		sess->alloc_scsi_task_failed++;
		vmk_SpinlockUnlock(sess->lock);
		goto cmd_not_accepted;
	}

	scsi_task->scsi_cmd = cmd;
	scsi_task->ilun = iscsilun;

	vmk_ListInsert(&scsi_task->link, vmk_ListAtRear(&sess->pend_cmd_list));
	sess->pend_cmd_count++;

	if (sess->hba->max_scsi_task_queued < sess->pend_cmd_count) {
		sess->hba->max_scsi_task_queued = sess->pend_cmd_count;
	}
	sess->total_cmds_queued++;

	vmk_SpinlockUnlock(sess->lock);
	if (vmk_AtomicRead64(&conn->worker_enabled)) {
		vmk_AtomicWrite64(&conn->lastSched,12);
		QFLE3I_DBG(DBG_IO, conn->sess->hba,
				  "scheduling conn tasklet, iscsi_conn_cid:0x%x, "
				  "scsi_cmd:%p lun:%d W:0x%x cmdID:0x%lx I:%p pendcc:%d\n",
				  conn->iscsi_conn_cid, cmd, iscsilun->lun_id,
				  cmd->worldId, cmd->cmdId.serialNumber,
				  cmd->cmdId.initiator, sess->pend_cmd_count);
		ql_vmk_tasklet_schedule(&conn->conn_tasklet);
	}
	return VMK_OK;

cmd_not_accepted:
	QFLE3I_DBG(DBG_IO, hba, "retun with 0x%x, cmd:%p cmdID:0x%lx\n",
					VMK_WOULD_BLOCK, cmd, cmd->cmdId.serialNumber);
	return VMK_WOULD_BLOCK;

dev_offline:
dev_not_found:

	cmd->status.host = VMK_SCSI_HOST_NO_CONNECT;
	cmd->bytesXferred = 0;
	QFLE3I_DBG(DBG_IO, hba, "complete cmd:%p cmdID:0x%lx with host_status:0x%x\n",
					cmd, cmd->cmdId.serialNumber, cmd->status.host);
	cmd->done(cmd);
	return VMK_OK;
}

/*
 * qfle3i_send_tmf_wait_cmpl - executes scsi command abort process
 *
 * @sc: 		SCSI-ML command pointer
 *
 * initiate command abort process by requesting CNIC to send
 *	an iSCSI TMF request to target
 */
static int qfle3i_send_tmf_wait_cmpl(struct qfle3i_sess *sess)
{
	int rc = 0;
	struct qfle3i_cmd *tmf_cmd = sess->scsi_tmf_cmd;
	struct qfle3i_conn *conn = sess->lead_conn;
	struct qfle3i_hba *hba = sess->hba;

	ADD_STATS_64(sess->hba, tx_pdus, 1);
	tmf_cmd->tmf_response = ISCSI_TMF_RSP_REJECTED;

	/* Schedule the tasklet to send out the TMF pdu */
	vmk_AtomicWrite64(&sess->tmf_pending, 1);
	if (vmk_AtomicRead64(&conn->worker_enabled)) {
		QFLE3I_DBG(DBG_TMF, hba, "sess %p ep_iscsi_cid:0x%x\n",
			  sess, conn->ep->ep_iscsi_cid);

		QFLE3I_DBG(DBG_TMF, conn->sess->hba,
				  "scheduling conn tasklet\n");
		ql_vmk_tasklet_schedule(&conn->conn_tasklet);
	}

#define QFLE3I_TMF_TIMEOUT	(60 * VMK_MSEC_PER_SEC)
	/* Now we wait here */
	rc = ql_vmk_wait_for_completion(&sess->er_wait,
					((sess->recovery_state != 0) ||
 					(vmk_AtomicRead64(&tmf_cmd->cmd_state) !=
  					ISCSI_CMD_STATE_INITIATED)),
					QFLE3I_TMF_TIMEOUT);

#if 0
	VK: find alternate in native
	if (signal_pending(current))
		flush_signals(current);
#endif
	if (rc == VMK_TIMEOUT) {
		/** TIMEOUT **/
		if (conn->ep)
			QFLE3I_DBG(DBG_TMF, hba, "TMF timed out for sess %p"
				  " ep_iscsi_cid:0x%x\n", sess, conn->ep->ep_iscsi_cid);
		vmk_AtomicWrite64(&tmf_cmd->cmd_state, ISCSI_CMD_STATE_TMF_TIMEOUT);
		vmk_AtomicWrite64(&sess->tmf_pending, 0);
		qfle3i_do_iscsi_sess_recovery(sess, VMK_SCSI_HOST_RESET, 1);
		return -1;
	}

	if (vmk_AtomicRead64(&sess->tmf_pending)) {
		QFLE3I_DBG(DBG_TMF, hba, "qfle3i: is tmf still pending\n");
		vmk_AtomicWrite64(&sess->tmf_pending, 0);
	}

	if (tmf_cmd->tmf_response == ISCSI_TMF_RSP_COMPLETE) {
		/* normal success case */
		return 0;
	} else if (tmf_cmd->tmf_response == ISCSI_TMF_RSP_NO_TASK) {
		if (tmf_cmd->tmf_ref_cmd->scsi_cmd == tmf_cmd->tmf_ref_sc) {
			if (vmk_AtomicRead64(&tmf_cmd->tmf_ref_cmd->cmd_state) == ISCSI_CMD_STATE_COMPLETED) {
				/* task completed while tmf request is pending, driver is
				 * holding on to the completion 
				 */
				return 0;
			} else {
				/* missing command, do session recovery */
				goto do_recovery;
			}
		} else {
			return 0; /* command already completed */
		}
	}

do_recovery:
	QFLE3I_DBG(DBG_TMF, hba, "tmf failed for sess %p cmd %p\n",
		    sess, tmf_cmd);
	qfle3i_do_iscsi_sess_recovery(sess, VMK_SCSI_HOST_RESET, 1);
	return -1;
}

/*
 * qfle3i_initiate_target_reset- executes scsi command target reset
 *
 * @sc: 		SCSI-ML command pointer
 *
 * initiate command abort process by requesting CNIC to send
 *	an iSCSI TMF request to target
 */
static int qfle3i_initiate_target_reset(struct qfle3i_sess *sess)
{
	struct qfle3i_cmd *tmf_cmd;
	struct qfle3i_hba *hba;

	hba = sess->hba;
	tmf_cmd = sess->scsi_tmf_cmd;
	vmk_AtomicWrite64(&sess->tmf_active, 1);
	tmf_cmd->conn = sess->lead_conn;
	tmf_cmd->scsi_cmd = NULL;
	tmf_cmd->iscsi_opcode = ISCSI_OP_SCSI_TMFUNC | ISCSI_OP_IMMEDIATE;
	tmf_cmd->tmf_func = ISCSI_TM_FUNC_TARGET_WARM_RESET;
	tmf_cmd->tmf_lun = 0;
	tmf_cmd->ilun = NULL;
	tmf_cmd->tmf_ref_itt = ISCSI_RESERVED_TAG;
	tmf_cmd->tmf_ref_cmd = NULL;
	tmf_cmd->tmf_ref_sc = NULL;
	vmk_AtomicWrite64(&tmf_cmd->cmd_state, ISCSI_CMD_STATE_INITIATED);

	QFLE3I_DBG(DBG_TMF, hba, "sess %p, conn %p\n",
		  sess, sess->lead_conn);
	return 0;
}

/*
 * qfle3i_initiate_lun_reset- executes scsi command abort process
 *
 * @sc: 		SCSI-ML command pointer
 *
 * initiate command abort process by requesting CNIC to send
 *	an iSCSI TMF request to target
 */
static int qfle3i_initiate_lun_reset(qfle3i_iscsilun *ilun,
				    vmk_ScsiTaskMgmt *taskMgmt)
{
	struct qfle3i_cmd *tmf_cmd;
	struct qfle3i_hba *hba;
	qfle3i_sess *sess = ilun->sess;

	if (!sess) {
		vmk_LogMessage("%s:%d TMF session=NULL\n", __func__, __LINE__);
		return VMK_FAILURE;
	}

	hba = sess->hba;
	tmf_cmd = sess->scsi_tmf_cmd;
	vmk_AtomicWrite64(&sess->tmf_active, 1);
	tmf_cmd->conn = sess->lead_conn;
	tmf_cmd->scsi_cmd = NULL;
	tmf_cmd->iscsi_opcode = ISCSI_OP_SCSI_TMFUNC | ISCSI_OP_IMMEDIATE;
	tmf_cmd->tmf_func = ISCSI_TM_FUNC_LOGICAL_UNIT_RESET;
	tmf_cmd->ilun = ilun;
	tmf_cmd->tmf_lun = ilun->lun_id;
	tmf_cmd->tmf_ref_itt = ISCSI_RESERVED_TAG;
	tmf_cmd->tmf_ref_cmd = NULL;
	tmf_cmd->tmf_ref_sc = NULL;
	vmk_AtomicWrite64(&tmf_cmd->cmd_state, ISCSI_CMD_STATE_INITIATED);
	return 0;
}


/*
 * qfle3i_initiate_abort_cmd - executes scsi command abort process
 *
 * @sc: 		SCSI-ML command pointer
 *
 * initiate command abort process by requesting CNIC to send
 *	an iSCSI TMF request to target. Called with sess->lock held
 */
static int qfle3i_initiate_abort_cmd(struct qfle3i_sess *sess, struct qfle3i_cmd *cmd)
{
	vmk_ScsiCommand *sc = NULL;
	struct qfle3i_cmd *tmf_cmd;
	struct qfle3i_hba *hba;

	hba = sess->hba;

	if (!cmd || !cmd->scsi_cmd) {
		/* command already completed to scsi mid-layer */
		QFLE3I_DBG(DBG_TMF, hba,
			  "sess %p %x:%x:%x cmd no longer active.\n",
			  sess, sess->channel_id, sess->target_id,
			  cmd->ilun->lun_id);
		return VMK_NOT_FOUND;
	}

	sc = cmd->scsi_cmd;
	tmf_cmd = sess->scsi_tmf_cmd;
	vmk_AtomicWrite64(&sess->tmf_active, 1);
	if (vmk_AtomicRead64(&cmd->cmd_state) == ISCSI_CMD_STATE_ABORT_REQ)
		vmk_AtomicWrite64(&cmd->cmd_state, ISCSI_CMD_STATE_ABORT_PEND);
	vmk_AtomicWrite64(&tmf_cmd->cmd_state, ISCSI_CMD_STATE_INITIATED);
	tmf_cmd->conn = sess->lead_conn;
	tmf_cmd->scsi_cmd = NULL;
	tmf_cmd->iscsi_opcode = ISCSI_OP_SCSI_TMFUNC | ISCSI_OP_IMMEDIATE;
	tmf_cmd->tmf_func = ISCSI_TM_FUNC_ABORT_TASK;
	tmf_cmd->ilun = cmd->ilun;
	tmf_cmd->tmf_lun = cmd->ilun->lun_id;
	tmf_cmd->tmf_ref_itt = cmd->req.itt;
	tmf_cmd->tmf_ref_cmd = cmd;
	tmf_cmd->tmf_ref_sc = cmd->scsi_cmd;
	QFLE3I_DBG(DBG_TMF, hba, "[%lx] sess %p sc %p aborting active cmd "
		  "ref_cmd %p ref_scsi_cmd %p\n", vmk_GetTimerCycles(),
		  sess, sc, tmf_cmd->tmf_ref_cmd, tmf_cmd->tmf_ref_sc);

	return VMK_EINPROGRESS;
}


static void qfle3i_tmf_wait_dev_offlined(struct qfle3i_sess *sess)
{
	int rc = 0;

#define QFLE3I_WAIT_OFFLINE_TIMEOUT	(4 * VMK_MSEC_PER_SEC)
	/* Either 'conn' object can get removed (session logged out) or
	 * 'ep' alone is destroyed (session in error recovery), driver need
	 * to handle both conditions.
	 */
	/* Confirm wait condition one more time */
	rc = ql_vmk_wait_for_completion(
					&sess->er_wait,
					(vmk_AtomicRead64(&sess->device_offline) ||
					(sess->lead_conn ? !sess->lead_conn->ep : 1)),
					QFLE3I_WAIT_OFFLINE_TIMEOUT);

	if (rc == VMK_TIMEOUT) {
		QFLE3I_DBG(DBG_TMF, sess->hba, "Timeout, sess:%p\n", sess);
	}
	else {
		QFLE3I_DBG(DBG_TMF, sess->hba, "No Timeout, sess:%p\n", sess);

	}
}

/*
 * qfle3i_execute_tmf_cmd - executes scsi tmf
 *
 * @sc: 		SCSI-ML command pointer
 *
 * initiate scsi tmf, support ABORT_TASK and LUN_RESET
 */
static int qfle3i_execute_tmf_cmd(qfle3i_iscsilun *ilun,
				vmk_ScsiTaskMgmt *taskMgmt, int tmf_func,
				qfle3i_cmd *reset_cmd)
{
	int active_cmds_failed = 0;
	int pend_cmds_failed = 0;
	struct qfle3i_hba *hba = NULL;
	struct qfle3i_cmd *cmd = NULL;
	int rc = VMK_FAILURE;
	int wait_rc = 0;
	qfle3i_sess *sess = ilun->sess;

	if (!sess) {
		vmk_LogMessage("%s:%d TMF session=NULL\n", __func__, __LINE__);
		return VMK_FAILURE;
	}
	hba = sess->hba;

	QFLE3I_DBG(DBG_TMF, hba, "start, sess %p scsi_cmdId:0x%lx tmf_func %d\n",
		  sess, taskMgmt->cmdId.serialNumber, tmf_func);

	vmk_SemaLock(&sess->tmf_mutex);

	if (vmk_BitVectorTest(sess->hba->adapter_state, ADAPTER_STATE_GOING_DOWN) ||
	    vmk_BitVectorTest(sess->hba->adapter_state, ADAPTER_STATE_LINK_DOWN)) {
		QFLE3I_DBG(DBG_TMF, hba, "sess %p scsi_cmdID:0x%lx\n",
			  sess, taskMgmt->cmdId.serialNumber);
		vmk_SemaUnlock(&sess->tmf_mutex);
		return VMK_FAILURE;
	}

	vmk_SpinlockLock(sess->lock);
	if (sess->recovery_state || !is_sess_active(sess)) {
		QFLE3I_DBG(DBG_TMF, hba, "sess %p, scsi_cmdID:0x%lx not active.\n",
			  sess, taskMgmt->cmdId.serialNumber);
		/* better to wait till device is offlined to avoid ABORT storm
		 */
		vmk_SpinlockUnlock(sess->lock);
		qfle3i_tmf_wait_dev_offlined(sess);
		vmk_SemaUnlock(&sess->tmf_mutex);
		return VMK_FAILURE;
	}

	vmk_AtomicWrite64(&sess->tmf_active, 1);

	if (tmf_func == ISCSI_TM_FUNC_ABORT_TASK) {
		cmd = qfle3i_scsi_cmd_in_active_list(sess, taskMgmt->cmdId,
										taskMgmt->worldId, ilun->lun_id);
		if (VMK_UNLIKELY(!cmd) || (cmd->ilun->lun_id != ilun->lun_id)) {
			/* command already completed to scsi mid-layer */
			PRINT_ERR(hba,
					"sess %p %x:%x:%x scsi_cmdID:0x%lx cmd no longer active.\n",
					sess, sess->channel_id, sess->target_id, ilun->lun_id,
					taskMgmt->cmdId.serialNumber);
			rc = VMK_OK;
			vmk_SpinlockUnlock(sess->lock);
			goto done;
		}

		rc = qfle3i_initiate_abort_cmd(sess, cmd);
		if (rc == VMK_NOT_FOUND) {
			/* cmd not active */
			rc = VMK_OK;
			vmk_SpinlockUnlock(sess->lock);
			goto done;
		}
		rc = VMK_OK;
	} else 	if (tmf_func == ISCSI_TM_FUNC_VIRT_RESET) {
		cmd = reset_cmd;
		rc = qfle3i_initiate_abort_cmd(sess, cmd);
		if (rc == VMK_NOT_FOUND) {
			/* cmd not active */
			rc = VMK_OK;
			vmk_SpinlockUnlock(sess->lock);
			goto done;
		}
		rc = VMK_OK;
	}  else if (tmf_func == ISCSI_TM_FUNC_LOGICAL_UNIT_RESET) {
		qfle3i_initiate_lun_reset(ilun, taskMgmt);
	} else if (tmf_func == ISCSI_TM_FUNC_TARGET_WARM_RESET) {
		qfle3i_initiate_target_reset(sess);
	} else {
		PRINT_ALERT(sess->hba, "unknown Task Mgmt Command:%x\n",
		       tmf_func);
		rc = VMK_FAILURE;
		vmk_SpinlockUnlock(sess->lock);
		goto done;
	}
	vmk_SpinlockUnlock(sess->lock);

	QFLE3I_DBG(DBG_TMF, hba, "tmf wait......., sess %p scsi_cmdID:0x%lx\n",
					sess, taskMgmt->cmdId.serialNumber);

	if (qfle3i_send_tmf_wait_cmpl(sess)) {
		/* TMF request timeout */
		rc = VMK_FAILURE;
		goto done;
	}

	sess->cmd_cleanup_req = 0;
	sess->cmd_cleanup_cmpl = 0;

	if (sess->scsi_tmf_cmd->tmf_response == ISCSI_TMF_RSP_COMPLETE) {
		if ((tmf_func == ISCSI_TM_FUNC_ABORT_TASK) ||
			(tmf_func == ISCSI_TM_FUNC_VIRT_RESET)) {
			if (cmd->scsi_status_rcvd) {
				/* cmd completed while TMF was active.
				 * Now it's safe to complete command
				 * to SCSI-ML
				 */
				qfle3i_complete_cmd(sess, cmd);
			} else {
				vmk_SpinlockLock(sess->lock);
				qfle3i_cleanup_task_context(sess, cmd, VMK_SCSI_HOST_ABORT);
				vmk_SpinlockUnlock(sess->lock);
			}
			active_cmds_failed = 1;
		} else if (tmf_func == ISCSI_TM_FUNC_LOGICAL_UNIT_RESET) {
			/* Pend queue is already flushed before issuing send TMF
			 * request on wire. This is just a redundant flush which
			 * should do allow us to detect any command queued while
			 * TMF is active
			 */
			pend_cmds_failed = qfle3i_flush_pend_queue(sess, ilun, VMK_SCSI_HOST_RESET);
			active_cmds_failed = qfle3i_flush_cmd_queue(sess, ilun, VMK_SCSI_HOST_RESET, 1);

		} else if (tmf_func == ISCSI_TM_FUNC_TARGET_WARM_RESET) {
			/* pend queue- Same comments as LUN RESET holds good here */
			pend_cmds_failed = qfle3i_flush_pend_queue(sess, NULL, VMK_SCSI_HOST_RESET);
			active_cmds_failed = qfle3i_flush_cmd_queue(sess, NULL, VMK_SCSI_HOST_RESET, 1);
		}
		rc = VMK_OK;
	} else if ((sess->scsi_tmf_cmd->tmf_response == ISCSI_TMF_RSP_NO_TASK) &&
		   ((tmf_func == ISCSI_TM_FUNC_ABORT_TASK) || 
		   (tmf_func == ISCSI_TM_FUNC_VIRT_RESET))) {
		if (!cmd->scsi_cmd ||
		    (cmd->scsi_cmd != sess->scsi_tmf_cmd->tmf_ref_sc)) {
			/* command already completed, later case cmd is being
			 * reused for a different I/O
			 */
			rc = VMK_OK;
		} else if (cmd->scsi_status_rcvd) {
			/* cmd completed while TMF was active. Now it's safe
			 * to complete the command back to SCSI-ML
			 */
			QFLE3I_DBG(DBG_TMF, hba, "cmd is completed\n");
			qfle3i_complete_cmd(sess, cmd);
			rc = VMK_OK;
		} else {
			/* we should never step into this code path as missing command
			 * will trigger session recovery in  qfle3i_send_tmf_wait_cmpl()
			 */
			PRINT(sess->hba, "%s: TMF_ABORT completed with NO_TASK,"
			      " but ITT %x is pending.\n", /*sess->hba->netdev->name*/
			      "netdev", sess->scsi_tmf_cmd->tmf_ref_itt);
			rc = VMK_FAILURE;
		}
	} else {
		rc = VMK_FAILURE;
	}

	wait_rc = ql_vmk_wait_for_completion(&sess->er_wait,
					(!is_sess_active(sess) ||
					(sess->cmd_cleanup_req == sess->cmd_cleanup_cmpl)),
					30 * VMK_MSEC_PER_SEC);

	if (!is_sess_active(sess)) {
		/* session went into recovery due to protocol error, there won't
		 * be any CQ completions, active command cleanup will continue
		 * in ep_disconnect()
		 */
		QFLE3I_DBG(DBG_TMF, hba, "sess %p in recovery\n", sess);
		rc = VMK_FAILURE;
	} else if (wait_rc == VMK_TIMEOUT) {
		QFLE3I_DBG(DBG_TMF, hba,
			  "cleanup did not complete in 30 seconds\n");
		/* If TCP layer is working fine, CMD_CLEANUP should complete
		 * 'Cuz all CMD before TMF REQ would have been TCP ACK'ed.
		 * If there is a problem with the TCP layer, TMF request should
		 * have timed out triggering session recovery
		 */
#if 0
		qfle3i_print_cqe(sess->lead_conn);
		qfle3i_print_sqe(sess->lead_conn);
#endif
		PRINT(hba, "%s: sess %p - ITT cleanup request timed out\n",
		      /*hba->netdev->name*/ "netdev", sess);
		/* Force session recovery */
		qfle3i_do_iscsi_sess_recovery(sess, VMK_SCSI_HOST_RESET, 1);
		rc = VMK_FAILURE;
	}

#if 0
	if (signal_pending(current))
		flush_signals(current);
#endif

	QFLE3I_DBG(DBG_TMF, hba, "sess %p async cmd cleanup. req %d, comp %d\n",
		  sess, sess->cmd_cleanup_req, sess->cmd_cleanup_cmpl);

done:
	QFLE3I_DBG(DBG_TMF, hba, "sess %p cmds stats, AC=%d, PD=%d,"
		  " Q=%d, S=%d, D=%d, F=%d, CC=%d\n", sess,
		  active_cmds_failed, pend_cmds_failed,
		  sess->total_cmds_queued, sess->total_cmds_sent,
		  sess->total_cmds_completed, sess->total_cmds_failed,
		  sess->total_cmds_completed_by_chip);

	vmk_CPUMemFenceReadWrite();
	vmk_AtomicWrite64(&sess->tmf_active, 0);
	vmk_SemaUnlock(&sess->tmf_mutex);

	return rc;
}

static void qfle3i_wait_for_tmf_completion(struct qfle3i_sess *sess)
{
	int lpcnt = 200;

	while (lpcnt-- && vmk_AtomicRead64(&sess->tmf_active))
		vmk_WorldSleep(100*VMK_USEC_PER_MSEC);
}

/*
 * qfle3i_eh_device_reset - 'eh_device_reset_handler' entry point
 *
 * @sc: 		SCSI-ML command pointer
 *
 * SCSI host reset handler - iSCSI session recovery
 */
VMK_ReturnStatus qfle3i_eh_device_reset(qfle3i_iscsilun *ilun,
					vmk_ScsiTaskMgmt *taskMgmt)
{
	int rc = 0;
	struct qfle3i_sess *sess = ilun->sess;
	int tmf_func;

	if (!sess || !sess->lead_conn || !sess->lead_conn->ep ||
	    vmk_AtomicRead64(&sess->lead_conn->stop_state))
		return VMK_FAILURE;

	QFLE3I_DBG(DBG_TMF, sess->hba, "device/lun reset, ep_iscsi_cid:0x%x, lun %x\n",
					sess->lead_conn->ep->ep_iscsi_cid, ilun->lun_id);

	/* We do send tmf_func value to firmware, new VMK values will
	   be a problem in that case, mapping it to old legacy values,
	   so fw can handle these requests properly.
	*/
	if (taskMgmt->type == VMK_SCSI_TASKMGMT_LUN_RESET)
		tmf_func = ISCSI_TM_FUNC_LOGICAL_UNIT_RESET;
	else
		tmf_func = ISCSI_TM_FUNC_TARGET_WARM_RESET;

	rc = qfle3i_execute_tmf_cmd(ilun, taskMgmt, tmf_func, NULL);
	return rc;
}
struct qfle3i_cmd *get_active_cmd_for_virt_reset(
					qfle3i_sess *sess,
					vmk_ScsiTaskMgmt *taskMgmt)
{
	struct qfle3i_cmd *cmd, *tmp_cmd;
	struct qfle3i_hba *hba = sess->hba;

	ql_vmk_list_each_entry_safe(cmd, tmp_cmd, &sess->active_cmd_list,
							link, struct qfle3i_cmd) {
		if ((vmk_AtomicRead64(&cmd->cmd_state) != ISCSI_CMD_STATE_INITIATED) ||
			!cmd->conn->ep)
			continue;

		if ((cmd->scsi_cmd->worldId == taskMgmt->worldId) &&
			((taskMgmt->cmdId.initiator == VMK_SCSI_TASKMGMT_ANY_INITIATOR) ||
			(cmd->scsi_cmd->cmdId.initiator == taskMgmt->cmdId.initiator)))
			return cmd;
	}
	return NULL;
}


#define CMD_COMPLETED	VMK_OK
static int virtual_reset_grace_time(qfle3i_cmd *cmd)
{
	int lpcnt = 1000;
	while (lpcnt &&
			(vmk_AtomicRead64(&cmd->cmd_state) != ISCSI_CMD_STATE_COMPLETED)) {
		lpcnt--;
		vmk_WorldSleep(10 * VMK_USEC_PER_MSEC);
	}

	if (lpcnt)
		return CMD_COMPLETED;
	else
		return !CMD_COMPLETED;
}


VMK_ReturnStatus qfle3i_eh_virtual_reset(
					qfle3i_iscsilun *ilun,
					vmk_ScsiTaskMgmt *taskMgmt)
{
	VMK_ReturnStatus reason = VMK_OK;
	vmk_ListLinks failed_cmds;
	struct qfle3i_cmd *cmd, *tmp_cmd;
	struct qfle3i_scsi_task *scsi_task, *tmp_task;
	struct qfle3i_sess *sess = ilun->sess;
	struct qfle3i_hba *hba = sess->hba;
	qfle3i_iscsilun *lilun;

	vmk_ListInit(&failed_cmds);

	vmk_SpinlockLock(sess->lock);
	ql_vmk_list_each_entry_safe(scsi_task, tmp_task, &sess->pend_cmd_list,
							link, struct qfle3i_scsi_task) {
		if ((scsi_task->scsi_cmd->worldId == taskMgmt->worldId) &&
			((taskMgmt->cmdId.initiator == VMK_SCSI_TASKMGMT_ANY_INITIATOR) ||
			(scsi_task->scsi_cmd->cmdId.initiator == taskMgmt->cmdId.initiator))) {

			scsi_task->ilun->tmf_in_progress = 1;
			sess->pend_cmd_count--;

			vmk_ListRemove(&scsi_task->link);
			vmk_ListInit(&scsi_task->link);
			vmk_ListInsert(&scsi_task->link, vmk_ListAtRear(&failed_cmds));
		}
	}
	vmk_SpinlockUnlock(sess->lock);

	ql_vmk_list_each_entry_safe(scsi_task, tmp_task, &failed_cmds,
							link, struct qfle3i_scsi_task) {
		lilun = scsi_task->ilun;
		lilun->tmf_in_progress = 1;
		scsi_task->scsi_cmd->status.host = VMK_SCSI_HOST_ABORT;
		qfle3i_return_failed_command(sess, scsi_task->scsi_cmd,
				vmk_SgGetDataLen(scsi_task->scsi_cmd->sgArray),
				VMK_SCSI_HOST_ABORT);

		QFLE3I_DBG(DBG_TMF, hba,
				"sess %p %x:%x:%x scsi_cmdID:0x%lx worldID:0x%x sc:%p "
				"aborted from pending queue.\n", sess,
				sess->channel_id, sess->target_id,
				lilun->lun_id, taskMgmt->cmdId.serialNumber,
				scsi_task->scsi_cmd->worldId, scsi_task->scsi_cmd);

		vmk_SpinlockLock(sess->lock);
		qfle3i_free_scsi_task(sess, scsi_task);
		vmk_SpinlockUnlock(sess->lock);
		lilun->tmf_in_progress = 0;
	}

	/** It wasn't in the pending queue... and it still has no cmd object
	 * it must have completed out.
	 */
	vmk_SpinlockLock(sess->lock);
retry:
	cmd = get_active_cmd_for_virt_reset(sess, taskMgmt);
	if (!cmd || !cmd->scsi_cmd) {
		/* active command list loop is not good, debug this */
		PRINT_INFO(sess->hba, "No cmds left for virtual_reset. \n");
		goto loop_exit;
	}

	if (sess->recovery_state || !is_sess_active(sess) ||
		vmk_AtomicRead64(&sess->lead_conn->stop_state)) {
		PRINT_INFO(sess->hba,
			  "sess:%p, sc:%p scsi_cmdID:0x%lx worldID:0x%x not active.\n",
			  sess, cmd->scsi_cmd, taskMgmt->cmdId.serialNumber, cmd->scsi_cmd->worldId);
		reason = VMK_FAILURE;
		goto loop_exit;
	}

	lilun = cmd->ilun;
	lilun->tmf_in_progress = 1;

	/* Set cmd_state so that command will not be completed to SCSI-ML
	 * if SCSI_RESP is rcvd for this command
	 */
	vmk_AtomicWrite64(&cmd->cmd_state, ISCSI_CMD_STATE_ABORT_REQ);
	QFLE3I_DBG(DBG_TMF, hba,
			"sess %p %x:%x:%x scsi_cmdID:0x%lx worldID:0x%x sc:%p abort task.\n",
			sess, sess->channel_id, sess->target_id,
			lilun->lun_id, taskMgmt->cmdId.serialNumber,
			cmd->scsi_cmd->worldId, cmd->scsi_cmd);

	vmk_SpinlockUnlock(sess->lock);

	/* Give some grace time to command to complete */
	reason = virtual_reset_grace_time(cmd);
	if (reason == CMD_COMPLETED)
		qfle3i_complete_cmd(sess, cmd);
	else
		reason = qfle3i_execute_tmf_cmd(lilun, taskMgmt, ISCSI_TM_FUNC_VIRT_RESET, cmd);

	lilun->tmf_in_progress = 0;
	vmk_SpinlockLock(sess->lock);
	goto retry;
loop_exit:
	vmk_SpinlockUnlock(sess->lock);
	return reason;
}

static VMK_ReturnStatus qfle3i_eh_abort(
					qfle3i_iscsilun *ilun,
					vmk_ScsiTaskMgmt *taskMgmt)
{
	int reason;
	struct qfle3i_cmd *cmd;
	struct qfle3i_scsi_task *scsi_task;
	struct qfle3i_sess *sess = ilun->sess;
	struct qfle3i_hba *hba = sess->hba;


	/*
	 * we can ALWAYS abort from the pending queue
	 * since it has not made it to the chip yet
	 * NOTE: the queue has to be protected via spin lock
	 */
	vmk_SpinlockLock(sess->lock);

	scsi_task = qfle3i_scsi_cmd_in_pend_list(sess, taskMgmt->cmdId,
										taskMgmt->worldId, ilun->lun_id);
	if (scsi_task) {
		scsi_task->scsi_cmd->status.host = VMK_SCSI_HOST_ABORT;
		sess->pend_cmd_count--;

		qfle3i_return_failed_command(sess, scsi_task->scsi_cmd,
						vmk_SgGetDataLen(scsi_task->scsi_cmd->sgArray),
						VMK_SCSI_HOST_ABORT);

		qfle3i_free_scsi_task(sess, scsi_task);
		vmk_SpinlockUnlock(sess->lock);

		QFLE3I_DBG(DBG_TMF, hba,
			"sess %p %x:%x:%x scsi_cmdID:0x%lx sc %p aborted from "
			"pending queue.\n", sess,
			sess->channel_id, sess->target_id,
			ilun->lun_id, taskMgmt->cmdId.serialNumber,
			scsi_task->scsi_cmd);
		return VMK_OK;
	}

	/** It wasn't in the pending queue... and it still has no cmd object
	 * it must have completed out.
	 */
	cmd = qfle3i_scsi_cmd_in_active_list(sess, taskMgmt->cmdId,
										taskMgmt->worldId, ilun->lun_id);
	if (VMK_UNLIKELY(!cmd) || (cmd->ilun->lun_id != ilun->lun_id)) {
		/* command already completed to scsi mid-layer */
		QFLE3I_DBG(DBG_TMF, sess->hba,
			  "sess %p %x:%x:%x scsi_cmdID:0x%lx cmd no longer active.\n",
			  sess, sess->channel_id, sess->target_id, ilun->lun_id,
			  taskMgmt->cmdId.serialNumber);
		vmk_SpinlockUnlock(sess->lock);
		return VMK_OK;
	}

	if ((vmk_AtomicRead64(&cmd->cmd_state) != ISCSI_CMD_STATE_INITIATED) ||
	    !cmd->conn->ep) {
		/* Command completion is being processed, fail the abort request
		 * Second condition should never be true unless SCSI layer is
		 * out of sync
		 */
		vmk_SpinlockUnlock(sess->lock);
		return VMK_FAILURE;
	}
	/* Set cmd_state so that command will not be completed to SCSI-ML
	 * if SCSI_RESP is rcvd for this command
	 */
	vmk_AtomicWrite64(&cmd->cmd_state, ISCSI_CMD_STATE_ABORT_REQ);

	QFLE3I_DBG(DBG_TMF, hba,
			"sess %p %x:%x:%x scsi_cmdID:0x%lx sc %p abort task.\n",
			sess, sess->channel_id, sess->target_id,
			ilun->lun_id, taskMgmt->cmdId.serialNumber, cmd->scsi_cmd);

	vmk_SpinlockUnlock(sess->lock);

	/* We do send tmf_func value to firmware, new VMK values will
	   be a problem in that case, mapping it to old legacy values,
	   so fw can handle these requests properly.
	*/
	reason = qfle3i_execute_tmf_cmd(ilun, taskMgmt, ISCSI_TM_FUNC_ABORT_TASK, NULL);
	return reason;
}

static void set_tmf_progress_for_all_luns(qfle3i_sess *sess)
{
	struct qfle3i_iscsilun *iscsilun = NULL;

	vmk_SpinlockLock(sess->iscsilun_lock);

	ql_vmk_list_each_entry(iscsilun, &sess->iscsilun_list, link, qfle3i_iscsilun) {
		if (iscsilun)
			iscsilun->tmf_in_progress = 1;
	}
	vmk_SpinlockUnlock(sess->iscsilun_lock);
}

static void reset_tmf_progress_for_all_luns(qfle3i_sess *sess)
{
	struct qfle3i_iscsilun *iscsilun = NULL;

	vmk_SpinlockLock(sess->iscsilun_lock);

	ql_vmk_list_each_entry(iscsilun, &sess->iscsilun_list, link, qfle3i_iscsilun) {
		if (iscsilun)
			iscsilun->tmf_in_progress = 0;
	}
	vmk_SpinlockUnlock(sess->iscsilun_lock);
}

static VMK_ReturnStatus
qfle3i_scsiTaskMgmt(void *client_data, vmk_ScsiTaskMgmt *taskMgmt,
   void *device_data)
{
	VMK_ReturnStatus status = VMK_FAILURE;
	struct qfle3i_hba *hba = (struct qfle3i_hba *)client_data;
	struct qfle3i_iscsilun *ilun = (qfle3i_iscsilun *)device_data;
	struct qfle3i_sess *sess = NULL;

	if (!client_data) {
		PRINT_ERR(hba, "Error: NULL client_data.\n");
		return VMK_BAD_PARAM;
	}

	if (!device_data) {
		PRINT_ERR(hba, "Error: NULL device_data.\n");
		return VMK_BAD_PARAM;
	}

	if (!taskMgmt){
		PRINT_ERR(hba, "Error: NULL taskMgmt.\n");
		return VMK_BAD_PARAM;
	}

	if (!ilun) {
		PRINT_ERR(hba, "Error: NULL device_data.\n");
		return VMK_BAD_PARAM;
	}
	sess = ilun->sess;
	if (!sess || !sess->lead_conn || !sess->lead_conn->ep ||
	    vmk_AtomicRead64(&sess->lead_conn->stop_state)) {
		PRINT_ERR(hba, "Error: sess/conn not present or stop-conn in progress.\n");
		return VMK_OK;
	}

	/* let IO requests for this specific LUN return with busy status */
	ilun->tmf_in_progress = 1;
	switch (taskMgmt->type) {
		case VMK_SCSI_TASKMGMT_ABORT:
			QFLE3I_DBG(DBG_TMF, hba, "TaskMgmt abort on serial num:0x%lx\n",
				taskMgmt->cmdId.serialNumber);
			status = qfle3i_eh_abort(ilun, taskMgmt);
			break;
		case VMK_SCSI_TASKMGMT_LUN_RESET:
			QFLE3I_DBG(DBG_TMF, hba, "TaskMgmt lun reset.\n");
			status = qfle3i_eh_device_reset(ilun, taskMgmt);
			break;
		case VMK_SCSI_TASKMGMT_DEVICE_RESET:
			QFLE3I_DBG(DBG_TMF, hba, "Task Mgmt device/target reset.\n");
			set_tmf_progress_for_all_luns(sess);
			status = qfle3i_eh_device_reset(ilun, taskMgmt);
			reset_tmf_progress_for_all_luns(sess);
			break;
		case VMK_SCSI_TASKMGMT_BUS_RESET:
			QFLE3I_DBG(DBG_TMF, hba, "TaskMgmt bus reset, not supported.\n");
			status = VMK_NOT_SUPPORTED;
			break;
		case VMK_SCSI_TASKMGMT_VIRT_RESET:
			QFLE3I_DBG(DBG_TMF, hba, "TaskMgmt virt reset, W:0x%x CmdID:0x%lx I:%p\n",
							taskMgmt->worldId, taskMgmt->cmdId.serialNumber,
							taskMgmt->cmdId.initiator);
			status = qfle3i_eh_virtual_reset(ilun, taskMgmt);
			break;
		default:
			PRINT_ERR(hba, "Unknown task management type:0x%x\n",
				taskMgmt->type);
			return VMK_OK;
	}

	ilun->tmf_in_progress = 0;
	return status;
}

static VMK_ReturnStatus
qfle3i_scsiDumpCommand(qfle3i_hba *adapter, vmk_ScsiCommand *vmkCmd,
   void *deviceData)
{
   return VMK_NOT_SUPPORTED;
}

static VMK_ReturnStatus
qfle3i_procInfo(void *clientData,
   char *buf, vmk_ByteCountSmall offset,
   vmk_ByteCountSmall count, vmk_ByteCountSmall * nbytes, int isWrite)
{
  DRV_DEBUG(0, "qfle3i: proc info is not supported.\n");
  return VMK_NOT_SUPPORTED;
}

static void
qfle3i_scsiDumpQueue(qfle3i_hba *adapter)
{

}

static void
qfle3i_scsiDumpPollHandler(qfle3i_hba *adapter)
{

}

static VMK_ReturnStatus
qfle3i_ioctl(void *client_data, void *deviceData,
   vmk_uint32 fileFlags, vmk_uint32 cmd, vmk_VA userArgsPtr,
   vmk_IoctlCallerSize callerSize, vmk_int32 *drvErr)
{
  DRV_DEBUG(0, "qfle3i: ioctl is not supported.\n");
  return VMK_NOT_SUPPORTED;
}

static VMK_ReturnStatus qfle3i_discover(
				void *client_data, vmk_ScanAction action,
   				int channel_id, int target_id,
				int lun_id, void **device_data)
{
	VMK_ReturnStatus status = VMK_OK;
	struct qfle3i_hba *hba;
	struct qfle3i_sess *sess = NULL;
	struct qfle3i_iscsilun *iscsilun = NULL;

	hba = (qfle3i_hba *)client_data;
	if (!hba) {
		vmk_LogMessage("invalid param\n");
		return VMK_BAD_PARAM;
	}

	switch (action) {
		case VMK_SCSI_SCAN_CREATE_PATH:

			/* get the session associated with these IDs */
			sess = qfle3i_get_session(hba, channel_id, target_id);
			if (!sess) {
				return VMK_NO_CONNECT;
			}

			if (sess->state != QFLE3I_SESS_IN_FFP) {
				PRINT_ERR(hba, "iscsi session:%p  not in FFP.\n", sess);
				return VMK_NO_CONNECT;
			}

			QFLE3I_DBG(DBG_CONN_SETUP, hba, "Action:%d, H:%d, C:%d, T:%d, L:%d\n",
						action, hba->host_num, channel_id, target_id, lun_id);

			/* check if lun already exists */
			iscsilun = qfle3i_get_iscsilun(sess, lun_id);
			if (iscsilun) {
				PRINT_INFO(hba, "Path exists %s:%d:%d:%d\n",
					hba->scsiAdapter->name.string, channel_id,
					target_id, lun_id);
				return VMK_EXISTS;
			}

			iscsilun = qfle3i_allocate_lun(sess, channel_id,
										target_id, lun_id);
			break;

		case VMK_SCSI_SCAN_CONFIGURE_PATH:
			return status;

		case VMK_SCSI_SCAN_DESTROY_PATH:
			QFLE3I_DBG(DBG_CONN_SETUP, hba, "Action:%d, H:%d, C:%d, T:%d, L:%d\n",
						action, hba->host_num, channel_id, target_id, lun_id);

			sess = qfle3i_get_session(hba, channel_id, target_id);
			if (!sess) {
				PRINT_ERR(hba, "qfle3i_get_session failed.\n");
				return VMK_OK;
			}

			/* check if lun already exists */
			iscsilun = qfle3i_get_iscsilun(sess, lun_id);
			if (iscsilun) {
				qfle3i_free_iscsilun(iscsilun);
				iscsilun = NULL;
			} else {
				PRINT_ERR(hba, "No lun with id:%d exists.\n", lun_id);
				return VMK_OK;
			}

			if (sess->state == QFLE3I_SESS_DESTROYED) {
				if (vmk_ListIsEmpty(&sess->iscsilun_list))
					qfle3i_iscsi_sess_release(sess);
			}
			break;
		default:
			return VMK_NOT_IMPLEMENTED;
   }
	*device_data = (void *)iscsilun;
	PRINT_INFO(hba, "device_data ptr to mid-layer:%p\n", *device_data);
	return status;
}

static int
qfle3i_modifyQueueDepth(void *clientData, int qdepth, void *deviceData)
{
	return 0;
}

static int
qfle3i_query_qdepth(void *client_data, void *device_data)
{
		qfle3i_iscsilun *iscsilun = (qfle3i_iscsilun *)device_data;
		return iscsilun->q_depth;
}


static VMK_ReturnStatus
qfle3i_checkTarget(void *clientData, int channel, int target_id)
{
   VMK_ReturnStatus status = VMK_OK;

   return status;
}

VMK_ReturnStatus qfle3i_set_rdp_info(vmk_uint64 cookie, void *stringIn)
{
	struct qfle3i_hba *hba = (struct qfle3i_hba  *) cookie;
	char *buffer = stringIn;

	vmk_LogMessage("%s: %d:  qfle3i_set_rdp_info not supported\n",
			__func__, __LINE__);

	return VMK_NOT_SUPPORTED;
}

#define SPEED_CAP_10G_MASK     (1<<30)
#define SPEED_CAP_CONFIG_MASK  (1<<17)
#define OPR_SPEED_10G_MASK     (1<<30)

VMK_ReturnStatus qfle3i_get_rdp_info(vmk_uint64 cookie, void *stringOut)
{
	struct qfle3i_hba *hba = (struct qfle3i_hba *) cookie;
	int rc;
	char *buffer = NULL;
	char *finalBuffer = (char *) stringOut;
	vmk_uint64 finalLength = MAX_KEY_VAL_STRING_LEN;
	vmk_uint64 length = 0;
	struct sfp_diagnostic sfp_diag;
	struct link_error_status lesb;


	buffer = vmk_HeapAlloc(qfle3i_driver_info.heap_id, finalLength + 1);
	if (!buffer) {
		vmk_LogMessage("Failed to allocate the buffer for storing Value.");
		return VMK_FAILURE;
	}

	vmk_Memset(buffer, 0, finalLength);

	/* read RDP information */
	rc = hba->cnic->drv_get_sfp_diagnostic(hba->cnic, &sfp_diag);
	if (rc != VMK_OK) {
		PRINT_ERR(hba, "failed to read SFP parameters\n");
		goto end;
	}
	QFLE3I_DBG(DBG_APP, hba, "temp:0x%x vcc:0x%x txb:0x%x txp:0x%x rxp:0x%x"
					"cap:0x%x opr:0x%x\n",
					sfp_diag.temperature, sfp_diag.vcc, sfp_diag.tx_bias,
					sfp_diag.tx_power, sfp_diag.rx_power, sfp_diag.speed_cap,
					sfp_diag.opr_speed);

	vmk_StringFormat(buffer+length, (finalLength - length), NULL,
			"\n3001: %d: %02hhx:%02hhx:%02hhx:%02hhx:%02hhx:%02hhx\n",
			VMK_ETH_ADDR_LENGTH,
			hba->mac_addr[0], hba->mac_addr[1], hba->mac_addr[2],
			hba->mac_addr[3], hba->mac_addr[4], hba->mac_addr[5]);
	length = vmk_Strnlen(buffer, finalLength);

	vmk_StringFormat(buffer+length, (finalLength - length), NULL,
			"4001: %lu: 0x%08x\n", sizeof(sfp_diag.temperature), sfp_diag.temperature);
	length = vmk_Strnlen(buffer, finalLength);

	vmk_StringFormat(buffer+length, (finalLength - length), NULL,
			"4002: %lu: 0x%08x\n", sizeof(sfp_diag.vcc), sfp_diag.vcc);
	length = vmk_Strnlen(buffer, finalLength);

	vmk_StringFormat(buffer+length, (finalLength - length), NULL,
			"4003: %lu: 0x%08x\n", sizeof(sfp_diag.tx_bias), sfp_diag.tx_bias);
	length = vmk_Strnlen(buffer, finalLength);

	vmk_StringFormat(buffer+length, (finalLength - length), NULL,
			"4004: %lu: 0x%08x\n", sizeof(sfp_diag.tx_power), sfp_diag.tx_power);
	length = vmk_Strnlen(buffer, finalLength);

	vmk_StringFormat(buffer+length, (finalLength - length), NULL,
			"4005: %lu: 0x%08x\n", sizeof(sfp_diag.rx_power), sfp_diag.rx_power);
	length = vmk_Strnlen(buffer, finalLength);

	vmk_StringFormat(buffer+length, (finalLength - length), NULL,
					 "4006: %lu: 0x%08x\n", sizeof(sfp_diag.speed_cap),
					 (SPEED_CAP_10G_MASK | SPEED_CAP_CONFIG_MASK));
	length = vmk_Strnlen(buffer, finalLength);

	vmk_StringFormat(buffer+length, (finalLength - length), NULL,
			"4007: %lu: 0x%08x\n", sizeof(sfp_diag.opr_speed), OPR_SPEED_10G_MASK);
	length = vmk_Strnlen(buffer, finalLength);

end:
	if (!length) {
		vmk_LogMessage("length is zero.");
		return VMK_FAILURE;
	} else if (length < finalLength) {
		vmk_StringCopy(finalBuffer, buffer, length);
	} else {
		vmk_LogMessage("Too much data! Cannot write full data in mgmt_info. Data truncated.");
		vmk_StringCopy(finalBuffer, buffer, length);
	}

	vmk_HeapFree(qfle3i_driver_info.heap_id, buffer);
	return VMK_OK;
}

VMK_ReturnStatus qfle3i_set_adapter_info(vmk_uint64 cookie, void *stringIn)
{
	struct qfle3i_hba *hba = (struct qfle3i_hba  *) cookie;
	char *buffer = stringIn;

	vmk_LogMessage("%s: %d: qfle3i_set_adapter_info is not supported\n",
			__func__, __LINE__);

	return VMK_NOT_SUPPORTED;

}


VMK_ReturnStatus qfle3i_get_adapter_info(vmk_uint64 cookie, void *stringOut)
{
	struct qfle3i_hba *hba = (struct qfle3i_hba *) cookie;
	char *buffer = NULL;
	char *finalBuffer = (char *) stringOut;
	struct qfle3i_sess *sess = NULL;
	unsigned char *ptr;
	vmk_Bool first_sess = 1;
	struct qfle3i_conn *conn;
	struct qfle3i_endpoint *ep;
	vmk_uint64 finalLength = MAX_KEY_VAL_STRING_LEN;
	vmk_uint64 length = 0;

	buffer = vmk_HeapAlloc(qfle3i_driver_info.heap_id, finalLength + 1);
	if (!buffer) {
		vmk_LogMessage("Failed to allocate the buffer for storing Value.");
		return VMK_FAILURE;
	}
	vmk_Memset(buffer, 0, finalLength);

	if (hba == NULL) {
		vmk_LogMessage("wrong cookie parameter.\n");
		return VMK_FAILURE;
	}

	vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\nDriver Version: %s-%s\n", QFLE3I_DRIVER_NAME, qfle3i_version_str);
	length = vmk_Strnlen(buffer, finalLength);

	vmk_UplinkSharedData *sharedData;
	sharedData = hba->cnic->uplinkSharedData;
	vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\tFW Ver: %s\n", vmk_NameToString(&sharedData->driverInfo.firmwareVersion));
	length = vmk_Strnlen(buffer, finalLength);

	vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\tHBA: %s:%p binded to %s\n",
			vmk_NameToString(&hba->vmhba_name), hba, vmk_NameToString(&hba->vmnic_name));
	length = vmk_Strnlen(buffer, finalLength);

	if (vmk_BitVectorTest(hba->adapter_state, ADAPTER_STATE_UP))
		vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\tAdapter State: UP\n");
	else if (vmk_BitVectorTest(hba->adapter_state, ADAPTER_STATE_GOING_DOWN))
		vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\tAdapter State: DOWN\n");
	else if (vmk_BitVectorTest(hba->adapter_state, ADAPTER_STATE_LINK_DOWN))
		vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\tAdapter State: LINK DOWN\n");
	else if (vmk_BitVectorTest(hba->adapter_state, ADAPTER_STATE_INIT_FAILED))
		vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\tAdapter State: INIT_FAILED\n");
	length = vmk_Strnlen(buffer, finalLength);

	vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\tPCI_DEV_ADDR:[%04x:%02x:%02x:%02x]\n",
			hba->sbdf.seg, hba->sbdf.bus,
			hba->sbdf.dev, hba->sbdf.fn);
	length = vmk_Strnlen(buffer, finalLength);

	vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\tISP: ISP%04x\n", hba->pcidev_id.deviceID);
	length = vmk_Strnlen(buffer, finalLength);

	vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\tHost Num: %d\n", hba->host_num);
	length = vmk_Strnlen(buffer, finalLength);

	vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\tPort Speed: %d\n", sharedData->link.speed);
	length = vmk_Strnlen(buffer, finalLength);

	if (sharedData->link.state == VMK_LINK_STATE_UP)
		vmk_StringFormat(buffer+length, finalLength - length, NULL,
				"\tPort State: UP\n");
	else
		vmk_StringFormat(buffer+length, finalLength - length, NULL,
				"\tPort State: DOWN\n");
	length = vmk_Strnlen(buffer, finalLength);

	vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\tMTU: %d\n", hba->mtu_supported);
	length = vmk_Strnlen(buffer, finalLength);

	vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\tActive Endpoints: %d, Active Sess: %d\n",
			hba->ofld_conns_active, hba->num_active_sess);
	length = vmk_Strnlen(buffer, finalLength);

	vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\tmax sqes:%d, cqes:%d rqes:%d\n",
			hba->max_sqes, hba->max_cqes, hba->max_rqes);
	length = vmk_Strnlen(buffer, finalLength);

	if (hba->name_string)
		vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\n\tIQN Name: %s\n", hba->name_string);
	else
		vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\n\tIQN Name: %s\n", "NULL");
	length = vmk_Strnlen(buffer, finalLength);

	vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\tmac_addr: %02hhx:%02hhx:%02hhx:%02hhx:%02hhx:%02hhx\n",
			hba->mac_addr[0], hba->mac_addr[1], hba->mac_addr[2],
			hba->mac_addr[3], hba->mac_addr[4], hba->mac_addr[5]);
	length = vmk_Strnlen(buffer, finalLength);

	vmk_SpinlockLock(hba->lock);
	ql_vmk_list_each_entry(sess, &hba->active_sess, link,
		struct qfle3i_sess) {

		if (!sess->lead_conn || !sess->lead_conn->ep)
			continue;

		conn = sess->lead_conn;
		ep = conn->ep;

		if (first_sess) {
			ptr = (unsigned char *) &ep->cm_sk->src_ip;
			if (vmk_BitVectorTest(ep->cm_sk->flags, SK_F_IPV6))
				vmk_StringFormat(buffer+length, finalLength - length, NULL,
					"\tsrc_ip:%04x:%04x:%04x:%04x:%04x:%04x:%04x:%04x \n",
					ptr[0], ptr[1], ptr[2], ptr[3], ptr[4], ptr[5], ptr[6], ptr[7]);
			 else
				vmk_StringFormat(buffer+length, finalLength - length, NULL,
					"\tsrc_ip:%u.%u.%u.%u \n",
					ptr[0], ptr[1], ptr[2], ptr[3]);
			length = vmk_Strnlen(buffer, finalLength);

			vmk_StringFormat(buffer+length, finalLength - length, NULL,
				"\n ### Sess/Conn List ### \n");
			length = vmk_Strnlen(buffer, finalLength);
			first_sess = 0;
		}

		vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\n\tsess:%p, conn:%p cid:0x%x channel_id:%d target_id:%d \n",
			sess, sess->lead_conn, conn->iscsi_conn_cid,
			sess->channel_id, sess->target_id);
		length = vmk_Strnlen(buffer, finalLength);

		vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\t\tstate:0x%x recov_state:0x%lx\n",
			sess->state, sess->recovery_state);
		length = vmk_Strnlen(buffer, finalLength);

		if (sess->target_name) {
			vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\t\ttgt name:%s\n", sess->target_name);
			length = vmk_Strnlen(buffer, finalLength);
		}

		ptr = (unsigned char *) &ep->cm_sk->dst_ip;
		if (vmk_BitVectorTest(ep->cm_sk->flags, SK_F_IPV6))
			vmk_StringFormat(buffer+length, finalLength - length, NULL,
				"\t\tdst_ip:%04x:%04x:%04x:%04x:%04x:%04x:%04x:%04x \n",
				ptr[0], ptr[1], ptr[2], ptr[3], ptr[4], ptr[5], ptr[6], ptr[7]);
		 else
			vmk_StringFormat(buffer+length, finalLength - length, NULL,
				"\t\tdst_ip:%u.%u.%u.%u \n",
				ptr[0], ptr[1], ptr[2], ptr[3]);
		length = vmk_Strnlen(buffer, finalLength);

		struct qfle3i_iscsilun *iscsilun = NULL;
		vmk_uint32 lun_list[QFLE3I_MAX_LUNS];
		vmk_uint32 num_luns = 0;
		vmk_uint32 i = 0;
		vmk_SpinlockLock(sess->iscsilun_lock);
		ql_vmk_list_each_entry(iscsilun, &sess->iscsilun_list, link, qfle3i_iscsilun) {
			if (iscsilun) {
				lun_list[num_luns] = iscsilun->lun_id;
				num_luns++;
			}
		}
		vmk_SpinlockUnlock(sess->iscsilun_lock);

		vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\t\ttotal_luns:%d, lun_ids:", num_luns);
		length = vmk_Strnlen(buffer, finalLength);

		for (i = 0; i < num_luns; i++) {
			vmk_StringFormat(buffer+length, finalLength - length, NULL,
				"%d, ", lun_list[i]);
			length = vmk_Strnlen(buffer, finalLength);
		}

		vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\n\t\thdr_dig:%d, data_dig:%d\n",
			conn->header_digest_en, conn->data_digest_en);
		length = vmk_Strnlen(buffer, finalLength);


		vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\t\tcmds/sess:%d, free_cmds:%d, total_allocated_cmds:%d, total_freed_cmds:%d\n",
			sess->allocated_cmds, sess->num_free_cmds,
			sess->total_cmds_allocated, sess->total_cmds_freed);
		length = vmk_Strnlen(buffer, finalLength);

		vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\t\tlogin_req:%d, login_resp:%d, nop_req:%d, nop_res:%d, logout_req:%d, logout_res:%d\n",
			conn->num_login_req_pdus, conn->num_login_resp_pdus, conn->num_nopout_pdus,
			conn->num_nopin_pdus, conn->num_logout_req_pdus, conn->num_logout_resp_pdus);
		length = vmk_Strnlen(buffer, finalLength);

		vmk_StringFormat(buffer+length, finalLength - length, NULL,
			"\t\tscsi_cmds:%d, scsi_resp:%d, wr_cmds:%d, rd_cmds:%d, abrt_req:%d, abrt_res:%d\n",
			conn->num_scsi_cmd_pdus, conn->num_scsi_resp_pdus, conn->num_dataout_pdus,
			conn->num_datain_pdus, conn->num_tmf_req_pdus, conn->num_tmf_resp_pdus);
		length = vmk_Strnlen(buffer, finalLength);
	}
	vmk_SpinlockUnlock(hba->lock);



	if (length < finalLength)
		vmk_StringCopy(finalBuffer, buffer, length);
	else {
		vmk_LogMessage("Too much data! Cannot write full data in mgmt_info. Data truncated.");
		vmk_StringCopy(finalBuffer, buffer, length);
	}


	vmk_HeapFree(qfle3i_driver_info.heap_id, buffer);

	return VMK_OK;
}

VMK_ReturnStatus qfle3iKeyValueInitPerHba(qfle3i_hba *hba)
{
	vmk_Name kvName, vmhbaName;
	VMK_ReturnStatus vmk_stat = VMK_OK;
	vmk_MgmtProps mgmtProps;

	QFLE3I_DBG(DBG_INIT, hba, "Enter\n");

	vmk_stat = vmk_NameInitialize(&kvName,
			vmk_ScsiGetAdapterName(hba->scsiAdapter));

	vmk_NameInitialize(&kvSig.name, vmk_NameToString(&kvName));

	mgmtProps.modId = vmk_ModuleCurrentID;
	mgmtProps.heapId =  vmk_ModuleGetHeapID(vmk_ModuleCurrentID);
	mgmtProps.sig = &kvSig;
	mgmtProps.cleanupFn = NULL;
	mgmtProps.sessionAnnounceFn = NULL;
	mgmtProps.sessionCleanupFn = NULL;
	mgmtProps.handleCookie = (vmk_uint64) hba;
	vmk_stat = vmk_MgmtInit(&mgmtProps,
			&hba->kvMgmtHandle);
	if (vmk_stat != VMK_OK)
		return vmk_stat;

	vmk_NameInitialize(&vmhbaName, "ADAPTER");
	vmk_stat = vmk_MgmtAddKey(hba->kvMgmtHandle,
			VMK_MGMT_KEY_TYPE_STRING,
			&vmhbaName,
			qfle3i_get_adapter_info,
			qfle3i_set_adapter_info);
	if (vmk_stat != VMK_OK) {
		vmk_MgmtDestroy(hba->kvMgmtHandle);
		hba->mgmtKeyAdded = VMK_FALSE;
		return vmk_stat;
	}

	vmk_NameInitialize(&vmhbaName, "SSAN_RDP");
	vmk_stat = vmk_MgmtAddKey(hba->kvMgmtHandle,
			VMK_MGMT_KEY_TYPE_STRING,
			&vmhbaName,
			qfle3i_get_rdp_info,
			qfle3i_set_rdp_info);
	if (vmk_stat != VMK_OK) {
		vmk_MgmtDestroy(hba->kvMgmtHandle);
		hba->mgmtKeyAdded = VMK_FALSE;
		return vmk_stat;
	}
	return vmk_stat;
}

static void
qfle3i_ScsiNotifyIOAllowed(vmk_Device device, vmk_Bool isAllowed)
{
	vmk_ScsiAdapter *scsiAdapter = NULL;
	qfle3i_hba *hba = NULL;
	VMK_ReturnStatus status = VMK_OK;

	status = vmk_DeviceGetRegistrationData(device,
				(vmk_AddrCookie *)&scsiAdapter);
	if ((status != VMK_OK) || (scsiAdapter == NULL)) {
		PRINT_ERR(hba, "Can't get scsi adapter, status:%s\n",
						vmk_StatusToString(status));
		return;
	}
	hba = (qfle3i_hba *)scsiAdapter->clientData;

	QFLE3I_DBG(DBG_INIT, hba, "hba:%p hostnum:%d,isAllowed:%d\n",
					hba, hba->host_num, isAllowed);

	if (isAllowed == VMK_FALSE) {
		return;
	}

	if (hba->isRegisteredWithTransport == VMK_FALSE) {
		status = qfle3i_register_transport(hba);
		if (status != VMK_OK) {
			PRINT_INFO(hba, "qfle3i_register_transport failed. status:%s\n",
						vmk_StatusToString(status));
			return;
		}

		status = qfle3i_register_adapter(hba);
		if (status != VMK_OK) {
			PRINT_ERR(hba, "Can't register iscsi adapter, status:%s\n",
							vmk_StatusToString(status));
			qfle3i_unregister_transport(hba);
			return;
		} else {
			PRINT_NOTICE(hba, "Init key value.\n ");
			if ((hba->mgmtKeyAdded == VMK_FALSE)) {
				status = qfle3iKeyValueInitPerHba(hba);
				if (status != VMK_OK) {
					qfle3i_unreg_iscsi_adapter(hba);
					if (hba->iscsiTransport)
						qfle3i_unregister_transport(hba);

					PRINT_ERR(hba, "Can't register qfle3iKeyValueInitPerHba, "
							"status:%s\n",
							vmk_StatusToString(status));
					return;
				}
				hba->mgmtKeyAdded = VMK_TRUE;
			}
		}
		hba->isRegisteredWithTransport = VMK_TRUE;
	}
	return;
}

VMK_ReturnStatus qfle3i_setup_scsiadapter(qfle3i_hba *hba)
{
	vmk_DMAEngineProps dmaProps;
	VMK_ReturnStatus status = VMK_OK;
#if (VMWARE_ESX_DDK_VERSION >= 60000)
	vmk_ScsiAdapterCapabilities scsiAdapterCap;
#endif

	hba->scsiAdapter->device                 = hba->cnic->ldev;
	hba->scsiAdapter->hostMaxSectors         = QFLE3I_MAX_IO_SECTORS;
	hba->scsiAdapter->qDepthPtr              = &hba->q_depth;
	hba->scsiAdapter->command                = (vmk_ScsiAdapterCommand)qfle3i_queuecommand;
	hba->scsiAdapter->taskMgmt               = (vmk_ScsiAdapterTaskMgmt)qfle3i_scsiTaskMgmt;
	hba->scsiAdapter->dumpCommand            = (vmk_ScsiAdapterDumpCommand)qfle3i_scsiDumpCommand;
	//hba->scsiAdapter->close =
	hba->scsiAdapter->procInfo               = qfle3i_procInfo;
	hba->scsiAdapter->dumpQueue              = (vmk_ScsiAdapterDumpQueue)qfle3i_scsiDumpQueue;
	hba->scsiAdapter->dumpPollHandler        = (vmk_ScsiAdapterDumpPollHandler)qfle3i_scsiDumpPollHandler;
	hba->scsiAdapter->dumpPollHandlerData    = hba;
	hba->scsiAdapter->ioctl                  = qfle3i_ioctl;
	hba->scsiAdapter->discover               = qfle3i_discover;
#if (VMWARE_ESX_DDK_VERSION <= 65000)
	hba->scsiAdapter->vportop                = NULL;
	hba->scsiAdapter->vportDiscover          = NULL;
#endif
	hba->scsiAdapter->modifyDeviceQueueDepth = qfle3i_modifyQueueDepth;
	hba->scsiAdapter->queryDeviceQueueDepth  = qfle3i_query_qdepth;
	hba->scsiAdapter->checkTarget            = qfle3i_checkTarget;
	hba->scsiAdapter->targetId               = -1;
	hba->scsiAdapter->flags                  = VMK_SCSI_ADAPTER_FLAG_REGISTER_WITHOUT_SCAN;
	hba->scsiAdapter->moduleID               = vmk_ModuleCurrentID;
	hba->scsiAdapter->clientData             = hba;
	hba->scsiAdapter->channels               = 64;
	hba->scsiAdapter->maxTargets             = QFLE3I_MAX_TARGET;
	hba->scsiAdapter->maxLUNs                = QFLE3I_MAX_LUNS;
	hba->scsiAdapter->paeCapable             = VMK_TRUE;
	hba->scsiAdapter->maxCmdLen              = QFLE3I_MAX_CMD_LEN;
	hba->scsiAdapter->notifyIOAllowed        = qfle3i_ScsiNotifyIOAllowed;
	/*
	   vmk_uint8 tmp_name[32];
	   vmk_Snprintf(tmp_name, 32, "%s_%d", QFLE3I_SCSI_HBA_NAME, hba->instance);
	   vmk_NameInitialize(&hba->scsiAdapter->name, tmp_name);
	*/
	//hba->scsiAdapter->mgmtAdapter.transport = VMK_STORAGE_ADAPTER_ISCSI_VENDOR_SUPPLIED_IMA;

#if (VMWARE_ESX_DDK_VERSION >= 60000)
	/*
	 * Set second level addressing capability support. If there is a failure
	 * just issue a warning as we do not want to disable the complete adapter.
	 */
	scsiAdapterCap = VMK_SCSI_ADAPTER_CAP_SECONDLEVEL_ADDRESSING;

	status = vmk_ScsiAdapterSetCapabilities(hba->scsiAdapter, scsiAdapterCap);
	if (status != VMK_OK) {
		PRINT_WARNING(hba, "Unable to set adapter capabilities. Reason: %s",
				vmk_StatusToString(status));
		status = VMK_OK;
	}
#endif

	vmk_NameInitialize(&hba->scsiAdapter->driverName, QFLE3I_DRIVER_NAME);
	hba->scsiAdapter->engine = hba->dmaEngine;

	QFLE3I_DBG(DBG_INIT, hba, "qfle3i: scsi adapter setup is done.\n");
	return status;
}
