/*
 * Decompiled with CFR 0.152.
 */
package org.apache.rocketmq.tieredstore.core;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.Scheduler;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import org.apache.rocketmq.common.BoundaryType;
import org.apache.rocketmq.common.message.MessageQueue;
import org.apache.rocketmq.store.GetMessageResult;
import org.apache.rocketmq.store.GetMessageStatus;
import org.apache.rocketmq.store.MessageFilter;
import org.apache.rocketmq.store.QueryMessageResult;
import org.apache.rocketmq.store.SelectMappedBufferResult;
import org.apache.rocketmq.tieredstore.MessageStoreConfig;
import org.apache.rocketmq.tieredstore.TieredMessageStore;
import org.apache.rocketmq.tieredstore.common.GetMessageResultExt;
import org.apache.rocketmq.tieredstore.common.SelectBufferResult;
import org.apache.rocketmq.tieredstore.core.MessageStoreFetcher;
import org.apache.rocketmq.tieredstore.exception.TieredStoreException;
import org.apache.rocketmq.tieredstore.file.FlatFileStore;
import org.apache.rocketmq.tieredstore.file.FlatMessageFile;
import org.apache.rocketmq.tieredstore.index.IndexItem;
import org.apache.rocketmq.tieredstore.index.IndexService;
import org.apache.rocketmq.tieredstore.metadata.MetadataStore;
import org.apache.rocketmq.tieredstore.metadata.entity.TopicMetadata;
import org.apache.rocketmq.tieredstore.util.MessageFormatUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MessageStoreFetcherImpl
implements MessageStoreFetcher {
    private static final Logger log = LoggerFactory.getLogger((String)"RocketmqTieredStore");
    protected static final String CACHE_KEY_FORMAT = "%s@%d@%d";
    private final String brokerName;
    private final MetadataStore metadataStore;
    private final MessageStoreConfig storeConfig;
    private final TieredMessageStore messageStore;
    private final IndexService indexService;
    private final FlatFileStore flatFileStore;
    private final long memoryMaxSize;
    private final Cache<String, SelectBufferResult> fetcherCache;

    public MessageStoreFetcherImpl(TieredMessageStore messageStore) {
        this(messageStore, messageStore.getStoreConfig(), messageStore.getFlatFileStore(), messageStore.getIndexService());
    }

    public MessageStoreFetcherImpl(TieredMessageStore messageStore, MessageStoreConfig storeConfig, FlatFileStore flatFileStore, IndexService indexService) {
        this.storeConfig = storeConfig;
        this.brokerName = storeConfig.getBrokerName();
        this.flatFileStore = flatFileStore;
        this.messageStore = messageStore;
        this.indexService = indexService;
        this.metadataStore = flatFileStore.getMetadataStore();
        this.memoryMaxSize = (long)((double)Runtime.getRuntime().maxMemory() * storeConfig.getReadAheadCacheSizeThresholdRate());
        this.fetcherCache = this.initCache(storeConfig);
        log.info("MessageStoreFetcher init success, brokerName={}", (Object)storeConfig.getBrokerName());
    }

    private Cache<String, SelectBufferResult> initCache(MessageStoreConfig storeConfig) {
        return Caffeine.newBuilder().scheduler(Scheduler.systemScheduler()).expireAfterAccess(storeConfig.getReadAheadCacheExpireDuration(), TimeUnit.MILLISECONDS).maximumWeight(this.memoryMaxSize).weigher((key, buffer) -> buffer.getSize()).recordStats().build();
    }

    public Cache<String, SelectBufferResult> getFetcherCache() {
        return this.fetcherCache;
    }

    protected void putMessageToCache(FlatMessageFile flatFile, long offset, SelectBufferResult result) {
        MessageQueue mq = flatFile.getMessageQueue();
        this.fetcherCache.put((Object)String.format(CACHE_KEY_FORMAT, mq.getTopic(), mq.getQueueId(), offset), (Object)result);
    }

    protected SelectBufferResult getMessageFromCache(FlatMessageFile flatFile, long offset) {
        MessageQueue mq = flatFile.getMessageQueue();
        SelectBufferResult buffer = (SelectBufferResult)this.fetcherCache.getIfPresent((Object)String.format(CACHE_KEY_FORMAT, mq.getTopic(), mq.getQueueId(), offset));
        if (buffer == null) {
            return null;
        }
        long count = buffer.getAccessCount().incrementAndGet();
        if (count % 1000L == 0L) {
            log.warn("MessageFetcher fetch same offset message too many times, topic={}, queueId={}, offset={}, count={}", new Object[]{mq.getTopic(), mq.getQueueId(), offset, count});
        }
        return new SelectBufferResult(buffer.getByteBuffer().asReadOnlyBuffer(), buffer.getStartOffset(), buffer.getSize(), buffer.getTagCode());
    }

    protected GetMessageResultExt getMessageFromCache(FlatMessageFile flatFile, long offset, int maxCount, MessageFilter messageFilter) {
        GetMessageResultExt result = new GetMessageResultExt();
        int interval = this.storeConfig.getReadAheadMessageCountThreshold();
        long end = offset + (long)interval;
        for (long current = offset; current < end; ++current) {
            SelectBufferResult buffer = this.getMessageFromCache(flatFile, current);
            if (buffer == null) {
                result.setNextBeginOffset(current);
                break;
            }
            result.setNextBeginOffset(current + 1L);
            if (messageFilter != null && (!messageFilter.isMatchedByConsumeQueue(Long.valueOf(buffer.getTagCode()), null) || !messageFilter.isMatchedByCommitLog(buffer.getByteBuffer().slice(), null))) continue;
            SelectMappedBufferResult bufferResult = new SelectMappedBufferResult(buffer.getStartOffset(), buffer.getByteBuffer(), buffer.getSize(), null);
            result.addMessageExt(bufferResult, current, buffer.getTagCode());
            if (result.getMessageCount() == maxCount) break;
        }
        result.setStatus(result.getMessageCount() > 0 ? GetMessageStatus.FOUND : GetMessageStatus.NO_MATCHED_MESSAGE);
        result.setMinOffset(flatFile.getConsumeQueueMinOffset());
        result.setMaxOffset(flatFile.getConsumeQueueCommitOffset());
        return result;
    }

    protected CompletableFuture<Long> fetchMessageThenPutToCache(FlatMessageFile flatFile, long queueOffset, int batchSize) {
        MessageQueue mq = flatFile.getMessageQueue();
        return this.getMessageFromTieredStoreAsync(flatFile, queueOffset, batchSize).thenApply(result -> {
            if (result.getStatus() == GetMessageStatus.OFFSET_OVERFLOW_ONE || result.getStatus() == GetMessageStatus.OFFSET_OVERFLOW_BADLY) {
                return -1L;
            }
            if (result.getStatus() != GetMessageStatus.FOUND) {
                log.warn("MessageFetcher prefetch message then put to cache failed, result={}, topic={}, queue={}, queue offset={}, batch size={}", new Object[]{result.getStatus(), mq.getTopic(), mq.getQueueId(), queueOffset, batchSize});
                return -1L;
            }
            List offsetList = result.getMessageQueueOffset();
            List<Long> tagCodeList = result.getTagCodeList();
            List msgList = result.getMessageMapedList();
            for (int i = 0; i < offsetList.size(); ++i) {
                SelectMappedBufferResult msg = (SelectMappedBufferResult)msgList.get(i);
                SelectBufferResult bufferResult = new SelectBufferResult(msg.getByteBuffer(), msg.getStartOffset(), msg.getSize(), tagCodeList.get(i));
                this.putMessageToCache(flatFile, queueOffset + (long)i, bufferResult);
            }
            return (Long)offsetList.get(offsetList.size() - 1);
        });
    }

    public CompletableFuture<GetMessageResult> getMessageFromCacheAsync(FlatMessageFile flatFile, String group, long queueOffset, int maxCount, MessageFilter messageFilter) {
        MessageQueue mq = flatFile.getMessageQueue();
        GetMessageResultExt result = this.getMessageFromCache(flatFile, queueOffset, maxCount, messageFilter);
        if (GetMessageStatus.FOUND.equals((Object)result.getStatus())) {
            log.debug("MessageFetcher cache hit, group={}, topic={}, queueId={}, offset={}, maxCount={}, resultSize={}, lag={}", new Object[]{group, mq.getTopic(), mq.getQueueId(), queueOffset, maxCount, result.getMessageCount(), result.getMaxOffset() - result.getNextBeginOffset()});
            return CompletableFuture.completedFuture(result);
        }
        log.debug("MessageFetcher cache miss, group={}, topic={}, queueId={}, offset={}, maxCount={}, lag={}", new Object[]{group, mq.getTopic(), mq.getQueueId(), queueOffset, maxCount, result.getMaxOffset() - result.getNextBeginOffset()});
        int fetchSize = maxCount == 1 ? 32 : this.storeConfig.getReadAheadMessageCountThreshold();
        return this.fetchMessageThenPutToCache(flatFile, queueOffset, fetchSize).thenApply(maxOffset -> this.getMessageFromCache(flatFile, queueOffset, maxCount, messageFilter));
    }

    public CompletableFuture<GetMessageResultExt> getMessageFromTieredStoreAsync(FlatMessageFile flatFile, long queueOffset, int batchSize) {
        CompletableFuture<ByteBuffer> readConsumeQueueFuture;
        GetMessageResultExt result = new GetMessageResultExt();
        result.setMinOffset(flatFile.getConsumeQueueMinOffset());
        result.setMaxOffset(flatFile.getConsumeQueueCommitOffset());
        if (queueOffset < result.getMinOffset()) {
            result.setStatus(GetMessageStatus.OFFSET_TOO_SMALL);
            result.setNextBeginOffset(result.getMinOffset());
            return CompletableFuture.completedFuture(result);
        }
        if (queueOffset == result.getMaxOffset()) {
            result.setStatus(GetMessageStatus.OFFSET_OVERFLOW_ONE);
            result.setNextBeginOffset(queueOffset);
            return CompletableFuture.completedFuture(result);
        }
        if (queueOffset > result.getMaxOffset()) {
            result.setStatus(GetMessageStatus.OFFSET_OVERFLOW_BADLY);
            result.setNextBeginOffset(result.getMaxOffset());
            return CompletableFuture.completedFuture(result);
        }
        if (queueOffset < result.getMaxOffset()) {
            batchSize = Math.min(batchSize, (int)Math.min(result.getMaxOffset() - queueOffset, (long)this.storeConfig.getReadAheadMessageCountThreshold()));
        }
        try {
            readConsumeQueueFuture = flatFile.getConsumeQueueAsync(queueOffset, batchSize);
        }
        catch (TieredStoreException e2) {
            switch (e2.getErrorCode()) {
                default: 
            }
            result.setStatus(GetMessageStatus.OFFSET_FOUND_NULL);
            result.setNextBeginOffset(queueOffset);
            return CompletableFuture.completedFuture(result);
        }
        int finalBatchSize = batchSize;
        CompletionStage readCommitLogFuture = readConsumeQueueFuture.thenCompose(cqBuffer -> {
            long firstCommitLogOffset = MessageFormatUtil.getCommitLogOffsetFromItem(cqBuffer);
            cqBuffer.position(cqBuffer.remaining() - 20);
            long lastCommitLogOffset = MessageFormatUtil.getCommitLogOffsetFromItem(cqBuffer);
            if (lastCommitLogOffset < firstCommitLogOffset) {
                log.error("MessageFetcher#getMessageFromTieredStoreAsync, last offset is smaller than first offset, topic={} queueId={}, offset={}, firstOffset={}, lastOffset={}", new Object[]{flatFile.getMessageQueue().getTopic(), flatFile.getMessageQueue().getQueueId(), queueOffset, firstCommitLogOffset, lastCommitLogOffset});
                return CompletableFuture.completedFuture(ByteBuffer.allocate(0));
            }
            long length = lastCommitLogOffset - firstCommitLogOffset + (long)MessageFormatUtil.getSizeFromItem(cqBuffer);
            while (cqBuffer.limit() > 20 && length > (long)this.storeConfig.getReadAheadMessageSizeThreshold()) {
                cqBuffer.limit(cqBuffer.position());
                cqBuffer.position(cqBuffer.limit() - 20);
                length = MessageFormatUtil.getCommitLogOffsetFromItem(cqBuffer) - firstCommitLogOffset + (long)MessageFormatUtil.getSizeFromItem(cqBuffer);
            }
            int messageCount = cqBuffer.position() / 20 + 1;
            log.info("MessageFetcher#getMessageFromTieredStoreAsync, topic={}, queueId={}, broker offset={}-{}, offset={}, expect={}, actually={}, lag={}", new Object[]{flatFile.getMessageQueue().getTopic(), flatFile.getMessageQueue().getQueueId(), result.getMinOffset(), result.getMaxOffset(), queueOffset, finalBatchSize, messageCount, result.getMaxOffset() - queueOffset});
            return flatFile.getCommitLogAsync(firstCommitLogOffset, (int)length);
        });
        return ((CompletableFuture)readConsumeQueueFuture.thenCombine(readCommitLogFuture, (cqBuffer, msgBuffer) -> {
            List<SelectBufferResult> bufferList = MessageFormatUtil.splitMessageBuffer(cqBuffer, msgBuffer);
            int requestSize = cqBuffer.remaining() / 20;
            if (bufferList.isEmpty()) {
                result.setStatus(GetMessageStatus.NO_MATCHED_MESSAGE);
                result.setNextBeginOffset(queueOffset + (long)requestSize);
            } else {
                result.setStatus(GetMessageStatus.FOUND);
                result.setNextBeginOffset(queueOffset + (long)requestSize);
                for (SelectBufferResult bufferResult : bufferList) {
                    ByteBuffer slice = bufferResult.getByteBuffer().slice();
                    slice.limit(bufferResult.getSize());
                    SelectMappedBufferResult msg = new SelectMappedBufferResult(bufferResult.getStartOffset(), bufferResult.getByteBuffer(), bufferResult.getSize(), null);
                    result.addMessageExt(msg, MessageFormatUtil.getQueueOffset(slice), bufferResult.getTagCode());
                }
            }
            return result;
        })).exceptionally(e -> {
            MessageQueue mq = flatFile.getMessageQueue();
            log.warn("MessageFetcher#getMessageFromTieredStoreAsync failed, topic={} queueId={}, offset={}, batchSize={}", new Object[]{mq.getTopic(), mq.getQueueId(), queueOffset, finalBatchSize, e});
            result.setStatus(GetMessageStatus.OFFSET_FOUND_NULL);
            result.setNextBeginOffset(queueOffset);
            return result;
        });
    }

    @Override
    public CompletableFuture<GetMessageResult> getMessageAsync(String group, String topic, int queueId, long queueOffset, int maxCount, MessageFilter messageFilter) {
        boolean cacheBusy;
        GetMessageResult result = new GetMessageResult();
        FlatMessageFile flatFile = this.flatFileStore.getFlatFile(new MessageQueue(topic, this.brokerName, queueId));
        if (flatFile == null) {
            result.setNextBeginOffset(queueOffset);
            result.setStatus(GetMessageStatus.NO_MATCHED_LOGIC_QUEUE);
            return CompletableFuture.completedFuture(result);
        }
        result.setMinOffset(flatFile.getConsumeQueueMinOffset());
        result.setMaxOffset(flatFile.getConsumeQueueCommitOffset());
        if (result.getMaxOffset() <= 0L) {
            result.setStatus(GetMessageStatus.NO_MESSAGE_IN_QUEUE);
            result.setNextBeginOffset(queueOffset);
            return CompletableFuture.completedFuture(result);
        }
        if (queueOffset < result.getMinOffset()) {
            result.setStatus(GetMessageStatus.OFFSET_TOO_SMALL);
            result.setNextBeginOffset(result.getMinOffset());
            return CompletableFuture.completedFuture(result);
        }
        if (queueOffset == result.getMaxOffset()) {
            result.setStatus(GetMessageStatus.OFFSET_OVERFLOW_ONE);
            result.setNextBeginOffset(result.getMaxOffset());
            return CompletableFuture.completedFuture(result);
        }
        if (queueOffset > result.getMaxOffset()) {
            result.setStatus(GetMessageStatus.OFFSET_OVERFLOW_BADLY);
            result.setNextBeginOffset(result.getMaxOffset());
            return CompletableFuture.completedFuture(result);
        }
        boolean bl = cacheBusy = (double)this.fetcherCache.estimatedSize() > (double)this.memoryMaxSize * 0.8;
        if (this.storeConfig.isReadAheadCacheEnable() && !cacheBusy) {
            return this.getMessageFromCacheAsync(flatFile, group, queueOffset, maxCount, messageFilter);
        }
        return this.getMessageFromTieredStoreAsync(flatFile, queueOffset, maxCount).thenApply(messageResultExt -> messageResultExt.doFilterMessage(messageFilter));
    }

    @Override
    public CompletableFuture<Long> getEarliestMessageTimeAsync(String topic, int queueId) {
        FlatMessageFile flatFile = this.flatFileStore.getFlatFile(new MessageQueue(topic, this.brokerName, queueId));
        if (flatFile == null) {
            return CompletableFuture.completedFuture(-1L);
        }
        int length = 64;
        return flatFile.getCommitLogAsync(flatFile.getCommitLogMinOffset(), length).thenApply(MessageFormatUtil::getStoreTimeStamp);
    }

    @Override
    public CompletableFuture<Long> getMessageStoreTimeStampAsync(String topic, int queueId, long queueOffset) {
        FlatMessageFile flatFile = this.flatFileStore.getFlatFile(new MessageQueue(topic, this.brokerName, queueId));
        if (flatFile == null) {
            return CompletableFuture.completedFuture(-1L);
        }
        return ((CompletableFuture)((CompletableFuture)flatFile.getConsumeQueueAsync(queueOffset).thenComposeAsync(cqItem -> {
            long commitLogOffset = MessageFormatUtil.getCommitLogOffsetFromItem(cqItem);
            int size = MessageFormatUtil.getSizeFromItem(cqItem);
            return flatFile.getCommitLogAsync(commitLogOffset, size);
        }, (Executor)this.messageStore.getStoreExecutor().bufferFetchExecutor)).thenApply(MessageFormatUtil::getStoreTimeStamp)).exceptionally(e -> {
            log.error("MessageStoreFetcherImpl#getMessageStoreTimeStampAsync: get or decode message failed, topic={}, queue={}, offset={}", new Object[]{topic, queueId, queueOffset, e});
            return -1L;
        });
    }

    @Override
    public long getOffsetInQueueByTime(String topic, int queueId, long timestamp, BoundaryType type) {
        FlatMessageFile flatFile = this.flatFileStore.getFlatFile(new MessageQueue(topic, this.brokerName, queueId));
        if (flatFile == null) {
            return -1L;
        }
        return flatFile.getQueueOffsetByTimeAsync(timestamp, type).join();
    }

    @Override
    public CompletableFuture<QueryMessageResult> queryMessageAsync(String topic, String key, int maxCount, long begin, long end) {
        long topicId;
        try {
            TopicMetadata topicMetadata = this.metadataStore.getTopic(topic);
            if (topicMetadata == null) {
                log.info("MessageFetcher#queryMessageAsync, topic metadata not found, topic={}", (Object)topic);
                return CompletableFuture.completedFuture(new QueryMessageResult());
            }
            topicId = topicMetadata.getTopicId();
        }
        catch (Exception e) {
            log.error("MessageFetcher#queryMessageAsync, get topic id failed, topic={}", (Object)topic, (Object)e);
            return CompletableFuture.completedFuture(new QueryMessageResult());
        }
        CompletableFuture<List<IndexItem>> future = this.indexService.queryAsync(topic, key, maxCount, begin, end);
        return ((CompletableFuture)future.thenCompose(indexItemList -> {
            ArrayList<CompletionStage> futureList = new ArrayList<CompletionStage>(maxCount);
            for (IndexItem indexItem : indexItemList) {
                FlatMessageFile flatFile;
                if (topicId != (long)indexItem.getTopicId() || (flatFile = this.flatFileStore.getFlatFile(new MessageQueue(topic, this.brokerName, indexItem.getQueueId()))) == null) continue;
                CompletionStage getMessageFuture = flatFile.getCommitLogAsync(indexItem.getOffset(), indexItem.getSize()).thenApply(messageBuffer -> new SelectMappedBufferResult(indexItem.getOffset(), messageBuffer, indexItem.getSize(), null));
                futureList.add(getMessageFuture);
                if (futureList.size() < maxCount) continue;
                break;
            }
            return CompletableFuture.allOf(futureList.toArray(new CompletableFuture[0])).thenApply(v -> {
                QueryMessageResult result = new QueryMessageResult();
                futureList.forEach(f -> f.thenAccept(arg_0 -> ((QueryMessageResult)result).addMessage(arg_0)));
                return result;
            });
        })).whenComplete((result, throwable) -> {
            if (result != null) {
                log.info("MessageFetcher#queryMessageAsync, query result={}, topic={}, topicId={}, key={}, maxCount={}, timestamp={}-{}", new Object[]{result.getMessageBufferList().size(), topic, topicId, key, maxCount, begin, end});
            }
        });
    }
}

