#include <stdlib.h>
#include <string.h>

#ifndef _WIN32
# include <sched.h>
# include <sys/types.h>
# include <unistd.h>
# include <dirent.h>
# include <time.h>
# define IS_SLASH(c) ((c) == '/')
#else
# include <io.h>
# define IS_SLASH(c) ((c) == '/' || (c) == '\\')
#endif /* _WIN32 */

#define UVWASI__READDIR_NUM_ENTRIES 1
#define UVWASI__MAX_SYMLINK_FOLLOWS 32

#include "uvwasi.h"
#include "uvwasi_alloc.h"
#include "uv.h"
#include "uv_mapping.h"
#include "fd_table.h"
#include "clocks.h"
#include "wasi_rights.h"

/* TODO(cjihrig): PATH_MAX_BYTES shouldn't be stack allocated. On Windows, paths
   can be 32k long, and this PATH_MAX_BYTES is an artificial limitation. */
#ifdef _WIN32
/* MAX_PATH is in characters, not bytes. Make sure we have enough headroom. */
# define PATH_MAX_BYTES (MAX_PATH * 4)
#else
# include <limits.h>
# define PATH_MAX_BYTES (PATH_MAX)
#endif

/* IBMi PASE does not support posix_fadvise() */
#ifdef __PASE__
# undef POSIX_FADV_NORMAL
#endif

static void* default_malloc(size_t size, void* mem_user_data) {
  return malloc(size);
}

static void default_free(void* ptr, void* mem_user_data) {
  free(ptr);
}

static void* default_calloc(size_t nmemb, size_t size, void* mem_user_data) {
  return calloc(nmemb, size);
}

static void* default_realloc(void* ptr, size_t size, void* mem_user_data) {
  return realloc(ptr, size);
}

void* uvwasi__malloc(const uvwasi_t* uvwasi, size_t size) {
  return uvwasi->allocator->malloc(size, uvwasi->allocator->mem_user_data);
}

void uvwasi__free(const uvwasi_t* uvwasi, void* ptr) {
  uvwasi->allocator->free(ptr, uvwasi->allocator->mem_user_data);
}

void* uvwasi__calloc(const uvwasi_t* uvwasi, size_t nmemb, size_t size) {
  return uvwasi->allocator->calloc(nmemb,
                                   size,
                                   uvwasi->allocator->mem_user_data);
}

void* uvwasi__realloc(const uvwasi_t* uvwasi, void* ptr, size_t size) {
  return uvwasi->allocator->realloc(ptr,
                                    size,
                                    uvwasi->allocator->mem_user_data);
}

static const uvwasi_mem_t default_allocator = {
  NULL,
  default_malloc,
  default_free,
  default_calloc,
  default_realloc,
};


static int uvwasi__is_absolute_path(const char* path, size_t path_len) {
  /* It's expected that only Unix style paths will be generated by WASI. */
  return path != NULL && path_len > 0 && path[0] == '/';
}


static char* uvwasi__strchr_slash(const char* s) {
  /* strchr() that identifies /, as well as \ on Windows. */
  do {
    if (IS_SLASH(*s))
      return (char*) s;
  } while (*s++);

  return NULL;
}


static uvwasi_errno_t uvwasi__normalize_path(const char* path,
                                             size_t path_len,
                                             char* normalized_path,
                                             size_t normalized_len) {
  const char* cur;
  char* ptr;
  char* next;
  size_t cur_len;

  if (path_len > normalized_len)
    return UVWASI_ENOBUFS;

  normalized_path[0] = '\0';
  ptr = normalized_path;
  for (cur = path; cur != NULL; cur = next + 1) {
    next = uvwasi__strchr_slash(cur);
    cur_len = (next == NULL) ? strlen(cur) : (size_t) (next - cur);

    if (cur_len == 0 || (cur_len == 1 && cur[0] == '.'))
      continue;

    if (cur_len == 2 && cur[0] == '.' && cur[1] == '.') {
      while (!IS_SLASH(*ptr) && ptr != normalized_path)
        ptr--;
      *ptr = '\0';
      continue;
    }

    *ptr = '/';
    ptr++;
    memcpy(ptr, cur, cur_len);
    ptr += cur_len;
    *ptr = '\0';

    if (next == NULL)
      break;
  }

  return UVWASI_ESUCCESS;
}


static uvwasi_errno_t uvwasi__resolve_path_to_host(
                                              const uvwasi_t* uvwasi,
                                              const struct uvwasi_fd_wrap_t* fd,
                                              const char* path,
                                              size_t path_len,
                                              char** resolved_path,
                                              size_t* resolved_len
                                            ) {
  /* Return the normalized path, but resolved to the host's real path. */
  int real_path_len;
  int fake_path_len;
#ifdef _WIN32
  size_t i;
#endif /* _WIN32 */

  real_path_len = strlen(fd->real_path);
  fake_path_len = strlen(fd->path);
  *resolved_len = path_len - fake_path_len + real_path_len;
  *resolved_path = uvwasi__malloc(uvwasi, *resolved_len + 1);

  if (*resolved_path == NULL)
    return UVWASI_ENOMEM;

  memcpy(*resolved_path, fd->real_path, real_path_len);
  memcpy(*resolved_path + real_path_len,
         path + fake_path_len,
         path_len - fake_path_len + 1);

#ifdef _WIN32
  /* Replace / with \ on Windows. */
  for (i = real_path_len; i < *resolved_len; i++) {
    if ((*resolved_path)[i] == '/')
      (*resolved_path)[i] = '\\';
  }
#endif /* _WIN32 */

  return UVWASI_ESUCCESS;
}


static uvwasi_errno_t uvwasi__normalize_absolute_path(
                                              const uvwasi_t* uvwasi,
                                              const struct uvwasi_fd_wrap_t* fd,
                                              const char* path,
                                              size_t path_len,
                                              char** normalized_path,
                                              size_t* normalized_len
                                            ) {
  uvwasi_errno_t err;
  char* abs_path;
  int abs_size;

  *normalized_path = NULL;
  *normalized_len = 0;
  abs_size = path_len + 1;
  abs_path = uvwasi__malloc(uvwasi, abs_size);
  if (abs_path == NULL) {
    err = UVWASI_ENOMEM;
    goto exit;
  }

  /* Normalize the input path first. */
  err = uvwasi__normalize_path(path,
                               path_len,
                               abs_path,
                               path_len);
  if (err != UVWASI_ESUCCESS)
    goto exit;

  /* Once the input is normalized, ensure that it is still sandboxed. */
  if (abs_path != strstr(abs_path, fd->path)) {
    err = UVWASI_ENOTCAPABLE;
    goto exit;
  }

  *normalized_path = abs_path;
  *normalized_len = abs_size - 1;
  return UVWASI_ESUCCESS;

exit:
  uvwasi__free(uvwasi, abs_path);
  return err;
}


static uvwasi_errno_t uvwasi__normalize_relative_path(
                                              const uvwasi_t* uvwasi,
                                              const struct uvwasi_fd_wrap_t* fd,
                                              const char* path,
                                              size_t path_len,
                                              char** normalized_path,
                                              size_t* normalized_len
                                            ) {
  uvwasi_errno_t err;
  char* abs_path;
  int abs_size;
  int r;

  *normalized_path = NULL;
  *normalized_len = 0;
  abs_size = path_len + strlen(fd->path) + 2;
  abs_path = uvwasi__malloc(uvwasi, abs_size);
  if (abs_path == NULL) {
    err = UVWASI_ENOMEM;
    goto exit;
  }

  /* Resolve the relative path to an absolute path based on fd's fake path. */
  r = snprintf(abs_path, abs_size, "%s/%s", fd->path, path);
  if (r <= 0) {
    err = uvwasi__translate_uv_error(uv_translate_sys_error(errno));
    goto exit;
  }

  err = uvwasi__normalize_absolute_path(uvwasi,
                                        fd,
                                        abs_path,
                                        abs_size - 1,
                                        normalized_path,
                                        normalized_len);
exit:
  uvwasi__free(uvwasi, abs_path);
  return err;
}


static uvwasi_errno_t uvwasi__resolve_path(const uvwasi_t* uvwasi,
                                           const struct uvwasi_fd_wrap_t* fd,
                                           const char* path,
                                           size_t path_len,
                                           char* resolved_path,
                                           uvwasi_lookupflags_t flags) {
  uv_fs_t req;
  uvwasi_errno_t err;
  const char* input;
  char* host_path;
  char* normalized_path;
  char* link_target;
  size_t input_len;
  size_t host_path_len;
  size_t normalized_len;
  int follow_count;
  int r;

  input = path;
  input_len = path_len;
  link_target = NULL;
  follow_count = 0;
  host_path = NULL;

start:
  normalized_path = NULL;
  err = UVWASI_ESUCCESS;

  if (1 == uvwasi__is_absolute_path(input, input_len)) {
    err = uvwasi__normalize_absolute_path(uvwasi,
                                          fd,
                                          input,
                                          input_len,
                                          &normalized_path,
                                          &normalized_len);
  } else {
    err = uvwasi__normalize_relative_path(uvwasi,
                                          fd,
                                          input,
                                          input_len,
                                          &normalized_path,
                                          &normalized_len);
  }

  if (err != UVWASI_ESUCCESS)
    goto exit;

  uvwasi__free(uvwasi, host_path);
  err = uvwasi__resolve_path_to_host(uvwasi,
                                     fd,
                                     normalized_path,
                                     normalized_len,
                                     &host_path,
                                     &host_path_len);
  if (err != UVWASI_ESUCCESS)
    goto exit;

  /* TODO(cjihrig): Currently performing a bounds check here. The TODO is to
     stop allocating resolved_path in every caller and instead return the
     path allocated in this function. */
  if (host_path_len > PATH_MAX_BYTES) {
    err = UVWASI_ENOBUFS;
    goto exit;
  }

  if ((flags & UVWASI_LOOKUP_SYMLINK_FOLLOW) == UVWASI_LOOKUP_SYMLINK_FOLLOW) {
    r = uv_fs_readlink(NULL, &req, host_path, NULL);

    if (r != 0) {
#ifdef _WIN32
      /* uv_fs_readlink() returns UV__UNKNOWN on Windows. Try to get a better
         error using uv_fs_stat(). */
      if (r == UV__UNKNOWN) {
        uv_fs_req_cleanup(&req);
        r = uv_fs_stat(NULL, &req, host_path, NULL);

        if (r == 0) {
          if (uvwasi__stat_to_filetype(&req.statbuf) !=
              UVWASI_FILETYPE_SYMBOLIC_LINK) {
            r = UV_EINVAL;
          }
        }

        // Fall through.
      }
#endif /* _WIN32 */

      /* Don't report UV_EINVAL or UV_ENOENT. They mean that either the file
          does not exist, or it is not a symlink. Both are OK. */
      if (r != UV_EINVAL && r != UV_ENOENT)
        err = uvwasi__translate_uv_error(r);

      uv_fs_req_cleanup(&req);
      goto exit;
    }

    /* Clean up memory and follow the link, unless it's time to return ELOOP. */
    follow_count++;
    if (follow_count >= UVWASI__MAX_SYMLINK_FOLLOWS) {
      uv_fs_req_cleanup(&req);
      err = UVWASI_ELOOP;
      goto exit;
    }

    input_len = strlen(req.ptr);
    uvwasi__free(uvwasi, link_target);
    link_target = uvwasi__malloc(uvwasi, input_len + 1);
    if (link_target == NULL) {
      uv_fs_req_cleanup(&req);
      err = UVWASI_ENOMEM;
      goto exit;
    }

    memcpy(link_target, req.ptr, input_len + 1);
    input = link_target;
    uvwasi__free(uvwasi, normalized_path);
    uv_fs_req_cleanup(&req);
    goto start;
  }

exit:
  if (err == UVWASI_ESUCCESS)
    memcpy(resolved_path, host_path, host_path_len + 1);

  uvwasi__free(uvwasi, link_target);
  uvwasi__free(uvwasi, normalized_path);
  uvwasi__free(uvwasi, host_path);
  return err;
}


static uvwasi_errno_t uvwasi__lseek(uv_file fd,
                                    uvwasi_filedelta_t offset,
                                    uvwasi_whence_t whence,
                                    uvwasi_filesize_t* newoffset) {
  int real_whence;

  if (whence == UVWASI_WHENCE_CUR)
    real_whence = SEEK_CUR;
  else if (whence == UVWASI_WHENCE_END)
    real_whence = SEEK_END;
  else if (whence == UVWASI_WHENCE_SET)
    real_whence = SEEK_SET;
  else
    return UVWASI_EINVAL;

#ifdef _WIN32
  int64_t r;

  r = _lseeki64(fd, offset, real_whence);
  if (-1L == r)
    return uvwasi__translate_uv_error(uv_translate_sys_error(errno));
#else
  off_t r;

  r = lseek(fd, offset, real_whence);
  if ((off_t) -1 == r)
    return uvwasi__translate_uv_error(uv_translate_sys_error(errno));
#endif /* _WIN32 */

  *newoffset = r;
  return UVWASI_ESUCCESS;
}


static uvwasi_errno_t uvwasi__setup_iovs(const uvwasi_t* uvwasi,
                                         uv_buf_t** buffers,
                                         const uvwasi_iovec_t* iovs,
                                         size_t iovs_len) {
  uv_buf_t* bufs;
  size_t i;

  if ((iovs_len * sizeof(*bufs)) / (sizeof(*bufs)) != iovs_len)
    return UVWASI_ENOMEM;

  bufs = uvwasi__malloc(uvwasi, iovs_len * sizeof(*bufs));
  if (bufs == NULL)
    return UVWASI_ENOMEM;

  for (i = 0; i < iovs_len; ++i)
    bufs[i] = uv_buf_init(iovs[i].buf, iovs[i].buf_len);

  *buffers = bufs;
  return UVWASI_ESUCCESS;
}


static uvwasi_errno_t uvwasi__setup_ciovs(const uvwasi_t* uvwasi,
                                          uv_buf_t** buffers,
                                          const uvwasi_ciovec_t* iovs,
                                          size_t iovs_len) {
  uv_buf_t* bufs;
  size_t i;

  if ((iovs_len * sizeof(*bufs)) / (sizeof(*bufs)) != iovs_len)
    return UVWASI_ENOMEM;

  bufs = uvwasi__malloc(uvwasi, iovs_len * sizeof(*bufs));
  if (bufs == NULL)
    return UVWASI_ENOMEM;

  for (i = 0; i < iovs_len; ++i)
    bufs[i] = uv_buf_init((char*)iovs[i].buf, iovs[i].buf_len);

  *buffers = bufs;
  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_init(uvwasi_t* uvwasi, uvwasi_options_t* options) {
  uv_fs_t realpath_req;
  uv_fs_t open_req;
  uvwasi_errno_t err;
  size_t args_size;
  size_t size;
  size_t offset;
  size_t env_count;
  size_t env_buf_size;
  size_t i;
  int r;

  if (uvwasi == NULL || options == NULL || options->fd_table_size == 0)
    return UVWASI_EINVAL;

  uvwasi->allocator = options->allocator;
  if (uvwasi->allocator == NULL)
    uvwasi->allocator = &default_allocator;

  uvwasi->argv_buf = NULL;
  uvwasi->argv = NULL;
  uvwasi->env_buf = NULL;
  uvwasi->env = NULL;
  uvwasi->fds.fds = NULL;

  args_size = 0;
  for (i = 0; i < options->argc; ++i)
    args_size += strlen(options->argv[i]) + 1;

  uvwasi->argc = options->argc;
  uvwasi->argv_buf_size = args_size;

  if (args_size > 0) {
    uvwasi->argv_buf = uvwasi__malloc(uvwasi, args_size);
    if (uvwasi->argv_buf == NULL) {
      err = UVWASI_ENOMEM;
      goto exit;
    }

    uvwasi->argv = uvwasi__calloc(uvwasi, options->argc, sizeof(char*));
    if (uvwasi->argv == NULL) {
      err = UVWASI_ENOMEM;
      goto exit;
    }

    offset = 0;
    for (i = 0; i < options->argc; ++i) {
      size = strlen(options->argv[i]) + 1;
      memcpy(uvwasi->argv_buf + offset, options->argv[i], size);
      uvwasi->argv[i] = uvwasi->argv_buf + offset;
      offset += size;
    }
  }

  env_count = 0;
  env_buf_size = 0;
  if (options->envp != NULL) {
    while (options->envp[env_count] != NULL) {
      env_buf_size += strlen(options->envp[env_count]) + 1;
      env_count++;
    }
  }

  uvwasi->envc = env_count;
  uvwasi->env_buf_size = env_buf_size;

  if (env_buf_size > 0) {
    uvwasi->env_buf = uvwasi__malloc(uvwasi, env_buf_size);
    if (uvwasi->env_buf == NULL) {
      err = UVWASI_ENOMEM;
      goto exit;
    }

    uvwasi->env = uvwasi__calloc(uvwasi, env_count, sizeof(char*));
    if (uvwasi->env == NULL) {
      err = UVWASI_ENOMEM;
      goto exit;
    }

    offset = 0;
    for (i = 0; i < env_count; ++i) {
      size = strlen(options->envp[i]) + 1;
      memcpy(uvwasi->env_buf + offset, options->envp[i], size);
      uvwasi->env[i] = uvwasi->env_buf + offset;
      offset += size;
    }
  }

  for (i = 0; i < options->preopenc; ++i) {
    if (options->preopens[i].real_path == NULL ||
        options->preopens[i].mapped_path == NULL) {
      err = UVWASI_EINVAL;
      goto exit;
    }
  }

  err = uvwasi_fd_table_init(uvwasi, options);
  if (err != UVWASI_ESUCCESS)
    goto exit;

  for (i = 0; i < options->preopenc; ++i) {
    r = uv_fs_realpath(NULL,
                       &realpath_req,
                       options->preopens[i].real_path,
                       NULL);
    if (r != 0) {
      err = uvwasi__translate_uv_error(r);
      uv_fs_req_cleanup(&realpath_req);
      goto exit;
    }

    r = uv_fs_open(NULL, &open_req, realpath_req.ptr, 0, 0666, NULL);
    if (r < 0) {
      err = uvwasi__translate_uv_error(r);
      uv_fs_req_cleanup(&realpath_req);
      uv_fs_req_cleanup(&open_req);
      goto exit;
    }

    err = uvwasi_fd_table_insert_preopen(uvwasi,
                                         &uvwasi->fds,
                                         open_req.result,
                                         options->preopens[i].mapped_path,
                                         realpath_req.ptr);
    uv_fs_req_cleanup(&realpath_req);
    uv_fs_req_cleanup(&open_req);

    if (err != UVWASI_ESUCCESS)
      goto exit;
  }

  return UVWASI_ESUCCESS;

exit:
  uvwasi_destroy(uvwasi);
  return err;
}


void uvwasi_destroy(uvwasi_t* uvwasi) {
  if (uvwasi == NULL)
    return;

  uvwasi_fd_table_free(uvwasi, &uvwasi->fds);
  uvwasi__free(uvwasi, uvwasi->argv_buf);
  uvwasi__free(uvwasi, uvwasi->argv);
  uvwasi__free(uvwasi, uvwasi->env_buf);
  uvwasi__free(uvwasi, uvwasi->env);
  uvwasi->argv_buf = NULL;
  uvwasi->argv = NULL;
  uvwasi->env_buf = NULL;
  uvwasi->env = NULL;
}


uvwasi_errno_t uvwasi_embedder_remap_fd(uvwasi_t* uvwasi,
                                        const uvwasi_fd_t fd,
                                        uv_file new_host_fd) {
  struct uvwasi_fd_wrap_t* wrap;
  uvwasi_errno_t err;

  if (uvwasi == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds, fd, &wrap, 0, 0);
  if (err != UVWASI_ESUCCESS)
    return err;

  wrap->fd = new_host_fd;
  uv_mutex_unlock(&wrap->mutex);
  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_args_get(uvwasi_t* uvwasi, char** argv, char* argv_buf) {
  size_t i;

  if (uvwasi == NULL || argv == NULL || argv_buf == NULL)
    return UVWASI_EINVAL;

  for (i = 0; i < uvwasi->argc; ++i) {
    argv[i] = argv_buf + (uvwasi->argv[i] - uvwasi->argv_buf);
  }

  memcpy(argv_buf, uvwasi->argv_buf, uvwasi->argv_buf_size);
  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_args_sizes_get(uvwasi_t* uvwasi,
                                     size_t* argc,
                                     size_t* argv_buf_size) {
  if (uvwasi == NULL || argc == NULL || argv_buf_size == NULL)
    return UVWASI_EINVAL;

  *argc = uvwasi->argc;
  *argv_buf_size = uvwasi->argv_buf_size;
  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_clock_res_get(uvwasi_t* uvwasi,
                                    uvwasi_clockid_t clock_id,
                                    uvwasi_timestamp_t* resolution) {
  if (uvwasi == NULL || resolution == NULL)
    return UVWASI_EINVAL;

  switch (clock_id) {
    case UVWASI_CLOCK_MONOTONIC:
    case UVWASI_CLOCK_REALTIME:
      *resolution = 1;  /* Nanosecond precision. */
      return UVWASI_ESUCCESS;
    case UVWASI_CLOCK_PROCESS_CPUTIME_ID:
      return uvwasi__clock_getres_process_cputime(resolution);
    case UVWASI_CLOCK_THREAD_CPUTIME_ID:
      return uvwasi__clock_getres_thread_cputime(resolution);
    default:
      return UVWASI_EINVAL;
  }
}


uvwasi_errno_t uvwasi_clock_time_get(uvwasi_t* uvwasi,
                                     uvwasi_clockid_t clock_id,
                                     uvwasi_timestamp_t precision,
                                     uvwasi_timestamp_t* time) {
  if (uvwasi == NULL || time == NULL)
    return UVWASI_EINVAL;

  switch (clock_id) {
    case UVWASI_CLOCK_MONOTONIC:
      *time = uv_hrtime();
      return UVWASI_ESUCCESS;
    case UVWASI_CLOCK_REALTIME:
      return uvwasi__clock_gettime_realtime(time);
    case UVWASI_CLOCK_PROCESS_CPUTIME_ID:
      return uvwasi__clock_gettime_process_cputime(time);
    case UVWASI_CLOCK_THREAD_CPUTIME_ID:
      return uvwasi__clock_gettime_thread_cputime(time);
    default:
      return UVWASI_EINVAL;
  }
}


uvwasi_errno_t uvwasi_environ_get(uvwasi_t* uvwasi,
                                  char** environment,
                                  char* environ_buf) {
  size_t i;

  if (uvwasi == NULL || environment == NULL || environ_buf == NULL)
    return UVWASI_EINVAL;

  for (i = 0; i < uvwasi->envc; ++i) {
    environment[i] = environ_buf + (uvwasi->env[i] - uvwasi->env_buf);
  }

  memcpy(environ_buf, uvwasi->env_buf, uvwasi->env_buf_size);
  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_environ_sizes_get(uvwasi_t* uvwasi,
                                        size_t* environ_count,
                                        size_t* environ_buf_size) {
  if (uvwasi == NULL || environ_count == NULL || environ_buf_size == NULL)
    return UVWASI_EINVAL;

  *environ_count = uvwasi->envc;
  *environ_buf_size = uvwasi->env_buf_size;
  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_fd_advise(uvwasi_t* uvwasi,
                                uvwasi_fd_t fd,
                                uvwasi_filesize_t offset,
                                uvwasi_filesize_t len,
                                uvwasi_advice_t advice) {
  struct uvwasi_fd_wrap_t* wrap;
  uvwasi_errno_t err;
#ifdef POSIX_FADV_NORMAL
  int mapped_advice;
  int r;
#endif /* POSIX_FADV_NORMAL */

  if (uvwasi == NULL)
    return UVWASI_EINVAL;

  switch (advice) {
    case UVWASI_ADVICE_DONTNEED:
#ifdef POSIX_FADV_NORMAL
      mapped_advice = POSIX_FADV_DONTNEED;
#endif /* POSIX_FADV_NORMAL */
      break;
    case UVWASI_ADVICE_NOREUSE:
#ifdef POSIX_FADV_NORMAL
      mapped_advice = POSIX_FADV_NOREUSE;
#endif /* POSIX_FADV_NORMAL */
      break;
    case UVWASI_ADVICE_NORMAL:
#ifdef POSIX_FADV_NORMAL
      mapped_advice = POSIX_FADV_NORMAL;
#endif /* POSIX_FADV_NORMAL */
      break;
    case UVWASI_ADVICE_RANDOM:
#ifdef POSIX_FADV_NORMAL
      mapped_advice = POSIX_FADV_RANDOM;
#endif /* POSIX_FADV_NORMAL */
      break;
    case UVWASI_ADVICE_SEQUENTIAL:
#ifdef POSIX_FADV_NORMAL
      mapped_advice = POSIX_FADV_SEQUENTIAL;
#endif /* POSIX_FADV_NORMAL */
      break;
    case UVWASI_ADVICE_WILLNEED:
#ifdef POSIX_FADV_NORMAL
      mapped_advice = POSIX_FADV_WILLNEED;
#endif /* POSIX_FADV_NORMAL */
      break;
    default:
      return UVWASI_EINVAL;
  }

  err = uvwasi_fd_table_get(&uvwasi->fds, fd, &wrap, UVWASI_RIGHT_FD_ADVISE, 0);
  if (err != UVWASI_ESUCCESS)
    return err;

  err = UVWASI_ESUCCESS;

#ifdef POSIX_FADV_NORMAL
  r = posix_fadvise(wrap->fd, offset, len, mapped_advice);
  if (r != 0)
    err = uvwasi__translate_uv_error(uv_translate_sys_error(r));
#endif /* POSIX_FADV_NORMAL */
  uv_mutex_unlock(&wrap->mutex);
  return err;
}


uvwasi_errno_t uvwasi_fd_allocate(uvwasi_t* uvwasi,
                                  uvwasi_fd_t fd,
                                  uvwasi_filesize_t offset,
                                  uvwasi_filesize_t len) {
#if !defined(__POSIX__)
  uv_fs_t req;
  uint64_t st_size;
#endif /* !__POSIX__ */
  struct uvwasi_fd_wrap_t* wrap;
  uvwasi_errno_t err;
  int r;

  if (uvwasi == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds,
                            fd,
                            &wrap,
                            UVWASI_RIGHT_FD_ALLOCATE,
                            0);
  if (err != UVWASI_ESUCCESS)
    return err;

  /* Try to use posix_fallocate(). If that's not an option, fall back to the
     race condition prone combination of fstat() + ftruncate(). */
#if defined(__POSIX__)
  r = posix_fallocate(wrap->fd, offset, len);
  if (r != 0) {
    err = uvwasi__translate_uv_error(uv_translate_sys_error(r));
    goto exit;
  }
#else
  r = uv_fs_fstat(NULL, &req, wrap->fd, NULL);
  st_size = req.statbuf.st_size;
  uv_fs_req_cleanup(&req);
  if (r != 0) {
    err = uvwasi__translate_uv_error(r);
    goto exit;
  }

  if (st_size < offset + len) {
    r = uv_fs_ftruncate(NULL, &req, wrap->fd, offset + len, NULL);
    if (r != 0) {
      err = uvwasi__translate_uv_error(r);
      goto exit;
    }
  }
#endif /* __POSIX__ */

  err = UVWASI_ESUCCESS;
exit:
  uv_mutex_unlock(&wrap->mutex);
  return err;
}


uvwasi_errno_t uvwasi_fd_close(uvwasi_t* uvwasi, uvwasi_fd_t fd) {
  struct uvwasi_fd_wrap_t* wrap;
  uvwasi_errno_t err;
  uv_fs_t req;
  int r;

  if (uvwasi == NULL)
    return UVWASI_EINVAL;

  uvwasi_fd_table_lock(&uvwasi->fds);

  err = uvwasi_fd_table_get_nolock(&uvwasi->fds, fd, &wrap, 0, 0);
  if (err != UVWASI_ESUCCESS)
    goto exit;

  r = uv_fs_close(NULL, &req, wrap->fd, NULL);
  uv_mutex_unlock(&wrap->mutex);
  uv_fs_req_cleanup(&req);

  if (r != 0) {
    err = uvwasi__translate_uv_error(r);
    goto exit;
  }

  err = uvwasi_fd_table_remove_nolock(uvwasi, &uvwasi->fds, fd);

exit:
  uvwasi_fd_table_unlock(&uvwasi->fds);
  return err;
}


uvwasi_errno_t uvwasi_fd_datasync(uvwasi_t* uvwasi, uvwasi_fd_t fd) {
  struct uvwasi_fd_wrap_t* wrap;
  uvwasi_errno_t err;
  uv_fs_t req;
  int r;

  if (uvwasi == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds,
                            fd,
                            &wrap,
                            UVWASI_RIGHT_FD_DATASYNC,
                            0);
  if (err != UVWASI_ESUCCESS)
    return err;

  r = uv_fs_fdatasync(NULL, &req, wrap->fd, NULL);
  uv_mutex_unlock(&wrap->mutex);
  uv_fs_req_cleanup(&req);

  if (r != 0)
    return uvwasi__translate_uv_error(r);

  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_fd_fdstat_get(uvwasi_t* uvwasi,
                                    uvwasi_fd_t fd,
                                    uvwasi_fdstat_t* buf) {
  struct uvwasi_fd_wrap_t* wrap;
  uvwasi_errno_t err;
#ifndef _WIN32
  int r;
#endif

  if (uvwasi == NULL || buf == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds, fd, &wrap, 0, 0);
  if (err != UVWASI_ESUCCESS)
    return err;

  buf->fs_filetype = wrap->type;
  buf->fs_rights_base = wrap->rights_base;
  buf->fs_rights_inheriting = wrap->rights_inheriting;
#ifdef _WIN32
  buf->fs_flags = 0;  /* TODO(cjihrig): Missing Windows support. */
#else
  r = fcntl(wrap->fd, F_GETFL);
  if (r < 0) {
    err = uvwasi__translate_uv_error(uv_translate_sys_error(errno));
    uv_mutex_unlock(&wrap->mutex);
    return err;
  }
  buf->fs_flags = r;
#endif /* _WIN32 */

  uv_mutex_unlock(&wrap->mutex);
  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_fd_fdstat_set_flags(uvwasi_t* uvwasi,
                                          uvwasi_fd_t fd,
                                          uvwasi_fdflags_t flags) {
#ifdef _WIN32
  /* TODO(cjihrig): Missing Windows support. */
  return UVWASI_ENOSYS;
#else
  struct uvwasi_fd_wrap_t* wrap;
  uvwasi_errno_t err;
  int mapped_flags;
  int r;

  if (uvwasi == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds,
                            fd,
                            &wrap,
                            UVWASI_RIGHT_FD_FDSTAT_SET_FLAGS,
                            0);
  if (err != UVWASI_ESUCCESS)
    return err;

  mapped_flags = 0;

  if ((flags & UVWASI_FDFLAG_APPEND) == UVWASI_FDFLAG_APPEND)
    mapped_flags |= O_APPEND;

  if ((flags & UVWASI_FDFLAG_DSYNC) == UVWASI_FDFLAG_DSYNC)
#ifdef O_DSYNC
    mapped_flags |= O_DSYNC;
#else
    mapped_flags |= O_SYNC;
#endif /* O_DSYNC */

  if ((flags & UVWASI_FDFLAG_NONBLOCK) == UVWASI_FDFLAG_NONBLOCK)
    mapped_flags |= O_NONBLOCK;

  if ((flags & UVWASI_FDFLAG_RSYNC) == UVWASI_FDFLAG_RSYNC)
#ifdef O_RSYNC
    mapped_flags |= O_RSYNC;
#else
    mapped_flags |= O_SYNC;
#endif /* O_RSYNC */

  if ((flags & UVWASI_FDFLAG_SYNC) == UVWASI_FDFLAG_SYNC)
    mapped_flags |= O_SYNC;

  r = fcntl(wrap->fd, F_SETFL, mapped_flags);
  if (r < 0)
    err = uvwasi__translate_uv_error(uv_translate_sys_error(errno));
  else
    err = UVWASI_ESUCCESS;

  uv_mutex_unlock(&wrap->mutex);
  return err;
#endif /* _WIN32 */
}


uvwasi_errno_t uvwasi_fd_fdstat_set_rights(uvwasi_t* uvwasi,
                                           uvwasi_fd_t fd,
                                           uvwasi_rights_t fs_rights_base,
                                           uvwasi_rights_t fs_rights_inheriting
                                          ) {
  struct uvwasi_fd_wrap_t* wrap;
  uvwasi_errno_t err;

  if (uvwasi == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds, fd, &wrap, 0, 0);
  if (err != UVWASI_ESUCCESS)
    return err;

  /* Check for attempts to add new permissions. */
  if ((fs_rights_base | wrap->rights_base) > wrap->rights_base) {
    err = UVWASI_ENOTCAPABLE;
    goto exit;
  }

  if ((fs_rights_inheriting | wrap->rights_inheriting) >
      wrap->rights_inheriting) {
    err = UVWASI_ENOTCAPABLE;
    goto exit;
  }

  wrap->rights_base = fs_rights_base;
  wrap->rights_inheriting = fs_rights_inheriting;
  err = UVWASI_ESUCCESS;
exit:
  uv_mutex_unlock(&wrap->mutex);
  return err;
}


uvwasi_errno_t uvwasi_fd_filestat_get(uvwasi_t* uvwasi,
                                      uvwasi_fd_t fd,
                                      uvwasi_filestat_t* buf) {
  struct uvwasi_fd_wrap_t* wrap;
  uv_fs_t req;
  uvwasi_errno_t err;
  int r;

  if (uvwasi == NULL || buf == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds,
                            fd,
                            &wrap,
                            UVWASI_RIGHT_FD_FILESTAT_GET,
                            0);
  if (err != UVWASI_ESUCCESS)
    return err;

  r = uv_fs_fstat(NULL, &req, wrap->fd, NULL);
  if (r != 0) {
    err = uvwasi__translate_uv_error(r);
    goto exit;
  }

  uvwasi__stat_to_filestat(&req.statbuf, buf);
  err = UVWASI_ESUCCESS;
exit:
  uv_mutex_unlock(&wrap->mutex);
  uv_fs_req_cleanup(&req);
  return err;
}


uvwasi_errno_t uvwasi_fd_filestat_set_size(uvwasi_t* uvwasi,
                                           uvwasi_fd_t fd,
                                           uvwasi_filesize_t st_size) {
  /* TODO(cjihrig): uv_fs_ftruncate() takes an int64_t. st_size is uint64_t. */
  struct uvwasi_fd_wrap_t* wrap;
  uv_fs_t req;
  uvwasi_errno_t err;
  int r;

  if (uvwasi == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds,
                            fd,
                            &wrap,
                            UVWASI_RIGHT_FD_FILESTAT_SET_SIZE,
                            0);
  if (err != UVWASI_ESUCCESS)
    return err;

  r = uv_fs_ftruncate(NULL, &req, wrap->fd, st_size, NULL);
  uv_mutex_unlock(&wrap->mutex);
  uv_fs_req_cleanup(&req);

  if (r != 0)
    return uvwasi__translate_uv_error(r);

  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_fd_filestat_set_times(uvwasi_t* uvwasi,
                                            uvwasi_fd_t fd,
                                            uvwasi_timestamp_t st_atim,
                                            uvwasi_timestamp_t st_mtim,
                                            uvwasi_fstflags_t fst_flags) {
  /* TODO(cjihrig): libuv does not currently support nanosecond precision. */
  struct uvwasi_fd_wrap_t* wrap;
  uv_fs_t req;
  uvwasi_errno_t err;
  int r;

  if (uvwasi == NULL)
    return UVWASI_EINVAL;

  if (fst_flags & ~(UVWASI_FILESTAT_SET_ATIM | UVWASI_FILESTAT_SET_ATIM_NOW |
                    UVWASI_FILESTAT_SET_MTIM | UVWASI_FILESTAT_SET_MTIM_NOW)) {
    return UVWASI_EINVAL;
  }

  err = uvwasi_fd_table_get(&uvwasi->fds,
                            fd,
                            &wrap,
                            UVWASI_RIGHT_FD_FILESTAT_SET_TIMES,
                            0);
  if (err != UVWASI_ESUCCESS)
    return err;

  /* TODO(cjihrig): st_atim and st_mtim should not be unconditionally passed. */
  r = uv_fs_futime(NULL, &req, wrap->fd, st_atim, st_mtim, NULL);
  uv_mutex_unlock(&wrap->mutex);
  uv_fs_req_cleanup(&req);

  if (r != 0)
    return uvwasi__translate_uv_error(r);

  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_fd_pread(uvwasi_t* uvwasi,
                               uvwasi_fd_t fd,
                               const uvwasi_iovec_t* iovs,
                               size_t iovs_len,
                               uvwasi_filesize_t offset,
                               size_t* nread) {
  struct uvwasi_fd_wrap_t* wrap;
  uv_buf_t* bufs;
  uv_fs_t req;
  uvwasi_errno_t err;
  size_t uvread;
  int r;

  if (uvwasi == NULL || iovs == NULL || nread == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds,
                            fd,
                            &wrap,
                            UVWASI_RIGHT_FD_READ | UVWASI_RIGHT_FD_SEEK,
                            0);
  if (err != UVWASI_ESUCCESS)
    return err;

  err = uvwasi__setup_iovs(uvwasi, &bufs, iovs, iovs_len);
  if (err != UVWASI_ESUCCESS) {
    uv_mutex_unlock(&wrap->mutex);
    return err;
  }

  r = uv_fs_read(NULL, &req, wrap->fd, bufs, iovs_len, offset, NULL);
  uv_mutex_unlock(&wrap->mutex);
  uvread = req.result;
  uv_fs_req_cleanup(&req);
  uvwasi__free(uvwasi, bufs);

  if (r < 0)
    return uvwasi__translate_uv_error(r);

  *nread = uvread;
  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_fd_prestat_get(uvwasi_t* uvwasi,
                                     uvwasi_fd_t fd,
                                     uvwasi_prestat_t* buf) {
  struct uvwasi_fd_wrap_t* wrap;
  uvwasi_errno_t err;

  if (uvwasi == NULL || buf == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds, fd, &wrap, 0, 0);
  if (err != UVWASI_ESUCCESS)
    return err;
  if (wrap->preopen != 1) {
    err = UVWASI_EINVAL;
    goto exit;
  }

  buf->pr_type = UVWASI_PREOPENTYPE_DIR;
  buf->u.dir.pr_name_len = strlen(wrap->path) + 1;
  err = UVWASI_ESUCCESS;
exit:
  uv_mutex_unlock(&wrap->mutex);
  return err;
}


uvwasi_errno_t uvwasi_fd_prestat_dir_name(uvwasi_t* uvwasi,
                                          uvwasi_fd_t fd,
                                          char* path,
                                          size_t path_len) {
  struct uvwasi_fd_wrap_t* wrap;
  uvwasi_errno_t err;
  size_t size;

  if (uvwasi == NULL || path == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds, fd, &wrap, 0, 0);
  if (err != UVWASI_ESUCCESS)
    return err;
  if (wrap->preopen != 1) {
    err = UVWASI_EBADF;
    goto exit;
  }

  size = strlen(wrap->path) + 1;
  if (size > path_len) {
    err = UVWASI_ENOBUFS;
    goto exit;
  }

  memcpy(path, wrap->path, size);
  err = UVWASI_ESUCCESS;
exit:
  uv_mutex_unlock(&wrap->mutex);
  return err;
}


uvwasi_errno_t uvwasi_fd_pwrite(uvwasi_t* uvwasi,
                                uvwasi_fd_t fd,
                                const uvwasi_ciovec_t* iovs,
                                size_t iovs_len,
                                uvwasi_filesize_t offset,
                                size_t* nwritten) {
  struct uvwasi_fd_wrap_t* wrap;
  uv_buf_t* bufs;
  uv_fs_t req;
  uvwasi_errno_t err;
  size_t uvwritten;
  int r;

  if (uvwasi == NULL || iovs == NULL || nwritten == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds,
                            fd,
                            &wrap,
                            UVWASI_RIGHT_FD_WRITE | UVWASI_RIGHT_FD_SEEK,
                            0);
  if (err != UVWASI_ESUCCESS)
    return err;

  err = uvwasi__setup_ciovs(uvwasi, &bufs, iovs, iovs_len);
  if (err != UVWASI_ESUCCESS) {
    uv_mutex_unlock(&wrap->mutex);
    return err;
  }

  r = uv_fs_write(NULL, &req, wrap->fd, bufs, iovs_len, offset, NULL);
  uv_mutex_unlock(&wrap->mutex);
  uvwritten = req.result;
  uv_fs_req_cleanup(&req);
  uvwasi__free(uvwasi, bufs);

  if (r < 0)
    return uvwasi__translate_uv_error(r);

  *nwritten = uvwritten;
  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_fd_read(uvwasi_t* uvwasi,
                              uvwasi_fd_t fd,
                              const uvwasi_iovec_t* iovs,
                              size_t iovs_len,
                              size_t* nread) {
  struct uvwasi_fd_wrap_t* wrap;
  uv_buf_t* bufs;
  uv_fs_t req;
  uvwasi_errno_t err;
  size_t uvread;
  int r;

  if (uvwasi == NULL || iovs == NULL || nread == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds, fd, &wrap, UVWASI_RIGHT_FD_READ, 0);
  if (err != UVWASI_ESUCCESS)
    return err;

  err = uvwasi__setup_iovs(uvwasi, &bufs, iovs, iovs_len);
  if (err != UVWASI_ESUCCESS) {
    uv_mutex_unlock(&wrap->mutex);
    return err;
  }

  r = uv_fs_read(NULL, &req, wrap->fd, bufs, iovs_len, -1, NULL);
  uv_mutex_unlock(&wrap->mutex);
  uvread = req.result;
  uv_fs_req_cleanup(&req);
  uvwasi__free(uvwasi, bufs);

  if (r < 0)
    return uvwasi__translate_uv_error(r);

  *nread = uvread;
  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_fd_readdir(uvwasi_t* uvwasi,
                                 uvwasi_fd_t fd,
                                 void* buf,
                                 size_t buf_len,
                                 uvwasi_dircookie_t cookie,
                                 size_t* bufused) {
  /* TODO(cjihrig): Support Windows where seekdir() and telldir() are used. */
  /* TODO(cjihrig): Avoid opening and closing the directory on each call. */
  struct uvwasi_fd_wrap_t* wrap;
  uvwasi_dirent_t dirent;
  uv_dirent_t dirents[UVWASI__READDIR_NUM_ENTRIES];
  uv_dir_t* dir;
  uv_fs_t req;
  uvwasi_errno_t err;
  size_t name_len;
  size_t available;
  size_t size_to_cp;
  long tell;
  int i;
  int r;

  if (uvwasi == NULL || buf == NULL || bufused == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds,
                            fd,
                            &wrap,
                            UVWASI_RIGHT_FD_READDIR,
                            0);
  if (err != UVWASI_ESUCCESS)
    return err;

  /* Open the directory. */
  r = uv_fs_opendir(NULL, &req, wrap->real_path, NULL);
  if (r != 0) {
    uv_mutex_unlock(&wrap->mutex);
    return uvwasi__translate_uv_error(r);
  }

  /* Setup for reading the directory. */
  dir = req.ptr;
  dir->dirents = dirents;
  dir->nentries = UVWASI__READDIR_NUM_ENTRIES;
  uv_fs_req_cleanup(&req);

#ifndef _WIN32
  /* TODO(cjihrig): Need a Windows equivalent of this logic. */
  /* Seek to the proper location in the directory. */
  if (cookie != UVWASI_DIRCOOKIE_START)
    seekdir(dir->dir, cookie);
#endif

  /* Read the directory entries into the provided buffer. */
  err = UVWASI_ESUCCESS;
  *bufused = 0;
  while (0 != (r = uv_fs_readdir(NULL, &req, dir, NULL))) {
    if (r < 0) {
      err = uvwasi__translate_uv_error(r);
      uv_fs_req_cleanup(&req);
      goto exit;
    }

    for (i = 0; i < r; i++) {
      /* TODO(cjihrig): This should probably be serialized to the buffer
         consistently across platforms. In other words, d_next should always
         be 8 bytes, d_ino should always be 8 bytes, d_namlen should always be
         4 bytes, and d_type should always be 1 byte. */
#ifndef _WIN32
      tell = telldir(dir->dir);
      if (tell < 0) {
        err = uvwasi__translate_uv_error(uv_translate_sys_error(errno));
        uv_fs_req_cleanup(&req);
        goto exit;
      }
#else
      tell = 0; /* TODO(cjihrig): Need to support Windows. */
#endif /* _WIN32 */

      name_len = strlen(dirents[i].name);
      dirent.d_next = (uvwasi_dircookie_t) tell;
      /* TODO(cjihrig): Missing ino libuv (and Windows) support. fstat()? */
      dirent.d_ino = 0;
      dirent.d_namlen = name_len;

      switch (dirents[i].type) {
        case UV_DIRENT_FILE:
          dirent.d_type = UVWASI_FILETYPE_REGULAR_FILE;
          break;
        case UV_DIRENT_DIR:
          dirent.d_type = UVWASI_FILETYPE_DIRECTORY;
          break;
        case UV_DIRENT_SOCKET:
          dirent.d_type = UVWASI_FILETYPE_SOCKET_STREAM;
          break;
        case UV_DIRENT_LINK:
          dirent.d_type = UVWASI_FILETYPE_SYMBOLIC_LINK;
          break;
        case UV_DIRENT_CHAR:
          dirent.d_type = UVWASI_FILETYPE_CHARACTER_DEVICE;
          break;
        case UV_DIRENT_BLOCK:
          dirent.d_type = UVWASI_FILETYPE_BLOCK_DEVICE;
          break;
        case UV_DIRENT_FIFO:
        case UV_DIRENT_UNKNOWN:
        default:
          dirent.d_type = UVWASI_FILETYPE_UNKNOWN;
          break;
      }

      /* Write dirent to the buffer. */
      available = buf_len - *bufused;
      size_to_cp = sizeof(dirent) > available ? available : sizeof(dirent);
      memcpy((char*)buf + *bufused, &dirent, size_to_cp);
      *bufused += size_to_cp;
      /* Write the entry name to the buffer. */
      available = buf_len - *bufused;
      size_to_cp = name_len > available ? available : name_len;
      memcpy((char*)buf + *bufused, &dirents[i].name, size_to_cp);
      *bufused += size_to_cp;
    }

    uv_fs_req_cleanup(&req);

    if (*bufused >= buf_len)
      break;
  }

exit:
  /* Close the directory. */
  r = uv_fs_closedir(NULL, &req, dir, NULL);
  uv_mutex_unlock(&wrap->mutex);
  uv_fs_req_cleanup(&req);
  if (r != 0)
    return uvwasi__translate_uv_error(r);

  return err;
}


uvwasi_errno_t uvwasi_fd_renumber(uvwasi_t* uvwasi,
                                  uvwasi_fd_t from,
                                  uvwasi_fd_t to) {
  if (uvwasi == NULL)
    return UVWASI_EINVAL;

  return uvwasi_fd_table_renumber(uvwasi, &uvwasi->fds, to, from);
}


uvwasi_errno_t uvwasi_fd_seek(uvwasi_t* uvwasi,
                              uvwasi_fd_t fd,
                              uvwasi_filedelta_t offset,
                              uvwasi_whence_t whence,
                              uvwasi_filesize_t* newoffset) {
  struct uvwasi_fd_wrap_t* wrap;
  uvwasi_errno_t err;

  if (uvwasi == NULL || newoffset == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds, fd, &wrap, UVWASI_RIGHT_FD_SEEK, 0);
  if (err != UVWASI_ESUCCESS)
    return err;

  err = uvwasi__lseek(wrap->fd, offset, whence, newoffset);
  uv_mutex_unlock(&wrap->mutex);
  return err;
}


uvwasi_errno_t uvwasi_fd_sync(uvwasi_t* uvwasi, uvwasi_fd_t fd) {
  struct uvwasi_fd_wrap_t* wrap;
  uv_fs_t req;
  uvwasi_errno_t err;
  int r;

  if (uvwasi == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds,
                            fd,
                            &wrap,
                            UVWASI_RIGHT_FD_SYNC,
                            0);
  if (err != UVWASI_ESUCCESS)
    return err;

  r = uv_fs_fsync(NULL, &req, wrap->fd, NULL);
  uv_mutex_unlock(&wrap->mutex);
  uv_fs_req_cleanup(&req);

  if (r != 0)
    return uvwasi__translate_uv_error(r);

  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_fd_tell(uvwasi_t* uvwasi,
                              uvwasi_fd_t fd,
                              uvwasi_filesize_t* offset) {
  struct uvwasi_fd_wrap_t* wrap;
  uvwasi_errno_t err;

  if (uvwasi == NULL || offset == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds, fd, &wrap, UVWASI_RIGHT_FD_TELL, 0);
  if (err != UVWASI_ESUCCESS)
    return err;

  err = uvwasi__lseek(wrap->fd, 0, UVWASI_WHENCE_CUR, offset);
  uv_mutex_unlock(&wrap->mutex);
  return err;
}


uvwasi_errno_t uvwasi_fd_write(uvwasi_t* uvwasi,
                               uvwasi_fd_t fd,
                               const uvwasi_ciovec_t* iovs,
                               size_t iovs_len,
                               size_t* nwritten) {
  struct uvwasi_fd_wrap_t* wrap;
  uv_buf_t* bufs;
  uv_fs_t req;
  uvwasi_errno_t err;
  size_t uvwritten;
  int r;

  if (uvwasi == NULL || iovs == NULL || nwritten == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds, fd, &wrap, UVWASI_RIGHT_FD_WRITE, 0);
  if (err != UVWASI_ESUCCESS)
    return err;

  err = uvwasi__setup_ciovs(uvwasi, &bufs, iovs, iovs_len);
  if (err != UVWASI_ESUCCESS) {
    uv_mutex_unlock(&wrap->mutex);
    return err;
  }

  r = uv_fs_write(NULL, &req, wrap->fd, bufs, iovs_len, -1, NULL);
  uv_mutex_unlock(&wrap->mutex);
  uvwritten = req.result;
  uv_fs_req_cleanup(&req);
  uvwasi__free(uvwasi, bufs);

  if (r < 0)
    return uvwasi__translate_uv_error(r);

  *nwritten = uvwritten;
  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_path_create_directory(uvwasi_t* uvwasi,
                                            uvwasi_fd_t fd,
                                            const char* path,
                                            size_t path_len) {
  char resolved_path[PATH_MAX_BYTES];
  struct uvwasi_fd_wrap_t* wrap;
  uv_fs_t req;
  uvwasi_errno_t err;
  int r;

  if (uvwasi == NULL || path == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds,
                            fd,
                            &wrap,
                            UVWASI_RIGHT_PATH_CREATE_DIRECTORY,
                            0);
  if (err != UVWASI_ESUCCESS)
    return err;

  err = uvwasi__resolve_path(uvwasi, wrap, path, path_len, resolved_path, 0);
  if (err != UVWASI_ESUCCESS)
    goto exit;

  r = uv_fs_mkdir(NULL, &req, resolved_path, 0777, NULL);
  uv_fs_req_cleanup(&req);

  if (r != 0) {
    err = uvwasi__translate_uv_error(r);
    goto exit;
  }

  err = UVWASI_ESUCCESS;
exit:
  uv_mutex_unlock(&wrap->mutex);
  return err;
}


uvwasi_errno_t uvwasi_path_filestat_get(uvwasi_t* uvwasi,
                                        uvwasi_fd_t fd,
                                        uvwasi_lookupflags_t flags,
                                        const char* path,
                                        size_t path_len,
                                        uvwasi_filestat_t* buf) {
  char resolved_path[PATH_MAX_BYTES];
  struct uvwasi_fd_wrap_t* wrap;
  uv_fs_t req;
  uvwasi_errno_t err;
  int r;

  if (uvwasi == NULL || path == NULL || buf == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds,
                            fd,
                            &wrap,
                            UVWASI_RIGHT_PATH_FILESTAT_GET,
                            0);
  if (err != UVWASI_ESUCCESS)
    return err;

  err = uvwasi__resolve_path(uvwasi,
                             wrap,
                             path,
                             path_len,
                             resolved_path,
                             flags);
  if (err != UVWASI_ESUCCESS)
    goto exit;

  r = uv_fs_stat(NULL, &req, resolved_path, NULL);
  if (r != 0) {
    uv_fs_req_cleanup(&req);
    err = uvwasi__translate_uv_error(r);
    goto exit;
  }

  uvwasi__stat_to_filestat(&req.statbuf, buf);
  uv_fs_req_cleanup(&req);
  err = UVWASI_ESUCCESS;
exit:
  uv_mutex_unlock(&wrap->mutex);
  return err;
}


uvwasi_errno_t uvwasi_path_filestat_set_times(uvwasi_t* uvwasi,
                                              uvwasi_fd_t fd,
                                              uvwasi_lookupflags_t flags,
                                              const char* path,
                                              size_t path_len,
                                              uvwasi_timestamp_t st_atim,
                                              uvwasi_timestamp_t st_mtim,
                                              uvwasi_fstflags_t fst_flags) {
  /* TODO(cjihrig): libuv does not currently support nanosecond precision. */
  char resolved_path[PATH_MAX_BYTES];
  struct uvwasi_fd_wrap_t* wrap;
  uv_fs_t req;
  uvwasi_errno_t err;
  int r;

  if (uvwasi == NULL || path == NULL)
    return UVWASI_EINVAL;

  if (fst_flags & ~(UVWASI_FILESTAT_SET_ATIM | UVWASI_FILESTAT_SET_ATIM_NOW |
                    UVWASI_FILESTAT_SET_MTIM | UVWASI_FILESTAT_SET_MTIM_NOW)) {
    return UVWASI_EINVAL;
  }

  err = uvwasi_fd_table_get(&uvwasi->fds,
                            fd,
                            &wrap,
                            UVWASI_RIGHT_PATH_FILESTAT_SET_TIMES,
                            0);
  if (err != UVWASI_ESUCCESS)
    return err;

  err = uvwasi__resolve_path(uvwasi,
                             wrap,
                             path,
                             path_len,
                             resolved_path,
                             flags);
  if (err != UVWASI_ESUCCESS)
    goto exit;

  /* TODO(cjihrig): st_atim and st_mtim should not be unconditionally passed. */
  r = uv_fs_utime(NULL, &req, resolved_path, st_atim, st_mtim, NULL);
  uv_fs_req_cleanup(&req);

  if (r != 0) {
    err = uvwasi__translate_uv_error(r);
    goto exit;
  }

  err = UVWASI_ESUCCESS;
exit:
  uv_mutex_unlock(&wrap->mutex);
  return err;
}


uvwasi_errno_t uvwasi_path_link(uvwasi_t* uvwasi,
                                uvwasi_fd_t old_fd,
                                uvwasi_lookupflags_t old_flags,
                                const char* old_path,
                                size_t old_path_len,
                                uvwasi_fd_t new_fd,
                                const char* new_path,
                                size_t new_path_len) {
  char resolved_old_path[PATH_MAX_BYTES];
  char resolved_new_path[PATH_MAX_BYTES];
  struct uvwasi_fd_wrap_t* old_wrap;
  struct uvwasi_fd_wrap_t* new_wrap;
  uvwasi_errno_t err;
  uv_fs_t req;
  int r;

  if (uvwasi == NULL || old_path == NULL || new_path == NULL)
    return UVWASI_EINVAL;

  uvwasi_fd_table_lock(&uvwasi->fds);

  if (old_fd == new_fd) {
    err = uvwasi_fd_table_get_nolock(&uvwasi->fds,
                                     old_fd,
                                     &old_wrap,
                                     UVWASI_RIGHT_PATH_LINK_SOURCE |
                                     UVWASI_RIGHT_PATH_LINK_TARGET,
                                     0);
    new_wrap = old_wrap;
  } else {
    err = uvwasi_fd_table_get_nolock(&uvwasi->fds,
                                     old_fd,
                                     &old_wrap,
                                     UVWASI_RIGHT_PATH_LINK_SOURCE,
                                     0);
    if (err != UVWASI_ESUCCESS) {
      uvwasi_fd_table_unlock(&uvwasi->fds);
      return err;
    }

    err = uvwasi_fd_table_get_nolock(&uvwasi->fds,
                                     new_fd,
                                     &new_wrap,
                                     UVWASI_RIGHT_PATH_LINK_TARGET,
                                     0);
    if (err != UVWASI_ESUCCESS)
      uv_mutex_unlock(&old_wrap->mutex);
  }

  uvwasi_fd_table_unlock(&uvwasi->fds);

  if (err != UVWASI_ESUCCESS)
    return err;

  err = uvwasi__resolve_path(uvwasi,
                             old_wrap,
                             old_path,
                             old_path_len,
                             resolved_old_path,
                             old_flags);
  if (err != UVWASI_ESUCCESS)
    goto exit;

  err = uvwasi__resolve_path(uvwasi,
                             new_wrap,
                             new_path,
                             new_path_len,
                             resolved_new_path,
                             0);
  if (err != UVWASI_ESUCCESS)
    goto exit;

  r = uv_fs_link(NULL, &req, resolved_old_path, resolved_new_path, NULL);
  uv_fs_req_cleanup(&req);
  if (r != 0) {
    err = uvwasi__translate_uv_error(r);
    goto exit;
  }

  err = UVWASI_ESUCCESS;
exit:
  uv_mutex_unlock(&new_wrap->mutex);
  if (old_fd != new_fd)
    uv_mutex_unlock(&old_wrap->mutex);
  return err;
}


uvwasi_errno_t uvwasi_path_open(uvwasi_t* uvwasi,
                                uvwasi_fd_t dirfd,
                                uvwasi_lookupflags_t dirflags,
                                const char* path,
                                size_t path_len,
                                uvwasi_oflags_t o_flags,
                                uvwasi_rights_t fs_rights_base,
                                uvwasi_rights_t fs_rights_inheriting,
                                uvwasi_fdflags_t fs_flags,
                                uvwasi_fd_t* fd) {
  char resolved_path[PATH_MAX_BYTES];
  uvwasi_rights_t needed_inheriting;
  uvwasi_rights_t needed_base;
  uvwasi_rights_t max_base;
  uvwasi_rights_t max_inheriting;
  struct uvwasi_fd_wrap_t* dirfd_wrap;
  struct uvwasi_fd_wrap_t *wrap;
  uvwasi_filetype_t filetype;
  uvwasi_errno_t err;
  uv_fs_t req;
  int flags;
  int read;
  int write;
  int r;

  if (uvwasi == NULL || path == NULL || fd == NULL)
    return UVWASI_EINVAL;

  read = 0 != (fs_rights_base & (UVWASI_RIGHT_FD_READ |
                                 UVWASI_RIGHT_FD_READDIR));
  write = 0 != (fs_rights_base & (UVWASI_RIGHT_FD_DATASYNC |
                                  UVWASI_RIGHT_FD_WRITE |
                                  UVWASI_RIGHT_FD_ALLOCATE |
                                  UVWASI_RIGHT_FD_FILESTAT_SET_SIZE));
  flags = write ? read ? UV_FS_O_RDWR : UV_FS_O_WRONLY : UV_FS_O_RDONLY;
  needed_base = UVWASI_RIGHT_PATH_OPEN;
  needed_inheriting = fs_rights_base | fs_rights_inheriting;

  if ((o_flags & UVWASI_O_CREAT) != 0) {
    flags |= UV_FS_O_CREAT;
    needed_base |= UVWASI_RIGHT_PATH_CREATE_FILE;
  }
  if ((o_flags & UVWASI_O_DIRECTORY) != 0)
    flags |= UV_FS_O_DIRECTORY;
  if ((o_flags & UVWASI_O_EXCL) != 0)
    flags |= UV_FS_O_EXCL;
  if ((o_flags & UVWASI_O_TRUNC) != 0) {
    flags |= UV_FS_O_TRUNC;
    needed_base |= UVWASI_RIGHT_PATH_FILESTAT_SET_SIZE;
  }

  if ((fs_flags & UVWASI_FDFLAG_APPEND) != 0)
    flags |= UV_FS_O_APPEND;
  if ((fs_flags & UVWASI_FDFLAG_DSYNC) != 0) {
    flags |= UV_FS_O_DSYNC;
    needed_inheriting |= UVWASI_RIGHT_FD_DATASYNC;
  }
  if ((fs_flags & UVWASI_FDFLAG_NONBLOCK) != 0)
    flags |= UV_FS_O_NONBLOCK;
  if ((fs_flags & UVWASI_FDFLAG_RSYNC) != 0) {
#ifdef O_RSYNC
    flags |= O_RSYNC; /* libuv has no UV_FS_O_RSYNC. */
#else
    flags |= UV_FS_O_SYNC;
#endif
    needed_inheriting |= UVWASI_RIGHT_FD_SYNC;
  }
  if ((fs_flags & UVWASI_FDFLAG_SYNC) != 0) {
    flags |= UV_FS_O_SYNC;
    needed_inheriting |= UVWASI_RIGHT_FD_SYNC;
  }
  if (write && (flags & (UV_FS_O_APPEND | UV_FS_O_TRUNC)) == 0)
    needed_inheriting |= UVWASI_RIGHT_FD_SEEK;

  err = uvwasi_fd_table_get(&uvwasi->fds,
                            dirfd,
                            &dirfd_wrap,
                            needed_base,
                            needed_inheriting);
  if (err != UVWASI_ESUCCESS)
    return err;

  err = uvwasi__resolve_path(uvwasi,
                             dirfd_wrap,
                             path,
                             path_len,
                             resolved_path,
                             dirflags);
  if (err != UVWASI_ESUCCESS) {
    uv_mutex_unlock(&dirfd_wrap->mutex);
    return err;
  }

  r = uv_fs_open(NULL, &req, resolved_path, flags, 0666, NULL);
  uv_mutex_unlock(&dirfd_wrap->mutex);
  uv_fs_req_cleanup(&req);

  if (r < 0)
    return uvwasi__translate_uv_error(r);

  /* Not all platforms support UV_FS_O_DIRECTORY, so get the file type and check
     it here. */
  err = uvwasi__get_filetype_by_fd(r, &filetype);
  if (err != UVWASI_ESUCCESS)
    goto close_file_and_error_exit;

  if ((o_flags & UVWASI_O_DIRECTORY) != 0 &&
      filetype != UVWASI_FILETYPE_DIRECTORY) {
    err = UVWASI_ENOTDIR;
    goto close_file_and_error_exit;
  }

  err = uvwasi__get_rights(r, flags, filetype, &max_base, &max_inheriting);
  if (err != UVWASI_ESUCCESS)
    goto close_file_and_error_exit;

  err = uvwasi_fd_table_insert(uvwasi,
                               &uvwasi->fds,
                               r,
                               resolved_path,
                               resolved_path,
                               filetype,
                               fs_rights_base & max_base,
                               fs_rights_inheriting & max_inheriting,
                               0,
                               &wrap);
  if (err != UVWASI_ESUCCESS)
    goto close_file_and_error_exit;

  *fd = wrap->id;
  uv_mutex_unlock(&wrap->mutex);
  return UVWASI_ESUCCESS;

close_file_and_error_exit:
  uv_fs_close(NULL, &req, r, NULL);
  uv_fs_req_cleanup(&req);
  return err;
}


uvwasi_errno_t uvwasi_path_readlink(uvwasi_t* uvwasi,
                                    uvwasi_fd_t fd,
                                    const char* path,
                                    size_t path_len,
                                    char* buf,
                                    size_t buf_len,
                                    size_t* bufused) {
  char resolved_path[PATH_MAX_BYTES];
  struct uvwasi_fd_wrap_t* wrap;
  uvwasi_errno_t err;
  uv_fs_t req;
  size_t len;
  int r;

  if (uvwasi == NULL || path == NULL || buf == NULL || bufused == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds,
                            fd,
                            &wrap,
                            UVWASI_RIGHT_PATH_READLINK,
                            0);
  if (err != UVWASI_ESUCCESS)
    return err;

  err = uvwasi__resolve_path(uvwasi, wrap, path, path_len, resolved_path, 0);
  if (err != UVWASI_ESUCCESS) {
    uv_mutex_unlock(&wrap->mutex);
    return err;
  }

  r = uv_fs_readlink(NULL, &req, resolved_path, NULL);
  uv_mutex_unlock(&wrap->mutex);
  if (r != 0) {
    uv_fs_req_cleanup(&req);
    return uvwasi__translate_uv_error(r);
  }

  len = strnlen(req.ptr, buf_len);
  if (len >= buf_len) {
    uv_fs_req_cleanup(&req);
    return UVWASI_ENOBUFS;
  }

  memcpy(buf, req.ptr, len);
  buf[len] = '\0';
  *bufused = len + 1;
  uv_fs_req_cleanup(&req);
  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_path_remove_directory(uvwasi_t* uvwasi,
                                            uvwasi_fd_t fd,
                                            const char* path,
                                            size_t path_len) {
  char resolved_path[PATH_MAX_BYTES];
  struct uvwasi_fd_wrap_t* wrap;
  uv_fs_t req;
  uvwasi_errno_t err;
  int r;

  if (uvwasi == NULL || path == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds,
                            fd,
                            &wrap,
                            UVWASI_RIGHT_PATH_REMOVE_DIRECTORY,
                            0);
  if (err != UVWASI_ESUCCESS)
    return err;

  err = uvwasi__resolve_path(uvwasi, wrap, path, path_len, resolved_path, 0);
  if (err != UVWASI_ESUCCESS) {
    uv_mutex_unlock(&wrap->mutex);
    return err;
  }

  r = uv_fs_rmdir(NULL, &req, resolved_path, NULL);
  uv_mutex_unlock(&wrap->mutex);
  uv_fs_req_cleanup(&req);

  if (r != 0)
    return uvwasi__translate_uv_error(r);

  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_path_rename(uvwasi_t* uvwasi,
                                  uvwasi_fd_t old_fd,
                                  const char* old_path,
                                  size_t old_path_len,
                                  uvwasi_fd_t new_fd,
                                  const char* new_path,
                                  size_t new_path_len) {
  char resolved_old_path[PATH_MAX_BYTES];
  char resolved_new_path[PATH_MAX_BYTES];
  struct uvwasi_fd_wrap_t* old_wrap;
  struct uvwasi_fd_wrap_t* new_wrap;
  uvwasi_errno_t err;
  uv_fs_t req;
  int r;

  if (uvwasi == NULL || old_path == NULL || new_path == NULL)
    return UVWASI_EINVAL;

  uvwasi_fd_table_lock(&uvwasi->fds);

  if (old_fd == new_fd) {
    err = uvwasi_fd_table_get_nolock(&uvwasi->fds,
                                     old_fd,
                                     &old_wrap,
                                     UVWASI_RIGHT_PATH_RENAME_SOURCE |
                                     UVWASI_RIGHT_PATH_RENAME_TARGET,
                                     0);
    new_wrap = old_wrap;
  } else {
    err = uvwasi_fd_table_get_nolock(&uvwasi->fds,
                                     old_fd,
                                     &old_wrap,
                                     UVWASI_RIGHT_PATH_RENAME_SOURCE,
                                     0);
    if (err != UVWASI_ESUCCESS) {
      uvwasi_fd_table_unlock(&uvwasi->fds);
      return err;
    }

    err = uvwasi_fd_table_get_nolock(&uvwasi->fds,
                                     new_fd,
                                     &new_wrap,
                                     UVWASI_RIGHT_PATH_RENAME_TARGET,
                                     0);
    if (err != UVWASI_ESUCCESS)
      uv_mutex_unlock(&old_wrap->mutex);
  }

  uvwasi_fd_table_unlock(&uvwasi->fds);

  if (err != UVWASI_ESUCCESS)
    return err;

  err = uvwasi__resolve_path(uvwasi,
                             old_wrap,
                             old_path,
                             old_path_len,
                             resolved_old_path,
                             0);
  if (err != UVWASI_ESUCCESS)
    goto exit;

  err = uvwasi__resolve_path(uvwasi,
                             new_wrap,
                             new_path,
                             new_path_len,
                             resolved_new_path,
                             0);
  if (err != UVWASI_ESUCCESS)
    goto exit;

  r = uv_fs_rename(NULL, &req, resolved_old_path, resolved_new_path, NULL);
  uv_fs_req_cleanup(&req);
  if (r != 0) {
    err = uvwasi__translate_uv_error(r);
    goto exit;
  }

  err = UVWASI_ESUCCESS;
exit:
  uv_mutex_unlock(&new_wrap->mutex);
  if (old_fd != new_fd)
    uv_mutex_unlock(&old_wrap->mutex);

  return err;
}


uvwasi_errno_t uvwasi_path_symlink(uvwasi_t* uvwasi,
                                   const char* old_path,
                                   size_t old_path_len,
                                   uvwasi_fd_t fd,
                                   const char* new_path,
                                   size_t new_path_len) {
  char resolved_new_path[PATH_MAX_BYTES];
  struct uvwasi_fd_wrap_t* wrap;
  uvwasi_errno_t err;
  uv_fs_t req;
  int r;

  if (uvwasi == NULL || old_path == NULL || new_path == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds,
                            fd,
                            &wrap,
                            UVWASI_RIGHT_PATH_SYMLINK,
                            0);
  if (err != UVWASI_ESUCCESS)
    return err;

  err = uvwasi__resolve_path(uvwasi,
                             wrap,
                             new_path,
                             new_path_len,
                             resolved_new_path,
                             0);
  if (err != UVWASI_ESUCCESS) {
    uv_mutex_unlock(&wrap->mutex);
    return err;
  }

  /* Windows support may require setting the flags option. */
  r = uv_fs_symlink(NULL, &req, old_path, resolved_new_path, 0, NULL);
  uv_mutex_unlock(&wrap->mutex);
  uv_fs_req_cleanup(&req);
  if (r != 0)
    return uvwasi__translate_uv_error(r);

  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_path_unlink_file(uvwasi_t* uvwasi,
                                       uvwasi_fd_t fd,
                                       const char* path,
                                       size_t path_len) {
  char resolved_path[PATH_MAX_BYTES];
  struct uvwasi_fd_wrap_t* wrap;
  uv_fs_t req;
  uvwasi_errno_t err;
  int r;

  if (uvwasi == NULL || path == NULL)
    return UVWASI_EINVAL;

  err = uvwasi_fd_table_get(&uvwasi->fds,
                            fd,
                            &wrap,
                            UVWASI_RIGHT_PATH_UNLINK_FILE,
                            0);
  if (err != UVWASI_ESUCCESS)
    return err;

  err = uvwasi__resolve_path(uvwasi, wrap, path, path_len, resolved_path, 0);
  if (err != UVWASI_ESUCCESS) {
    uv_mutex_unlock(&wrap->mutex);
    return err;
  }

  r = uv_fs_unlink(NULL, &req, resolved_path, NULL);
  uv_mutex_unlock(&wrap->mutex);
  uv_fs_req_cleanup(&req);

  if (r != 0)
    return uvwasi__translate_uv_error(r);

  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_poll_oneoff(uvwasi_t* uvwasi,
                                  const uvwasi_subscription_t* in,
                                  uvwasi_event_t* out,
                                  size_t nsubscriptions,
                                  size_t* nevents) {
  /* TODO(cjihrig): Implement this. */
  return UVWASI_ENOTSUP;
}


uvwasi_errno_t uvwasi_proc_exit(uvwasi_t* uvwasi, uvwasi_exitcode_t rval) {
  exit(rval);
  return UVWASI_ESUCCESS; /* This doesn't happen. */
}


uvwasi_errno_t uvwasi_proc_raise(uvwasi_t* uvwasi, uvwasi_signal_t sig) {
  int r;

  if (uvwasi == NULL)
    return UVWASI_EINVAL;

  r = uvwasi__translate_to_uv_signal(sig);
  if (r == -1)
    return UVWASI_ENOSYS;

  r = uv_kill(uv_os_getpid(), r);
  if (r != 0)
    return uvwasi__translate_uv_error(r);

  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_random_get(uvwasi_t* uvwasi, void* buf, size_t buf_len) {
  int r;

  if (uvwasi == NULL || buf == NULL)
    return UVWASI_EINVAL;

  r = uv_random(NULL, NULL, buf, buf_len, 0, NULL);
  if (r != 0)
    return uvwasi__translate_uv_error(r);

  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_sched_yield(uvwasi_t* uvwasi) {
  if (uvwasi == NULL)
    return UVWASI_EINVAL;

#ifndef _WIN32
  if (0 != sched_yield())
    return uvwasi__translate_uv_error(uv_translate_sys_error(errno));
#else
  SwitchToThread();
#endif /* _WIN32 */

  return UVWASI_ESUCCESS;
}


uvwasi_errno_t uvwasi_sock_recv(uvwasi_t* uvwasi,
                                uvwasi_fd_t sock,
                                const uvwasi_iovec_t* ri_data,
                                size_t ri_data_len,
                                uvwasi_riflags_t ri_flags,
                                size_t* ro_datalen,
                                uvwasi_roflags_t* ro_flags) {
  /* TODO(cjihrig): Waiting to implement, pending
                    https://github.com/WebAssembly/WASI/issues/4 */
  return UVWASI_ENOTSUP;
}


uvwasi_errno_t uvwasi_sock_send(uvwasi_t* uvwasi,
                                uvwasi_fd_t sock,
                                const uvwasi_ciovec_t* si_data,
                                size_t si_data_len,
                                uvwasi_siflags_t si_flags,
                                size_t* so_datalen) {
  /* TODO(cjihrig): Waiting to implement, pending
                    https://github.com/WebAssembly/WASI/issues/4 */
  return UVWASI_ENOTSUP;
}


uvwasi_errno_t uvwasi_sock_shutdown(uvwasi_t* uvwasi,
                                    uvwasi_fd_t sock,
                                    uvwasi_sdflags_t how) {
  /* TODO(cjihrig): Waiting to implement, pending
                    https://github.com/WebAssembly/WASI/issues/4 */
  return UVWASI_ENOTSUP;
}


const char* uvwasi_embedder_err_code_to_string(uvwasi_errno_t code) {
  switch (code) {
#define V(errcode) case errcode: return #errcode;
    V(UVWASI_E2BIG)
    V(UVWASI_EACCES)
    V(UVWASI_EADDRINUSE)
    V(UVWASI_EADDRNOTAVAIL)
    V(UVWASI_EAFNOSUPPORT)
    V(UVWASI_EAGAIN)
    V(UVWASI_EALREADY)
    V(UVWASI_EBADF)
    V(UVWASI_EBADMSG)
    V(UVWASI_EBUSY)
    V(UVWASI_ECANCELED)
    V(UVWASI_ECHILD)
    V(UVWASI_ECONNABORTED)
    V(UVWASI_ECONNREFUSED)
    V(UVWASI_ECONNRESET)
    V(UVWASI_EDEADLK)
    V(UVWASI_EDESTADDRREQ)
    V(UVWASI_EDOM)
    V(UVWASI_EDQUOT)
    V(UVWASI_EEXIST)
    V(UVWASI_EFAULT)
    V(UVWASI_EFBIG)
    V(UVWASI_EHOSTUNREACH)
    V(UVWASI_EIDRM)
    V(UVWASI_EILSEQ)
    V(UVWASI_EINPROGRESS)
    V(UVWASI_EINTR)
    V(UVWASI_EINVAL)
    V(UVWASI_EIO)
    V(UVWASI_EISCONN)
    V(UVWASI_EISDIR)
    V(UVWASI_ELOOP)
    V(UVWASI_EMFILE)
    V(UVWASI_EMLINK)
    V(UVWASI_EMSGSIZE)
    V(UVWASI_EMULTIHOP)
    V(UVWASI_ENAMETOOLONG)
    V(UVWASI_ENETDOWN)
    V(UVWASI_ENETRESET)
    V(UVWASI_ENETUNREACH)
    V(UVWASI_ENFILE)
    V(UVWASI_ENOBUFS)
    V(UVWASI_ENODEV)
    V(UVWASI_ENOENT)
    V(UVWASI_ENOEXEC)
    V(UVWASI_ENOLCK)
    V(UVWASI_ENOLINK)
    V(UVWASI_ENOMEM)
    V(UVWASI_ENOMSG)
    V(UVWASI_ENOPROTOOPT)
    V(UVWASI_ENOSPC)
    V(UVWASI_ENOSYS)
    V(UVWASI_ENOTCONN)
    V(UVWASI_ENOTDIR)
    V(UVWASI_ENOTEMPTY)
    V(UVWASI_ENOTRECOVERABLE)
    V(UVWASI_ENOTSOCK)
    V(UVWASI_ENOTSUP)
    V(UVWASI_ENOTTY)
    V(UVWASI_ENXIO)
    V(UVWASI_EOVERFLOW)
    V(UVWASI_EOWNERDEAD)
    V(UVWASI_EPERM)
    V(UVWASI_EPIPE)
    V(UVWASI_EPROTO)
    V(UVWASI_EPROTONOSUPPORT)
    V(UVWASI_EPROTOTYPE)
    V(UVWASI_ERANGE)
    V(UVWASI_EROFS)
    V(UVWASI_ESPIPE)
    V(UVWASI_ESRCH)
    V(UVWASI_ESTALE)
    V(UVWASI_ETIMEDOUT)
    V(UVWASI_ETXTBSY)
    V(UVWASI_EXDEV)
    V(UVWASI_ENOTCAPABLE)
    V(UVWASI_ESUCCESS)
#undef V
    default:
      return "UVWASI_UNKNOWN_ERROR";
  }
}
