package util

import (
	"bytes"
	"testing"
	"text/template"

	"github.com/stretchr/testify/assert"
)

func TestIsValidJSONNumber(t *testing.T) {
	cases := []struct {
		input  string
		expect bool
	}{
		{
			input:  "12345",
			expect: true,
		},
		{
			input:  "340282366920938463463374607431768211455",
			expect: true,
		},
		{
			input:  "340282366920938463463374607431768211455E340282366920938463463374607431768211455",
			expect: true,
		},
		{
			input:  "abc123",
			expect: false,
		},
		{
			input:  "123\n",
			expect: false,
		},
		{
			input:  "\n123",
			expect: false,
		},
	}

	for i, c := range cases {
		actual := IsValidJSONNumber(c.input)
		t.Logf("%d: in=%v actual=%v expect=%v", i, c.input, actual, c.expect)
		assert.Equal(t, c.expect, actual)
	}
}

func TestUnescape(t *testing.T) {
	cases := []struct {
		input  string
		output string
	}{
		{
			input:  "\\a\\b\\e\\f\\n\\r\\t\\v\\\\\\'\\\"\\?",
			output: string([]rune{0x7, 0x8, 0x1b, 0x0c, 0x0a, 0x0d, 0x09, 0x0b, 0x5c, 0x27, 0x22, 0x3f}),
		},
	}

	for i, c := range cases {
		unescaped, err := Unescape(c.input)
		assert.Nil(t, err)
		t.Logf("%d: %v %v", i, []byte(c.output), []byte(unescaped))
		assert.Equal(t, c.output, unescaped)
	}
}

func TestStringToValue(t *testing.T) {
	cases := []struct {
		input  string
		output interface{}
	}{
		{
			input:  "true",
			output: true,
		},
		{
			input:  "TRUE",
			output: true,
		},
		{
			input:  "yes",
			output: true,
		},
		{
			input:  "no",
			output: false,
		},
		{
			input:  "false",
			output: false,
		},
		{
			input:  "1",
			output: 1,
		},
		{
			input:  "0xdeadbeef",
			output: 3735928559,
		},
		{
			input:  "3.14159",
			output: float64(3.14159),
		},
		{
			input:  "foobar",
			output: "foobar",
		},
	}

	for _, c := range cases {
		actual := StringToValue(c.input)
		assert.Equal(t, actual, c.output)
	}
}

func TestValueToString(t *testing.T) {
	cases := []struct {
		input  interface{}
		output string
	}{
		{
			input:  true,
			output: "true",
		},
		{
			input:  false,
			output: "false",
		},
		{
			input:  123,
			output: "123",
		},
		{
			input:  float64(123),
			output: "123",
		},
		{
			input:  123.456,
			output: "123.456",
		},
		{
			input:  "hi",
			output: "hi",
		},
		{
			input:  'x',
			output: "x",
		},
		{
			input:  []int{1, 2, 3},
			output: "[1,2,3]",
		},
		{
			input:  map[string]int{"foo": 1, "bar": 5},
			output: `{"bar":5,"foo":1}`,
		},
	}

	for _, c := range cases {
		actual := ValueToString(c.input)
		assert.Equal(t, c.output, actual)
	}
}

func TestTruthy(t *testing.T) {
	cases := []struct {
		input  string
		output bool
	}{
		{
			input:  "true",
			output: true,
		},
		{
			input:  "TRUE",
			output: true,
		},
		{
			input:  "1",
			output: true,
		},
		{
			input:  "T",
			output: true,
		},
		{
			input:  "Y",
			output: true,
		},
		{
			input:  "YeS",
			output: true,
		},
		{
			input:  "\tYeS\n   ",
			output: true,
		},
		{
			input:  "hi",
			output: false,
		},
		{
			input:  "",
			output: false,
		},
		{
			input:  "0",
			output: false,
		},
		{
			input:  "false",
			output: false,
		},
	}

	for _, c := range cases {
		actual := Truthy(c.input)
		assert.Equal(t, c.output, actual)
	}
}

func TestTabularize(t *testing.T) {
	cases := []struct {
		input  interface{}
		expect [][]string
	}{
		{
			input:  "foo",
			expect: [][]string{[]string{"foo"}},
		},
		{
			input:  [][]interface{}{[]interface{}{"foo", 1, 7.3, -1}, []interface{}{"bar", "baz"}},
			expect: [][]string{[]string{"foo", "1", "7.3", "-1"}, []string{"bar", "baz"}},
		},
		{
			input:  [][]int{[]int{1, 2, 3, 4}, []int{7, 8, 9}},
			expect: [][]string{[]string{"1", "2", "3", "4"}, []string{"7", "8", "9"}},
		},
		{
			input:  [][]interface{}{[]interface{}{map[string]string{"foo": "bar"}}, []interface{}{1, 2, 3}},
			expect: [][]string{[]string{`{"foo":"bar"}`}, []string{"1", "2", "3"}},
		},
		{
			input: []map[string]interface{}{
				map[string]interface{}{
					"foo": 123,
					"bar": 456,
				},
				map[string]interface{}{
					"foo":  111,
					"quux": 999,
				},
			},
			expect: [][]string{
				[]string{"bar", "foo", "quux"},
				[]string{"456", "123", ""},
				[]string{"", "111", "999"},
			},
		},
		{
			input: []map[string]interface{}{
				map[string]interface{}{
					"foo": 123,
					"bar": 456,
					"baz": []string{"aaa", "bbb", "ccc"},
				},
				map[string]interface{}{
					"foo":  111,
					"quux": 999,
					"baz":  []int{1, 2},
				},
			},
			expect: [][]string{
				[]string{"bar", "baz.0", "baz.1", "baz.2", "foo", "quux"},
				[]string{"456", "aaa", "bbb", "ccc", "123", ""},
				[]string{"", "1", "2", "", "111", "999"},
			},
		},
	}

	for _, c := range cases {
		actual, err := Tabularize(c.input, true)
		assert.Nil(t, err)
		assert.Equal(t, c.expect, actual)
	}
}

func TestMapPath(t *testing.T) {
	p := ParseMapPath(`foo.bar\.baz.quux`)
	assert.Equal(t, MapPath{path: []string{"foo", `bar.baz`, "quux"}}, p)
	assert.Equal(t, `foo.bar\.baz.quux`, p.String())
	assert.Equal(t, MapPath{path: []string{"foo", `bar.baz`, "quux", "spam"}}, p.Child("spam"))
	assert.Equal(t, MapPath{path: []string{`bar.baz`, "quux"}}, p.Suffix())
}

func TestKeyspace(t *testing.T) {
	cases := []struct {
		input  interface{}
		expect []MapPath
	}{
		{
			input:  map[string]string{"foo": "bar", "baz": "quux"},
			expect: []MapPath{MapPath{path: []string{"foo"}}, MapPath{path: []string{"baz"}}},
		},
		{
			input: map[string]interface{}{"foo": "bar", "baz": "quux", "spam": map[string]string{"abc": "xyz", "123": "456"}},
			expect: []MapPath{
				MapPath{path: []string{"foo"}},
				MapPath{path: []string{"baz"}},
				MapPath{path: []string{"spam", "abc"}},
				MapPath{path: []string{"spam", "123"}},
			},
		},
		{
			input: map[string]interface{}{
				"foo":  "bar",
				"baz":  "quux",
				"spam": map[string]string{"abc": "xyz", "123": "456"},
				"list": []interface{}{5, "aaaa", 3.14149},
			},
			expect: []MapPath{
				MapPath{path: []string{"foo"}},
				MapPath{path: []string{"baz"}},
				MapPath{path: []string{"spam", "abc"}},
				MapPath{path: []string{"spam", "123"}},
				MapPath{path: []string{"list", "0"}},
				MapPath{path: []string{"list", "1"}},
				MapPath{path: []string{"list", "2"}},
			},
		},
	}

	for _, c := range cases {
		actual := Keyspace(MapPath{}, c.input)
		assert.ElementsMatch(t, c.expect, actual)
	}
}

func TestAccess(t *testing.T) {
	cases := []struct {
		input  interface{}
		path   MapPath
		expect interface{}
	}{
		{
			input:  map[string]string{"foo": "bar", "baz": "quux"},
			path:   MapPath{path: []string{"foo"}},
			expect: "bar",
		},
		{
			input:  map[string]interface{}{"foo": "bar", "baz": "quux", "spam": map[string]string{"abc": "xyz", "123": "456"}},
			path:   MapPath{path: []string{"spam", "123"}},
			expect: "456",
		},
		{
			input: map[string]interface{}{
				"foo":  "bar",
				"baz":  "quux",
				"spam": map[string]string{"abc": "xyz", "123": "456"},
				"list": []interface{}{5, "aaaa", 3.14149},
			},
			path:   MapPath{path: []string{"list", "1"}},
			expect: "aaaa",
		},
		{
			input: map[string]interface{}{
				"foo":  "bar",
				"baz":  "quux",
				"spam": map[string]string{"abc": "xyz", "123": "456"},
				"list": []interface{}{5, "aaaa", 3.14149},
			},
			path:   MapPath{path: []string{"spam"}},
			expect: map[string]string{"abc": "xyz", "123": "456"},
		},
	}

	for _, c := range cases {
		actual, ok := c.path.Access(c.input)
		assert.True(t, ok)
		assert.Equal(t, c.expect, actual)
	}
}

func TestTemplateFuncs(t *testing.T) {
	cases := []struct {
		template  string
		data      map[string]interface{}
		expect    string
		expectErr string
	}{

		{
			template: "hello {{.x}}",
			data:     map[string]interface{}{"x": "world"},
			expect:   "hello world",
		},
		{
			template: `{{lower "aaa BBB cCc"}}`,
			data:     map[string]interface{}{},
			expect:   "aaa bbb ccc",
		},
		{
			template: `{{upper "aaa BBB cCc"}}`,
			data:     map[string]interface{}{},
			expect:   "AAA BBB CCC",
		},
		{
			template: `{{replace "aaa BBB cCc" "aaa" "xxx"}}`,
			data:     map[string]interface{}{},
			expect:   "xxx BBB cCc",
		},
		{
			template: `{{reverse "abc"}}`,
			data:     map[string]interface{}{},
			expect:   "cba",
		},
		{
			template: `{{trim "abc aaa bbb aaa" " ab"}}`,
			data:     map[string]interface{}{},
			expect:   "c",
		},
		{
			template: `{{trim "  abc  " " "}}`,
			data:     map[string]interface{}{},
			expect:   "abc",
		},
		{
			template: `{{trim_left "aaabbbaaa" "a"}}`,
			data:     map[string]interface{}{},
			expect:   "bbbaaa",
		},
		{
			template: `{{trim_right "aaabbbaaa" "a"}}`,
			data:     map[string]interface{}{},
			expect:   "aaabbb",
		},
		{
			template: `{{trim_space "  abc    "}}`,
			data:     map[string]interface{}{},
			expect:   "abc",
		},
		{
			template: `{{trim_prefix "xyzabc" "xyz"}}`,
			data:     map[string]interface{}{},
			expect:   "abc",
		},
		{
			template: `{{trim_prefix "xyzabc" "yz"}}`,
			data:     map[string]interface{}{},
			expect:   "xyzabc",
		},
		{
			template: `{{trim_suffix "abcxyz" "xyz"}}`,
			data:     map[string]interface{}{},
			expect:   "abc",
		},
		{
			template: `{{trim_suffix "abcxyz" "xy"}}`,
			data:     map[string]interface{}{},
			expect:   "abcxyz",
		},
		{
			template: `{{repeat "abc" 3}}`,
			data:     map[string]interface{}{},
			expect:   "abcabcabc",
		},
		{
			template: `{{sprintf "%03d %2.2f" 7 5.678}}`,
			data:     map[string]interface{}{},
			expect:   "007 5.68",
		},
		{
			template: `{{resub "abc abbc abbbc" "b+" "x"}}`,
			data:     map[string]interface{}{},
			expect:   "axc axc axc",
		},
		{
			template:  `{{resub "abc abbc abbbc" "+" "x"}}`,
			data:      map[string]interface{}{},
			expect:    "",
			expectErr: `template: :1:2: executing "" at <resub "abc abbc abbbc" "+" "x">: error calling resub: template function 'resub' failed: failed to compile regex: error parsing regexp: missing argument to repetition operator: ` + "`+`",
		},
		{
			template:  `{{resub 7 "+" "x"}}`,
			data:      map[string]interface{}{},
			expect:    "",
			expectErr: "template: :1:2: executing \"\" at <resub 7 \"+\" \"x\">: error calling resub: template function 'resub' has 1 or more arguments of incorrect types (HINT: consider using 'str' or 'sprintf', e.g. '{{ lower ( str .somekey ) }}')\nargument 0 is 'int', not string",
		},
		{
			template: `{{quote "abc"}}`,
			data:     map[string]interface{}{},
			expect:   `"abc"`,
		},
		{
			template: "{{quote `\"abc\"`}}",
			data:     map[string]interface{}{},
			expect:   `"\"abc\""`,
		},
		{
			template:  `{{unquote "abc"}}`,
			data:      map[string]interface{}{},
			expect:    "",
			expectErr: `template: :1:2: executing "" at <unquote "abc">: error calling unquote: template function 'unquote' failed: invalid syntax`,
		},
		{
			template: "{{unquote `\"abc\"`}}",
			data:     map[string]interface{}{},
			expect:   `abc`,
		},
		{
			template:  "{{quote .x}}",
			data:      map[string]any{"x": 1.2},
			expect:    "",
			expectErr: "template: :1:2: executing \"\" at <quote .x>: error calling quote: template function 'quote' requires a string argument, not float64 (HINT: consider using 'str' or 'sprintf', e.g. '{{ lower ( str .somekey ) }}')",
		},
		{
			template: "{{quote (str .x)}}",
			data:     map[string]any{"x": 1.2},
			expect:   `"1.2"`,
		},
	}

	for _, c := range cases {
		tmpl, err := template.New("").Funcs(GetTemplateFuncs()).Parse(string(c.template))
		assert.Nil(t, err)

		buf := &bytes.Buffer{}
		err = tmpl.Execute(buf, c.data)
		if c.expectErr != "" {
			assert.NotNil(t, err)
			if err != nil {
				assert.Equal(t, c.expectErr, err.Error())
			}
		} else {
			assert.Nil(t, err)
		}

		assert.Equal(t, c.expect, buf.String())
	}
}
