# bitfinex
--
    import "github.com/bitfinexcom/bitfinex-api-go/v1"

Package bitfinex is the official client to access to bitfinex.com API

## Usage

```go
const (
	// BaseURL is the v1 REST endpoint.
	BaseURL = "https://api.bitfinex.com/v1/"
	// WebSocketURL the v1 Websocket endpoint.
	WebSocketURL = "wss://api-pub.bitfinex.com/ws/"
)
```

```go
const (
	LEND = "lend"
	LOAN = "loan"
)
```

```go
const (
	OrderTypeMarket               = "market"
	OrderTypeLimit                = "limit"
	OrderTypeStop                 = "stop"
	OrderTypeTrailingStop         = "trailing-stop"
	OrderTypeFillOrKill           = "fill-or-kill"
	OrderTypeExchangeMarket       = "exchange market"
	OrderTypeExchangeLimit        = "exchange limit"
	OrderTypeExchangeStop         = "exchange stop"
	OrderTypeExchangeTrailingStop = "exchange trailing-stop"
	OrderTypeExchangeFillOrKill   = "exchange fill-or-kill"
)
```
Order types that the API can return.

```go
const (
	WALLET_TRADING  = "trading"
	WALLET_EXCHANGE = "exchange"
	WALLET_DEPOSIT  = "deposit"
)
```

```go
const (
	// Pairs
	BTCUSD = "BTCUSD"
	LTCUSD = "LTCUSD"
	LTCBTC = "LTCBTC"
	ETHUSD = "ETHUSD"
	ETHBTC = "ETHBTC"
	ETCUSD = "ETCUSD"
	ETCBTC = "ETCBTC"
	BFXUSD = "BFXUSD"
	BFXBTC = "BFXBTC"
	ZECUSD = "ZECUSD"
	ZECBTC = "ZECBTC"
	XMRUSD = "XMRUSD"
	XMRBTC = "XMRBTC"
	RRTUSD = "RRTUSD"
	RRTBTC = "RRTBTC"
	XRPUSD = "XRPUSD"
	XRPBTC = "XRPBTC"
	EOSETH = "EOSETH"
	EOSUSD = "EOSUSD"
	EOSBTC = "EOSBTC"
	IOTUSD = "IOTUSD"
	IOTBTC = "IOTBTC"
	IOTETH = "IOTETH"
	BCCBTC = "BCCBTC"
	BCUBTC = "BCUBTC"
	BCCUSD = "BCCUSD"
	BCUUSD = "BCUUSD"

	// Channels
	ChanBook   = "book"
	ChanTrade  = "trades"
	ChanTicker = "ticker"
)
```
Pairs available

#### type AccountInfo

```go
type AccountInfo struct {
	MakerFees float64 `json:"maker_fees,string"`
	TakerFees float64 `json:"taker_fees,string"`
	Fees      []AccountPairFee
}
```


#### type AccountPairFee

```go
type AccountPairFee struct {
	Pair      string
	MakerFees float64 `json:"maker_fees,string"`
	TakerFees float64 `json:"taker_fees,string"`
}
```


#### type AccountService

```go
type AccountService struct {
}
```


#### func (*AccountService) Info

```go
func (a *AccountService) Info() (AccountInfo, error)
```
GET account_infos

#### func (*AccountService) KeyPermission

```go
func (a *AccountService) KeyPermission() (Permissions, error)
```

#### func (*AccountService) Summary

```go
func (a *AccountService) Summary() (Summary, error)
```

#### type ActiveOffer

```go
type ActiveOffer struct {
	ID              int64
	Currency        string
	Rate            string
	Period          int
	Direction       string
	Timestamp       string
	IsLive          bool   `json:"is_live"`
	IsCancelled     bool   `json:"is_cancelled"`
	OriginalAmount  string `json:"original_amount"`
	RemainingAmount string `json:"remaining_amount"`
	ExecutedAmount  string `json:"executed_amount"`
}
```


#### type Balance

```go
type Balance struct {
	Currency    string
	Amount      string
	Balance     string
	Description string
	Timestamp   string
}
```


#### type BalancesService

```go
type BalancesService struct {
}
```


#### func (*BalancesService) All

```go
func (b *BalancesService) All() ([]WalletBalance, error)
```
GET balances

#### type BankAccount

```go
type BankAccount struct {
	AccountName   string // Account name
	AccountNumber string // Account number or IBAN
	BankName      string // Bank Name
	BankAddress   string // Bank Address
	BankCity      string // Bank City
	BankCountry   string // Bank Country
	SwiftCode     string // SWIFT Code
}
```


#### type Client

```go
type Client struct {
	// Base URL for API requests.
	BaseURL                *url.URL
	WebSocketURL           string
	WebSocketTLSSkipVerify bool

	// Auth data
	APIKey    string
	APISecret string

	// Services
	Pairs         *PairsService
	Stats         *StatsService
	Ticker        *TickerService
	Account       *AccountService
	Balances      *BalancesService
	Offers        *OffersService
	Credits       *CreditsService
	Deposit       *DepositService
	Lendbook      *LendbookService
	MarginInfo    *MarginInfoService
	MarginFunding *MarginFundingService
	OrderBook     *OrderBookService
	Orders        *OrderService
	Trades        *TradesService
	Positions     *PositionsService
	History       *HistoryService
	WebSocket     *WebSocketService
	Wallet        *WalletService
}
```

Client manages all the communication with the Bitfinex API.

#### func  NewClient

```go
func NewClient() *Client
```
NewClient creates new Bitfinex.com API client.

#### func (*Client) Auth

```go
func (c *Client) Auth(key string, secret string) *Client
```
Auth sets api key and secret for usage is requests that requires authentication.

#### type Credit

```go
type Credit struct {
	Id        int
	Currency  string
	Status    string
	Rate      float64 `json:",string"`
	Period    float64
	Amount    float64 `json:",string"`
	Timestamp string
}
```


#### type CreditsService

```go
type CreditsService struct {
}
```


#### func (*CreditsService) All

```go
func (c *CreditsService) All() ([]Credit, error)
```
Returns an array of Credit

#### type DepositResponse

```go
type DepositResponse struct {
	Result   string
	Method   string
	Currency string
	Address  string
}
```


#### func (*DepositResponse) Success

```go
func (d *DepositResponse) Success() (bool, error)
```

#### type DepositService

```go
type DepositService struct {
}
```


#### func (*DepositService) New

```go
func (s *DepositService) New(method, walletName string, renew int) (DepositResponse, error)
```

#### type ErrorResponse

```go
type ErrorResponse struct {
	Response *Response
	Message  string `json:"message"`
}
```

ErrorResponse is the custom error type that is returned if the API returns an
error.

#### func (*ErrorResponse) Error

```go
func (r *ErrorResponse) Error() string
```

#### type HistoryService

```go
type HistoryService struct {
}
```


#### func (*HistoryService) Balance

```go
func (s *HistoryService) Balance(currency, wallet string, since, until time.Time, limit int) ([]Balance, error)
```

#### func (*HistoryService) Movements

```go
func (s *HistoryService) Movements(currency, method string, since, until time.Time, limit int) ([]Movement, error)
```

#### func (*HistoryService) Trades

```go
func (s *HistoryService) Trades(pair string, since, until time.Time, limit int, reverse bool) ([]PastTrade, error)
```

#### type KeyPerm

```go
type KeyPerm struct {
	Read  bool
	Write bool
}
```


#### type Lend

```go
type Lend struct {
	Rate      string
	Amount    string
	Period    int
	Timestamp string
	Frr       string
}
```


#### func (*Lend) ParseTime

```go
func (el *Lend) ParseTime() (*time.Time, error)
```

#### type Lendbook

```go
type Lendbook struct {
	Bids []Lend
	Asks []Lend
}
```


#### type LendbookService

```go
type LendbookService struct {
}
```


#### func (*LendbookService) Get

```go
func (s *LendbookService) Get(currency string, limitBids, limitAsks int) (Lendbook, error)
```
GET /lendbook/:currency

#### func (*LendbookService) Lends

```go
func (s *LendbookService) Lends(currency string) ([]Lends, error)
```
GET /lends/:currency

#### type Lends

```go
type Lends struct {
	Rate       string
	AmountLent string `json:"amount_lent"`
	AmountUsed string `json:"amount_used"`
	Timestamp  int64
}
```


#### func (*Lends) Time

```go
func (el *Lends) Time() *time.Time
```

#### type MarginFundingService

```go
type MarginFundingService struct {
}
```


#### func (*MarginFundingService) Cancel

```go
func (s *MarginFundingService) Cancel(offerId int64) (MarginOffer, error)
```

#### func (*MarginFundingService) Credits

```go
func (s *MarginFundingService) Credits() ([]ActiveOffer, error)
```

#### func (*MarginFundingService) NewLend

```go
func (s *MarginFundingService) NewLend(currency string, amount, rate float64, period int) (MarginOffer, error)
```

#### func (*MarginFundingService) NewLoan

```go
func (s *MarginFundingService) NewLoan(currency string, amount, rate float64, period int) (MarginOffer, error)
```

#### func (*MarginFundingService) Offers

```go
func (s *MarginFundingService) Offers() ([]ActiveOffer, error)
```

#### func (*MarginFundingService) Status

```go
func (s *MarginFundingService) Status(offerId int64) (MarginOffer, error)
```

#### type MarginInfo

```go
type MarginInfo struct {
	MarginBalance     float64       `json:"margin_balance,string"`
	TradableBalance   float64       `json:"tradable_balance,string"`
	UnrealizedPl      float64       `json:"unrealized_pl,string"`
	UnrealizedSwap    float64       `json:"unrealized_swap,string"`
	NetValue          float64       `json:"net_value,string"`
	RequiredMargin    float64       `json:"required_margin,string"`
	Leverage          float64       `json:"leverage,string"`
	MarginRequirement float64       `json:"margin_requirement,string"`
	MarginLimits      []MarginLimit `json:"margin_limits,string"`
	Message           string        `json:"message"`
}
```


#### type MarginInfoService

```go
type MarginInfoService struct {
}
```


#### func (*MarginInfoService) All

```go
func (s *MarginInfoService) All() ([]MarginInfo, error)
```
GET /margin_infos

#### type MarginLimit

```go
type MarginLimit struct {
	OnPair            string  `json:"on_pair"`
	InitialMargin     float64 `json:"initial_margin,string"`
	MarginRequirement float64 `json:"margin_requirement,string"`
	TradableBalance   float64 `json:"tradable_balance,string"`
}
```


#### type MarginOffer

```go
type MarginOffer struct {
	ID              int64
	Currency        string
	Rate            string
	Period          int
	Direction       string
	Timestamp       string
	IsLive          bool   `json:"is_live"`
	IsCancelled     bool   `json:"is_cancelled"`
	OriginalAmount  string `json:"original_amount"`
	RemainingAmount string `json:"remaining_amount"`
	ExecutedAmount  string `json:"executed_amount"`
	OfferId         int
}
```


#### type Movement

```go
type Movement struct {
	ID          int64 `json:",int"`
	Currency    string
	Method      string
	Type        string
	Amount      string
	Description string
	Status      string
	Timestamp   string
}
```


#### type MultipleOrderResponse

```go
type MultipleOrderResponse struct {
	Orders []Order `json:"order_ids"`
	Status string
}
```

MultipleOrderResponse bundles orders returned by the CreateMulti method.

#### type Offer

```go
type Offer struct {
	Id              int64
	Currency        string
	Rate            string
	Period          int64
	Direction       string
	Timestamp       string
	IsLive          bool   `json:"is_live"`
	IsCancelled     bool   `json:"is_cancelled"`
	OriginalAmount  string `json:"original_amount:string"`
	RemainingAmount string `json:"remaining_amount:string"`
	ExecutedAmount  string `json:"executed_amount:string"`
	OfferId         int64  `json:"offer_id"`
}
```


#### type OffersService

```go
type OffersService struct {
}
```


#### func (*OffersService) Cancel

```go
func (s *OffersService) Cancel(offerId int64) (Offer, error)
```

#### func (*OffersService) New

```go
func (s *OffersService) New(currency string, amount, rate float64, period int64, direction string) (Offer, error)
```
Create new offer for LEND or LOAN a currency, use LEND or LOAN constants as
direction

#### func (*OffersService) Status

```go
func (s *OffersService) Status(offerId int64) (Offer, error)
```

#### type Order

```go
type Order struct {
	ID                int64
	Symbol            string
	Exchange          string
	Price             string
	AvgExecutionPrice string `json:"avg_execution_price"`
	Side              string
	Type              string
	Timestamp         string
	IsLive            bool   `json:"is_live"`
	IsCanceled        bool   `json:"is_cancelled"`
	IsHidden          bool   `json:"is_hidden"`
	WasForced         bool   `json:"was_forced"`
	OriginalAmount    string `json:"original_amount"`
	RemainingAmount   string `json:"remaining_amount"`
	ExecutedAmount    string `json:"executed_amount"`
}
```

Order represents one order on the bitfinex platform.

#### type OrderBook

```go
type OrderBook struct {
	Bids []OrderBookEntry
	Asks []OrderBookEntry
}
```


#### type OrderBookEntry

```go
type OrderBookEntry struct {
	Price     string
	Rate      string
	Amount    string
	Period    int
	Timestamp string
	Frr       string
}
```


#### func (*OrderBookEntry) ParseTime

```go
func (el *OrderBookEntry) ParseTime() (*time.Time, error)
```

#### type OrderBookService

```go
type OrderBookService struct {
}
```


#### func (*OrderBookService) Get

```go
func (s *OrderBookService) Get(pair string, limitBids, limitAsks int, noGroup bool) (OrderBook, error)
```
GET /book

#### type OrderService

```go
type OrderService struct {
}
```

OrderService manages the Order endpoint.

#### func (*OrderService) All

```go
func (s *OrderService) All() ([]Order, error)
```
All returns all orders for the authenticated account.

#### func (*OrderService) Cancel

```go
func (s *OrderService) Cancel(orderID int64) error
```
Cancel the order with id `orderID`.

#### func (*OrderService) CancelAll

```go
func (s *OrderService) CancelAll() error
```
CancelAll active orders for the authenticated account.

#### func (*OrderService) CancelMulti

```go
func (s *OrderService) CancelMulti(orderIDS []int64) (string, error)
```
CancelMulti allows batch cancellation of orders.

#### func (*OrderService) Create

```go
func (s *OrderService) Create(symbol string, amount float64, price float64, orderType string) (*Order, error)
```
Create a new order.

#### func (*OrderService) CreateMulti

```go
func (s *OrderService) CreateMulti(orders []SubmitOrder) (MultipleOrderResponse, error)
```
CreateMulti allows batch creation of orders.

#### func (*OrderService) Replace

```go
func (s *OrderService) Replace(orderID int64, useRemaining bool, newOrder SubmitOrder) (Order, error)
```
Replace an Order

#### func (*OrderService) Status

```go
func (s *OrderService) Status(orderID int64) (Order, error)
```
Status retrieves the given order from the API.

#### type Pair

```go
type Pair struct {
	Pair             string
	PricePrecision   int     `json:"price_precision,int"`
	InitialMargin    float64 `json:"initial_margin,string"`
	MinimumMargin    float64 `json:"minimum_margin,string"`
	MaximumOrderSize float64 `json:"maximum_order_size,string"`
	MinimumOrderSize float64 `json:"minimum_order_size,string"`
	Expiration       string
	Margin           bool
}
```

Detailed Pair

#### type PairsService

```go
type PairsService struct {
}
```


#### func (*PairsService) All

```go
func (p *PairsService) All() ([]string, error)
```
Get all Pair names as array of strings

#### func (*PairsService) AllDetailed

```go
func (p *PairsService) AllDetailed() ([]Pair, error)
```
Return a list of detailed pairs

#### type PastTrade

```go
type PastTrade struct {
	Price       string
	Amount      string
	Timestamp   string
	Exchange    string
	Type        string
	FeeCurrency string `json:"fee_currency"`
	FeeAmount   string `json:"fee_amount"`
	TID         int64
	OrderId     int64 `json:"order_id,int"`
}
```


#### type Permissions

```go
type Permissions struct {
	Account   KeyPerm
	History   KeyPerm
	Orders    KeyPerm
	Positions KeyPerm
	Funding   KeyPerm
	Wallets   KeyPerm
	Withdraw  KeyPerm
}
```


#### type Position

```go
type Position struct {
	ID        int
	Symbol    string
	Amount    string
	Status    string
	Base      string
	Timestamp string
	Swap      string
	Pl        string
}
```

Position structure

#### func (*Position) ParseTime

```go
func (p *Position) ParseTime() (*time.Time, error)
```

#### type PositionsService

```go
type PositionsService struct {
}
```

PositionsService structure

#### func (*PositionsService) All

```go
func (b *PositionsService) All() ([]Position, error)
```
All - gets all positions

#### func (*PositionsService) Claim

```go
func (b *PositionsService) Claim(positionId int, amount string) (Position, error)
```
Claim a position

#### type Response

```go
type Response struct {
	Response *http.Response
	Body     []byte
}
```

Response is wrapper for standard http.Response and provides more methods.

#### func (*Response) String

```go
func (r *Response) String() string
```
String converts response body to string. An empty string will be returned if
error.

#### type Stats

```go
type Stats struct {
	Period int64
	Volume float64 `json:"volume,string"`
}
```


#### type StatsService

```go
type StatsService struct {
}
```


#### func (*StatsService) All

```go
func (s *StatsService) All(pair string, period, volume string) ([]Stats, error)
```
All(pair) - Volume stats for specified pair

#### type SubmitOrder

```go
type SubmitOrder struct {
	Symbol string
	Amount float64
	Price  float64
	Type   string
}
```

SubmitOrder is an order to be created on the bitfinex platform.

#### type Summary

```go
type Summary struct {
	TradeVolume   SummaryVolume `json:"trade_vol_30d"`
	FundingProfit SummaryProfit `json:"funding_profit_30d"`
	MakerFee      string        `json:"maker_fee"`
	TakerFee      string        `json:"taker_fee"`
}
```


#### type SummaryProfit

```go
type SummaryProfit struct {
	Currency string `json:"curr"`
	Volume   string `json:"amount"`
}
```


#### type SummaryVolume

```go
type SummaryVolume struct {
	Currency string `json:"curr"`
	Volume   string `json:"vol"`
}
```


#### type TermData

```go
type TermData struct {
	// Data term. E.g: ps, ws, ou, etc... See official documentation for more details.
	Term string
	// Data will contain different number of elements for each term.
	// Examples:
	// Term: ws, Data: ["exchange","BTC",0.01410829,0]
	// Term: oc, Data: [0,"BTCUSD",0,-0.01,"","CANCELED",270,0,"2015-10-15T11:26:13Z",0]
	Data  []interface{}
	Error string
}
```


#### func (*TermData) HasError

```go
func (c *TermData) HasError() bool
```

#### type Tick

```go
type Tick struct {
	Mid       string
	Bid       string
	Ask       string
	LastPrice string `json:"last_price"`
	Low       string
	High      string
	Volume    string
	Timestamp string
}
```


#### func (*Tick) ParseTime

```go
func (el *Tick) ParseTime() (*time.Time, error)
```
ParseTime - return Timestamp in time.Time format

#### type TickerService

```go
type TickerService struct {
}
```


#### func (*TickerService) Get

```go
func (s *TickerService) Get(pair string) (Tick, error)
```
Get(pair) - return last Tick for specified pair

#### type Trade

```go
type Trade struct {
	Price     string
	Amount    string
	Exchange  string
	Type      string
	Timestamp int64
	TradeId   int64 `json:"tid,int"`
}
```


#### func (*Trade) Time

```go
func (el *Trade) Time() *time.Time
```

#### type TradesService

```go
type TradesService struct {
}
```


#### func (*TradesService) All

```go
func (s *TradesService) All(pair string, timestamp time.Time, limitTrades int) ([]Trade, error)
```

#### type TransferStatus

```go
type TransferStatus struct {
	Status  string
	Message string
}
```


#### type WalletBalance

```go
type WalletBalance struct {
	Type      string
	Currency  string
	Amount    string
	Available string
}
```


#### type WalletService

```go
type WalletService struct {
}
```


#### func (*WalletService) Transfer

```go
func (c *WalletService) Transfer(amount float64, currency, from, to string) ([]TransferStatus, error)
```
Transfer funds between wallets

#### func (*WalletService) WithdrawCrypto

```go
func (c *WalletService) WithdrawCrypto(amount float64, currency, wallet, destinationAddress string) ([]WithdrawStatus, error)
```
Withdraw a cryptocurrency to a digital wallet

#### func (*WalletService) WithdrawWire

```go
func (c *WalletService) WithdrawWire(amount float64, expressWire bool, wallet string, beneficiaryBank, intermediaryBank BankAccount, message string) ([]WithdrawStatus, error)
```

#### type WebSocketService

```go
type WebSocketService struct {
}
```

WebSocketService allow to connect and receive stream data from bitfinex.com ws
service. nolint:megacheck,structcheck

#### func  NewWebSocketService

```go
func NewWebSocketService(c *Client) *WebSocketService
```
NewWebSocketService returns a WebSocketService using the given client.

#### func (*WebSocketService) AddSubscribe

```go
func (w *WebSocketService) AddSubscribe(channel string, pair string, c chan []float64)
```

#### func (*WebSocketService) ClearSubscriptions

```go
func (w *WebSocketService) ClearSubscriptions()
```

#### func (*WebSocketService) Close

```go
func (w *WebSocketService) Close()
```
Close web socket connection

#### func (*WebSocketService) Connect

```go
func (w *WebSocketService) Connect() error
```
Connect create new bitfinex websocket connection

#### func (*WebSocketService) ConnectPrivate

```go
func (w *WebSocketService) ConnectPrivate(ch chan TermData)
```

#### func (*WebSocketService) Subscribe

```go
func (w *WebSocketService) Subscribe() error
```
Subscribe allows to subsribe to channels and watch for new updates. This method
supports next channels: book, trade, ticker.

#### type WithdrawStatus

```go
type WithdrawStatus struct {
	Status       string
	Message      string
	WithdrawalID int `json:"withdrawal_id"`
}
```
