package spanish_vocab_test

import (
	"testing"

	"github.com/kljensen/snowball/spanish"
)

// Test a large set of words for which we know
// the correct stemmed form.
//
func Test_SpanishVocabulary(t *testing.T) {
	testCases := []struct {
		in  string
		out string
	}{
		{"a", "a"},
		{"aarón", "aaron"},
		{"abaco", "abac"},
		{"abajo", "abaj"},
		{"abandera", "abander"},
		{"abandona", "abandon"},
		{"abandonada", "abandon"},
		{"abandonadas", "abandon"},
		{"abandonado", "abandon"},
		{"abandonados", "abandon"},
		{"abandonamos", "abandon"},
		{"abandonan", "abandon"},
		{"abandonar", "abandon"},
		{"abandonarlo", "abandon"},
		{"abandonaron", "abandon"},
		{"abandono", "abandon"},
		{"abandonó", "abandon"},
		{"abaratar", "abarat"},
		{"abarca", "abarc"},
		{"abarcamos", "abarc"},
		{"abarcan", "abarc"},
		{"abarcar", "abarc"},
		{"abarcará", "abarc"},
		{"abarcarán", "abarc"},
		{"abarcaría", "abarc"},
		{"abarcó", "abarc"},
		{"abaroa", "abaro"},
		{"abarroteros", "abarroter"},
		{"abarrotó", "abarrot"},
		{"abastece", "abastec"},
		{"abastecedora", "abastecedor"},
		{"abastecer", "abastec"},
		{"abastecimiento", "abastec"},
		{"abastecimientos", "abastec"},
		{"abasto", "abast"},
		{"abatares", "abatar"},
		{"abatida", "abat"},
		{"abatido", "abat"},
		{"abatidos", "abat"},
		{"abatimiento", "abat"},
		{"abatió", "abat"},
		{"abatir", "abat"},
		{"abatirnos", "abat"},
		{"abatirse", "abat"},
		{"abba", "abba"},
		{"abbud", "abbud"},
		{"abc", "abc"},
		{"abdicación", "abdic"},
		{"abdicar", "abdic"},
		{"abeba", "abeb"},
		{"abedrop", "abedrop"},
		{"abel", "abel"},
		{"abelardo", "abelard"},
		{"aberración", "aberr"},
		{"aberrante", "aberr"},
		{"abidjan", "abidj"},
		{"abiel", "abiel"},
		{"abierta", "abiert"},
		{"abiertamente", "abiert"},
		{"abiertas", "abiert"},
		{"abierto", "abiert"},
		{"abiertos", "abiert"},
		{"abigeo", "abige"},
		{"abimael", "abimael"},
		{"abióticos", "abiot"},
		{"abismo", "abism"},
		{"ablandar", "abland"},
		{"ablandó", "abland"},
		{"abluciones", "ablucion"},
		{"abnegación", "abneg"},
		{"aboagay", "aboagay"},
		{"abocó", "aboc"},
		{"abogada", "abog"},
		{"abogadas", "abog"},
		{"abogadete", "abogadet"},
		{"abogadillo", "abogadill"},
		{"abogado", "abog"},
		{"abogados", "abog"},
		{"abolición", "abolicion"},
		{"abolirá", "abol"},
		{"abominan", "abomin"},
		{"abono", "abon"},
		{"abonos", "abon"},
		{"aborda", "abord"},
		{"abordaban", "abord"},
		{"abordada", "abord"},
		{"abordadas", "abord"},
		{"abordan", "abord"},
		{"abordando", "abord"},
		{"abordar", "abord"},
		{"abordará", "abord"},
		{"abordarán", "abord"},
		{"abordaron", "abord"},
		{"aborde", "abord"},
		{"abordo", "abord"},
		{"abordó", "abord"},
		{"abortados", "abort"},
		{"abortiva", "abort"},
		{"aborto", "abort"},
		{"abortos", "abort"},
		{"abra", "abra"},
		{"abraham", "abraham"},
		{"abrasivo", "abras"},
		{"abrazarlo", "abraz"},
		{"abrazarse", "abraz"},
		{"abrazo", "abraz"},
		{"abre", "abre"},
		{"abrego", "abreg"},
		{"abren", "abren"},
		{"abreu", "abreu"},
		{"abreviar", "abrevi"},
		{"abrían", "abri"},
		{"abridor", "abridor"},
		{"abriendo", "abriend"},
		{"abrieron", "abrieron"},
		{"abrigan", "abrig"},
		{"abrigo", "abrig"},
		{"abrigos", "abrig"},
		{"abril", "abril"},
		{"abriles", "abril"},
		{"abrio", "abri"},
		{"abrió", "abri"},
		{"abrir", "abrir"},
		{"abrira", "abrir"},
		{"abrirá", "abrir"},
		{"abrirán", "abriran"},
		{"abriría", "abrir"},
		{"abrirla", "abrirl"},
		{"abrirlos", "abrirl"},
		{"abrirnos", "abrirn"},
		{"abrirse", "abrirs"},
		{"abrogación", "abrog"},
		{"abrogados", "abrog"},
		{"abrogar", "abrog"},
		{"abrupto", "abrupt"},
		{"abs", "abs"},
		{"absoletas", "absolet"},
		{"absoluta", "absolut"},
		{"absolutamente", "absolut"},
		{"absoluto", "absolut"},
		{"absolutoria", "absolutori"},
		{"absolvió", "absolv"},
		{"absorba", "absorb"},
		{"absorber", "absorb"},
		{"absorbiendo", "absorb"},
		{"absorver", "absorv"},
		{"abstencionismo", "abstencion"},
		{"abstendrá", "abstendr"},
		{"abstenga", "absteng"},
		{"abstengan", "absteng"},
		{"abstenido", "absten"},
		{"abstracción", "abstraccion"},
		{"abstracta", "abstract"},
		{"abstractos", "abstract"},
		{"abstuviera", "abstuv"},
		{"absuelto", "absuelt"},
		{"absurdas", "absurd"},
		{"absurdo", "absurd"},
		{"abucheado", "abuch"},
		{"abuchear", "abuch"},
		{"abuchearon", "abuch"},
		{"abucheo", "abuche"},
		{"abucheos", "abuche"},
		{"abuela", "abuel"},
		{"abuelas", "abuel"},
		{"abuelita", "abuelit"},
		{"abuelitas", "abuelit"},
		{"abuelito", "abuelit"},
		{"abuelitos", "abuelit"},
		{"abuelo", "abuel"},
		{"abuelos", "abuel"},
		{"abultar", "abult"},
		{"abunda", "abund"},
		{"abundan", "abund"},
		{"abundancia", "abund"},
		{"abundante", "abund"},
		{"abundantes", "abund"},
		{"abundar", "abund"},
		{"abundaron", "abund"},
		{"abundó", "abund"},
		{"abur", "abur"},
		{"aburridas", "aburr"},
		{"abuso", "abus"},
		{"abusos", "abus"},
		{"acá", "aca"},
		{"acaba", "acab"},
		{"acababa", "acab"},
		{"acabado", "acab"},
		{"acabados", "acab"},
		{"acabamos", "acab"},
		{"acaban", "acab"},
		{"acabando", "acab"},
		{"acabar", "acab"},
		{"acabará", "acab"},
		{"acabaría", "acab"},
		{"acabaron", "acab"},
		{"acabe", "acab"},
		{"acabo", "acab"},
		{"acabó", "acab"},
		{"academia", "academi"},
		{"académica", "academ"},
		{"académicamente", "academ"},
		{"académicas", "academ"},
		{"académico", "academ"},
		{"académicos", "academ"},
		{"academik", "academik"},
		{"academy", "academy"},
		{"acaeció", "acaec"},
		{"acallando", "acall"},
		{"acallar", "acall"},
		{"acámbaro", "acambar"},
		{"acampar", "acamp"},
		{"acapara", "acap"},
		{"acaparado", "acapar"},
		{"acaparadores", "acapar"},
		{"acaparar", "acapar"},
		{"acaparó", "acapar"},
		{"acapulco", "acapulc"},
		{"acariciarte", "acariciart"},
		{"acarreado", "acarr"},
		{"acarreadores", "acarr"},
		{"acarreados", "acarr"},
		{"acarrear", "acarr"},
		{"acaso", "acas"},
		{"acatar", "acat"},
		{"acataré", "acat"},
		{"acatarlas", "acat"},
		{"acaten", "acat"},
		{"acaudaladas", "acaudal"},
		{"accedan", "acced"},
		{"accede", "acced"},
		{"acceder", "acced"},
		{"accederán", "acced"},
		{"accedido", "acced"},
		{"accedieron", "acced"},
		{"accesará", "acces"},
		{"accesible", "acces"},
		{"accesibles", "acces"},
		{"acceso", "acces"},
		{"accesorio", "accesori"},
		{"accesorios", "accesori"},
		{"accesos", "acces"},
		{"accidentado", "accident"},
		{"accidentados", "accident"},
		{"accidental", "accidental"},
		{"accidentarnos", "accident"},
		{"accidente", "accident"},
		{"accidentes", "accident"},
		{"accion", "accion"},
		{"acción", "accion"},
		{"accionar", "accion"},
		{"accionaria", "accionari"},
		{"accionarias", "accionari"},
		{"accionario", "accionari"},
		{"accione", "accion"},
		{"acciones", "accion"},
		{"accionista", "accion"},
		{"accionistas", "accion"},
		{"accival", "accival"},
		{"accra", "accra"},
		{"aceite", "aceit"},
		{"aceites", "aceit"},
		{"aceleradas", "aceler"},
		{"acelerado", "aceler"},
		{"acelerar", "aceler"},
		{"acelerara", "aceler"},
		{"acelerará", "aceler"},
		{"acelerarse", "aceler"},
		{"aceleró", "aceler"},
		{"acendrada", "acendr"},
		{"acentuada", "acentu"},
		{"acentuadas", "acentu"},
		{"acentuar", "acentu"},
		{"acentuó", "acentu"},
		{"acepta", "acept"},
		{"aceptaba", "acept"},
		{"aceptaban", "acept"},
		{"aceptable", "acept"},
		{"aceptación", "acept"},
		{"aceptada", "acept"},
		{"aceptadas", "acept"},
		{"aceptado", "acept"},
		{"aceptados", "acept"},
		{"aceptamos", "acept"},
		{"aceptan", "acept"},
		{"aceptar", "acept"},
		{"aceptara", "acept"},
		{"aceptarían", "acept"},
		{"aceptarla", "acept"},
		{"aceptaron", "acept"},
		{"aceptas", "acept"},
		{"acepte", "acept"},
		{"acepté", "acept"},
		{"acepten", "acept"},
		{"acepto", "acept"},
		{"aceptó", "acept"},
		{"acer", "acer"},
		{"acera", "acer"},
		{"acerca", "acerc"},
		{"acercaba", "acerc"},
		{"acercaban", "acerc"},
		{"acercado", "acerc"},
		{"acercamiento", "acerc"},
		{"acercamientos", "acerc"},
		{"acercan", "acerc"},
		{"acercando", "acerc"},
		{"acercándonos", "acerc"},
		{"acercándose", "acerc"},
		{"acercar", "acerc"},
		{"acercarán", "acerc"},
		{"acercaría", "acerc"},
		{"acercarnos", "acerc"},
		{"acercaron", "acerc"},
		{"acercarse", "acerc"},
		{"acercó", "acerc"},
		{"acereras", "acerer"},
		{"acereros", "acerer"},
		{"acero", "acer"},
		{"acerque", "acerqu"},
		{"acerqué", "acerqu"},
		{"acerquen", "acerqu"},
		{"acertada", "acert"},
		{"acertadamente", "acert"},
		{"acertado", "acert"},
		{"acervo", "acerv"},
		{"acetato", "acetat"},
		{"acevedo", "aceved"},
		{"aceves", "acev"},
		{"achacan", "achac"},
		{"achacar", "achac"},
		{"achacaron", "achac"},
		{"achacársela", "achac"},
		{"achaques", "achaqu"},
		{"acheson", "acheson"},
		{"achichincles", "achichincl"},
		{"achieve", "achiev"},
		{"achique", "achiqu"},
		{"achy", "achy"},
		{"acicalar", "acical"},
		{"acicate", "acicat"},
		{"ácida", "acid"},
		{"ácidas", "acid"},
		{"acidez", "acidez"},
		{"ácido", "acid"},
		{"ácidos", "acid"},
		{"acierto", "aciert"},
		{"aciertos", "aciert"},
		{"aclamada", "aclam"},
		{"aclamado", "aclam"},
		{"aclara", "aclar"},
		{"aclaración", "aclar"},
		{"aclaraciones", "aclar"},
		{"aclarado", "aclar"},
		{"aclaramos", "aclar"},
		{"aclarando", "aclar"},
		{"aclarar", "aclar"},
		{"aclarara", "aclar"},
		{"aclararon", "aclar"},
		{"aclaratorias", "aclaratori"},
		{"aclaren", "aclar"},
		{"aclaró", "aclar"},
		{"aclimatado", "aclimat"},
		{"aclimatándose", "aclimat"},
		{"acne", "acne"},
		{"acné", "acne"},
		{"acogida", "acog"},
		{"acojan", "acoj"},
		{"acomete", "acomet"},
		{"acomodado", "acomod"},
		{"acomodados", "acomod"},
		{"acomodar", "acomod"},
		{"acomodarse", "acomod"},
		{"acompaña", "acompañ"},
		{"acompañaba", "acompañ"},
		{"acompañaban", "acompañ"},
		{"acompañada", "acompañ"},
		{"acompañadas", "acompañ"},
		{"acompañado", "acompañ"},
		{"acompañados", "acompañ"},
		{"acompañan", "acompañ"},
		{"acompañando", "acompañ"},
		{"acompañante", "acompañ"},
		{"acompañantes", "acompañ"},
		{"acompañar", "acompañ"},
		{"acompañará", "acompañ"},
		{"acompañarán", "acompañ"},
		{"acompañaría", "acompañ"},
		{"acompañarían", "acompañ"},
		{"acompañaron", "acompañ"},
		{"acompañe", "acompañ"},
		{"acompañó", "acompañ"},
		{"acomplejados", "acomplej"},
		{"acondicionadas", "acondicion"},
		{"acondicionados", "acondicion"},
		{"acondicionamiento", "acondicion"},
		{"acongoje", "acongoj"},
		{"aconseja", "aconsej"},
		{"aconsejan", "aconsej"},
		{"aconsejar", "aconsej"},
		{"aconsejará", "aconsej"},
		{"aconsejaron", "aconsej"},
		{"aconséjeme", "aconsejem"},
		{"aconsejó", "aconsej"},
		{"acontecer", "acontec"},
		{"acontecimiento", "acontec"},
		{"acontecimientos", "acontec"},
		{"aconteció", "acontec"},
		{"acopio", "acopi"},
		{"acoplamiento", "acopl"},
		{"acoplamientos", "acopl"},
		{"acoplando", "acopl"},
		{"acordada", "acord"},
		{"acordado", "acord"},
		{"acordados", "acord"},
		{"acordaron", "acord"},
		{"acorde", "acord"},
		{"acordé", "acord"},
		{"acordes", "acord"},
		{"acordó", "acord"},
		{"acorraló", "acorral"},
		{"acortado", "acort"},
		{"acortaron", "acort"},
		{"acortó", "acort"},
		{"acoso", "acos"},
		{"acosta", "acost"},
		{"acostada", "acost"},
		{"acostumbra", "acostumbr"},
		{"acostumbraba", "acostumbr"},
		{"acostumbraban", "acostumbr"},
		{"acostumbradas", "acostumbr"},
		{"acostumbrado", "acostumbr"},
		{"acostumbrados", "acostumbr"},
		{"acostumbramos", "acostumbr"},
		{"acostumbran", "acostumbr"},
		{"acostumbrando", "acostumbr"},
		{"acostumbrar", "acostumbr"},
		{"acostumbrarme", "acostumbr"},
		{"acostumbrarnos", "acostumbr"},
		{"acota", "acot"},
		{"acotamiento", "acot"},
		{"acrecentarlo", "acrecent"},
		{"acrecienta", "acrecient"},
		{"acredita", "acredit"},
		{"acreditable", "acredit"},
		{"acreditación", "acredit"},
		{"acreditada", "acredit"},
		{"acreditado", "acredit"},
		{"acreditados", "acredit"},
		{"acreditó", "acredit"},
		{"acreedor", "acreedor"},
		{"acreedores", "acreedor"},
		{"acribillado", "acribill"},
		{"acribillan", "acribill"},
		{"acrobacias", "acrobaci"},
		{"acróbata", "acrobat"},
		{"acta", "acta"},
		{"actas", "actas"},
		{"actitud", "actitud"},
		{"actitudes", "actitud"},
		{"activa", "activ"},
		{"activación", "activ"},
		{"activamente", "activ"},
		{"activas", "activ"},
		{"actividad", "activ"},
		{"actividades", "activ"},
		{"activista", "activ"},
		{"activistas", "activ"},
		{"activo", "activ"},
		{"activos", "activ"},
		{"acto", "acto"},
		{"actor", "actor"},
		{"actoral", "actoral"},
		{"actores", "actor"},
		{"actos", "actos"},
		{"actrices", "actric"},
		{"actriz", "actriz"},
		{"actúa", "actu"},
		{"actuaba", "actu"},
		{"actuación", "actuacion"},
		{"actuaciones", "actuacion"},
		{"actuado", "actu"},
		{"actual", "actual"},
		{"actuales", "actual"},
		{"actualice", "actualic"},
		{"actualidad", "actual"},
		{"actualización", "actualiz"},
		{"actualizaciones", "actualiz"},
		{"actualizada", "actualiz"},
		{"actualizando", "actualiz"},
		{"actualizar", "actualiz"},
		{"actualmente", "actual"},
		{"actúan", "actu"},
		{"actuando", "actu"},
		{"actuar", "actu"},
		{"actuará", "actu"},
		{"actuarán", "actu"},
		{"actuaron", "actu"},
		{"actúe", "actu"},
		{"actúen", "actu"},
		{"actuó", "actu"},
		{"acuarios", "acuari"},
		{"acuática", "acuat"},
		{"acuciantes", "acuci"},
		{"acuda", "acud"},
		{"acudan", "acud"},
		{"acude", "acud"},
		{"acuden", "acud"},
		{"acudí", "acud"},
		{"acudía", "acud"},
		{"acudían", "acud"},
		{"acudido", "acud"},
		{"acudiera", "acud"},
		{"acudieron", "acud"},
		{"acudió", "acud"},
		{"acudir", "acud"},
		{"acudirá", "acud"},
		{"acudirán", "acud"},
		{"acudiremos", "acud"},
		{"acudirían", "acud"},
		{"acueducto", "acueduct"},
		{"acuerda", "acuerd"},
		{"acuerdas", "acuerd"},
		{"acuérdese", "acuerdes"},
		{"acuerdo", "acuerd"},
		{"acuerdos", "acuerd"},
		{"acuesto", "acuest"},
		{"acumula", "acumul"},
		{"acumulación", "acumul"},
		{"acumulada", "acumul"},
		{"acumuladas", "acumul"},
		{"acumulado", "acumul"},
		{"acumulados", "acumul"},
		{"acumulan", "acumul"},
		{"acumulando", "acumul"},
		{"acumular", "acumul"},
		{"acumuló", "acumul"},
		{"acuña", "acuñ"},
		{"acupuntura", "acupuntur"},
		{"acusa", "acus"},
		{"acusaban", "acus"},
		{"acusación", "acus"},
		{"acusaciones", "acus"},
		{"acusada", "acus"},
		{"acusadas", "acus"},
		{"acusado", "acus"},
		{"acusador", "acus"},
		{"acusadoras", "acus"},
		{"acusadores", "acus"},
		{"acusados", "acus"},
		{"acusan", "acus"},
		{"acusándolos", "acus"},
		{"acusarlo", "acus"},
		{"acusaron", "acus"},
		{"acusatorias", "acusatori"},
		{"acusó", "acus"},
		{"acústica", "acust"},
		{"acústicas", "acust"},
		{"ad", "ad"},
		{"adam", "adam"},
		{"adame", "adam"},
		{"adams", "adams"},
		{"adamson", "adamson"},
		{"adán", "adan"},
		{"adaptación", "adapt"},
		{"adaptada", "adapt"},
		{"adaptado", "adapt"},
		{"adaptados", "adapt"},
		{"adaptan", "adapt"},
		{"adaptando", "adapt"},
		{"adaptándose", "adapt"},
		{"adaptar", "adapt"},
		{"adaptarán", "adapt"},
		{"adaptarse", "adapt"},
		{"adapten", "adapt"},
		{"adaptó", "adapt"},
		{"adarme", "adarm"},
		{"addis", "addis"},
		{"adecuación", "adecu"},
		{"adecuaciones", "adecu"},
		{"adecuada", "adecu"},
		{"adecuadamente", "adecu"},
		{"adecuadas", "adecu"},
		{"adecuado", "adecu"},
		{"adecuados", "adecu"},
		{"adecuar", "adecu"},
		{"adecuarla", "adecu"},
		{"adecuarlos", "adecu"},
		{"adecuarse", "adecu"},
		{"adecuarte", "adecuart"},
		{"adela", "adel"},
		{"adelaida", "adel"},
		{"adelanta", "adelant"},
		{"adelantada", "adelant"},
		{"adelantado", "adelant"},
		{"adelantados", "adelant"},
		{"adelantamos", "adelant"},
		{"adelantando", "adelant"},
		{"adelantar", "adelant"},
		{"adelantarnos", "adelant"},
		{"adelantaron", "adelant"},
		{"adelantarse", "adelant"},
		{"adelantase", "adelant"},
		{"adelante", "adel"},
		{"adelanto", "adelant"},
		{"adelantó", "adelant"},
		{"adelantos", "adelant"},
		{"adelgazado", "adelgaz"},
		{"adelgazar", "adelgaz"},
		{"ademán", "ademan"},
		{"ademanes", "ademan"},
		{"ademas", "adem"},
		{"además", "ademas"},
		{"adentrarme", "adentr"},
		{"adentrarnos", "adentr"},
		{"adentro", "adentr"},
		{"adeptos", "adept"},
		{"aderezara", "aderez"},
		{"adese", "ades"},
		{"adeudaba", "adeud"},
		{"adeudo", "adeud"},
		{"adeudos", "adeud"},
		{"adherencia", "adherent"},
		{"adherida", "adher"},
		{"adheridos", "adher"},
		{"adherirse", "adher"},
		{"adhesión", "adhesion"},
		{"adhesiva", "adhes"},
		{"adhiera", "adhier"},
		{"adicional", "adicional"},
		{"adicionales", "adicional"},
		{"adicionalmente", "adicional"},
		{"adiciones", "adicion"},
		{"adicto", "adict"},
		{"adictos", "adict"},
		{"adiestramiento", "adiestr"},
		{"adiós", "adios"},
		{"aditamento", "aditament"},
		{"adivinación", "adivin"},
		{"adivinó", "adivin"},
		{"adivinos", "adivin"},
		{"adjetivo", "adjet"},
		{"adjudicación", "adjud"},
		{"adjudican", "adjudic"},
		{"adjudicaron", "adjudic"},
		{"adjudicarse", "adjudic"},
		{"adjudicó", "adjudic"},
		{"adjuntando", "adjunt"},
		{"adjunto", "adjunt"},
		{"administra", "administr"},
		{"administración", "administr"},
		{"administraciones", "administr"},
		{"administrador", "administr"},
		{"administradora", "administr"},
		{"administradores", "administr"},
		{"administrar", "administr"},
		{"administrativa", "administr"},
		{"administrativas", "administr"},
		{"administrativo", "administr"},
		{"administrativos", "administr"},
		{"admiraban", "admir"},
		{"admiración", "admir"},
		{"admiradoras", "admir"},
		{"admiran", "admir"},
		{"admirar", "admir"},
		{"admiro", "admir"},
		{"admiró", "admir"},
		{"admisión", "admision"},
		{"admiten", "admit"},
		{"admitidas", "admit"},
		{"admitió", "admit"},
		{"admitir", "admit"},
		{"admito", "admit"},
		{"adoleciendo", "adolec"},
		{"adoleció", "adolec"},
		{"adolescencia", "adolescent"},
		{"adolescente", "adolescent"},
		{"adolescentes", "adolescent"},
		{"adolfo", "adolf"},
		{"adolorida", "adolor"},
		{"adopción", "adopcion"},
		{"adopta", "adopt"},
		{"adoptada", "adopt"},
		{"adoptadas", "adopt"},
		{"adoptado", "adopt"},
		{"adoptando", "adopt"},
		{"adoptar", "adopt"},
		{"adoptará", "adopt"},
		{"adoptaran", "adopt"},
		{"adoptarán", "adopt"},
		{"adoptaron", "adopt"},
		{"adopte", "adopt"},
		{"adopten", "adopt"},
		{"adoptó", "adopt"},
		{"adoquín", "adoquin"},
		{"adora", "ador"},
		{"adoradores", "ador"},
		{"adoratorio", "adoratori"},
		{"adornadas", "adorn"},
		{"adornan", "adorn"},
		{"adornar", "adorn"},
		{"adornó", "adorn"},
		{"adornos", "adorn"},
		{"adoro", "ador"},
		{"adquiera", "adqu"},
		{"adquiere", "adquier"},
		{"adquieren", "adquier"},
		{"adquirí", "adquir"},
		{"adquirida", "adquir"},
		{"adquiridas", "adquir"},
		{"adquirido", "adquir"},
		{"adquirientes", "adquirient"},
		{"adquiriera", "adquir"},
		{"adquirieran", "adquir"},
		{"adquirieron", "adquir"},
		{"adquirió", "adquir"},
		{"adquirir", "adquir"},
		{"adquirirá", "adquir"},
		{"adquisición", "adquisicion"},
		{"adquisiciones", "adquisicion"},
		{"adquisitivo", "adquisit"},
		{"adr", "adr"},
		{"adrián", "adrian"},
		{"adriana", "adrian"},
		{"adrs", "adrs"},
		{"adscrita", "adscrit"},
		{"adscrito", "adscrit"},
		{"aduana", "aduan"},
		{"aduanal", "aduanal"},
		{"aduanales", "aduanal"},
		{"aductor", "aductor"},
		{"adueñarse", "adueñ"},
		{"adulta", "adult"},
		{"adultas", "adult"},
		{"adulto", "adult"},
		{"adultos", "adult"},
		{"advantaje", "advantaj"},
		{"adversario", "adversari"},
		{"adversarios", "adversari"},
		{"adversas", "advers"},
		{"adversidades", "advers"},
		{"adversos", "advers"},
		{"advertencia", "advertent"},
		{"advertido", "advert"},
		{"advertimos", "advert"},
		{"advertir", "advert"},
		{"advertirán", "advert"},
		{"advertirle", "advert"},
		{"advertising", "advertising"},
		{"advierta", "adviert"},
		{"advierte", "adviert"},
		{"advierten", "adviert"},
		{"advierto", "adviert"},
		{"advirtiendo", "advirt"},
		{"advirtiéndoles", "advirt"},
		{"advirtieron", "advirt"},
		{"advirtió", "advirt"},
		{"aérea", "aere"},
		{"aéreas", "aer"},
		{"aéreo", "aere"},
		{"aéreos", "aere"},
		{"aerodinámica", "aerodinam"},
		{"aerodinámico", "aerodinam"},
		{"aerofotogrametría", "aerofotogrametr"},
		{"aerógrafo", "aerograf"},
		{"aerolínea", "aeroline"},
		{"aerolíneas", "aerolin"},
		{"aerolito", "aerolit"},
		{"aeroméxico", "aeromex"},
		{"aeronáutica", "aeronaut"},
		{"aeronave", "aeronav"},
		{"aeronaves", "aeronav"},
		{"aeroperú", "aeroperu"},
		{"aeroportuario", "aeroportuari"},
		{"aeropuerto", "aeropuert"},
		{"aeropuertos", "aeropuert"},
		{"aerosol", "aerosol"},
		{"aerosoles", "aerosol"},
		{"aerovias", "aerovi"},
		{"aerovías", "aerov"},
		{"afa", "afa"},
		{"afán", "afan"},
		{"afanes", "afan"},
		{"afear", "afe"},
		{"afección", "afeccion"},
		{"afecciones", "afeccion"},
		{"afecta", "afect"},
		{"afectaba", "afect"},
		{"afectación", "afect"},
		{"afectaciones", "afect"},
		{"afectad", "afect"},
		{"afectada", "afect"},
		{"afectadas", "afect"},
		{"afectado", "afect"},
		{"afectados", "afect"},
		{"afectan", "afect"},
		{"afectando", "afect"},
		{"afectar", "afect"},
		{"afectará", "afect"},
		{"afectarán", "afect"},
		{"afectaría", "afect"},
		{"afectaron", "afect"},
		{"afectarse", "afect"},
		{"afecte", "afect"},
		{"afectivas", "afect"},
		{"afectivo", "afect"},
		{"afecto", "afect"},
		{"afectó", "afect"},
		{"afectos", "afect"},
		{"afeen", "afe"},
		{"afeitar", "afeit"},
		{"aferrarse", "aferr"},
		{"affaire", "affair"},
		{"afianzadora", "afianz"},
		{"afianzarse", "afianz"},
		{"afición", "aficion"},
		{"aficionadas", "aficion"},
		{"aficionado", "aficion"},
		{"aficionados", "aficion"},
		{"aficiones", "aficion"},
		{"afilada", "afil"},
		{"afiliación", "afili"},
		{"afiliada", "afili"},
		{"afiliadas", "afili"},
		{"afiliado", "afili"},
		{"afiliados", "afili"},
		{"afiliaron", "afili"},
		{"afín", "afin"},
		{"afinación", "afin"},
		{"afinaciones", "afin"},
		{"afinando", "afin"},
		{"afinar", "afin"},
		{"afines", "afin"},
		{"afinidad", "afin"},
		{"afinidades", "afin"},
		{"afinó", "afin"},
		{"afirma", "afirm"},
		{"afirmación", "afirm"},
		{"afirmaciones", "afirm"},
		{"afirmado", "afirm"},
		{"afirman", "afirm"},
		{"afirmando", "afirm"},
		{"afirmar", "afirm"},
		{"afirmaron", "afirm"},
		{"afirmativa", "afirm"},
		{"afirmó", "afirm"},
		{"afl", "afl"},
		{"aflicción", "afliccion"},
		{"afligida", "aflig"},
		{"afligido", "aflig"},
		{"afloja", "afloj"},
		{"aflojadas", "afloj"},
		{"aflojamos", "afloj"},
		{"aflojar", "afloj"},
		{"aflora", "aflor"},
		{"afluencia", "afluenci"},
		{"aforismo", "afor"},
		{"afortunada", "afortun"},
		{"afortunadamente", "afortun"},
		{"afortunado", "afortun"},
		{"afp", "afp"},
		{"afrenta", "afrent"},
		{"africa", "afric"},
		{"africano", "african"},
		{"africanos", "african"},
		{"afro", "afro"},
		{"afroamericanos", "afroamerican"},
		{"afrobrasileño", "afrobrasileñ"},
		{"afrobrasileños", "afrobrasileñ"},
		{"afrodisiaco", "afrodisiac"},
		{"afrodisiacos", "afrodisiac"},
		{"afrodisio", "afrodisi"},
		{"afrontar", "afront"},
		{"afrontará", "afront"},
		{"afrontaron", "afront"},
		{"afronten", "afront"},
		{"afuera", "afuer"},
		{"afueras", "afuer"},
		{"agachó", "agach"},
		{"agarra", "agarr"},
		{"agarrada", "agarr"},
		{"agarrar", "agarr"},
		{"agarraron", "agarr"},
		{"agarre", "agarr"},
		{"agárrese", "agarres"},
		{"age", "age"},
		{"agencia", "agenci"},
		{"agenciaron", "agenci"},
		{"agencias", "agenci"},
		{"agenda", "agend"},
		{"agendas", "agend"},
		{"agenl", "agenl"},
		{"agente", "agent"},
		{"agentes", "agent"},
		{"ágil", "agil"},
		{"agilidad", "agil"},
		{"agilización", "agiliz"},
		{"agilizando", "agiliz"},
		{"agilizar", "agiliz"},
		{"agilizara", "agiliz"},
		{"agilizará", "agiliz"},
		{"agio", "agi"},
		{"agiotistas", "agiot"},
		{"agitación", "agit"},
		{"agitadas", "agit"},
		{"agitadores", "agit"},
		{"agitando", "agit"},
		{"aglomeraciones", "aglomer"},
		{"aglutina", "aglutin"},
		{"aglutinar", "aglutin"},
		{"agobiado", "agobi"},
		{"agonía", "agon"},
		{"agonizante", "agoniz"},
		{"agora", "agor"},
		{"agosto", "agost"},
		{"agotadas", "agot"},
		{"agotado", "agot"},
		{"agotamiento", "agot"},
		{"agotándolos", "agot"},
		{"agotaron", "agot"},
		{"agotarse", "agot"},
		{"agote", "agot"},
		{"agraciado", "agraci"},
		{"agrada", "agrad"},
		{"agradable", "agrad"},
		{"agradece", "agradec"},
		{"agradecemos", "agradec"},
		{"agradecer", "agradec"},
		{"agradecerán", "agradec"},
		{"agradecería", "agradec"},
		{"agradeciendo", "agradec"},
		{"agradecieron", "agradec"},
		{"agradecimiento", "agradec"},
		{"agradeció", "agradec"},
		{"agradezco", "agradezc"},
		{"agrado", "agrad"},
		{"agramón", "agramon"},
		{"agrandar", "agrand"},
		{"agrandarla", "agrand"},
		{"agraria", "agrari"},
		{"agrario", "agrari"},
		{"agraristas", "agrar"},
		{"agrava", "agrav"},
		{"agravación", "agrav"},
		{"agravado", "agrav"},
		{"agravaron", "agrav"},
		{"agraviado", "agravi"},
		{"agravios", "agravi"},
		{"agredido", "agred"},
		{"agredidos", "agred"},
		{"agredieron", "agred"},
		{"agrega", "agreg"},
		{"agregada", "agreg"},
		{"agregado", "agreg"},
		{"agregaduría", "agregadur"},
		{"agregan", "agreg"},
		{"agregando", "agreg"},
		{"agregar", "agreg"},
		{"agregaron", "agreg"},
		{"agregarse", "agreg"},
		{"agrego", "agreg"},
		{"agregó", "agreg"},
		{"agréguele", "agreguel"},
		{"agremiada", "agremi"},
		{"agremiados", "agremi"},
		{"agresión", "agresion"},
		{"agresiones", "agresion"},
		{"agresiva", "agres"},
		{"agresivamente", "agres"},
		{"agresivas", "agres"},
		{"agresividad", "agres"},
		{"agresivo", "agres"},
		{"agresivos", "agres"},
		{"agresor", "agresor"},
		{"agresores", "agresor"},
		{"agrícola", "agricol"},
		{"agrícolas", "agricol"},
		{"agricultores", "agricultor"},
		{"agricultura", "agricultur"},
		{"agridyne", "agridyn"},
		{"agrio", "agri"},
		{"agro", "agro"},
		{"agropecuaria", "agropecuari"},
		{"agropecuario", "agropecuari"},
		{"agropecuarios", "agropecuari"},
		{"agroquímico", "agroquim"},
		{"agroquímicos", "agroquim"},
		{"agrupa", "agrup"},
		{"agrupación", "agrup"},
		{"agrupaciones", "agrup"},
		{"agrupadas", "agrup"},
		{"agrupamiento", "agrup"},
		{"agrupar", "agrup"},
		{"agua", "agu"},
		{"aguacate", "aguacat"},
		{"aguacates", "aguacat"},
		{"agualeguas", "agualegu"},
		{"aguanta", "aguant"},
		{"aguantar", "aguant"},
		{"aguantarse", "aguant"},
		{"aguantó", "aguant"},
		{"aguardaba", "aguard"},
		{"aguardaban", "aguard"},
		{"aguardan", "aguard"},
		{"aguas", "agu"},
		{"aguascalientes", "aguascalient"},
		{"aguaytía", "aguayt"},
		{"aguda", "agud"},
		{"agudas", "agud"},
		{"agudiza", "agudiz"},
		{"agudizado", "agudiz"},
		{"agudo", "agud"},
		{"agudos", "agud"},
		{"aguila", "aguil"},
		{"águila", "aguil"},
		{"aguilar", "aguil"},
		{"aguilas", "aguil"},
		{"aguilera", "aguiler"},
		{"aguinaldo", "aguinald"},
		{"aguirre", "aguirr"},
		{"agüites", "agüit"},
		{"aguja", "aguj"},
		{"agujeros", "agujer"},
		{"agujetas", "agujet"},
		{"agustín", "agustin"},
		{"agusto", "agust"},
		{"ah", "ah"},
		{"ahí", "ahi"},
		{"ahmsa", "ahmsa"},
		{"ahogados", "ahog"},
		{"ahogando", "ahog"},
		{"ahogaron", "ahog"},
		{"ahondando", "ahond"},
		{"ahondar", "ahond"},
		{"ahora", "ahor"},
		{"ahorita", "ahorit"},
		{"ahorra", "ahorr"},
		{"ahorrado", "ahorr"},
		{"ahorrador", "ahorr"},
		{"ahorradores", "ahorr"},
		{"ahorramos", "ahorr"},
		{"ahorrar", "ahorr"},
		{"ahorrará", "ahorr"},
		{"ahorrarán", "ahorr"},
		{"ahorraron", "ahorr"},
		{"ahorrativa", "ahorr"},
		{"ahorrativo", "ahorr"},
		{"ahorro", "ahorr"},
		{"ahorros", "ahorr"},
		{"ahuehuete", "ahuehuet"},
		{"ahujas", "ahuj"},
		{"ahuyenta", "ahuyent"},
		{"aída", "aid"},
		{"aikman", "aikm"},
		{"air", "air"},
		{"airadamente", "airad"},
		{"airbus", "airbus"},
		{"aire", "air"},
		{"airecombustible", "airecombust"},
		{"aires", "air"},
		{"airways", "airways"},
		{"ais", "ais"},
		{"aislacionismo", "aislacion"},
		{"aislada", "aisl"},
		{"aisladas", "aisl"},
		{"aislados", "aisl"},
		{"aislamiento", "aislamient"},
		{"aislarla", "aisl"},
		{"aislarlo", "aisl"},
		{"aispuro", "aispur"},
		{"aiv", "aiv"},
		{"aix", "aix"},
		{"ajalpan", "ajalp"},
		{"ajena", "ajen"},
		{"ajenas", "ajen"},
		{"ajeno", "ajen"},
		{"ajenos", "ajen"},
		{"ajusco", "ajusc"},
		{"ajusta", "ajust"},
		{"ajustados", "ajust"},
		{"ajustan", "ajust"},
		{"ajustando", "ajust"},
		{"ajustándolo", "ajust"},
		{"ajustar", "ajust"},
		{"ajustarse", "ajust"},
		{"ajuste", "ajust"},
		{"ajustes", "ajust"},
		{"ajusticiado", "ajustici"},
		{"ak", "ak"},
		{"akita", "akit"},
		{"al", "al"},
		{"ala", "ala"},
		{"alababa", "alab"},
		{"alabanza", "alab"},
		{"alabarderos", "alabarder"},
		{"alabó", "alab"},
		{"aladino", "aladin"},
		{"aladro", "aladr"},
		{"alain", "alain"},
		{"alambre", "alambr"},
		{"alameda", "alamed"},
		{"alamo", "alam"},
		{"alamos", "alam"},
		{"alan", "alan"},
		{"alanís", "alan"},
		{"alarcón", "alarcon"},
		{"alargadas", "alarg"},
		{"alargar", "alarg"},
		{"alarido", "alar"},
		{"alarma", "alarm"},
		{"alarmado", "alarm"},
		{"alarmados", "alarm"},
		{"alarmante", "alarm"},
		{"alarmaron", "alarm"},
		{"alarme", "alarm"},
		{"alarmismo", "alarm"},
		{"alarmó", "alarm"},
		{"alaska", "alask"},
		{"alatriste", "alatr"},
		{"alavez", "alavez"},
		{"alba", "alba"},
		{"albacete", "albacet"},
		{"albán", "alban"},
		{"albañil", "albañil"},
		{"albañiles", "albañil"},
		{"albay", "albay"},
		{"albee", "albe"},
		{"alberca", "alberc"},
		{"alberga", "alberg"},
		{"albergar", "alberg"},
		{"albergará", "alberg"},
		{"albergue", "alberg"},
		{"albergues", "alberg"},
		{"albert", "albert"},
		{"alberto", "albert"},
		{"albiazul", "albiazul"},
		{"albiazules", "albiazul"},
		{"albiceleste", "albicelest"},
		{"alborotada", "alborot"},
		{"alboroto", "alborot"},
		{"alborotos", "alborot"},
		{"album", "album"},
		{"álbum", "album"},
		{"albur", "albur"},
		{"alcachofa", "alcachof"},
		{"alcalá", "alcal"},
		{"alcalde", "alcald"},
		{"alcaldes", "alcald"},
		{"alcaldesa", "alcaldes"},
		{"alcaldía", "alcald"},
		{"alcaldías", "alcald"},
		{"alcalinidad", "alcalin"},
		{"alcalinos", "alcalin"},
		{"alcance", "alcanc"},
		{"alcancé", "alcanc"},
		{"alcancemos", "alcanc"},
		{"alcancen", "alcanc"},
		{"alcances", "alcanc"},
		{"alcántara", "alcant"},
		{"alcantarilla", "alcantarill"},
		{"alcantarillado", "alcantarill"},
		{"alcanza", "alcanz"},
		{"alcanzaba", "alcanz"},
		{"alcanzaban", "alcanz"},
		{"alcanzada", "alcanz"},
		{"alcanzado", "alcanz"},
		{"alcanzados", "alcanz"},
		{"alcanzan", "alcanz"},
		{"alcanzando", "alcanz"},
		{"alcanzar", "alcanz"},
		{"alcanzara", "alcanz"},
		{"alcanzará", "alcanz"},
		{"alcanzaré", "alcanz"},
		{"alcanzaremos", "alcanz"},
		{"alcanzaría", "alcanz"},
		{"alcanzarlos", "alcanz"},
		{"alcanzaron", "alcanz"},
		{"alcanzó", "alcanz"},
		{"alcerreza", "alcerrez"},
		{"alcocer", "alcoc"},
		{"alcohol", "alcohol"},
		{"alcoholicas", "alcohol"},
		{"aldape", "aldap"},
		{"aldea", "alde"},
		{"aldeas", "alde"},
		{"aledañas", "aledañ"},
		{"alega", "aleg"},
		{"alegando", "aleg"},
		{"alegar", "aleg"},
		{"alegara", "aleg"},
		{"alegaron", "aleg"},
		{"alegatos", "alegat"},
		{"alegó", "aleg"},
		{"alegoría", "alegor"},
		{"alegórico", "alegor"},
		{"alegóricos", "alegor"},
		{"alegraron", "alegr"},
		{"alegre", "alegr"},
		{"alegremente", "alegr"},
		{"alegría", "alegr"},
		{"alegró", "alegr"},
		{"aleja", "alej"},
		{"alejaban", "alej"},
		{"alejado", "alej"},
		{"alejamiento", "alej"},
		{"alejándonos", "alej"},
		{"alejándose", "alej"},
		{"alejandra", "alejandr"},
		{"alejandro", "alejandr"},
		{"alejar", "alej"},
		{"alejara", "alej"},
		{"alejaron", "alej"},
		{"alejo", "alej"},
		{"alejó", "alej"},
		{"alemán", "aleman"},
		{"alemana", "aleman"},
		{"alemanas", "aleman"},
		{"alemanes", "aleman"},
		{"alemania", "alemani"},
		{"alenia", "aleni"},
		{"alentador", "alent"},
		{"alentadora", "alent"},
		{"alentadores", "alent"},
		{"alentar", "alent"},
		{"alentó", "alent"},
		{"alergia", "alergi"},
		{"alergología", "alergolog"},
		{"alerta", "alert"},
		{"alertado", "alert"},
		{"alerten", "alert"},
		{"alessio", "alessi"},
		{"alex", "alex"},
		{"alexander", "alexand"},
		{"alexandr", "alexandr"},
		{"alexis", "alexis"},
		{"alfa", "alfa"},
		{"alfabético", "alfabet"},
		{"alfarería", "alfar"},
		{"alfaro", "alfar"},
		{"alfilerazos", "alfileraz"},
		{"alfio", "alfi"},
		{"alfombra", "alfombr"},
		{"alfonso", "alfons"},
		{"alfred", "alfred"},
		{"alfredo", "alfred"},
		{"algas", "algas"},
		{"algebra", "algebr"},
		{"álgebra", "algebr"},
		{"algo", "algo"},
		{"algodón", "algodon"},
		{"alguien", "algui"},
		{"algún", "algun"},
		{"alguna", "algun"},
		{"algunas", "algun"},
		{"alguno", "algun"},
		{"algunos", "algun"},
		{"ali", "ali"},
		{"aliada", "ali"},
		{"aliadas", "ali"},
		{"aliado", "ali"},
		{"aliados", "ali"},
		{"aliándose", "ali"},
		{"alianza", "alianz"},
		{"alianzas", "alianz"},
		{"aliar", "ali"},
		{"aliaron", "ali"},
		{"aliarse", "ali"},
		{"alice", "alic"},
		{"alicea", "alice"},
		{"alicia", "alici"},
		{"aliemos", "ali"},
		{"alienta", "alient"},
		{"alientan", "alient"},
		{"aliento", "alient"},
		{"alientos", "alient"},
		{"aligerar", "aliger"},
		{"aligerarla", "aliger"},
		{"alija", "alij"},
		{"alimañas", "alimañ"},
		{"alimentación", "aliment"},
		{"alimentada", "aliment"},
		{"alimentado", "aliment"},
		{"alimentador", "aliment"},
		{"alimentadores", "aliment"},
		{"alimentar", "aliment"},
		{"alimenticia", "alimentici"},
		{"alimenticio", "alimentici"},
		{"alimenticios", "alimentici"},
		{"alimento", "aliment"},
		{"alimentó", "aliment"},
		{"alimentos", "aliment"},
		{"alineacion", "alineacion"},
		{"alineación", "alin"},
		{"alineaciones", "alin"},
		{"alineado", "alin"},
		{"alineó", "aline"},
		{"alistan", "alist"},
		{"aliuska", "aliusk"},
		{"aliviar", "alivi"},
		{"aliviará", "alivi"},
		{"aliviaría", "alivi"},
		{"alivio", "alivi"},
		{"aljibes", "aljib"},
		{"alla", "alla"},
		{"allá", "alla"},
		{"allegadas", "alleg"},
		{"allegado", "alleg"},
		{"allegados", "alleg"},
		{"allegar", "alleg"},
		{"allegarle", "alleg"},
		{"allegarles", "alleg"},
		{"allegarse", "alleg"},
		{"allende", "allend"},
		{"allí", "alli"},
		{"alma", "alma"},
		{"almacena", "almacen"},
		{"almacenada", "almacen"},
		{"almacenadora", "almacen"},
		{"almacenaje", "almacenaj"},
		{"almacenamiento", "almacen"},
		{"almacenar", "almacen"},
		{"almacene", "almacen"},
		{"almacenes", "almacen"},
		{"almaguer", "almagu"},
		{"almaraz", "almaraz"},
		{"almas", "almas"},
		{"almohada", "almoh"},
		{"almonte", "almont"},
		{"almorzar", "almorz"},
		{"almorzase", "almorz"},
		{"almuerzo", "almuerz"},
		{"almuerzos", "almuerz"},
		{"alonso", "alons"},
		{"alpargatado", "alpargat"},
		{"alquibenceno", "alquibencen"},
		{"alquilado", "alquil"},
		{"alquiler", "alquil"},
		{"alquimia", "alquimi"},
		{"alrededor", "alrededor"},
		{"alrededores", "alrededor"},
		{"alta", "alta"},
		{"altadena", "altaden"},
		{"altamente", "alt"},
		{"altamira", "altamir"},
		{"altamirano", "altamiran"},
		{"altas", "altas"},
		{"altavista", "altav"},
		{"altera", "alter"},
		{"alteración", "alter"},
		{"alteraciones", "alter"},
		{"alterado", "alter"},
		{"alterados", "alter"},
		{"alterar", "alter"},
		{"alterarán", "alter"},
		{"alteraron", "alter"},
		{"alterarse", "alter"},
		{"altercado", "alterc"},
		{"altere", "alter"},
		{"alterna", "altern"},
		{"alternando", "altern"},
		{"alternar", "altern"},
		{"alternará", "altern"},
		{"alternativa", "altern"},
		{"alternativas", "altern"},
		{"alternó", "altern"},
		{"altibajos", "altibaj"},
		{"altímetro", "altimetr"},
		{"altiplano", "altiplan"},
		{"altísimos", "altis"},
		{"altisonantes", "altison"},
		{"altitud", "altitud"},
		{"alto", "alto"},
		{"altos", "altos"},
		{"altruista", "altruist"},
		{"altura", "altur"},
		{"alturas", "altur"},
		{"alud", "alud"},
		{"aludir", "alud"},
		{"alumbraba", "alumbr"},
		{"alumbraban", "alumbr"},
		{"alumbrado", "alumbr"},
		{"alumbró", "alumbr"},
		{"aluminio", "alumini"},
		{"alumna", "alumn"},
		{"alumnado", "alumn"},
		{"alumnas", "alumn"},
		{"alumno", "alumn"},
		{"alumnos", "alumn"},
		{"aluncinógenos", "aluncinogen"},
		{"alusión", "alusion"},
		{"alusiva", "alus"},
		{"alvarado", "alvar"},
		{"alvarez", "alvarez"},
		{"alvaro", "alvar"},
		{"alvelais", "alvelais"},
		{"alvez", "alvez"},
		{"alza", "alza"},
		{"alzados", "alzad"},
		{"alzamiento", "alzamient"},
		{"alzas", "alzas"},
		{"alzheimer", "alzheim"},
		{"alzo", "alzo"},
		{"am", "am"},
		{"ama", "ama"},
		{"amaba", "amab"},
		{"amaban", "amab"},
		{"amabilidad", "amabil"},
		{"amable", "amabl"},
		{"amablemente", "amabl"},
		{"amables", "amabl"},
		{"amadeo", "amade"},
		{"amado", "amad"},
		{"amador", "amador"},
		{"amagaron", "amag"},
		{"amagó", "amag"},
		{"amamos", "amam"},
		{"amanda", "amand"},
		{"amando", "amand"},
		{"amanece", "amanec"},
		{"amanecer", "amanec"},
		{"amante", "amant"},
		{"amantes", "amant"},
		{"amanuense", "amanuens"},
		{"amapola", "amapol"},
		{"amar", "amar"},
		{"amaré", "amar"},
		{"amarga", "amarg"},
		{"amargarnos", "amarg"},
		{"amargas", "amarg"},
		{"amargos", "amarg"},
		{"amarilla", "amarill"},
		{"amarillas", "amarill"},
		{"amarillo", "amarill"},
		{"amarillos", "amarill"},
		{"amarme", "amarm"},
		{"amaro", "amar"},
		{"amarrado", "amarr"},
		{"amarrando", "amarr"},
		{"amarren", "amarr"},
		{"amarres", "amarr"},
		{"amarró", "amarr"},
		{"amaru", "amaru"},
		{"amas", "amas"},
		{"amasando", "amas"},
		{"amateur", "amateur"},
		{"amateurs", "amateurs"},
		{"amatista", "amat"},
		{"amatistas", "amat"},
		{"amatorias", "amatori"},
		{"ambas", "ambas"},
		{"ambassador", "ambass"},
		{"ambición", "ambicion"},
		{"ambicionan", "ambicion"},
		{"ambiciosa", "ambici"},
		{"ambicioso", "ambici"},
		{"ambiciosos", "ambici"},
		{"ambientación", "ambient"},
		{"ambiental", "ambiental"},
		{"ambientales", "ambiental"},
		{"ambientalista", "ambiental"},
		{"ambientalistas", "ambiental"},
		{"ambientalmente", "ambiental"},
		{"ambientar", "ambient"},
		{"ambiente", "ambient"},
		{"ambientes", "ambient"},
		{"ambigüedades", "ambigüedad"},
		{"ámbito", "ambit"},
		{"ámbitos", "ambit"},
		{"ambos", "ambos"},
		{"ambriz", "ambriz"},
		{"ambrose", "ambros"},
		{"ambrosio", "ambrosi"},
		{"ambulancia", "ambul"},
		{"ambulancias", "ambul"},
		{"ambulantes", "ambul"},
		{"ambulatorio", "ambulatori"},
		{"amda", "amda"},
		{"amen", "amen"},
		{"amén", "amen"},
		{"amenace", "amenac"},
		{"amenaqzado", "amenaqz"},
		{"amenaza", "amenaz"},
		{"amenazaba", "amenaz"},
		{"amenazadas", "amenaz"},
		{"amenazado", "amenaz"},
		{"amenazados", "amenaz"},
		{"amenazan", "amenaz"},
		{"amenazando", "amenaz"},
		{"amenazar", "amenaz"},
		{"amenazará", "amenaz"},
		{"amenazaron", "amenaz"},
		{"amenazas", "amenaz"},
		{"amenazó", "amenaz"},
		{"amenizador", "ameniz"},
		{"ameno", "amen"},
		{"america", "amer"},
		{"américa", "amer"},
		{"american", "americ"},
		{"americana", "american"},
		{"americanas", "american"},
		{"americanismo", "american"},
		{"americanista", "american"},
		{"americanistas", "american"},
		{"americano", "american"},
		{"americanos", "american"},
		{"américas", "amer"},
		{"américo", "amer"},
		{"amerimex", "amerimex"},
		{"ameritaba", "amerit"},
		{"ames", "ames"},
		{"ametralladoras", "ametrall"},
		{"amga", "amga"},
		{"amieva", "amiev"},
		{"amiga", "amig"},
		{"amigabilidad", "amig"},
		{"amigable", "amig"},
		{"amigas", "amig"},
		{"amigo", "amig"},
		{"amigos", "amig"},
		{"amigote", "amigot"},
		{"amiguismos", "amigu"},
		{"amiguita", "amiguit"},
		{"aminoró", "aminor"},
		{"amistad", "amist"},
		{"amistades", "amistad"},
		{"amistosa", "amist"},
		{"amistosas", "amist"},
		{"amistoso", "amist"},
		{"amistosos", "amist"},
		{"amnistía", "amnist"},
		{"amo", "amo"},
		{"amoldan", "amold"},
		{"amonestaba", "amonest"},
		{"amonestados", "amonest"},
		{"amoniaco", "amoniac"},
		{"amonio", "amoni"},
		{"amontonada", "amonton"},
		{"amor", "amor"},
		{"amores", "amor"},
		{"amoroso", "amor"},
		{"amortiguadores", "amortigu"},
		{"amosca", "amosc"},
		{"amotinados", "amotin"},
		{"amoxcalli", "amoxcalli"},
		{"amp", "amp"},
		{"amparado", "ampar"},
		{"amparar", "ampar"},
		{"ampararnos", "ampar"},
		{"ampare", "ampar"},
		{"amparo", "ampar"},
		{"amparos", "ampar"},
		{"amplia", "ampli"},
		{"amplía", "ampli"},
		{"ampliación", "ampliacion"},
		{"ampliaciones", "ampliacion"},
		{"ampliada", "ampli"},
		{"ampliadas", "ampli"},
		{"ampliado", "ampli"},
		{"ampliados", "ampli"},
		{"ampliamente", "ampli"},
		{"ampliando", "ampli"},
		{"ampliar", "ampli"},
		{"ampliará", "ampli"},
		{"ampliarán", "ampli"},
		{"ampliarla", "ampli"},
		{"ampliarse", "ampli"},
		{"amplias", "ampli"},
		{"amplíe", "ampli"},
		{"amplio", "ampli"},
		{"amplios", "ampli"},
		{"amplitud", "amplitud"},
		{"ampudia", "ampudi"},
		{"amputaron", "amput"},
		{"amueblar", "amuebl"},
		{"amuelan", "amuel"},
		{"an", "an"},
		{"ana", "ana"},
		{"anacaborca", "anacaborc"},
		{"anacrónico", "anacron"},
		{"añade", "añad"},
		{"añadido", "añad"},
		{"añadiendo", "añad"},
		{"añadieron", "añad"},
		{"añadió", "añad"},
		{"anáhuac", "anahuac"},
		{"analfabetas", "analfabet"},
		{"analfabetismo", "analfabet"},
		{"analice", "analic"},
		{"analicen", "analic"},
		{"análisis", "analisis"},
		{"analista", "anal"},
		{"analistas", "anal"},
		{"analiza", "analiz"},
		{"analizadas", "analiz"},
		{"analizado", "analiz"},
		{"analizados", "analiz"},
		{"analizamos", "analiz"},
		{"analizando", "analiz"},
		{"analizándolo", "analiz"},
		{"analizar", "analiz"},
		{"analizará", "analiz"},
		{"analizarán", "analiz"},
		{"analizaría", "analiz"},
		{"analizarla", "analiz"},
		{"analizarlo", "analiz"},
		{"analizaron", "analiz"},
		{"analizarse", "analiz"},
		{"analizó", "analiz"},
		{"análogas", "analog"},
		{"analogías", "analog"},
		{"anamup", "anamup"},
		{"anarquía", "anarqu"},
		{"anatemas", "anatem"},
		{"ancestrales", "ancestral"},
		{"ancestro", "ancestr"},
		{"ancestros", "ancestr"},
		{"ancha", "ancha"},
		{"anchas", "anchas"},
		{"ancho", "ancho"},
		{"anciana", "ancian"},
		{"ancianas", "ancian"},
		{"ancianitos", "ancianit"},
		{"anciano", "ancian"},
		{"ancianos", "ancian"},
		{"ancira", "ancir"},
		{"ancló", "anclo"},
		{"and", "and"},
		{"anda", "anda"},
		{"andaba", "andab"},
		{"andaban", "andab"},
		{"andamos", "andam"},
		{"andan", "andan"},
		{"andar", "andar"},
		{"andaría", "andar"},
		{"ande", "ande"},
		{"andei", "andei"},
		{"anderson", "anderson"},
		{"andes", "andes"},
		{"andina", "andin"},
		{"andino", "andin"},
		{"ando", "ando"},
		{"andrade", "andrad"},
		{"andrea", "andre"},
		{"andres", "andres"},
		{"andrés", "andres"},
		{"andrew", "andrew"},
		{"anduvo", "anduv"},
		{"anécdota", "anecdot"},
		{"anemia", "anemi"},
		{"anémica", "anem"},
		{"anestesia", "anestesi"},
		{"anexada", "anex"},
		{"anexamos", "anex"},
		{"anexas", "anex"},
		{"anexos", "anex"},
		{"anfibios", "anfibi"},
		{"anfisbena", "anfisben"},
		{"anfiteatro", "anfiteatr"},
		{"anfitrión", "anfitrion"},
		{"ánfora", "anfor"},
		{"ánforas", "anfor"},
		{"angel", "angel"},
		{"ángel", "angel"},
		{"angeles", "angel"},
		{"ángeles", "angel"},
		{"angélica", "angel"},
		{"angélicas", "angel"},
		{"angelina", "angelin"},
		{"angelino", "angelin"},
		{"angelito", "angelit"},
		{"angelitos", "angelit"},
		{"angelo", "angel"},
		{"angelus", "angelus"},
		{"angiosperma", "angiosperm"},
		{"angiospermas", "angiosperm"},
		{"anglicana", "anglican"},
		{"anglo", "anglo"},
		{"angola", "angol"},
		{"angoleña", "angoleñ"},
		{"angoleño", "angoleñ"},
		{"angoleños", "angoleñ"},
		{"angongueo", "angongue"},
		{"angostura", "angostur"},
		{"ángulo", "angul"},
		{"angustia", "angusti"},
		{"angustiada", "angusti"},
		{"angustias", "angusti"},
		{"angustió", "angust"},
		{"anhatl", "anhatl"},
		{"anhelado", "anhel"},
		{"anhelo", "anhel"},
		{"anillo", "anill"},
		{"anima", "anim"},
		{"animación", "anim"},
		{"animal", "animal"},
		{"animales", "animal"},
		{"animalización", "animaliz"},
		{"animar", "anim"},
		{"animarán", "anim"},
		{"anímica", "anim"},
		{"anímicamente", "anim"},
		{"ánimo", "anim"},
		{"ánimos", "anim"},
		{"aniversario", "aniversari"},
		{"ann", "ann"},
		{"anna", "anna"},
		{"año", "año"},
		{"anoche", "anoch"},
		{"anochecer", "anochec"},
		{"anonas", "anon"},
		{"anónima", "anonim"},
		{"anónimas", "anonim"},
		{"anonimato", "anonimat"},
		{"anónimo", "anonim"},
		{"anónimos", "anon"},
		{"años", "años"},
		{"añosfueron", "añosfueron"},
		{"anotaba", "anot"},
		{"anotación", "anot"},
		{"anotaciones", "anot"},
		{"anotado", "anot"},
		{"anotados", "anot"},
		{"anotando", "anot"},
		{"anotar", "anot"},
		{"anotaron", "anot"},
		{"anotarse", "anot"},
		{"anote", "anot"},
		{"anoten", "anot"},
		{"anotó", "anot"},
		{"anselmo", "anselm"},
		{"ansias", "ansi"},
		{"ansiedad", "ansied"},
		{"ansioso", "ansios"},
		{"anstee", "anste"},
		{"antad", "antad"},
		{"antaño", "antañ"},
		{"ante", "ante"},
		{"anteayer", "anteay"},
		{"antebrazo", "antebraz"},
		{"antecedente", "antecedent"},
		{"antecedentes", "antecedent"},
		{"antecesores", "antecesor"},
		{"antemano", "anteman"},
		{"anteo", "ante"},
		{"anteojos", "anteoj"},
		{"antepasado", "antepas"},
		{"antepenúltimo", "antepenultim"},
		{"anterior", "anterior"},
		{"anteriores", "anterior"},
		{"anterioridad", "anterior"},
		{"anteriormente", "anterior"},
		{"antes", "antes"},
		{"antesala", "antesal"},
		{"antesalista", "antesal"},
		{"anthony", "anthony"},
		{"anti", "anti"},
		{"antiaéreas", "antiaer"},
		{"anticipaba", "anticip"},
		{"anticipación", "anticip"},
		{"anticipada", "anticip"},
		{"anticipadamente", "anticip"},
		{"anticipadas", "anticip"},
		{"anticipado", "anticip"},
		{"anticipamos", "anticip"},
		{"anticipando", "anticip"},
		{"anticipar", "anticip"},
		{"anticiparon", "anticip"},
		{"anticipos", "anticip"},
		{"anticonstitucional", "anticonstitucional"},
		{"anticontaminante", "anticontamin"},
		{"anticontaminantes", "anticontamin"},
		{"anticristo", "anticrist"},
		{"anticuarios", "anticuari"},
		{"antidemocrática", "antidemocrat"},
		{"antiderrapante", "antiderrap"},
		{"antidoping", "antidoping"},
		{"antidoto", "antidot"},
		{"antídotos", "antidot"},
		{"antidrogas", "antidrog"},
		{"antidumping", "antidumping"},
		{"antier", "anti"},
		{"antiestrés", "antiestres"},
		{"antigua", "antigu"},
		{"antiguas", "antigu"},
		{"antigüedad", "antigüed"},
		{"antigüedades", "antigüedad"},
		{"antiguo", "antigu"},
		{"antiguos", "antigu"},
		{"antillano", "antillan"},
		{"antillanos", "antillan"},
		{"antillas", "antill"},
		{"antimonopolio", "antimonopoli"},
		{"antinarcóticos", "antinarcot"},
		{"antinépota", "antinepot"},
		{"antioqueña", "antioqueñ"},
		{"antiparasitaria", "antiparasitari"},
		{"antipatía", "antipat"},
		{"antisocial", "antisocial"},
		{"antisubversiva", "antisubvers"},
		{"antítesis", "antitesis"},
		{"antoja", "antoj"},
		{"antojo", "antoj"},
		{"antonelli", "antonelli"},
		{"antonin", "antonin"},
		{"antonini", "antonini"},
		{"antonio", "antoni"},
		{"antorchista", "antorch"},
		{"antropología", "antropolog"},
		{"antropólogo", "antropolog"},
		{"anual", "anual"},
		{"anuales", "anual"},
		{"anualmente", "anual"},
		{"anulación", "anul"},
		{"anuladas", "anul"},
		{"anulados", "anul"},
		{"anularlo", "anul"},
		{"anuncia", "anunci"},
		{"anunciaba", "anunci"},
		{"anunciaban", "anunci"},
		{"anunciada", "anunci"},
		{"anunciadas", "anunci"},
		{"anunciado", "anunci"},
		{"anunciados", "anunci"},
		{"anuncian", "anunci"},
		{"anunciando", "anunci"},
		{"anunciar", "anunci"},
		{"anunciará", "anunci"},
		{"anunciarlo", "anunci"},
		{"anunciaron", "anunci"},
		{"anuncio", "anunci"},
		{"anunció", "anunc"},
		{"anuncios", "anunci"},
		{"anzuelo", "anzuel"},
		{"anzuelos", "anzuel"},
		{"ap", "ap"},
		{"apabulló", "apabull"},
		{"apaches", "apach"},
		{"apacible", "apac"},
		{"apaga", "apag"},
		{"apagados", "apag"},
		{"apagan", "apag"},
		{"apagará", "apag"},
		{"apagó", "apag"},
		{"apagón", "apagon"},
		{"apaleó", "apale"},
		{"aparadores", "apar"},
		{"aparato", "aparat"},
		{"aparatos", "aparat"},
		{"aparatosa", "aparat"},
		{"aparcar", "aparc"},
		{"aparece", "aparec"},
		{"aparecen", "aparec"},
		{"aparecer", "aparec"},
		{"aparecía", "aparec"},
		{"aparecidas", "aparec"},
		{"aparecido", "aparec"},
		{"apareciendo", "aparec"},
		{"apareciera", "aparec"},
		{"aparecieron", "aparec"},
		{"apareció", "aparec"},
		{"aparentan", "aparent"},
		{"aparentar", "aparent"},
		{"aparente", "aparent"},
		{"aparentemente", "aparent"},
		{"aparentes", "aparent"},
		{"aparentó", "aparent"},
		{"aparezca", "aparezc"},
		{"aparezcan", "aparezc"},
		{"aparición", "aparicion"},
		{"apariciones", "aparicion"},
		{"apariencia", "aparient"},
		{"apariencias", "aparient"},
		{"apartado", "apart"},
		{"apartados", "apart"},
		{"apartamento", "apartament"},
		{"apartamentos", "apartament"},
		{"apartan", "apart"},
		{"aparte", "apart"},
		{"apasionadamente", "apasion"},
		{"apasionado", "apasion"},
		{"apasionamiento", "apasion"},
		{"apasionantes", "apasion"},
		{"apatía", "apat"},
		{"apáticas", "apat"},
		{"apáticos", "apat"},
		{"apedrean", "apedr"},
		{"apegada", "apeg"},
		{"apegadas", "apeg"},
		{"apegan", "apeg"},
		{"apegarnos", "apeg"},
		{"apegarse", "apeg"},
		{"apego", "apeg"},
		{"apelación", "apel"},
		{"apelaciones", "apel"},
		{"apelar", "apel"},
		{"apelarán", "apel"},
		{"apelativos", "apel"},
		{"apellido", "apell"},
		{"apelo", "apel"},
		{"apenado", "apen"},
		{"apenaron", "apen"},
		{"apenas", "apen"},
		{"apéndice", "apendic"},
		{"apendicitis", "apendicitis"},
		{"apertura", "apertur"},
		{"apetito", "apetit"},
		{"apizaco", "apizac"},
		{"aplacarlos", "aplac"},
		{"aplasta", "aplast"},
		{"aplastada", "aplast"},
		{"aplastaron", "aplast"},
		{"aplaude", "aplaud"},
		{"aplaudiendo", "aplaud"},
		{"aplaudieran", "aplaud"},
		{"aplaudieron", "aplaud"},
		{"aplaudió", "aplaud"},
		{"aplaudir", "aplaud"},
		{"aplaudirán", "aplaud"},
		{"aplauso", "aplaus"},
		{"aplausos", "aplaus"},
		{"aplaza", "aplaz"},
		{"aplazada", "aplaz"},
		{"aplazamiento", "aplaz"},
		{"aplazó", "aplaz"},
		{"aplica", "aplic"},
		{"aplicables", "aplic"},
		{"aplicación", "aplic"},
		{"aplicaciones", "aplic"},
		{"aplicada", "aplic"},
		{"aplicado", "aplic"},
		{"aplicados", "aplic"},
		{"aplican", "aplic"},
		{"aplicar", "aplic"},
		{"aplicara", "aplic"},
		{"aplicará", "aplic"},
		{"aplicáramos", "aplic"},
		{"aplicarán", "aplic"},
		{"aplicaríamos", "aplic"},
		{"aplicarla", "aplic"},
		{"aplicarle", "aplic"},
		{"aplicarlos", "aplic"},
		{"aplicaron", "aplic"},
		{"aplicarse", "aplic"},
		{"aplico", "aplic"},
		{"aplicó", "aplic"},
		{"aplique", "apliqu"},
		{"apliquen", "apliqu"},
		{"apo", "apo"},
		{"apocada", "apoc"},
		{"apocalipsis", "apocalipsis"},
		{"apocalíptica", "apocalipt"},
		{"apocalípticas", "apocalipt"},
		{"apócrifo", "apocrif"},
		{"apoda", "apod"},
		{"apodaban", "apod"},
		{"apodaca", "apodac"},
		{"apodado", "apod"},
		{"apoderado", "apoder"},
		{"apoderándose", "apoder"},
		{"apoderaron", "apoder"},
		{"apoderó", "apoder"},
		{"apodo", "apod"},
		{"apogeo", "apoge"},
		{"apoplejía", "apoplej"},
		{"apoquinar", "apoquin"},
		{"aporta", "aport"},
		{"aportación", "aport"},
		{"aportaciones", "aport"},
		{"aportado", "aport"},
		{"aportamos", "aport"},
		{"aportan", "aport"},
		{"aportando", "aport"},
		{"aportar", "aport"},
		{"aportará", "aport"},
		{"aportaron", "aport"},
		{"aporte", "aport"},
		{"aporten", "aport"},
		{"aportó", "aport"},
		{"aposentos", "aposent"},
		{"apostados", "apost"},
		{"apostamos", "apost"},
		{"apostar", "apost"},
		{"apostemos", "apost"},
		{"apostó", "apost"},
		{"apóstoles", "apostol"},
		{"apostólica", "apostol"},
		{"apostólico", "apostol"},
		{"apoya", "apoy"},
		{"apoyaban", "apoy"},
		{"apoyada", "apoy"},
		{"apoyado", "apoy"},
		{"apoyados", "apoy"},
		{"apoyamos", "apoy"},
		{"apoyan", "apoy"},
		{"apoyando", "apoy"},
		{"apoyándola", "apoy"},
		{"apoyándonos", "apoy"},
		{"apoyándose", "apoy"},
		{"apoyar", "apoy"},
		{"apoyara", "apoy"},
		{"apoyará", "apoy"},
		{"apoyarán", "apoy"},
		{"apoyaremos", "apoy"},
		{"apoyaría", "apoy"},
		{"apoyarian", "apoyari"},
		{"apoyarlo", "apoy"},
		{"apoyaron", "apoy"},
		{"apoyasen", "apoy"},
		{"apoye", "apoy"},
		{"apoyen", "apoy"},
		{"apoyo", "apoy"},
		{"apoyó", "apoy"},
		{"apoyos", "apoy"},
		{"appel", "appel"},
		{"apple", "apple"},
		{"apra", "apra"},
		{"aprecia", "apreci"},
		{"apreciación", "apreci"},
		{"apreciada", "apreci"},
		{"apreciado", "apreci"},
		{"apreciar", "apreci"},
		{"apreciarlo", "apreci"},
		{"apreciaron", "apreci"},
		{"apreciarse", "apreci"},
		{"aprecio", "apreci"},
		{"apreció", "aprec"},
		{"aprehendido", "aprehend"},
		{"aprehension", "aprehension"},
		{"aprehensión", "aprehension"},
		{"aprehensiones", "aprehension"},
		{"aprehensores", "aprehensor"},
		{"aprenda", "aprend"},
		{"aprendan", "aprend"},
		{"aprende", "aprend"},
		{"aprenden", "aprend"},
		{"aprender", "aprend"},
		{"aprenderá", "aprend"},
		{"aprenderlo", "aprend"},
		{"aprendices", "aprendic"},
		{"aprendido", "aprend"},
		{"aprendiendo", "aprend"},
		{"aprendieron", "aprend"},
		{"aprendió", "aprend"},
		{"aprendiste", "aprend"},
		{"aprendizaje", "aprendizaj"},
		{"aprestos", "aprest"},
		{"apresura", "apresur"},
		{"apresuradamente", "apresur"},
		{"apresurado", "apresur"},
		{"apresuramiento", "apresur"},
		{"apresurando", "apresur"},
		{"apresurar", "apresur"},
		{"apresuró", "apresur"},
		{"apretado", "apret"},
		{"apretamos", "apret"},
		{"apretar", "apret"},
		{"apretarlo", "apret"},
		{"apretaron", "apret"},
		{"apretó", "apret"},
		{"apretón", "apreton"},
		{"aprieta", "apriet"},
		{"apriete", "apriet"},
		{"aprieten", "apriet"},
		{"aprietos", "apriet"},
		{"apro", "apro"},
		{"aprobación", "aprob"},
		{"aprobada", "aprob"},
		{"aprobadas", "aprob"},
		{"aprobado", "aprob"},
		{"aprobados", "aprob"},
		{"aprobar", "aprob"},
		{"aprobará", "aprob"},
		{"aprobarán", "aprob"},
		{"aprobaron", "aprob"},
		{"aprobarse", "aprob"},
		{"aprobatorio", "aprobatori"},
		{"aprobó", "aprob"},
		{"apropiada", "apropi"},
		{"apropiadamente", "apropi"},
		{"apropiadas", "apropi"},
		{"apropiado", "apropi"},
		{"apropiados", "apropi"},
		{"apropian", "apropi"},
		{"aprovecha", "aprovech"},
		{"aprovechado", "aprovech"},
		{"aprovechamiento", "aprovech"},
		{"aprovechan", "aprovech"},
		{"aprovechando", "aprovech"},
		{"aprovechar", "aprovech"},
		{"aprovecharán", "aprovech"},
		{"aprovecharla", "aprovech"},
		{"aprovecharlas", "aprovech"},
		{"aprovecharon", "aprovech"},
		{"aprovecharse", "aprovech"},
		{"aproveche", "aprovech"},
		{"aprovechó", "aprovech"},
		{"aproximaba", "aproxim"},
		{"aproximación", "aproxim"},
		{"aproximada", "aproxim"},
		{"aproximadamente", "aproxim"},
		{"aproximado", "aproxim"},
		{"aproximados", "aproxim"},
		{"aproximan", "aproxim"},
		{"aprueba", "aprueb"},
		{"aprueban", "aprueb"},
		{"apruebe", "aprueb"},
		{"aprueben", "aprueb"},
		{"aptas", "aptas"},
		{"aptitud", "aptitud"},
		{"aptitudes", "aptitud"},
		{"aptos", "aptos"},
		{"apuesta", "apuest"},
		{"apuestas", "apuest"},
		{"apuesto", "apuest"},
		{"apuestos", "apuest"},
		{"apunta", "apunt"},
		{"apuntalado", "apuntal"},
		{"apuntalar", "apuntal"},
		{"apuntan", "apunt"},
		{"apuntar", "apunt"},
		{"apuntaron", "apunt"},
		{"apuntes", "apunt"},
		{"apuntó", "apunt"},
		{"aquel", "aquel"},
		{"aquél", "aquel"},
		{"aquelarre", "aquelarr"},
		{"aquelarres", "aquelarr"},
		{"aquella", "aquell"},
		{"aquellas", "aquell"},
		{"aquéllas", "aquell"},
		{"aquello", "aquell"},
		{"aquellos", "aquell"},
		{"aquéllos", "aquell"},
		{"aqui", "aqui"},
		{"aquí", "aqu"},
		{"aquiles", "aquil"},
		{"aquino", "aquin"},
		{"arabe", "arab"},
		{"árabe", "arab"},
		{"árabes", "arab"},
		{"arábiga", "arabig"},
		{"arado", "arad"},
		{"aragón", "aragon"},
		{"aragua", "aragu"},
		{"araiza", "araiz"},
		{"arana", "aran"},
		{"arañas", "arañ"},
		{"arancel", "arancel"},
		{"arancelarias", "arancelari"},
		{"aranceles", "arancel"},
		{"aranda", "arand"},
		{"arango", "arang"},
		{"aranjuez", "aranjuez"},
		{"aras", "aras"},
		{"arau", "arau"},
		{"araujo", "arauj"},
		{"arbitraje", "arbitraj"},
		{"arbitral", "arbitral"},
		{"arbitraria", "arbitrari"},
		{"arbitrarias", "arbitrari"},
		{"arbitrariedades", "arbitrariedad"},
		{"arbitrario", "arbitrari"},
		{"arbitrio", "arbitri"},
		{"árbitro", "arbitr"},
		{"árbitros", "arbitr"},
		{"árbol", "arbol"},
		{"arboles", "arbol"},
		{"árboles", "arbol"},
		{"arbusto", "arbust"},
		{"arcadio", "arcadi"},
		{"arcángeles", "arcangel"},
		{"archaelogy", "archaelogy"},
		{"archiconservadores", "archiconserv"},
		{"archiduque", "archiduqu"},
		{"archipiélago", "archipielag"},
		{"archivada", "archiv"},
		{"archivo", "archiv"},
		{"archundia", "archundi"},
		{"arcilla", "arcill"},
		{"arcillistas", "arcill"},
		{"arco", "arco"},
		{"arcoiris", "arcoiris"},
		{"arcusa", "arcus"},
		{"ardía", "ardi"},
		{"ardiente", "ardient"},
		{"ardientes", "ardient"},
		{"arduo", "ardu"},
		{"área", "are"},
		{"areas", "are"},
		{"áreas", "are"},
		{"arellanes", "arellan"},
		{"arena", "aren"},
		{"arenal", "arenal"},
		{"areoportuarias", "areoportuari"},
		{"aretes", "aret"},
		{"arévalo", "areval"},
		{"argentina", "argentin"},
		{"argentinabrasil", "argentinabrasil"},
		{"argentino", "argentin"},
		{"argentinos", "argentin"},
		{"argoytia", "argoyti"},
		{"argudín", "argudin"},
		{"arguelles", "arguell"},
		{"argumenta", "argument"},
		{"argumentando", "argument"},
		{"argumentar", "argument"},
		{"argumentarán", "argument"},
		{"argumentaron", "argument"},
		{"argumento", "argument"},
		{"argumentó", "argument"},
		{"argumentos", "argument"},
		{"arias", "ari"},
		{"ariel", "ariel"},
		{"ariete", "ariet"},
		{"arista", "arist"},
		{"aristócratas", "aristocrat"},
		{"aristóteles", "aristotel"},
		{"aritmética", "aritmet"},
		{"aritmético", "aritmet"},
		{"arizmendi", "arizmendi"},
		{"arizona", "arizon"},
		{"arkansas", "arkans"},
		{"arlette", "arlett"},
		{"arma", "arma"},
		{"armada", "armad"},
		{"armadas", "armad"},
		{"armadillo", "armadill"},
		{"armado", "armad"},
		{"armadoras", "armador"},
		{"armados", "armad"},
		{"armagedón", "armagedon"},
		{"armamento", "armament"},
		{"armamentos", "armament"},
		{"armando", "armand"},
		{"armani", "armani"},
		{"armar", "armar"},
		{"armas", "armas"},
		{"armella", "armell"},
		{"armendáriz", "armendariz"},
		{"armenio", "armeni"},
		{"armillita", "armillit"},
		{"armisticio", "armistici"},
		{"armó", "armo"},
		{"armonía", "armon"},
		{"armónica", "armon"},
		{"armónico", "armon"},
		{"armoniosa", "armoni"},
		{"armonización", "armoniz"},
		{"armonizar", "armoniz"},
		{"armonizarse", "armoniz"},
		{"armstrong", "armstrong"},
		{"army", "army"},
		{"arnaldo", "arnald"},
		{"arnoldo", "arnold"},
		{"arnulfo", "arnulf"},
		{"aromas", "arom"},
		{"aromatizantes", "aromatiz"},
		{"aromatizar", "aromatiz"},
		{"aros", "aros"},
		{"arq", "arq"},
		{"arqueóloga", "arqueolog"},
		{"arqueología", "arqueolog"},
		{"arqueológica", "arqueolog"},
		{"arqueológicas", "arqueolog"},
		{"arqueológico", "arqueolog"},
		{"arqueológicos", "arqueolog"},
		{"arqueólogos", "arqueolog"},
		{"arquero", "arquer"},
		{"arquidiócesis", "arquidiocesis"},
		{"arquitecto", "arquitect"},
		{"arquitectónicamente", "arquitecton"},
		{"", ""},
		{"arquitectónicas", "arquitecton"},
		{"arquitectónico", "arquitecton"},
		{"arquitectos", "arquitect"},
		{"arquitectura", "arquitectur"},
		{"arquitecturas", "arquitectur"},
		{"arracadas", "arrac"},
		{"arrachera", "arracher"},
		{"arraigada", "arraig"},
		{"arraigado", "arraig"},
		{"arraigo", "arraig"},
		{"arrambide", "arrambid"},
		{"arranca", "arranc"},
		{"arrancados", "arranc"},
		{"arrancando", "arranc"},
		{"arrancar", "arranc"},
		{"arrancará", "arranc"},
		{"arrancarán", "arranc"},
		{"arrancaremos", "arranc"},
		{"arrancaron", "arranc"},
		{"arrancárselo", "arranc"},
		{"arrancó", "arranc"},
		{"arranque", "arranqu"},
		{"arrasaron", "arras"},
		{"arrastrada", "arrastr"},
		{"arrastrados", "arrastr"},
		{"arrastrando", "arrastr"},
		{"arrastre", "arrastr"},
		{"arrastró", "arrastr"},
		{"arreaga", "arreag"},
		{"arrear", "arre"},
		{"arrebatada", "arrebat"},
		{"arrebatar", "arrebat"},
		{"arrebatara", "arrebat"},
		{"arrebató", "arrebat"},
		{"arredondo", "arredond"},
		{"arredra", "arredr"},
		{"arregla", "arregl"},
		{"arreglada", "arregl"},
		{"arreglan", "arregl"},
		{"arreglar", "arregl"},
		{"arreglaran", "arregl"},
		{"arreglarse", "arregl"},
		{"arreglas", "arregl"},
		{"arreglo", "arregl"},
		{"arregló", "arregl"},
		{"arreglos", "arregl"},
		{"arremangó", "arremang"},
		{"arremetió", "arremet"},
		{"arrendaba", "arrend"},
		{"arrendador", "arrend"},
		{"arrendadora", "arrend"},
		{"arrendadoras", "arrend"},
		{"arrendamiento", "arrend"},
		{"arrendamientos", "arrend"},
		{"arrendatario", "arrendatari"},
		{"arreola", "arreol"},
		{"arrepentido", "arrepent"},
		{"arrepentimiento", "arrepent"},
		{"arrepiente", "arrepient"},
		{"arrestado", "arrest"},
		{"arrestados", "arrest"},
		{"arrested", "arrest"},
		{"arresto", "arrest"},
		{"arrestó", "arrest"},
		{"arrestos", "arrest"},
		{"arriaga", "arriag"},
		{"arriba", "arrib"},
		{"arribo", "arrib"},
		{"arribó", "arrib"},
		{"arriesga", "arriesg"},
		{"arriesgada", "arriesg"},
		{"arriesgado", "arriesg"},
		{"arriesgamos", "arriesg"},
		{"arriesgando", "arriesg"},
		{"arriesgar", "arriesg"},
		{"arriesgarían", "arriesg"},
		{"arriesgarlo", "arriesg"},
		{"arriesgarse", "arriesg"},
		{"arrimo", "arrim"},
		{"arrivederci", "arrivederci"},
		{"arrobamiento", "arrob"},
		{"arrodillar", "arrodill"},
		{"arrogante", "arrog"},
		{"arroja", "arroj"},
		{"arrojaban", "arroj"},
		{"arrojada", "arroj"},
		{"arrojado", "arroj"},
		{"arrojados", "arroj"},
		{"arrojan", "arroj"},
		{"arrojando", "arroj"},
		{"arrojar", "arroj"},
		{"arrojarlos", "arroj"},
		{"arrojaron", "arroj"},
		{"arrojen", "arroj"},
		{"arrojó", "arroj"},
		{"arrolladas", "arroll"},
		{"arrollado", "arroll"},
		{"arrolló", "arroll"},
		{"arropados", "arrop"},
		{"arrópese", "arropes"},
		{"arroyo", "arroy"},
		{"arroyó", "arroy"},
		{"arroyos", "arroy"},
		{"arroz", "arroz"},
		{"arrugas", "arrug"},
		{"arruinado", "arruin"},
		{"arruinan", "arruin"},
		{"arruinarse", "arruin"},
		{"arrutinamos", "arrutin"},
		{"arsenio", "arseni"},
		{"art", "art"},
		{"arte", "arte"},
		{"arteaga", "arteag"},
		{"artefacto", "artefact"},
		{"artemio", "artemi"},
		{"arteria", "arteri"},
		{"arterias", "arteri"},
		{"arterioesclerosis", "arterioesclerosis"},
		{"arteroesclerosis", "arteroesclerosis"},
		{"artes", "artes"},
		{"artesanal", "artesanal"},
		{"artesanales", "artesanal"},
		{"artesanas", "artesan"},
		{"artesanías", "artesan"},
		{"artesianos", "artesian"},
		{"arthur", "arthur"},
		{"articulación", "articul"},
		{"articulado", "articul"},
		{"artículo", "articul"},
		{"artículos", "articul"},
		{"artífice", "artific"},
		{"artificial", "artificial"},
		{"artificiales", "artificial"},
		{"artificialmente", "artificial"},
		{"artificio", "artifici"},
		{"artillería", "artill"},
		{"artilleros", "artiller"},
		{"artista", "artist"},
		{"artistas", "artist"},
		{"artística", "artist"},
		{"artísticas", "artist"},
		{"artístico", "artist"},
		{"artísticos", "artist"},
		{"arturo", "artur"},
		{"arvernse", "arverns"},
		{"aryca", "aryca"},
		{"arzobispal", "arzobispal"},
		{"arzobispo", "arzobisp"},
		{"as", "as"},
		{"asa", "asa"},
		{"asada", "asad"},
		{"asadas", "asad"},
		{"asaltaba", "asalt"},
		{"asaltada", "asalt"},
		{"asaltado", "asalt"},
		{"asaltados", "asalt"},
		{"asaltan", "asalt"},
		{"asaltante", "asalt"},
		{"asaltantes", "asalt"},
		{"asaltar", "asalt"},
		{"asaltaron", "asalt"},
		{"asalto", "asalt"},
		{"asaltó", "asalt"},
		{"asaltos", "asalt"},
		{"asamblea", "asamble"},
		{"asambleas", "asambl"},
		{"asambleísas", "asambleis"},
		{"ascencio", "ascenci"},
		{"ascendencia", "ascendent"},
		{"ascendente", "ascendent"},
		{"ascenderá", "ascend"},
		{"ascenderían", "ascend"},
		{"ascendido", "ascend"},
		{"ascendiendo", "ascend"},
		{"ascendieron", "ascend"},
		{"ascendió", "ascend"},
		{"ascenso", "ascens"},
		{"ascensos", "ascens"},
		{"asciende", "asciend"},
		{"ascienden", "asciend"},
		{"aseados", "ase"},
		{"asegura", "asegur"},
		{"aseguraba", "asegur"},
		{"asegurada", "asegur"},
		{"aseguradas", "asegur"},
		{"asegurado", "asegur"},
		{"aseguradora", "asegur"},
		{"asegurados", "asegur"},
		{"aseguramiento", "asegur"},
		{"aseguran", "asegur"},
		{"asegurar", "asegur"},
		{"aseguraremos", "asegur"},
		{"asegurarle", "asegur"},
		{"aseguraron", "asegur"},
		{"aseguren", "asegur"},
		{"asegúrese", "asegures"},
		{"aseguró", "asegur"},
		{"asemeja", "asemej"},
		{"asemejaba", "asemej"},
		{"asemex", "asemex"},
		{"asención", "asencion"},
		{"asentadas", "asent"},
		{"asentado", "asent"},
		{"asentamiento", "asent"},
		{"asentamientos", "asent"},
		{"asentándose", "asent"},
		{"asentó", "asent"},
		{"asertivo", "asert"},
		{"ases", "ases"},
		{"asesinada", "asesin"},
		{"asesinado", "asesin"},
		{"asesinar", "asesin"},
		{"asesinaron", "asesin"},
		{"asesinato", "asesinat"},
		{"asesinatos", "asesinat"},
		{"asesino", "asesin"},
		{"asesinó", "asesin"},
		{"asesinos", "asesin"},
		{"asesor", "asesor"},
		{"asesora", "asesor"},
		{"asesoraban", "asesor"},
		{"asesorado", "asesor"},
		{"asesoramiento", "asesor"},
		{"asesorar", "asesor"},
		{"asesorarlo", "asesor"},
		{"asesores", "asesor"},
		{"asesoría", "asesor"},
		{"asestadas", "asest"},
		{"asestó", "asest"},
		{"asevera", "asever"},
		{"aseverando", "asever"},
		{"aseveraron", "asever"},
		{"aseveró", "asever"},
		{"asfalto", "asfalt"},
		{"asfixia", "asfixi"},
		{"ashida", "ashid"},
		{"asi", "asi"},
		{"así", "asi"},
		{"asia", "asi"},
		{"asiático", "asiat"},
		{"asic", "asic"},
		{"asiduo", "asidu"},
		{"asienta", "asient"},
		{"asiento", "asient"},
		{"asientos", "asient"},
		{"asigna", "asign"},
		{"asignación", "asign"},
		{"asignadas", "asign"},
		{"asignado", "asign"},
		{"asignados", "asign"},
		{"asignan", "asign"},
		{"asignando", "asign"},
		{"asignar", "asign"},
		{"asignaron", "asign"},
		{"asignen", "asign"},
		{"asimétrica", "asimetr"},
		{"asimilación", "asimil"},
		{"asimilar", "asimil"},
		{"asimilemos", "asimil"},
		{"asimismo", "asim"},
		{"asímismo", "asim"},
		{"asista", "asist"},
		{"asistan", "asist"},
		{"asiste", "asist"},
		{"asisten", "asist"},
		{"asistencia", "asistent"},
		{"asistenciales", "asistencial"},
		{"asistencias", "asistent"},
		{"asistentes", "asistent"},
		{"asistían", "asist"},
		{"asistiendo", "asist"},
		{"asistiéndose", "asist"},
		{"asistiera", "asist"},
		{"asistieron", "asist"},
		{"asistió", "asist"},
		{"asistir", "asist"},
		{"asistirá", "asist"},
		{"asistirán", "asist"},
		{"asitex", "asitex"},
		{"asocia", "asoci"},
		{"asociación", "asoci"},
		{"asociaciones", "asoci"},
		{"asociada", "asoci"},
		{"asociado", "asoci"},
		{"asociados", "asoci"},
		{"asocian", "asoci"},
		{"asociar", "asoci"},
		{"asociaron", "asoci"},
		{"asociarse", "asoci"},
		{"asombra", "asombr"},
		{"asombradas", "asombr"},
		{"asómbrese", "asombres"},
		{"asombro", "asombr"},
		{"asombrosa", "asombr"},
		{"asombrosamente", "asombr"},
		{"asómese", "asomes"},
		{"asomo", "asom"},
		{"asomó", "asom"},
		{"asotiation", "asotiation"},
		{"aspavientos", "aspavient"},
		{"aspe", "aspe"},
		{"aspecto", "aspect"},
		{"aspectos", "aspect"},
		{"ásperamente", "asper"},
		{"áspero", "asper"},
		{"aspersión", "aspersion"},
		{"aspes", "aspes"},
		{"aspetia", "aspeti"},
		{"aspira", "aspir"},
		{"aspiraciones", "aspir"},
		{"aspiran", "aspir"},
		{"aspirante", "aspir"},
		{"aspirantes", "aspir"},
		{"aspirar", "aspir"},
		{"aspirara", "aspir"},
		{"assessment", "assessment"},
		{"associates", "associat"},
		{"association", "association"},
		{"astado", "astad"},
		{"asteroide", "asteroid"},
		{"asteroides", "asteroid"},
		{"astro", "astro"},
		{"astroditomas", "astroditom"},
		{"astrología", "astrolog"},
		{"astrológico", "astrolog"},
		{"astronautas", "astronaut"},
		{"astronómicos", "astronom"},
		{"astros", "astros"},
		{"asuma", "asum"},
		{"asuman", "asum"},
		{"asume", "asum"},
		{"asumen", "asum"},
		{"asumido", "asum"},
		{"asumiendo", "asum"},
		{"asumiera", "asum"},
		{"asumieron", "asum"},
		{"asumimos", "asum"},
		{"asumió", "asum"},
		{"asumir", "asum"},
		{"asumirá", "asum"},
		{"asuntito", "asuntit"},
		{"asunto", "asunt"},
		{"asuntos", "asunt"},
		{"asustada", "asust"},
		{"asustado", "asust"},
		{"asustar", "asust"},
		{"asuste", "asust"},
		{"asusten", "asust"},
		{"at", "at"},
		{"ataca", "atac"},
		{"atacada", "atac"},
		{"atacado", "atac"},
		{"atacados", "atac"},
		{"atacan", "atac"},
		{"atacando", "atac"},
		{"atacándolo", "atac"},
		{"atacante", "atac"},
		{"atacantes", "atac"},
		{"atacar", "atac"},
		{"atacará", "atac"},
		{"atacarán", "atac"},
		{"atacarlos", "atac"},
		{"atacaron", "atac"},
		{"atacas", "atac"},
		{"atacó", "atac"},
		{"atadas", "atad"},
		{"atados", "atad"},
		{"atanasio", "atanasi"},
		{"atañe", "atañ"},
		{"ataque", "ataqu"},
		{"ataques", "ataqu"},
		{"atardecer", "atardec"},
		{"atarlos", "atarl"},
		{"ataron", "ataron"},
		{"atascadas", "atasc"},
		{"atascón", "atascon"},
		{"atasta", "atast"},
		{"ataviado", "atavi"},
		{"ataviados", "atavi"},
		{"atávicos", "atav"},
		{"atemorizante", "atemoriz"},
		{"atención", "atencion"},
		{"atenciones", "atencion"},
		{"atendamos", "atend"},
		{"atender", "atend"},
		{"atenderá", "atend"},
		{"atenderán", "atend"},
		{"atenderían", "atend"},
		{"atenderlos", "atend"},
		{"atendía", "atend"},
		{"atendida", "atend"},
		{"atendidas", "atend"},
		{"atendido", "atend"},
		{"atendidos", "atend"},
		{"atendieran", "atend"},
		{"atendieron", "atend"},
		{"atendió", "atend"},
		{"atenemos", "aten"},
		{"atenta", "atent"},
		{"atentado", "atent"},
		{"atentados", "atent"},
		{"atentamente", "atent"},
		{"atentar", "atent"},
		{"atenten", "atent"},
		{"atento", "atent"},
		{"atentos", "atent"},
		{"aterra", "aterr"},
		{"aterrados", "aterr"},
		{"aterrizaje", "aterrizaj"},
		{"aterrorizada", "aterroriz"},
		{"ates", "ates"},
		{"athlone", "athlon"},
		{"atiborran", "atiborr"},
		{"atienda", "atiend"},
		{"atiende", "atiend"},
		{"atienden", "atiend"},
		{"atinada", "atin"},
		{"atinadamente", "atin"},
		{"atinados", "atin"},
		{"atípico", "atip"},
		{"atlanta", "atlant"},
		{"atlante", "atlant"},
		{"atlantic", "atlantic"},
		{"atlántico", "atlant"},
		{"atlantis", "atlantis"},
		{"atlantista", "atlant"},
		{"atlantistas", "atlant"},
		{"atlas", "atlas"},
		{"atleta", "atlet"},
		{"atletas", "atlet"},
		{"atlético", "atlet"},
		{"atletismo", "atlet"},
		{"atmósfera", "atmosfer"},
		{"atmosférica", "atmosfer"},
		{"atmosférico", "atmosfer"},
		{"atolladero", "atollader"},
		{"atolondradas", "atolondr"},
		{"atómica", "atom"},
		{"atómicas", "atom"},
		{"atomizador", "atomiz"},
		{"atónitos", "atonit"},
		{"atópica", "atop"},
		{"atosigaba", "atosig"},
		{"atp", "atp"},
		{"atracados", "atrac"},
		{"atracción", "atraccion"},
		{"atraco", "atrac"},
		{"atractiva", "atract"},
		{"atractivas", "atract"},
		{"atractivo", "atract"},
		{"atractivos", "atract"},
		{"atrae", "atra"},
		{"atraen", "atra"},
		{"atraer", "atra"},
		{"atraídos", "atraid"},
		{"atrajo", "atraj"},
		{"atrapadas", "atrap"},
		{"atrapado", "atrap"},
		{"atrapados", "atrap"},
		{"atrapan", "atrap"},
		{"atrápelo", "atrapel"},
		{"atrás", "atras"},
		{"atrasaditos", "atrasadit"},
		{"atrasado", "atras"},
		{"atrasados", "atras"},
		{"atrasarnos", "atras"},
		{"atrasen", "atras"},
		{"atraso", "atras"},
		{"atravesado", "atraves"},
		{"atravesaron", "atraves"},
		{"atraviesa", "atravies"},
		{"atraviesan", "atravies"},
		{"atrevería", "atrev"},
		{"atrevida", "atrev"},
		{"atrevido", "atrev"},
		{"atrevo", "atrev"},
		{"atribuciones", "atribu"},
		{"atribuible", "atribu"},
		{"atribuibles", "atribu"},
		{"atribuido", "atribu"},
		{"atribuirse", "atribu"},
		{"atribulado", "atribul"},
		{"atribulados", "atribul"},
		{"atributos", "atribut"},
		{"atribuye", "atribu"},
		{"atribuyó", "atribu"},
		{"atrofia", "atrofi"},
		{"atropella", "atropell"},
		{"atropellado", "atropell"},
		{"atropellar", "atropell"},
		{"atropelló", "atropell"},
		{"atte", "atte"},
		{"atuendos", "atuend"},
		{"atzcapotzalco", "atzcapotzalc"},
		{"auburn", "auburn"},
		{"audacia", "audaci"},
		{"audaz", "audaz"},
		{"audelo", "audel"},
		{"audición", "audicion"},
		{"audicionar", "audicion"},
		{"audiencia", "audienci"},
		{"audífonos", "audifon"},
		{"audio", "audi"},
		{"audiovisuales", "audiovisual"},
		{"auditiva", "audit"},
		{"auditivo", "audit"},
		{"auditores", "auditor"},
		{"auditoria", "auditori"},
		{"auditoría", "auditor"},
		{"auditorías", "auditor"},
		{"auditorio", "auditori"},
		{"auditorium", "auditorium"},
		{"auge", "aug"},
		{"augura", "augur"},
		{"augurio", "auguri"},
		{"augurios", "auguri"},
		{"auguró", "augur"},
		{"august", "august"},
		{"augusto", "august"},
		{"augustus", "augustus"},
		{"aula", "aul"},
		{"aulas", "aul"},
		{"aumenta", "aument"},
		{"aumentaba", "aument"},
		{"aumentado", "aument"},
		{"aumentar", "aument"},
		{"aumentara", "aument"},
		{"aumentará", "aument"},
		{"aumentarán", "aument"},
		{"aumentaron", "aument"},
		{"aumente", "aument"},
		{"aumenten", "aument"},
		{"aumento", "aument"},
		{"aumentó", "aument"},
		{"aumentos", "aument"},
		{"aun", "aun"},
		{"aún", "aun"},
		{"aunado", "aun"},
		{"aunque", "aunqu"},
		{"aurelio", "aureli"},
		{"aureola", "aureol"},
		{"auriazul", "auriazul"},
		{"aurio", "auri"},
		{"aurora", "auror"},
		{"aurrera", "aurrer"},
		{"aurrerá", "aurr"},
		{"aurreras", "aurrer"},
		{"ausencia", "ausenci"},
		{"ausentarse", "ausent"},
		{"ausente", "ausent"},
		{"ausentes", "ausent"},
		{"ausentismo", "ausent"},
		{"auspicia", "auspici"},
		{"auspiciadoras", "auspici"},
		{"auspiciará", "auspici"},
		{"auspicio", "auspici"},
		{"auspicios", "auspici"},
		{"austeras", "auster"},
		{"austeridad", "auster"},
		{"austero", "auster"},
		{"austeros", "auster"},
		{"austin", "austin"},
		{"australia", "australi"},
		{"australian", "australi"},
		{"australianos", "australian"},
		{"austriacos", "austriac"},
		{"autárticos", "autart"},
		{"auténticamente", "autent"},
		{"auténtico", "autent"},
		{"auténticos", "autent"},
		{"auto", "aut"},
		{"autoanálisis", "autoanalisis"},
		{"autobalance", "autobalanc"},
		{"autobiografía", "autobiograf"},
		{"autobús", "autobus"},
		{"autobuses", "autobus"},
		{"autocalifica", "autocalif"},
		{"autocalificación", "autocalif"},
		{"autocalificada", "autocalific"},
		{"autocalificar", "autocalific"},
		{"autocontrol", "autocontrol"},
		{"autocrítica", "autocrit"},
		{"autódromo", "autodrom"},
		{"autoestima", "autoestim"},
		{"autoevaluación", "autoevalu"},
		{"autofinanciamiento", "autofinanci"},
		{"autofinanciamientos", "autofinanci"},
		{"autogol", "autogol"},
		{"autógrafos", "autograf"},
		{"automática", "automat"},
		{"automáticamente", "automat"},
		{"automáticas", "automat"},
		{"automático", "automat"},
		{"automatización", "automatiz"},
		{"automatizado", "automatiz"},
		{"automatizados", "automatiz"},
		{"automotor", "automotor"},
		{"automotores", "automotor"},
		{"automotrices", "automotric"},
		{"automotriz", "automotriz"},
		{"automovil", "automovil"},
		{"automóvil", "automovil"},
		{"automoviles", "automovil"},
		{"automóviles", "automovil"},
		{"automovilista", "automovil"},
		{"automovilistas", "automovil"},
		{"automovilística", "automovilist"},
		{"automovilístico", "automovilist"},
		{"automovilísticos", "automovilist"},
		{"autónoma", "autonom"},
		{"autonomía", "autonom"},
		{"autónomo", "autonom"},
		{"autónomos", "autonom"},
		{"autopartes", "autopart"},
		{"autopista", "autop"},
		{"autopistas", "autop"},
		{"autopsia", "autopsi"},
		{"autor", "autor"},
		{"autora", "autor"},
		{"autoregulación", "autoregul"},
		{"autores", "autor"},
		{"autorice", "autoric"},
		{"autoridad", "autor"},
		{"autoridades", "autor"},
		{"autoritaria", "autoritari"},
		{"autoritario", "autoritari"},
		{"autoritarismo", "autoritar"},
		{"autoritarismos", "autoritar"},
		{"autoriza", "autoriz"},
		{"autorización", "autoriz"},
		{"autorizaciones", "autoriz"},
		{"autorizada", "autoriz"},
		{"autorizadas", "autoriz"},
		{"autorizado", "autoriz"},
		{"autorizados", "autoriz"},
		{"autorizan", "autoriz"},
		{"autorizando", "autoriz"},
		{"autorizar", "autoriz"},
		{"autorizaran", "autoriz"},
		{"autorizarse", "autoriz"},
		{"autorizó", "autoriz"},
		{"autos", "aut"},
		{"autoservicio", "autoservici"},
		{"autoservicios", "autoservici"},
		{"autosuficiencia", "autosuficient"},
		{"autosuficientes", "autosuficient"},
		{"autotanques", "autotanqu"},
		{"autotransporte", "autotransport"},
		{"autotransportistas", "autotransport"},
		{"autrey", "autrey"},
		{"auxilia", "auxili"},
		{"auxiliado", "auxili"},
		{"auxiliados", "auxili"},
		{"auxiliando", "auxili"},
		{"auxiliar", "auxili"},
		{"auxiliares", "auxiliar"},
		{"auxiliarla", "auxili"},
		{"auxiliaron", "auxili"},
		{"auxilio", "auxili"},
		{"auxilios", "auxili"},
		{"av", "av"},
		{"aval", "aval"},
		{"avala", "aval"},
		{"avalaban", "aval"},
		{"avaladas", "aval"},
		{"avalan", "aval"},
		{"avalanchas", "avalanch"},
		{"avalando", "aval"},
		{"avalar", "aval"},
		{"avalaremos", "aval"},
		{"avalaron", "aval"},
		{"avale", "aval"},
		{"avalúo", "avalu"},
		{"avalúos", "avalu"},
		{"avance", "avanc"},
		{"avances", "avanc"},
		{"avanza", "avanz"},
		{"avanzaba", "avanz"},
		{"avanzada", "avanz"},
		{"avanzadas", "avanz"},
		{"avanzado", "avanz"},
		{"avanzados", "avanz"},
		{"avanzan", "avanz"},
		{"avanzando", "avanz"},
		{"avanzar", "avanz"},
		{"avanzarán", "avanz"},
		{"avanzaron", "avanz"},
		{"avanzó", "avanz"},
		{"avaro", "avar"},
		{"ave", "ave"},
		{"avecina", "avecin"},
		{"avecinado", "avecin"},
		{"avelar", "avel"},
		{"avelina", "avelin"},
		{"avenida", "aven"},
		{"avenidas", "aven"},
		{"aventaba", "avent"},
		{"aventaja", "aventaj"},
		{"aventó", "avent"},
		{"aventura", "aventur"},
		{"aventuradas", "aventur"},
		{"aventurado", "aventur"},
		{"aventuras", "aventur"},
		{"avergonzado", "avergonz"},
		{"avergüencen", "avergüenc"},
		{"averiguación", "averigu"},
		{"averiguaciones", "averigu"},
		{"averiguando", "averigu"},
		{"averiguar", "averigu"},
		{"averió", "aver"},
		{"averoes", "aver"},
		{"aves", "aves"},
		{"avestruz", "avestruz"},
		{"aviación", "aviacion"},
		{"aviacsa", "aviacs"},
		{"aviadurismo", "aviadur"},
		{"ávida", "avid"},
		{"avidez", "avidez"},
		{"ávido", "avid"},
		{"avientan", "avient"},
		{"avila", "avil"},
		{"avilán", "avilan"},
		{"avilés", "aviles"},
		{"avío", "avi"},
		{"avión", "avion"},
		{"aviones", "avion"},
		{"avisamos", "avis"},
		{"avisará", "avis"},
		{"avisarles", "avis"},
		{"avisaron", "avis"},
		{"aviso", "avis"},
		{"avisó", "avis"},
		{"avisora", "avisor"},
		{"avocara", "avoc"},
		{"awards", "awards"},
		{"axilar", "axil"},
		{"ay", "ay"},
		{"ayacucho", "ayacuch"},
		{"ayala", "ayal"},
		{"ayarzagoitia", "ayarzagoiti"},
		{"ayense", "ayens"},
		{"ayer", "ayer"},
		{"ayuda", "ayud"},
		{"ayudaba", "ayud"},
		{"ayudada", "ayud"},
		{"ayudado", "ayud"},
		{"ayudados", "ayud"},
		{"ayudamos", "ayud"},
		{"ayudan", "ayud"},
		{"ayudando", "ayud"},
		{"ayudándole", "ayud"},
		{"ayudándose", "ayud"},
		{"ayudante", "ayud"},
		{"ayudantes", "ayud"},
		{"ayudar", "ayud"},
		{"ayudara", "ayud"},
		{"ayudará", "ayud"},
		{"ayudaran", "ayud"},
		{"ayudaría", "ayud"},
		{"ayudarle", "ayud"},
		{"ayudarlo", "ayud"},
		{"ayudarlos", "ayud"},
		{"ayudaron", "ayud"},
		{"ayude", "ayud"},
		{"ayuden", "ayud"},
		{"ayudó", "ayud"},
		{"ayuntamiento", "ayunt"},
		{"ayuntamientos", "ayunt"},
		{"azáleas", "azal"},
		{"azalia", "azali"},
		{"azar", "azar"},
		{"azcárraga", "azcarrag"},
		{"azcua", "azcu"},
		{"azhamar", "azham"},
		{"azido", "azid"},
		{"azorrillan", "azorrill"},
		{"azotaba", "azot"},
		{"azotado", "azot"},
		{"azotando", "azot"},
		{"azteca", "aztec"},
		{"aztecas", "aztec"},
		{"aztlán", "aztlan"},
		{"azucar", "azuc"},
		{"azúcar", "azuc"},
		{"azúcares", "azucar"},
		{"azucena", "azucen"},
		{"azufre", "azufr"},
		{"azul", "azul"},
		{"azúl", "azul"},
		{"azules", "azul"},
		{"azulgrana", "azulgran"},
		{"azulosos", "azul"},
		{"azuma", "azum"},
		{"azumah", "azumah"},
		{"b", "b"},
		{"bab", "bab"},
		{"babalucas", "babaluc"},
		{"babilonia", "babiloni"},
		{"baby", "baby"},
		{"baccantes", "baccant"},
		{"bacco", "bacc"},
		{"bacheando", "bach"},
		{"bachilleres", "bachiller"},
		{"bacteria", "bacteri"},
		{"badmington", "badmington"},
		{"bae", "bae"},
		{"báez", "baez"},
		{"bafsa", "bafs"},
		{"bagaje", "bagaj"},
		{"bagdad", "bagd"},
		{"bahamas", "baham"},
		{"bahía", "bah"},
		{"baila", "bail"},
		{"bailable", "bailabl"},
		{"bailando", "bail"},
		{"bailar", "bail"},
		{"bailarines", "bailarin"},
		{"bailarlos", "bail"},
		{"baile", "bail"},
		{"bailes", "bail"},
		{"bailleres", "bailler"},
		{"baja", "baj"},
		{"bajacaliforniano", "bajacalifornian"},
		{"bajadas", "baj"},
		{"bajado", "baj"},
		{"bajan", "baj"},
		{"bajando", "baj"},
		{"bajar", "baj"},
		{"bajará", "baj"},
		{"bajarle", "baj"},
		{"bajaron", "baj"},
		{"bajarse", "baj"},
		{"bajas", "baj"},
		{"baje", "baj"},
		{"bajío", "baji"},
		{"bajista", "bajist"},
		{"bajo", "baj"},
		{"bajó", "baj"},
		{"bajonazo", "bajonaz"},
		{"bajos", "baj"},
		{"bala", "bal"},
		{"balaceados", "balac"},
		{"balacearlo", "balac"},
		{"balaceó", "balace"},
		{"balacera", "balacer"},
		{"balaceras", "balacer"},
		{"baladas", "bal"},
		{"balance", "balanc"},
		{"balanceadas", "balanc"},
		{"balanceado", "balanc"},
		{"balancear", "balanc"},
		{"balanza", "balanz"},
		{"balas", "bal"},
		{"balazo", "balaz"},
		{"balazos", "balaz"},
		{"balbina", "balbin"},
		{"balcanes", "balcan"},
		{"balcón", "balcon"},
		{"balcones", "balcon"},
		{"balderas", "balder"},
		{"baldío", "baldi"},
		{"baldomero", "baldomer"},
		{"baleada", "bal"},
		{"balística", "balist"},
		{"ball", "ball"},
		{"ballard", "ballard"},
		{"ballén", "ballen"},
		{"ballena", "ballen"},
		{"ballet", "ballet"},
		{"balompié", "balompi"},
		{"balón", "balon"},
		{"balones", "balon"},
		{"balsa", "bals"},
		{"baltazar", "baltaz"},
		{"baltimore", "baltimor"},
		{"baluarte", "baluart"},
		{"bamba", "bamb"},
		{"bambalinas", "bambalin"},
		{"baña", "bañ"},
		{"bañada", "bañ"},
		{"banamex", "banamex"},
		{"bañan", "bañ"},
		{"bananero", "bananer"},
		{"bañar", "bañ"},
		{"bañarse", "bañ"},
		{"banca", "banc"},
		{"bancaria", "bancari"},
		{"bancarias", "bancari"},
		{"bancario", "bancari"},
		{"bancarios", "bancari"},
		{"bancas", "banc"},
		{"banco", "banc"},
		{"bancomer", "bancom"},
		{"bancomext", "bancomext"},
		{"bancos", "banc"},
		{"bancrecer", "bancrec"},
		{"band", "band"},
		{"banda", "band"},
		{"bandas", "band"},
		{"bandazos", "bandaz"},
		{"bandeja", "bandej"},
		{"bandera", "bander"},
		{"banderas", "bander"},
		{"banderazo", "banderaz"},
		{"banderillas", "banderill"},
		{"banderillero", "banderiller"},
		{"banderín", "banderin"},
		{"banderita", "banderit"},
		{"bandidos", "band"},
		{"bando", "band"},
		{"bandos", "band"},
		{"bank", "bank"},
		{"bankers", "bankers"},
		{"banking", "banking"},
		{"bankshares", "bankshar"},
		{"baño", "bañ"},
		{"banoro", "banor"},
		{"baños", "bañ"},
		{"banpaís", "banp"},
		{"banquero", "banquer"},
		{"banqueros", "banquer"},
		{"banqueta", "banquet"},
		{"banquetas", "banquet"},
		{"banquillo", "banquill"},
		{"banrural", "banrural"},
		{"banxico", "banxic"},
		{"bar", "bar"},
		{"barahona", "barahon"},
		{"barajas", "baraj"},
		{"baranda", "barand"},
		{"barandal", "barandal"},
		{"barandilla", "barandill"},
		{"barata", "barat"},
		{"baratas", "barat"},
		{"barato", "barat"},
		{"barba", "barb"},
		{"barbacoa", "barbaco"},
		{"barbados", "barb"},
		{"bárbara", "barb"},
		{"barbarabbe", "barbarabb"},
		{"barbaridades", "barbar"},
		{"barbarie", "barbari"},
		{"barberena", "barberen"},
		{"barbosa", "barbos"},
		{"barcelona", "barcelon"},
		{"barceloneses", "barcelones"},
		{"barco", "barc"},
		{"barcos", "barc"},
		{"barda", "bard"},
		{"bardas", "bard"},
		{"bares", "bar"},
		{"barinas", "barin"},
		{"barocio", "baroci"},
		{"barón", "baron"},
		{"barra", "barr"},
		{"barragán", "barragan"},
		{"barranca", "barranc"},
		{"barranco", "barranc"},
		{"barras", "barr"},
		{"barre", "barr"},
		{"barreira", "barreir"},
		{"barrenechea", "barreneche"},
		{"barrera", "barrer"},
		{"barreras", "barrer"},
		{"barrerse", "barr"},
		{"barres", "barr"},
		{"barreto", "barret"},
		{"barrett", "barrett"},
		{"barría", "barr"},
		{"barrial", "barrial"},
		{"barriéndose", "barr"},
		{"barrientos", "barrient"},
		{"barrileras", "barriler"},
		{"barriles", "barril"},
		{"barrio", "barri"},
		{"barrios", "barri"},
		{"barro", "barr"},
		{"barroca", "barroc"},
		{"barroco", "barroc"},
		{"barros", "barr"},
		{"barrote", "barrot"},
		{"barry", "barry"},
		{"bart", "bart"},
		{"bartlett", "bartlett"},
		{"bartletts", "bartletts"},
		{"bartolo", "bartol"},
		{"basa", "bas"},
		{"basaba", "bas"},
		{"basada", "bas"},
		{"basadas", "bas"},
		{"basado", "bas"},
		{"basados", "bas"},
		{"basamos", "bas"},
		{"basándose", "bas"},
		{"basañez", "basañez"},
		{"basáñez", "basañez"},
		{"basar", "bas"},
		{"basará", "bas"},
		{"basarán", "bas"},
		{"basarse", "bas"},
		{"basave", "basav"},
		{"basay", "basay"},
		{"basch", "basch"},
		{"báscula", "bascul"},
		{"básculas", "bascul"},
		{"base", "bas"},
		{"basen", "bas"},
		{"bases", "bas"},
		{"básica", "basic"},
		{"básicamente", "basic"},
		{"basicas", "basic"},
		{"básicas", "basic"},
		{"básico", "basic"},
		{"básicos", "basic"},
		{"basile", "basil"},
		{"basílica", "basil"},
		{"basilio", "basili"},
		{"basó", "bas"},
		{"basquetbol", "basquetbol"},
		{"basset", "basset"},
		{"basta", "bast"},
		{"bastado", "bast"},
		{"bastan", "bast"},
		{"bastante", "bastant"},
		{"bastantes", "bastant"},
		{"bastaría", "bast"},
		{"baste", "bast"},
		{"bastó", "bast"},
		{"basualdo", "basuald"},
		{"basura", "basur"},
		{"basuras", "basur"},
		{"basurero", "basurer"},
		{"basureros", "basurer"},
		{"bat", "bat"},
		{"bata", "bat"},
		{"batalla", "batall"},
		{"batallan", "batall"},
		{"batallar", "batall"},
		{"batallaron", "batall"},
		{"batallas", "batall"},
		{"batallen", "batall"},
		{"batalló", "batall"},
		{"batallón", "batallon"},
		{"batas", "bat"},
		{"bate", "bat"},
		{"bateador", "bateador"},
		{"bateadores", "bateador"},
		{"bateo", "bate"},
		{"bateó", "bate"},
		{"batería", "bat"},
		{"baterías", "bat"},
		{"baterista", "bater"},
		{"batieron", "bat"},
		{"bátik", "batik"},
		{"batir", "bat"},
		{"batis", "batis"},
		{"batistuta", "batistut"},
		{"bátiz", "batiz"},
		{"batocletti", "batocletti"},
		{"battaglia", "battagli"},
		{"baudelaire", "baudelair"},
		{"bautista", "bautist"},
		{"bautizado", "bautiz"},
		{"bautizaría", "bautiz"},
		{"bautizo", "bautiz"},
		{"bautizos", "bautiz"},
		{"bayerische", "bayerisch"},
		{"baz", "baz"},
		{"bazaine", "bazain"},
		{"bazaldúa", "bazaldu"},
		{"bazán", "bazan"},
		{"bazar", "baz"},
		{"bazdresch", "bazdresch"},
		{"bc", "bc"},
		{"bch", "bch"},
		{"beach", "beach"},
		{"beamish", "beamish"},
		{"beatriz", "beatriz"},
		{"beauvais", "beauvais"},
		{"beauvaisis", "beauvaisis"},
		{"beaver", "beav"},
		{"bebé", "beb"},
		{"bebedero", "bebeder"},
		{"beber", "beb"},
		{"bebeto", "bebet"},
		{"bebidas", "beb"},
		{"bebió", "beb"},
		{"bebita", "bebit"},
		{"bebop", "bebop"},
		{"beca", "bec"},
		{"becado", "bec"},
		{"becario", "becari"},
		{"becarios", "becari"},
		{"becas", "bec"},
		{"becerra", "becerr"},
		{"becerril", "becerril"},
		{"becker", "beck"},
		{"becthel", "becthel"},
		{"bedoya", "bedoy"},
		{"beethoven", "beethov"},
		{"beginning", "beginning"},
		{"begiristain", "begiristain"},
		{"begoña", "begoñ"},
		{"begonia", "begoni"},
		{"beige", "beig"},
		{"beirut", "beirut"},
		{"beisbol", "beisbol"},
		{"beisbolero", "beisboler"},
		{"beisbolistas", "beisbol"},
		{"beisbolísticos", "beisbolist"},
		{"beisborama", "beisboram"},
		{"bejarano", "bejaran"},
		{"belair", "bel"},
		{"belaúnde", "belaund"},
		{"belden", "beld"},
		{"belén", "belen"},
		{"belga", "belg"},
		{"belgas", "belg"},
		{"bélgica", "belgic"},
		{"bélica", "belic"},
		{"belice", "belic"},
		{"beliceño", "beliceñ"},
		{"belicosos", "belic"},
		{"bella", "bell"},
		{"bellamente", "bell"},
		{"bellas", "bell"},
		{"beller", "bell"},
		{"belleza", "bellez"},
		{"bello", "bell"},
		{"bellone", "bellon"},
		{"belmont", "belmont"},
		{"ben", "ben"},
		{"benavente", "benavent"},
		{"benavides", "benavid"},
		{"benchmark", "benchmark"},
		{"benchmarking", "benchmarking"},
		{"benchmarks", "benchmarks"},
		{"bend", "bend"},
		{"bendesky", "bendesky"},
		{"bendición", "bendicion"},
		{"bendiciones", "bendicion"},
		{"bendijo", "bendij"},
		{"bendita", "bendit"},
		{"bendito", "bendit"},
		{"beneficencia", "beneficent"},
		{"beneficia", "benefici"},
		{"beneficiada", "benefici"},
		{"beneficiado", "benefici"},
		{"beneficiados", "benefici"},
		{"benefician", "benefici"},
		{"beneficiando", "benefici"},
		{"beneficiar", "benefici"},
		{"beneficiará", "benefici"},
		{"beneficiarán", "benefici"},
		{"beneficiaria", "beneficiari"},
		{"beneficiaría", "benefici"},
		{"beneficien", "benefici"},
		{"beneficio", "benefici"},
		{"benefició", "benefic"},
		{"beneficios", "benefici"},
		{"beneficiosas", "benefici"},
		{"beneficioso", "benefici"},
		{"beneplácito", "beneplacit"},
		{"benevolencia", "benevolent"},
		{"bengoa", "bengo"},
		{"benigno", "benign"},
		{"benítez", "benitez"},
		{"benito", "benit"},
		{"benjamín", "benjamin"},
		{"bennis", "bennis"},
		{"benotto", "benott"},
		{"benz", "benz"},
		{"benzeno", "benzen"},
		{"berchelmann", "berchelmann"},
		{"beregovoy", "beregovoy"},
		{"berenice", "berenic"},
		{"beresford", "beresford"},
		{"bérgamo", "bergam"},
		{"bergerac", "bergerac"},
		{"berges", "berg"},
		{"beria", "beri"},
		{"berlanga", "berlang"},
		{"berlín", "berlin"},
		{"berlinguer", "berlingu"},
		{"berman", "berm"},
		{"bermudas", "bermud"},
		{"bermúdez", "bermudez"},
		{"bernabe", "bernab"},
		{"bernabé", "bernab"},
		{"bernabeu", "bernabeu"},
		{"bernal", "bernal"},
		{"bernanos", "bernan"},
		{"bernard", "bernard"},
		{"bernardino", "bernardin"},
		{"bernardo", "bernard"},
		{"bernuncio", "bernunci"},
		{"berrinche", "berrinch"},
		{"berrinchito", "berrinchit"},
		{"berrueto", "berruet"},
		{"berta", "bert"},
		{"berumen", "berum"},
		{"besando", "bes"},
		{"besar", "bes"},
		{"besara", "bes"},
		{"besarla", "bes"},
		{"besarle", "bes"},
		{"bese", "bes"},
		{"beso", "bes"},
		{"besó", "bes"},
		{"best", "best"},
		{"bestia", "besti"},
		{"bestias", "besti"},
		{"betancourt", "betancourt"},
		{"betania", "betani"},
		{"bi", "bi"},
		{"bianchezi", "bianchezi"},
		{"bianchini", "bianchini"},
		{"biberón", "biberon"},
		{"bibi", "bibi"},
		{"biblia", "bibli"},
		{"biblias", "bibli"},
		{"bíblica", "biblic"},
		{"bíblico", "biblic"},
		{"bibliografía", "bibliograf"},
		{"bibliográficas", "bibliograf"},
		{"bibliográfico", "bibliograf"},
		{"biblioteca", "bibliotec"},
		{"bibliotecas", "bibliotec"},
		{"bicampeón", "bicampeon"},
		{"bicentenario", "bicentenari"},
		{"bíceps", "biceps"},
		{"bichos", "bich"},
		{"bicicleta", "biciclet"},
		{"bicicletas", "biciclet"},
		{"bien", "bien"},
		{"bienal", "bienal"},
		{"bienales", "bienal"},
		{"bienes", "bien"},
		{"bienestar", "bienest"},
		{"bienio", "bieni"},
		{"bienvenida", "bienven"},
		{"bienvenidas", "bienven"},
		{"bienvenido", "bienven"},
		{"bienvenidos", "bienven"},
		{"bierhaus", "bierhaus"},
		{"big", "big"},
		{"bigfoot", "bigfoot"},
		{"bigleaguers", "bigleaguers"},
		{"bigote", "bigot"},
		{"bigotes", "bigot"},
		{"bike", "bik"},
		{"bilardo", "bilard"},
		{"bilateral", "bilateral"},
		{"bilaterales", "bilateral"},
		{"bilateralmente", "bilateral"},
		{"bilbao", "bilba"},
		{"bilimek", "bilimek"},
		{"bill", "bill"},
		{"billar", "bill"},
		{"billete", "billet"},
		{"billetes", "billet"},
		{"billón", "billon"},
		{"billones", "billon"},
		{"bills", "bills"},
		{"billy", "billy"},
		{"bin", "bin"},
		{"binacional", "binacional"},
		{"bing", "bing"},
		{"biodegradable", "biodegrad"},
		{"biodegradables", "biodegrad"},
		{"biología", "biolog"},
		{"biológica", "biolog"},
		{"biológico", "biolog"},
		{"biólogo", "biolog"},
		{"biondi", "biondi"},
		{"biotecnología", "biotecnolog"},
		{"biotecnológica", "biotecnolog"},
		{"bióticos", "biotic"},
		{"bipolar", "bipol"},
		{"bird", "bird"},
		{"birks", "birks"},
		{"birlos", "birl"},
		{"bis", "bis"},
		{"bisabuela", "bisabuel"},
		{"biscaine", "biscain"},
		{"bismarck", "bismarck"},
		{"bissonnette", "bissonnett"},
		{"biznietos", "bizniet"},
		{"bjoern", "bjoern"},
		{"black", "black"},
		{"blackhawks", "blackhawks"},
		{"bladimir", "bladim"},
		{"blaise", "blais"},
		{"blanca", "blanc"},
		{"blancas", "blanc"},
		{"blanco", "blanc"},
		{"blancos", "blanc"},
		{"blancq", "blancq"},
		{"blandir", "bland"},
		{"blando", "bland"},
		{"blanquiazul", "blanquiazul"},
		{"blanquiazules", "blanquiazul"},
		{"blanquita", "blanquit"},
		{"blas", "blas"},
		{"blasa", "blas"},
		{"blasco", "blasc"},
		{"blasio", "blasi"},
		{"blasón", "blason"},
		{"blatidos", "blat"},
		{"block", "block"},
		{"blocker", "block"},
		{"bloque", "bloqu"},
		{"bloquea", "bloque"},
		{"bloqueadas", "bloqu"},
		{"bloqueado", "bloqu"},
		{"bloqueadoras", "bloqueador"},
		{"bloqueados", "bloqu"},
		{"bloquean", "bloqu"},
		{"bloqueando", "bloqu"},
		{"bloquear", "bloqu"},
		{"bloquearon", "bloqu"},
		{"bloqueo", "bloque"},
		{"bloqueos", "bloque"},
		{"bloques", "bloqu"},
		{"bloxcom", "bloxcom"},
		{"blue", "blu"},
		{"blues", "blu"},
		{"bm", "bm"},
		{"bmw", "bmw"},
		{"bob", "bob"},
		{"bobby", "bobby"},
		{"bobina", "bobin"},
		{"boca", "boc"},
		{"bocanadas", "bocan"},
		{"bocanegra", "bocanegr"},
		{"bocas", "boc"},
		{"bochini", "bochini"},
		{"bocho", "boch"},
		{"bochornos", "bochorn"},
		{"bochos", "boch"},
		{"bocina", "bocin"},
		{"bocinas", "bocin"},
		{"boda", "bod"},
		{"bodas", "bod"},
		{"bodega", "bodeg"},
		{"bodegas", "bodeg"},
		{"bofetón", "bofeton"},
		{"boga", "bog"},
		{"bogota", "bogot"},
		{"bogotá", "bogot"},
		{"bogue", "bog"},
		{"bohemio", "bohemi"},
		{"boicot", "boicot"},
		{"boicotearlas", "boicot"},
		{"boinas", "boin"},
		{"bola", "bol"},
		{"bolaños", "bolañ"},
		{"bolas", "bol"},
		{"boleados", "bol"},
		{"boleando", "bol"},
		{"boleo", "bole"},
		{"boleta", "bolet"},
		{"boletas", "bolet"},
		{"boletín", "boletin"},
		{"boletines", "boletin"},
		{"boleto", "bolet"},
		{"boletos", "bolet"},
		{"bolillo", "bolill"},
		{"bolillos", "bolill"},
		{"bolitas", "bolit"},
		{"bolivar", "boliv"},
		{"bolívar", "boliv"},
		{"bolivia", "bolivi"},
		{"boloña", "boloñ"},
		{"bolsa", "bols"},
		{"bolsas", "bols"},
		{"bolsillo", "bolsill"},
		{"bolsillos", "bolsill"},
		{"bolso", "bols"},
		{"bomba", "bomb"},
		{"bombardeado", "bombard"},
		{"bombardeos", "bombarde"},
		{"bombas", "bomb"},
		{"bombear", "bomb"},
		{"bombeo", "bombe"},
		{"bomberos", "bomber"},
		{"bondades", "bondad"},
		{"bondadosas", "bondad"},
		{"bondadoso", "bondad"},
		{"bone", "bon"},
		{"bonifacio", "bonifaci"},
		{"bonilla", "bonill"},
		{"bonita", "bonit"},
		{"bonitas", "bonit"},
		{"bonito", "bonit"},
		{"bonitos", "bonit"},
		{"bonn", "bonn"},
		{"bonos", "bon"},
		{"boom", "boom"},
		{"boomers", "boomers"},
		{"borax", "borax"},
		{"bordadores", "bordador"},
		{"bordados", "bord"},
		{"borde", "bord"},
		{"bordo", "bord"},
		{"bordos", "bord"},
		{"borge", "borg"},
		{"boricua", "boricu"},
		{"boris", "boris"},
		{"borja", "borj"},
		{"borla", "borl"},
		{"bormann", "bormann"},
		{"borrables", "borrabl"},
		{"borrachín", "borrachin"},
		{"borrachito", "borrachit"},
		{"borracho", "borrach"},
		{"borrador", "borrador"},
		{"borraja", "borraj"},
		{"borrando", "borr"},
		{"borrar", "borr"},
		{"borrego", "borreg"},
		{"borregos", "borreg"},
		{"borreguitos", "borreguit"},
		{"borrero", "borrer"},
		{"borroso", "borros"},
		{"borsalino", "borsalin"},
		{"borssum", "borssum"},
		{"bosch", "bosch"},
		{"boscosos", "boscos"},
		{"bosnia", "bosni"},
		{"bosnio", "bosni"},
		{"bosque", "bosqu"},
		{"bosquejarse", "bosquej"},
		{"bosques", "bosqu"},
		{"bossier", "bossi"},
		{"boston", "boston"},
		{"bostoniano", "bostonian"},
		{"bota", "bot"},
		{"botana", "botan"},
		{"botanero", "botaner"},
		{"botaneros", "botaner"},
		{"botas", "bot"},
		{"bote", "bot"},
		{"botella", "botell"},
		{"botellas", "botell"},
		{"botello", "botell"},
		{"botes", "bot"},
		{"botín", "botin"},
		{"botines", "botin"},
		{"botón", "boton"},
		{"botones", "boton"},
		{"bougthon", "bougthon"},
		{"boulevard", "boulevard"},
		{"bourbon", "bourbon"},
		{"bours", "bours"},
		{"boutique", "boutiqu"},
		{"boutiques", "boutiqu"},
		{"boutros", "boutr"},
		{"bovary", "bovary"},
		{"bovino", "bovin"},
		{"bowman", "bowm"},
		{"bowthorpe", "bowthorp"},
		{"box", "box"},
		{"boxeador", "boxeador"},
		{"boxeadores", "boxeador"},
		{"boxeo", "boxe"},
		{"boxeó", "boxe"},
		{"boxística", "boxist"},
		{"boy", "boy"},
		{"boyas", "boy"},
		{"boyle", "boyl"},
		{"bracamontes", "bracamont"},
		{"braceaba", "brac"},
		{"bracho", "brach"},
		{"bracket", "bracket"},
		{"bradenton", "bradenton"},
		{"brambila", "brambil"},
		{"branco", "branc"},
		{"brando", "brand"},
		{"brandt", "brandt"},
		{"brasero", "braser"},
		{"brasil", "brasil"},
		{"brasileira", "brasileir"},
		{"brasileña", "brasileñ"},
		{"brasileño", "brasileñ"},
		{"brasileños", "brasileñ"},
		{"brasilia", "brasili"},
		{"brass", "brass"},
		{"brasso", "brass"},
		{"braun", "braun"},
		{"bravo", "brav"},
		{"bravos", "brav"},
		{"bravura", "bravur"},
		{"brazo", "braz"},
		{"brazos", "braz"},
		{"brazzaville", "brazzavill"},
		{"breaky", "breaky"},
		{"brecha", "brech"},
		{"brechas", "brech"},
		{"bremer", "brem"},
		{"bremond", "bremond"},
		{"brener", "bren"},
		{"bretaña", "bretañ"},
		{"breton", "breton"},
		{"bretón", "breton"},
		{"breve", "brev"},
		{"brevedad", "breved"},
		{"brevemente", "brevement"},
		{"breves", "brev"},
		{"brevísima", "brevisim"},
		{"brian", "bri"},
		{"bridan", "brid"},
		{"brigada", "brig"},
		{"brigadas", "brig"},
		{"brillaban", "brill"},
		{"brillante", "brillant"},
		{"brillantes", "brillant"},
		{"brillantez", "brillantez"},
		{"brillo", "brill"},
		{"brinca", "brinc"},
		{"brincar", "brinc"},
		{"brincarse", "brinc"},
		{"brinckerhoff", "brinckerhoff"},
		{"brincó", "brinc"},
		{"brinda", "brind"},
		{"brindado", "brind"},
		{"brindar", "brind"},
		{"brindará", "brind"},
		{"brindaría", "brind"},
		{"brindarle", "brind"},
		{"brindarles", "brind"},
		{"brindarme", "brind"},
		{"brindaron", "brind"},
		{"brinde", "brind"},
		{"brinden", "brind"},
		{"brindó", "brind"},
		{"bringas", "bring"},
		{"briones", "brion"},
		{"briosos", "brios"},
		{"brisas", "bris"},
		{"britania", "britani"},
		{"británica", "britan"},
		{"británico", "britan"},
		{"británicos", "britan"},
		{"british", "british"},
		{"brito", "brit"},
		{"brizio", "brizi"},
		{"broad", "bro"},
		{"brocha", "broch"},
		{"broche", "broch"},
		{"brock", "brock"},
		{"brocker", "brock"},
		{"broma", "brom"},
		{"bromas", "brom"},
		{"bromeaban", "brom"},
		{"bromeando", "brom"},
		{"bromista", "bromist"},
		{"bromosas", "brom"},
		{"brompton", "brompton"},
		{"bronca", "bronc"},
		{"broncas", "bronc"},
		{"bronce", "bronc"},
		{"bronco", "bronc"},
		{"broncón", "broncon"},
		{"broncos", "bronc"},
		{"bronquitis", "bronquitis"},
		{"brooks", "brooks"},
		{"bros", "bros"},
		{"brotan", "brot"},
		{"brotaron", "brot"},
		{"brothers", "brothers"},
		{"brown", "brown"},
		{"brownsville", "brownsvill"},
		{"bruce", "bruc"},
		{"bruces", "bruc"},
		{"brugera", "bruger"},
		{"bruja", "bruj"},
		{"brunas", "brun"},
		{"brunch", "brunch"},
		{"bruno", "brun"},
		{"brusca", "brusc"},
		{"bruscas", "brusc"},
		{"bruselas", "brusel"},
		{"brusquedad", "brusqued"},
		{"brutal", "brutal"},
		{"brutales", "brutal"},
		{"brutalmente", "brutal"},
		{"bruto", "brut"},
		{"brutos", "brut"},
		{"bubka", "bubk"},
		{"bucareli", "bucareli"},
		{"buckingham", "buckingham"},
		{"bucolia", "bucoli"},
		{"bud", "bud"},
		{"buda", "bud"},
		{"buddleia", "buddlei"},
		{"buen", "buen"},
		{"buena", "buen"},
		{"buenas", "buen"},
		{"buenaventura", "buenaventur"},
		{"buendía", "buend"},
		{"bueno", "buen"},
		{"buenos", "buen"},
		{"buerón", "bueron"},
		{"buey", "buey"},
		{"bueyes", "buey"},
		{"búfalo", "bufal"},
		{"bufferin", "bufferin"},
		{"bugno", "bugn"},
		{"buho", "buh"},
		{"buick", "buick"},
		{"buitres", "buitr"},
		{"bulbos", "bulb"},
		{"bulgaria", "bulgari"},
		{"búlgaro", "bulgar"},
		{"bunbun", "bunbun"},
		{"bunker", "bunk"},
		{"buque", "buqu"},
		{"buques", "buqu"},
		{"burel", "burel"},
		{"burelo", "burel"},
		{"burger", "burg"},
		{"burgos", "burg"},
		{"burgueño", "burgueñ"},
		{"burla", "burl"},
		{"burlaba", "burl"},
		{"burlan", "burl"},
		{"burlando", "burl"},
		{"burlar", "burl"},
		{"burlarte", "burlart"},
		{"burlas", "burl"},
		{"burló", "burl"},
		{"burocracia", "burocraci"},
		{"burócrata", "burocrat"},
		{"burócratas", "burocrat"},
		{"burocráticas", "burocrat"},
		{"burocráticos", "burocrat"},
		{"burocratismos", "burocrat"},
		{"burro", "burr"},
		{"burros", "burr"},
		{"bursatil", "bursatil"},
		{"bursátil", "bursatil"},
		{"bursátiles", "bursatil"},
		{"bursatilidad", "bursatil"},
		{"bursatilización", "bursatiliz"},
		{"burzatilización", "burzatiliz"},
		{"busca", "busc"},
		{"buscaba", "busc"},
		{"buscaban", "busc"},
		{"buscado", "busc"},
		{"buscador", "buscador"},
		{"buscadores", "buscador"},
		{"buscados", "busc"},
		{"buscamos", "busc"},
		{"buscan", "busc"},
		{"buscando", "busc"},
		{"buscar", "busc"},
		{"buscara", "busc"},
		{"buscará", "busc"},
		{"buscaran", "busc"},
		{"buscarán", "busc"},
		{"buscaras", "busc"},
		{"buscaremos", "busc"},
		{"buscaría", "busc"},
		{"buscarla", "busc"},
		{"buscarle", "busc"},
		{"buscarlo", "busc"},
		{"buscaron", "busc"},
		{"buscarse", "busc"},
		{"busco", "busc"},
		{"buscó", "busc"},
		{"bush", "bush"},
		{"business", "business"},
		{"busque", "busqu"},
		{"búsqueda", "busqued"},
		{"búsquedas", "busqued"},
		{"búsquele", "busquel"},
		{"busquen", "busqu"},
		{"bustamante", "bustam"},
		{"bustillos", "bustill"},
		{"bustinzer", "bustinz"},
		{"busto", "bust"},
		{"butacas", "butac"},
		{"butano", "butan"},
		{"butcher", "butch"},
		{"butler", "butl"},
		{"by", "by"},
		{"bytes", "bytes"},
		{"c", "c"},
		{"caad", "caad"},
		{"cab", "cab"},
		{"cabada", "cab"},
		{"cabal", "cabal"},
		{"cabales", "cabal"},
		{"caballada", "caball"},
		{"caballero", "caballer"},
		{"caballeros", "caballer"},
		{"caballo", "caball"},
		{"caballos", "caball"},
		{"cabalmente", "cabal"},
		{"cabaña", "cabañ"},
		{"cabañas", "cabañ"},
		{"cabanillas", "cabanill"},
		{"cabaret", "cabaret"},
		{"cabe", "cab"},
		{"cabecillas", "cabecill"},
		{"cabelleras", "cabeller"},
		{"cabello", "cabell"},
		{"cabellos", "cabell"},
		{"cabestany", "cabestany"},
		{"cabeza", "cabez"},
		{"cabida", "cab"},
		{"cabiedes", "cabied"},
		{"cabildo", "cabild"},
		{"cabildos", "cabild"},
		{"cabina", "cabin"},
		{"cable", "cabl"},
		{"cableado", "cabl"},
		{"cables", "cabl"},
		{"cablevisión", "cablevision"},
		{"cabo", "cab"},
		{"cabos", "cab"},
		{"cabras", "cabr"},
		{"cabrera", "cabrer"},
		{"cabrito", "cabrit"},
		{"cacallacas", "cacallac"},
		{"cacao", "caca"},
		{"cacareaban", "cacar"},
		{"cacería", "cac"},
		{"cacerías", "cac"},
		{"cacerío", "caceri"},
		{"caceríos", "caceri"},
		{"cacerolas", "cacerol"},
		{"cachetearon", "cachet"},
		{"cachetes", "cachet"},
		{"caciques", "caciqu"},
		{"caciquiles", "caciquil"},
		{"cada", "cad"},
		{"cadapa", "cadap"},
		{"cadáver", "cadav"},
		{"cadáveres", "cadaver"},
		{"cadena", "caden"},
		{"cadenas", "caden"},
		{"cadencia", "cadenci"},
		{"cadereyta", "cadereyt"},
		{"cadete", "cadet"},
		{"cadetes", "cadet"},
		{"cadis", "cadis"},
		{"cádiz", "cadiz"},
		{"cadmio", "cadmi"},
		{"caducidad", "caduc"},
		{"cae", "cae"},
		{"caemos", "caem"},
		{"caen", "caen"},
		{"caer", "caer"},
		{"caerle", "caerl"},
		{"café", "caf"},
		{"cafecito", "cafecit"},
		{"cafés", "cafes"},
		{"cafetal", "cafetal"},
		{"cafetería", "cafet"},
		{"cafeteros", "cafeter"},
		{"cafú", "cafu"},
		{"caía", "cai"},
		{"caída", "caid"},
		{"caídas", "caid"},
		{"caído", "caid"},
		{"caifanes", "caifan"},
		{"caigan", "caig"},
		{"caintra", "caintr"},
		{"cairo", "cair"},
		{"caja", "caj"},
		{"cajal", "cajal"},
		{"cajas", "caj"},
		{"cajón", "cajon"},
		{"cajuela", "cajuel"},
		{"cala", "cal"},
		{"calabaza", "calabaz"},
		{"calamidades", "calam"},
		{"calcamonía", "calcamon"},
		{"calcamonías", "calcamon"},
		{"calcetines", "calcetin"},
		{"calcinado", "calcin"},
		{"calcomanía", "calcoman"},
		{"calcula", "calcul"},
		{"calculadas", "calcul"},
		{"calculado", "calcul"},
		{"calculadora", "calcul"},
		{"calculadoras", "calcul"},
		{"calculando", "calcul"},
		{"calcular", "calcul"},
		{"calcularon", "calcul"},
		{"calcule", "calcul"},
		{"calcúlele", "calculel"},
		{"calculo", "calcul"},
		{"cálculo", "calcul"},
		{"calculó", "calcul"},
		{"cálculos", "calcul"},
		{"caldeado", "cald"},
		{"caldeen", "cald"},
		{"caldera", "calder"},
		{"calderón", "calderon"},
		{"calderoni", "calderoni"},
		{"caldillo", "caldill"},
		{"calefacción", "calefaccion"},
		{"calendario", "calendari"},
		{"calendarios", "calendari"},
		{"calentamiento", "calent"},
		{"calentando", "calent"},
		{"calentar", "calent"},
		{"calentura", "calentur"},
		{"calgene", "calgen"},
		{"calibración", "calibr"},
		{"calibradas", "calibr"},
		{"calibrar", "calibr"},
		{"calibre", "calibr"},
		{"cálida", "cal"},
		{"calidad", "calid"},
		{"calidades", "calidad"},
		{"cálido", "cal"},
		{"calientan", "calient"},
		{"caliente", "calient"},
		{"calientes", "calient"},
		{"califica", "calif"},
		{"calificación", "calif"},
		{"calificaciones", "calif"},
		{"calificada", "calific"},
		{"calificadas", "calific"},
		{"calificado", "calific"},
		{"calificadora", "calif"},
		{"calificados", "calific"},
		{"calificamos", "calific"},
		{"califican", "calific"},
		{"calificar", "calific"},
		{"calificará", "calific"},
		{"calificaría", "calific"},
		{"calificaron", "calific"},
		{"calificativo", "calific"},
		{"calificativos", "calific"},
		{"calificó", "calific"},
		{"califique", "califiqu"},
		{"california", "californi"},
		{"californiano", "californian"},
		{"californias", "californi"},
		{"caligula", "caligul"},
		{"calígula", "caligul"},
		{"calla", "call"},
		{"callada", "call"},
		{"calladito", "calladit"},
		{"callado", "call"},
		{"callar", "call"},
		{"callarlo", "call"},
		{"calle", "call"},
		{"callejera", "callejer"},
		{"callejeros", "callejer"},
		{"callejoneadas", "callejon"},
		{"calles", "call"},
		{"calloway", "calloway"},
		{"calma", "calm"},
		{"calmar", "calm"},
		{"calmarla", "calm"},
		{"cálmecac", "calmecac"},
		{"calo", "cal"},
		{"caloca", "caloc"},
		{"calor", "calor"},
		{"calpan", "calp"},
		{"calpixques", "calpixqu"},
		{"calpullis", "calpullis"},
		{"calumnia", "calumni"},
		{"calumnias", "calumni"},
		{"caluroso", "calur"},
		{"calvillo", "calvill"},
		{"calzada", "calz"},
		{"calzado", "calz"},
		{"calzoncillo", "calzoncill"},
		{"cama", "cam"},
		{"camacho", "camach"},
		{"camaleónica", "camaleon"},
		{"camaleónico", "camaleon"},
		{"cámara", "cam"},
		{"camaradería", "camarad"},
		{"cámaras", "cam"},
		{"camarero", "camarer"},
		{"camareros", "camarer"},
		{"camargo", "camarg"},
		{"camarlengo", "camarleng"},
		{"camarón", "camaron"},
		{"camarote", "camarot"},
		{"camas", "cam"},
		{"cambia", "cambi"},
		{"cambiado", "cambi"},
		{"cambiamos", "cambi"},
		{"cambian", "cambi"},
		{"cambiando", "cambi"},
		{"cambiante", "cambiant"},
		{"cambiantes", "cambiant"},
		{"cambiar", "cambi"},
		{"cambiara", "cambi"},
		{"cambiará", "cambi"},
		{"cambiaría", "cambi"},
		{"cambiarían", "cambi"},
		{"cambiarias", "cambiari"},
		{"cambiario", "cambiari"},
		{"cambiarle", "cambi"},
		{"cambiarlos", "cambi"},
		{"cambiaron", "cambi"},
		{"cambie", "cambi"},
		{"cambié", "cambi"},
		{"cambio", "cambi"},
		{"cambió", "camb"},
		{"cambios", "cambi"},
		{"cambridge", "cambridg"},
		{"camdomble", "camdombl"},
		{"camelia", "cameli"},
		{"camellón", "camellon"},
		{"camellones", "camellon"},
		{"camelo", "camel"},
		{"camerino", "camerin"},
		{"camet", "camet"},
		{"camexa", "camex"},
		{"caminaba", "camin"},
		{"caminaban", "camin"},
		{"caminan", "camin"},
		{"caminando", "camin"},
		{"caminar", "camin"},
		{"caminaron", "camin"},
		{"caminata", "caminat"},
		{"camine", "camin"},
		{"camino", "camin"},
		{"caminó", "camin"},
		{"caminos", "camin"},
		{"camión", "camion"},
		{"camioneras", "camioner"},
		{"camioneros", "camioner"},
		{"camiones", "camion"},
		{"camioneta", "camionet"},
		{"camionetas", "camionet"},
		{"camisa", "camis"},
		{"camiseta", "camiset"},
		{"camisetas", "camiset"},
		{"camisitas", "camisit"},
		{"camisón", "camison"},
		{"camisones", "camison"},
		{"camorristas", "camorr"},
		{"camotlán", "camotlan"},
		{"camp", "camp"},
		{"campal", "campal"},
		{"campamento", "campament"},
		{"campamentos", "campament"},
		{"campaña", "campañ"},
		{"campañas", "campañ"},
		{"campechanos", "campechan"},
		{"campeche", "campech"},
		{"campeon", "campeon"},
		{"campeón", "campeon"},
		{"campeona", "campeon"},
		{"campeonato", "campeonat"},
		{"campeonatos", "campeonat"},
		{"campeones", "campeon"},
		{"campesina", "campesin"},
		{"campesinos", "campesin"},
		{"campestre", "campestr"},
		{"campo", "camp"},
		{"campos", "camp"},
		{"camposeco", "camposec"},
		{"campus", "campus"},
		{"campuzano", "campuzan"},
		{"camuflajes", "camuflaj"},
		{"canacintra", "canacintr"},
		{"canaco", "canac"},
		{"canada", "can"},
		{"cañada", "cañ"},
		{"canadá", "canad"},
		{"canadian", "canadi"},
		{"canadiense", "canadiens"},
		{"canadienses", "canadiens"},
		{"canal", "canal"},
		{"canalera", "canaler"},
		{"canalero", "canaler"},
		{"canales", "canal"},
		{"canalicen", "canalic"},
		{"canalización", "canaliz"},
		{"canalizado", "canaliz"},
		{"canalizados", "canaliz"},
		{"canalizando", "canaliz"},
		{"canalizara", "canaliz"},
		{"cañamar", "cañam"},
		{"canapés", "canapes"},
		{"canasta", "canast"},
		{"canasto", "canast"},
		{"cancela", "cancel"},
		{"cancelaba", "cancel"},
		{"cancelación", "cancel"},
		{"cancelada", "cancel"},
		{"cancelados", "cancel"},
		{"cancelando", "cancel"},
		{"cancelar", "cancel"},
		{"cancelarán", "cancel"},
		{"canceló", "cancel"},
		{"cáncer", "canc"},
		{"cancha", "canch"},
		{"canchas", "canch"},
		{"canciller", "cancill"},
		{"cancilleres", "canciller"},
		{"cancillería", "cancill"},
		{"canción", "cancion"},
		{"cancioncita", "cancioncit"},
		{"canciones", "cancion"},
		{"cancun", "cancun"},
		{"cancún", "cancun"},
		{"candelaria", "candelari"},
		{"candelero", "candeler"},
		{"candentes", "candent"},
		{"cándida", "cand"},
		{"candidata", "candidat"},
		{"candidato", "candidat"},
		{"candidatos", "candidat"},
		{"candidatura", "candidatur"},
		{"candidaturas", "candidatur"},
		{"candidiasis", "candidiasis"},
		{"candil", "candil"},
		{"candor", "candor"},
		{"candy", "candy"},
		{"cañedo", "cañed"},
		{"canela", "canel"},
		{"canels", "canels"},
		{"cañete", "cañet"},
		{"caníbales", "canibal"},
		{"caniggia", "caniggi"},
		{"canijo", "canij"},
		{"canion", "canion"},
		{"canje", "canj"},
		{"cano", "can"},
		{"canoga", "canog"},
		{"canolas", "canol"},
		{"cañonazos", "cañonaz"},
		{"cañones", "cañon"},
		{"cansada", "cans"},
		{"cansadas", "cans"},
		{"cansado", "cans"},
		{"cansados", "cans"},
		{"cansancio", "cansanci"},
		{"canseco", "cansec"},
		{"canta", "cant"},
		{"cantaba", "cant"},
		{"cantaban", "cant"},
		{"cantan", "cant"},
		{"cantando", "cant"},
		{"cantante", "cantant"},
		{"cantantes", "cantant"},
		{"cantar", "cant"},
		{"cantaron", "cant"},
		{"cantera", "canter"},
		{"cantería", "cant"},
		{"cantidad", "cantid"},
		{"cantidades", "cantidad"},
		{"cantina", "cantin"},
		{"cantinas", "cantin"},
		{"cantinflas", "cantinfl"},
		{"canto", "cant"},
		{"cantó", "cant"},
		{"cantos", "cant"},
		{"cantú", "cantu"},
		{"caoba", "caob"},
		{"caón", "caon"},
		{"caona", "caon"},
		{"caos", "caos"},
		{"caótica", "caotic"},
		{"capa", "cap"},
		{"capaces", "capac"},
		{"capacidad", "capac"},
		{"capacidades", "capac"},
		{"capacita", "capacit"},
		{"capacitación", "capacit"},
		{"capacitada", "capacit"},
		{"capacitado", "capacit"},
		{"capacitamos", "capacit"},
		{"capacitan", "capacit"},
		{"capacitando", "capacit"},
		{"capacitar", "capacit"},
		{"capacitarse", "capacit"},
		{"capas", "cap"},
		{"capatacio", "capataci"},
		{"capaz", "capaz"},
		{"caperucita", "caperucit"},
		{"capetillo", "capetill"},
		{"capfce", "capfc"},
		{"capilla", "capill"},
		{"capital", "capital"},
		{"capitales", "capital"},
		{"capitalina", "capitalin"},
		{"capitalino", "capitalin"},
		{"capitalinos", "capitalin"},
		{"capitalista", "capital"},
		{"capitalistas", "capital"},
		{"capitalización", "capitaliz"},
		{"capitalizador", "capitaliz"},
		{"capitalizar", "capitaliz"},
		{"capitalizaron", "capitaliz"},
		{"capitán", "capitan"},
		{"capitanes", "capitan"},
		{"capitolio", "capitoli"},
		{"capitonada", "capiton"},
		{"capitonadas", "capiton"},
		{"capitonados", "capiton"},
		{"capitulación", "capitul"},
		{"capítulo", "capitul"},
		{"capítulos", "capitul"},
		{"capo", "cap"},
		{"caporal", "caporal"},
		{"capos", "cap"},
		{"capote", "capot"},
		{"cappel", "cappel"},
		{"capricho", "caprich"},
		{"caprichos", "caprich"},
		{"caprichosas", "caprich"},
		{"caprichosos", "caprich"},
		{"captación", "captacion"},
		{"captado", "capt"},
		{"captamos", "capt"},
		{"captar", "capt"},
		{"captará", "capt"},
		{"captaron", "capt"},
		{"capten", "capt"},
		{"captivos", "captiv"},
		{"captó", "capt"},
		{"captura", "captur"},
		{"capturados", "captur"},
		{"capturó", "captur"},
		{"capucha", "capuch"},
		{"car", "car"},
		{"cara", "car"},
		{"caracas", "carac"},
		{"caracol", "caracol"},
		{"carácter", "caract"},
		{"caracteres", "caracter"},
		{"característica", "caracterist"},
		{"características", "caracterist"},
		{"característicos", "caracterist"},
		{"caracteriza", "caracteriz"},
		{"caracterizadas", "caracteriz"},
		{"caracterizado", "caracteriz"},
		{"caracterizan", "caracteriz"},
		{"caracterizar", "caracteriz"},
		{"caracterizaron", "caracteriz"},
		{"caracterizó", "caracteriz"},
		{"carajo", "caraj"},
		{"caramona", "caramon"},
		{"caraqueña", "caraqueñ"},
		{"caras", "car"},
		{"carátula", "caratul"},
		{"caravana", "caravan"},
		{"caravanas", "caravan"},
		{"caravia", "caravi"},
		{"caray", "caray"},
		{"carbajal", "carbajal"},
		{"carballido", "carball"},
		{"carballo", "carball"},
		{"carbón", "carbon"},
		{"carbonáceo", "carbonace"},
		{"carbonel", "carbonel"},
		{"carbono", "carbon"},
		{"carburador", "carbur"},
		{"carcajada", "carcaj"},
		{"carcajadas", "carcaj"},
		{"carcaño", "carcañ"},
		{"cárcel", "carcel"},
		{"cárceles", "carcel"},
		{"carcomido", "carcom"},
		{"cárcova", "carcov"},
		{"card", "card"},
		{"cardán", "cardan"},
		{"cardenal", "cardenal"},
		{"cardenales", "cardenal"},
		{"cárdenas", "carden"},
		{"cardenismo", "carden"},
		{"cardenista", "carden"},
		{"cardenistas", "carden"},
		{"cárdeno", "carden"},
		{"cardiaca", "cardiac"},
		{"cardíaca", "cardiac"},
		{"cardiaco", "cardiac"},
		{"cardíaco", "cardiac"},
		{"cardíacos", "cardiac"},
		{"cardiología", "cardiolog"},
		{"cardiólogo", "cardiolog"},
		{"cardiopulmonar", "cardiopulmon"},
		{"cardona", "cardon"},
		{"careca", "carec"},
		{"carece", "carec"},
		{"carecen", "carec"},
		{"carecer", "carec"},
		{"carecía", "carec"},
		{"carecían", "carec"},
		{"careció", "carec"},
		{"carencia", "carenci"},
		{"carencias", "carenci"},
		{"carente", "carent"},
		{"carentes", "carent"},
		{"careo", "care"},
		{"carga", "carg"},
		{"cargada", "carg"},
		{"cargadas", "carg"},
		{"cargado", "carg"},
		{"cargados", "carg"},
		{"cargamento", "cargament"},
		{"cargamentos", "cargament"},
		{"cargando", "carg"},
		{"cargar", "carg"},
		{"cargaran", "carg"},
		{"cargarle", "carg"},
		{"cargaron", "carg"},
		{"cargas", "carg"},
		{"cargo", "carg"},
		{"cargó", "carg"},
		{"cargos", "carg"},
		{"cargueros", "carguer"},
		{"carha", "carh"},
		{"caribe", "carib"},
		{"caribeña", "caribeñ"},
		{"caribeño", "caribeñ"},
		{"caribeños", "caribeñ"},
		{"caricatura", "caricatur"},
		{"caricaturas", "caricatur"},
		{"caricias", "carici"},
		{"caricom", "caricom"},
		{"caridad", "carid"},
		{"carina", "carin"},
		{"cariño", "cariñ"},
		{"cariñosidad", "cariñ"},
		{"cariñoso", "cariñ"},
		{"carioca", "carioc"},
		{"cariotipos", "cariotip"},
		{"carisma", "carism"},
		{"carismático", "carismat"},
		{"cáritas", "carit"},
		{"caritativas", "carit"},
		{"carl", "carl"},
		{"carla", "carl"},
		{"carlitos", "carlit"},
		{"carlos", "carl"},
		{"carlota", "carlot"},
		{"carman", "carm"},
		{"carmen", "carm"},
		{"carmona", "carmon"},
		{"carnaval", "carnaval"},
		{"carnavaleros", "carnavaler"},
		{"carnavales", "carnaval"},
		{"carnavalescas", "carnavalesc"},
		{"carnavalito", "carnavalit"},
		{"carne", "carn"},
		{"carneros", "carner"},
		{"carnes", "carn"},
		{"carnestolendas", "carnestolend"},
		{"carnet", "carnet"},
		{"carnicería", "carnic"},
		{"carnicero", "carnicer"},
		{"cárnicos", "carnic"},
		{"caro", "car"},
		{"carolina", "carolin"},
		{"caronte", "caront"},
		{"caros", "car"},
		{"carpa", "carp"},
		{"carpera", "carper"},
		{"carpeta", "carpet"},
		{"carpintería", "carpint"},
		{"carpizo", "carpiz"},
		{"carr", "carr"},
		{"carrancismo", "carranc"},
		{"carrancistas", "carranc"},
		{"carranza", "carranz"},
		{"carrasquel", "carrasquel"},
		{"carreño", "carreñ"},
		{"carreón", "carreon"},
		{"carrera", "carrer"},
		{"carreras", "carrer"},
		{"carrero", "carrer"},
		{"carreta", "carret"},
		{"carretera", "carreter"},
		{"carreteras", "carreter"},
		{"carretero", "carreter"},
		{"carreteros", "carreter"},
		{"carretón", "carreton"},
		{"carrier", "carri"},
		{"carriles", "carril"},
		{"carrillo", "carrill"},
		{"carrington", "carrington"},
		{"carrión", "carrion"},
		{"carrito", "carrit"},
		{"carrizalejo", "carrizalej"},
		{"carrizales", "carrizal"},
		{"carro", "carr"},
		{"carrocería", "carroc"},
		{"carrocerías", "carroc"},
		{"carros", "carr"},
		{"carroza", "carroz"},
		{"carrozas", "carroz"},
		{"carruaje", "carruaj"},
		{"carruajes", "carruaj"},
		{"carrujo", "carruj"},
		{"carrujos", "carruj"},
		{"carso", "cars"},
		{"carta", "cart"},
		{"cartagena", "cartagen"},
		{"cartas", "cart"},
		{"cartay", "cartay"},
		{"cartel", "cartel"},
		{"cártel", "cartel"},
		{"cartelera", "carteler"},
		{"carteles", "cartel"},
		{"cartelones", "cartelon"},
		{"carter", "cart"},
		{"cartera", "carter"},
		{"carteras", "carter"},
		{"cartilla", "cartill"},
		{"cartón", "carton"},
		{"cartucho", "cartuch"},
		{"carvajal", "carvajal"},
		{"casa", "cas"},
		{"casaca", "casac"},
		{"casada", "cas"},
		{"casadas", "cas"},
		{"casado", "cas"},
		{"casados", "cas"},
		{"casamentero", "casamenter"},
		{"casando", "cas"},
		{"casarme", "cas"},
		{"casaron", "cas"},
		{"casarse", "cas"},
		{"casas", "cas"},
		{"casashabitación", "casashabit"},
		{"casavan", "casav"},
		{"cascabel", "cascabel"},
		{"cascabeles", "cascabel"},
		{"cascada", "casc"},
		{"cáscara", "casc"},
		{"cascos", "casc"},
		{"caseras", "caser"},
		{"casero", "caser"},
		{"caseros", "caser"},
		{"caseta", "caset"},
		{"casetas", "caset"},
		{"cash", "cash"},
		{"casi", "casi"},
		{"casilla", "casill"},
		{"casillas", "casill"},
		{"casillero", "casiller"},
		{"casimir", "casim"},
		{"casino", "casin"},
		{"casio", "casi"},
		{"casita", "casit"},
		{"caso", "cas"},
		{"casó", "cas"},
		{"casos", "cas"},
		{"casquillo", "casquill"},
		{"casquillos", "casquill"},
		{"cassete", "casset"},
		{"cassette", "cassett"},
		{"cassettes", "cassett"},
		{"casta", "cast"},
		{"castañas", "castañ"},
		{"castañeda", "castañed"},
		{"castaño", "castañ"},
		{"castañón", "castañon"},
		{"castaños", "castañ"},
		{"castellano", "castellan"},
		{"castellanos", "castellan"},
		{"castidad", "castid"},
		{"castiga", "castig"},
		{"castigadas", "castig"},
		{"castigados", "castig"},
		{"castigar", "castig"},
		{"castigará", "castig"},
		{"castigo", "castig"},
		{"castigue", "castig"},
		{"castilla", "castill"},
		{"castilleja", "castillej"},
		{"castillejos", "castillej"},
		{"castillo", "castill"},
		{"castor", "castor"},
		{"castores", "castor"},
		{"castrejón", "castrejon"},
		{"castro", "castr"},
		{"castruita", "castruit"},
		{"cástulo", "castul"},
		{"casual", "casual"},
		{"casuales", "casual"},
		{"casualidad", "casual"},
		{"casuística", "casuist"},
		{"cat", "cat"},
		{"catalán", "catalan"},
		{"catalanes", "catalan"},
		{"catalina", "catalin"},
		{"catalítico", "catalit"},
		{"catalíticos", "catalit"},
		{"catalizador", "cataliz"},
		{"catalogan", "catalog"},
		{"catálogo", "catalog"},
		{"catamo", "catam"},
		{"cataño", "catañ"},
		{"catarata", "catarat"},
		{"cataratas", "catarat"},
		{"catarina", "catarin"},
		{"catarro", "catarr"},
		{"catarros", "catarr"},
		{"catarsis", "catarsis"},
		{"catársis", "catarsis"},
		{"catastral", "catastral"},
		{"catastrales", "catastral"},
		{"catastro", "catastr"},
		{"catástrofe", "catastrof"},
		{"catastrófica", "catastrof"},
		{"cátchers", "catchers"},
		{"cateadas", "cat"},
		{"cátedra", "catedr"},
		{"catedral", "catedral"},
		{"catedrática", "catedrat"},
		{"catedrático", "catedrat"},
		{"catedráticos", "catedrat"},
		{"categoría", "categor"},
		{"categorías", "categor"},
		{"cateos", "cate"},
		{"catequesis", "catequesis"},
		{"católica", "catol"},
		{"catolicismo", "catolic"},
		{"católico", "catol"},
		{"católicos", "catol"},
		{"catón", "caton"},
		{"catorce", "catorc"},
		{"catrerina", "catrerin"},
		{"catsup", "catsup"},
		{"cauce", "cauc"},
		{"cauces", "cauc"},
		{"caucional", "caucional"},
		{"caudalosos", "caudal"},
		{"caudillo", "caudill"},
		{"causa", "caus"},
		{"causada", "caus"},
		{"causadas", "caus"},
		{"causado", "caus"},
		{"causados", "caus"},
		{"causaefecto", "causaefect"},
		{"causales", "causal"},
		{"causan", "caus"},
		{"causando", "caus"},
		{"causante", "causant"},
		{"causantes", "causant"},
		{"causar", "caus"},
		{"causará", "caus"},
		{"causaron", "caus"},
		{"causas", "caus"},
		{"cause", "caus"},
		{"causó", "caus"},
		{"cautela", "cautel"},
		{"cautelosa", "cautel"},
		{"cautivarlos", "cautiv"},
		{"cautivas", "cautiv"},
		{"cautivos", "cautiv"},
		{"cavazos", "cavaz"},
		{"cavernaria", "cavernari"},
		{"cavidad", "cavid"},
		{"cayeran", "cayer"},
		{"cayeron", "cayeron"},
		{"cayetano", "cayetan"},
		{"cayman", "caym"},
		{"cayo", "cay"},
		{"cayó", "cay"},
		{"caza", "caz"},
		{"cazador", "cazador"},
		{"cazadoras", "cazador"},
		{"cazadores", "cazador"},
		{"cazar", "caz"},
		{"cázares", "cazar"},
		{"cazaux", "cazaux"},
		{"cazuela", "cazuel"},
		{"cbinbur", "cbinbur"},
		{"cbs", "cbs"},
		{"cbtis", "cbtis"},
		{"ccd", "ccd"},
		{"cce", "cce"},
		{"ccinlac", "ccinlac"},
		{"ccolonia", "ccoloni"},
		{"ccp", "ccp"},
		{"cd", "cd"},
		{"ce", "ce"},
		{"cear", "cear"},
		{"cebada", "ceb"},
		{"cebatis", "cebatis"},
		{"cebolla", "ceboll"},
		{"cebollas", "ceboll"},
		{"cebrián", "cebrian"},
		{"cecilia", "cecili"},
		{"ceda", "ced"},
		{"cedart", "cedart"},
		{"cede", "ced"},
		{"cedeco", "cedec"},
		{"cedeño", "cedeñ"},
		{"ceder", "ced"},
		{"cederán", "ced"},
		{"cedería", "ced"},
		{"cedido", "ced"},
		{"cedieron", "ced"},
		{"cedió", "ced"},
		{"cedral", "cedral"},
		{"cédula", "cedul"},
		{"cédulas", "cedul"},
		{"cee", "cee"},
		{"cegarse", "ceg"},
		{"ceja", "cej"},
		{"celadora", "celador"},
		{"celadores", "celador"},
		{"celaya", "celay"},
		{"celda", "celd"},
		{"celebérrima", "celeberrim"},
		{"celebra", "celebr"},
		{"celebraba", "celebr"},
		{"celebración", "celebr"},
		{"celebraciones", "celebr"},
		{"celebrada", "celebr"},
		{"celebradas", "celebr"},
		{"celebrado", "celebr"},
		{"celebrados", "celebr"},
		{"celebran", "celebr"},
		{"celebrar", "celebr"},
		{"celebrará", "celebr"},
		{"celebrarlo", "celebr"},
		{"celebraron", "celebr"},
		{"celebrarse", "celebr"},
		{"celebre", "celebr"},
		{"célebre", "celebr"},
		{"celébrelas", "celebrel"},
		{"célebres", "celebr"},
		{"celebridad", "celebr"},
		{"celebridades", "celebr"},
		{"celebró", "celebr"},
		{"celeridad", "celer"},
		{"celestes", "celest"},
		{"celestial", "celestial"},
		{"celia", "celi"},
		{"celiberia", "celiberi"},
		{"celio", "celi"},
		{"celo", "cel"},
		{"celos", "cel"},
		{"celso", "cels"},
		{"celta", "celt"},
		{"célula", "celul"},
		{"celular", "celul"},
		{"celulares", "celular"},
		{"células", "celul"},
		{"cemcaspe", "cemcasp"},
		{"cement", "cement"},
		{"cementeras", "cementer"},
		{"cementero", "cementer"},
		{"cemento", "cement"},
		{"cementos", "cement"},
		{"cemex", "cemex"},
		{"cen", "cen"},
		{"cena", "cen"},
		{"cenamos", "cen"},
		{"cenar", "cen"},
		{"cenas", "cen"},
		{"cenefa", "cenef"},
		{"cenefas", "cenef"},
		{"ceñido", "ceñ"},
		{"ceniza", "ceniz"},
		{"cenizas", "ceniz"},
		{"cenlac", "cenlac"},
		{"cenó", "cen"},
		{"censo", "cens"},
		{"censura", "censur"},
		{"censurable", "censur"},
		{"censurada", "censur"},
		{"censuradas", "censur"},
		{"censurado", "censur"},
		{"censurar", "censur"},
		{"censuró", "censur"},
		{"centavo", "centav"},
		{"centavos", "centav"},
		{"centenar", "centen"},
		{"centenares", "centenar"},
		{"centenario", "centenari"},
		{"centeno", "centen"},
		{"center", "cent"},
		{"centésimas", "centesim"},
		{"centígrados", "centigr"},
		{"centímetros", "centimetr"},
		{"centradas", "centr"},
		{"central", "central"},
		{"centrales", "central"},
		{"centramos", "centr"},
		{"centrándose", "centr"},
		{"centrará", "centr"},
		{"centrarse", "centr"},
		{"centre", "centr"},
		{"céntricas", "centric"},
		{"céntrico", "centric"},
		{"céntricos", "centric"},
		{"centro", "centr"},
		{"centroamérica", "centroamer"},
		{"centroamericana", "centroamerican"},
		{"centroamericano", "centroamerican"},
		{"centroamericanos", "centroamerican"},
		{"centroderechista", "centroderech"},
		{"centros", "centr"},
		{"century", "century"},
		{"cepa", "cep"},
		{"cepal", "cepal"},
		{"cepeda", "ceped"},
		{"cepillan", "cepill"},
		{"cepillo", "cepill"},
		{"cerámica", "ceram"},
		{"cerca", "cerc"},
		{"cercados", "cerc"},
		{"cercana", "cercan"},
		{"cercanas", "cercan"},
		{"cercando", "cerc"},
		{"cercanía", "cercan"},
		{"cercano", "cercan"},
		{"cercanos", "cercan"},
		{"cercar", "cerc"},
		{"cerciorarse", "cercior"},
		{"cerda", "cerd"},
		{"cerdas", "cerd"},
		{"cerdo", "cerd"},
		{"cerdos", "cerd"},
		{"cereales", "cereal"},
		{"cerebral", "cerebral"},
		{"cerebrales", "cerebral"},
		{"cerebro", "cerebr"},
		{"cerebros", "cerebr"},
		{"ceremonia", "ceremoni"},
		{"ceremonias", "ceremoni"},
		{"ceremoniosamente", "ceremoni"},
		{"cereso", "ceres"},
		{"cereza", "cerez"},
		{"cero", "cer"},
		{"cerraba", "cerr"},
		{"cerraban", "cerr"},
		{"cerrada", "cerr"},
		{"cerradas", "cerr"},
		{"cerrado", "cerr"},
		{"cerrados", "cerr"},
		{"cerradura", "cerradur"},
		{"cerraduras", "cerradur"},
		{"cerralvo", "cerralv"},
		{"cerramos", "cerr"},
		{"cerrando", "cerr"},
		{"cerrar", "cerr"},
		{"cerrará", "cerr"},
		{"cerrarán", "cerr"},
		{"cerrarlos", "cerr"},
		{"cerraron", "cerr"},
		{"cerrazón", "cerrazon"},
		{"cerrito", "cerrit"},
		{"cerro", "cerr"},
		{"cerró", "cerr"},
		{"cerrojo", "cerroj"},
		{"cerros", "cerr"},
		{"certamen", "certam"},
		{"certera", "certer"},
		{"certero", "certer"},
		{"certeros", "certer"},
		{"certeza", "certez"},
		{"certidumbre", "certidumbr"},
		{"certificado", "certific"},
		{"certificados", "certific"},
		{"certucha", "certuch"},
		{"cerval", "cerval"},
		{"cervantes", "cervant"},
		{"cervantino", "cervantin"},
		{"cerveceras", "cervecer"},
		{"cervecería", "cervec"},
		{"cervera", "cerver"},
		{"cerveza", "cervez"},
		{"cervezas", "cervez"},
		{"cesa", "ces"},
		{"cesados", "ces"},
		{"cesan", "ces"},
		{"cesar", "ces"},
		{"césar", "ces"},
		{"cesará", "ces"},
		{"cesáreo", "cesare"},
		{"cesaría", "ces"},
		{"cese", "ces"},
		{"cesión", "cesion"},
		{"cesó", "ces"},
		{"cesto", "cest"},
		{"cetemista", "cetem"},
		{"cetemistas", "cetem"},
		{"cetes", "cet"},
		{"cetro", "cetr"},
		{"cetros", "cetr"},
		{"ceu", "ceu"},
		{"cevallos", "cevall"},
		{"cezanne", "cezann"},
		{"cfe", "cfe"},
		{"ch", "ch"},
		{"chagoyos", "chagoy"},
		{"chagui", "chagui"},
		{"chains", "chains"},
		{"chal", "chal"},
		{"chaleco", "chalec"},
		{"chalina", "chalin"},
		{"chalmer", "chalm"},
		{"chamaca", "chamac"},
		{"chamarra", "chamarr"},
		{"chamarras", "chamarr"},
		{"chamba", "chamb"},
		{"chambelanes", "chambelan"},
		{"chamois", "chamois"},
		{"chamorro", "chamorr"},
		{"champúes", "champu"},
		{"chamula", "chamul"},
		{"chamuscado", "chamusc"},
		{"chan", "chan"},
		{"chance", "chanc"},
		{"chanel", "chanel"},
		{"chango", "chang"},
		{"chantilly", "chantilly"},
		{"chanza", "chanz"},
		{"chapa", "chap"},
		{"chapala", "chapal"},
		{"chapo", "chap"},
		{"chapoteaderos", "chapoteader"},
		{"chapultepec", "chapultepec"},
		{"chapuzas", "chapuz"},
		{"chaqueta", "chaquet"},
		{"chara", "char"},
		{"character", "charact"},
		{"charchina", "charchin"},
		{"charchino", "charchin"},
		{"charco", "charc"},
		{"charla", "charl"},
		{"charlaron", "charl"},
		{"charlas", "charl"},
		{"charles", "charl"},
		{"charlie", "charli"},
		{"charló", "charl"},
		{"charly", "charly"},
		{"charnela", "charnel"},
		{"charnell", "charnell"},
		{"charreada", "charr"},
		{"charreamos", "charr"},
		{"charreando", "charr"},
		{"charro", "charr"},
		{"charros", "charr"},
		{"chase", "chas"},
		{"chasis", "chasis"},
		{"chatarra", "chatarr"},
		{"chatarrero", "chatarrer"},
		{"chato", "chat"},
		{"chaumu", "chaumu"},
		{"chauvet", "chauvet"},
		{"chavarría", "chavarr"},
		{"chavas", "chav"},
		{"chavero", "chaver"},
		{"chavez", "chavez"},
		{"chávez", "chavez"},
		{"chavitas", "chavit"},
		{"chavo", "chav"},
		{"chavos", "chav"},
		{"che", "che"},
		{"checa", "chec"},
		{"checar", "chec"},
		{"checo", "chec"},
		{"checoslovaquia", "checoslovaqui"},
		{"chedraoui", "chedraoui"},
		{"chefs", "chefs"},
		{"cheliabinsk", "cheliabinsk"},
		{"chelo", "chel"},
		{"chemical", "chemical"},
		{"chemicalweek", "chemicalweek"},
		{"chemise", "chemis"},
		{"chepo", "chep"},
		{"cheque", "chequ"},
		{"chequeo", "cheque"},
		{"cheques", "chequ"},
		{"cheraw", "cheraw"},
		{"chesca", "chesc"},
		{"chester", "chest"},
		{"chetumal", "chetumal"},
		{"chetumaleños", "chetumaleñ"},
		{"chevrolet", "chevrolet"},
		{"cheyene", "cheyen"},
		{"cheyenne", "cheyenn"},
		{"chi", "chi"},
		{"chía", "chi"},
		{"chiapaneca", "chiapanec"},
		{"chiapas", "chiap"},
		{"chiba", "chib"},
		{"chic", "chic"},
		{"chica", "chic"},
		{"chicago", "chicag"},
		{"chicana", "chican"},
		{"chicano", "chican"},
		{"chicas", "chic"},
		{"chicharrones", "chicharron"},
		{"chichen", "chich"},
		{"chichimecas", "chichimec"},
		{"chicles", "chicl"},
		{"chico", "chic"},
		{"chicos", "chic"},
		{"chicotear", "chicot"},
		{"chicuelinas", "chicuelin"},
		{"chido", "chid"},
		{"chifón", "chifon"},
		{"chiguagua", "chiguagu"},
		{"chihuahua", "chihuahu"},
		{"chihuahuense", "chihuahuens"},
		{"chiítas", "chiit"},
		{"chila", "chil"},
		{"chilac", "chilac"},
		{"chilacayota", "chilacayot"},
		{"chilangolandia", "chilangolandi"},
		{"chilaquiles", "chilaquil"},
		{"chile", "chil"},
		{"chileno", "chilen"},
		{"chilenos", "chilen"},
		{"chiles", "chil"},
		{"chili", "chili"},
		{"chilpancingo", "chilpancing"},
		{"chima", "chim"},
		{"chin", "chin"},
		{"china", "chin"},
		{"chinche", "chinch"},
		{"chinches", "chinch"},
		{"chino", "chin"},
		{"chinos", "chin"},
		{"chip", "chip"},
		{"chipinque", "chipinqu"},
		{"chips", "chips"},
		{"chiquita", "chiquit"},
		{"chirac", "chirac"},
		{"chirinos", "chirin"},
		{"chisec", "chisec"},
		{"chismosa", "chismos"},
		{"chispa", "chisp"},
		{"chispas", "chisp"},
		{"chispazo", "chispaz"},
		{"chispeando", "chisp"},
		{"chistes", "chist"},
		{"chiva", "chiv"},
		{"chivas", "chiv"},
		{"chivo", "chiv"},
		{"chocado", "choc"},
		{"chocan", "choc"},
		{"chocar", "choc"},
		{"chocarán", "choc"},
		{"chocaron", "choc"},
		{"chocó", "choc"},
		{"chocolate", "chocolat"},
		{"chocolates", "chocolat"},
		{"chofer", "chof"},
		{"choferes", "chofer"},
		{"chontalpa", "chontalp"},
		{"chopin", "chopin"},
		{"choque", "choqu"},
		{"choques", "choqu"},
		{"chorizo", "choriz"},
		{"chorreando", "chorr"},
		{"chorro", "chorr"},
		{"chraysler", "chraysl"},
		{"chris", "chris"},
		{"christi", "christi"},
		{"christian", "christi"},
		{"christie", "christi"},
		{"christoph", "christoph"},
		{"christopher", "christoph"},
		{"chrysler", "chrysler"},
		{"chucho", "chuch"},
		{"chuck", "chuck"},
		{"chuletón", "chuleton"},
		{"chung", "chung"},
		{"chupa", "chup"},
		{"chupaban", "chup"},
		{"chupaletada", "chupalet"},
		{"chupando", "chup"},
		{"chupándose", "chup"},
		{"chuparse", "chup"},
		{"chupón", "chupon"},
		{"churubusco", "churubusc"},
		{"churumbela", "churumbel"},
		{"chusca", "chusc"},
		{"chyba", "chyba"},
		{"cia", "cia"},
		{"ciangherotti", "ciangherotti"},
		{"cibaeñas", "cibaeñ"},
		{"cibao", "ciba"},
		{"cicatrización", "cicatriz"},
		{"cíclicamente", "ciclic"},
		{"cíclico", "ciclic"},
		{"ciclismo", "ciclism"},
		{"ciclista", "ciclist"},
		{"ciclistas", "ciclist"},
		{"ciclístico", "ciclist"},
		{"ciclísticos", "ciclist"},
		{"ciclo", "cicl"},
		{"ciclónica", "ciclon"},
		{"ciclos", "cicl"},
		{"ciclosporina", "ciclosporin"},
		{"cide", "cid"},
		{"cielo", "ciel"},
		{"cien", "cien"},
		{"ciencia", "cienci"},
		{"ciencias", "cienci"},
		{"ciénega", "cieneg"},
		{"científica", "cientif"},
		{"científicas", "cientif"},
		{"científico", "cientif"},
		{"científicos", "cientif"},
		{"ciento", "cient"},
		{"cientos", "cient"},
		{"ciernes", "ciern"},
		{"cierra", "cierr"},
		{"cierran", "cierr"},
		{"cierre", "cierr"},
		{"cierres", "cierr"},
		{"cierta", "ciert"},
		{"ciertamente", "ciert"},
		{"ciertas", "ciert"},
		{"cierto", "ciert"},
		{"ciertos", "ciert"},
		{"cifra", "cifr"},
		{"cifrarse", "cifr"},
		{"cifras", "cifr"},
		{"cigarrera", "cigarrer"},
		{"cigarrero", "cigarrer"},
		{"cigarrillos", "cigarrill"},
		{"cigarros", "cigarr"},
		{"cihuatepixque", "cihuatepixqu"},
		{"cilindros", "cilindr"},
		{"cima", "cim"},
		{"cimarrones", "cimarron"},
		{"cimbraría", "cimbr"},
		{"cimiento", "cimient"},
		{"cimientos", "cimient"},
		{"cin", "cin"},
		{"cincelado", "cincel"},
		{"cinco", "cinc"},
		{"cincuenta", "cincuent"},
		{"cine", "cin"},
		{"cineasta", "cineast"},
		{"cinemascope", "cinemascop"},
		{"cinematografía", "cinematograf"},
		{"cinematográfica", "cinematograf"},
		{"cinematográficas", "cinematograf"},
		{"cinematográfico", "cinematograf"},
		{"cines", "cin"},
		{"cineterraza", "cineterraz"},
		{"cinética", "cinet"},
		{"cinismo", "cinism"},
		{"cinta", "cint"},
		{"cintas", "cint"},
		{"cintermex", "cintermex"},
		{"cinto", "cint"},
		{"cintos", "cint"},
		{"cintura", "cintur"},
		{"cinturón", "cinturon"},
		{"cinturones", "cinturon"},
		{"cio", "cio"},
		{"cipriano", "ciprian"},
		{"circo", "circ"},
		{"circos", "circ"},
		{"circuito", "circuit"},
		{"circuitos", "circuit"},
		{"circula", "circul"},
		{"circulaba", "circul"},
		{"circulaban", "circul"},
		{"circulación", "circul"},
		{"circulan", "circul"},
		{"circulando", "circul"},
		{"circular", "circul"},
		{"circulará", "circul"},
		{"circularán", "circul"},
		{"circularía", "circul"},
		{"circularon", "circul"},
		{"circulatoria", "circulatori"},
		{"circulatorios", "circulatori"},
		{"círculo", "circul"},
		{"circuló", "circul"},
		{"círculos", "circul"},
		{"circunscribe", "circunscrib"},
		{"circunscrito", "circunscrit"},
		{"circunstancia", "circunst"},
		{"circunstancias", "circunst"},
		{"circunvalación", "circunval"},
		{"circurret", "circurret"},
		{"cirios", "ciri"},
		{"cirugía", "cirug"},
		{"cirujanos", "cirujan"},
		{"cirules", "cirul"},
		{"cisneros", "cisner"},
		{"cita", "cit"},
		{"citada", "cit"},
		{"citadas", "cit"},
		{"citado", "cit"},
		{"citados", "cit"},
		{"citamos", "cit"},
		{"citan", "cit"},
		{"citar", "cit"},
		{"citará", "cit"},
		{"citas", "cit"},
		{"citation", "citation"},
		{"citatorios", "citatori"},
		{"cite", "cit"},
		{"citen", "cit"},
		{"cito", "cit"},
		{"citó", "cit"},
		{"cítricos", "citric"},
		{"city", "city"},
		{"ciudad", "ciud"},
		{"ciudadana", "ciudadan"},
		{"ciudadanas", "ciudadan"},
		{"ciudadanía", "ciudadan"},
		{"ciudadaníapanprd", "ciudadaniapanprd"},
		{"ciudadanías", "ciudadan"},
		{"ciudadano", "ciudadan"},
		{"ciudadanos", "ciudadan"},
		{"ciudades", "ciudad"},
		{"civic", "civic"},
		{"cívica", "civic"},
		{"cívicas", "civic"},
		{"cívico", "civic"},
		{"cívicos", "civic"},
		{"civil", "civil"},
		{"civiles", "civil"},
		{"civilización", "civiliz"},
		{"civilizaciones", "civiliz"},
		{"civilizada", "civiliz"},
		{"civilizador", "civiliz"},
		{"civilizadora", "civiliz"},
		{"civilizar", "civiliz"},
		{"civismo", "civism"},
		{"clair", "cla"},
		{"clamor", "clamor"},
		{"clan", "clan"},
		{"clandestinos", "clandestin"},
		{"clanes", "clan"},
		{"clapton", "clapton"},
		{"clara", "clar"},
		{"claramente", "clar"},
		{"claras", "clar"},
		{"claridad", "clarid"},
		{"clarifiquen", "clarifiqu"},
		{"clariond", "clariond"},
		{"clarísima", "clarisim"},
		{"clarividencia", "clarivident"},
		{"clarke", "clark"},
		{"claro", "clar"},
		{"claros", "clar"},
		{"claroscuros", "claroscur"},
		{"clase", "clas"},
		{"clases", "clas"},
		{"clásica", "clasic"},
		{"clásicas", "clasic"},
		{"clasico", "clasic"},
		{"clásico", "clasic"},
		{"clásicos", "clasic"},
		{"clasificación", "clasif"},
		{"clasificaciones", "clasif"},
		{"clasificada", "clasific"},
		{"clasificadas", "clasific"},
		{"clasificado", "clasific"},
		{"clasificados", "clasific"},
		{"clasificando", "clasific"},
		{"clasificar", "clasific"},
		{"clasificatorias", "clasificatori"},
		{"classic", "classic"},
		{"claude", "claud"},
		{"claudia", "claudi"},
		{"claudicar", "claudic"},
		{"claudico", "claudic"},
		{"claudio", "claudi"},
		{"claustro", "claustr"},
		{"cláusula", "clausul"},
		{"cláusulas", "clausul"},
		{"clausura", "clausur"},
		{"clausurar", "clausur"},
		{"clausurará", "clausur"},
		{"clavada", "clav"},
		{"clavar", "clav"},
		{"clavaremos", "clav"},
		{"clave", "clav"},
		{"claves", "clav"},
		{"clavó", "clav"},
		{"claxon", "claxon"},
		{"cleary", "cleary"},
		{"clemente", "clement"},
		{"cleotilde", "cleotild"},
		{"clérigo", "clerig"},
		{"clero", "cler"},
		{"cliente", "client"},
		{"clientela", "clientel"},
		{"clientes", "client"},
		{"clima", "clim"},
		{"climáticas", "climat"},
		{"clímax", "climax"},
		{"clínica", "clinic"},
		{"clínicas", "clinic"},
		{"clínico", "clinic"},
		{"clínicos", "clinic"},
		{"clinton", "clinton"},
		{"clone", "clon"},
		{"clones", "clon"},
		{"clos", "clos"},
		{"closet", "closet"},
		{"cloud", "cloud"},
		{"clouthier", "clouthi"},
		{"clowes", "clow"},
		{"club", "club"},
		{"clubes", "club"},
		{"cma", "cma"},
		{"cmb", "cmb"},
		{"cmos", "cmos"},
		{"cms", "cms"},
		{"cna", "cna"},
		{"cnca", "cnca"},
		{"cndh", "cndh"},
		{"cnet", "cnet"},
		{"cnn", "cnn"},
		{"cnpa", "cnpa"},
		{"cnsf", "cnsf"},
		{"cnv", "cnv"},
		{"co", "co"},
		{"coach", "coach"},
		{"coaches", "coach"},
		{"coachman", "coachm"},
		{"coadyuvar", "coadyuv"},
		{"coah", "coah"},
		{"coahuila", "coahuil"},
		{"coahuilense", "coahuilens"},
		{"coahuilenses", "coahuilens"},
		{"coalición", "coalicion"},
		{"coartados", "coart"},
		{"coatepec", "coatepec"},
		{"coatlicue", "coatlicu"},
		{"coautor", "coautor"},
		{"cobán", "coban"},
		{"cobardía", "cobard"},
		{"cobertura", "cobertur"},
		{"cobija", "cobij"},
		{"cobijas", "cobij"},
		{"cobol", "cobol"},
		{"cobos", "cob"},
		{"cobra", "cobr"},
		{"cobraba", "cobr"},
		{"cobrado", "cobr"},
		{"cobramos", "cobr"},
		{"cobran", "cobr"},
		{"cobrando", "cobr"},
		{"cobrar", "cobr"},
		{"cobrara", "cobr"},
		{"cobraran", "cobr"},
		{"cobrarán", "cobr"},
		{"cobraré", "cobr"},
		{"cobrarles", "cobr"},
		{"cobrarte", "cobrart"},
		{"cobras", "cobr"},
		{"cobre", "cobr"},
		{"cobren", "cobr"},
		{"cobro", "cobr"},
		{"cobró", "cobr"},
		{"cobros", "cobr"},
		{"coca", "coc"},
		{"cocaína", "cocain"},
		{"cocainómano", "cocainoman"},
		{"coche", "coch"},
		{"cochero", "cocher"},
		{"coches", "coch"},
		{"cochinilla", "cochinill"},
		{"cocina", "cocin"},
		{"cocinar", "cocin"},
		{"cocineta", "cocinet"},
		{"coco", "coc"},
		{"cocó", "coc"},
		{"cocoa", "coco"},
		{"cóconas", "cocon"},
		{"cocoyoc", "cocoyoc"},
		{"coctel", "coctel"},
		{"codeme", "codem"},
		{"codesal", "codesal"},
		{"codicia", "codici"},
		{"codiciado", "codici"},
		{"código", "codig"},
		{"códigos", "codig"},
		{"codo", "cod"},
		{"coece", "coec"},
		{"coello", "coell"},
		{"coempresario", "coempresari"},
		{"coequipero", "coequiper"},
		{"coercitivas", "coercit"},
		{"coexistan", "coexist"},
		{"coexistir", "coexist"},
		{"cofre", "cofr"},
		{"coger", "cog"},
		{"coggi", "coggi"},
		{"cogido", "cog"},
		{"cognitivo", "cognit"},
		{"cohabita", "cohabit"},
		{"cohecho", "cohech"},
		{"cohen", "coh"},
		{"coherencia", "coherent"},
		{"coherente", "coherent"},
		{"cohetes", "cohet"},
		{"cohetitos", "cohetit"},
		{"coincide", "coincid"},
		{"coinciden", "coincid"},
		{"coincidencia", "coincident"},
		{"coincidencias", "coincident"},
		{"coincidentemente", "coincident"},
		{"coincidentes", "coincident"},
		{"coincidiera", "coincid"},
		{"coincidieron", "coincid"},
		{"coincidió", "coincid"},
		{"coincidirán", "coincid"},
		{"coinversión", "coinversion"},
		{"coinversiones", "coinversion"},
		{"coinvertir", "coinvert"},
		{"cojo", "coj"},
		{"col", "col"},
		{"cola", "col"},
		{"colabora", "colabor"},
		{"colaboración", "colabor"},
		{"colaborado", "colabor"},
		{"colaborador", "colabor"},
		{"colaboradores", "colabor"},
		{"colaboran", "colabor"},
		{"colaborando", "colabor"},
		{"colaborar", "colabor"},
		{"colaborara", "colabor"},
		{"colaboraran", "colabor"},
		{"colaboraría", "colabor"},
		{"colaboraron", "colabor"},
		{"colabore", "colabor"},
		{"colapsaron", "colaps"},
		{"colas", "col"},
		{"colcha", "colch"},
		{"colchón", "colchon"},
		{"colchoneros", "colchoner"},
		{"colchones", "colchon"},
		{"coldwell", "coldwell"},
		{"cole", "col"},
		{"coleadero", "coleader"},
		{"colección", "coleccion"},
		{"colecciones", "coleccion"},
		{"coleccionismo", "coleccion"},
		{"coleccionista", "coleccion"},
		{"coleccionistas", "coleccion"},
		{"colecta", "colect"},
		{"colectiva", "colect"},
		{"colectivamente", "colect"},
		{"colectividad", "colect"},
		{"colectivo", "colect"},
		{"colectivos", "colect"},
		{"colega", "coleg"},
		{"colegas", "coleg"},
		{"colegiado", "colegi"},
		{"colegiatura", "colegiatur"},
		{"colegio", "colegi"},
		{"colegios", "colegi"},
		{"cólera", "coler"},
		{"colesterol", "colesterol"},
		{"colgaba", "colg"},
		{"colgaban", "colg"},
		{"colgada", "colg"},
		{"colgadas", "colg"},
		{"colgado", "colg"},
		{"colgados", "colg"},
		{"colgó", "colg"},
		{"colgón", "colgon"},
		{"colima", "colim"},
		{"colin", "colin"},
		{"colín", "colin"},
		{"colina", "colin"},
		{"colinas", "colin"},
		{"coliseo", "colise"},
		{"collado", "coll"},
		{"colleen", "coll"},
		{"colleges", "colleg"},
		{"collin", "collin"},
		{"collor", "collor"},
		{"colmada", "colm"},
		{"colmo", "colm"},
		{"coloca", "coloc"},
		{"colocación", "coloc"},
		{"colocaciones", "coloc"},
		{"colocadas", "coloc"},
		{"colocado", "coloc"},
		{"colocadores", "coloc"},
		{"colocados", "coloc"},
		{"colocan", "coloc"},
		{"colocando", "coloc"},
		{"colocándose", "coloc"},
		{"colocar", "coloc"},
		{"colocara", "coloc"},
		{"colocará", "coloc"},
		{"colocarán", "coloc"},
		{"colocarla", "coloc"},
		{"colocarme", "coloc"},
		{"colocarnos", "coloc"},
		{"colocaron", "coloc"},
		{"colocarse", "coloc"},
		{"colocó", "coloc"},
		{"colofón", "colofon"},
		{"colombia", "colombi"},
		{"colombiano", "colombian"},
		{"colombianos", "colombian"},
		{"colón", "colon"},
		{"colonia", "coloni"},
		{"colonial", "colonial"},
		{"coloniales", "colonial"},
		{"colonias", "coloni"},
		{"colonizadora", "coloniz"},
		{"colonizando", "coloniz"},
		{"coloque", "coloqu"},
		{"coloquialmente", "coloquial"},
		{"coloquio", "coloqui"},
		{"color", "color"},
		{"colorado", "color"},
		{"colorantes", "color"},
		{"colores", "color"},
		{"colorido", "color"},
		{"coloridos", "color"},
		{"colorísticos", "colorist"},
		{"colosio", "colosi"},
		{"coloso", "colos"},
		{"colosos", "colos"},
		{"coludidos", "colud"},
		{"columbia", "columbi"},
		{"columna", "column"},
		{"columnabreves", "columnabrev"},
		{"columnaconsultorio", "columnaconsultori"},
		{"columnaempresa", "columnaempres"},
		{"columnafricase", "columnafric"},
		{"columnafricasé", "columnafricas"},
		{"columnamirador", "columnamir"},
		{"columnas", "column"},
		{"columneja", "columnej"},
		{"columnista", "column"},
		{"colunga", "colung"},
		{"com", "com"},
		{"comadre", "comadr"},
		{"comadrita", "comadrit"},
		{"comanda", "comand"},
		{"comandado", "comand"},
		{"comandados", "comand"},
		{"comandancia", "comand"},
		{"comandante", "comand"},
		{"comande", "comand"},
		{"comandó", "comand"},
		{"comandos", "comand"},
		{"comas", "com"},
		{"combate", "combat"},
		{"combaten", "combat"},
		{"combates", "combat"},
		{"combatido", "combat"},
		{"combatió", "combat"},
		{"combatir", "combat"},
		{"combatirán", "combat"},
		{"combatividad", "combat"},
		{"combativo", "combat"},
		{"combina", "combin"},
		{"combinación", "combin"},
		{"combinaciones", "combin"},
		{"combinada", "combin"},
		{"combinadas", "combin"},
		{"combinado", "combin"},
		{"combinados", "combin"},
		{"combinan", "combin"},
		{"combinando", "combin"},
		{"combinar", "combin"},
		{"combinará", "combin"},
		{"combinarlo", "combin"},
		{"combinarlos", "combin"},
		{"combinen", "combin"},
		{"combis", "combis"},
		{"combustible", "combust"},
		{"combustibles", "combust"},
		{"combustión", "combustion"},
		{"come", "com"},
		{"comedia", "comedi"},
		{"comediante", "comedi"},
		{"comediógrafo", "comediograf"},
		{"comedor", "comedor"},
		{"comedores", "comedor"},
		{"comemos", "com"},
		{"comensal", "comensal"},
		{"comenta", "coment"},
		{"comentaba", "coment"},
		{"comentábamos", "coment"},
		{"comentaban", "coment"},
		{"comentabas", "coment"},
		{"comentada", "coment"},
		{"comentadas", "coment"},
		{"comentado", "coment"},
		{"comentan", "coment"},
		{"comentando", "coment"},
		{"comentar", "coment"},
		{"comentaré", "coment"},
		{"comentario", "comentari"},
		{"comentarios", "comentari"},
		{"comentaristas", "comentar"},
		{"comentarla", "coment"},
		{"comentarle", "coment"},
		{"comentaron", "coment"},
		{"comentarse", "coment"},
		{"comente", "coment"},
		{"comentó", "coment"},
		{"comenzaba", "comenz"},
		{"comenzaban", "comenz"},
		{"comenzado", "comenz"},
		{"comenzamos", "comenz"},
		{"comenzar", "comenz"},
		{"comenzara", "comenz"},
		{"comenzará", "comenz"},
		{"comenzarán", "comenz"},
		{"comenzaré", "comenz"},
		{"comenzaremos", "comenz"},
		{"comenzaron", "comenz"},
		{"comenzó", "comenz"},
		{"comer", "com"},
		{"comerá", "com"},
		{"comerciaba", "comerci"},
		{"comercial", "comercial"},
		{"comerciales", "comercial"},
		{"comercializa", "comercializ"},
		{"comercialización", "comercializ"},
		{"comercializadora", "comercializ"},
		{"comercializadoras", "comercializ"},
		{"comercializan", "comercializ"},
		{"comercializar", "comercializ"},
		{"comercializaría", "comercializ"},
		{"comercializarlo", "comercializ"},
		{"comercialmente", "comercial"},
		{"comerciante", "comerci"},
		{"comerciantes", "comerci"},
		{"comerciar", "comerci"},
		{"comercio", "comerci"},
		{"comercios", "comerci"},
		{"comernos", "com"},
		{"cometa", "comet"},
		{"cometan", "comet"},
		{"cometarios", "cometari"},
		{"cometas", "comet"},
		{"comete", "comet"},
		{"cometemos", "comet"},
		{"cometen", "comet"},
		{"cometer", "comet"},
		{"cometeríamos", "comet"},
		{"cometerse", "comet"},
		{"cometía", "comet"},
		{"cometidas", "comet"},
		{"cometido", "comet"},
		{"cometidos", "comet"},
		{"cometiendo", "comet"},
		{"cometiera", "comet"},
		{"cometieron", "comet"},
		{"cometió", "comet"},
		{"comezón", "comezon"},
		{"comía", "com"},
		{"cómica", "comic"},
		{"cómicas", "comic"},
		{"comicios", "comici"},
		{"cómico", "comic"},
		{"cómicos", "comic"},
		{"comida", "com"},
		{"comidas", "com"},
		{"comience", "comienc"},
		{"comiencen", "comienc"},
		{"comienza", "comienz"},
		{"comienzan", "comienz"},
		{"comienzo", "comienz"},
		{"comienzos", "comienz"},
		{"comillas", "comill"},
		{"cominfinanciero", "cominfinancier"},
		{"comino", "comin"},
		{"comió", "com"},
		{"comisaría", "comis"},
		{"comisión", "comision"},
		{"comisionado", "comision"},
		{"comisionados", "comision"},
		{"comisiones", "comision"},
		{"comisionó", "comision"},
		{"comité", "comit"},
		{"comités", "comites"},
		{"comitiva", "comit"},
		{"comitivas", "comit"},
		{"commerce", "commerc"},
		{"communications", "communications"},
		{"como", "com"},
		{"cómo", "com"},
		{"cómoda", "comod"},
		{"cómodamente", "comod"},
		{"cómodas", "comod"},
		{"comodato", "comodat"},
		{"comodidad", "comod"},
		{"comodidades", "comod"},
		{"comodines", "comodin"},
		{"comodísimos", "comodis"},
		{"cómodo", "comod"},
		{"comodones", "comodon"},
		{"cómodos", "comod"},
		{"compact", "compact"},
		{"compacto", "compact"},
		{"compactos", "compact"},
		{"compadrazgos", "compadrazg"},
		{"compañera", "compañer"},
		{"compañeras", "compañer"},
		{"compañerismo", "compañer"},
		{"compañero", "compañer"},
		{"compañeros", "compañer"},
		{"companía", "compan"},
		{"compañía", "compañ"},
		{"compañías", "compañ"},
		{"company", "company"},
		{"compaq", "compaq"},
		{"compara", "comp"},
		{"comparable", "compar"},
		{"comparación", "compar"},
		{"comparaciones", "compar"},
		{"comparado", "compar"},
		{"comparados", "compar"},
		{"comparan", "comp"},
		{"comparando", "compar"},
		{"comparándolas", "compar"},
		{"comparándose", "compar"},
		{"comparar", "compar"},
		{"compararse", "compar"},
		{"comparársele", "compararsel"},
		{"comparativamente", "compar"},
		{"comparecencia", "comparecent"},
		{"comparecer", "comparec"},
		{"comparecerán", "comparec"},
		{"comparecía", "comparec"},
		{"comparó", "compar"},
		{"comparsa", "compars"},
		{"comparsas", "compars"},
		{"comparte", "compart"},
		{"comparten", "compart"},
		{"compartido", "compart"},
		{"compartimentos", "compartiment"},
		{"compartimiento", "compart"},
		{"compartió", "compart"},
		{"compartir", "compart"},
		{"comparto", "compart"},
		{"compás", "compas"},
		{"compasión", "compasion"},
		{"compasiva", "compas"},
		{"compatibilizar", "compatibiliz"},
		{"compatible", "compat"},
		{"compatibles", "compat"},
		{"compatriota", "compatriot"},
		{"compatriotas", "compatriot"},
		{"compelen", "compel"},
		{"compendio", "compendi"},
		{"compensa", "compens"},
		{"compensación", "compens"},
		{"compensaciones", "compens"},
		{"compensado", "compens"},
		{"compensar", "compens"},
		{"compensatorio", "compensatori"},
		{"compensatorios", "compensatori"},
		{"competa", "compet"},
		{"compete", "compet"},
		{"competencia", "competent"},
		{"competencial", "competencial"},
		{"competencias", "competent"},
		{"competente", "competent"},
		{"competentes", "competent"},
		{"competidor", "competidor"},
		{"competidoras", "competidor"},
		{"competidores", "competidor"},
		{"competir", "compet"},
		{"competirán", "compet"},
		{"competiremos", "compet"},
		{"competirle", "compet"},
		{"competitiva", "competit"},
		{"competitivas", "competit"},
		{"competitive", "competitiv"},
		{"competitividad", "competit"},
		{"competitivo", "competit"},
		{"competitivos", "competit"},
		{"compilador", "compil"},
		{"compilando", "compil"},
		{"compilar", "compil"},
		{"compite", "compit"},
		{"compiten", "compit"},
		{"compitiendo", "compit"},
		{"compito", "compit"},
		{"complacer", "complac"},
		{"complacido", "complac"},
		{"complació", "complac"},
		{"complazca", "complazc"},
		{"compleja", "complej"},
		{"complejas", "complej"},
		{"complejidad", "complej"},
		{"complejo", "complej"},
		{"complejos", "complej"},
		{"complementada", "complement"},
		{"complementan", "complement"},
		{"complementar", "complement"},
		{"complementaria", "complementari"},
		{"complementaría", "complement"},
		{"complementarias", "complementari"},
		{"complementarios", "complementari"},
		{"complementarlo", "complement"},
		{"complemento", "complement"},
		{"completa", "complet"},
		{"completadas", "complet"},
		{"completamente", "complet"},
		{"completar", "complet"},
		{"completarían", "complet"},
		{"completaron", "complet"},
		{"completarse", "complet"},
		{"completas", "complet"},
		{"completo", "complet"},
		{"completó", "complet"},
		{"completos", "complet"},
		{"complexión", "complexion"},
		{"complica", "complic"},
		{"complicación", "complic"},
		{"complicaciones", "complic"},
		{"complicadas", "complic"},
		{"complicado", "complic"},
		{"complicados", "complic"},
		{"complican", "complic"},
		{"complicar", "complic"},
		{"complicarse", "complic"},
		{"cómplice", "complic"},
		{"cómplices", "complic"},
		{"complicidad", "complic"},
		{"complicó", "complic"},
		{"compone", "compon"},
		{"componen", "compon"},
		{"componendas", "componend"},
		{"componente", "component"},
		{"componentes", "component"},
		{"componga", "compong"},
		{"componían", "compon"},
		{"comportamiento", "comport"},
		{"comportamientos", "comport"},
		{"comportan", "comport"},
		{"comportará", "comport"},
		{"composición", "composicion"},
		{"composiciones", "composicion"},
		{"compositor", "compositor"},
		{"compositores", "compositor"},
		{"compra", "compr"},
		{"compraba", "compr"},
		{"compradas", "compr"},
		{"comprado", "compr"},
		{"comprador", "comprador"},
		{"compradora", "comprador"},
		{"compradores", "comprador"},
		{"compramos", "compr"},
		{"compran", "compr"},
		{"comprando", "compr"},
		{"comprar", "compr"},
		{"comprara", "compr"},
		{"comprará", "compr"},
		{"comprarlas", "compr"},
		{"comprarle", "compr"},
		{"comprarles", "compr"},
		{"comprarme", "compr"},
		{"compraron", "compr"},
		{"comprarse", "compr"},
		{"comprárselo", "compr"},
		{"compras", "compr"},
		{"compraventa", "compravent"},
		{"compre", "compr"},
		{"compré", "compr"},
		{"comprenda", "comprend"},
		{"comprendan", "comprend"},
		{"comprendas", "comprend"},
		{"comprende", "comprend"},
		{"comprendemos", "comprend"},
		{"comprenden", "comprend"},
		{"comprender", "comprend"},
		{"comprenderá", "comprend"},
		{"comprendía", "comprend"},
		{"comprendiera", "comprend"},
		{"comprendieron", "comprend"},
		{"comprendo", "comprend"},
		{"comprensible", "comprens"},
		{"comprensión", "comprension"},
		{"comprensivo", "comprens"},
		{"compresor", "compresor"},
		{"comprimida", "comprim"},
		{"compró", "compr"},
		{"comprobado", "comprob"},
		{"comprobante", "comprob"},
		{"comprobar", "comprob"},
		{"comprobaron", "comprob"},
		{"comprobarse", "comprob"},
		{"comprobó", "comprob"},
		{"comprometan", "compromet"},
		{"compromete", "compromet"},
		{"comprometen", "compromet"},
		{"comprometernos", "compromet"},
		{"comprometerse", "compromet"},
		{"comprometía", "compromet"},
		{"comprometían", "compromet"},
		{"comprometido", "compromet"},
		{"comprometidos", "compromet"},
		{"comprometieran", "compromet"},
		{"comprometieron", "compromet"},
		{"comprometimos", "compromet"},
		{"comprometió", "compromet"},
		{"compromiso", "compromis"},
		{"compromisos", "compromis"},
		{"comprueba", "comprueb"},
		{"comprueban", "comprueb"},
		{"compuertas", "compuert"},
		{"compuesta", "compuest"},
		{"compuesto", "compuest"},
		{"compulsiva", "compuls"},
		{"compuso", "compus"},
		{"computación", "comput"},
		{"computacional", "computacional"},
		{"computacionales", "computacional"},
		{"computadora", "comput"},
		{"computadoras", "comput"},
		{"computados", "comput"},
		{"computarizado", "computariz"},
		{"computarizados", "computariz"},
		{"computec", "computec"},
		{"computer", "comput"},
		{"computo", "comput"},
		{"cómputo", "comput"},
		{"comun", "comun"},
		{"común", "comun"},
		{"comunales", "comunal"},
		{"comunes", "comun"},
		{"comunica", "comun"},
		{"comunicación", "comun"},
		{"comunicaciones", "comun"},
		{"comunicado", "comunic"},
		{"comunicados", "comunic"},
		{"comunican", "comunic"},
		{"comunicándose", "comunic"},
		{"comunicantes", "comun"},
		{"comunicar", "comunic"},
		{"comunicara", "comunic"},
		{"comunicarse", "comunic"},
		{"comunicativa", "comunic"},
		{"comunicativos", "comunic"},
		{"comunicó", "comunic"},
		{"comunidad", "comun"},
		{"comunidades", "comun"},
		{"comuniqué", "comuniqu"},
		{"comuníqueme", "comuniquem"},
		{"comuniquen", "comuniqu"},
		{"comuníquese", "comuniques"},
		{"comunista", "comun"},
		{"comunistas", "comun"},
		{"comunitarias", "comunitari"},
		{"comunitario", "comunitari"},
		{"comunitarios", "comunitari"},
		{"con", "con"},
		{"conacyt", "conacyt"},
		{"conadeip", "conadeip"},
		{"conatos", "conat"},
		{"concacaf", "concacaf"},
		{"concamin", "concamin"},
		{"concamín", "concamin"},
		{"concanaco", "concanac"},
		{"concebían", "conceb"},
		{"concebida", "conceb"},
		{"concebido", "conceb"},
		{"conceder", "conced"},
		{"concederá", "conced"},
		{"concedían", "conced"},
		{"concedida", "conced"},
		{"concedido", "conced"},
		{"concediendo", "conced"},
		{"concediera", "conced"},
		{"concedió", "conced"},
		{"concem", "concem"},
		{"concentra", "concentr"},
		{"concentración", "concentr"},
		{"concentraciones", "concentr"},
		{"concentrada", "concentr"},
		{"concentradas", "concentr"},
		{"concentrado", "concentr"},
		{"concentrados", "concentr"},
		{"concentran", "concentr"},
		{"concentrará", "concentr"},
		{"concentraran", "concentr"},
		{"concentraremos", "concentr"},
		{"concentrarse", "concentr"},
		{"concentre", "concentr"},
		{"concentró", "concentr"},
		{"concepción", "concepcion"},
		{"concepciones", "concepcion"},
		{"concepto", "concept"},
		{"conceptos", "concept"},
		{"conceptual", "conceptual"},
		{"concert", "concert"},
		{"concertación", "concert"},
		{"concertaciones", "concert"},
		{"concertadamente", "concert"},
		{"concertado", "concert"},
		{"concertando", "concert"},
		{"concertar", "concert"},
		{"concesión", "concesion"},
		{"concesionadas", "concesion"},
		{"concesionado", "concesion"},
		{"concesionarias", "concesionari"},
		{"concesionario", "concesionari"},
		{"concesionarios", "concesionari"},
		{"concesiones", "concesion"},
		{"concha", "conch"},
		{"conchello", "conchell"},
		{"conchita", "conchit"},
		{"concibe", "concib"},
		{"concibió", "concib"},
		{"conciencia", "concienci"},
		{"conciencias", "concienci"},
		{"concientizar", "concientiz"},
		{"concierne", "conciern"},
		{"concierto", "conciert"},
		{"conciertos", "conciert"},
		{"conciliación", "concili"},
		{"conciliador", "concili"},
		{"conciliar", "concili"},
		{"concilio", "concili"},
		{"conciudadanos", "conciudadan"},
		{"concluida", "conclu"},
		{"concluido", "conclu"},
		{"concluído", "concluid"},
		{"concluir", "conclu"},
		{"concluirá", "conclu"},
		{"concluirán", "conclu"},
		{"concluirla", "conclu"},
		{"conclusion", "conclusion"},
		{"conclusión", "conclusion"},
		{"conclusiones", "conclusion"},
		{"concluya", "conclu"},
		{"concluyamos", "conclu"},
		{"concluyan", "conclu"},
		{"concluye", "conclu"},
		{"concluyen", "conclu"},
		{"concluyeron", "conclu"},
		{"concluyó", "conclu"},
		{"concordar", "concord"},
		{"concordia", "concordi"},
		{"concordó", "concord"},
		{"concreción", "concrecion"},
		{"concreta", "concret"},
		{"concretado", "concret"},
		{"concretamente", "concret"},
		{"concretándolo", "concret"},
		{"concretar", "concret"},
		{"concretarse", "concret"},
		{"concretas", "concret"},
		{"concrete", "concret"},
		{"concretera", "concreter"},
		{"concreteras", "concreter"},
		{"concreto", "concret"},
		{"concretó", "concret"},
		{"concretos", "concret"},
		{"concuerda", "concuerd"},
		{"concupiscente", "concupiscent"},
		{"concurran", "concurr"},
		{"concurrencia", "concurrent"},
		{"concurrió", "concurr"},
		{"concursan", "concurs"},
		{"concursante", "concurs"},
		{"concursar", "concurs"},
		{"concurso", "concurs"},
		{"concursos", "concurs"},
		{"condado", "cond"},
		{"condados", "cond"},
		{"condde", "condd"},
		{"conde", "cond"},
		{"condecoración", "condecor"},
		{"condecorado", "condecor"},
		{"condena", "conden"},
		{"condenaban", "conden"},
		{"condenada", "conden"},
		{"condenadas", "conden"},
		{"condenado", "conden"},
		{"condenados", "conden"},
		{"condenamos", "conden"},
		{"condenan", "conden"},
		{"condenar", "conden"},
		{"condenaron", "conden"},
		{"condenas", "conden"},
		{"condene", "conden"},
		{"condenó", "conden"},
		{"condesa", "condes"},
		{"condición", "condicion"},
		{"condicionado", "condicion"},
		{"condicional", "condicional"},
		{"condicionamientos", "condicion"},
		{"condicionan", "condicion"},
		{"condicionar", "condicion"},
		{"condiciones", "condicion"},
		{"condicionó", "condicion"},
		{"condominios", "condomini"},
		{"condones", "condon"},
		{"conducción", "conduccion"},
		{"conduce", "conduc"},
		{"conducen", "conduc"},
		{"conducentes", "conducent"},
		{"conducía", "conduc"},
		{"conducida", "conduc"},
		{"conducido", "conduc"},
		{"conducidos", "conduc"},
		{"conduciendo", "conduc"},
		{"conduciéndonos", "conduc"},
		{"conduciera", "conduc"},
		{"conducimos", "conduc"},
		{"conducir", "conduc"},
		{"conducirá", "conduc"},
		{"conducta", "conduct"},
		{"conductas", "conduct"},
		{"conducto", "conduct"},
		{"conductor", "conductor"},
		{"conductores", "conductor"},
		{"conductuales", "conductual"},
		{"condujeron", "condujeron"},
		{"condujo", "conduj"},
		{"conduzca", "conduzc"},
		{"conduzcan", "conduzc"},
		{"conecta", "conect"},
		{"conectaba", "conect"},
		{"conectado", "conect"},
		{"conectar", "conect"},
		{"conectará", "conect"},
		{"conectarse", "conect"},
		{"conecte", "conect"},
		{"conectividad", "conect"},
		{"conejos", "conej"},
		{"conexiones", "conexion"},
		{"confección", "confeccion"},
		{"confeccionados", "confeccion"},
		{"confederación", "confeder"},
		{"conferencia", "conferent"},
		{"conferenciantes", "conferenci"},
		{"conferencias", "conferent"},
		{"conferencista", "conferenc"},
		{"conferencistas", "conferenc"},
		{"confesamos", "confes"},
		{"confesar", "confes"},
		{"confesaron", "confes"},
		{"confesión", "confesion"},
		{"confesó", "confes"},
		{"confesores", "confesor"},
		{"confeti", "confeti"},
		{"confía", "conf"},
		{"confiaba", "confi"},
		{"confiabilidad", "confiabil"},
		{"confiable", "confiabl"},
		{"confiables", "confiabl"},
		{"confiado", "confi"},
		{"confiados", "confi"},
		{"confiamos", "confi"},
		{"confían", "conf"},
		{"confiana", "confian"},
		{"confiant", "confiant"},
		{"confianza", "confianz"},
		{"confiar", "confi"},
		{"confiará", "confi"},
		{"confiarnos", "confi"},
		{"confiarse", "confi"},
		{"confidencias", "confident"},
		{"confidente", "confident"},
		{"confíen", "conf"},
		{"confiera", "conf"},
		{"confiesa", "confies"},
		{"confiesan", "confies"},
		{"configura", "configur"},
		{"configuración", "configur"},
		{"configuraciones", "configur"},
		{"configurada", "configur"},
		{"configuradas", "configur"},
		{"configurado", "configur"},
		{"configurados", "configur"},
		{"configurar", "configur"},
		{"confinado", "confin"},
		{"confio", "confi"},
		{"confió", "conf"},
		{"confirma", "confirm"},
		{"confirmación", "confirm"},
		{"confirmada", "confirm"},
		{"confirmado", "confirm"},
		{"confirman", "confirm"},
		{"confirmar", "confirm"},
		{"confirmaron", "confirm"},
		{"confirmó", "confirm"},
		{"conflicto", "conflict"},
		{"conflictos", "conflict"},
		{"confluyen", "conflu"},
		{"conforma", "conform"},
		{"conformaban", "conform"},
		{"conformación", "conform"},
		{"conformada", "conform"},
		{"conforman", "conform"},
		{"conformar", "conform"},
		{"conformara", "conform"},
		{"conformarse", "conform"},
		{"conforme", "conform"},
		{"conformen", "conform"},
		{"conformista", "conform"},
		{"conformo", "conform"},
		{"confort", "confort"},
		{"confrontación", "confront"},
		{"confrontaciones", "confront"},
		{"confrontando", "confront"},
		{"confrontar", "confront"},
		{"confucio", "confuci"},
		{"confunde", "confund"},
		{"confunden", "confund"},
		{"confundido", "confund"},
		{"confundidos", "confund"},
		{"confundimos", "confund"},
		{"confundir", "confund"},
		{"confundirse", "confund"},
		{"confusión", "confusion"},
		{"congelación", "congel"},
		{"congeladas", "congel"},
		{"congelados", "congel"},
		{"congelarse", "congel"},
		{"congéneres", "congener"},
		{"congénitos", "congenit"},
		{"congestión", "congestion"},
		{"congestionadas", "congestion"},
		{"congestionado", "congestion"},
		{"congestionamiento", "congestion"},
		{"congestionamientos", "congestion"},
		{"conglomerado", "conglomer"},
		{"conglomerados", "conglomer"},
		{"conglomerará", "conglomer"},
		{"congo", "cong"},
		{"congoja", "congoj"},
		{"congregación", "congreg"},
		{"congregada", "congreg"},
		{"congregados", "congreg"},
		{"congregara", "congreg"},
		{"congregaron", "congreg"},
		{"congresista", "congres"},
		{"congresistas", "congres"},
		{"congreso", "congres"},
		{"congresos", "congres"},
		{"congruencia", "congruenci"},
		{"congruente", "congruent"},
		{"congruentes", "congruent"},
		{"conjuga", "conjug"},
		{"conjugada", "conjug"},
		{"conjugaron", "conjug"},
		{"conjugarse", "conjug"},
		{"conjunción", "conjuncion"},
		{"conjunta", "conjunt"},
		{"conjuntamente", "conjunt"},
		{"conjuntar", "conjunt"},
		{"conjuntarán", "conjunt"},
		{"conjuntarse", "conjunt"},
		{"conjuntas", "conjunt"},
		{"conjuntivitis", "conjuntivitis"},
		{"conjunto", "conjunt"},
		{"conjuntos", "conjunt"},
		{"conjura", "conjur"},
		{"conjurarse", "conjur"},
		{"conlleva", "conllev"},
		{"conllevan", "conllev"},
		{"conmemorar", "conmemor"},
		{"conmemorativa", "conmemor"},
		{"conmemorativos", "conmemor"},
		{"conmigo", "conmig"},
		{"conminado", "conmin"},
		{"conmoción", "conmocion"},
		{"conmovedor", "conmovedor"},
		{"conmovedora", "conmovedor"},
		{"conmovieron", "conmov"},
		{"connor", "connor"},
		{"connotación", "connot"},
		{"connotados", "connot"},
		{"cono", "con"},
		{"conoce", "conoc"},
		{"conocedor", "conocedor"},
		{"conocedora", "conocedor"},
		{"conocedoras", "conocedor"},
		{"conocedores", "conocedor"},
		{"conocemos", "conoc"},
		{"conocen", "conoc"},
		{"conocer", "conoc"},
		{"conocerá", "conoc"},
		{"conocerán", "conoc"},
		{"conocería", "conoc"},
		{"conocerían", "conoc"},
		{"conocerlo", "conoc"},
		{"conocerse", "conoc"},
		{"conoces", "conoc"},
		{"conocí", "conoc"},
		{"conocía", "conoc"},
		{"conocían", "conoc"},
		{"conocida", "conoc"},
		{"conocidas", "conoc"},
		{"conocido", "conoc"},
		{"conocidos", "conoc"},
		{"conociendo", "conoc"},
		{"conociéndose", "conoc"},
		{"conocieron", "conoc"},
		{"conocimiento", "conoc"},
		{"conocimientos", "conoc"},
		{"conocimos", "conoc"},
		{"conoció", "conoc"},
		{"conos", "con"},
		{"conotados", "conot"},
		{"conozca", "conozc"},
		{"conozcan", "conozc"},
		{"conozco", "conozc"},
		{"conquista", "conquist"},
		{"conquistar", "conquist"},
		{"conquistas", "conquist"},
		{"consagrados", "consagr"},
		{"consagraron", "consagr"},
		{"consagró", "consagr"},
		{"consciente", "conscient"},
		{"conscientemente", "conscient"},
		{"conscientes", "conscient"},
		{"consecuencia", "consecuent"},
		{"consecuencias", "consecuent"},
		{"consecuente", "consecuent"},
		{"consecutiva", "consecut"},
		{"consecutivas", "consecut"},
		{"consecutivo", "consecut"},
		{"consecutivos", "consecut"},
		{"conseguía", "consegu"},
		{"conseguida", "consegu"},
		{"conseguido", "consegu"},
		{"conseguir", "consegu"},
		{"conseguirle", "consegu"},
		{"conseguirlo", "consegu"},
		{"conseguirlos", "consegu"},
		{"consejero", "consejer"},
		{"consejeros", "consejer"},
		{"consejo", "consej"},
		{"consejos", "consej"},
		{"consenso", "consens"},
		{"consensual", "consensual"},
		{"consentido", "consent"},
		{"consentidos", "consent"},
		{"consentimiento", "consent"},
		{"conserva", "conserv"},
		{"conservaba", "conserv"},
		{"conservaban", "conserv"},
		{"conservación", "conserv"},
		{"conservacionistas", "conservacion"},
		{"conservado", "conserv"},
		{"conservador", "conserv"},
		{"conservadores", "conserv"},
		{"conservados", "conserv"},
		{"conservan", "conserv"},
		{"conservando", "conserv"},
		{"conservar", "conserv"},
		{"conservarlo", "conserv"},
		{"conservarse", "conserv"},
		{"conserven", "conserv"},
		{"conservó", "conserv"},
		{"considera", "consider"},
		{"consideraba", "consider"},
		{"consideraban", "consider"},
		{"considerable", "consider"},
		{"considerablemente", "consider"},
		{"considerables", "consider"},
		{"consideración", "consider"},
		{"consideraciones", "consider"},
		{"considerada", "consider"},
		{"consideradas", "consider"},
		{"considerado", "consider"},
		{"considerados", "consider"},
		{"consideramos", "consider"},
		{"consideran", "consider"},
		{"considerando", "consider"},
		{"considerar", "consider"},
		{"considerara", "consider"},
		{"considerará", "consider"},
		{"consideraría", "consider"},
		{"considerarlas", "consider"},
		{"considerarlo", "consider"},
		{"consideraron", "consider"},
		{"considerarse", "consider"},
		{"considere", "consider"},
		{"consideren", "consider"},
		{"consideres", "consider"},
		{"considero", "consider"},
		{"consideró", "consider"},
		{"consiga", "consig"},
		{"consigna", "consign"},
		{"consignación", "consign"},
		{"consignado", "consign"},
		{"consignados", "consign"},
		{"consignará", "consign"},
		{"consignó", "consign"},
		{"consigo", "consig"},
		{"consigue", "consig"},
		{"consiguen", "consig"},
		{"consigues", "consig"},
		{"consiguiendo", "consigu"},
		{"consiguiente", "consiguient"},
		{"consiguieron", "consigu"},
		{"consiguió", "consigu"},
		{"consiste", "cons"},
		{"consisten", "consist"},
		{"consistencia", "consistent"},
		{"consistente", "consistent"},
		{"consistieron", "consist"},
		{"consistirá", "consist"},
		{"consistiría", "consist"},
		{"consolación", "consol"},
		{"consolarla", "consol"},
		{"consolida", "consol"},
		{"consolidación", "consolid"},
		{"consolidad", "consol"},
		{"consolidadas", "consolid"},
		{"consolidado", "consolid"},
		{"consolidar", "consolid"},
		{"consolidarlo", "consolid"},
		{"consolidaron", "consolid"},
		{"consolidarse", "consolid"},
		{"consolidó", "consolid"},
		{"consorcio", "consorci"},
		{"consorcios", "consorci"},
		{"conspiración", "conspir"},
		{"consta", "const"},
		{"constan", "const"},
		{"constancia", "constanci"},
		{"constancias", "constanci"},
		{"constante", "constant"},
		{"constantemente", "constant"},
		{"constantes", "constant"},
		{"constantiniana", "constantinian"},
		{"constantino", "constantin"},
		{"constará", "const"},
		{"constata", "constat"},
		{"constatar", "constat"},
		{"conste", "const"},
		{"consterna", "constern"},
		{"constitucion", "constitucion"},
		{"constitución", "constitu"},
		{"constitucional", "constitucional"},
		{"constitucionales", "constitucional"},
		{"constitucionalmente", "constitucional"},
		{"constituciones", "constitu"},
		{"constituía", "constitu"},
		{"constituida", "constitu"},
		{"constituido", "constitu"},
		{"constituidos", "constitu"},
		{"constituir", "constitu"},
		{"constituirá", "constitu"},
		{"constituirán", "constitu"},
		{"constituiría", "constitu"},
		{"constituirse", "constitu"},
		{"constitutiva", "constitut"},
		{"constitutivo", "constitut"},
		{"constitutivos", "constitut"},
		{"constituya", "constitu"},
		{"constituye", "constitu"},
		{"constituyen", "constitu"},
		{"constituyente", "constituyent"},
		{"constituyentes", "constituyent"},
		{"constituyeron", "constitu"},
		{"constituyó", "constitu"},
		{"construcción", "construccion"},
		{"construcciones", "construccion"},
		{"constructivo", "construct"},
		{"constructor", "constructor"},
		{"constructora", "constructor"},
		{"constructoras", "constructor"},
		{"constructores", "constructor"},
		{"construida", "constru"},
		{"construidas", "constru"},
		{"construido", "constru"},
		{"construir", "constru"},
		{"construirá", "constru"},
		{"construirán", "constru"},
		{"construirían", "constru"},
		{"construirlo", "constru"},
		{"construirse", "constru"},
		{"construya", "constru"},
		{"construyan", "constru"},
		{"construye", "constru"},
		{"construyen", "constru"},
		{"construyendo", "constru"},
		{"construyó", "constru"},
		{"consuelo", "consuel"},
		{"consuetudinario", "consuetudinari"},
		{"cónsul", "consul"},
		{"consulado", "consul"},
		{"consulados", "consul"},
		{"consular", "consul"},
		{"cónsules", "consul"},
		{"consulta", "consult"},
		{"consultadas", "consult"},
		{"consultado", "consult"},
		{"consultados", "consult"},
		{"consultar", "consult"},
		{"consultaron", "consult"},
		{"consultas", "consult"},
		{"consulte", "consult"},
		{"consultivo", "consult"},
		{"consultor", "consultor"},
		{"consultora", "consultor"},
		{"consultoría", "consultor"},
		{"consultorio", "consultori"},
		{"consultoriobursátil", "consultoriobursatil"},
		{"", ""},
		{"consumados", "consum"},
		{"consumaran", "consum"},
		{"consumida", "consum"},
		{"consumido", "consum"},
		{"consumidor", "consumidor"},
		{"consumidores", "consumidor"},
		{"consumiendo", "consum"},
		{"consumió", "consum"},
		{"consumir", "consum"},
		{"consumirán", "consum"},
		{"consumismo", "consum"},
		{"consumista", "consum"},
		{"consumo", "consum"},
		{"consumó", "consum"},
		{"contaba", "cont"},
		{"contaban", "cont"},
		{"contabilidad", "contabil"},
		{"contabilizada", "contabiliz"},
		{"contabilizan", "contabiliz"},
		{"contabilizaron", "contabiliz"},
		{"contable", "contabl"},
		{"contactado", "contact"},
		{"contactar", "contact"},
		{"contactarse", "contact"},
		{"contacto", "contact"},
		{"contactos", "contact"},
		{"contadas", "cont"},
		{"contado", "cont"},
		{"contador", "contador"},
		{"contadores", "contador"},
		{"contaduría", "contadur"},
		{"contagia", "contagi"},
		{"contagio", "contagi"},
		{"contagios", "contagi"},
		{"contagiosas", "contagi"},
		{"contamina", "contamin"},
		{"contaminación", "contamin"},
		{"contaminador", "contamin"},
		{"contaminadores", "contamin"},
		{"contaminan", "contamin"},
		{"contaminante", "contamin"},
		{"contaminantes", "contamin"},
		{"contaminar", "contamin"},
		{"contamine", "contamin"},
		{"contaminen", "contamin"},
		{"contamos", "cont"},
		{"contando", "cont"},
		{"contar", "cont"},
		{"contara", "cont"},
		{"contará", "cont"},
		{"contarán", "cont"},
		{"contaré", "cont"},
		{"contempla", "contempl"},
		{"contemplaba", "contempl"},
		{"contemplación", "contempl"},
		{"contemplada", "contempl"},
		{"contempladas", "contempl"},
		{"contemplado", "contempl"},
		{"contemplados", "contempl"},
		{"contemplan", "contempl"},
		{"contemplando", "contempl"},
		{"contemplar", "contempl"},
		{"contemplaron", "contempl"},
		{"contemplarse", "contempl"},
		{"contemple", "contempl"},
		{"contemporánea", "contemporane"},
		{"contemporáneas", "contemporan"},
		{"contemporáneo", "contemporane"},
		{"contemporáneos", "contemporane"},
		{"contención", "contencion"},
		{"contencioso", "contenci"},
		{"contender", "contend"},
		{"contenderán", "contend"},
		{"contendientes", "contendient"},
		{"contener", "conten"},
		{"contenga", "conteng"},
		{"contengan", "conteng"},
		{"contenía", "conten"},
		{"contenían", "conten"},
		{"contenidas", "conten"},
		{"contenido", "conten"},
		{"contenidos", "conten"},
		{"contenta", "content"},
		{"contentamos", "content"},
		{"contentas", "content"},
		{"contento", "content"},
		{"contentos", "content"},
		{"conteo", "conte"},
		{"contesta", "contest"},
		{"contestaban", "contest"},
		{"contestan", "contest"},
		{"contestar", "contest"},
		{"contestara", "contest"},
		{"contestarle", "contest"},
		{"contestarles", "contest"},
		{"contestatario", "contestatari"},
		{"contesté", "contest"},
		{"contestó", "contest"},
		{"contexto", "context"},
		{"conti", "conti"},
		{"contienda", "contiend"},
		{"contiene", "contien"},
		{"contienen", "contien"},
		{"contigo", "contig"},
		{"contiguo", "contigu"},
		{"continental", "continental"},
		{"continente", "continent"},
		{"continentes", "continent"},
		{"contingencia", "contingent"},
		{"contingencias", "contingent"},
		{"contingente", "contingent"},
		{"contingentes", "contingent"},
		{"continua", "continu"},
		{"contínua", "continu"},
		{"continúa", "continu"},
		{"continuaba", "continu"},
		{"continuaban", "continu"},
		{"continuación", "continu"},
		{"continuada", "continu"},
		{"continuado", "continu"},
		{"continuamente", "continu"},
		{"continúan", "continu"},
		{"continuar", "continu"},
		{"continuara", "continu"},
		{"continuará", "continu"},
		{"continuaran", "continu"},
		{"continuarán", "continu"},
		{"continuaron", "continu"},
		{"continuas", "continu"},
		{"continúe", "continu"},
		{"continuen", "continu"},
		{"continuidad", "continu"},
		{"continuo", "continu"},
		{"contínuo", "continu"},
		{"continuó", "continu"},
		{"continuos", "continu"},
		{"contó", "cont"},
		{"contorsionarse", "contorsion"},
		{"contra", "contr"},
		{"contraataque", "contraataqu"},
		{"contraataques", "contraataqu"},
		{"contrabandistas", "contraband"},
		{"contrabando", "contrab"},
		{"contracción", "contraccion"},
		{"contractual", "contractual"},
		{"contractura", "contractur"},
		{"contradecir", "contradec"},
		{"contradecirle", "contradec"},
		{"contrademanda", "contrademand"},
		{"contradicción", "contradiccion"},
		{"contradicciones", "contradiccion"},
		{"contradice", "contradic"},
		{"contradicen", "contradic"},
		{"contradictorias", "contradictori"},
		{"contradigan", "contradig"},
		{"contraer", "contr"},
		{"contragolpear", "contragolp"},
		{"contraídos", "contraid"},
		{"contrajo", "contraj"},
		{"contralor", "contralor"},
		{"contraloría", "contralor"},
		{"contralorías", "contralor"},
		{"contrantes", "contrant"},
		{"contraparte", "contrapart"},
		{"contraposiciones", "contraposicion"},
		{"contraproducente", "contraproducent"},
		{"contraproducentes", "contraproducent"},
		{"contrapuesta", "contrapuest"},
		{"contraria", "contrari"},
		{"contrarias", "contrari"},
		{"contrario", "contrari"},
		{"contrarios", "contrari"},
		{"contrarresta", "contrarrest"},
		{"contrarrestar", "contrarrest"},
		{"contrasta", "contrast"},
		{"contrastan", "contrast"},
		{"contraste", "contr"},
		{"contrastes", "contrast"},
		{"contrataban", "contrat"},
		{"contratación", "contrat"},
		{"contrataciones", "contrat"},
		{"contratadas", "contrat"},
		{"contratado", "contrat"},
		{"contratados", "contrat"},
		{"contratan", "contrat"},
		{"contratantes", "contrat"},
		{"contratar", "contrat"},
		{"contratarle", "contrat"},
		{"contratarlo", "contrat"},
		{"contrataron", "contrat"},
		{"contraten", "contrat"},
		{"contratiempo", "contratiemp"},
		{"contratiempos", "contratiemp"},
		{"contratista", "contrat"},
		{"contratistas", "contrat"},
		{"contrato", "contrat"},
		{"contrató", "contrat"},
		{"contratos", "contrat"},
		{"contrayente", "contrayent"},
		{"contreras", "contrer"},
		{"contribución", "contribu"},
		{"contribuciones", "contribu"},
		{"contribuido", "contribu"},
		{"contribuir", "contribu"},
		{"contribuirá", "contribu"},
		{"contribuiría", "contribu"},
		{"contribuyendo", "contribu"},
		{"contribuyente", "contribuyent"},
		{"contribuyentes", "contribuyent"},
		{"contribuyeron", "contribu"},
		{"contribuyó", "contribu"},
		{"contrincante", "contrinc"},
		{"contrincantes", "contrinc"},
		{"contritos", "contrit"},
		{"control", "control"},
		{"controla", "control"},
		{"controlables", "control"},
		{"controlada", "control"},
		{"controlado", "control"},
		{"controladora", "control"},
		{"controlando", "control"},
		{"controlar", "control"},
		{"controlarlo", "control"},
		{"controlaron", "control"},
		{"controlarse", "control"},
		{"controle", "control"},
		{"controles", "control"},
		{"controversia", "controversi"},
		{"controversias", "controversi"},
		{"controvertida", "controvert"},
		{"controvertido", "controvert"},
		{"contry", "contry"},
		{"contubernio", "contuberni"},
		{"contundencia", "contundent"},
		{"contundente", "contundent"},
		{"contundentemente", "contundent"},
		{"contundentes", "contundent"},
		{"conturbado", "conturb"},
		{"conurbada", "conurb"},
		{"conurbadas", "conurb"},
		{"convalecencia", "convalecent"},
		{"convaleciente", "convalecient"},
		{"convence", "convenc"},
		{"convencer", "convenc"},
		{"convencerle", "convenc"},
		{"convencerse", "convenc"},
		{"convencida", "convenc"},
		{"convencidas", "convenc"},
		{"convencido", "convenc"},
		{"convencidos", "convenc"},
		{"convencieron", "convenc"},
		{"convencimiento", "convenc"},
		{"convencimos", "convenc"},
		{"convenció", "convenc"},
		{"convención", "convencion"},
		{"convencional", "convencional"},
		{"convencionales", "convencional"},
		{"convenciones", "convencion"},
		{"convenencieros", "convenencier"},
		{"convenga", "conveng"},
		{"convenía", "conven"},
		{"convenidos", "conven"},
		{"conveniencia", "convenient"},
		{"conveniente", "convenient"},
		{"convenientes", "convenient"},
		{"convenio", "conveni"},
		{"convenios", "conveni"},
		{"convento", "convent"},
		{"convenzo", "convenz"},
		{"convergencia", "convergent"},
		{"conversación", "convers"},
		{"conversaciones", "convers"},
		{"conversarán", "convers"},
		{"conversaron", "convers"},
		{"conversión", "conversion"},
		{"conversó", "convers"},
		{"convertibles", "convert"},
		{"convertido", "convert"},
		{"convertidor", "convertidor"},
		{"convertidores", "convertidor"},
		{"convertir", "convert"},
		{"convertirá", "convert"},
		{"convertiría", "convert"},
		{"convertirnos", "convert"},
		{"convertirse", "convert"},
		{"convicción", "conviccion"},
		{"convicciones", "conviccion"},
		{"conviene", "convien"},
		{"convierta", "conviert"},
		{"conviertan", "conviert"},
		{"convierte", "conviert"},
		{"convierten", "conviert"},
		{"convincente", "convincent"},
		{"convino", "convin"},
		{"convirtiendo", "convirt"},
		{"convirtiéndola", "convirt"},
		{"convirtiéndolas", "convirt"},
		{"convirtiéndose", "convirt"},
		{"convirtiera", "convirt"},
		{"convirtieron", "convirt"},
		{"convirtió", "convirt"},
		{"convivencia", "convivent"},
		{"convivio", "convivi"},
		{"convivir", "conviv"},
		{"convocación", "convoc"},
		{"convocada", "convoc"},
		{"convocadas", "convoc"},
		{"convocado", "convoc"},
		{"convocados", "convoc"},
		{"convocan", "convoc"},
		{"convocar", "convoc"},
		{"convocará", "convoc"},
		{"convocaron", "convoc"},
		{"convocatoria", "convocatori"},
		{"convocatorias", "convocatori"},
		{"convocó", "convoc"},
		{"convoqué", "convoqu"},
		{"convoquen", "convoqu"},
		{"convoy", "convoy"},
		{"convoyes", "convoy"},
		{"convulsionados", "convulsion"},
		{"convulsivas", "convuls"},
		{"cony", "cony"},
		{"conyugal", "conyugal"},
		{"conyugales", "conyugal"},
		{"cooper", "coop"},
		{"cooperación", "cooper"},
		{"cooperar", "cooper"},
		{"cooperará", "cooper"},
		{"cooperaron", "cooper"},
		{"cooperativa", "cooper"},
		{"cooperen", "cooper"},
		{"cooppel", "cooppel"},
		{"cooptación", "cooptacion"},
		{"coordenadas", "coorden"},
		{"coordina", "coordin"},
		{"coordinación", "coordin"},
		{"coordinada", "coordin"},
		{"coordinadas", "coordin"},
		{"coordinador", "coordin"},
		{"coordinadora", "coordin"},
		{"coordinadores", "coordin"},
		{"coordinados", "coordin"},
		{"coordinan", "coordin"},
		{"coordinando", "coordin"},
		{"coordinar", "coordin"},
		{"coordinarán", "coordin"},
		{"coordinaron", "coordin"},
		{"copa", "cop"},
		{"copán", "copan"},
		{"coparmex", "coparmex"},
		{"copas", "cop"},
		{"copia", "copi"},
		{"copiar", "copi"},
		{"copias", "copi"},
		{"copiemos", "copi"},
		{"copioso", "copios"},
		{"coppel", "coppel"},
		{"cópula", "copul"},
		{"coraje", "coraj"},
		{"corán", "coran"},
		{"coraza", "coraz"},
		{"corazon", "corazon"},
		{"corazón", "corazon"},
		{"corazones", "corazon"},
		{"corbalá", "corbal"},
		{"corbata", "corbat"},
		{"corbatas", "corbat"},
		{"cordelia", "cordeli"},
		{"corderos", "corder"},
		{"cordial", "cordial"},
		{"cordillera", "cordiller"},
		{"córdoba", "cordob"},
		{"cordobés", "cordobes"},
		{"cordón", "cordon"},
		{"cordura", "cordur"},
		{"corea", "core"},
		{"coreados", "cor"},
		{"coreana", "corean"},
		{"coreógrafa", "coreograf"},
		{"coreografías", "coreograf"},
		{"coreográfica", "coreograf"},
		{"corinter", "corint"},
		{"cornalina", "cornalin"},
		{"córneas", "corn"},
		{"cornejo", "cornej"},
		{"cornoyer", "cornoy"},
		{"coro", "cor"},
		{"corona", "coron"},
		{"coronacion", "coronacion"},
		{"coronación", "coron"},
		{"coronada", "coron"},
		{"coronado", "coron"},
		{"coronas", "coron"},
		{"coronelas", "coronel"},
		{"coronó", "coron"},
		{"coros", "cor"},
		{"corp", "corp"},
		{"corporación", "corpor"},
		{"corporaciones", "corpor"},
		{"corporal", "corporal"},
		{"corporation", "corporation"},
		{"corporativa", "corpor"},
		{"corporativas", "corpor"},
		{"corporativo", "corpor"},
		{"corporativos", "corpor"},
		{"corpórea", "corpore"},
		{"corpus", "corpus"},
		{"corra", "corr"},
		{"corral", "corral"},
		{"corrales", "corral"},
		{"corran", "corr"},
		{"corre", "corr"},
		{"correajes", "correaj"},
		{"correcaminos", "correcamin"},
		{"corrección", "correccion"},
		{"correcciones", "correccion"},
		{"correcta", "correct"},
		{"correctamente", "correct"},
		{"correctas", "correct"},
		{"correcto", "correct"},
		{"corredor", "corredor"},
		{"corredores", "corredor"},
		{"correduría", "corredur"},
		{"corregible", "correg"},
		{"corregido", "correg"},
		{"corregidora", "corregidor"},
		{"corregir", "correg"},
		{"corregirla", "correg"},
		{"corregirlo", "correg"},
		{"correligionario", "correligionari"},
		{"correligionarios", "correligionari"},
		{"correlon", "correlon"},
		{"correlones", "correlon"},
		{"corremos", "corr"},
		{"corren", "corr"},
		{"correo", "corre"},
		{"correr", "corr"},
		{"correrá", "corr"},
		{"correrán", "corr"},
		{"correrían", "corr"},
		{"correrlos", "corr"},
		{"corres", "corr"},
		{"correspondan", "correspond"},
		{"corresponde", "correspond"},
		{"corresponden", "correspond"},
		{"correspondencia", "correspondent"},
		{"corresponder", "correspond"},
		{"correspondía", "correspond"},
		{"correspondido", "correspond"},
		{"correspondiente", "correspondient"},
		{"correspondientes", "correspondient"},
		{"correspondieron", "correspond"},
		{"correspondió", "correspond"},
		{"corresponsal", "corresponsal"},
		{"correveidile", "correveidil"},
		{"corrí", "corr"},
		{"corría", "corr"},
		{"corrían", "corr"},
		{"corrida", "corr"},
		{"corridas", "corr"},
		{"corrido", "corr"},
		{"corridos", "corr"},
		{"corriendo", "corr"},
		{"corriente", "corrient"},
		{"corrientes", "corrient"},
		{"corrieron", "corr"},
		{"corrige", "corrig"},
		{"corrigen", "corrig"},
		{"corrijan", "corrij"},
		{"corrillo", "corrill"},
		{"corrió", "corr"},
		{"corripio", "corripi"},
		{"corro", "corr"},
		{"corrobora", "corrobor"},
		{"corroboraban", "corrobor"},
		{"corroboraron", "corrobor"},
		{"corrompa", "corromp"},
		{"corrompe", "corromp"},
		{"corrosión", "corrosion"},
		{"corrupción", "corrupcion"},
		{"corruptela", "corruptel"},
		{"corrupto", "corrupt"},
		{"corruptos", "corrupt"},
		{"corta", "cort"},
		{"cortada", "cort"},
		{"cortadillo", "cortadill"},
		{"cortadillos", "cortadill"},
		{"cortado", "cort"},
		{"cortadores", "cortador"},
		{"cortando", "cort"},
		{"cortantes", "cortant"},
		{"cortar", "cort"},
		{"cortarle", "cort"},
		{"cortarlos", "cort"},
		{"cortarnos", "cort"},
		{"cortárselo", "cort"},
		{"cortas", "cort"},
		{"cortázar", "cortaz"},
		{"corte", "cort"},
		{"cortedad", "corted"},
		{"cortejada", "cortej"},
		{"cortejar", "cortej"},
		{"cortés", "cortes"},
		{"cortesanas", "cortesan"},
		{"cortesía", "cortes"},
		{"cortez", "cortez"},
		{"corteza", "cortez"},
		{"cortina", "cortin"},
		{"cortinas", "cortin"},
		{"cortines", "cortin"},
		{"cortito", "cortit"},
		{"corto", "cort"},
		{"cortó", "cort"},
		{"cortoplazo", "cortoplaz"},
		{"cortos", "cort"},
		{"coruña", "coruñ"},
		{"corvette", "corvett"},
		{"corvi", "corvi"},
		{"cosa", "cos"},
		{"cosas", "cos"},
		{"cosaspeores", "cosaspeor"},
		{"cosecha", "cosech"},
		{"cosechando", "cosech"},
		{"cosechas", "cosech"},
		{"coser", "cos"},
		{"cosméticos", "cosmet"},
		{"cósmica", "cosmic"},
		{"cósmico", "cosmic"},
		{"cosmovisión", "cosmovision"},
		{"coso", "cos"},
		{"coss", "coss"},
		{"cossío", "cossi"},
		{"costa", "cost"},
		{"costaba", "cost"},
		{"costado", "cost"},
		{"costados", "cost"},
		{"costal", "costal"},
		{"costales", "costal"},
		{"costándole", "cost"},
		{"costar", "cost"},
		{"costará", "cost"},
		{"costarán", "cost"},
		{"costarricense", "costarricens"},
		{"costas", "cost"},
		{"costeable", "costeabl"},
		{"costee", "coste"},
		{"costera", "coster"},
		{"costeras", "coster"},
		{"costero", "coster"},
		{"costilla", "costill"},
		{"costner", "costn"},
		{"costo", "cost"},
		{"costó", "cost"},
		{"costos", "cost"},
		{"costosas", "cost"},
		{"costoso", "costos"},
		{"costosos", "costos"},
		{"costumbre", "costumbr"},
		{"costumbres", "costumbr"},
		{"costura", "costur"},
		{"costuras", "costur"},
		{"costureras", "costurer"},
		{"cota", "cot"},
		{"cotejo", "cotej"},
		{"cotero", "coter"},
		{"cotidiana", "cotidian"},
		{"cotidianas", "cotidian"},
		{"cotidianeidad", "cotidian"},
		{"cotidiano", "cotidian"},
		{"cotidianos", "cotidian"},
		{"cotiza", "cotiz"},
		{"cotización", "cotiz"},
		{"cotizaciones", "cotiz"},
		{"cotizado", "cotiz"},
		{"cotizados", "cotiz"},
		{"cotizan", "cotiz"},
		{"cotizar", "cotiz"},
		{"cotorreo", "cotorre"},
		{"couches", "couch"},
		{"cougar", "coug"},
		{"country", "country"},
		{"covarrubias", "covarrubi"},
		{"cover", "cov"},
		{"covers", "covers"},
		{"cóvers", "covers"},
		{"covian", "covi"},
		{"cowboy", "cowboy"},
		{"cox", "cox"},
		{"coyolxauhqui", "coyolxauhqui"},
		{"coyotaje", "coyotaj"},
		{"coyote", "coyot"},
		{"coyotepec", "coyotepec"},
		{"coyuca", "coyuc"},
		{"coyuntura", "coyuntur"},
		{"cozumel", "cozumel"},
		{"cpp", "cpp"},
		{"crack", "crack"},
		{"cráneo", "crane"},
		{"cráter", "crat"},
		{"cráteres", "crater"},
		{"cravioto", "craviot"},
		{"crawford", "crawford"},
		{"crayones", "crayon"},
		{"crea", "cre"},
		{"creacion", "creacion"},
		{"creación", "creacion"},
		{"creaciones", "creacion"},
		{"creada", "cre"},
		{"creadas", "cre"},
		{"creado", "cre"},
		{"creador", "creador"},
		{"creadores", "creador"},
		{"creame", "cream"},
		{"creamos", "cre"},
		{"crean", "cre"},
		{"creando", "cre"},
		{"crear", "cre"},
		{"creara", "cre"},
		{"creará", "cre"},
		{"crearán", "cre"},
		{"crearía", "cre"},
		{"crearles", "cre"},
		{"crearon", "cre"},
		{"creativa", "creativ"},
		{"creativas", "creativ"},
		{"creatividad", "creativ"},
		{"creativo", "creativ"},
		{"creativos", "creativ"},
		{"crece", "crec"},
		{"crecen", "crec"},
		{"crecer", "crec"},
		{"crecerá", "crec"},
		{"crecerán", "crec"},
		{"creceremos", "crec"},
		{"crecerían", "crec"},
		{"creces", "crec"},
		{"crecí", "crec"},
		{"crecido", "crec"},
		{"creciendo", "crec"},
		{"creciente", "crecient"},
		{"crecieron", "crec"},
		{"crecimiento", "crecimient"},
		{"crecimientos", "crecimient"},
		{"creció", "crec"},
		{"credencial", "credencial"},
		{"credenciales", "credencial"},
		{"credencialización", "credencializ"},
		{"credencializados", "credencializ"},
		{"credibilidad", "credibil"},
		{"crediticia", "creditici"},
		{"crediticias", "creditici"},
		{"crediticio", "creditici"},
		{"crediticios", "creditici"},
		{"crédito", "credit"},
		{"creditos", "credit"},
		{"créditos", "credit"},
		{"cree", "cre"},
		{"créeme", "creem"},
		{"creemos", "cre"},
		{"creen", "cre"},
		{"creencia", "creenci"},
		{"creencias", "creenci"},
		{"creer", "cre"},
		{"creerán", "cre"},
		{"creería", "cre"},
		{"creerme", "cre"},
		{"creía", "cre"},
		{"creíamos", "cre"},
		{"creíble", "creibl"},
		{"creíbles", "creibl"},
		{"creído", "creid"},
		{"creímos", "creim"},
		{"cremas", "crem"},
		{"cremi", "cremi"},
		{"creo", "cre"},
		{"creó", "cre"},
		{"creole", "creol"},
		{"cretinismo", "cretin"},
		{"creyendo", "creyend"},
		{"creyó", "crey"},
		{"crezca", "crezc"},
		{"criada", "cri"},
		{"criadas", "cri"},
		{"criadero", "criader"},
		{"criadita", "criadit"},
		{"criadores", "criador"},
		{"criatura", "criatur"},
		{"criaturas", "criatur"},
		{"crimen", "crim"},
		{"crímenes", "crimen"},
		{"criminal", "criminal"},
		{"criminales", "criminal"},
		{"criminalidad", "criminal"},
		{"criminalista", "criminal"},
		{"criocirugía", "criocirug"},
		{"criollos", "crioll"},
		{"crisantemo", "crisantem"},
		{"crisantemos", "crisant"},
		{"crisis", "crisis"},
		{"cristal", "cristal"},
		{"cristales", "cristal"},
		{"cristalización", "cristaliz"},
		{"cristalizado", "cristaliz"},
		{"cristiana", "cristian"},
		{"cristianismo", "cristian"},
		{"cristiano", "cristian"},
		{"cristianos", "cristian"},
		{"cristina", "cristin"},
		{"cristo", "crist"},
		{"cristobal", "cristobal"},
		{"criterio", "criteri"},
		{"criterios", "criteri"},
		{"critica", "critic"},
		{"crítica", "critic"},
		{"criticaban", "critic"},
		{"criticado", "critic"},
		{"criticamente", "critic"},
		{"critican", "critic"},
		{"criticando", "critic"},
		{"criticar", "critic"},
		{"criticará", "critic"},
		{"criticarlos", "critic"},
		{"criticaron", "critic"},
		{"críticas", "critic"},
		{"crítico", "critic"},
		{"criticó", "critic"},
		{"críticos", "critic"},
		{"croacia", "croaci"},
		{"croc", "croc"},
		{"cromadas", "crom"},
		{"cromadoras", "cromador"},
		{"cromados", "crom"},
		{"cromera", "cromer"},
		{"crónica", "cronic"},
		{"crónicas", "cronic"},
		{"cronista", "cronist"},
		{"cronistas", "cronist"},
		{"cronometraron", "cronometr"},
		{"cronómetro", "cronometr"},
		{"cronometró", "cronometr"},
		{"cronómetros", "cronometr"},
		{"croquetas", "croquet"},
		{"croquis", "croquis"},
		{"croquistas", "croquist"},
		{"croux", "croux"},
		{"cruce", "cruc"},
		{"crucero", "crucer"},
		{"cruceros", "crucer"},
		{"cruces", "cruc"},
		{"crucial", "crucial"},
		{"cruda", "crud"},
		{"crudo", "crud"},
		{"crue", "cru"},
		{"cruel", "cruel"},
		{"cruenta", "cruent"},
		{"cruise", "cruis"},
		{"cruz", "cruz"},
		{"cruza", "cruz"},
		{"cruzada", "cruz"},
		{"cruzadas", "cruz"},
		{"cruzado", "cruz"},
		{"cruzados", "cruz"},
		{"cruzan", "cruz"},
		{"cruzando", "cruz"},
		{"cruzar", "cruz"},
		{"cruzará", "cruz"},
		{"cruzaron", "cruz"},
		{"cruzas", "cruz"},
		{"cruzó", "cruz"},
		{"crysel", "crysel"},
		{"cse", "cse"},
		{"csg", "csg"},
		{"ctm", "ctm"},
		{"cu", "cu"},
		{"cuadra", "cuadr"},
		{"cuadrado", "cuadr"},
		{"cuadrados", "cuadr"},
		{"cuadrangular", "cuadrangul"},
		{"cuadras", "cuadr"},
		{"cuadrilátero", "cuadrilater"},
		{"cuadriláteros", "cuadrilater"},
		{"cuadrilla", "cuadrill"},
		{"cuadro", "cuadr"},
		{"cuadros", "cuadr"},
		{"cuadruplica", "cuadrupl"},
		{"cuadruplicó", "cuadruplic"},
		{"cuajar", "cuaj"},
		{"cuajarle", "cuaj"},
		{"cuajimalpa", "cuajimalp"},
		{"cual", "cual"},
		{"cuál", "cual"},
		{"cuales", "cual"},
		{"cuáles", "cual"},
		{"cualesquier", "cualesqui"},
		{"cualidad", "cualid"},
		{"cualidades", "cualidad"},
		{"cualquier", "cualqui"},
		{"cualquiera", "cualqu"},
		{"cuando", "cuand"},
		{"cuándo", "cuand"},
		{"cuánta", "cuant"},
		{"cuantas", "cuant"},
		{"cuántas", "cuant"},
		{"cuantifica", "cuantif"},
		{"cuantiosamente", "cuanti"},
		{"cuantiosos", "cuantios"},
		{"cuanto", "cuant"},
		{"cuánto", "cuant"},
		{"cuantos", "cuant"},
		{"cuántos", "cuant"},
		{"cuarenta", "cuarent"},
		{"cuarta", "cuart"},
		{"cuartel", "cuartel"},
		{"cuarteles", "cuartel"},
		{"cuarteto", "cuartet"},
		{"cuartillas", "cuartill"},
		{"cuartitos", "cuartit"},
		{"cuarto", "cuart"},
		{"cuartos", "cuart"},
		{"cuarzo", "cuarz"},
		{"cuatacho", "cuatach"},
		{"cuate", "cuat"},
		{"cuatro", "cuatr"},
		{"cuauhtémoc", "cuauhtemoc"},
		{"cuautitlán", "cuautitlan"},
		{"cuba", "cub"},
		{"cubana", "cuban"},
		{"cubanas", "cuban"},
		{"cubano", "cuban"},
		{"cubanos", "cuban"},
		{"cubas", "cub"},
		{"cubero", "cuber"},
		{"cúbicos", "cubic"},
		{"cubículos", "cubicul"},
		{"cubierta", "cubiert"},
		{"cubiertas", "cubiert"},
		{"cubierto", "cubiert"},
		{"cubiertos", "cubiert"},
		{"cubra", "cubr"},
		{"cubran", "cubr"},
		{"cubre", "cubr"},
		{"cubren", "cubr"},
		{"cubría", "cubr"},
		{"cubrían", "cubr"},
		{"cubriendo", "cubr"},
		{"cubriéndonos", "cubr"},
		{"cubrimos", "cubr"},
		{"cubrió", "cubr"},
		{"cubrir", "cubr"},
		{"cubrirá", "cubr"},
		{"cubrirán", "cubr"},
		{"cuca", "cuc"},
		{"cuchara", "cuch"},
		{"cucharaditas", "cucharadit"},
		{"cucharas", "cuch"},
		{"cuchillada", "cuchill"},
		{"cuchillería", "cuchill"},
		{"cuchillo", "cuchill"},
		{"cuchillos", "cuchill"},
		{"cuco", "cuc"},
		{"cuelgan", "cuelg"},
		{"cuelgue", "cuelg"},
		{"cuellar", "cuell"},
		{"cuéllar", "cuell"},
		{"cuello", "cuell"},
		{"cuenca", "cuenc"},
		{"cuenta", "cuent"},
		{"cuentan", "cuent"},
		{"cuentas", "cuent"},
		{"cuente", "cuent"},
		{"cuenten", "cuent"},
		{"cuento", "cuent"},
		{"cuentos", "cuent"},
		{"cuerda", "cuerd"},
		{"cuerdas", "cuerd"},
		{"cuerdo", "cuerd"},
		{"cuernavaca", "cuernavac"},
		{"cuerno", "cuern"},
		{"cuernos", "cuern"},
		{"cuero", "cuer"},
		{"cuerpo", "cuerp"},
		{"cuerpos", "cuerp"},
		{"cuervo", "cuerv"},
		{"cuesta", "cuest"},
		{"cuestan", "cuest"},
		{"cuestas", "cuest"},
		{"cuestion", "cuestion"},
		{"cuestión", "cuestion"},
		{"cuestiona", "cuestion"},
		{"cuestionada", "cuestion"},
		{"cuestionado", "cuestion"},
		{"cuestionados", "cuestion"},
		{"cuestionamiento", "cuestion"},
		{"cuestionamientos", "cuestion"},
		{"cuestionan", "cuestion"},
		{"cuestiones", "cuestion"},
		{"cuestionó", "cuestion"},
		{"cueva", "cuev"},
		{"cuevas", "cuev"},
		{"cuicalli", "cuicalli"},
		{"cuicuilco", "cuicuilc"},
		{"cuida", "cuid"},
		{"cuidaba", "cuid"},
		{"cuidad", "cuid"},
		{"cuidadas", "cuid"},
		{"cuidado", "cuid"},
		{"cuidador", "cuidador"},
		{"cuidados", "cuid"},
		{"cuidadosa", "cuidad"},
		{"cuidadosamente", "cuidad"},
		{"cuidadoso", "cuidad"},
		{"cuidan", "cuid"},
		{"cuidando", "cuid"},
		{"cuidar", "cuid"},
		{"cuidará", "cuid"},
		{"cuidarlo", "cuid"},
		{"cuide", "cuid"},
		{"cuiden", "cuid"},
		{"cuitlahuac", "cuitlahuac"},
		{"culebra", "culebr"},
		{"culiacan", "culiac"},
		{"culiacán", "culiacan"},
		{"culinarias", "culinari"},
		{"culminada", "culmin"},
		{"culminado", "culmin"},
		{"culminante", "culmin"},
		{"culminará", "culmin"},
		{"culminarán", "culmin"},
		{"culminó", "culmin"},
		{"culpa", "culp"},
		{"culpabilidad", "culpabil"},
		{"culpable", "culpabl"},
		{"culpables", "culpabl"},
		{"culpar", "culp"},
		{"culpó", "culp"},
		{"culposos", "culpos"},
		{"culta", "cult"},
		{"cultiva", "cultiv"},
		{"cultivable", "cultiv"},
		{"cultivadas", "cultiv"},
		{"cultivadores", "cultiv"},
		{"cultivan", "cultiv"},
		{"cultivar", "cultiv"},
		{"cultivo", "cultiv"},
		{"cultivos", "cultiv"},
		{"culto", "cult"},
		{"cultos", "cult"},
		{"cultura", "cultur"},
		{"cultural", "cultural"},
		{"culturales", "cultural"},
		{"culturas", "cultur"},
		{"culturización", "culturiz"},
		{"cumaná", "cuman"},
		{"cumbre", "cumbr"},
		{"cumbres", "cumbr"},
		{"cumpla", "cumpl"},
		{"cumplan", "cumpl"},
		{"cumple", "cumpl"},
		{"cumpleaños", "cumpleañ"},
		{"cumplen", "cumpl"},
		{"cumplí", "cumpl"},
		{"cumplía", "cumpl"},
		{"cumplían", "cumpl"},
		{"cumplidas", "cumpl"},
		{"cumplido", "cumpl"},
		{"cumplidoras", "cumplidor"},
		{"cumplidos", "cumpl"},
		{"cumpliendo", "cumpl"},
		{"cumpliéramos", "cumpl"},
		{"cumplieron", "cumpl"},
		{"cumplimiento", "cumplimient"},
		{"cumplimientos", "cumplimient"},
		{"cumplimos", "cumpl"},
		{"cumplió", "cumpl"},
		{"cumplir", "cumpl"},
		{"cumplirá", "cumpl"},
		{"cumplirles", "cumpl"},
		{"cumplirse", "cumpl"},
		{"cumplírsele", "cumplirsel"},
		{"cuñada", "cuñ"},
		{"cuñado", "cuñ"},
		{"cuñados", "cuñ"},
		{"cunda", "cund"},
		{"cundió", "cund"},
		{"cuño", "cuñ"},
		{"cuota", "cuot"},
		{"cuotas", "cuot"},
		{"cupertino", "cupertin"},
		{"cupo", "cup"},
		{"cupón", "cupon"},
		{"cúpula", "cupul"},
		{"cúpulas", "cupul"},
		{"cura", "cur"},
		{"curaban", "cur"},
		{"curandero", "curander"},
		{"curar", "cur"},
		{"cure", "cur"},
		{"curia", "curi"},
		{"curiosamente", "curi"},
		{"curiosas", "curi"},
		{"curiosidad", "curi"},
		{"curiosidades", "curi"},
		{"curioso", "curios"},
		{"curriculum", "curriculum"},
		{"currículum", "curriculum"},
		{"curro", "curr"},
		{"cursada", "curs"},
		{"cursando", "curs"},
		{"cursaron", "curs"},
		{"curso", "curs"},
		{"cursos", "curs"},
		{"curva", "curv"},
		{"curvatura", "curvatur"},
		{"cuspinera", "cuspiner"},
		{"custodia", "custodi"},
		{"custodiado", "custodi"},
		{"custodian", "custodi"},
		{"custodiar", "custodi"},
		{"customline", "customlin"},
		{"cutáneas", "cutan"},
		{"cutzamala", "cutzamal"},
		{"cuya", "cuy"},
		{"cuyas", "cuy"},
		{"cuyo", "cuy"},
		{"cuyos", "cuy"},
		{"cydsa", "cydsa"},
		{"cyrus", "cyrus"},
		{"d", "d"},
		{"da", "da"},
		{"daba", "dab"},
		{"daban", "dab"},
		{"dacarett", "dacarett"},
		{"dad", "dad"},
		{"dada", "dad"},
		{"dadas", "dad"},
		{"dado", "dad"},
		{"dados", "dad"},
		{"dagoberto", "dagobert"},
		{"dahn", "dahn"},
		{"daihim", "daihim"},
		{"daily", "daily"},
		{"daimatsu", "daimatsu"},
		{"daimler", "daiml"},
		{"dalia", "dali"},
		{"dallas", "dall"},
		{"daltonismo", "dalton"},
		{"dama", "dam"},
		{"damas", "dam"},
		{"dame", "dam"},
		{"damnificada", "damnific"},
		{"damnificados", "damnific"},
		{"damos", "dam"},
		{"dan", "dan"},
		{"daña", "dañ"},
		{"dañaban", "dañ"},
		{"dañada", "dañ"},
		{"dañadas", "dañ"},
		{"dañado", "dañ"},
		{"dañados", "dañ"},
		{"dañan", "dañ"},
		{"dañar", "dañ"},
		{"dañará", "dañ"},
		{"dañaran", "dañ"},
		{"dañarse", "dañ"},
		{"dando", "dand"},
		{"dándole", "dandol"},
		{"dándonos", "dandon"},
		{"daniel", "daniel"},
		{"daniela", "daniel"},
		{"dañino", "dañin"},
		{"dañinos", "dañin"},
		{"danny", "danny"},
		{"daño", "dañ"},
		{"daños", "dañ"},
		{"dante", "dant"},
		{"danza", "danz"},
		{"danzar", "danz"},
		{"danzas", "danz"},
		{"dar", "dar"},
		{"dará", "dar"},
		{"darán", "daran"},
		{"darás", "daras"},
		{"darci", "darci"},
		{"daré", "dar"},
		{"daría", "dar"},
		{"darían", "dar"},
		{"darío", "dari"},
		{"darla", "darl"},
		{"darle", "darl"},
		{"darles", "darl"},
		{"darnos", "darn"},
		{"darse", "dars"},
		{"dárselo", "darsel"},
		{"dárselos", "darsel"},
		{"das", "das"},
		{"dasa", "das"},
		{"dasd", "dasd"},
		{"data", "dat"},
		{"date", "dat"},
		{"dato", "dat"},
		{"datos", "dat"},
		{"datsun", "datsun"},
		{"datum", "datum"},
		{"davalillo", "davalill"},
		{"dávalos", "daval"},
		{"dave", "dav"},
		{"david", "dav"},
		{"dávila", "davil"},
		{"davis", "davis"},
		{"day", "day"},
		{"days", "days"},
		{"daza", "daz"},
		{"ddf", "ddf"},
		{"de", "de"},
		{"dé", "de"},
		{"dean", "dean"},
		{"deba", "deb"},
		{"debajo", "debaj"},
		{"deban", "deb"},
		{"debate", "debat"},
		{"debatido", "debat"},
		{"debatir", "debat"},
		{"debatirán", "debat"},
		{"debe", "deb"},
		{"debemos", "deb"},
		{"deben", "deb"},
		{"deber", "deb"},
		{"deberá", "deb"},
		{"deberán", "deb"},
		{"deberás", "deb"},
		{"deberes", "deber"},
		{"debería", "deb"},
		{"deberíamos", "deb"},
		{"deberían", "deb"},
		{"debes", "deb"},
		{"debí", "deb"},
		{"debía", "deb"},
		{"debíamos", "deb"},
		{"debían", "deb"},
		{"debidamente", "debid"},
		{"debido", "deb"},
		{"debiera", "deb"},
		{"debieran", "deb"},
		{"debieron", "deb"},
		{"débil", "debil"},
		{"debiles", "debil"},
		{"débiles", "debil"},
		{"debilidad", "debil"},
		{"debilidades", "debil"},
		{"debilita", "debilit"},
		{"debilitado", "debilit"},
		{"debilitar", "debilit"},
		{"debilitarnos", "debilit"},
		{"debilite", "debilit"},
		{"debió", "deb"},
		{"débito", "debit"},
		{"debo", "deb"},
		{"debut", "debut"},
		{"debutado", "debut"},
		{"debutante", "debut"},
		{"debutarán", "debut"},
		{"debutó", "debut"},
		{"década", "dec"},
		{"décadas", "dec"},
		{"decadencia", "decadent"},
		{"decadente", "decadent"},
		{"decaer", "dec"},
		{"decaería", "dec"},
		{"decaído", "decaid"},
		{"decaiga", "decaig"},
		{"decanini", "decanini"},
		{"decano", "decan"},
		{"decapitada", "decapit"},
		{"decapitados", "decapit"},
		{"decapitó", "decapit"},
		{"decayó", "decay"},
		{"decenas", "decen"},
		{"decenso", "decens"},
		{"decente", "decent"},
		{"decepción", "decepcion"},
		{"decepciona", "decepcion"},
		{"decepcionada", "decepcion"},
		{"decepcionadas", "decepcion"},
		{"decepcionado", "decepcion"},
		{"decía", "dec"},
		{"decíamos", "dec"},
		{"decían", "dec"},
		{"decida", "dec"},
		{"decidan", "decid"},
		{"decide", "decid"},
		{"deciden", "decid"},
		{"decidí", "decid"},
		{"decidida", "decid"},
		{"decididamente", "decidid"},
		{"decidido", "decid"},
		{"decidieron", "decid"},
		{"decidimos", "decid"},
		{"decidió", "decid"},
		{"decidir", "decid"},
		{"decidirá", "decid"},
		{"decidirán", "decid"},
		{"decidiría", "decid"},
		{"decidirse", "decid"},
		{"decido", "dec"},
		{"décima", "decim"},
		{"décimas", "decim"},
		{"décimo", "decim"},
		{"decimos", "dec"},
		{"decir", "dec"},
		{"decirle", "dec"},
		{"decirles", "dec"},
		{"decirlo", "dec"},
		{"decirme", "dec"},
		{"decirnos", "dec"},
		{"decirse", "dec"},
		{"decírselo", "dec"},
		{"decision", "decision"},
		{"decisión", "decision"},
		{"decisiones", "decision"},
		{"decisioón", "decisioon"},
		{"decisiva", "decis"},
		{"decisivo", "decis"},
		{"decisivos", "decis"},
		{"declamaba", "declam"},
		{"declara", "decl"},
		{"declaraban", "declar"},
		{"declaración", "declar"},
		{"declaraciones", "declar"},
		{"declarada", "declar"},
		{"declaradas", "declar"},
		{"declarado", "declar"},
		{"declarante", "declar"},
		{"declarar", "declar"},
		{"declararon", "declar"},
		{"declararse", "declar"},
		{"declaratoria", "declaratori"},
		{"declare", "declar"},
		{"declaren", "declar"},
		{"declaró", "declar"},
		{"declinación", "declin"},
		{"declinar", "declin"},
		{"declinó", "declin"},
		{"decomisadas", "decomis"},
		{"decomisados", "decomis"},
		{"decomisan", "decomis"},
		{"decomisaron", "decomis"},
		{"decomisó", "decomis"},
		{"decoración", "decor"},
		{"decorado", "decor"},
		{"decorados", "decor"},
		{"decorar", "decor"},
		{"decorativo", "decor"},
		{"decrecimiento", "decrec"},
		{"decremento", "decrement"},
		{"decrescenzo", "decrescenz"},
		{"decretado", "decret"},
		{"decretan", "decret"},
		{"decretaron", "decret"},
		{"decretarse", "decret"},
		{"decreto", "decret"},
		{"decretó", "decret"},
		{"decretos", "decret"},
		{"dedazo", "dedaz"},
		{"dedazos", "dedaz"},
		{"dedeó", "dede"},
		{"dedica", "dedic"},
		{"dedicación", "dedic"},
		{"dedicada", "dedic"},
		{"dedicadas", "dedic"},
		{"dedicado", "dedic"},
		{"dedicados", "dedic"},
		{"dedicamos", "dedic"},
		{"dedican", "dedic"},
		{"dedicando", "dedic"},
		{"dedicar", "dedic"},
		{"dedicará", "dedic"},
		{"dedicaran", "dedic"},
		{"dedicaré", "dedic"},
		{"dedicarme", "dedic"},
		{"dedicarnos", "dedic"},
		{"dedicaron", "dedic"},
		{"dedicarse", "dedic"},
		{"dedicarte", "dedicart"},
		{"dedico", "dedic"},
		{"dedicó", "dedic"},
		{"dedillo", "dedill"},
		{"dedique", "dediqu"},
		{"dediqué", "dediqu"},
		{"dedo", "ded"},
		{"dedos", "ded"},
		{"deducibles", "deduc"},
		{"deduje", "deduj"},
		{"dedúzcale", "deduzcal"},
		{"deduzco", "deduzc"},
		{"dee", "dee"},
		{"defecto", "defect"},
		{"defectos", "defect"},
		{"defender", "defend"},
		{"defenderá", "defend"},
		{"defenderán", "defend"},
		{"defenderse", "defend"},
		{"defendía", "defend"},
		{"defendida", "defend"},
		{"defendido", "defend"},
		{"defendiendo", "defend"},
		{"defendió", "defend"},
		{"defenestradas", "defenestr"},
		{"defeños", "defeñ"},
		{"defensa", "defens"},
		{"defensas", "defens"},
		{"defensiva", "defens"},
		{"defensivas", "defens"},
		{"defensivo", "defens"},
		{"defensor", "defensor"},
		{"defensora", "defensor"},
		{"defensores", "defensor"},
		{"defensorías", "defensor"},
		{"deferencia", "deferent"},
		{"deficiencia", "deficient"},
		{"deficiencias", "deficient"},
		{"deficiente", "deficient"},
		{"deficientes", "deficient"},
		{"deficit", "deficit"},
		{"déficit", "deficit"},
		{"defienda", "defiend"},
		{"defiende", "defiend"},
		{"defienden", "defiend"},
		{"defina", "defin"},
		{"define", "defin"},
		{"definen", "defin"},
		{"definía", "defin"},
		{"definición", "definicion"},
		{"definiciones", "definicion"},
		{"definida", "defin"},
		{"definidas", "defin"},
		{"definido", "defin"},
		{"definidos", "defin"},
		{"definiendo", "defin"},
		{"definió", "defin"},
		{"definir", "defin"},
		{"definirá", "defin"},
		{"definirán", "defin"},
		{"definiría", "defin"},
		{"definirse", "defin"},
		{"definitiva", "definit"},
		{"definitivamente", "definit"},
		{"definitivas", "definit"},
		{"definitivo", "definit"},
		{"definitivos", "definit"},
		{"deforma", "deform"},
		{"deformaciones", "deform"},
		{"defraudador", "defraud"},
		{"defraudadores", "defraud"},
		{"defraudados", "defraud"},
		{"defraudando", "defraud"},
		{"defraudar", "defraud"},
		{"defraudaré", "defraud"},
		{"defraudarlos", "defraud"},
		{"degollado", "degoll"},
		{"degradación", "degrad"},
		{"degradado", "degrad"},
		{"degradando", "degrad"},
		{"degüello", "degüell"},
		{"dehesa", "dehes"},
		{"deidad", "deid"},
		{"deidades", "deidad"},
		{"deificaron", "deific"},
		{"déis", "deis"},
		{"deja", "dej"},
		{"dejaba", "dej"},
		{"dejaban", "dej"},
		{"dejada", "dej"},
		{"dejadas", "dej"},
		{"dejado", "dej"},
		{"dejados", "dej"},
		{"déjame", "dejam"},
		{"dejamos", "dej"},
		{"dejan", "dej"},
		{"dejando", "dej"},
		{"dejándolas", "dej"},
		{"dejar", "dej"},
		{"dejara", "dej"},
		{"dejará", "dej"},
		{"dejaran", "dej"},
		{"dejarán", "dej"},
		{"dejaría", "dej"},
		{"dejarla", "dej"},
		{"dejarlo", "dej"},
		{"dejarlos", "dej"},
		{"dejarme", "dej"},
		{"dejaron", "dej"},
		{"dejarse", "dej"},
		{"deje", "dej"},
		{"déjeme", "dejem"},
		{"dejemos", "dej"},
		{"dejen", "dej"},
		{"dejó", "dej"},
		{"del", "del"},
		{"delano", "delan"},
		{"delante", "delant"},
		{"delantera", "delanter"},
		{"delanteras", "delanter"},
		{"delantero", "delanter"},
		{"delanteros", "delanter"},
		{"delatan", "delat"},
		{"delató", "delat"},
		{"delaware", "delawar"},
		{"dele", "del"},
		{"déle", "del"},
		{"delega", "deleg"},
		{"delegación", "deleg"},
		{"delegaciones", "deleg"},
		{"delegada", "deleg"},
		{"delegado", "deleg"},
		{"delegados", "deleg"},
		{"deleitan", "deleit"},
		{"deleitar", "deleit"},
		{"deleite", "deleit"},
		{"delfines", "delfin"},
		{"delgada", "delg"},
		{"delgado", "delg"},
		{"delgados", "delg"},
		{"delia", "deli"},
		{"deliberaciones", "deliber"},
		{"deliberada", "deliber"},
		{"deliberadamente", "deliber"},
		{"deliberado", "deliber"},
		{"delicada", "delic"},
		{"delicadas", "delic"},
		{"delicado", "delic"},
		{"delicias", "delici"},
		{"delictivos", "delict"},
		{"delimitada", "delimit"},
		{"delimitar", "delimit"},
		{"delimitó", "delimit"},
		{"delincuencia", "delincuent"},
		{"delincuente", "delincuent"},
		{"delincuentes", "delincuent"},
		{"delinea", "deline"},
		{"delinquen", "delinqu"},
		{"delinquir", "delinqu"},
		{"deliquio", "deliqui"},
		{"delirante", "delir"},
		{"delirio", "deliri"},
		{"delito", "delit"},
		{"delitos", "delit"},
		{"deliverpool", "deliverpool"},
		{"dell", "dell"},
		{"delon", "delon"},
		{"delors", "delors"},
		{"delsur", "delsur"},
		{"delta", "delt"},
		{"demagogia", "demagogi"},
		{"demagógicamente", "demagog"},
		{"demagógico", "demagog"},
		{"demagógicos", "demagog"},
		{"demanda", "demand"},
		{"demandaba", "demand"},
		{"demandado", "demand"},
		{"demandan", "demand"},
		{"demandando", "demand"},
		{"demandante", "demand"},
		{"demandantes", "demand"},
		{"demandar", "demand"},
		{"demandará", "demand"},
		{"demandaron", "demand"},
		{"demandas", "demand"},
		{"demandé", "demand"},
		{"demandó", "demand"},
		{"demarcación", "demarc"},
		{"demás", "demas"},
		{"demasía", "demas"},
		{"demasiada", "demasi"},
		{"demasiadas", "demasi"},
		{"demasiado", "demasi"},
		{"demasiados", "demasi"},
		{"déme", "dem"},
		{"demencia", "demenci"},
		{"demencial", "demencial"},
		{"deming", "deming"},
		{"democracia", "democraci"},
		{"democracias", "democraci"},
		{"demócrata", "democrat"},
		{"demócratas", "democrat"},
		{"democrática", "democrat"},
		{"democráticamente", "democrat"},
		{"democráticas", "democrat"},
		{"democrático", "democrat"},
		{"democráticos", "democrat"},
		{"democratización", "democratiz"},
		{"democratizador", "democratiz"},
		{"democratizante", "democratiz"},
		{"democratizar", "democratiz"},
		{"democratizarla", "democratiz"},
		{"democristiano", "democristian"},
		{"demográfica", "demograf"},
		{"demográficas", "demograf"},
		{"demoníaca", "demoniac"},
		{"demonio", "demoni"},
		{"demonios", "demoni"},
		{"demora", "demor"},
		{"demorados", "demor"},
		{"demorar", "demor"},
		{"demostración", "demostr"},
		{"demostraciones", "demostr"},
		{"demostrado", "demostr"},
		{"demostrando", "demostr"},
		{"demostrar", "demostr"},
		{"demostrará", "demostr"},
		{"demostrarle", "demostr"},
		{"demostrarnos", "demostr"},
		{"demostraron", "demostr"},
		{"demostremos", "demostr"},
		{"demostró", "demostr"},
		{"demuestra", "demuestr"},
		{"demuestran", "demuestr"},
		{"demuestre", "demuestr"},
		{"demuestren", "demuestr"},
		{"den", "den"},
		{"denny", "denny"},
		{"denomina", "denomin"},
		{"denominada", "denomin"},
		{"denominadas", "denomin"},
		{"denominado", "denomin"},
		{"denominados", "denomin"},
		{"denostada", "denost"},
		{"denotan", "denot"},
		{"denotar", "denot"},
		{"denotaron", "denot"},
		{"denotó", "denot"},
		{"densa", "dens"},
		{"dense", "dens"},
		{"densidad", "densid"},
		{"densos", "dens"},
		{"dental", "dental"},
		{"dentales", "dental"},
		{"dentelladas", "dentell"},
		{"dentista", "dentist"},
		{"dentro", "dentr"},
		{"denuncia", "denunci"},
		{"denunciada", "denunci"},
		{"denunciado", "denunci"},
		{"denunciados", "denunci"},
		{"denuncian", "denunci"},
		{"denunciante", "denunci"},
		{"denunciantes", "denunci"},
		{"denunciar", "denunci"},
		{"denunciará", "denunci"},
		{"denunciarán", "denunci"},
		{"denunciaron", "denunci"},
		{"denuncias", "denunci"},
		{"denuncie", "denunci"},
		{"denuncié", "denunci"},
		{"denúncielo", "denunciel"},
		{"denunció", "denunc"},
		{"denzel", "denzel"},
		{"deparado", "depar"},
		{"deparará", "depar"},
		{"departamentales", "departamental"},
		{"departamento", "departament"},
		{"departamentos", "departament"},
		{"departió", "depart"},
		{"depende", "depend"},
		{"dependen", "depend"},
		{"dependencia", "dependent"},
		{"dependencias", "dependent"},
		{"depender", "depend"},
		{"dependerá", "depend"},
		{"dependes", "depend"},
		{"dependía", "depend"},
		{"dependiendo", "depend"},
		{"dependiente", "dependient"},
		{"dependientes", "dependient"},
		{"deplazamiento", "deplaz"},
		{"deportaciones", "deport"},
		{"deportado", "deport"},
		{"deporte", "deport"},
		{"deportes", "deport"},
		{"deportista", "deport"},
		{"deportistas", "deport"},
		{"deportiva", "deport"},
		{"deportivas", "deport"},
		{"deportivo", "deport"},
		{"deportivos", "deport"},
		{"deposita", "deposit"},
		{"depositada", "deposit"},
		{"depositado", "deposit"},
		{"depositados", "deposit"},
		{"depositan", "deposit"},
		{"depositantes", "deposit"},
		{"depositar", "deposit"},
		{"depositario", "depositari"},
		{"depositarse", "deposit"},
		{"deposite", "deposit"},
		{"deposítele", "depositel"},
		{"depósito", "deposit"},
		{"depósitos", "deposit"},
		{"depredadora", "depred"},
		{"depredadores", "depred"},
		{"depresión", "depresion"},
		{"depresiones", "depresion"},
		{"deprimente", "depriment"},
		{"deprimido", "deprim"},
		{"deprimidos", "deprim"},
		{"depuracion", "depuracion"},
		{"depuración", "depur"},
		{"depuradísima", "depuradisim"},
		{"depurado", "depur"},
		{"depurar", "depur"},
		{"der", "der"},
		{"derecha", "derech"},
		{"derechas", "derech"},
		{"derechazo", "derechaz"},
		{"derechazos", "derechaz"},
		{"derecho", "derech"},
		{"derechohabientes", "derechohabient"},
		{"derechos", "derech"},
		{"deregios", "deregi"},
		{"derek", "derek"},
		{"deri", "deri"},
		{"deriva", "deriv"},
		{"derivación", "deriv"},
		{"derivadas", "deriv"},
		{"derivado", "deriv"},
		{"derivados", "deriv"},
		{"derivan", "deriv"},
		{"derivar", "deriv"},
		{"derivará", "deriv"},
		{"derivaron", "deriv"},
		{"derivó", "deriv"},
		{"derma", "derm"},
		{"dermatitis", "dermatitis"},
		{"dermatología", "dermatolog"},
		{"dermatológica", "dermatolog"},
		{"derrama", "derram"},
		{"derramado", "derram"},
		{"derramamiento", "derram"},
		{"derraman", "derram"},
		{"derramar", "derram"},
		{"derrame", "derram"},
		{"derrames", "derram"},
		{"derrapante", "derrap"},
		{"derrapar", "derrap"},
		{"derrape", "derrap"},
		{"derredor", "derredor"},
		{"derretirse", "derret"},
		{"derriba", "derrib"},
		{"derribado", "derrib"},
		{"derribar", "derrib"},
		{"derribó", "derrib"},
		{"derrochando", "derroch"},
		{"derroche", "derroch"},
		{"derrota", "derrot"},
		{"derrotado", "derrot"},
		{"derrotados", "derrot"},
		{"derrotando", "derrot"},
		{"derrotar", "derrot"},
		{"derrotarlo", "derrot"},
		{"derrotaron", "derrot"},
		{"derrotas", "derrot"},
		{"derrotero", "derroter"},
		{"derrotó", "derrot"},
		{"derrumban", "derrumb"},
		{"derrumbe", "derrumb"},
		{"derrumbes", "derrumb"},
		{"desaceleración", "desaceler"},
		{"desaciertos", "desaciert"},
		{"desacredita", "desacredit"},
		{"desacreditados", "desacredit"},
		{"desactivar", "desactiv"},
		{"desactivaron", "desactiv"},
		{"desacuerdo", "desacuerd"},
		{"desacuerdos", "desacuerd"},
		{"desafiar", "desafi"},
		{"desafíe", "desafi"},
		{"desafinada", "desafin"},
		{"desafío", "desafi"},
		{"desafió", "desaf"},
		{"desafíos", "desafi"},
		{"desaforados", "desafor"},
		{"desafortunadamente", "desafortun"},
		{"desafortunado", "desafortun"},
		{"desafortunados", "desafortun"},
		{"desagradable", "desagrad"},
		{"desagradables", "desagrad"},
		{"desagüe", "desagü"},
		{"desahogar", "desahog"},
		{"desahogo", "desahog"},
		{"desahogos", "desahog"},
		{"desahucio", "desahuci"},
		{"desairado", "desair"},
		{"desaire", "desair"},
		{"desalentado", "desalent"},
		{"desalentadores", "desalent"},
		{"desalentarán", "desalent"},
		{"desalentó", "desalent"},
		{"desaliento", "desalient"},
		{"desalientos", "desalient"},
		{"desalojados", "desaloj"},
		{"desalojar", "desaloj"},
		{"desalojará", "desaloj"},
		{"desalojaran", "desaloj"},
		{"desalojaron", "desaloj"},
		{"desalojo", "desaloj"},
		{"desalojó", "desaloj"},
		{"desalojos", "desaloj"},
		{"desamor", "desamor"},
		{"desamparados", "desampar"},
		{"desangraba", "desangr"},
		{"desangrante", "desangr"},
		{"desanima", "desanim"},
		{"desanimada", "desanim"},
		{"desánimo", "desanim"},
		{"desaparecen", "desaparec"},
		{"desaparecer", "desaparec"},
		{"desaparecerá", "desaparec"},
		{"desaparecerán", "desaparec"},
		{"desaparecidas", "desaparec"},
		{"desaparecido", "desaparec"},
		{"desaparecidos", "desaparec"},
		{"desaparecieron", "desaparec"},
		{"desapareció", "desaparec"},
		{"desaparezca", "desaparezc"},
		{"desaparezcan", "desaparezc"},
		{"desaparición", "desaparicion"},
		{"desapariciones", "desaparicion"},
		{"desapercibida", "desapercib"},
		{"desaprobación", "desaprob"},
		{"desaprovechando", "desaprovech"},
		{"desaprovechar", "desaprovech"},
		{"desaprovechó", "desaprovech"},
		{"desaprueba", "desaprueb"},
		{"desarme", "des"},
		{"desarrolla", "desarroll"},
		{"desarrollada", "desarroll"},
		{"desarrolladas", "desarroll"},
		{"desarrollado", "desarroll"},
		{"desarrolladores", "desarroll"},
		{"desarrollados", "desarroll"},
		{"desarrollamos", "desarroll"},
		{"desarrollan", "desarroll"},
		{"desarrollando", "desarroll"},
		{"desarrollar", "desarroll"},
		{"desarrollará", "desarroll"},
		{"desarrollaría", "desarroll"},
		{"desarrollarnos", "desarroll"},
		{"desarrollaron", "desarroll"},
		{"desarrollarse", "desarroll"},
		{"desarrolle", "desarroll"},
		{"desarrollen", "desarroll"},
		{"desarrollo", "desarroll"},
		{"desarrolló", "desarroll"},
		{"desastres", "desastr"},
		{"desata", "desat"},
		{"desatado", "desat"},
		{"desatan", "desat"},
		{"desatar", "desat"},
		{"desatención", "desatencion"},
		{"desatinos", "desatin"},
		{"desató", "desat"},
		{"desayunar", "desayun"},
		{"desayuno", "desayun"},
		{"desayunos", "desayun"},
		{"desazolvar", "desazolv"},
		{"desbalancean", "desbalanc"},
		{"desbancarlo", "desbanc"},
		{"desbaratado", "desbarat"},
		{"desbordado", "desbord"},
		{"desbordados", "desbord"},
		{"desbordamiento", "desbord"},
		{"desbordando", "desbord"},
		{"desbordará", "desbord"},
		{"desbordes", "desbord"},
		{"desbordó", "desbord"},
		{"desburocratizar", "desburocratiz"},
		{"descabellada", "descabell"},
		{"descabelladas", "descabell"},
		{"descabello", "descabell"},
		{"descalabro", "descalabr"},
		{"descalificada", "descalific"},
		{"descalificado", "descalific"},
		{"descalzos", "descalz"},
		{"descansaban", "descans"},
		{"descansan", "descans"},
		{"descansando", "descans"},
		{"descansar", "descans"},
		{"descansarán", "descans"},
		{"descanso", "descans"},
		{"descansó", "descans"},
		{"descansos", "descans"},
		{"descapitaliza", "descapitaliz"},
		{"descarga", "descarg"},
		{"descargan", "descarg"},
		{"descargar", "descarg"},
		{"descargas", "descarg"},
		{"descargo", "descarg"},
		{"descargue", "descarg"},
		{"descarnado", "descarn"},
		{"descarrilamiento", "descarril"},
		{"descarriló", "descarril"},
		{"descarta", "descart"},
		{"descartaba", "descart"},
		{"descartado", "descart"},
		{"descartan", "descart"},
		{"descartando", "descart"},
		{"descartaron", "descart"},
		{"descartó", "descart"},
		{"descastados", "descast"},
		{"descendencia", "descendent"},
		{"descender", "descend"},
		{"descenderá", "descend"},
		{"descendientes", "descendient"},
		{"descendió", "descend"},
		{"descenso", "descens"},
		{"descensos", "descens"},
		{"descentralización", "descentraliz"},
		{"descentralizarse", "descentraliz"},
		{"desciende", "desciend"},
		{"descienden", "desciend"},
		{"descomponía", "descompon"},
		{"descomposición", "descomposicion"},
		{"descomposturas", "descompostur"},
		{"descompuesto", "descompuest"},
		{"descomunal", "descomunal"},
		{"desconcertado", "desconcert"},
		{"desconcertados", "desconcert"},
		{"desconcertante", "desconcert"},
		{"desconcierta", "desconciert"},
		{"desconcierto", "desconciert"},
		{"desconfianza", "desconfi"},
		{"desconfiara", "desconfi"},
		{"descongelación", "descongel"},
		{"descongelar", "descongel"},
		{"descongestionar", "descongestion"},
		{"desconoce", "desconoc"},
		{"desconocen", "desconoc"},
		{"desconocer", "desconoc"},
		{"desconocerlos", "desconoc"},
		{"desconocía", "desconoc"},
		{"desconocida", "desconoc"},
		{"desconocidas", "desconoc"},
		{"desconocido", "desconoc"},
		{"desconocidos", "desconoc"},
		{"desconocieron", "desconoc"},
		{"desconocimiento", "desconoc"},
		{"desconoció", "desconoc"},
		{"desconozco", "desconozc"},
		{"descontando", "descont"},
		{"descontar", "descont"},
		{"descontentas", "descontent"},
		{"descontento", "descontent"},
		{"descontentos", "descontent"},
		{"descontrol", "descontrol"},
		{"descrédito", "descredit"},
		{"descreído", "descreid"},
		{"describe", "describ"},
		{"describen", "describ"},
		{"describió", "describ"},
		{"describir", "describ"},
		{"descripción", "descripcion"},
		{"descriptores", "descriptor"},
		{"descrita", "descrit"},
		{"descrito", "descrit"},
		{"descritos", "descrit"},
		{"descubierta", "descubiert"},
		{"descubiertas", "descubiert"},
		{"descubierto", "descubiert"},
		{"descubiertos", "descubiert"},
		{"descubran", "descubr"},
		{"descubre", "descubr"},
		{"descubridor", "descubridor"},
		{"descubriendo", "descubr"},
		{"descubrieron", "descubr"},
		{"descubrimiento", "descubr"},
		{"descubrimientos", "descubr"},
		{"descubrió", "descubr"},
		{"descubrir", "descubr"},
		{"descuento", "descuent"},
		{"descuentos", "descuent"},
		{"descuidado", "descuid"},
		{"descuidan", "descuid"},
		{"descuide", "descuid"},
		{"descuido", "descu"},
		{"descuidos", "descu"},
		{"desde", "desd"},
		{"desdecir", "desdec"},
		{"desdeñar", "desdeñ"},
		{"desdeño", "desdeñ"},
		{"desdice", "desdic"},
		{"desdichada", "desdich"},
		{"desdichado", "desdich"},
		{"dése", "des"},
		{"desea", "dese"},
		{"deseaban", "des"},
		{"deseable", "deseabl"},
		{"deseada", "des"},
		{"deseamos", "des"},
		{"desean", "des"},
		{"deseando", "des"},
		{"deseándolo", "des"},
		{"desearle", "des"},
		{"desearon", "des"},
		{"deseas", "des"},
		{"desecha", "desech"},
		{"desechaban", "desech"},
		{"desechar", "desech"},
		{"desecharon", "desech"},
		{"desecho", "desech"},
		{"desechó", "desech"},
		{"desechos", "desech"},
		{"desee", "dese"},
		{"deseen", "des"},
		{"desembarcar", "desembarc"},
		{"desembarque", "desembarqu"},
		{"desembocar", "desemboc"},
		{"desempaña", "desempañ"},
		{"desempate", "desempat"},
		{"desempeña", "desempeñ"},
		{"desempeñado", "desempeñ"},
		{"desempeñan", "desempeñ"},
		{"desempeñando", "desempeñ"},
		{"desempeñar", "desempeñ"},
		{"desempeñarse", "desempeñ"},
		{"desempeño", "desempeñ"},
		{"desempleado", "desempl"},
		{"desempleados", "desempl"},
		{"desempleo", "desemple"},
		{"desencajado", "desencaj"},
		{"desencantados", "desencant"},
		{"desencanto", "desencant"},
		{"desenfreno", "desenfren"},
		{"desengaña", "desengañ"},
		{"desenmascaran", "desenmasc"},
		{"desentendido", "desentend"},
		{"desenterrados", "desenterr"},
		{"desentonado", "desenton"},
		{"desentonados", "desenton"},
		{"desenvolverse", "desenvolv"},
		{"desenvolvimiento", "desenvolv"},
		{"desenvuelve", "desenvuelv"},
		{"deseo", "dese"},
		{"deseos", "dese"},
		{"desequilibrada", "desequilibr"},
		{"desequilibrado", "desequilibr"},
		{"desequilibrio", "desequilibri"},
		{"desequilibrios", "desequilibri"},
		{"deserciones", "desercion"},
		{"desertar", "desert"},
		{"desesperación", "desesper"},
		{"desesperada", "desesper"},
		{"desesperadamente", "desesper"},
		{"desesperados", "desesper"},
		{"desesperan", "desesper"},
		{"desestabilizar", "desestabiliz"},
		{"desestímulos", "desestimul"},
		{"desfasan", "desfas"},
		{"desfavorable", "desfavor"},
		{"desfiguró", "desfigur"},
		{"desfiguros", "desfigur"},
		{"desfilan", "desfil"},
		{"desfile", "desfil"},
		{"desfiles", "desfil"},
		{"desfondaría", "desfond"},
		{"desgañitan", "desgañit"},
		{"desgarra", "desgarr"},
		{"desgarradoramente", "desgarrador"},
		{"desgarradoras", "desgarr"},
		{"desgarre", "desgarr"},
		{"desgarros", "desgarr"},
		{"desgastante", "desgast"},
		{"desgastar", "desgast"},
		{"desgaste", "desg"},
		{"desglosado", "desgl"},
		{"desglosados", "desgl"},
		{"desgracia", "desgraci"},
		{"desgraciadamente", "desgraci"},
		{"desgracias", "desgraci"},
		{"deshacer", "deshac"},
		{"deshacerse", "deshac"},
		{"desharía", "desh"},
		{"deshilachar", "deshilach"},
		{"deshilvanando", "deshilvan"},
		{"deshizo", "deshiz"},
		{"deshonestidad", "deshonest"},
		{"deshonesto", "deshonest"},
		{"deshumanizó", "deshumaniz"},
		{"desierto", "desiert"},
		{"desiertos", "desiert"},
		{"designación", "design"},
		{"designaciones", "design"},
		{"designada", "design"},
		{"designadas", "design"},
		{"designado", "design"},
		{"designador", "design"},
		{"designados", "design"},
		{"designar", "design"},
		{"designe", "design"},
		{"designio", "designi"},
		{"designó", "design"},
		{"desigualdad", "desiguald"},
		{"desigualdades", "desigualdad"},
		{"desincorporación", "desincorpor"},
		{"desincorporó", "desincorpor"},
		{"desindigenización", "desindigeniz"},
		{"desinformación", "desinform"},
		{"desinformados", "desinform"},
		{"desintegración", "desintegr"},
		{"desintegrar", "desintegr"},
		{"desintegrará", "desintegr"},
		{"desintegrarse", "desintegr"},
		{"desinterés", "desinteres"},
		{"desinterviniendo", "desintervin"},
		{"desiste", "des"},
		{"desistieran", "desist"},
		{"desistieron", "desist"},
		{"desistió", "desist"},
		{"desistir", "desist"},
		{"desistirse", "desist"},
		{"deslaves", "deslav"},
		{"desleal", "desleal"},
		{"desleales", "desleal"},
		{"deslices", "deslic"},
		{"desligarme", "deslig"},
		{"deslindar", "deslind"},
		{"deslindarse", "deslind"},
		{"deslinden", "deslind"},
		{"deslizando", "desliz"},
		{"deslizarse", "desliz"},
		{"deslizó", "desliz"},
		{"deslucido", "desluc"},
		{"deslumbra", "deslumbr"},
		{"desmantelamiento", "desmantel"},
		{"desmarque", "desmarqu"},
		{"desmembrada", "desmembr"},
		{"desmembró", "desmembr"},
		{"desmerece", "desmerec"},
		{"desmintió", "desmint"},
		{"desmitificado", "desmitific"},
		{"desmontadas", "desmont"},
		{"desmontarlo", "desmont"},
		{"desmoralizar", "desmoraliz"},
		{"desmoralizarnos", "desmoraliz"},
		{"desmoronando", "desmoron"},
		{"desmotivada", "desmotiv"},
		{"desnaquización", "desnaquiz"},
		{"desnivel", "desnivel"},
		{"desnudarla", "desnud"},
		{"desnudaron", "desnud"},
		{"desnudos", "desnud"},
		{"desnutrición", "desnutricion"},
		{"desobedece", "desobedec"},
		{"desocupación", "desocup"},
		{"desocupar", "desocup"},
		{"desolación", "desol"},
		{"desolador", "desol"},
		{"desorden", "desord"},
		{"desordenado", "desorden"},
		{"desórdenes", "desorden"},
		{"desorganizados", "desorganiz"},
		{"desorienta", "desorient"},
		{"desorientación", "desorient"},
		{"desorientada", "desorient"},
		{"desorientados", "desorient"},
		{"despachar", "despach"},
		{"despacho", "despach"},
		{"despacio", "despaci"},
		{"despacito", "despacit"},
		{"desparramaron", "desparram"},
		{"despavorido", "despavor"},
		{"despedazadas", "despedaz"},
		{"despedía", "desped"},
		{"despedida", "desped"},
		{"despedidas", "desped"},
		{"despedido", "desped"},
		{"despedidos", "desped"},
		{"despedir", "desped"},
		{"despedirlas", "desped"},
		{"despedirlos", "desped"},
		{"despedirse", "desped"},
		{"despegara", "despeg"},
		{"despegue", "despeg"},
		{"despeinada", "despein"},
		{"despeinarse", "despein"},
		{"despejando", "despej"},
		{"despejar", "despej"},
		{"despensa", "despens"},
		{"despensas", "despens"},
		{"desperdiciar", "desperdici"},
		{"desperdicio", "desperdici"},
		{"despertado", "despert"},
		{"despertando", "despert"},
		{"despertar", "despert"},
		{"despertaron", "despert"},
		{"despertó", "despert"},
		{"despidan", "despid"},
		{"despide", "despid"},
		{"despidiendo", "despid"},
		{"despidió", "despid"},
		{"despido", "desp"},
		{"despidos", "desp"},
		{"despierta", "despiert"},
		{"despiertan", "despiert"},
		{"despierte", "despiert"},
		{"desplace", "desplac"},
		{"desplazaba", "desplaz"},
		{"desplazado", "desplaz"},
		{"desplazados", "desplaz"},
		{"desplazamiento", "desplaz"},
		{"desplazamientos", "desplaz"},
		{"desplazan", "desplaz"},
		{"desplazando", "desplaz"},
		{"desplazándolo", "desplaz"},
		{"desplazar", "desplaz"},
		{"desplazarla", "desplaz"},
		{"desplegado", "despleg"},
		{"desplegará", "despleg"},
		{"desplegarán", "despleg"},
		{"despliegue", "desplieg"},
		{"desplieguen", "desplieg"},
		{"despoblado", "despobl"},
		{"despojaba", "despoj"},
		{"despojado", "despoj"},
		{"despojando", "despoj"},
		{"despojar", "despoj"},
		{"despojarlo", "despoj"},
		{"despojaron", "despoj"},
		{"despojo", "despoj"},
		{"despojó", "despoj"},
		{"despotismo", "despot"},
		{"despreciable", "despreci"},
		{"despreciar", "despreci"},
		{"despreciarse", "despreci"},
		{"desprende", "desprend"},
		{"desprenderse", "desprend"},
		{"desprendía", "desprend"},
		{"desprendió", "desprend"},
		{"desprestigiando", "desprestigi"},
		{"desprestigio", "desprestigi"},
		{"desproporcionadas", "desproporcion"},
		{"desproporcionado", "desproporcion"},
		{"despropósito", "desproposit"},
		{"desprotegerán", "desproteg"},
		{"desprovisto", "desprovist"},
		{"después", "despues"},
		{"desquita", "desquit"},
		{"desregulación", "desregul"},
		{"desregulado", "desregul"},
		{"desregulando", "desregul"},
		{"desrregulación", "desrregul"},
		{"destaca", "destac"},
		{"destacaba", "destac"},
		{"destacada", "destac"},
		{"destacadas", "destac"},
		{"destacado", "destac"},
		{"destacados", "destac"},
		{"destacamentados", "destacament"},
		{"destacan", "destac"},
		{"destacando", "destac"},
		{"destacar", "destac"},
		{"destacaron", "destac"},
		{"destacó", "destac"},
		{"destanteados", "destant"},
		{"destapan", "destap"},
		{"destaparon", "destap"},
		{"destape", "destap"},
		{"destapes", "destap"},
		{"destaque", "destaqu"},
		{"destemplada", "destempl"},
		{"destemplado", "destempl"},
		{"desterrarlas", "desterr"},
		{"destinaban", "destin"},
		{"destinada", "destin"},
		{"destinadas", "destin"},
		{"destinado", "destin"},
		{"destinados", "destin"},
		{"destinan", "destin"},
		{"destinar", "destin"},
		{"destinarán", "destin"},
		{"destinarían", "destin"},
		{"destinarlos", "destin"},
		{"destinaron", "destin"},
		{"destinatario", "destinatari"},
		{"destinen", "destin"},
		{"destino", "destin"},
		{"destinó", "destin"},
		{"destinos", "destin"},
		{"destitución", "destitu"},
		{"destreza", "destrez"},
		{"destrozado", "destroz"},
		{"destrozados", "destroz"},
		{"destrucción", "destruccion"},
		{"destructivo", "destruct"},
		{"destruida", "destru"},
		{"destruir", "destru"},
		{"destruirá", "destru"},
		{"destruirlo", "destru"},
		{"destruye", "destru"},
		{"destruyeron", "destru"},
		{"destruyó", "destru"},
		{"desulfuración", "desulfur"},
		{"desvanecen", "desvanec"},
		{"desvanecieran", "desvanec"},
		{"desveladas", "desvel"},
		{"desvelos", "desvel"},
		{"desventaja", "desventaj"},
		{"desventajosa", "desventaj"},
		{"desventajosas", "desventaj"},
		{"desviaciones", "desviacion"},
		{"desviadas", "desvi"},
		{"desviado", "desvi"},
		{"desviándose", "desvi"},
		{"desviar", "desvi"},
		{"desviaremos", "desvi"},
		{"desviaría", "desvi"},
		{"desviarse", "desvi"},
		{"desvíe", "desvi"},
		{"desvinculaba", "desvincul"},
		{"desvincular", "desvincul"},
		{"desvió", "desv"},
		{"desvirtúa", "desvirtu"},
		{"desvirtuación", "desvirtu"},
		{"desvirtuar", "desvirtu"},
		{"detalla", "detall"},
		{"detalladamente", "detall"},
		{"detalladas", "detall"},
		{"detallado", "detall"},
		{"detalle", "detall"},
		{"detalles", "detall"},
		{"detallista", "detall"},
		{"detallistas", "detall"},
		{"detallitos", "detallit"},
		{"detalló", "detall"},
		{"detecciones", "deteccion"},
		{"detecnología", "detecnolog"},
		{"detecta", "detect"},
		{"detectaba", "detect"},
		{"detectada", "detect"},
		{"detectado", "detect"},
		{"detectados", "detect"},
		{"detectan", "detect"},
		{"detectando", "detect"},
		{"detectar", "detect"},
		{"detectara", "detect"},
		{"detectaron", "detect"},
		{"detectó", "detect"},
		{"detención", "detencion"},
		{"detenciones", "detencion"},
		{"detendrá", "detendr"},
		{"detener", "deten"},
		{"detenerlo", "deten"},
		{"detenerme", "deten"},
		{"detenerse", "deten"},
		{"detengan", "deteng"},
		{"detenida", "deten"},
		{"detenidas", "deten"},
		{"detenido", "deten"},
		{"detenidos", "deten"},
		{"detenta", "detent"},
		{"detentando", "detent"},
		{"detergente", "detergent"},
		{"detergentes", "detergent"},
		{"deterioradas", "deterior"},
		{"deterioran", "deterior"},
		{"deterioro", "deterior"},
		{"determina", "determin"},
		{"determinación", "determin"},
		{"determinaciones", "determin"},
		{"determinada", "determin"},
		{"determinado", "determin"},
		{"determinados", "determin"},
		{"determinan", "determin"},
		{"determinante", "determin"},
		{"determinantemente", "determin"},
		{"determinantes", "determin"},
		{"determinar", "determin"},
		{"determinará", "determin"},
		{"determinaran", "determin"},
		{"determinaron", "determin"},
		{"determinarse", "determin"},
		{"determine", "determin"},
		{"determinó", "determin"},
		{"deterner", "detern"},
		{"detestan", "detest"},
		{"detiene", "detien"},
		{"detienen", "detien"},
		{"detonación", "deton"},
		{"detrás", "detras"},
		{"detuviera", "detuv"},
		{"detuvieran", "detuv"},
		{"detuvieron", "detuv"},
		{"detuvo", "detuv"},
		{"deuda", "deud"},
		{"deudas", "deud"},
		{"deuteronomio", "deuteronomi"},
		{"devaluación", "devalu"},
		{"devaluado", "devalu"},
		{"devalúan", "devalu"},
		{"devanándose", "devan"},
		{"devant", "devant"},
		{"devastada", "devast"},
		{"devastador", "devast"},
		{"develará", "devel"},
		{"development", "development"},
		{"devenido", "deven"},
		{"devenir", "deven"},
		{"devoción", "devocion"},
		{"devociones", "devocion"},
		{"devolución", "devolu"},
		{"devolver", "devolv"},
		{"devolvería", "devolv"},
		{"devolverle", "devolv"},
		{"devolvió", "devolv"},
		{"devuelto", "devuelt"},
		{"devueltos", "devuelt"},
		{"devuelvan", "devuelv"},
		{"devuelve", "devuelv"},
		{"df", "df"},
		{"dgo", "dgo"},
		{"dhl", "dhl"},
		{"di", "di"},
		{"dí", "di"},
		{"dia", "dia"},
		{"día", "dia"},
		{"diabetes", "diabet"},
		{"diablo", "diabl"},
		{"diablos", "diabl"},
		{"diadema", "diadem"},
		{"diagnosticamos", "diagnostic"},
		{"diagnosticar", "diagnostic"},
		{"diagnóstico", "diagnost"},
		{"diagnósticos", "diagnost"},
		{"diagrama", "diagram"},
		{"dialogado", "dialog"},
		{"dialogamos", "dialog"},
		{"dialogar", "dialog"},
		{"dialogaron", "dialog"},
		{"diálogo", "dialog"},
		{"diálogos", "dialog"},
		{"diamante", "diamant"},
		{"diámetro", "diametr"},
		{"diana", "dian"},
		{"diaria", "diari"},
		{"diariamente", "diari"},
		{"diarias", "diari"},
		{"diario", "diari"},
		{"diarios", "diari"},
		{"días", "dias"},
		{"díaz", "diaz"},
		{"dibujando", "dibuj"},
		{"dibujantes", "dibuj"},
		{"dibujo", "dibuj"},
		{"dibujó", "dibuj"},
		{"dibujos", "dibuj"},
		{"dicamex", "dicamex"},
		{"diccionario", "diccionari"},
		{"dice", "dic"},
		{"dicen", "dic"},
		{"dices", "dic"},
		{"dicha", "dich"},
		{"dichas", "dich"},
		{"dicho", "dich"},
		{"dichos", "dich"},
		{"diciembre", "diciembr"},
		{"diciendo", "dic"},
		{"diciéndome", "dic"},
		{"dickens", "dickens"},
		{"dicom", "dicom"},
		{"dicta", "dict"},
		{"dictada", "dict"},
		{"dictadas", "dict"},
		{"dictado", "dict"},
		{"dictador", "dictador"},
		{"dictadura", "dictadur"},
		{"dictaduras", "dictadur"},
		{"dictamen", "dictam"},
		{"dictaminaron", "dictamin"},
		{"dictaminó", "dictamin"},
		{"dictar", "dict"},
		{"dictarnos", "dict"},
		{"dictaron", "dict"},
		{"dictatorial", "dictatorial"},
		{"dictó", "dict"},
		{"didáctico", "didact"},
		{"diego", "dieg"},
		{"dientes", "dient"},
		{"diera", "dier"},
		{"dieran", "dier"},
		{"dieron", "dieron"},
		{"diesel", "diesel"},
		{"diestro", "diestr"},
		{"dieta", "diet"},
		{"diez", "diez"},
		{"díez", "diez"},
		{"diezmado", "diezm"},
		{"dif", "dif"},
		{"difamación", "difam"},
		{"difamador", "difam"},
		{"difame", "difam"},
		{"diferencia", "diferent"},
		{"diferenciado", "diferenci"},
		{"diferencial", "diferencial"},
		{"diferenciales", "diferencial"},
		{"diferenciando", "diferenci"},
		{"diferenciar", "diferenci"},
		{"diferenciarse", "diferenci"},
		{"diferencias", "diferent"},
		{"diferendos", "diferend"},
		{"diferente", "diferent"},
		{"diferentes", "diferent"},
		{"diferida", "difer"},
		{"diferidos", "difer"},
		{"dificil", "dificil"},
		{"difícil", "dificil"},
		{"difíciles", "dificil"},
		{"dificilmente", "dificil"},
		{"difícilmente", "dificil"},
		{"dificultaba", "dificult"},
		{"dificultad", "dificult"},
		{"dificultades", "dificultad"},
		{"dificultado", "dificult"},
		{"dificultan", "dificult"},
		{"dificultar", "dificult"},
		{"dificultó", "dificult"},
		{"difieren", "difier"},
		{"difunda", "difund"},
		{"difunde", "difund"},
		{"difunden", "difund"},
		{"difundida", "difund"},
		{"difundido", "difund"},
		{"difundir", "difund"},
		{"difundirá", "difund"},
		{"difundirlo", "difund"},
		{"difunto", "difunt"},
		{"difuntos", "difunt"},
		{"difusión", "difusion"},
		{"diga", "dig"},
		{"dígale", "digal"},
		{"dígales", "digal"},
		{"dígame", "digam"},
		{"digamos", "dig"},
		{"digan", "dig"},
		{"digerir", "diger"},
		{"digestión", "digestion"},
		{"digestivos", "digest"},
		{"digital", "digital"},
		{"digitales", "digital"},
		{"digitalización", "digitaliz"},
		{"dígito", "digit"},
		{"digna", "dign"},
		{"dignas", "dign"},
		{"dignidad", "dignid"},
		{"dignifica", "dignif"},
		{"dignificar", "dignific"},
		{"dignity", "dignity"},
		{"digno", "dign"},
		{"dignos", "dign"},
		{"dignosticar", "dignostic"},
		{"digo", "dig"},
		{"dihigo", "dihig"},
		{"dihlgo", "dihlg"},
		{"dije", "dij"},
		{"dijera", "dijer"},
		{"dijeran", "dijer"},
		{"dijeron", "dijeron"},
		{"dijes", "dij"},
		{"dijiste", "dij"},
		{"dijo", "dij"},
		{"dilatadas", "dilat"},
		{"dilatar", "dilat"},
		{"dilatorias", "dilatori"},
		{"díle", "dil"},
		{"dilema", "dilem"},
		{"diligencia", "diligent"},
		{"diligencias", "diligent"},
		{"diligente", "diligent"},
		{"dillo", "dill"},
		{"dilucidar", "dilucid"},
		{"diluido", "dilu"},
		{"diluir", "dilu"},
		{"dimas", "dim"},
		{"dimensión", "dimension"},
		{"dimensiones", "dimension"},
		{"dimino", "dimin"},
		{"diminutas", "diminut"},
		{"diminutos", "diminut"},
		{"dimisión", "dimision"},
		{"dimitir", "dimit"},
		{"dimitri", "dimitri"},
		{"dimos", "dim"},
		{"dina", "din"},
		{"dinamarca", "dinamarc"},
		{"dinámica", "dinam"},
		{"dinámicas", "dinam"},
		{"dinámico", "dinam"},
		{"dinámicos", "dinam"},
		{"dinamismo", "dinam"},
		{"dinamiteros", "dinamiter"},
		{"dinastía", "dinast"},
		{"dinero", "diner"},
		{"dineros", "diner"},
		{"dinorah", "dinorah"},
		{"dinosaurios", "dinosauri"},
		{"dio", "dio"},
		{"diócesis", "diocesis"},
		{"dior", "dior"},
		{"diorita", "diorit"},
		{"dios", "dios"},
		{"diosa", "dios"},
		{"dioses", "dios"},
		{"dióxido", "diox"},
		{"diplomacia", "diplomaci"},
		{"diplomado", "diplom"},
		{"diplomados", "diplom"},
		{"diplomática", "diplomat"},
		{"diplomáticas", "diplomat"},
		{"diplomático", "diplomat"},
		{"diplomáticos", "diplomat"},
		{"diputa", "diput"},
		{"diputación", "diput"},
		{"diputaciones", "diput"},
		{"diputada", "diput"},
		{"diputado", "diput"},
		{"diputados", "diput"},
		{"dirá", "dir"},
		{"dirán", "diran"},
		{"diré", "dir"},
		{"dirección", "direccion"},
		{"direccionales", "direccional"},
		{"direcciones", "direccion"},
		{"direct", "direct"},
		{"directa", "direct"},
		{"directamente", "direct"},
		{"directas", "direct"},
		{"directiva", "direct"},
		{"directivas", "direct"},
		{"directivo", "direct"},
		{"directivos", "direct"},
		{"directo", "direct"},
		{"director", "director"},
		{"directora", "director"},
		{"directores", "director"},
		{"directorio", "directori"},
		{"directorios", "directori"},
		{"directos", "direct"},
		{"diría", "dir"},
		{"dirige", "dirig"},
		{"dirigencia", "dirigent"},
		{"dirigente", "dirigent"},
		{"dirigentes", "dirigent"},
		{"dirigía", "dirig"},
		{"dirigían", "dirig"},
		{"dirigible", "dirig"},
		{"dirigida", "dirig"},
		{"dirigidas", "dirig"},
		{"dirigido", "dirig"},
		{"dirigidos", "dirig"},
		{"dirigiendo", "dirig"},
		{"dirigieron", "dirig"},
		{"dirigió", "dirig"},
		{"dirigir", "dirig"},
		{"dirigirá", "dirig"},
		{"dirigirla", "dirig"},
		{"dirigirme", "dirig"},
		{"dirigirse", "dirig"},
		{"dirija", "dirij"},
		{"dirimir", "dirim"},
		{"disc", "disc"},
		{"discapacidad", "discapac"},
		{"discapacidades", "discapac"},
		{"discapacitada", "discapacit"},
		{"discapacitadas", "discapacit"},
		{"discapacitado", "discapacit"},
		{"discapacitados", "discapacit"},
		{"disciplina", "disciplin"},
		{"disciplinada", "disciplin"},
		{"disciplinadas", "disciplin"},
		{"disciplinaria", "disciplinari"},
		{"disciplinario", "disciplinari"},
		{"disciplinas", "disciplin"},
		{"discípulos", "discipul"},
		{"disco", "disc"},
		{"discografía", "discograf"},
		{"discográfico", "discograf"},
		{"discos", "disc"},
		{"discoteca", "discotec"},
		{"discotecas", "discotec"},
		{"discrepancia", "discrep"},
		{"discreto", "discret"},
		{"discretos", "discret"},
		{"discriminación", "discrimin"},
		{"discs", "discs"},
		{"disculpa", "disculp"},
		{"disculpaba", "disculp"},
		{"disculpe", "disculp"},
		{"discurso", "discurs"},
		{"discursos", "discurs"},
		{"discusión", "discusion"},
		{"discusiones", "discusion"},
		{"discuta", "discut"},
		{"discutían", "discut"},
		{"discutido", "discut"},
		{"discutieron", "discut"},
		{"discutió", "discut"},
		{"discutir", "discut"},
		{"discutirse", "discut"},
		{"diseña", "diseñ"},
		{"diseñada", "diseñ"},
		{"diseñadas", "diseñ"},
		{"diseñado", "diseñ"},
		{"diseñador", "diseñ"},
		{"diseñadores", "diseñ"},
		{"diseñados", "diseñ"},
		{"diseñan", "diseñ"},
		{"diseñando", "diseñ"},
		{"diseñar", "diseñ"},
		{"diseñaron", "diseñ"},
		{"diseñe", "diseñ"},
		{"diseño", "diseñ"},
		{"diseñó", "diseñ"},
		{"diseños", "diseñ"},
		{"disertará", "disert"},
		{"disfrace", "disfrac"},
		{"disfraces", "disfrac"},
		{"disfrazada", "disfraz"},
		{"disfrazados", "disfraz"},
		{"disfrazar", "disfraz"},
		{"disfrazaron", "disfraz"},
		{"disfrazarse", "disfraz"},
		{"disfruta", "disfrut"},
		{"disfrutando", "disfrut"},
		{"disfrutar", "disfrut"},
		{"disfrutará", "disfrut"},
		{"disfrutaron", "disfrut"},
		{"disfrutarse", "disfrut"},
		{"disfrute", "disfrut"},
		{"disfrutó", "disfrut"},
		{"disgregulación", "disgregul"},
		{"disgusta", "disgust"},
		{"disgustado", "disgust"},
		{"disgustarle", "disgust"},
		{"disgusto", "disgust"},
		{"disidente", "disident"},
		{"disidentes", "disident"},
		{"disimular", "disimul"},
		{"disipar", "disip"},
		{"disipó", "disip"},
		{"diskete", "disket"},
		{"diskette", "diskett"},
		{"diskettes", "diskett"},
		{"disléxico", "dislex"},
		{"disminución", "disminu"},
		{"disminuidas", "disminu"},
		{"disminuido", "disminu"},
		{"disminuir", "disminu"},
		{"disminuiría", "disminu"},
		{"disminuye", "disminu"},
		{"disminuyendo", "disminu"},
		{"disminuyó", "disminu"},
		{"disney", "disney"},
		{"disolvió", "disolv"},
		{"disparaba", "dispar"},
		{"disparado", "dispar"},
		{"disparan", "disp"},
		{"disparando", "dispar"},
		{"disparar", "dispar"},
		{"dispararon", "dispar"},
		{"dispararse", "dispar"},
		{"disparate", "disparat"},
		{"disparates", "disparat"},
		{"disparo", "dispar"},
		{"disparó", "dispar"},
		{"disparos", "dispar"},
		{"dispendio", "dispendi"},
		{"dispendiosa", "dispendi"},
		{"dispensa", "dispens"},
		{"dispersan", "dispers"},
		{"dispersión", "dispersion"},
		{"disperso", "dispers"},
		{"dispersó", "dispers"},
		{"dispersos", "dispers"},
		{"displays", "displays"},
		{"displicencia", "displicent"},
		{"dispone", "dispon"},
		{"disponen", "dispon"},
		{"disponga", "dispong"},
		{"dispóngase", "dispong"},
		{"dispongo", "dispong"},
		{"disponía", "dispon"},
		{"disponibilidad", "disponibil"},
		{"disponible", "dispon"},
		{"disponibles", "dispon"},
		{"disposición", "disposicion"},
		{"disposiciones", "disposicion"},
		{"dispositivo", "disposit"},
		{"dispositivos", "disposit"},
		{"dispuesta", "dispuest"},
		{"dispuesto", "dispuest"},
		{"dispuestos", "dispuest"},
		{"dispusieron", "dispus"},
		{"dispuso", "dispus"},
		{"disputa", "disput"},
		{"disputadas", "disput"},
		{"disputado", "disput"},
		{"disputar", "disput"},
		{"disputará", "disput"},
		{"disputarán", "disput"},
		{"disputarían", "disput"},
		{"disputaron", "disput"},
		{"disputas", "disput"},
		{"disputó", "disput"},
		{"disquera", "disquer"},
		{"disqueras", "disquer"},
		{"distancia", "distanci"},
		{"distancias", "distanci"},
		{"distensión", "distension"},
		{"distinción", "distincion"},
		{"distinciones", "distincion"},
		{"distingos", "disting"},
		{"distinguen", "disting"},
		{"distinguida", "distingu"},
		{"distinguidas", "distingu"},
		{"distinguido", "distingu"},
		{"distinguidos", "distingu"},
		{"distinguimos", "distingu"},
		{"distinguir", "distingu"},
		{"distinta", "distint"},
		{"distintas", "distint"},
		{"distintivo", "distint"},
		{"distinto", "distint"},
		{"distintos", "distint"},
		{"distorsione", "distorsion"},
		{"distorsiones", "distorsion"},
		{"distraen", "distr"},
		{"distraer", "distr"},
		{"distraída", "distraid"},
		{"distrajo", "distraj"},
		{"distribución", "distribu"},
		{"distribuía", "distribu"},
		{"distribuidas", "distribu"},
		{"distribuido", "distribu"},
		{"distribuidor", "distribuidor"},
		{"distribuidora", "distribuidor"},
		{"distribuidoras", "distribuidor"},
		{"distribuidores", "distribuidor"},
		{"distribuidos", "distribu"},
		{"distribuimos", "distribu"},
		{"distribuir", "distribu"},
		{"distribuirán", "distribu"},
		{"distribuirla", "distribu"},
		{"distribuirse", "distribu"},
		{"distribution", "distribution"},
		{"distribuya", "distribu"},
		{"distribuye", "distribu"},
		{"distribuyen", "distribu"},
		{"distribuyendo", "distribu"},
		{"distribuyó", "distribu"},
		{"distritales", "distrital"},
		{"distrito", "distrit"},
		{"distritos", "distrit"},
		{"distrofia", "distrofi"},
		{"disturbios", "disturbi"},
		{"disyuntiva", "disyunt"},
		{"dit", "dit"},
		{"diva", "div"},
		{"diván", "divan"},
		{"divergencias", "divergent"},
		{"diversa", "divers"},
		{"diversas", "divers"},
		{"diversidad", "divers"},
		{"diversificación", "diversif"},
		{"diversificándose", "diversific"},
		{"diversificó", "diversific"},
		{"diversión", "diversion"},
		{"diverso", "divers"},
		{"diversos", "divers"},
		{"divertía", "divert"},
		{"divertida", "divert"},
		{"divertidas", "divert"},
		{"divertido", "divert"},
		{"divertir", "divert"},
		{"divertirnos", "divert"},
		{"divertirse", "divert"},
		{"divide", "divid"},
		{"dividendos", "dividend"},
		{"dividido", "divid"},
		{"divididos", "divid"},
		{"dividiéndose", "divid"},
		{"dividió", "divid"},
		{"dividir", "divid"},
		{"dividirnos", "divid"},
		{"divina", "divin"},
		{"divinas", "divin"},
		{"divino", "divin"},
		{"divirtieran", "divirt"},
		{"divisas", "divis"},
		{"división", "division"},
		{"divisional", "divisional"},
		{"divisiones", "division"},
		{"divisoria", "divisori"},
		{"divorciado", "divorci"},
		{"divorciarme", "divorci"},
		{"divorcio", "divorci"},
		{"divorció", "divorc"},
		{"divorcios", "divorci"},
		{"divulgar", "divulg"},
		{"diz", "diz"},
		{"dizque", "dizqu"},
		{"dizzie", "dizzi"},
		{"dls", "dls"},
		{"dnx", "dnx"},
		{"dobla", "dobl"},
		{"doblaje", "doblaj"},
		{"doblajes", "doblaj"},
		{"doblar", "dobl"},
		{"doble", "dobl"},
		{"doblegado", "dobleg"},
		{"doblegamiento", "dobleg"},
		{"doblegando", "dobleg"},
		{"doblegar", "dobleg"},
		{"doblemente", "doblement"},
		{"dobles", "dobl"},
		{"doblinger", "dobling"},
		{"doblista", "doblist"},
		{"doblistas", "doblist"},
		{"dobson", "dobson"},
		{"doce", "doc"},
		{"docena", "docen"},
		{"docenas", "docen"},
		{"docencia", "docenci"},
		{"docente", "docent"},
		{"docentes", "docent"},
		{"dócil", "docil"},
		{"dock", "dock"},
		{"docta", "doct"},
		{"doctor", "doctor"},
		{"doctora", "doctor"},
		{"doctorados", "doctor"},
		{"doctoral", "doctoral"},
		{"doctores", "doctor"},
		{"doctoró", "doctor"},
		{"doctrina", "doctrin"},
		{"doctrinario", "doctrinari"},
		{"doctrinas", "doctrin"},
		{"documentación", "document"},
		{"documentada", "document"},
		{"documentadas", "document"},
		{"documental", "documental"},
		{"documentales", "documental"},
		{"documenté", "document"},
		{"documento", "document"},
		{"documentos", "document"},
		{"dodgers", "dodgers"},
		{"dog", "dog"},
		{"doger", "dog"},
		{"dogs", "dogs"},
		{"dólar", "dol"},
		{"dolares", "dolar"},
		{"dólares", "dolar"},
		{"dolce", "dolc"},
		{"doliéndose", "dol"},
		{"dolo", "dol"},
		{"dolor", "dolor"},
		{"dolores", "dolor"},
		{"dolorido", "dolor"},
		{"dolorosa", "dolor"},
		{"dolosa", "dolos"},
		{"doman", "dom"},
		{"domar", "dom"},
		{"domene", "domen"},
		{"doméstica", "domest"},
		{"domesticados", "domestic"},
		{"domesticaron", "domestic"},
		{"domésticas", "domest"},
		{"doméstico", "domest"},
		{"domésticos", "domest"},
		{"domiciliado", "domicili"},
		{"domiciliaria", "domiciliari"},
		{"domiciliarias", "domiciliari"},
		{"domicilio", "domicili"},
		{"domicilios", "domicili"},
		{"domina", "domin"},
		{"dominada", "domin"},
		{"dominado", "domin"},
		{"dominador", "domin"},
		{"dominan", "domin"},
		{"dominando", "domin"},
		{"dominantes", "domin"},
		{"dominar", "domin"},
		{"dominarlo", "domin"},
		{"dominen", "domin"},
		{"domingo", "doming"},
		{"domingos", "doming"},
		{"domínguez", "dominguez"},
		{"dominical", "dominical"},
		{"dominicana", "dominican"},
		{"dominicano", "dominican"},
		{"dominicanos", "dominican"},
		{"dominio", "domini"},
		{"dominó", "domin"},
		{"domizzi", "domizzi"},
		{"don", "don"},
		{"dona", "don"},
		{"doña", "doñ"},
		{"donadas", "don"},
		{"donado", "don"},
		{"donados", "don"},
		{"donald", "donald"},
		{"donaldo", "donald"},
		{"donan", "don"},
		{"donantes", "donant"},
		{"donar", "don"},
		{"donará", "don"},
		{"donas", "don"},
		{"donativo", "donat"},
		{"donativos", "donat"},
		{"doncella", "doncell"},
		{"donde", "dond"},
		{"dónde", "dond"},
		{"dones", "don"},
		{"donna", "donn"},
		{"donó", "don"},
		{"doors", "doors"},
		{"doquier", "doqui"},
		{"dora", "dor"},
		{"dorada", "dor"},
		{"doradas", "dor"},
		{"dorado", "dor"},
		{"dorados", "dor"},
		{"dorar", "dor"},
		{"doria", "dori"},
		{"doris", "doris"},
		{"dormí", "dorm"},
		{"dormía", "dorm"},
		{"dormir", "dorm"},
		{"dormirán", "dorm"},
		{"dormirme", "dorm"},
		{"dormitó", "dormit"},
		{"dorsal", "dorsal"},
		{"dorsets", "dorsets"},
		{"dorsey", "dorsey"},
		{"dos", "dos"},
		{"dosamantes", "dosam"},
		{"doscientas", "doscient"},
		{"dosis", "dosis"},
		{"dotación", "dotacion"},
		{"dotadas", "dot"},
		{"dotar", "dot"},
		{"dotes", "dot"},
		{"dow", "dow"},
		{"down", "down"},
		{"doy", "doy"},
		{"dr", "dr"},
		{"dra", "dra"},
		{"drácula", "dracul"},
		{"drag", "drag"},
		{"dragón", "dragon"},
		{"dragster", "dragst"},
		{"drake", "drak"},
		{"drama", "dram"},
		{"dramas", "dram"},
		{"dramática", "dramat"},
		{"dramáticamente", "dramat"},
		{"dramático", "dramat"},
		{"dramáticos", "dramat"},
		{"dramatismo", "dramat"},
		{"dramaturgia", "dramaturgi"},
		{"drástica", "drastic"},
		{"drásticamente", "drastic"},
		{"drásticas", "drastic"},
		{"drásticos", "drastic"},
		{"drenaje", "drenaj"},
		{"drew", "drew"},
		{"droga", "drog"},
		{"drogadicción", "drogadiccion"},
		{"drogadicto", "drogadict"},
		{"drogadictos", "drogadict"},
		{"drogado", "drog"},
		{"drogan", "drog"},
		{"drogarse", "drog"},
		{"drogas", "drog"},
		{"du", "du"},
		{"dual", "dual"},
		{"duarte", "duart"},
		{"ducharme", "duch"},
		{"ductilidad", "ductil"},
		{"duda", "dud"},
		{"dudaba", "dud"},
		{"dudaban", "dud"},
		{"dudamos", "dud"},
		{"dudas", "dud"},
		{"dudo", "dud"},
		{"dudosos", "dudos"},
		{"duela", "duel"},
		{"duele", "duel"},
		{"duelen", "duel"},
		{"duelo", "duel"},
		{"dueña", "dueñ"},
		{"dueño", "dueñ"},
		{"dueños", "dueñ"},
		{"duke", "duk"},
		{"dulce", "dulc"},
		{"dulces", "dulc"},
		{"dulles", "dull"},
		{"dulzura", "dulzur"},
		{"dumping", "dumping"},
		{"dune", "dun"},
		{"dunkin", "dunkin"},
		{"duo", "duo"},
		{"dúo", "duo"},
		{"dupla", "dupl"},
		{"duplicación", "duplic"},
		{"duplicado", "duplic"},
		{"duplicados", "duplic"},
		{"duplicar", "duplic"},
		{"duplicará", "duplic"},
		{"duplicidad", "duplic"},
		{"duplicidades", "duplic"},
		{"duplicó", "duplic"},
		{"dupont", "dupont"},
		{"duque", "duqu"},
		{"duques", "duqu"},
		{"duquesa", "duques"},
		{"dura", "dur"},
		{"duraban", "dur"},
		{"durabilidad", "durabil"},
		{"duración", "duracion"},
		{"duradero", "durader"},
		{"duramente", "dur"},
		{"duran", "dur"},
		{"durán", "duran"},
		{"durango", "durang"},
		{"duranguense", "duranguens"},
		{"durante", "durant"},
		{"durar", "dur"},
		{"durara", "dur"},
		{"durará", "dur"},
		{"durarán", "dur"},
		{"duras", "dur"},
		{"durazno", "durazn"},
		{"durbin", "durbin"},
		{"dúrcal", "durcal"},
		{"dure", "dur"},
		{"duren", "dur"},
		{"dureza", "durez"},
		{"durmió", "durm"},
		{"duro", "dur"},
		{"duró", "dur"},
		{"duros", "dur"},
		{"dury", "dury"},
		{"dwyer", "dwyer"},
		{"dye", "dye"},
		{"dzul", "dzul"},
		{"e", "e"},
		{"eagles", "eagl"},
		{"earl", "earl"},
		{"eastwood", "eastwood"},
		{"eau", "eau"},
		{"ebanista", "eban"},
		{"ebanitos", "ebanit"},
		{"eberhard", "eberhard"},
		{"ebriedad", "ebried"},
		{"ebrio", "ebri"},
		{"ebullición", "ebullicion"},
		{"echa", "echa"},
		{"echaban", "echab"},
		{"echada", "echad"},
		{"echado", "echad"},
		{"echan", "echan"},
		{"echando", "echand"},
		{"echándole", "echandol"},
		{"echao", "echa"},
		{"echar", "echar"},
		{"echara", "echar"},
		{"echarán", "echaran"},
		{"echarle", "echarl"},
		{"echarnos", "echarn"},
		{"echas", "echas"},
		{"échate", "echat"},
		{"echavarría", "echavarr"},
		{"eche", "eche"},
		{"echen", "echen"},
		{"echeverría", "echeverr"},
		{"echó", "echo"},
		{"echoes", "echo"},
		{"eckhard", "eckhard"},
		{"eclesiásticas", "eclesiast"},
		{"eclesíasticos", "eclesiast"},
		{"eco", "eco"},
		{"ecología", "ecolog"},
		{"ecológica", "ecolog"},
		{"ecológicas", "ecolog"},
		{"ecológico", "ecolog"},
		{"ecológicos", "ecolog"},
		{"ecologista", "ecolog"},
		{"ecologísta", "ecologist"},
		{"ecologistas", "ecolog"},
		{"econometristas", "econometr"},
		{"economia", "economi"},
		{"economía", "econom"},
		{"economías", "econom"},
		{"economic", "economic"},
		{"económica", "econom"},
		{"económicamente", "econom"},
		{"económicas", "econom"},
		{"económico", "econom"},
		{"económicos", "econom"},
		{"economist", "economist"},
		{"economista", "econom"},
		{"economistas", "econom"},
		{"ecosistema", "ecosistem"},
		{"ecosistemas", "ecosistem"},
		{"ecotaxi", "ecotaxi"},
		{"ecotaxis", "ecotaxis"},
		{"ecotaxistas", "ecotax"},
		{"ecotécnicas", "ecotecn"},
		{"ecuaciones", "ecuacion"},
		{"ecuador", "ecuador"},
		{"ecuánime", "ecuanim"},
		{"ecuatoriana", "ecuatorian"},
		{"ecuatoriano", "ecuatorian"},
		{"edad", "edad"},
		{"edades", "edad"},
		{"edecán", "edecan"},
		{"edelmiro", "edelmir"},
		{"eden", "eden"},
		{"edén", "eden"},
		{"eder", "eder"},
		{"edgar", "edgar"},
		{"edgardo", "edgard"},
		{"edición", "edicion"},
		{"ediciones", "edicion"},
		{"edifica", "edif"},
		{"edificación", "edif"},
		{"edificaciones", "edif"},
		{"edificar", "edific"},
		{"edificio", "edifici"},
		{"edificios", "edifici"},
		{"edifico", "edif"},
		{"edil", "edil"},
		{"edilberto", "edilbert"},
		{"ediles", "edil"},
		{"edison", "edison"},
		{"edit", "edit"},
		{"edita", "edit"},
		{"editada", "edit"},
		{"editado", "edit"},
		{"editados", "edit"},
		{"editar", "edit"},
		{"edith", "edith"},
		{"editó", "edit"},
		{"editor", "editor"},
		{"editoras", "editor"},
		{"editores", "editor"},
		{"editorial", "editorial"},
		{"editoriales", "editorial"},
		{"editorialista", "editorial"},
		{"editorialy", "editorialy"},
		{"edmundo", "edmund"},
		{"edomex", "edomex"},
		{"edu", "edu"},
		{"edú", "edu"},
		{"eduard", "eduard"},
		{"eduardo", "eduard"},
		{"educación", "educ"},
		{"educada", "educ"},
		{"educador", "educ"},
		{"educadores", "educ"},
		{"educando", "educ"},
		{"educar", "educ"},
		{"educaron", "educ"},
		{"educativa", "educ"},
		{"educativas", "educ"},
		{"educativo", "educ"},
		{"educativos", "educ"},
		{"eduque", "eduqu"},
		{"edward", "edward"},
		{"efe", "efe"},
		{"efectiva", "efect"},
		{"efectivamente", "efect"},
		{"efectivas", "efect"},
		{"efectividad", "efect"},
		{"efectivo", "efect"},
		{"efectivos", "efect"},
		{"efecto", "efect"},
		{"efectos", "efect"},
		{"efectúa", "efectu"},
		{"efectuaba", "efectu"},
		{"efectuada", "efectu"},
		{"efectuadas", "efectu"},
		{"efectuado", "efectu"},
		{"efectúan", "efectu"},
		{"efectuar", "efectu"},
		{"efectuará", "efectu"},
		{"efectuarán", "efectu"},
		{"efectuaría", "efectu"},
		{"efectuaron", "efectu"},
		{"efectuarse", "efectu"},
		{"efectúen", "efectu"},
		{"efectuo", "efectu"},
		{"efectúo", "efectu"},
		{"efectuó", "efectu"},
		{"eficaces", "eficac"},
		{"eficacia", "eficaci"},
		{"eficaz", "eficaz"},
		{"eficazmente", "eficaz"},
		{"eficiencia", "eficient"},
		{"eficientarlas", "eficient"},
		{"eficiente", "eficient"},
		{"eficientemente", "eficient"},
		{"eficientes", "eficient"},
		{"eficientización", "eficientiz"},
		{"efigie", "efigi"},
		{"efímero", "efimer"},
		{"efluentes", "efluent"},
		{"efraín", "efrain"},
		{"efusivamente", "efus"},
		{"egca", "egca"},
		{"eggert", "eggert"},
		{"égida", "egid"},
		{"egipcios", "egipci"},
		{"egipto", "egipt"},
		{"ego", "ego"},
		{"egocentrismo", "egocentr"},
		{"egoísmo", "egoism"},
		{"egoísta", "egoist"},
		{"egresaban", "egres"},
		{"egresada", "egres"},
		{"egresados", "egres"},
		{"egresan", "egres"},
		{"egresaron", "egres"},
		{"egresen", "egres"},
		{"egreso", "egres"},
		{"egresos", "egres"},
		{"eiffel", "eiffel"},
		{"eight", "eight"},
		{"eje", "eje"},
		{"ejecución", "ejecu"},
		{"ejecuta", "ejecut"},
		{"ejecutaba", "ejecut"},
		{"ejecutable", "ejecut"},
		{"ejecutada", "ejecut"},
		{"ejecutado", "ejecut"},
		{"ejecutados", "ejecut"},
		{"ejecutando", "ejecut"},
		{"ejecutantes", "ejecut"},
		{"ejecutar", "ejecut"},
		{"ejecutarán", "ejecut"},
		{"ejecutarse", "ejecut"},
		{"ejecutiva", "ejecut"},
		{"ejecutivas", "ejecut"},
		{"ejecutivo", "ejecut"},
		{"ejecutivos", "ejecut"},
		{"ejecutó", "ejecut"},
		{"ejemplar", "ejempl"},
		{"ejemplares", "ejemplar"},
		{"ejemplarmente", "ejemplar"},
		{"ejemplificó", "ejemplific"},
		{"ejemplifique", "ejemplifiqu"},
		{"ejemplo", "ejempl"},
		{"ejemplos", "ejempl"},
		{"ejerce", "ejerc"},
		{"ejercen", "ejerc"},
		{"ejercer", "ejerc"},
		{"ejercerán", "ejerc"},
		{"ejercían", "ejerc"},
		{"ejercicio", "ejercici"},
		{"ejercicios", "ejercici"},
		{"ejerciendo", "ejerc"},
		{"ejerciera", "ejerc"},
		{"ejercitación", "ejercit"},
		{"ejercito", "ejercit"},
		{"ejército", "ejercit"},
		{"ejércitos", "ejercit"},
		{"ejidatarios", "ejidatari"},
		{"ejido", "ejid"},
		{"el", "el"},
		{"él", "el"},
		{"elaboración", "elabor"},
		{"elaborada", "elabor"},
		{"elaboradas", "elabor"},
		{"elaborado", "elabor"},
		{"elaborados", "elabor"},
		{"elaborar", "elabor"},
		{"elaborarán", "elabor"},
		{"elabore", "elabor"},
		{"elaboró", "elabor"},
		{"elásticos", "elast"},
		{"elba", "elba"},
		{"elcrimen", "elcrim"},
		{"eldridge", "eldridg"},
		{"eleanor", "eleanor"},
		{"elección", "eleccion"},
		{"eleccionario", "eleccionari"},
		{"elecciones", "eleccion"},
		{"electa", "elect"},
		{"electo", "elect"},
		{"elector", "elector"},
		{"electorado", "elector"},
		{"electoral", "electoral"},
		{"electorales", "electoral"},
		{"electoralmente", "electoral"},
		{"electores", "elector"},
		{"electos", "elect"},
		{"eléctrica", "electr"},
		{"eléctricas", "electr"},
		{"electricidad", "electr"},
		{"electricistas", "electric"},
		{"eléctrico", "electr"},
		{"eléctricos", "electr"},
		{"electroacústica", "electroacust"},
		{"electrodomésticos", "electrodomest"},
		{"electromagnéticas", "electromagnet"},
		{"electronic", "electronic"},
		{"electrónica", "electron"},
		{"electrónicas", "electron"},
		{"electrónico", "electron"},
		{"electrónicos", "electron"},
		{"elefante", "elef"},
		{"elegancia", "eleg"},
		{"elegante", "eleg"},
		{"elegantemente", "eleg"},
		{"elegantes", "eleg"},
		{"elegidas", "eleg"},
		{"elegido", "eleg"},
		{"elegidos", "eleg"},
		{"elegir", "eleg"},
		{"elegirá", "eleg"},
		{"elegirán", "eleg"},
		{"elegirse", "eleg"},
		{"elektra", "elektr"},
		{"elemental", "elemental"},
		{"elementales", "elemental"},
		{"elemento", "element"},
		{"elementos", "element"},
		{"elena", "elen"},
		{"elenco", "elenc"},
		{"elencos", "elenc"},
		{"elepé", "elep"},
		{"elevaban", "elev"},
		{"elevación", "elev"},
		{"elevada", "elev"},
		{"elevadas", "elev"},
		{"elevado", "elev"},
		{"elevados", "elev"},
		{"elevamos", "elev"},
		{"elevan", "elev"},
		{"elevar", "elev"},
		{"elevaran", "elev"},
		{"elevaría", "elev"},
		{"elevarse", "elev"},
		{"elevó", "elev"},
		{"elia", "eli"},
		{"elías", "eli"},
		{"elida", "elid"},
		{"elie", "eli"},
		{"elier", "eli"},
		{"eligen", "elig"},
		{"eligieron", "elig"},
		{"eligió", "elig"},
		{"eligirá", "elig"},
		{"eligirán", "elig"},
		{"elija", "elij"},
		{"elijan", "elij"},
		{"elimina", "elimin"},
		{"eliminación", "elimin"},
		{"eliminada", "elimin"},
		{"eliminado", "elimin"},
		{"eliminar", "elimin"},
		{"eliminará", "elimin"},
		{"eliminarán", "elimin"},
		{"eliminarían", "elimin"},
		{"eliminarla", "elimin"},
		{"eliminarse", "elimin"},
		{"eliminatoria", "eliminatori"},
		{"eliminatorias", "eliminatori"},
		{"eliminatorios", "eliminatori"},
		{"elimine", "elimin"},
		{"eliminó", "elimin"},
		{"eliodoro", "eliodor"},
		{"elisa", "elis"},
		{"eliseo", "elise"},
		{"élite", "elit"},
		{"eliud", "eliud"},
		{"elizondo", "elizond"},
		{"ella", "ella"},
		{"ellas", "ellas"},
		{"ellington", "ellington"},
		{"ello", "ello"},
		{"ellos", "ellos"},
		{"elmer", "elmer"},
		{"elmo", "elmo"},
		{"elocuencia", "elocuent"},
		{"elogiado", "elogi"},
		{"elogio", "elogi"},
		{"elogios", "elogi"},
		{"elogiosas", "elogi"},
		{"eloisa", "elois"},
		{"eloísa", "elois"},
		{"elosúa", "elosu"},
		{"elotitos", "elotit"},
		{"eloxochitlán", "eloxochitlan"},
		{"eloy", "eloy"},
		{"elton", "elton"},
		{"elude", "elud"},
		{"eludir", "elud"},
		{"elva", "elva"},
		{"elvis", "elvis"},
		{"em", "em"},
		{"emana", "eman"},
		{"emanadas", "eman"},
		{"emanciparse", "emancip"},
		{"ematografía", "ematograf"},
		{"embajada", "embaj"},
		{"embajadas", "embaj"},
		{"embajador", "embaj"},
		{"embajadora", "embaj"},
		{"embajadores", "embaj"},
		{"embalaje", "embalaj"},
		{"embarazada", "embaraz"},
		{"embarazarse", "embaraz"},
		{"embarazo", "embaraz"},
		{"embarazosa", "embaraz"},
		{"embarazoso", "embaraz"},
		{"embarcación", "embarc"},
		{"embarcaciones", "embarc"},
		{"embarcadas", "embarc"},
		{"embarcar", "embarc"},
		{"embargaron", "embarg"},
		{"embargo", "embarg"},
		{"embarrarnos", "embarr"},
		{"embestía", "embest"},
		{"embestida", "embest"},
		{"embestidas", "embest"},
		{"embestido", "embest"},
		{"embistiendo", "embist"},
		{"embistió", "embist"},
		{"emblema", "emblem"},
		{"embolsó", "embols"},
		{"emboscada", "embosc"},
		{"emboscado", "embosc"},
		{"emboscados", "embosc"},
		{"emboscan", "embosc"},
		{"embotelladora", "embotell"},
		{"embotelladoras", "embotell"},
		{"embotellamientos", "embotell"},
		{"embravecidas", "embravec"},
		{"embriagantes", "embriag"},
		{"embriagarse", "embriag"},
		{"embriagas", "embriag"},
		{"embutidos", "embut"},
		{"emerge", "emerg"},
		{"emergencia", "emergent"},
		{"emergencias", "emergent"},
		{"emergente", "emergent"},
		{"emergentes", "emergent"},
		{"emerson", "emerson"},
		{"emes", "emes"},
		{"emigramos", "emigr"},
		{"emigrantes", "emigr"},
		{"emigrar", "emigr"},
		{"emigraron", "emigr"},
		{"emiliano", "emilian"},
		{"emilio", "emili"},
		{"eminencia", "eminent"},
		{"eminentemente", "eminent"},
		{"emirato", "emirat"},
		{"emisión", "emision"},
		{"emisiones", "emision"},
		{"emisora", "emisor"},
		{"emisoras", "emisor"},
		{"emisores", "emisor"},
		{"emita", "emit"},
		{"emitan", "emit"},
		{"emiten", "emit"},
		{"emitían", "emit"},
		{"emitida", "emit"},
		{"emitidas", "emit"},
		{"emitido", "emit"},
		{"emitidos", "emit"},
		{"emitiendo", "emit"},
		{"emitieron", "emit"},
		{"emitió", "emit"},
		{"emitir", "emit"},
		{"emitirá", "emit"},
		{"emitirán", "emit"},
		{"emitiría", "emit"},
		{"emmanuel", "emmanuel"},
		{"emmitth", "emmitth"},
		{"emoción", "emocion"},
		{"emocionada", "emocion"},
		{"emocionado", "emocion"},
		{"emocionados", "emocion"},
		{"emocional", "emocional"},
		{"emocionante", "emocion"},
		{"emocionaron", "emocion"},
		{"emociones", "emocion"},
		{"emocionó", "emocion"},
		{"emory", "emory"},
		{"emotivamente", "emot"},
		{"empacadores", "empac"},
		{"empacados", "empac"},
		{"empacamex", "empacamex"},
		{"empadronados", "empadron"},
		{"empalagó", "empalag"},
		{"empanadas", "empan"},
		{"empantanadas", "empantan"},
		{"empantanar", "empantan"},
		{"empape", "empap"},
		{"empapé", "empap"},
		{"empaquetado", "empaquet"},
		{"empatado", "empat"},
		{"empatados", "empat"},
		{"empatar", "empat"},
		{"empatarles", "empat"},
		{"empataron", "empat"},
		{"empate", "empat"},
		{"empates", "empat"},
		{"empató", "empat"},
		{"empecé", "empec"},
		{"empecinan", "empecin"},
		{"empeñado", "empeñ"},
		{"empeñados", "empeñ"},
		{"empeño", "empeñ"},
		{"empeoran", "empeor"},
		{"empeorar", "empeor"},
		{"emperador", "emper"},
		{"emperatriz", "emperatriz"},
		{"empero", "emper"},
		{"empezaba", "empez"},
		{"empezaban", "empez"},
		{"empezado", "empez"},
		{"empezamos", "empez"},
		{"empezando", "empez"},
		{"empezar", "empez"},
		{"empezará", "empez"},
		{"empezarán", "empez"},
		{"empezarlos", "empez"},
		{"empezaron", "empez"},
		{"empezó", "empez"},
		{"empiece", "empiec"},
		{"empiecen", "empiec"},
		{"empieza", "empiez"},
		{"empiezan", "empiez"},
		{"emplazamiento", "emplaz"},
		{"emplea", "emple"},
		{"empleada", "emple"},
		{"empleadas", "emple"},
		{"empleado", "emple"},
		{"empleados", "emple"},
		{"empleando", "emple"},
		{"emplear", "emple"},
		{"empleo", "emple"},
		{"empleó", "emple"},
		{"empleos", "emple"},
		{"empobrecido", "empobrec"},
		{"emporio", "empori"},
		{"emprendedor", "emprendedor"},
		{"emprendedores", "emprendedor"},
		{"emprender", "emprend"},
		{"emprenderá", "emprend"},
		{"emprendido", "emprend"},
		{"emprendiendo", "emprend"},
		{"emprendimos", "emprend"},
		{"emprendió", "emprend"},
		{"empresa", "empres"},
		{"empresaria", "empresari"},
		{"empresarial", "empresarial"},
		{"empresariales", "empresarial"},
		{"empresario", "empresari"},
		{"empresarios", "empresari"},
		{"empresas", "empres"},
		{"empréstitos", "emprestit"},
		{"empujamos", "empuj"},
		{"empujan", "empuj"},
		{"empujando", "empuj"},
		{"empujar", "empuj"},
		{"empujarlo", "empuj"},
		{"emular", "emul"},
		{"en", "en"},
		{"enajenación", "enajen"},
		{"enamora", "enamor"},
		{"enamorado", "enamor"},
		{"enamoró", "enamor"},
		{"encabece", "encabec"},
		{"encabeza", "encabez"},
		{"encabezada", "encabez"},
		{"encabezado", "encabez"},
		{"encabezados", "encabez"},
		{"encabezar", "encabez"},
		{"encabezará", "encabez"},
		{"encabezaron", "encabez"},
		{"encabezó", "encabez"},
		{"encadenaban", "encaden"},
		{"encadenado", "encaden"},
		{"encaja", "encaj"},
		{"encajan", "encaj"},
		{"encaje", "encaj"},
		{"encajes", "encaj"},
		{"encallados", "encall"},
		{"encallar", "encall"},
		{"encaminadas", "encamin"},
		{"encaminará", "encamin"},
		{"encaminaron", "encamin"},
		{"encañonaron", "encañon"},
		{"encanta", "encant"},
		{"encantada", "encant"},
		{"encantador", "encant"},
		{"encantados", "encant"},
		{"encantan", "encant"},
		{"encantaría", "encant"},
		{"encanto", "encant"},
		{"encantos", "encant"},
		{"encapricha", "encaprich"},
		{"encara", "encar"},
		{"encarcelado", "encarcel"},
		{"encarcelados", "encarcel"},
		{"encarcelamiento", "encarcel"},
		{"encarecerá", "encarec"},
		{"encarga", "encarg"},
		{"encargaba", "encarg"},
		{"encargaban", "encarg"},
		{"encargada", "encarg"},
		{"encargadas", "encarg"},
		{"encargado", "encarg"},
		{"encargados", "encarg"},
		{"encargar", "encarg"},
		{"encargara", "encarg"},
		{"encargará", "encarg"},
		{"encargarán", "encarg"},
		{"encargaron", "encarg"},
		{"encargo", "encarg"},
		{"encargó", "encarg"},
		{"encargue", "encarg"},
		{"encargué", "encarg"},
		{"encarna", "encarn"},
		{"encarnado", "encarn"},
		{"encarnan", "encarn"},
		{"encarnizan", "encarniz"},
		{"encarno", "encarn"},
		{"encarnó", "encarn"},
		{"encauce", "encauc"},
		{"encendedor", "encendedor"},
		{"encendedores", "encendedor"},
		{"encendido", "encend"},
		{"encendidos", "encend"},
		{"encerar", "encer"},
		{"enceren", "encer"},
		{"encerrado", "encerr"},
		{"encerrados", "encerr"},
		{"encerrarnos", "encerr"},
		{"encerraron", "encerr"},
		{"encerrona", "encerron"},
		{"enchalecados", "enchalec"},
		{"enchinchar", "enchinch"},
		{"enciclopedia", "enciclopedi"},
		{"enciclopedias", "enciclopedi"},
		{"enciclopédico", "encicloped"},
		{"encienda", "enciend"},
		{"encierra", "encierr"},
		{"encierro", "encierr"},
		{"encima", "encim"},
		{"encke", "encke"},
		{"enclaustrados", "enclaustr"},
		{"encomendadas", "encomend"},
		{"encomendando", "encomend"},
		{"encomiable", "encomi"},
		{"encomienda", "encomiend"},
		{"encomiende", "encomiend"},
		{"encontraba", "encontr"},
		{"encontraban", "encontr"},
		{"encontrada", "encontr"},
		{"encontradas", "encontr"},
		{"encontrado", "encontr"},
		{"encontrados", "encontr"},
		{"encontramos", "encontr"},
		{"encontrar", "encontr"},
		{"encontrara", "encontr"},
		{"encontrará", "encontr"},
		{"encontraran", "encontr"},
		{"encontrarán", "encontr"},
		{"encontrarás", "encontr"},
		{"encontraremos", "encontr"},
		{"encontrarle", "encontr"},
		{"encontrarlo", "encontr"},
		{"encontrarme", "encontr"},
		{"encontraron", "encontr"},
		{"encontrarse", "encontr"},
		{"encontrársele", "encontrarsel"},
		{"encontré", "encontr"},
		{"encontró", "encontr"},
		{"encriptación", "encript"},
		{"encrucijada", "encrucij"},
		{"encuadrados", "encuadr"},
		{"encuadren", "encuadr"},
		{"encubierta", "encubiert"},
		{"encubiertos", "encubiert"},
		{"encuentra", "encuentr"},
		{"encuentran", "encuentr"},
		{"encuentras", "encuentr"},
		{"encuentre", "encuentr"},
		{"encuentro", "encuentr"},
		{"encuentros", "encuentr"},
		{"encuesta", "encuest"},
		{"encuestadora", "encuest"},
		{"encuestados", "encuest"},
		{"encuestaron", "encuest"},
		{"encuestas", "encuest"},
		{"encueste", "encuest"},
		{"encumbran", "encumbr"},
		{"endeavour", "endeavour"},
		{"endebles", "endebl"},
		{"enderezadas", "enderez"},
		{"endeudamiento", "endeud"},
		{"endiosado", "endi"},
		{"endiosamiento", "endi"},
		{"endocrinología", "endocrinolog"},
		{"endocrinólogo", "endocrinolog"},
		{"endulce", "endulc"},
		{"endulzada", "endulz"},
		{"enduro", "endur"},
		{"enemiga", "enemig"},
		{"enemigo", "enemig"},
		{"enemigos", "enemig"},
		{"energética", "energet"},
		{"energético", "energet"},
		{"energéticos", "energet"},
		{"energía", "energ"},
		{"enérgica", "energ"},
		{"enérgicas", "energ"},
		{"enérgico", "energ"},
		{"enero", "ener"},
		{"enervante", "enerv"},
		{"enervantes", "enerv"},
		{"enésima", "enesim"},
		{"enfadaría", "enfad"},
		{"énfasis", "enfasis"},
		{"enfatiza", "enfatiz"},
		{"enfatizará", "enfatiz"},
		{"enfatizó", "enfatiz"},
		{"enferma", "enferm"},
		{"enfermarse", "enferm"},
		{"enfermé", "enferm"},
		{"enfermedad", "enfermed"},
		{"enfermedades", "enfermedad"},
		{"enfermera", "enfermer"},
		{"enfermeras", "enfermer"},
		{"enfermo", "enferm"},
		{"enfermos", "enferm"},
		{"enfiladas", "enfil"},
		{"enfilamos", "enfil"},
		{"enfoca", "enfoc"},
		{"enfocadas", "enfoc"},
		{"enfocado", "enfoc"},
		{"enfocados", "enfoc"},
		{"enfocan", "enfoc"},
		{"enfocar", "enfoc"},
		{"enfocará", "enfoc"},
		{"enfocarán", "enfoc"},
		{"enfocarse", "enfoc"},
		{"enfoque", "enfoqu"},
		{"enfoquen", "enfoqu"},
		{"enfóquese", "enfoques"},
		{"enfrascados", "enfrasc"},
		{"enfrenta", "enfrent"},
		{"enfrentaba", "enfrent"},
		{"enfrentada", "enfrent"},
		{"enfrentado", "enfrent"},
		{"enfrentamiento", "enfrent"},
		{"enfrentamientos", "enfrent"},
		{"enfrentamos", "enfrent"},
		{"enfrentan", "enfrent"},
		{"enfrentando", "enfrent"},
		{"enfrentándose", "enfrent"},
		{"enfrentar", "enfrent"},
		{"enfrentará", "enfrent"},
		{"enfrentarán", "enfrent"},
		{"enfrentaremos", "enfrent"},
		{"enfrentarlos", "enfrent"},
		{"enfrentarme", "enfrent"},
		{"enfrentaron", "enfrent"},
		{"enfrentarse", "enfrent"},
		{"enfrente", "enfrent"},
		{"enfrenten", "enfrent"},
		{"enfrentó", "enfrent"},
		{"enfriamiento", "enfriamient"},
		{"enfundaron", "enfund"},
		{"enfurecieron", "enfurec"},
		{"engalana", "engalan"},
		{"engañaba", "engañ"},
		{"engañado", "engañ"},
		{"engañar", "engañ"},
		{"engancharle", "enganch"},
		{"engaño", "engañ"},
		{"engañosas", "engañ"},
		{"engendra", "engendr"},
		{"engineering", "engineering"},
		{"englutidos", "englut"},
		{"engomados", "engom"},
		{"engordar", "engord"},
		{"engordó", "engord"},
		{"engorrosas", "engorr"},
		{"engorrosos", "engorr"},
		{"engracia", "engraci"},
		{"engranaje", "engranaj"},
		{"engrandecer", "engrandec"},
		{"engrane", "engran"},
		{"engrasen", "engras"},
		{"enhorabuena", "enhorabuen"},
		{"enigmático", "enigmat"},
		{"enjambre", "enjambr"},
		{"enjuiciado", "enjuici"},
		{"enjuiciar", "enjuici"},
		{"enlace", "enlac"},
		{"enlatados", "enlat"},
		{"enlaza", "enlaz"},
		{"enlazadas", "enlaz"},
		{"enlazado", "enlaz"},
		{"enlistar", "enlist"},
		{"enlisten", "enlist"},
		{"enmarañado", "enmarañ"},
		{"enmarca", "enmarc"},
		{"enmarcarlos", "enmarc"},
		{"enmarcaron", "enmarc"},
		{"enmarcarse", "enmarc"},
		{"enmascarado", "enmascar"},
		{"enmedio", "enmedi"},
		{"enmendar", "enmend"},
		{"enoc", "enoc"},
		{"enojaran", "enoj"},
		{"enojo", "enoj"},
		{"enojó", "enoj"},
		{"enorme", "enorm"},
		{"enormemente", "enorm"},
		{"enormes", "enorm"},
		{"enraizada", "enraiz"},
		{"enrazado", "enraz"},
		{"enrazados", "enraz"},
		{"enreden", "enred"},
		{"enrico", "enric"},
		{"enrique", "enriqu"},
		{"enriquece", "enriquec"},
		{"enriquecer", "enriquec"},
		{"enriquecerse", "enriquec"},
		{"enriquecida", "enriquec"},
		{"enriquecimiento", "enriquec"},
		{"enriqueció", "enriquec"},
		{"enríquez", "enriquez"},
		{"ensambladora", "ensambl"},
		{"ensamblar", "ensambl"},
		{"ensamble", "ensambl"},
		{"ensambles", "ensambl"},
		{"ensanchado", "ensanch"},
		{"ensangrentados", "ensangrent"},
		{"ensaya", "ensay"},
		{"ensayado", "ensay"},
		{"ensayaron", "ensay"},
		{"ensayista", "ensay"},
		{"ensayistas", "ensay"},
		{"ensayo", "ensay"},
		{"enseguida", "ensegu"},
		{"enseña", "enseñ"},
		{"enseñaba", "enseñ"},
		{"enseñaban", "enseñ"},
		{"ensenada", "ensen"},
		{"enseñado", "enseñ"},
		{"enseñan", "enseñ"},
		{"enseñando", "enseñ"},
		{"enseñanza", "enseñ"},
		{"enseñanzas", "enseñ"},
		{"enseñar", "enseñ"},
		{"enseñarles", "enseñ"},
		{"enseñen", "enseñ"},
		{"enseñes", "enseñ"},
		{"enseñó", "enseñ"},
		{"ensoberbecerse", "ensoberbec"},
		{"ensombrecían", "ensombrec"},
		{"ensorberbecera", "ensorberbecer"},
		{"ensordecedores", "ensordecedor"},
		{"ensuciar", "ensuci"},
		{"entabladas", "entabl"},
		{"entablar", "entabl"},
		{"entablarse", "entabl"},
		{"ente", "ente"},
		{"entenados", "enten"},
		{"entendederas", "entendeder"},
		{"entendemos", "entend"},
		{"entender", "entend"},
		{"entenderá", "entend"},
		{"entenderé", "entend"},
		{"entenderíamos", "entend"},
		{"entendí", "entend"},
		{"entendible", "entend"},
		{"entendibles", "entend"},
		{"entendido", "entend"},
		{"entendimiento", "entend"},
		{"entera", "enter"},
		{"enterada", "enter"},
		{"enterado", "enter"},
		{"enterados", "enter"},
		{"enteramente", "enter"},
		{"enteramos", "enter"},
		{"enterar", "enter"},
		{"enterarse", "enter"},
		{"entere", "enter"},
		{"enteremos", "enter"},
		{"enteren", "enter"},
		{"entereza", "enterez"},
		{"entero", "enter"},
		{"enteró", "enter"},
		{"enteros", "enter"},
		{"enterrarlos", "enterr"},
		{"entes", "entes"},
		{"entidad", "entid"},
		{"entidades", "entidad"},
		{"entienda", "entiend"},
		{"entiendan", "entiend"},
		{"entiende", "entiend"},
		{"entienden", "entiend"},
		{"entierro", "entierr"},
		{"entonado", "enton"},
		{"entonces", "entonc"},
		{"entonen", "enton"},
		{"entorno", "entorn"},
		{"entorpece", "entorpec"},
		{"entra", "entra"},
		{"entraban", "entrab"},
		{"entrada", "entrad"},
		{"entradas", "entrad"},
		{"entrado", "entrad"},
		{"entramos", "entram"},
		{"entran", "entran"},
		{"entraña", "entrañ"},
		{"entrañaría", "entrañ"},
		{"entrando", "entrand"},
		{"entrándole", "entrandol"},
		{"entrante", "entrant"},
		{"entrar", "entrar"},
		{"entrará", "entrar"},
		{"entraran", "entrar"},
		{"entrarán", "entraran"},
		{"entraría", "entrar"},
		{"entrarían", "entrar"},
		{"entrarle", "entrarl"},
		{"entraron", "entraron"},
		{"entre", "entre"},
		{"entré", "entre"},
		{"entrecano", "entrecan"},
		{"entrecomillas", "entrecomill"},
		{"entrecruzan", "entrecruz"},
		{"entrega", "entreg"},
		{"entregada", "entreg"},
		{"entregadas", "entreg"},
		{"entregado", "entreg"},
		{"entregados", "entreg"},
		{"entregan", "entreg"},
		{"entregándoles", "entreg"},
		{"entregar", "entreg"},
		{"entregara", "entreg"},
		{"entregará", "entreg"},
		{"entregaran", "entreg"},
		{"entregarán", "entreg"},
		{"entregaría", "entreg"},
		{"entregarían", "entreg"},
		{"entregarla", "entreg"},
		{"entregarle", "entreg"},
		{"entregarme", "entreg"},
		{"entregaron", "entreg"},
		{"entregarse", "entreg"},
		{"entregársela", "entreg"},
		{"entregas", "entreg"},
		{"entrego", "entreg"},
		{"entregó", "entreg"},
		{"entregue", "entreg"},
		{"entreguen", "entreg"},
		{"entréguenos", "entreguen"},
		{"entrelazadas", "entrelaz"},
		{"entrelíneas", "entrelin"},
		{"entren", "entren"},
		{"entrena", "entren"},
		{"entrenaba", "entren"},
		{"entrenado", "entren"},
		{"entrenador", "entren"},
		{"entrenadores", "entren"},
		{"entrenados", "entren"},
		{"entrenamiento", "entren"},
		{"entrenamientos", "entren"},
		{"entrenan", "entren"},
		{"entrenando", "entren"},
		{"entrenar", "entren"},
		{"entrenará", "entren"},
		{"entrenarán", "entren"},
		{"entrenarlo", "entren"},
		{"entrenarlos", "entren"},
		{"entrenaron", "entren"},
		{"entrenó", "entren"},
		{"entretejida", "entretej"},
		{"entretener", "entreten"},
		{"entretenían", "entreten"},
		{"entretenimiento", "entreten"},
		{"entretenimientos", "entreten"},
		{"entrever", "entrev"},
		{"entrevista", "entrev"},
		{"entrevistaba", "entrevist"},
		{"entrevistada", "entrevist"},
		{"entrevistado", "entrevist"},
		{"entrevistados", "entrevist"},
		{"entrevistar", "entrevist"},
		{"entrevistaron", "entrevist"},
		{"entrevistarse", "entrevist"},
		{"entrevistas", "entrev"},
		{"entrevistó", "entrevist"},
		{"entró", "entro"},
		{"entrometiera", "entromet"},
		{"entusiasmada", "entusiasm"},
		{"entusiasmado", "entusiasm"},
		{"entusiasmo", "entusiasm"},
		{"entusiasta", "entusiast"},
		{"entusiastas", "entusiast"},
		{"enumera", "enumer"},
		{"enumerará", "enumer"},
		{"envasadora", "envas"},
		{"envase", "envas"},
		{"envases", "envas"},
		{"envejecen", "envejec"},
		{"envejecimiento", "envejec"},
		{"envenenada", "envenen"},
		{"envenenarla", "envenen"},
		{"envia", "envi"},
		{"envía", "envi"},
		{"enviada", "envi"},
		{"enviadas", "envi"},
		{"enviado", "envi"},
		{"enviados", "envi"},
		{"enviamos", "envi"},
		{"envían", "envi"},
		{"enviando", "envi"},
		{"enviar", "envi"},
		{"enviará", "envi"},
		{"enviarán", "envi"},
		{"enviaría", "envi"},
		{"enviarlas", "envi"},
		{"enviaron", "envi"},
		{"envidia", "envidi"},
		{"envidiables", "envidi"},
		{"envidiamos", "envidi"},
		{"envidias", "envidi"},
		{"envidiosos", "envidi"},
		{"envíe", "envi"},
		{"envio", "envi"},
		{"envío", "envi"},
		{"envió", "envi"},
		{"envoltorio", "envoltori"},
		{"envoltura", "envoltur"},
		{"envolturas", "envoltur"},
		{"envolverse", "envolv"},
		{"envolviéndose", "envolv"},
		{"envueltos", "envuelt"},
		{"enyesado", "enyes"},
		{"eory", "eory"},
		{"epazote", "epazot"},
		{"episcopado", "episcop"},
		{"episcopal", "episcopal"},
		{"episodio", "episodi"},
		{"episodios", "episodi"},
		{"epístola", "epistol"},
		{"epistolar", "epistol"},
		{"época", "epoc"},
		{"épocas", "epoc"},
		{"equals", "equals"},
		{"equidad", "equid"},
		{"equilibrada", "equilibr"},
		{"equilibrado", "equilibr"},
		{"equilibrar", "equilibr"},
		{"equilibrarlos", "equilibr"},
		{"equilibrio", "equilibri"},
		{"equipada", "equip"},
		{"equipado", "equip"},
		{"equipados", "equip"},
		{"equipaje", "equipaj"},
		{"equipan", "equip"},
		{"equipar", "equip"},
		{"equipararnos", "equipar"},
		{"equipment", "equipment"},
		{"equipo", "equip"},
		{"equipos", "equip"},
		{"equisetum", "equisetum"},
		{"equitación", "equit"},
		{"equivale", "equival"},
		{"equivalen", "equival"},
		{"equivalente", "equivalent"},
		{"equivalentes", "equivalent"},
		{"equivoca", "equivoc"},
		{"equivocaba", "equivoc"},
		{"equivocaban", "equivoc"},
		{"equivocaciones", "equivoc"},
		{"equivocados", "equivoc"},
		{"equivocarse", "equivoc"},
		{"equivocó", "equivoc"},
		{"equivoqué", "equivoqu"},
		{"er", "er"},
		{"era", "era"},
		{"eramos", "eram"},
		{"eran", "eran"},
		{"erario", "erari"},
		{"eres", "eres"},
		{"erhard", "erhard"},
		{"eric", "eric"},
		{"erick", "erick"},
		{"erigido", "erig"},
		{"erika", "erik"},
		{"ermita", "ermit"},
		{"ernesto", "ernest"},
		{"erogaron", "erog"},
		{"eros", "eros"},
		{"erosionada", "erosion"},
		{"erosionadas", "erosion"},
		{"eróticas", "erot"},
		{"erraban", "errab"},
		{"erradica", "errad"},
		{"erradicación", "errad"},
		{"errado", "errad"},
		{"errática", "errat"},
		{"erráticos", "errat"},
		{"errónea", "errone"},
		{"erróneamente", "erron"},
		{"erróneo", "errone"},
		{"error", "error"},
		{"errores", "error"},
		{"erupción", "erupcion"},
		{"es", "es"},
		{"esa", "esa"},
		{"ésa", "esa"},
		{"esai", "esai"},
		{"esas", "esas"},
		{"ésas", "esas"},
		{"esbozo", "esboz"},
		{"escabrosos", "escabr"},
		{"escala", "escal"},
		{"escalada", "escal"},
		{"escalafón", "escalafon"},
		{"escalante", "escal"},
		{"escalar", "escal"},
		{"escalera", "escaler"},
		{"escaleras", "escaler"},
		{"escalinata", "escalinat"},
		{"escalón", "escalon"},
		{"escalona", "escalon"},
		{"escalonados", "escalon"},
		{"escalones", "escalon"},
		{"escamilla", "escamill"},
		{"escampavías", "escampav"},
		{"escándalo", "escandal"},
		{"escándalos", "escandal"},
		{"escandalosas", "escandal"},
		{"escaños", "escañ"},
		{"escapa", "escap"},
		{"escapada", "escap"},
		{"escapado", "escap"},
		{"escapando", "escap"},
		{"escapar", "escap"},
		{"escaparon", "escap"},
		{"escape", "escap"},
		{"escapó", "escap"},
		{"escarmiento", "escarmient"},
		{"escasa", "escas"},
		{"escasas", "escas"},
		{"escasez", "escasez"},
		{"escaso", "escas"},
		{"escasos", "escas"},
		{"escena", "escen"},
		{"escenario", "escenari"},
		{"escenarios", "escenari"},
		{"escenas", "escen"},
		{"escenciales", "escencial"},
		{"escénico", "escen"},
		{"escenificará", "escenific"},
		{"escenificarán", "escenific"},
		{"escenografía", "escenograf"},
		{"escenógrafos", "escenograf"},
		{"escisiones", "escision"},
		{"esclarecer", "esclarec"},
		{"esclarecimiento", "esclarec"},
		{"esclarezca", "esclarezc"},
		{"esclava", "esclav"},
		{"esclavas", "esclav"},
		{"esclavitud", "esclavitud"},
		{"esclavo", "esclav"},
		{"escoba", "escob"},
		{"escobar", "escob"},
		{"escobas", "escob"},
		{"escobedo", "escobed"},
		{"escobera", "escober"},
		{"escocés", "escoces"},
		{"escoge", "escog"},
		{"escoger", "escog"},
		{"escogería", "escog"},
		{"escoges", "escog"},
		{"escogido", "escog"},
		{"escogiendo", "escog"},
		{"escogieron", "escog"},
		{"escogimos", "escog"},
		{"escogió", "escog"},
		{"escojan", "escoj"},
		{"escolar", "escol"},
		{"escolares", "escolar"},
		{"escolarización", "escolariz"},
		{"escollo", "escoll"},
		{"escoltado", "escolt"},
		{"escombrera", "escombrer"},
		{"escombros", "escombr"},
		{"escondan", "escond"},
		{"esconde", "escond"},
		{"esconden", "escond"},
		{"esconder", "escond"},
		{"esconderse", "escond"},
		{"escondida", "escond"},
		{"escondidas", "escond"},
		{"escondido", "escond"},
		{"escondió", "escond"},
		{"escondite", "escondit"},
		{"escondrijos", "escondrij"},
		{"escore", "escor"},
		{"escoriaciones", "escori"},
		{"escorpión", "escorpion"},
		{"escote", "escot"},
		{"escotes", "escot"},
		{"escriba", "escrib"},
		{"escríbanos", "escriban"},
		{"escribe", "escrib"},
		{"escriben", "escrib"},
		{"escribía", "escrib"},
		{"escribiendo", "escrib"},
		{"escribieran", "escrib"},
		{"escribieron", "escrib"},
		{"escribió", "escrib"},
		{"escribir", "escrib"},
		{"escribirá", "escrib"},
		{"escribirlas", "escrib"},
		{"escribirle", "escrib"},
		{"escribirse", "escrib"},
		{"escribo", "escrib"},
		{"escrita", "escrit"},
		{"escritas", "escrit"},
		{"escrito", "escrit"},
		{"escritor", "escritor"},
		{"escritora", "escritor"},
		{"escritores", "escritor"},
		{"escritorio", "escritori"},
		{"escritos", "escrit"},
		{"escritura", "escritur"},
		{"escrituras", "escritur"},
		{"escrutinio", "escrutini"},
		{"escuadra", "escuadr"},
		{"escuadras", "escuadr"},
		{"escuadrones", "escuadron"},
		{"escuálidas", "escual"},
		{"escucha", "escuch"},
		{"escuchaba", "escuch"},
		{"escuchaban", "escuch"},
		{"escuchadas", "escuch"},
		{"escuchado", "escuch"},
		{"escuchan", "escuch"},
		{"escuchando", "escuch"},
		{"escuchándolas", "escuch"},
		{"escuchar", "escuch"},
		{"escuchará", "escuch"},
		{"escucharán", "escuch"},
		{"escucharla", "escuch"},
		{"escucharlos", "escuch"},
		{"escucharon", "escuch"},
		{"escuche", "escuch"},
		{"escuché", "escuch"},
		{"escuchen", "escuch"},
		{"escuchó", "escuch"},
		{"escudería", "escud"},
		{"escudo", "escud"},
		{"escudriñan", "escudriñ"},
		{"escuela", "escuel"},
		{"escuelas", "escuel"},
		{"escuetamente", "escuet"},
		{"escultor", "escultor"},
		{"escultórico", "escultor"},
		{"escultura", "escultur"},
		{"esculturas", "escultur"},
		{"escupir", "escup"},
		{"escurridizo", "escurridiz"},
		{"escurridizos", "escurridiz"},
		{"escurridos", "escurr"},
		{"esde", "esde"},
		{"ese", "ese"},
		{"ése", "ese"},
		{"esencia", "esenci"},
		{"esencial", "esencial"},
		{"esencialmente", "esencial"},
		{"esfera", "esfer"},
		{"esférico", "esfer"},
		{"esforzamos", "esforz"},
		{"esforzarnos", "esforz"},
		{"esforzaron", "esforz"},
		{"esfuercen", "esfuerc"},
		{"esfuerzo", "esfuerz"},
		{"esfuerzos", "esfuerz"},
		{"esfuman", "esfum"},
		{"esgrimía", "esgrim"},
		{"esguince", "esguinc"},
		{"eslabonó", "eslabon"},
		{"eslava", "eslav"},
		{"esmaltes", "esmalt"},
		{"esmeralda", "esmerald"},
		{"esmeraldas", "esmerald"},
		{"esmeran", "esmer"},
		{"eso", "eso"},
		{"esos", "esos"},
		{"ésos", "esos"},
		{"espacial", "espacial"},
		{"espaciales", "espacial"},
		{"espacio", "espaci"},
		{"espacios", "espaci"},
		{"espada", "espad"},
		{"espalda", "espald"},
		{"espaldas", "espald"},
		{"españa", "españ"},
		{"español", "español"},
		{"española", "español"},
		{"españolas", "español"},
		{"españoles", "español"},
		{"españolitos", "españolit"},
		{"espansión", "espansion"},
		{"espanta", "espant"},
		{"espantadas", "espant"},
		{"esparcimiento", "esparc"},
		{"esparció", "esparc"},
		{"esparza", "esparz"},
		{"espasmo", "espasm"},
		{"especial", "especial"},
		{"especiales", "especial"},
		{"especialidad", "especial"},
		{"especialidades", "especial"},
		{"especialista", "especial"},
		{"especialistas", "especial"},
		{"especializa", "especializ"},
		{"especialización", "especializ"},
		{"especializada", "especializ"},
		{"especializadas", "especializ"},
		{"especializado", "especializ"},
		{"especializados", "especializ"},
		{"especialmente", "especial"},
		{"especie", "especi"},
		{"especies", "especi"},
		{"especifica", "especif"},
		{"específica", "especif"},
		{"especificaba", "especific"},
		{"especificaciones", "especif"},
		{"especificado", "especific"},
		{"específicamente", "especif"},
		{"especifican", "especific"},
		{"especificar", "especific"},
		{"específicas", "especif"},
		{"específico", "especif"},
		{"especificó", "especific"},
		{"específicos", "especif"},
		{"especifiquen", "especifiqu"},
		{"espectacular", "espectacul"},
		{"espectaculares", "espectacular"},
		{"espectacularmente", "espectacular"},
		{"espectáculo", "espectacul"},
		{"espectáculos", "espectacul"},
		{"espectador", "espect"},
		{"espectadores", "espect"},
		{"espectro", "espectr"},
		{"espectrógrafo", "espectrograf"},
		{"espectrómetro", "espectrometr"},
		{"espectroredioeléctrico", "espectroredioelectr"},
		{"", ""},
		{"especulador", "especul"},
		{"especuladores", "especul"},
		{"especular", "especul"},
		{"especulativos", "especul"},
		{"especuló", "especul"},
		{"espejismo", "espej"},
		{"espejo", "espej"},
		{"espera", "esper"},
		{"esperaba", "esper"},
		{"esperábamos", "esper"},
		{"esperaban", "esper"},
		{"esperada", "esper"},
		{"esperado", "esper"},
		{"esperados", "esper"},
		{"esperamos", "esper"},
		{"esperan", "esper"},
		{"esperando", "esper"},
		{"esperanza", "esper"},
		{"esperanzado", "esperanz"},
		{"esperanzados", "esperanz"},
		{"esperanzas", "esper"},
		{"esperar", "esper"},
		{"esperara", "esper"},
		{"esperará", "esper"},
		{"esperarán", "esper"},
		{"esperaré", "esper"},
		{"esperaremos", "esper"},
		{"esperaría", "esper"},
		{"esperarlos", "esper"},
		{"esperaron", "esper"},
		{"esperarse", "esper"},
		{"espere", "esper"},
		{"esperemos", "esper"},
		{"espero", "esper"},
		{"esperó", "esper"},
		{"esperpénticas", "esperpent"},
		{"espesa", "espes"},
		{"espesor", "espesor"},
		{"espetar", "espet"},
		{"espía", "espi"},
		{"espían", "espi"},
		{"espiando", "espi"},
		{"espina", "espin"},
		{"espinal", "espinal"},
		{"espinosa", "espin"},
		{"espinoso", "espin"},
		{"espinoza", "espinoz"},
		{"espionaje", "espionaj"},
		{"espiridión", "espiridion"},
		{"espíritu", "espiritu"},
		{"espiritual", "espiritual"},
		{"espirituales", "espiritual"},
		{"espiritualidad", "espiritual"},
		{"espiritualmente", "espiritual"},
		{"espíritus", "espiritus"},
		{"esplendor", "esplendor"},
		{"esplendores", "esplendor"},
		{"espontánea", "espontane"},
		{"espontáneamente", "espontan"},
		{"espontáneo", "espontane"},
		{"esporádica", "esporad"},
		{"esposa", "espos"},
		{"esposas", "espos"},
		{"esposo", "espos"},
		{"esposos", "espos"},
		{"espuela", "espuel"},
		{"espuma", "espum"},
		{"esqueleto", "esquelet"},
		{"esquema", "esquem"},
		{"esquemas", "esquem"},
		{"esquer", "esqu"},
		{"esquiar", "esqui"},
		{"esquimal", "esquimal"},
		{"esquina", "esquin"},
		{"esquinas", "esquin"},
		{"esquivar", "esquiv"},
		{"esquivel", "esquivel"},
		{"esquizofrenia", "esquizofreni"},
		{"esta", "esta"},
		{"ésta", "esta"},
		{"está", "esta"},
		{"estaba", "estab"},
		{"estábamos", "estab"},
		{"estaban", "estab"},
		{"estabilidad", "estabil"},
		{"estabilizar", "estabiliz"},
		{"estable", "establ"},
		{"establece", "establec"},
		{"establecemos", "establec"},
		{"establecen", "establec"},
		{"establecer", "establec"},
		{"establecerá", "establec"},
		{"establecerse", "establec"},
		{"establecía", "establec"},
		{"establecían", "establec"},
		{"establecida", "establec"},
		{"establecidas", "establec"},
		{"establecido", "establec"},
		{"establecidos", "establec"},
		{"estableciendo", "establec"},
		{"establecieron", "establec"},
		{"establecimiento", "establec"},
		{"establecimientos", "establec"},
		{"estableció", "establec"},
		{"establezca", "establezc"},
		{"establezcan", "establezc"},
		{"estación", "estacion"},
		{"estacionado", "estacion"},
		{"estacionados", "estacion"},
		{"estacionamiento", "estacion"},
		{"estacionamientos", "estacion"},
		{"estacionar", "estacion"},
		{"estacionarios", "estacionari"},
		{"estacionarse", "estacion"},
		{"estaciones", "estacion"},
		{"estacionó", "estacion"},
		{"estacón", "estacon"},
		{"estadía", "estad"},
		{"estadio", "estadi"},
		{"estadios", "estadi"},
		{"estadista", "estad"},
		{"estadistas", "estad"},
		{"estadística", "estadist"},
		{"estadísticas", "estadist"},
		{"estado", "estad"},
		{"estados", "estad"},
		{"estadounidense", "estadounidens"},
		{"estadounidenses", "estadounidens"},
		{"estafan", "estaf"},
		{"estalla", "estall"},
		{"estallar", "estall"},
		{"estallaría", "estall"},
		{"estallido", "estall"},
		{"estalló", "estall"},
		{"estambre", "estambr"},
		{"estamos", "estam"},
		{"estampas", "estamp"},
		{"estampilla", "estampill"},
		{"estampita", "estampit"},
		{"están", "estan"},
		{"estancadas", "estanc"},
		{"estancamiento", "estanc"},
		{"estancarse", "estanc"},
		{"estancia", "estanci"},
		{"estancias", "estanci"},
		{"estandar", "estand"},
		{"estándar", "estand"},
		{"estándares", "estandar"},
		{"estandarizadas", "estandariz"},
		{"estandarte", "estandart"},
		{"estando", "estand"},
		{"estanques", "estanqu"},
		{"estanzuela", "estanzuel"},
		{"estar", "estar"},
		{"estará", "estar"},
		{"estarán", "estaran"},
		{"estaremos", "estar"},
		{"estaría", "estar"},
		{"estaríamos", "estar"},
		{"estarían", "estar"},
		{"estarle", "estarl"},
		{"estarse", "estars"},
		{"estas", "estas"},
		{"éstas", "estas"},
		{"estás", "estas"},
		{"estatal", "estatal"},
		{"estatales", "estatal"},
		{"estatua", "estatu"},
		{"estatuas", "estatu"},
		{"estatura", "estatur"},
		{"estatuto", "estatut"},
		{"estatutos", "estatut"},
		{"este", "este"},
		{"éste", "este"},
		{"esté", "este"},
		{"esteban", "esteb"},
		{"estelar", "estel"},
		{"estelares", "estelar"},
		{"estelarizado", "estelariz"},
		{"estelarizar", "estelariz"},
		{"estemos", "estem"},
		{"estén", "esten"},
		{"estéreo", "estere"},
		{"estereotipan", "estereotip"},
		{"esterlinas", "esterlin"},
		{"esternón", "esternon"},
		{"estética", "estet"},
		{"estéticamente", "estet"},
		{"estéticas", "estet"},
		{"estéticos", "estet"},
		{"estévez", "estevez"},
		{"esthela", "esthel"},
		{"esther", "esther"},
		{"estigma", "estigm"},
		{"estilo", "estil"},
		{"estilos", "estil"},
		{"estima", "estim"},
		{"estimaba", "estim"},
		{"estimación", "estim"},
		{"estimaciones", "estim"},
		{"estimada", "estim"},
		{"estimado", "estim"},
		{"estimados", "estim"},
		{"estimamos", "estim"},
		{"estiman", "estim"},
		{"estimando", "estim"},
		{"estimarás", "estim"},
		{"estimaron", "estim"},
		{"estimé", "estim"},
		{"estimó", "estim"},
		{"estimula", "estimul"},
		{"estimulación", "estimul"},
		{"estimulada", "estimul"},
		{"estimulado", "estimul"},
		{"estimulados", "estimul"},
		{"estimular", "estimul"},
		{"estimulo", "estimul"},
		{"estímulo", "estimul"},
		{"estímulos", "estimul"},
		{"estipula", "estipul"},
		{"estipulaba", "estipul"},
		{"estira", "estir"},
		{"estiró", "estir"},
		{"estirón", "estiron"},
		{"estirpe", "estirp"},
		{"esto", "esto"},
		{"ésto", "esto"},
		{"estocada", "estoc"},
		{"estocolmo", "estocolm"},
		{"estomacal", "estomacal"},
		{"estómago", "estomag"},
		{"estopa", "estop"},
		{"estoperoles", "estoperol"},
		{"estorba", "estorb"},
		{"estorbaba", "estorb"},
		{"estorbe", "estorb"},
		{"estos", "estos"},
		{"éstos", "estos"},
		{"estoy", "estoy"},
		{"estrada", "estrad"},
		{"estrado", "estrad"},
		{"estragos", "estrag"},
		{"estrangulado", "estrangul"},
		{"estrasburgo", "estrasburg"},
		{"estratagemas", "estratagem"},
		{"estratega", "estrateg"},
		{"estrategia", "estrategi"},
		{"estrategias", "estrategi"},
		{"estratégica", "estrateg"},
		{"estratégicas", "estrateg"},
		{"estratégico", "estrateg"},
		{"estratégicos", "estrateg"},
		{"estratos", "estrat"},
		{"estratosféricas", "estratosfer"},
		{"estrecha", "estrech"},
		{"estrechándose", "estrech"},
		{"estrechar", "estrech"},
		{"estrecharon", "estrech"},
		{"estrechas", "estrech"},
		{"estrecho", "estrech"},
		{"estrechó", "estrech"},
		{"estrechos", "estrech"},
		{"estrella", "estrell"},
		{"estrellado", "estrell"},
		{"estrellamos", "estrell"},
		{"estrellar", "estrell"},
		{"estrellas", "estrell"},
		{"estrelló", "estrell"},
		{"estremeció", "estremec"},
		{"estrena", "estren"},
		{"estrenado", "estren"},
		{"estrenan", "estren"},
		{"estrenando", "estren"},
		{"estrenándose", "estren"},
		{"estrenar", "estren"},
		{"estrenará", "estren"},
		{"estrenaron", "estren"},
		{"estreno", "estren"},
		{"estrenó", "estren"},
		{"estrenos", "estren"},
		{"estrépito", "estrepit"},
		{"estrepitosa", "estrepit"},
		{"estrepitosamente", "estrepit"},
		{"estribo", "estrib"},
		{"estricta", "estrict"},
		{"estrictamente", "estrict"},
		{"estrictas", "estrict"},
		{"estricto", "estrict"},
		{"estrictos", "estrict"},
		{"estropeadas", "estrop"},
		{"estropear", "estrop"},
		{"estructura", "estructur"},
		{"estructuración", "estructur"},
		{"estructural", "estructural"},
		{"estructuras", "estructur"},
		{"estruendo", "estruend"},
		{"estudia", "estudi"},
		{"estudiaba", "estudi"},
		{"estudiaban", "estudi"},
		{"estudiados", "estudi"},
		{"estudian", "estudi"},
		{"estudiando", "estudi"},
		{"estudiante", "estudi"},
		{"estudiantes", "estudi"},
		{"estudiantil", "estudiantil"},
		{"estudiantiles", "estudiantil"},
		{"estudiar", "estudi"},
		{"estudiará", "estudi"},
		{"estudiaron", "estudi"},
		{"estudiarse", "estudi"},
		{"estudie", "estudi"},
		{"estudié", "estudi"},
		{"estudien", "estudi"},
		{"estudio", "estudi"},
		{"estudió", "estud"},
		{"estudios", "estudi"},
		{"estudioso", "estudi"},
		{"estudiosos", "estudi"},
		{"estupefacción", "estupefaccion"},
		{"estupefacta", "estupefact"},
		{"estupendo", "estupend"},
		{"estúpida", "estup"},
		{"estupideces", "estupidec"},
		{"estuve", "estuv"},
		{"estuviera", "estuv"},
		{"estuviéramos", "estuv"},
		{"estuvieran", "estuv"},
		{"estuvieron", "estuv"},
		{"estuviese", "estuv"},
		{"estuvimos", "estuv"},
		{"estuvo", "estuv"},
		{"etapa", "etap"},
		{"etapas", "etap"},
		{"etc", "etc"},
		{"etcétera", "etceter"},
		{"etchohuaquila", "etchohuaquil"},
		{"eternidad", "etern"},
		{"eterno", "etern"},
		{"ethernet", "ethernet"},
		{"ethnology", "ethnology"},
		{"ética", "etic"},
		{"éticas", "etic"},
		{"etienne", "etienn"},
		{"etíope", "etiop"},
		{"etiopía", "etiop"},
		{"etiqueta", "etiquet"},
		{"etiquetas", "etiquet"},
		{"etnia", "etni"},
		{"étnica", "etnic"},
		{"étnicas", "etnic"},
		{"étnico", "etnic"},
		{"etnografía", "etnograf"},
		{"etnohistórico", "etnohistor"},
		{"ettiene", "ettien"},
		{"eu", "eu"},
		{"eucalipto", "eucalipt"},
		{"eudelio", "eudeli"},
		{"euforia", "eufori"},
		{"eugéne", "eugen"},
		{"eugenia", "eugeni"},
		{"eugenio", "eugeni"},
		{"eugui", "eugui"},
		{"eulalio", "eulali"},
		{"eurípides", "euripid"},
		{"eurobonos", "eurobon"},
		{"europa", "europ"},
		{"europea", "europe"},
		{"europeas", "europ"},
		{"europeo", "europe"},
		{"europeos", "europe"},
		{"eurovisión", "eurovision"},
		{"eusebio", "eusebi"},
		{"eustolio", "eustoli"},
		{"eva", "eva"},
		{"evacuación", "evacu"},
		{"evacuados", "evacu"},
		{"evacuan", "evacu"},
		{"evacuar", "evacu"},
		{"evadía", "evad"},
		{"evadió", "evad"},
		{"evadir", "evad"},
		{"evaluacion", "evaluacion"},
		{"evaluación", "evalu"},
		{"evaluaciones", "evalu"},
		{"evaluado", "evalu"},
		{"evaluador", "evalu"},
		{"evaluados", "evalu"},
		{"evalúan", "evalu"},
		{"evaluar", "evalu"},
		{"evaluarán", "evalu"},
		{"evaluaron", "evalu"},
		{"evangélicos", "evangel"},
		{"evangelización", "evangeliz"},
		{"evangelizan", "evangeliz"},
		{"evangelizar", "evangeliz"},
		{"evasión", "evasion"},
		{"evasivas", "evas"},
		{"evasivo", "evas"},
		{"evento", "event"},
		{"eventos", "event"},
		{"eventual", "eventual"},
		{"eventuales", "eventual"},
		{"eventualmente", "eventual"},
		{"ever", "ever"},
		{"everardo", "everard"},
		{"everman", "everm"},
		{"everton", "everton"},
		{"evidencia", "evident"},
		{"evidenciado", "evidenci"},
		{"evidencian", "evidenci"},
		{"evidencias", "evident"},
		{"evidenció", "evidenc"},
		{"evidente", "evident"},
		{"evidentemente", "evident"},
		{"evidentes", "evident"},
		{"evita", "evit"},
		{"evitado", "evit"},
		{"evitan", "evit"},
		{"evitando", "evit"},
		{"evitar", "evit"},
		{"evitara", "evit"},
		{"evitará", "evit"},
		{"evitaremos", "evit"},
		{"evitarla", "evit"},
		{"evitarle", "evit"},
		{"evitarlo", "evit"},
		{"evitarse", "evit"},
		{"evite", "evit"},
		{"eviten", "evit"},
		{"evitó", "evit"},
		{"evocación", "evoc"},
		{"evolución", "evolu"},
		{"evolucionando", "evolucion"},
		{"evolucionara", "evolucion"},
		{"evolucionarios", "evolucionari"},
		{"evolucionaron", "evolucion"},
		{"evolucionen", "evolucion"},
		{"ex", "ex"},
		{"exacta", "exact"},
		{"exactamente", "exact"},
		{"exactas", "exact"},
		{"exactitud", "exactitud"},
		{"exacto", "exact"},
		{"exageraciones", "exager"},
		{"exagerada", "exager"},
		{"exagerado", "exager"},
		{"exagerados", "exager"},
		{"exageraron", "exager"},
		{"exalumnas", "exalumn"},
		{"exalumnos", "exalumn"},
		{"examen", "exam"},
		{"exámenes", "examen"},
		{"examina", "examin"},
		{"examinado", "examin"},
		{"examinados", "examin"},
		{"examinando", "examin"},
		{"examinar", "examin"},
		{"examinen", "examin"},
		{"examinó", "examin"},
		{"exc", "exc"},
		{"excandidato", "excandidat"},
		{"excatedrático", "excatedrat"},
		{"excavaciones", "excav"},
		{"excavando", "excav"},
		{"excede", "exced"},
		{"exceden", "exced"},
		{"excedernos", "exced"},
		{"excediendo", "exced"},
		{"excedieron", "exced"},
		{"excel", "excel"},
		{"excelencia", "excelent"},
		{"excelente", "excelent"},
		{"excelentemente", "excelent"},
		{"excelentes", "excelent"},
		{"excelsa", "excels"},
		{"excélsior", "excelsior"},
		{"excepción", "excepcion"},
		{"excepcional", "excepcional"},
		{"excepcionales", "excepcional"},
		{"excepciones", "excepcion"},
		{"excepto", "except"},
		{"excesiva", "exces"},
		{"excesivamente", "exces"},
		{"excesivo", "exces"},
		{"excesivos", "exces"},
		{"exceso", "exces"},
		{"excesos", "exces"},
		{"exchange", "exchang"},
		{"excita", "excit"},
		{"excitación", "excit"},
		{"excitan", "excit"},
		{"excitante", "excit"},
		{"excitantes", "excit"},
		{"excitarse", "excit"},
		{"exclama", "exclam"},
		{"exclamación", "exclam"},
		{"exclamó", "exclam"},
		{"excluidos", "exclu"},
		{"exclusión", "exclusion"},
		{"exclusiva", "exclus"},
		{"exclusivamente", "exclus"},
		{"exclusivas", "exclus"},
		{"exclusividad", "exclus"},
		{"exclusividades", "exclus"},
		{"exclusivo", "exclus"},
		{"exclusivos", "exclus"},
		{"excluyendo", "exclu"},
		{"excolega", "excoleg"},
		{"excomunión", "excomunion"},
		{"excursiones", "excursion"},
		{"excusarse", "excus"},
		{"excusas", "excus"},
		{"exención", "exencion"},
		{"exenta", "exent"},
		{"exento", "exent"},
		{"exhala", "exhal"},
		{"exhaustiva", "exhaust"},
		{"exhibe", "exhib"},
		{"exhiben", "exhib"},
		{"exhibía", "exhib"},
		{"exhibición", "exhibicion"},
		{"exhibiciones", "exhibicion"},
		{"exhibida", "exhib"},
		{"exhibidas", "exhib"},
		{"exhibido", "exhib"},
		{"exhibiendo", "exhib"},
		{"exhibiera", "exhib"},
		{"exhibiere", "exhibier"},
		{"exhibieren", "exhibier"},
		{"exhibimos", "exhib"},
		{"exhibió", "exhib"},
		{"exhibirá", "exhib"},
		{"exhibirán", "exhib"},
		{"exhibirse", "exhib"},
		{"exhoneración", "exhoner"},
		{"exhorta", "exhort"},
		{"exhortaban", "exhort"},
		{"exhortan", "exhort"},
		{"exhortando", "exhort"},
		{"exhorto", "exhort"},
		{"exhortó", "exhort"},
		{"exige", "exig"},
		{"exigen", "exig"},
		{"exigencia", "exigent"},
		{"exigencias", "exigent"},
		{"exigente", "exigent"},
		{"exigentes", "exigent"},
		{"exigían", "exig"},
		{"exigida", "exig"},
		{"exigido", "exig"},
		{"exigiendo", "exig"},
		{"exigieran", "exig"},
		{"exigieron", "exig"},
		{"exigimos", "exig"},
		{"exigió", "exig"},
		{"exigir", "exig"},
		{"exigira", "exigir"},
		{"exigirá", "exig"},
		{"exigiría", "exig"},
		{"exigirle", "exig"},
		{"exigirles", "exig"},
		{"exigirse", "exig"},
		{"exija", "exij"},
		{"exijan", "exij"},
		{"exilio", "exili"},
		{"exime", "exim"},
		{"eximir", "exim"},
		{"eximirlos", "exim"},
		{"exista", "exist"},
		{"existan", "exist"},
		{"existe", "exist"},
		{"existen", "exist"},
		{"existencia", "existent"},
		{"existente", "existent"},
		{"existentes", "existent"},
		{"existía", "exist"},
		{"existían", "exist"},
		{"existido", "exist"},
		{"existiendo", "exist"},
		{"existió", "exist"},
		{"existir", "exist"},
		{"existirán", "exist"},
		{"exito", "exit"},
		{"éxito", "exit"},
		{"éxitos", "exit"},
		{"exitosa", "exit"},
		{"exitosamente", "exit"},
		{"exitosas", "exit"},
		{"exitoso", "exit"},
		{"exitosos", "exit"},
		{"éxodo", "exod"},
		{"exonera", "exoner"},
		{"exóticas", "exot"},
		{"exoticos", "exot"},
		{"exóticos", "exot"},
		{"expander", "expand"},
		{"expandera", "expander"},
		{"expandido", "expand"},
		{"expandieron", "expand"},
		{"expandirán", "expand"},
		{"expandirse", "expand"},
		{"expansion", "expansion"},
		{"expansión", "expansion"},
		{"expansiva", "expans"},
		{"expectación", "expect"},
		{"expectativa", "expect"},
		{"expectativas", "expect"},
		{"expedición", "expedicion"},
		{"expedidas", "exped"},
		{"expediente", "expedient"},
		{"expedientes", "expedient"},
		{"expedir", "exped"},
		{"expedirse", "exped"},
		{"expedito", "expedit"},
		{"expenden", "expend"},
		{"expendio", "expendi"},
		{"experiencia", "experient"},
		{"experiencias", "experient"},
		{"experimenta", "experiment"},
		{"experimentación", "experiment"},
		{"experimentadas", "experiment"},
		{"experimentado", "experiment"},
		{"experimentados", "experiment"},
		{"experimental", "experimental"},
		{"experimentar", "experiment"},
		{"experimentarse", "experiment"},
		{"experimento", "experiment"},
		{"experimentó", "experiment"},
		{"experto", "expert"},
		{"expertos", "expert"},
		{"expidan", "expid"},
		{"expide", "expid"},
		{"expidieron", "expid"},
		{"expira", "expir"},
		{"expirar", "expir"},
		{"explanada", "explan"},
		{"explica", "explic"},
		{"explicaban", "explic"},
		{"explicación", "explic"},
		{"explicaciones", "explic"},
		{"explicado", "explic"},
		{"explicamos", "explic"},
		{"explican", "explic"},
		{"explicando", "explic"},
		{"explicar", "explic"},
		{"explicara", "explic"},
		{"explicará", "explic"},
		{"explicarles", "explic"},
		{"explicarlo", "explic"},
		{"explicaron", "explic"},
		{"explicó", "explic"},
		{"explique", "expliqu"},
		{"explora", "explor"},
		{"exploraban", "explor"},
		{"exploración", "explor"},
		{"explorado", "explor"},
		{"exploran", "explor"},
		{"explorar", "explor"},
		{"explosión", "explosion"},
		{"explosiva", "explos"},
		{"explosividad", "explos"},
		{"explosivo", "explos"},
		{"explosivos", "explos"},
		{"explotación", "explot"},
		{"explotado", "explot"},
		{"explotan", "explot"},
		{"explotar", "explot"},
		{"explotará", "explot"},
		{"explotarse", "explot"},
		{"explotó", "explot"},
		{"expo", "expo"},
		{"expondrá", "expondr"},
		{"expondrán", "expondran"},
		{"expondremos", "expondr"},
		{"expone", "expon"},
		{"exponentes", "exponent"},
		{"exponer", "expon"},
		{"exponernos", "expon"},
		{"exponerse", "expon"},
		{"exponga", "expong"},
		{"expongan", "expong"},
		{"exponían", "expon"},
		{"exponiéndose", "expon"},
		{"expoplaza", "expoplaz"},
		{"exporta", "export"},
		{"exportación", "export"},
		{"exportaciones", "export"},
		{"exportado", "export"},
		{"exportador", "export"},
		{"exportadoras", "export"},
		{"exportadores", "export"},
		{"exportados", "export"},
		{"exportamos", "export"},
		{"exportan", "export"},
		{"exportando", "export"},
		{"exportar", "export"},
		{"exporters", "exporters"},
		{"exportó", "export"},
		{"exposición", "exposicion"},
		{"exposiciones", "exposicion"},
		{"expresa", "expres"},
		{"expresada", "expres"},
		{"expresadas", "expres"},
		{"expresado", "expres"},
		{"expresamente", "expres"},
		{"expresan", "expres"},
		{"expresando", "expres"},
		{"expresar", "expres"},
		{"expresaron", "expres"},
		{"expresarse", "expres"},
		{"expresé", "expres"},
		{"expresidentes", "expresident"},
		{"expresión", "expresion"},
		{"expresiones", "expresion"},
		{"expresiva", "expres"},
		{"expreso", "expres"},
		{"expresó", "expres"},
		{"express", "express"},
		{"expropiar", "expropi"},
		{"expuesta", "expuest"},
		{"expuesto", "expuest"},
		{"expuestos", "expuest"},
		{"expulsada", "expuls"},
		{"expulsado", "expuls"},
		{"expulsados", "expuls"},
		{"expulsar", "expuls"},
		{"expulsarlo", "expuls"},
		{"expulsarme", "expuls"},
		{"expulsaron", "expuls"},
		{"expulsión", "expulsion"},
		{"expusiera", "expus"},
		{"expusieron", "expus"},
		{"expuso", "expus"},
		{"exquisitamente", "exquisit"},
		{"exquisiteces", "exquisitec"},
		{"exquisitos", "exquisit"},
		{"exsecretario", "exsecretari"},
		{"extendemos", "extend"},
		{"extender", "extend"},
		{"extenderá", "extend"},
		{"extenderse", "extend"},
		{"extendido", "extend"},
		{"extendieron", "extend"},
		{"extendió", "extend"},
		{"extensa", "extens"},
		{"extensas", "extens"},
		{"extensión", "extension"},
		{"extensiones", "extension"},
		{"extensivo", "extens"},
		{"extenso", "extens"},
		{"extenuación", "extenu"},
		{"extenuar", "extenu"},
		{"exterior", "exterior"},
		{"exteriores", "exterior"},
		{"externa", "extern"},
		{"externado", "extern"},
		{"externaron", "extern"},
		{"externas", "extern"},
		{"externo", "extern"},
		{"externó", "extern"},
		{"externos", "extern"},
		{"extiendan", "extiend"},
		{"extiende", "extiend"},
		{"extienden", "extiend"},
		{"extinción", "extincion"},
		{"extinguidores", "extinguidor"},
		{"extinta", "extint"},
		{"extirpar", "extirp"},
		{"extirparon", "extirp"},
		{"extorsionando", "extorsion"},
		{"extorsionar", "extorsion"},
		{"extorsionaron", "extorsion"},
		{"extorsiones", "extorsion"},
		{"extosionando", "extosion"},
		{"extra", "extra"},
		{"extrabajadores", "extrabaj"},
		{"extracción", "extraccion"},
		{"extracto", "extract"},
		{"extradición", "extradicion"},
		{"extraer", "extra"},
		{"extraerle", "extra"},
		{"extraescolar", "extraescol"},
		{"extraída", "extraid"},
		{"extraído", "extraid"},
		{"extrajo", "extraj"},
		{"extraña", "extrañ"},
		{"extrañaba", "extrañ"},
		{"extrañados", "extrañ"},
		{"extrañas", "extrañ"},
		{"extrañe", "extrañ"},
		{"extrañeza", "extrañez"},
		{"extranjera", "extranjer"},
		{"extranjeras", "extranjer"},
		{"extranjero", "extranjer"},
		{"extranjeros", "extranjer"},
		{"extraño", "extrañ"},
		{"extraños", "extrañ"},
		{"extraoficial", "extraoficial"},
		{"extraoficiales", "extraoficial"},
		{"extraoficialmente", "extraoficial"},
		{"extraordinaria", "extraordinari"},
		{"extraordinariamente", "extraordinari"},
		{"extraordinarias", "extraordinari"},
		{"extraordinario", "extraordinari"},
		{"extraordinarios", "extraordinari"},
		{"extras", "extras"},
		{"extraterrestres", "extraterrestr"},
		{"extravagantes", "extravag"},
		{"extraviadas", "extravi"},
		{"extraviado", "extravi"},
		{"extravío", "extravi"},
		{"extravió", "extrav"},
		{"extrema", "extrem"},
		{"extremadamente", "extrem"},
		{"extremo", "extrem"},
		{"extremos", "extrem"},
		{"exvotos", "exvot"},
		{"exxon", "exxon"},
		{"eye", "eye"},
		{"ezeta", "ezet"},
		{"f", "f"},
		{"fabián", "fabian"},
		{"fabiana", "fabian"},
		{"fabio", "fabi"},
		{"fabrica", "fabric"},
		{"fábrica", "fabric"},
		{"fabricación", "fabric"},
		{"fabricadas", "fabric"},
		{"fabricado", "fabric"},
		{"fabricante", "fabric"},
		{"fabricantes", "fabric"},
		{"fabricar", "fabric"},
		{"fabricará", "fabric"},
		{"fabricarla", "fabric"},
		{"fábricas", "fabric"},
		{"fabricó", "fabric"},
		{"fabrique", "fabriqu"},
		{"fabulosas", "fabul"},
		{"fabuloso", "fabul"},
		{"fabulosos", "fabul"},
		{"facciones", "faccion"},
		{"faceta", "facet"},
		{"facha", "fach"},
		{"fachada", "fach"},
		{"fácil", "facil"},
		{"fáciles", "facil"},
		{"facilidad", "facil"},
		{"facilidades", "facil"},
		{"facilita", "facilit"},
		{"facilitaba", "facilit"},
		{"facilitados", "facilit"},
		{"facilitando", "facilit"},
		{"facilitar", "facilit"},
		{"facilitará", "facilit"},
		{"facilitarán", "facilit"},
		{"facilitarles", "facilit"},
		{"facilitarlos", "facilit"},
		{"facilite", "facilit"},
		{"faciliten", "facilit"},
		{"fácilmente", "facil"},
		{"factible", "factibl"},
		{"factibles", "factibl"},
		{"facto", "fact"},
		{"factor", "factor"},
		{"factorajes", "factoraj"},
		{"factores", "factor"},
		{"facts", "facts"},
		{"factura", "factur"},
		{"facturación", "factur"},
		{"facturada", "factur"},
		{"facturados", "factur"},
		{"facturar", "factur"},
		{"facturas", "factur"},
		{"facturó", "factur"},
		{"faculta", "facult"},
		{"facultad", "facult"},
		{"facultades", "facultad"},
		{"facultados", "facult"},
		{"facultativos", "facult"},
		{"facundo", "facund"},
		{"faena", "faen"},
		{"faenas", "faen"},
		{"fagotistas", "fagot"},
		{"fahr", "fahr"},
		{"fairfax", "fairfax"},
		{"faisán", "faisan"},
		{"faja", "faj"},
		{"fajarse", "faj"},
		{"fajín", "fajin"},
		{"fal", "fal"},
		{"falacci", "falacci"},
		{"falcón", "falcon"},
		{"falda", "fald"},
		{"faldas", "fald"},
		{"falla", "fall"},
		{"fallado", "fall"},
		{"fallan", "fall"},
		{"fallar", "fall"},
		{"fallaron", "fall"},
		{"fallas", "fall"},
		{"fallece", "fallec"},
		{"fallecido", "fallec"},
		{"fallecidos", "fallec"},
		{"falleciera", "fallec"},
		{"fallecieron", "fallec"},
		{"falleció", "fallec"},
		{"fallen", "fall"},
		{"fallida", "fall"},
		{"fallido", "fall"},
		{"fallo", "fall"},
		{"falló", "fall"},
		{"falls", "falls"},
		{"falsa", "fals"},
		{"falsas", "fals"},
		{"falsete", "falset"},
		{"falsificación", "falsif"},
		{"falsificados", "falsific"},
		{"falsificó", "falsific"},
		{"falso", "fals"},
		{"falsos", "fals"},
		{"falta", "falt"},
		{"faltaba", "falt"},
		{"faltaban", "falt"},
		{"faltado", "falt"},
		{"faltan", "falt"},
		{"faltando", "falt"},
		{"faltándole", "falt"},
		{"faltante", "faltant"},
		{"faltar", "falt"},
		{"faltara", "falt"},
		{"faltas", "falt"},
		{"falte", "falt"},
		{"falté", "falt"},
		{"falto", "falt"},
		{"faltó", "falt"},
		{"faltriquera", "faltriquer"},
		{"fama", "fam"},
		{"familia", "famili"},
		{"familiar", "famili"},
		{"familiares", "familiar"},
		{"familiarice", "familiaric"},
		{"familiarizado", "familiariz"},
		{"familiarizarse", "familiariz"},
		{"familias", "famili"},
		{"famosa", "famos"},
		{"famosas", "fam"},
		{"famoso", "famos"},
		{"famosos", "famos"},
		{"fanaticada", "fanatic"},
		{"fanáticas", "fanat"},
		{"fanático", "fanat"},
		{"fanáticos", "fanat"},
		{"fanfarrias", "fanfarri"},
		{"fango", "fang"},
		{"fans", "fans"},
		{"fantasía", "fantas"},
		{"fantasmas", "fantasm"},
		{"farías", "far"},
		{"farinas", "farin"},
		{"farmaceútica", "farmaceut"},
		{"farmacia", "farmaci"},
		{"farmacias", "farmaci"},
		{"farmacodependientes", "farmacodependient"},
		{"farmacología", "farmacolog"},
		{"farmar", "farm"},
		{"farrugia", "farrugi"},
		{"farsa", "fars"},
		{"farsante", "farsant"},
		{"fascinación", "fascin"},
		{"fascinante", "fascin"},
		{"fase", "fas"},
		{"fases", "fas"},
		{"fast", "fast"},
		{"fastuosos", "fastuos"},
		{"fatal", "fatal"},
		{"fatales", "fatal"},
		{"fatalmente", "fatal"},
		{"fatiga", "fatig"},
		{"fatigado", "fatig"},
		{"fátima", "fatim"},
		{"fauna", "faun"},
		{"faustino", "faustin"},
		{"fausto", "faust"},
		{"favor", "favor"},
		{"favorable", "favor"},
		{"favorablemente", "favor"},
		{"favorables", "favor"},
		{"favorece", "favorec"},
		{"favorecer", "favorec"},
		{"favorecerá", "favorec"},
		{"favorecerán", "favorec"},
		{"favorecía", "favorec"},
		{"favorecían", "favorec"},
		{"favorecieron", "favorec"},
		{"favorecimiento", "favorec"},
		{"favoreció", "favorec"},
		{"favores", "favor"},
		{"favorezcan", "favorezc"},
		{"favorita", "favorit"},
		{"favoritas", "favorit"},
		{"favoritismo", "favorit"},
		{"favorito", "favorit"},
		{"favoritos", "favorit"},
		{"fax", "fax"},
		{"faxes", "fax"},
		{"fbi", "fbi"},
		{"fda", "fda"},
		{"fe", "fe"},
		{"fea", "fea"},
		{"feas", "feas"},
		{"feat", "feat"},
		{"feb", "feb"},
		{"febrero", "febrer"},
		{"fecha", "fech"},
		{"fechada", "fech"},
		{"fechado", "fech"},
		{"fechados", "fech"},
		{"fechas", "fech"},
		{"fechorías", "fechor"},
		{"fecunda", "fecund"},
		{"fecundación", "fecund"},
		{"fecundando", "fecund"},
		{"fecundidad", "fecund"},
		{"fecundínez", "fecundinez"},
		{"federación", "feder"},
		{"federal", "federal"},
		{"federales", "federal"},
		{"federalismo", "federal"},
		{"federalización", "federaliz"},
		{"federativa", "feder"},
		{"federativas", "feder"},
		{"federativo", "feder"},
		{"federica", "feder"},
		{"federico", "feder"},
		{"feeder", "feed"},
		{"fehacientemente", "fehacient"},
		{"feldhaus", "feldhaus"},
		{"felguérez", "felguerez"},
		{"felices", "felic"},
		{"feliciano", "felician"},
		{"felicidad", "felic"},
		{"felicitación", "felicit"},
		{"felicitaciones", "felicit"},
		{"felicitada", "felicit"},
		{"felicitadas", "felicit"},
		{"felicitado", "felicit"},
		{"felicitar", "felicit"},
		{"felicitarla", "felicit"},
		{"felicitarlo", "felicit"},
		{"felicito", "felicit"},
		{"felicitó", "felicit"},
		{"felícitos", "felicit"},
		{"feligreses", "feligres"},
		{"felina", "felin"},
		{"felino", "felin"},
		{"felinos", "felin"},
		{"felipe", "felip"},
		{"félix", "felix"},
		{"feliz", "feliz"},
		{"femam", "femam"},
		{"femenil", "femenil"},
		{"femeniles", "femenil"},
		{"femenina", "femenin"},
		{"femenino", "femenin"},
		{"femeninos", "femenin"},
		{"femineidad", "femin"},
		{"feminismo", "femin"},
		{"feminista", "femin"},
		{"feministas", "femin"},
		{"femsa", "fems"},
		{"fendick", "fendick"},
		{"feneciendo", "fenec"},
		{"fenómeno", "fenomen"},
		{"fenómenos", "fenomen"},
		{"feo", "feo"},
		{"fer", "fer"},
		{"fergie", "fergi"},
		{"ferguson", "ferguson"},
		{"feria", "feri"},
		{"ferias", "feri"},
		{"fermín", "fermin"},
		{"fernanda", "fernand"},
		{"fernandez", "fernandez"},
		{"fernández", "fernandez"},
		{"fernando", "fern"},
		{"ferniza", "ferniz"},
		{"feroz", "feroz"},
		{"ferrara", "ferr"},
		{"ferrat", "ferrat"},
		{"férreo", "ferre"},
		{"ferretería", "ferret"},
		{"ferreterías", "ferret"},
		{"ferriño", "ferriñ"},
		{"ferrocarril", "ferrocarril"},
		{"ferrocarrileros", "ferrocarriler"},
		{"ferrocarriles", "ferrocarril"},
		{"ferronales", "ferronal"},
		{"ferroviaria", "ferroviari"},
		{"ferroviarias", "ferroviari"},
		{"ferroviario", "ferroviari"},
		{"ferroviarios", "ferroviari"},
		{"fértiles", "fertil"},
		{"fertilidad", "fertil"},
		{"fertilizante", "fertiliz"},
		{"fertilizantes", "fertiliz"},
		{"fesperman", "fesperm"},
		{"festeja", "festej"},
		{"festejada", "festej"},
		{"festejar", "festej"},
		{"festejaron", "festej"},
		{"festejo", "festej"},
		{"festejó", "festej"},
		{"festejos", "festej"},
		{"festiva", "festiv"},
		{"festival", "festival"},
		{"festivales", "festival"},
		{"festividades", "festiv"},
		{"festivo", "festiv"},
		{"festivos", "festiv"},
		{"fetales", "fetal"},
		{"fetos", "fet"},
		{"feudales", "feudal"},
		{"ffcc", "ffcc"},
		{"fían", "fian"},
		{"fianza", "fianz"},
		{"fib", "fib"},
		{"fibra", "fibr"},
		{"fibras", "fibr"},
		{"fibrosarcomas", "fibrosarcom"},
		{"ficción", "ficcion"},
		{"ficha", "fich"},
		{"fichado", "fich"},
		{"ficóloga", "ficolog"},
		{"fictícios", "fictici"},
		{"fideicomiso", "fideicomis"},
		{"fideicomisos", "fideicomis"},
		{"fidel", "fidel"},
		{"fidelidad", "fidel"},
		{"fidencio", "fidenci"},
		{"fidenor", "fidenor"},
		{"fidler", "fidl"},
		{"fiebre", "fiebr"},
		{"fiel", "fiel"},
		{"fieles", "fiel"},
		{"fierro", "fierr"},
		{"fiesta", "fiest"},
		{"fiestas", "fiest"},
		{"fiestota", "fiestot"},
		{"fifa", "fif"},
		{"fife", "fif"},
		{"fignon", "fignon"},
		{"figueroa", "figuero"},
		{"figura", "figur"},
		{"figuraba", "figur"},
		{"figurado", "figur"},
		{"figuran", "figur"},
		{"figuraría", "figur"},
		{"figuras", "figur"},
		{"figurativamente", "figur"},
		{"fija", "fij"},
		{"fijación", "fijacion"},
		{"fijadas", "fij"},
		{"fijado", "fij"},
		{"fijados", "fij"},
		{"fijamos", "fij"},
		{"fijan", "fij"},
		{"fijar", "fij"},
		{"fijará", "fij"},
		{"fijaron", "fij"},
		{"fijarse", "fij"},
		{"fijas", "fij"},
		{"fíjate", "fijat"},
		{"fije", "fij"},
		{"fíjese", "fijes"},
		{"fijeza", "fijez"},
		{"fijo", "fij"},
		{"fijó", "fij"},
		{"fijos", "fij"},
		{"fila", "fil"},
		{"filadelfia", "filadelfi"},
		{"filas", "fil"},
		{"filatelia", "filateli"},
		{"filatélica", "filatel"},
		{"filatélico", "filatel"},
		{"filatélicos", "filatel"},
		{"filatelismo", "filatel"},
		{"filatelista", "filatel"},
		{"filatelistas", "filatel"},
		{"filete", "filet"},
		{"filial", "filial"},
		{"filiales", "filial"},
		{"filiberto", "filibert"},
		{"filipinas", "filipin"},
		{"filipino", "filipin"},
		{"film", "film"},
		{"filmación", "filmacion"},
		{"filmado", "film"},
		{"filmados", "film"},
		{"filmar", "film"},
		{"filmaron", "film"},
		{"filme", "film"},
		{"fílmicos", "filmic"},
		{"filosofía", "filosof"},
		{"filosófica", "filosof"},
		{"filósofo", "filosof"},
		{"filósofos", "filosof"},
		{"filtración", "filtracion"},
		{"filtraciones", "filtracion"},
		{"filtro", "filtr"},
		{"filtros", "filtr"},
		{"fin", "fin"},
		{"fina", "fin"},
		{"final", "final"},
		{"finales", "final"},
		{"finalice", "finalic"},
		{"finalidad", "final"},
		{"finalista", "final"},
		{"finalistas", "final"},
		{"finaliza", "finaliz"},
		{"finalizado", "finaliz"},
		{"finalizar", "finaliz"},
		{"finalizara", "finaliz"},
		{"finalizará", "finaliz"},
		{"finalizaron", "finaliz"},
		{"finalizó", "finaliz"},
		{"finalmente", "final"},
		{"finamex", "finamex"},
		{"financiadas", "financi"},
		{"financiado", "financi"},
		{"financiados", "financi"},
		{"financial", "financial"},
		{"financiamiento", "financi"},
		{"financiamientos", "financi"},
		{"financiar", "financi"},
		{"financiera", "financ"},
		{"financieras", "financ"},
		{"financiero", "financier"},
		{"financieros", "financier"},
		{"financing", "financing"},
		{"financió", "financ"},
		{"finanzas", "finanz"},
		{"finas", "fin"},
		{"finca", "finc"},
		{"fincado", "finc"},
		{"fincó", "finc"},
		{"fines", "fin"},
		{"fingiendo", "fing"},
		{"fingieron", "fing"},
		{"fingir", "fing"},
		{"finiquitación", "finiquit"},
		{"finiquito", "finiquit"},
		{"finishing", "finishing"},
		{"finland", "finland"},
		{"fino", "fin"},
		{"finos", "fin"},
		{"fira", "fir"},
		{"firma", "firm"},
		{"firmaba", "firm"},
		{"firmaban", "firm"},
		{"firmada", "firm"},
		{"firmadas", "firm"},
		{"firmado", "firm"},
		{"firmados", "firm"},
		{"firmamos", "firm"},
		{"firman", "firm"},
		{"firmante", "firmant"},
		{"firmantes", "firmant"},
		{"firmar", "firm"},
		{"firmara", "firm"},
		{"firmará", "firm"},
		{"firmarán", "firm"},
		{"firmaron", "firm"},
		{"firmarse", "firm"},
		{"firmas", "firm"},
		{"firme", "firm"},
		{"firmemente", "firmement"},
		{"firmen", "firm"},
		{"firmes", "firm"},
		{"firmó", "firm"},
		{"first", "first"},
		{"fiscal", "fiscal"},
		{"fiscales", "fiscal"},
		{"fiscalía", "fiscal"},
		{"fiscalización", "fiscaliz"},
		{"fiscalizado", "fiscaliz"},
		{"fischer", "fisch"},
		{"fisco", "fisc"},
		{"física", "fisic"},
		{"físicamente", "fisic"},
		{"físicas", "fisic"},
		{"físico", "fisic"},
		{"físicodeportiva", "fisicodeport"},
		{"físicos", "fisic"},
		{"fisiológicamente", "fisiolog"},
		{"fisk", "fisk"},
		{"fisonomía", "fisonom"},
		{"fisura", "fisur"},
		{"fitopecuarias", "fitopecuari"},
		{"flaco", "flac"},
		{"flagrante", "flagrant"},
		{"flamante", "flamant"},
		{"flamenco", "flamenc"},
		{"flandes", "fland"},
		{"flannery", "flannery"},
		{"flanquear", "flanqu"},
		{"flaquito", "flaquit"},
		{"flaubert", "flaubert"},
		{"flavio", "flavi"},
		{"flavorite", "flavorit"},
		{"flavr", "flavr"},
		{"flecha", "flech"},
		{"flechazos", "flechaz"},
		{"flecos", "flec"},
		{"flemming", "flemming"},
		{"fleteros", "fleter"},
		{"fletes", "flet"},
		{"flexibilidad", "flexibil"},
		{"flexible", "flexibl"},
		{"flexibles", "flexibl"},
		{"flojos", "floj"},
		{"flor", "flor"},
		{"flora", "flor"},
		{"floración", "floracion"},
		{"florales", "floral"},
		{"florecita", "florecit"},
		{"florencia", "florenci"},
		{"flores", "flor"},
		{"florida", "flor"},
		{"flota", "flot"},
		{"flotación", "flotacion"},
		{"flotando", "flot"},
		{"flotantes", "flotant"},
		{"flotilla", "flotill"},
		{"flotillas", "flotill"},
		{"floyd", "floyd"},
		{"fluctuación", "fluctuacion"},
		{"fluctúan", "fluctu"},
		{"fluidez", "fluidez"},
		{"fluido", "flu"},
		{"fluído", "fluid"},
		{"flujo", "fluj"},
		{"flujos", "fluj"},
		{"flume", "flum"},
		{"fluordaniel", "fluordaniel"},
		{"fluye", "flu"},
		{"fm", "fm"},
		{"fmf", "fmf"},
		{"fmn", "fmn"},
		{"fna", "fna"},
		{"foco", "foc"},
		{"focos", "foc"},
		{"fofo", "fof"},
		{"fogatas", "fogat"},
		{"foggia", "foggi"},
		{"fogueo", "fogue"},
		{"folclórica", "folclor"},
		{"folio", "foli"},
		{"folklore", "folklor"},
		{"folklórica", "folklor"},
		{"folklórico", "folklor"},
		{"folklorismo", "folklor"},
		{"folleto", "follet"},
		{"folletos", "follet"},
		{"fomenta", "foment"},
		{"fomentaba", "foment"},
		{"fomentada", "foment"},
		{"fomentado", "foment"},
		{"fomentan", "foment"},
		{"fomentar", "foment"},
		{"fomento", "foment"},
		{"fomerrey", "fomerrey"},
		{"fonda", "fond"},
		{"fondear", "fond"},
		{"fondistas", "fondist"},
		{"fondo", "fond"},
		{"fondos", "fond"},
		{"fonética", "fonet"},
		{"fonhapo", "fonhap"},
		{"fonméxico", "fonmex"},
		{"fonseca", "fonsec"},
		{"food", "food"},
		{"foods", "foods"},
		{"foquitos", "foquit"},
		{"for", "for"},
		{"foránea", "forane"},
		{"foráneo", "forane"},
		{"foráneos", "forane"},
		{"forcados", "forc"},
		{"force", "forc"},
		{"ford", "ford"},
		{"forest", "forest"},
		{"forestación", "forest"},
		{"forestal", "forestal"},
		{"forestales", "forestal"},
		{"forista", "forist"},
		{"foristas", "forist"},
		{"forja", "forj"},
		{"forjan", "forj"},
		{"forjarse", "forj"},
		{"forma", "form"},
		{"formaban", "form"},
		{"formación", "formacion"},
		{"formada", "form"},
		{"formado", "form"},
		{"formados", "form"},
		{"formal", "formal"},
		{"formales", "formal"},
		{"formalidades", "formal"},
		{"formalización", "formaliz"},
		{"formalizar", "formaliz"},
		{"formalizará", "formaliz"},
		{"formalizarlo", "formaliz"},
		{"formalmente", "formal"},
		{"formamos", "form"},
		{"forman", "form"},
		{"formando", "form"},
		{"formar", "form"},
		{"formará", "form"},
		{"formarán", "form"},
		{"formarlo", "form"},
		{"formaron", "form"},
		{"formarse", "form"},
		{"formas", "form"},
		{"formativos", "format"},
		{"formato", "format"},
		{"formatos", "format"},
		{"forme", "form"},
		{"formen", "form"},
		{"formó", "form"},
		{"fórmula", "formul"},
		{"formulación", "formul"},
		{"formulado", "formul"},
		{"formulan", "formul"},
		{"formular", "formul"},
		{"fórmulas", "formul"},
		{"formuló", "formul"},
		{"fornica", "fornic"},
		{"fornicación", "fornic"},
		{"fornicadora", "fornic"},
		{"fornicadores", "fornic"},
		{"fornicados", "fornic"},
		{"foro", "for"},
		{"foros", "for"},
		{"forrada", "forr"},
		{"forro", "forr"},
		{"fors", "fors"},
		{"fortalece", "fortalec"},
		{"fortalecemos", "fortalec"},
		{"fortalecen", "fortalec"},
		{"fortalecer", "fortalec"},
		{"fortalecidas", "fortalec"},
		{"fortalecido", "fortalec"},
		{"fortaleciendo", "fortalec"},
		{"fortaleciéndose", "fortalec"},
		{"fortalecimiento", "fortalec"},
		{"fortaleció", "fortalec"},
		{"fortaleza", "fortalez"},
		{"fortalezca", "fortalezc"},
		{"fortalezcan", "fortalezc"},
		{"fortín", "fortin"},
		{"fortines", "fortin"},
		{"fortuna", "fortun"},
		{"fortunas", "fortun"},
		{"fortunato", "fortunat"},
		{"forzada", "forz"},
		{"forzadas", "forz"},
		{"forzar", "forz"},
		{"forzaron", "forz"},
		{"forzosa", "forzos"},
		{"forzosamente", "forz"},
		{"forzoso", "forzos"},
		{"fosas", "fos"},
		{"fosforescente", "fosforescent"},
		{"fosforescentes", "fosforescent"},
		{"fósiles", "fosil"},
		{"foster", "fost"},
		{"foto", "fot"},
		{"fotocopiadoras", "fotocopi"},
		{"fotocredencialización", "fotocredencializ"},
		{"", ""},
		{"fotocredencializadas", "fotocredencializ"},
		{"", ""},
		{"fotocredencializados", "fotocredencializ"},
		{"", ""},
		{"fotocredencializar", "fotocredencializ"},
		{"fotografía", "fotograf"},
		{"fotografiar", "fotografi"},
		{"fotografías", "fotograf"},
		{"fotográfica", "fotograf"},
		{"fotográficas", "fotograf"},
		{"fotógrafo", "fotograf"},
		{"fotógrafos", "fotograf"},
		{"fotomecánica", "fotomecan"},
		{"fotoquímicos", "fotoquim"},
		{"fotos", "fot"},
		{"found", "found"},
		{"foundation", "foundation"},
		{"fovi", "fovi"},
		{"fovissste", "fovissst"},
		{"fox", "fox"},
		{"foxpro", "foxpr"},
		{"fracasado", "fracas"},
		{"fracasar", "fracas"},
		{"fracasaron", "fracas"},
		{"fracase", "frac"},
		{"fracaso", "fracas"},
		{"fracasó", "fracas"},
		{"fracasos", "fracas"},
		{"fracc", "fracc"},
		{"fracción", "fraccion"},
		{"fraccionadas", "fraccion"},
		{"fraccionador", "fraccion"},
		{"fraccionadores", "fraccion"},
		{"fraccionamiento", "fraccion"},
		{"fraccionamientos", "fraccion"},
		{"fraccionarios", "fraccionari"},
		{"fracciones", "fraccion"},
		{"fractura", "fractur"},
		{"fracturado", "fractur"},
		{"fracturados", "fractur"},
		{"fracturarse", "fractur"},
		{"fracturas", "fractur"},
		{"fragancias", "fraganci"},
		{"frágil", "fragil"},
		{"frágiles", "fragil"},
		{"fragmenta", "fragment"},
		{"fragmentación", "fragment"},
		{"fragmentos", "fragment"},
		{"fragor", "fragor"},
		{"fraguando", "fragu"},
		{"franca", "franc"},
		{"francamente", "franc"},
		{"francas", "franc"},
		{"france", "franc"},
		{"francés", "frances"},
		{"francesa", "frances"},
		{"franceses", "frances"},
		{"francia", "franci"},
		{"francisco", "francisc"},
		{"franco", "franc"},
		{"francos", "franc"},
		{"francotirador", "francotir"},
		{"franela", "franel"},
		{"frank", "frank"},
		{"franklin", "franklin"},
		{"franklyn", "franklyn"},
		{"franqueó", "franque"},
		{"franqueza", "franquez"},
		{"franquicia", "franquici"},
		{"franquiciaba", "franquici"},
		{"franquicias", "franquici"},
		{"franz", "franz"},
		{"frapi", "frapi"},
		{"frase", "fras"},
		{"fraseo", "frase"},
		{"frases", "fras"},
		{"frasquito", "frasquit"},
		{"fraterna", "fratern"},
		{"fraternal", "fraternal"},
		{"fraternidad", "fratern"},
		{"fraude", "fraud"},
		{"fraudes", "fraud"},
		{"fraudulenta", "fraudulent"},
		{"fraudulentas", "fraudulent"},
		{"fraudulento", "fraudulent"},
		{"frausto", "fraust"},
		{"frayre", "frayr"},
		{"frecuencia", "frecuenci"},
		{"frecuencias", "frecuenci"},
		{"frecuentada", "frecuent"},
		{"frecuentado", "frecuent"},
		{"frecuentan", "frecuent"},
		{"frecuentando", "frecuent"},
		{"frecuentaran", "frecuent"},
		{"frecuente", "frecuent"},
		{"frecuentemente", "frecuent"},
		{"frecuentes", "frecuent"},
		{"frederick", "frederick"},
		{"freedom", "freedom"},
		{"freeport", "freeport"},
		{"fregado", "freg"},
		{"fregonería", "fregon"},
		{"frenado", "fren"},
		{"frenados", "fren"},
		{"frenar", "fren"},
		{"freno", "fren"},
		{"frenó", "fren"},
		{"frenos", "fren"},
		{"frente", "frent"},
		{"frentes", "frent"},
		{"fresas", "fres"},
		{"fresca", "fresc"},
		{"frescas", "fresc"},
		{"fresco", "fresc"},
		{"frescura", "frescur"},
		{"fresh", "fresh"},
		{"freud", "freud"},
		{"fría", "fri"},
		{"frias", "fri"},
		{"frías", "fri"},
		{"fricase", "fric"},
		{"fricciones", "friccion"},
		{"frida", "frid"},
		{"fried", "fri"},
		{"frijol", "frijol"},
		{"frijoles", "frijol"},
		{"frío", "fri"},
		{"friolentas", "friolent"},
		{"friolentos", "friolent"},
		{"fríos", "fri"},
		{"frisco", "frisc"},
		{"fritos", "frit"},
		{"fritz", "fritz"},
		{"frivolidad", "frivol"},
		{"frontal", "frontal"},
		{"frontalmente", "frontal"},
		{"frontera", "fronter"},
		{"fronteras", "fronter"},
		{"fronteriza", "fronteriz"},
		{"fronterizo", "fronteriz"},
		{"fronterizos", "fronteriz"},
		{"fructificar", "fructific"},
		{"fructuoso", "fructuos"},
		{"frustración", "frustracion"},
		{"frustrada", "frustr"},
		{"frustradas", "frustr"},
		{"frustrado", "frustr"},
		{"fruta", "frut"},
		{"frutales", "frutal"},
		{"frutas", "frut"},
		{"fruto", "frut"},
		{"frutos", "frut"},
		{"fstse", "fstse"},
		{"fucsia", "fucsi"},
		{"fue", "fue"},
		{"fuego", "fueg"},
		{"fuehrer", "fuehr"},
		{"fuente", "fuent"},
		{"fuentes", "fuent"},
		{"fuera", "fuer"},
		{"fueran", "fuer"},
		{"fuere", "fuer"},
		{"fuero", "fuer"},
		{"fueron", "fueron"},
		{"fuerte", "fuert"},
		{"fuertemente", "fuertement"},
		{"fuertes", "fuert"},
		{"fuerza", "fuerz"},
		{"fuerzas", "fuerz"},
		{"fuese", "fues"},
		{"fuesen", "fues"},
		{"fuezas", "fuez"},
		{"fuga", "fug"},
		{"fui", "fui"},
		{"fuiste", "fuist"},
		{"fujimori", "fujimori"},
		{"fujimorista", "fujimor"},
		{"fukuoka", "fukuok"},
		{"fulano", "fulan"},
		{"fulgor", "fulgor"},
		{"fuller", "full"},
		{"fulminante", "fulmin"},
		{"fulminantemente", "fulmin"},
		{"fulminó", "fulmin"},
		{"fuman", "fum"},
		{"fumando", "fum"},
		{"fumarolas", "fumarol"},
		{"función", "funcion"},
		{"funciona", "funcion"},
		{"funcionaba", "funcion"},
		{"funcional", "funcional"},
		{"funcionalidad", "funcional"},
		{"funcionalidades", "funcional"},
		{"funcionamiento", "funcion"},
		{"funcionan", "funcion"},
		{"funcionando", "funcion"},
		{"funcionar", "funcion"},
		{"funcionará", "funcion"},
		{"funcionaran", "funcion"},
		{"funcionarán", "funcion"},
		{"funcionaria", "funcionari"},
		{"funcionarias", "funcionari"},
		{"funcionario", "funcionari"},
		{"funcionarios", "funcionari"},
		{"funcionaron", "funcion"},
		{"funcione", "funcion"},
		{"funciones", "funcion"},
		{"funcionó", "funcion"},
		{"fund", "fund"},
		{"funda", "fund"},
		{"fundación", "fundacion"},
		{"fundaciones", "fundacion"},
		{"fundada", "fund"},
		{"fundado", "fund"},
		{"fundador", "fundador"},
		{"fundadora", "fundador"},
		{"fundadoras", "fundador"},
		{"fundadores", "fundador"},
		{"fundamenta", "fundament"},
		{"fundamentados", "fundament"},
		{"fundamental", "fundamental"},
		{"fundamentales", "fundamental"},
		{"fundamentalismo", "fundamental"},
		{"fundamentalmente", "fundamental"},
		{"fundamento", "fundament"},
		{"fundamentos", "fundament"},
		{"fundar", "fund"},
		{"fundara", "fund"},
		{"fundaran", "fund"},
		{"fundarán", "fund"},
		{"fundaron", "fund"},
		{"fundarse", "fund"},
		{"funde", "fund"},
		{"funden", "fund"},
		{"fundente", "fundent"},
		{"fundición", "fundicion"},
		{"fundiciones", "fundicion"},
		{"fundidora", "fundidor"},
		{"fundó", "fund"},
		{"funerales", "funeral"},
		{"funge", "fung"},
		{"fungía", "fung"},
		{"fungían", "fung"},
		{"fungió", "fung"},
		{"fungir", "fung"},
		{"funk", "funk"},
		{"funny", "funny"},
		{"furgones", "furgon"},
		{"furia", "furi"},
		{"furibunda", "furibund"},
		{"furibundo", "furibund"},
		{"furioso", "furios"},
		{"furtiva", "furtiv"},
		{"furtivo", "furtiv"},
		{"furtivos", "furtiv"},
		{"fusión", "fusion"},
		{"fusionada", "fusion"},
		{"fusiones", "fusion"},
		{"futbol", "futbol"},
		{"futbolista", "futbol"},
		{"futbolistas", "futbol"},
		{"futbolística", "futbolist"},
		{"futbolísticamente", "futbolist"},
		{"futbolístico", "futbolist"},
		{"futbolísticos", "futbolist"},
		{"futura", "futur"},
		{"futuras", "futur"},
		{"futureando", "futur"},
		{"futurismo", "futur"},
		{"futurista", "futur"},
		{"futuro", "futur"},
		{"futuros", "futur"},
		{"fxg", "fxg"},
		{"g", "g"},
		{"gabanna", "gabann"},
		{"gabinete", "gabinet"},
		{"gabriel", "gabriel"},
		{"gabriela", "gabriel"},
		{"gado", "gad"},
		{"gafete", "gafet"},
		{"gail", "gail"},
		{"gala", "gal"},
		{"galán", "galan"},
		{"galanes", "galan"},
		{"galante", "galant"},
		{"galardón", "galardon"},
		{"galardonada", "galardon"},
		{"galardonado", "galardon"},
		{"galardonados", "galardon"},
		{"galardones", "galardon"},
		{"galarza", "galarz"},
		{"galeana", "galean"},
		{"galenos", "galen"},
		{"galeones", "galeon"},
		{"galeras", "galer"},
		{"galería", "gal"},
		{"galerías", "gal"},
		{"gales", "gal"},
		{"galicia", "galici"},
		{"galileo", "galile"},
		{"galimanty", "galimanty"},
		{"galindo", "galind"},
		{"gallardo", "gallard"},
		{"gallego", "galleg"},
		{"gallegos", "galleg"},
		{"galleta", "gallet"},
		{"galletitas", "galletit"},
		{"gallina", "gallin"},
		{"gallinas", "gallin"},
		{"gallinero", "galliner"},
		{"gallo", "gall"},
		{"gallont", "gallont"},
		{"gallos", "gall"},
		{"galvak", "galvak"},
		{"galvamet", "galvamet"},
		{"galván", "galvan"},
		{"galvanizada", "galvaniz"},
		{"gama", "gam"},
		{"gámez", "gamez"},
		{"gamma", "gamm"},
		{"gamuza", "gamuz"},
		{"gana", "gan"},
		{"ganaban", "gan"},
		{"ganada", "gan"},
		{"ganadas", "gan"},
		{"ganadera", "ganader"},
		{"ganadería", "ganad"},
		{"ganadero", "ganader"},
		{"ganaderos", "ganader"},
		{"ganado", "gan"},
		{"ganador", "ganador"},
		{"ganadora", "ganador"},
		{"ganadores", "ganador"},
		{"ganados", "gan"},
		{"ganamos", "gan"},
		{"ganan", "gan"},
		{"ganancia", "gananci"},
		{"ganancias", "gananci"},
		{"ganando", "gan"},
		{"ganar", "gan"},
		{"ganará", "gan"},
		{"ganarán", "gan"},
		{"ganarla", "gan"},
		{"ganarle", "gan"},
		{"ganarles", "gan"},
		{"ganaron", "gan"},
		{"ganarse", "gan"},
		{"ganas", "gan"},
		{"gancho", "ganch"},
		{"ganchó", "ganch"},
		{"ganchos", "ganch"},
		{"gane", "gan"},
		{"ganen", "gan"},
		{"gangas", "gang"},
		{"gannet", "gannet"},
		{"ganó", "gan"},
		{"gaona", "gaon"},
		{"garantía", "garant"},
		{"garantías", "garant"},
		{"garantice", "garantic"},
		{"garantices", "garantic"},
		{"garantiza", "garantiz"},
		{"garantizaba", "garantiz"},
		{"garantizaban", "garantiz"},
		{"garantizada", "garantiz"},
		{"garantizadas", "garantiz"},
		{"garantizar", "garantiz"},
		{"garantizarán", "garantiz"},
		{"garantizaron", "garantiz"},
		{"garantizarse", "garantiz"},
		{"garantizo", "garantiz"},
		{"garcía", "garc"},
		{"gardenia", "gardeni"},
		{"garduño", "garduñ"},
		{"garfield", "garfield"},
		{"garganta", "gargant"},
		{"gargantilla", "gargantill"},
		{"gari", "gari"},
		{"garibay", "garibay"},
		{"garitos", "garit"},
		{"garma", "garm"},
		{"garras", "garr"},
		{"garrido", "garr"},
		{"garrison", "garrison"},
		{"garrocha", "garroch"},
		{"garrotero", "garroter"},
		{"garroteros", "garroter"},
		{"gary", "gary"},
		{"garza", "garz"},
		{"gas", "gas"},
		{"gasca", "gasc"},
		{"gasera", "gaser"},
		{"gaseras", "gaser"},
		{"gases", "gas"},
		{"gasoducto", "gasoduct"},
		{"gasóleo", "gasole"},
		{"gasolina", "gasolin"},
		{"gasolinas", "gasolin"},
		{"gasolinera", "gasoliner"},
		{"gasolinerías", "gasolin"},
		{"gaspar", "gasp"},
		{"gasparini", "gasparini"},
		{"gasset", "gasset"},
		{"gastado", "gast"},
		{"gastados", "gast"},
		{"gastando", "gast"},
		{"gastar", "gast"},
		{"gastarán", "gast"},
		{"gastarla", "gast"},
		{"gastaron", "gast"},
		{"gaste", "gast"},
		{"gasto", "gast"},
		{"gastó", "gast"},
		{"gastolfo", "gastolf"},
		{"gastón", "gaston"},
		{"gastona", "gaston"},
		{"gastos", "gast"},
		{"gastritis", "gastritis"},
		{"gastroenterología", "gastroenterolog"},
		{"gatauline", "gataulin"},
		{"gates", "gat"},
		{"gatica", "gatic"},
		{"gato", "gat"},
		{"gatorade", "gatorad"},
		{"gatos", "gat"},
		{"gatt", "gatt"},
		{"gaucho", "gauch"},
		{"gauna", "gaun"},
		{"gavaldon", "gavaldon"},
		{"gavaldón", "gavaldon"},
		{"gavica", "gavic"},
		{"gaviria", "gaviri"},
		{"gaytán", "gaytan"},
		{"gazal", "gazal"},
		{"gds", "gds"},
		{"gehard", "gehard"},
		{"geiser", "geis"},
		{"gelasio", "gelasi"},
		{"gélida", "gel"},
		{"gélido", "gel"},
		{"gemebundo", "gemebund"},
		{"gemelas", "gemel"},
		{"gemelos", "gemel"},
		{"gemido", "gem"},
		{"gemidos", "gem"},
		{"genaro", "genar"},
		{"gendarme", "gend"},
		{"gene", "gen"},
		{"genealógico", "genealog"},
		{"genera", "gener"},
		{"generación", "gener"},
		{"generaciones", "gener"},
		{"generada", "gener"},
		{"generado", "gener"},
		{"generador", "gener"},
		{"generadora", "gener"},
		{"generadores", "gener"},
		{"generados", "gener"},
		{"general", "general"},
		{"generales", "general"},
		{"generalidad", "general"},
		{"generalista", "general"},
		{"generalizaciones", "generaliz"},
		{"generalizada", "generaliz"},
		{"generalizado", "generaliz"},
		{"generalizados", "generaliz"},
		{"generalizar", "generaliz"},
		{"generalmente", "general"},
		{"generan", "gener"},
		{"generando", "gener"},
		{"generandoles", "gener"},
		{"generar", "gener"},
		{"generará", "gener"},
		{"generarán", "gener"},
		{"generaron", "gener"},
		{"generarse", "gener"},
		{"genérico", "gener"},
		{"generis", "generis"},
		{"género", "gener"},
		{"generó", "gener"},
		{"géneros", "gener"},
		{"generosa", "gener"},
		{"generosas", "gener"},
		{"generoso", "gener"},
		{"genes", "gen"},
		{"genesio", "genesi"},
		{"genetec", "genetec"},
		{"genética", "genet"},
		{"genéticamente", "genet"},
		{"genético", "genet"},
		{"genéticos", "genet"},
		{"genetistas", "genet"},
		{"genial", "genial"},
		{"geniales", "genial"},
		{"genio", "geni"},
		{"genios", "geni"},
		{"genital", "genital"},
		{"genitales", "genital"},
		{"geno", "gen"},
		{"genocidio", "genocidi"},
		{"genpham", "genpham"},
		{"gente", "gent"},
		{"gentes", "gent"},
		{"gentor", "gentor"},
		{"genuinamente", "genuin"},
		{"geo", "geo"},
		{"geográficas", "geograf"},
		{"geográfico", "geograf"},
		{"geographic", "geographic"},
		{"geológicos", "geolog"},
		{"geólogos", "geolog"},
		{"george", "georg"},
		{"georges", "georg"},
		{"gephard", "gephard"},
		{"gephardt", "gephardt"},
		{"gerald", "gerald"},
		{"gerardo", "gerard"},
		{"gerda", "gerd"},
		{"gerencia", "gerenci"},
		{"gerencial", "gerencial"},
		{"gerenciales", "gerencial"},
		{"gerencias", "gerenci"},
		{"gerente", "gerent"},
		{"gerentes", "gerent"},
		{"germán", "german"},
		{"germen", "germ"},
		{"geroncio", "geronci"},
		{"gerónimo", "geronim"},
		{"gestado", "gest"},
		{"gestando", "gest"},
		{"gestión", "gestion"},
		{"gestionando", "gestion"},
		{"gestionar", "gestion"},
		{"gestiones", "gestion"},
		{"gesto", "gest"},
		{"gestor", "gestor"},
		{"gestoras", "gestor"},
		{"gestoría", "gestor"},
		{"gfinbur", "gfinbur"},
		{"ghali", "ghali"},
		{"ghana", "ghan"},
		{"ghanés", "ghanes"},
		{"ghersi", "ghersi"},
		{"giancana", "giancan"},
		{"gianni", "gianni"},
		{"gielgud", "gielgud"},
		{"gigante", "gigant"},
		{"gigantes", "gigant"},
		{"gigantesco", "gigantesc"},
		{"gigantescos", "gigantesc"},
		{"gigantezca", "gigantezc"},
		{"gijón", "gijon"},
		{"gil", "gil"},
		{"gilbert", "gilbert"},
		{"gilberto", "gilbert"},
		{"gime", "gim"},
		{"gimnasia", "gimnasi"},
		{"gimnasio", "gimnasi"},
		{"gimnospermas", "gimnosperm"},
		{"ginebra", "ginebr"},
		{"ginecobstetricia", "ginecobstetrici"},
		{"ginecólogo", "ginecolog"},
		{"giovanni", "giovanni"},
		{"gira", "gir"},
		{"giraba", "gir"},
		{"girado", "gir"},
		{"giran", "gir"},
		{"girar", "gir"},
		{"girara", "gir"},
		{"girará", "gir"},
		{"giraron", "gir"},
		{"giras", "gir"},
		{"giro", "gir"},
		{"giró", "gir"},
		{"girolamo", "girolam"},
		{"girolano", "girolan"},
		{"giros", "gir"},
		{"gises", "gis"},
		{"giuseppe", "giusepp"},
		{"gladiador", "gladiador"},
		{"gladiadores", "gladiador"},
		{"glamorosa", "glamor"},
		{"glamour", "glamour"},
		{"glamur", "glamur"},
		{"glándula", "glandul"},
		{"glen", "glen"},
		{"glenn", "glenn"},
		{"glioblastomas", "glioblastom"},
		{"global", "global"},
		{"globales", "global"},
		{"globalización", "globaliz"},
		{"globalizado", "globaliz"},
		{"globalizarse", "globaliz"},
		{"globe", "glob"},
		{"globero", "glober"},
		{"globo", "glob"},
		{"globos", "glob"},
		{"gloria", "glori"},
		{"glorias", "glori"},
		{"glorieta", "gloriet"},
		{"gloriosísima", "gloriosisim"},
		{"glosas", "glos"},
		{"goave", "goav"},
		{"gober", "gob"},
		{"gobernaban", "gobern"},
		{"gobernabilidad", "gobern"},
		{"gobernación", "gobern"},
		{"gobernada", "gobern"},
		{"gobernado", "gobern"},
		{"gobernador", "gobern"},
		{"gobernadora", "gobern"},
		{"gobernadores", "gobern"},
		{"gobernante", "gobern"},
		{"gobernantes", "gobern"},
		{"gobernar", "gobern"},
		{"gobernarse", "gobern"},
		{"gobernícolas", "gobernicol"},
		{"gobernó", "gobern"},
		{"gobienro", "gobienr"},
		{"gobierna", "gobiern"},
		{"gobierne", "gobiern"},
		{"gobierno", "gobiern"},
		{"gobiernos", "gobi"},
		{"goce", "goc"},
		{"gocen", "goc"},
		{"goddard", "goddard"},
		{"godínez", "godinez"},
		{"goering", "goering"},
		{"goeter", "goet"},
		{"goethe", "goeth"},
		{"gol", "gol"},
		{"golazo", "golaz"},
		{"golden", "gold"},
		{"goldsmith", "goldsmith"},
		{"goleador", "goleador"},
		{"goleadora", "goleador"},
		{"goleadores", "goleador"},
		{"golearon", "gol"},
		{"goleo", "gole"},
		{"goles", "gol"},
		{"golf", "golf"},
		{"golfistas", "golfist"},
		{"golfo", "golf"},
		{"golondrinas", "golondrin"},
		{"golpe", "golp"},
		{"golpeaba", "golp"},
		{"golpeado", "golp"},
		{"golpeador", "golpeador"},
		{"golpeados", "golp"},
		{"golpean", "golp"},
		{"golpear", "golp"},
		{"golpeara", "golp"},
		{"golpearlo", "golp"},
		{"golpearon", "golp"},
		{"golpeo", "golpe"},
		{"golpeó", "golpe"},
		{"golpes", "golp"},
		{"golpiza", "golpiz"},
		{"goma", "gom"},
		{"gomes", "gom"},
		{"gómez", "gomez"},
		{"gomiz", "gomiz"},
		{"gonzalez", "gonzalez"},
		{"gonzález", "gonzalez"},
		{"gonzalitos", "gonzalit"},
		{"gonzalo", "gonzal"},
		{"goosen", "goos"},
		{"gor", "gor"},
		{"gorbachov", "gorbachov"},
		{"gorda", "gord"},
		{"gordillo", "gordill"},
		{"gordo", "gord"},
		{"gordolobo", "gordolob"},
		{"gordon", "gordon"},
		{"gorgona", "gorgon"},
		{"gorgonia", "gorgoni"},
		{"gorila", "goril"},
		{"gorilas", "goril"},
		{"gorios", "gori"},
		{"gorra", "gorr"},
		{"gorras", "gorr"},
		{"gorro", "gorr"},
		{"gortari", "gortari"},
		{"gortázar", "gortaz"},
		{"gorvachov", "gorvachov"},
		{"gota", "got"},
		{"gotas", "got"},
		{"goteo", "gote"},
		{"goteras", "goter"},
		{"gourmet", "gourmet"},
		{"gourmets", "gourmets"},
		{"goya", "goy"},
		{"goyesca", "goyesc"},
		{"goyesco", "goyesc"},
		{"goyo", "goy"},
		{"goza", "goz"},
		{"gozado", "goz"},
		{"gozan", "goz"},
		{"gozar", "goz"},
		{"gozo", "goz"},
		{"gozoso", "gozos"},
		{"graba", "grab"},
		{"grababan", "grab"},
		{"grabación", "grabacion"},
		{"grabaciones", "grabacion"},
		{"grabada", "grab"},
		{"grabadas", "grab"},
		{"grabadista", "grabad"},
		{"grabado", "grab"},
		{"grabadora", "grabador"},
		{"grabadoras", "grabador"},
		{"grabados", "grab"},
		{"grabamos", "grab"},
		{"graban", "grab"},
		{"grabar", "grab"},
		{"grabara", "grab"},
		{"grabará", "grab"},
		{"grabarle", "grab"},
		{"grabaron", "grab"},
		{"grabó", "grab"},
		{"gracia", "graci"},
		{"graciano", "gracian"},
		{"gracias", "graci"},
		{"graciela", "graciel"},
		{"gracilmente", "gracil"},
		{"gradas", "grad"},
		{"graderío", "graderi"},
		{"grado", "grad"},
		{"gradocon", "gradocon"},
		{"grados", "grad"},
		{"graduación", "graduacion"},
		{"graduado", "gradu"},
		{"graduados", "gradu"},
		{"gradual", "gradual"},
		{"gradualmente", "gradual"},
		{"graduaron", "gradu"},
		{"graduarse", "gradu"},
		{"grafenberg", "grafenberg"},
		{"grafiar", "grafi"},
		{"gráfica", "grafic"},
		{"gráficas", "grafic"},
		{"gráfico", "grafic"},
		{"grafito", "grafit"},
		{"grajales", "grajal"},
		{"grajeda", "grajed"},
		{"gramajo", "gramaj"},
		{"grammont", "grammont"},
		{"grammy", "grammy"},
		{"grammys", "grammys"},
		{"gramos", "gram"},
		{"grampus", "grampus"},
		{"gran", "gran"},
		{"grana", "gran"},
		{"graña", "grañ"},
		{"granada", "gran"},
		{"granadas", "gran"},
		{"granaderos", "granader"},
		{"granados", "gran"},
		{"granalguina", "granalguin"},
		{"granata", "granat"},
		{"granate", "granat"},
		{"grand", "grand"},
		{"grande", "grand"},
		{"grandecito", "grandecit"},
		{"grandes", "grand"},
		{"grandeza", "grandez"},
		{"grandguillhomme", "grandguillhomm"},
		{"grandiosa", "grandios"},
		{"grandiosas", "grandi"},
		{"grandioso", "grandios"},
		{"grandísimo", "grandisim"},
		{"grandota", "grandot"},
		{"grandote", "grandot"},
		{"grandotes", "grandot"},
		{"granero", "graner"},
		{"granillo", "granill"},
		{"granito", "granit"},
		{"granja", "granj"},
		{"granjas", "granj"},
		{"granjeros", "granjer"},
		{"granma", "granm"},
		{"grano", "gran"},
		{"granos", "gran"},
		{"gratificaciones", "gratif"},
		{"gratificante", "gratif"},
		{"gratis", "gratis"},
		{"gratitud", "gratitud"},
		{"grato", "grat"},
		{"gratuita", "gratuit"},
		{"gratuitamente", "gratuit"},
		{"gratuito", "gratuit"},
		{"gravados", "grav"},
		{"gravamen", "gravam"},
		{"gravámenes", "gravamen"},
		{"gravar", "grav"},
		{"grave", "grav"},
		{"gravedad", "graved"},
		{"gravemente", "gravement"},
		{"graves", "grav"},
		{"gravísimo", "gravisim"},
		{"gravitacionales", "gravitacional"},
		{"gravoso", "gravos"},
		{"gray", "gray"},
		{"greatest", "greatest"},
		{"grecia", "greci"},
		{"grecorromano", "grecorroman"},
		{"greenbelt", "greenbelt"},
		{"greenspan", "greensp"},
		{"greg", "greg"},
		{"gregorianos", "gregorian"},
		{"gregorio", "gregori"},
		{"gregory", "gregory"},
		{"greña", "greñ"},
		{"griega", "grieg"},
		{"griego", "grieg"},
		{"griegos", "grieg"},
		{"grieta", "griet"},
		{"grijalva", "grijalv"},
		{"grill", "grill"},
		{"grilletes", "grillet"},
		{"gripa", "grip"},
		{"gripal", "gripal"},
		{"gripe", "grip"},
		{"gris", "gris"},
		{"griselda", "griseld"},
		{"grises", "gris"},
		{"grita", "grit"},
		{"gritaba", "grit"},
		{"gritaban", "grit"},
		{"gritando", "grit"},
		{"gritar", "grit"},
		{"gritarle", "grit"},
		{"gritaron", "grit"},
		{"gritería", "grit"},
		{"grito", "grit"},
		{"gritó", "grit"},
		{"gritos", "grit"},
		{"gro", "gro"},
		{"grosera", "groser"},
		{"grosería", "gros"},
		{"groserías", "gros"},
		{"grotewold", "grotewold"},
		{"group", "group"},
		{"grrrrr", "grrrrr"},
		{"grúa", "gru"},
		{"grúas", "gru"},
		{"gruending", "gruending"},
		{"gruesa", "grues"},
		{"grueso", "grues"},
		{"gruesos", "grues"},
		{"grumos", "grum"},
		{"grupo", "grup"},
		{"grupos", "grup"},
		{"guacamole", "guacamol"},
		{"guadalajara", "guadalaj"},
		{"guadalcanal", "guadalcanal"},
		{"guadalupe", "guadalup"},
		{"guajardo", "guajard"},
		{"guamerú", "guameru"},
		{"guan", "guan"},
		{"guanajuatense", "guanajuatens"},
		{"guanajuato", "guanajuat"},
		{"guango", "guang"},
		{"guantes", "guant"},
		{"guapa", "guap"},
		{"guapo", "guap"},
		{"guarda", "guard"},
		{"guardacostas", "guardacost"},
		{"guardada", "guard"},
		{"guardadita", "guardadit"},
		{"guardados", "guard"},
		{"guardameta", "guardamet"},
		{"guardan", "guard"},
		{"guardar", "guard"},
		{"guardará", "guard"},
		{"guardaran", "guard"},
		{"guardarán", "guard"},
		{"guardarlas", "guard"},
		{"guardarropa", "guardarrop"},
		{"guardarse", "guard"},
		{"guardas", "guard"},
		{"guardería", "guard"},
		{"guardia", "guardi"},
		{"guardián", "guardian"},
		{"guardias", "guardi"},
		{"guardó", "guard"},
		{"guat", "guat"},
		{"guatemala", "guatemal"},
		{"guatemalteco", "guatemaltec"},
		{"guatemaltecos", "guatemaltec"},
		{"gubernamental", "gubernamental"},
		{"gubernamentales", "gubernamental"},
		{"gubernatura", "gubernatur"},
		{"guccione", "guccion"},
		{"güedad", "güed"},
		{"güera", "güer"},
		{"güerito", "güerit"},
		{"güeritos", "güerit"},
		{"güero", "güer"},
		{"guerra", "guerr"},
		{"guerras", "guerr"},
		{"guerrerense", "guerrerens"},
		{"guerreresnses", "guerreresns"},
		{"guerrero", "guerrer"},
		{"guerrilla", "guerrill"},
		{"guerrillera", "guerriller"},
		{"guerrillero", "guerriller"},
		{"guerrilleros", "guerriller"},
		{"guetiérrez", "guetierrez"},
		{"guevara", "guev"},
		{"guevarista", "guevar"},
		{"guía", "gui"},
		{"guiada", "gui"},
		{"guiadas", "gui"},
		{"guiar", "gui"},
		{"guiaron", "gui"},
		{"guíe", "gui"},
		{"guillermina", "guillermin"},
		{"guillermo", "guillerm"},
		{"guillespie", "guillespi"},
		{"guillotina", "guillotin"},
		{"guío", "gui"},
		{"guiomar", "guiom"},
		{"guión", "guion"},
		{"guiones", "guion"},
		{"guitarra", "guitarr"},
		{"guitarrista", "guitarr"},
		{"guns", "guns"},
		{"gurmets", "gurmets"},
		{"gurría", "gurr"},
		{"gurrola", "gurrol"},
		{"gusano", "gusan"},
		{"gusta", "gust"},
		{"gustaba", "gust"},
		{"gustaban", "gust"},
		{"gustado", "gust"},
		{"gustamos", "gust"},
		{"gustan", "gust"},
		{"gustando", "gust"},
		{"gustar", "gust"},
		{"gustaría", "gust"},
		{"gustavo", "gustav"},
		{"guste", "gust"},
		{"gusten", "gust"},
		{"gusto", "gust"},
		{"gustó", "gust"},
		{"gustos", "gust"},
		{"gutierrez", "gutierrez"},
		{"gutiérrez", "gutierrez"},
		{"guzmán", "guzman"},
		{"gyetko", "gyetk"},
		{"h", "h"},
		{"ha", "ha"},
		{"habana", "haban"},
		{"haber", "hab"},
		{"haberlas", "hab"},
		{"haberle", "hab"},
		{"haberles", "hab"},
		{"haberlo", "hab"},
		{"haberlos", "hab"},
		{"haberse", "hab"},
		{"había", "hab"},
		{"habíamos", "hab"},
		{"habían", "hab"},
		{"habidas", "hab"},
		{"habido", "hab"},
		{"habidos", "hab"},
		{"habiendo", "hab"},
		{"hábil", "habil"},
		{"hábiles", "habil"},
		{"habilidad", "habil"},
		{"habilidades", "habil"},
		{"habilitado", "habilit"},
		{"habilitar", "habilit"},
		{"habita", "habit"},
		{"habitaban", "habit"},
		{"habitables", "habit"},
		{"habitación", "habit"},
		{"habitacional", "habitacional"},
		{"habitaciones", "habit"},
		{"habitadas", "habit"},
		{"habitado", "habit"},
		{"habitantes", "habit"},
		{"habitarán", "habit"},
		{"hábito", "habit"},
		{"hábitos", "habit"},
		{"habituados", "habitu"},
		{"habitual", "habitual"},
		{"habla", "habl"},
		{"hablaba", "habl"},
		{"hablábamos", "habl"},
		{"hablado", "habl"},
		{"hablador", "hablador"},
		{"hablados", "habl"},
		{"háblame", "hablam"},
		{"hablamos", "habl"},
		{"hablan", "habl"},
		{"hablando", "habl"},
		{"hablándose", "habl"},
		{"hablar", "habl"},
		{"hablara", "habl"},
		{"hablará", "habl"},
		{"hablarán", "habl"},
		{"hablarle", "habl"},
		{"hablarme", "habl"},
		{"hablarnos", "habl"},
		{"hablaron", "habl"},
		{"hable", "habl"},
		{"hablé", "habl"},
		{"hablen", "habl"},
		{"hablo", "habl"},
		{"habló", "habl"},
		{"habra", "habr"},
		{"habrá", "habr"},
		{"habrán", "habran"},
		{"habre", "habr"},
		{"habremos", "habr"},
		{"habría", "habr"},
		{"habríamos", "habr"},
		{"habrían", "habr"},
		{"hace", "hac"},
		{"hacemos", "hac"},
		{"hacen", "hac"},
		{"hacendarias", "hacendari"},
		{"hacendario", "hacendari"},
		{"hacendoso", "hacend"},
		{"hacer", "hac"},
		{"hacerla", "hac"},
		{"hacerle", "hac"},
		{"hacerles", "hac"},
		{"hacerlo", "hac"},
		{"hacerlos", "hac"},
		{"hacerme", "hac"},
		{"hacernos", "hac"},
		{"hacerse", "hac"},
		{"haces", "hac"},
		{"hacha", "hach"},
		{"hacia", "haci"},
		{"hacía", "hac"},
		{"hacíamos", "hac"},
		{"hacían", "hac"},
		{"hacienda", "haciend"},
		{"haciendas", "haciend"},
		{"haciendo", "hac"},
		{"haciéndola", "hac"},
		{"haciéndolas", "hac"},
		{"haciéndoles", "hac"},
		{"haciéndolo", "hac"},
		{"haciéndolos", "hac"},
		{"haciéndose", "hac"},
		{"hacinamiento", "hacin"},
		{"haddad", "hadd"},
		{"haemophilus", "haemophilus"},
		{"haga", "hag"},
		{"hágalas", "hagal"},
		{"hágalo", "hagal"},
		{"hagamos", "hag"},
		{"hagámosla", "hagamosl"},
		{"hagámoslo", "hagamosl"},
		{"hagan", "hag"},
		{"hago", "hag"},
		{"hainan", "hain"},
		{"haití", "hait"},
		{"haitianos", "haitian"},
		{"halagadora", "halag"},
		{"halago", "halag"},
		{"halcion", "halcion"},
		{"halcón", "halcon"},
		{"halen", "hal"},
		{"hall", "hall"},
		{"halla", "hall"},
		{"hallaba", "hall"},
		{"hallado", "hall"},
		{"hallados", "hall"},
		{"hallamos", "hall"},
		{"hallan", "hall"},
		{"hallar", "hall"},
		{"hallarle", "hall"},
		{"hallazgo", "hallazg"},
		{"hallazgos", "hallazg"},
		{"halle", "hall"},
		{"halley", "halley"},
		{"halógeno", "halogen"},
		{"hamaca", "hamac"},
		{"hambre", "hambr"},
		{"hambres", "hambr"},
		{"hambriento", "hambrient"},
		{"hambrientos", "hambrient"},
		{"hamburguesa", "hamburgues"},
		{"hamburguesas", "hamburgues"},
		{"hamel", "hamel"},
		{"hamilton", "hamilton"},
		{"hampa", "hamp"},
		{"hamponio", "hamponi"},
		{"han", "han"},
		{"hanchette", "hanchett"},
		{"handicap", "handicap"},
		{"hangar", "hang"},
		{"hank", "hank"},
		{"hannover", "hannov"},
		{"hanover", "hanov"},
		{"hans", "hans"},
		{"hanse", "hans"},
		{"hará", "har"},
		{"harald", "harald"},
		{"harán", "haran"},
		{"harás", "haras"},
		{"harbaugh", "harbaugh"},
		{"hard", "hard"},
		{"hardcore", "hardcor"},
		{"hardware", "hardwar"},
		{"haré", "har"},
		{"harefield", "harefield"},
		{"haremos", "har"},
		{"haría", "har"},
		{"haríamos", "har"},
		{"harían", "har"},
		{"harina", "harin"},
		{"harlem", "harlem"},
		{"haro", "har"},
		{"harper", "harp"},
		{"harry", "harry"},
		{"hartazgo", "hartazg"},
		{"hartmut", "hartmut"},
		{"harto", "hart"},
		{"harvard", "harvard"},
		{"harvest", "harvest"},
		{"harvey", "harvey"},
		{"has", "has"},
		{"hassan", "hass"},
		{"hasta", "hast"},
		{"haugen", "haug"},
		{"hausser", "hauss"},
		{"havelange", "havelang"},
		{"hawai", "hawai"},
		{"hawaiano", "hawaian"},
		{"hay", "hay"},
		{"haya", "hay"},
		{"hayamos", "hay"},
		{"hayan", "hay"},
		{"hayas", "hay"},
		{"hayat", "hayat"},
		{"haz", "haz"},
		{"hazaña", "hazañ"},
		{"hboc", "hboc"},
		{"hc", "hc"},
		{"he", "he"},
		{"headsh", "headsh"},
		{"health", "health"},
		{"heart", "heart"},
		{"hearts", "hearts"},
		{"heavy", "heavy"},
		{"heberto", "hebert"},
		{"hebilla", "hebill"},
		{"hebillas", "hebill"},
		{"hecha", "hech"},
		{"hechas", "hech"},
		{"hechizada", "hechiz"},
		{"hecho", "hech"},
		{"hechos", "hech"},
		{"hectárea", "hectare"},
		{"hectáreas", "hectar"},
		{"hector", "hector"},
		{"héctor", "hector"},
		{"hedrick", "hedrick"},
		{"hegel", "hegel"},
		{"heidi", "heidi"},
		{"heladas", "hel"},
		{"helado", "hel"},
		{"helados", "hel"},
		{"helen", "hel"},
		{"helicóptero", "helicopter"},
		{"helicópteros", "helicopter"},
		{"hembras", "hembr"},
		{"hemerográficas", "hemerograf"},
		{"hemiciclo", "hemicicl"},
		{"hemisferio", "hemisferi"},
		{"hemoglobina", "hemoglobin"},
		{"hemorragias", "hemorragi"},
		{"hemos", "hem"},
		{"henderson", "henderson"},
		{"henequenera", "henequener"},
		{"henk", "henk"},
		{"henrique", "henriqu"},
		{"henry", "henry"},
		{"hepática", "hepat"},
		{"hepburn", "hepburn"},
		{"heráclito", "heraclit"},
		{"heralcor", "heralcor"},
		{"herald", "herald"},
		{"heráldico", "herald"},
		{"herbicidas", "herbic"},
		{"herbívoros", "herbivor"},
		{"herce", "herc"},
		{"hércules", "hercul"},
		{"herdez", "herdez"},
		{"hérdez", "herdez"},
		{"heredados", "hered"},
		{"heredero", "hereder"},
		{"herederos", "hereder"},
		{"heredia", "heredi"},
		{"hereditario", "hereditari"},
		{"herencia", "herenci"},
		{"heriberto", "heribert"},
		{"herida", "her"},
		{"heridas", "her"},
		{"herido", "her"},
		{"heridos", "her"},
		{"herir", "her"},
		{"herirla", "her"},
		{"heritage", "heritag"},
		{"herman", "herm"},
		{"hermana", "herman"},
		{"hermanas", "herman"},
		{"hermandad", "hermand"},
		{"hermanitas", "hermanit"},
		{"hermann", "hermann"},
		{"hermano", "herman"},
		{"hermanó", "herman"},
		{"hermanos", "herman"},
		{"hermenegildo", "hermenegild"},
		{"hermética", "hermet"},
		{"hermeticidad", "hermet"},
		{"hermetismo", "hermet"},
		{"hermila", "hermil"},
		{"hermosa", "hermos"},
		{"hermosamente", "herm"},
		{"hermosas", "herm"},
		{"hermosillo", "hermosill"},
		{"hermoso", "hermos"},
		{"hernán", "hernan"},
		{"hernández", "hernandez"},
		{"hernandéz", "hernandez"},
		{"hernando", "hern"},
		{"héroe", "hero"},
		{"héroes", "her"},
		{"heróicos", "heroic"},
		{"heroísmo", "heroism"},
		{"herradura", "herradur"},
		{"herramental", "herramental"},
		{"herramienta", "herramient"},
		{"herramientas", "herramient"},
		{"herrera", "herrer"},
		{"herzfeld", "herzfeld"},
		{"herzog", "herzog"},
		{"heterosexual", "heterosexual"},
		{"heterosexuales", "heterosexual"},
		{"hewlett", "hewlett"},
		{"hewlettpackard", "hewlettpackard"},
		{"hgo", "hgo"},
		{"híbrido", "hibr"},
		{"híbridos", "hibr"},
		{"hice", "hic"},
		{"hiciera", "hic"},
		{"hicieran", "hic"},
		{"hicieron", "hic"},
		{"hiciesen", "hic"},
		{"hicimos", "hic"},
		{"hiciste", "hic"},
		{"hidalgo", "hidalg"},
		{"hidalgos", "hidalg"},
		{"hidalguense", "hidalguens"},
		{"hidalguenses", "hidalguens"},
		{"hidráulica", "hidraul"},
		{"hidráulicos", "hidraul"},
		{"hidrocarburo", "hidrocarbur"},
		{"hidrocarburos", "hidrocarbur"},
		{"hidroelectricidad", "hidroelectr"},
		{"hidrológica", "hidrolog"},
		{"hidrológicos", "hidrolog"},
		{"hielo", "hiel"},
		{"hierba", "hierb"},
		{"hierbas", "hierb"},
		{"hierro", "hierr"},
		{"hierros", "hierr"},
		{"hígado", "hig"},
		{"hígados", "hig"},
		{"higiene", "higien"},
		{"higinio", "higini"},
		{"hija", "hij"},
		{"hijas", "hij"},
		{"hijo", "hij"},
		{"hijos", "hij"},
		{"hilario", "hilari"},
		{"hilburn", "hilburn"},
		{"hilda", "hild"},
		{"hilera", "hiler"},
		{"hill", "hill"},
		{"hillary", "hillary"},
		{"hills", "hills"},
		{"hilo", "hil"},
		{"hilos", "hil"},
		{"hilvanó", "hilvan"},
		{"himenia", "himeni"},
		{"himno", "himn"},
		{"hinca", "hinc"},
		{"hincapié", "hincapi"},
		{"hines", "hin"},
		{"hinojosa", "hinoj"},
		{"hipersensibilidad", "hipersensibil"},
		{"hipertensión", "hipertension"},
		{"hipertiroidismo", "hipertiroid"},
		{"hipo", "hip"},
		{"hipólito", "hipolit"},
		{"hiponacte", "hiponact"},
		{"hipoteca", "hipotec"},
		{"hipotecada", "hipotec"},
		{"hipotecaria", "hipotecari"},
		{"hipotecarias", "hipotecari"},
		{"hipotecario", "hipotecari"},
		{"hipótesis", "hipotesis"},
		{"hipotéticamente", "hipotet"},
		{"hipotéticos", "hipotet"},
		{"hippie", "hippi"},
		{"hippies", "hippi"},
		{"hiriendo", "hir"},
		{"hirió", "hir"},
		{"hirofumi", "hirofumi"},
		{"hiroshima", "hiroshim"},
		{"hispana", "hispan"},
		{"hispánicas", "hispan"},
		{"hispanio", "hispani"},
		{"hispanos", "hispan"},
		{"histeria", "histeri"},
		{"histéricamente", "hister"},
		{"histérico", "hister"},
		{"historia", "histori"},
		{"historiadores", "histori"},
		{"historial", "historial"},
		{"historias", "histori"},
		{"histórica", "histor"},
		{"históricamente", "histor"},
		{"históricas", "histor"},
		{"historico", "histor"},
		{"histórico", "histor"},
		{"históricos", "histor"},
		{"hit", "hit"},
		{"hitazo", "hitaz"},
		{"hitler", "hitl"},
		{"hits", "hits"},
		{"hizo", "hiz"},
		{"ho", "ho"},
		{"hobbie", "hobbi"},
		{"hoc", "hoc"},
		{"hocico", "hocic"},
		{"hogar", "hog"},
		{"hogares", "hogar"},
		{"hogueras", "hoguer"},
		{"hoja", "hoj"},
		{"hojalata", "hojalat"},
		{"hojas", "hoj"},
		{"hojeo", "hoje"},
		{"hojita", "hojit"},
		{"hola", "hol"},
		{"holanda", "holand"},
		{"holandés", "holandes"},
		{"holanes", "holan"},
		{"holding", "holding"},
		{"holgada", "holg"},
		{"holgado", "holg"},
		{"holguín", "holguin"},
		{"holliday", "holliday"},
		{"hollywood", "hollywood"},
		{"holt", "holt"},
		{"holzinger", "holzing"},
		{"homar", "hom"},
		{"hombre", "hombr"},
		{"hombres", "hombr"},
		{"hombro", "hombr"},
		{"home", "hom"},
		{"homenaje", "homenaj"},
		{"homenajeado", "homenaj"},
		{"homeóstasis", "homeostasis"},
		{"homero", "homer"},
		{"homicida", "homic"},
		{"homicidas", "homic"},
		{"homicidio", "homicidi"},
		{"homicidios", "homicidi"},
		{"homilía", "homil"},
		{"homologación", "homolog"},
		{"homologando", "homolog"},
		{"homologarlas", "homolog"},
		{"homologarlos", "homolog"},
		{"homologarnos", "homolog"},
		{"homólogo", "homolog"},
		{"homólogos", "homolog"},
		{"homónimo", "homonim"},
		{"homosexual", "homosexual"},
		{"homosexuales", "homosexual"},
		{"homosexualismo", "homosexual"},
		{"homs", "homs"},
		{"honda", "hond"},
		{"hondo", "hond"},
		{"honduras", "hondur"},
		{"hondureño", "hondureñ"},
		{"hondureños", "hondureñ"},
		{"honesta", "honest"},
		{"honestamente", "honest"},
		{"honestidad", "honest"},
		{"honesto", "honest"},
		{"honestos", "honest"},
		{"hong", "hong"},
		{"honor", "honor"},
		{"honorabilidad", "honor"},
		{"honorable", "honor"},
		{"honorables", "honor"},
		{"honorarios", "honorari"},
		{"honores", "honor"},
		{"honorífica", "honorif"},
		{"honrada", "honr"},
		{"honradez", "honradez"},
		{"honrados", "honr"},
		{"honrarlo", "honr"},
		{"hopkins", "hopkins"},
		{"hora", "hor"},
		{"horacio", "horaci"},
		{"horario", "horari"},
		{"horarios", "horari"},
		{"horas", "hor"},
		{"horcada", "horc"},
		{"horchata", "horchat"},
		{"horizon", "horizon"},
		{"horizontal", "horizontal"},
		{"horizonte", "horizont"},
		{"horizontes", "horizont"},
		{"hormona", "hormon"},
		{"hormonas", "hormon"},
		{"horneados", "horn"},
		{"hornos", "horn"},
		{"horrenda", "horrend"},
		{"horrendo", "horrend"},
		{"horribles", "horribl"},
		{"horror", "horror"},
		{"horrorizado", "horroriz"},
		{"hortalizas", "hortaliz"},
		{"horwath", "horwath"},
		{"hospedaba", "hosped"},
		{"hospederos", "hospeder"},
		{"hospicio", "hospici"},
		{"hospital", "hospital"},
		{"hospitalario", "hospitalari"},
		{"hospitalarios", "hospitalari"},
		{"hospitales", "hospital"},
		{"hospitalización", "hospitaliz"},
		{"hospitalizado", "hospitaliz"},
		{"hospitalizados", "hospitaliz"},
		{"hostile", "hostil"},
		{"hostilidades", "hostil"},
		{"hot", "hot"},
		{"hotel", "hotel"},
		{"hotelera", "hoteler"},
		{"hoteleros", "hoteler"},
		{"hoteles", "hotel"},
		{"houston", "houston"},
		{"howard", "howard"},
		{"hoy", "hoy"},
		{"hoyo", "hoy"},
		{"hoyos", "hoy"},
		{"hp", "hp"},
		{"hristo", "hrist"},
		{"huacho", "huach"},
		{"huahutla", "huahutl"},
		{"huamantla", "huamantl"},
		{"huambo", "huamb"},
		{"huang", "huang"},
		{"huapango", "huapang"},
		{"huasteca", "huastec"},
		{"huautla", "huautl"},
		{"huayami", "huayami"},
		{"hub", "hub"},
		{"hubiera", "hub"},
		{"hubieran", "hub"},
		{"hubiere", "hubier"},
		{"hubiese", "hub"},
		{"hubiesen", "hub"},
		{"hubo", "hub"},
		{"huecos", "huec"},
		{"huehuetenango", "huehuetenang"},
		{"huehuetlán", "huehuetlan"},
		{"huehuetlatoa", "huehuetlato"},
		{"huela", "huel"},
		{"huele", "huel"},
		{"huelen", "huel"},
		{"huelga", "huelg"},
		{"huelguistas", "huelguist"},
		{"huella", "huell"},
		{"huellas", "huell"},
		{"huérfanos", "huerfan"},
		{"huerta", "huert"},
		{"huertas", "huert"},
		{"huerto", "huert"},
		{"huertos", "huert"},
		{"huesca", "huesc"},
		{"hueso", "hues"},
		{"huésped", "huesp"},
		{"huestes", "huest"},
		{"huevazos", "huevaz"},
		{"huevo", "huev"},
		{"huevos", "huev"},
		{"hufbauer", "hufbau"},
		{"hugo", "hug"},
		{"huichapan", "huichap"},
		{"huichol", "huichol"},
		{"huicholas", "huichol"},
		{"huicholes", "huichol"},
		{"huidizo", "huidiz"},
		{"huinalá", "huinal"},
		{"huir", "huir"},
		{"huitzilopochtli", "huitzilopochtli"},
		{"huitzuco", "huitzuc"},
		{"hules", "hul"},
		{"humana", "human"},
		{"humanas", "human"},
		{"humanidad", "human"},
		{"humanidades", "human"},
		{"humanista", "human"},
		{"humanitario", "humanitari"},
		{"humanitarios", "humanitari"},
		{"humaniza", "humaniz"},
		{"humanizar", "humaniz"},
		{"humano", "human"},
		{"humanos", "human"},
		{"humberto", "humbert"},
		{"humeante", "humeant"},
		{"humedad", "humed"},
		{"humildad", "humild"},
		{"humilde", "humild"},
		{"humillados", "humill"},
		{"humillante", "humill"},
		{"humillantemente", "humill"},
		{"humillar", "humill"},
		{"humo", "hum"},
		{"humor", "humor"},
		{"humorista", "humor"},
		{"hundida", "hund"},
		{"hundidas", "hund"},
		{"hundido", "hund"},
		{"hundidos", "hund"},
		{"hundieron", "hund"},
		{"hundimientos", "hundimient"},
		{"hundió", "hund"},
		{"hundir", "hund"},
		{"húngaro", "hungar"},
		{"hungo", "hung"},
		{"hungría", "hungr"},
		{"huntsville", "huntsvill"},
		{"hurtadillas", "hurtadill"},
		{"hurtado", "hurt"},
		{"hurtic", "hurtic"},
		{"huss", "huss"},
		{"hussein", "hussein"},
		{"huyeron", "huyeron"},
		{"huyó", "huy"},
		{"hylsa", "hylsa"},
		{"i", "i"},
		{"iaaf", "iaaf"},
		{"iacocca", "iacocc"},
		{"iannuzzelli", "iannuzzelli"},
		{"iba", "iba"},
		{"íbamos", "ibam"},
		{"iban", "iban"},
		{"ibargüengoitia", "ibargüengoiti"},
		{"ibarra", "ibarr"},
		{"ibéricos", "iber"},
		{"ibm", "ibm"},
		{"ibor", "ibor"},
		{"ica", "ica"},
		{"iceberg", "iceberg"},
		{"ichi", "ichi"},
		{"ici", "ici"},
		{"iconografía", "iconograf"},
		{"iconográfica", "iconograf"},
		{"id", "id"},
		{"ida", "ida"},
		{"idaho", "idah"},
		{"idalia", "idali"},
		{"idea", "ide"},
		{"ideal", "ideal"},
		{"ideales", "ideal"},
		{"idealistas", "ideal"},
		{"ideario", "ideari"},
		{"ideas", "ide"},
		{"ídem", "idem"},
		{"idéntica", "ident"},
		{"idéntico", "ident"},
		{"identidad", "ident"},
		{"identifica", "identif"},
		{"identificación", "identif"},
		{"identificada", "identific"},
		{"identificadas", "identific"},
		{"identificado", "identific"},
		{"identificados", "identific"},
		{"identifican", "identific"},
		{"identificando", "identific"},
		{"identificar", "identific"},
		{"identificarlo", "identific"},
		{"identificaron", "identific"},
		{"identificarse", "identific"},
		{"identificó", "identific"},
		{"identifique", "identifiqu"},
		{"identifiquen", "identifiqu"},
		{"ideó", "ide"},
		{"ideología", "ideolog"},
		{"ideologías", "ideolog"},
		{"ideológica", "ideolog"},
		{"ideológicas", "ideolog"},
		{"ideológico", "ideolog"},
		{"idioma", "idiom"},
		{"idiomas", "idiom"},
		{"idiosincrasia", "idiosincrasi"},
		{"idiotas", "idiot"},
		{"ido", "ido"},
		{"idólatra", "idolatr"},
		{"idolatría", "idolatr"},
		{"ídolo", "idol"},
		{"ídolos", "idol"},
		{"idóneo", "idone"},
		{"ife", "ife"},
		{"ifigenia", "ifigeni"},
		{"ifigenios", "ifigeni"},
		{"iglesia", "iglesi"},
		{"iglesiaestado", "iglesiaest"},
		{"iglesias", "iglesi"},
		{"ignacio", "ignaci"},
		{"ignición", "ignicion"},
		{"ignominia", "ignomini"},
		{"ignora", "ignor"},
		{"ignoran", "ignor"},
		{"ignorancia", "ignor"},
		{"ignorando", "ignor"},
		{"ignorante", "ignor"},
		{"ignorar", "ignor"},
		{"ignore", "ignor"},
		{"igor", "igor"},
		{"igual", "igual"},
		{"iguala", "igual"},
		{"igualada", "igual"},
		{"igualadas", "igual"},
		{"igualan", "igual"},
		{"igualar", "igual"},
		{"igualarlo", "igual"},
		{"igualarnos", "igual"},
		{"igualaron", "igual"},
		{"igualdad", "iguald"},
		{"iguales", "igual"},
		{"igualita", "igualit"},
		{"igualmente", "igual"},
		{"ii", "ii"},
		{"iii", "iii"},
		{"il", "il"},
		{"ildefonso", "ildefons"},
		{"ilegal", "ilegal"},
		{"ilegales", "ilegal"},
		{"ilegalmente", "ilegal"},
		{"ilesa", "iles"},
		{"ileso", "iles"},
		{"ilícita", "ilicit"},
		{"ilícito", "ilicit"},
		{"ilícitos", "ilicit"},
		{"ilimitada", "ilimit"},
		{"ilimitadas", "ilimit"},
		{"ilimitado", "ilimit"},
		{"ilimitados", "ilimit"},
		{"illinois", "illinois"},
		{"ilógica", "ilog"},
		{"ilógico", "ilog"},
		{"iluminación", "ilumin"},
		{"iluminadas", "ilumin"},
		{"iluminaron", "ilumin"},
		{"iluminó", "ilumin"},
		{"ilusas", "ilus"},
		{"ilusión", "ilusion"},
		{"ilusiones", "ilusion"},
		{"ilustra", "ilustr"},
		{"ilustraciones", "ilustr"},
		{"ilustran", "ilustr"},
		{"ilustrar", "ilustr"},
		{"ilustraría", "ilustr"},
		{"im", "im"},
		{"imagen", "imag"},
		{"imágenes", "imagen"},
		{"imagina", "imagin"},
		{"imaginación", "imagin"},
		{"imaginar", "imagin"},
		{"imagínense", "imaginens"},
		{"imagínese", "imagines"},
		{"imagino", "imagin"},
		{"imaginó", "imagin"},
		{"imán", "iman"},
		{"imax", "imax"},
		{"imbécil", "imbecil"},
		{"imecas", "imec"},
		{"imelda", "imeld"},
		{"imevisión", "imevision"},
		{"imitable", "imit"},
		{"imitación", "imit"},
		{"imitaciones", "imit"},
		{"imitan", "imit"},
		{"imitar", "imit"},
		{"imitarlo", "imit"},
		{"imp", "imp"},
		{"impaciencia", "impacient"},
		{"impacta", "impact"},
		{"impactar", "impact"},
		{"impactaría", "impact"},
		{"impactaron", "impact"},
		{"impacten", "impact"},
		{"impacto", "impact"},
		{"impactó", "impact"},
		{"impactos", "impact"},
		{"imparcial", "imparcial"},
		{"imparcialidad", "imparcial"},
		{"impartan", "impart"},
		{"imparte", "impart"},
		{"imparten", "impart"},
		{"impartía", "impart"},
		{"impartida", "impart"},
		{"impartiendo", "impart"},
		{"impartieran", "impart"},
		{"impartimos", "impart"},
		{"impartir", "impart"},
		{"impartirá", "impart"},
		{"impartirán", "impart"},
		{"impasse", "impass"},
		{"impecable", "impec"},
		{"impedido", "imped"},
		{"impedimento", "impediment"},
		{"impedimentos", "impediment"},
		{"impedir", "imped"},
		{"impedirá", "imped"},
		{"impedirán", "imped"},
		{"impediría", "imped"},
		{"impedirle", "imped"},
		{"imperan", "imper"},
		{"imperdonable", "imperdon"},
		{"imperfecto", "imperfect"},
		{"imperial", "imperial"},
		{"imperiales", "imperial"},
		{"imperialismo", "imperial"},
		{"imperialistas", "imperial"},
		{"imperio", "imperi"},
		{"impersonal", "impersonal"},
		{"impida", "impid"},
		{"impide", "impid"},
		{"impiden", "impid"},
		{"impidiera", "impid"},
		{"impidieron", "impid"},
		{"impidió", "impid"},
		{"implacable", "implac"},
		{"implacables", "implac"},
		{"implantación", "implant"},
		{"implantado", "implant"},
		{"implantados", "implant"},
		{"implantando", "implant"},
		{"implantaremos", "implant"},
		{"implanté", "implant"},
		{"implementación", "implement"},
		{"implementada", "implement"},
		{"implementan", "implement"},
		{"implementando", "implement"},
		{"implementar", "implement"},
		{"implementarán", "implement"},
		{"implementarla", "implement"},
		{"implementarse", "implement"},
		{"implementó", "implement"},
		{"implementos", "implement"},
		{"implica", "implic"},
		{"implicación", "implic"},
		{"implicaciones", "implic"},
		{"implicados", "implic"},
		{"implican", "implic"},
		{"implicando", "implic"},
		{"implicar", "implic"},
		{"implicará", "implic"},
		{"implicaran", "implic"},
		{"implícito", "implicit"},
		{"implicó", "implic"},
		{"impolutos", "impolut"},
		{"impondrá", "impondr"},
		{"impondrán", "impondran"},
		{"impondría", "impondr"},
		{"impone", "impon"},
		{"imponemos", "impon"},
		{"imponen", "impon"},
		{"imponer", "impon"},
		{"imponerse", "impon"},
		{"impongan", "impong"},
		{"imponiendo", "impon"},
		{"imponiéndose", "impon"},
		{"importa", "import"},
		{"importaba", "import"},
		{"importación", "import"},
		{"importaciones", "import"},
		{"importadas", "import"},
		{"importado", "import"},
		{"importadora", "import"},
		{"importados", "import"},
		{"importan", "import"},
		{"importancia", "import"},
		{"importando", "import"},
		{"importante", "import"},
		{"importantemente", "import"},
		{"importantes", "import"},
		{"importantísimas", "importantisim"},
		{"importantísimo", "importantisim"},
		{"importantísimos", "importantis"},
		{"importar", "import"},
		{"importará", "import"},
		{"importaría", "import"},
		{"importe", "import"},
		{"importó", "import"},
		{"imposibilidad", "imposibil"},
		{"imposibilitada", "imposibilit"},
		{"imposible", "impos"},
		{"imposición", "imposicion"},
		{"imposiciones", "imposicion"},
		{"impositivos", "imposit"},
		{"impostergable", "imposterg"},
		{"impostergables", "imposterg"},
		{"impotencia", "impotent"},
		{"impotente", "impotent"},
		{"impregnada", "impregn"},
		{"impregnan", "impregn"},
		{"imprenta", "imprent"},
		{"impresa", "impres"},
		{"impresas", "impres"},
		{"impresión", "impresion"},
		{"impresionable", "impresion"},
		{"impresionada", "impresion"},
		{"impresionado", "impresion"},
		{"impresionante", "impresion"},
		{"impresionantes", "impresion"},
		{"impresiones", "impresion"},
		{"impresionismo", "impresion"},
		{"impresionista", "impresion"},
		{"impresionó", "impresion"},
		{"impreso", "impres"},
		{"impresora", "impresor"},
		{"impresoras", "impresor"},
		{"impresores", "impresor"},
		{"impresos", "impres"},
		{"imprevisión", "imprevision"},
		{"imprimió", "imprim"},
		{"imprimir", "imprim"},
		{"imprimirse", "imprim"},
		{"improductiva", "improduct"},
		{"impropiedades", "impropiedad"},
		{"improvisación", "improvis"},
		{"improvisada", "improvis"},
		{"improvisados", "improvis"},
		{"imprudencia", "imprudent"},
		{"imprudencial", "imprudencial"},
		{"imprudencialmente", "imprudencial"},
		{"imprudentes", "imprudent"},
		{"impuesta", "impuest"},
		{"impuestas", "impuest"},
		{"impuesto", "impuest"},
		{"impuestos", "impuest"},
		{"impugnaciones", "impugn"},
		{"impugnado", "impugn"},
		{"impulsa", "impuls"},
		{"impulsan", "impuls"},
		{"impulsando", "impuls"},
		{"impulsar", "impuls"},
		{"impulse", "impuls"},
		{"impulsivo", "impuls"},
		{"impulso", "impuls"},
		{"impulsos", "impuls"},
		{"impunemente", "impun"},
		{"impunes", "impun"},
		{"impunidad", "impun"},
		{"impunidades", "impun"},
		{"impuntualidad", "impuntual"},
		{"impuras", "impur"},
		{"impusieron", "impus"},
		{"impuso", "impus"},
		{"imputaciones", "imput"},
		{"imputado", "imput"},
		{"imsa", "imsa"},
		{"imss", "imss"},
		{"in", "in"},
		{"inaccesible", "inacces"},
		{"inaccesibles", "inacces"},
		{"inacción", "inaccion"},
		{"inaceptable", "inacept"},
		{"inactividad", "inact"},
		{"inadecuada", "inadecu"},
		{"inadmisible", "inadmis"},
		{"inadvertidas", "inadvert"},
		{"inadvertidos", "inadvert"},
		{"inah", "inah"},
		{"inalámbrico", "inalambr"},
		{"inalienables", "inalien"},
		{"inamovible", "inamov"},
		{"inamovibles", "inamov"},
		{"inaplicables", "inaplic"},
		{"inasistencia", "inasistent"},
		{"inaudito", "inaudit"},
		{"inaugura", "inaugur"},
		{"inauguración", "inaugur"},
		{"inauguraciones", "inaugur"},
		{"inaugurada", "inaugur"},
		{"inaugurado", "inaugur"},
		{"inaugural", "inaugural"},
		{"inauguran", "inaugur"},
		{"inaugurar", "inaugur"},
		{"inaugurará", "inaugur"},
		{"inauguró", "inaugur"},
		{"inba", "inba"},
		{"inc", "inc"},
		{"incalculable", "incalcul"},
		{"incapaces", "incapac"},
		{"incapacidad", "incapac"},
		{"incapacitado", "incapacit"},
		{"incapacitados", "incapacit"},
		{"incapaz", "incapaz"},
		{"incautada", "incaut"},
		{"incautado", "incaut"},
		{"incautan", "incaut"},
		{"incauten", "incaut"},
		{"incautó", "incaut"},
		{"incendiarias", "incendiari"},
		{"incendio", "incendi"},
		{"incendió", "incend"},
		{"incendios", "incendi"},
		{"incentivar", "incentiv"},
		{"incentivo", "incent"},
		{"incentivos", "incent"},
		{"incertidumbre", "incertidumbr"},
		{"incesante", "inces"},
		{"incesantemente", "inces"},
		{"incian", "inci"},
		{"incide", "incid"},
		{"inciden", "incid"},
		{"incidencia", "incident"},
		{"incidental", "incidental"},
		{"incidente", "incident"},
		{"incidentes", "incident"},
		{"incidido", "incid"},
		{"incierto", "inciert"},
		{"inciertos", "inciert"},
		{"incinerador", "inciner"},
		{"incisos", "incis"},
		{"incitación", "incit"},
		{"incitador", "incit"},
		{"incitaron", "incit"},
		{"inclán", "inclan"},
		{"inclemencias", "inclement"},
		{"inclina", "inclin"},
		{"inclinaba", "inclin"},
		{"inclinación", "inclin"},
		{"inclinaciones", "inclin"},
		{"inclinada", "inclin"},
		{"inclinados", "inclin"},
		{"inclinamos", "inclin"},
		{"inclinan", "inclin"},
		{"inclinándose", "inclin"},
		{"inclinó", "inclin"},
		{"incluía", "inclu"},
		{"incluida", "inclu"},
		{"incluidas", "inclu"},
		{"incluido", "inclu"},
		{"incluidos", "inclu"},
		{"incluimos", "inclu"},
		{"incluir", "inclu"},
		{"incluír", "incluir"},
		{"incluirá", "inclu"},
		{"incluirán", "inclu"},
		{"incluiría", "inclu"},
		{"incluirlo", "inclu"},
		{"incluirlos", "inclu"},
		{"inclusión", "inclusion"},
		{"inclusive", "inclusiv"},
		{"incluso", "inclus"},
		{"incluya", "inclu"},
		{"incluyan", "inclu"},
		{"incluye", "inclu"},
		{"incluyen", "inclu"},
		{"incluyendo", "inclu"},
		{"incluyeron", "inclu"},
		{"incluyó", "inclu"},
		{"incoherencias", "incoherent"},
		{"incómodo", "incomod"},
		{"incompetentes", "incompetent"},
		{"incompleta", "incomplet"},
		{"incompleto", "incomplet"},
		{"incomprendidos", "incomprend"},
		{"incomprensibles", "incomprens"},
		{"incomprensión", "incomprension"},
		{"incomunicación", "incomun"},
		{"incomunicada", "incomunic"},
		{"incomunicadas", "incomunic"},
		{"incondicional", "incondicional"},
		{"incondicionales", "incondicional"},
		{"inconformaron", "inconform"},
		{"inconformes", "inconform"},
		{"inconformidad", "inconform"},
		{"inconformidades", "inconform"},
		{"incongruencias", "incongruent"},
		{"incongruente", "incongruent"},
		{"inconmovibles", "inconmov"},
		{"inconsciente", "inconscient"},
		{"inconscientes", "inconscient"},
		{"inconsecuencia", "inconsecuent"},
		{"inconsistencia", "inconsistent"},
		{"inconsistente", "inconsistent"},
		{"inconstitucionales", "inconstitucional"},
		{"inconstitucionalidad", "inconstitucional"},
		{"", ""},
		{"incontenible", "inconten"},
		{"incontrolable", "incontrol"},
		{"incontrolables", "incontrol"},
		{"incontrolado", "incontrol"},
		{"inconveniente", "inconvenient"},
		{"inconvenientes", "inconvenient"},
		{"incora", "incor"},
		{"incorpora", "incorpor"},
		{"incorporación", "incorpor"},
		{"incorporados", "incorpor"},
		{"incorporan", "incorpor"},
		{"incorporando", "incorpor"},
		{"incorporar", "incorpor"},
		{"incorporaron", "incorpor"},
		{"incorporarse", "incorpor"},
		{"incorporated", "incorporat"},
		{"incorpore", "incorpor"},
		{"incorporen", "incorpor"},
		{"incorporó", "incorpor"},
		{"incosteables", "incost"},
		{"incredulidad", "incredul"},
		{"increíble", "increibl"},
		{"increíblemente", "increibl"},
		{"increíbles", "increibl"},
		{"incrementa", "increment"},
		{"incrementadas", "increment"},
		{"incrementado", "increment"},
		{"incrementando", "increment"},
		{"incrementándose", "increment"},
		{"incrementar", "increment"},
		{"incrementará", "increment"},
		{"incrementarles", "increment"},
		{"incrementaron", "increment"},
		{"incrementarse", "increment"},
		{"incremente", "increment"},
		{"incrementen", "increment"},
		{"incremento", "increment"},
		{"incrementó", "increment"},
		{"incrementos", "increment"},
		{"increparan", "increp"},
		{"incriminar", "incrimin"},
		{"incrustación", "incrust"},
		{"incrustaciones", "incrust"},
		{"incrustó", "incrust"},
		{"incubación", "incub"},
		{"incuestionable", "incuestion"},
		{"incuestionables", "incuestion"},
		{"inculcar", "inculc"},
		{"incumbe", "incumb"},
		{"incumple", "incumpl"},
		{"incumplen", "incumpl"},
		{"incumplidas", "incumpl"},
		{"incumplido", "incumpl"},
		{"incumplieron", "incumpl"},
		{"incumplimiento", "incumpl"},
		{"incumplió", "incumpl"},
		{"incurran", "incurr"},
		{"incurre", "incurr"},
		{"incurrido", "incurr"},
		{"incurrieron", "incurr"},
		{"incurrió", "incurr"},
		{"incurrir", "incurr"},
		{"incursión", "incursion"},
		{"incursiona", "incursion"},
		{"incursionamos", "incursion"},
		{"incursionar", "incursion"},
		{"inda", "inda"},
		{"indagación", "indag"},
		{"indagando", "indag"},
		{"indebido", "indeb"},
		{"indebidos", "indeb"},
		{"indecibles", "indec"},
		{"indecisa", "indecis"},
		{"indecisión", "indecision"},
		{"indeciso", "indecis"},
		{"indefensión", "indefension"},
		{"indefinida", "indefin"},
		{"indefinidamente", "indefinid"},
		{"indefinido", "indefin"},
		{"indejo", "indej"},
		{"indemnización", "indemniz"},
		{"indemnizaciones", "indemniz"},
		{"indemnizar", "indemniz"},
		{"indemnizó", "indemniz"},
		{"independencia", "independent"},
		{"independiente", "independient"},
		{"independientemente", "independient"},
		{"independientes", "independient"},
		{"independiza", "independiz"},
		{"independizó", "independiz"},
		{"indeseado", "indes"},
		{"indeterminado", "indetermin"},
		{"india", "indi"},
		{"indiana", "indian"},
		{"indianapolis", "indianapolis"},
		{"indica", "indic"},
		{"indicaba", "indic"},
		{"indicaban", "indic"},
		{"indicaciones", "indic"},
		{"indicada", "indic"},
		{"indicado", "indic"},
		{"indicador", "indic"},
		{"indicadores", "indic"},
		{"indican", "indic"},
		{"indicando", "indic"},
		{"indicándome", "indic"},
		{"indicar", "indic"},
		{"indicará", "indic"},
		{"indicarán", "indic"},
		{"indicaron", "indic"},
		{"indicativa", "indic"},
		{"indicators", "indicators"},
		{"índice", "indic"},
		{"índices", "indic"},
		{"indicio", "indici"},
		{"indicios", "indici"},
		{"indico", "indic"},
		{"indicó", "indic"},
		{"indiferencia", "indiferent"},
		{"indiferente", "indiferent"},
		{"indiferentes", "indiferent"},
		{"indígena", "indigen"},
		{"indígenas", "indigen"},
		{"indigencia", "indigent"},
		{"indigente", "indigent"},
		{"indigentes", "indigent"},
		{"indignación", "indign"},
		{"indignado", "indign"},
		{"indignidad", "indign"},
		{"indigno", "indign"},
		{"indignó", "indign"},
		{"índigo", "indig"},
		{"indio", "indi"},
		{"indique", "indiqu"},
		{"indirecta", "indirect"},
		{"indirectamente", "indirect"},
		{"indirectas", "indirect"},
		{"indisciplina", "indisciplin"},
		{"indiscreciones", "indiscrecion"},
		{"indiscriminadamente", "indiscrimin"},
		{"indiscutible", "indiscut"},
		{"indiscutibles", "indiscut"},
		{"indispensable", "indispens"},
		{"indispensables", "indispens"},
		{"indistintos", "indistint"},
		{"individual", "individual"},
		{"individuales", "individual"},
		{"individualidades", "individual"},
		{"individualista", "individual"},
		{"individualistas", "individual"},
		{"individualmente", "individual"},
		{"individuo", "individu"},
		{"individuos", "individu"},
		{"índole", "indol"},
		{"indonesia", "indonesi"},
		{"indubitable", "indubit"},
		{"inducción", "induccion"},
		{"induce", "induc"},
		{"inducir", "induc"},
		{"inducirlo", "induc"},
		{"indudablemente", "indud"},
		{"indulgencias", "indulgent"},
		{"indulto", "indult"},
		{"indultó", "indult"},
		{"indumentaria", "indumentari"},
		{"industria", "industri"},
		{"industrial", "industrial"},
		{"industriales", "industrial"},
		{"industrialización", "industrializ"},
		{"industrializadas", "industrializ"},
		{"industrializados", "industrializ"},
		{"industrias", "industri"},
		{"industries", "industri"},
		{"industriosa", "industri"},
		{"indy", "indy"},
		{"inea", "ine"},
		{"inédita", "inedit"},
		{"inéditas", "inedit"},
		{"inédito", "inedit"},
		{"ineficaces", "ineficac"},
		{"ineficacia", "ineficaci"},
		{"ineficaz", "ineficaz"},
		{"ineficiencia", "ineficient"},
		{"ineficiencias", "ineficient"},
		{"ineficiente", "ineficient"},
		{"inegi", "inegi"},
		{"inelegancias", "ineleg"},
		{"ineludible", "inelud"},
		{"ineptas", "inept"},
		{"inequívoca", "inequivoc"},
		{"inercia", "inerci"},
		{"inés", "ines"},
		{"inescrutable", "inescrut"},
		{"inestabilidad", "inest"},
		{"inestable", "inest"},
		{"inevitable", "inevit"},
		{"inevitablemente", "inevit"},
		{"inevitables", "inevit"},
		{"inexactitudes", "inexactitud"},
		{"inexistente", "inexistent"},
		{"inexorablemente", "inexor"},
		{"inexplicable", "inexplic"},
		{"inexplicablemente", "inexplic"},
		{"inexploradas", "inexplor"},
		{"infactibilidad", "infactibil"},
		{"infalibilidad", "infalibil"},
		{"infalsificables", "infalsific"},
		{"infame", "infam"},
		{"infamia", "infami"},
		{"infancia", "infanci"},
		{"infante", "infant"},
		{"infantes", "infant"},
		{"infantil", "infantil"},
		{"infantiles", "infantil"},
		{"infarto", "infart"},
		{"infección", "infeccion"},
		{"infecciones", "infeccion"},
		{"infectología", "infectolog"},
		{"inferido", "infer"},
		{"inferior", "inferior"},
		{"inferiores", "inferior"},
		{"inferioridad", "inferior"},
		{"infernal", "infernal"},
		{"infernales", "infernal"},
		{"infestadas", "infest"},
		{"infidelidad", "infidel"},
		{"infidelidades", "infidel"},
		{"infiel", "infiel"},
		{"infierno", "infiern"},
		{"infiltrados", "infiltr"},
		{"infiltraron", "infiltr"},
		{"infinidad", "infin"},
		{"infinitum", "infinitum"},
		{"inflaba", "inflab"},
		{"inflación", "inflacion"},
		{"inflacionario", "inflacionari"},
		{"inflacionarios", "inflacionari"},
		{"inflamación", "inflam"},
		{"inflar", "inflar"},
		{"infle", "infle"},
		{"influencia", "influenci"},
		{"influenciada", "influenci"},
		{"influencias", "influenci"},
		{"influenzae", "influenza"},
		{"influido", "influ"},
		{"influir", "influ"},
		{"influiría", "influ"},
		{"influjo", "influj"},
		{"influyen", "influ"},
		{"influyente", "influyent"},
		{"influyó", "influ"},
		{"infonavit", "infonavit"},
		{"informa", "inform"},
		{"informaba", "inform"},
		{"información", "inform"},
		{"informaciones", "inform"},
		{"informada", "inform"},
		{"informado", "inform"},
		{"informados", "inform"},
		{"informal", "informal"},
		{"informan", "inform"},
		{"informándome", "inform"},
		{"informante", "inform"},
		{"informantes", "inform"},
		{"informar", "inform"},
		{"informará", "inform"},
		{"informarán", "inform"},
		{"informarle", "inform"},
		{"informaron", "inform"},
		{"informarse", "inform"},
		{"informatica", "informat"},
		{"informática", "informat"},
		{"informático", "informat"},
		{"informativa", "inform"},
		{"informativas", "inform"},
		{"informativo", "inform"},
		{"informativos", "inform"},
		{"informe", "inform"},
		{"informé", "inform"},
		{"informes", "inform"},
		{"informó", "inform"},
		{"inforrey", "inforrey"},
		{"infortunadamente", "infortun"},
		{"infortunado", "infortun"},
		{"infracción", "infraccion"},
		{"infraccionados", "infraccion"},
		{"infraccionar", "infraccion"},
		{"infractor", "infractor"},
		{"infractores", "infractor"},
		{"infraestructura", "infraestructur"},
		{"infrarrojas", "infrarroj"},
		{"infrarrojos", "infrarroj"},
		{"infundados", "infund"},
		{"infundios", "infundi"},
		{"infundir", "infund"},
		{"infusión", "infusion"},
		{"infusiones", "infusion"},
		{"ing", "ing"},
		{"ingando", "ingand"},
		{"ingeniería", "ingeni"},
		{"ingeniero", "ingenier"},
		{"ingenieros", "ingenier"},
		{"ingenio", "ingeni"},
		{"ingeniosa", "ingeni"},
		{"ingenioso", "ingeni"},
		{"ingenua", "ingenu"},
		{"ingenuas", "ingenu"},
		{"ingenuo", "ingenu"},
		{"ingerencia", "ingerent"},
		{"ingerir", "inger"},
		{"inglaterra", "inglaterr"},
		{"inglés", "ingles"},
		{"inglesa", "ingles"},
		{"inglesas", "ingles"},
		{"ingleses", "ingles"},
		{"ingratos", "ingrat"},
		{"ingrediente", "ingredient"},
		{"ingredientes", "ingredient"},
		{"ingresa", "ingres"},
		{"ingresaba", "ingres"},
		{"ingresado", "ingres"},
		{"ingresan", "ingres"},
		{"ingresando", "ingres"},
		{"ingresar", "ingres"},
		{"ingresara", "ingres"},
		{"ingresaron", "ingres"},
		{"ingreso", "ingres"},
		{"ingresó", "ingres"},
		{"ingresos", "ingres"},
		{"inhabilitados", "inhabilit"},
		{"inhabilitan", "inhabilit"},
		{"inhalaciones", "inhal"},
		{"inhalámbricos", "inhalambr"},
		{"inherente", "inherent"},
		{"inherentes", "inherent"},
		{"inhibiendo", "inhib"},
		{"inicado", "inic"},
		{"inicará", "inic"},
		{"inicia", "inici"},
		{"iniciaba", "inici"},
		{"iniciación", "inici"},
		{"iniciada", "inici"},
		{"iniciadas", "inici"},
		{"iniciado", "inici"},
		{"iniciadores", "inici"},
		{"inicial", "inicial"},
		{"iniciales", "inicial"},
		{"inicialización", "inicializ"},
		{"inicialmente", "inicial"},
		{"inician", "inici"},
		{"iniciando", "inici"},
		{"iniciar", "inici"},
		{"iniciara", "inici"},
		{"iniciará", "inici"},
		{"iniciarán", "inici"},
		{"iniciaría", "inici"},
		{"iniciarían", "inici"},
		{"iniciarme", "inici"},
		{"iniciaron", "inici"},
		{"iniciarse", "inici"},
		{"iniciativa", "inici"},
		{"iniciativas", "inici"},
		{"inicie", "inici"},
		{"iniciemos", "inici"},
		{"inicien", "inici"},
		{"inicio", "inici"},
		{"inició", "inic"},
		{"inicios", "inici"},
		{"inimaginables", "inimagin"},
		{"ininteligibles", "inintelig"},
		{"ininterrumpida", "ininterrump"},
		{"injerencia", "injerent"},
		{"injude", "injud"},
		{"injurias", "injuri"},
		{"injusta", "injust"},
		{"injustas", "injust"},
		{"injusticia", "injustici"},
		{"injusticias", "injustici"},
		{"injustificada", "injustific"},
		{"injustificados", "injustific"},
		{"injusto", "injust"},
		{"injustos", "injust"},
		{"inmadurez", "inmadurez"},
		{"inmaduros", "inmadur"},
		{"inmediaciones", "inmedi"},
		{"inmediata", "inmediat"},
		{"inmediatamente", "inmediat"},
		{"inmediatas", "inmediat"},
		{"inmediato", "inmediat"},
		{"inmediatos", "inmediat"},
		{"inmenso", "inmens"},
		{"inmensos", "inmens"},
		{"inmigrantes", "inmigr"},
		{"inminencia", "inminent"},
		{"inminente", "inminent"},
		{"inmiscuidos", "inmiscu"},
		{"inmiscuirse", "inmiscu"},
		{"inmobiliaria", "inmobiliari"},
		{"inmobiliarias", "inmobiliari"},
		{"inmobiliario", "inmobiliari"},
		{"inmobiliarios", "inmobiliari"},
		{"inmorales", "inmoral"},
		{"inmortal", "inmortal"},
		{"inmortales", "inmortal"},
		{"inmortalidad", "inmortal"},
		{"inmovilizó", "inmoviliz"},
		{"inmueble", "inmuebl"},
		{"inmuebles", "inmuebl"},
		{"inmunes", "inmun"},
		{"inmunidad", "inmun"},
		{"inmunológica", "inmunolog"},
		{"inmutable", "inmut"},
		{"inn", "inn"},
		{"innecesaria", "innecesari"},
		{"innecesarias", "innecesari"},
		{"innecesario", "innecesari"},
		{"innecesarios", "innecesari"},
		{"innova", "innov"},
		{"innovación", "innov"},
		{"innovaciones", "innov"},
		{"innovador", "innov"},
		{"innovadora", "innov"},
		{"innovadores", "innov"},
		{"innovar", "innov"},
		{"inntienen", "inntien"},
		{"innumerables", "innumer"},
		{"inobservancia", "inobserv"},
		{"inocencia", "inocent"},
		{"inocente", "inocent"},
		{"inocentes", "inocent"},
		{"inocular", "inocul"},
		{"inocularle", "inocul"},
		{"inofensivos", "inofens"},
		{"inolvidable", "inolvid"},
		{"inoperantes", "inoper"},
		{"inquiere", "inquier"},
		{"inquieta", "inquiet"},
		{"inquietas", "inquiet"},
		{"inquieto", "inquiet"},
		{"inquietos", "inquiet"},
		{"inquietud", "inquietud"},
		{"inquietudes", "inquietud"},
		{"inquilinaria", "inquilinari"},
		{"inquilino", "inquilin"},
		{"inquilinos", "inquilin"},
		{"inquina", "inquin"},
		{"inquisición", "inquisicion"},
		{"insaciable", "insaci"},
		{"insalvables", "insalv"},
		{"inscribe", "inscrib"},
		{"inscribió", "inscrib"},
		{"inscribir", "inscrib"},
		{"inscribirse", "inscrib"},
		{"inscripción", "inscripcion"},
		{"inscripciones", "inscripcion"},
		{"inscrita", "inscrit"},
		{"inscritas", "inscrit"},
		{"inscrito", "inscrit"},
		{"inscritos", "inscrit"},
		{"insecticida", "insectic"},
		{"insecto", "insect"},
		{"insectos", "insect"},
		{"inseguridad", "insegur"},
		{"insen", "insen"},
		{"insensatez", "insensatez"},
		{"inservibles", "inserv"},
		{"insiders", "insiders"},
		{"insidiosos", "insidi"},
		{"insignia", "insigni"},
		{"insignificante", "insignif"},
		{"insinúa", "insinu"},
		{"insinuó", "insinu"},
		{"insípidos", "insip"},
		{"insista", "insist"},
		{"insiste", "insist"},
		{"insisten", "insist"},
		{"insistencia", "insistent"},
		{"insistente", "insistent"},
		{"insistía", "insist"},
		{"insistían", "insist"},
		{"insistieron", "insist"},
		{"insistió", "insist"},
		{"insistir", "insist"},
		{"insistirá", "insist"},
		{"insistiré", "insist"},
		{"insistirse", "insist"},
		{"insististe", "insist"},
		{"insisto", "insist"},
		{"insólito", "insolit"},
		{"insospechado", "insospech"},
		{"inspección", "inspeccion"},
		{"inspeccionaba", "inspeccion"},
		{"inspeccionado", "inspeccion"},
		{"inspecciones", "inspeccion"},
		{"inspector", "inspector"},
		{"inspectores", "inspector"},
		{"inspiraba", "inspir"},
		{"inspiración", "inspir"},
		{"inspiraciones", "inspir"},
		{"inspirada", "inspir"},
		{"inspirado", "inspir"},
		{"instala", "instal"},
		{"instalación", "instal"},
		{"instalaciones", "instal"},
		{"instalada", "instal"},
		{"instaladas", "instal"},
		{"instalado", "instal"},
		{"instalados", "instal"},
		{"instalamos", "instal"},
		{"instalan", "instal"},
		{"instalando", "instal"},
		{"instalar", "instal"},
		{"instalará", "instal"},
		{"instalarán", "instal"},
		{"instalarlo", "instal"},
		{"instalarlos", "instal"},
		{"instalaron", "instal"},
		{"instalarse", "instal"},
		{"instalen", "instal"},
		{"instaló", "instal"},
		{"instancia", "instanci"},
		{"instancias", "instanci"},
		{"instándolos", "instandol"},
		{"instantánea", "instantane"},
		{"instantáneamente", "instantan"},
		{"instante", "instant"},
		{"instantes", "instant"},
		{"instará", "instar"},
		{"instaurada", "instaur"},
		{"instinto", "instint"},
		{"institución", "institu"},
		{"institucional", "institucional"},
		{"institucionales", "institucional"},
		{"institucionalizar", "institucionaliz"},
		{"institucionalmente", "institucional"},
		{"instituciones", "institu"},
		{"instituida", "institu"},
		{"instituír", "instituir"},
		{"institute", "institut"},
		{"instituto", "institut"},
		{"institutos", "institut"},
		{"instituya", "institu"},
		{"instituyan", "institu"},
		{"instó", "insto"},
		{"instrucción", "instruccion"},
		{"instrucciones", "instruccion"},
		{"instructor", "instructor"},
		{"instructora", "instructor"},
		{"instrumentación", "instrument"},
		{"instrumentado", "instrument"},
		{"instrumentados", "instrument"},
		{"instrumental", "instrumental"},
		{"instrumentar", "instrument"},
		{"instrumentaran", "instrument"},
		{"instrumentistas", "instrument"},
		{"instrumento", "instrument"},
		{"instrumentó", "instrument"},
		{"instrumentos", "instrument"},
		{"instruments", "instruments"},
		{"instruye", "instru"},
		{"instruyó", "instru"},
		{"insubordinación", "insubordin"},
		{"insubstancial", "insubstancial"},
		{"insuficiencia", "insuficient"},
		{"insuficiente", "insuficient"},
		{"insuficientes", "insuficient"},
		{"insulta", "insult"},
		{"insultaba", "insult"},
		{"insultando", "insult"},
		{"insulté", "insult"},
		{"insulto", "insult"},
		{"insultos", "insult"},
		{"insumo", "insum"},
		{"insuperable", "insuper"},
		{"insurgencia", "insurgent"},
		{"insurgentes", "insurgent"},
		{"insurrección", "insurreccion"},
		{"intachable", "intach"},
		{"intacta", "intact"},
		{"intactas", "intact"},
		{"intacto", "intact"},
		{"integra", "integr"},
		{"integración", "integr"},
		{"integracionista", "integracion"},
		{"integrada", "integr"},
		{"integradas", "integr"},
		{"integrado", "integr"},
		{"integrados", "integr"},
		{"integral", "integral"},
		{"integrales", "integral"},
		{"íntegramente", "integr"},
		{"integran", "integr"},
		{"integrante", "integr"},
		{"integrantes", "integr"},
		{"integrar", "integr"},
		{"integrará", "integr"},
		{"integrarán", "integr"},
		{"integraría", "integr"},
		{"integrarlo", "integr"},
		{"integrarlos", "integr"},
		{"integraron", "integr"},
		{"integrarse", "integr"},
		{"integridad", "integr"},
		{"integristas", "integr"},
		{"íntegro", "integr"},
		{"integró", "integr"},
		{"intel", "intel"},
		{"intelectual", "intelectual"},
		{"intelectuales", "intelectual"},
		{"intelectualmente", "intelectual"},
		{"inteligencia", "inteligent"},
		{"inteligente", "inteligent"},
		{"inteligentes", "inteligent"},
		{"intempestivamente", "intempest"},
		{"intempestivo", "intempest"},
		{"intención", "intencion"},
		{"intencionado", "intencion"},
		{"intencional", "intencional"},
		{"intencionales", "intencional"},
		{"intenciones", "intencion"},
		{"intensa", "intens"},
		{"intensamente", "intens"},
		{"intensas", "intens"},
		{"intensidad", "intens"},
		{"intensificado", "intensific"},
		{"intensificando", "intensific"},
		{"intensificar", "intensific"},
		{"intensificó", "intensific"},
		{"intensifiquen", "intensifiqu"},
		{"intensión", "intension"},
		{"intensivamente", "intens"},
		{"intenso", "intens"},
		{"intensos", "intens"},
		{"intenta", "intent"},
		{"intentaba", "intent"},
		{"intentaban", "intent"},
		{"intentado", "intent"},
		{"intentamos", "intent"},
		{"intentan", "intent"},
		{"intentando", "intent"},
		{"intentar", "intent"},
		{"intentará", "intent"},
		{"intentarán", "intent"},
		{"intentaré", "intent"},
		{"intentarlo", "intent"},
		{"intentaron", "intent"},
		{"intento", "intent"},
		{"intentó", "intent"},
		{"intentos", "intent"},
		{"inter", "inter"},
		{"interacción", "interaccion"},
		{"interaciones", "inter"},
		{"interactiva", "interact"},
		{"interactivo", "interact"},
		{"interamericana", "interamerican"},
		{"interbancaria", "interbancari"},
		{"interbancarias", "interbancari"},
		{"interbancario", "interbancari"},
		{"intercambian", "intercambi"},
		{"intercambiando", "intercambi"},
		{"intercambiar", "intercambi"},
		{"intercambié", "intercambi"},
		{"intercambio", "intercambi"},
		{"intercambió", "intercamb"},
		{"interceder", "interced"},
		{"intercedido", "interced"},
		{"intercepción", "intercepcion"},
		{"interceptado", "intercept"},
		{"interceptados", "intercept"},
		{"intercomunicará", "intercomunic"},
		{"interconexión", "interconexion"},
		{"interconstruido", "interconstru"},
		{"intercontinental", "intercontinental"},
		{"intercostal", "intercostal"},
		{"intercostero", "intercoster"},
		{"interdependiente", "interdependient"},
		{"interdisciplinario", "interdisciplinari"},
		{"interés", "interes"},
		{"interesa", "interes"},
		{"interesaba", "interes"},
		{"interesada", "interes"},
		{"interesadas", "interes"},
		{"interesado", "interes"},
		{"interesados", "interes"},
		{"interesan", "interes"},
		{"interesante", "interes"},
		{"interesantes", "interes"},
		{"interesara", "interes"},
		{"interesaría", "interes"},
		{"interesarme", "interes"},
		{"interesaron", "interes"},
		{"interescuadras", "interescuadr"},
		{"interese", "interes"},
		{"interesen", "interes"},
		{"intereses", "interes"},
		{"interfase", "interf"},
		{"interferencia", "interferent"},
		{"interfieran", "interf"},
		{"interfieren", "interfier"},
		{"interfronterizo", "interfronteriz"},
		{"intergrupales", "intergrupal"},
		{"interina", "interin"},
		{"interinato", "interinat"},
		{"interinatos", "interinat"},
		{"interino", "interin"},
		{"interior", "interior"},
		{"interiores", "interior"},
		{"interioridad", "interior"},
		{"interiormente", "interior"},
		{"interlocutora", "interlocutor"},
		{"intermatrix", "intermatrix"},
		{"intermedia", "intermedi"},
		{"intermediación", "intermedi"},
		{"intermediaria", "intermediari"},
		{"intermediarias", "intermediari"},
		{"intermediario", "intermediari"},
		{"intermediarios", "intermediari"},
		{"intermediarista", "intermediar"},
		{"intermedio", "intermedi"},
		{"intermedios", "intermedi"},
		{"intermitentes", "intermitent"},
		{"interna", "intern"},
		{"internacional", "internacional"},
		{"internacionales", "internacional"},
		{"internacionalización", "internacionaliz"},
		{"", ""},
		{"internacionalmente", "internacional"},
		{"internado", "intern"},
		{"internamente", "intern"},
		{"internarnos", "intern"},
		{"internaron", "intern"},
		{"internas", "intern"},
		{"international", "international"},
		{"interno", "intern"},
		{"internó", "intern"},
		{"internos", "intern"},
		{"interpeló", "interpel"},
		{"interponer", "interpon"},
		{"interpreta", "interpret"},
		{"interpretaba", "interpret"},
		{"interpretación", "interpret"},
		{"interpretaciones", "interpret"},
		{"interpretado", "interpret"},
		{"interpretamos", "interpret"},
		{"interpretan", "interpret"},
		{"interpretando", "interpret"},
		{"interpretándolo", "interpret"},
		{"interpretar", "interpret"},
		{"interpretara", "interpret"},
		{"interpretará", "interpret"},
		{"interpretarán", "interpret"},
		{"interpretaremos", "interpret"},
		{"interpretarlas", "interpret"},
		{"interpretarse", "interpret"},
		{"interprete", "interpret"},
		{"intérprete", "interpret"},
		{"interpreten", "interpret"},
		{"intérpretes", "interpret"},
		{"interpretó", "interpret"},
		{"interpuestas", "interpuest"},
		{"interpuestos", "interpuest"},
		{"interpuso", "interpus"},
		{"interrelacionarse", "interrelacion"},
		{"interrogada", "interrog"},
		{"interrogado", "interrog"},
		{"interrogados", "interrog"},
		{"interrogante", "interrog"},
		{"interrogar", "interrog"},
		{"interrogó", "interrog"},
		{"interrumpa", "interrump"},
		{"interrumpe", "interrump"},
		{"interrumpía", "interrump"},
		{"interrumpido", "interrump"},
		{"interrumpidos", "interrump"},
		{"interrumpieron", "interrump"},
		{"interrumpió", "interrump"},
		{"interrumpir", "interrump"},
		{"interrupción", "interrupcion"},
		{"interruptor", "interruptor"},
		{"interticket", "interticket"},
		{"intervalos", "interval"},
		{"intervención", "intervencion"},
		{"intervenciones", "intervencion"},
		{"intervencionismo", "intervencion"},
		{"intervendrá", "intervendr"},
		{"intervenga", "interveng"},
		{"intervengan", "interveng"},
		{"intervenido", "interven"},
		{"intervenidos", "interven"},
		{"intervenir", "interven"},
		{"intervenirse", "interven"},
		{"interventores", "interventor"},
		{"interviene", "intervien"},
		{"intervienen", "intervien"},
		{"interviniera", "intervin"},
		{"intervinieron", "intervin"},
		{"intervino", "intervin"},
		{"intestino", "intestin"},
		{"íntima", "intim"},
		{"íntimas", "intim"},
		{"intimate", "intimat"},
		{"intimidad", "intim"},
		{"intimidados", "intimid"},
		{"intimidando", "intimid"},
		{"íntimo", "intim"},
		{"íntimos", "intim"},
		{"intolerable", "intoler"},
		{"intolerancia", "intoler"},
		{"intoxicación", "intox"},
		{"intoxicados", "intoxic"},
		{"intoxicando", "intoxic"},
		{"intralox", "intralox"},
		{"intramuscular", "intramuscul"},
		{"intransigencia", "intransigent"},
		{"intransigentemente", "intransigent"},
		{"intrascendente", "intrascendent"},
		{"intrascendentes", "intrascendent"},
		{"intrincadas", "intrinc"},
		{"intrínseca", "intrinsec"},
		{"introducción", "introduccion"},
		{"introducido", "introduc"},
		{"introduciendo", "introduc"},
		{"introducir", "introduc"},
		{"introducirán", "introduc"},
		{"introducirles", "introduc"},
		{"introducirse", "introduc"},
		{"introductorio", "introductori"},
		{"introdujera", "introdujer"},
		{"introdujeron", "introdujeron"},
		{"introdujo", "introduj"},
		{"introduzca", "introduzc"},
		{"introspectivo", "introspect"},
		{"introvertido", "introvert"},
		{"intrusión", "intrusion"},
		{"intuición", "intuicion"},
		{"intuitiva", "intuit"},
		{"inundación", "inund"},
		{"inundaciones", "inund"},
		{"inundadas", "inund"},
		{"inundado", "inund"},
		{"inusitada", "inusit"},
		{"inusitado", "inusit"},
		{"inusual", "inusual"},
		{"inútil", "inutil"},
		{"inútiles", "inutil"},
		{"inútilmente", "inutil"},
		{"invadidas", "invad"},
		{"invadido", "invad"},
		{"invadieron", "invad"},
		{"invadió", "invad"},
		{"invadir", "invad"},
		{"invalidan", "invalid"},
		{"invaluable", "invalu"},
		{"invasión", "invasion"},
		{"invasiones", "invasion"},
		{"invencible", "invenc"},
		{"invención", "invencion"},
		{"inventado", "invent"},
		{"inventando", "invent"},
		{"inventar", "invent"},
		{"inventario", "inventari"},
		{"invente", "invent"},
		{"inventé", "invent"},
		{"inventiva", "invent"},
		{"invento", "invent"},
		{"inventó", "invent"},
		{"inverlat", "inverlat"},
		{"inverméxico", "invermex"},
		{"invernadero", "invernader"},
		{"invernal", "invernal"},
		{"inverosímil", "inverosimil"},
		{"inversa", "invers"},
		{"inversión", "inversion"},
		{"inversiones", "inversion"},
		{"inversionista", "inversion"},
		{"inversionistas", "inversion"},
		{"invertía", "invert"},
		{"invertidas", "invert"},
		{"invertido", "invert"},
		{"invertidos", "invert"},
		{"invertimos", "invert"},
		{"invertir", "invert"},
		{"invertirá", "invert"},
		{"invertirán", "invert"},
		{"invertiremos", "invert"},
		{"invertirían", "invert"},
		{"invertirla", "invert"},
		{"invertirle", "invert"},
		{"investidura", "investidur"},
		{"investiga", "investig"},
		{"investigaban", "investig"},
		{"investigación", "investig"},
		{"investigaciones", "investig"},
		{"investigado", "investig"},
		{"investigador", "investig"},
		{"investigadora", "investig"},
		{"investigadores", "investig"},
		{"investigados", "investig"},
		{"investigan", "investig"},
		{"investigando", "investig"},
		{"investigar", "investig"},
		{"investigará", "investig"},
		{"investigaran", "investig"},
		{"investigarse", "investig"},
		{"investigó", "investig"},
		{"investigue", "investig"},
		{"investiguen", "investig"},
		{"invicta", "invict"},
		{"invicto", "invict"},
		{"invictos", "invict"},
		{"invierno", "inviern"},
		{"invierta", "inviert"},
		{"invierte", "inviert"},
		{"invierten", "inviert"},
		{"invirtiendo", "invirt"},
		{"invirtieron", "invirt"},
		{"invirtió", "invirt"},
		{"invisiblemente", "invis"},
		{"invita", "invit"},
		{"invitaba", "invit"},
		{"invitación", "invit"},
		{"invitaciones", "invit"},
		{"invitadas", "invit"},
		{"invitado", "invit"},
		{"invitados", "invit"},
		{"invitamos", "invit"},
		{"invitan", "invit"},
		{"invitando", "invit"},
		{"invitar", "invit"},
		{"invitarlo", "invit"},
		{"invitaron", "invit"},
		{"invite", "invit"},
		{"inviten", "invit"},
		{"invito", "invit"},
		{"invitó", "invit"},
		{"invocando", "invoc"},
		{"involucra", "involucr"},
		{"involucrada", "involucr"},
		{"involucradas", "involucr"},
		{"involucrado", "involucr"},
		{"involucrados", "involucr"},
		{"involucramiento", "involucr"},
		{"involucran", "involucr"},
		{"involucrar", "involucr"},
		{"involucrarlo", "involucr"},
		{"involucrarme", "involucr"},
		{"involucrarse", "involucr"},
		{"involucre", "involucr"},
		{"involucro", "involucr"},
		{"involuntario", "involuntari"},
		{"inyección", "inyeccion"},
		{"inyectar", "inyect"},
		{"inyecte", "inyect"},
		{"inyectó", "inyect"},
		{"ionizados", "ioniz"},
		{"ionizantes", "ioniz"},
		{"ip", "ip"},
		{"ipiña", "ipiñ"},
		{"ir", "ir"},
		{"ira", "ira"},
		{"irá", "ira"},
		{"iracheta", "irachet"},
		{"iraheta", "irahet"},
		{"iran", "iran"},
		{"irán", "iran"},
		{"iraní", "iran"},
		{"iraníes", "iran"},
		{"irapuato", "irapuat"},
		{"iraq", "iraq"},
		{"iraquí", "iraqu"},
		{"iraquíes", "iraqu"},
		{"iremos", "irem"},
		{"iría", "iri"},
		{"irían", "iri"},
		{"iriarte", "iriart"},
		{"irigoyen", "irigoy"},
		{"irineo", "irine"},
		{"irlandés", "irlandes"},
		{"irlandeses", "irlandes"},
		{"irlos", "irlos"},
		{"irma", "irma"},
		{"irme", "irme"},
		{"irnos", "irnos"},
		{"ironía", "iron"},
		{"irónico", "iron"},
		{"ironizaba", "ironiz"},
		{"irracional", "irracional"},
		{"irrealistas", "irreal"},
		{"irrealizables", "irrealiz"},
		{"irreconocible", "irreconoc"},
		{"irrecuperable", "irrecuper"},
		{"irredentos", "irredent"},
		{"irrefutable", "irrefut"},
		{"irregular", "irregul"},
		{"irregulares", "irregular"},
		{"irregularidad", "irregular"},
		{"irregularidades", "irregular"},
		{"irrelevante", "irrelev"},
		{"irrelevantes", "irrelev"},
		{"irremisiblemente", "irremis"},
		{"irrenunciabilidad", "irrenunci"},
		{"irresoluto", "irresolut"},
		{"irresolutos", "irresolut"},
		{"irresponsabilidad", "irrespons"},
		{"irresponsable", "irrespons"},
		{"irresponsables", "irrespons"},
		{"irreversible", "irrevers"},
		{"irrita", "irrit"},
		{"irritación", "irrit"},
		{"irritaciones", "irrit"},
		{"irritado", "irrit"},
		{"irritan", "irrit"},
		{"irritantes", "irrit"},
		{"irritarse", "irrit"},
		{"irruga", "irrug"},
		{"irse", "irse"},
		{"irte", "irte"},
		{"irvin", "irvin"},
		{"isaac", "isaac"},
		{"isabel", "isabel"},
		{"isabela", "isabel"},
		{"isaias", "isai"},
		{"isidro", "isidr"},
		{"isla", "isla"},
		{"islam", "islam"},
		{"islámico", "islam"},
		{"island", "island"},
		{"islandia", "islandi"},
		{"islas", "islas"},
		{"isleño", "isleñ"},
		{"isleños", "isleñ"},
		{"isley", "isley"},
		{"ismael", "ismael"},
		{"israel", "israel"},
		{"israelí", "israel"},
		{"israelíes", "israel"},
		{"issste", "issste"},
		{"isssteleon", "isssteleon"},
		{"isssteleón", "isssteleon"},
		{"istmo", "istmo"},
		{"italia", "itali"},
		{"italiana", "italian"},
		{"italiano", "italian"},
		{"italianos", "italian"},
		{"itam", "itam"},
		{"itamaraty", "itamaraty"},
		{"itesm", "itesm"},
		{"itinerante", "itiner"},
		{"itinerantes", "itiner"},
		{"itinerario", "itinerari"},
		{"itinerarios", "itinerari"},
		{"itoloca", "itoloc"},
		{"itsmo", "itsmo"},
		{"itzá", "itza"},
		{"iv", "iv"},
		{"iva", "iva"},
		{"iván", "ivan"},
		{"ivo", "ivo"},
		{"iwai", "iwai"},
		{"ix", "ix"},
		{"ixalapa", "ixalap"},
		{"ixcaquixtla", "ixcaquixtl"},
		{"izar", "izar"},
		{"izetbegovic", "izetbegovic"},
		{"izquierda", "izquierd"},
		{"izquierdas", "izquierd"},
		{"izquierdo", "izquierd"},
		{"iztapalapa", "iztapalap"},
		{"izvestia", "izvesti"},
		{"j", "j"},
		{"jaafar", "jaaf"},
		{"jaboneras", "jaboner"},
		{"jacinta", "jacint"},
		{"jack", "jack"},
		{"jackeline", "jackelin"},
		{"jackie", "jacki"},
		{"jackson", "jackson"},
		{"jacobo", "jacob"},
		{"jacobs", "jacobs"},
		{"jacques", "jacqu"},
		{"jactancio", "jactanci"},
		{"jacte", "jact"},
		{"jade", "jad"},
		{"jaguares", "jaguar"},
		{"jaguarundis", "jaguarundis"},
		{"jaibos", "jaib"},
		{"jaime", "jaim"},
		{"jalaba", "jal"},
		{"jalado", "jal"},
		{"jalando", "jal"},
		{"jalapeños", "jalapeñ"},
		{"jalar", "jal"},
		{"jalarse", "jal"},
		{"jale", "jal"},
		{"jalisciense", "jalisciens"},
		{"jalisco", "jalisc"},
		{"jaló", "jal"},
		{"jalones", "jalon"},
		{"jam", "jam"},
		{"jamaiquina", "jamaiquin"},
		{"jamás", "jamas"},
		{"james", "jam"},
		{"jamón", "jamon"},
		{"jana", "jan"},
		{"janeiro", "janeir"},
		{"janice", "janic"},
		{"japon", "japon"},
		{"japón", "japon"},
		{"japonés", "japones"},
		{"japonesa", "japones"},
		{"japonesas", "japones"},
		{"japoneses", "japones"},
		{"jaque", "jaqu"},
		{"jaramillo", "jaramill"},
		{"jardín", "jardin"},
		{"jardinería", "jardin"},
		{"jardinero", "jardiner"},
		{"jardines", "jardin"},
		{"jarocha", "jaroch"},
		{"jarochas", "jaroch"},
		{"jarra", "jarr"},
		{"jarro", "jarr"},
		{"jarry", "jarry"},
		{"jason", "jason"},
		{"jasso", "jass"},
		{"jaula", "jaul"},
		{"javier", "javi"},
		{"jazz", "jazz"},
		{"jazzista", "jazzist"},
		{"jc", "jc"},
		{"jcpenney", "jcpenney"},
		{"jean", "jean"},
		{"jeans", "jeans"},
		{"jefa", "jef"},
		{"jefatura", "jefatur"},
		{"jefaturados", "jefatur"},
		{"jefe", "jef"},
		{"jefes", "jef"},
		{"jeff", "jeff"},
		{"jefferson", "jefferson"},
		{"jehová", "jehov"},
		{"jerarcas", "jerarc"},
		{"jerarquía", "jerarqu"},
		{"jeremie", "jeremi"},
		{"jeremy", "jeremy"},
		{"jerga", "jerg"},
		{"jerónimo", "jeronim"},
		{"jersey", "jersey"},
		{"jesés", "jeses"},
		{"jesse", "jess"},
		{"jessica", "jessic"},
		{"jesús", "jesus"},
		{"jethro", "jethr"},
		{"jetta", "jett"},
		{"jim", "jim"},
		{"jiménez", "jimenez"},
		{"jimmy", "jimmy"},
		{"jinete", "jinet"},
		{"jinetear", "jinet"},
		{"jinetes", "jinet"},
		{"jingles", "jingl"},
		{"jo", "jo"},
		{"joachim", "joachim"},
		{"joao", "joa"},
		{"joaquín", "joaquin"},
		{"jobs", "jobs"},
		{"joe", "joe"},
		{"joel", "joel"},
		{"john", "john"},
		{"johnny", "johnny"},
		{"johns", "johns"},
		{"johnson", "johnson"},
		{"jol", "jol"},
		{"jolalpán", "jolalpan"},
		{"jolopo", "jolop"},
		{"jonas", "jon"},
		{"jones", "jon"},
		{"jonrón", "jonron"},
		{"jorge", "jorg"},
		{"jornada", "jorn"},
		{"jornadas", "jorn"},
		{"joroba", "jorob"},
		{"jorobas", "jorob"},
		{"josé", "jos"},
		{"josefina", "josefin"},
		{"joselito", "joselit"},
		{"joseph", "joseph"},
		{"jour", "jour"},
		{"journal", "journal"},
		{"joven", "jov"},
		{"jóven", "jov"},
		{"jovencita", "jovencit"},
		{"jovencitas", "jovencit"},
		{"jovencito", "jovencit"},
		{"jovencitos", "jovencit"},
		{"jóvenes", "joven"},
		{"jovial", "jovial"},
		{"joya", "joy"},
		{"joyas", "joy"},
		{"joyera", "joyer"},
		{"joyería", "joy"},
		{"joyero", "joyer"},
		{"joyeros", "joyer"},
		{"joyitas", "joyit"},
		{"jr", "jr"},
		{"juan", "juan"},
		{"juana", "juan"},
		{"juarez", "juarez"},
		{"juárez", "juarez"},
		{"jubilado", "jubil"},
		{"jubilados", "jubil"},
		{"júbilo", "jubil"},
		{"jubilosa", "jubil"},
		{"judicial", "judicial"},
		{"judiciales", "judicial"},
		{"judío", "judi"},
		{"judíos", "judi"},
		{"judith", "judith"},
		{"judoka", "judok"},
		{"jueceo", "juece"},
		{"jueces", "juec"},
		{"juega", "jueg"},
		{"juegan", "jueg"},
		{"juegas", "jueg"},
		{"juego", "jueg"},
		{"juegos", "jueg"},
		{"juegue", "jueg"},
		{"jueguen", "jueg"},
		{"jueves", "juev"},
		{"juez", "juez"},
		{"jugaba", "jug"},
		{"jugaban", "jug"},
		{"jugada", "jug"},
		{"jugado", "jug"},
		{"jugador", "jugador"},
		{"jugadoras", "jugador"},
		{"jugadores", "jugador"},
		{"jugados", "jug"},
		{"jugando", "jug"},
		{"jugar", "jug"},
		{"jugará", "jug"},
		{"jugaran", "jug"},
		{"jugarán", "jug"},
		{"jugaremos", "jug"},
		{"jugaría", "jug"},
		{"jugaron", "jug"},
		{"jugo", "jug"},
		{"jugó", "jug"},
		{"jugosa", "jugos"},
		{"jugué", "jug"},
		{"juguete", "juguet"},
		{"juguetes", "juguet"},
		{"juguetito", "juguetit"},
		{"juicio", "juici"},
		{"juicios", "juici"},
		{"julia", "juli"},
		{"julián", "julian"},
		{"julieta", "juliet"},
		{"julio", "juli"},
		{"julius", "julius"},
		{"juncal", "juncal"},
		{"junco", "junc"},
		{"jung", "jung"},
		{"junio", "juni"},
		{"junior", "junior"},
		{"junquia", "junqui"},
		{"junta", "junt"},
		{"juntando", "junt"},
		{"juntar", "junt"},
		{"juntaron", "junt"},
		{"juntas", "junt"},
		{"junto", "junt"},
		{"juntos", "junt"},
		{"jurado", "jur"},
		{"juramentado", "jurament"},
		{"juramentaron", "jurament"},
		{"juramentó", "jurament"},
		{"jurídica", "jurid"},
		{"jurídicamente", "jurid"},
		{"jurídicas", "jurid"},
		{"jurídico", "jurid"},
		{"jurídicos", "jurid"},
		{"jurisdicción", "jurisdiccion"},
		{"juristas", "jurist"},
		{"juro", "jur"},
		{"justa", "just"},
		{"justice", "justic"},
		{"justicia", "justici"},
		{"justiciero", "justicier"},
		{"justifica", "justif"},
		{"justificable", "justific"},
		{"justificación", "justif"},
		{"justificada", "justific"},
		{"justificadas", "justific"},
		{"justifican", "justific"},
		{"justificando", "justific"},
		{"justificantes", "justif"},
		{"justificar", "justific"},
		{"justificará", "justific"},
		{"justificarán", "justific"},
		{"justificare", "justificar"},
		{"justificó", "justific"},
		{"justifique", "justifiqu"},
		{"justino", "justin"},
		{"justo", "just"},
		{"justos", "just"},
		{"juvenil", "juvenil"},
		{"juveniles", "juvenil"},
		{"juventino", "juventin"},
		{"juventud", "juventud"},
		{"juzgaban", "juzg"},
		{"juzgada", "juzg"},
		{"juzgado", "juzg"},
		{"juzgador", "juzgador"},
		{"juzgados", "juzg"},
		{"juzgar", "juzg"},
		{"juzgarlo", "juzg"},
		{"juzgue", "juzg"},
		{"k", "k"},
		{"kahlo", "kahl"},
		{"kaiman", "kaim"},
		{"kansai", "kansai"},
		{"kantor", "kantor"},
		{"kaoteng", "kaoteng"},
		{"karam", "karam"},
		{"karate", "karat"},
		{"karcz", "karcz"},
		{"kardex", "kardex"},
		{"karen", "kar"},
		{"karina", "karin"},
		{"karl", "karl"},
		{"karla", "karl"},
		{"katheyn", "katheyn"},
		{"katia", "kati"},
		{"kaveh", "kaveh"},
		{"kay", "kay"},
		{"kb", "kb"},
		{"kbytes", "kbytes"},
		{"keldvsh", "keldvsh"},
		{"kelly", "kelly"},
		{"kelso", "kels"},
		{"kempes", "kemp"},
		{"kenia", "keni"},
		{"keniano", "kenian"},
		{"kenianos", "kenian"},
		{"keniata", "keniat"},
		{"kennan", "kenn"},
		{"kennedy", "kennedy"},
		{"kenneth", "kenneth"},
		{"kenny", "kenny"},
		{"kensington", "kensington"},
		{"kent", "kent"},
		{"kentenich", "kentenich"},
		{"kenworth", "kenworth"},
		{"kerosina", "kerosin"},
		{"kevin", "kevin"},
		{"key", "key"},
		{"keynesianismo", "keynesian"},
		{"kg", "kg"},
		{"kgb", "kgb"},
		{"khasbulatov", "khasbulatov"},
		{"kibeho", "kibeh"},
		{"kidman", "kidm"},
		{"kiko", "kik"},
		{"kilo", "kil"},
		{"kilogramos", "kilogr"},
		{"kilometraje", "kilometraj"},
		{"kilometro", "kilometr"},
		{"kilómetro", "kilometr"},
		{"kilómetros", "kilometr"},
		{"kilos", "kil"},
		{"kim", "kim"},
		{"kima", "kim"},
		{"king", "king"},
		{"kinshasa", "kinshas"},
		{"kipper", "kipp"},
		{"kissinger", "kissing"},
		{"kit", "kit"},
		{"km", "km"},
		{"kmart", "kmart"},
		{"knight", "knight"},
		{"ko", "ko"},
		{"koblenz", "koblenz"},
		{"koeman", "koem"},
		{"kong", "kong"},
		{"korshak", "korshak"},
		{"kosovo", "kosov"},
		{"krafchick", "krafchick"},
		{"kremlin", "kremlin"},
		{"kresak", "kresak"},
		{"kretch", "kretch"},
		{"krieger", "krieg"},
		{"krill", "krill"},
		{"kristof", "kristof"},
		{"kruger", "krug"},
		{"kruschev", "kruschev"},
		{"kubli", "kubli"},
		{"kuehnheim", "kuehnheim"},
		{"kumate", "kumat"},
		{"kuniaki", "kuniaki"},
		{"kuperlmann", "kuperlmann"},
		{"kuramura", "kuramur"},
		{"kurdos", "kurd"},
		{"kutugata", "kutugat"},
		{"kuwait", "kuwait"},
		{"kuwaití", "kuwait"},
		{"kytty", "kytty"},
		{"l", "l"},
		{"la", "la"},
		{"labastida", "labast"},
		{"laberinto", "laberint"},
		{"labiche", "labich"},
		{"labor", "labor"},
		{"labora", "labor"},
		{"laboral", "laboral"},
		{"laborales", "laboral"},
		{"laborando", "labor"},
		{"laborar", "labor"},
		{"laboratorio", "laboratori"},
		{"laboratorios", "laboratori"},
		{"laborde", "labord"},
		{"labores", "labor"},
		{"laborioso", "labori"},
		{"laboró", "labor"},
		{"labra", "labr"},
		{"labrecque", "labrecqu"},
		{"lacónicos", "lacon"},
		{"lácteos", "lacte"},
		{"lada", "lad"},
		{"ladarse", "lad"},
		{"ladas", "lad"},
		{"ladatel", "ladatel"},
		{"ladera", "lader"},
		{"laderas", "lader"},
		{"ladino", "ladin"},
		{"ladinos", "ladin"},
		{"lado", "lad"},
		{"lados", "lad"},
		{"ladrón", "ladron"},
		{"ladrones", "ladron"},
		{"lagerfeld", "lagerfeld"},
		{"lago", "lag"},
		{"lagos", "lag"},
		{"lágrimas", "lagrim"},
		{"lagüera", "lagüer"},
		{"laguna", "lagun"},
		{"lagunas", "lagun"},
		{"lagunera", "laguner"},
		{"laguneros", "laguner"},
		{"laicos", "laic"},
		{"lajas", "laj"},
		{"lajous", "lajous"},
		{"lake", "lak"},
		{"lala", "lal"},
		{"lamadrid", "lamadr"},
		{"lamarche", "lamarch"},
		{"lambisconería", "lambiscon"},
		{"lambretón", "lambreton"},
		{"lamen", "lam"},
		{"lamenta", "lament"},
		{"lamentable", "lament"},
		{"lamentablemente", "lament"},
		{"lamentables", "lament"},
		{"lamentaron", "lament"},
		{"lamento", "lament"},
		{"lamentó", "lament"},
		{"lámina", "lamin"},
		{"laminadora", "lamin"},
		{"láminas", "lamin"},
		{"lamosa", "lamos"},
		{"lámparas", "lamp"},
		{"lana", "lan"},
		{"lanari", "lanari"},
		{"lance", "lanc"},
		{"lancé", "lanc"},
		{"lancen", "lanc"},
		{"lancha", "lanch"},
		{"lanchero", "lancher"},
		{"landois", "landois"},
		{"landru", "landru"},
		{"lang", "lang"},
		{"langange", "langang"},
		{"langley", "langley"},
		{"lantana", "lantan"},
		{"lantastic", "lantastic"},
		{"lanz", "lanz"},
		{"lanza", "lanz"},
		{"lanzaba", "lanz"},
		{"lanzada", "lanz"},
		{"lanzadas", "lanz"},
		{"lanzado", "lanz"},
		{"lanzador", "lanzador"},
		{"lanzadores", "lanzador"},
		{"lanzagranadas", "lanzagran"},
		{"lanzamiento", "lanzamient"},
		{"lanzamientos", "lanzamient"},
		{"lanzar", "lanz"},
		{"lanzará", "lanz"},
		{"lanzaran", "lanz"},
		{"lanzarán", "lanz"},
		{"lanzaron", "lanz"},
		{"lanzarse", "lanz"},
		{"lanzó", "lanz"},
		{"lápiz", "lapiz"},
		{"lapso", "laps"},
		{"lara", "lar"},
		{"lareau", "lareau"},
		{"laredo", "lared"},
		{"laredos", "lared"},
		{"lares", "lar"},
		{"larga", "larg"},
		{"largamente", "larg"},
		{"largas", "larg"},
		{"largo", "larg"},
		{"largometraje", "largometraj"},
		{"largometrajes", "largometraj"},
		{"largos", "larg"},
		{"larguirucho", "larguiruch"},
		{"larguísimo", "larguisim"},
		{"larios", "lari"},
		{"larralde", "larrald"},
		{"larry", "larry"},
		{"las", "las"},
		{"lascivo", "lasciv"},
		{"laser", "las"},
		{"láser", "las"},
		{"lasercard", "lasercard"},
		{"láseres", "laser"},
		{"lasser", "lass"},
		{"lástima", "lastim"},
		{"lastimada", "lastim"},
		{"lastimadas", "lastim"},
		{"lastimar", "lastim"},
		{"lastimeramente", "lastimer"},
		{"lata", "lat"},
		{"latas", "lat"},
		{"latente", "latent"},
		{"lateral", "lateral"},
		{"laterales", "lateral"},
		{"lateralidad", "lateral"},
		{"latido", "lat"},
		{"latin", "latin"},
		{"latina", "latin"},
		{"latino", "latin"},
		{"latinoamérica", "latinoamer"},
		{"latinoamericanas", "latinoamerican"},
		{"latinoamericano", "latinoamerican"},
		{"latinoamericanos", "latinoamerican"},
		{"latinos", "latin"},
		{"latitudes", "latitud"},
		{"latón", "laton"},
		{"laudos", "laud"},
		{"laudrup", "laudrup"},
		{"laura", "laur"},
		{"laureano", "laurean"},
		{"lauren", "laur"},
		{"laurence", "laurenc"},
		{"laurent", "laurent"},
		{"lava", "lav"},
		{"lavacoches", "lavacoch"},
		{"lavaderos", "lavader"},
		{"lavado", "lav"},
		{"lavadora", "lavador"},
		{"lavalle", "lavall"},
		{"lavan", "lav"},
		{"lavar", "lav"},
		{"lavarlo", "lav"},
		{"lavercombe", "lavercomb"},
		{"lavolpe", "lavolp"},
		{"laxitud", "laxitud"},
		{"layla", "layl"},
		{"lázaro", "lazar"},
		{"lazcano", "lazcan"},
		{"lazo", "laz"},
		{"lazos", "laz"},
		{"le", "le"},
		{"leal", "leal"},
		{"leales", "leal"},
		{"lealtad", "lealt"},
		{"léase", "leas"},
		{"leblanc", "leblanc"},
		{"lección", "leccion"},
		{"lecciones", "leccion"},
		{"leche", "lech"},
		{"lechería", "lech"},
		{"lechero", "lecher"},
		{"lecho", "lech"},
		{"lechuga", "lechug"},
		{"leckie", "lecki"},
		{"lecomte", "lecomt"},
		{"lector", "lector"},
		{"lectores", "lector"},
		{"lectura", "lectur"},
		{"lecturas", "lectur"},
		{"ledezma", "ledezm"},
		{"ledo", "led"},
		{"lee", "lee"},
		{"leeds", "leeds"},
		{"leen", "leen"},
		{"leer", "leer"},
		{"leerán", "leeran"},
		{"leerlo", "leerl"},
		{"legado", "leg"},
		{"legal", "legal"},
		{"legales", "legal"},
		{"legalidad", "legal"},
		{"legalizar", "legaliz"},
		{"legalizó", "legaliz"},
		{"legalmente", "legal"},
		{"legar", "leg"},
		{"legendario", "legendari"},
		{"legisla", "legisl"},
		{"legislación", "legisl"},
		{"legislador", "legisl"},
		{"legisladora", "legisl"},
		{"legisladores", "legisl"},
		{"legislan", "legisl"},
		{"legislar", "legisl"},
		{"legislativa", "legisl"},
		{"legislativas", "legisl"},
		{"legislativo", "legisl"},
		{"legislativos", "legisl"},
		{"legislatura", "legislatur"},
		{"legislaturas", "legislatur"},
		{"legisle", "legisl"},
		{"legislen", "legisl"},
		{"legista", "legist"},
		{"legítima", "legitim"},
		{"legítimadefensa", "legitimadefens"},
		{"legítimamente", "legitim"},
		{"legitimidad", "legitim"},
		{"legítimo", "legitim"},
		{"legorreta", "legorret"},
		{"legrá", "legr"},
		{"leguas", "legu"},
		{"legue", "leg"},
		{"legumbres", "legumbr"},
		{"lehnert", "lehnert"},
		{"leí", "lei"},
		{"leía", "lei"},
		{"leída", "leid"},
		{"leído", "leid"},
		{"lejana", "lejan"},
		{"lejanas", "lejan"},
		{"lejano", "lejan"},
		{"lejanos", "lejan"},
		{"lejos", "lej"},
		{"lema", "lem"},
		{"lemas", "lem"},
		{"lemus", "lemus"},
		{"leña", "leñ"},
		{"lencería", "lenc"},
		{"lendl", "lendl"},
		{"leñero", "leñer"},
		{"lengua", "lengu"},
		{"lenguaje", "lenguaj"},
		{"lennard", "lennard"},
		{"lenny", "lenny"},
		{"lenta", "lent"},
		{"lentamente", "lent"},
		{"lentejuela", "lentejuel"},
		{"lentejuelas", "lentejuel"},
		{"lentes", "lent"},
		{"lentitud", "lentitud"},
		{"lento", "lent"},
		{"lentos", "lent"},
		{"leo", "leo"},
		{"león", "leon"},
		{"leonard", "leonard"},
		{"leonarda", "leonard"},
		{"leonardo", "leonard"},
		{"leones", "leon"},
		{"leonor", "leonor"},
		{"leopoldo", "leopold"},
		{"leotardos", "leotard"},
		{"léoz", "leoz"},
		{"lerma", "lerm"},
		{"lermas", "lerm"},
		{"lerna", "lern"},
		{"leroux", "leroux"},
		{"les", "les"},
		{"lesbianas", "lesbian"},
		{"lesión", "lesion"},
		{"lesiona", "lesion"},
		{"lesionada", "lesion"},
		{"lesionadas", "lesion"},
		{"lesionado", "lesion"},
		{"lesionados", "lesion"},
		{"lesionan", "lesion"},
		{"lesionando", "lesion"},
		{"lesionar", "lesion"},
		{"lesiones", "lesion"},
		{"lesiono", "lesion"},
		{"lesionó", "lesion"},
		{"lessing", "lessing"},
		{"letal", "letal"},
		{"letales", "letal"},
		{"leticia", "letici"},
		{"letra", "letr"},
		{"letras", "letr"},
		{"letrero", "letrer"},
		{"letreros", "letrer"},
		{"letrina", "letrin"},
		{"letrinas", "letrin"},
		{"letrista", "letrist"},
		{"lett", "lett"},
		{"leucemia", "leucemi"},
		{"levanta", "levant"},
		{"levantaba", "levant"},
		{"levantada", "levant"},
		{"levantado", "levant"},
		{"levantados", "levant"},
		{"levantamiento", "levant"},
		{"levantando", "levant"},
		{"levantándose", "levant"},
		{"levantante", "levant"},
		{"levantar", "levant"},
		{"levantaran", "levant"},
		{"levantarán", "levant"},
		{"levantarás", "levant"},
		{"levantarme", "levant"},
		{"levantaron", "levant"},
		{"levantarse", "levant"},
		{"levante", "levant"},
		{"levanten", "levant"},
		{"levantó", "levant"},
		{"levati", "levati"},
		{"leve", "lev"},
		{"levemente", "levement"},
		{"leves", "lev"},
		{"levin", "levin"},
		{"levín", "levin"},
		{"levítico", "levit"},
		{"levy", "levy"},
		{"ley", "ley"},
		{"leyenda", "leyend"},
		{"leyendas", "leyend"},
		{"leyendo", "leyend"},
		{"leyeron", "leyeron"},
		{"leyes", "ley"},
		{"leyó", "ley"},
		{"leyva", "leyv"},
		{"lianas", "lian"},
		{"libanés", "libanes"},
		{"libanesa", "libanes"},
		{"líbano", "liban"},
		{"libera", "liber"},
		{"liberación", "liber"},
		{"liberada", "liber"},
		{"liberado", "liber"},
		{"liberal", "liberal"},
		{"liberales", "liberal"},
		{"liberalismo", "liberal"},
		{"liberalización", "liberaliz"},
		{"liberalizar", "liberaliz"},
		{"liberan", "liber"},
		{"liberar", "liber"},
		{"liberará", "liber"},
		{"liberarlo", "liber"},
		{"liberaron", "liber"},
		{"liberia", "liberi"},
		{"libero", "liber"},
		{"líbero", "liber"},
		{"liberó", "liber"},
		{"libertad", "libert"},
		{"libertades", "libertad"},
		{"libertador", "libert"},
		{"libertario", "libertari"},
		{"libidinosamente", "libidin"},
		{"libidio", "libidi"},
		{"libramiento", "libramient"},
		{"libran", "libr"},
		{"librando", "libr"},
		{"librará", "libr"},
		{"librarla", "libr"},
		{"librarse", "libr"},
		{"librarte", "librart"},
		{"libras", "libr"},
		{"libre", "libr"},
		{"librerías", "libr"},
		{"libres", "libr"},
		{"libreta", "libret"},
		{"librito", "librit"},
		{"libro", "libr"},
		{"libró", "libr"},
		{"libros", "libr"},
		{"lic", "lic"},
		{"licea", "lice"},
		{"licencia", "licenci"},
		{"licenciado", "licenci"},
		{"licencias", "licenci"},
		{"licenciaturas", "licenciatur"},
		{"licha", "lich"},
		{"licitación", "licit"},
		{"licitaciones", "licit"},
		{"licitará", "licit"},
		{"lícito", "licit"},
		{"liconsa", "licons"},
		{"licor", "licor"},
		{"licuada", "licu"},
		{"lid", "lid"},
		{"líder", "lid"},
		{"liderato", "liderat"},
		{"liderazgo", "liderazg"},
		{"liderazgos", "liderazg"},
		{"lidereados", "lider"},
		{"liderear", "lider"},
		{"líderes", "lider"},
		{"lideresa", "lideres"},
		{"lideresas", "lideres"},
		{"lidia", "lidi"},
		{"lidiado", "lidi"},
		{"lidiaron", "lidi"},
		{"lidice", "lidic"},
		{"lidió", "lid"},
		{"lido", "lid"},
		{"lienzo", "lienz"},
		{"liera", "lier"},
		{"lievin", "lievin"},
		{"lifane", "lifan"},
		{"life", "lif"},
		{"liga", "lig"},
		{"ligados", "lig"},
		{"ligamayorista", "ligamayor"},
		{"ligamento", "ligament"},
		{"ligar", "lig"},
		{"ligarde", "ligard"},
		{"ligaron", "lig"},
		{"ligas", "lig"},
		{"ligera", "liger"},
		{"ligeramente", "liger"},
		{"ligeras", "liger"},
		{"ligereza", "ligerez"},
		{"ligero", "liger"},
		{"light", "light"},
		{"ligne", "lign"},
		{"ligó", "lig"},
		{"ligue", "lig"},
		{"liguilla", "liguill"},
		{"liliana", "lilian"},
		{"lima", "lim"},
		{"limar", "lim"},
		{"limita", "limit"},
		{"limitaba", "limit"},
		{"limitaciones", "limit"},
		{"limitado", "limit"},
		{"limitados", "limit"},
		{"limitan", "limit"},
		{"limitar", "limit"},
		{"limitará", "limit"},
		{"limite", "limit"},
		{"límite", "limit"},
		{"limited", "limit"},
		{"límites", "limit"},
		{"limitó", "limit"},
		{"limítrofes", "limitrof"},
		{"limits", "limits"},
		{"limón", "limon"},
		{"limonada", "limon"},
		{"limosine", "limosin"},
		{"limosna", "limosn"},
		{"limosnita", "limosnit"},
		{"limpia", "limpi"},
		{"limpiamente", "limpi"},
		{"limpiando", "limpi"},
		{"limpiaparabrisas", "limpiaparabris"},
		{"limpiar", "limpi"},
		{"limpiaron", "limpi"},
		{"limpias", "limpi"},
		{"limpieza", "limpiez"},
		{"limpio", "limpi"},
		{"lina", "lin"},
		{"linares", "linar"},
		{"linces", "linc"},
		{"linchar", "linch"},
		{"lincoln", "lincoln"},
		{"linda", "lind"},
		{"linduras", "lindur"},
		{"line", "lin"},
		{"linea", "line"},
		{"línea", "line"},
		{"lineal", "lineal"},
		{"lineamiento", "lineamient"},
		{"lineamientos", "lineamient"},
		{"líneas", "lin"},
		{"lineker", "linek"},
		{"liner", "lin"},
		{"ling", "ling"},
		{"lingada", "ling"},
		{"linterna", "lintern"},
		{"líos", "lios"},
		{"liquidación", "liquid"},
		{"liquidaciones", "liquid"},
		{"liquidar", "liquid"},
		{"liquidará", "liquid"},
		{"liquidez", "liquidez"},
		{"liquido", "liqu"},
		{"líquido", "liqu"},
		{"líquidos", "liqu"},
		{"lira", "lir"},
		{"lisa", "lis"},
		{"lista", "list"},
		{"listado", "list"},
		{"listados", "list"},
		{"listas", "list"},
		{"listo", "list"},
		{"listón", "liston"},
		{"listos", "list"},
		{"literalmente", "literal"},
		{"literaria", "literari"},
		{"literarias", "literari"},
		{"literarios", "literari"},
		{"literatos", "literat"},
		{"literatura", "literatur"},
		{"litigios", "litigi"},
		{"litoral", "litoral"},
		{"litorales", "litoral"},
		{"litro", "litr"},
		{"litros", "litr"},
		{"livas", "liv"},
		{"liverpool", "liverpool"},
		{"livestock", "livestock"},
		{"liviano", "livian"},
		{"llaguno", "llagun"},
		{"llama", "llam"},
		{"llamaba", "llam"},
		{"llamaban", "llam"},
		{"llamada", "llam"},
		{"llamadas", "llam"},
		{"llamado", "llam"},
		{"llamados", "llam"},
		{"llamáis", "llam"},
		{"llamamiento", "llamamient"},
		{"llamamos", "llam"},
		{"llaman", "llam"},
		{"llamando", "llam"},
		{"llamar", "llam"},
		{"llamara", "llam"},
		{"llamará", "llam"},
		{"llamaría", "llam"},
		{"llamarían", "llam"},
		{"llamarlo", "llam"},
		{"llamarnos", "llam"},
		{"llamaron", "llam"},
		{"llamarse", "llam"},
		{"llamas", "llam"},
		{"llamativa", "llamat"},
		{"llamativas", "llamat"},
		{"llamativos", "llamat"},
		{"llame", "llam"},
		{"llamen", "llam"},
		{"llamo", "llam"},
		{"llamó", "llam"},
		{"llanamente", "llan"},
		{"llanas", "llan"},
		{"llanes", "llan"},
		{"llano", "llan"},
		{"llanos", "llan"},
		{"llanta", "llant"},
		{"llantas", "llant"},
		{"llanto", "llant"},
		{"llave", "llav"},
		{"llaveros", "llaver"},
		{"llaves", "llav"},
		{"lle", "lle"},
		{"llega", "lleg"},
		{"llegaba", "lleg"},
		{"llegaban", "lleg"},
		{"llegada", "lleg"},
		{"llegadas", "lleg"},
		{"llegado", "lleg"},
		{"llegados", "lleg"},
		{"llegamos", "lleg"},
		{"llegan", "lleg"},
		{"llegando", "lleg"},
		{"llegar", "lleg"},
		{"llegara", "lleg"},
		{"llegará", "lleg"},
		{"llegaran", "lleg"},
		{"llegarán", "lleg"},
		{"llegaré", "lleg"},
		{"llegaremos", "lleg"},
		{"llegaría", "lleg"},
		{"llegarían", "lleg"},
		{"llegaron", "lleg"},
		{"llegarse", "lleg"},
		{"llegas", "lleg"},
		{"llego", "lleg"},
		{"llegó", "lleg"},
		{"llegue", "lleg"},
		{"lleguemos", "lleg"},
		{"lleguen", "lleg"},
		{"llena", "llen"},
		{"llenado", "llen"},
		{"llenan", "llen"},
		{"llenando", "llen"},
		{"llenar", "llen"},
		{"llenará", "llen"},
		{"llenarla", "llen"},
		{"llenarlas", "llen"},
		{"llenaron", "llen"},
		{"llenarse", "llen"},
		{"llenas", "llen"},
		{"lleno", "llen"},
		{"llenó", "llen"},
		{"llenos", "llen"},
		{"llergo", "llerg"},
		{"lleva", "llev"},
		{"llevaba", "llev"},
		{"llevaban", "llev"},
		{"llevada", "llev"},
		{"llevadas", "llev"},
		{"llevado", "llev"},
		{"llevados", "llev"},
		{"llevamos", "llev"},
		{"llevan", "llev"},
		{"llevando", "llev"},
		{"llevándolos", "llev"},
		{"llevándose", "llev"},
		{"llevar", "llev"},
		{"llevara", "llev"},
		{"llevará", "llev"},
		{"llevaran", "llev"},
		{"llevarán", "llev"},
		{"llevaría", "llev"},
		{"llevarían", "llev"},
		{"llevarla", "llev"},
		{"llevarlas", "llev"},
		{"llevarle", "llev"},
		{"llevarlo", "llev"},
		{"llevarlos", "llev"},
		{"llevaron", "llev"},
		{"llevarse", "llev"},
		{"llevárselo", "llev"},
		{"llevas", "llev"},
		{"llevase", "llev"},
		{"lleve", "llev"},
		{"llévelo", "llevel"},
		{"lleven", "llev"},
		{"llevo", "llev"},
		{"llevó", "llev"},
		{"llora", "llor"},
		{"lloraba", "llor"},
		{"lloran", "llor"},
		{"llorando", "llor"},
		{"llorar", "llor"},
		{"lloraré", "llor"},
		{"lloró", "llor"},
		{"llorosa", "lloros"},
		{"llosa", "llos"},
		{"llover", "llov"},
		{"lloviendo", "llov"},
		{"llovió", "llov"},
		{"llovizna", "llovizn"},
		{"lloviznas", "llovizn"},
		{"llueva", "lluev"},
		{"llueve", "lluev"},
		{"llueven", "lluev"},
		{"lluvia", "lluvi"},
		{"lluvias", "lluvi"},
		{"lluvioso", "lluvios"},
		{"lluviosos", "lluvios"},
		{"lmkf", "lmkf"},
		{"lo", "lo"},
		{"loable", "loabl"},
		{"loaeza", "loaez"},
		{"lobeira", "lobeir"},
		{"lobo", "lob"},
		{"lobos", "lob"},
		{"lóbulo", "lobul"},
		{"loca", "loc"},
		{"locación", "locacion"},
		{"local", "local"},
		{"locales", "local"},
		{"localice", "localic"},
		{"localidad", "local"},
		{"localidades", "local"},
		{"localiza", "localiz"},
		{"localización", "localiz"},
		{"localizada", "localiz"},
		{"localizadas", "localiz"},
		{"localizado", "localiz"},
		{"localizados", "localiz"},
		{"localizar", "localiz"},
		{"localizará", "localiz"},
		{"localizaron", "localiz"},
		{"localizó", "localiz"},
		{"localmente", "local"},
		{"locka", "lock"},
		{"loco", "loc"},
		{"locos", "loc"},
		{"locura", "locur"},
		{"lodazal", "lodazal"},
		{"lodo", "lod"},
		{"lofton", "lofton"},
		{"lógica", "logic"},
		{"lógicamente", "logic"},
		{"lógico", "logic"},
		{"logo", "log"},
		{"logotipo", "logotip"},
		{"logra", "logr"},
		{"lograba", "logr"},
		{"lograban", "logr"},
		{"lograda", "logr"},
		{"logradas", "logr"},
		{"logrado", "logr"},
		{"logrados", "logr"},
		{"logramos", "logr"},
		{"logran", "logr"},
		{"logrando", "logr"},
		{"lograr", "logr"},
		{"lograra", "logr"},
		{"logrará", "logr"},
		{"lograrán", "logr"},
		{"lograremos", "logr"},
		{"lograrlo", "logr"},
		{"lograron", "logr"},
		{"lograrse", "logr"},
		{"logre", "logr"},
		{"logré", "logr"},
		{"logren", "logr"},
		{"logro", "logr"},
		{"logró", "logr"},
		{"logronés", "logrones"},
		{"logroñés", "logroñes"},
		{"logros", "logr"},
		{"lolita", "lolit"},
		{"loma", "lom"},
		{"lomas", "lom"},
		{"lombardo", "lombard"},
		{"lomelí", "lomel"},
		{"lomos", "lom"},
		{"lona", "lon"},
		{"lonche", "lonch"},
		{"lonches", "lonch"},
		{"londinense", "londinens"},
		{"londres", "londr"},
		{"longevo", "longev"},
		{"longitud", "longitud"},
		{"longoria", "longori"},
		{"lope", "lop"},
		{"lopez", "lopez"},
		{"lópez", "lopez"},
		{"lópezfigueroa", "lopezfiguero"},
		{"lopezportillista", "lopezportill"},
		{"loredo", "lored"},
		{"lorena", "loren"},
		{"lorenzo", "lorenz"},
		{"loreto", "loret"},
		{"los", "los"},
		{"losada", "los"},
		{"lote", "lot"},
		{"lotería", "lot"},
		{"lotes", "lot"},
		{"lotus", "lotus"},
		{"louis", "louis"},
		{"louisiana", "louisian"},
		{"lourdes", "lourd"},
		{"lover", "lov"},
		{"loya", "loy"},
		{"loyola", "loyol"},
		{"lozano", "lozan"},
		{"lsqb", "lsqb"},
		{"ltd", "ltd"},
		{"lubricación", "lubric"},
		{"lucas", "luc"},
		{"luce", "luc"},
		{"lucen", "luc"},
		{"lucero", "lucer"},
		{"luces", "luc"},
		{"lucha", "luch"},
		{"luchaba", "luch"},
		{"luchado", "luch"},
		{"luchador", "luchador"},
		{"luchadores", "luchador"},
		{"luchando", "luch"},
		{"luchar", "luch"},
		{"luchará", "luch"},
		{"lucharon", "luch"},
		{"luchas", "luch"},
		{"lucho", "luch"},
		{"luchó", "luch"},
		{"lucía", "luc"},
		{"luciano", "lucian"},
		{"lúcidamente", "lucid"},
		{"lucido", "luc"},
		{"luciendo", "luc"},
		{"lucieron", "luc"},
		{"lucila", "lucil"},
		{"lucimiento", "lucimient"},
		{"lucio", "luci"},
		{"lució", "luc"},
		{"lucir", "luc"},
		{"lucrar", "lucr"},
		{"lucro", "lucr"},
		{"luctuosas", "luctu"},
		{"lúdica", "ludic"},
		{"lúdicos", "ludic"},
		{"ludmila", "ludmil"},
		{"luego", "lueg"},
		{"lueguito", "lueguit"},
		{"luévano", "luevan"},
		{"lugar", "lug"},
		{"lugares", "lugar"},
		{"lugo", "lug"},
		{"luis", "luis"},
		{"luisa", "luis"},
		{"luiz", "luiz"},
		{"lujo", "luj"},
		{"lujos", "luj"},
		{"lujoso", "lujos"},
		{"lumbreras", "lumbrer"},
		{"luminosas", "lumin"},
		{"luminosidad", "lumin"},
		{"luminoso", "lumin"},
		{"luminotécnicos", "luminotecn"},
		{"luna", "lun"},
		{"lunar", "lun"},
		{"lunáticos", "lunat"},
		{"lunes", "lun"},
		{"luneta", "lunet"},
		{"lupe", "lup"},
		{"lupercio", "luperci"},
		{"lupita", "lupit"},
		{"lúpulo", "lupul"},
		{"lusinchi", "lusinchi"},
		{"lutero", "luter"},
		{"luto", "lut"},
		{"luviano", "luvian"},
		{"luxemburgo", "luxemburg"},
		{"luz", "luz"},
		{"luzcan", "luzc"},
		{"luzón", "luzon"},
		{"lwdvig", "lwdvig"},
		{"lydia", "lydi"},
		{"lynch", "lynch"},
		{"lyon", "lyon"},
		{"m", "m"},
		{"ma", "ma"},
		{"mac", "mac"},
		{"macabro", "macabr"},
		{"macartismo", "macart"},
		{"macedonia", "macedoni"},
		{"macedonio", "macedoni"},
		{"machado", "mach"},
		{"machetero", "macheter"},
		{"machiria", "machiri"},
		{"machismo", "machism"},
		{"machista", "machist"},
		{"macho", "mach"},
		{"machuca", "machuc"},
		{"macías", "mac"},
		{"macintosh", "macintosh"},
		{"macizo", "maciz"},
		{"macotela", "macotel"},
		{"macroeconómica", "macroeconom"},
		{"macroeconómicas", "macroeconom"},
		{"macroplaza", "macroplaz"},
		{"macrópolis", "macropolis"},
		{"macumba", "macumb"},
		{"madamme", "madamm"},
		{"madariaga", "madariag"},
		{"madeja", "madej"},
		{"mademoiselle", "mademoisell"},
		{"madera", "mader"},
		{"maderámenes", "maderamen"},
		{"maderas", "mader"},
		{"madero", "mader"},
		{"madrazo", "madraz"},
		{"madre", "madr"},
		{"madres", "madr"},
		{"madrid", "madr"},
		{"madrigal", "madrigal"},
		{"madril", "madril"},
		{"madrileñas", "madrileñ"},
		{"madrileño", "madrileñ"},
		{"madrileños", "madrileñ"},
		{"madrina", "madrin"},
		{"madrinas", "madrin"},
		{"madrugada", "madrug"},
		{"madura", "madur"},
		{"maduración", "madur"},
		{"madurar", "madur"},
		{"madure", "madur"},
		{"madurez", "madurez"},
		{"maduro", "madur"},
		{"maduros", "madur"},
		{"maestra", "maestr"},
		{"maestras", "maestr"},
		{"maestría", "maestr"},
		{"maestrías", "maestr"},
		{"maestro", "maestr"},
		{"maestros", "maestr"},
		{"mafia", "mafi"},
		{"mafiosos", "mafios"},
		{"magala", "magal"},
		{"magaña", "magañ"},
		{"magdalena", "magdalen"},
		{"magia", "magi"},
		{"magic", "magic"},
		{"mágica", "magic"},
		{"mágicas", "magic"},
		{"mágico", "magic"},
		{"magisterial", "magisterial"},
		{"magisteriales", "magisterial"},
		{"magisterio", "magisteri"},
		{"magistrado", "magistr"},
		{"magistrados", "magistr"},
		{"magistral", "magistral"},
		{"magistrales", "magistral"},
		{"magistralmente", "magistral"},
		{"magma", "magm"},
		{"magna", "magn"},
		{"magnesia", "magnesi"},
		{"magnética", "magnet"},
		{"magnéticamente", "magnet"},
		{"magnéticas", "magnet"},
		{"magnéticos", "magnet"},
		{"magneto", "magnet"},
		{"magnetómetro", "magnetometr"},
		{"magnífica", "magnif"},
		{"magníficas", "magnif"},
		{"magnificencia", "magnificent"},
		{"magnífico", "magnif"},
		{"magníficos", "magnif"},
		{"magnitud", "magnitud"},
		{"magno", "magn"},
		{"magnolia", "magnoli"},
		{"magnum", "magnum"},
		{"magotteaux", "magotteaux"},
		{"mague", "mag"},
		{"mahoney", "mahoney"},
		{"mahony", "mahony"},
		{"maiz", "maiz"},
		{"maíz", "maiz"},
		{"majadería", "majad"},
		{"majadero", "majader"},
		{"majdel", "majdel"},
		{"majestad", "majest"},
		{"majestuosas", "majestu"},
		{"majestuosidad", "majestu"},
		{"majestuoso", "majestu"},
		{"maksim", "maksim"},
		{"mal", "mal"},
		{"mala", "mal"},
		{"malagón", "malagon"},
		{"malas", "mal"},
		{"malcolm", "malcolm"},
		{"malcriada", "malcri"},
		{"maldad", "mald"},
		{"maldades", "maldad"},
		{"maldición", "maldicion"},
		{"maldiciones", "maldicion"},
		{"maldito", "maldit"},
		{"maldonado", "maldon"},
		{"maleantes", "maleant"},
		{"males", "mal"},
		{"maletas", "malet"},
		{"malicia", "malici"},
		{"maligna", "malign"},
		{"malignas", "malign"},
		{"malignidad", "malign"},
		{"maligno", "malign"},
		{"malinche", "malinch"},
		{"malinchista", "malinch"},
		{"malintencionada", "malintencion"},
		{"malla", "mall"},
		{"mallas", "mall"},
		{"malo", "mal"},
		{"malograrlas", "malogr"},
		{"malos", "mal"},
		{"malta", "malt"},
		{"maltratado", "maltrat"},
		{"maltratados", "maltrat"},
		{"maltrato", "maltrat"},
		{"maltrecha", "maltrech"},
		{"malva", "malv"},
		{"malvada", "malv"},
		{"malvavisco", "malvavisc"},
		{"malversación", "malvers"},
		{"malversaciones", "malvers"},
		{"malvivientes", "malvivient"},
		{"mamá", "mam"},
		{"mamás", "mamas"},
		{"mamíferos", "mamifer"},
		{"mamilas", "mamil"},
		{"mamparas", "mamp"},
		{"maña", "mañ"},
		{"maná", "man"},
		{"management", "management"},
		{"mánager", "manag"},
		{"managers", "managers"},
		{"mañana", "mañan"},
		{"mañanas", "mañan"},
		{"manazos", "manaz"},
		{"mancha", "manch"},
		{"manchada", "manch"},
		{"manchar", "manch"},
		{"manchas", "manch"},
		{"mancillados", "mancill"},
		{"mancomunada", "mancomun"},
		{"mancuso", "mancus"},
		{"manda", "mand"},
		{"mandadas", "mand"},
		{"mandado", "mand"},
		{"mandamiento", "mandamient"},
		{"mandamientos", "mandamient"},
		{"mandamos", "mand"},
		{"mandan", "mand"},
		{"mandar", "mand"},
		{"mandará", "mand"},
		{"mandaríamos", "mand"},
		{"mandarinas", "mandarin"},
		{"mandarlas", "mand"},
		{"mandarlo", "mand"},
		{"mandaron", "mand"},
		{"mandatario", "mandatari"},
		{"mandatarios", "mandatari"},
		{"mandato", "mandat"},
		{"mandatos", "mandat"},
		{"manden", "mand"},
		{"mandíbula", "mandibul"},
		{"mandíbulas", "mandibul"},
		{"mandilones", "mandilon"},
		{"mando", "mand"},
		{"mandó", "mand"},
		{"mandos", "mand"},
		{"mandril", "mandril"},
		{"maneja", "manej"},
		{"manejaba", "manej"},
		{"manejable", "manej"},
		{"manejada", "manej"},
		{"manejadas", "manej"},
		{"manejado", "manej"},
		{"manejador", "manej"},
		{"manejadores", "manej"},
		{"manejados", "manej"},
		{"manejamos", "manej"},
		{"manejan", "manej"},
		{"manejando", "manej"},
		{"manejar", "manej"},
		{"manejará", "manej"},
		{"manejaron", "manej"},
		{"manejarse", "manej"},
		{"maneje", "manej"},
		{"manejen", "manej"},
		{"manejes", "manej"},
		{"manejo", "manej"},
		{"manejó", "manej"},
		{"manejos", "manej"},
		{"manente", "manent"},
		{"manera", "maner"},
		{"maneras", "maner"},
		{"manga", "mang"},
		{"manganas", "mangan"},
		{"mangas", "mang"},
		{"manglares", "manglar"},
		{"manguera", "manguer"},
		{"manhattan", "manhatt"},
		{"manía", "man"},
		{"maniatar", "maniat"},
		{"manieristas", "manier"},
		{"manifestaban", "manifest"},
		{"manifestación", "manifest"},
		{"manifestaciones", "manifest"},
		{"manifestada", "manifest"},
		{"manifestadas", "manifest"},
		{"manifestado", "manifest"},
		{"manifestantes", "manifest"},
		{"manifestar", "manifest"},
		{"manifestaran", "manifest"},
		{"manifestaron", "manifest"},
		{"manifestarse", "manifest"},
		{"manifestó", "manifest"},
		{"manifiesta", "manifiest"},
		{"manifiesto", "manifiest"},
		{"manifold", "manifold"},
		{"manila", "manil"},
		{"maniobra", "maniobr"},
		{"maniobras", "maniobr"},
		{"manipulación", "manipul"},
		{"manipulado", "manipul"},
		{"manipular", "manipul"},
		{"manita", "manit"},
		{"manjar", "manj"},
		{"manjarrez", "manjarrez"},
		{"manneck", "manneck"},
		{"mano", "man"},
		{"manolita", "manolit"},
		{"manolo", "manol"},
		{"manos", "man"},
		{"mañosamente", "mañ"},
		{"manotas", "manot"},
		{"manrique", "manriqu"},
		{"manríquez", "manriquez"},
		{"mansedumbre", "mansedumbr"},
		{"mansión", "mansion"},
		{"mansiones", "mansion"},
		{"manso", "mans"},
		{"mansos", "mans"},
		{"mansur", "mansur"},
		{"mantas", "mant"},
		{"mantendrá", "mantendr"},
		{"mantenemos", "manten"},
		{"mantener", "manten"},
		{"mantenerlas", "manten"},
		{"mantenerlo", "manten"},
		{"mantenerlos", "manten"},
		{"mantenerme", "manten"},
		{"mantenernos", "manten"},
		{"mantenerse", "manten"},
		{"mantenga", "manteng"},
		{"mantengan", "manteng"},
		{"mantenía", "manten"},
		{"mantenidas", "manten"},
		{"mantenido", "manten"},
		{"manteniendo", "manten"},
		{"manteniéndose", "manten"},
		{"mantenimiento", "manten"},
		{"mantiene", "mantien"},
		{"mantienen", "mantien"},
		{"manto", "mant"},
		{"mantuvieron", "mantuv"},
		{"mantuvo", "mantuv"},
		{"manu", "manu"},
		{"manual", "manual"},
		{"manuales", "manual"},
		{"manualidades", "manual"},
		{"manuel", "manuel"},
		{"manufactura", "manufactur"},
		{"manufacturar", "manufactur"},
		{"manufacturera", "manufacturer"},
		{"manufactureros", "manufacturer"},
		{"many", "many"},
		{"manzana", "manzan"},
		{"manzanas", "manzan"},
		{"manzanillo", "manzanill"},
		{"maoísta", "maoist"},
		{"maoístas", "maoist"},
		{"mapa", "map"},
		{"mapas", "map"},
		{"mapinfo", "mapinf"},
		{"maqueta", "maquet"},
		{"maquilada", "maquil"},
		{"maquilador", "maquil"},
		{"maquiladora", "maquil"},
		{"maquiladoras", "maquil"},
		{"maquilados", "maquil"},
		{"maquilas", "maquil"},
		{"maquillado", "maquill"},
		{"maquillaje", "maquillaj"},
		{"maquillarlas", "maquill"},
		{"máquina", "maquin"},
		{"maquinaria", "maquinari"},
		{"maquinas", "maquin"},
		{"máquinas", "maquin"},
		{"maquinista", "maquin"},
		{"mar", "mar"},
		{"mara", "mar"},
		{"maradona", "maradon"},
		{"maraldo", "marald"},
		{"marasmo", "marasm"},
		{"maratón", "maraton"},
		{"maravilla", "maravill"},
		{"maravillando", "maravill"},
		{"maravillosa", "maravill"},
		{"maravillosamente", "maravill"},
		{"maravillosas", "maravill"},
		{"maravillosos", "maravill"},
		{"marc", "marc"},
		{"marca", "marc"},
		{"marcaba", "marc"},
		{"marcación", "marcacion"},
		{"marcada", "marc"},
		{"marcadas", "marc"},
		{"marcado", "marc"},
		{"marcador", "marcador"},
		{"marcadores", "marcador"},
		{"marcan", "marc"},
		{"marcando", "marc"},
		{"marcar", "marc"},
		{"marcará", "marc"},
		{"marcaron", "marc"},
		{"marcas", "marc"},
		{"marcela", "marcel"},
		{"marcelino", "marcelin"},
		{"marcelo", "marcel"},
		{"marcha", "march"},
		{"marchado", "march"},
		{"marchan", "march"},
		{"marcharían", "march"},
		{"marcharon", "march"},
		{"marcharse", "march"},
		{"marchas", "march"},
		{"marche", "march"},
		{"marchó", "march"},
		{"marcial", "marcial"},
		{"marciano", "marcian"},
		{"marcio", "marci"},
		{"marco", "marc"},
		{"marcó", "marc"},
		{"marcos", "marc"},
		{"marcus", "marcus"},
		{"marcusse", "marcuss"},
		{"mareadas", "mar"},
		{"maremoto", "maremot"},
		{"mares", "mar"},
		{"marfil", "marfil"},
		{"marga", "marg"},
		{"margáin", "margain"},
		{"margaret", "margaret"},
		{"margaretha", "margareth"},
		{"margarita", "margarit"},
		{"margen", "marg"},
		{"márgen", "marg"},
		{"margenes", "margen"},
		{"margina", "margin"},
		{"marginación", "margin"},
		{"marginaciones", "margin"},
		{"marginadas", "margin"},
		{"marginados", "margin"},
		{"marginal", "marginal"},
		{"marginara", "margin"},
		{"marginó", "margin"},
		{"marguerite", "marguerit"},
		{"margules", "margul"},
		{"mari", "mari"},
		{"maria", "mari"},
		{"maría", "mar"},
		{"mariachi", "mariachi"},
		{"marian", "mari"},
		{"mariana", "marian"},
		{"marianas", "marian"},
		{"mariani", "mariani"},
		{"mariano", "marian"},
		{"marianos", "marian"},
		{"mariateguista", "mariategu"},
		{"maribel", "maribel"},
		{"maricela", "maricel"},
		{"maridito", "maridit"},
		{"marido", "mar"},
		{"maridos", "mar"},
		{"marie", "mari"},
		{"mariela", "mariel"},
		{"mariguana", "mariguan"},
		{"marihuana", "marihuan"},
		{"marijo", "marij"},
		{"marilyn", "marilyn"},
		{"marin", "marin"},
		{"marín", "marin"},
		{"marina", "marin"},
		{"marinas", "marin"},
		{"marinera", "mariner"},
		{"marineros", "mariner"},
		{"marino", "marin"},
		{"marinos", "marin"},
		{"mario", "mari"},
		{"marion", "marion"},
		{"marionetero", "marioneter"},
		{"mariposas", "marip"},
		{"mariscal", "mariscal"},
		{"mariscales", "mariscal"},
		{"marisma", "marism"},
		{"marissa", "mariss"},
		{"marita", "marit"},
		{"marítima", "maritim"},
		{"marítimo", "maritim"},
		{"marketing", "marketing"},
		{"marley", "marley"},
		{"marlines", "marlin"},
		{"marlon", "marlon"},
		{"mármol", "marmol"},
		{"marque", "marqu"},
		{"marqués", "marques"},
		{"marquesina", "marquesin"},
		{"márquez", "marquez"},
		{"marquis", "marquis"},
		{"marranos", "marran"},
		{"marsalis", "marsalis"},
		{"marshal", "marshal"},
		{"marshall", "marshall"},
		{"mart", "mart"},
		{"marte", "mart"},
		{"martellotto", "martellott"},
		{"martens", "martens"},
		{"martes", "mart"},
		{"martha", "marth"},
		{"martí", "mart"},
		{"martillo", "martill"},
		{"martin", "martin"},
		{"martín", "martin"},
		{"martinez", "martinez"},
		{"martínez", "martinez"},
		{"martinica", "martin"},
		{"martiniqués", "martiniques"},
		{"mártires", "martir"},
		{"martirio", "martiri"},
		{"marv", "marv"},
		{"marx", "marx"},
		{"mary", "mary"},
		{"maryland", "maryland"},
		{"marzo", "marz"},
		{"mas", "mas"},
		{"más", "mas"},
		{"masa", "mas"},
		{"masaaki", "masaaki"},
		{"masacre", "masacr"},
		{"masakre", "masakr"},
		{"masas", "mas"},
		{"mascadas", "masc"},
		{"máscara", "masc"},
		{"máscaras", "masc"},
		{"mascota", "mascot"},
		{"mascotas", "mascot"},
		{"masculina", "masculin"},
		{"masculino", "masculin"},
		{"masiva", "masiv"},
		{"masivamente", "masiv"},
		{"masivas", "masiv"},
		{"masivo", "masiv"},
		{"masivos", "masiv"},
		{"masoquismo", "masoqu"},
		{"mass", "mass"},
		{"massa", "mass"},
		{"massacessi", "massacessi"},
		{"massachusetts", "massachusetts"},
		{"massachussets", "massachussets"},
		{"massachussetts", "massachussetts"},
		{"massieu", "massieu"},
		{"masso", "mass"},
		{"masterd", "masterd"},
		{"masturbación", "masturb"},
		{"mata", "mat"},
		{"matado", "mat"},
		{"matador", "matador"},
		{"matadores", "matador"},
		{"matamoros", "matamor"},
		{"matando", "mat"},
		{"matanzas", "matanz"},
		{"matar", "mat"},
		{"mátar", "mat"},
		{"mataran", "mat"},
		{"mataría", "mat"},
		{"matarla", "mat"},
		{"matarlo", "mat"},
		{"matarme", "mat"},
		{"matarnos", "mat"},
		{"mataron", "mat"},
		{"matarse", "mat"},
		{"mate", "mat"},
		{"matemática", "matemat"},
		{"matemáticas", "matemat"},
		{"matemático", "matemat"},
		{"maten", "mat"},
		{"mateo", "mate"},
		{"mateos", "mate"},
		{"mater", "mat"},
		{"materia", "materi"},
		{"material", "material"},
		{"materiales", "material"},
		{"materializarse", "materializ"},
		{"materias", "materi"},
		{"materna", "matern"},
		{"maternal", "maternal"},
		{"maternidad", "matern"},
		{"matías", "mat"},
		{"matices", "matic"},
		{"matilde", "matild"},
		{"matineé", "matine"},
		{"matizada", "matiz"},
		{"mató", "mat"},
		{"matorrales", "matorral"},
		{"matos", "mat"},
		{"matriarcado", "matriarc"},
		{"matrices", "matric"},
		{"matrícula", "matricul"},
		{"matriculen", "matricul"},
		{"matrimonial", "matrimonial"},
		{"matrimoniales", "matrimonial"},
		{"matrimonio", "matrimoni"},
		{"matrimonios", "matrimoni"},
		{"matrix", "matrix"},
		{"matriz", "matriz"},
		{"matt", "matt"},
		{"matthew", "matthew"},
		{"matutina", "matutin"},
		{"matutino", "matutin"},
		{"maurice", "mauric"},
		{"mauricio", "maurici"},
		{"mauro", "maur"},
		{"max", "max"},
		{"maxi", "maxi"},
		{"máxima", "maxim"},
		{"máxime", "maxim"},
		{"maximiliano", "maximilian"},
		{"máximo", "maxim"},
		{"máximos", "max"},
		{"maximova", "maximov"},
		{"may", "may"},
		{"mayas", "may"},
		{"mayela", "mayel"},
		{"mayito", "mayit"},
		{"mayo", "may"},
		{"mayón", "mayon"},
		{"mayonesa", "mayones"},
		{"mayor", "mayor"},
		{"mayoreo", "mayore"},
		{"mayores", "mayor"},
		{"mayoría", "mayor"},
		{"mayorista", "mayor"},
		{"mayoristas", "mayor"},
		{"mayoritaria", "mayoritari"},
		{"mayoritario", "mayoritari"},
		{"mayoritarios", "mayoritari"},
		{"mayos", "may"},
		{"mayrán", "mayran"},
		{"mays", "mays"},
		{"mayúscula", "mayuscul"},
		{"mayúsculas", "mayuscul"},
		{"mayúsculo", "mayuscul"},
		{"mayweather", "mayweath"},
		{"maza", "maz"},
		{"mazatlan", "mazatl"},
		{"mazatlán", "mazatlan"},
		{"mazatleco", "mazatlec"},
		{"mc", "mc"},
		{"mcallen", "mcall"},
		{"mccann", "mccann"},
		{"mcdonalds", "mcdonalds"},
		{"mcdowell", "mcdowell"},
		{"mcenroe", "mcenro"},
		{"mcgregor", "mcgregor"},
		{"mckesson", "mckesson"},
		{"mcleod", "mcleod"},
		{"md", "md"},
		{"me", "me"},
		{"mea", "mea"},
		{"meat", "meat"},
		{"mecánica", "mecan"},
		{"mecánico", "mecan"},
		{"mecánicos", "mecan"},
		{"mecanismo", "mecan"},
		{"mecanismos", "mecan"},
		{"mecenazgo", "mecenazg"},
		{"medalla", "medall"},
		{"medallas", "medall"},
		{"medallista", "medall"},
		{"medallón", "medallon"},
		{"medellín", "medellin"},
		{"media", "medi"},
		{"mediación", "mediacion"},
		{"mediadora", "mediador"},
		{"mediados", "medi"},
		{"mediahermana", "mediaherman"},
		{"mediana", "median"},
		{"medianas", "median"},
		{"mediano", "median"},
		{"medianoche", "medianoch"},
		{"medianos", "median"},
		{"mediante", "mediant"},
		{"medias", "medi"},
		{"mediata", "mediat"},
		{"médica", "medic"},
		{"medical", "medical"},
		{"medicamentos", "medicament"},
		{"médicas", "medic"},
		{"medicina", "medicin"},
		{"medicinales", "medicinal"},
		{"medicinas", "medicin"},
		{"medición", "medicion"},
		{"mediciones", "medicion"},
		{"médico", "medic"},
		{"médicos", "medic"},
		{"medida", "med"},
		{"medidas", "med"},
		{"medido", "med"},
		{"medievales", "medieval"},
		{"medina", "medin"},
		{"medio", "medi"},
		{"mediocampista", "mediocamp"},
		{"mediocampistas", "mediocamp"},
		{"mediocre", "mediocr"},
		{"mediocres", "mediocr"},
		{"mediocridad", "mediocr"},
		{"mediodía", "mediod"},
		{"mediohermano", "medioherman"},
		{"medios", "medi"},
		{"medir", "med"},
		{"medirá", "med"},
		{"medirán", "med"},
		{"medirse", "med"},
		{"meditación", "medit"},
		{"meditado", "medit"},
		{"medite", "medit"},
		{"mediterránea", "mediterrane"},
		{"mediterráneo", "mediterrane"},
		{"medrano", "medran"},
		{"medugorie", "medugori"},
		{"médula", "medul"},
		{"medusia", "medusi"},
		{"mefistófeles", "mefistofel"},
		{"megabits", "megabits"},
		{"megabytes", "megabyt"},
		{"megahertz", "megahertz"},
		{"mejía", "mej"},
		{"mejilla", "mejill"},
		{"mejillas", "mejill"},
		{"mejor", "mejor"},
		{"mejora", "mejor"},
		{"mejorada", "mejor"},
		{"mejorado", "mejor"},
		{"mejorados", "mejor"},
		{"mejoramiento", "mejor"},
		{"mejoran", "mejor"},
		{"mejorando", "mejor"},
		{"mejorar", "mejor"},
		{"mejorará", "mejor"},
		{"mejoraran", "mejor"},
		{"mejorarle", "mejor"},
		{"mejorarlos", "mejor"},
		{"mejoras", "mejor"},
		{"mejore", "mejor"},
		{"mejores", "mejor"},
		{"mejoría", "mejor"},
		{"mejrar", "mejr"},
		{"melanoma", "melanom"},
		{"melchor", "melchor"},
		{"melé", "mel"},
		{"meléndez", "melendez"},
		{"melgar", "melg"},
		{"melian", "meli"},
		{"melisa", "melis"},
		{"mella", "mell"},
		{"mellizos", "melliz"},
		{"mello", "mell"},
		{"melodía", "melod"},
		{"melódica", "melod"},
		{"melón", "melon"},
		{"melquiades", "melquiad"},
		{"membretado", "membret"},
		{"membrete", "membret"},
		{"memo", "mem"},
		{"memorable", "memor"},
		{"memorándum", "memorandum"},
		{"memoria", "memori"},
		{"memorias", "memori"},
		{"memorizar", "memoriz"},
		{"mena", "men"},
		{"ménagier", "menagi"},
		{"menchú", "menchu"},
		{"mención", "mencion"},
		{"menciona", "mencion"},
		{"mencionada", "mencion"},
		{"mencionadas", "mencion"},
		{"mencionado", "mencion"},
		{"mencionados", "mencion"},
		{"mencionan", "mencion"},
		{"mencionar", "mencion"},
		{"mencionaré", "mencion"},
		{"mencionaremos", "mencion"},
		{"mencionaron", "mencion"},
		{"mencione", "mencion"},
		{"mencioné", "mencion"},
		{"menciones", "mencion"},
		{"mencionó", "mencion"},
		{"méndez", "mendez"},
		{"mendigo", "mendig"},
		{"mendiomas", "mendiom"},
		{"mendoza", "mendoz"},
		{"menem", "menem"},
		{"meneses", "menes"},
		{"mengano", "mengan"},
		{"menguadas", "mengu"},
		{"meninges", "mening"},
		{"meningitis", "meningitis"},
		{"meningoencefalitis", "meningoencefalitis"},
		{"menopausia", "menopausi"},
		{"menopáusica", "menopaus"},
		{"menor", "menor"},
		{"menores", "menor"},
		{"menos", "men"},
		{"menosprecian", "menospreci"},
		{"menospreciarnos", "menospreci"},
		{"menotti", "menotti"},
		{"mensaje", "mensaj"},
		{"mensajería", "mensaj"},
		{"mensajes", "mensaj"},
		{"mensual", "mensual"},
		{"mensuales", "mensual"},
		{"mensualidades", "mensual"},
		{"mensualmente", "mensual"},
		{"menta", "ment"},
		{"mental", "mental"},
		{"mentales", "mental"},
		{"mentalidad", "mental"},
		{"mentalidades", "mental"},
		{"mente", "ment"},
		{"mentha", "menth"},
		{"mentimos", "ment"},
		{"mentir", "ment"},
		{"mentiras", "mentir"},
		{"mentirosas", "mentir"},
		{"mentiroso", "mentir"},
		{"mentor", "mentor"},
		{"mentores", "mentor"},
		{"menú", "menu"},
		{"menudeo", "menude"},
		{"menudo", "menud"},
		{"mera", "mer"},
		{"meramente", "mer"},
		{"mercadear", "mercad"},
		{"mercadeo", "mercade"},
		{"mercader", "mercad"},
		{"mercaderes", "mercader"},
		{"mercado", "merc"},
		{"mercados", "merc"},
		{"mercadotecnia", "mercadotecni"},
		{"mercadotencia", "mercadotent"},
		{"mercancía", "mercanc"},
		{"mercancías", "mercanc"},
		{"mercante", "mercant"},
		{"mercantes", "mercant"},
		{"mercantil", "mercantil"},
		{"mercantiles", "mercantil"},
		{"mercantilista", "mercantil"},
		{"mercedes", "merced"},
		{"mercomún", "mercomun"},
		{"mercosur", "mercosur"},
		{"mercurio", "mercuri"},
		{"merece", "merec"},
		{"merecedor", "merecedor"},
		{"merecedores", "merecedor"},
		{"merecen", "merec"},
		{"merecerlo", "merec"},
		{"merecía", "merec"},
		{"merecida", "merec"},
		{"merecido", "merec"},
		{"merecimientos", "merec"},
		{"mereció", "merec"},
		{"merengue", "mereng"},
		{"merengues", "mereng"},
		{"merezcan", "merezc"},
		{"merge", "merg"},
		{"mérida", "mer"},
		{"mérito", "merit"},
		{"meritorio", "meritori"},
		{"méritos", "merit"},
		{"merlene", "merlen"},
		{"merlis", "merlis"},
		{"merman", "merm"},
		{"mermeladas", "mermel"},
		{"mero", "mer"},
		{"merrill", "merrill"},
		{"mes", "mes"},
		{"mesa", "mes"},
		{"mesas", "mes"},
		{"mese", "mes"},
		{"meseros", "meser"},
		{"meses", "mes"},
		{"mesiánica", "mesian"},
		{"mesilla", "mesill"},
		{"mesoamericana", "mesoamerican"},
		{"mesoamericanas", "mesoamerican"},
		{"mesoamericano", "mesoamerican"},
		{"mesoamericanos", "mesoamerican"},
		{"mestre", "mestr"},
		{"meta", "met"},
		{"metabolismo", "metabol"},
		{"metafísica", "metafis"},
		{"metáfora", "metafor"},
		{"metal", "metal"},
		{"metalero", "metaler"},
		{"metales", "metal"},
		{"metálica", "metal"},
		{"metálico", "metal"},
		{"metálicos", "metal"},
		{"metallica", "metall"},
		{"metamucil", "metamucil"},
		{"metan", "met"},
		{"metano", "metan"},
		{"metas", "met"},
		{"metate", "metat"},
		{"mete", "met"},
		{"meten", "met"},
		{"meteorito", "meteorit"},
		{"meteorológica", "meteorolog"},
		{"meteorológico", "meteorolog"},
		{"meteorológicos", "meteorolog"},
		{"meteoros", "meteor"},
		{"meter", "met"},
		{"meterle", "met"},
		{"meterlo", "met"},
		{"meternos", "met"},
		{"meterse", "met"},
		{"metí", "met"},
		{"metían", "met"},
		{"meticulosamente", "meticul"},
		{"metido", "met"},
		{"metidos", "met"},
		{"metiendo", "met"},
		{"metiera", "met"},
		{"metió", "met"},
		{"metodista", "metod"},
		{"método", "metod"},
		{"metodología", "metodolog"},
		{"metodologías", "metodolog"},
		{"métodos", "metod"},
		{"metrallas", "metrall"},
		{"metralleta", "metrallet"},
		{"metralletas", "metrallet"},
		{"métricas", "metric"},
		{"metro", "metr"},
		{"metrología", "metrolog"},
		{"metrópoli", "metropoli"},
		{"metropolitana", "metropolitan"},
		{"metropolitano", "metropolitan"},
		{"metropolitanos", "metropolitan"},
		{"metrorrey", "metrorrey"},
		{"metros", "metr"},
		{"metternich", "metternich"},
		{"metzelaars", "metzelaars"},
		{"mexica", "mexic"},
		{"mexicali", "mexicali"},
		{"mexican", "mexic"},
		{"mexicana", "mexican"},
		{"mexicanas", "mexican"},
		{"mexicanidad", "mexican"},
		{"mexicanismo", "mexican"},
		{"mexicano", "mexican"},
		{"mexicanos", "mexican"},
		{"mexicas", "mexic"},
		{"mexicayótl", "mexicayotl"},
		{"mexico", "mexic"},
		{"méxico", "mexic"},
		{"méxicoestadounidense", "mexicoestadounidens"},
		{"", ""},
		{"méxiconorteamericanos", "mexiconorteamerican"},
		{"", ""},
		{"méxicoquerétaro", "mexicoqueretar"},
		{"mexinox", "mexinox"},
		{"mexiquense", "mexiquens"},
		{"meyer", "mey"},
		{"meza", "mez"},
		{"mezcal", "mezcal"},
		{"mezcla", "mezcl"},
		{"mezclada", "mezcl"},
		{"mezcladas", "mezcl"},
		{"mezclado", "mezcl"},
		{"mezclar", "mezcl"},
		{"mezclilla", "mezclill"},
		{"mezclo", "mezcl"},
		{"mezquinas", "mezquin"},
		{"mezquino", "mezquin"},
		{"mezquinos", "mezquin"},
		{"mezquital", "mezquital"},
		{"mg", "mg"},
		{"mi", "mi"},
		{"mí", "mi"},
		{"mia", "mia"},
		{"miami", "miami"},
		{"mich", "mich"},
		{"michael", "michael"},
		{"michel", "michel"},
		{"michelle", "michell"},
		{"michoacán", "michoacan"},
		{"michoacanazo", "michoacanaz"},
		{"michoacano", "michoacan"},
		{"michoacanos", "michoacan"},
		{"mickey", "mickey"},
		{"mico", "mic"},
		{"micosis", "micosis"},
		{"micro", "micr"},
		{"microbus", "microbus"},
		{"microbús", "microbus"},
		{"microbuses", "microbus"},
		{"microcomputadoras", "microcomput"},
		{"microelectronics", "microelectronics"},
		{"microempresa", "microempres"},
		{"microempresario", "microempresari"},
		{"microempresarios", "microempresari"},
		{"microempresas", "microempres"},
		{"micrófono", "microfon"},
		{"microindustriales", "microindustrial"},
		{"microprocesador", "microproces"},
		{"micros", "micr"},
		{"microsoft", "microsoft"},
		{"microsystems", "microsystems"},
		{"middletown", "middletown"},
		{"mide", "mid"},
		{"midieron", "mid"},
		{"midió", "mid"},
		{"mie", "mie"},
		{"miedo", "mied"},
		{"miedos", "mied"},
		{"miel", "miel"},
		{"mielenización", "mieleniz"},
		{"miembro", "miembr"},
		{"miembros", "miembr"},
		{"mientan", "mient"},
		{"miente", "mient"},
		{"mientras", "mientr"},
		{"mier", "mier"},
		{"miercoles", "miercol"},
		{"miércoles", "miercol"},
		{"migajas", "migaj"},
		{"migdalia", "migdali"},
		{"mignón", "mignon"},
		{"migración", "migracion"},
		{"migraciones", "migracion"},
		{"migrante", "migrant"},
		{"miguel", "miguel"},
		{"miguelito", "miguelit"},
		{"mijail", "mijail"},
		{"mijares", "mijar"},
		{"mike", "mik"},
		{"mil", "mil"},
		{"milagro", "milagr"},
		{"milagros", "milagr"},
		{"milagrosos", "milagr"},
		{"milan", "mil"},
		{"milán", "milan"},
		{"milanés", "milanes"},
		{"mildred", "mildr"},
		{"mile", "mil"},
		{"milenios", "mileni"},
		{"miles", "mil"},
		{"milésimas", "milesim"},
		{"mililitros", "mililitr"},
		{"milímetros", "milimetr"},
		{"milita", "milit"},
		{"militan", "milit"},
		{"militancia", "milit"},
		{"militante", "milit"},
		{"militantes", "milit"},
		{"militar", "milit"},
		{"militares", "militar"},
		{"militarización", "militariz"},
		{"militarizada", "militariz"},
		{"milla", "mill"},
		{"millar", "mill"},
		{"millares", "millar"},
		{"millas", "mill"},
		{"miller", "mill"},
		{"milli", "milli"},
		{"millón", "millon"},
		{"millonarias", "millonari"},
		{"millonario", "millonari"},
		{"millonarios", "millonari"},
		{"millones", "millon"},
		{"milmo", "milm"},
		{"milton", "milton"},
		{"mima", "mim"},
		{"mimi", "mimi"},
		{"mimí", "mim"},
		{"mimiahuápam", "mimiahuapam"},
		{"mímica", "mimic"},
		{"min", "min"},
		{"mina", "min"},
		{"mindes", "mind"},
		{"minera", "miner"},
		{"mineral", "mineral"},
		{"minerales", "mineral"},
		{"mineros", "miner"},
		{"minerva", "minerv"},
		{"minesota", "minesot"},
		{"mini", "mini"},
		{"minibuses", "minibus"},
		{"minicomputadoras", "minicomput"},
		{"minicumbre", "minicumbr"},
		{"minilab", "minilab"},
		{"minima", "minim"},
		{"mínima", "minim"},
		{"mínimas", "minim"},
		{"minimizando", "minimiz"},
		{"minimizar", "minimiz"},
		{"mínimo", "minim"},
		{"mínimos", "min"},
		{"ministerial", "ministerial"},
		{"ministerio", "ministeri"},
		{"ministro", "ministr"},
		{"ministros", "ministr"},
		{"minisubmarinos", "minisubmarin"},
		{"minivolei", "minivolei"},
		{"minivoleibol", "minivoleibol"},
		{"minnesota", "minnesot"},
		{"minor", "minor"},
		{"minorías", "minor"},
		{"minoristas", "minor"},
		{"minoritaria", "minoritari"},
		{"minoritarios", "minoritari"},
		{"mintió", "mint"},
		{"minton", "minton"},
		{"mintonette", "mintonett"},
		{"minuciosa", "minuci"},
		{"minúsculos", "minuscul"},
		{"minusvalía", "minusval"},
		{"minusválidos", "minusval"},
		{"minuta", "minut"},
		{"minuto", "minut"},
		{"minutos", "minut"},
		{"mío", "mio"},
		{"miocardio", "miocardi"},
		{"miopía", "miop"},
		{"míos", "mios"},
		{"mira", "mir"},
		{"miraba", "mir"},
		{"mirada", "mir"},
		{"miradita", "miradit"},
		{"mirador", "mirador"},
		{"miragoane", "miragoan"},
		{"mírale", "miral"},
		{"miramón", "miramon"},
		{"miramontes", "miramont"},
		{"miranda", "mirand"},
		{"mirando", "mir"},
		{"mirar", "mir"},
		{"miras", "mir"},
		{"miravalle", "miravall"},
		{"mire", "mir"},
		{"miré", "mir"},
		{"mireles", "mirel"},
		{"miren", "mir"},
		{"mireya", "mirey"},
		{"miriam", "miriam"},
		{"mirielle", "miriell"},
		{"miró", "mir"},
		{"miroslava", "miroslav"},
		{"mirren", "mirr"},
		{"mis", "mis"},
		{"misa", "mis"},
		{"misas", "mis"},
		{"miscelanea", "miscelane"},
		{"miserables", "miser"},
		{"miseria", "miseri"},
		{"misericordiosos", "misericordi"},
		{"mishima", "mishim"},
		{"misión", "mision"},
		{"misionero", "misioner"},
		{"misiones", "mision"},
		{"misma", "mism"},
		{"mismas", "mism"},
		{"mismo", "mism"},
		{"mismos", "mism"},
		{"miss", "miss"},
		{"missael", "missael"},
		{"missisipi", "missisipi"},
		{"mississippi", "mississippi"},
		{"missouri", "missouri"},
		{"misterio", "misteri"},
		{"misterios", "misteri"},
		{"misteriosa", "misteri"},
		{"misterioso", "misteri"},
		{"místico", "mistic"},
		{"mit", "mit"},
		{"mitad", "mit"},
		{"mitades", "mitad"},
		{"mítica", "mitic"},
		{"mítico", "mitic"},
		{"mitigantes", "mitig"},
		{"mitin", "mitin"},
		{"mito", "mit"},
		{"mitología", "mitolog"},
		{"mitológico", "mitolog"},
		{"mitos", "mit"},
		{"mitotes", "mitot"},
		{"mitras", "mitr"},
		{"mitsukoshi", "mitsukoshi"},
		{"mitterrand", "mitterrand"},
		{"mitzy", "mitzy"},
		{"mixtas", "mixt"},
		{"mka", "mka"},
		{"mnp", "mnp"},
		{"moais", "moais"},
		{"moaré", "moar"},
		{"mobiliario", "mobiliari"},
		{"mobutu", "mobutu"},
		{"moca", "moc"},
		{"mochila", "mochil"},
		{"mochilas", "mochil"},
		{"mochis", "mochis"},
		{"moción", "mocion"},
		{"moctezuma", "moctezum"},
		{"moda", "mod"},
		{"modales", "modal"},
		{"modalidad", "modal"},
		{"modalidades", "modal"},
		{"modas", "mod"},
		{"mode", "mod"},
		{"modelo", "model"},
		{"modelos", "model"},
		{"módem", "modem"},
		{"modera", "moder"},
		{"moderación", "moder"},
		{"moderada", "moder"},
		{"moderado", "moder"},
		{"moderados", "moder"},
		{"moderna", "modern"},
		{"modernas", "modern"},
		{"modernicen", "modernic"},
		{"modernidad", "modern"},
		{"modernismo", "modern"},
		{"modernización", "moderniz"},
		{"modernizada", "moderniz"},
		{"modernizador", "moderniz"},
		{"modernizar", "moderniz"},
		{"modernizarla", "moderniz"},
		{"modernizarse", "moderniz"},
		{"moderno", "modern"},
		{"modernos", "mod"},
		{"modesta", "modest"},
		{"modestas", "modest"},
		{"modestia", "modesti"},
		{"modesto", "modest"},
		{"modestos", "modest"},
		{"modifica", "modif"},
		{"modificación", "modif"},
		{"modificaciones", "modif"},
		{"modificada", "modific"},
		{"modificadas", "modific"},
		{"modificados", "modific"},
		{"modificando", "modific"},
		{"modificar", "modific"},
		{"modificarse", "modific"},
		{"modificó", "modific"},
		{"modista", "modist"},
		{"modo", "mod"},
		{"modos", "mod"},
		{"modulada", "modul"},
		{"modular", "modul"},
		{"módulo", "modul"},
		{"módulos", "modul"},
		{"moisés", "moises"},
		{"mol", "mol"},
		{"moldeado", "mold"},
		{"moldearon", "mold"},
		{"mole", "mol"},
		{"molesta", "molest"},
		{"molestada", "molest"},
		{"molestados", "molest"},
		{"molestan", "molest"},
		{"molestar", "molest"},
		{"molestarán", "molest"},
		{"molestarse", "molest"},
		{"moleste", "molest"},
		{"molestia", "molesti"},
		{"molestias", "molesti"},
		{"molesto", "molest"},
		{"molestó", "molest"},
		{"molestos", "molest"},
		{"molida", "mol"},
		{"moliere", "molier"},
		{"molina", "molin"},
		{"molino", "molin"},
		{"moll", "moll"},
		{"mollejas", "mollej"},
		{"molones", "molon"},
		{"mombasa", "mombas"},
		{"momentánea", "momentane"},
		{"momentáneamente", "momentan"},
		{"momentáneo", "momentane"},
		{"momento", "moment"},
		{"momentos", "moment"},
		{"monarca", "monarc"},
		{"monarquía", "monarqu"},
		{"monárquica", "monarqu"},
		{"monas", "mon"},
		{"moncada", "monc"},
		{"monclova", "monclov"},
		{"mondongo", "mondong"},
		{"moneda", "moned"},
		{"monedas", "moned"},
		{"monederos", "moneder"},
		{"monetaria", "monetari"},
		{"monetarias", "monetari"},
		{"monetario", "monetari"},
		{"mónica", "monic"},
		{"monipodio", "monipodi"},
		{"monitor", "monitor"},
		{"monitoreo", "monitore"},
		{"monitoreos", "monitore"},
		{"monitores", "monitor"},
		{"monjaras", "monj"},
		{"monjaraz", "monjaraz"},
		{"monjas", "monj"},
		{"monk", "monk"},
		{"moño", "moñ"},
		{"monogollas", "monogoll"},
		{"monolito", "monolit"},
		{"monoplaza", "monoplaz"},
		{"monoplazas", "monoplaz"},
		{"monopolio", "monopoli"},
		{"monopolios", "monopoli"},
		{"monóxido", "monox"},
		{"monroy", "monroy"},
		{"monsivais", "monsivais"},
		{"monsiváis", "monsiv"},
		{"monstruosidad", "monstru"},
		{"monstruosos", "monstruos"},
		{"monta", "mont"},
		{"montada", "mont"},
		{"montado", "mont"},
		{"montaje", "montaj"},
		{"montajes", "montaj"},
		{"montalbán", "montalban"},
		{"montalvo", "montalv"},
		{"montana", "montan"},
		{"montaña", "montañ"},
		{"montañas", "montañ"},
		{"montando", "mont"},
		{"montaño", "montañ"},
		{"montar", "mont"},
		{"monte", "mont"},
		{"monteleone", "monteleon"},
		{"montemayor", "montemayor"},
		{"montemorelos", "montemorel"},
		{"monten", "mont"},
		{"montepíos", "montepi"},
		{"montera", "monter"},
		{"montero", "monter"},
		{"monterrey", "monterrey"},
		{"monterreynuevo", "monterreynuev"},
		{"monterrubio", "monterrubi"},
		{"montes", "mont"},
		{"montgomery", "montgomery"},
		{"months", "months"},
		{"montículo", "monticul"},
		{"montiel", "montiel"},
		{"monto", "mont"},
		{"montó", "mont"},
		{"montón", "monton"},
		{"montos", "mont"},
		{"montoya", "montoy"},
		{"montreal", "montreal"},
		{"montura", "montur"},
		{"monumental", "monumental"},
		{"monumentalidad", "monumental"},
		{"monumento", "monument"},
		{"monumentos", "monument"},
		{"moodys", "moodys"},
		{"mor", "mor"},
		{"mora", "mor"},
		{"morada", "mor"},
		{"morado", "mor"},
		{"moraes", "mor"},
		{"moral", "moral"},
		{"morales", "moral"},
		{"moralidad", "moral"},
		{"moralistas", "moral"},
		{"moralización", "moraliz"},
		{"moralizarse", "moraliz"},
		{"moralmente", "moral"},
		{"moranchel", "moranchel"},
		{"moratoria", "moratori"},
		{"morbilidad", "morbil"},
		{"morbosa", "morbos"},
		{"morbosos", "morbos"},
		{"mordida", "mord"},
		{"mordidas", "mord"},
		{"mordiendo", "mord"},
		{"mordiscos", "mordisc"},
		{"morelenses", "morelens"},
		{"morelia", "moreli"},
		{"morell", "morell"},
		{"morelos", "morel"},
		{"morena", "moren"},
		{"morenas", "moren"},
		{"morenitos", "morenit"},
		{"moreno", "moren"},
		{"moresa", "mores"},
		{"morgan", "morg"},
		{"moría", "mor"},
		{"morillo", "morill"},
		{"morín", "morin"},
		{"morir", "mor"},
		{"morirse", "mor"},
		{"morón", "moron"},
		{"morones", "moron"},
		{"moroso", "moros"},
		{"morozoff", "morozoff"},
		{"morris", "morris"},
		{"morrison", "morrison"},
		{"morros", "morr"},
		{"mortal", "mortal"},
		{"mortem", "mortem"},
		{"moruecos", "moruec"},
		{"mos", "mos"},
		{"moschino", "moschin"},
		{"moscovita", "moscovit"},
		{"moscu", "moscu"},
		{"moscú", "moscu"},
		{"mosquito", "mosquit"},
		{"mostaza", "mostaz"},
		{"mostraba", "mostr"},
		{"mostraban", "mostr"},
		{"mostrada", "mostr"},
		{"mostrado", "mostr"},
		{"mostrando", "mostr"},
		{"mostrándome", "mostr"},
		{"mostrándose", "mostr"},
		{"mostrar", "mostr"},
		{"mostrará", "mostr"},
		{"mostrarle", "mostr"},
		{"mostrarles", "mostr"},
		{"mostrarlo", "mostr"},
		{"mostraron", "mostr"},
		{"mostrarse", "mostr"},
		{"mostró", "mostr"},
		{"mota", "mot"},
		{"moteles", "motel"},
		{"motiva", "motiv"},
		{"motivación", "motiv"},
		{"motivacional", "motivacional"},
		{"motivacionales", "motivacional"},
		{"motivaciones", "motiv"},
		{"motivado", "motiv"},
		{"motivador", "motiv"},
		{"motivante", "motiv"},
		{"motivar", "motiv"},
		{"motivarán", "motiv"},
		{"motivarlas", "motiv"},
		{"motivarlo", "motiv"},
		{"motivarlos", "motiv"},
		{"motive", "motiv"},
		{"motivo", "motiv"},
		{"motivó", "motiv"},
		{"motivos", "motiv"},
		{"moto", "mot"},
		{"motocicleta", "motociclet"},
		{"motociclismo", "motocicl"},
		{"motociclista", "motocicl"},
		{"motociclistas", "motocicl"},
		{"motocross", "motocross"},
		{"motolinía", "motolin"},
		{"motor", "motor"},
		{"motorenwerke", "motorenwerk"},
		{"motores", "motor"},
		{"motors", "motors"},
		{"motos", "mot"},
		{"motriz", "motriz"},
		{"mountain", "mountain"},
		{"mouse", "mous"},
		{"moussavi", "moussavi"},
		{"movamos", "mov"},
		{"mover", "mov"},
		{"moveré", "mov"},
		{"movernos", "mov"},
		{"moverse", "mov"},
		{"movían", "mov"},
		{"movida", "mov"},
		{"movido", "mov"},
		{"moviendo", "mov"},
		{"moviéndose", "mov"},
		{"movieron", "mov"},
		{"móvil", "movil"},
		{"móviles", "movil"},
		{"movilidad", "movil"},
		{"moviliza", "moviliz"},
		{"movilización", "moviliz"},
		{"movilizaciones", "moviliz"},
		{"movilizar", "moviliz"},
		{"movilizó", "moviliz"},
		{"movimiento", "movimient"},
		{"movimientos", "movimient"},
		{"movió", "mov"},
		{"moya", "moy"},
		{"moyobamba", "moyobamb"},
		{"moyssén", "moyssen"},
		{"moza", "moz"},
		{"mozart", "mozart"},
		{"mozo", "moz"},
		{"mozos", "moz"},
		{"mr", "mr"},
		{"mstislav", "mstislav"},
		{"mtito", "mtit"},
		{"mtley", "mtley"},
		{"mu", "mu"},
		{"mucha", "much"},
		{"muchacha", "muchach"},
		{"muchachas", "muchach"},
		{"muchacho", "muchach"},
		{"muchachos", "muchach"},
		{"muchas", "much"},
		{"muchedumbre", "muchedumbr"},
		{"muchísima", "muchisim"},
		{"muchísimas", "muchisim"},
		{"muchísimo", "muchisim"},
		{"muchísimos", "muchis"},
		{"mucho", "much"},
		{"muchos", "much"},
		{"muda", "mud"},
		{"mudanza", "mudanz"},
		{"mudará", "mud"},
		{"mudarse", "mud"},
		{"mueble", "muebl"},
		{"muebles", "muebl"},
		{"mueca", "muec"},
		{"muelle", "muell"},
		{"muelles", "muell"},
		{"muera", "muer"},
		{"muérdago", "muerdag"},
		{"muere", "muer"},
		{"muerta", "muert"},
		{"muertas", "muert"},
		{"muerte", "muert"},
		{"muertes", "muert"},
		{"muerto", "muert"},
		{"muertos", "muert"},
		{"muestra", "muestr"},
		{"muestran", "muestr"},
		{"muestrario", "muestrari"},
		{"muestras", "muestr"},
		{"muestre", "muestr"},
		{"muestren", "muestr"},
		{"muestreo", "muestre"},
		{"mueve", "muev"},
		{"mueven", "muev"},
		{"mugre", "mugr"},
		{"muguerza", "muguerz"},
		{"muguiro", "muguir"},
		{"mujer", "muj"},
		{"mujercita", "mujercit"},
		{"mujeres", "mujer"},
		{"mula", "mul"},
		{"mulato", "mulat"},
		{"muleta", "mulet"},
		{"muletas", "mulet"},
		{"muletazos", "muletaz"},
		{"mulino", "mulin"},
		{"mulitas", "mulit"},
		{"muller", "mull"},
		{"multa", "mult"},
		{"multadas", "mult"},
		{"multado", "mult"},
		{"multados", "mult"},
		{"multar", "mult"},
		{"multas", "mult"},
		{"multichip", "multichip"},
		{"multicolor", "multicolor"},
		{"multiculturalismo", "multicultural"},
		{"multilateral", "multilateral"},
		{"multilaterales", "multilateral"},
		{"multimedia", "multimedi"},
		{"multimedios", "multimedi"},
		{"multimillonariamente", "multimillonari"},
		{"", ""},
		{"multimillonario", "multimillonari"},
		{"multimillonarios", "multimillonari"},
		{"multimodal", "multimodal"},
		{"multinacionales", "multinacional"},
		{"multipanel", "multipanel"},
		{"múltiple", "multipl"},
		{"múltiples", "multipl"},
		{"multiplicación", "multipl"},
		{"multiplicado", "multiplic"},
		{"multiplicar", "multiplic"},
		{"multiplicaron", "multiplic"},
		{"multiplicarse", "multiplic"},
		{"multiplicó", "multiplic"},
		{"multipliquen", "multipliqu"},
		{"múltiplos", "multipl"},
		{"multitarea", "multitare"},
		{"multitud", "multitud"},
		{"multitudes", "multitud"},
		{"multitudinaria", "multitudinari"},
		{"mundanales", "mundanal"},
		{"mundial", "mundial"},
		{"mundiales", "mundial"},
		{"mundialista", "mundial"},
		{"mundialmente", "mundial"},
		{"mundillo", "mundill"},
		{"mundo", "mund"},
		{"mundos", "mund"},
		{"mundy", "mundy"},
		{"muñeca", "muñec"},
		{"munguía", "mungu"},
		{"munich", "munich"},
		{"municipal", "municipal"},
		{"municipales", "municipal"},
		{"munícipe", "municip"},
		{"municipio", "municipi"},
		{"municipios", "municipi"},
		{"muñiz", "muñiz"},
		{"muñoz", "muñoz"},
		{"mural", "mural"},
		{"muralismo", "mural"},
		{"murat", "murat"},
		{"muriendo", "mur"},
		{"muriera", "mur"},
		{"murieron", "mur"},
		{"murillo", "murill"},
		{"murió", "mur"},
		{"muro", "mur"},
		{"muros", "mur"},
		{"muscular", "muscul"},
		{"musculares", "muscular"},
		{"músculo", "muscul"},
		{"museo", "muse"},
		{"museografía", "museograf"},
		{"museógrafo", "museograf"},
		{"museos", "muse"},
		{"museum", "museum"},
		{"music", "music"},
		{"musica", "music"},
		{"música", "music"},
		{"musical", "musical"},
		{"musicales", "musical"},
		{"músico", "music"},
		{"músicos", "music"},
		{"muslo", "musl"},
		{"muslos", "musl"},
		{"mussa", "muss"},
		{"musset", "musset"},
		{"mustang", "mustang"},
		{"musulmán", "musulman"},
		{"musulmana", "musulman"},
		{"musulmanas", "musulman"},
		{"musulmanes", "musulman"},
		{"mutación", "mutacion"},
		{"mutilados", "mutil"},
		{"mutua", "mutu"},
		{"mutualidad", "mutual"},
		{"mutualista", "mutual"},
		{"mutuamente", "mutu"},
		{"mutuas", "mutu"},
		{"mutuo", "mutu"},
		{"muy", "muy"},
		{"múzquiz", "muzquiz"},
		{"my", "my"},
		{"myers", "myers"},
		{"mystic", "mystic"},
		{"n", "n"},
		{"na", "na"},
		{"nabor", "nabor"},
		{"nace", "nac"},
		{"nacen", "nac"},
		{"nacer", "nac"},
		{"nacho", "nach"},
		{"nací", "nac"},
		{"nacida", "nac"},
		{"nacido", "nac"},
		{"nacidos", "nac"},
		{"naciera", "nac"},
		{"nacieran", "nac"},
		{"nacieron", "nac"},
		{"nacimiento", "nacimient"},
		{"nacimos", "nac"},
		{"nació", "nac"},
		{"nación", "nacion"},
		{"nacional", "nacional"},
		{"nacionales", "nacional"},
		{"nacionalidad", "nacional"},
		{"nacionalista", "nacional"},
		{"nacionalización", "nacionaliz"},
		{"nacionalizada", "nacionaliz"},
		{"naciones", "nacion"},
		{"nada", "nad"},
		{"nadador", "nadador"},
		{"nadadores", "nadador"},
		{"nadando", "nad"},
		{"nadar", "nad"},
		{"nadie", "nadi"},
		{"nadores", "nador"},
		{"nafin", "nafin"},
		{"nafinsa", "nafins"},
		{"nafta", "naft"},
		{"naftacorporation", "naftacorporation"},
		{"nagoya", "nagoy"},
		{"nahuas", "nahu"},
		{"nailon", "nailon"},
		{"nairobi", "nairobi"},
		{"naju", "naju"},
		{"nancy", "nancy"},
		{"nandin", "nandin"},
		{"nandín", "nandin"},
		{"napoleón", "napoleon"},
		{"nápoles", "napol"},
		{"naranja", "naranj"},
		{"naranjas", "naranj"},
		{"naranjo", "naranj"},
		{"narco", "narc"},
		{"narcoanálisis", "narcoanalisis"},
		{"narcos", "narc"},
		{"narcotraficante", "narcotraf"},
		{"narcotraficantes", "narcotraf"},
		{"narcotráfico", "narcotraf"},
		{"nariz", "nariz"},
		{"narozhilenko", "narozhilenk"},
		{"narra", "narr"},
		{"narrado", "narr"},
		{"narrador", "narrador"},
		{"narran", "narr"},
		{"narrativa", "narrat"},
		{"narrativo", "narrat"},
		{"narró", "narr"},
		{"nasa", "nas"},
		{"nasal", "nasal"},
		{"nasales", "nasal"},
		{"nat", "nat"},
		{"nata", "nat"},
		{"natación", "natacion"},
		{"natal", "natal"},
		{"natalicio", "natalici"},
		{"natalie", "natali"},
		{"natalio", "natali"},
		{"nate", "nat"},
		{"national", "national"},
		{"nativa", "nativ"},
		{"nativas", "nativ"},
		{"natividad", "nativ"},
		{"nativo", "nativ"},
		{"nativos", "nativ"},
		{"nato", "nat"},
		{"natural", "natural"},
		{"naturales", "natural"},
		{"naturaleza", "naturalez"},
		{"naturalmente", "natural"},
		{"nature", "natur"},
		{"naturista", "natur"},
		{"naucalpan", "naucalp"},
		{"naufragaron", "naufrag"},
		{"naufragio", "naufragi"},
		{"naufragios", "naufragi"},
		{"náufrago", "naufrag"},
		{"naufragó", "naufrag"},
		{"náutico", "nautic"},
		{"náuticos", "nautic"},
		{"nava", "nav"},
		{"navaja", "navaj"},
		{"navajas", "navaj"},
		{"navajazo", "navajaz"},
		{"navajearon", "navaj"},
		{"naval", "naval"},
		{"navales", "naval"},
		{"navarrete", "navarret"},
		{"navarro", "navarr"},
		{"nave", "nav"},
		{"naveda", "naved"},
		{"navegación", "naveg"},
		{"navegar", "naveg"},
		{"naves", "nav"},
		{"navidad", "navid"},
		{"naviera", "nav"},
		{"navieras", "nav"},
		{"navieros", "navier"},
		{"navío", "navi"},
		{"navíos", "navi"},
		{"navismo", "navism"},
		{"navista", "navist"},
		{"navojoa", "navojo"},
		{"nayarit", "nayarit"},
		{"nazis", "nazis"},
		{"nbc", "nbc"},
		{"near", "near"},
		{"neblina", "neblin"},
		{"necaxa", "necax"},
		{"necaxistas", "necax"},
		{"necedad", "neced"},
		{"necesaria", "necesari"},
		{"necesariamente", "necesari"},
		{"necesarias", "necesari"},
		{"necesario", "necesari"},
		{"necesarios", "necesari"},
		{"necesidad", "neces"},
		{"necesidades", "neces"},
		{"necesita", "necesit"},
		{"necesitaba", "necesit"},
		{"necesitábamos", "necesit"},
		{"necesitaban", "necesit"},
		{"necesitado", "necesit"},
		{"necesitamos", "necesit"},
		{"necesitan", "necesit"},
		{"necesitará", "necesit"},
		{"necesitarán", "necesit"},
		{"necesitaría", "necesit"},
		{"necesitarían", "necesit"},
		{"necesitarlas", "necesit"},
		{"necesite", "necesit"},
		{"necesiten", "necesit"},
		{"necesito", "necesit"},
		{"necesitó", "necesit"},
		{"necia", "neci"},
		{"necrófilo", "necrofil"},
		{"nedda", "nedd"},
		{"nefasto", "nefast"},
		{"nefrología", "nefrolog"},
		{"negaba", "neg"},
		{"negaban", "neg"},
		{"negada", "neg"},
		{"negado", "neg"},
		{"negando", "neg"},
		{"negar", "neg"},
		{"negara", "neg"},
		{"negarla", "neg"},
		{"negarle", "neg"},
		{"negaron", "neg"},
		{"negarse", "neg"},
		{"negativa", "negat"},
		{"negativamente", "negat"},
		{"negativas", "negat"},
		{"negativo", "negat"},
		{"negativos", "negat"},
		{"negligencia", "negligent"},
		{"negó", "neg"},
		{"negocia", "negoci"},
		{"negociables", "negoci"},
		{"negociación", "negoci"},
		{"negociaciones", "negoci"},
		{"negociada", "negoci"},
		{"negociado", "negoci"},
		{"negociador", "negoci"},
		{"negociadora", "negoci"},
		{"negociadores", "negoci"},
		{"negociando", "negoci"},
		{"negociantes", "negoci"},
		{"negociar", "negoci"},
		{"negociarían", "negoci"},
		{"negociarlo", "negoci"},
		{"negociaron", "negoci"},
		{"negociazo", "negociaz"},
		{"negocio", "negoci"},
		{"negoció", "negoc"},
		{"negocios", "negoci"},
		{"negra", "negr"},
		{"negras", "negr"},
		{"negrete", "negret"},
		{"negro", "negr"},
		{"negromex", "negromex"},
		{"negroponte", "negropont"},
		{"negros", "negr"},
		{"nehru", "nehru"},
		{"neil", "neil"},
		{"nejedly", "nejedly"},
		{"nelson", "nelson"},
		{"neófito", "neofit"},
		{"neográfica", "neograf"},
		{"neoleonés", "neoleones"},
		{"neoliberal", "neoliberal"},
		{"neoliberales", "neoliberal"},
		{"neoliberalismo", "neoliberal"},
		{"neoloneses", "neolones"},
		{"neomexicanismo", "neomexican"},
		{"neón", "neon"},
		{"neopopulistas", "neopopul"},
		{"neoproteccionista", "neoproteccion"},
		{"neosectores", "neosector"},
		{"neoyorkina", "neoyorkin"},
		{"neoyorkino", "neoyorkin"},
		{"neoyorquina", "neoyorquin"},
		{"neoyorquinas", "neoyorquin"},
		{"neoyorquino", "neoyorquin"},
		{"népotas", "nepot"},
		{"nepotismo", "nepot"},
		{"neptuno", "neptun"},
		{"nereus", "nereus"},
		{"neri", "neri"},
		{"neruda", "nerud"},
		{"nerviosa", "nervios"},
		{"nerviosas", "nervi"},
		{"nerviosismo", "nervi"},
		{"nervioso", "nervios"},
		{"nerviosos", "nervios"},
		{"nestlé", "nestl"},
		{"nestor", "nestor"},
		{"neta", "net"},
		{"netamente", "net"},
		{"netas", "net"},
		{"neto", "net"},
		{"netos", "net"},
		{"network", "network"},
		{"neumología", "neumolog"},
		{"neumonía", "neumon"},
		{"neurastenia", "neurasteni"},
		{"neurobiólogo", "neurobiolog"},
		{"neurocirujanos", "neurocirujan"},
		{"neurología", "neurolog"},
		{"neurológicas", "neurolog"},
		{"neurológico", "neurolog"},
		{"neurona", "neuron"},
		{"neuronales", "neuronal"},
		{"neuronas", "neuron"},
		{"neurosis", "neurosis"},
		{"neutral", "neutral"},
		{"neutralizar", "neutraliz"},
		{"neutro", "neutr"},
		{"neutros", "neutr"},
		{"neuville", "neuvill"},
		{"nevando", "nev"},
		{"new", "new"},
		{"newagers", "newagers"},
		{"newagismo", "newag"},
		{"newport", "newport"},
		{"newquist", "newquist"},
		{"news", "news"},
		{"newsweek", "newsweek"},
		{"newton", "newton"},
		{"nexos", "nex"},
		{"next", "next"},
		{"nextstep", "nextstep"},
		{"ney", "ney"},
		{"neza", "nez"},
		{"nfl", "nfl"},
		{"nhra", "nhra"},
		{"ni", "ni"},
		{"nicaragua", "nicaragu"},
		{"nicaragüense", "nicaragüens"},
		{"nicho", "nich"},
		{"nichols", "nichols"},
		{"nichos", "nich"},
		{"nickin", "nickin"},
		{"ñico", "ñic"},
		{"nicola", "nicol"},
		{"nicolás", "nicolas"},
		{"nicole", "nicol"},
		{"nidas", "nid"},
		{"nido", "nid"},
		{"nidos", "nid"},
		{"niebla", "niebl"},
		{"niega", "nieg"},
		{"niegan", "nieg"},
		{"niegue", "nieg"},
		{"nieguen", "nieg"},
		{"nieta", "niet"},
		{"nietecita", "nietecit"},
		{"nieto", "niet"},
		{"nietos", "niet"},
		{"nieve", "niev"},
		{"nieves", "niev"},
		{"nikko", "nikk"},
		{"nilda", "nild"},
		{"nina", "nin"},
		{"niña", "niñ"},
		{"niñas", "niñ"},
		{"niñez", "niñez"},
		{"ningún", "ningun"},
		{"ninguna", "ningun"},
		{"ninguno", "ningun"},
		{"niño", "niñ"},
		{"niños", "niñ"},
		{"nipón", "nipon"},
		{"níquel", "niquel"},
		{"nironenko", "nironenk"},
		{"nirvana", "nirvan"},
		{"nísperos", "nisper"},
		{"nissan", "niss"},
		{"nissho", "nissh"},
		{"nitrógeno", "nitrogen"},
		{"nivea", "nive"},
		{"nivel", "nivel"},
		{"nivelar", "nivel"},
		{"niveles", "nivel"},
		{"nixon", "nixon"},
		{"nl", "nl"},
		{"no", "no"},
		{"nó", "no"},
		{"nobel", "nobel"},
		{"nobilísimo", "nobilisim"},
		{"noble", "nobl"},
		{"nobleza", "noblez"},
		{"nocaut", "nocaut"},
		{"nocauts", "nocauts"},
		{"noche", "noch"},
		{"nochebuena", "nochebuen"},
		{"noches", "noch"},
		{"noción", "nocion"},
		{"nociones", "nocion"},
		{"nocivas", "nociv"},
		{"nocivo", "nociv"},
		{"nocturna", "nocturn"},
		{"nocturnas", "nocturn"},
		{"nocturno", "nocturn"},
		{"noé", "noe"},
		{"nogalar", "nogal"},
		{"nogales", "nogal"},
		{"nolte", "nolt"},
		{"nomás", "nomas"},
		{"nombra", "nombr"},
		{"nombradas", "nombr"},
		{"nombrado", "nombr"},
		{"nombrados", "nombr"},
		{"nombramiento", "nombramient"},
		{"nombramientos", "nombramient"},
		{"nombrar", "nombr"},
		{"nombraria", "nombrari"},
		{"nombrarlos", "nombr"},
		{"nombre", "nombr"},
		{"nombré", "nombr"},
		{"nombren", "nombr"},
		{"nombres", "nombr"},
		{"nombró", "nombr"},
		{"nómina", "nomin"},
		{"nominación", "nomin"},
		{"nominaciones", "nomin"},
		{"nominada", "nomin"},
		{"nominado", "nomin"},
		{"nominados", "nomin"},
		{"nominal", "nominal"},
		{"nominales", "nominal"},
		{"nominará", "nomin"},
		{"nonagenario", "nonagenari"},
		{"nopales", "nopal"},
		{"noquear", "noqu"},
		{"noqueará", "noqu"},
		{"nora", "nor"},
		{"norah", "norah"},
		{"noratlántico", "noratlant"},
		{"norberta", "norbert"},
		{"norberto", "norbert"},
		{"norcarolinos", "norcarolin"},
		{"norcoreano", "norcorean"},
		{"noreste", "norest"},
		{"norestense", "norestens"},
		{"norias", "nori"},
		{"noriega", "norieg"},
		{"norma", "norm"},
		{"normal", "normal"},
		{"normales", "normal"},
		{"normalidad", "normal"},
		{"normalmente", "normal"},
		{"normandía", "normand"},
		{"normar", "norm"},
		{"normas", "norm"},
		{"normatividad", "normat"},
		{"norme", "norm"},
		{"noroccidental", "noroccidental"},
		{"norris", "norris"},
		{"norte", "nort"},
		{"norteamérica", "norteamer"},
		{"norteamericana", "norteamerican"},
		{"norteamericanas", "norteamerican"},
		{"norteamericano", "norteamerican"},
		{"norteamericanos", "norteamerican"},
		{"norteña", "norteñ"},
		{"norteñas", "norteñ"},
		{"norteño", "norteñ"},
		{"norteños", "norteñ"},
		{"northern", "northern"},
		{"norwick", "norwick"},
		{"nos", "nos"},
		{"nosocomio", "nosocomi"},
		{"nosotras", "nosotr"},
		{"nosotros", "nosotr"},
		{"nostalgias", "nostalgi"},
		{"nota", "not"},
		{"notable", "notabl"},
		{"notablemente", "notabl"},
		{"notables", "notabl"},
		{"notado", "not"},
		{"notamos", "not"},
		{"notar", "not"},
		{"notaria", "notari"},
		{"notaría", "not"},
		{"notariado", "notari"},
		{"notario", "notari"},
		{"notas", "not"},
		{"noticia", "notici"},
		{"noticias", "notici"},
		{"noticiero", "noticier"},
		{"noticiosa", "notici"},
		{"noticiosos", "notici"},
		{"notificación", "notif"},
		{"notificado", "notific"},
		{"notificar", "notific"},
		{"notificarán", "notific"},
		{"notificarle", "notific"},
		{"notificaron", "notific"},
		{"notificó", "notific"},
		{"notimex", "notimex"},
		{"notó", "not"},
		{"notoria", "notori"},
		{"notoriamente", "notori"},
		{"notorias", "notori"},
		{"notorio", "notori"},
		{"nova", "nov"},
		{"novacek", "novacek"},
		{"novar", "nov"},
		{"novatas", "novat"},
		{"novatez", "novatez"},
		{"novato", "novat"},
		{"novatos", "novat"},
		{"novedad", "noved"},
		{"novedades", "novedad"},
		{"novedosas", "noved"},
		{"novedoso", "noved"},
		{"novedosos", "noved"},
		{"novel", "novel"},
		{"novela", "novel"},
		{"novelas", "novel"},
		{"novelista", "novel"},
		{"novell", "novell"},
		{"novena", "noven"},
		{"novenas", "noven"},
		{"noveno", "noven"},
		{"novia", "novi"},
		{"novias", "novi"},
		{"noviazgo", "noviazg"},
		{"noviembre", "noviembr"},
		{"novillada", "novill"},
		{"novilladas", "novill"},
		{"novillero", "noviller"},
		{"novilleros", "noviller"},
		{"novillo", "novill"},
		{"novillos", "novill"},
		{"novio", "novi"},
		{"novios", "novi"},
		{"novísimo", "novisim"},
		{"novoa", "novo"},
		{"nox", "nox"},
		{"noyola", "noyol"},
		{"np", "np"},
		{"npp", "npp"},
		{"nt", "nt"},
		{"ntx", "ntx"},
		{"nube", "nub"},
		{"nubla", "nubl"},
		{"nublados", "nubl"},
		{"nuca", "nuc"},
		{"nucleado", "nucl"},
		{"nuclear", "nucl"},
		{"núcleo", "nucle"},
		{"núcleos", "nucle"},
		{"nudillos", "nudill"},
		{"nudos", "nud"},
		{"nueces", "nuec"},
		{"nuera", "nuer"},
		{"nuestra", "nuestr"},
		{"nuestras", "nuestr"},
		{"nuestro", "nuestr"},
		{"nuestros", "nuestr"},
		{"nueva", "nuev"},
		{"nuevamente", "nuev"},
		{"nuevas", "nuev"},
		{"nueve", "nuev"},
		{"nuevo", "nuev"},
		{"nuevoleonés", "nuevoleones"},
		{"nuevoleonesa", "nuevoleones"},
		{"nuevoleoneses", "nuevoleones"},
		{"nuevos", "nuev"},
		{"nuff", "nuff"},
		{"nui", "nui"},
		{"nula", "nul"},
		{"nulas", "nul"},
		{"nulo", "nul"},
		{"numerales", "numeral"},
		{"numeralia", "numerali"},
		{"numérica", "numer"},
		{"numeritos", "numerit"},
		{"numero", "numer"},
		{"número", "numer"},
		{"numeros", "numer"},
		{"números", "numer"},
		{"numerosa", "numer"},
		{"numerosas", "numer"},
		{"numerosos", "numer"},
		{"nunca", "nunc"},
		{"nunciatura", "nunciatur"},
		{"nuncio", "nunci"},
		{"nuñez", "nuñez"},
		{"núñez", "nuñez"},
		{"nunn", "nunn"},
		{"nupcial", "nupcial"},
		{"nupcias", "nupci"},
		{"nuremberg", "nuremberg"},
		{"nurser", "nurs"},
		{"nutrición", "nutricion"},
		{"nutrientes", "nutrient"},
		{"ny", "ny"},
		{"nye", "nye"},
		{"nylon", "nylon"},
		{"nyrup", "nyrup"},
		{"nyt", "nyt"},
		{"o", "o"},
		{"ó", "ó"},
		{"oajaca", "oajac"},
		{"oaxaca", "oaxac"},
		{"oaxaqueñas", "oaxaqueñ"},
		{"obdulia", "obduli"},
		{"obdulio", "obduli"},
		{"obedece", "obedec"},
		{"obedecen", "obedec"},
		{"obedecer", "obedec"},
		{"obedecerlo", "obedec"},
		{"obedecía", "obedec"},
		{"obedecían", "obedec"},
		{"obedeció", "obedec"},
		{"obediencia", "obedient"},
		{"obispado", "obisp"},
		{"obispo", "obisp"},
		{"obispos", "obisp"},
		{"objeciones", "objecion"},
		{"objeta", "objet"},
		{"objetar", "objet"},
		{"objetare", "objetar"},
		{"objetiva", "objet"},
		{"objetividad", "objet"},
		{"objetivo", "objet"},
		{"objetivos", "objet"},
		{"objeto", "objet"},
		{"objetos", "objet"},
		{"obliga", "oblig"},
		{"obligaba", "oblig"},
		{"obligaban", "oblig"},
		{"obligación", "oblig"},
		{"obligaciones", "oblig"},
		{"obligada", "oblig"},
		{"obligado", "oblig"},
		{"obligados", "oblig"},
		{"obligan", "oblig"},
		{"obligando", "oblig"},
		{"obligándolo", "oblig"},
		{"obligar", "oblig"},
		{"obligará", "oblig"},
		{"obligaría", "oblig"},
		{"obligarían", "oblig"},
		{"obligaron", "oblig"},
		{"obligatoria", "obligatori"},
		{"obligatoriamente", "obligatori"},
		{"obligatoriedad", "obligatoried"},
		{"obligatorio", "obligatori"},
		{"obligatorios", "obligatori"},
		{"obligo", "oblig"},
		{"obligó", "oblig"},
		{"obligue", "oblig"},
		{"oboe", "obo"},
		{"oboístas", "oboist"},
		{"obra", "obra"},
		{"obrador", "obrador"},
		{"obran", "obran"},
		{"obrar", "obrar"},
		{"obras", "obras"},
		{"obregón", "obregon"},
		{"obrera", "obrer"},
		{"obrero", "obrer"},
		{"obreros", "obrer"},
		{"obscena", "obscen"},
		{"obscenas", "obscen"},
		{"obscenos", "obscen"},
		{"obscurece", "obscurec"},
		{"obscuro", "obscur"},
		{"obscuros", "obscur"},
		{"obsequian", "obsequi"},
		{"obsequiar", "obsequi"},
		{"obsequiarle", "obsequi"},
		{"obsequiaron", "obsequi"},
		{"obsequió", "obsequ"},
		{"observa", "observ"},
		{"observaba", "observ"},
		{"observaban", "observ"},
		{"observable", "observ"},
		{"observación", "observ"},
		{"observaciones", "observ"},
		{"observada", "observ"},
		{"observadas", "observ"},
		{"observado", "observ"},
		{"observador", "observ"},
		{"observadora", "observ"},
		{"observadores", "observ"},
		{"observamos", "observ"},
		{"observan", "observ"},
		{"observancia", "observ"},
		{"observando", "observ"},
		{"observar", "observ"},
		{"observaron", "observ"},
		{"observé", "observ"},
		{"observo", "observ"},
		{"observó", "observ"},
		{"obsesión", "obsesion"},
		{"obsesionado", "obsesion"},
		{"obsesionarse", "obsesion"},
		{"obsesione", "obsesion"},
		{"obsesiones", "obsesion"},
		{"obsesiva", "obses"},
		{"obsidiana", "obsidian"},
		{"obsolescencia", "obsolescent"},
		{"obstaculiza", "obstaculiz"},
		{"obstaculización", "obstaculiz"},
		{"obstaculizado", "obstaculiz"},
		{"obstáculo", "obstacul"},
		{"obstáculos", "obstacul"},
		{"obstante", "obstant"},
		{"obstinación", "obstin"},
		{"obstinado", "obstin"},
		{"obstine", "obstin"},
		{"obstrucción", "obstruccion"},
		{"obtención", "obtencion"},
		{"obtendrá", "obtendr"},
		{"obtendrán", "obtendran"},
		{"obtener", "obten"},
		{"obtenerla", "obten"},
		{"obtenerlas", "obten"},
		{"obtenerse", "obten"},
		{"obtenga", "obteng"},
		{"obtengan", "obteng"},
		{"obtengo", "obteng"},
		{"obtenía", "obten"},
		{"obtenida", "obten"},
		{"obtenidas", "obten"},
		{"obtenido", "obten"},
		{"obtenidos", "obten"},
		{"obteniendo", "obten"},
		{"obtiene", "obtien"},
		{"obtienen", "obtien"},
		{"obtuviera", "obtuv"},
		{"obtuvieron", "obtuv"},
		{"obtuvimos", "obtuv"},
		{"obtuvo", "obtuv"},
		{"obuses", "obus"},
		{"obvia", "obvi"},
		{"obviamente", "obvi"},
		{"obvio", "obvi"},
		{"oca", "oca"},
		{"ocampo", "ocamp"},
		{"ocasión", "ocasion"},
		{"ocasiona", "ocasion"},
		{"ocasionadas", "ocasion"},
		{"ocasionado", "ocasion"},
		{"ocasionados", "ocasion"},
		{"ocasionalmente", "ocasional"},
		{"ocasionar", "ocasion"},
		{"ocasionará", "ocasion"},
		{"ocasionarle", "ocasion"},
		{"ocasionaron", "ocasion"},
		{"ocasionarse", "ocasion"},
		{"ocasiones", "ocasion"},
		{"ocasionó", "ocasion"},
		{"occidental", "occidental"},
		{"occidentales", "occidental"},
		{"occidente", "occident"},
		{"occipital", "occipital"},
		{"occisa", "occis"},
		{"occiso", "occis"},
		{"occisos", "occis"},
		{"oceanía", "ocean"},
		{"océanica", "ocean"},
		{"oceánico", "ocean"},
		{"oceano", "ocean"},
		{"océano", "ocean"},
		{"océanos", "ocean"},
		{"ocejo", "ocej"},
		{"oceransky", "oceransky"},
		{"ochenta", "ochent"},
		{"ochentas", "ochent"},
		{"ocho", "ocho"},
		{"ochoa", "ocho"},
		{"ochos", "ochos"},
		{"ochosi", "ochosi"},
		{"ocio", "oci"},
		{"ocremente", "ocrement"},
		{"octava", "octav"},
		{"octavio", "octavi"},
		{"octavo", "octav"},
		{"octubre", "octubr"},
		{"oculta", "ocult"},
		{"ocultaba", "ocult"},
		{"ocultar", "ocult"},
		{"ocultaron", "ocult"},
		{"ocultas", "ocult"},
		{"oculten", "ocult"},
		{"oculto", "ocult"},
		{"ocultó", "ocult"},
		{"ocultos", "ocult"},
		{"ocupa", "ocup"},
		{"ocupaba", "ocup"},
		{"ocupaban", "ocup"},
		{"ocupación", "ocup"},
		{"ocupacional", "ocupacional"},
		{"ocupada", "ocup"},
		{"ocupado", "ocup"},
		{"ocupados", "ocup"},
		{"ocupan", "ocup"},
		{"ocupantes", "ocup"},
		{"ocupar", "ocup"},
		{"ocuparon", "ocup"},
		{"ocuparse", "ocup"},
		{"ocupen", "ocup"},
		{"ocupó", "ocup"},
		{"ocurra", "ocurr"},
		{"ocurran", "ocurr"},
		{"ocurre", "ocurr"},
		{"ocurren", "ocurr"},
		{"ocurrencia", "ocurrent"},
		{"ocurría", "ocurr"},
		{"ocurrida", "ocurr"},
		{"ocurridas", "ocurr"},
		{"ocurrido", "ocurr"},
		{"ocurriendo", "ocurr"},
		{"ocurrieron", "ocurr"},
		{"ocurrió", "ocurr"},
		{"ocurrir", "ocurr"},
		{"ocurrirán", "ocurr"},
		{"odio", "odi"},
		{"odisea", "odise"},
		{"odriozola", "odriozol"},
		{"oea", "oea"},
		{"oecd", "oecd"},
		{"oeste", "oest"},
		{"of", "of"},
		{"ofelia", "ofeli"},
		{"ofende", "ofend"},
		{"ofenden", "ofend"},
		{"ofender", "ofend"},
		{"ofenderá", "ofend"},
		{"ofendidas", "ofend"},
		{"ofendido", "ofend"},
		{"ofendió", "ofend"},
		{"ofensa", "ofens"},
		{"ofensiva", "ofens"},
		{"ofensivas", "ofens"},
		{"ofensivo", "ofens"},
		{"ofensivos", "ofens"},
		{"oferentes", "oferent"},
		{"oferta", "ofert"},
		{"ofertas", "ofert"},
		{"ofertó", "ofert"},
		{"officio", "offici"},
		{"oficial", "oficial"},
		{"oficiales", "oficial"},
		{"oficialía", "oficial"},
		{"oficialista", "oficial"},
		{"oficializar", "oficializ"},
		{"oficializo", "oficializ"},
		{"oficializó", "oficializ"},
		{"oficialmente", "oficial"},
		{"oficiaría", "ofici"},
		{"oficina", "oficin"},
		{"oficinas", "oficin"},
		{"oficinesco", "oficinesc"},
		{"oficio", "ofici"},
		{"oficiosas", "ofici"},
		{"ofrece", "ofrec"},
		{"ofrecen", "ofrec"},
		{"ofrecer", "ofrec"},
		{"ofrecerá", "ofrec"},
		{"ofrecerán", "ofrec"},
		{"ofreceré", "ofrec"},
		{"ofrecería", "ofrec"},
		{"ofrecerla", "ofrec"},
		{"ofrecerle", "ofrec"},
		{"ofrecerles", "ofrec"},
		{"ofrecerlo", "ofrec"},
		{"ofrecía", "ofrec"},
		{"ofrecían", "ofrec"},
		{"ofrecida", "ofrec"},
		{"ofrecidas", "ofrec"},
		{"ofrecido", "ofrec"},
		{"ofrecidos", "ofrec"},
		{"ofreciendo", "ofrec"},
		{"ofreciera", "ofrec"},
		{"ofrecieran", "ofrec"},
		{"ofrecieron", "ofrec"},
		{"ofrecimiento", "ofrec"},
		{"ofrecimientos", "ofrec"},
		{"ofreció", "ofrec"},
		{"ofrendas", "ofrend"},
		{"ofrezca", "ofrezc"},
		{"ofrezcan", "ofrezc"},
		{"ofrezco", "ofrezc"},
		{"oftalmología", "oftalmolog"},
		{"oh", "oh"},
		{"ohio", "ohi"},
		{"oído", "oid"},
		{"oídos", "oid"},
		{"oiga", "oig"},
		{"oigame", "oigam"},
		{"oímos", "oim"},
		{"oír", "oir"},
		{"ojalá", "ojal"},
		{"ojales", "ojal"},
		{"ojeda", "ojed"},
		{"ojeras", "ojer"},
		{"ojo", "ojo"},
		{"ojos", "ojos"},
		{"ojuelos", "ojuel"},
		{"oklahoma", "oklahom"},
		{"ola", "ola"},
		{"olacyr", "olacyr"},
		{"olas", "olas"},
		{"olaya", "olay"},
		{"olé", "ole"},
		{"olea", "ole"},
		{"oleaginoso", "oleagin"},
		{"oleaginosos", "oleagin"},
		{"oleaje", "oleaj"},
		{"óleo", "ole"},
		{"olga", "olga"},
		{"oligopolio", "oligopoli"},
		{"olimpiadas", "olimpi"},
		{"olímpica", "olimp"},
		{"olímpico", "olimp"},
		{"olímpicos", "olimp"},
		{"olimpo", "olimp"},
		{"olivares", "olivar"},
		{"olivas", "oliv"},
		{"oliver", "oliv"},
		{"olivera", "oliver"},
		{"oliveras", "oliver"},
		{"oliverio", "oliveri"},
		{"olivia", "olivi"},
		{"olivier", "olivi"},
		{"olivo", "oliv"},
		{"ollas", "ollas"},
		{"olloqui", "olloqui"},
		{"olmedo", "olmed"},
		{"olor", "olor"},
		{"olores", "olor"},
		{"olvera", "olver"},
		{"olvida", "olvid"},
		{"olvidada", "olvid"},
		{"olvidadizo", "olvidadiz"},
		{"olvidado", "olvid"},
		{"olvidamos", "olvid"},
		{"olvidan", "olvid"},
		{"olvidando", "olvid"},
		{"olvidándose", "olvid"},
		{"olvidar", "olvid"},
		{"olvidaron", "olvid"},
		{"olvídese", "olvides"},
		{"olvido", "olvid"},
		{"olvidó", "olvid"},
		{"omán", "oman"},
		{"omaña", "omañ"},
		{"omar", "omar"},
		{"omb", "omb"},
		{"ominoso", "omin"},
		{"omiso", "omis"},
		{"omiten", "omit"},
		{"omitir", "omit"},
		{"omito", "omit"},
		{"ómnibus", "omnibus"},
		{"omnimax", "omnimax"},
		{"oñate", "oñat"},
		{"once", "once"},
		{"onceavo", "onceav"},
		{"oncena", "oncen"},
		{"onda", "onda"},
		{"ondarza", "ondarz"},
		{"ondas", "ondas"},
		{"one", "one"},
		{"onix", "onix"},
		{"ontario", "ontari"},
		{"ontiveros", "ontiver"},
		{"onu", "onu"},
		{"onzas", "onzas"},
		{"opa", "opa"},
		{"ópalos", "opal"},
		{"opción", "opcion"},
		{"opcionado", "opcion"},
		{"opcional", "opcional"},
		{"opciones", "opcion"},
		{"opera", "oper"},
		{"ópera", "oper"},
		{"operaba", "oper"},
		{"operación", "oper"},
		{"operacionales", "operacional"},
		{"operaciones", "oper"},
		{"operada", "oper"},
		{"operadas", "oper"},
		{"operado", "oper"},
		{"operador", "oper"},
		{"operadora", "oper"},
		{"operadoras", "oper"},
		{"operadores", "oper"},
		{"operados", "oper"},
		{"operan", "oper"},
		{"operando", "oper"},
		{"operar", "oper"},
		{"operará", "oper"},
		{"operaría", "oper"},
		{"operarios", "operari"},
		{"operas", "oper"},
		{"operation", "operation"},
		{"operativa", "oper"},
		{"operativas", "oper"},
		{"operativo", "oper"},
		{"operativos", "oper"},
		{"operen", "oper"},
		{"opina", "opin"},
		{"opinaban", "opin"},
		{"opinar", "opin"},
		{"opinas", "opin"},
		{"opinen", "opin"},
		{"opinión", "opinion"},
		{"opiniones", "opinion"},
		{"opiniónfinanciera", "opinionfinanc"},
		{"opino", "opin"},
		{"opinó", "opin"},
		{"opondrá", "opondr"},
		{"opone", "opon"},
		{"oponen", "opon"},
		{"oponentes", "oponent"},
		{"oponga", "opong"},
		{"opongo", "opong"},
		{"oponíamos", "opon"},
		{"oportuna", "oportun"},
		{"oportunamente", "oportun"},
		{"oportunidad", "oportun"},
		{"oportunidades", "oportun"},
		{"oportuno", "oportun"},
		{"oposición", "oposicion"},
		{"opositor", "opositor"},
		{"opositores", "opositor"},
		{"oprimir", "oprim"},
		{"opta", "opta"},
		{"optaba", "optab"},
		{"optar", "optar"},
		{"optarse", "optars"},
		{"optativo", "optat"},
		{"óptica", "optic"},
		{"ópticas", "optic"},
		{"óptico", "optic"},
		{"optimismo", "optim"},
		{"optimista", "optim"},
		{"optimistas", "optim"},
		{"optimización", "optimiz"},
		{"optimizar", "optimiz"},
		{"óptimo", "optim"},
		{"optó", "opto"},
		{"opuesta", "opuest"},
		{"opuesto", "opuest"},
		{"opuestos", "opuest"},
		{"opusieron", "opus"},
		{"or", "or"},
		{"oraciones", "oracion"},
		{"orador", "orador"},
		{"oral", "oral"},
		{"oralia", "orali"},
		{"oralmente", "oral"},
		{"oramos", "oram"},
		{"orangután", "orangutan"},
		{"orangutanes", "orangutan"},
		{"oraría", "orar"},
		{"orbe", "orbe"},
		{"órbita", "orbit"},
		{"orbitar", "orbit"},
		{"orco", "orco"},
		{"ordás", "ordas"},
		{"ordaz", "ordaz"},
		{"orden", "orden"},
		{"ordena", "orden"},
		{"ordenaban", "orden"},
		{"ordenada", "orden"},
		{"ordenadas", "orden"},
		{"ordenado", "orden"},
		{"ordenamos", "orden"},
		{"ordenar", "orden"},
		{"ordeñarás", "ordeñ"},
		{"ordenaron", "orden"},
		{"ordene", "orden"},
		{"ordenen", "orden"},
		{"ordenes", "orden"},
		{"órdenes", "orden"},
		{"ordenó", "orden"},
		{"ordiales", "ordial"},
		{"ordinaria", "ordinari"},
		{"ordinario", "ordinari"},
		{"ordóñez", "ordoñez"},
		{"orduño", "orduñ"},
		{"orea", "ore"},
		{"oregón", "oregon"},
		{"oreja", "orej"},
		{"orejas", "orej"},
		{"orejeras", "orejer"},
		{"orfebres", "orfebr"},
		{"orgánicas", "organ"},
		{"organice", "organic"},
		{"organismo", "organ"},
		{"organismos", "organ"},
		{"organiza", "organiz"},
		{"organización", "organiz"},
		{"organizacional", "organizacional"},
		{"organizaciones", "organiz"},
		{"organizada", "organiz"},
		{"organizadas", "organiz"},
		{"organizado", "organiz"},
		{"organizador", "organiz"},
		{"organizadora", "organiz"},
		{"organizadoras", "organiz"},
		{"organizadores", "organiz"},
		{"organizados", "organiz"},
		{"organizamos", "organiz"},
		{"organizan", "organiz"},
		{"organizando", "organiz"},
		{"organizar", "organiz"},
		{"organizará", "organiz"},
		{"organizaría", "organiz"},
		{"organizaron", "organiz"},
		{"organizarse", "organiz"},
		{"organizativa", "organiz"},
		{"organizó", "organiz"},
		{"órgano", "organ"},
		{"órganos", "organ"},
		{"orgásmico", "orgasm"},
		{"orgasmo", "orgasm"},
		{"orgía", "orgi"},
		{"orgullo", "orgull"},
		{"orgullosa", "orgull"},
		{"orgullosamente", "orgull"},
		{"orgulloso", "orgull"},
		{"orgullosos", "orgull"},
		{"oriana", "orian"},
		{"orienta", "orient"},
		{"orientació", "orientac"},
		{"orientación", "orient"},
		{"orientada", "orient"},
		{"orientadas", "orient"},
		{"orientado", "orient"},
		{"orientadoras", "orient"},
		{"orientadores", "orient"},
		{"orientados", "orient"},
		{"oriental", "oriental"},
		{"orientales", "oriental"},
		{"orientan", "orient"},
		{"orientando", "orient"},
		{"orientar", "orient"},
		{"orientarte", "orientart"},
		{"oriente", "orient"},
		{"origen", "orig"},
		{"orígenes", "origen"},
		{"origina", "origin"},
		{"originada", "origin"},
		{"originadas", "origin"},
		{"originado", "origin"},
		{"originados", "origin"},
		{"original", "original"},
		{"originales", "original"},
		{"originalidad", "original"},
		{"originalmente", "original"},
		{"originan", "origin"},
		{"originando", "origin"},
		{"originar", "origin"},
		{"originará", "origin"},
		{"originario", "originari"},
		{"originarios", "originari"},
		{"originarla", "origin"},
		{"originaron", "origin"},
		{"origine", "origin"},
		{"originó", "origin"},
		{"orilla", "orill"},
		{"orillando", "orill"},
		{"orillar", "orill"},
		{"orillas", "orill"},
		{"orilló", "orill"},
		{"orina", "orin"},
		{"orinar", "orin"},
		{"orioles", "oriol"},
		{"orishas", "orish"},
		{"orizaba", "oriz"},
		{"orlando", "orland"},
		{"ornelas", "ornel"},
		{"ornellas", "ornell"},
		{"oro", "oro"},
		{"oroinia", "oroini"},
		{"oropel", "oropel"},
		{"oropeza", "oropez"},
		{"orozco", "orozc"},
		{"orozquiana", "orozquian"},
		{"orquesta", "orquest"},
		{"orquestas", "orquest"},
		{"orta", "orta"},
		{"ortega", "orteg"},
		{"ortigoza", "ortigoz"},
		{"ortiz", "ortiz"},
		{"ortíz", "ortiz"},
		{"ortodoxa", "ortodox"},
		{"ortodoxas", "ortodox"},
		{"ortodoxia", "ortodoxi"},
		{"ortografía", "ortograf"},
		{"ortopedia", "ortopedi"},
		{"orven", "orven"},
		{"orwell", "orwell"},
		{"os", "os"},
		{"osadía", "osad"},
		{"osados", "osad"},
		{"osaka", "osak"},
		{"osasuna", "osasun"},
		{"oscar", "oscar"},
		{"oscila", "oscil"},
		{"oscilaba", "oscil"},
		{"oscilaciones", "oscil"},
		{"oscilan", "oscil"},
		{"oscura", "oscur"},
		{"oscuras", "oscur"},
		{"oscurecer", "oscurec"},
		{"oscuridad", "oscur"},
		{"oscuro", "oscur"},
		{"oscuros", "oscur"},
		{"osorio", "osori"},
		{"osos", "osos"},
		{"osservatore", "osservator"},
		{"ostensible", "ostens"},
		{"ostensiblemente", "ostens"},
		{"ostenta", "ostent"},
		{"ostentar", "ostent"},
		{"osuna", "osun"},
		{"osvaldo", "osvald"},
		{"oswald", "oswald"},
		{"oswaldo", "oswald"},
		{"otáez", "otaez"},
		{"otan", "otan"},
		{"ote", "ote"},
		{"oteando", "ote"},
		{"othon", "othon"},
		{"othón", "othon"},
		{"otila", "otil"},
		{"otilia", "otili"},
		{"otitis", "otitis"},
		{"otoño", "otoñ"},
		{"otorga", "otorg"},
		{"otorgada", "otorg"},
		{"otorgadas", "otorg"},
		{"otorgado", "otorg"},
		{"otorgados", "otorg"},
		{"otorgamiento", "otorg"},
		{"otorgan", "otorg"},
		{"otorgando", "otorg"},
		{"otorgar", "otorg"},
		{"otorgará", "otorg"},
		{"otorgaran", "otorg"},
		{"otorgarán", "otorg"},
		{"otorgarle", "otorg"},
		{"otorgarles", "otorg"},
		{"otorgarnos", "otorg"},
		{"otorgaron", "otorg"},
		{"otorgó", "otorg"},
		{"otorguen", "otorg"},
		{"otra", "otra"},
		{"otras", "otras"},
		{"otro", "otro"},
		{"otrora", "otror"},
		{"otros", "otros"},
		{"ottawa", "ottaw"},
		{"ottey", "ottey"},
		{"otto", "otto"},
		{"out", "out"},
		{"outs", "outs"},
		{"ovación", "ovacion"},
		{"ovacionada", "ovacion"},
		{"ovacionado", "ovacion"},
		{"ovacionados", "ovacion"},
		{"ovacionan", "ovacion"},
		{"ovacionarlo", "ovacion"},
		{"ovaciones", "ovacion"},
		{"ovacionó", "ovacion"},
		{"oval", "oval"},
		{"ovalle", "ovall"},
		{"ovejas", "ovej"},
		{"over", "over"},
		{"ovidio", "ovidi"},
		{"oviedo", "ovied"},
		{"ovoide", "ovoid"},
		{"ovulación", "ovul"},
		{"owen", "owen"},
		{"oxford", "oxford"},
		{"oxida", "oxid"},
		{"oxidando", "oxid"},
		{"oxidar", "oxid"},
		{"óxidos", "oxid"},
		{"oxígeno", "oxigen"},
		{"oxxo", "oxxo"},
		{"oyen", "oyen"},
		{"oyentes", "oyent"},
		{"oyeran", "oyer"},
		{"oyó", "oyo"},
		{"oziel", "oziel"},
		{"ozono", "ozon"},
		{"p", "p"},
		{"pa", "pa"},
		{"pabellón", "pabellon"},
		{"pabilos", "pabil"},
		{"pablo", "pabl"},
		{"pacheco", "pachec"},
		{"pacho", "pach"},
		{"pachoneos", "pachone"},
		{"pachuca", "pachuc"},
		{"paciencia", "pacienci"},
		{"paciente", "pacient"},
		{"pacientes", "pacient"},
		{"pacífica", "pacif"},
		{"pacifico", "pacif"},
		{"pacífico", "pacif"},
		{"packaged", "packag"},
		{"packard", "packard"},
		{"paco", "pac"},
		{"pactada", "pact"},
		{"pactarán", "pact"},
		{"pacto", "pact"},
		{"pactó", "pact"},
		{"padece", "padec"},
		{"padecemos", "padec"},
		{"padecen", "padec"},
		{"padecer", "padec"},
		{"padeciendo", "padec"},
		{"padecieron", "padec"},
		{"padecimiento", "padec"},
		{"padecimientos", "padec"},
		{"padilla", "padill"},
		{"padre", "padr"},
		{"padrecito", "padrecit"},
		{"padres", "padr"},
		{"padrino", "padrin"},
		{"padró", "padr"},
		{"padrón", "padron"},
		{"padrones", "padron"},
		{"páez", "paez"},
		{"pag", "pag"},
		{"pág", "pag"},
		{"paga", "pag"},
		{"pagaba", "pag"},
		{"pagadas", "pag"},
		{"pagado", "pag"},
		{"pagados", "pag"},
		{"pagamos", "pag"},
		{"pagan", "pag"},
		{"pagando", "pag"},
		{"paganismo", "pagan"},
		{"pagano", "pagan"},
		{"paganos", "pagan"},
		{"pagar", "pag"},
		{"pagara", "pag"},
		{"pagará", "pag"},
		{"pagaran", "pag"},
		{"pagarán", "pag"},
		{"pagarían", "pag"},
		{"pagarla", "pag"},
		{"pagarle", "pag"},
		{"pagarnos", "pag"},
		{"pagaron", "pag"},
		{"pagarte", "pagart"},
		{"página", "pagin"},
		{"páginas", "pagin"},
		{"pago", "pag"},
		{"pagó", "pag"},
		{"pagos", "pag"},
		{"pague", "pag"},
		{"paguen", "pag"},
		{"país", "pais"},
		{"paisaje", "paisaj"},
		{"paisajes", "paisaj"},
		{"paisajista", "paisaj"},
		{"paisajístico", "paisajist"},
		{"paises", "pais"},
		{"países", "pais"},
		{"paja", "paj"},
		{"pájaro", "pajar"},
		{"pájaros", "pajar"},
		{"pala", "pal"},
		{"palabra", "palabr"},
		{"palabras", "palabr"},
		{"palabrería", "palabr"},
		{"palacetes", "palacet"},
		{"palacio", "palaci"},
		{"palacios", "palaci"},
		{"paladar", "palad"},
		{"palanca", "palanc"},
		{"palapa", "palap"},
		{"palas", "pal"},
		{"palaú", "palau"},
		{"palazuelos", "palazuel"},
		{"palco", "palc"},
		{"palcos", "palc"},
		{"pale", "pal"},
		{"palencia", "palenci"},
		{"palenque", "palenqu"},
		{"palenques", "palenqu"},
		{"paleros", "paler"},
		{"palestino", "palestin"},
		{"palestinos", "palestin"},
		{"palestra", "palestr"},
		{"paletones", "paleton"},
		{"pálido", "pal"},
		{"palillo", "palill"},
		{"paliza", "paliz"},
		{"palm", "palm"},
		{"palma", "palm"},
		{"palmas", "palm"},
		{"palmeras", "palmer"},
		{"palmitos", "palmit"},
		{"palo", "pal"},
		{"paloma", "palom"},
		{"palomas", "palom"},
		{"palomilla", "palomill"},
		{"palomitas", "palomit"},
		{"palomo", "palom"},
		{"palos", "pal"},
		{"palou", "palou"},
		{"palpable", "palpabl"},
		{"pam", "pam"},
		{"pamela", "pamel"},
		{"pampas", "pamp"},
		{"pamplona", "pamplon"},
		{"pan", "pan"},
		{"pana", "pan"},
		{"panacea", "panace"},
		{"panadería", "panad"},
		{"pañales", "pañal"},
		{"panamá", "panam"},
		{"panameño", "panameñ"},
		{"panamericana", "panamerican"},
		{"panamericanos", "panamerican"},
		{"pancarta", "pancart"},
		{"pancartas", "pancart"},
		{"pancho", "panch"},
		{"pancracio", "pancraci"},
		{"pancrasio", "pancrasi"},
		{"pandilla", "pandill"},
		{"pandillas", "pandill"},
		{"pandilleril", "pandilleril"},
		{"pandillerismo", "pandiller"},
		{"pandillero", "pandiller"},
		{"pandilleros", "pandiller"},
		{"panecillos", "panecill"},
		{"panel", "panel"},
		{"paneles", "panel"},
		{"panes", "pan"},
		{"paniagua", "paniagu"},
		{"pánico", "panic"},
		{"panista", "panist"},
		{"panistas", "panist"},
		{"panorama", "panoram"},
		{"panorámicas", "panoram"},
		{"panorámicos", "panoram"},
		{"panqué", "panqu"},
		{"pantalla", "pantall"},
		{"pantallas", "pantall"},
		{"pantalón", "pantalon"},
		{"pantalones", "pantalon"},
		{"panteón", "panteon"},
		{"panteones", "panteon"},
		{"panteras", "panter"},
		{"pantorrilla", "pantorrill"},
		{"pantro", "pantr"},
		{"pants", "pants"},
		{"pañuelos", "pañuel"},
		{"panzas", "panz"},
		{"paola", "paol"},
		{"papa", "pap"},
		{"papá", "pap"},
		{"papal", "papal"},
		{"papalotla", "papalotl"},
		{"papás", "papas"},
		{"papasquiaroa", "papasquiaro"},
		{"papel", "papel"},
		{"papelería", "papel"},
		{"papelerías", "papel"},
		{"papeles", "papel"},
		{"papitas", "papit"},
		{"paquete", "paquet"},
		{"paquetería", "paquet"},
		{"paquetes", "paquet"},
		{"paquiro", "paquir"},
		{"par", "par"},
		{"para", "par"},
		{"paraba", "par"},
		{"parabólicas", "parabol"},
		{"parabrisas", "parabris"},
		{"paracaídas", "paracaid"},
		{"parada", "par"},
		{"paradas", "par"},
		{"parado", "par"},
		{"paradojas", "paradoj"},
		{"paradójicamente", "paradoj"},
		{"parados", "par"},
		{"paraestatal", "paraestatal"},
		{"parafinas", "parafin"},
		{"parafraseando", "parafras"},
		{"paráfrasis", "parafrasis"},
		{"paraguay", "paraguay"},
		{"paraguayo", "paraguay"},
		{"paraíso", "parais"},
		{"paraje", "paraj"},
		{"paralela", "paralel"},
		{"paralelamente", "paralel"},
		{"paralelas", "paralel"},
		{"paralelismo", "paralel"},
		{"paralelo", "paralel"},
		{"paralelos", "paralel"},
		{"parálisis", "paralisis"},
		{"paralítico", "paralit"},
		{"paralizan", "paraliz"},
		{"paralizando", "paraliz"},
		{"paralizante", "paraliz"},
		{"paralizar", "paraliz"},
		{"paralizaron", "paraliz"},
		{"paralizó", "paraliz"},
		{"parámetro", "parametr"},
		{"parámetros", "parametr"},
		{"paran", "par"},
		{"parando", "par"},
		{"paraninfo", "paraninf"},
		{"parar", "par"},
		{"parara", "par"},
		{"pararme", "par"},
		{"pararse", "par"},
		{"parás", "paras"},
		{"parcial", "parcial"},
		{"parciales", "parcial"},
		{"parcialidad", "parcial"},
		{"parcialmente", "parcial"},
		{"parece", "parec"},
		{"parecemos", "parec"},
		{"parecen", "parec"},
		{"parecer", "parec"},
		{"parecería", "parec"},
		{"parecerían", "parec"},
		{"parecerse", "parec"},
		{"parecía", "parec"},
		{"parecían", "parec"},
		{"parecida", "parec"},
		{"parecidas", "parec"},
		{"parecido", "parec"},
		{"pareciendo", "parec"},
		{"parecieron", "parec"},
		{"pareciese", "parec"},
		{"pareció", "parec"},
		{"pared", "par"},
		{"paredes", "pared"},
		{"pareja", "parej"},
		{"parejas", "parej"},
		{"parejos", "parej"},
		{"paremos", "par"},
		{"paréntesis", "parentesis"},
		{"pares", "par"},
		{"parezca", "parezc"},
		{"parga", "parg"},
		{"pariente", "parient"},
		{"parientes", "parient"},
		{"parietal", "parietal"},
		{"paris", "paris"},
		{"parís", "par"},
		{"parisino", "parisin"},
		{"park", "park"},
		{"parkas", "park"},
		{"parker", "park"},
		{"parlamentaria", "parlamentari"},
		{"parlamentarias", "parlamentari"},
		{"parlamentario", "parlamentari"},
		{"parlamentarios", "parlamentari"},
		{"parlamento", "parlament"},
		{"parm", "parm"},
		{"parmista", "parmist"},
		{"paro", "par"},
		{"paró", "par"},
		{"parodiando", "parodi"},
		{"paros", "par"},
		{"párpados", "parp"},
		{"parque", "parqu"},
		{"parquedad", "parqued"},
		{"parques", "parqu"},
		{"parquimetrero", "parquimetrer"},
		{"parquímetro", "parquimetr"},
		{"parquímetros", "parquimetr"},
		{"parra", "parr"},
		{"párrafo", "parraf"},
		{"parranda", "parrand"},
		{"parricidas", "parric"},
		{"parricidio", "parricidi"},
		{"parrilla", "parrill"},
		{"párroco", "parroc"},
		{"parroquia", "parroqui"},
		{"parroquiales", "parroquial"},
		{"parroquiano", "parroquian"},
		{"parroquias", "parroqui"},
		{"parsons", "parsons"},
		{"parte", "part"},
		{"parteaguas", "parteagu"},
		{"parten", "part"},
		{"partes", "part"},
		{"participa", "particip"},
		{"participaba", "particip"},
		{"participaban", "particip"},
		{"participación", "particip"},
		{"participaciones", "particip"},
		{"participado", "particip"},
		{"participamos", "particip"},
		{"participan", "particip"},
		{"participando", "particip"},
		{"participante", "particip"},
		{"participantes", "particip"},
		{"participar", "particip"},
		{"participara", "particip"},
		{"participará", "particip"},
		{"participarán", "particip"},
		{"participarían", "particip"},
		{"participaron", "particip"},
		{"participativos", "particip"},
		{"participe", "particip"},
		{"participen", "particip"},
		{"partícipes", "particip"},
		{"participó", "particip"},
		{"particular", "particul"},
		{"particulares", "particular"},
		{"particularmente", "particular"},
		{"partículas", "particul"},
		{"partida", "part"},
		{"partidaria", "partidari"},
		{"partidario", "partidari"},
		{"partidarios", "partidari"},
		{"partidas", "part"},
		{"partidazo", "partidaz"},
		{"partidero", "partider"},
		{"partidistas", "partid"},
		{"partido", "part"},
		{"partidos", "part"},
		{"partiendo", "part"},
		{"partiera", "part"},
		{"partieron", "part"},
		{"partió", "part"},
		{"partir", "part"},
		{"partirá", "part"},
		{"partirán", "part"},
		{"parto", "part"},
		{"partos", "part"},
		{"parturienta", "parturient"},
		{"parvada", "parv"},
		{"pasa", "pas"},
		{"pasaba", "pas"},
		{"pasaban", "pas"},
		{"pasacassettes", "pasacassett"},
		{"pasada", "pas"},
		{"pasadas", "pas"},
		{"pasado", "pas"},
		{"pasadores", "pasador"},
		{"pasados", "pas"},
		{"pasaje", "pasaj"},
		{"pasajera", "pasajer"},
		{"pasajeros", "pasajer"},
		{"pasajes", "pasaj"},
		{"pasamayo", "pasamay"},
		{"pasamos", "pas"},
		{"pasan", "pas"},
		{"pasando", "pas"},
		{"pasante", "pasant"},
		{"pasaporte", "pasaport"},
		{"pasar", "pas"},
		{"pasara", "pas"},
		{"pasará", "pas"},
		{"pasarán", "pas"},
		{"pasarela", "pasarel"},
		{"pasarelas", "pasarel"},
		{"pasaría", "pas"},
		{"pasarían", "pas"},
		{"pasarle", "pas"},
		{"pasarles", "pas"},
		{"pasaron", "pas"},
		{"pasarse", "pas"},
		{"pasatiempo", "pasatiemp"},
		{"pascal", "pascal"},
		{"pasco", "pasc"},
		{"pascua", "pascu"},
		{"pascual", "pascual"},
		{"pascuense", "pascuens"},
		{"pase", "pas"},
		{"pasé", "pas"},
		{"pasean", "pas"},
		{"pasear", "pas"},
		{"pasearon", "pas"},
		{"pásele", "pasel"},
		{"pasen", "pas"},
		{"paseño", "paseñ"},
		{"paseo", "pase"},
		{"paseó", "pase"},
		{"paseos", "pase"},
		{"pases", "pas"},
		{"pasillo", "pasill"},
		{"pasillos", "pasill"},
		{"pasión", "pasion"},
		{"pasional", "pasional"},
		{"pasiones", "pasion"},
		{"pasiva", "pasiv"},
		{"pasivo", "pasiv"},
		{"pasivos", "pasiv"},
		{"paso", "pas"},
		{"pasó", "pas"},
		{"pasos", "pas"},
		{"pasquinal", "pasquinal"},
		{"passarella", "passarell"},
		{"pasta", "past"},
		{"pastillas", "pastill"},
		{"pasto", "past"},
		{"pastor", "pastor"},
		{"pastora", "pastor"},
		{"pastoral", "pastoral"},
		{"pastoriza", "pastoriz"},
		{"pata", "pat"},
		{"patadas", "pat"},
		{"patas", "pat"},
		{"pateadores", "pateador"},
		{"patearon", "pat"},
		{"patentara", "patent"},
		{"patentarlo", "patent"},
		{"patente", "patent"},
		{"patentes", "patent"},
		{"paternal", "paternal"},
		{"paternalismo", "paternal"},
		{"paternalista", "paternal"},
		{"paternalistas", "paternal"},
		{"paternos", "pat"},
		{"patéticamente", "patet"},
		{"patéticos", "patet"},
		{"path", "path"},
		{"patinamos", "patin"},
		{"patiñazo", "patiñaz"},
		{"patiño", "patiñ"},
		{"patio", "pati"},
		{"patito", "patit"},
		{"pato", "pat"},
		{"patológico", "patolog"},
		{"patos", "pat"},
		{"patria", "patri"},
		{"patriarca", "patriarc"},
		{"patriarcas", "patriarc"},
		{"patricia", "patrici"},
		{"patricio", "patrici"},
		{"patricios", "patrici"},
		{"patrick", "patrick"},
		{"patrimonial", "patrimonial"},
		{"patrimoniales", "patrimonial"},
		{"patrimonio", "patrimoni"},
		{"patrio", "patri"},
		{"patrios", "patri"},
		{"patriot", "patriot"},
		{"patrocinada", "patrocin"},
		{"patrocinado", "patrocin"},
		{"patrocinador", "patrocin"},
		{"patrocinadores", "patrocin"},
		{"patrocinio", "patrocini"},
		{"patrón", "patron"},
		{"patrona", "patron"},
		{"patronal", "patronal"},
		{"patronales", "patronal"},
		{"patronato", "patronat"},
		{"patrones", "patron"},
		{"patrulla", "patrull"},
		{"patrullas", "patrull"},
		{"patrulleros", "patruller"},
		{"patty", "patty"},
		{"paty", "paty"},
		{"paul", "paul"},
		{"paúl", "paul"},
		{"paulatina", "paulatin"},
		{"paulatinamente", "paulatin"},
		{"paulatino", "paulatin"},
		{"paulina", "paulin"},
		{"paulino", "paulin"},
		{"paullada", "paull"},
		{"paulo", "paul"},
		{"pausa", "paus"},
		{"pausada", "paus"},
		{"pausadamente", "pausad"},
		{"pausas", "paus"},
		{"pauta", "paut"},
		{"pavimentación", "paviment"},
		{"pavimentada", "paviment"},
		{"pavimentado", "paviment"},
		{"pavimentados", "paviment"},
		{"pavimentar", "paviment"},
		{"pavimento", "paviment"},
		{"pavor", "pavor"},
		{"payán", "payan"},
		{"payasadas", "payas"},
		{"paz", "paz"},
		{"pc", "pc"},
		{"pclab", "pclab"},
		{"pdcc", "pdcc"},
		{"pdm", "pdm"},
		{"peabody", "peabody"},
		{"peaje", "peaj"},
		{"pearl", "pearl"},
		{"peatones", "peaton"},
		{"pecado", "pec"},
		{"pecador", "pecador"},
		{"pecadores", "pecador"},
		{"pecaminosa", "pecamin"},
		{"pecará", "pec"},
		{"pece", "pec"},
		{"peces", "pec"},
		{"pechera", "pecher"},
		{"pecho", "pech"},
		{"pecina", "pecin"},
		{"pectoral", "pectoral"},
		{"pectorales", "pectoral"},
		{"peculiar", "peculi"},
		{"peculiares", "peculiar"},
		{"peculiaridad", "peculiar"},
		{"peculiaridades", "peculiar"},
		{"pedagogía", "pedagog"},
		{"pedagógica", "pedagog"},
		{"pedagógicas", "pedagog"},
		{"pedagógicos", "pedagog"},
		{"pedalista", "pedal"},
		{"pedalistas", "pedal"},
		{"pedazos", "pedaz"},
		{"pedía", "ped"},
		{"pedían", "ped"},
		{"pediatra", "pediatr"},
		{"pediatría", "pediatr"},
		{"pediátrica", "pediatr"},
		{"pedida", "ped"},
		{"pedido", "ped"},
		{"pedidores", "pedidor"},
		{"pedidos", "ped"},
		{"pedimento", "pediment"},
		{"pedimentos", "pediment"},
		{"pedimos", "ped"},
		{"pedir", "ped"},
		{"pedirá", "ped"},
		{"pedirán", "ped"},
		{"pediré", "ped"},
		{"pediría", "ped"},
		{"pedirle", "ped"},
		{"pedirles", "ped"},
		{"pedirlo", "ped"},
		{"pedomo", "pedom"},
		{"pedrada", "pedr"},
		{"pedradas", "pedr"},
		{"pedraza", "pedraz"},
		{"pedreras", "pedrer"},
		{"pedrería", "pedr"},
		{"pedrín", "pedrin"},
		{"pedro", "pedr"},
		{"pega", "peg"},
		{"pegada", "peg"},
		{"pegado", "peg"},
		{"pegamento", "pegament"},
		{"pegando", "peg"},
		{"pegar", "peg"},
		{"pegaran", "peg"},
		{"pegarle", "peg"},
		{"pegaron", "peg"},
		{"pegó", "peg"},
		{"pegote", "pegot"},
		{"pegotes", "pegot"},
		{"pegue", "peg"},
		{"peinado", "pein"},
		{"peinó", "pein"},
		{"pelado", "pel"},
		{"peláez", "pelaez"},
		{"pelar", "pel"},
		{"peldaños", "peldañ"},
		{"pelé", "pel"},
		{"pelea", "pele"},
		{"peleado", "pel"},
		{"pelean", "pel"},
		{"peleando", "pel"},
		{"pelear", "pel"},
		{"peleará", "pel"},
		{"pelearan", "pel"},
		{"pelearon", "pel"},
		{"peleas", "pel"},
		{"peleen", "pel"},
		{"peleles", "pelel"},
		{"peleo", "pele"},
		{"peleó", "pele"},
		{"pelicula", "pelicul"},
		{"película", "pelicul"},
		{"películas", "pelicul"},
		{"peligro", "peligr"},
		{"peligros", "peligr"},
		{"peligrosa", "peligr"},
		{"peligrosamente", "peligr"},
		{"peligrosas", "peligr"},
		{"peligroso", "peligr"},
		{"peligrosos", "peligr"},
		{"pelletier", "pelleti"},
		{"pello", "pell"},
		{"pelo", "pel"},
		{"pelos", "pel"},
		{"pelota", "pelot"},
		{"pelotero", "peloter"},
		{"peloteros", "peloter"},
		{"pelotón", "peloton"},
		{"peluche", "peluch"},
		{"peluquerías", "peluqu"},
		{"pemex", "pemex"},
		{"pena", "pen"},
		{"peña", "peñ"},
		{"penal", "penal"},
		{"penales", "penal"},
		{"penalidad", "penal"},
		{"penalmente", "penal"},
		{"penalty", "penalty"},
		{"penas", "pen"},
		{"peñascos", "peñasc"},
		{"pendiente", "pendient"},
		{"pendientes", "pendient"},
		{"pendleton", "pendleton"},
		{"penetracion", "penetracion"},
		{"penetración", "penetr"},
		{"penetrados", "penetr"},
		{"penetran", "penetr"},
		{"penetrante", "penetr"},
		{"penetrar", "penetr"},
		{"peniche", "penich"},
		{"península", "peninsul"},
		{"peninsulares", "peninsular"},
		{"penitenciarios", "penitenciari"},
		{"penn", "penn"},
		{"pennsylvania", "pennsylvani"},
		{"penoso", "penos"},
		{"pensaba", "pens"},
		{"pensabamos", "pensab"},
		{"pensaban", "pens"},
		{"pensado", "pens"},
		{"pensadores", "pensador"},
		{"pensamiento", "pensamient"},
		{"pensamos", "pens"},
		{"pensando", "pens"},
		{"pensantes", "pensant"},
		{"pensar", "pens"},
		{"pensarse", "pens"},
		{"pensativa", "pensat"},
		{"pensé", "pens"},
		{"pensemos", "pens"},
		{"pensilvania", "pensilvani"},
		{"pensión", "pension"},
		{"pensionados", "pension"},
		{"pensiones", "pension"},
		{"pensó", "pens"},
		{"pentágono", "pentagon"},
		{"pentathlón", "pentathlon"},
		{"pentecostés", "pentecostes"},
		{"pentotal", "pentotal"},
		{"penúltimo", "penultim"},
		{"penurias", "penuri"},
		{"peor", "peor"},
		{"peores", "peor"},
		{"pepe", "pep"},
		{"peppers", "peppers"},
		{"pequeña", "pequeñ"},
		{"pequeñas", "pequeñ"},
		{"pequeñísima", "pequeñisim"},
		{"pequeñísimo", "pequeñisim"},
		{"pequeñita", "pequeñit"},
		{"pequeño", "pequeñ"},
		{"pequeños", "pequeñ"},
		{"per", "per"},
		{"perales", "peral"},
		{"peraza", "peraz"},
		{"percance", "percanc"},
		{"percances", "percanc"},
		{"percata", "percat"},
		{"percatado", "percat"},
		{"percatarse", "percat"},
		{"percató", "percat"},
		{"percepción", "percepcion"},
		{"percepciones", "percepcion"},
		{"perciban", "percib"},
		{"percibe", "percib"},
		{"perciben", "percib"},
		{"percusiones", "percusion"},
		{"percusionistas", "percusion"},
		{"perdedores", "perdedor"},
		{"perdemos", "perd"},
		{"perder", "perd"},
		{"perderá", "perd"},
		{"perderán", "perd"},
		{"perderían", "perd"},
		{"perderla", "perd"},
		{"perderlo", "perd"},
		{"perdí", "perd"},
		{"perdición", "perdicion"},
		{"perdida", "perd"},
		{"pérdida", "perd"},
		{"perdidas", "perd"},
		{"pérdidas", "perd"},
		{"perdido", "perd"},
		{"perdidos", "perd"},
		{"perdiendo", "perd"},
		{"perdiera", "perd"},
		{"perdieran", "perd"},
		{"perdieron", "perd"},
		{"perdió", "perd"},
		{"perdomo", "perdom"},
		{"perdón", "perdon"},
		{"perdonarlos", "perdon"},
		{"perdones", "perdon"},
		{"perea", "pere"},
		{"perecieron", "perec"},
		{"pereda", "pered"},
		{"peregrina", "peregrin"},
		{"peregrinación", "peregrin"},
		{"peregrino", "peregrin"},
		{"perenne", "perenn"},
		{"pereyra", "pereyr"},
		{"perez", "perez"},
		{"pérez", "perez"},
		{"perfección", "perfeccion"},
		{"perfeccionamiento", "perfeccion"},
		{"perfeccionar", "perfeccion"},
		{"perfeccionarlo", "perfeccion"},
		{"perfeccione", "perfeccion"},
		{"perfectamente", "perfect"},
		{"perfecto", "perfect"},
		{"perfectos", "perfect"},
		{"perfidiana", "perfidian"},
		{"perfil", "perfil"},
		{"perfila", "perfil"},
		{"perfilaba", "perfil"},
		{"perfilan", "perfil"},
		{"perfiles", "perfil"},
		{"perfoliata", "perfoliat"},
		{"perforación", "perfor"},
		{"perforado", "perfor"},
		{"perforar", "perfor"},
		{"performance", "performanc"},
		{"pergaminos", "pergamin"},
		{"pericia", "perici"},
		{"periciales", "pericial"},
		{"pericos", "peric"},
		{"periférica", "perifer"},
		{"periféricas", "perifer"},
		{"periférico", "perifer"},
		{"periféricos", "perifer"},
		{"perigord", "perigord"},
		{"perillas", "perill"},
		{"perimetral", "perimetral"},
		{"periódicamente", "period"},
		{"periódicas", "period"},
		{"períodico", "period"},
		{"periódico", "period"},
		{"periódicos", "period"},
		{"periodismo", "period"},
		{"periodista", "period"},
		{"periodistas", "period"},
		{"periodística", "periodist"},
		{"periodísticas", "periodist"},
		{"periodístico", "periodist"},
		{"periodo", "period"},
		{"período", "period"},
		{"periodos", "period"},
		{"períodos", "period"},
		{"perjudica", "perjud"},
		{"perjudicada", "perjudic"},
		{"perjudicado", "perjudic"},
		{"perjudicados", "perjudic"},
		{"perjudican", "perjudic"},
		{"perjudicar", "perjudic"},
		{"perjudicará", "perjudic"},
		{"perjudicial", "perjudicial"},
		{"perjudicó", "perjudic"},
		{"perjuicio", "perjuici"},
		{"perjuicios", "perjuici"},
		{"perla", "perl"},
		{"perlas", "perl"},
		{"permanece", "permanec"},
		{"permanecen", "permanec"},
		{"permanecer", "permanec"},
		{"permanecerá", "permanec"},
		{"permanecerán", "permanec"},
		{"permaneceremos", "permanec"},
		{"permanecía", "permanec"},
		{"permanecían", "permanec"},
		{"permanecido", "permanec"},
		{"permaneciendo", "permanec"},
		{"permanecieran", "permanec"},
		{"permanecieron", "permanec"},
		{"permaneció", "permanec"},
		{"permanencia", "permanent"},
		{"permanente", "permanent"},
		{"permanentemente", "permanent"},
		{"permanentes", "permanent"},
		{"permanezca", "permanezc"},
		{"permanezcan", "permanezc"},
		{"permisibles", "permis"},
		{"permisionario", "permisionari"},
		{"permisionarios", "permisionari"},
		{"permiso", "permis"},
		{"permisos", "permis"},
		{"permita", "permit"},
		{"permítame", "permitam"},
		{"permitan", "permit"},
		{"permite", "permit"},
		{"permiten", "permit"},
		{"permitía", "permit"},
		{"permitida", "permit"},
		{"permitidas", "permit"},
		{"permitido", "permit"},
		{"permitiendo", "permit"},
		{"permitiéndole", "permit"},
		{"permitiera", "permit"},
		{"permitieron", "permit"},
		{"permitió", "permit"},
		{"permitir", "permit"},
		{"permitirá", "permit"},
		{"permitirán", "permit"},
		{"permitiré", "permit"},
		{"permitiría", "permit"},
		{"permitirían", "permit"},
		{"permitirles", "permit"},
		{"permitirse", "permit"},
		{"pernada", "pern"},
		{"perniciosas", "pernici"},
		{"pernicioso", "pernici"},
		{"pernoctó", "pernoct"},
		{"pero", "per"},
		{"perpetra", "perpetr"},
		{"perpetrada", "perpetr"},
		{"perpetrados", "perpetr"},
		{"perpetua", "perpetu"},
		{"perpetúan", "perpetu"},
		{"perpetuarse", "perpetu"},
		{"perplejidad", "perplej"},
		{"perplejo", "perplej"},
		{"perredista", "perred"},
		{"perredistas", "perred"},
		{"perro", "perr"},
		{"perros", "perr"},
		{"perry", "perry"},
		{"persecución", "persecu"},
		{"persecutoria", "persecutori"},
		{"persecutorias", "persecutori"},
		{"perseguida", "persegu"},
		{"perseguidor", "perseguidor"},
		{"perseguidores", "perseguidor"},
		{"perseguidos", "persegu"},
		{"perseguirlos", "persegu"},
		{"persiga", "persig"},
		{"persiste", "pers"},
		{"persisten", "persist"},
		{"persistencia", "persistent"},
		{"persistente", "persistent"},
		{"persistir", "persist"},
		{"persona", "person"},
		{"personaje", "personaj"},
		{"personajes", "personaj"},
		{"personal", "personal"},
		{"personales", "personal"},
		{"personalidad", "personal"},
		{"personalidades", "personal"},
		{"personalizar", "personaliz"},
		{"personalizarlo", "personaliz"},
		{"personalmente", "personal"},
		{"personas", "person"},
		{"personifica", "personif"},
		{"personificará", "personific"},
		{"perspectiva", "perspect"},
		{"perspectivas", "perspect"},
		{"perspicacia", "perspicaci"},
		{"persuadir", "persuad"},
		{"pertenece", "pertenec"},
		{"pertenecen", "pertenec"},
		{"pertenecer", "pertenec"},
		{"pertenecía", "pertenec"},
		{"perteneciendo", "pertenec"},
		{"perteneciente", "pertenecient"},
		{"pertenecientes", "pertenecient"},
		{"pertenecieron", "pertenec"},
		{"perteneció", "pertenec"},
		{"pertenencias", "pertenent"},
		{"pértiga", "pertig"},
		{"pertinaces", "pertinac"},
		{"pertinente", "pertinent"},
		{"pertinentes", "pertinent"},
		{"perturba", "perturb"},
		{"perturbación", "perturb"},
		{"perturbemos", "perturb"},
		{"perturbó", "perturb"},
		{"perú", "peru"},
		{"peruana", "peruan"},
		{"peruano", "peruan"},
		{"peruanos", "peruan"},
		{"perversas", "pervers"},
		{"perversión", "perversion"},
		{"perversiones", "perversion"},
		{"perversos", "pervers"},
		{"pervivencia", "pervivent"},
		{"pesa", "pes"},
		{"pesadas", "pes"},
		{"pesadez", "pesadez"},
		{"pesadilla", "pesadill"},
		{"pesado", "pes"},
		{"pesados", "pes"},
		{"pesaje", "pesaj"},
		{"pesar", "pes"},
		{"pesas", "pes"},
		{"pesca", "pesc"},
		{"pescadores", "pescador"},
		{"pescados", "pesc"},
		{"pese", "pes"},
		{"peseras", "peser"},
		{"peseta", "peset"},
		{"pesetas", "peset"},
		{"pésima", "pesim"},
		{"pesimista", "pesim"},
		{"pesimistas", "pesim"},
		{"pésimo", "pesim"},
		{"peso", "pes"},
		{"pesó", "pes"},
		{"pesos", "pes"},
		{"pespuntes", "pespunt"},
		{"pesqueira", "pesqueir"},
		{"pesquería", "pesqu"},
		{"pesqueros", "pesquer"},
		{"pesticida", "pestic"},
		{"pesticidas", "pestic"},
		{"petate", "petat"},
		{"pete", "pet"},
		{"peter", "pet"},
		{"petición", "peticion"},
		{"peticiones", "peticion"},
		{"petit", "petit"},
		{"peto", "pet"},
		{"petra", "petr"},
		{"petrel", "petrel"},
		{"petróleo", "petrole"},
		{"petróleos", "petrole"},
		{"petrolera", "petroler"},
		{"petroleras", "petroler"},
		{"petrolero", "petroler"},
		{"petroleros", "petroler"},
		{"petrolizar", "petroliz"},
		{"petroquímica", "petroquim"},
		{"petroquímicas", "petroquim"},
		{"petrov", "petrov"},
		{"peyote", "peyot"},
		{"peza", "pez"},
		{"pfcrn", "pfcrn"},
		{"pfcyp", "pfcyp"},
		{"pfeiffer", "pfeiff"},
		{"pgjdf", "pgjdf"},
		{"pgr", "pgr"},
		{"ph", "ph"},
		{"philip", "philip"},
		{"philippe", "philipp"},
		{"phillips", "phillips"},
		{"philos", "phil"},
		{"piales", "pial"},
		{"pianista", "pianist"},
		{"piano", "pian"},
		{"pib", "pib"},
		{"picaba", "pic"},
		{"picado", "pic"},
		{"picadores", "picador"},
		{"pican", "pic"},
		{"picante", "picant"},
		{"pichardo", "pichard"},
		{"pichichis", "pichichis"},
		{"pick", "pick"},
		{"pico", "pic"},
		{"picos", "pic"},
		{"pictórica", "pictor"},
		{"pictóricas", "pictor"},
		{"pida", "pid"},
		{"pidan", "pid"},
		{"pidas", "pid"},
		{"pide", "pid"},
		{"piden", "pid"},
		{"pidiendo", "pid"},
		{"pidiéndoles", "pid"},
		{"pidieron", "pid"},
		{"pidió", "pid"},
		{"pido", "pid"},
		{"pie", "pie"},
		{"piedad", "pied"},
		{"piedra", "piedr"},
		{"piedras", "piedr"},
		{"piedrecita", "piedrecit"},
		{"piedritas", "piedrit"},
		{"piel", "piel"},
		{"pieles", "piel"},
		{"pienda", "piend"},
		{"piensa", "piens"},
		{"piensan", "piens"},
		{"piense", "piens"},
		{"piensen", "piens"},
		{"pienso", "piens"},
		{"pierce", "pierc"},
		{"pierda", "pierd"},
		{"pierdan", "pierd"},
		{"pierde", "pierd"},
		{"pierden", "pierd"},
		{"pierdo", "pierd"},
		{"pierna", "piern"},
		{"piernas", "piern"},
		{"pierre", "pierr"},
		{"pies", "pies"},
		{"pieza", "piez"},
		{"piezas", "piez"},
		{"pifia", "pifi"},
		{"pigmento", "pigment"},
		{"pigmentosa", "pigment"},
		{"pila", "pil"},
		{"pilar", "pil"},
		{"pilares", "pilar"},
		{"pilas", "pil"},
		{"píldora", "pildor"},
		{"píldoras", "pildor"},
		{"pillma", "pillm"},
		{"pilón", "pilon"},
		{"piloncillo", "piloncill"},
		{"piloteando", "pilot"},
		{"piloteó", "pilote"},
		{"piloto", "pilot"},
		{"pilotos", "pilot"},
		{"pimentel", "pimentel"},
		{"piña", "piñ"},
		{"pinacoteca", "pinacotec"},
		{"pinal", "pinal"},
		{"piñaolayas", "piñaolay"},
		{"piñas", "piñ"},
		{"pinchado", "pinch"},
		{"pinchazo", "pinchaz"},
		{"pinchó", "pinch"},
		{"pino", "pin"},
		{"pinochet", "pinochet"},
		{"pinos", "pin"},
		{"pinta", "pint"},
		{"pintaban", "pint"},
		{"pintada", "pint"},
		{"pintadas", "pint"},
		{"pintado", "pint"},
		{"pintados", "pint"},
		{"pintan", "pint"},
		{"pintando", "pint"},
		{"pintar", "pint"},
		{"pintarlas", "pint"},
		{"pintarlo", "pint"},
		{"pintaron", "pint"},
		{"pintas", "pint"},
		{"pintor", "pintor"},
		{"pintora", "pintor"},
		{"pintores", "pintor"},
		{"pintura", "pintur"},
		{"pinturas", "pintur"},
		{"pinzón", "pinzon"},
		{"pío", "pio"},
		{"pionera", "pioner"},
		{"pionero", "pioner"},
		{"pioneros", "pioner"},
		{"pipa", "pip"},
		{"piquetes", "piquet"},
		{"piramidal", "piramidal"},
		{"pirata", "pirat"},
		{"piratas", "pirat"},
		{"pirate", "pirat"},
		{"pirateando", "pirat"},
		{"piretro", "piretr"},
		{"piri", "piri"},
		{"piropeado", "pirop"},
		{"pirulina", "pirulin"},
		{"pisaba", "pis"},
		{"pisado", "pis"},
		{"pisando", "pis"},
		{"pisen", "pis"},
		{"piso", "pis"},
		{"pisos", "pis"},
		{"pisoteaban", "pisot"},
		{"pisoteada", "pisot"},
		{"pisoteando", "pisot"},
		{"pisotear", "pisot"},
		{"pisoteen", "pisot"},
		{"pista", "pist"},
		{"pistas", "pist"},
		{"pistola", "pistol"},
		{"pistolas", "pistol"},
		{"pitas", "pit"},
		{"pitcheado", "pitch"},
		{"pitcheo", "pitche"},
		{"pitcher", "pitch"},
		{"pítcher", "pitch"},
		{"piti", "piti"},
		{"pito", "pit"},
		{"pitol", "pitol"},
		{"pitoncitos", "pitoncit"},
		{"pits", "pits"},
		{"pittsburgh", "pittsburgh"},
		{"pitufos", "pituf"},
		{"pivote", "pivot"},
		{"pizarro", "pizarr"},
		{"pizarrón", "pizarron"},
		{"pizpireta", "pizpiret"},
		{"pizuto", "pizut"},
		{"pizza", "pizz"},
		{"pizzas", "pizz"},
		{"pjf", "pjf"},
		{"placa", "plac"},
		{"placas", "plac"},
		{"place", "plac"},
		{"plácemes", "placem"},
		{"placer", "plac"},
		{"placet", "placet"},
		{"plácida", "plac"},
		{"plácidamente", "placid"},
		{"placita", "placit"},
		{"plaga", "plag"},
		{"plagadas", "plag"},
		{"plagas", "plag"},
		{"plagiada", "plagi"},
		{"plaguicidas", "plaguic"},
		{"plainfield", "plainfield"},
		{"plan", "plan"},
		{"plana", "plan"},
		{"planas", "plan"},
		{"planauto", "planaut"},
		{"planchar", "planch"},
		{"planchará", "planch"},
		{"planchas", "planch"},
		{"planea", "plane"},
		{"planeaban", "plan"},
		{"planeación", "planeacion"},
		{"planeada", "plan"},
		{"planeadas", "plan"},
		{"planeado", "plan"},
		{"planeados", "plan"},
		{"planean", "plan"},
		{"planeando", "plan"},
		{"planear", "plan"},
		{"planee", "plane"},
		{"planes", "plan"},
		{"planeta", "planet"},
		{"planetaria", "planetari"},
		{"planificación", "planif"},
		{"planilla", "planill"},
		{"planillas", "planill"},
		{"planiria", "planiri"},
		{"plano", "plan"},
		{"planos", "plan"},
		{"planta", "plant"},
		{"plantaciones", "plantacion"},
		{"plantado", "plant"},
		{"plantados", "plant"},
		{"plantara", "plant"},
		{"plantaron", "plant"},
		{"plantas", "plant"},
		{"plantea", "plante"},
		{"planteada", "plant"},
		{"planteadas", "plant"},
		{"planteado", "plant"},
		{"planteados", "plant"},
		{"planteamientos", "planteamient"},
		{"planteamos", "plant"},
		{"plantean", "plant"},
		{"plantear", "plant"},
		{"plantearan", "plant"},
		{"plantearon", "plant"},
		{"plantearse", "plant"},
		{"plantee", "plante"},
		{"plantéeles", "planteel"},
		{"planteen", "plant"},
		{"plantel", "plantel"},
		{"planteles", "plantel"},
		{"planteó", "plante"},
		{"plantón", "planton"},
		{"plantones", "planton"},
		{"plascencia", "plascenci"},
		{"plasma", "plasm"},
		{"plasmado", "plasm"},
		{"plasmar", "plasm"},
		{"plasme", "plasm"},
		{"plasmó", "plasm"},
		{"plastas", "plast"},
		{"plástica", "plastic"},
		{"plásticas", "plastic"},
		{"plástico", "plastic"},
		{"plásticos", "plastic"},
		{"plastilina", "plastilin"},
		{"plata", "plat"},
		{"plataforma", "plataform"},
		{"plataformas", "plataform"},
		{"plataneras", "plataner"},
		{"plátanos", "platan"},
		{"platas", "plat"},
		{"plate", "plat"},
		{"plateadas", "plat"},
		{"plateas", "plat"},
		{"platería", "plat"},
		{"plática", "platic"},
		{"platicaba", "platic"},
		{"platicábamos", "platic"},
		{"platicaban", "platic"},
		{"platicado", "platic"},
		{"platicamos", "platic"},
		{"platicando", "platic"},
		{"platicar", "platic"},
		{"platicas", "platic"},
		{"pláticas", "platic"},
		{"platicó", "platic"},
		{"platillo", "platill"},
		{"platillos", "platill"},
		{"platinos", "platin"},
		{"platique", "platiqu"},
		{"plato", "plat"},
		{"plausibles", "plausibl"},
		{"playa", "play"},
		{"playas", "play"},
		{"playback", "playback"},
		{"playera", "player"},
		{"playeras", "player"},
		{"playtex", "playtex"},
		{"plaza", "plaz"},
		{"plazas", "plaz"},
		{"plazo", "plaz"},
		{"plazos", "plaz"},
		{"plebiscito", "plebiscit"},
		{"plegables", "plegabl"},
		{"pleitesía", "pleites"},
		{"pleito", "pleit"},
		{"pleitos", "pleit"},
		{"plena", "plen"},
		{"plenamente", "plen"},
		{"plenaria", "plenari"},
		{"pleno", "plen"},
		{"plomería", "plom"},
		{"plomo", "plom"},
		{"pluma", "plum"},
		{"plumas", "plum"},
		{"plumones", "plumon"},
		{"plural", "plural"},
		{"plusmarca", "plusmarc"},
		{"plusmarcas", "plusmarc"},
		{"plusvalía", "plusval"},
		{"plutarco", "plutarc"},
		{"pluvial", "pluvial"},
		{"pluviales", "pluvial"},
		{"plymouth", "plymouth"},
		{"pn", "pn"},
		{"po", "po"},
		{"población", "poblacion"},
		{"poblacional", "poblacional"},
		{"poblaciones", "poblacion"},
		{"poblada", "pobl"},
		{"pobladas", "pobl"},
		{"poblado", "pobl"},
		{"pobladores", "poblador"},
		{"poblados", "pobl"},
		{"poblano", "poblan"},
		{"poblanos", "poblan"},
		{"poblaron", "pobl"},
		{"pobre", "pobr"},
		{"pobres", "pobr"},
		{"pobreza", "pobrez"},
		{"poca", "poc"},
		{"pocas", "poc"},
		{"pocho", "poch"},
		{"pochtecas", "pochtec"},
		{"poco", "poc"},
		{"pocos", "poc"},
		{"poda", "pod"},
		{"podamos", "pod"},
		{"podemos", "pod"},
		{"poder", "pod"},
		{"poderes", "poder"},
		{"poderío", "poderi"},
		{"poderlas", "pod"},
		{"poderosa", "poder"},
		{"poderoso", "poder"},
		{"poderosos", "poder"},
		{"podía", "pod"},
		{"podíamos", "pod"},
		{"podían", "pod"},
		{"podido", "pod"},
		{"podrá", "podr"},
		{"podrán", "podran"},
		{"podrás", "podras"},
		{"podré", "podr"},
		{"podremos", "podr"},
		{"podría", "podr"},
		{"podríamos", "podr"},
		{"podrían", "podr"},
		{"podrías", "podr"},
		{"poema", "poem"},
		{"poemario", "poemari"},
		{"poemas", "poem"},
		{"poesía", "poes"},
		{"poeta", "poet"},
		{"poético", "poetic"},
		{"poetisa", "poetis"},
		{"point", "point"},
		{"pókar", "pok"},
		{"póker", "pok"},
		{"polanco", "polanc"},
		{"polar", "pol"},
		{"polémica", "polem"},
		{"polémicas", "polem"},
		{"polemizan", "polemiz"},
		{"polen", "pol"},
		{"poleo", "pole"},
		{"policarpo", "policarp"},
		{"policia", "polici"},
		{"policía", "polic"},
		{"policiaca", "policiac"},
		{"policíaca", "policiac"},
		{"policiacas", "policiac"},
		{"policíacas", "policiac"},
		{"policiaco", "policiac"},
		{"policíaco", "policiac"},
		{"policiacos", "policiac"},
		{"policíacos", "policiac"},
		{"policial", "policial"},
		{"policiales", "policial"},
		{"policias", "polici"},
		{"policías", "polic"},
		{"polifonía", "polifon"},
		{"poligono", "poligon"},
		{"polígono", "poligon"},
		{"polilla", "polill"},
		{"polinésico", "polines"},
		{"polinización", "poliniz"},
		{"polio", "poli"},
		{"poliomielitis", "poliomielitis"},
		{"politécnico", "politecn"},
		{"politeísta", "politeist"},
		{"politeístas", "politeist"},
		{"política", "polit"},
		{"políticas", "polit"},
		{"politico", "polit"},
		{"político", "polit"},
		{"políticos", "polit"},
		{"politiquero", "politiquer"},
		{"politólogos", "politolog"},
		{"poliuretano", "poliuretan"},
		{"polivalente", "polivalent"},
		{"pollo", "poll"},
		{"pollution", "pollution"},
		{"polo", "pol"},
		{"polonia", "poloni"},
		{"polveras", "polver"},
		{"polvo", "polv"},
		{"polvos", "polv"},
		{"pomadas", "pom"},
		{"pomares", "pomar"},
		{"pomperrier", "pomperri"},
		{"pompeyo", "pompey"},
		{"pompidou", "pompidou"},
		{"ponce", "ponc"},
		{"ponchando", "ponch"},
		{"ponche", "ponch"},
		{"pond", "pond"},
		{"pondera", "ponder"},
		{"ponderada", "ponder"},
		{"pondrá", "pondr"},
		{"pondrán", "pondran"},
		{"pondrás", "pondras"},
		{"pondremos", "pondr"},
		{"pondrían", "pondr"},
		{"ponds", "ponds"},
		{"pone", "pon"},
		{"ponemos", "pon"},
		{"ponen", "pon"},
		{"ponencia", "ponenci"},
		{"ponencias", "ponenci"},
		{"ponente", "ponent"},
		{"ponentes", "ponent"},
		{"poner", "pon"},
		{"ponerla", "pon"},
		{"ponerle", "pon"},
		{"ponerles", "pon"},
		{"ponerlo", "pon"},
		{"ponerlos", "pon"},
		{"ponernos", "pon"},
		{"ponerse", "pon"},
		{"pones", "pon"},
		{"ponga", "pong"},
		{"pongamos", "pong"},
		{"pongan", "pong"},
		{"póngase", "pong"},
		{"pongo", "pong"},
		{"ponía", "pon"},
		{"poniatowska", "poniatowsk"},
		{"poniendo", "pon"},
		{"poniente", "ponient"},
		{"pons", "pons"},
		{"pontiac", "pontiac"},
		{"pontier", "ponti"},
		{"pontificado", "pontific"},
		{"pontífice", "pontific"},
		{"pontificia", "pontifici"},
		{"pontificio", "pontifici"},
		{"pony", "pony"},
		{"pop", "pop"},
		{"popa", "pop"},
		{"pope", "pop"},
		{"popo", "pop"},
		{"poppe", "popp"},
		{"popular", "popul"},
		{"populares", "popular"},
		{"popularidad", "popular"},
		{"popularísimo", "popularisim"},
		{"populismo", "popul"},
		{"populista", "popul"},
		{"populistas", "popul"},
		{"populosos", "popul"},
		{"poquísimo", "poquisim"},
		{"poquitas", "poquit"},
		{"poquitito", "poquitit"},
		{"poquito", "poquit"},
		{"por", "por"},
		{"porcelana", "porcelan"},
		{"porcentaje", "porcentaj"},
		{"porcentajes", "porcentaj"},
		{"porcentual", "porcentual"},
		{"porcentuales", "porcentual"},
		{"porcícolas", "porcicol"},
		{"porciles", "porcil"},
		{"porción", "porcion"},
		{"porcioncita", "porcioncit"},
		{"porciones", "porcion"},
		{"porfesión", "porfesion"},
		{"porfía", "porf"},
		{"porfiriato", "porfiriat"},
		{"porfirio", "porfiri"},
		{"pormenores", "pormenor"},
		{"pormenorizados", "pormenoriz"},
		{"pornográficas", "pornograf"},
		{"pornográficos", "pornograf"},
		{"porque", "porqu"},
		{"porqué", "porqu"},
		{"porras", "porr"},
		{"porros", "porr"},
		{"porta", "port"},
		{"portaba", "port"},
		{"portaban", "port"},
		{"portabilidad", "portabil"},
		{"portación", "portacion"},
		{"portador", "portador"},
		{"portafolio", "portafoli"},
		{"portal", "portal"},
		{"portan", "port"},
		{"portando", "port"},
		{"portar", "port"},
		{"portarlas", "port"},
		{"portátil", "portatil"},
		{"portátiles", "portatil"},
		{"portavoz", "portavoz"},
		{"porte", "port"},
		{"porten", "port"},
		{"porter", "port"},
		{"portería", "port"},
		{"portero", "porter"},
		{"portezuela", "portezuel"},
		{"portillo", "portill"},
		{"portland", "portland"},
		{"portuaria", "portuari"},
		{"portuarias", "portuari"},
		{"portuario", "portuari"},
		{"portuarios", "portuari"},
		{"portugal", "portugal"},
		{"portugués", "portugues"},
		{"porvenir", "porven"},
		{"posada", "pos"},
		{"posee", "pose"},
		{"poseedor", "poseedor"},
		{"poseen", "pos"},
		{"poseer", "pos"},
		{"poseían", "pos"},
		{"poseidón", "poseidon"},
		{"poses", "pos"},
		{"posesión", "posesion"},
		{"posesionar", "posesion"},
		{"posesionase", "posesion"},
		{"posesivas", "poses"},
		{"posgrado", "posgr"},
		{"posgrados", "posgr"},
		{"posibilidad", "posibil"},
		{"posibilidades", "posibil"},
		{"posibilitan", "posibilit"},
		{"posible", "posibl"},
		{"posiblemente", "posibl"},
		{"posibles", "posibl"},
		{"posición", "posicion"},
		{"posicionado", "posicion"},
		{"posicionamiento", "posicion"},
		{"posiciones", "posicion"},
		{"posicionó", "posicion"},
		{"positiva", "posit"},
		{"positivamente", "posit"},
		{"positivas", "posit"},
		{"positivo", "posit"},
		{"positivos", "posit"},
		{"pospuesto", "pospuest"},
		{"pospuso", "pospus"},
		{"post", "post"},
		{"postal", "postal"},
		{"poste", "post"},
		{"postelectoral", "postelectoral"},
		{"postelectorales", "postelectoral"},
		{"postergación", "posterg"},
		{"posterior", "posterior"},
		{"posteriores", "posterior"},
		{"posteriormente", "posterior"},
		{"posters", "posters"},
		{"postes", "post"},
		{"postguerra", "postguerr"},
		{"postor", "postor"},
		{"postores", "postor"},
		{"postre", "postr"},
		{"postulaban", "postul"},
		{"postulación", "postul"},
		{"postulada", "postul"},
		{"postulado", "postul"},
		{"postulados", "postul"},
		{"postular", "postul"},
		{"postularse", "postul"},
		{"postuló", "postul"},
		{"póstumo", "postum"},
		{"postura", "postur"},
		{"posturas", "postur"},
		{"potabilizadora", "potabiliz"},
		{"potabilizadoras", "potabiliz"},
		{"potable", "potabl"},
		{"potencia", "potenci"},
		{"potencial", "potencial"},
		{"potenciales", "potencial"},
		{"potencialmente", "potencial"},
		{"potente", "potent"},
		{"potestad", "potest"},
		{"potomac", "potomac"},
		{"potosi", "potosi"},
		{"potosí", "potos"},
		{"potosino", "potosin"},
		{"potrillo", "potrill"},
		{"potrillos", "potrill"},
		{"potros", "potr"},
		{"pottlesberghe", "pottlesbergh"},
		{"poul", "poul"},
		{"powell", "powell"},
		{"powerbook", "powerbook"},
		{"powered", "power"},
		{"pozas", "poz"},
		{"pozo", "poz"},
		{"pozole", "pozol"},
		{"pozos", "poz"},
		{"ppc", "ppc"},
		{"pps", "pps"},
		{"practica", "practic"},
		{"práctica", "practic"},
		{"practicaban", "practic"},
		{"practicable", "practic"},
		{"practicadas", "practic"},
		{"prácticamente", "practic"},
		{"practicamos", "practic"},
		{"practican", "practic"},
		{"practicar", "practic"},
		{"practicaran", "practic"},
		{"practicaron", "practic"},
		{"practicas", "practic"},
		{"prácticas", "practic"},
		{"práctico", "practic"},
		{"practicó", "practic"},
		{"prácticos", "practic"},
		{"practique", "practiqu"},
		{"practiquen", "practiqu"},
		{"praderas", "prader"},
		{"prado", "prad"},
		{"prampolini", "prampolini"},
		{"praviel", "praviel"},
		{"praxedis", "praxedis"},
		{"prd", "prd"},
		{"pre", "pre"},
		{"preámbulo", "preambul"},
		{"prebendas", "prebend"},
		{"precandidato", "precandidat"},
		{"precaria", "precari"},
		{"precario", "precari"},
		{"precaución", "precaucion"},
		{"precautorio", "precautori"},
		{"precavido", "precav"},
		{"precavidos", "precav"},
		{"precedente", "precedent"},
		{"precedentes", "precedent"},
		{"precedidas", "preced"},
		{"precepto", "precept"},
		{"preceptos", "precept"},
		{"preciados", "preci"},
		{"precio", "preci"},
		{"precios", "preci"},
		{"preciosas", "preci"},
		{"precioso", "precios"},
		{"precipitación", "precipit"},
		{"precipitaciones", "precipit"},
		{"precipitadamente", "precipit"},
		{"precipitadas", "precipit"},
		{"precipitándose", "precipit"},
		{"precipitaron", "precipit"},
		{"precipitarse", "precipit"},
		{"precipitó", "precipit"},
		{"precisa", "precis"},
		{"precisado", "precis"},
		{"precisamente", "precis"},
		{"precísamente", "precis"},
		{"precisan", "precis"},
		{"precisando", "precis"},
		{"precisar", "precis"},
		{"precisaron", "precis"},
		{"precisión", "precision"},
		{"precisiones", "precision"},
		{"preciso", "precis"},
		{"precisó", "precis"},
		{"precisos", "precis"},
		{"preclaros", "preclar"},
		{"precolombina", "precolombin"},
		{"preconiza", "preconiz"},
		{"precursora", "precursor"},
		{"precursores", "precursor"},
		{"predecir", "predec"},
		{"predestinado", "predestin"},
		{"predeterminado", "predetermin"},
		{"predial", "predial"},
		{"predica", "predic"},
		{"predicación", "predic"},
		{"predicador", "predic"},
		{"predicar", "predic"},
		{"predicciones", "prediccion"},
		{"predice", "predic"},
		{"predijo", "predij"},
		{"predio", "predi"},
		{"predispone", "predispon"},
		{"predomina", "predomin"},
		{"predominan", "predomin"},
		{"predominante", "predomin"},
		{"predominantes", "predomin"},
		{"predominaron", "predomin"},
		{"preelectorales", "preelectoral"},
		{"preerupción", "preerupcion"},
		{"preescolar", "preescol"},
		{"preestablecido", "preestablec"},
		{"preestablecidos", "preestablec"},
		{"prefecto", "prefect"},
		{"preferencia", "preferent"},
		{"preferencial", "preferencial"},
		{"preferencias", "preferent"},
		{"preferentemente", "preferent"},
		{"prefería", "pref"},
		{"preferible", "prefer"},
		{"preferida", "prefer"},
		{"preferirán", "prefer"},
		{"preferiría", "prefer"},
		{"prefiera", "pref"},
		{"prefieran", "pref"},
		{"prefiere", "prefier"},
		{"prefieren", "prefier"},
		{"prefiero", "prefier"},
		{"prefijado", "prefij"},
		{"prefirieron", "prefir"},
		{"prefirió", "prefir"},
		{"pregrado", "pregr"},
		{"pregunta", "pregunt"},
		{"preguntaba", "pregunt"},
		{"preguntamos", "pregunt"},
		{"preguntan", "pregunt"},
		{"preguntando", "pregunt"},
		{"preguntar", "pregunt"},
		{"preguntarle", "pregunt"},
		{"preguntarnos", "pregunt"},
		{"preguntaron", "pregunt"},
		{"preguntarse", "pregunt"},
		{"preguntas", "pregunt"},
		{"pregunte", "pregunt"},
		{"pregúnteles", "preguntel"},
		{"pregunto", "pregunt"},
		{"preguntó", "pregunt"},
		{"prehispánica", "prehispan"},
		{"prehispánicas", "prehispan"},
		{"prehispánico", "prehispan"},
		{"prejuicios", "prejuici"},
		{"prejunior", "prejunior"},
		{"prelado", "prel"},
		{"prelavado", "prelav"},
		{"preliminar", "prelimin"},
		{"preliminares", "preliminar"},
		{"prematuramente", "prematur"},
		{"prematuro", "prematur"},
		{"premestizo", "premestiz"},
		{"premiaba", "premi"},
		{"premiación", "premiacion"},
		{"premiadas", "premi"},
		{"premiado", "premi"},
		{"premiados", "premi"},
		{"premiaron", "premi"},
		{"premie", "premi"},
		{"premier", "premi"},
		{"premio", "premi"},
		{"premió", "prem"},
		{"premios", "premi"},
		{"premisa", "premis"},
		{"premoderno", "premodern"},
		{"premonición", "premonicion"},
		{"premundialistas", "premundial"},
		{"prenda", "prend"},
		{"prendarios", "prendari"},
		{"prendas", "prend"},
		{"prende", "prend"},
		{"prendedores", "prendedor"},
		{"prender", "prend"},
		{"prendido", "prend"},
		{"prendieron", "prend"},
		{"prensa", "prens"},
		{"prensado", "prens"},
		{"preocupa", "preocup"},
		{"preocupaba", "preocup"},
		{"preocupación", "preocup"},
		{"preocupaciones", "preocup"},
		{"preocupada", "preocup"},
		{"preocupadas", "preocup"},
		{"preocupado", "preocup"},
		{"preocupados", "preocup"},
		{"preocupan", "preocup"},
		{"preocupar", "preocup"},
		{"preocupara", "preocup"},
		{"preocupará", "preocup"},
		{"preocuparía", "preocup"},
		{"preocuparle", "preocup"},
		{"preocuparon", "preocup"},
		{"preocuparse", "preocup"},
		{"preocupe", "preocup"},
		{"preocupen", "preocup"},
		{"preocupes", "preocup"},
		{"prepara", "prep"},
		{"preparaba", "prepar"},
		{"preparaban", "prepar"},
		{"preparación", "prepar"},
		{"preparada", "prepar"},
		{"preparadas", "prepar"},
		{"preparado", "prepar"},
		{"preparados", "prepar"},
		{"preparamos", "prepar"},
		{"preparan", "prep"},
		{"preparando", "prepar"},
		{"preparar", "prepar"},
		{"prepararán", "prepar"},
		{"prepararla", "prepar"},
		{"prepararse", "prepar"},
		{"prepárate", "preparat"},
		{"preparativos", "prepar"},
		{"preparatoria", "preparatori"},
		{"preparatorias", "preparatori"},
		{"preparatorio", "preparatori"},
		{"preparatorios", "preparatori"},
		{"prepárele", "preparel"},
		{"preparen", "prepar"},
		{"prepárese", "prepares"},
		{"preparó", "prepar"},
		{"preponderantemente", "preponder"},
		{"prepotencia", "prepotent"},
		{"prepotente", "prepotent"},
		{"prepotentes", "prepotent"},
		{"preproducción", "preproduccion"},
		{"prerequisitos", "prerequisit"},
		{"prerregistro", "prerregistr"},
		{"prerrogativa", "prerrog"},
		{"prerrogativas", "prerrog"},
		{"presa", "pres"},
		{"presas", "pres"},
		{"prescindir", "prescind"},
		{"prescriben", "prescrib"},
		{"prescritos", "prescrit"},
		{"preseas", "pres"},
		{"preselección", "preseleccion"},
		{"presencia", "presenci"},
		{"presenciado", "presenci"},
		{"presenciados", "presenci"},
		{"presencial", "presencial"},
		{"presenciar", "presenci"},
		{"presenciaron", "presenci"},
		{"presenció", "presenc"},
		{"presenta", "present"},
		{"presentaba", "present"},
		{"presentaban", "present"},
		{"presentación", "present"},
		{"presentaciones", "present"},
		{"presentada", "present"},
		{"presentadas", "present"},
		{"presentado", "present"},
		{"presentador", "present"},
		{"presentadores", "present"},
		{"presentados", "present"},
		{"presentamos", "present"},
		{"presentan", "present"},
		{"presentando", "present"},
		{"presentándolo", "present"},
		{"presentándose", "present"},
		{"presentar", "present"},
		{"presentara", "present"},
		{"presentará", "present"},
		{"presentáramos", "present"},
		{"presentarán", "present"},
		{"presentaremos", "present"},
		{"presentarla", "present"},
		{"presentarlo", "present"},
		{"presentarlos", "present"},
		{"presentarme", "present"},
		{"presentaron", "present"},
		{"presentarse", "present"},
		{"presente", "present"},
		{"presenten", "present"},
		{"presentes", "present"},
		{"presentí", "present"},
		{"presentó", "present"},
		{"preservación", "preserv"},
		{"preservado", "preserv"},
		{"preservar", "preserv"},
		{"preservarle", "preserv"},
		{"presida", "pres"},
		{"preside", "presid"},
		{"presiden", "presid"},
		{"presidencia", "president"},
		{"presidenciable", "presidenci"},
		{"presidenciables", "presidenci"},
		{"presidencial", "presidencial"},
		{"presidenciales", "presidencial"},
		{"presidencialismo", "presidencial"},
		{"presidencialista", "presidencial"},
		{"presidenta", "president"},
		{"presidente", "president"},
		{"presidentes", "president"},
		{"presidida", "presid"},
		{"presidido", "presid"},
		{"presidió", "presid"},
		{"presidir", "presid"},
		{"presidirá", "presid"},
		{"presidium", "presidium"},
		{"presión", "presion"},
		{"presionado", "presion"},
		{"presionando", "presion"},
		{"presionar", "presion"},
		{"presionará", "presion"},
		{"presionarlos", "presion"},
		{"presionaron", "presion"},
		{"presione", "presion"},
		{"presiones", "presion"},
		{"presley", "presley"},
		{"preso", "pres"},
		{"presos", "pres"},
		{"pressey", "pressey"},
		{"presta", "prest"},
		{"prestación", "prestacion"},
		{"prestaciones", "prestacion"},
		{"prestada", "prest"},
		{"prestado", "prest"},
		{"prestadores", "prestador"},
		{"préstamo", "prestam"},
		{"préstamos", "prest"},
		{"prestan", "prest"},
		{"prestancia", "prestanci"},
		{"prestar", "prest"},
		{"prestaría", "prest"},
		{"prestaron", "prest"},
		{"preste", "prest"},
		{"prestigiado", "prestigi"},
		{"prestigian", "prestigi"},
		{"prestigio", "prestigi"},
		{"prestigiosos", "prestigi"},
		{"prestó", "prest"},
		{"presume", "presum"},
		{"presumiblemente", "presum"},
		{"presumida", "presum"},
		{"presumieron", "presum"},
		{"presumir", "presum"},
		{"presunción", "presuncion"},
		{"presunta", "presunt"},
		{"presuntamente", "presunt"},
		{"presuntas", "presunt"},
		{"presunto", "presunt"},
		{"presuntos", "presunt"},
		{"presuntuoso", "presuntu"},
		{"presupuestadas", "presupuest"},
		{"presupuestal", "presupuestal"},
		{"presupuestario", "presupuestari"},
		{"presupuesto", "presupuest"},
		{"presurosamente", "presur"},
		{"preta", "pret"},
		{"pretel", "pretel"},
		{"pretemporada", "pretempor"},
		{"pretenda", "pretend"},
		{"pretende", "pretend"},
		{"pretendemos", "pretend"},
		{"pretenden", "pretend"},
		{"pretender", "pretend"},
		{"pretendía", "pretend"},
		{"pretendían", "pretend"},
		{"pretendido", "pretend"},
		{"pretendiente", "pretendient"},
		{"pretendieran", "pretend"},
		{"pretendieron", "pretend"},
		{"pretendió", "pretend"},
		{"pretensión", "pretension"},
		{"pretexto", "pretext"},
		{"prevalece", "prevalec"},
		{"prevalecen", "prevalec"},
		{"prevaleciendo", "prevalec"},
		{"prevalecientes", "prevalecient"},
		{"prevalezca", "prevalezc"},
		{"preve", "prev"},
		{"prevé", "prev"},
		{"prevee", "preve"},
		{"prevención", "prevencion"},
		{"prevenida", "preven"},
		{"prevenir", "preven"},
		{"prevenirse", "preven"},
		{"preventiva", "prevent"},
		{"preventivas", "prevent"},
		{"preventivo", "prevent"},
		{"prever", "prev"},
		{"preverificación", "preverif"},
		{"previa", "previ"},
		{"previamente", "previ"},
		{"previas", "previ"},
		{"previendo", "prev"},
		{"previéndolo", "prev"},
		{"previene", "previen"},
		{"previno", "previn"},
		{"previo", "previ"},
		{"previó", "prev"},
		{"previos", "previ"},
		{"previsible", "previs"},
		{"previsión", "prevision"},
		{"prevista", "previst"},
		{"previstas", "previst"},
		{"previsto", "previst"},
		{"previstos", "previst"},
		{"prevoisin", "prevoisin"},
		{"pri", "pri"},
		{"pría", "pri"},
		{"price", "pric"},
		{"prieto", "priet"},
		{"prigione", "prigion"},
		{"prigobierno", "prigobiern"},
		{"priísmo", "priism"},
		{"priísta", "priist"},
		{"priístas", "priist"},
		{"prima", "prim"},
		{"primaria", "primari"},
		{"primarias", "primari"},
		{"primario", "primari"},
		{"primarios", "primari"},
		{"primas", "prim"},
		{"primates", "primat"},
		{"primavera", "primaver"},
		{"prime", "prim"},
		{"primer", "prim"},
		{"primera", "primer"},
		{"primeramente", "primer"},
		{"primeras", "primer"},
		{"primeriza", "primeriz"},
		{"primero", "primer"},
		{"primeros", "primer"},
		{"primex", "primex"},
		{"primitivas", "primit"},
		{"primitivismo", "primitiv"},
		{"primitivo", "primit"},
		{"primitivos", "primit"},
		{"primo", "prim"},
		{"primogénita", "primogenit"},
		{"primogénito", "primogenit"},
		{"primordial", "primordial"},
		{"primordiales", "primordial"},
		{"princesa", "princes"},
		{"princesas", "princes"},
		{"princeton", "princeton"},
		{"principal", "principal"},
		{"principales", "principal"},
		{"principalmente", "principal"},
		{"príncipe", "princip"},
		{"príncipes", "princip"},
		{"principesco", "principesc"},
		{"principiante", "principi"},
		{"principio", "principi"},
		{"principios", "principi"},
		{"prioridad", "priorid"},
		{"prioridades", "prioridad"},
		{"prioritaria", "prioritari"},
		{"prioritario", "prioritari"},
		{"prioritarios", "prioritari"},
		{"prisa", "pris"},
		{"prisión", "prision"},
		{"prisionero", "prisioner"},
		{"prisioneros", "prisioner"},
		{"pristina", "pristin"},
		{"priva", "priv"},
		{"privacidad", "privac"},
		{"privada", "priv"},
		{"privadas", "priv"},
		{"privado", "priv"},
		{"privados", "priv"},
		{"privan", "priv"},
		{"privar", "priv"},
		{"privativa", "privat"},
		{"privativo", "privat"},
		{"privatización", "privatiz"},
		{"privatizaciones", "privatiz"},
		{"privatizado", "privatiz"},
		{"privatizar", "privatiz"},
		{"privilegiada", "privilegi"},
		{"privilegiado", "privilegi"},
		{"privilegio", "privilegi"},
		{"privilegios", "privilegi"},
		{"privó", "priv"},
		{"pro", "pro"},
		{"probabilidades", "probabil"},
		{"probable", "probabl"},
		{"probablemente", "probabl"},
		{"probables", "probabl"},
		{"probada", "prob"},
		{"probado", "prob"},
		{"probados", "prob"},
		{"probar", "prob"},
		{"probara", "prob"},
		{"probarle", "prob"},
		{"probarlo", "prob"},
		{"probaron", "prob"},
		{"probarse", "prob"},
		{"problema", "problem"},
		{"problemas", "problem"},
		{"problemática", "problemat"},
		{"problemático", "problemat"},
		{"probó", "prob"},
		{"probursa", "proburs"},
		{"proceda", "proced"},
		{"procede", "proced"},
		{"proceden", "proced"},
		{"procedencia", "procedent"},
		{"procedente", "procedent"},
		{"procedentes", "procedent"},
		{"proceder", "proced"},
		{"procederá", "proced"},
		{"procedimentales", "procedimental"},
		{"procedimiento", "proced"},
		{"procedimientos", "proced"},
		{"procéntrico", "procentr"},
		{"procentro", "procentr"},
		{"procesa", "proces"},
		{"procesaba", "proces"},
		{"procesada", "proces"},
		{"procesado", "proces"},
		{"procesador", "proces"},
		{"procesadora", "proces"},
		{"procesadores", "proces"},
		{"procesados", "proces"},
		{"procesal", "procesal"},
		{"procesales", "procesal"},
		{"procesamiento", "proces"},
		{"procesando", "proces"},
		{"proceso", "proces"},
		{"procesos", "proces"},
		{"proclama", "proclam"},
		{"proclamaciones", "proclam"},
		{"proclamado", "proclam"},
		{"proclamó", "proclam"},
		{"procreación", "procreacion"},
		{"procura", "procur"},
		{"procuración", "procur"},
		{"procurador", "procur"},
		{"procuradora", "procur"},
		{"procuraduria", "procuraduri"},
		{"procuraduría", "procuradur"},
		{"procuramos", "procur"},
		{"procurar", "procur"},
		{"procurarán", "procur"},
		{"procurarse", "procur"},
		{"procure", "procur"},
		{"procuro", "procur"},
		{"procuró", "procur"},
		{"prodigiosa", "prodigi"},
		{"producción", "produccion"},
		{"producciones", "produccion"},
		{"produce", "produc"},
		{"producen", "produc"},
		{"producían", "produc"},
		{"producida", "produc"},
		{"producidas", "produc"},
		{"producido", "produc"},
		{"producidos", "produc"},
		{"produciendo", "produc"},
		{"producir", "produc"},
		{"producirá", "produc"},
		{"producirían", "produc"},
		{"producirse", "produc"},
		{"productiva", "product"},
		{"productivas", "product"},
		{"productividad", "product"},
		{"productivo", "product"},
		{"productivos", "product"},
		{"producto", "product"},
		{"productor", "productor"},
		{"productora", "productor"},
		{"productoras", "productor"},
		{"productores", "productor"},
		{"productos", "product"},
		{"produjera", "produjer"},
		{"produjeron", "produjeron"},
		{"produjo", "produj"},
		{"produzca", "produzc"},
		{"produzcan", "produzc"},
		{"proeza", "proez"},
		{"profeco", "profec"},
		{"profesa", "profes"},
		{"profesan", "profes"},
		{"profesión", "profesion"},
		{"profesional", "profesional"},
		{"profesionales", "profesional"},
		{"profesionalismo", "profesional"},
		{"profesionalización", "profesionaliz"},
		{"", ""},
		{"profesionalmente", "profesional"},
		{"profesionista", "profesion"},
		{"profesionistas", "profesion"},
		{"profesor", "profesor"},
		{"profesora", "profesor"},
		{"profesorado", "profesor"},
		{"profesores", "profesor"},
		{"profeta", "profet"},
		{"profetas", "profet"},
		{"profr", "profr"},
		{"prófugo", "profug"},
		{"prófugos", "profug"},
		{"profunda", "profund"},
		{"profundamente", "profund"},
		{"profundas", "profund"},
		{"profundicen", "profundic"},
		{"profundidad", "profund"},
		{"profundidades", "profund"},
		{"profundizado", "profundiz"},
		{"profundizamos", "profundiz"},
		{"profundizar", "profundiz"},
		{"profundizará", "profundiz"},
		{"profundo", "profund"},
		{"profundos", "profund"},
		{"progenitora", "progenitor"},
		{"programa", "program"},
		{"programación", "program"},
		{"programada", "program"},
		{"programadas", "program"},
		{"programado", "program"},
		{"programados", "program"},
		{"programan", "program"},
		{"programando", "program"},
		{"programar", "program"},
		{"programas", "program"},
		{"programático", "programat"},
		{"programmer", "programm"},
		{"programó", "program"},
		{"progresando", "progres"},
		{"progresarán", "progres"},
		{"progresista", "progres"},
		{"progresivamente", "progres"},
		{"progresivo", "progres"},
		{"progreso", "progres"},
		{"progresos", "progres"},
		{"prohiba", "prohib"},
		{"prohiban", "prohib"},
		{"prohíbe", "prohib"},
		{"prohibía", "prohib"},
		{"prohibición", "prohibicion"},
		{"prohibiciones", "prohibicion"},
		{"prohibida", "prohib"},
		{"prohibidas", "prohib"},
		{"prohibido", "prohib"},
		{"prohibidos", "prohib"},
		{"prohibieron", "prohib"},
		{"prohibió", "prohib"},
		{"prohibir", "prohib"},
		{"proletaria", "proletari"},
		{"prolíder", "prolid"},
		{"proliferación", "prolifer"},
		{"proliferan", "prolifer"},
		{"prolijas", "prolij"},
		{"prologan", "prolog"},
		{"prólogo", "prolog"},
		{"prolongación", "prolong"},
		{"prolongada", "prolong"},
		{"prolongadas", "prolong"},
		{"prolongado", "prolong"},
		{"prolongan", "prolong"},
		{"prolongar", "prolong"},
		{"prolongarse", "prolong"},
		{"prolongó", "prolong"},
		{"promediar", "promedi"},
		{"promedio", "promedi"},
		{"promesa", "promes"},
		{"promesas", "promes"},
		{"prometan", "promet"},
		{"promete", "promet"},
		{"prometedor", "prometedor"},
		{"prometen", "promet"},
		{"prometerle", "promet"},
		{"prometerles", "promet"},
		{"prometidas", "promet"},
		{"prometido", "promet"},
		{"prometieron", "promet"},
		{"prometimos", "promet"},
		{"prometió", "promet"},
		{"prometo", "promet"},
		{"promex", "promex"},
		{"promiscuidad", "promiscu"},
		{"promiscuo", "promiscu"},
		{"promoción", "promocion"},
		{"promociona", "promocion"},
		{"promocionado", "promocion"},
		{"promocional", "promocional"},
		{"promocionales", "promocional"},
		{"promocionan", "promocion"},
		{"promocionando", "promocion"},
		{"promocionar", "promocion"},
		{"promocionarlo", "promocion"},
		{"promocionarse", "promocion"},
		{"promociones", "promocion"},
		{"promotor", "promotor"},
		{"promotora", "promotor"},
		{"promotoras", "promotor"},
		{"promotores", "promotor"},
		{"promovemos", "promov"},
		{"promover", "promov"},
		{"promoverá", "promov"},
		{"promoverán", "promov"},
		{"promovía", "promov"},
		{"promovida", "promov"},
		{"promovido", "promov"},
		{"promovidos", "promov"},
		{"promoviendo", "promov"},
		{"promovieron", "promov"},
		{"promovió", "promov"},
		{"prompt", "prompt"},
		{"promueva", "promuev"},
		{"promueve", "promuev"},
		{"promueven", "promuev"},
		{"promulgación", "promulg"},
		{"promulgada", "promulg"},
		{"promvido", "promv"},
		{"pronasol", "pronasol"},
		{"pronostica", "pronost"},
		{"pronosticada", "pronostic"},
		{"pronosticado", "pronostic"},
		{"pronosticaron", "pronostic"},
		{"pronostico", "pronost"},
		{"pronóstico", "pronost"},
		{"pronósticos", "pronost"},
		{"pronta", "pront"},
		{"pronto", "pront"},
		{"pronuciados", "pronuci"},
		{"pronuncia", "pronunci"},
		{"pronunciadas", "pronunci"},
		{"pronunciamiento", "pronunci"},
		{"pronunciamos", "pronunci"},
		{"pronunciar", "pronunci"},
		{"pronunciaron", "pronunci"},
		{"pronuncie", "pronunci"},
		{"pronunció", "pronunc"},
		{"propagación", "propag"},
		{"propaganda", "propagand"},
		{"propagandísticas", "propagandist"},
		{"propagaron", "propag"},
		{"propensas", "propens"},
		{"propia", "propi"},
		{"propiamente", "propi"},
		{"propias", "propi"},
		{"propiciado", "propici"},
		{"propiciando", "propici"},
		{"propiciar", "propici"},
		{"propiciará", "propici"},
		{"propiciarse", "propici"},
		{"propicien", "propici"},
		{"propicio", "propici"},
		{"propició", "propic"},
		{"propiedad", "propied"},
		{"propiedades", "propiedad"},
		{"propietaria", "propietari"},
		{"propietario", "propietari"},
		{"propietarios", "propietari"},
		{"propinándole", "propin"},
		{"propinarle", "propin"},
		{"propinaron", "propin"},
		{"propinas", "propin"},
		{"propinó", "propin"},
		{"propio", "propi"},
		{"propios", "propi"},
		{"propone", "propon"},
		{"proponemos", "propon"},
		{"proponen", "propon"},
		{"proponer", "propon"},
		{"proponerle", "propon"},
		{"proponernos", "propon"},
		{"proponga", "propong"},
		{"propongo", "propong"},
		{"proponían", "propon"},
		{"proporción", "proporcion"},
		{"proporciona", "proporcion"},
		{"proporcionaba", "proporcion"},
		{"proporcionada", "proporcion"},
		{"proporcionadas", "proporcion"},
		{"proporcionado", "proporcion"},
		{"proporcionados", "proporcion"},
		{"proporcional", "proporcional"},
		{"proporcionan", "proporcion"},
		{"proporcionando", "proporcion"},
		{"proporcionar", "proporcion"},
		{"proporcionará", "proporcion"},
		{"proporcionaría", "proporcion"},
		{"proporcionarle", "proporcion"},
		{"proporcionarles", "proporcion"},
		{"proporcione", "proporcion"},
		{"proporcionen", "proporcion"},
		{"proporciones", "proporcion"},
		{"proporcionó", "proporcion"},
		{"proposición", "proposicion"},
		{"propositivas", "proposit"},
		{"propósito", "proposit"},
		{"propósitos", "proposit"},
		{"propuesta", "propuest"},
		{"propuestas", "propuest"},
		{"propuesto", "propuest"},
		{"propuestos", "propuest"},
		{"propulsor", "propulsor"},
		{"propuse", "propus"},
		{"propusieron", "propus"},
		{"propuso", "propus"},
		{"prórroga", "prorrog"},
		{"prorrogada", "prorrog"},
		{"prorrumpe", "prorrump"},
		{"prosa", "pros"},
		{"prose", "pros"},
		{"proseguid", "prosegu"},
		{"proseguir", "prosegu"},
		{"proselitismo", "proselit"},
		{"prosiga", "prosig"},
		{"prosigue", "prosig"},
		{"prosiguió", "prosigu"},
		{"prosinecky", "prosinecky"},
		{"prospectiva", "prospect"},
		{"prospecto", "prospect"},
		{"prospectos", "prospect"},
		{"prosperado", "prosper"},
		{"prosperan", "prosper"},
		{"prospere", "prosper"},
		{"prosperidades", "prosper"},
		{"protagónicos", "protagon"},
		{"protagonismo", "protagon"},
		{"protagonista", "protagon"},
		{"protagonistas", "protagon"},
		{"protagonizado", "protagoniz"},
		{"protagonizados", "protagoniz"},
		{"protagonizará", "protagoniz"},
		{"protagonizaron", "protagoniz"},
		{"protagonizó", "protagoniz"},
		{"protección", "proteccion"},
		{"proteccionistas", "proteccion"},
		{"protective", "protectiv"},
		{"protecto", "protect"},
		{"protector", "protector"},
		{"protectora", "protector"},
		{"protectorado", "protector"},
		{"protectores", "protector"},
		{"protege", "proteg"},
		{"protegen", "proteg"},
		{"proteger", "proteg"},
		{"protegerán", "proteg"},
		{"protegerían", "proteg"},
		{"protegerse", "proteg"},
		{"protegía", "proteg"},
		{"protegido", "proteg"},
		{"protegidos", "proteg"},
		{"protegieron", "proteg"},
		{"protein", "protein"},
		{"proteína", "protein"},
		{"proteínas", "protein"},
		{"proteja", "protej"},
		{"protejan", "protej"},
		{"protesta", "protest"},
		{"protestada", "protest"},
		{"protestan", "protest"},
		{"protestantes", "protest"},
		{"protestar", "protest"},
		{"protestaron", "protest"},
		{"protestas", "protest"},
		{"proteste", "protest"},
		{"protesten", "protest"},
		{"protestó", "protest"},
		{"protocolo", "protocol"},
		{"prototipo", "prototip"},
		{"prototipos", "prototip"},
		{"provea", "prove"},
		{"provecho", "provech"},
		{"provee", "prove"},
		{"proveedor", "proveedor"},
		{"proveedores", "proveedor"},
		{"proveeduría", "proveedur"},
		{"proveer", "prov"},
		{"provenía", "proven"},
		{"proveniente", "provenient"},
		{"provenientes", "provenient"},
		{"proverbio", "proverbi"},
		{"proviene", "provien"},
		{"provienen", "provien"},
		{"provincia", "provinci"},
		{"provincial", "provincial"},
		{"provinciales", "provincial"},
		{"provinciana", "provincian"},
		{"provincias", "provinci"},
		{"provisional", "provisional"},
		{"provisionales", "provisional"},
		{"provisiones", "provision"},
		{"provoca", "provoc"},
		{"provocaba", "provoc"},
		{"provocación", "provoc"},
		{"provocado", "provoc"},
		{"provocador", "provoc"},
		{"provocados", "provoc"},
		{"provocan", "provoc"},
		{"provocando", "provoc"},
		{"provocar", "provoc"},
		{"provocará", "provoc"},
		{"provocaran", "provoc"},
		{"provocarán", "provoc"},
		{"provocaron", "provoc"},
		{"provocativa", "provoc"},
		{"provocó", "provoc"},
		{"provoque", "provoqu"},
		{"provoquen", "provoqu"},
		{"provotel", "provotel"},
		{"próxima", "proxim"},
		{"próximamente", "proxim"},
		{"próximas", "proxim"},
		{"próximo", "proxim"},
		{"próximos", "prox"},
		{"proyección", "proyeccion"},
		{"proyecta", "proyect"},
		{"proyectaba", "proyect"},
		{"proyectaban", "proyect"},
		{"proyectada", "proyect"},
		{"proyectado", "proyect"},
		{"proyectan", "proyect"},
		{"proyectar", "proyect"},
		{"proyectará", "proyect"},
		{"proyectarse", "proyect"},
		{"proyectil", "proyectil"},
		{"proyectito", "proyectit"},
		{"proyecto", "proyect"},
		{"proyectó", "proyect"},
		{"proyectos", "proyect"},
		{"prudente", "prudent"},
		{"prudentemente", "prudent"},
		{"prudentes", "prudent"},
		{"prueba", "prueb"},
		{"pruebas", "prueb"},
		{"pruebita", "pruebit"},
		{"pruneda", "pruned"},
		{"prusia", "prusi"},
		{"prusiana", "prusian"},
		{"prusianos", "prusian"},
		{"ps", "ps"},
		{"pseudolíderes", "pseudolider"},
		{"psicoanalista", "psicoanal"},
		{"psicoanalítica", "psicoanalit"},
		{"psicoanalítico", "psicoanalit"},
		{"psicóloga", "psicolog"},
		{"psicólogas", "psicolog"},
		{"psicología", "psicolog"},
		{"psicológica", "psicolog"},
		{"psicológicamente", "psicolog"},
		{"psicológicas", "psicolog"},
		{"psicológico", "psicolog"},
		{"psicológicos", "psicolog"},
		{"psicólogo", "psicolog"},
		{"psicoterapéutico", "psicoterapeut"},
		{"psicoterapia", "psicoterapi"},
		{"psicotrópicas", "psicotrop"},
		{"psicotrópicos", "psicotrop"},
		{"psique", "psiqu"},
		{"psiquiátrico", "psiquiatr"},
		{"psíquico", "psiquic"},
		{"psoriasis", "psoriasis"},
		{"pte", "pte"},
		{"púber", "pub"},
		{"public", "public"},
		{"publica", "public"},
		{"pública", "public"},
		{"publicación", "public"},
		{"publicaciones", "public"},
		{"publicada", "public"},
		{"publicadas", "public"},
		{"publicado", "public"},
		{"públicamente", "public"},
		{"publican", "public"},
		{"publicando", "public"},
		{"publicar", "public"},
		{"publicará", "public"},
		{"publicaron", "public"},
		{"públicas", "public"},
		{"publicidad", "public"},
		{"publicistas", "public"},
		{"publicitada", "publicit"},
		{"publicitaria", "publicitari"},
		{"publicitario", "publicitari"},
		{"público", "public"},
		{"publicó", "public"},
		{"públicos", "public"},
		{"publio", "publi"},
		{"publique", "publiqu"},
		{"publishing", "publishing"},
		{"puche", "puch"},
		{"pude", "pud"},
		{"pudiendo", "pud"},
		{"pudiera", "pud"},
		{"pudieran", "pud"},
		{"pudieron", "pud"},
		{"pudiese", "pud"},
		{"pudimos", "pud"},
		{"pudo", "pud"},
		{"pudor", "pudor"},
		{"pue", "pue"},
		{"puebla", "puebl"},
		{"pueblerina", "pueblerin"},
		{"pueblerino", "pueblerin"},
		{"pueblito", "pueblit"},
		{"pueblo", "puebl"},
		{"pueblos", "puebl"},
		{"pueda", "pued"},
		{"puedan", "pued"},
		{"puede", "pued"},
		{"pueden", "pued"},
		{"puedes", "pued"},
		{"puedo", "pued"},
		{"puente", "puent"},
		{"puentes", "puent"},
		{"puerta", "puert"},
		{"puertas", "puert"},
		{"puerto", "puert"},
		{"puertorriqueño", "puertorriqueñ"},
		{"puertos", "puert"},
		{"pues", "pues"},
		{"puesta", "puest"},
		{"puestas", "puest"},
		{"puesteros", "puester"},
		{"puesto", "puest"},
		{"puestos", "puest"},
		{"púgil", "pugil"},
		{"pugilismo", "pugil"},
		{"pugilista", "pugil"},
		{"pugilística", "pugilist"},
		{"pugnan", "pugn"},
		{"pugnando", "pugn"},
		{"pugnará", "pugn"},
		{"pugnaré", "pugn"},
		{"pugnas", "pugn"},
		{"puig", "puig"},
		{"pujante", "pujant"},
		{"pulcro", "pulcr"},
		{"pulgadas", "pulg"},
		{"pulgar", "pulg"},
		{"pulido", "pul"},
		{"pulimos", "pul"},
		{"pulir", "pul"},
		{"pulirlos", "pul"},
		{"pulitzer", "pulitz"},
		{"pulmonares", "pulmonar"},
		{"pulsar", "puls"},
		{"pulso", "puls"},
		{"pululan", "pulul"},
		{"pulverizado", "pulveriz"},
		{"pulverizadora", "pulveriz"},
		{"pulverizar", "pulveriz"},
		{"pulverizó", "pulveriz"},
		{"pum", "pum"},
		{"pumas", "pum"},
		{"pump", "pump"},
		{"puñado", "puñ"},
		{"puñaladas", "puñal"},
		{"puñales", "puñal"},
		{"punk", "punk"},
		{"puño", "puñ"},
		{"punongbayan", "punongbay"},
		{"punta", "punt"},
		{"puntadón", "puntadon"},
		{"puntapies", "puntapi"},
		{"puntas", "punt"},
		{"punteando", "punt"},
		{"punteó", "punte"},
		{"puntería", "punt"},
		{"punterías", "punt"},
		{"puntero", "punter"},
		{"punteros", "punter"},
		{"punto", "punt"},
		{"puntos", "punt"},
		{"puntual", "puntual"},
		{"puntualizaciones", "puntualiz"},
		{"puntualizó", "puntualiz"},
		{"puntualmente", "puntual"},
		{"punzo", "punz"},
		{"punzocortantes", "punzocort"},
		{"punzones", "punzon"},
		{"pupila", "pupil"},
		{"pupilas", "pupil"},
		{"pupilo", "pupil"},
		{"pura", "pur"},
		{"puramente", "pur"},
		{"puras", "pur"},
		{"pureza", "purez"},
		{"purga", "purg"},
		{"purgar", "purg"},
		{"puro", "pur"},
		{"purón", "puron"},
		{"puros", "pur"},
		{"purpúreos", "purpure"},
		{"puse", "pus"},
		{"pusiera", "pus"},
		{"pusiéramos", "pus"},
		{"pusieran", "pus"},
		{"pusieron", "pus"},
		{"pusiese", "pus"},
		{"pusilánime", "pusilanim"},
		{"pusimos", "pus"},
		{"puso", "pus"},
		{"putz", "putz"},
		{"pvc", "pvc"},
		{"pyramid", "pyram"},
		{"q", "q"},
		{"qabrija", "qabrij"},
		{"qi", "qi"},
		{"que", "que"},
		{"qué", "que"},
		{"quebec", "quebec"},
		{"quebradas", "quebr"},
		{"quebrado", "quebr"},
		{"quebrados", "quebr"},
		{"quebrantar", "quebrant"},
		{"quebrar", "quebr"},
		{"quebraron", "quebr"},
		{"quebró", "quebr"},
		{"queda", "qued"},
		{"quedaba", "qued"},
		{"quedaban", "qued"},
		{"quedado", "qued"},
		{"quedamos", "qued"},
		{"quedan", "qued"},
		{"quedando", "qued"},
		{"quedándoles", "qued"},
		{"quedar", "qued"},
		{"quedara", "qued"},
		{"quedará", "qued"},
		{"quedarán", "qued"},
		{"quedaría", "qued"},
		{"quedaríamos", "qued"},
		{"quedarnos", "qued"},
		{"quedaron", "qued"},
		{"quedarse", "qued"},
		{"quedas", "qued"},
		{"quede", "qued"},
		{"queden", "qued"},
		{"quedes", "qued"},
		{"quedó", "qued"},
		{"quehacer", "quehac"},
		{"quehaceres", "quehacer"},
		{"queja", "quej"},
		{"quejaba", "quej"},
		{"quejado", "quej"},
		{"quejamos", "quej"},
		{"quejan", "quej"},
		{"quejando", "quej"},
		{"quejándose", "quej"},
		{"quejaron", "quej"},
		{"quejas", "quej"},
		{"quejó", "quej"},
		{"quejoso", "quejos"},
		{"quejosos", "quejos"},
		{"quela", "quel"},
		{"quelonio", "queloni"},
		{"quemacocos", "quemacoc"},
		{"quemado", "quem"},
		{"quemaduras", "quemadur"},
		{"quemas", "quem"},
		{"quemó", "quem"},
		{"querella", "querell"},
		{"queremos", "quer"},
		{"querendas", "querend"},
		{"querer", "quer"},
		{"quererlo", "quer"},
		{"queretanos", "queretan"},
		{"queretaro", "queretar"},
		{"querétaro", "queretar"},
		{"querétaroi", "queretaroi"},
		{"quería", "quer"},
		{"queríamos", "quer"},
		{"querían", "quer"},
		{"queridas", "quer"},
		{"querido", "quer"},
		{"queriendo", "quer"},
		{"querrá", "querr"},
		{"querrían", "querr"},
		{"queso", "ques"},
		{"quesos", "ques"},
		{"quesque", "quesqu"},
		{"quezada", "quez"},
		{"quiebra", "quiebr"},
		{"quien", "qui"},
		{"quién", "quien"},
		{"quienes", "quien"},
		{"quiénes", "quien"},
		{"quiera", "quier"},
		{"quieran", "quier"},
		{"quieras", "quier"},
		{"quiere", "quier"},
		{"quieren", "quier"},
		{"quieres", "quier"},
		{"quiero", "quier"},
		{"quiet", "quiet"},
		{"quihubo", "quihub"},
		{"quijano", "quijan"},
		{"quijote", "quijot"},
		{"química", "quimic"},
		{"químicas", "quimic"},
		{"químico", "quimic"},
		{"quince", "quinc"},
		{"quinceañero", "quinceañer"},
		{"quincena", "quincen"},
		{"quincenales", "quincenal"},
		{"quinielas", "quiniel"},
		{"quinta", "quint"},
		{"quintana", "quintan"},
		{"quintanilla", "quintanill"},
		{"quinteto", "quintet"},
		{"quintetos", "quintet"},
		{"quinto", "quint"},
		{"quiroga", "quirog"},
		{"quiroz", "quiroz"},
		{"quirúrgica", "quirurg"},
		{"quirúrgicamente", "quirurg"},
		{"quise", "quis"},
		{"quisiera", "quis"},
		{"quisiéramos", "quis"},
		{"quisieran", "quis"},
		{"quisieron", "quis"},
		{"quiso", "quis"},
		{"quisquillosos", "quisquill"},
		{"quita", "quit"},
		{"quitaba", "quit"},
		{"quitadme", "quitadm"},
		{"quitado", "quit"},
		{"quitan", "quit"},
		{"quitando", "quit"},
		{"quitándole", "quit"},
		{"quitándolos", "quit"},
		{"quitándose", "quit"},
		{"quitar", "quit"},
		{"quitara", "quit"},
		{"quitarían", "quit"},
		{"quitarle", "quit"},
		{"quitarles", "quit"},
		{"quitarnos", "quit"},
		{"quitaron", "quit"},
		{"quitarse", "quit"},
		{"quite", "quit"},
		{"quiten", "quit"},
		{"quitó", "quit"},
		{"quitos", "quit"},
		{"quizá", "quiz"},
		{"quizás", "quizas"},
		{"quo", "quo"},
		{"qwarge", "qwarg"},
		{"r", "r"},
		{"rabaté", "rabat"},
		{"rabatté", "rabatt"},
		{"rabbit", "rabbit"},
		{"rabia", "rabi"},
		{"rabiar", "rabi"},
		{"rabino", "rabin"},
		{"rabo", "rab"},
		{"racha", "rach"},
		{"rachas", "rach"},
		{"racial", "racial"},
		{"raciales", "racial"},
		{"racimo", "racim"},
		{"racing", "racing"},
		{"raciocinios", "raciocini"},
		{"radar", "rad"},
		{"radares", "radar"},
		{"radiación", "radiacion"},
		{"radiaciones", "radiacion"},
		{"radiador", "radiador"},
		{"radial", "radial"},
		{"radiales", "radial"},
		{"radiante", "radiant"},
		{"radiantes", "radiant"},
		{"radica", "radic"},
		{"radicada", "radic"},
		{"radicado", "radic"},
		{"radical", "radical"},
		{"radicalismo", "radical"},
		{"radicalización", "radicaliz"},
		{"radicalmente", "radical"},
		{"radicar", "radic"},
		{"radio", "radi"},
		{"radioactiva", "radioact"},
		{"radioactividad", "radioact"},
		{"radioescucha", "radioescuch"},
		{"radiofónica", "radiofon"},
		{"radiofónico", "radiofon"},
		{"radiografías", "radiograf"},
		{"radiología", "radiolog"},
		{"radios", "radi"},
		{"radko", "radk"},
		{"rads", "rads"},
		{"rafael", "rafael"},
		{"ráfagas", "rafag"},
		{"rafi", "rafi"},
		{"raí", "rai"},
		{"raíces", "raic"},
		{"raiz", "raiz"},
		{"raíz", "raiz"},
		{"rajado", "raj"},
		{"rajko", "rajk"},
		{"rajmunel", "rajmunel"},
		{"rakenel", "rakenel"},
		{"ram", "ram"},
		{"rama", "ram"},
		{"ramana", "raman"},
		{"ramas", "ram"},
		{"ramificaciones", "ramif"},
		{"ramificado", "ramific"},
		{"ramírez", "ramirez"},
		{"ramiro", "ramir"},
		{"ramo", "ram"},
		{"ramon", "ramon"},
		{"ramón", "ramon"},
		{"ramos", "ram"},
		{"rampas", "ramp"},
		{"rams", "rams"},
		{"ranchera", "rancher"},
		{"rancheras", "rancher"},
		{"rancherita", "rancherit"},
		{"rancho", "ranch"},
		{"rangel", "rangel"},
		{"rango", "rang"},
		{"rangos", "rang"},
		{"rano", "ran"},
		{"ranqueadas", "ranqu"},
		{"ranulfo", "ranulf"},
		{"rap", "rap"},
		{"rapa", "rap"},
		{"rapamontes", "rapamont"},
		{"rapar", "rap"},
		{"raphael", "raphael"},
		{"rápida", "rap"},
		{"rápidamente", "rapid"},
		{"rápidas", "rap"},
		{"rapidez", "rapidez"},
		{"rápido", "rap"},
		{"rápidos", "rap"},
		{"rapport", "rapport"},
		{"raquet", "raquet"},
		{"raquítica", "raquit"},
		{"raquítico", "raquit"},
		{"rara", "rar"},
		{"raraku", "raraku"},
		{"raro", "rar"},
		{"raros", "rar"},
		{"rasca", "rasc"},
		{"rascacielos", "rascaciel"},
		{"rascón", "rascon"},
		{"rasero", "raser"},
		{"rasgaría", "rasg"},
		{"rasgos", "rasg"},
		{"rasmussen", "rasmuss"},
		{"raspar", "rasp"},
		{"rastrearlas", "rastr"},
		{"rastro", "rastr"},
		{"rateros", "rater"},
		{"ratificación", "ratif"},
		{"ratificado", "ratific"},
		{"ratificar", "ratific"},
		{"ratificará", "ratific"},
		{"ratifico", "ratif"},
		{"ratificó", "ratific"},
		{"rating", "rating"},
		{"ratito", "ratit"},
		{"rato", "rat"},
		{"raton", "raton"},
		{"ratón", "raton"},
		{"ratoncito", "ratoncit"},
		{"ratones", "raton"},
		{"ratos", "rat"},
		{"rauda", "raud"},
		{"raúl", "raul"},
		{"raulito", "raulit"},
		{"ravel", "ravel"},
		{"ravines", "ravin"},
		{"ray", "ray"},
		{"raya", "ray"},
		{"rayada", "ray"},
		{"rayaditos", "rayadit"},
		{"rayado", "ray"},
		{"rayados", "ray"},
		{"rayando", "ray"},
		{"rayas", "ray"},
		{"rayen", "ray"},
		{"raymonde", "raymond"},
		{"raymundo", "raymund"},
		{"rayo", "ray"},
		{"rayón", "rayon"},
		{"rayones", "rayon"},
		{"rayos", "ray"},
		{"raytheon", "raytheon"},
		{"raza", "raz"},
		{"razón", "razon"},
		{"razonable", "razon"},
		{"razonablemente", "razon"},
		{"razonables", "razon"},
		{"razonada", "razon"},
		{"razonamiento", "razon"},
		{"razonan", "razon"},
		{"razones", "razon"},
		{"razos", "raz"},
		{"re", "re"},
		{"reabierta", "reabiert"},
		{"reabra", "reabr"},
		{"reabrir", "reabr"},
		{"reabrirse", "reabr"},
		{"reacaudarían", "reacaud"},
		{"reacción", "reaccion"},
		{"reaccionar", "reaccion"},
		{"reaccionaria", "reaccionari"},
		{"reaccionarios", "reaccionari"},
		{"reacciones", "reaccion"},
		{"reaccionó", "reaccion"},
		{"reacomodados", "reacomod"},
		{"reactivar", "reactiv"},
		{"reactivarán", "reactiv"},
		{"reactor", "reactor"},
		{"readaptación", "readapt"},
		{"readecuación", "readecu"},
		{"reafirmar", "reafirm"},
		{"reafirmaron", "reafirm"},
		{"reafirmé", "reafirm"},
		{"reafirmó", "reafirm"},
		{"reagan", "reag"},
		{"reajustados", "reajust"},
		{"reajustarlas", "reajust"},
		{"reajuste", "reajust"},
		{"real", "real"},
		{"realce", "realc"},
		{"reales", "real"},
		{"realeza", "realez"},
		{"realice", "realic"},
		{"realicé", "realic"},
		{"realicen", "realic"},
		{"realidad", "realid"},
		{"realidades", "realidad"},
		{"realismo", "realism"},
		{"realista", "realist"},
		{"realistas", "realist"},
		{"realísticamente", "realist"},
		{"realiza", "realiz"},
		{"realizaba", "realiz"},
		{"realizaban", "realiz"},
		{"realización", "realiz"},
		{"realizada", "realiz"},
		{"realizadas", "realiz"},
		{"realizado", "realiz"},
		{"realizador", "realiz"},
		{"realizados", "realiz"},
		{"realizamos", "realiz"},
		{"realizan", "realiz"},
		{"realizando", "realiz"},
		{"realizar", "realiz"},
		{"realizara", "realiz"},
		{"realizará", "realiz"},
		{"realizarán", "realiz"},
		{"realizaremos", "realiz"},
		{"realizarían", "realiz"},
		{"realizarlo", "realiz"},
		{"realizarnos", "realiz"},
		{"realizaron", "realiz"},
		{"realizarse", "realiz"},
		{"realizó", "realiz"},
		{"realmente", "realment"},
		{"realziados", "realzi"},
		{"reanimar", "reanim"},
		{"reanudándose", "reanud"},
		{"reanudar", "reanud"},
		{"reanudarán", "reanud"},
		{"reanudó", "reanud"},
		{"reaparecer", "reaparec"},
		{"reaparecerá", "reaparec"},
		{"reapareció", "reaparec"},
		{"reaparezca", "reaparezc"},
		{"reaparición", "reaparicion"},
		{"reaprehensión", "reaprehension"},
		{"reavalúo", "reavalu"},
		{"rebajar", "rebaj"},
		{"rebaño", "rebañ"},
		{"rebasa", "rebas"},
		{"rebasada", "rebas"},
		{"rebasado", "rebas"},
		{"rebasados", "rebas"},
		{"rebasan", "rebas"},
		{"rebasar", "rebas"},
		{"rebasará", "rebas"},
		{"rebasaron", "rebas"},
		{"rebasen", "reb"},
		{"rebasó", "rebas"},
		{"rebeca", "rebec"},
		{"rebelar", "rebel"},
		{"rebelde", "rebeld"},
		{"rebeldes", "rebeld"},
		{"rebelión", "rebelion"},
		{"rebobinar", "rebobin"},
		{"rebolledo", "rebolled"},
		{"rebonato", "rebonat"},
		{"rebordados", "rebord"},
		{"rebotará", "rebot"},
		{"rebotarán", "rebot"},
		{"rebote", "rebot"},
		{"recaba", "rec"},
		{"recaban", "rec"},
		{"recado", "rec"},
		{"recados", "rec"},
		{"recalcar", "recalc"},
		{"recalcitrante", "recalcitr"},
		{"recalcó", "recalc"},
		{"recalibren", "recalibr"},
		{"recámara", "recam"},
		{"recapacite", "recapacit"},
		{"recapitalización", "recapitaliz"},
		{"recargaron", "recarg"},
		{"recargó", "recarg"},
		{"recarpetear", "recarpet"},
		{"recarpeteo", "recarpete"},
		{"recaudaban", "recaud"},
		{"recaudando", "recaud"},
		{"recaudar", "recaud"},
		{"recaudarían", "recaud"},
		{"recaudaron", "recaud"},
		{"recaude", "recaud"},
		{"recaudó", "recaud"},
		{"rece", "rec"},
		{"recelar", "recel"},
		{"recelos", "recel"},
		{"recepción", "recepcion"},
		{"receptor", "receptor"},
		{"receptores", "receptor"},
		{"receptoría", "receptor"},
		{"recesión", "recesion"},
		{"recesiones", "recesion"},
		{"recesivo", "reces"},
		{"receso", "reces"},
		{"receta", "recet"},
		{"rechace", "rechac"},
		{"rechaza", "rechaz"},
		{"rechazada", "rechaz"},
		{"rechazadas", "rechaz"},
		{"rechazamos", "rechaz"},
		{"rechazan", "rechaz"},
		{"rechazando", "rechaz"},
		{"rechazar", "rechaz"},
		{"rechazaron", "rechaz"},
		{"rechazo", "rechaz"},
		{"rechazó", "rechaz"},
		{"rechifla", "rechifl"},
		{"rechiflas", "rechifl"},
		{"rechinan", "rechin"},
		{"rechinar", "rechin"},
		{"reciba", "recib"},
		{"reciban", "recib"},
		{"recibe", "recib"},
		{"reciben", "recib"},
		{"recibía", "recib"},
		{"recibían", "recib"},
		{"recibida", "recib"},
		{"recibidas", "recib"},
		{"recibido", "recib"},
		{"recibidos", "recib"},
		{"recibiendo", "recib"},
		{"recibiera", "recib"},
		{"recibieron", "recib"},
		{"recibimiento", "recib"},
		{"recibimos", "recib"},
		{"recibió", "recib"},
		{"recibir", "recib"},
		{"recibirá", "recib"},
		{"recibirán", "recib"},
		{"recibiría", "recib"},
		{"recibirían", "recib"},
		{"recibirla", "recib"},
		{"recibirlas", "recib"},
		{"recibirlo", "recib"},
		{"recibirnos", "recib"},
		{"recibo", "recib"},
		{"recibos", "recib"},
		{"reciclaje", "reciclaj"},
		{"reciclamos", "recicl"},
		{"reciclando", "recicl"},
		{"reciclar", "recicl"},
		{"recién", "recien"},
		{"reciente", "recient"},
		{"recientemente", "recient"},
		{"recientes", "recient"},
		{"recientísima", "recientisim"},
		{"recinto", "recint"},
		{"recintos", "recint"},
		{"recipiente", "recipient"},
		{"recíprocas", "reciproc"},
		{"reciprocidad", "reciproc"},
		{"recíproco", "reciproc"},
		{"recirculado", "recircul"},
		{"recircularon", "recircul"},
		{"recitar", "recit"},
		{"recitarle", "recit"},
		{"recitó", "recit"},
		{"reclama", "reclam"},
		{"reclamaba", "reclam"},
		{"reclamación", "reclam"},
		{"reclamadas", "reclam"},
		{"reclamados", "reclam"},
		{"reclaman", "reclam"},
		{"reclamando", "reclam"},
		{"reclamar", "reclam"},
		{"reclamara", "reclam"},
		{"reclamarle", "reclam"},
		{"reclamarles", "reclam"},
		{"reclamaron", "reclam"},
		{"reclamen", "reclam"},
		{"reclamó", "reclam"},
		{"reclamos", "recl"},
		{"recluido", "reclu"},
		{"recluidos", "reclu"},
		{"reclusorio", "reclusori"},
		{"reclusorios", "reclusori"},
		{"reclutadas", "reclut"},
		{"recocido", "recoc"},
		{"recogen", "recog"},
		{"recoger", "recog"},
		{"recogerla", "recog"},
		{"recogerle", "recog"},
		{"recogida", "recog"},
		{"recogidas", "recog"},
		{"recogido", "recog"},
		{"recogidos", "recog"},
		{"recogieron", "recog"},
		{"recogió", "recog"},
		{"recolección", "recoleccion"},
		{"recolectando", "recolect"},
		{"recolector", "recolector"},
		{"recomedaciones", "recomed"},
		{"recomendación", "recomend"},
		{"recomendaciones", "recomend"},
		{"recomendadas", "recomend"},
		{"recomendado", "recomend"},
		{"recomendamos", "recomend"},
		{"recomendar", "recomend"},
		{"recomendara", "recomend"},
		{"recomendaron", "recomend"},
		{"recomendó", "recomend"},
		{"recomienda", "recomiend"},
		{"recomiendan", "recomiend"},
		{"recomiendo", "recom"},
		{"recompensa", "recompens"},
		{"recompra", "recompr"},
		{"reconciliacion", "reconciliacion"},
		{"reconciliación", "reconcili"},
		{"reconcilió", "reconcil"},
		{"reconducción", "reconduccion"},
		{"reconformación", "reconform"},
		{"reconfortante", "reconfort"},
		{"reconoce", "reconoc"},
		{"reconocemos", "reconoc"},
		{"reconocer", "reconoc"},
		{"reconocerlas", "reconoc"},
		{"reconocerlos", "reconoc"},
		{"reconocerse", "reconoc"},
		{"reconocía", "reconoc"},
		{"reconocida", "reconoc"},
		{"reconocidamente", "reconocid"},
		{"reconocidas", "reconoc"},
		{"reconocido", "reconoc"},
		{"reconocidos", "reconoc"},
		{"reconociendo", "reconoc"},
		{"reconocieron", "reconoc"},
		{"reconocimiento", "reconoc"},
		{"reconocimientos", "reconoc"},
		{"reconoció", "reconoc"},
		{"reconozca", "reconozc"},
		{"reconozcan", "reconozc"},
		{"reconozco", "reconozc"},
		{"reconsidere", "reconsider"},
		{"reconstrucción", "reconstruccion"},
		{"reconstruir", "reconstru"},
		{"reconstruirán", "reconstru"},
		{"reconversión", "reconversion"},
		{"recopa", "recop"},
		{"recopila", "recopil"},
		{"recopilación", "recopil"},
		{"recopiló", "recopil"},
		{"record", "record"},
		{"récord", "record"},
		{"recordaba", "record"},
		{"recordado", "record"},
		{"recordar", "record"},
		{"recordarán", "record"},
		{"recordemos", "record"},
		{"recordó", "record"},
		{"récords", "records"},
		{"recorre", "recorr"},
		{"recorrer", "recorr"},
		{"recorrida", "recorr"},
		{"recorrido", "recorr"},
		{"recorridos", "recorr"},
		{"recorrieron", "recorr"},
		{"recorrió", "recorr"},
		{"recortar", "recort"},
		{"recorte", "recort"},
		{"recortes", "recort"},
		{"recostado", "recost"},
		{"recostó", "recost"},
		{"recreación", "recreacion"},
		{"recreará", "recr"},
		{"recreativas", "recreat"},
		{"recreativo", "recreat"},
		{"recrimina", "recrimin"},
		{"recriminaba", "recrimin"},
		{"recriminados", "recrimin"},
		{"recronometraron", "recronometr"},
		{"recrudecer", "recrudec"},
		{"recrudecieron", "recrudec"},
		{"recrudeció", "recrudec"},
		{"recta", "rect"},
		{"rectas", "rect"},
		{"rectificada", "rectific"},
		{"rectificar", "rectific"},
		{"rectitud", "rectitud"},
		{"rector", "rector"},
		{"rectoría", "rector"},
		{"rectos", "rect"},
		{"recuerda", "recuerd"},
		{"recuerdan", "recuerd"},
		{"recuerde", "recuerd"},
		{"recuérdele", "recuerdel"},
		{"recuerden", "recuerd"},
		{"recuerdo", "recuerd"},
		{"recuerdos", "recuerd"},
		{"recupera", "recuper"},
		{"recuperable", "recuper"},
		{"recuperación", "recuper"},
		{"recuperado", "recuper"},
		{"recuperando", "recuper"},
		{"recuperar", "recuper"},
		{"recuperarla", "recuper"},
		{"recuperaron", "recuper"},
		{"recuperarse", "recuper"},
		{"recupere", "recuper"},
		{"recurre", "recurr"},
		{"recurrido", "recurr"},
		{"recurrieron", "recurr"},
		{"recurrir", "recurr"},
		{"recurrirá", "recurr"},
		{"recurso", "recurs"},
		{"recursos", "recurs"},
		{"red", "red"},
		{"redacción", "redaccion"},
		{"redactado", "redact"},
		{"redactaron", "redact"},
		{"redada", "red"},
		{"redefinir", "redefin"},
		{"redes", "red"},
		{"redescuentos", "redescuent"},
		{"redilas", "redil"},
		{"rediseñar", "rediseñ"},
		{"rediseños", "rediseñ"},
		{"redistribución", "redistribu"},
		{"reditúa", "reditu"},
		{"redituable", "reditu"},
		{"redobla", "redobl"},
		{"redoblado", "redobl"},
		{"redoblar", "redobl"},
		{"redonda", "redond"},
		{"redondear", "redond"},
		{"redondez", "redondez"},
		{"redondo", "redond"},
		{"rédord", "redord"},
		{"reduccion", "reduccion"},
		{"reducción", "reduccion"},
		{"reducciones", "reduccion"},
		{"reduce", "reduc"},
		{"reducen", "reduc"},
		{"reducía", "reduc"},
		{"reducida", "reduc"},
		{"reducidas", "reduc"},
		{"reducido", "reduc"},
		{"reduciendo", "reduc"},
		{"reducir", "reduc"},
		{"reducirá", "reduc"},
		{"reducirlas", "reduc"},
		{"reducirles", "reduc"},
		{"reducirse", "reduc"},
		{"redujeron", "redujeron"},
		{"redujo", "reduj"},
		{"redunde", "redund"},
		{"reduzca", "reduzc"},
		{"reduzcan", "reduzc"},
		{"reed", "reed"},
		{"reedición", "reedicion"},
		{"reeditará", "reedit"},
		{"reelección", "reeleccion"},
		{"reelecto", "reelect"},
		{"reelegido", "reeleg"},
		{"reelegirse", "reeleg"},
		{"reemplaza", "reemplaz"},
		{"reemplazable", "reemplaz"},
		{"reemplazado", "reemplaz"},
		{"reemplazados", "reemplaz"},
		{"reemplazar", "reemplaz"},
		{"reemplazo", "reemplaz"},
		{"reencarnación", "reencarn"},
		{"reencuentro", "reencuentr"},
		{"reestablecieron", "reestablec"},
		{"reestrenan", "reestren"},
		{"reestreno", "reestren"},
		{"reestructuración", "reestructur"},
		{"reestructurar", "reestructur"},
		{"reestructurará", "reestructur"},
		{"reestructure", "reestructur"},
		{"reestudiarse", "reestudi"},
		{"refacción", "refaccion"},
		{"refaccionarias", "refaccionari"},
		{"refaccionarios", "refaccionari"},
		{"refacciones", "refaccion"},
		{"referencia", "referent"},
		{"referencias", "referent"},
		{"referéndum", "referendum"},
		{"referente", "referent"},
		{"referentes", "referent"},
		{"réferi", "referi"},
		{"referí", "refer"},
		{"refería", "ref"},
		{"referían", "ref"},
		{"referidos", "refer"},
		{"referirme", "refer"},
		{"referirse", "refer"},
		{"refiera", "ref"},
		{"refiere", "refier"},
		{"refieren", "refier"},
		{"refiero", "refier"},
		{"refinación", "refin"},
		{"refinadora", "refin"},
		{"refineria", "refineri"},
		{"refinerías", "refin"},
		{"refiriéndose", "refir"},
		{"refirió", "refir"},
		{"reflectores", "reflector"},
		{"refleja", "reflej"},
		{"reflejaba", "reflej"},
		{"reflejaban", "reflej"},
		{"reflejada", "reflej"},
		{"reflejado", "reflej"},
		{"reflejan", "reflej"},
		{"reflejando", "reflej"},
		{"reflejar", "reflej"},
		{"reflejará", "reflej"},
		{"reflejo", "reflej"},
		{"reflejó", "reflej"},
		{"reflejos", "reflej"},
		{"reflexión", "reflexion"},
		{"reflexiona", "reflexion"},
		{"reflexionar", "reflexion"},
		{"reflexionen", "reflexion"},
		{"reflexiones", "reflexion"},
		{"reflexionó", "reflexion"},
		{"reforestarlas", "reforest"},
		{"reforma", "reform"},
		{"reformado", "reform"},
		{"reformar", "reform"},
		{"reformas", "reform"},
		{"reforme", "reform"},
		{"reformó", "reform"},
		{"reforzada", "reforz"},
		{"reforzadas", "reforz"},
		{"reforzado", "reforz"},
		{"reforzamiento", "reforz"},
		{"reforzando", "reforz"},
		{"reforzar", "reforz"},
		{"reforzarlas", "reforz"},
		{"reforzó", "reforz"},
		{"refractarias", "refractari"},
		{"refrán", "refran"},
		{"refranes", "refran"},
		{"refrenamiento", "refren"},
		{"refrendaron", "refrend"},
		{"refrendo", "refrend"},
		{"refrendó", "refrend"},
		{"refresca", "refresc"},
		{"refrescos", "refresc"},
		{"refriegas", "refrieg"},
		{"refuercen", "refuerc"},
		{"refuerzan", "refuerz"},
		{"refuerzo", "refuerz"},
		{"refuerzos", "refuerz"},
		{"refugiados", "refugi"},
		{"refugiar", "refugi"},
		{"refugiarse", "refugi"},
		{"refugio", "refugi"},
		{"refugió", "refug"},
		{"regadas", "reg"},
		{"regalado", "regal"},
		{"regalan", "regal"},
		{"regalando", "regal"},
		{"regalar", "regal"},
		{"regalarnos", "regal"},
		{"regalárselo", "regal"},
		{"regalárselos", "regal"},
		{"regalo", "regal"},
		{"regaló", "regal"},
		{"regalos", "regal"},
		{"regañadientes", "regañadient"},
		{"regañan", "regañ"},
		{"regaños", "regañ"},
		{"regente", "regent"},
		{"regia", "regi"},
		{"regido", "reg"},
		{"regidor", "regidor"},
		{"regidores", "regidor"},
		{"régimen", "regim"},
		{"regímenes", "regimen"},
		{"regina", "regin"},
		{"regio", "regi"},
		{"regiomontana", "regiomontan"},
		{"regiomontanas", "regiomontan"},
		{"regiomontano", "regiomontan"},
		{"regiomontanos", "regiomontan"},
		{"region", "region"},
		{"región", "region"},
		{"regional", "regional"},
		{"regionales", "regional"},
		{"regionalismo", "regional"},
		{"regiones", "region"},
		{"regios", "regi"},
		{"regiria", "regiri"},
		{"regirnos", "reg"},
		{"regirse", "reg"},
		{"registra", "registr"},
		{"registraba", "registr"},
		{"registrada", "registr"},
		{"registradas", "registr"},
		{"registrado", "registr"},
		{"registrador", "registr"},
		{"registrados", "registr"},
		{"registran", "registr"},
		{"registrandose", "registr"},
		{"registrándose", "registr"},
		{"registrar", "registr"},
		{"registrará", "registr"},
		{"registrarán", "registr"},
		{"registraron", "registr"},
		{"registrarse", "registr"},
		{"registre", "registr"},
		{"registren", "registr"},
		{"registro", "registr"},
		{"registró", "registr"},
		{"registros", "registr"},
		{"regla", "regl"},
		{"reglamentación", "reglament"},
		{"reglamentar", "reglament"},
		{"reglamentarios", "reglamentari"},
		{"reglamente", "regl"},
		{"reglamento", "reglament"},
		{"reglamentos", "reglament"},
		{"reglas", "regl"},
		{"regresa", "regres"},
		{"regresaba", "regres"},
		{"regresaban", "regres"},
		{"regresan", "regres"},
		{"regresando", "regres"},
		{"regresar", "regres"},
		{"regresara", "regres"},
		{"regresará", "regres"},
		{"regresaran", "regres"},
		{"regresarán", "regres"},
		{"regresaremos", "regres"},
		{"regresaría", "regres"},
		{"regresaron", "regres"},
		{"regresarse", "regres"},
		{"regresas", "regres"},
		{"regrese", "regres"},
		{"regreselo", "regresel"},
		{"regresen", "regres"},
		{"regreses", "regres"},
		{"regrésese", "regreses"},
		{"regresiva", "regres"},
		{"regreso", "regres"},
		{"regresó", "regres"},
		{"reguera", "reguer"},
		{"regula", "regul"},
		{"regulación", "regul"},
		{"regulaciones", "regul"},
		{"regulador", "regul"},
		{"reguladora", "regul"},
		{"reguladores", "regul"},
		{"regular", "regul"},
		{"regulares", "regular"},
		{"regularización", "regulariz"},
		{"regularizar", "regulariz"},
		{"regularizará", "regulariz"},
		{"regularizó", "regulariz"},
		{"regularmente", "regular"},
		{"regulatoria", "regulatori"},
		{"regule", "regul"},
		{"rehabilitación", "rehabilit"},
		{"rehabilitaciones", "rehabilit"},
		{"rehabilitado", "rehabilit"},
		{"rehacer", "rehac"},
		{"rehén", "rehen"},
		{"rehenes", "rehen"},
		{"rehiletes", "rehilet"},
		{"rehuir", "rehu"},
		{"rehusaban", "rehus"},
		{"rehusaron", "rehus"},
		{"reía", "rei"},
		{"reían", "rei"},
		{"reimplantó", "reimplant"},
		{"reina", "rein"},
		{"reinado", "rein"},
		{"reinante", "reinant"},
		{"reincidan", "reincid"},
		{"reincidencia", "reincident"},
		{"reincorporará", "reincorpor"},
		{"reineros", "reiner"},
		{"reingresó", "reingres"},
		{"reiniciar", "reinici"},
		{"reiniciaron", "reinici"},
		{"reino", "rein"},
		{"reinstalación", "reinstal"},
		{"reintegración", "reintegr"},
		{"reintegrar", "reintegr"},
		{"reintegrarme", "reintegr"},
		{"reinterpretar", "reinterpret"},
		{"reinvertirá", "reinvert"},
		{"reír", "reir"},
		{"reírnos", "reirn"},
		{"reírse", "reirs"},
		{"reiteración", "reiter"},
		{"reiteradamente", "reiter"},
		{"reiterará", "reiter"},
		{"reiteraron", "reiter"},
		{"reiteró", "reiter"},
		{"reivindicación", "reivind"},
		{"reivindicar", "reivindic"},
		{"reja", "rej"},
		{"rejas", "rej"},
		{"rejón", "rejon"},
		{"rejoneador", "rejon"},
		{"rejones", "rejon"},
		{"rejuvenecedores", "rejuvenecedor"},
		{"relación", "relacion"},
		{"relaciona", "relacion"},
		{"relacionada", "relacion"},
		{"relacionadas", "relacion"},
		{"relacionado", "relacion"},
		{"relacionados", "relacion"},
		{"relacionaron", "relacion"},
		{"relacionarse", "relacion"},
		{"relaciones", "relacion"},
		{"relacionó", "relacion"},
		{"relajado", "relaj"},
		{"relajarse", "relaj"},
		{"relamían", "relam"},
		{"relampaguee", "relampague"},
		{"relanzamiento", "relanz"},
		{"relata", "relat"},
		{"relatar", "relat"},
		{"relativa", "relat"},
		{"relativamente", "relat"},
		{"relativas", "relat"},
		{"relativo", "relat"},
		{"relativos", "relat"},
		{"relato", "relat"},
		{"relató", "relat"},
		{"relatos", "relat"},
		{"relegada", "releg"},
		{"relegado", "releg"},
		{"relegue", "releg"},
		{"relevado", "relev"},
		{"relevancia", "relev"},
		{"relevantes", "relev"},
		{"relevar", "relev"},
		{"relevista", "relev"},
		{"relevo", "relev"},
		{"relevó", "relev"},
		{"relevos", "relev"},
		{"relieve", "reliev"},
		{"relieves", "reliev"},
		{"religión", "religion"},
		{"religiones", "religion"},
		{"religiosa", "religi"},
		{"religiosamente", "religi"},
		{"religiosas", "religi"},
		{"religioso", "religi"},
		{"religiosos", "religi"},
		{"rellenadas", "rellen"},
		{"rellenarán", "rellen"},
		{"relleno", "rellen"},
		{"reloj", "reloj"},
		{"relojito", "relojit"},
		{"reluciente", "relucient"},
		{"remache", "remach"},
		{"remarcó", "remarc"},
		{"rematar", "remat"},
		{"remate", "remat"},
		{"remates", "remat"},
		{"rembolso", "rembols"},
		{"remediada", "remedi"},
		{"remediar", "remedi"},
		{"remedio", "remedi"},
		{"remedios", "remedi"},
		{"remineralizante", "remineraliz"},
		{"reminiscencia", "reminiscent"},
		{"remite", "remit"},
		{"remitente", "remitent"},
		{"remitidas", "remit"},
		{"remitidos", "remit"},
		{"remitieron", "remit"},
		{"remodelación", "remodel"},
		{"remodelada", "remodel"},
		{"remolques", "remolqu"},
		{"remontan", "remont"},
		{"rémoras", "remor"},
		{"remota", "remot"},
		{"remotamente", "remot"},
		{"remotas", "remot"},
		{"remoto", "remot"},
		{"remotos", "remot"},
		{"remover", "remov"},
		{"removibles", "remov"},
		{"removida", "remov"},
		{"removidos", "remov"},
		{"removió", "remov"},
		{"remueve", "remuev"},
		{"remuneración", "remuner"},
		{"renace", "renac"},
		{"renació", "renac"},
		{"renales", "renal"},
		{"renato", "renat"},
		{"rencillas", "rencill"},
		{"renconformación", "renconform"},
		{"rencor", "rencor"},
		{"rencores", "rencor"},
		{"rencorosos", "rencor"},
		{"rendía", "rend"},
		{"rendido", "rend"},
		{"rendimiento", "rendimient"},
		{"rendimientos", "rendimient"},
		{"rendir", "rend"},
		{"rendirse", "rend"},
		{"rendón", "rendon"},
		{"rené", "ren"},
		{"renee", "rene"},
		{"renegar", "reneg"},
		{"renegociación", "renegoci"},
		{"renegociado", "renegoci"},
		{"renegociar", "renegoci"},
		{"renglón", "renglon"},
		{"renglones", "renglon"},
		{"reñida", "reñ"},
		{"reñidas", "reñ"},
		{"renombradas", "renombr"},
		{"renombre", "renombr"},
		{"renovables", "renov"},
		{"renovación", "renov"},
		{"renovada", "renov"},
		{"renovado", "renov"},
		{"renovadores", "renov"},
		{"renovados", "renov"},
		{"renovando", "renov"},
		{"renovar", "renov"},
		{"renovarvía", "renovarv"},
		{"renta", "rent"},
		{"rentabilidad", "rentabil"},
		{"rentable", "rentabl"},
		{"rentables", "rentabl"},
		{"rentada", "rent"},
		{"rentadas", "rent"},
		{"rentados", "rent"},
		{"rentando", "rent"},
		{"rentarlas", "rent"},
		{"rentas", "rent"},
		{"rentería", "rent"},
		{"renueve", "renuev"},
		{"renuncia", "renunci"},
		{"renunciar", "renunci"},
		{"renunciara", "renunci"},
		{"renunciarán", "renunci"},
		{"renunciaría", "renunci"},
		{"renunciaron", "renunci"},
		{"renunció", "renunc"},
		{"reo", "reo"},
		{"reordenamiento", "reorden"},
		{"reordenar", "reorden"},
		{"reordene", "reorden"},
		{"reorganice", "reorganic"},
		{"reorganización", "reorganiz"},
		{"reorganizar", "reorganiz"},
		{"reorganizarnos", "reorganiz"},
		{"reorganizarse", "reorganiz"},
		{"reos", "reos"},
		{"reparación", "repar"},
		{"reparaciones", "repar"},
		{"reparadas", "repar"},
		{"reparando", "repar"},
		{"reparar", "repar"},
		{"repararlas", "repar"},
		{"reparos", "repar"},
		{"reparte", "repart"},
		{"repartido", "repart"},
		{"repartidor", "repartidor"},
		{"repartidos", "repart"},
		{"repartiendo", "repart"},
		{"repartieron", "repart"},
		{"repartimos", "repart"},
		{"repartir", "repart"},
		{"repartirán", "repart"},
		{"repartirlos", "repart"},
		{"repartirse", "repart"},
		{"reparto", "repart"},
		{"repatriación", "repatri"},
		{"repatriados", "repatri"},
		{"repavimentación", "repaviment"},
		{"repavimentar", "repaviment"},
		{"repente", "repent"},
		{"repentina", "repentin"},
		{"repentinamente", "repentin"},
		{"repercusión", "repercusion"},
		{"repercuten", "repercut"},
		{"repertorio", "repertori"},
		{"repetición", "repeticion"},
		{"repeticiones", "repeticion"},
		{"repetidamente", "repetid"},
		{"repetidas", "repet"},
		{"repetidísimas", "repetidisim"},
		{"repetidoras", "repetidor"},
		{"repetimos", "repet"},
		{"repetir", "repet"},
		{"repetira", "repetir"},
		{"repetirá", "repet"},
		{"repetirán", "repet"},
		{"repetiría", "repet"},
		{"repetirse", "repet"},
		{"repita", "repit"},
		{"repitan", "repit"},
		{"repite", "repit"},
		{"repitiéndome", "repit"},
		{"repitió", "repit"},
		{"repito", "repit"},
		{"repleta", "replet"},
		{"repleto", "replet"},
		{"replica", "replic"},
		{"réplica", "replic"},
		{"replicar", "replic"},
		{"replicarle", "replic"},
		{"réplicas", "replic"},
		{"repollos", "repoll"},
		{"reponen", "repon"},
		{"reponer", "repon"},
		{"reporta", "report"},
		{"reportadas", "report"},
		{"reportado", "report"},
		{"reportados", "report"},
		{"reportaje", "reportaj"},
		{"reportan", "report"},
		{"reportar", "report"},
		{"reportara", "report"},
		{"reportarán", "report"},
		{"reportaron", "report"},
		{"reporte", "report"},
		{"reportera", "reporter"},
		{"reportero", "reporter"},
		{"reporteros", "reporter"},
		{"reportes", "report"},
		{"repórtese", "reportes"},
		{"reportó", "report"},
		{"reportos", "report"},
		{"reposo", "repos"},
		{"reprendido", "reprend"},
		{"represalias", "represali"},
		{"representa", "represent"},
		{"representaba", "represent"},
		{"representación", "represent"},
		{"representada", "represent"},
		{"representadas", "represent"},
		{"representado", "represent"},
		{"representan", "represent"},
		{"representando", "represent"},
		{"representante", "represent"},
		{"representantes", "represent"},
		{"representar", "represent"},
		{"representará", "represent"},
		{"representarán", "represent"},
		{"representaría", "represent"},
		{"representativas", "represent"},
		{"representatividad", "representat"},
		{"representativo", "represent"},
		{"representativos", "represent"},
		{"representen", "represent"},
		{"representó", "represent"},
		{"represión", "represion"},
		{"represivo", "repres"},
		{"represivos", "repres"},
		{"represores", "represor"},
		{"reprimir", "reprim"},
		{"reprobadas", "reprob"},
		{"reprobado", "reprob"},
		{"reprobaron", "reprob"},
		{"reprobó", "reprob"},
		{"reprocha", "reproch"},
		{"reproches", "reproch"},
		{"reprochó", "reproch"},
		{"reproducción", "reproduccion"},
		{"reproducciones", "reproduccion"},
		{"reproduce", "reproduc"},
		{"reproducen", "reproduc"},
		{"reproducir", "reproduc"},
		{"reproducirse", "reproduc"},
		{"reproductor", "reproductor"},
		{"reprodujo", "reproduj"},
		{"reproduzca", "reproduzc"},
		{"reprueba", "reprueb"},
		{"reprueban", "reprueb"},
		{"reptiles", "reptil"},
		{"república", "republ"},
		{"republicanas", "republican"},
		{"republicano", "republican"},
		{"repúblicanos", "republican"},
		{"repúblicas", "republ"},
		{"repudiando", "repudi"},
		{"repudio", "repudi"},
		{"repuesto", "repuest"},
		{"repugnancia", "repugn"},
		{"repuntando", "repunt"},
		{"repunte", "repunt"},
		{"reputación", "reput"},
		{"requería", "requ"},
		{"requerida", "requer"},
		{"requerido", "requer"},
		{"requeridos", "requer"},
		{"requerimiento", "requer"},
		{"requerimientos", "requer"},
		{"requerimos", "requer"},
		{"requerirá", "requer"},
		{"requerirán", "requer"},
		{"requiera", "requ"},
		{"requieran", "requ"},
		{"requiere", "requier"},
		{"requieren", "requier"},
		{"requinto", "requint"},
		{"requirieran", "requir"},
		{"requirió", "requir"},
		{"requisito", "requisit"},
		{"requisitos", "requisit"},
		{"resalta", "resalt"},
		{"resaltando", "resalt"},
		{"resaltar", "resalt"},
		{"resaltaron", "resalt"},
		{"resalte", "resalt"},
		{"resaltó", "resalt"},
		{"resarcirán", "resarc"},
		{"resbaladero", "resbalader"},
		{"resbaladizo", "resbaladiz"},
		{"resbalosas", "resbal"},
		{"rescata", "rescat"},
		{"rescatable", "rescat"},
		{"rescatada", "rescat"},
		{"rescatadas", "rescat"},
		{"rescatado", "rescat"},
		{"rescatados", "rescat"},
		{"rescatar", "rescat"},
		{"rescate", "rescat"},
		{"rescates", "rescat"},
		{"rescindirles", "rescind"},
		{"rescindirlo", "rescind"},
		{"rescindirse", "rescind"},
		{"research", "research"},
		{"reseca", "resec"},
		{"reseña", "reseñ"},
		{"reseñas", "reseñ"},
		{"reséndez", "resendez"},
		{"resentidos", "resent"},
		{"resentimos", "resent"},
		{"reserva", "reserv"},
		{"reservaciones", "reserv"},
		{"reservadas", "reserv"},
		{"reservado", "reserv"},
		{"reservados", "reserv"},
		{"reservar", "reserv"},
		{"reservas", "reserv"},
		{"resguardado", "resguard"},
		{"resguardando", "resguard"},
		{"resguardarlo", "resguard"},
		{"reside", "resid"},
		{"residen", "resid"},
		{"residencia", "resident"},
		{"residencial", "residencial"},
		{"residenciales", "residencial"},
		{"residencias", "resident"},
		{"residente", "resident"},
		{"residentes", "resident"},
		{"residió", "resid"},
		{"residuales", "residual"},
		{"residuos", "residu"},
		{"resiente", "resient"},
		{"resignación", "resign"},
		{"resignada", "resign"},
		{"resignado", "resign"},
		{"resigno", "resign"},
		{"resillas", "resill"},
		{"resintiendo", "resint"},
		{"resintiera", "resint"},
		{"resiste", "res"},
		{"resistencia", "resistent"},
		{"resistente", "resistent"},
		{"resistentes", "resistent"},
		{"resistidas", "resist"},
		{"resistiera", "resist"},
		{"resistió", "resist"},
		{"resolucion", "resolucion"},
		{"resolución", "resolu"},
		{"resoluciones", "resolu"},
		{"resolver", "resolv"},
		{"resolverá", "resolv"},
		{"resolverla", "resolv"},
		{"resolverlas", "resolv"},
		{"resolverlo", "resolv"},
		{"resolverse", "resolv"},
		{"resolviendo", "resolv"},
		{"resolviéndole", "resolv"},
		{"resolvieron", "resolv"},
		{"resolvió", "resolv"},
		{"resonante", "reson"},
		{"resortes", "resort"},
		{"respalda", "respald"},
		{"respaldaba", "respald"},
		{"respaldada", "respald"},
		{"respaldará", "respald"},
		{"respaldo", "respald"},
		{"respaldó", "respald"},
		{"respaldos", "respald"},
		{"respecta", "respect"},
		{"respectiva", "respect"},
		{"respectivamente", "respect"},
		{"respectivas", "respect"},
		{"respectivo", "respect"},
		{"respectivos", "respect"},
		{"respecto", "respect"},
		{"respeta", "respet"},
		{"respetaban", "respet"},
		{"respetable", "respet"},
		{"respetado", "respet"},
		{"respetados", "respet"},
		{"respetamos", "respet"},
		{"respetan", "respet"},
		{"respetando", "respet"},
		{"respetar", "respet"},
		{"respetará", "respet"},
		{"respetaron", "respet"},
		{"respetarse", "respet"},
		{"respete", "respet"},
		{"respeten", "respet"},
		{"respeto", "respet"},
		{"respetuosa", "respetu"},
		{"respetuoso", "respetu"},
		{"respiratoria", "respiratori"},
		{"respiratorias", "respiratori"},
		{"respiratorio", "respiratori"},
		{"respiro", "respir"},
		{"resplandecientes", "resplandecient"},
		{"resplandor", "resplandor"},
		{"responda", "respond"},
		{"respondan", "respond"},
		{"responde", "respond"},
		{"responden", "respond"},
		{"responder", "respond"},
		{"responderá", "respond"},
		{"responderle", "respond"},
		{"respondí", "respond"},
		{"respondido", "respond"},
		{"respondiera", "respond"},
		{"respondió", "respond"},
		{"responsabilice", "responsabilic"},
		{"responsabilidad", "respons"},
		{"responsabilidades", "respons"},
		{"responsabilizaba", "responsabiliz"},
		{"responsabilizarlo", "responsabiliz"},
		{"responsabilizaron", "responsabiliz"},
		{"responsabilizó", "responsabiliz"},
		{"responsable", "respons"},
		{"responsables", "respons"},
		{"respuesta", "respuest"},
		{"respuestas", "respuest"},
		{"resquicio", "resquici"},
		{"resta", "rest"},
		{"restablecer", "restablec"},
		{"restablecerse", "restablec"},
		{"restablezcan", "restablezc"},
		{"restan", "rest"},
		{"restante", "restant"},
		{"restantes", "restant"},
		{"restar", "rest"},
		{"restarle", "rest"},
		{"restas", "rest"},
		{"restauración", "restaur"},
		{"restaurant", "restaurant"},
		{"restaurante", "restaur"},
		{"restaurantera", "restauranter"},
		{"restaurantero", "restauranter"},
		{"restauranteros", "restauranter"},
		{"restaurantes", "restaur"},
		{"restaurarle", "restaur"},
		{"restitución", "restitu"},
		{"restituir", "restitu"},
		{"resto", "rest"},
		{"restos", "rest"},
		{"restregárselo", "restreg"},
		{"restricción", "restriccion"},
		{"restricciones", "restriccion"},
		{"restrictiva", "restrict"},
		{"restructuración", "restructur"},
		{"resuelta", "resuelt"},
		{"resuelto", "resuelt"},
		{"resueltos", "resuelt"},
		{"resuelva", "resuelv"},
		{"resuelve", "resuelv"},
		{"resuelven", "resuelv"},
		{"resulta", "result"},
		{"resultaba", "result"},
		{"resultado", "result"},
		{"resultados", "result"},
		{"resultan", "result"},
		{"resultando", "result"},
		{"resultante", "result"},
		{"resultar", "result"},
		{"resultara", "result"},
		{"resultaran", "result"},
		{"resultaría", "result"},
		{"resultaron", "result"},
		{"resulte", "result"},
		{"resulten", "result"},
		{"resulto", "result"},
		{"resultó", "result"},
		{"resumen", "resum"},
		{"resumido", "resum"},
		{"resumir", "resum"},
		{"resumirse", "resum"},
		{"resurgente", "resurgent"},
		{"resurgir", "resurg"},
		{"resurja", "resurj"},
		{"resurrección", "resurreccion"},
		{"retador", "retador"},
		{"retardatario", "retardatari"},
		{"retardo", "retard"},
		{"retazos", "retaz"},
		{"retener", "reten"},
		{"retengan", "reteng"},
		{"retenía", "reten"},
		{"retenido", "reten"},
		{"reticencia", "reticent"},
		{"reticencias", "reticent"},
		{"retiene", "retien"},
		{"retinitis", "retinitis"},
		{"retira", "retir"},
		{"retiraba", "retir"},
		{"retirada", "retir"},
		{"retirado", "retir"},
		{"retirados", "retir"},
		{"retiramos", "retir"},
		{"retirando", "retir"},
		{"retirándose", "retir"},
		{"retirar", "retir"},
		{"retirara", "retir"},
		{"retirará", "retir"},
		{"retirarán", "retir"},
		{"retiraré", "retir"},
		{"retiraría", "retir"},
		{"retirarlas", "retir"},
		{"retiraron", "retir"},
		{"retirarse", "retir"},
		{"retire", "retir"},
		{"retiro", "retir"},
		{"retiró", "retir"},
		{"retiros", "retir"},
		{"reto", "ret"},
		{"retó", "ret"},
		{"retomado", "retom"},
		{"retomados", "retom"},
		{"retomando", "retom"},
		{"retomar", "retom"},
		{"retomarán", "retom"},
		{"retomarla", "retom"},
		{"retomé", "retom"},
		{"retomo", "retom"},
		{"retomó", "retom"},
		{"retorcerse", "retorc"},
		{"retorcidos", "retorc"},
		{"retornados", "retorn"},
		{"retornaría", "retorn"},
		{"retorno", "retorn"},
		{"retornó", "retorn"},
		{"retos", "ret"},
		{"retracta", "retract"},
		{"retractado", "retract"},
		{"retractarán", "retract"},
		{"retractaré", "retract"},
		{"retrasada", "retras"},
		{"retrasado", "retras"},
		{"retrasan", "retras"},
		{"retrasando", "retras"},
		{"retrasar", "retras"},
		{"retrasará", "retras"},
		{"retrase", "retr"},
		{"retrasen", "retr"},
		{"retraso", "retras"},
		{"retrasó", "retras"},
		{"retrasos", "retras"},
		{"retratado", "retrat"},
		{"retratar", "retrat"},
		{"retratará", "retrat"},
		{"retrataron", "retrat"},
		{"retratarse", "retrat"},
		{"retrato", "retrat"},
		{"retrató", "retrat"},
		{"retratos", "retrat"},
		{"retribuido", "retribu"},
		{"retribuye", "retribu"},
		{"retroactiva", "retroact"},
		{"retroactivo", "retroact"},
		{"retroalimentan", "retroaliment"},
		{"retroceder", "retroced"},
		{"retrocedido", "retroced"},
		{"retroceso", "retroces"},
		{"retrospección", "retrospeccion"},
		{"retrospectiva", "retrospect"},
		{"retrotraerlas", "retrotr"},
		{"retrovisor", "retrovisor"},
		{"retuvieron", "retuv"},
		{"retuvo", "retuv"},
		{"reubicación", "reubic"},
		{"reubicadas", "reubic"},
		{"reubicado", "reubic"},
		{"reubicados", "reubic"},
		{"reubicar", "reubic"},
		{"reubicarlos", "reubic"},
		{"reubicarse", "reubic"},
		{"reúna", "reun"},
		{"reúnan", "reun"},
		{"reune", "reun"},
		{"reúne", "reun"},
		{"reunen", "reun"},
		{"reúnen", "reun"},
		{"reunía", "reun"},
		{"reunían", "reun"},
		{"reunido", "reun"},
		{"reunidos", "reun"},
		{"reunieron", "reun"},
		{"reunió", "reun"},
		{"reunion", "reunion"},
		{"reunión", "reunion"},
		{"reuniones", "reunion"},
		{"reunir", "reun"},
		{"reunirá", "reun"},
		{"reunirán", "reun"},
		{"reunirnos", "reun"},
		{"reunirse", "reun"},
		{"reuter", "reut"},
		{"reuters", "reuters"},
		{"reutilización", "reutiliz"},
		{"revaloración", "revalor"},
		{"revalorarlo", "revalor"},
		{"revaluándose", "revalu"},
		{"revancha", "revanch"},
		{"revela", "revel"},
		{"revelaba", "revel"},
		{"revelaban", "revel"},
		{"revelación", "revel"},
		{"revelada", "revel"},
		{"reveladas", "revel"},
		{"revelado", "revel"},
		{"revelador", "revel"},
		{"reveladora", "revel"},
		{"revelan", "revel"},
		{"revelarla", "revel"},
		{"revelaron", "revel"},
		{"reveló", "revel"},
		{"revendedores", "revendedor"},
		{"reventa", "revent"},
		{"reventado", "revent"},
		{"reventadores", "revent"},
		{"reventó", "revent"},
		{"reventón", "reventon"},
		{"reverencia", "reverent"},
		{"reverencial", "reverencial"},
		{"reverenciales", "reverencial"},
		{"reverenciarlo", "reverenci"},
		{"reverendo", "reverend"},
		{"reversa", "revers"},
		{"revertido", "revert"},
		{"revertir", "revert"},
		{"reveses", "reves"},
		{"revierten", "reviert"},
		{"revillagigedo", "revillagiged"},
		{"revisado", "revis"},
		{"revisados", "revis"},
		{"revisan", "revis"},
		{"revisando", "revis"},
		{"revisar", "revis"},
		{"revisará", "revis"},
		{"revisaran", "revis"},
		{"revisarán", "revis"},
		{"revisarla", "revis"},
		{"revisarlos", "revis"},
		{"revisaron", "revis"},
		{"revisarse", "revis"},
		{"revise", "revis"},
		{"revisen", "revis"},
		{"revisión", "revision"},
		{"revisiones", "revision"},
		{"reviso", "revis"},
		{"revista", "revist"},
		{"revistas", "revist"},
		{"revitalizar", "revitaliz"},
		{"revivir", "reviv"},
		{"revocar", "revoc"},
		{"revocó", "revoc"},
		{"revolcón", "revolcon"},
		{"revolución", "revolu"},
		{"revolucionaban", "revolucion"},
		{"revolucionar", "revolucion"},
		{"revolucionaria", "revolucionari"},
		{"revolucionarias", "revolucionari"},
		{"revolucionario", "revolucionari"},
		{"revolucionarios", "revolucionari"},
		{"revoluciones", "revolu"},
		{"revolucionó", "revolucion"},
		{"revolufia", "revolufi"},
		{"revolver", "revolv"},
		{"revólver", "revolv"},
		{"revólveres", "revolver"},
		{"revolvió", "revolv"},
		{"revuelo", "revuel"},
		{"revuelta", "revuelt"},
		{"revuelto", "revuelt"},
		{"revuelva", "revuelv"},
		{"revuelvo", "revuelv"},
		{"rey", "rey"},
		{"reyes", "rey"},
		{"reyhlander", "reyhland"},
		{"reyna", "reyn"},
		{"reynaldo", "reynald"},
		{"reynard", "reynard"},
		{"reynolds", "reynolds"},
		{"reynosa", "reynos"},
		{"reynoso", "reynos"},
		{"reza", "rez"},
		{"rezagada", "rezag"},
		{"rezago", "rezag"},
		{"rezagos", "rezag"},
		{"rezará", "rez"},
		{"rezaremos", "rez"},
		{"rezarlo", "rez"},
		{"rezarte", "rezart"},
		{"rezo", "rez"},
		{"rezonificación", "rezonif"},
		{"rhon", "rhon"},
		{"riachuelo", "riachuel"},
		{"riar", "riar"},
		{"riazor", "riazor"},
		{"rib", "rib"},
		{"ribadeneyra", "ribadeneyr"},
		{"ribbentrop", "ribbentrop"},
		{"riberas", "riber"},
		{"ribereña", "ribereñ"},
		{"ribereños", "ribereñ"},
		{"rica", "ric"},
		{"ricacho", "ricach"},
		{"ricachón", "ricachon"},
		{"ricardo", "ricard"},
		{"ricas", "ric"},
		{"richard", "richard"},
		{"richards", "richards"},
		{"richardson", "richardson"},
		{"ricky", "ricky"},
		{"rico", "ric"},
		{"ricos", "ric"},
		{"ridiculiza", "ridiculiz"},
		{"ridiculización", "ridiculiz"},
		{"ridiculizado", "ridiculiz"},
		{"ridiculizan", "ridiculiz"},
		{"ríe", "rie"},
		{"riegas", "rieg"},
		{"riego", "rieg"},
		{"rieles", "riel"},
		{"riendas", "riend"},
		{"rieron", "rieron"},
		{"ries", "ries"},
		{"riesgo", "riesg"},
		{"riesgos", "riesg"},
		{"riesgosa", "riesgos"},
		{"riesgosísimo", "riesgosisim"},
		{"rifa", "rif"},
		{"rifadas", "rif"},
		{"rifar", "rif"},
		{"rifas", "rif"},
		{"rifkin", "rifkin"},
		{"rige", "rig"},
		{"rigen", "rig"},
		{"rígidas", "rig"},
		{"rigidez", "rigidez"},
		{"rígidos", "rig"},
		{"rigoberta", "rigobert"},
		{"rigor", "rigor"},
		{"rigurosa", "rigur"},
		{"riguroso", "rigur"},
		{"rigurosos", "rigur"},
		{"rijan", "rij"},
		{"rime", "rim"},
		{"riña", "riñ"},
		{"riñas", "riñ"},
		{"rincón", "rincon"},
		{"rinconada", "rincon"},
		{"rinden", "rind"},
		{"rindiendo", "rind"},
		{"rindiera", "rind"},
		{"rindieran", "rind"},
		{"rindieron", "rind"},
		{"rindió", "rind"},
		{"riñen", "riñ"},
		{"rines", "rin"},
		{"ring", "ring"},
		{"rinker", "rink"},
		{"riñones", "riñon"},
		{"rio", "rio"},
		{"río", "rio"},
		{"rió", "rio"},
		{"riojas", "rioj"},
		{"ríos", "rios"},
		{"rippey", "rippey"},
		{"riqueza", "riquez"},
		{"riquezas", "riquez"},
		{"risa", "ris"},
		{"risas", "ris"},
		{"risca", "risc"},
		{"ríspido", "risp"},
		{"ristra", "ristr"},
		{"ristre", "ristr"},
		{"rita", "rit"},
		{"ritchie", "ritchi"},
		{"riteaid", "rite"},
		{"rítmica", "ritmic"},
		{"rítmico", "ritmic"},
		{"ritmo", "ritm"},
		{"ritmos", "ritm"},
		{"rito", "rit"},
		{"ritual", "ritual"},
		{"riva", "riv"},
		{"rivadeneira", "rivadeneir"},
		{"rival", "rival"},
		{"rivales", "rival"},
		{"rivapalacio", "rivapalaci"},
		{"river", "riv"},
		{"rivera", "river"},
		{"rivero", "river"},
		{"riverside", "riversid"},
		{"riviello", "riviell"},
		{"rizado", "riz"},
		{"rizzo", "rizz"},
		{"rkc", "rkc"},
		{"rke", "rke"},
		{"rkl", "rkl"},
		{"road", "road"},
		{"robada", "rob"},
		{"robadas", "rob"},
		{"robado", "rob"},
		{"robados", "rob"},
		{"roban", "rob"},
		{"robar", "rob"},
		{"robarle", "rob"},
		{"robe", "rob"},
		{"roben", "rob"},
		{"robert", "robert"},
		{"roberta", "robert"},
		{"roberto", "robert"},
		{"roberts", "roberts"},
		{"roble", "robl"},
		{"robledo", "robled"},
		{"robles", "robl"},
		{"robo", "rob"},
		{"robó", "rob"},
		{"robocops", "robocops"},
		{"robos", "rob"},
		{"robot", "robot"},
		{"robotizar", "robotiz"},
		{"robots", "robots"},
		{"robusta", "robust"},
		{"robustecer", "robustec"},
		{"rocaetti", "rocaetti"},
		{"rocard", "rocard"},
		{"rocas", "roc"},
		{"roce", "roc"},
		{"roces", "roc"},
		{"rocha", "roch"},
		{"roche", "roch"},
		{"rochester", "rochest"},
		{"rocío", "roci"},
		{"rock", "rock"},
		{"rockanroleros", "rockanroler"},
		{"rockeras", "rocker"},
		{"rockport", "rockport"},
		{"rocoso", "rocos"},
		{"rocosos", "rocos"},
		{"rod", "rod"},
		{"rodado", "rod"},
		{"rodaje", "rodaj"},
		{"rodal", "rodal"},
		{"rodarse", "rod"},
		{"rodarte", "rodart"},
		{"rodea", "rode"},
		{"rodeaba", "rod"},
		{"rodeado", "rod"},
		{"rodeados", "rod"},
		{"rodean", "rod"},
		{"rodear", "rod"},
		{"rodeará", "rod"},
		{"rodearon", "rod"},
		{"rodearse", "rod"},
		{"rodela", "rodel"},
		{"rodeo", "rode"},
		{"rodeos", "rode"},
		{"rodilla", "rodill"},
		{"rodillas", "rodill"},
		{"rodillera", "rodiller"},
		{"rodion", "rodion"},
		{"rodisonato", "rodisonat"},
		{"rodney", "rodney"},
		{"rodolfo", "rodolf"},
		{"rodrigo", "rodrig"},
		{"rodríguez", "rodriguez"},
		{"roedor", "roedor"},
		{"roedores", "roedor"},
		{"roel", "roel"},
		{"rogelio", "rogeli"},
		{"roger", "rog"},
		{"rogers", "rogers"},
		{"rogó", "rog"},
		{"rogozinski", "rogozinski"},
		{"roja", "roj"},
		{"rojas", "roj"},
		{"rojiblanca", "rojiblanc"},
		{"rojinegra", "rojinegr"},
		{"rojo", "roj"},
		{"rojos", "roj"},
		{"rol", "rol"},
		{"rolada", "rol"},
		{"rolando", "rol"},
		{"roll", "roll"},
		{"rolling", "rolling"},
		{"rollo", "roll"},
		{"rolls", "rolls"},
		{"rom", "rom"},
		{"roma", "rom"},
		{"román", "roman"},
		{"romana", "roman"},
		{"romance", "romanc"},
		{"romances", "romanc"},
		{"romano", "roman"},
		{"románticas", "romant"},
		{"romanticismo", "romantic"},
		{"romanticonas", "romanticon"},
		{"románticos", "romant"},
		{"romeo", "rome"},
		{"romero", "romer"},
		{"romilio", "romili"},
		{"romo", "rom"},
		{"rompe", "romp"},
		{"rompebrechas", "rompebrech"},
		{"rompen", "romp"},
		{"romper", "romp"},
		{"romperá", "romp"},
		{"romperian", "romperi"},
		{"romperles", "romp"},
		{"romperlo", "romp"},
		{"rompevientos", "rompevient"},
		{"rompiendo", "romp"},
		{"rompieron", "romp"},
		{"rompimiento", "rompimient"},
		{"rompió", "romp"},
		{"rómulo", "romul"},
		{"ronald", "ronald"},
		{"roncaba", "ronc"},
		{"ronda", "rond"},
		{"rondallas", "rondall"},
		{"rondando", "rond"},
		{"ronzón", "ronzon"},
		{"roo", "roo"},
		{"roosevelt", "roosevelt"},
		{"ropa", "rop"},
		{"ropajes", "ropaj"},
		{"roque", "roqu"},
		{"rosa", "ros"},
		{"rosales", "rosal"},
		{"rosalinda", "rosalind"},
		{"rosario", "rosari"},
		{"rosarios", "rosari"},
		{"rosas", "ros"},
		{"rosaura", "rosaur"},
		{"rosellini", "rosellini"},
		{"rosencof", "rosencof"},
		{"rosendo", "rosend"},
		{"roses", "ros"},
		{"rosibel", "rosibel"},
		{"rosilí", "rosil"},
		{"ross", "ross"},
		{"rossela", "rossel"},
		{"rossi", "rossi"},
		{"rossin", "rossin"},
		{"rossini", "rossini"},
		{"roster", "rost"},
		{"rosto", "rost"},
		{"rostro", "rostr"},
		{"rostros", "rostr"},
		{"rota", "rot"},
		{"rotación", "rotacion"},
		{"rotario", "rotari"},
		{"rotarios", "rotari"},
		{"rotativo", "rotat"},
		{"rotativos", "rotat"},
		{"rotemberg", "rotemberg"},
		{"roto", "rot"},
		{"rotonda", "rotond"},
		{"rotter", "rott"},
		{"rotuliano", "rotulian"},
		{"rotundamente", "rotund"},
		{"rotura", "rotur"},
		{"round", "round"},
		{"rounds", "rounds"},
		{"rousseau", "rousseau"},
		{"roy", "roy"},
		{"royce", "royc"},
		{"rsm", "rsm"},
		{"rsqb", "rsqb"},
		{"ru", "ru"},
		{"ruandeses", "ruandes"},
		{"rubén", "ruben"},
		{"rubí", "rub"},
		{"rubia", "rubi"},
		{"rubiales", "rubial"},
		{"rubiano", "rubian"},
		{"rubio", "rubi"},
		{"rubor", "rubor"},
		{"ruborizándose", "ruboriz"},
		{"rubro", "rubr"},
		{"rubros", "rubr"},
		{"ruco", "ruc"},
		{"rudos", "rud"},
		{"rudyard", "rudyard"},
		{"rueda", "rued"},
		{"ruedas", "rued"},
		{"ruedo", "rued"},
		{"ruega", "rueg"},
		{"ruego", "rueg"},
		{"ruelas", "ruel"},
		{"ruffo", "ruff"},
		{"rufino", "rufin"},
		{"ruge", "rug"},
		{"ruggeri", "ruggeri"},
		{"rugido", "rug"},
		{"rugir", "rug"},
		{"ruido", "ruid"},
		{"ruidosas", "ruid"},
		{"ruina", "ruin"},
		{"ruinas", "ruin"},
		{"ruindades", "ruindad"},
		{"ruisz", "ruisz"},
		{"ruiz", "ruiz"},
		{"ruíz", "ruiz"},
		{"rul", "rul"},
		{"rumania", "rumani"},
		{"rumanos", "ruman"},
		{"rumberas", "rumber"},
		{"rumbo", "rumb"},
		{"rumor", "rumor"},
		{"rumores", "rumor"},
		{"ruperto", "rupert"},
		{"ruptura", "ruptur"},
		{"rural", "rural"},
		{"rurales", "rural"},
		{"rusa", "rus"},
		{"rusas", "rus"},
		{"rusia", "rusi"},
		{"ruslan", "rusl"},
		{"ruso", "rus"},
		{"rusos", "rus"},
		{"rústica", "rustic"},
		{"rústico", "rustic"},
		{"rústicos", "rustic"},
		{"ruta", "rut"},
		{"rutas", "rut"},
		{"ruteador", "ruteador"},
		{"rutina", "rutin"},
		{"rutinario", "rutinari"},
		{"rutinas", "rutin"},
		{"rutskoi", "rutskoi"},
		{"ruvalcaba", "ruvalc"},
		{"ruy", "ruy"},
		{"s", "s"},
		{"sábado", "sab"},
		{"sábados", "sab"},
		{"sabastian", "sabasti"},
		{"sabatino", "sabatin"},
		{"sabe", "sab"},
		{"sabedores", "sabedor"},
		{"sabemos", "sab"},
		{"saben", "sab"},
		{"saber", "sab"},
		{"saberlo", "sab"},
		{"saberse", "sab"},
		{"sabes", "sab"},
		{"sabia", "sabi"},
		{"sabía", "sab"},
		{"sabíamos", "sab"},
		{"sabían", "sab"},
		{"sabidas", "sab"},
		{"sabido", "sab"},
		{"sabiduría", "sabidur"},
		{"sabiendas", "sabiend"},
		{"sabina", "sabin"},
		{"sabinas", "sabin"},
		{"sabino", "sabin"},
		{"sabio", "sabi"},
		{"sabios", "sabi"},
		{"sabor", "sabor"},
		{"saboreando", "sabor"},
		{"sabores", "sabor"},
		{"saborit", "saborit"},
		{"saborizantes", "saboriz"},
		{"sabotaje", "sabotaj"},
		{"sabrá", "sabr"},
		{"sabrán", "sabran"},
		{"sabremos", "sabr"},
		{"sabrosa", "sabros"},
		{"sabroso", "sabros"},
		{"saca", "sac"},
		{"sacada", "sac"},
		{"sacadas", "sac"},
		{"sacado", "sac"},
		{"sacamos", "sac"},
		{"sacando", "sac"},
		{"sacar", "sac"},
		{"sacara", "sac"},
		{"sacaran", "sac"},
		{"sacaremos", "sac"},
		{"sacarla", "sac"},
		{"sacarle", "sac"},
		{"sacarles", "sac"},
		{"sacarlo", "sac"},
		{"sacarme", "sac"},
		{"sacaron", "sac"},
		{"sacerdocio", "sacerdoci"},
		{"sacerdotal", "sacerdotal"},
		{"sacerdotales", "sacerdotal"},
		{"sacerdote", "sacerdot"},
		{"sacerdotes", "sacerdot"},
		{"saco", "sac"},
		{"sacó", "sac"},
		{"sacos", "sac"},
		{"sacra", "sacr"},
		{"sacramento", "sacrament"},
		{"sacrificado", "sacrific"},
		{"sacrificar", "sacrific"},
		{"sacrificarlo", "sacrific"},
		{"sacrificio", "sacrifici"},
		{"sacrificios", "sacrifici"},
		{"sacrificó", "sacrific"},
		{"sacristán", "sacristan"},
		{"sacristía", "sacrist"},
		{"sacro", "sacr"},
		{"sacs", "sacs"},
		{"sacudida", "sacud"},
		{"sacudimiento", "sacud"},
		{"sada", "sad"},
		{"sadam", "sadam"},
		{"sadomasoquistas", "sadomasoqu"},
		{"saenz", "saenz"},
		{"sáenz", "saenz"},
		{"saeta", "saet"},
		{"sagrada", "sagr"},
		{"sagradas", "sagr"},
		{"sagrado", "sagr"},
		{"sahuayo", "sahuay"},
		{"sain", "sain"},
		{"saint", "saint"},
		{"sainz", "sainz"},
		{"sáinz", "sainz"},
		{"saínz", "sainz"},
		{"sal", "sal"},
		{"sala", "sal"},
		{"salacio", "salaci"},
		{"salada", "sal"},
		{"salais", "salais"},
		{"salamandria", "salamandri"},
		{"salarial", "salarial"},
		{"salariales", "salarial"},
		{"salario", "salari"},
		{"salarios", "salari"},
		{"salas", "sal"},
		{"salaz", "salaz"},
		{"salazar", "salaz"},
		{"saldaña", "saldañ"},
		{"saldar", "sald"},
		{"saldarán", "sald"},
		{"saldívar", "saldiv"},
		{"saldo", "sald"},
		{"saldos", "sald"},
		{"saldrá", "saldr"},
		{"saldrán", "saldran"},
		{"saldría", "saldr"},
		{"sale", "sal"},
		{"salen", "sal"},
		{"salesianas", "salesian"},
		{"salga", "salg"},
		{"salgado", "salg"},
		{"salgamos", "salg"},
		{"salgan", "salg"},
		{"salgo", "salg"},
		{"salí", "sal"},
		{"salía", "sal"},
		{"salían", "sal"},
		{"salida", "sal"},
		{"salidas", "sal"},
		{"salido", "sal"},
		{"saliendo", "sal"},
		{"saliente", "salient"},
		{"saliera", "sal"},
		{"salieran", "sal"},
		{"salieron", "sal"},
		{"salimos", "sal"},
		{"salinas", "salin"},
		{"salinismo", "salin"},
		{"salinista", "salin"},
		{"salió", "sal"},
		{"salir", "sal"},
		{"salirse", "sal"},
		{"salmón", "salmon"},
		{"salmones", "salmon"},
		{"salmos", "salm"},
		{"salomon", "salomon"},
		{"salomón", "salomon"},
		{"salomónica", "salomon"},
		{"salomónico", "salomon"},
		{"salón", "salon"},
		{"salones", "salon"},
		{"salpicadas", "salpic"},
		{"salsa", "sals"},
		{"salsas", "sals"},
		{"salta", "salt"},
		{"saltándose", "salt"},
		{"saltaron", "salt"},
		{"salte", "salt"},
		{"saltillense", "saltillens"},
		{"saltillenses", "saltillens"},
		{"saltillo", "saltill"},
		{"salto", "salt"},
		{"saltó", "salt"},
		{"salubridad", "salubr"},
		{"salud", "salud"},
		{"saludable", "salud"},
		{"saludables", "salud"},
		{"saludando", "salud"},
		{"saludar", "salud"},
		{"saludarle", "salud"},
		{"saludarlo", "salud"},
		{"saludaron", "salud"},
		{"saludemos", "salud"},
		{"saludo", "salud"},
		{"saludó", "salud"},
		{"saludos", "salud"},
		{"salutación", "salut"},
		{"salva", "salv"},
		{"salvación", "salvacion"},
		{"salvado", "salv"},
		{"salvador", "salvador"},
		{"salvadoreña", "salvadoreñ"},
		{"salvadoreños", "salvadoreñ"},
		{"salvados", "salv"},
		{"salvaguardas", "salvaguard"},
		{"salvaje", "salvaj"},
		{"salvamentos", "salvament"},
		{"salvar", "salv"},
		{"salvara", "salv"},
		{"salvarle", "salv"},
		{"salvarse", "salv"},
		{"salve", "salv"},
		{"salvedades", "salvedad"},
		{"salvia", "salvi"},
		{"salvo", "salv"},
		{"salvó", "salv"},
		{"sam", "sam"},
		{"samaniego", "samanieg"},
		{"samoa", "samo"},
		{"sampetrinas", "sampetrin"},
		{"sampetrino", "sampetrin"},
		{"samuel", "samuel"},
		{"samuelito", "samuelit"},
		{"san", "san"},
		{"sana", "san"},
		{"sanchez", "sanchez"},
		{"sánchez", "sanchez"},
		{"sancho", "sanch"},
		{"sanción", "sancion"},
		{"sancionados", "sancion"},
		{"sancionan", "sancion"},
		{"sancionaría", "sancion"},
		{"sancionarlo", "sancion"},
		{"sancionarlos", "sancion"},
		{"sancionaron", "sancion"},
		{"sanciones", "sancion"},
		{"sandalio", "sandali"},
		{"sandinista", "sandin"},
		{"sandoval", "sandoval"},
		{"sandra", "sandr"},
		{"sandrine", "sandrin"},
		{"sandwich", "sandwich"},
		{"saneado", "san"},
		{"saneamiento", "saneamient"},
		{"sanear", "san"},
		{"saneara", "san"},
		{"sanearlas", "san"},
		{"sangrados", "sangr"},
		{"sangrar", "sangr"},
		{"sangre", "sangr"},
		{"sangría", "sangr"},
		{"sangrientas", "sangrient"},
		{"sangriento", "sangrient"},
		{"sanguinario", "sanguinari"},
		{"sanidad", "sanid"},
		{"sanidades", "sanidad"},
		{"sanitaria", "sanitari"},
		{"sanitario", "sanitari"},
		{"sanitarios", "sanitari"},
		{"sanjuana", "sanjuan"},
		{"sano", "san"},
		{"sansores", "sansor"},
		{"santa", "sant"},
		{"santana", "santan"},
		{"santiago", "santiag"},
		{"santidad", "santid"},
		{"santiesteban", "santiesteb"},
		{"santillán", "santillan"},
		{"santísimo", "santisim"},
		{"santiso", "santis"},
		{"santistas", "santist"},
		{"santo", "sant"},
		{"santos", "sant"},
		{"santuario", "santuari"},
		{"santurce", "santurc"},
		{"sao", "sao"},
		{"sapién", "sapien"},
		{"sapitos", "sapit"},
		{"sapo", "sap"},
		{"saqué", "saqu"},
		{"saqueadas", "saqu"},
		{"saquear", "saqu"},
		{"saqueo", "saque"},
		{"saqueos", "saque"},
		{"sar", "sar"},
		{"sarah", "sarah"},
		{"sarajevo", "sarajev"},
		{"sarcástica", "sarcast"},
		{"sargentos", "sargent"},
		{"sarh", "sarh"},
		{"sartenes", "sarten"},
		{"sastre", "sastr"},
		{"sat", "sat"},
		{"satanás", "satanas"},
		{"satélite", "satelit"},
		{"satelitel", "satelitel"},
		{"satélites", "satelit"},
		{"satín", "satin"},
		{"satírica", "satir"},
		{"satisfacción", "satisfaccion"},
		{"satisfacciones", "satisfaccion"},
		{"satisface", "satisfac"},
		{"satisfacer", "satisfac"},
		{"satisfactores", "satisfactor"},
		{"satisfactoria", "satisfactori"},
		{"satisfactoriamente", "satisfactori"},
		{"satisfactorias", "satisfactori"},
		{"satisfactorio", "satisfactori"},
		{"satisfagan", "satisfag"},
		{"satisfecho", "satisfech"},
		{"sato", "sat"},
		{"saturación", "satur"},
		{"saturada", "satur"},
		{"saturadas", "satur"},
		{"saturado", "satur"},
		{"saturados", "satur"},
		{"saturó", "satur"},
		{"sauceda", "sauced"},
		{"saucedo", "sauced"},
		{"sauco", "sauc"},
		{"saúl", "saul"},
		{"saunders", "saunders"},
		{"sauzal", "sauzal"},
		{"savimbi", "savimbi"},
		{"savotino", "savotin"},
		{"savoy", "savoy"},
		{"savr", "savr"},
		{"saxe", "sax"},
		{"saxofón", "saxofon"},
		{"saxofonista", "saxofon"},
		{"saya", "say"},
		{"sazón", "sazon"},
		{"sc", "sc"},
		{"scafadi", "scafadi"},
		{"scalia", "scali"},
		{"scatasta", "scatast"},
		{"scheider", "scheid"},
		{"schlesinghaus", "schlesinghaus"},
		{"schmal", "schmal"},
		{"schmall", "schmall"},
		{"schoenstat", "schoenstat"},
		{"schoenstatt", "schoenstatt"},
		{"schooll", "schooll"},
		{"schools", "schools"},
		{"schopenhahuer", "schopenhahu"},
		{"science", "scienc"},
		{"scott", "scott"},
		{"scout", "scout"},
		{"scrooge", "scroog"},
		{"sct", "sct"},
		{"sculley", "sculley"},
		{"se", "se"},
		{"sé", "se"},
		{"sea", "sea"},
		{"seamos", "seam"},
		{"sean", "sean"},
		{"seas", "seas"},
		{"seattle", "seattl"},
		{"sebastian", "sebasti"},
		{"sebastián", "sebastian"},
		{"seca", "sec"},
		{"secaba", "sec"},
		{"secadora", "secador"},
		{"secas", "sec"},
		{"sección", "seccion"},
		{"seccional", "seccional"},
		{"seccionales", "seccional"},
		{"seco", "sec"},
		{"secofi", "secofi"},
		{"secogef", "secogef"},
		{"secos", "sec"},
		{"secreta", "secret"},
		{"secretaria", "secretari"},
		{"secretaría", "secret"},
		{"secretarías", "secret"},
		{"secretario", "secretari"},
		{"secretarios", "secretari"},
		{"secretas", "secret"},
		{"secreto", "secret"},
		{"secretos", "secret"},
		{"sectas", "sect"},
		{"sector", "sector"},
		{"sectores", "sector"},
		{"sectoriales", "sectorial"},
		{"secuela", "secuel"},
		{"secuelas", "secuel"},
		{"secuencial", "secuencial"},
		{"secuestrada", "secuestr"},
		{"secuestradas", "secuestr"},
		{"secuestrado", "secuestr"},
		{"secuestrador", "secuestr"},
		{"secuestradores", "secuestr"},
		{"secuestro", "secuestr"},
		{"secuestró", "secuestr"},
		{"secuestros", "secuestr"},
		{"secular", "secul"},
		{"secundada", "secund"},
		{"secundados", "secund"},
		{"secundaria", "secundari"},
		{"secundarias", "secundari"},
		{"secundario", "secundari"},
		{"secundarios", "secundari"},
		{"secundaron", "secund"},
		{"securities", "securiti"},
		{"sed", "sed"},
		{"seda", "sed"},
		{"sedán", "sedan"},
		{"sedanes", "sedan"},
		{"sedante", "sedant"},
		{"sede", "sed"},
		{"sedesol", "sedesol"},
		{"sedientos", "sedient"},
		{"sedona", "sedon"},
		{"seducción", "seduccion"},
		{"seducir", "seduc"},
		{"seductoras", "seductor"},
		{"sedujo", "seduj"},
		{"seduop", "seduop"},
		{"seelene", "seelen"},
		{"segmentada", "segment"},
		{"segmento", "segment"},
		{"segmentos", "segment"},
		{"segovia", "segovi"},
		{"segueta", "seguet"},
		{"seguí", "segu"},
		{"seguía", "segu"},
		{"seguían", "segu"},
		{"seguida", "segu"},
		{"seguido", "segu"},
		{"seguidora", "seguidor"},
		{"seguidores", "seguidor"},
		{"seguidos", "segu"},
		{"seguimiento", "seguimient"},
		{"seguimos", "segu"},
		{"seguir", "segu"},
		{"seguirá", "segu"},
		{"seguirán", "segu"},
		{"seguiré", "segu"},
		{"seguiremos", "segu"},
		{"seguiría", "segu"},
		{"seguirlas", "segu"},
		{"seguirles", "segu"},
		{"seguirse", "segu"},
		{"según", "segun"},
		{"segunda", "segund"},
		{"segundas", "segund"},
		{"segundo", "segund"},
		{"segundones", "segundon"},
		{"segundos", "segund"},
		{"segura", "segur"},
		{"seguramente", "segur"},
		{"seguras", "segur"},
		{"seguri", "seguri"},
		{"seguridad", "segur"},
		{"seguro", "segur"},
		{"seguros", "segur"},
		{"seguy", "seguy"},
		{"seis", "seis"},
		{"seiyu", "seiyu"},
		{"seko", "sek"},
		{"selección", "seleccion"},
		{"seleccionada", "seleccion"},
		{"seleccionadas", "seleccion"},
		{"seleccionado", "seleccion"},
		{"seleccionados", "seleccion"},
		{"seleccionar", "seleccion"},
		{"seleccionarán", "seleccion"},
		{"selecciones", "seleccion"},
		{"selectivo", "select"},
		{"selecto", "select"},
		{"sellada", "sell"},
		{"sellar", "sell"},
		{"seller", "sell"},
		{"sello", "sell"},
		{"selva", "selv"},
		{"selvas", "selv"},
		{"semáforos", "semafor"},
		{"semana", "seman"},
		{"semanal", "semanal"},
		{"semanales", "semanal"},
		{"semanalmente", "semanal"},
		{"semanario", "semanari"},
		{"semanas", "seman"},
		{"semblantes", "semblant"},
		{"sembraban", "sembr"},
		{"sembrada", "sembr"},
		{"sembradío", "sembradi"},
		{"sembradíos", "sembradi"},
		{"sembradores", "sembrador"},
		{"sembrar", "sembr"},
		{"sembremos", "sembr"},
		{"semejante", "semej"},
		{"semejantes", "semej"},
		{"semejanza", "semej"},
		{"semejanzas", "semej"},
		{"semen", "sem"},
		{"semental", "semental"},
		{"semestral", "semestral"},
		{"semestrales", "semestral"},
		{"semestre", "semestr"},
		{"semestres", "semestr"},
		{"semi", "semi"},
		{"semiautomática", "semiautomat"},
		{"semicerrados", "semicerr"},
		{"semidesértica", "semidesert"},
		{"semidíos", "semidi"},
		{"semifinal", "semifinal"},
		{"semifinales", "semifinal"},
		{"semifinalistas", "semifinal"},
		{"semilla", "semill"},
		{"semillas", "semill"},
		{"seminal", "seminal"},
		{"seminario", "seminari"},
		{"seminarios", "seminari"},
		{"seminaristas", "seminar"},
		{"semioscuras", "semioscur"},
		{"semipreciosa", "semipreci"},
		{"semivacíos", "semivaci"},
		{"seña", "señ"},
		{"senado", "sen"},
		{"senador", "senador"},
		{"senadora", "senador"},
		{"senadores", "senador"},
		{"señal", "señal"},
		{"señala", "señal"},
		{"señalaban", "señal"},
		{"señaladas", "señal"},
		{"señalado", "señal"},
		{"señalamiento", "señal"},
		{"señalamientos", "señal"},
		{"señalamos", "señal"},
		{"señalan", "señal"},
		{"señalar", "señal"},
		{"señalarla", "señal"},
		{"señalaron", "señal"},
		{"señalarse", "señal"},
		{"señalen", "señal"},
		{"señales", "señal"},
		{"señalo", "señal"},
		{"señaló", "señal"},
		{"señas", "señ"},
		{"sencilla", "sencill"},
		{"sencillamente", "sencill"},
		{"sencillas", "sencill"},
		{"sencillez", "sencillez"},
		{"sencillo", "sencill"},
		{"sencillos", "sencill"},
		{"sendas", "send"},
		{"senderista", "sender"},
		{"senderistas", "sender"},
		{"sendero", "sender"},
		{"sendos", "send"},
		{"seneam", "seneam"},
		{"seno", "sen"},
		{"señor", "señor"},
		{"señora", "señor"},
		{"señoras", "señor"},
		{"señores", "señor"},
		{"señoría", "señor"},
		{"señorita", "señorit"},
		{"senos", "sen"},
		{"sensación", "sensacion"},
		{"sensacionalismo", "sensacional"},
		{"sensacionalista", "sensacional"},
		{"sensaciones", "sensacion"},
		{"sensata", "sensat"},
		{"sensatas", "sensat"},
		{"sensibilidad", "sensibil"},
		{"sensibilidades", "sensibil"},
		{"sensibilizar", "sensibiliz"},
		{"sensible", "sensibl"},
		{"sensiblemente", "sensibl"},
		{"sensibles", "sensibl"},
		{"sensores", "sensor"},
		{"sensual", "sensual"},
		{"sensualidad", "sensual"},
		{"sentadas", "sent"},
		{"sentado", "sent"},
		{"sentados", "sent"},
		{"sentamos", "sent"},
		{"sentando", "sent"},
		{"sentanes", "sentan"},
		{"sentar", "sent"},
		{"sentaran", "sent"},
		{"sentarían", "sent"},
		{"sentaron", "sent"},
		{"sentarse", "sent"},
		{"sentencia", "sentenci"},
		{"sentenciada", "sentenci"},
		{"sentenciado", "sentenci"},
		{"sentenciados", "sentenci"},
		{"sentenció", "sentenc"},
		{"sentí", "sent"},
		{"sentía", "sent"},
		{"sentían", "sent"},
		{"sentido", "sent"},
		{"sentimental", "sentimental"},
		{"sentimiento", "sentimient"},
		{"sentimientos", "sentimient"},
		{"sentimos", "sent"},
		{"sentir", "sent"},
		{"sentiría", "sent"},
		{"sentirse", "sent"},
		{"sentó", "sent"},
		{"sep", "sep"},
		{"sepa", "sep"},
		{"sepamos", "sep"},
		{"sepan", "sep"},
		{"separa", "sep"},
		{"separación", "separ"},
		{"separaciones", "separ"},
		{"separada", "separ"},
		{"separadas", "separ"},
		{"separado", "separ"},
		{"separador", "separ"},
		{"separados", "separ"},
		{"separar", "separ"},
		{"separará", "separ"},
		{"separarán", "separ"},
		{"separarlos", "separ"},
		{"separaron", "separ"},
		{"separarse", "separ"},
		{"separe", "separ"},
		{"separen", "separ"},
		{"sepelio", "sepeli"},
		{"sépticas", "septic"},
		{"septiembre", "septiembr"},
		{"séptima", "septim"},
		{"séptimo", "septim"},
		{"sepultado", "sepult"},
		{"sepultan", "sepult"},
		{"sepultó", "sepult"},
		{"sepúlveda", "sepulved"},
		{"séquito", "sequit"},
		{"ser", "ser"},
		{"sera", "ser"},
		{"será", "ser"},
		{"serán", "seran"},
		{"serbia", "serbi"},
		{"serbio", "serbi"},
		{"serbios", "serbi"},
		{"seremos", "ser"},
		{"serena", "seren"},
		{"serenata", "serenat"},
		{"serenidad", "seren"},
		{"seres", "ser"},
		{"serfin", "serfin"},
		{"serfín", "serfin"},
		{"sergei", "sergei"},
		{"sergi", "sergi"},
		{"sergio", "sergi"},
		{"seria", "seri"},
		{"sería", "ser"},
		{"serial", "serial"},
		{"seriamente", "seri"},
		{"seríamos", "ser"},
		{"serían", "ser"},
		{"serias", "seri"},
		{"serie", "seri"},
		{"seriedad", "seried"},
		{"series", "seri"},
		{"serio", "seri"},
		{"serios", "seri"},
		{"serlo", "serl"},
		{"sermón", "sermon"},
		{"sermonearlos", "sermon"},
		{"serna", "sern"},
		{"serpentinero", "serpentiner"},
		{"serpiente", "serpient"},
		{"serra", "serr"},
		{"serrano", "serran"},
		{"servia", "servi"},
		{"service", "servic"},
		{"servicial", "servicial"},
		{"servicio", "servici"},
		{"servicios", "servici"},
		{"servido", "serv"},
		{"servidor", "servidor"},
		{"servidores", "servidor"},
		{"servidumbre", "servidumbr"},
		{"serviles", "servil"},
		{"servilleta", "servillet"},
		{"servimos", "serv"},
		{"servín", "servin"},
		{"servios", "servi"},
		{"servir", "serv"},
		{"servirá", "serv"},
		{"servirán", "serv"},
		{"serviría", "serv"},
		{"servirnos", "serv"},
		{"sese", "ses"},
		{"sesentas", "sesent"},
		{"sesgo", "sesg"},
		{"sesión", "sesion"},
		{"sesiones", "sesion"},
		{"sesos", "ses"},
		{"set", "set"},
		{"setentas", "setent"},
		{"sets", "sets"},
		{"seudónimo", "seudonim"},
		{"seul", "seul"},
		{"seúl", "seul"},
		{"severa", "sever"},
		{"severamente", "sever"},
		{"severas", "sever"},
		{"severo", "sever"},
		{"severos", "sever"},
		{"sevilla", "sevill"},
		{"sexenio", "sexeni"},
		{"sexenios", "sexeni"},
		{"sexista", "sexist"},
		{"sexo", "sex"},
		{"sexopatía", "sexopat"},
		{"sexos", "sex"},
		{"sexta", "sext"},
		{"sextetas", "sextet"},
		{"sexto", "sext"},
		{"sexual", "sexual"},
		{"sexuales", "sexual"},
		{"sexualidad", "sexual"},
		{"sexualmente", "sexual"},
		{"sexy", "sexy"},
		{"sexys", "sexys"},
		{"sfara", "sfar"},
		{"sgpv", "sgpv"},
		{"shadow", "shadow"},
		{"shannon", "shannon"},
		{"shaw", "shaw"},
		{"shcp", "shcp"},
		{"sheehy", "sheehy"},
		{"sheila", "sheil"},
		{"shepard", "shepard"},
		{"sherel", "sherel"},
		{"sheriff", "sheriff"},
		{"sherman", "sherm"},
		{"sherrit", "sherrit"},
		{"shibata", "shibat"},
		{"shikang", "shikang"},
		{"shimizu", "shimizu"},
		{"shorts", "shorts"},
		{"show", "show"},
		{"shows", "shows"},
		{"showtime", "showtim"},
		{"shrine", "shrin"},
		{"si", "si"},
		{"sí", "si"},
		{"siberia", "siberi"},
		{"siberiana", "siberian"},
		{"sica", "sic"},
		{"siclo", "sicl"},
		{"siclos", "sicl"},
		{"sicológica", "sicolog"},
		{"sida", "sid"},
		{"sidente", "sident"},
		{"siderúrgicas", "siderurg"},
		{"sido", "sid"},
		{"siembra", "siembr"},
		{"siembran", "siembr"},
		{"siemens", "siemens"},
		{"siempre", "siempr"},
		{"siendo", "siend"},
		{"sienta", "sient"},
		{"sientan", "sient"},
		{"siente", "sient"},
		{"sienten", "sient"},
		{"siento", "sient"},
		{"sierra", "sierr"},
		{"sierva", "sierv"},
		{"siervo", "sierv"},
		{"siete", "siet"},
		{"siga", "sig"},
		{"sigamos", "sig"},
		{"sigan", "sig"},
		{"siglas", "sigl"},
		{"siglo", "sigl"},
		{"siglos", "sigl"},
		{"significa", "signif"},
		{"significación", "signif"},
		{"significado", "signific"},
		{"significados", "signific"},
		{"significan", "signific"},
		{"significar", "signific"},
		{"significará", "signific"},
		{"significaran", "signific"},
		{"significaría", "signific"},
		{"significativa", "signific"},
		{"significativamente", "signific"},
		{"significativas", "signific"},
		{"significativo", "signific"},
		{"significativos", "signific"},
		{"significo", "signif"},
		{"significó", "signific"},
		{"signifique", "signifiqu"},
		{"signo", "sign"},
		{"signos", "sign"},
		{"sigo", "sig"},
		{"sigue", "sig"},
		{"siguen", "sig"},
		{"siguiendo", "sigu"},
		{"siguiente", "siguient"},
		{"siguientes", "siguient"},
		{"siguiera", "sigu"},
		{"siguieran", "sigu"},
		{"siguieron", "sigu"},
		{"siguió", "sigu"},
		{"silbante", "silbant"},
		{"silbaron", "silb"},
		{"silbatazo", "silbataz"},
		{"silber", "silb"},
		{"silbes", "silb"},
		{"silencio", "silenci"},
		{"silencioso", "silenci"},
		{"silla", "sill"},
		{"sillas", "sill"},
		{"sillón", "sillon"},
		{"silos", "sil"},
		{"silueta", "siluet"},
		{"silva", "silv"},
		{"silvestre", "silvestr"},
		{"silvestris", "silvestris"},
		{"silvia", "silvi"},
		{"silviano", "silvian"},
		{"silvio", "silvi"},
		{"simbólica", "simbol"},
		{"simbólico", "simbol"},
		{"simbolización", "simboliz"},
		{"símbolo", "simbol"},
		{"símbolos", "simbol"},
		{"simeone", "simeon"},
		{"simeprode", "simeprod"},
		{"similar", "simil"},
		{"similares", "similar"},
		{"simón", "simon"},
		{"simpatía", "simpat"},
		{"simpático", "simpat"},
		{"simpatizantes", "simpatiz"},
		{"simple", "simpl"},
		{"simplemente", "simplement"},
		{"simples", "simpl"},
		{"simpliciano", "simplician"},
		{"simplificación", "simplif"},
		{"simplificado", "simplific"},
		{"simplista", "simplist"},
		{"simposiums", "simposiums"},
		{"simulación", "simul"},
		{"simulador", "simul"},
		{"simulados", "simul"},
		{"simular", "simul"},
		{"simultánea", "simultane"},
		{"simultáneamente", "simultan"},
		{"simultáneos", "simultane"},
		{"sin", "sin"},
		{"sinagua", "sinagu"},
		{"sinaí", "sina"},
		{"sinaloa", "sinalo"},
		{"sinaloense", "sinaloens"},
		{"sinaloenses", "sinaloens"},
		{"sinceramente", "sincer"},
		{"sincero", "sincer"},
		{"sinclair", "sincl"},
		{"sincretismo", "sincret"},
		{"sincronización", "sincroniz"},
		{"sincronizados", "sincroniz"},
		{"sindelfingen", "sindelfing"},
		{"sindical", "sindical"},
		{"sindicales", "sindical"},
		{"sindicalismo", "sindical"},
		{"sindicalista", "sindical"},
		{"sindicalistas", "sindical"},
		{"sindicato", "sindicat"},
		{"sindicatos", "sindicat"},
		{"sindicatura", "sindicatur"},
		{"síndico", "sindic"},
		{"síndicos", "sindic"},
		{"sindo", "sind"},
		{"síndrome", "sindrom"},
		{"sinfonía", "sinfon"},
		{"singapur", "singapur"},
		{"singer", "sing"},
		{"singles", "singl"},
		{"singlista", "singlist"},
		{"singular", "singul"},
		{"siniestra", "siniestr"},
		{"siniestro", "siniestr"},
		{"sinnúmero", "sinnumer"},
		{"sino", "sin"},
		{"sinodal", "sinodal"},
		{"sinónimo", "sinonim"},
		{"sinovial", "sinovial"},
		{"síntesis", "sintesis"},
		{"sintética", "sintet"},
		{"sintético", "sintet"},
		{"sintéticos", "sintet"},
		{"sintiéndome", "sint"},
		{"sintió", "sint"},
		{"síntoma", "sintom"},
		{"síntomas", "sintom"},
		{"sintomática", "sintomat"},
		{"sintomáticas", "sintomat"},
		{"sintomático", "sintomat"},
		{"sinuosos", "sinuos"},
		{"sinusitis", "sinusitis"},
		{"siquiatra", "siquiatr"},
		{"siquiera", "siqu"},
		{"sir", "sir"},
		{"sirgo", "sirg"},
		{"sirloin", "sirloin"},
		{"siro", "sir"},
		{"sirva", "sirv"},
		{"sirvan", "sirv"},
		{"sirve", "sirv"},
		{"sirven", "sirv"},
		{"sirves", "sirv"},
		{"sirvientas", "sirvient"},
		{"sirvieran", "sirv"},
		{"sirvieron", "sirv"},
		{"sirvió", "sirv"},
		{"siscohm", "siscohm"},
		{"sísmica", "sismic"},
		{"sísmicos", "sismic"},
		{"sismógrafos", "sismograf"},
		{"sismología", "sismolog"},
		{"sistefin", "sistefin"},
		{"sistema", "sistem"},
		{"sistemas", "sistem"},
		{"sistemática", "sistemat"},
		{"sistemáticas", "sistemat"},
		{"sistemático", "sistemat"},
		{"sister", "sist"},
		{"sitemáticamente", "sitemat"},
		{"sitiado", "siti"},
		{"sitiados", "siti"},
		{"sitio", "siti"},
		{"sitios", "siti"},
		{"situación", "situacion"},
		{"situaciones", "situacion"},
		{"situada", "situ"},
		{"situadas", "situ"},
		{"situado", "situ"},
		{"situándose", "situ"},
		{"situar", "situ"},
		{"situarlos", "situ"},
		{"situarse", "situ"},
		{"sixto", "sixt"},
		{"sketches", "sketch"},
		{"slam", "slam"},
		{"sledge", "sledg"},
		{"sli", "sli"},
		{"slim", "slim"},
		{"smc", "smc"},
		{"smith", "smith"},
		{"smuckers", "smuckers"},
		{"snte", "snte"},
		{"sobada", "sob"},
		{"sobando", "sob"},
		{"soberana", "soberan"},
		{"soberanía", "soberan"},
		{"soberano", "soberan"},
		{"soberbia", "soberbi"},
		{"sobornados", "soborn"},
		{"sobornando", "soborn"},
		{"soborno", "soborn"},
		{"sobornos", "soborn"},
		{"sobra", "sobr"},
		{"sobraba", "sobr"},
		{"sobradamente", "sobrad"},
		{"sobran", "sobr"},
		{"sobrando", "sobr"},
		{"sobrar", "sobr"},
		{"sobrarán", "sobr"},
		{"sobre", "sobr"},
		{"sobreasignación", "sobreasign"},
		{"sobrecalentada", "sobrecalent"},
		{"sobrecalentamiento", "sobrecalent"},
		{"sobrecalentaron", "sobrecalent"},
		{"sobrecarga", "sobrecarg"},
		{"sobrecargaron", "sobrecarg"},
		{"sobrecargas", "sobrecarg"},
		{"sobrecosto", "sobrecost"},
		{"sobrecostos", "sobrecost"},
		{"sobrecupo", "sobrecup"},
		{"sobrecupos", "sobrecup"},
		{"sobreexplotación", "sobreexplot"},
		{"sobregiro", "sobregir"},
		{"sobregiros", "sobregir"},
		{"sobrellevar", "sobrellev"},
		{"sobremanera", "sobremaner"},
		{"sobrenombre", "sobrenombr"},
		{"sobrepasa", "sobrepas"},
		{"sobrepasan", "sobrepas"},
		{"sobrepasar", "sobrepas"},
		{"sobrepase", "sobrep"},
		{"sobrepasó", "sobrepas"},
		{"sobrepoblación", "sobrepobl"},
		{"sobreponen", "sobrepon"},
		{"sobreponiéndose", "sobrepon"},
		{"sobreposición", "sobreposicion"},
		{"sobresale", "sobresal"},
		{"sobresalen", "sobresal"},
		{"sobresaliendo", "sobresal"},
		{"sobresaliente", "sobresalient"},
		{"sobresalió", "sobresal"},
		{"sobresalir", "sobresal"},
		{"sobresaltos", "sobresalt"},
		{"sobreseída", "sobreseid"},
		{"sobrestantes", "sobrest"},
		{"sobresuscripcion", "sobresuscripcion"},
		{"sobretasa", "sobretas"},
		{"sobretodo", "sobretod"},
		{"sobrevaluado", "sobrevalu"},
		{"sobrevenga", "sobreveng"},
		{"sobrevida", "sobrev"},
		{"sobreviven", "sobreviv"},
		{"sobrevivencia", "sobrevivent"},
		{"sobrevivido", "sobreviv"},
		{"sobrevivientes", "sobrevivient"},
		{"sobrevivieron", "sobreviv"},
		{"sobrevivir", "sobreviv"},
		{"sobrevuelan", "sobrevuel"},
		{"sobrevuelos", "sobrevuel"},
		{"sobriedad", "sobried"},
		{"sobrina", "sobrin"},
		{"sobrino", "sobrin"},
		{"sobrinos", "sobrin"},
		{"socavamos", "socav"},
		{"socavar", "socav"},
		{"soccer", "socc"},
		{"social", "social"},
		{"socialdemócrata", "socialdemocrat"},
		{"sociales", "social"},
		{"socialismo", "social"},
		{"socialista", "social"},
		{"socialmente", "social"},
		{"sociedad", "socied"},
		{"sociedades", "sociedad"},
		{"socio", "soci"},
		{"socioeconómica", "socioeconom"},
		{"socioeconómico", "socioeconom"},
		{"socióloga", "sociolog"},
		{"sociológico", "sociolog"},
		{"sociológicos", "sociolog"},
		{"sociólogo", "sociolog"},
		{"sociopolítica", "sociopolit"},
		{"socios", "soci"},
		{"socorrista", "socorr"},
		{"socorristas", "socorr"},
		{"socorro", "socorr"},
		{"socorros", "socorr"},
		{"sócrates", "socrat"},
		{"socrático", "socrat"},
		{"socráticos", "socrat"},
		{"sodas", "sod"},
		{"sódico", "sodic"},
		{"sodio", "sodi"},
		{"sofá", "sof"},
		{"sofía", "sof"},
		{"sofisticación", "sofist"},
		{"sofisticadas", "sofistic"},
		{"sofisticado", "sofistic"},
		{"sofisticados", "sofistic"},
		{"sofocación", "sofoc"},
		{"sofocado", "sofoc"},
		{"sofocarlo", "sofoc"},
		{"soft", "soft"},
		{"softbol", "softbol"},
		{"software", "softwar"},
		{"sogem", "sogem"},
		{"sojo", "soj"},
		{"sol", "sol"},
		{"sola", "sol"},
		{"solamente", "sol"},
		{"solana", "solan"},
		{"solapados", "solap"},
		{"solar", "sol"},
		{"solares", "solar"},
		{"solaris", "solaris"},
		{"solas", "sol"},
		{"solaza", "solaz"},
		{"solazo", "solaz"},
		{"soldada", "sold"},
		{"soldado", "sold"},
		{"soldador", "soldador"},
		{"soldados", "sold"},
		{"soldadura", "soldadur"},
		{"soldan", "sold"},
		{"soldando", "sold"},
		{"soldar", "sold"},
		{"soldarlos", "sold"},
		{"soleado", "sol"},
		{"soledad", "soled"},
		{"solemne", "solemn"},
		{"solemnidad", "solemn"},
		{"solía", "sol"},
		{"solían", "sol"},
		{"solicita", "solicit"},
		{"solicitaba", "solicit"},
		{"solicitada", "solicit"},
		{"solicitado", "solicit"},
		{"solicitados", "solicit"},
		{"solicitamos", "solicit"},
		{"solicitan", "solicit"},
		{"solicitando", "solicit"},
		{"solicitar", "solicit"},
		{"solicitará", "solicit"},
		{"solicitarán", "solicit"},
		{"solicitarle", "solicit"},
		{"solicitaron", "solicit"},
		{"solicite", "solicit"},
		{"solicité", "solicit"},
		{"solicito", "solicit"},
		{"solícito", "solicit"},
		{"solicitó", "solicit"},
		{"solicitud", "solicitud"},
		{"solicitudes", "solicitud"},
		{"sólida", "sol"},
		{"solidaria", "solidari"},
		{"solidaridad", "solidar"},
		{"solidario", "solidari"},
		{"sólidas", "sol"},
		{"solidez", "solidez"},
		{"sólido", "sol"},
		{"sólidos", "sol"},
		{"solís", "sol"},
		{"solista", "solist"},
		{"solistas", "solist"},
		{"solitario", "solitari"},
		{"solitarios", "solitari"},
		{"solito", "solit"},
		{"sollozando", "solloz"},
		{"solo", "sol"},
		{"sólo", "sol"},
		{"solórzano", "solorzan"},
		{"solos", "sol"},
		{"soltar", "solt"},
		{"soltara", "solt"},
		{"soltaran", "solt"},
		{"soltera", "solter"},
		{"solteras", "solter"},
		{"soltó", "solt"},
		{"solución", "solucion"},
		{"solucionan", "solucion"},
		{"solucionar", "solucion"},
		{"solucionarse", "solucion"},
		{"solucione", "solucion"},
		{"soluciones", "solucion"},
		{"solucionó", "solucion"},
		{"solvencia", "solvenci"},
		{"solventar", "solvent"},
		{"somalia", "somali"},
		{"sombra", "sombr"},
		{"sombras", "sombr"},
		{"sombrero", "sombrer"},
		{"sombreros", "sombrer"},
		{"sometan", "somet"},
		{"somete", "somet"},
		{"someter", "somet"},
		{"someterá", "somet"},
		{"someterán", "somet"},
		{"someterlos", "somet"},
		{"someterse", "somet"},
		{"sometidas", "somet"},
		{"sometido", "somet"},
		{"sometidos", "somet"},
		{"sometimiento", "somet"},
		{"somos", "som"},
		{"somoza", "somoz"},
		{"son", "son"},
		{"soñado", "soñ"},
		{"soñadores", "soñador"},
		{"sonados", "son"},
		{"soñados", "soñ"},
		{"soñando", "soñ"},
		{"soñar", "soñ"},
		{"sonda", "sond"},
		{"sondas", "sond"},
		{"sondeos", "sonde"},
		{"soñé", "soñ"},
		{"sonia", "soni"},
		{"sonido", "son"},
		{"sonidos", "son"},
		{"soñó", "soñ"},
		{"sonora", "sonor"},
		{"sonorense", "sonorens"},
		{"sonriente", "sonrient"},
		{"sonríes", "sonr"},
		{"sonrió", "sonr"},
		{"sonrisa", "sonris"},
		{"sonrisas", "sonris"},
		{"sony", "sony"},
		{"sonya", "sony"},
		{"sopesar", "sopes"},
		{"soplando", "sopl"},
		{"soplarse", "sopl"},
		{"soporta", "soport"},
		{"soportaba", "soport"},
		{"soportable", "soport"},
		{"soportado", "soport"},
		{"soportan", "soport"},
		{"soportar", "soport"},
		{"soportarán", "soport"},
		{"soporte", "soport"},
		{"soportes", "soport"},
		{"soportó", "soport"},
		{"sor", "sor"},
		{"sorda", "sord"},
		{"sordera", "sorder"},
		{"sórdidas", "sord"},
		{"sórdido", "sord"},
		{"sordos", "sord"},
		{"sorgo", "sorg"},
		{"soriana", "sorian"},
		{"soriano", "sorian"},
		{"soros", "sor"},
		{"sorprenda", "sorprend"},
		{"sorprende", "sorprend"},
		{"sorprenden", "sorprend"},
		{"sorprendente", "sorprendent"},
		{"sorprendentes", "sorprendent"},
		{"sorprender", "sorprend"},
		{"sorprenderá", "sorprend"},
		{"sorprendí", "sorprend"},
		{"sorprendidas", "sorprend"},
		{"sorprendido", "sorprend"},
		{"sorprendidos", "sorprend"},
		{"sorprendieron", "sorprend"},
		{"sorprendió", "sorprend"},
		{"sorpresa", "sorpres"},
		{"sorpresas", "sorpres"},
		{"sorpresiva", "sorpres"},
		{"sorpresivas", "sorpres"},
		{"sorpresivo", "sorpres"},
		{"sorpresivos", "sorpres"},
		{"sortea", "sorte"},
		{"sorteo", "sorte"},
		{"sorteos", "sorte"},
		{"sosa", "sos"},
		{"soslayó", "soslay"},
		{"sospecha", "sospech"},
		{"sospechaba", "sospech"},
		{"sospechamos", "sospech"},
		{"sospecho", "sospech"},
		{"sospechosa", "sospech"},
		{"sospechoso", "sospech"},
		{"sospechosos", "sospech"},
		{"sostén", "sosten"},
		{"sostendrá", "sostendr"},
		{"sostendrán", "sostendran"},
		{"sostener", "sosten"},
		{"sostenerla", "sosten"},
		{"sostenerse", "sosten"},
		{"sostengo", "sosteng"},
		{"sostenía", "sosten"},
		{"sostenían", "sosten"},
		{"sostenible", "sosten"},
		{"sostenida", "sosten"},
		{"sostenidas", "sosten"},
		{"sostenido", "sosten"},
		{"sosteniendo", "sosten"},
		{"sostiene", "sostien"},
		{"sostienen", "sostien"},
		{"sostuvieron", "sostuv"},
		{"sostuvo", "sostuv"},
		{"sotanero", "sotaner"},
		{"sotaneros", "sotaner"},
		{"sotero", "soter"},
		{"soto", "sot"},
		{"soundgarden", "soundgard"},
		{"south", "south"},
		{"southern", "southern"},
		{"southwest", "southwest"},
		{"soviet", "soviet"},
		{"soviética", "soviet"},
		{"soviético", "soviet"},
		{"soy", "soy"},
		{"soya", "soy"},
		{"sparcclassic", "sparcclassic"},
		{"sparcservers", "sparcservers"},
		{"sparcstations", "sparcstations"},
		{"sparrings", "sparrings"},
		{"special", "special"},
		{"spike", "spik"},
		{"spirit", "spirit"},
		{"spivakov", "spivakov"},
		{"sport", "sport"},
		{"sporting", "sporting"},
		{"spot", "spot"},
		{"sprenger", "spreng"},
		{"springbreakers", "springbreakers"},
		{"sprint", "sprint"},
		{"sprints", "sprints"},
		{"sr", "sr"},
		{"staff", "staff"},
		{"stalin", "stalin"},
		{"standard", "standard"},
		{"stanley", "stanley"},
		{"state", "stat"},
		{"station", "station"},
		{"statuo", "statu"},
		{"status", "status"},
		{"steels", "steels"},
		{"stéfano", "stefan"},
		{"stephan", "steph"},
		{"stephanopoulos", "stephanopoul"},
		{"stephen", "steph"},
		{"stereo", "stere"},
		{"stern", "stern"},
		{"steve", "stev"},
		{"steven", "stev"},
		{"stl", "stl"},
		{"stock", "stock"},
		{"stoichkov", "stoichkov"},
		{"stoickov", "stoickov"},
		{"stoker", "stok"},
		{"stólica", "stolic"},
		{"stone", "ston"},
		{"stor", "stor"},
		{"straffon", "straffon"},
		{"strauss", "strauss"},
		{"stravinsky", "stravinsky"},
		{"street", "street"},
		{"strictly", "strictly"},
		{"strikes", "strik"},
		{"strindberg", "strindberg"},
		{"studies", "studi"},
		{"stup", "stup"},
		{"stuttgart", "stuttgart"},
		{"su", "su"},
		{"suárez", "suarez"},
		{"suave", "suav"},
		{"suavemente", "suavement"},
		{"suaves", "suav"},
		{"suavidad", "suavid"},
		{"suavizamos", "suaviz"},
		{"suavizaron", "suaviz"},
		{"sub", "sub"},
		{"suba", "sub"},
		{"subacuática", "subacuat"},
		{"subalterna", "subaltern"},
		{"subalterno", "subaltern"},
		{"suban", "sub"},
		{"subaru", "subaru"},
		{"subasta", "subast"},
		{"subastas", "subast"},
		{"subcampeón", "subcampeon"},
		{"subcomandante", "subcomand"},
		{"subcomandantes", "subcomand"},
		{"subcomisión", "subcomision"},
		{"subcomité", "subcomit"},
		{"subconsciente", "subconscient"},
		{"subcontratación", "subcontrat"},
		{"subdelegado", "subdeleg"},
		{"subdesarrollada", "subdesarroll"},
		{"subdesarrollados", "subdesarroll"},
		{"subdirección", "subdireccion"},
		{"subdirector", "subdirector"},
		{"subdirectora", "subdirector"},
		{"subdistribuidores", "subdistribuidor"},
		{"subdivisión", "subdivision"},
		{"suben", "sub"},
		{"subes", "sub"},
		{"subespecialidades", "subespecial"},
		{"subestación", "subest"},
		{"subfacturación", "subfactur"},
		{"subgerente", "subgerent"},
		{"subí", "sub"},
		{"subían", "sub"},
		{"subida", "sub"},
		{"subidas", "sub"},
		{"subido", "sub"},
		{"subiendo", "sub"},
		{"subieran", "sub"},
		{"subieron", "sub"},
		{"subió", "sub"},
		{"subir", "sub"},
		{"subirá", "sub"},
		{"subirse", "sub"},
		{"súbita", "subit"},
		{"súbitamente", "subit"},
		{"subjetiva", "subjet"},
		{"subjetividad", "subjet"},
		{"sublíder", "sublid"},
		{"sublíderes", "sublider"},
		{"sublime", "sublim"},
		{"subliminal", "subliminal"},
		{"submarina", "submarin"},
		{"submarinas", "submarin"},
		{"submarino", "submarin"},
		{"submarinos", "submarin"},
		{"subordinado", "subordin"},
		{"subordinados", "subordin"},
		{"subprocurador", "subprocur"},
		{"subproductos", "subproduct"},
		{"subrayar", "subray"},
		{"subrayaron", "subray"},
		{"subrayó", "subray"},
		{"subregional", "subregional"},
		{"subsana", "subsan"},
		{"subsanables", "subsan"},
		{"subsecretaria", "subsecretari"},
		{"subsecretaría", "subsecret"},
		{"subsecretario", "subsecretari"},
		{"subsecretarios", "subsecretari"},
		{"subsectores", "subsector"},
		{"subsecuentes", "subsecuent"},
		{"subsidiada", "subsidi"},
		{"subsidiadas", "subsidi"},
		{"subsidiar", "subsidi"},
		{"subsidiaria", "subsidiari"},
		{"subsidiarias", "subsidiari"},
		{"subsidiarios", "subsidiari"},
		{"subsidio", "subsidi"},
		{"subsidios", "subsidi"},
		{"substancia", "substanci"},
		{"substancial", "substancial"},
		{"substanciales", "substancial"},
		{"substancias", "substanci"},
		{"subsuelo", "subsuel"},
		{"subtangente", "subtangent"},
		{"subterránea", "subterrane"},
		{"subterráneo", "subterrane"},
		{"suburban", "suburb"},
		{"suburbanos", "suburban"},
		{"suburbias", "suburbi"},
		{"suburbios", "suburbi"},
		{"subvención", "subvencion"},
		{"subversión", "subversion"},
		{"subversivo", "subvers"},
		{"subversivos", "subvers"},
		{"subvertir", "subvert"},
		{"subviertes", "subviert"},
		{"suceda", "suced"},
		{"sucede", "suced"},
		{"suceden", "suced"},
		{"suceder", "suced"},
		{"sucederá", "suced"},
		{"sucedería", "suced"},
		{"sucederlo", "suced"},
		{"sucedía", "suced"},
		{"sucedido", "suced"},
		{"sucediera", "suced"},
		{"sucedieron", "suced"},
		{"sucedió", "suced"},
		{"sucesion", "sucesion"},
		{"sucesión", "sucesion"},
		{"sucesivamente", "suces"},
		{"suceso", "suces"},
		{"sucesor", "sucesor"},
		{"sucesorio", "sucesori"},
		{"sucesos", "suces"},
		{"sucia", "suci"},
		{"suciedad", "sucied"},
		{"sucio", "suci"},
		{"sucios", "suci"},
		{"sucre", "sucr"},
		{"sucumbir", "sucumb"},
		{"sucursal", "sucursal"},
		{"sucursales", "sucursal"},
		{"sudamérica", "sudamer"},
		{"sudamericana", "sudamerican"},
		{"sudamericanas", "sudamerican"},
		{"sudamericano", "sudamerican"},
		{"sudar", "sud"},
		{"sudeste", "sudest"},
		{"sudoccidental", "sudoccidental"},
		{"sudoeste", "sudoest"},
		{"sudoración", "sudor"},
		{"sudoroso", "sudor"},
		{"suecia", "sueci"},
		{"sueco", "suec"},
		{"suecos", "suec"},
		{"suegra", "suegr"},
		{"suelas", "suel"},
		{"sueldo", "sueld"},
		{"sueldos", "sueld"},
		{"suele", "suel"},
		{"suelen", "suel"},
		{"suelo", "suel"},
		{"suelos", "suel"},
		{"sueltan", "suelt"},
		{"suena", "suen"},
		{"sueña", "sueñ"},
		{"suenen", "suen"},
		{"sueño", "sueñ"},
		{"sueños", "sueñ"},
		{"suero", "suer"},
		{"suerte", "suert"},
		{"suéter", "suet"},
		{"suéteres", "sueter"},
		{"suficiente", "suficient"},
		{"suficientemente", "suficient"},
		{"suficientes", "suficient"},
		{"sufragar", "sufrag"},
		{"sufragaron", "sufrag"},
		{"sufragio", "sufragi"},
		{"sufragios", "sufragi"},
		{"sufrague", "sufrag"},
		{"sufre", "sufr"},
		{"sufren", "sufr"},
		{"sufrí", "sufr"},
		{"sufría", "sufr"},
		{"sufrido", "sufr"},
		{"sufridos", "sufr"},
		{"sufriendo", "sufr"},
		{"sufrieron", "sufr"},
		{"sufrimiento", "sufrimient"},
		{"sufrió", "sufr"},
		{"sufrir", "sufr"},
		{"sugerencia", "sugerent"},
		{"sugerencias", "sugerent"},
		{"sugerentes", "sugerent"},
		{"sugería", "sug"},
		{"sugerido", "suger"},
		{"sugeridos", "suger"},
		{"sugerir", "suger"},
		{"sugestivos", "sugest"},
		{"sugiera", "sug"},
		{"sugieran", "sug"},
		{"sugiere", "sugier"},
		{"sugieren", "sugier"},
		{"sugiero", "sugier"},
		{"sugiriendo", "sugir"},
		{"sugirieron", "sugir"},
		{"sugirió", "sugir"},
		{"sui", "sui"},
		{"suicida", "suic"},
		{"suicidaron", "suicid"},
		{"suicidarse", "suicid"},
		{"suicidio", "suicidi"},
		{"suite", "suit"},
		{"suiza", "suiz"},
		{"suizos", "suiz"},
		{"sujeción", "sujecion"},
		{"sujetar", "sujet"},
		{"sujetarlo", "sujet"},
		{"sujetarse", "sujet"},
		{"sujeto", "sujet"},
		{"sujetos", "sujet"},
		{"sulaimán", "sulaiman"},
		{"sulfito", "sulfit"},
		{"sulfúrico", "sulfur"},
		{"sultana", "sultan"},
		{"sultanes", "sultan"},
		{"suma", "sum"},
		{"sumaban", "sum"},
		{"sumado", "sum"},
		{"sumados", "sum"},
		{"sumamente", "sum"},
		{"suman", "sum"},
		{"sumando", "sum"},
		{"sumar", "sum"},
		{"sumarán", "sum"},
		{"sumario", "sumari"},
		{"sumaron", "sum"},
		{"sumarse", "sum"},
		{"sumas", "sum"},
		{"sume", "sum"},
		{"sumen", "sum"},
		{"sumerge", "sumerg"},
		{"sumergibles", "sumerg"},
		{"sumergido", "sumerg"},
		{"sumergió", "sumerg"},
		{"sumerio", "sumeri"},
		{"sumido", "sum"},
		{"suministraba", "suministr"},
		{"suministrada", "suministr"},
		{"suministrar", "suministr"},
		{"suministro", "suministr"},
		{"suministró", "suministr"},
		{"sumisa", "sumis"},
		{"summers", "summers"},
		{"sumo", "sum"},
		{"sumó", "sum"},
		{"sun", "sun"},
		{"sung", "sung"},
		{"sunny", "sunny"},
		{"suntuoso", "suntuos"},
		{"suoo", "suo"},
		{"supe", "sup"},
		{"super", "sup"},
		{"súper", "sup"},
		{"supera", "super"},
		{"superación", "super"},
		{"superada", "super"},
		{"superado", "super"},
		{"superamas", "superam"},
		{"superamos", "super"},
		{"superan", "super"},
		{"superando", "super"},
		{"superar", "super"},
		{"superará", "super"},
		{"superarla", "super"},
		{"superarlo", "super"},
		{"superarlos", "super"},
		{"superarme", "super"},
		{"superaron", "super"},
		{"superavenida", "superaven"},
		{"superávit", "superavit"},
		{"superavitaria", "superavitari"},
		{"supercenter", "supercent"},
		{"supercenters", "supercenters"},
		{"superdotados", "superdot"},
		{"supere", "super"},
		{"superficial", "superficial"},
		{"superficiales", "superficial"},
		{"superficie", "superfici"},
		{"superficies", "superfici"},
		{"superfluas", "superflu"},
		{"superior", "superior"},
		{"superiores", "superior"},
		{"superioridad", "superior"},
		{"superiorsubaru", "superiorsubaru"},
		{"superlíder", "superlid"},
		{"superliderato", "superliderat"},
		{"superligero", "superliger"},
		{"superligeros", "superliger"},
		{"superliviano", "superlivian"},
		{"supermediano", "supermedian"},
		{"supermercado", "supermerc"},
		{"supermercados", "supermerc"},
		{"superó", "super"},
		{"superpluma", "superplum"},
		{"superpotencia", "superpotent"},
		{"supersecretarios", "supersecretari"},
		{"supersónicas", "superson"},
		{"supervisa", "supervis"},
		{"supervisar", "supervis"},
		{"supervisará", "supervis"},
		{"supervisarlos", "supervis"},
		{"supervisión", "supervision"},
		{"supervisó", "supervis"},
		{"supervisor", "supervisor"},
		{"supervisora", "supervisor"},
		{"supervivencia", "supervivent"},
		{"superwelter", "superwelt"},
		{"supieran", "sup"},
		{"supieron", "sup"},
		{"suple", "supl"},
		{"suplemento", "suplement"},
		{"suplente", "suplent"},
		{"suplentes", "suplent"},
		{"suplicantes", "suplic"},
		{"suplicar", "suplic"},
		{"suplicará", "suplic"},
		{"suplir", "supl"},
		{"supo", "sup"},
		{"supondría", "supondr"},
		{"supone", "supon"},
		{"suponemos", "supon"},
		{"suponer", "supon"},
		{"suponerse", "supon"},
		{"suponga", "supong"},
		{"supongo", "supong"},
		{"suponía", "supon"},
		{"suponiendo", "supon"},
		{"suposiciones", "suposicion"},
		{"supra", "supr"},
		{"suprema", "suprem"},
		{"supremacia", "supremaci"},
		{"supremacía", "supremac"},
		{"supremo", "suprem"},
		{"suprimir", "suprim"},
		{"suprimirlas", "suprim"},
		{"supuesta", "supuest"},
		{"supuestamente", "supuest"},
		{"supuestas", "supuest"},
		{"supuesto", "supuest"},
		{"supuestos", "supuest"},
		{"sur", "sur"},
		{"surcaba", "surc"},
		{"surcoreana", "surcorean"},
		{"surcoreanas", "surcorean"},
		{"surcoreanos", "surcorean"},
		{"sureste", "surest"},
		{"surge", "surg"},
		{"surgen", "surg"},
		{"surgida", "surg"},
		{"surgido", "surg"},
		{"surgidos", "surg"},
		{"surgiendo", "surg"},
		{"surgiera", "surg"},
		{"surgieron", "surg"},
		{"surgimiento", "surgimient"},
		{"surgio", "surgi"},
		{"surgió", "surg"},
		{"surgir", "surg"},
		{"surgirá", "surg"},
		{"surgirían", "surg"},
		{"surja", "surj"},
		{"surjan", "surj"},
		{"suroeste", "suroest"},
		{"surrealista", "surreal"},
		{"surta", "surt"},
		{"surtido", "surt"},
		{"surtiendo", "surt"},
		{"surtió", "surt"},
		{"surtir", "surt"},
		{"sus", "sus"},
		{"susan", "sus"},
		{"susana", "susan"},
		{"susceptible", "suscept"},
		{"susceptibles", "suscept"},
		{"suscitado", "suscit"},
		{"suscitó", "suscit"},
		{"suscribió", "suscrib"},
		{"suscribir", "suscrib"},
		{"suscripción", "suscripcion"},
		{"suscrita", "suscrit"},
		{"suscrito", "suscrit"},
		{"susiflor", "susiflor"},
		{"suspenda", "suspend"},
		{"suspende", "suspend"},
		{"suspenden", "suspend"},
		{"suspender", "suspend"},
		{"suspenderá", "suspend"},
		{"suspenderán", "suspend"},
		{"suspendida", "suspend"},
		{"suspendidas", "suspend"},
		{"suspendido", "suspend"},
		{"suspendidos", "suspend"},
		{"suspendiera", "suspend"},
		{"suspendieron", "suspend"},
		{"suspendió", "suspend"},
		{"suspendo", "suspend"},
		{"suspensión", "suspension"},
		{"suspensiones", "suspension"},
		{"suspiro", "suspir"},
		{"sustancia", "sustanci"},
		{"sustancial", "sustancial"},
		{"sustancias", "sustanci"},
		{"sustenta", "sustent"},
		{"sustentables", "sustent"},
		{"sustentación", "sustent"},
		{"sustentado", "sustent"},
		{"sustentar", "sustent"},
		{"sustente", "sustent"},
		{"sustento", "sustent"},
		{"sustitución", "sustitu"},
		{"sustituían", "sustitu"},
		{"sustituido", "sustitu"},
		{"sustituir", "sustitu"},
		{"sustituirá", "sustitu"},
		{"sustituiría", "sustitu"},
		{"sustitutiva", "sustitut"},
		{"sustituto", "sustitut"},
		{"sustituyamos", "sustitu"},
		{"sustituyan", "sustitu"},
		{"sustituye", "sustitu"},
		{"sustituyen", "sustitu"},
		{"sustituyendo", "sustitu"},
		{"sustituyó", "sustitu"},
		{"susto", "sust"},
		{"sustraía", "sustr"},
		{"sustrajo", "sustraj"},
		{"suya", "suy"},
		{"suyas", "suy"},
		{"suyo", "suy"},
		{"suyos", "suy"},
		{"svyt", "svyt"},
		{"swahili", "swahili"},
		{"swing", "swing"},
		{"swycord", "swycord"},
		{"sx", "sx"},
		{"sybilla", "sybill"},
		{"symington", "symington"},
		{"system", "system"},
		{"systems", "systems"},
		{"szekely", "szekely"},
		{"sznajder", "sznajd"},
		{"t", "t"},
		{"tabachín", "tabachin"},
		{"tabaco", "tabac"},
		{"tabasco", "tabasc"},
		{"tabe", "tab"},
		{"tabla", "tabl"},
		{"tablante", "tablant"},
		{"tablero", "tabler"},
		{"tableros", "tabler"},
		{"tablones", "tablon"},
		{"taboada", "tab"},
		{"tabulador", "tabul"},
		{"tachado", "tach"},
		{"tacho", "tach"},
		{"tachuelas", "tachuel"},
		{"tácita", "tacit"},
		{"tácitamente", "tacit"},
		{"taco", "tac"},
		{"tacón", "tacon"},
		{"tacos", "tac"},
		{"táctica", "tactic"},
		{"tácticas", "tactic"},
		{"tácticos", "tactic"},
		{"tacto", "tact"},
		{"taesa", "taes"},
		{"taffarel", "taffarel"},
		{"tagamet", "tagamet"},
		{"tagetes", "taget"},
		{"taiga", "taig"},
		{"tailandia", "tailandi"},
		{"taiwan", "taiw"},
		{"taiwán", "taiwan"},
		{"taiwandeses", "taiwandes"},
		{"tajante", "tajant"},
		{"takatoshi", "takatoshi"},
		{"take", "tak"},
		{"tal", "tal"},
		{"talada", "tal"},
		{"taladores", "talador"},
		{"taladro", "taladr"},
		{"tálamo", "talam"},
		{"talante", "talant"},
		{"talento", "talent"},
		{"talentos", "talent"},
		{"talentoso", "talent"},
		{"tales", "tal"},
		{"talismán", "talisman"},
		{"talla", "tall"},
		{"tallando", "tall"},
		{"tallar", "tall"},
		{"taller", "tall"},
		{"talleres", "taller"},
		{"talón", "talon"},
		{"talones", "talon"},
		{"talusa", "talus"},
		{"tamal", "tamal"},
		{"tamaño", "tamañ"},
		{"tamasopo", "tamasop"},
		{"tamaulipas", "tamaulip"},
		{"tamaulipeca", "tamaulipec"},
		{"tamayo", "tamay"},
		{"tambien", "tambi"},
		{"también", "tambien"},
		{"tambor", "tambor"},
		{"tambores", "tambor"},
		{"támesis", "tamesis"},
		{"tamez", "tamez"},
		{"tampico", "tampic"},
		{"tampoco", "tampoc"},
		{"tamps", "tamps"},
		{"tan", "tan"},
		{"tanasio", "tanasi"},
		{"tancanhuitz", "tancanhuitz"},
		{"tancredi", "tancredi"},
		{"tanda", "tand"},
		{"tane", "tan"},
		{"tanganhuato", "tanganhuat"},
		{"tangibles", "tangibl"},
		{"tanque", "tanqu"},
		{"tanques", "tanqu"},
		{"tanta", "tant"},
		{"tantas", "tant"},
		{"tanto", "tant"},
		{"tantos", "tant"},
		{"tanzanios", "tanzani"},
		{"tapa", "tap"},
		{"tapacalles", "tapacall"},
		{"tapachula", "tapachul"},
		{"tapar", "tap"},
		{"tapas", "tap"},
		{"tapatía", "tapat"},
		{"tapatías", "tapat"},
		{"tapatío", "tapati"},
		{"tapatíos", "tapati"},
		{"tapete", "tapet"},
		{"tapia", "tapi"},
		{"tapicería", "tapic"},
		{"tapitas", "tapit"},
		{"tapiz", "tapiz"},
		{"tapizados", "tapiz"},
		{"taponaron", "tapon"},
		{"taponeo", "tapone"},
		{"taquería", "taqu"},
		{"taquilla", "taquill"},
		{"taquillas", "taquill"},
		{"taquillera", "taquiller"},
		{"taquillero", "taquiller"},
		{"taquitos", "taquit"},
		{"tarado", "tar"},
		{"tarahumara", "tarahum"},
		{"tarahumaras", "tarahum"},
		{"tarasov", "tarasov"},
		{"tarciso", "tarcis"},
		{"tarda", "tard"},
		{"tardamos", "tard"},
		{"tardan", "tard"},
		{"tardándose", "tard"},
		{"tardanza", "tardanz"},
		{"tardar", "tard"},
		{"tardará", "tard"},
		{"tardarían", "tard"},
		{"tardaron", "tard"},
		{"tarde", "tard"},
		{"tardé", "tard"},
		{"tardes", "tard"},
		{"tardía", "tard"},
		{"tardo", "tard"},
		{"tardó", "tard"},
		{"tarea", "tare"},
		{"tareas", "tar"},
		{"tarifa", "tarif"},
		{"tarifario", "tarifari"},
		{"tarifarios", "tarifari"},
		{"tarifas", "tarif"},
		{"tarjeta", "tarjet"},
		{"tarjetahabientes", "tarjetahabient"},
		{"tarjetas", "tarjet"},
		{"tarjetón", "tarjeton"},
		{"tartajoso", "tartaj"},
		{"tasa", "tas"},
		{"tasas", "tas"},
		{"tascón", "tascon"},
		{"tasmania", "tasmani"},
		{"tataranietos", "tataraniet"},
		{"tatiana", "tatian"},
		{"tato", "tat"},
		{"tatuajes", "tatuaj"},
		{"tauro", "taur"},
		{"tavares", "tavar"},
		{"taxco", "taxc"},
		{"taxi", "taxi"},
		{"taxis", "taxis"},
		{"taxista", "taxist"},
		{"taxistas", "taxist"},
		{"taza", "taz"},
		{"tazas", "taz"},
		{"tazcón", "tazcon"},
		{"tazón", "tazon"},
		{"tazones", "tazon"},
		{"tcp", "tcp"},
		{"te", "te"},
		{"té", "te"},
		{"teaanque", "teaanqu"},
		{"team", "team"},
		{"teatral", "teatral"},
		{"teatrales", "teatral"},
		{"teatralmente", "teatral"},
		{"teatro", "teatr"},
		{"teatros", "teatr"},
		{"tec", "tec"},
		{"tecate", "tecat"},
		{"tech", "tech"},
		{"techado", "tech"},
		{"technologies", "technologi"},
		{"technology", "technology"},
		{"techo", "tech"},
		{"techos", "tech"},
		{"techumbre", "techumbr"},
		{"tecihuatlanque", "tecihuatlanqu"},
		{"tecla", "tecl"},
		{"tecladista", "teclad"},
		{"teclados", "tecl"},
		{"técnica", "tecnic"},
		{"técnicamente", "tecnic"},
		{"técnicas", "tecnic"},
		{"técnico", "tecnic"},
		{"técnicos", "tecnic"},
		{"tecnocracia", "tecnocraci"},
		{"tecnócratas", "tecnocrat"},
		{"tecnología", "tecnolog"},
		{"tecnologías", "tecnolog"},
		{"tecnológica", "tecnolog"},
		{"tecnológicamente", "tecnolog"},
		{"tecnológicas", "tecnolog"},
		{"tecnológico", "tecnolog"},
		{"tecnológicos", "tecnolog"},
		{"tecos", "tec"},
		{"tecpan", "tecp"},
		{"tecuniapa", "tecuniap"},
		{"teddy", "teddy"},
		{"teddys", "teddys"},
		{"tedioso", "tedios"},
		{"teherán", "teh"},
		{"tehuacan", "tehuac"},
		{"tehuacán", "tehuacan"},
		{"tehuitzingo", "tehuitzing"},
		{"teissier", "teissi"},
		{"tejado", "tej"},
		{"tejas", "tej"},
		{"tejeda", "tejed"},
		{"tejido", "tej"},
		{"tejidos", "tej"},
		{"tel", "tel"},
		{"tela", "tel"},
		{"telas", "tel"},
		{"tele", "tel"},
		{"telecom", "telecom"},
		{"telecomunicaciones", "telecomun"},
		{"telefonía", "telefon"},
		{"telefónica", "telefon"},
		{"telefónicamente", "telefon"},
		{"telefónicas", "telefon"},
		{"telefónico", "telefon"},
		{"telefónicos", "telefon"},
		{"telefonistas", "telefon"},
		{"teléfono", "telefon"},
		{"teléfonos", "telefon"},
		{"teleguía", "telegu"},
		{"telehogares", "telehogar"},
		{"telenovela", "telenovel"},
		{"telenovelas", "telenovel"},
		{"telesecundaria", "telesecundari"},
		{"teleseries", "teleseri"},
		{"televicine", "televicin"},
		{"televidente", "televident"},
		{"televidentes", "televident"},
		{"televisa", "televis"},
		{"televisar", "televis"},
		{"televisión", "television"},
		{"televisiones", "television"},
		{"televisiva", "televis"},
		{"televisivas", "televis"},
		{"televisivo", "televis"},
		{"televisivos", "televis"},
		{"televisora", "televisor"},
		{"telex", "telex"},
		{"téllez", "tellez"},
		{"tello", "tell"},
		{"telmex", "telmex"},
		{"teloloapan", "teloloap"},
		{"teloloapense", "teloloapens"},
		{"telones", "telon"},
		{"tema", "tem"},
		{"temas", "tem"},
		{"temática", "temat"},
		{"temáticas", "temat"},
		{"temblar", "tembl"},
		{"temblor", "temblor"},
		{"temblores", "temblor"},
		{"teme", "tem"},
		{"temecula", "temecul"},
		{"temen", "tem"},
		{"temeraria", "temerari"},
		{"temerariamente", "temerari"},
		{"temerarios", "temerari"},
		{"temeroso", "temer"},
		{"temía", "tem"},
		{"temible", "temibl"},
		{"temido", "tem"},
		{"temiendo", "tem"},
		{"temo", "tem"},
		{"temor", "temor"},
		{"temores", "temor"},
		{"témpano", "tempan"},
		{"temperamento", "temperament"},
		{"temperatura", "temperatur"},
		{"temperaturas", "temperatur"},
		{"tempestad", "tempest"},
		{"templado", "templ"},
		{"templados", "templ"},
		{"temple", "templ"},
		{"templete", "templet"},
		{"templo", "templ"},
		{"templos", "templ"},
		{"temporada", "tempor"},
		{"temporadas", "tempor"},
		{"temporal", "temporal"},
		{"temporales", "temporal"},
		{"temporalmente", "temporal"},
		{"temprana", "tempran"},
		{"tempranamente", "tempran"},
		{"tempranero", "tempraner"},
		{"temprano", "tempran"},
		{"tempranos", "tempran"},
		{"temulento", "temulent"},
		{"tención", "tencion"},
		{"tendajones", "tendajon"},
		{"tendencia", "tendenci"},
		{"tendencias", "tendenci"},
		{"tenderá", "tend"},
		{"tenderán", "tend"},
		{"tendido", "tend"},
		{"tendientes", "tendient"},
		{"tendimos", "tend"},
		{"tendió", "tend"},
		{"tendón", "tendon"},
		{"tendrá", "tendr"},
		{"tendrán", "tendran"},
		{"tendré", "tendr"},
		{"tendremos", "tendr"},
		{"tendría", "tendr"},
		{"tendríamos", "tendr"},
		{"tendrían", "tendr"},
		{"tenemos", "ten"},
		{"tenencia", "tenenci"},
		{"tenencias", "tenenci"},
		{"tener", "ten"},
		{"tenerife", "tenerif"},
		{"tenerlas", "ten"},
		{"tenessee", "tenesse"},
		{"tenga", "teng"},
		{"tengamos", "teng"},
		{"tengan", "teng"},
		{"tengas", "teng"},
		{"tengo", "teng"},
		{"tenia", "teni"},
		{"tenía", "ten"},
		{"teniamos", "teni"},
		{"teníamos", "ten"},
		{"tenían", "ten"},
		{"tenida", "ten"},
		{"tenido", "ten"},
		{"teniendo", "ten"},
		{"teniente", "tenient"},
		{"tenis", "tenis"},
		{"tenistas", "tenist"},
		{"tenístico", "tenist"},
		{"tennessee", "tennesse"},
		{"tenochtitlan", "tenochtitl"},
		{"tenor", "tenor"},
		{"tensa", "tens"},
		{"tensión", "tension"},
		{"tensiones", "tension"},
		{"tentación", "tentacion"},
		{"tentativa", "tentat"},
		{"teo", "teo"},
		{"teodora", "teodor"},
		{"teodoro", "teodor"},
		{"teología", "teolog"},
		{"teólogos", "teolog"},
		{"teoría", "teor"},
		{"teorías", "teor"},
		{"teórica", "teoric"},
		{"teóricamente", "teoric"},
		{"teóricas", "teoric"},
		{"teóricos", "teoric"},
		{"tepanco", "tepanc"},
		{"tepatitlán", "tepatitlan"},
		{"tepeaca", "tepeac"},
		{"tepehuanes", "tepehuan"},
		{"tepeyac", "tepeyac"},
		{"tepic", "tepic"},
		{"tepochcalli", "tepochcalli"},
		{"tepozotlán", "tepozotlan"},
		{"tequila", "tequil"},
		{"tequisquiápan", "tequisquiap"},
		{"terán", "teran"},
		{"terapéuticas", "terapeut"},
		{"terapéuticos", "terapeut"},
		{"terapia", "terapi"},
		{"terapias", "terapi"},
		{"tercer", "terc"},
		{"tercera", "tercer"},
		{"terceras", "tercer"},
		{"tercero", "tercer"},
		{"terceros", "tercer"},
		{"tercia", "terci"},
		{"tercias", "terci"},
		{"tercio", "terci"},
		{"terciopelo", "terciopel"},
		{"tercios", "terci"},
		{"teresa", "teres"},
		{"tergiversado", "tergivers"},
		{"termes", "term"},
		{"térmica", "termic"},
		{"térmicas", "termic"},
		{"termina", "termin"},
		{"terminación", "termin"},
		{"terminaciones", "termin"},
		{"terminada", "termin"},
		{"terminadas", "termin"},
		{"terminado", "termin"},
		{"terminados", "termin"},
		{"terminal", "terminal"},
		{"terminales", "terminal"},
		{"terminamos", "termin"},
		{"terminan", "termin"},
		{"terminando", "termin"},
		{"terminar", "termin"},
		{"terminará", "termin"},
		{"terminarán", "termin"},
		{"terminaría", "termin"},
		{"terminarían", "termin"},
		{"terminarlo", "termin"},
		{"terminaron", "termin"},
		{"terminarse", "termin"},
		{"terminator", "terminator"},
		{"termine", "termin"},
		{"terminemos", "termin"},
		{"terminen", "termin"},
		{"termino", "termin"},
		{"término", "termin"},
		{"terminó", "termin"},
		{"terminología", "terminolog"},
		{"términos", "termin"},
		{"termómetro", "termometr"},
		{"terna", "tern"},
		{"ternura", "ternur"},
		{"térprete", "terpret"},
		{"terracota", "terracot"},
		{"terranova", "terranov"},
		{"terrateniente", "terratenient"},
		{"terratenientes", "terratenient"},
		{"terrazas", "terraz"},
		{"terremoto", "terremot"},
		{"terrenales", "terrenal"},
		{"terreno", "terren"},
		{"terrenos", "terren"},
		{"terrestre", "terrestr"},
		{"terrible", "terribl"},
		{"terribles", "terribl"},
		{"territorial", "territorial"},
		{"territorio", "territori"},
		{"territorios", "territori"},
		{"terror", "terror"},
		{"terrorismo", "terror"},
		{"terrorista", "terror"},
		{"terroristas", "terror"},
		{"terry", "terry"},
		{"tés", "tes"},
		{"tesis", "tesis"},
		{"tésis", "tesis"},
		{"tesitura", "tesitur"},
		{"tesorería", "tesor"},
		{"tesorero", "tesorer"},
		{"tesoreros", "tesorer"},
		{"tesoro", "tesor"},
		{"tesoros", "tesor"},
		{"testigo", "testig"},
		{"testigos", "testig"},
		{"testimonial", "testimonial"},
		{"testimoniar", "testimoni"},
		{"testimonio", "testimoni"},
		{"testimonios", "testimoni"},
		{"testosterona", "testosteron"},
		{"tetela", "tetel"},
		{"tetramestres", "tetramestr"},
		{"texana", "texan"},
		{"texano", "texan"},
		{"texanos", "texan"},
		{"texas", "tex"},
		{"texcoco", "texcoc"},
		{"textil", "textil"},
		{"textilera", "textiler"},
		{"textiles", "textil"},
		{"texto", "text"},
		{"textos", "text"},
		{"textualmente", "textual"},
		{"texturas", "textur"},
		{"tez", "tez"},
		{"tezoyuca", "tezoyuc"},
		{"the", "the"},
		{"thelonius", "thelonius"},
		{"theodore", "theodor"},
		{"therese", "theres"},
		{"thiner", "thin"},
		{"things", "things"},
		{"thomas", "thom"},
		{"thompson", "thompson"},
		{"thomson", "thomson"},
		{"thorn", "thorn"},
		{"three", "thre"},
		{"thriller", "thrill"},
		{"thurlow", "thurlow"},
		{"thurman", "thurm"},
		{"ti", "ti"},
		{"tía", "tia"},
		{"tianguis", "tianguis"},
		{"tías", "tias"},
		{"tiberio", "tiberi"},
		{"tibet", "tibet"},
		{"tibio", "tibi"},
		{"tichavski", "tichavski"},
		{"ticktin", "ticktin"},
		{"tiempecito", "tiempecit"},
		{"tiempo", "tiemp"},
		{"tiempos", "tiemp"},
		{"tienda", "tiend"},
		{"tiendas", "tiend"},
		{"tiende", "tiend"},
		{"tienden", "tiend"},
		{"tiene", "tien"},
		{"tienen", "tien"},
		{"tienes", "tien"},
		{"tienta", "tient"},
		{"tiernamente", "tiern"},
		{"tierra", "tierr"},
		{"tierras", "tierr"},
		{"tigre", "tigr"},
		{"tigres", "tigr"},
		{"tigrillas", "tigrill"},
		{"tigrillos", "tigrill"},
		{"tijerina", "tijerin"},
		{"tijuana", "tijuan"},
		{"tijuanenses", "tijuanens"},
		{"tildar", "tild"},
		{"tim", "tim"},
		{"timbiriche", "timbirich"},
		{"timbre", "timbr"},
		{"timbres", "timbr"},
		{"time", "tim"},
		{"times", "tim"},
		{"tímida", "tim"},
		{"tímidamente", "timid"},
		{"tímido", "tim"},
		{"timón", "timon"},
		{"timonel", "timonel"},
		{"timorato", "timorat"},
		{"timoteo", "timote"},
		{"timothy", "timothy"},
		{"tino", "tin"},
		{"tinta", "tint"},
		{"tintas", "tint"},
		{"tinto", "tint"},
		{"tío", "tio"},
		{"tíos", "tios"},
		{"tiosulfato", "tiosulfat"},
		{"típicamente", "tipic"},
		{"típico", "tipic"},
		{"típicos", "tipic"},
		{"tipifica", "tipif"},
		{"tipo", "tip"},
		{"tipos", "tip"},
		{"tips", "tips"},
		{"tira", "tir"},
		{"tirada", "tir"},
		{"tiradero", "tirader"},
		{"tiraderos", "tirader"},
		{"tirado", "tir"},
		{"tirados", "tir"},
		{"tirando", "tir"},
		{"tiránica", "tiran"},
		{"tirantes", "tirant"},
		{"tirar", "tir"},
		{"tirarán", "tir"},
		{"tirarlos", "tir"},
		{"tiras", "tir"},
		{"tire", "tir"},
		{"tiro", "tir"},
		{"tiró", "tir"},
		{"tiroideas", "tiroid"},
		{"tiroides", "tiroid"},
		{"tiros", "tir"},
		{"tiroteo", "tirote"},
		{"tiroteos", "tirote"},
		{"tirzo", "tirz"},
		{"tisa", "tis"},
		{"titanic", "titanic"},
		{"titanica", "titan"},
		{"titánica", "titan"},
		{"titanio", "titani"},
		{"titipuchal", "titipuchal"},
		{"tito", "tit"},
		{"titula", "titul"},
		{"titulada", "titul"},
		{"titulado", "titul"},
		{"titular", "titul"},
		{"titulares", "titular"},
		{"titularidad", "titular"},
		{"título", "titul"},
		{"tituló", "titul"},
		{"títulos", "titul"},
		{"tixkokob", "tixkokob"},
		{"tlacamécatl", "tlacamecatl"},
		{"tlacotepec", "tlacotepec"},
		{"tlahuapán", "tlahuapan"},
		{"tlalchapa", "tlalchap"},
		{"tlalpan", "tlalp"},
		{"tlaltenango", "tlaltenang"},
		{"tlamacazques", "tlamacazqu"},
		{"tlatelolco", "tlatelolc"},
		{"tlatlahuquitepec", "tlatlahuquitepec"},
		{"tlatoani", "tlatoani"},
		{"tlatoanis", "tlatoanis"},
		{"tlaxcala", "tlaxcal"},
		{"tlaxcalteca", "tlaxcaltec"},
		{"tlaxcaltecas", "tlaxcaltec"},
		{"tlc", "tlc"},
		{"to", "to"},
		{"tobías", "tob"},
		{"tobillo", "tobill"},
		{"toca", "toc"},
		{"tocaba", "toc"},
		{"tocaban", "toc"},
		{"tocado", "toc"},
		{"tocan", "toc"},
		{"tocar", "toc"},
		{"tocará", "toc"},
		{"tocarán", "toc"},
		{"tocarle", "toc"},
		{"tocco", "tocc"},
		{"tocó", "toc"},
		{"toda", "tod"},
		{"todas", "tod"},
		{"todavía", "todav"},
		{"todd", "todd"},
		{"todo", "tod"},
		{"todos", "tod"},
		{"toilette", "toilett"},
		{"toji", "toji"},
		{"tokio", "toki"},
		{"toledo", "toled"},
		{"tolerada", "toler"},
		{"tolerancia", "toler"},
		{"tolerando", "toler"},
		{"tolerante", "toler"},
		{"tolerantes", "toler"},
		{"tolerarse", "toler"},
		{"toletero", "toleter"},
		{"toluca", "toluc"},
		{"tom", "tom"},
		{"toma", "tom"},
		{"tomaba", "tom"},
		{"tomaban", "tom"},
		{"tomada", "tom"},
		{"tomadas", "tom"},
		{"tomado", "tom"},
		{"tomadores", "tomador"},
		{"tomados", "tom"},
		{"toman", "tom"},
		{"tomando", "tom"},
		{"tomándola", "tom"},
		{"tomar", "tom"},
		{"tomara", "tom"},
		{"tomará", "tom"},
		{"tomaran", "tom"},
		{"tomarán", "tom"},
		{"tomaría", "tom"},
		{"tomarían", "tom"},
		{"tomarla", "tom"},
		{"tomarle", "tom"},
		{"tomarlo", "tom"},
		{"tomarnos", "tom"},
		{"tomaron", "tom"},
		{"tomarse", "tom"},
		{"tomas", "tom"},
		{"tomás", "tomas"},
		{"tomate", "tomat"},
		{"tome", "tom"},
		{"tomé", "tom"},
		{"tomemos", "tom"},
		{"tomen", "tom"},
		{"tómese", "tomes"},
		{"tomlinson", "tomlinson"},
		{"tomo", "tom"},
		{"tomó", "tom"},
		{"tona", "ton"},
		{"tonada", "ton"},
		{"tonalidades", "tonal"},
		{"tonelada", "tonel"},
		{"toneladas", "tonel"},
		{"tonelaje", "tonelaj"},
		{"tónica", "tonic"},
		{"tónicos", "tonic"},
		{"toninho", "toninh"},
		{"tono", "ton"},
		{"toño", "toñ"},
		{"tontería", "tont"},
		{"tonto", "tont"},
		{"tontos", "tont"},
		{"tony", "tony"},
		{"toole", "tool"},
		{"topa", "top"},
		{"toparon", "top"},
		{"toparse", "top"},
		{"topaz", "topaz"},
		{"tope", "top"},
		{"topes", "top"},
		{"topete", "topet"},
		{"topeteaba", "topet"},
		{"topia", "topi"},
		{"tópico", "topic"},
		{"tópicos", "topic"},
		{"topo", "top"},
		{"topografía", "topograf"},
		{"topologías", "topolog"},
		{"toque", "toqu"},
		{"toques", "toqu"},
		{"toqui", "toqui"},
		{"torá", "tor"},
		{"tórax", "torax"},
		{"torcer", "torc"},
		{"toreado", "tor"},
		{"toreados", "tor"},
		{"toreándolo", "tor"},
		{"torear", "tor"},
		{"toreara", "tor"},
		{"torearlo", "tor"},
		{"toreó", "tore"},
		{"torero", "torer"},
		{"toreros", "torer"},
		{"torio", "tori"},
		{"tormenta", "torment"},
		{"tormentas", "torment"},
		{"tornado", "torn"},
		{"tornados", "torn"},
		{"tornar", "torn"},
		{"tornen", "torn"},
		{"torneo", "torne"},
		{"torneos", "torne"},
		{"tornillo", "tornill"},
		{"tornillos", "tornill"},
		{"torniquete", "torniquet"},
		{"torno", "torn"},
		{"toro", "tor"},
		{"toronto", "toront"},
		{"toros", "tor"},
		{"torpedearon", "torped"},
		{"torpeza", "torpez"},
		{"torrado", "torr"},
		{"torralba", "torralb"},
		{"torre", "torr"},
		{"torrencial", "torrencial"},
		{"torrenciales", "torrencial"},
		{"torrente", "torrent"},
		{"torreon", "torreon"},
		{"torreón", "torreon"},
		{"torres", "torr"},
		{"torrescano", "torrescan"},
		{"torresmochas", "torresmoch"},
		{"torricelli", "torricelli"},
		{"tortícolis", "torticolis"},
		{"tortilla", "tortill"},
		{"tortillas", "tortill"},
		{"tortillero", "tortiller"},
		{"tortuga", "tortug"},
		{"tortugueras", "tortuguer"},
		{"tortuguismo", "tortugu"},
		{"torturados", "tortur"},
		{"torturaron", "tortur"},
		{"torturas", "tortur"},
		{"torturoso", "tortur"},
		{"tos", "tos"},
		{"toscano", "toscan"},
		{"tosferina", "tosferin"},
		{"tostado", "tost"},
		{"tota", "tot"},
		{"total", "total"},
		{"totales", "total"},
		{"totalidad", "total"},
		{"totalizó", "totaliz"},
		{"totalmente", "total"},
		{"totopos", "totop"},
		{"tottenham", "tottenham"},
		{"touché", "touch"},
		{"tour", "tour"},
		{"tovar", "tov"},
		{"tóxica", "toxic"},
		{"tóxicas", "toxic"},
		{"tóxicos", "toxic"},
		{"toyota", "toyot"},
		{"tpc", "tpc"},
		{"tqm", "tqm"},
		{"trabado", "trab"},
		{"trabaja", "trabaj"},
		{"trabajaba", "trabaj"},
		{"trabajaban", "trabaj"},
		{"trabajada", "trabaj"},
		{"trabajado", "trabaj"},
		{"trabajador", "trabaj"},
		{"trabajadora", "trabaj"},
		{"trabajadoras", "trabaj"},
		{"trabajadores", "trabaj"},
		{"trabajamos", "trabaj"},
		{"trabajan", "trabaj"},
		{"trabajando", "trabaj"},
		{"trabajar", "trabaj"},
		{"trabajara", "trabaj"},
		{"trabajará", "trabaj"},
		{"trabajarán", "trabaj"},
		{"trabajaron", "trabaj"},
		{"trabajemos", "trabaj"},
		{"trabajen", "trabaj"},
		{"trabajo", "trabaj"},
		{"trabajó", "trabaj"},
		{"trabajos", "trabaj"},
		{"trabas", "trab"},
		{"trabó", "trab"},
		{"tracción", "traccion"},
		{"tracen", "trac"},
		{"track", "track"},
		{"tractor", "tractor"},
		{"tradición", "tradicion"},
		{"tradicional", "tradicional"},
		{"tradicionales", "tradicional"},
		{"tradicionalmente", "tradicional"},
		{"tradiciones", "tradicion"},
		{"traducción", "traduccion"},
		{"traducciones", "traduccion"},
		{"traduce", "traduc"},
		{"traducida", "traduc"},
		{"traducido", "traduc"},
		{"traducidos", "traduc"},
		{"traducir", "traduc"},
		{"traduciría", "traduc"},
		{"traducirse", "traduc"},
		{"tradujo", "traduj"},
		{"trae", "tra"},
		{"traemos", "tra"},
		{"traen", "tra"},
		{"traer", "tra"},
		{"traerá", "tra"},
		{"traerán", "tra"},
		{"traeré", "tra"},
		{"traería", "tra"},
		{"traerla", "tra"},
		{"traernos", "tra"},
		{"traes", "tra"},
		{"traficante", "trafic"},
		{"trafico", "trafic"},
		{"tráfico", "trafic"},
		{"tragar", "trag"},
		{"tragarse", "trag"},
		{"tragedia", "tragedi"},
		{"trágica", "tragic"},
		{"trágico", "tragic"},
		{"tragó", "trag"},
		{"tragos", "trag"},
		{"trague", "trag"},
		{"traía", "tra"},
		{"traían", "tra"},
		{"traición", "traicion"},
		{"traicionado", "traicion"},
		{"traicionados", "traicion"},
		{"traidor", "traidor"},
		{"traídos", "traid"},
		{"traiga", "traig"},
		{"traigan", "traig"},
		{"traigo", "traig"},
		{"trailer", "trail"},
		{"trailero", "trailer"},
		{"traileros", "trailer"},
		{"trailers", "trailers"},
		{"traje", "traj"},
		{"trajeron", "trajeron"},
		{"trajes", "traj"},
		{"trajo", "traj"},
		{"trama", "tram"},
		{"tramitación", "tramit"},
		{"tramitando", "tramit"},
		{"tramitar", "tramit"},
		{"trámite", "tramit"},
		{"trámites", "tramit"},
		{"tramitó", "tramit"},
		{"tramitología", "tramitolog"},
		{"tramo", "tram"},
		{"tramos", "tram"},
		{"trampa", "tramp"},
		{"trampas", "tramp"},
		{"tramposo", "trampos"},
		{"trancas", "tranc"},
		{"trancazo", "trancaz"},
		{"trance", "tranc"},
		{"tranquila", "tranquil"},
		{"tranquilamente", "tranquil"},
		{"tranquilas", "tranquil"},
		{"tranquilidad", "tranquil"},
		{"tranquilizara", "tranquiliz"},
		{"tranquilizarlo", "tranquiliz"},
		{"tranquilizó", "tranquiliz"},
		{"tranquilo", "tranquil"},
		{"tranquilos", "tranquil"},
		{"trans", "trans"},
		{"transa", "trans"},
		{"transacción", "transaccion"},
		{"transacciones", "transaccion"},
		{"transamos", "trans"},
		{"transborda", "transbord"},
		{"transbordador", "transbord"},
		{"transbordadores", "transbord"},
		{"transbordar", "transbord"},
		{"transbordo", "transbord"},
		{"transcribe", "transcrib"},
		{"transcurra", "transcurr"},
		{"transcurrido", "transcurr"},
		{"transcurridos", "transcurr"},
		{"transcurriendo", "transcurr"},
		{"transcurrieron", "transcurr"},
		{"transcurrir", "transcurr"},
		{"transcurso", "transcurs"},
		{"transero", "transer"},
		{"transeros", "transer"},
		{"transeúnte", "transeunt"},
		{"transeúntes", "transeunt"},
		{"transexual", "transexual"},
		{"transexuales", "transexual"},
		{"transferencia", "transferent"},
		{"transferencias", "transferent"},
		{"transferir", "transfer"},
		{"transfieren", "transfier"},
		{"transfirió", "transfir"},
		{"transformación", "transform"},
		{"transformaciones", "transform"},
		{"transformado", "transform"},
		{"transformador", "transform"},
		{"transforman", "transform"},
		{"transformando", "transform"},
		{"transformar", "transform"},
		{"transformaren", "transformar"},
		{"transformaría", "transform"},
		{"transformaron", "transform"},
		{"transformarse", "transform"},
		{"transfronterizas", "transfronteriz"},
		{"transfusiones", "transfusion"},
		{"transición", "transicion"},
		{"transija", "transij"},
		{"transita", "transit"},
		{"transitaban", "transit"},
		{"transitada", "transit"},
		{"transitado", "transit"},
		{"transitados", "transit"},
		{"transitar", "transit"},
		{"transitará", "transit"},
		{"transitarán", "transit"},
		{"tránsito", "transit"},
		{"transitó", "transit"},
		{"transitorio", "transitori"},
		{"transmisión", "transmision"},
		{"transmisiones", "transmision"},
		{"transmita", "transmit"},
		{"transmitan", "transmit"},
		{"transmite", "transmit"},
		{"transmiten", "transmit"},
		{"transmitida", "transmit"},
		{"transmitido", "transmit"},
		{"transmitieran", "transmit"},
		{"transmitieron", "transmit"},
		{"transmitir", "transmit"},
		{"transmitirá", "transmit"},
		{"transmitirán", "transmit"},
		{"transnacional", "transnacional"},
		{"transnacionales", "transnacional"},
		{"transparencia", "transparent"},
		{"transparentar", "transparent"},
		{"transparentarse", "transparent"},
		{"transparente", "transparent"},
		{"transparentes", "transparent"},
		{"transpeninsular", "transpeninsul"},
		{"transport", "transport"},
		{"transporta", "transport"},
		{"transportaba", "transport"},
		{"transportaban", "transport"},
		{"transportación", "transport"},
		{"transportaciones", "transport"},
		{"transportada", "transport"},
		{"transportadoras", "transport"},
		{"transportan", "transport"},
		{"transportando", "transport"},
		{"transportar", "transport"},
		{"transportaría", "transport"},
		{"transportarse", "transport"},
		{"transporte", "transport"},
		{"transportes", "transport"},
		{"transportista", "transport"},
		{"transportistas", "transport"},
		{"transversales", "transversal"},
		{"tranvías", "tranv"},
		{"trapeadores", "trapeador"},
		{"trapeas", "trap"},
		{"tras", "tras"},
		{"trasatlántico", "trasatlant"},
		{"trascendencia", "trascendent"},
		{"trascendental", "trascendental"},
		{"trascendentales", "trascendental"},
		{"trascender", "trascend"},
		{"trasera", "traser"},
		{"traseras", "traser"},
		{"trasero", "traser"},
		{"traseros", "traser"},
		{"trasfondo", "trasfond"},
		{"trasfondos", "trasfond"},
		{"traslación", "traslacion"},
		{"traslada", "trasl"},
		{"trasladaba", "traslad"},
		{"trasladado", "traslad"},
		{"trasladados", "traslad"},
		{"trasladar", "traslad"},
		{"trasladarán", "traslad"},
		{"trasladarla", "traslad"},
		{"trasladarse", "traslad"},
		{"trasládese", "traslades"},
		{"traslado", "trasl"},
		{"trasladó", "traslad"},
		{"traslados", "trasl"},
		{"trasmisión", "trasmision"},
		{"trasmite", "trasmit"},
		{"trasmitirlo", "trasmit"},
		{"trasnacional", "trasnacional"},
		{"trasnacionales", "trasnacional"},
		{"trasnacionalización", "trasnacionaliz"},
		{"", ""},
		{"trasplantados", "trasplant"},
		{"trasplante", "trasplant"},
		{"trasplantes", "trasplant"},
		{"trasporte", "trasport"},
		{"traste", "trast"},
		{"trasteo", "traste"},
		{"trastoca", "trastoc"},
		{"trastocan", "trastoc"},
		{"trastorna", "trastorn"},
		{"trastornó", "trastorn"},
		{"trastornos", "trastorn"},
		{"trata", "trat"},
		{"trataba", "trat"},
		{"trataban", "trat"},
		{"tratada", "trat"},
		{"tratadas", "trat"},
		{"tratado", "trat"},
		{"tratadoras", "tratador"},
		{"tratados", "trat"},
		{"tratamiento", "tratamient"},
		{"tratamientos", "tratamient"},
		{"tratamos", "trat"},
		{"tratan", "trat"},
		{"tratando", "trat"},
		{"tratándose", "trat"},
		{"tratar", "trat"},
		{"tratara", "trat"},
		{"tratará", "trat"},
		{"tratarán", "trat"},
		{"trataremos", "trat"},
		{"trataría", "trat"},
		{"trataron", "trat"},
		{"tratarse", "trat"},
		{"tratas", "trat"},
		{"trate", "trat"},
		{"traté", "trat"},
		{"traten", "trat"},
		{"trato", "trat"},
		{"trató", "trat"},
		{"tratos", "trat"},
		{"traumatismo", "traumat"},
		{"traumatología", "traumatolog"},
		{"través", "traves"},
		{"travesaño", "travesañ"},
		{"travieso", "travies"},
		{"travis", "travis"},
		{"trayecto", "trayect"},
		{"trayectoria", "trayectori"},
		{"trayendo", "trayend"},
		{"trazadas", "traz"},
		{"trazado", "traz"},
		{"trazo", "traz"},
		{"trazó", "traz"},
		{"trazos", "traz"},
		{"trece", "trec"},
		{"trecho", "trech"},
		{"trechos", "trech"},
		{"tregua", "tregu"},
		{"treinta", "treint"},
		{"trejo", "trej"},
		{"tremenda", "tremend"},
		{"tremendamente", "tremend"},
		{"tremendas", "tremend"},
		{"tremendo", "tremend"},
		{"tremendos", "tremend"},
		{"tren", "tren"},
		{"trenes", "tren"},
		{"trenza", "trenz"},
		{"trepó", "trep"},
		{"tres", "tres"},
		{"tress", "tress"},
		{"treta", "tret"},
		{"treto", "tret"},
		{"trevi", "trevi"},
		{"treviñada", "treviñ"},
		{"treviñista", "treviñ"},
		{"treviño", "treviñ"},
		{"tri", "tri"},
		{"triana", "trian"},
		{"triángulo", "triangul"},
		{"tribuna", "tribun"},
		{"tribunal", "tribunal"},
		{"tribunales", "tribunal"},
		{"tribunas", "tribun"},
		{"tributarias", "tributari"},
		{"tributario", "tributari"},
		{"tributarle", "tribut"},
		{"tributos", "tribut"},
		{"tricampeón", "tricampeon"},
		{"tricolor", "tricolor"},
		{"tricolores", "tricolor"},
		{"tridimensional", "tridimensional"},
		{"trigésimo", "trigesim"},
		{"trigo", "trig"},
		{"trigon", "trigon"},
		{"trilateral", "trilateral"},
		{"trillado", "trill"},
		{"trimestre", "trimestr"},
		{"trinchera", "trincher"},
		{"trinidad", "trinid"},
		{"trío", "tri"},
		{"tríos", "tri"},
		{"tripa", "trip"},
		{"tripartita", "tripartit"},
		{"triple", "tripl"},
		{"tripleta", "triplet"},
		{"triplicaron", "triplic"},
		{"tripulación", "tripul"},
		{"tripulado", "tripul"},
		{"tripulados", "tripul"},
		{"tripulando", "tripul"},
		{"tripulantes", "tripul"},
		{"tripular", "tripul"},
		{"tripulará", "tripul"},
		{"triquiñuela", "triquiñuel"},
		{"tris", "tris"},
		{"tristán", "tristan"},
		{"triste", "trist"},
		{"tristemente", "tristement"},
		{"tristes", "trist"},
		{"tristeza", "tristez"},
		{"tritt", "tritt"},
		{"triunfa", "triunf"},
		{"triunfado", "triunf"},
		{"triunfador", "triunfador"},
		{"triunfadora", "triunfador"},
		{"triunfadores", "triunfador"},
		{"triunfal", "triunfal"},
		{"triunfalismo", "triunfal"},
		{"triunfalista", "triunfal"},
		{"triunfalmente", "triunfal"},
		{"triunfantes", "triunfant"},
		{"triunfar", "triunf"},
		{"triunfaría", "triunf"},
		{"triunfaron", "triunf"},
		{"triunfo", "triunf"},
		{"triunfos", "triunf"},
		{"trivial", "trivial"},
		{"trizados", "triz"},
		{"trofeo", "trofe"},
		{"trofeos", "trofe"},
		{"trolebús", "trolebus"},
		{"trompeta", "trompet"},
		{"trompetero", "trompeter"},
		{"trompetista", "trompet"},
		{"trompetística", "trompetist"},
		{"troncales", "troncal"},
		{"tronco", "tronc"},
		{"troncos", "tronc"},
		{"trono", "tron"},
		{"tropa", "trop"},
		{"tropas", "trop"},
		{"tropezar", "tropez"},
		{"tropezón", "tropezon"},
		{"tropicosas", "tropic"},
		{"tropiezos", "tropiez"},
		{"trota", "trot"},
		{"trotamundos", "trotamund"},
		{"trotando", "trot"},
		{"trotar", "trot"},
		{"trote", "trot"},
		{"trousseau", "trousseau"},
		{"trout", "trout"},
		{"troy", "troy"},
		{"troyanos", "troyan"},
		{"truenan", "truen"},
		{"truene", "truen"},
		{"trujillo", "trujill"},
		{"truman", "trum"},
		{"trump", "trump"},
		{"trust", "trust"},
		{"tryon", "tryon"},
		{"tsi", "tsi"},
		{"tsmbién", "tsmbien"},
		{"tsubame", "tsubam"},
		{"tsuru", "tsuru"},
		{"tu", "tu"},
		{"tú", "tu"},
		{"tubacero", "tubacer"},
		{"tubería", "tub"},
		{"tuberías", "tub"},
		{"tubo", "tub"},
		{"tubos", "tub"},
		{"tuercas", "tuerc"},
		{"tuerto", "tuert"},
		{"tulancingo", "tulancing"},
		{"tules", "tul"},
		{"tulin", "tulin"},
		{"tull", "tull"},
		{"tulle", "tull"},
		{"tum", "tum"},
		{"tumba", "tumb"},
		{"tumbado", "tumb"},
		{"tumbas", "tumb"},
		{"tumor", "tumor"},
		{"tumoral", "tumoral"},
		{"tumores", "tumor"},
		{"tumultuosa", "tumultu"},
		{"túnel", "tunel"},
		{"túneles", "tunel"},
		{"tunguska", "tungusk"},
		{"túnicas", "tunic"},
		{"tupac", "tupac"},
		{"tupamaros", "tupamar"},
		{"turbias", "turbi"},
		{"turbina", "turbin"},
		{"turbo", "turb"},
		{"turcos", "turc"},
		{"turismo", "turism"},
		{"turista", "turist"},
		{"turistas", "turist"},
		{"turística", "turist"},
		{"turísticas", "turist"},
		{"turistico", "turist"},
		{"turístico", "turist"},
		{"turísticos", "turist"},
		{"turna", "turn"},
		{"turnbull", "turnbull"},
		{"turneriano", "turnerian"},
		{"turno", "turn"},
		{"turnó", "turn"},
		{"turnos", "turn"},
		{"turquía", "turqu"},
		{"tus", "tus"},
		{"tutelar", "tutel"},
		{"tuve", "tuv"},
		{"tuviera", "tuv"},
		{"tuvieran", "tuv"},
		{"tuvieron", "tuv"},
		{"tuvimos", "tuv"},
		{"tuvo", "tuv"},
		{"tuxpan", "tuxp"},
		{"tuxtla", "tuxtl"},
		{"tuya", "tuy"},
		{"tuyo", "tuy"},
		{"tuzo", "tuz"},
		{"tuzos", "tuz"},
		{"tv", "tv"},
		{"tyson", "tyson"},
		{"u", "u"},
		{"uag", "uag"},
		{"uane", "uan"},
		{"uanl", "uanl"},
		{"uap", "uap"},
		{"uat", "uat"},
		{"ubica", "ubic"},
		{"ubicaba", "ubic"},
		{"ubicaban", "ubic"},
		{"ubicación", "ubic"},
		{"ubicada", "ubic"},
		{"ubicadas", "ubic"},
		{"ubicado", "ubic"},
		{"ubicados", "ubic"},
		{"ubican", "ubic"},
		{"ubicando", "ubic"},
		{"ubicándolo", "ubic"},
		{"ubicar", "ubic"},
		{"ubicara", "ubic"},
		{"ubicará", "ubic"},
		{"ubicarían", "ubic"},
		{"ubicarlos", "ubic"},
		{"ubicarnos", "ubic"},
		{"ubicaron", "ubic"},
		{"ubicarse", "ubic"},
		{"ubicársele", "ubicarsel"},
		{"ubicó", "ubic"},
		{"ucalli", "ucalli"},
		{"ucayali", "ucayali"},
		{"ucimme", "ucimm"},
		{"ucrania", "ucrani"},
		{"ucraniano", "ucranian"},
		{"udeg", "udeg"},
		{"udem", "udem"},
		{"udinese", "udines"},
		{"ugalde", "ugald"},
		{"ugarte", "ugart"},
		{"uglilia", "uglili"},
		{"ugocp", "ugocp"},
		{"uhagon", "uhagon"},
		{"uhf", "uhf"},
		{"úlcera", "ulcer"},
		{"ulises", "ulis"},
		{"ullami", "ullami"},
		{"ulloa", "ullo"},
		{"última", "ultim"},
		{"últimamente", "ultim"},
		{"últimas", "ultim"},
		{"ultimo", "ultim"},
		{"último", "ultim"},
		{"ultimos", "ultim"},
		{"últimos", "ultim"},
		{"ultra", "ultra"},
		{"ultrajó", "ultraj"},
		{"ultranza", "ultranz"},
		{"ultravioleta", "ultraviolet"},
		{"umbral", "umbral"},
		{"un", "un"},
		{"una", "una"},
		{"unacolumna", "unacolumn"},
		{"unam", "unam"},
		{"unamitas", "unamit"},
		{"unan", "unan"},
		{"unánime", "unanim"},
		{"unánimemente", "unanim"},
		{"unanimidad", "unanim"},
		{"unas", "unas"},
		{"uñas", "uñas"},
		{"undiano", "undian"},
		{"une", "une"},
		{"unesco", "unesc"},
		{"unforgettable", "unforgett"},
		{"unía", "uni"},
		{"única", "unic"},
		{"unicamente", "unic"},
		{"únicamente", "unic"},
		{"únicas", "unic"},
		{"unico", "unic"},
		{"único", "unic"},
		{"únicos", "unic"},
		{"unidad", "unid"},
		{"unidades", "unidad"},
		{"unidas", "unid"},
		{"unidimensional", "unidimensional"},
		{"unido", "unid"},
		{"unidos", "unid"},
		{"unieron", "unieron"},
		{"unifamiliares", "unifamiliar"},
		{"unificación", "unif"},
		{"unificadas", "unific"},
		{"unificado", "unific"},
		{"unificar", "unific"},
		{"uniformada", "uniform"},
		{"uniformado", "uniform"},
		{"uniformados", "uniform"},
		{"uniformar", "uniform"},
		{"uniforme", "uniform"},
		{"uniformes", "uniform"},
		{"unilateral", "unilateral"},
		{"unilaterales", "unilateral"},
		{"unilateralmente", "unilateral"},
		{"unimóvil", "unimovil"},
		{"unió", "uni"},
		{"unión", "union"},
		{"uniones", "union"},
		{"unir", "unir"},
		{"unirá", "unir"},
		{"uniría", "unir"},
		{"unirse", "unirs"},
		{"unisex", "unisex"},
		{"unitardos", "unitard"},
		{"unitario", "unitari"},
		{"united", "unit"},
		{"univ", "univ"},
		{"universal", "universal"},
		{"universales", "universal"},
		{"universalmente", "universal"},
		{"universidad", "univers"},
		{"universidades", "univers"},
		{"universitaria", "universitari"},
		{"universitarias", "universitari"},
		{"universitario", "universitari"},
		{"universitarios", "universitari"},
		{"university", "university"},
		{"universo", "univers"},
		{"unix", "unix"},
		{"uno", "uno"},
		{"unos", "unos"},
		{"unplugged", "unplugg"},
		{"unzueta", "unzuet"},
		{"up", "up"},
		{"upi", "upi"},
		{"ur", "ur"},
		{"urbana", "urban"},
		{"urbanas", "urban"},
		{"urbanistas", "urban"},
		{"urbanístico", "urbanist"},
		{"urbanísticos", "urbanist"},
		{"urbano", "urban"},
		{"urbanos", "urban"},
		{"urbieta", "urbiet"},
		{"urdió", "urdi"},
		{"urge", "urge"},
		{"urgen", "urgen"},
		{"urgencia", "urgenci"},
		{"urgencias", "urgenci"},
		{"urgente", "urgent"},
		{"urgentes", "urgent"},
		{"urgida", "urgid"},
		{"uribe", "urib"},
		{"urista", "urist"},
		{"urna", "urna"},
		{"urnas", "urnas"},
		{"uro", "uro"},
		{"urrea", "urre"},
		{"urss", "urss"},
		{"ursula", "ursul"},
		{"urtiz", "urtiz"},
		{"uruapan", "uruap"},
		{"uruguay", "uruguay"},
		{"uruguayo", "uruguay"},
		{"urzúa", "urzu"},
		{"usa", "usa"},
		{"usaba", "usab"},
		{"usaban", "usab"},
		{"usada", "usad"},
		{"usadas", "usad"},
		{"usado", "usad"},
		{"usados", "usad"},
		{"usan", "usan"},
		{"usando", "usand"},
		{"usar", "usar"},
		{"usara", "usar"},
		{"usará", "usar"},
		{"usarán", "usaran"},
		{"usarla", "usarl"},
		{"usarlos", "usarl"},
		{"usaron", "usaron"},
		{"usarse", "usars"},
		{"use", "use"},
		{"usen", "usen"},
		{"usis", "usis"},
		{"uso", "uso"},
		{"usó", "uso"},
		{"usos", "usos"},
		{"usted", "usted"},
		{"ustedes", "usted"},
		{"usuales", "usual"},
		{"usuario", "usuari"},
		{"usuarios", "usuari"},
		{"usura", "usur"},
		{"usureros", "usurer"},
		{"usurpación", "usurp"},
		{"útero", "uter"},
		{"útil", "util"},
		{"utilería", "util"},
		{"utilerías", "util"},
		{"utilice", "utilic"},
		{"utilidad", "util"},
		{"utilidades", "util"},
		{"utiliza", "utiliz"},
		{"utilizaba", "utiliz"},
		{"utilizaban", "utiliz"},
		{"utilización", "utiliz"},
		{"utilizada", "utiliz"},
		{"utilizadas", "utiliz"},
		{"utilizado", "utiliz"},
		{"utilizados", "utiliz"},
		{"utilizamos", "utiliz"},
		{"utilizan", "utiliz"},
		{"utilizando", "utiliz"},
		{"utilizar", "utiliz"},
		{"utilizará", "utiliz"},
		{"utilizarán", "utiliz"},
		{"utilizarla", "utiliz"},
		{"utilizarlo", "utiliz"},
		{"utilizaron", "utiliz"},
		{"utilizarse", "utiliz"},
		{"utilizó", "utiliz"},
		{"utopía", "utop"},
		{"utopías", "utop"},
		{"utópicas", "utop"},
		{"uy", "uy"},
		{"v", "v"},
		{"va", "va"},
		{"vaca", "vac"},
		{"vacacionar", "vacacion"},
		{"vacaciones", "vacacion"},
		{"vacas", "vac"},
		{"vacía", "vac"},
		{"vaciado", "vaci"},
		{"vacías", "vac"},
		{"vacila", "vacil"},
		{"vacilada", "vacil"},
		{"vacilan", "vacil"},
		{"vacilando", "vacil"},
		{"vacilante", "vacil"},
		{"vacilar", "vacil"},
		{"vaciló", "vacil"},
		{"vacilón", "vacilon"},
		{"vacío", "vaci"},
		{"vació", "vac"},
		{"vacíos", "vaci"},
		{"vacuna", "vacun"},
		{"vacunación", "vacun"},
		{"vacunas", "vacun"},
		{"vado", "vad"},
		{"vagabundo", "vagabund"},
		{"vagina", "vagin"},
		{"vagones", "vagon"},
		{"vagoneta", "vagonet"},
		{"vagos", "vag"},
		{"vaivenes", "vaiven"},
		{"vajilla", "vajill"},
		{"val", "val"},
		{"valde", "vald"},
		{"valdés", "valdes"},
		{"valdez", "valdez"},
		{"valdo", "vald"},
		{"valdrá", "valdr"},
		{"valdría", "valdr"},
		{"vale", "val"},
		{"valedores", "valedor"},
		{"valegorrista", "valegorr"},
		{"valemos", "val"},
		{"valen", "val"},
		{"valencia", "valenci"},
		{"valens", "valens"},
		{"valente", "valent"},
		{"valentia", "valenti"},
		{"valentía", "valent"},
		{"valentín", "valentin"},
		{"valentino", "valentin"},
		{"valenzuela", "valenzuel"},
		{"valer", "val"},
		{"valeri", "valeri"},
		{"valero", "valer"},
		{"valet", "valet"},
		{"valga", "valg"},
		{"valgan", "valg"},
		{"valía", "val"},
		{"validez", "validez"},
		{"valido", "val"},
		{"válido", "val"},
		{"valiente", "valient"},
		{"valieron", "val"},
		{"valió", "val"},
		{"valiosa", "valios"},
		{"valiosas", "vali"},
		{"valioso", "valios"},
		{"valiosos", "valios"},
		{"valium", "valium"},
		{"valla", "vall"},
		{"valladares", "valladar"},
		{"vallarta", "vallart"},
		{"vallas", "vall"},
		{"valle", "vall"},
		{"vallecano", "vallecan"},
		{"vallecillo", "vallecill"},
		{"vallecillos", "vallecill"},
		{"vallejo", "vallej"},
		{"valles", "vall"},
		{"valley", "valley"},
		{"valor", "valor"},
		{"valora", "valor"},
		{"valoraba", "valor"},
		{"valoración", "valor"},
		{"valoradas", "valor"},
		{"valorado", "valor"},
		{"valorados", "valor"},
		{"valorar", "valor"},
		{"valorarlo", "valor"},
		{"valoraron", "valor"},
		{"valore", "valor"},
		{"valores", "valor"},
		{"valorizada", "valoriz"},
		{"valparaíso", "valparais"},
		{"valtierra", "valtierr"},
		{"valuación", "valuacion"},
		{"valuadas", "valu"},
		{"value", "valu"},
		{"válvulas", "valvul"},
		{"vamos", "vam"},
		{"vampiro", "vampir"},
		{"vampiros", "vampir"},
		{"van", "van"},
		{"vandálicos", "vandal"},
		{"vandalismo", "vandal"},
		{"vanderley", "vanderley"},
		{"vanette", "vanett"},
		{"vanguardia", "vanguardi"},
		{"vanguardista", "vanguard"},
		{"vanguardistas", "vanguard"},
		{"vanidad", "vanid"},
		{"vanidades", "vanidad"},
		{"vanilli", "vanilli"},
		{"vánitas", "vanit"},
		{"vanity", "vanity"},
		{"vano", "van"},
		{"vanos", "van"},
		{"vapor", "vapor"},
		{"vaporizaciones", "vaporiz"},
		{"vaporizarlas", "vaporiz"},
		{"vaporosa", "vapor"},
		{"vaporosos", "vapor"},
		{"vaquera", "vaquer"},
		{"vaquerías", "vaqu"},
		{"vaquero", "vaquer"},
		{"vaqueros", "vaquer"},
		{"varada", "var"},
		{"varados", "var"},
		{"varela", "varel"},
		{"vargas", "varg"},
		{"varía", "var"},
		{"variable", "variabl"},
		{"variables", "variabl"},
		{"variación", "variacion"},
		{"variaciones", "variacion"},
		{"variada", "vari"},
		{"variado", "vari"},
		{"variados", "vari"},
		{"varían", "var"},
		{"variando", "vari"},
		{"variantes", "variant"},
		{"varias", "vari"},
		{"varicela", "varicel"},
		{"variedad", "varied"},
		{"variedades", "variedad"},
		{"variopinta", "variopint"},
		{"varios", "vari"},
		{"varita", "varit"},
		{"varón", "varon"},
		{"varones", "varon"},
		{"varonil", "varonil"},
		{"varoniles", "varonil"},
		{"vas", "vas"},
		{"vasco", "vasc"},
		{"vasconcelos", "vasconcel"},
		{"vaso", "vas"},
		{"vasos", "vas"},
		{"vásquez", "vasquez"},
		{"vasto", "vast"},
		{"vaticano", "vatican"},
		{"vaticinado", "vaticin"},
		{"vaticinar", "vaticin"},
		{"vaticinaron", "vaticin"},
		{"vaticinen", "vaticin"},
		{"vaticinios", "vaticini"},
		{"vaticinó", "vaticin"},
		{"vax", "vax"},
		{"vaxstations", "vaxstations"},
		{"vaya", "vay"},
		{"vayamos", "vay"},
		{"vayan", "vay"},
		{"vazquez", "vazquez"},
		{"vázquez", "vazquez"},
		{"vcr", "vcr"},
		{"ve", "ve"},
		{"vea", "vea"},
		{"veamos", "veam"},
		{"vean", "vean"},
		{"veces", "vec"},
		{"vecina", "vecin"},
		{"vecinal", "vecinal"},
		{"vecinales", "vecinal"},
		{"vecinas", "vecin"},
		{"vecindad", "vecind"},
		{"vecindario", "vecindari"},
		{"vecino", "vecin"},
		{"vecinos", "vecin"},
		{"vector", "vector"},
		{"vega", "veg"},
		{"vegetación", "veget"},
		{"vegetales", "vegetal"},
		{"vegetarianismo", "vegetarian"},
		{"vegetarianos", "vegetarian"},
		{"vegetativo", "veget"},
		{"vehemencia", "vehement"},
		{"vehemente", "vehement"},
		{"vehicular", "vehicul"},
		{"vehiculares", "vehicular"},
		{"vehículo", "vehicul"},
		{"vehículos", "vehicul"},
		{"veía", "vei"},
		{"veían", "vei"},
		{"veinte", "veint"},
		{"veinteañeros", "veinteañer"},
		{"veintena", "veinten"},
		{"veinticinco", "veinticinc"},
		{"veintiseis", "veintiseis"},
		{"vejarrón", "vejarron"},
		{"vejez", "vejez"},
		{"vela", "vel"},
		{"velada", "vel"},
		{"velador", "velador"},
		{"velando", "vel"},
		{"velarde", "velard"},
		{"velas", "vel"},
		{"velasco", "velasc"},
		{"velázquez", "velazquez"},
		{"velero", "veler"},
		{"vélez", "velez"},
		{"vellones", "vellon"},
		{"velocidad", "veloc"},
		{"velocidades", "veloc"},
		{"velocímetro", "velocimetr"},
		{"velos", "vel"},
		{"veloso", "velos"},
		{"vemos", "vem"},
		{"ven", "ven"},
		{"vena", "ven"},
		{"venado", "ven"},
		{"vence", "venc"},
		{"vencedor", "vencedor"},
		{"vencedores", "vencedor"},
		{"vencer", "venc"},
		{"vencerás", "venc"},
		{"vencerse", "venc"},
		{"vencida", "venc"},
		{"vencidas", "venc"},
		{"vencido", "venc"},
		{"vencidos", "venc"},
		{"vencieron", "venc"},
		{"vencimiento", "vencimient"},
		{"vencimientos", "vencimient"},
		{"venció", "venc"},
		{"venda", "vend"},
		{"vendados", "vend"},
		{"vendan", "vend"},
		{"vendaron", "vend"},
		{"vende", "vend"},
		{"vendedor", "vendedor"},
		{"vendedora", "vendedor"},
		{"vendedores", "vendedor"},
		{"vendemos", "vend"},
		{"venden", "vend"},
		{"vender", "vend"},
		{"venderá", "vend"},
		{"venderán", "vend"},
		{"venderemos", "vend"},
		{"vendería", "vend"},
		{"venderla", "vend"},
		{"venderlo", "vend"},
		{"venderse", "vend"},
		{"vendía", "vend"},
		{"vendida", "vend"},
		{"vendidas", "vend"},
		{"vendido", "vend"},
		{"vendidos", "vend"},
		{"vendiendo", "vend"},
		{"vendiéndose", "vend"},
		{"vendiera", "vend"},
		{"vendieron", "vend"},
		{"vendió", "vend"},
		{"vendo", "vend"},
		{"vendrá", "vendr"},
		{"vendrán", "vendran"},
		{"venegas", "veneg"},
		{"veneno", "venen"},
		{"veneraban", "vener"},
		{"veneración", "vener"},
		{"venezolana", "venezolan"},
		{"venezolano", "venezolan"},
		{"venezolanos", "venezolan"},
		{"venezuela", "venezuel"},
		{"venga", "veng"},
		{"vengadores", "vengador"},
		{"vengan", "veng"},
		{"venganza", "venganz"},
		{"vengo", "veng"},
		{"venía", "ven"},
		{"venían", "ven"},
		{"venicio", "venici"},
		{"venida", "ven"},
		{"venido", "ven"},
		{"venidos", "ven"},
		{"venimos", "ven"},
		{"venir", "ven"},
		{"venirse", "ven"},
		{"venta", "vent"},
		{"ventaja", "ventaj"},
		{"ventajas", "ventaj"},
		{"ventajosa", "ventaj"},
		{"ventajosas", "ventaj"},
		{"ventana", "ventan"},
		{"ventanas", "ventan"},
		{"ventanilla", "ventanill"},
		{"ventas", "vent"},
		{"ventila", "ventil"},
		{"ventilación", "ventil"},
		{"ventilar", "ventil"},
		{"ventilaron", "ventil"},
		{"ventilas", "ventil"},
		{"ventre", "ventr"},
		{"ventura", "ventur"},
		{"venus", "venus"},
		{"venustiano", "venustian"},
		{"veo", "veo"},
		{"ver", "ver"},
		{"vera", "ver"},
		{"verá", "ver"},
		{"veracruz", "veracruz"},
		{"veracruzana", "veracruzan"},
		{"veracruzano", "veracruzan"},
		{"veracruzanos", "veracruzan"},
		{"verán", "veran"},
		{"verano", "veran"},
		{"veras", "ver"},
		{"verbal", "verbal"},
		{"verbales", "verbal"},
		{"verbalista", "verbal"},
		{"verbalmente", "verbal"},
		{"verbo", "verb"},
		{"verdad", "verd"},
		{"verdadera", "verdader"},
		{"verdaderamente", "verdader"},
		{"verdaderas", "verdader"},
		{"verdadero", "verdader"},
		{"verdaderos", "verdader"},
		{"verdades", "verdad"},
		{"verde", "verd"},
		{"verdes", "verd"},
		{"verdirame", "verdiram"},
		{"verdugo", "verdug"},
		{"verduras", "verdur"},
		{"verduzco", "verduzc"},
		{"veré", "ver"},
		{"veredalta", "veredalt"},
		{"veredas", "vered"},
		{"veredicto", "veredict"},
		{"veremos", "ver"},
		{"vergonzante", "vergonz"},
		{"vergonzosos", "vergonz"},
		{"vergüenza", "vergüenz"},
		{"vería", "ver"},
		{"veríamos", "ver"},
		{"verificaba", "verific"},
		{"verificación", "verif"},
		{"verifican", "verific"},
		{"verificar", "verific"},
		{"verificaron", "verific"},
		{"verificó", "verific"},
		{"verla", "verl"},
		{"verlas", "verl"},
		{"verlo", "verl"},
		{"verlos", "verl"},
		{"verme", "verm"},
		{"verónica", "veron"},
		{"verónicas", "veron"},
		{"verrugas", "verrug"},
		{"versa", "vers"},
		{"versace", "versac"},
		{"versátil", "versatil"},
		{"versatilidad", "versatil"},
		{"verse", "vers"},
		{"versificación", "versif"},
		{"versión", "version"},
		{"versiones", "version"},
		{"versos", "vers"},
		{"versus", "versus"},
		{"verte", "vert"},
		{"vertebral", "vertebral"},
		{"vertical", "vertical"},
		{"vertida", "vert"},
		{"vertidas", "vert"},
		{"vertido", "vert"},
		{"vertientes", "vertient"},
		{"very", "very"},
		{"ves", "ves"},
		{"vespertino", "vespertin"},
		{"vestía", "vest"},
		{"vestida", "vest"},
		{"vestido", "vest"},
		{"vestidos", "vest"},
		{"vestidura", "vestidur"},
		{"vestiduras", "vestidur"},
		{"vestigio", "vestigi"},
		{"vestimos", "vest"},
		{"vestir", "vest"},
		{"vestirse", "vest"},
		{"vestuario", "vestuari"},
		{"vetado", "vet"},
		{"vetará", "vet"},
		{"veterano", "veteran"},
		{"veteranos", "veteran"},
		{"veterinarias", "veterinari"},
		{"veto", "vet"},
		{"vetó", "vet"},
		{"vez", "vez"},
		{"vi", "vi"},
		{"vía", "via"},
		{"viabilidad", "viabil"},
		{"viable", "viabl"},
		{"viaducto", "viaduct"},
		{"viaja", "viaj"},
		{"viajaba", "viaj"},
		{"viajaban", "viaj"},
		{"viajado", "viaj"},
		{"viajan", "viaj"},
		{"viajar", "viaj"},
		{"viajará", "viaj"},
		{"viajarán", "viaj"},
		{"viajarían", "viaj"},
		{"viajaron", "viaj"},
		{"viaje", "viaj"},
		{"viajen", "viaj"},
		{"viajero", "viajer"},
		{"viajeros", "viajer"},
		{"viajes", "viaj"},
		{"viajó", "viaj"},
		{"vial", "vial"},
		{"viales", "vial"},
		{"vialidad", "vialid"},
		{"vialidades", "vialidad"},
		{"viandas", "viand"},
		{"vianey", "vianey"},
		{"vías", "vias"},
		{"víbora", "vibor"},
		{"vibra", "vibr"},
		{"vibraciones", "vibracion"},
		{"vibradores", "vibrador"},
		{"vibrar", "vibr"},
		{"vicario", "vicari"},
		{"vice", "vic"},
		{"vicecanciller", "vicecancill"},
		{"vicente", "vicent"},
		{"vicepresidencia", "vicepresident"},
		{"vicepresidente", "vicepresident"},
		{"viceversa", "vicevers"},
		{"vicio", "vici"},
		{"vicios", "vici"},
		{"viciosas", "vici"},
		{"vícitma", "vicitm"},
		{"vicky", "vicky"},
		{"víctima", "victim"},
		{"victimado", "victim"},
		{"víctimas", "victim"},
		{"víctor", "victor"},
		{"victoria", "victori"},
		{"victoriana", "victorian"},
		{"victoriano", "victorian"},
		{"victorias", "victori"},
		{"victorioso", "victori"},
		{"victoriosos", "victori"},
		{"vida", "vid"},
		{"vidal", "vidal"},
		{"vidales", "vidal"},
		{"vidas", "vid"},
		{"vidente", "vident"},
		{"videntes", "vident"},
		{"video", "vide"},
		{"videocasetera", "videocaseter"},
		{"videocasetes", "videocaset"},
		{"videocentros", "videocentr"},
		{"videoclip", "videoclip"},
		{"videoclubes", "videoclub"},
		{"videohomes", "videohom"},
		{"videomax", "videomax"},
		{"videopelícula", "videopelicul"},
		{"videos", "vide"},
		{"vidrio", "vidri"},
		{"vidrios", "vidri"},
		{"vieira", "vieir"},
		{"vieja", "viej"},
		{"viejas", "viej"},
		{"viejecita", "viejecit"},
		{"viejita", "viejit"},
		{"viejito", "viejit"},
		{"viejitos", "viejit"},
		{"viejo", "viej"},
		{"viejón", "viejon"},
		{"viejos", "viej"},
		{"viena", "vien"},
		{"viendo", "viend"},
		{"viendose", "viendos"},
		{"viene", "vien"},
		{"vienen", "vien"},
		{"viento", "vient"},
		{"vientos", "vient"},
		{"vientre", "vientr"},
		{"vientres", "vientr"},
		{"viera", "vier"},
		{"viernes", "viern"},
		{"vieron", "vieron"},
		{"viesca", "viesc"},
		{"vietnam", "vietnam"},
		{"view", "view"},
		{"vigas", "vig"},
		{"vigencia", "vigenci"},
		{"vigente", "vigent"},
		{"vigentes", "vigent"},
		{"vigésima", "vigesim"},
		{"vigésimacuarta", "vigesimacuart"},
		{"vigésimo", "vigesim"},
		{"vigil", "vigil"},
		{"vigila", "vigil"},
		{"vigilaban", "vigil"},
		{"vigilada", "vigil"},
		{"vigiladas", "vigil"},
		{"vigilado", "vigil"},
		{"vigilados", "vigil"},
		{"vigilancia", "vigil"},
		{"vigilante", "vigil"},
		{"vigilantes", "vigil"},
		{"vigilar", "vigil"},
		{"vigilará", "vigil"},
		{"vigilarlas", "vigil"},
		{"vigile", "vigil"},
		{"vigilen", "vigil"},
		{"vigilia", "vigili"},
		{"vigo", "vig"},
		{"vigor", "vigor"},
		{"vigoroso", "vigor"},
		{"viii", "viii"},
		{"villa", "vill"},
		{"villacaña", "villacañ"},
		{"villada", "vill"},
		{"villagómez", "villagomez"},
		{"villagrán", "villagran"},
		{"villahermosa", "villaherm"},
		{"villalba", "villalb"},
		{"villaldama", "villaldam"},
		{"villalobos", "villalob"},
		{"villamar", "villam"},
		{"villán", "villan"},
		{"villanueva", "villanuev"},
		{"villar", "vill"},
		{"villarreal", "villarreal"},
		{"villasana", "villasan"},
		{"villaseñor", "villaseñor"},
		{"villauto", "villaut"},
		{"villegas", "villeg"},
		{"villuendas", "villuend"},
		{"vimos", "vim"},
		{"vimosa", "vimos"},
		{"viña", "viñ"},
		{"vincent", "vincent"},
		{"vincula", "vincul"},
		{"vinculación", "vincul"},
		{"vinculadas", "vincul"},
		{"vinculado", "vincul"},
		{"vinculados", "vincul"},
		{"vincular", "vincul"},
		{"vincularse", "vincul"},
		{"vinculo", "vincul"},
		{"vínculo", "vincul"},
		{"vínculos", "vincul"},
		{"vine", "vin"},
		{"viniendo", "vin"},
		{"viniera", "vin"},
		{"vinieran", "vin"},
		{"vinieron", "vin"},
		{"vino", "vin"},
		{"vinyard", "vinyard"},
		{"vio", "vio"},
		{"violación", "violacion"},
		{"violaciones", "violacion"},
		{"violada", "viol"},
		{"violado", "viol"},
		{"violador", "violador"},
		{"violadores", "violador"},
		{"violan", "viol"},
		{"violando", "viol"},
		{"violar", "viol"},
		{"violatorias", "violatori"},
		{"violencia", "violenci"},
		{"violenta", "violent"},
		{"violentada", "violent"},
		{"violentamente", "violent"},
		{"violentaría", "violent"},
		{"violentas", "violent"},
		{"violento", "violent"},
		{"violentos", "violent"},
		{"violeta", "violet"},
		{"violetina", "violetin"},
		{"violín", "violin"},
		{"violinista", "violin"},
		{"violó", "viol"},
		{"vips", "vips"},
		{"virgen", "virg"},
		{"vírgenes", "virgen"},
		{"virginia", "virgini"},
		{"virilidad", "viril"},
		{"virreinato", "virreinat"},
		{"virtual", "virtual"},
		{"virtud", "virtud"},
		{"virtudes", "virtud"},
		{"virtuosa", "virtuos"},
		{"virtuosos", "virtuos"},
		{"virulentos", "virulent"},
		{"visa", "vis"},
		{"visag", "visag"},
		{"viscatán", "viscatan"},
		{"viscencio", "viscenci"},
		{"viscerales", "visceral"},
		{"visibilidad", "visibil"},
		{"visible", "visibl"},
		{"visiblemente", "visibl"},
		{"visión", "vision"},
		{"visionarios", "visionari"},
		{"visiones", "vision"},
		{"visires", "visir"},
		{"visita", "visit"},
		{"visitaba", "visit"},
		{"visitadas", "visit"},
		{"visitado", "visit"},
		{"visitaduría", "visitadur"},
		{"visitan", "visit"},
		{"visitando", "visit"},
		{"visitante", "visit"},
		{"visitantes", "visit"},
		{"visitar", "visit"},
		{"visitara", "visit"},
		{"visitará", "visit"},
		{"visitarán", "visit"},
		{"visitaría", "visit"},
		{"visitarla", "visit"},
		{"visitaron", "visit"},
		{"visitas", "visit"},
		{"visite", "visit"},
		{"visiten", "visit"},
		{"visiteo", "visite"},
		{"visitiendo", "visit"},
		{"visitó", "visit"},
		{"viskin", "viskin"},
		{"vislumbra", "vislumbr"},
		{"vislumbran", "vislumbr"},
		{"vislumbrar", "vislumbr"},
		{"vislumbraron", "vislumbr"},
		{"vislumbró", "vislumbr"},
		{"visora", "visor"},
		{"visos", "vis"},
		{"víspera", "visper"},
		{"vísperas", "visper"},
		{"vista", "vist"},
		{"vistas", "vist"},
		{"vistazo", "vistaz"},
		{"visten", "vist"},
		{"vistiendo", "vist"},
		{"vistió", "vist"},
		{"visto", "vist"},
		{"vistos", "vist"},
		{"vistoso", "vistos"},
		{"vistosos", "vistos"},
		{"visual", "visual"},
		{"visuales", "visual"},
		{"visualice", "visualic"},
		{"visualizaban", "visualiz"},
		{"visualización", "visualiz"},
		{"visualizado", "visualiz"},
		{"visualizar", "visualiz"},
		{"visualmente", "visual"},
		{"vital", "vital"},
		{"vitales", "vital"},
		{"vitalicio", "vitalici"},
		{"vitalidad", "vital"},
		{"vitamina", "vitamin"},
		{"vitaminas", "vitamin"},
		{"vite", "vit"},
		{"vito", "vit"},
		{"vítores", "vitor"},
		{"vitro", "vitr"},
		{"vitroles", "vitrol"},
		{"vittorio", "vittori"},
		{"vituperado", "vituper"},
		{"vituperados", "vituper"},
		{"viuda", "viud"},
		{"viudas", "viud"},
		{"viudo", "viud"},
		{"viva", "viv"},
		{"vivales", "vival"},
		{"vivan", "viv"},
		{"vivas", "viv"},
		{"vive", "viv"},
		{"viven", "viv"},
		{"víveres", "viver"},
		{"vivero", "viver"},
		{"viví", "viv"},
		{"vivía", "viv"},
		{"vivíamos", "viv"},
		{"vivían", "viv"},
		{"vivible", "vivibl"},
		{"vivido", "viv"},
		{"vividos", "viv"},
		{"vivienda", "viviend"},
		{"viviendas", "viviend"},
		{"viviendo", "viv"},
		{"viviente", "vivient"},
		{"vivientes", "vivient"},
		{"viviera", "viv"},
		{"vivieran", "viv"},
		{"vivieron", "viv"},
		{"vivillos", "vivill"},
		{"vivimos", "viv"},
		{"vivió", "viv"},
		{"vivir", "viv"},
		{"vivirá", "viv"},
		{"vivirse", "viv"},
		{"vivo", "viv"},
		{"vivos", "viv"},
		{"vl", "vl"},
		{"vladimir", "vladim"},
		{"vocabulario", "vocabulari"},
		{"vocación", "vocacion"},
		{"vocacional", "vocacional"},
		{"vocaciones", "vocacion"},
		{"vocal", "vocal"},
		{"vocalía", "vocal"},
		{"vocalista", "vocal"},
		{"vocera", "vocer"},
		{"vocero", "vocer"},
		{"voceros", "vocer"},
		{"voces", "voc"},
		{"voice", "voic"},
		{"voices", "voic"},
		{"voladeros", "volader"},
		{"volando", "vol"},
		{"volante", "volant"},
		{"volantes", "volant"},
		{"volar", "vol"},
		{"volará", "vol"},
		{"volaría", "vol"},
		{"volátil", "volatil"},
		{"volátiles", "volatil"},
		{"volatilidad", "volatil"},
		{"volcadura", "volcadur"},
		{"volcamos", "volc"},
		{"volcán", "volcan"},
		{"volcanes", "volcan"},
		{"volcánica", "volcan"},
		{"volcánico", "volcan"},
		{"volcar", "volc"},
		{"volcaron", "volc"},
		{"volcó", "volc"},
		{"voleibol", "voleibol"},
		{"volga", "volg"},
		{"volkswagen", "volkswag"},
		{"volmer", "volm"},
		{"voltaire", "voltair"},
		{"voltea", "volte"},
		{"voltear", "volt"},
		{"volteó", "volte"},
		{"volteretas", "volteret"},
		{"voltios", "volti"},
		{"volts", "volts"},
		{"voluble", "volubl"},
		{"volumen", "volum"},
		{"volumenes", "volumen"},
		{"volúmenes", "volumen"},
		{"voluminosísima", "voluminosisim"},
		{"voluntad", "volunt"},
		{"voluntades", "voluntad"},
		{"voluntaria", "voluntari"},
		{"voluntariado", "voluntari"},
		{"voluntariamente", "voluntari"},
		{"voluntarias", "voluntari"},
		{"voluntario", "voluntari"},
		{"voluntarios", "voluntari"},
		{"voluntarioso", "voluntari"},
		{"volvamos", "volv"},
		{"volver", "volv"},
		{"volverá", "volv"},
		{"volverán", "volv"},
		{"volveré", "volv"},
		{"volvería", "volv"},
		{"volverlas", "volv"},
		{"volverse", "volv"},
		{"volvía", "volv"},
		{"volviendo", "volv"},
		{"volvieron", "volv"},
		{"volvimos", "volv"},
		{"volvió", "volv"},
		{"volvo", "volv"},
		{"volvoramírez", "volvoramirez"},
		{"vomitar", "vomit"},
		{"vómitos", "vomit"},
		{"von", "von"},
		{"vorazmente", "voraz"},
		{"vosso", "voss"},
		{"vota", "vot"},
		{"votación", "votacion"},
		{"votaciones", "votacion"},
		{"votantes", "votant"},
		{"votar", "vot"},
		{"votara", "vot"},
		{"votará", "vot"},
		{"votarán", "vot"},
		{"votaría", "vot"},
		{"votaron", "vot"},
		{"votarse", "vot"},
		{"vote", "vot"},
		{"voten", "vot"},
		{"voto", "vot"},
		{"votó", "vot"},
		{"votos", "vot"},
		{"voy", "voy"},
		{"voz", "voz"},
		{"vs", "vs"},
		{"vudú", "vudu"},
		{"vuelas", "vuel"},
		{"vuelca", "vuelc"},
		{"vuele", "vuel"},
		{"vuelo", "vuel"},
		{"vuelos", "vuel"},
		{"vuelta", "vuelt"},
		{"vueltas", "vuelt"},
		{"vuelto", "vuelt"},
		{"vuelva", "vuelv"},
		{"vuelvan", "vuelv"},
		{"vuelve", "vuelv"},
		{"vuelven", "vuelv"},
		{"vuelvo", "vuelv"},
		{"vuestro", "vuestr"},
		{"vulcanología", "vulcanolog"},
		{"vulcanólogos", "vulcanolog"},
		{"vulgar", "vulg"},
		{"vulgares", "vulgar"},
		{"vulgaridad", "vulgar"},
		{"vulgo", "vulg"},
		{"vulnerabilidad", "vulner"},
		{"vulnerable", "vulner"},
		{"vulnerables", "vulner"},
		{"vulneran", "vulner"},
		{"vw", "vw"},
		{"w", "w"},
		{"wa", "wa"},
		{"wal", "wal"},
		{"walberto", "walbert"},
		{"walcott", "walcott"},
		{"wall", "wall"},
		{"wallace", "wallac"},
		{"walt", "walt"},
		{"walter", "walt"},
		{"walterio", "walteri"},
		{"walton", "walton"},
		{"ward", "ward"},
		{"warner", "warn"},
		{"warrants", "warrants"},
		{"warren", "warr"},
		{"washington", "washington"},
		{"waterer", "water"},
		{"we", "we"},
		{"weisackaer", "weisack"},
		{"weiss", "weiss"},
		{"welch", "welch"},
		{"wells", "wells"},
		{"welsh", "welsh"},
		{"wenceslao", "wencesla"},
		{"wendy", "wendy"},
		{"west", "west"},
		{"western", "western"},
		{"westh", "westh"},
		{"westinghouse", "westinghous"},
		{"wetherell", "wetherell"},
		{"weule", "weul"},
		{"whataburger", "whataburg"},
		{"whisky", "whisky"},
		{"widnall", "widnall"},
		{"wiesel", "wiesel"},
		{"wildlife", "wildlif"},
		{"wilford", "wilford"},
		{"william", "william"},
		{"williams", "williams"},
		{"williamson", "williamson"},
		{"willie", "willi"},
		{"wilson", "wilson"},
		{"windows", "windows"},
		{"winfield", "winfield"},
		{"winston", "winston"},
		{"winstone", "winston"},
		{"wisconsin", "wisconsin"},
		{"woldenberg", "woldenberg"},
		{"wong", "wong"},
		{"woody", "woody"},
		{"worcester", "worcest"},
		{"word", "word"},
		{"works", "works"},
		{"world", "world"},
		{"worldnet", "worldnet"},
		{"x", "x"},
		{"xavier", "xavi"},
		{"xenotransplantes", "xenotranspl"},
		{"xenotrasplante", "xenotraspl"},
		{"xenotrasplantes", "xenotraspl"},
		{"xicoténcatl", "xicotencatl"},
		{"xiii", "xiii"},
		{"xilitla", "xilitl"},
		{"xiv", "xiv"},
		{"xix", "xix"},
		{"xochimilco", "xochimilc"},
		{"xochitl", "xochitl"},
		{"xóchitl", "xochitl"},
		{"xv", "xv"},
		{"xvi", "xvi"},
		{"xvii", "xvii"},
		{"xviii", "xviii"},
		{"xx", "xx"},
		{"xxi", "xxi"},
		{"xxiii", "xxiii"},
		{"xxvi", "xxvi"},
		{"y", "y"},
		{"ya", "ya"},
		{"yacía", "yac"},
		{"yacimiento", "yacimient"},
		{"yacimientos", "yacimient"},
		{"yañez", "yañez"},
		{"yanni", "yanni"},
		{"yardas", "yard"},
		{"yarmuch", "yarmuch"},
		{"yaroslao", "yarosla"},
		{"yate", "yat"},
		{"ycosas", "ycos"},
		{"years", "years"},
		{"yeltsin", "yeltsin"},
		{"yendo", "yend"},
		{"yerbaniz", "yerbaniz"},
		{"yerena", "yeren"},
		{"yerno", "yern"},
		{"yeso", "yes"},
		{"yidish", "yidish"},
		{"ylizaliturri", "ylizaliturri"},
		{"yo", "yo"},
		{"yodo", "yod"},
		{"yoga", "yog"},
		{"yogurt", "yogurt"},
		{"yolanda", "yoland"},
		{"yoltectl", "yoltectl"},
		{"yonke", "yonk"},
		{"yonquero", "yonquer"},
		{"york", "york"},
		{"yorky", "yorky"},
		{"youshimatz", "youshimatz"},
		{"yrapuato", "yrapuat"},
		{"yuc", "yuc"},
		{"yucatán", "yucatan"},
		{"yucateca", "yucatec"},
		{"yucateco", "yucatec"},
		{"yucatecos", "yucatec"},
		{"yugoslavia", "yugoslavi"},
		{"yugoslavo", "yugoslav"},
		{"yukio", "yuki"},
		{"yunez", "yunez"},
		{"yunta", "yunt"},
		{"yuri", "yuri"},
		{"yuxtapuesto", "yuxtapuest"},
		{"yvonne", "yvonn"},
		{"zabludowsky", "zabludowsky"},
		{"zacarías", "zac"},
		{"zacatecas", "zacatec"},
		{"zacatepec", "zacatepec"},
		{"zafra", "zafr"},
		{"zag", "zag"},
		{"zaga", "zag"},
		{"zagas", "zag"},
		{"zague", "zag"},
		{"zaguero", "zaguer"},
		{"zagueros", "zaguer"},
		{"zahnle", "zahnl"},
		{"zaid", "zaid"},
		{"zaire", "zair"},
		{"zalea", "zale"},
		{"zambrano", "zambran"},
		{"zamora", "zamor"},
		{"zamorano", "zamoran"},
		{"zamudio", "zamudi"},
		{"zanabria", "zanabri"},
		{"zancadilla", "zancadill"},
		{"zancudos", "zancud"},
		{"zanja", "zanj"},
		{"zanjas", "zanj"},
		{"zapalinamé", "zapalinam"},
		{"zapata", "zapat"},
		{"zapatera", "zapater"},
		{"zapatilla", "zapatill"},
		{"zapatitos", "zapatit"},
		{"zapatos", "zapat"},
		{"zapoteco", "zapotec"},
		{"zar", "zar"},
		{"zaragoza", "zaragoz"},
		{"zarape", "zarap"},
		{"zarapeados", "zarap"},
		{"zárate", "zarat"},
		{"zarazúa", "zarazu"},
		{"zarco", "zarc"},
		{"zarpó", "zarp"},
		{"ze", "ze"},
		{"zedillo", "zedill"},
		{"zelanda", "zeland"},
		{"zeltún", "zeltun"},
		{"zenith", "zenith"},
		{"zenón", "zenon"},
		{"zepa", "zep"},
		{"zepeda", "zeped"},
		{"zertuche", "zertuch"},
		{"zig", "zig"},
		{"zimmerman", "zimmerm"},
		{"zinser", "zins"},
		{"zínser", "zins"},
		{"zona", "zon"},
		{"zonas", "zon"},
		{"zoológico", "zoolog"},
		{"zoológicos", "zoolog"},
		{"zorkin", "zorkin"},
		{"zorrilla", "zorrill"},
		{"zorros", "zorr"},
		{"zotoluca", "zotoluc"},
		{"zotoluco", "zotoluc"},
		{"zotolucos", "zotoluc"},
		{"zuazua", "zuazu"},
		{"zubillaga", "zubillag"},
		{"zubizarreta", "zubizarret"},
		{"zulia", "zuli"},
		{"zúñiga", "zuñig"},
		{"zurcos", "zurc"},
		{"zurda", "zurd"},
		{"zurdo", "zurd"},
		{"zurdos", "zurd"},
		{"zurita", "zurit"},
		{"zutano", "zutan"},
	}
	for _, testCase := range testCases {
		result := spanish.Stem(testCase.in, true)
		if result != testCase.out {
			t.Errorf("Expected %v -> %v, but got %v", testCase.in, testCase.out, result)
		}
	}
}
