// Copyright 2024 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package pagespeedonline provides access to the PageSpeed Insights API.
//
// For product documentation, see: https://developers.google.com/speed/docs/insights/v5/about
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/pagespeedonline/v5"
//	...
//	ctx := context.Background()
//	pagespeedonlineService, err := pagespeedonline.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	pagespeedonlineService, err := pagespeedonline.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	pagespeedonlineService, err := pagespeedonline.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package pagespeedonline // import "google.golang.org/api/pagespeedonline/v5"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version

const apiId = "pagespeedonline:v5"
const apiName = "pagespeedonline"
const apiVersion = "v5"
const basePath = "https://pagespeedonline.googleapis.com/"
const basePathTemplate = "https://pagespeedonline.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://pagespeedonline.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// Associate you with your personal info on Google
	OpenIDScope = "openid"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"openid",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s, err := New(client)
	if err != nil {
		return nil, err
	}
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	s := &Service{client: client, BasePath: basePath}
	s.Pagespeedapi = NewPagespeedapiService(s)
	return s, nil
}

type Service struct {
	client    *http.Client
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Pagespeedapi *PagespeedapiService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewPagespeedapiService(s *Service) *PagespeedapiService {
	rs := &PagespeedapiService{s: s}
	return rs
}

type PagespeedapiService struct {
	s *Service
}

// AuditRefs: A light reference to an audit by id, used to group and weight
// audits in a given category.
type AuditRefs struct {
	// Acronym: The conventional acronym for the audit/metric.
	Acronym string `json:"acronym,omitempty"`
	// Group: The category group that the audit belongs to (optional).
	Group string `json:"group,omitempty"`
	// Id: The audit ref id.
	Id string `json:"id,omitempty"`
	// RelevantAudits: Any audit IDs closely relevant to this one.
	RelevantAudits []string `json:"relevantAudits,omitempty"`
	// Weight: The weight this audit's score has on the overall category score.
	Weight float64 `json:"weight,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Acronym") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Acronym") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *AuditRefs) MarshalJSON() ([]byte, error) {
	type NoMethod AuditRefs
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

func (s *AuditRefs) UnmarshalJSON(data []byte) error {
	type NoMethod AuditRefs
	var s1 struct {
		Weight gensupport.JSONFloat64 `json:"weight"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Weight = float64(s1.Weight)
	return nil
}

// Bucket: A proportion of data in the total distribution, bucketed by a
// min/max percentage. Each bucket's range is bounded by min <= x < max, In
// millisecond.
type Bucket struct {
	// Max: Upper bound for a bucket's range.
	Max int64 `json:"max,omitempty"`
	// Min: Lower bound for a bucket's range.
	Min int64 `json:"min,omitempty"`
	// Proportion: The proportion of data in this bucket.
	Proportion float64 `json:"proportion,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Max") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Max") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *Bucket) MarshalJSON() ([]byte, error) {
	type NoMethod Bucket
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

func (s *Bucket) UnmarshalJSON(data []byte) error {
	type NoMethod Bucket
	var s1 struct {
		Proportion gensupport.JSONFloat64 `json:"proportion"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Proportion = float64(s1.Proportion)
	return nil
}

// Categories: The categories in a Lighthouse run.
type Categories struct {
	// Accessibility: The accessibility category, containing all accessibility
	// related audits.
	Accessibility *LighthouseCategoryV5 `json:"accessibility,omitempty"`
	// BestPractices: The best practices category, containing all best practices
	// related audits.
	BestPractices *LighthouseCategoryV5 `json:"best-practices,omitempty"`
	// Performance: The performance category, containing all performance related
	// audits.
	Performance *LighthouseCategoryV5 `json:"performance,omitempty"`
	// Pwa: The Progressive-Web-App (PWA) category, containing all pwa related
	// audits.
	Pwa *LighthouseCategoryV5 `json:"pwa,omitempty"`
	// Seo: The Search-Engine-Optimization (SEO) category, containing all seo
	// related audits.
	Seo *LighthouseCategoryV5 `json:"seo,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Accessibility") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Accessibility") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *Categories) MarshalJSON() ([]byte, error) {
	type NoMethod Categories
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

// CategoryGroupV5: Message containing a category
type CategoryGroupV5 struct {
	// Description: The description of what the category is grouping
	Description string `json:"description,omitempty"`
	// Title: The human readable title of the group
	Title string `json:"title,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Description") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Description") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *CategoryGroupV5) MarshalJSON() ([]byte, error) {
	type NoMethod CategoryGroupV5
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

// ConfigSettings: Message containing the configuration settings for the
// Lighthouse run.
type ConfigSettings struct {
	// Channel: How Lighthouse was run, e.g. from the Chrome extension or from the
	// npm module.
	Channel string `json:"channel,omitempty"`
	// EmulatedFormFactor: The form factor the emulation should use. This field is
	// deprecated, form_factor should be used instead.
	EmulatedFormFactor string `json:"emulatedFormFactor,omitempty"`
	// FormFactor: How Lighthouse should interpret this run in regards to scoring
	// performance metrics and skipping mobile-only tests in desktop.
	FormFactor string `json:"formFactor,omitempty"`
	// Locale: The locale setting.
	Locale string `json:"locale,omitempty"`
	// OnlyCategories: List of categories of audits the run should conduct.
	OnlyCategories interface{} `json:"onlyCategories,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Channel") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Channel") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *ConfigSettings) MarshalJSON() ([]byte, error) {
	type NoMethod ConfigSettings
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

// Environment: Message containing environment configuration for a Lighthouse
// run.
type Environment struct {
	// BenchmarkIndex: The benchmark index number that indicates rough device
	// class.
	BenchmarkIndex float64 `json:"benchmarkIndex,omitempty"`
	// Credits: The version of libraries with which these results were generated.
	// Ex: axe-core.
	Credits map[string]string `json:"credits,omitempty"`
	// HostUserAgent: The user agent string of the version of Chrome used.
	HostUserAgent string `json:"hostUserAgent,omitempty"`
	// NetworkUserAgent: The user agent string that was sent over the network.
	NetworkUserAgent string `json:"networkUserAgent,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BenchmarkIndex") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BenchmarkIndex") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *Environment) MarshalJSON() ([]byte, error) {
	type NoMethod Environment
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

func (s *Environment) UnmarshalJSON(data []byte) error {
	type NoMethod Environment
	var s1 struct {
		BenchmarkIndex gensupport.JSONFloat64 `json:"benchmarkIndex"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.BenchmarkIndex = float64(s1.BenchmarkIndex)
	return nil
}

// I18n: Message containing the i18n data for the LHR - Version 1.
type I18n struct {
	// RendererFormattedStrings: Internationalized strings that are formatted to
	// the locale in configSettings.
	RendererFormattedStrings *RendererFormattedStrings `json:"rendererFormattedStrings,omitempty"`
	// ForceSendFields is a list of field names (e.g. "RendererFormattedStrings")
	// to unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "RendererFormattedStrings") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *I18n) MarshalJSON() ([]byte, error) {
	type NoMethod I18n
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

// LhrEntity: Message containing an Entity.
type LhrEntity struct {
	// Category: Optional. An optional category name for the entity.
	Category string `json:"category,omitempty"`
	// Homepage: Optional. An optional homepage URL of the entity.
	Homepage string `json:"homepage,omitempty"`
	// IsFirstParty: Optional. An optional flag indicating if the entity is the
	// first party.
	IsFirstParty bool `json:"isFirstParty,omitempty"`
	// IsUnrecognized: Optional. An optional flag indicating if the entity is not
	// recognized.
	IsUnrecognized bool `json:"isUnrecognized,omitempty"`
	// Name: Required. Name of the entity.
	Name string `json:"name,omitempty"`
	// Origins: Required. A list of URL origin strings that belong to this entity.
	Origins []string `json:"origins,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Category") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Category") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *LhrEntity) MarshalJSON() ([]byte, error) {
	type NoMethod LhrEntity
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

// LighthouseAuditResultV5: An audit's result object in a Lighthouse result.
type LighthouseAuditResultV5 struct {
	// Description: The description of the audit.
	Description string `json:"description,omitempty"`
	// Details: Freeform details section of the audit.
	Details googleapi.RawMessage `json:"details,omitempty"`
	// DisplayValue: The value that should be displayed on the UI for this audit.
	DisplayValue string `json:"displayValue,omitempty"`
	// ErrorMessage: An error message from a thrown error inside the audit.
	ErrorMessage string `json:"errorMessage,omitempty"`
	// Explanation: An explanation of the errors in the audit.
	Explanation string `json:"explanation,omitempty"`
	// Id: The audit's id.
	Id string `json:"id,omitempty"`
	// NumericUnit: The unit of the numeric_value field. Used to format the numeric
	// value for display.
	NumericUnit string `json:"numericUnit,omitempty"`
	// NumericValue: A numeric value that has a meaning specific to the audit, e.g.
	// the number of nodes in the DOM or the timestamp of a specific load event.
	// More information can be found in the audit details, if present.
	NumericValue float64 `json:"numericValue,omitempty"`
	// Score: The score of the audit, can be null.
	Score interface{} `json:"score,omitempty"`
	// ScoreDisplayMode: The enumerated score display mode.
	ScoreDisplayMode string `json:"scoreDisplayMode,omitempty"`
	// Title: The human readable title.
	Title string `json:"title,omitempty"`
	// Warnings: Possible warnings that occurred in the audit, can be null.
	Warnings interface{} `json:"warnings,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Description") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Description") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *LighthouseAuditResultV5) MarshalJSON() ([]byte, error) {
	type NoMethod LighthouseAuditResultV5
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

func (s *LighthouseAuditResultV5) UnmarshalJSON(data []byte) error {
	type NoMethod LighthouseAuditResultV5
	var s1 struct {
		NumericValue gensupport.JSONFloat64 `json:"numericValue"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.NumericValue = float64(s1.NumericValue)
	return nil
}

// LighthouseCategoryV5: A Lighthouse category.
type LighthouseCategoryV5 struct {
	// AuditRefs: An array of references to all the audit members of this category.
	AuditRefs []*AuditRefs `json:"auditRefs,omitempty"`
	// Description: A more detailed description of the category and its importance.
	Description string `json:"description,omitempty"`
	// Id: The string identifier of the category.
	Id string `json:"id,omitempty"`
	// ManualDescription: A description for the manual audits in the category.
	ManualDescription string `json:"manualDescription,omitempty"`
	// Score: The overall score of the category, the weighted average of all its
	// audits. (The category's score, can be null.)
	Score interface{} `json:"score,omitempty"`
	// Title: The human-friendly name of the category.
	Title string `json:"title,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AuditRefs") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AuditRefs") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *LighthouseCategoryV5) MarshalJSON() ([]byte, error) {
	type NoMethod LighthouseCategoryV5
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

// LighthouseResultV5: The Lighthouse result object.
type LighthouseResultV5 struct {
	// Audits: Map of audits in the LHR.
	Audits map[string]LighthouseAuditResultV5 `json:"audits,omitempty"`
	// Categories: Map of categories in the LHR.
	Categories *Categories `json:"categories,omitempty"`
	// CategoryGroups: Map of category groups in the LHR.
	CategoryGroups map[string]CategoryGroupV5 `json:"categoryGroups,omitempty"`
	// ConfigSettings: The configuration settings for this LHR.
	ConfigSettings *ConfigSettings `json:"configSettings,omitempty"`
	// Entities: Entity classification data.
	Entities []*LhrEntity `json:"entities,omitempty"`
	// Environment: Environment settings that were used when making this LHR.
	Environment *Environment `json:"environment,omitempty"`
	// FetchTime: The time that this run was fetched.
	FetchTime string `json:"fetchTime,omitempty"`
	// FinalDisplayedUrl: URL displayed on the page after Lighthouse finishes.
	FinalDisplayedUrl string `json:"finalDisplayedUrl,omitempty"`
	// FinalUrl: The final resolved url that was audited.
	FinalUrl string `json:"finalUrl,omitempty"`
	// FullPageScreenshot: Screenshot data of the full page, along with node rects
	// relevant to the audit results.
	FullPageScreenshot interface{} `json:"fullPageScreenshot,omitempty"`
	// I18n: The internationalization strings that are required to render the LHR.
	I18n *I18n `json:"i18n,omitempty"`
	// LighthouseVersion: The lighthouse version that was used to generate this
	// LHR.
	LighthouseVersion string `json:"lighthouseVersion,omitempty"`
	// MainDocumentUrl: URL of the main document request of the final navigation.
	MainDocumentUrl string `json:"mainDocumentUrl,omitempty"`
	// RequestedUrl: The original requested url.
	RequestedUrl string `json:"requestedUrl,omitempty"`
	// RunWarnings: List of all run warnings in the LHR. Will always output to at
	// least `[]`.
	RunWarnings []interface{} `json:"runWarnings,omitempty"`
	// RuntimeError: A top-level error message that, if present, indicates a
	// serious enough problem that this Lighthouse result may need to be discarded.
	RuntimeError *RuntimeError `json:"runtimeError,omitempty"`
	// StackPacks: The Stack Pack advice strings.
	StackPacks []*StackPack `json:"stackPacks,omitempty"`
	// Timing: Timing information for this LHR.
	Timing *Timing `json:"timing,omitempty"`
	// UserAgent: The user agent that was used to run this LHR.
	UserAgent string `json:"userAgent,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Audits") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Audits") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *LighthouseResultV5) MarshalJSON() ([]byte, error) {
	type NoMethod LighthouseResultV5
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

// PagespeedApiLoadingExperienceV5: The CrUX loading experience object that
// contains CrUX data breakdowns.
type PagespeedApiLoadingExperienceV5 struct {
	// Id: The url, pattern or origin which the metrics are on.
	Id string `json:"id,omitempty"`
	// InitialUrl: The requested URL, which may differ from the resolved "id".
	InitialUrl string `json:"initial_url,omitempty"`
	// Metrics: The map of .
	Metrics map[string]UserPageLoadMetricV5 `json:"metrics,omitempty"`
	// OriginFallback: True if the result is an origin fallback from a page, false
	// otherwise.
	OriginFallback bool `json:"origin_fallback,omitempty"`
	// OverallCategory: The human readable speed "category" of the id.
	OverallCategory string `json:"overall_category,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Id") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Id") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *PagespeedApiLoadingExperienceV5) MarshalJSON() ([]byte, error) {
	type NoMethod PagespeedApiLoadingExperienceV5
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

// PagespeedApiPagespeedResponseV5: The Pagespeed API response object.
type PagespeedApiPagespeedResponseV5 struct {
	// AnalysisUTCTimestamp: The UTC timestamp of this analysis.
	AnalysisUTCTimestamp string `json:"analysisUTCTimestamp,omitempty"`
	// CaptchaResult: The captcha verify result
	CaptchaResult string `json:"captchaResult,omitempty"`
	// Id: Canonicalized and final URL for the document, after following page
	// redirects (if any).
	Id string `json:"id,omitempty"`
	// Kind: Kind of result.
	Kind string `json:"kind,omitempty"`
	// LighthouseResult: Lighthouse response for the audit url as an object.
	LighthouseResult *LighthouseResultV5 `json:"lighthouseResult,omitempty"`
	// LoadingExperience: Metrics of end users' page loading experience.
	LoadingExperience *PagespeedApiLoadingExperienceV5 `json:"loadingExperience,omitempty"`
	// OriginLoadingExperience: Metrics of the aggregated page loading experience
	// of the origin
	OriginLoadingExperience *PagespeedApiLoadingExperienceV5 `json:"originLoadingExperience,omitempty"`
	// Version: The version of PageSpeed used to generate these results.
	Version *PagespeedVersion `json:"version,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AnalysisUTCTimestamp") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AnalysisUTCTimestamp") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *PagespeedApiPagespeedResponseV5) MarshalJSON() ([]byte, error) {
	type NoMethod PagespeedApiPagespeedResponseV5
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

// PagespeedVersion: The Pagespeed Version object.
type PagespeedVersion struct {
	// Major: The major version number of PageSpeed used to generate these results.
	Major string `json:"major,omitempty"`
	// Minor: The minor version number of PageSpeed used to generate these results.
	Minor string `json:"minor,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Major") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Major") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *PagespeedVersion) MarshalJSON() ([]byte, error) {
	type NoMethod PagespeedVersion
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

// RendererFormattedStrings: Message holding the formatted strings used in the
// renderer.
type RendererFormattedStrings struct {
	// AuditGroupExpandTooltip: The tooltip text on an expandable chevron icon.
	AuditGroupExpandTooltip string `json:"auditGroupExpandTooltip,omitempty"`
	// CalculatorLink: Text link pointing to the Lighthouse scoring calculator.
	// This link immediately follows a sentence stating the performance score is
	// calculated from the perf metrics.
	CalculatorLink string `json:"calculatorLink,omitempty"`
	// CrcInitialNavigation: The label for the initial request in a critical
	// request chain.
	CrcInitialNavigation string `json:"crcInitialNavigation,omitempty"`
	// CrcLongestDurationLabel: The label for values shown in the summary of
	// critical request chains.
	CrcLongestDurationLabel string `json:"crcLongestDurationLabel,omitempty"`
	// DropdownCopyJSON: Option in a dropdown menu that copies the Lighthouse JSON
	// object to the system clipboard.
	DropdownCopyJSON string `json:"dropdownCopyJSON,omitempty"`
	// DropdownDarkTheme: Option in a dropdown menu that toggles the themeing of
	// the report between Light(default) and Dark themes.
	DropdownDarkTheme string `json:"dropdownDarkTheme,omitempty"`
	// DropdownPrintExpanded: Option in a dropdown menu that opens a full
	// Lighthouse report in a print dialog.
	DropdownPrintExpanded string `json:"dropdownPrintExpanded,omitempty"`
	// DropdownPrintSummary: Option in a dropdown menu that opens a small, summary
	// report in a print dialog.
	DropdownPrintSummary string `json:"dropdownPrintSummary,omitempty"`
	// DropdownSaveGist: Option in a dropdown menu that saves the current report as
	// a new GitHub Gist.
	DropdownSaveGist string `json:"dropdownSaveGist,omitempty"`
	// DropdownSaveHTML: Option in a dropdown menu that saves the Lighthouse report
	// HTML locally to the system as a '.html' file.
	DropdownSaveHTML string `json:"dropdownSaveHTML,omitempty"`
	// DropdownSaveJSON: Option in a dropdown menu that saves the Lighthouse JSON
	// object to the local system as a '.json' file.
	DropdownSaveJSON string `json:"dropdownSaveJSON,omitempty"`
	// DropdownViewer: Option in a dropdown menu that opens the current report in
	// the Lighthouse Viewer Application.
	DropdownViewer string `json:"dropdownViewer,omitempty"`
	// ErrorLabel: The label shown next to an audit or metric that has had an
	// error.
	ErrorLabel string `json:"errorLabel,omitempty"`
	// ErrorMissingAuditInfo: The error string shown next to an erroring audit.
	ErrorMissingAuditInfo string `json:"errorMissingAuditInfo,omitempty"`
	// FooterIssue: Label for button to create an issue against the Lighthouse
	// GitHub project.
	FooterIssue string `json:"footerIssue,omitempty"`
	// LabDataTitle: The title of the lab data performance category.
	LabDataTitle string `json:"labDataTitle,omitempty"`
	// LsPerformanceCategoryDescription: The disclaimer shown under performance
	// explaining that the network can vary.
	LsPerformanceCategoryDescription string `json:"lsPerformanceCategoryDescription,omitempty"`
	// ManualAuditsGroupTitle: The heading shown above a list of audits that were
	// not computerd in the run.
	ManualAuditsGroupTitle string `json:"manualAuditsGroupTitle,omitempty"`
	// NotApplicableAuditsGroupTitle: The heading shown above a list of audits that
	// do not apply to a page.
	NotApplicableAuditsGroupTitle string `json:"notApplicableAuditsGroupTitle,omitempty"`
	// OpportunityResourceColumnLabel: The heading for the estimated page load
	// savings opportunity of an audit.
	OpportunityResourceColumnLabel string `json:"opportunityResourceColumnLabel,omitempty"`
	// OpportunitySavingsColumnLabel: The heading for the estimated page load
	// savings of opportunity audits.
	OpportunitySavingsColumnLabel string `json:"opportunitySavingsColumnLabel,omitempty"`
	// PassedAuditsGroupTitle: The heading that is shown above a list of audits
	// that are passing.
	PassedAuditsGroupTitle string `json:"passedAuditsGroupTitle,omitempty"`
	// RuntimeDesktopEmulation: Descriptive explanation for emulation setting when
	// emulating a generic desktop form factor, as opposed to a mobile-device like
	// form factor.
	RuntimeDesktopEmulation string `json:"runtimeDesktopEmulation,omitempty"`
	// RuntimeMobileEmulation: Descriptive explanation for emulation setting when
	// emulating a Nexus 5X mobile device.
	RuntimeMobileEmulation string `json:"runtimeMobileEmulation,omitempty"`
	// RuntimeNoEmulation: Descriptive explanation for emulation setting when no
	// device emulation is set.
	RuntimeNoEmulation string `json:"runtimeNoEmulation,omitempty"`
	// RuntimeSettingsAxeVersion: Label for a row in a table that shows the version
	// of the Axe library used
	RuntimeSettingsAxeVersion string `json:"runtimeSettingsAxeVersion,omitempty"`
	// RuntimeSettingsBenchmark: Label for a row in a table that shows the
	// estimated CPU power of the machine running Lighthouse. Example row values:
	// 532, 1492, 783.
	RuntimeSettingsBenchmark string `json:"runtimeSettingsBenchmark,omitempty"`
	// RuntimeSettingsCPUThrottling: Label for a row in a table that describes the
	// CPU throttling conditions that were used during a Lighthouse run, if any.
	RuntimeSettingsCPUThrottling string `json:"runtimeSettingsCPUThrottling,omitempty"`
	// RuntimeSettingsChannel: Label for a row in a table that shows in what tool
	// Lighthouse is being run (e.g. The lighthouse CLI, Chrome DevTools,
	// Lightrider, WebPageTest, etc).
	RuntimeSettingsChannel string `json:"runtimeSettingsChannel,omitempty"`
	// RuntimeSettingsDevice: Label for a row in a table that describes the kind of
	// device that was emulated for the Lighthouse run. Example values for row
	// elements: 'No Emulation', 'Emulated Desktop', etc.
	RuntimeSettingsDevice string `json:"runtimeSettingsDevice,omitempty"`
	// RuntimeSettingsFetchTime: Label for a row in a table that shows the time at
	// which a Lighthouse run was conducted; formatted as a timestamp, e.g. Jan 1,
	// 1970 12:00 AM UTC.
	RuntimeSettingsFetchTime string `json:"runtimeSettingsFetchTime,omitempty"`
	// RuntimeSettingsNetworkThrottling: Label for a row in a table that describes
	// the network throttling conditions that were used during a Lighthouse run, if
	// any.
	RuntimeSettingsNetworkThrottling string `json:"runtimeSettingsNetworkThrottling,omitempty"`
	// RuntimeSettingsTitle: Title of the Runtime settings table in a Lighthouse
	// report. Runtime settings are the environment configurations that a specific
	// report used at auditing time.
	RuntimeSettingsTitle string `json:"runtimeSettingsTitle,omitempty"`
	// RuntimeSettingsUA: Label for a row in a table that shows the User Agent that
	// was detected on the Host machine that ran Lighthouse.
	RuntimeSettingsUA string `json:"runtimeSettingsUA,omitempty"`
	// RuntimeSettingsUANetwork: Label for a row in a table that shows the User
	// Agent that was used to send out all network requests during the Lighthouse
	// run.
	RuntimeSettingsUANetwork string `json:"runtimeSettingsUANetwork,omitempty"`
	// RuntimeSettingsUrl: Label for a row in a table that shows the URL that was
	// audited during a Lighthouse run.
	RuntimeSettingsUrl string `json:"runtimeSettingsUrl,omitempty"`
	// RuntimeUnknown: Descriptive explanation for a runtime setting that is set to
	// an unknown value.
	RuntimeUnknown string `json:"runtimeUnknown,omitempty"`
	// ScorescaleLabel: The label that explains the score gauges scale (0-49,
	// 50-89, 90-100).
	ScorescaleLabel string `json:"scorescaleLabel,omitempty"`
	// ShowRelevantAudits: Label preceding a radio control for filtering the list
	// of audits. The radio choices are various performance metrics (FCP, LCP,
	// TBT), and if chosen, the audits in the report are hidden if they are not
	// relevant to the selected metric.
	ShowRelevantAudits string `json:"showRelevantAudits,omitempty"`
	// SnippetCollapseButtonLabel: The label for the button to show only a few
	// lines of a snippet
	SnippetCollapseButtonLabel string `json:"snippetCollapseButtonLabel,omitempty"`
	// SnippetExpandButtonLabel: The label for the button to show all lines of a
	// snippet
	SnippetExpandButtonLabel string `json:"snippetExpandButtonLabel,omitempty"`
	// ThirdPartyResourcesLabel: This label is for a filter checkbox above a table
	// of items
	ThirdPartyResourcesLabel string `json:"thirdPartyResourcesLabel,omitempty"`
	// ThrottlingProvided: Descriptive explanation for environment throttling that
	// was provided by the runtime environment instead of provided by Lighthouse
	// throttling.
	ThrottlingProvided string `json:"throttlingProvided,omitempty"`
	// ToplevelWarningsMessage: The label shown preceding important warnings that
	// may have invalidated an entire report.
	ToplevelWarningsMessage string `json:"toplevelWarningsMessage,omitempty"`
	// VarianceDisclaimer: The disclaimer shown below a performance metric value.
	VarianceDisclaimer string `json:"varianceDisclaimer,omitempty"`
	// ViewTreemapLabel: Label for a button that opens the Treemap App
	ViewTreemapLabel string `json:"viewTreemapLabel,omitempty"`
	// WarningAuditsGroupTitle: The heading that is shown above a list of audits
	// that have warnings
	WarningAuditsGroupTitle string `json:"warningAuditsGroupTitle,omitempty"`
	// WarningHeader: The label shown above a bulleted list of warnings.
	WarningHeader string `json:"warningHeader,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AuditGroupExpandTooltip") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AuditGroupExpandTooltip") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *RendererFormattedStrings) MarshalJSON() ([]byte, error) {
	type NoMethod RendererFormattedStrings
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

// RuntimeError: Message containing a runtime error config.
type RuntimeError struct {
	// Code: The enumerated Lighthouse Error code.
	Code string `json:"code,omitempty"`
	// Message: A human readable message explaining the error code.
	Message string `json:"message,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Code") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Code") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *RuntimeError) MarshalJSON() ([]byte, error) {
	type NoMethod RuntimeError
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

// StackPack: Message containing Stack Pack information.
type StackPack struct {
	// Descriptions: The stack pack advice strings.
	Descriptions map[string]string `json:"descriptions,omitempty"`
	// IconDataURL: The stack pack icon data uri.
	IconDataURL string `json:"iconDataURL,omitempty"`
	// Id: The stack pack id.
	Id string `json:"id,omitempty"`
	// Title: The stack pack title.
	Title string `json:"title,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Descriptions") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Descriptions") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *StackPack) MarshalJSON() ([]byte, error) {
	type NoMethod StackPack
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

// Timing: Message containing the performance timing data for the Lighthouse
// run.
type Timing struct {
	// Total: The total duration of Lighthouse's run.
	Total float64 `json:"total,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Total") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Total") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *Timing) MarshalJSON() ([]byte, error) {
	type NoMethod Timing
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

func (s *Timing) UnmarshalJSON(data []byte) error {
	type NoMethod Timing
	var s1 struct {
		Total gensupport.JSONFloat64 `json:"total"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Total = float64(s1.Total)
	return nil
}

// UserPageLoadMetricV5: A CrUX metric object for a single metric and form
// factor.
type UserPageLoadMetricV5 struct {
	// Category: The category of the specific time metric.
	Category string `json:"category,omitempty"`
	// Distributions: Metric distributions. Proportions should sum up to 1.
	Distributions []*Bucket `json:"distributions,omitempty"`
	// FormFactor: Identifies the form factor of the metric being collected.
	FormFactor string `json:"formFactor,omitempty"`
	// Median: The median number of the metric, in millisecond.
	Median int64 `json:"median,omitempty"`
	// MetricId: Identifies the type of the metric.
	MetricId string `json:"metricId,omitempty"`
	// Percentile: We use this field to store certain percentile value for this
	// metric. For v4, this field contains pc50. For v5, this field contains pc90.
	Percentile int64 `json:"percentile,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Category") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Category") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s *UserPageLoadMetricV5) MarshalJSON() ([]byte, error) {
	type NoMethod UserPageLoadMetricV5
	return gensupport.MarshalJSON(NoMethod(*s), s.ForceSendFields, s.NullFields)
}

type PagespeedapiRunpagespeedCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Runpagespeed: Runs PageSpeed analysis on the page at the specified URL, and
// returns PageSpeed scores, a list of suggestions to make that page faster,
// and other information.
//
// - url: The URL to fetch and analyze.
func (r *PagespeedapiService) Runpagespeed(url string) *PagespeedapiRunpagespeedCall {
	c := &PagespeedapiRunpagespeedCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.urlParams_.Set("url", url)
	return c
}

// CaptchaToken sets the optional parameter "captchaToken": The captcha token
// passed when filling out a captcha.
func (c *PagespeedapiRunpagespeedCall) CaptchaToken(captchaToken string) *PagespeedapiRunpagespeedCall {
	c.urlParams_.Set("captchaToken", captchaToken)
	return c
}

// Category sets the optional parameter "category": A Lighthouse category to
// run; if none are given, only Performance category will be run
//
// Possible values:
//
//	"CATEGORY_UNSPECIFIED" - Default UNDEFINED category.
//	"ACCESSIBILITY" - Accessibility (a11y), category pertaining to a website's
//
// capacity to be accessible to all users.
//
//	"BEST_PRACTICES" - Best Practices, category pertaining to a website's
//
// conformance to web best practice.
//
//	"PERFORMANCE" - Performance, category pertaining to a website's
//
// performance.
//
//	"PWA" - Progressive Web App (PWA), category pertaining to a website's
//
// ability to be run as a PWA.
//
//	"SEO" - Search Engine Optimization (SEO), category pertaining to a
//
// website's ability to be indexed by search engines.
func (c *PagespeedapiRunpagespeedCall) Category(category ...string) *PagespeedapiRunpagespeedCall {
	c.urlParams_.SetMulti("category", append([]string{}, category...))
	return c
}

// Locale sets the optional parameter "locale": The locale used to localize
// formatted results
func (c *PagespeedapiRunpagespeedCall) Locale(locale string) *PagespeedapiRunpagespeedCall {
	c.urlParams_.Set("locale", locale)
	return c
}

// Strategy sets the optional parameter "strategy": The analysis strategy
// (desktop or mobile) to use, and desktop is the default
//
// Possible values:
//
//	"STRATEGY_UNSPECIFIED" - UNDEFINED.
//	"DESKTOP" - Fetch and analyze the URL for desktop browsers.
//	"MOBILE" - Fetch and analyze the URL for mobile devices.
func (c *PagespeedapiRunpagespeedCall) Strategy(strategy string) *PagespeedapiRunpagespeedCall {
	c.urlParams_.Set("strategy", strategy)
	return c
}

// UtmCampaign sets the optional parameter "utm_campaign": Campaign name for
// analytics.
func (c *PagespeedapiRunpagespeedCall) UtmCampaign(utmCampaign string) *PagespeedapiRunpagespeedCall {
	c.urlParams_.Set("utm_campaign", utmCampaign)
	return c
}

// UtmSource sets the optional parameter "utm_source": Campaign source for
// analytics.
func (c *PagespeedapiRunpagespeedCall) UtmSource(utmSource string) *PagespeedapiRunpagespeedCall {
	c.urlParams_.Set("utm_source", utmSource)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PagespeedapiRunpagespeedCall) Fields(s ...googleapi.Field) *PagespeedapiRunpagespeedCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *PagespeedapiRunpagespeedCall) IfNoneMatch(entityTag string) *PagespeedapiRunpagespeedCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PagespeedapiRunpagespeedCall) Context(ctx context.Context) *PagespeedapiRunpagespeedCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PagespeedapiRunpagespeedCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PagespeedapiRunpagespeedCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "pagespeedonline/v5/runPagespeed")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "pagespeedonline.pagespeedapi.runpagespeed" call.
// Any non-2xx status code is an error. Response headers are in either
// *PagespeedApiPagespeedResponseV5.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *PagespeedapiRunpagespeedCall) Do(opts ...googleapi.CallOption) (*PagespeedApiPagespeedResponseV5, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &PagespeedApiPagespeedResponseV5{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}
